package Bencher::Scenario::LevenshteinModules;

our $DATE = '2016-01-11'; # DATE
our $VERSION = '0.07'; # VERSION

use 5.010001;
use strict;
use utf8;
use warnings;

our $scenario = {
    summary => 'Benchmark various modules calculating the Levenshtein edit distance',
    modules => {
        'Text::Levenshtein' => 0.11,
    },
    participants => [
        {
            fcall_template => "PERLANCAR::Text::Levenshtein::editdist(<word1>, <word2>)",
            tags => ['no_unicode_support'],
        },
        {
            fcall_template => "Text::Levenshtein::fastdistance(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::Levenshtein::XS::distance(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::LevenshteinXS::distance(<word1>, <word2>)",
            tags => ['no_unicode_support'],
        },
        {
            fcall_template => "Text::Levenshtein::Damerau::PP::pp_edistance(<word1>, <word2>)",
        },
        {
            fcall_template => "Text::Levenshtein::Damerau::XS::xs_edistance(<word1>, <word2>)",
        },
    ],
    datasets => [
        { name=>"a",       args => {word1=>"a"      , word2=>"aa"},      result => 1 },
        { name=>"foo",     args => {word1=>"foo"    , word2=>"bar"},     result => 3 },
        { name=>"program", args => {word1=>"program", word2=>"porgram"}, result => 2 },
        { name=>"reve"   , args => {word1=>"reve"   , word2=>"rêves"},   result => 3, tags=>['unicode'], exclude_participant_tags=>['no_unicode_support'] },
        { name=>"euro"   , args => {word1=>"Euro"   , word2=>"€uro"},    result => 1, tags=>['unicode'], exclude_participant_tags=>['no_unicode_support'] },
    ],
    on_result_failure => 'warn',
};

1;
# ABSTRACT: Benchmark various modules calculating the Levenshtein edit distance

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::LevenshteinModules - Benchmark various modules calculating the Levenshtein edit distance

=head1 VERSION

This document describes version 0.07 of Bencher::Scenario::LevenshteinModules (from Perl distribution Bencher-Scenario-LevenshteinModules), released on 2016-01-11.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m LevenshteinModules

To run module startup overhead benchmark:

 % bencher --module-startup -m LevenshteinModules

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 BENCHMARKED MODULES

L<PERLANCAR::Text::Levenshtein> 0.02

L<Text::Levenshtein> 0.13

L<Text::Levenshtein::Damerau::PP> 0.25

L<Text::Levenshtein::Damerau::XS> 3.0

L<Text::Levenshtein::XS> 0.502

L<Text::LevenshteinXS> 0.03

=head1 BENCHMARK PARTICIPANTS

=over

=item * PERLANCAR::Text::Levenshtein::editdist (perl_code)

Function call template:

 PERLANCAR::Text::Levenshtein::editdist(<word1>, <word2>)



=item * Text::Levenshtein::fastdistance (perl_code)

Function call template:

 Text::Levenshtein::fastdistance(<word1>, <word2>)



=item * Text::Levenshtein::XS::distance (perl_code)

Function call template:

 Text::Levenshtein::XS::distance(<word1>, <word2>)



=item * Text::LevenshteinXS::distance (perl_code)

Function call template:

 Text::LevenshteinXS::distance(<word1>, <word2>)



=item * Text::Levenshtein::Damerau::PP::pp_edistance (perl_code)

Function call template:

 Text::Levenshtein::Damerau::PP::pp_edistance(<word1>, <word2>)



=item * Text::Levenshtein::Damerau::XS::xs_edistance (perl_code)

Function call template:

 Text::Levenshtein::Damerau::XS::xs_edistance(<word1>, <word2>)



=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.22.0 >>, CPU: I<< Intel(R) Core(TM) i7-4770 CPU @ 3.40GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.0 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark with default option:

 +-----+----------------------------------------------+---------+-------------+------------+---------+---------+-----------------------------------+
 | seq | participant                                  | dataset | rate        | time       | errors  | samples | notes                             |
 +-----+----------------------------------------------+---------+-------------+------------+---------+---------+-----------------------------------+
 | 5   | Text::Levenshtein::fastdistance              | program | 1.89e+04    | 52.9μs     | 2.1e-08 | 32      |                                   |
 | 18  | Text::Levenshtein::Damerau::PP::pp_edistance | program | 2.25e+04    | 44.4μs     | 5.3e-08 | 20      | Result (1) is not as expected (2) |
 | 2   | PERLANCAR::Text::Levenshtein::editdist       | program | 3.019e+04   | 33.12μs    | 1e-08   | 34      |                                   |
 | 6   | Text::Levenshtein::fastdistance              | reve    | 4.38e+04    | 22.8μs     | 2.8e-08 | 29      | Result (2) is not as expected (3) |
 | 19  | Text::Levenshtein::Damerau::PP::pp_edistance | reve    | 4.937e+04   | 20.26μs    | 5.6e-09 | 28      | Result (2) is not as expected (3) |
 | 7   | Text::Levenshtein::fastdistance              | euro    | 5.08e+04    | 19.7μs     | 6.7e-09 | 20      |                                   |
 | 20  | Text::Levenshtein::Damerau::PP::pp_edistance | euro    | 5.15e+04    | 19.4μs     | 6.7e-09 | 20      |                                   |
 | 4   | Text::Levenshtein::fastdistance              | foo     | 89590.9     | 11.1619μs  | 0       | 20      |                                   |
 | 17  | Text::Levenshtein::Damerau::PP::pp_edistance | foo     | 9.32e+04    | 10.7μs     | 1.1e-08 | 31      |                                   |
 | 1   | PERLANCAR::Text::Levenshtein::editdist       | foo     | 114265      | 8.75158μs  | 0       | 20      |                                   |
 | 16  | Text::Levenshtein::Damerau::PP::pp_edistance | a       | 2.86e+05    | 3.5μs      | 1.7e-09 | 20      |                                   |
 | 3   | Text::Levenshtein::fastdistance              | a       | 2.88e+05    | 3.47μs     | 6.7e-09 | 31      |                                   |
 | 0   | PERLANCAR::Text::Levenshtein::editdist       | a       | 4.37e+05    | 2.29μs     | 3.3e-09 | 20      |                                   |
 | 23  | Text::Levenshtein::Damerau::XS::xs_edistance | program | 5.71e+05    | 1.75μs     | 6.8e-10 | 30      | Result (1) is not as expected (2) |
 | 10  | Text::Levenshtein::XS::distance              | program | 6.3e+05     | 1.59μs     | 3.3e-09 | 20      |                                   |
 | 24  | Text::Levenshtein::Damerau::XS::xs_edistance | reve    | 7.71e+05    | 1.3μs      | 1.5e-09 | 25      | Result (2) is not as expected (3) |
 | 25  | Text::Levenshtein::Damerau::XS::xs_edistance | euro    | 8.11e+05    | 1.23μs     | 1.7e-09 | 20      |                                   |
 | 11  | Text::Levenshtein::XS::distance              | reve    | 823523      | 1.21429μs  | 0       | 26      | Result (2) is not as expected (3) |
 | 12  | Text::Levenshtein::XS::distance              | euro    | 9.13e+05    | 1.09μs     | 1.6e-09 | 21      |                                   |
 | 22  | Text::Levenshtein::Damerau::XS::xs_edistance | foo     | 9.2e+05     | 1.09μs     | 4.2e-10 | 20      |                                   |
 | 9   | Text::Levenshtein::XS::distance              | foo     | 1.02051e+06 | 0.979898μs | 0       | 20      |                                   |
 | 21  | Text::Levenshtein::Damerau::XS::xs_edistance | a       | 1.27499e+06 | 0.784323μs | 0       | 20      |                                   |
 | 8   | Text::Levenshtein::XS::distance              | a       | 1.28723e+06 | 0.776861μs | 0       | 20      |                                   |
 | 15  | Text::LevenshteinXS::distance                | program | 4.382e+06   | 0.2282μs   | 1.1e-11 | 20      |                                   |
 | 14  | Text::LevenshteinXS::distance                | foo     | 9.31e+06    | 0.107μs    | 4.3e-11 | 29      |                                   |
 | 13  | Text::LevenshteinXS::distance                | a       | 1.16e+07    | 0.0861μs   | 1.7e-10 | 31      |                                   |
 +-----+----------------------------------------------+---------+-------------+------------+---------+---------+-----------------------------------+


Benchmark module startup overhead:

 +-----+--------------------------------+--------+-------------------+---------+---------+-------+
 | seq | participant                    | time   | mod_overhead_time | errors  | samples | notes |
 +-----+--------------------------------+--------+-------------------+---------+---------+-------+
 | 4   | Text::LevenshteinXS            | 8.35ms | 5.98ms            | 2.1e-05 | 20      |       |
 | 2   | Text::Levenshtein              | 7ms    | 4.7ms             | 3.5e-05 | 20      |       |
 | 3   | Text::Levenshtein::XS          | 4.8ms  | 2.5ms             | 2.4e-05 | 20      |       |
 | 5   | Text::Levenshtein::Damerau::PP | 4.08ms | 1.71ms            | 1.3e-05 | 20      |       |
 | 6   | Text::Levenshtein::Damerau::XS | 3.73ms | 1.35ms            | 1e-05   | 20      |       |
 | 1   | PERLANCAR::Text::Levenshtein   | 2.6ms  | 0.24ms            | 8.7e-06 | 22      |       |
 | 0   | perl -e1 (baseline)            | 2.37ms | 0ms               | 7.4e-06 | 21      |       |
 +-----+--------------------------------+--------+-------------------+---------+---------+-------+

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenario-LevenshteinModules>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenario-LevenshteinModules>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenario-LevenshteinModules>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
