package App::BIP39Utils;

our $DATE = '2018-01-01'; # DATE
our $VERSION = '0.001'; # VERSION

use 5.010001;
use strict;
use warnings;

our %SPEC;

$SPEC{gen_bip39_mnemonic_phrase} = {
    v => 1.1,
    summary => 'Generate BIP39 mnemonic phrase',
    args => {
        language => {
            summary => 'Pick which language wordlist to use',
            schema => ['str*', match=>qr/\A\w{2}(?:-\w+)?\z/],
            default => 'en',
            description => <<'_',

Will retrieve wordlist from `WordList::<LANG_CODE>::BIP39` Perl module.

For Chinese (simplified), use `zh-simplified`. For Chinese (traditional), use
`zh-traditional`.

_
        },
        length => {
            summary => 'Number of words to produce',
            schema => ['posint*', max=>128],
            default => 12,
        },
    },
};
sub gen_bip39_mnemonic_phrase {
    require Math::Random::Secure;

    my %args = @_;

    my ($langcode, $variant) = ($args{language} // 'en') =~ /\A(\w{2})(?:-(\w+))?\z/
        or return [400, "Invalid language '$args{language}', please specify a 2-digit language code"];
    my $mod = "WordList::".uc($langcode).($variant ? "::".ucfirst(lc($variant)) : "")."::BIP39";
    (my $mod_pm = "$mod.pm") =~ s!::!/!g;
    require $mod_pm;

    my @all_words = $mod->new->all_words;

    my @words;
    my $length = $args{length} // 12;
    for (1..$length) {
        my $word = $all_words[@all_words * Math::Random::Secure::rand()];
        redo if grep { $word eq $_ } @words;
        push @words, $word;
    }

    [200, "OK", join(" ", @words)];
}

1;
# ABSTRACT: Collection of CLI utilities related to BIP39

__END__

=pod

=encoding UTF-8

=head1 NAME

App::BIP39Utils - Collection of CLI utilities related to BIP39

=head1 VERSION

This document describes version 0.001 of App::BIP39Utils (from Perl distribution App-BIP39Utils), released on 2018-01-01.

=head1 DESCRIPTION

This distribution provides the following command-line utilities related to
BIP39:

=over

=item * L<gen-bip39-mnemonic-phrase>

=back

Keywords: bitcoin, cryptocurrency, BIP, bitcoin improvement proposal, mnemonic
phrase.

=head1 FUNCTIONS


=head2 gen_bip39_mnemonic_phrase

Usage:

 gen_bip39_mnemonic_phrase(%args) -> [status, msg, result, meta]

Generate BIP39 mnemonic phrase.

This function is not exported.

Arguments ('*' denotes required arguments):

=over 4

=item * B<language> => I<str> (default: "en")

Pick which language wordlist to use.

Will retrieve wordlist from C<< WordList::E<lt>LANG_CODEE<gt>::BIP39 >> Perl module.

For Chinese (simplified), use C<zh-simplified>. For Chinese (traditional), use
C<zh-traditional>.

=item * B<length> => I<posint> (default: 12)

Number of words to produce.

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (any)

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/App-BIP39Utils>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-App-BIP39Utils>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=App-BIP39Utils>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 SEE ALSO

L<https://en.bitcoin.it/wiki/Mnemonic_phrase>

L<https://github.com/bitcoin/bips/blob/master/bip-0039.mediawiki>

L<WordList::EN::BIP39> and BIP39 for the other languages in
C<WordList::*::BIP39>.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2018 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
