#!/usr/local/bin/perl
#
#Perl wrappers for Slurm sshare cmd

package Slurm::Sshare;
use strict;
use warnings;
use base qw(Class::Accessor);
use Carp qw(carp croak);

use version; our $VERSION = qw(1.0.0);

#-------------------------------------------------------------------
#	Globals
#-------------------------------------------------------------------

my $VERBOSE;
#By default, use the sshare command found in your PATH
my $SSHARE_CMD = 'sshare'; 
#IF you wish to default to something else, uncomment and change line below
#$SSHARE_CMD = '/usr/local/bin/sshare'; 

#These are intended for regression tests only
my $_last_raw_output;
sub _sshare_last_raw_output($)
{       return $_last_raw_output;
}

#And because each sshare_list can result in multiple calls to sshare,
#a variant which saves all output from give sshare_list command
my $_sslist_last_raw_output;
sub _sshare_list_last_raw_output($)
{	return $_sslist_last_raw_output;
}

#-------------------------------------------------------------------
#	Class methods
#-------------------------------------------------------------------

sub verbose($;$)
{	my $class = shift;
	my $new = shift;
	if ( defined $new )
	{	$VERBOSE = $new?1:0;
	}
	return $VERBOSE;
}

sub sshare($;$)
{	my $class = shift;
	my $new = shift;
	if ( $new )
	{	$SSHARE_CMD = $new;
	}
	return $SSHARE_CMD;
}

#-------------------------------------------------------------------
#	Accessors
#-------------------------------------------------------------------

my @rw_accessors = qw(
);

my @sshare_fields_in_order = qw(
	account
	user
	raw_shares
	normalized_shares
	raw_usage
	normalized_usage
	effective_usage
	fairshare
	grpcpumins
	cpurunmins
);

my @ro_accessors = (@sshare_fields_in_order, 'cluster');

__PACKAGE__->mk_accessors(@rw_accessors);
__PACKAGE__->mk_ro_accessors(@ro_accessors);


my @required_parms = qw(
);

#-------------------------------------------------------------------
#	Constructors, etc
#-------------------------------------------------------------------

sub new($;@)
{	my $class = shift;
	$class = ref($class) if ref($class);
	my @args = @_;

	my $obj = {};
	bless $obj, $class;

	$obj->_parse_args(@args);
	$obj->_set_defaults;
	$obj->_init;

	return $obj;
}

sub _parse_args($@)
{	my $obj = shift;
	my %args = @_;

	my ($arg, $meth, $val);
	RWARG: foreach $arg (@rw_accessors)
	{	next RWARG unless exists $args{$arg};
		$val = delete $args{$arg};
		next RWARG unless defined $val;
		$meth = $arg;
		$obj->$meth($val);
	}

	ROARG: foreach $arg (@ro_accessors)
	{	next ROARG unless exists $args{$arg};
		$val = delete $args{$arg};
		next ROARG unless defined $val;
		$meth = $arg;
		$obj->set($meth,$val);
	}


	#Warn about unknown arguments
	if ( scalar(keys %args) )
	{	my $tmp = join ", ", (keys %args);
		croak "Unrecognized arguments [ $tmp ] to constructor at ";
	};
}

sub _set_defaults($)
{	my $obj = shift;
	return;
}

sub _init($)
{	my $obj = shift;

	my ($fld, $meth, $val);
	foreach $fld (@required_parms)
	{	$meth = $fld;
		$val = $obj->$meth;
		unless ( defined $val )
		{	croak "Missing required argument $fld";
		}
	}

}

#-------------------------------------------------------------------
#	Class methods to Run basic sshare commands
#-------------------------------------------------------------------

sub _noshell_backticks($$@)
#Calls an external command using pipes and forks so no shell gets invoked
#Returns ($err, @out) where $err is the error is the exit status of
#the command, and @out is the list of output returned, line by line.
#If $mode is 0, only STDOUT is returned in @out, 
#If $mode is non-zero, STDERR is dupped onto STDOUT and also returned.
{	my $class = shift;
	my $mode = shift;
	my @cmd = @_;

	my ($err, @out, $PIPE, $res);
	my $chd_excode=254;

	if ( $res = open($PIPE, "-|" )  )
	{	#Parent
		if ( ! defined $res )
		{	my $tmp = join ' ', @cmd;
			die "Pipe to '$tmp' failed: $!";
		}
		@out = <$PIPE>;
		$res = close $PIPE;
		$err = $?;
		if ( $err && ( ($err >> 8) == $chd_excode ) )
		{	#We (probably?) got an exception running exec in child process
			#Re-raise the exception
			my $exc = join '', @out;
			die $exc;
		}
		return ($err, @out);
	} else
	{	#Child
		#Duplicate stderr onto stdout if so requested
		if ( $mode )
		{	unless ( open(STDERR, '>&STDOUT') )
			{	print "Cannot dup stderr to stdout in child";
				die "Cannot dup stderr to stdout in child";
			}
		}
		#Wrap exec in an eval, and exit on error, not die.
		#Otherwise, if _noshell_backticks_ is put in an eval block,
		#an exception raised by exec (e.g. taint issues) will result in
		#BOTH child and parent continuing from the user's eval block.
		#See e.g http://www.perlmonks.org/?node_id=166538
		eval { exec { $cmd[0] } @cmd; #Not subject to shell escapes
		};
		#We only reach here if exec raised an exception
		warn "$@" if $@;
		exit $chd_excode;
	}
}

sub _run_generic_sshare_cmd_always($@)
#Run a generic sshare cmd, returning results
#Does NOT honors dryrun.
#This should only be called directly for commands which do NOT
#modify the database
{	my $class = shift;
	my @args = @_;

	my $cmd = $class->sshare;
	my $mode = 1;
	if ( $class->verbose )
	{	#Verbose mode:
		#Output command before executing it
		my @tmp = ($class->sshare, @args);
		my $tmpcmd = join ' ', @tmp;
		print STDERR "[VERBOSE] $tmpcmd\n";
	}
	my ($err, @out ) = $class->_noshell_backticks($mode, $cmd, @args);
        $_last_raw_output = [ @out ];
	return ($err, @out);
}

sub run_generic_sshare_cmd($@)
#Run a generic sshare cmd, returning results
#No dryrun mode, as sshare is view only anyway.  Also, no --readonly flag.
#
#Returns list ref of output, line by line on success.
#Returns non-ref error message on error.
{	my $class = shift;
	my @args = @_;

	my ($err, @out) = $class->_run_generic_sshare_cmd_always(@args);
	if ( $err )
	{	my $errstr = "Exit code: $err";
		my $output = join "\n", @out;
		$errstr .= "\n$output" if $output;
		return $errstr;
	}
	#return $err if $err;
	return [ @out ];
	
}

sub run_generic_sshare_list_command($@)
#Runs a sshare list command, returns output as a list ref of list refs
#of the fields in order. 
#(Tried using hash refs, but those get abbreviated.  Ugh!)
#'list' should be included in the command spec
#
#On error, returns non-ref error message/error code
#
#Appends --parsable2 and --noheader to the command
{	my $class = shift;
	my @cmd = @_;

	push @cmd, '--parsable2', '--noheader', '--long';
	my $lines = $class->run_generic_sshare_cmd(@cmd);

	return $lines unless $lines && ref($lines) eq 'ARRAY';

	my $results = [];
	LINE: foreach my $line (@$lines)
	{	chomp $line;
		next LINE unless $line =~ /\|/;
		my @values = split /\|/, $line;
		push @$results, \@values;
	}

	return $results;
}

#-------------------------------------------------------------------
#	Class methods to generate Slurm::Sshare object from output of
#	sshare command
#-------------------------------------------------------------------

sub _sshare_fields_in_order($)
#Should return a list ref of field names in order sacctmgr will return them
{	my $class = shift;
	return [ @sshare_fields_in_order];
}

sub new_from_sshare_record($$$)
#Generates a new instance from a list ref as obtained from the
#sshare command output
{	my $class = shift;
	my $record = shift;
	my $cluster = shift;

	my $fields = $class->_sshare_fields_in_order;
	my @record = @$record;

	my @newargs = ();
	@newargs = (cluster => $cluster) if $cluster;

	foreach my $fld (@$fields)
	{	my $val = shift @record;
		$val =~ s/^ *// if defined $val;
		$fld = lc $fld;
		push @newargs, $fld, $val if defined $val && $val ne '';
	}

	my $obj = $class->new(@newargs);
	return $obj;
}

#-------------------------------------------------------------------
#	High level commands
#-------------------------------------------------------------------

sub sshare_list($@)
#Gets sshare output.
#Input parameters are key=>value pairs and can include:
#	accounts: list of accounts to get information for
#	clusters: list of clusters to get information for. 
#	users: list of users to get information for
#
#Values can be either list refs or CSV strings.  For users, if the single
#string 'ALL' is given (not part of a list), then all users will be listed.
#
#On success, returns a list ref of Slurm::Sshare objects corresponding
#to the output of the sshare command.  Possibly an empty list.
#
#On error, either aborts or returns non-ref true value (text describing
#the error)
{	my $class = shift;
	my %where = @_;

	my $me = 'sshare_list';
	my @args = ();
	my $tmp;

	my $accounts = delete $where{accounts};
	my $clusters = delete $where{clusters};
	my $users = delete $where{users};
	croak "$me: Extraneous arguments [" . (join ', ', (keys %where) ) .
		"] given, aborting at " if scalar(%where);

	if ( scalar(keys %where) )
	{	$tmp = join ", ", (keys %where);
		croak "$me: Unrecognized arguments [ $tmp ], aborting.\n";
	}

	if ( $accounts )
	{	unless ( ref($accounts) eq 'ARRAY' )
		{	$accounts = [ split /\s*,\s*/, $accounts ];
		}
	}
	if ( $accounts && scalar(@$accounts) )
	{	$tmp = join ",", @$accounts;
		push @args, "--accounts=$tmp";
	}

	if ( $clusters )
	{	unless ( ref($clusters) eq 'ARRAY' )
		{	$clusters = [ split /\s*,\s*/, $clusters ];
		}
	}
	#unless ( $clusters && scalar(@$clusters) )
	#{	croak "$me: At least one cluster must be specified at ";
	#}

	if ( $users )
	{	unless ( ref($users) eq 'ARRAY' )
		{	$users = [ split /\s*,\s*/, $users ];
		}
	}
	if ( $users && scalar(@$users) )
	{	if ( $users->[0] eq 'ALL' )
		{	push @args, '--all';
		} else
		{	$tmp = join ",", @$users;
			push @args, "--users=$tmp";
		}
	}


	#Save raw output for debugging purposes
	$_sslist_last_raw_output = [];

	my @objects = ();

	if ( $clusters )
	{   # One or more clusters specified; run sshare on each specified cluster
	    # and label results based on the cluster
	    foreach my $cluster (@$clusters)
	    {	my $tmplist = $class->run_generic_sshare_list_command(
			"--clusters=$cluster", @args);
		push @$_sslist_last_raw_output, @{$class->_sshare_last_raw_output};

		unless ( $tmplist && ref($tmplist) )
		{	return "Error in sshare cmd for $cluster: $tmplist";
		}

		foreach my $rec (@$tmplist)
		{	
			my $obj = $class->new_from_sshare_record($rec,$cluster);
			push @objects, $obj;
		}
	    }
	} else
	{	#No clusters specified
	    	my $tmplist = $class->run_generic_sshare_list_command(@args);
		push @$_sslist_last_raw_output, @{$class->_sshare_last_raw_output};

		unless ( $tmplist && ref($tmplist) )
		{	return "Error in sshare cmd: $tmplist";
		}

		foreach my $rec (@$tmplist)
		{	
			my $obj = $class->new_from_sshare_record($rec);
			push @objects, $obj;
		}
	}
		
	return [@objects];
}
	
sub usage_for_account_in_cluster($@)
#Returns usage stats for specified account in specified cluster
#Takes key => value pairs:
#	cluster => name of cluster, REQUIRED
#	account => name of account, REQUIRED
#	users => list ref of users to return data for.  If undef, return
#		data for all users found.  To get no user data, set to []
#	nowarnings => BOOLEAN.  If true, wanrings about not finding account rows
#		are suppressed
#
#On success, returns 
# [ $cpusec_used, $cpumin_limit, $cpumin_avail, $userdata ]
#where $userdata is a hash ref keyed on username with value being
#	the total number of cpusecs used by that user for all associations
#	with that account/cluster.  Might be undef if no userdata req/found
#On error, aborts or returns nonref text string (error message)
{	my $class = shift;
	my %args = @_;
	my $me = 'usage_for_account_in_cluster';

	my $cluster = delete $args{cluster};
	my $account = delete $args{account};
	my $users = delete $args{users};
	my $nowarn = delete $args{nowarnings};
	croak "$me: Missing required parm cluster at " unless $cluster;
	croak "$me: Missing required parm account at " unless $account;
	croak "$me: Invalid value for 'users', expecting list ref at "
		if $users && ref($users) ne 'ARRAY';
	croak "$me: Extraneous arguments [" . (join ', ', (keys %args) ) .
		"] given, aborting at " if scalar(%args);
	

	my $userdata;
	my $tmpusers = $users;
	$tmpusers = 'ALL' unless defined $tmpusers;
	my $list = $class->sshare_list(
		clusters=>[$cluster], accounts=>[$account], users=>$tmpusers,
	);
	unless ( $list )
	{	carp "$me: Unable to find $account in $cluster for sshare at " unless $nowarn;
		return [ 0, 0, 0, $userdata ];
	}
	return  "$me: Error running sshare for cl $cluster, acc $account: $list"
		unless ( ref($list) );

	#We want entries w/out user
	my @list = grep { ! $_->user } @$list;

	unless ( @list )
	{	carp "$me: Unable to find $account in $cluster for sshare (2) at " unless $nowarn;;
		return [ 0, 0, 0, $userdata ];
	}
	if ( scalar(@list) > 1 ) 
	{	#Always warn on this! (???), not a normal situation??
		carp "$me: Multiple lines found for $account in $cluster for sshare, only using first at ";
	}
	my $rec = $list->[0];

	my $cpusec_used = $rec->raw_usage;
	my $cpumin_used = $cpusec_used/60;
	my $cpumin_limit = $rec->grpcpumins;
	$cpumin_limit = undef unless length($cpumin_limit);
	my $cpumin_avail = 0;
	$cpumin_avail = $cpumin_limit - $cpumin_used if $cpumin_limit;

	my @tmp;
	unless ( defined $users )
	{	#Default to all users found
		@tmp = map { $_->user } @$list;
		@tmp = grep { $_ } @tmp;
		my %tmp = map { $_ => $_ } @tmp;
		@tmp = values %tmp;
		$users = [ @tmp ];
	}

	USER: foreach my $user ( @$users )
	{	@tmp = grep { $_->user && $_->user eq $user } @$list;
		next USER unless @tmp;
		my $user_secs = 0;
		USERREC: foreach my $rec (@tmp)
		{	my $tmp = $rec->raw_usage || 0;
			$user_secs += $tmp;
		}
		$userdata = {} unless $userdata && ref($userdata) eq 'HASH';
		$userdata->{$user} = $user_secs;
	}
	
	return [ $cpusec_used, $cpumin_limit, $cpumin_avail, $userdata ];
}

sub usage_for_user_account_in_cluster($@)
#Returns the cpusecs used by user in account in cluster
#Takes key => value pairs:
#	user => username of user, REQUIRED
#	cluster => name of cluster, REQUIRED
#	account => name of account, REQUIRED
#	nowarnings => BOOLEAN.  If true, wanrings about not finding account rows
#		are suppressed
#
#On success, returns $cpusec_used
#On error, aborts or returns nonref text string (error message)
{	my $class = shift;
	my %args = @_;
	my $me = 'usage_for_user_account_in_cluster';

	my $user = delete $args{user};
	my $cluster = delete $args{cluster};
	my $account = delete $args{account};
	my $nowarn = delete $args{nowarnings};
	croak "$me: Missing required parm user at " unless $user;
	croak "$me: Missing required parm cluster at " unless $cluster;
	croak "$me: Missing required parm account at " unless $account;
	croak "$me: Extraneous arguments [" . (join ', ', (keys %args) ) .
		"] given, aborting at " if scalar(%args);

	my $list = $class->sshare_list(
		users=>[$user], clusters=>[$cluster], accounts=>[$account] 
	);
	unless ( $list )
	{	carp "$me: Unable to find $user for $account in $cluster  " .
			"for sshare at " unless $nowarn;
		return 0;
	}
	croak "$me: Error running sshare for (user=$user, cluster=$cluster, " .
		" account $account): $list" unless ( ref($list) );

	my @list = grep { $_->user && $_->user eq $user } @$list;
	#We will have a record for each association, sum over them
	my $cpusecs = 0;
	foreach my $rec (@list)
	{	my $tmp = $rec->raw_usage;
		$cpusecs += $tmp;
	}
	return $cpusecs;
}

#
1;
__END__

=head1 NAME

Slurm::Sshare - wrapper around the Slurm sshare command

=head1 SYNOPSIS

  use Slurm::Sshare;

  my $Sshare = 'Slurm::Sshare';
  $Sshare->verbose(1); #Print sshare commands as running them

  my $shares = $Sshare->sshare_list(clusters=>'dt2');

  foreach $share (@$shares)
  {	print $share->cluster, $share->account, $share->user, 
		$share->raw_usage, $share->grpcpumins, "\n";
  }

  ...


=head1 DESCRIPTION

This is a wrapper around the Slurm B<sshare> command.  Basically, it allows
the Slurm B<sshare> command to called from within Perl, with some processing
of the output into a more Perlish form, thereby enabling Perl scripts to have
access to the share information for Slurm associations.  The B<sshare> command is called using
forks and pipes, so no additional shell is spawned, and shell expansion is not
done, making things a bit more secure.

The interface to this package is object oriented, mainly to reduce namespace pollution.

Since the B<sshare> command is only useful when Slurm is running with the priority/multifactor
plugin, this Perl class has the same restrictions on its usefulness.


=head2 Class Data Members

There are a couple of class data members that control the behavior of this package, that
can be controlled by the following accessor/mutators:

=over 4

=item B<sshare>:

The path to the Slurm B<sshare> command.  Normally, this defaults to just "sshare", i.e. it will
look for B<sshare> in your current path.  Systems staff can set a different default by changing the
value of B<$SSHARE_CMD> at the top of this file.

=item B<verbose>:

If this is set, the module will work in B<verbose> mode, which means that every B<sshare> command will be
printed to STDERR before execution.  Default is false.  If you wish to explicitly set this to false, you
will need to provide a defined but false value (e.g. 0, but not undef) to the mutator.

=back

These methods are both  accessors and mutators, so in order to turn off verbose mode 
you need to supply a defined but false argument (e.g. 0) to the B<verbose> function; 
if the value undef is provided, the call will be treated as a pure accessor 
(rather than mutator) call, and the new value will B<not> be set.

Because the Slurm B<sshare> command only reads the Slurm databases and does not update them, there
is no B<dryrun> mode for this class.

=head2 CONSTRUCTOR and DATA members

Typically, one does not need to explicitly call the constructor for this class; the main methods
for external consumption are class methods, and might return one or more instances of the class.
But we include this section for completeness, and also to discuss the data members of this class.

The constructor B<new> takes key => value pairs to set the initial value of data members.
The instance data members are:

=over 4

=item B<account>:

The account for this association.

=item B<user>:

The user for this association.

=item B<cluster>:

The cluster for this association.

=item B<raw_shares>:

The raw shares assigned to this user/account.

=item B<normalized_shares>:

The shares assigned to this user/account, normalized to the total number of assigned shares.

=item B<raw_usage>:

The number of cpu-seconds of all the jobs that charged this account by this user.  This number will
decay over time when PriorityDecayHalfLife is defined.

=item B<normalized_usage>:

The number of cpu-seconds of all the jobs that charged this account by this user, normalized to the
total number of cpu-seconds of all jobs run on the cluster, subject to PriorityDecayHalfLife when defined.

=item B<effective_usage>:

Like B<normalized_usage>, but augmented to account for usage from sibling accounts.

=item B<fairshare>:

The fair share factorfor this account/user.

=item B<grpcpumins>: 

The CPU minutes limit set on the account.

=item B<cpurunmins>:

The estimated (based an walltime limits) number of CPU minutes needed to complete all currently running jobs
for this user/account.

=back

Note that the B<cluster> data member is B<not> included in the B<sshare> output, but will be filled in if
possible by the B<sshare_list> command if it is called on a specific cluster.  

The instance data members above are associated with read-only accessors with the same name.  Normally they
will be set when instances are created from parsing the output of the B<sshare> command.

=head2 The B<sshare_list> method

This class methods runs the B<sshare> command with the appropriate arguments, and returns an array reference
of instances of this class representing the rows of output returned by the B<sshare> command.  In addition
to the class (an instance can also be used) invocant, it takes a (possibly empty) list of key => value
pairs to provide arguments to the B<sshare> command, with the following keys recognized:

=over 4

=item B<clusters>: 

A list of clusters to issue the B<sshare> command to.  I.e., the B<--clusters> argument to the B<sshare> command.

=item B<accounts>: 

A list of accounts to report on.  The B<--accounts> argument to the B<sshare> command.

=item B<users>: 

A list of users to report on.  The special value B<ALL>, if given as the only user, will result in the B<--all> flag passed to B<sshare>, otherwise this becomes the B<--users> argument to B<sshare>.

=back

For all the lists above, you can give either a list ref or a scalar CSV string.

The B<accounts> argument is converted (if neccessary) to a scalar CSV string and added to the 
argument list of the B<sshare> command with the B<--accounts> flag.  The B<users> argument is handled similarly,
except that if the user list is 'ALL', instead of setting the B<--users> flag the B<-all> flag will be added
to the argument list of the B<sshare> command.

The B<clusters> argument, if given, is handled specially.  For each cluster specified, the B<sshare> command
is invoked with the B<--clusters> argument set to that cluster name (along with any flags from the
B<accounts> and B<users> arguments), and the results for each single cluster
invocation of B<sshare> are parsed separately, and passed the name of the cluster the command was issued to
in order to set the B<cluster> data member.  If the B<clusters> argument is not given, the B<sshare> command
is invoked just once, without any B<--clusters> argument, and the B<cluster> data member will B<NOT> be set
when the output of the B<sshare> command is parsed.

The B<sshare_list> command will raise an exception if called with improper arguments.  On more transient errors
(like the sshare command errored), it will return a non-reference scalar error text.  Otherwise, on success
it returns a (possibly empty) list of B<Slurm::Sshare> instances representing each association the sshare
reported on.  

=head2 The B<usage_for_account_in_cluster> method

This class method runs and parses the appropriate B<sshare> commands to compute the usage for a specific Slurm
allocation account in a specific cluster.   This makes certain assumptions on how the associations, and limits
for the associations, are defined in Slurm.  It is assumed that there in a given cluster, there is a single
association for the allocation account without an user set, and that the usage limit for the allocation account
is set soley in the B<GrpCPUMins> field of this association.  It is believed that this is a fairly common 
arrangement, but it could fail if, for example, there are associations without an user set for this allocation 
account for specific partitions, with or without B<GrpCPUMins> set on the per-partition associations.  This
assumption also fails if the allocation account has limits imposed on it from parent associations.

The method takes its arguments as key => value pairs, recognizing the following keys:

=over 4

=item B<cluster>:

The name of the cluster to get information for.  The underlying B<sshare_list> command will be directed at
that cluster.  This parameter is REQUIRED.

=item B<account>:

The name of the allocation account to get information for.  This parameter is REQUIRED.

=item B<users>:

This is a list ref of users to return data for.  If omitted or undef, data will be returned for all
users found.  To return data for no user, set this to an empty list ref (e.g. []).

=item B<nowarnings>:

This takes a standard Perl boolean value.  If true, warnings to STDERR for the account not being
found in sshare output are suppressed.  Default is false (display warnings).

=back 

The method will raise a fatal exception on certain errors, mainly for errors related to being invoked 
improperly (e.g. required parameters are missing, invalid data type for users, etc).   For less serious
errors (e.g. the underlying sshare did not succeed), an non-reference scalar error text string will
be returned.  On success, an array reference with the following elements will be returned:

=over 4

=item B<cpusec_used> : 

the total number of cpusecs used by all jobs charged to the specified account/cluster.

=item B<cpumin_limit> : 

the GrpCPUMins limit for this account/cluster (see caveats above).

=item B<cpumin_unused>: 

the difference between B<cpumin_limit> and B<cpusec_used> (in CPU-minutes)

=item B<used_by_username>: 

a hash reference, keyed by username, of the total number of CPU seconds used by jobs in all associations with
the specified account/cluster and username.  If the B<users> array reference was given, it will be restricted
to users in the specified list.

=back

In certain suspicious situations (e.g. no associations for the specified account in the specified cluster
could be found, or some of the assumptions discussed above do not appear to be holding), the method will
try to return a sensible answer, but will generate a warning to STDERR.  E.g., if no assocation for the
account is found, a warning will be generated but data (with all values 0) will be returned.


=head2 The B<usage_for_user_account_in_cluster> method

This class method will invoke the Slurm B<sshare> command to get usage information for a specific user
in a specified allocation account and cluster.  The B<sshare> command will normally return a separate
line for each association associated with this user/account/cluster (i.e., a separate line for each
partition the user is associated with).  This method will sum up the CPU seconds used for this user
over all the associations, and return it.

The method takes its arguments as key => value pairs, recognizing the following keys:

=over 4

=item B<user>: the name of the user to get usage for.  REQUIRED.

=item B<account>: the name of the account to get usage for.  REQUIRED.

=item B<cluster>: the name of the cluster to direct B<sshare> commands at.  REQUIRED.

=item B<nowarnings>: if true, warnings to STDERR are suppressed.  Default is false.

=back

All errors in this routine will result in an exception being raised; this includes errors running the
B<sshare> command.  On success, the sum of the CPU seconds consumed for all associations with the
specified user, account, and cluster will be returned.  On some suspicious cases the method will return
the best value it can get but also print a warning to STDERR (unless nowarnings is set to true).  E.g. if no associations
found for the specified user, account, and cluster, the method will print a warning and return 0 CPU-seconds.

=head2 EXPORT

None.  OO interface only.

=head1 SEE ALSO

Slurmdb

Slurm::Sacctmgr

=head1 AUTHOR

Tom Payerle, payerle@umd.edu

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2014-2016 by the University of Maryland

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

