package Chart::Plotly::Trace::Table::Domain;
use Moose;
use MooseX::ExtraArgs;
use Moose::Util::TypeConstraints qw(enum union);
if ( !defined Moose::Util::TypeConstraints::find_type_constraint('PDL') ) {
    Moose::Util::TypeConstraints::type('PDL');
}

our $VERSION = '0.026';    # VERSION

# ABSTRACT: This attribute is one of the possible options for the trace table.

sub TO_JSON {
    my $self       = shift;
    my $extra_args = $self->extra_args // {};
    my $meta       = $self->meta;
    my %hash       = %$self;
    for my $name ( sort keys %hash ) {
        my $attr = $meta->get_attribute($name);
        if ( defined $attr ) {
            my $value = $hash{$name};
            my $type  = $attr->type_constraint;
            if ( $type && $type->equals('Bool') ) {
                $hash{$name} = $value ? \1 : \0;
            }
        }
    }
    %hash = ( %hash, %$extra_args );
    delete $hash{'extra_args'};
    if ( $self->can('type') && ( !defined $hash{'type'} ) ) {
        $hash{type} = $self->type();
    }
    return \%hash;
}

has column => (
        is            => "rw",
        isa           => "Int",
        documentation => "If there is a layout grid, use the domain for this column in the grid for this table trace .",
);

has row => (
           is            => "rw",
           isa           => "Int",
           documentation => "If there is a layout grid, use the domain for this row in the grid for this table trace .",
);

has x => ( is            => "rw",
           isa           => "ArrayRef|PDL",
           documentation => "Sets the horizontal domain of this table trace (in plot fraction).",
);

has y => ( is            => "rw",
           isa           => "ArrayRef|PDL",
           documentation => "Sets the vertical domain of this table trace (in plot fraction).",
);

__PACKAGE__->meta->make_immutable();
1;

__END__

=pod

=encoding utf-8

=head1 NAME

Chart::Plotly::Trace::Table::Domain - This attribute is one of the possible options for the trace table.

=head1 VERSION

version 0.026

=head1 SYNOPSIS

 use Chart::Plotly qw(show_plot);
 use Chart::Plotly::Trace::Table;
 # Example data from: https://plot.ly/javascript/table/#basic-table
 my $table = Chart::Plotly::Trace::Table->new(
 
     header => {
         values => [ [ "EXPENSES" ], [ "Q1" ],
             [ "Q2" ], [ "Q3" ], [ "Q4" ] ],
         align  => "center",
         line   => { width => 1, color => 'black' },
         fill   => { color => "grey" },
         font   => { family => "Arial", size => 12, color => "white" }
     },
     cells  => {
         values => [
             [ 'Salaries', 'Office', 'Merchandise', 'Legal', 'TOTAL' ],
             [ 1200000, 20000, 80000, 2000, 12120000 ],
             [ 1300000, 20000, 70000, 2000, 130902000 ],
             [ 1300000, 20000, 120000, 2000, 131222000 ],
             [ 1400000, 20000, 90000, 2000, 14102000 ] ],
         align  => "center",
         line   => { color => "black", width => 1 },
         font   => { family => "Arial", size => 11, color => [ "black" ] }
     }
 );
 
 show_plot([ $table ]);

=head1 DESCRIPTION

This attribute is part of the possible options for the trace table.

This file has been autogenerated from the official plotly.js source.

If you like Plotly, please support them: L<https://plot.ly/> 
Open source announcement: L<https://plot.ly/javascript/open-source-announcement/>

Full reference: L<https://plot.ly/javascript/reference/#table>

=head1 DISCLAIMER

This is an unofficial Plotly Perl module. Currently I'm not affiliated in any way with Plotly. 
But I think plotly.js is a great library and I want to use it with perl.

=head1 METHODS

=head2 TO_JSON

Serialize the trace to JSON. This method should be called only by L<JSON> serializer.

=head1 ATTRIBUTES

=over

=item * column

If there is a layout grid, use the domain for this column in the grid for this table trace .

=item * row

If there is a layout grid, use the domain for this row in the grid for this table trace .

=item * x

Sets the horizontal domain of this table trace (in plot fraction).

=item * y

Sets the vertical domain of this table trace (in plot fraction).

=back

=head1 AUTHOR

Pablo Rodríguez González <pablo.rodriguez.gonzalez@gmail.com>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2019 by Pablo Rodríguez González.

This is free software, licensed under:

  The MIT (X11) License

=cut
