package Chart::Plotly::Trace::Pie::Domain;
use Moose;
use MooseX::ExtraArgs;
use Moose::Util::TypeConstraints qw(enum union);
if ( !defined Moose::Util::TypeConstraints::find_type_constraint('PDL') ) {
    Moose::Util::TypeConstraints::type('PDL');
}

our $VERSION = '0.024';    # VERSION

# ABSTRACT: This attribute is one of the possible options for the trace pie.

sub TO_JSON {
    my $self       = shift;
    my $extra_args = $self->extra_args // {};
    my $meta       = $self->meta;
    my %hash       = %$self;
    for my $name ( sort keys %hash ) {
        my $attr = $meta->get_attribute($name);
        if ( defined $attr ) {
            my $value = $hash{$name};
            my $type  = $attr->type_constraint;
            if ( $type && $type->equals('Bool') ) {
                $hash{$name} = $value ? \1 : \0;
            }
        }
    }
    %hash = ( %hash, %$extra_args );
    delete $hash{'extra_args'};
    if ( $self->can('type') && ( !defined $hash{'type'} ) ) {
        $hash{type} = $self->type();
    }
    return \%hash;
}

has column => (
          is            => "rw",
          isa           => "Int",
          documentation => "If there is a layout grid, use the domain for this column in the grid for this pie trace .",
);

has row => ( is            => "rw",
             isa           => "Int",
             documentation => "If there is a layout grid, use the domain for this row in the grid for this pie trace .",
);

has x => ( is            => "rw",
           isa           => "ArrayRef|PDL",
           documentation => "Sets the horizontal domain of this pie trace (in plot fraction).",
);

has y => ( is            => "rw",
           isa           => "ArrayRef|PDL",
           documentation => "Sets the vertical domain of this pie trace (in plot fraction).",
);

__PACKAGE__->meta->make_immutable();
1;

__END__

=pod

=encoding utf-8

=head1 NAME

Chart::Plotly::Trace::Pie::Domain - This attribute is one of the possible options for the trace pie.

=head1 VERSION

version 0.024

=head1 SYNOPSIS

 use HTML::Show;
 use Chart::Plotly;
 use Chart::Plotly::Trace::Pie;
 my @labels = ( "ants", "bees", "crickets", "dragonflies", "earwigs" );
 my $pie = Chart::Plotly::Trace::Pie->new( labels => \@labels, values => [ map { int( rand() * 10 ) } @labels ] );
 
 HTML::Show::show( Chart::Plotly::render_full_html( data => [$pie] ) );

=head1 DESCRIPTION

This attribute is part of the possible options for the trace pie.

This file has been autogenerated from the official plotly.js source.

If you like Plotly, please support them: L<https://plot.ly/> 
Open source announcement: L<https://plot.ly/javascript/open-source-announcement/>

Full reference: L<https://plot.ly/javascript/reference/#pie>

=head1 DISCLAIMER

This is an unofficial Plotly Perl module. Currently I'm not affiliated in any way with Plotly. 
But I think plotly.js is a great library and I want to use it with perl.

=head1 METHODS

=head2 TO_JSON

Serialize the trace to JSON. This method should be called only by L<JSON> serializer.

=head1 ATTRIBUTES

=over

=item * column

If there is a layout grid, use the domain for this column in the grid for this pie trace .

=item * row

If there is a layout grid, use the domain for this row in the grid for this pie trace .

=item * x

Sets the horizontal domain of this pie trace (in plot fraction).

=item * y

Sets the vertical domain of this pie trace (in plot fraction).

=back

=head1 AUTHOR

Pablo Rodríguez González <pablo.rodriguez.gonzalez@gmail.com>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2018 by Pablo Rodríguez González.

This is free software, licensed under:

  The MIT (X11) License

=cut
