
package PBS::Warp::Warp1_7 ;
use PBS::Debug ;

use strict ;
use warnings ;

use 5.006 ;
 
require Exporter ;
use AutoLoader qw(AUTOLOAD) ;

our @ISA = qw(Exporter) ;
our %EXPORT_TAGS = ('all' => [ qw() ]) ;
our @EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } ) ;
our @EXPORT = qw() ;
our $VERSION = '0.01' ;

#-------------------------------------------------------------------------------

use PBS::Output ;
use PBS::Log ;
use PBS::Digest ;
use PBS::Constants ;
use PBS::Plugin;
use PBS::Warp;

use Cwd ;
use File::Path;
use Data::Dumper ;
use Data::Compare ;
use Data::TreeDumper ;
use Digest::MD5 qw(md5_hex) ;
use Time::HiRes qw(gettimeofday tv_interval) ;

#-----------------------------------------------------------------------------------------------------------------------

sub WarpPbs
{
my ($targets, $pbs_config, $parent_config) = @_ ;

my $warp_signature = PBS::Warp::GetWarpSignature($targets, $pbs_config) ;
my $warp_path = $pbs_config->{BUILD_DIRECTORY} ;
my $warp_file= "$warp_path/Pbsfile_$warp_signature.warp1_7.blob" ;

$PBS::pbs_run_information->{WARP_1_7}{FILE} = $warp_file ;
PrintInfo "Warp file name: '$warp_file'\n" if defined $pbs_config->{DISPLAY_WARP_FILE_NAME} ;

my ($nodes, $node_names, $global_pbs_config, $insertion_file_names) ;
my ($warp_1_7_version, $number_of_nodes_in_the_dependency_tree, $warp_configuration) ;

my $run_in_warp_mode = 1 ;

my $t0_warp_check ;
my $t0_warp = [gettimeofday];

# Loading of warp file can be eliminated if:
# we add the pbsfiles to the watched files
# we are registred with the watch server (it will have the nodes already)

if(-e $warp_file)
	{
	my $blob = LoadWarpBlob($warp_file) ;
	
	$nodes                = $blob->{'nodes'};
	$node_names           = $blob->{'node_names'};
	$global_pbs_config    = $blob->{'global_pbs_config'};
	$insertion_file_names = $blob->{'insertion_file_names'};
	$warp_1_7_version     = $blob->{'warp_1_7_version'};
	$warp_configuration   = $blob->{'warp_configuration'};
	$number_of_nodes_in_the_dependency_tree = $blob->{'number_of_nodes_in_the_dependency_tree'};
		
	$PBS::pbs_run_information->{WARP_1_7}{SIZE} = -s $warp_file ;
	if($number_of_nodes_in_the_dependency_tree)
		{
		$PBS::pbs_run_information->{WARP_1_7}{SIZE_PER_NODE} = int((-s $warp_file) / $number_of_nodes_in_the_dependency_tree) ;
		}
	else
		{
		$PBS::pbs_run_information->{WARP_1_7}{SIZE_PER_NODE} = 'No node in the warp tree' ;
		$run_in_warp_mode = 0 ;
		}
		
	$PBS::pbs_run_information->{WARP_1_7}{VERSION} = $warp_1_7_version ;
	$PBS::pbs_run_information->{WARP_1_7}{NODES_IN_DEPENDENCY_GRAPH} = $number_of_nodes_in_the_dependency_tree	;
	
	if($pbs_config->{DISPLAY_WARP_TIME})
		{
		my $warp_load_time = tv_interval($t0_warp, [gettimeofday]) ;
		
		PrintInfo(sprintf("Warp load time: %0.2f s.\n", $warp_load_time)) ;
		$PBS::pbs_run_information->{WARP_1_7}{LOAD_TIME} = $warp_load_time ;
		}
		
	$t0_warp_check = [gettimeofday];
	
	PrintInfo "Verifying warp: $number_of_nodes_in_the_dependency_tree nodes ...\n" ;
	
	unless(defined $warp_1_7_version)
		{
		PrintWarning2("Warp: bad version. Warp file needs to be rebuilt.\n") ;
		$run_in_warp_mode = 0 ;
		}
		
	unless($warp_1_7_version == $VERSION)
		{
		PrintWarning2("Warp: bad version. Warp file needs to be rebuilt.\n") ;
		$run_in_warp_mode = 0 ;
		}
		
	# check if all pbs files are still the same
	if(0 == CheckFilesMD5($warp_configuration, 1))
		{
		PrintWarning("Warp: Differences in Pbsfiles. Warp file needs to be rebuilt.\n") ;
		$run_in_warp_mode = 0 ;
		}
	}
else
	{
	PrintWarning("Warp file '$warp_file' doesn't exist.\n") ;
	$run_in_warp_mode = 0 ;
	}

# use filewatching or default MD5 checking
my $IsFileModified = RunUniquePluginSub($pbs_config, 'GetWatchedFilesChecker', $pbs_config, $warp_signature, $nodes) ;

# skip all tests if nothing is modified
if($run_in_warp_mode && defined $IsFileModified  && '' eq ref $IsFileModified  && 0 == $IsFileModified )
	{
	if($pbs_config->{DISPLAY_WARP_TIME})
		{
		my $warp_verification_time = tv_interval($t0_warp_check, [gettimeofday]) ;
		PrintInfo(sprintf("Warp verification time: %0.2f s.\n", $warp_verification_time)) ;
		$PBS::pbs_run_information->{WARP_1_7}{VERIFICATION_TIME} = $warp_verification_time ;
		
		my $warp_total_time = tv_interval($t0_warp, [gettimeofday]) ;
		PrintInfo(sprintf("Warp total time: %0.2f s.\n", $warp_total_time)) ;
		$PBS::pbs_run_information->{WARP_1_7}{TOTAL_TIME} = $warp_total_time ;
		}
		
	PrintInfo("Warp: Up to date.\n") ;
	return (BUILD_SUCCESS, "Warp: Up to date", {READ_ME => "Up to date warp doesn't have any tree"}, $nodes) ;
	}

$IsFileModified ||= \&PBS::Digest::IsFileModified ;
my @build_result ;
if($run_in_warp_mode)
	{
	my ($number_of_removed_nodes)
		= Dewarpify($pbs_config, $nodes, $node_names, $insertion_file_names, $global_pbs_config, $IsFileModified) ;
	
	if($pbs_config->{DISPLAY_WARP_TIME})
		{
		my $warp_verification_time = tv_interval($t0_warp_check, [gettimeofday]) ;
		PrintInfo(sprintf("Warp verification time: %0.2f s.\n", $warp_verification_time)) ;
		$PBS::pbs_run_information->{WARP_1_7}{VERIFICATION_TIME} = $warp_verification_time ;
		
		my $warp_total_time = tv_interval($t0_warp, [gettimeofday]) ;
		PrintInfo(sprintf("Warp total time: %0.2f s.\n", $warp_total_time)) ;
		$PBS::pbs_run_information->{WARP_1_7}{TOTAL_TIME} = $warp_total_time ;
		}
		
	if($number_of_removed_nodes)
		{
		if(defined $pbs_config->{DISPLAY_WARP_BUILD_SEQUENCE})
			{
			}
			
		eval "use PBS::PBS" ;
		die $@ if $@ ;
		
		unless($pbs_config->{DISPLAY_WARP_GENERATED_WARNINGS})
			{
			$pbs_config->{NO_LINK_INFO} = 1 ;
			$pbs_config->{NO_LOCAL_MATCHING_RULES_INFO} = 1 ;
			}
			
		# we can't  generate a warp file while warping.
		# The warp configuration (pbsfiles md5) would be truncated
		# to the files used during the warp
		delete $pbs_config->{GENERATE_WARP1_5_FILE} ;
		
		# much of the "normal" node attributes are stripped in warp nodes
		# let the rest of the system know about this (ex graph generator)
		$pbs_config->{IN_WARP} = 1 ;
		my ($build_result, $build_message) ;
		my $new_dependency_tree ;
		
		eval
			{
			# PBS will link to the  warp nodes instead for regenerating them
			my $node_plural = '' ; $node_plural = 's' if $number_of_removed_nodes > 1 ;
			
			PrintInfo "Running PBS in warp mode. $number_of_removed_nodes node$node_plural to rebuild.\n" ;
			($build_result, $build_message, $new_dependency_tree)
				= PBS::PBS::Pbs
					(
					  $pbs_config->{PBSFILE}
					, ''    # parent package
					, $pbs_config
					, $parent_config
					, $targets
					, $nodes
					, "warp_tree"
					, DEPEND_CHECK_AND_BUILD
					) ;
			} ;
			
		if($@)
			{
			if($@ =~ /^BUILD_FAILED/)
				{
				# this exception occures only when a Builder fails so we can generate a warp file
				GenerateWarpFile
					(
					  $targets, $new_dependency_tree, $nodes
					, $pbs_config, $warp_configuration
					) ;
				}
				
			# died during depend or check
			die $@ ;
			}
		else
			{
			GenerateWarpFile
				(
				  $targets, $new_dependency_tree, $nodes
				, $pbs_config, $warp_configuration
				) ;
			}
			
		@build_result = ($build_result, $build_message, $new_dependency_tree, $nodes) ;
		}
	else
		{
		PrintInfo("Warp: Up to date.\n") ;
		@build_result = (BUILD_SUCCESS, "Warp: Up to date", {READ_ME => "Up to date warp doesn't have any tree"}, $nodes) ;
		}
	}
else
	{
	#eurk hack we could dispense with!
	# this is not needed but the subpses are travesed an extra time
	
	my ($dependency_tree_snapshot, $inserted_nodes_snapshot) ;
	
	$pbs_config->{INTERMEDIATE_WARP_WRITE} = 
		sub
		{
		my $dependency_tree = shift ;
		my $inserted_nodes = shift ;
		
		($dependency_tree_snapshot, $inserted_nodes_snapshot) = ($dependency_tree, $inserted_nodes) ;
		
		GenerateWarpFile
			(
			  $targets
			, $dependency_tree
			, $inserted_nodes
			, $pbs_config
			) ;
		} ;
		
	my ($build_result, $build_message, $dependency_tree, $inserted_nodes) ;
	eval
		{
		($build_result, $build_message, $dependency_tree, $inserted_nodes)
			= PBS::PBS::Pbs
				(
				$pbs_config->{PBSFILE}
				, ''    # parent package
				, $pbs_config
				, $parent_config
				, $targets
				, undef # inserted files
				, "root_WARP_1_7_NEEDS_REBUILD_pbs_$pbs_config->{PBSFILE}" # tree name
				, DEPEND_CHECK_AND_BUILD
				) ;
		} ;
		
		if($@)
			{
			if($@ =~ /^BUILD_FAILED/)
				{
				# this exception occures only when a Builder fails so we can generate a warp file
				GenerateWarpFile
					(
					  $targets
					, $dependency_tree_snapshot
					, $inserted_nodes_snapshot
					, $pbs_config
					) ;
				}
				
			die $@ ;
			}
		else
			{
			GenerateWarpFile
				(
				  $targets
				, $dependency_tree
				, $inserted_nodes
				, $pbs_config
				) ;
			}
			
	@build_result = ($build_result, $build_message, $dependency_tree, $inserted_nodes) ;
	}

if($build_result[0])
	{
	# force a refresh after we build files and generated events
	# note that the synch should be by file not global
	RunUniquePluginSub($pbs_config, 'ClearWatchedFilesList', $pbs_config, $warp_signature) ;
	}
	
return(@build_result) ;
}

#---------------------------------------------------------------------------------------------------------------------------

sub LoadWarpBlob
{
my ($warp_file) = @_ ;

use GDBM_File ;

my $t0_warp_generate =  [gettimeofday] ;

tie my %db, 'GDBM_File', $warp_file, &GDBM_WRCREAT, 0640;

my $blob = thaw($db{__BLOB}) ;

PrintInfo(sprintf("Loaded warp 1.7 Blob in: %0.2f s.\n", tv_interval($t0_warp_generate, [gettimeofday]))) ;

#~ print DumpTree $value, 'all files md5:' ;

#~ use Data::Compare ;

#~ if(Compare($value, $nodes))
	#~ {
	#~ print "same data\n" ;
	#~ }
#~ else
	#~ {
	#~ print "different data\n" ;
	#~ }

return($blob) ;
}

#-------------------------------------------------------------------------------

sub Dewarpify
{
# TODO explain "dry tree", node removal, ...

my $t0_warpify = [gettimeofday] ;

my ($pbs_config, $nodes, $node_names, $insertion_file_names, $global_pbs_config, $IsFileModified) = @_ ;

my $number_of_removed_nodes = 0 ;

# check md5 and remove all nodes that would trigger
my $node_verified = 0 ;
my $node_existed = 0 ;
for my $node (keys %$nodes)
	{
	if($pbs_config->{DISPLAY_WARP_CHECKED_NODES})	
		{
		PrintDebug "Warp checking: '$node'.\n" ;
		}
	else
		{
		#~ PrintInfo "\r$node_verified" ;
		}
		
	$node_verified++ ;
	
	next unless exists $nodes->{$node} ; # can have been removed by one of its dependencies
	
	$node_existed++ ;
	
	my $remove_this_node = 0 ;
	
	if('VIRTUAL' eq $nodes->{$node}{__MD5})
		{
		# virtual nodes don't have MD5
		}
	else
		{
		# rebuild the build name
		if(exists $nodes->{$node}{__LOCATION})
			{
			$nodes->{$node}{__BUILD_NAME} = $nodes->{$node}{__LOCATION} . substr($node, 1) ;
			}
		else
			{
			$nodes->{$node}{__BUILD_NAME} = $node ;
			}
			
		$remove_this_node += $IsFileModified->($pbs_config, $nodes->{$node}{__BUILD_NAME}, $nodes->{$node}{__MD5}) ;
		}
		
	$remove_this_node++ if(exists $nodes->{$node}{__FORCED}) ;
	
	if($remove_this_node) #and its dependents and its triggerer if any
		{
		my @nodes_to_remove = ($node) ;
		
		while(@nodes_to_remove)
			{
			my @dependent_nodes ;
			
			for my $node_to_remove (grep{ exists $nodes->{$_} } @nodes_to_remove)
				{
				if($pbs_config->{DISPLAY_WARP_TRIGGERED_NODES})	
					{
					PrintDebug "Warp: Removing node '$node_to_remove'\n" ;
					}
				
				push @dependent_nodes, grep{ exists $nodes->{$_} } map {$node_names->[$_]} @{$nodes->{$node_to_remove}{__DEPENDENT}} ;
				
				# remove triggering node and its dependents
				if(exists $nodes->{$node_to_remove}{__TRIGGER_INSERTED})
					{
					my $trigerring_node = $nodes->{$node_to_remove}{__TRIGGER_INSERTED} ;
					push @dependent_nodes, grep{ exists $nodes->{$_} } map {$node_names->[$_]} @{$nodes->{$trigerring_node}{__DEPENDENT}} ;
					delete $nodes->{$trigerring_node} ;
					}
					
				delete $nodes->{$node_to_remove} ;
				
				$number_of_removed_nodes++ ;
				}
				
			if($pbs_config->{DISPLAY_WARP_TRIGGERED_NODES})	
				{
				PrintDebug '-' x 30 . "\n" ;
				}
				
			@nodes_to_remove = @dependent_nodes ;
			}
		}
	else
		{
		# rebuild the data PBS needs from the warp file
		$nodes->{$node}{__NAME} = $node ;
		$nodes->{$node}{__BUILD_DONE} = "Field set in warp 1.7" ;
		$nodes->{$node}{__DEPENDED}++ ;
		$nodes->{$node}{__CHECKED}++ ; # pbs will not check any node (and its subtree) which is marked as checked
		
		$nodes->{$node}{__PBS_CONFIG} = $global_pbs_config unless exists $nodes->{$node}{__PBS_CONFIG} ;
		
		$nodes->{$node}{__INSERTED_AT}{INSERTION_FILE} = $insertion_file_names->[$nodes->{$node}{__INSERTED_AT}{INSERTION_FILE}] ;
		
		unless(exists $nodes->{$node}{__DEPENDED_AT})
			{
			$nodes->{$node}{__DEPENDED_AT} = $nodes->{$node}{__INSERTED_AT}{INSERTION_FILE} ;
			}
			
		#let our dependent nodes know about their dependencies
		#this needed when regenerating the warp file from partial warp data
		for my $dependent (map {$node_names->[$_]} @{$nodes->{$node}{__DEPENDENT}})
			{
			if(exists $nodes->{$dependent})
				{
				$nodes->{$dependent}{$node}++ ;
				}
			}
		}
	}

if($pbs_config->{DISPLAY_WARP_TRIGGERED_NODES})	
	{
	PrintInfo "\rNodes: $node_verified verified: $node_existed\n" ;
	}
else
	{
	PrintInfo "\r" ;
	}

$PBS::pbs_run_information->{WARP_1_7}{DEWARPIFY} = tv_interval($t0_warpify, [gettimeofday]) ;

if(scalar(keys %$nodes))
	{
	$PBS::pbs_run_information->{WARP_1_7}{DEWARPIFY_PER_NODE} = $PBS::pbs_run_information->{WARP_1_7}{DEWARPIFY} / scalar(keys %$nodes) ;
	}
else
	{
	$PBS::pbs_run_information->{WARP_1_7}{DEWARPIFY_PER_NODE} = "N/A, no nodes!" ;
	}

return($number_of_removed_nodes) ;
}

#-----------------------------------------------------------------------------------------------------------------------


sub GenerateWarpBlob
{
# indexing the node name  saves another 10% in size
# indexing the location name saves another 10% in size

my $t0_warp_generate =  [gettimeofday] ;

my ($header, $blob, $warp_file) = @_ ;

use GDBM_File ;
tie my %db, 'GDBM_File', $warp_file, &GDBM_WRCREAT, 0640;

use Storable  qw(freeze thaw);

my %nodes_md5 ;

$db{__BLOB} = freeze $blob ;
$db{__BLOB_HEADER} = freeze \$header ;

my $blob_generation_time = tv_interval($t0_warp_generate, [gettimeofday]) ;
PrintInfo(sprintf("Warp 1.7 blob generated in: %0.2f s.\n", $blob_generation_time)) ;
$PBS::pbs_run_information->{WARP_1_7}{BLOB_GENERATION_TIME} = $blob_generation_time ;
}

#-----------------------------------------------------------------------------------------------------------------------

sub GenerateWarpFile
{
# indexing the node name  saves another 10% in size
# indexing the location name saves another 10% in size

my ($targets, $dependency_tree, $inserted_nodes, $pbs_config, $warp_configuration) = @_ ;

$warp_configuration = PBS::Warp::GetWarpConfiguration($pbs_config, $warp_configuration) ; #$warp_configuration can be undef or from a warp file

PrintInfo("Generating warp file               \n") ;
my $t0_warp_generate =  [gettimeofday] ;

my $warp_signature = PBS::Warp::GetWarpSignature($targets, $pbs_config) ;
my $warp_path = $pbs_config->{BUILD_DIRECTORY} ;
mkpath($warp_path) unless(-e $warp_path) ;

my $warp_file= "$warp_path/Pbsfile_$warp_signature.warp1_7.blob" ;

my $header = PBS::Log::GetHeader('Warp', $pbs_config) ;

my $number_of_nodes_in_the_dependency_tree = keys %$inserted_nodes ;

my $global_pbs_config = # cache to reduce warp file size
	{
	  BUILD_DIRECTORY    => $pbs_config->{BUILD_DIRECTORY}
	, SOURCE_DIRECTORIES => $pbs_config->{SOURCE_DIRECTORIES}
	} ;
	
my ($nodes, $node_names, $insertion_file_names) = WarpifyTree($inserted_nodes, $global_pbs_config) ;

my $blob =
	{
	'global_pbs_config'    => $global_pbs_config,
	'nodes'                => $nodes,
	'node_names'           => $node_names,
	'insertion_file_names' => $insertion_file_names,
	'warp_configuration'   => $warp_configuration,
	'warp_1_7_version'     => $VERSION,
	'number_of_nodes_in_the_dependency_tree' => $number_of_nodes_in_the_dependency_tree,
	} ;


GenerateWarpBlob($header, $blob, $warp_file) ;

if($pbs_config->{DISPLAY_WARP_TIME})
	{
	my $warp_generation_time = tv_interval($t0_warp_generate, [gettimeofday]) ;
	PrintInfo(sprintf("Warp total time: %0.2f s.\n", $warp_generation_time)) ;
	$PBS::pbs_run_information->{WARP_1_7}{GENERATION_TIME} = $warp_generation_time ;
	}
}

#-----------------------------------------------------------------------------------------------------------------------

sub WarpifyTree
{
my $inserted_nodes = shift ;
my $global_pbs_config = shift ;

my ($package, $file_name, $line) = caller() ;

my (%nodes, @node_names, %nodes_index) ;
my (@insertion_file_names, %insertion_file_index) ;

my $t0_warpify=  [gettimeofday] ;

for my $node (keys %$inserted_nodes)
	{
	# this doesn't work with LOCAL_NODES
	
	if(exists $inserted_nodes->{$node}{__VIRTUAL})
		{
		$nodes{$node}{__VIRTUAL} = 1 ;
		}
	else
		{
		# here some attempt to start handling AddDependency and micro warps
		#$nodes{$node}{__DIGEST} = GetDigest($inserted_nodes->{$node}) ;
		}
		
	if(exists $inserted_nodes->{$node}{__FORCED})
		{
		$nodes{$node}{__FORCED} = 1 ;
		}

	if(!exists $inserted_nodes->{$node}{__VIRTUAL} && $node =~ /^\.(.*)/)
		{
		($nodes{$node}{__LOCATION}) = ($inserted_nodes->{$node}{__BUILD_NAME} =~ /^(.*)$1$/) ;
		}
		
	#this can also be reduced for a +/- 10% reduction
	if(exists $inserted_nodes->{$node}{__INSERTED_AT}{ORIGINAL_INSERTION_DATA}{INSERTING_NODE})
		{
		$nodes{$node}{__INSERTED_AT}{INSERTING_NODE} = $inserted_nodes->{$node}{__INSERTED_AT}{ORIGINAL_INSERTION_DATA}{INSERTING_NODE}
		}
	else
		{
		$nodes{$node}{__INSERTED_AT}{INSERTING_NODE} = $inserted_nodes->{$node}{__INSERTED_AT}{INSERTING_NODE} ;
		}
	
	$nodes{$node}{__INSERTED_AT}{INSERTION_RULE} = $inserted_nodes->{$node}{__INSERTED_AT}{INSERTION_RULE} ;
	
	if(exists $inserted_nodes->{$node}{__DEPENDED_AT})
		{
		if($inserted_nodes->{$node}{__INSERTED_AT}{INSERTION_FILE} ne $inserted_nodes->{$node}{__DEPENDED_AT})
			{
			$nodes{$node}{__DEPENDED_AT} = $inserted_nodes->{$node}{__DEPENDED_AT} ;
			}
		}
		
	#reduce amount of data by indexing Insertion files (Pbsfile)
	my $insertion_file = $inserted_nodes->{$node}{__INSERTED_AT}{INSERTION_FILE} ;
	
	unless (exists $insertion_file_index{$insertion_file})
		{
		push @insertion_file_names, $insertion_file ;
		$insertion_file_index{$insertion_file} = $#insertion_file_names ;
		}
		
	$nodes{$node}{__INSERTED_AT}{INSERTION_FILE} = $insertion_file_index{$insertion_file} ;
	
	if
		(
		   $inserted_nodes->{$node}{__PBS_CONFIG}{BUILD_DIRECTORY}  ne $global_pbs_config->{BUILD_DIRECTORY}
		|| !Compare($inserted_nodes->{$node}{__PBS_CONFIG}{SOURCE_DIRECTORIES}, $global_pbs_config->{SOURCE_DIRECTORIES})
		)
		{
		$nodes{$node}{__PBS_CONFIG}{BUILD_DIRECTORY} = $inserted_nodes->{$node}{__PBS_CONFIG}{BUILD_DIRECTORY} ;
		$nodes{$node}{__PBS_CONFIG}{SOURCE_DIRECTORIES} = [@{$inserted_nodes->{$node}{__PBS_CONFIG}{SOURCE_DIRECTORIES}}] ; 
		}
		
	if(exists $inserted_nodes->{$node}{__BUILD_DONE})
		{
		if(exists $inserted_nodes->{$node}{__VIRTUAL})
			{
			$nodes{$node}{__MD5} = 'VIRTUAL' ;
			}
		else
			{
			if(exists $inserted_nodes->{$node}{__INSERTED_AT}{INSERTION_TIME})
				{
				# this is a new node
				if(defined $inserted_nodes->{$node}{__MD5} && $inserted_nodes->{$node}{__MD5} ne 'not built yet')
					{
					$nodes{$node}{__MD5} = $inserted_nodes->{$node}{__MD5} ;
					}
				else
					{
					if(defined (my $current_md5 = GetFileMD5($inserted_nodes->{$node}{__BUILD_NAME})))
						{
						$nodes{$node}{__MD5} = $inserted_nodes->{$node}{__MD5} = $current_md5 ;
						}
					else
						{
						die ERROR("Can't open '$node' to compute MD5 digest: $!") ;
						}
					}
				}
			else
				{
				# use the old md5
				$nodes{$node}{__MD5} = $inserted_nodes->{$node}{__MD5} ;
				}
			}
		}
	else
		{
		$nodes{$node}{__MD5} = 'not built yet' ; 
		}
		
	unless (exists $nodes_index{$node})
		{
		push @node_names, $node ;
		$nodes_index{$node} = $#node_names;
		}
		
	for my $dependency (keys %{$inserted_nodes->{$node}})
		{
		next if $dependency =~ /^__/ ;
		
		push @{$nodes{$dependency}{__DEPENDENT}}, $nodes_index{$node} ;
		}
		
	if (exists $inserted_nodes->{$node}{__TRIGGER_INSERTED})
		{
		$nodes{$node}{__TRIGGER_INSERTED} = $inserted_nodes->{$node}{__TRIGGER_INSERTED} ;
		}
	}

$PBS::pbs_run_information->{WARP_1_7}{WARPIFY} = tv_interval($t0_warpify, [gettimeofday]) ;
$PBS::pbs_run_information->{WARP_1_7}{WARPIFY_PER_NODE} = $PBS::pbs_run_information->{WARP_1_7}{WARPIFY} / scalar(keys %$inserted_nodes) ;

return(\%nodes, \@node_names, \@insertion_file_names) ;
}

#--------------------------------------------------------------------------------------------------

1 ;

__END__
=head1 NAME

PBS::Warp::Warp1_7  -

=head1 DESCRIPTION

=head2 EXPORT

None.

=head1 AUTHOR

Khemir Nadim ibn Hamouda. nadim@khemir.net

=head1 SEE ALSO

B<PBS::Information>.

=cut
