use strict;
use warnings;

use FindBin;
use lib $FindBin::Bin;
use silktest;

use Test::More tests => 6;

use Net::Silk qw( :basic );

BEGIN { use_ok( SILK_IPWILDCARD_CLASS ) }

sub make_wc { new_ok(SILK_IPWILDCARD_CLASS, \@_ ) }

sub new_ip { SILK_IPADDR_CLASS->new(shift) }

###

sub test_construction {

  plan tests => 61;

  make_wc("0.0.0.0");
  make_wc("255.255.255.255");
  make_wc("     255.255.255.255");
  make_wc("255.255.255.255     ");
  make_wc("   255.255.255.255  ");
  make_wc("0.0.0.0/31");
  make_wc("255.255.255.254-255");
  make_wc("3,2,1.4.5.6");
  make_wc("0.0.0.1,31,51,71,91,101,121,141,161,181,211,231,251");
  make_wc("0,255.0,255.0,255.0,255");
  make_wc("1.1.128.0/22");
  make_wc("128.x.0.0");
  make_wc("128.0-255.0.0");
  make_wc("128.0,128-255,1-127.0.0");
  make_wc("128.0,128,129-253,255-255,254,1-127.0.0");
  make_wc("128.0,128-255,1-127.0.0  ");
  make_wc("  128.0,128-255,1-127.0.0  ");
  make_wc("  128.0,128-255,,1-127.0.0  ");
  make_wc("0.0.0.0");

  SKIP: {
    skip("ipv6 not enabled", 42) unless SILK_IPV6_ENABLED;
    make_wc("0:0:0:0:0:0:0:0");
    make_wc("::");
    make_wc("::0.0.0.0");
    make_wc("1:2:3:4:5:6:7:8");
    make_wc("1:203:405:607:809:a0b:c0d:e0f");
    make_wc("1:203:405:607:809:a0b:12.13.14.15");
    make_wc("::FFFF");
    make_wc("::FFFF:FFFF");
    make_wc("::0.0.255.255");
    make_wc("::255.255.255.255");
    make_wc("FFFF::");
    make_wc("0,FFFF::0,FFFF");
    make_wc("::FFFF:0,10.0.0.0,10");
    make_wc("::FFFF:0.0,160.0,160.0");
    make_wc("0:0:0:0:0:0:0:0/127");
    make_wc("::/127");
    make_wc("0:0:0:0:0:0:0:0/110");
    make_wc("0:0:0:0:0:0:0:0/95");
    make_wc("0:ffff::0/127");
    make_wc("0:ffff::0.0.0.0,1");
    make_wc("0:ffff::0.0.0.0-10");
    make_wc("0:ffff::0.0.0.x");
    make_wc("::ffff:0:0:0:0:0:0/110");
    make_wc("0:ffff::/112");
    make_wc("0:ffff:0:0:0:0:0:x");
    make_wc("0:ffff:0:0:0:0:0:x");
    make_wc("0:ffff:0:0:0:0:0:0-ffff");
    make_wc("0:ffff:0:0:0:0:0.0.x.x");
    make_wc("0:ffff:0:0:0:0:0.0.0-255.128-254,0-126,255,127");
    make_wc("0:ffff:0:0:0:0:0.0.128-254,0-126,255,127.x");
    make_wc("0:ffff:0:0:0:0:0.0.0.0/112");
    make_wc("0:ffff:0:0:0:0:0.0,1.x.x");
    make_wc("0:ffff:0:0:0:0:0:0-10,10-20,24,23,22,21,25-ffff");
    make_wc("0:ffff::x");
    make_wc("0:ffff:0:0:0:0:0:aaab-ffff,aaaa-aaaa,0-aaa9");
    make_wc("0:ffff:0:0:0:0:0:ff00/120");
    make_wc("0:ffff:0:0:0:0:0:ffff/120");
    make_wc("::ff00:0/104");
    make_wc("::x");
    make_wc("x::");
    make_wc("x::10.10.10.10");
    make_wc("::");
  }

}

###

sub fail_bad_wc_str {
  eval { SILK_IPWILDCARD_CLASS->new(@_) };
  like($@, qr/error.*parsing.*wildcard/, "reject bad str");
}

sub test_bad_strings {

  plan tests => 43;

  fail_bad_wc_str('0.0.0.0/33');
  fail_bad_wc_str('0.0.0.2-0');
  fail_bad_wc_str('0.0.0.256');
  fail_bad_wc_str('0.0.256.0');
  fail_bad_wc_str('0.0.0256.0');
  fail_bad_wc_str('0.256.0.0');
  fail_bad_wc_str('0.0.0.0.0');
  fail_bad_wc_str('0.0.x.0/31');
  fail_bad_wc_str('0.0.x.0:0');
  fail_bad_wc_str('0.0.0,1.0/31');
  fail_bad_wc_str('0.0.0-1.0/31');
  fail_bad_wc_str('0.0.0-1-.0');
  fail_bad_wc_str('0.0.0--1.0');
  fail_bad_wc_str('0.0.0.0 junk');
  fail_bad_wc_str('0.0.-0-1.0');
  fail_bad_wc_str('0.0.-1.0');
  fail_bad_wc_str('0.0.0..0');
  fail_bad_wc_str('.0.0.0.0');
  fail_bad_wc_str('0.0.0.0.');
  fail_bad_wc_str('1-FF::/16');
  fail_bad_wc_str('1,2::/16');
  fail_bad_wc_str('1::2::3');
  fail_bad_wc_str(':1::');
  fail_bad_wc_str(':1:2:3:4:5:6:7:8');
  fail_bad_wc_str('1:2:3:4:5:6:7:8:');
  fail_bad_wc_str('1:2:3:4:5:6:7.8.9:10');
  fail_bad_wc_str('1:2:3:4:5:6:7:8.9.10.11');
  fail_bad_wc_str(':');
  fail_bad_wc_str('1:2:3:4:5:6:7');
  fail_bad_wc_str('1:2:3:4:5:6:7/16');
  fail_bad_wc_str('FFFFF::');
  fail_bad_wc_str('::FFFFF');
  fail_bad_wc_str('1:FFFFF::7:8');
  fail_bad_wc_str('1:AAAA-FFFF0::');
  fail_bad_wc_str('FFFFF-AAAA::');
  fail_bad_wc_str('FFFF-AAAA::');
  fail_bad_wc_str('2-1::');
  fail_bad_wc_str('1:FFFF-0::');
  fail_bad_wc_str('1::FFFF-AAAA');
  fail_bad_wc_str(':::');
  fail_bad_wc_str('1:2:3:$::');
  fail_bad_wc_str('1.2.3.4:ffff::');
  fail_bad_wc_str('x');

}

###

sub t_contains {
  my($wc, $spec) = @_;
  ok($wc->contains($spec), "wc contains $spec");
  my $ip = SILK_IPADDR_CLASS->new($spec);
  ok($wc->contains($ip), "wc contains ip($ip)");
}

sub t_no_contains {
  my($wc, $spec) = @_;
  ok(! $wc->contains($spec), "wc contains $spec");
  my $ip = SILK_IPADDR_CLASS->new($spec);
  ok(! $wc->contains($ip), "wc contains ip($ip)");
}

sub test_containment {

  plan tests => 142;

  my $wild;

  $wild = SILK_IPWILDCARD_CLASS->new("0.0.0.0");
  t_contains   ($wild => "0.0.0.0");
  t_no_contains($wild => "0.0.0.1");

  $wild = SILK_IPWILDCARD_CLASS->new("0.0.0.0/31");
  t_contains   ($wild => "0.0.0.0");
  t_contains   ($wild => "0.0.0.1");
  t_no_contains($wild => "0.0.0.2");

  $wild = SILK_IPWILDCARD_CLASS->new("255.255.255.254-255");
  t_contains   ($wild => "255.255.255.254");
  t_contains   ($wild => "255.255.255.255");
  t_no_contains($wild => "255.255.255.253");

  $wild = SILK_IPWILDCARD_CLASS->new("3,2,1.4.5.6");
  t_contains   ($wild => "1.4.5.6");
  t_contains   ($wild => "2.4.5.6");
  t_contains   ($wild => "3.4.5.6");
  t_no_contains($wild => "4.4.5.6");

  $wild = SILK_IPWILDCARD_CLASS->new("0,255.0,255.0,255.0,255");
  t_contains   ($wild => "0.0.0.0");
  t_contains   ($wild => "0.0.0.255");
  t_contains   ($wild => "0.0.255.0");
  t_contains   ($wild => "0.255.0.0");
  t_contains   ($wild => "255.0.0.0");
  t_contains   ($wild => "255.255.0.0");
  t_contains   ($wild => "255.0.255.0");
  t_contains   ($wild => "255.0.0.255");
  t_contains   ($wild => "0.255.0.255");
  t_contains   ($wild => "0.255.255.0");
  t_contains   ($wild => "0.0.255.255");
  t_contains   ($wild => "0.255.255.255");
  t_contains   ($wild => "255.0.255.255");
  t_contains   ($wild => "255.255.0.255");
  t_contains   ($wild => "255.255.255.0");
  t_contains   ($wild => "255.255.255.255");

  t_no_contains($wild => "255.255.255.254");
  t_no_contains($wild => "255.255.254.255");
  t_no_contains($wild => "255.254.255.255");
  t_no_contains($wild => "254.255.255.255");

  t_contains   ($wild => "0.0.0.0");
  t_contains   ($wild => "0.0.0.255");
  t_contains   ($wild => "0.0.255.0");
  t_contains   ($wild => "0.255.0.0");
  t_contains   ($wild => "255.0.0.0");
  t_contains   ($wild => "255.255.0.0");
  t_contains   ($wild => "255.0.255.0");
  t_contains   ($wild => "255.0.0.255");
  t_contains   ($wild => "0.255.0.255");
  t_contains   ($wild => "0.255.255.0");
  t_contains   ($wild => "0.0.255.255");
  t_contains   ($wild => "0.255.255.255");
  t_contains   ($wild => "255.0.255.255");
  t_contains   ($wild => "255.255.0.255");
  t_contains   ($wild => "255.255.255.0");
  t_contains   ($wild => "255.255.255.255");

  t_no_contains($wild => "255.255.255.254");
  t_no_contains($wild => "255.255.254.255");
  t_no_contains($wild => "255.254.255.255");
  t_no_contains($wild => "254.255.255.255");

  SKIP: {
    skip("ipv6 not enabled", 38) unless SILK_IPV6_ENABLED;

    $wild = SILK_IPWILDCARD_CLASS->new("::");
    t_contains   ($wild => "::");
    t_no_contains($wild => "::1");

    $wild = SILK_IPWILDCARD_CLASS->new("::/127");
    t_contains   ($wild => "::");
    t_contains   ($wild => "::1");
    t_no_contains($wild => "::2");

    $wild = SILK_IPWILDCARD_CLASS->new("0:ffff::0.0.0.0,1");
    t_contains   ($wild => "0:ffff::0.0.0.0");
    t_contains   ($wild => "0:ffff::0.0.0.1");
    t_no_contains($wild => "0:ffff::0.0.0.2");

    $wild = SILK_IPWILDCARD_CLASS->new(
      "0:ffff:0:0:0:0:0.253-254.125-126,255.x");
    t_contains   ($wild => "0:ffff::0.253.125.1");
    t_contains   ($wild => "0:ffff::0.254.125.2");
    t_contains   ($wild => "0:ffff::0.253.126.3");
    t_contains   ($wild => "0:ffff::0.254.126.4");
    t_contains   ($wild => "0:ffff::0.253.255.5");
    t_contains   ($wild => "0:ffff::0.254.255.6");
    t_no_contains($wild => "0:ffff::0.255.255.7");

    $wild = SILK_IPWILDCARD_CLASS->new("0.0.0.0");
    t_contains   ($wild => "::ffff:0:0");
    t_no_contains($wild => "::");

    $wild = SILK_IPWILDCARD_CLASS->new("::ffff:0:0");
    t_contains   ($wild => "0.0.0.0");

    $wild = SILK_IPWILDCARD_CLASS->new("::");
    t_no_contains($wild => "0.0.0.0");
  }
}

###

sub t_ipv4_iter {
  my $wc = SILK_IPWILDCARD_CLASS->new(shift);
  ok(!$wc->is_ipv6, "is ipv4 wildcard");
  my $iter = $wc->iter;
  isa_ok($iter, 'CODE');
  my $i = 0;
  for (my $item = $iter->()) {
    is($item, new_ip($_[$i]), "item match $item");
    ++$i;
  }
  my @wc = $iter->();
}

sub t_ipv6_iter {
  my $wc = SILK_IPWILDCARD_CLASS->new(shift);
  ok($wc->is_ipv6, "is ipv6 wildcard");
  my $iter = $wc->iter;
  isa_ok($iter, 'CODE');
  my $i = 0;
  for (my $item = $iter->()) {
    is($item, new_ip($_[$i]), "item match $item");
    ++$i;
  }
  my @wc = $iter->();
}

sub test_iteration {

  plan tests => 27;

  t_ipv4_iter("0.0.0.0" => "0.0.0.0");
  t_ipv4_iter("0.0.0.0/31" => "0.0.0.0", "0.0.0.1");
  t_ipv4_iter("255.255.255.254-255" =>
              "255.255.255.254", "255.255.255.255");
  t_ipv4_iter("3,2,1.4.5.6" => 
              "1.4.5.6", "2.4.5.6", "3.4.5.6");
  t_ipv4_iter("0,255.0,255.0,255.0,255" =>
              "0.0.0.0",
              "0.0.0.255",
              "0.0.255.0",
              "0.0.255.255",
              "0.255.0.0",
              "0.255.0.255",
              "0.255.255.0",
              "0.255.255.255",
              "255.0.0.0",
              "255.0.0.255",
              "255.0.255.0",
              "255.0.255.255",
              "255.255.0.0",
              "255.255.0.255",
              "255.255.255.0",
              "255.255.255.255");

  SKIP: {
    skip("ipv6 not enabled", 12) unless SILK_IPV6_ENABLED;

    t_ipv6_iter("::" => "::");
    t_ipv6_iter("::/127" => "::0", "::1");
    t_ipv6_iter("0:ffff::0.0.0.0,1" => "0:ffff::0", "0:ffff::1");
    t_ipv6_iter("0:ffff::0.253-254.125-126,255.1" =>
                "0:ffff::0.253.125.1",
                "0:ffff::0.253.126.1",
                "0:ffff::0.253.255.1",
                "0:ffff::0.254.125.1",
                "0:ffff::0.254.126.1",
                "0:ffff::0.254.255.1");
  }

}

###

sub t_isipv4 {
  my $wc = SILK_IPWILDCARD_CLASS->new(shift);
  ok(!$wc->is_ipv6, "is ipv4 wildcard");
}

sub t_isipv6 {
  my $wc = SILK_IPWILDCARD_CLASS->new(shift);
  ok($wc->is_ipv6, "is ipv6 wildcard");
}

sub test_isipv6 {

  plan tests => 9;

  t_isipv4("0.0.0.0");
  t_isipv4("0.0.0.0/31");
  t_isipv4("255.255.255.254-255");
  t_isipv4("3,2,1.4.5.6");
  t_isipv4("0,255.0,255.0,255.0,255");

  SKIP: {
    skip("ipv6 not enabled", 4) unless SILK_IPV6_ENABLED;
    t_isipv6("::");
    t_isipv6("::/127");
    t_isipv6("0:ffff::0.0.0.0,1");
    t_isipv6("0:ffff:0:0:0:0:0.253-254.125-126,255.x");
  }
}

###

sub test_all {
  subtest "construction" => \&test_construction;
  subtest "bad strings"  => \&test_bad_strings;
  subtest "containment"  => \&test_containment;
  subtest "iteration"    => \&test_iteration;
  subtest "isipv6"       => \&test_isipv6;
}

test_all();

###
