# Copyright 2015 Kevin Ryde
#
# This file is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 3, or (at your option) any later
# version.
#
# This file is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
# or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
# for more details.
#
# You should have received a copy of the GNU General Public License along
# with this file.  See the file COPYING.  If not, see
# <http://www.gnu.org/licenses/>.

package Graph::Maker::Beineke;
use 5.004;
use strict;
use Carp 'croak';
use base 'Graph::Maker';
use Graph::Maker::Star;
use Graph::Maker::Wheel;

use vars '$VERSION';
$VERSION = 5;

# uncomment this to run the ### lines
# use Smart::Comments;

my @data
  = (undef, # 0
     undef, # 1
     [
      # G2    1--\
      #      / \  \
      #     2---3  5
      #      \ /  /
      #       4--/
      # https://hog.grinvin.org/ViewGraphInfo.action?id=438
      [1,2],[1,3],[1,5],
      [2,3],[2,4],
      [3,4],
      [5,4],
     ],
     [
      # G3 = K5-e complete 5 less one edge
      #      1
      #     /|\
      #    / 2 \
      #   / /|\ \
      #  / / 3 \ \
      #  |/ / \ \ |
      #  4--------5
      # https://hog.grinvin.org/ViewGraphInfo.action?id=450
      #
      [1,2],[1,4],[1,5],
      [2,3],[2,4],[2,5],
      [3,4],[3,5],
      [4,5],
     ],
     [
      # G4   1----5
      #     / \
      #    2---3
      #     \ /
      #      4----6
      # https://hog.grinvin.org/ViewGraphInfo.action?id=922
      #
      [1,2],[1,3],[1,5],
      [2,3],[2,4],
      [3,4],
      [4,6],
     ],
     [
      # G5    1
      #      /|\
      #     / 3 \
      #    / / \ \
      #    2-----4
      #     \    /
      #      \  /
      #       5----6
      # https://hog.grinvin.org/ViewGraphInfo.action?id=21099
      #
      [1,2],[1,5],[1,3],
      [2,5],[2,3],[2,4],
      [3,4],[3,5],
      [4,6],
     ],
     [
      # G6    1
      #      /|\
      #     / 3 \
      #    / / \ \
      #    2-----4
      #    \ \ / /
      #     \ 5 /
      #      \|/
      #       6
      # https://hog.grinvin.org/ViewGraphInfo.action?id=744
      #
      [1,2],[1,5],[1,3],
      [2,5],
      [2,3],[2,6],[2,4],
      [3,5],[3,6],[3,4],
      [4,6],
     ],
     [
      # G7   1----5
      #     / \   |
      #    2---3  |
      #     \ /   |
      #      4----6
      # https://hog.grinvin.org/ViewGraphInfo.action?id=21093
      #
      [1,2],[1,3],[1,5],
      [2,3],[2,4],
      [3,4],
      [4,6],
      [5,6],
     ],
     [
      # G8 1---2
      #    | / |
      #    3---4
      #    | / |
      #    5---6
      # https://hog.grinvin.org/ViewGraphInfo.action?id=21096
      #
      [1,2],[1,3],
      [2,3],[2,4],
      [3,4],[3,5],
      [4,5],[4,6],
      [5,6],
     ],
    );

sub _default_graph_maker {
  require Graph;
  Graph->new(@_);
}

sub init {
  my ($self, %params) = @_;

  my $G = delete($params{'G'}) || 0;
  ### $G

  if ($G == 1) {
    # G1 = claw = star 3
    # https://hog.grinvin.org/ViewGraphInfo.action?id=204
    my $graph = Graph::Maker->new('star', N=>4, %params);
    $graph->set_graph_attribute (name => "Beineke G1");
    return $graph;
  }

  if ($G == 9) {
    # G9 = wheel 6
    #        2        6 vertices
    #      / | \     10 edges
    #    4---1---3
    #     \ / \  /
    #      5---6
    # https://hog.grinvin.org/ViewGraphInfo.action?id=204
    #
    my $graph = Graph::Maker->new('wheel', N=>6, %params);
    $graph->set_graph_attribute (name => "Beineke G9");
    return $graph;
  }

  my $edge_aref = $data[$G]
    || croak "Unrecognised G: ", $G;

  my $graph_maker = delete($params{'graph_maker'});
  $graph_maker ||= \&_default_graph_maker;
  my $graph = $graph_maker->(%params);

  $graph->set_graph_attribute (name => "Beineke G$G");
  my $directed = $graph->is_directed;

  foreach my $edge (@$edge_aref) {
    ### $edge
    $graph->add_edge(@$edge);
    if ($directed) { $graph->add_edge(reverse @$edge); }
  }
  return $graph;
}

Graph::Maker->add_factory_type('beineke' => __PACKAGE__);
1;

=for stopwords Ryde

=head1 NAME

Graph::Maker::Beineke - create beineke tree graph

=for test_synopsis my ($graph)

=head1 SYNOPSIS

 use Graph::Maker::Beineke;
 $graph = Graph::Maker->new ('beineke', G => 9);

=head1 DESCRIPTION

C<Graph::Maker::Beineke> creates C<Graph.pm> graphs of the 9 graphs given by
Beineke

=over

Lowell W. Beineke, "Characterizations of Derived Graphs", Journal of
Combinatorial Theory, volume 9, 1970, pages 129-135.
L<http://www.sciencedirect.com/science/article/pii/S0021980070800199>

=back

Beineke shows that line graphs are characterized as all graphs which do not
contain as induced sub-graphs any of the following graphs

                                    
    G1    2           G2   1--\          G3    1                 
          |               / \  \              /|\                
          1              2---3  5            / 2 \               
         / \              \ /  /            / /|\ \              
        3   4              4--/            / / 3 \ \             
                                           |/ / \ \ |            
    claw = 4-star                          4--------5            
                                                                 
                                  K5-e complete 5 less one edge
                                         

    G4   1----5       G5    1            G6    1
        / \                /|\                /|\
       2---3              / 3 \              / 3 \
        \ /              / / \ \            / / \ \
         4----6          2-----4            2-----4
                          \   /             \ \ / /
                           \ /               \ 5 /
                            5----6            \|/
                                               6

    G7   1----5       G8  1---2          G9    2
        / \   |           | / |              / | \
       2---3  |           3---4            4---1---3
        \ /   |           | / |             \ / \  /
         4----6           5---6              5---6

                                            6-wheel

G1 is the claw (4-star) and is created by C<Graph::Maker::Star>.  G9 is the
6-wheel and is created by C<Graph::Maker::Wheel>.

=head1 FUNCTIONS

=over

=item C<$graph = Graph::Maker-E<gt>new('beineke', key =E<gt> value, ...)>

The key/value parameters are

    G =>  integer 1 to 9

Other parameters are passed to C<Graph-E<gt>new()>.

=back

=head1 SEE ALSO

L<Graph::Maker>,
L<Graph::Maker::Star>,
L<Graph::Maker::Wheel>

=head1 LICENSE

Copyright 2015 Kevin Ryde

This file is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 3, or (at your option) any later
version.

This file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
more details.

You should have received a copy of the GNU General Public License along with
This file.  If not, see L<http://www.gnu.org/licenses/>.

=cut
