#!/usr/bin/perl -w

# Copyright 2015 Kevin Ryde
#
# This file is part of Graph-Graph6.
#
# Graph-Graph6 is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 3, or (at your option) any later
# version.
#
# Graph-Graph6 is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
# more details.
#
# You should have received a copy of the GNU General Public License along
# with Graph-Graph6.  If not, see <http://www.gnu.org/licenses/>.

use strict;
use Test;

my $have_graph;
BEGIN {  # before warnings checking
  # before warnings checking since Graph.pm 0.96 is not safe to non-numeric
  # version number from Storable.pm 
  $have_graph = eval { require Graph; 1 };
}

use lib 't';
use MyTestHelpers;
BEGIN { MyTestHelpers::nowarnings() }

my $test_count = (tests => 8)[1];
plan tests => $test_count;

if (! $have_graph) {
  foreach (1 .. $test_count) {
    skip ('no Graph.pm', 1, 1);
  }
  exit 0;
}

my $have_graph_writer = eval { require Graph::Writer; 1 };
if (! $have_graph_writer) {
  foreach (1 .. $test_count) {
    skip ('no Graph::Writer', 1, 1);
  }
  exit 0;
}

{
  my $have_file_temp = eval { require File::Temp; 1 };
  if (! $have_file_temp) {
    foreach (1 .. $test_count) {
      skip ('no File::Temp', 1, 1);
    }
    exit 0;
  }
}

require Graph::Reader::Graph6;

#------------------------------------------------------------------------------
{
  my $want_version = 2;
  ok ($Graph::Reader::Graph6::VERSION, $want_version, 'VERSION variable');
  ok (Graph::Reader::Graph6->VERSION,  $want_version, 'VERSION class method');
  ok (eval { Graph::Reader::Graph6->VERSION($want_version); 1 }, 1,
      "VERSION class check $want_version");
  my $check_version = $want_version + 1000;
  ok (! eval { Graph::Reader::Graph6->VERSION($check_version); 1 }, 1,
      "VERSION class check $check_version");
}

#------------------------------------------------------------------------------

{
  # formats.txt graph6 example

  my $tempfh = File::Temp->new;
  print $tempfh chr(68),chr(81),chr(99),"\n";
  close $tempfh or die "Oops error closing tempfile: $!";
  my $filename = $tempfh->filename;

  my $parser = Graph::Reader::Graph6->new;
  my $graph = $parser->read_graph($filename);
  ok (join(',', sort $graph->vertices),
      '0,1,2,3,4');  # 5 vertices

  my @edges = $graph->edges; # [$name1,$name2]
  my @edge_names = map { join('=', @$_) } @edges;
  ok (join(',',sort @edge_names),
      '0=2,0=4,1=3,3=4');
}

{
  # formats.txt sparse6 example

  my $tempfh = File::Temp->new;
  print $tempfh ':Fa@x^', "\n";
  my $filename = $tempfh->filename;
  seek $tempfh, 0, 0 or die "Oopss, error seeking tempfile: $!";

  my $parser = Graph::Reader::Graph6->new;
  my $graph = $parser->read_graph($tempfh);
  ok (join(',', sort $graph->vertices),
      '0,1,2,3,4,5,6');  # 7 vertices

  my @edges = $graph->edges; # [$name1,$name2]
  my @edge_names = map { join('=', @$_) } @edges;
  ok (join(',',sort @edge_names),
      '0=1,0=2,1=2,5=6');

  close $tempfh or die "Oops error closing tempfile: $!";
}

#------------------------------------------------------------------------------
exit 0;
