#!/usr/bin/perl -w

# Copyright 2015 Kevin Ryde
#
# This file is part of Graph-Graph6.
#
# Graph-Graph6 is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 3, or (at your option) any later
# version.
#
# Graph-Graph6 is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
# more details.
#
# You should have received a copy of the GNU General Public License along
# with Graph-Graph6.  If not, see <http://www.gnu.org/licenses/>.

use strict;
use Test;

# uncomment this to run the ### lines
# use Smart::Comments;


my $test_count = (tests => 20)[1];
plan tests => $test_count;

{
  my $have_graph_easy = eval { require Graph::Easy; 1 };
  if (! $have_graph_easy) {
    foreach (1 .. $test_count) {
      skip ('no Graph::Easy', 1, 1);
    }
    exit 0;
  }
}
{
  my $have_graph_easy_parser = eval { require Graph::Easy::Parser; 1 };
  if (! $have_graph_easy_parser) {
    foreach (1 .. $test_count) {
      skip ('no Graph::Easy::Parser', 1, 1);
    }
    exit 0;
  }
}
{
  my $have_file_temp = eval { require File::Temp; 1 };
  if (! $have_file_temp) {
    foreach (1 .. $test_count) {
      skip ('no File::Temp', 1, 1);
    }
    exit 0;
  }
}

require Graph::Easy::Parser::Graph6;

#------------------------------------------------------------------------------
{
  my $want_version = 2;
  ok ($Graph::Easy::Parser::Graph6::VERSION, $want_version, 'VERSION variable');
  ok (Graph::Easy::Parser::Graph6->VERSION,  $want_version, 'VERSION class method');
  ok (eval { Graph::Easy::Parser::Graph6->VERSION($want_version); 1 }, 1,
      "VERSION class check $want_version");
  my $check_version = $want_version + 1000;
  ok (! eval { Graph::Easy::Parser::Graph6->VERSION($check_version); 1 }, 1,
      "VERSION class check $check_version");
}

#------------------------------------------------------------------------------
# from_text()

sub graph_vertices_str {
  my ($graph) = @_;
  if (! defined $graph) {
    return 'undef';
  }
  return join(',', sort map{$_->name} $graph->nodes);
}
sub graph_edges_str {
  my ($graph) = @_;
  if (! defined $graph) {
    return 'undef';
  }
  my @edges = $graph->edges;
  my @edge_names = map { join('=',map{$_->name} $_->nodes) } @edges;
  return join(',',sort @edge_names);
}

{
  ### from_text() formats.txt sparse6 example ...
  my $parser = Graph::Easy::Parser::Graph6->new;
  my $graph = $parser->from_text(':Fa@x^'."\n");
  ok (defined $graph, 1);

  ok (graph_vertices_str($graph),
      '0,1,2,3,4,5,6');  # 7 vertices
  ok (graph_edges_str($graph),
      '0=1,0=2,1=2,5=6');
}

{
  ### from_text() class method ...
  my $graph = Graph::Easy::Parser::Graph6->from_text(':'.chr(63+11));
  ok (defined $graph, 1);

  ok (graph_vertices_str($graph),
      '00,01,02,03,04,05,06,07,08,09,10');  # 11 vertices
  ok (graph_edges_str($graph),
      '');
}

{
  ### EOF ...
  my $parser = Graph::Easy::Parser::Graph6->new;
  my $graph = $parser->from_text('');
  ok ($graph, undef);
  ok ($parser->error, '');
}
{
  ### bad input ...
  my $parser = Graph::Easy::Parser::Graph6->new (fatal_errors => 0);
  my $graph = $parser->from_text('!!invalid');
  ok (defined $graph, 1);  # partially read graph
  ok ($parser->error, 'Unrecognised character: !');
}

#------------------------------------------------------------------------------
# from_file()

{
  ### from_file() formats.txt graph6 example ...

  my $tempfh = File::Temp->new;
  print $tempfh chr(68),chr(81),chr(99),"\n";
  close $tempfh or die "Oops error closing tempfile: $!";
  my $filename = $tempfh->filename;

  my $parser = Graph::Easy::Parser::Graph6->new;
  my $graph = $parser->from_file($filename);
  ok (defined $graph, 1);

  ok (graph_vertices_str($graph),
      '0,1,2,3,4');  # 5 vertices
  ok (graph_edges_str($graph),
      '0=2,0=4,1=3,3=4');
}

{
  ### from_file() class method

  my $tempfh = File::Temp->new;
  print $tempfh chr(63+2), chr(63+63);
  close $tempfh or die "Oops error closing tempfile: $!";
  my $filename = $tempfh->filename;

  my $parser = Graph::Easy::Parser::Graph6->new;
  my $graph = $parser->from_file($filename);
  ok (defined $graph, 1);

  ok (graph_vertices_str($graph),
      '0,1');  # 2 vertices
  ok (graph_edges_str($graph),
      '0=1');
}

#------------------------------------------------------------------------------
exit 0;
