use strict;
use warnings;

use PDL;
use PDL::Finance::Talib;
use Test::More;
use Test::Number::Delta relative => 1e-9;

sub pdl2aref {
  my $pdl = shift;
  my @out = ();
  my @in = $pdl->list;
  my $beginning = 1;
  for (@in) {
    next if $beginning && $_ && $_ eq 'BAD';
    $beginning = 0;
    push @out, $_;
  }
  return \@out;
}

my $O  = pdl([192.65, 194.15, 192.82, 191.31, 192, 193.85, 194.03, 194.36, 203.5, 203.91, 204.45, 204.85, 204.34, 203.69, 203.32, 204.65, 204.19, 204.31, 200.39, 200.62, 199.97]);
my $L  = pdl([191.28, 192.9, 191.75, 190.39, 191.35, 193.24, 193.8, 194.01, 203.36, 203.08, 204.13, 204.29, 203.64, 203.19, 202.96, 203.84, 203.57, 202.51, 199.56, 198.68, 199.68]);
my $H  = pdl([192.96, 195, 193.28, 192.73, 193.18, 194.46, 195, 196.08, 208.58, 205.06, 205.18, 206.22, 205.73, 204.88, 204.47, 205.35, 205.02, 204.75, 201.29, 200.91, 202.09]);
my $C  = pdl([192.88, 194.45, 192.62, 192.5, 192.59, 193.65, 194.47, 196.08, 204.72, 204.42, 204.97, 204.93, 203.9, 203.52, 203.07, 205.18, 203.79, 202.79, 201.02, 199.74, 201.68]);
my $V  = pdl([3608100, 3880400, 4172200, 4172100, 2962900, 3881600, 4559600, 7172300, 12530500, 4447200, 3358900, 2823700, 3617100, 2998400, 3091400, 3370700, 3188800, 3636900, 3624200, 3076700, 2893300]);
my $R  = pdl([528.55, 521, 502.68, 498.3, 494.64, 510.31, 498.52, 504.56, 508.81, 460, 451.69, 437.83, 458.5, 457, 456.98, 459.11, 453.91, 444.05, 456.47, 463.25, 474]);
my $P  = pdl([5.7, 5.2, 5.5, 4.9, 9.4, 5.7, 5.2, 7.8, 1.9, 4.7, 6.2, 4.1, 6.2, 2.6, 9.2, 4.4, 4.9, 9.4, 6.5, 4.7, 8.1]);
my $Z  = pdl([0.57, 0.552, 0.555, 0.549, 0.594, 0.557, 0.552, 0.578, 0.519, 0.547, 0.562, 0.541, 0.562, 0.526, 0.592, 0.544, 0.549, 0.594, 0.565, 0.547, 0.581]);
my $R0 = pdl([515.52, 519.02, 498.51, 483.38, 492.5, 502.03, 496.4, 496.63, 504.77, 450.25, 435, 435.86, 452.12, 454.5, 454.98, 448.35, 442, 442.22, 452.58, 454.12, 468.25]);
my $R1 = pdl([528.72, 525.32, 507.5, 498.99, 509.44, 510.75, 502.22, 507.88, 514.99, 465.73, 456.23, 453.21, 460.2, 462.6, 459.28, 459.48, 455.94, 459.74, 466.5, 470, 478.81]);

my $C4  = pdl([192.88, 194.45, 192.62, 192.5, 192.59, 193.65, 194.47, 196.08, 204.72, 204.42, 204.97, 204.93, 203.9, 203.52, 203.07, 205.18, 203.79, 202.79, 201.02, 199.74, 201.68,
               192.88, 194.45, 192.62, 192.5, 192.59, 193.65, 194.47, 196.08, 204.72, 204.42, 204.97, 204.93, 203.9, 203.52, 203.07, 205.18, 203.79, 202.79, 201.02, 199.74, 201.68,
               192.88, 194.45, 192.62, 192.5, 192.59, 193.65, 194.47, 196.08, 204.72, 204.42, 204.97, 204.93, 203.9, 203.52, 203.07, 205.18, 203.79, 202.79, 201.02, 199.74, 201.68,
               192.88, 194.45, 192.62, 192.5, 192.59, 193.65, 194.47, 196.08, 204.72, 204.42, 204.97, 204.93, 203.9, 203.52, 203.07, 205.18, 203.79, 202.79, 201.02, 199.74, 201.68,
              ]);
my $len = 21;

my $optInTimePeriod = 10;               # [Number of period] - integer; valid range: min=2 max=100000
my $optInAcceleration = 0.02;           # [Acceleration Factor used up to the Maximum value] - real number; valid range: min=0 max=3e+037
my $optInAccelerationInitLong = 0.02;   # [Acceleration Factor initial value for the Long direction] - real number; valid range: min=0 max=3e+037
my $optInAccelerationInitShort = 0.02;  # [Acceleration Factor initial value for the Short direction] - real number; valid range: min=0 max=3e+037
my $optInAccelerationLong = 0.02;       # [Acceleration Factor for the Long direction] - real number; valid range: min=0 max=3e+037
my $optInAccelerationMaxLong = 0.2;     # [Acceleration Factor maximum value for the Long direction] - real number; valid range: min=0 max=3e+037
my $optInAccelerationMaxShort = 0.2;    # [Acceleration Factor maximum value for the Short direction] - real number; valid range: min=0 max=3e+037
my $optInAccelerationShort = 0.02;      # [Acceleration Factor for the Short direction] - real number; valid range: min=0 max=3e+037
my $optInFastD_MAType = 2;              # [Type of Moving Average for Fast-D] - integer; valid values: 0=SMA 1=EMA 2=WMA 3=DEMA 4=TEMA 5=TRIMA 6=KAMA 7=MAMA 8=T3
my $optInFastD_Period = 3;              # [Smoothing for making the Fast-D line. Usually set to 3] - integer; valid range: min=1 max=100000
my $optInFastK_Period = 5;              # [Time period for building the Fast-K line] - integer; valid range: min=1 max=100000
my $optInFastLimit = 0.5;               # [Upper limit use in the adaptive algorithm] - real number; valid range: min=0.01 max=0.99
my $optInFastMAType = 1;                # [Type of Moving Average for fast MA] - integer; valid values: 0=SMA 1=EMA 2=WMA 3=DEMA 4=TEMA 5=TRIMA 6=KAMA 7=MAMA 8=T3
my $optInFastPeriod = 2;                # [Number of period for the fast MA] - integer; valid range: min=2 max=100000
my $optInMAType = 2;                    # [Type of Moving Average] - integer; valid values: 0=SMA 1=EMA 2=WMA 3=DEMA 4=TEMA 5=TRIMA 6=KAMA 7=MAMA 8=T3
my $optInMaxPeriod = 4;                 # [Value higher than maximum will be changed to Maximum period] - integer; valid range: min=2 max=100000
my $optInMaximum = 0.2;                 # [Acceleration Factor Maximum value] - real number; valid range: min=0 max=3e+037
my $optInMinPeriod = 2;                 # [Value less than minimum will be changed to Minimum period] - integer; valid range: min=2 max=100000
my $optInNbDev = 1;                     # [Nb of deviations] - real number; valid range: min=-3e+037 max=3e+037
my $optInNbDevUp = 2;                   # [Deviation multiplier for upper band] - real number; valid range: min=-3e+037 max=3e+037
my $optInNbDevDn = 2;                   # [Deviation multiplier for lower band] - real number; valid range: min=-3e+037 max=3e+037
my $optInOffsetOnReverse = 1;           # [Percent offset added/removed to initial stop on short/long reversal] - real number; valid range: min=0 max=3e+037
my $optInPenetration = 0.3;             # [Percentage of penetration of a candle within another candle] - real number; valid range: min=0 max=3e+037
my $optInSignalMAType = 3;              # [Type of Moving Average for signal line] - integer; valid values: 0=SMA 1=EMA 2=WMA 3=DEMA 4=TEMA 5=TRIMA 6=KAMA 7=MAMA 8=T3
my $optInSignalPeriod = 5;              # [Smoothing for the signal line (nb of period)] - integer; valid range: min=1 max=100000
my $optInSlowD_MAType = 0;              # [Type of Moving Average for Slow-D] - integer; valid values: 0=SMA 1=EMA 2=WMA 3=DEMA 4=TEMA 5=TRIMA 6=KAMA 7=MAMA 8=T3
my $optInSlowD_Period = 3;              # [Smoothing for making the Slow-D line] - integer; valid range: min=1 max=100000
my $optInSlowK_MAType = 1;              # [Type of Moving Average for Slow-K] - integer; valid values: 0=SMA 1=EMA 2=WMA 3=DEMA 4=TEMA 5=TRIMA 6=KAMA 7=MAMA 8=T3
my $optInSlowK_Period = 3;              # [Smoothing for making the Slow-K line. Usually set to 3] - integer; valid range: min=1 max=100000
my $optInSlowLimit = 0.05;              # [Lower limit use in the adaptive algorithm] - real number; valid range: min=0.01 max=0.99
my $optInSlowMAType = 2;                # [Type of Moving Average for slow MA] - integer; valid values: 0=SMA 1=EMA 2=WMA 3=DEMA 4=TEMA 5=TRIMA 6=KAMA 7=MAMA 8=T3
my $optInSlowPeriod = 5;                # [Number of period for the slow MA] - integer; valid range: min=2 max=100000
my $optInStartValue = -1;               # [Start value and direction. 0 for Auto, >0 for Long, <0 for Short] - real number; valid range: min=-3e+037 max=3e+037
my $optInTimePeriod1 = 3;               # [Number of bars for 1st period.] - integer; valid range: min=1 max=100000
my $optInTimePeriod2 = 6;               # [Number of bars for 2nd period.] - integer; valid range: min=1 max=100000
my $optInTimePeriod3 = 9;               # [Number of bars for 3rd period.] - integer; valid range: min=1 max=100000
my $optInVFactor = 0.7;                 # [Volume Factor] - real number; valid range: min=0 max=1

{
  my ($outRealUpperBand, $outRealMiddleBand, $outRealLowerBand) = ta_bbands($R, $optInTimePeriod, $optInNbDevUp, $optInNbDevDn, $optInMAType);
  my $outRealUpperBand_expected = [ 531.382459065198, 529.141277746873, 527.040443924978, 523.275250907033, 519.499916178695, 515.78599254102, 507.721697662976, 501.328856942882, 489.953066465923, 467.240149984415, 469.733303882044, 477.622372238609 ] ;
  delta_ok pdl2aref($outRealUpperBand), $outRealUpperBand_expected, 'ta_bbands:outRealUpperBand';
  my $outRealMiddleBand_expected = [ 496.586545454545, 487.305272727273, 476.901454545455, 471.768, 467.165090909091, 463.309454545455, 460.525818181818, 457.727636363636, 453.947818181818, 453.526363636364, 455.289272727273, 458.947636363636 ] ;
  delta_ok pdl2aref($outRealMiddleBand), $outRealMiddleBand_expected, 'ta_bbands:outRealMiddleBand';
  my $outRealLowerBand_expected = [ 461.790631843893, 445.469267707673, 426.762465165931, 420.260749092967, 414.830265639487, 410.832916549889, 413.32993870066, 414.126415784391, 417.942569897714, 439.812577288312, 440.845241572502, 440.272900488663 ] ;
  delta_ok pdl2aref($outRealLowerBand), $outRealLowerBand_expected, 'ta_bbands:outRealLowerBand';
}
{
  my $outReal = ta_dema($R, $optInTimePeriod);
  my $outReal_expected = [ 446.264867850754, 449.634703241312, 455.894354924534 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_dema:outReal';
}
{
  my $outReal = ta_ema($R, $optInTimePeriod);
  my $outReal_expected = [ 502.737, 493.455727272727, 483.341958677686, 478.825238918107, 474.857013660269, 471.60664754022, 469.334529805635, 466.530069840974, 462.442784415342, 461.356823612553, 461.70103750118, 463.937212500965 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_ema:outReal';
}
{
  my $outReal = ta_ht_trendline($C4);
  my $outReal_expected = [ 199.609373508376, 199.412626576322, 199.158957180501, 198.978311594203, 198.69943115942, 198.457685185185, 198.255900793651, 198.106143906221, 198.13074966247, 198.38207955665, 198.844288669951, 199.44006656632, 199.966790822843, 200.320885734636, 200.402267065527, 200.337241253561, 200.216686053512, 200.09053826087, 199.921477272727, 199.788662431771, 199.709073781291 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_ht_trendline:outReal';
}
{
  my $outReal = ta_kama($R, $optInTimePeriod);
  my $outReal_expected = [ 458.506775048664, 454.58992410124, 454.850349839919, 454.983985900066, 455.09679050285, 455.526360725362, 455.374045895411, 453.717872020146, 454.001820676163, 454.074666656218, 455.092387626505 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_kama:outReal';
}
{
  my $outReal = ta_ma($R, $optInTimePeriod, $optInMAType);
  my $outReal_expected = [ 496.586545454545, 487.305272727273, 476.901454545455, 471.768, 467.165090909091, 463.309454545455, 460.525818181818, 457.727636363636, 453.947818181818, 453.526363636364, 455.289272727273, 458.947636363636 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_ma:outReal';
}
{
  my ($outMAMA, $outFAMA) = ta_mama($C4, $optInFastLimit, $optInSlowLimit);
  my $outMAMA_expected = [ 200.320202970611, 200.640472541101, 200.784448914046, 200.898726468344, 201.112790144926, 201.24665063768, 201.323818105796, 201.171909052898, 201.100313600253, 201.12929792024, 200.716833024228, 197.583416512114, 197.225861154254, 196.989568096541, 194.789784048271, 194.219892024135, 194.232397422929, 194.324777551782, 194.844538674193, 195.323311740483, 195.805646153459, 200.36782307673, 200.599422298671, 200.745451183738, 200.861678624551, 201.077594693323, 201.213214958657, 201.292054210724, 201.156027105362, 201.085225750094, 201.114964462589, 200.70321623946, 197.57660811973, 197.235255855986, 196.998493063187, 194.794246531593, 194.222123265797, 194.234517102507, 194.326791247381, 194.846451685012, 195.325129100762, 195.807372645724, 200.368686322862, 200.601696361973, 200.747611543874, 200.86373096668, 201.079544418346, 201.215067197429, 201.293813837558, 201.156906918779, 201.08606157284, 201.115758494198 ] ;
  delta_ok pdl2aref($outMAMA), $outMAMA_expected, 'ta_mama:outMAMA';
  my $outFAMA_expected = [ 185.947659046097, 186.604911542356, 186.959399976648, 187.307883138941, 187.65300581409, 187.99284693468, 188.326121213958, 191.537568173693, 191.776636809357, 192.010453337129, 192.228112829307, 193.566938750009, 193.698729755905, 193.781000714421, 194.033196547884, 194.079870416947, 194.083683592096, 194.089710941088, 194.108581634416, 194.138949887068, 194.180617293727, 195.727418739478, 195.887143501023, 196.00860119309, 196.129928128877, 196.253619792988, 196.37760967213, 196.500470785595, 197.664359865537, 197.74988151265, 197.834008586399, 197.905738777725, 197.823456113227, 197.803201991406, 197.783084268201, 197.035874834049, 196.332436941986, 196.279988945999, 196.231159003534, 196.196541320571, 196.174756015075, 196.165571430841, 197.216350153847, 197.328039418024, 197.41352872117, 197.499783777308, 197.589277793334, 197.679922528436, 197.770269811165, 198.616929088068, 198.678657400187, 198.739584927538 ] ;
  delta_ok pdl2aref($outFAMA), $outFAMA_expected, 'ta_mama:outFAMA';
}
{
  my $outReal = ta_mavp($R, $P, $optInMinPeriod, $optInMaxPeriod, $optInMAType);
  my $outReal_expected = [ 507.179, 499.982, 502.444, 501.259, 502.906, 507.393333333333, 487.407, 470.894, 453.52, 451.087, 457.5, 455.375, 457.988, 456.393, 451.313, 452.496, 456.442, 464.274 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_mavp:outReal';
}
{
  my $outReal = ta_midpoint($R, $optInTimePeriod);
  my $outReal_expected = [ 494.275, 486.345, 474.07, 474.07, 474.07, 474.07, 473.32, 473.32, 473.32, 448.915, 450.54, 455.915 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_midpoint:outReal';
}
{
  my $outReal = ta_midprice($H, $L, $optInTimePeriod);
  my $outReal_expected = [ 199.485, 199.485, 199.485, 199.485, 199.965, 200.91, 201.19, 201.295, 205.545, 202.89, 202.45, 202.45 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_midprice:outReal';
}
{
  my $outReal = ta_sar($H, $L, $optInAcceleration, $optInMaximum);
  my $outReal_expected = [ 191.28, 191.3544, 195, 194.9078, 194.817444, 190.39, 190.4822, 190.706112, 191.77854528, 192.7866325632, 193.734234609408, 194.624980532844, 195.462281700873, 196.249344798821, 196.989184110891, 197.684633064238, 198.338355080384, 198.952853775561, 208.58, 208.382 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_sar:outReal';
}
{
  my $outReal = ta_sarext($H, $L, $optInStartValue, $optInOffsetOnReverse, $optInAccelerationInitLong, $optInAccelerationLong, $optInAccelerationMaxLong, $optInAccelerationInitShort, $optInAccelerationShort, $optInAccelerationMaxShort);
  my $outReal_expected = [ 0, 3.9, 7.722, 11.46756, 15.1382088, 18.735444624, 22.26073573152, 29.2135063022592, 39.9754959241236, 50.0917661686762, 59.6010601985557, 68.5397965866423, 76.9422087914438, 84.8404762639572, 92.2648476881197, 99.2437568268325, 105.803931417223, 111.970495532189, 117.767065800258, 123.215841852242 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_sarext:outReal';
}
{
  my $outReal = ta_sma($R, $optInTimePeriod);
  my $outReal_expected = [ 502.737, 495.051, 486.734, 482.316, 478.186, 474.42, 469.3, 464.839, 458.788, 453.554, 453.879, 456.11 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_sma:outReal';
}
{
  my $outReal = ta_t3($C4, $optInTimePeriod, $optInVFactor);
  my $outReal_expected = [ 197.638273754142, 198.556994963186, 199.471855561586, 200.376909544093, 201.220617131579, 201.949847397994, 202.503178714023, 202.840030671057, 203.014705843419, 202.866107694896, 202.414046644353, 201.680252516647, 200.72959492192, 199.644869506272, 198.529222789293, 197.47368616612, 196.567288377112, 196.045860823443, 195.955248387165, 196.263154119585, 196.886809870158, 197.70509732408, 198.610498737665, 199.514707378849, 200.411246576441, 201.248149724079, 201.97194307946, 202.52093004509, 202.854309945144, 203.026209115744 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_t3:outReal';
}
{
  my $outReal = ta_tema($C4, $optInTimePeriod);}
{
  my $outReal = ta_trima($R, $optInTimePeriod);
  my $outReal_expected = [ 502.730666666667, 498.677666666667, 493.209666666667, 486.955666666667, 478.912, 470.417333333333, 463.324666666667, 458.722, 455.660666666667, 454.117, 454.509333333333, 455.498666666667 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_trima:outReal';
}
{
  my $outReal = ta_wma($R, $optInTimePeriod);
  my $outReal_expected = [ 496.586545454545, 487.305272727273, 476.901454545455, 471.768, 467.165090909091, 463.309454545455, 460.525818181818, 457.727636363636, 453.947818181818, 453.526363636364, 455.289272727273, 458.947636363636 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_wma:outReal';
}
{
  my $outReal = ta_atr($H, $L, $C, $optInTimePeriod);
  my $outReal_expected = [ 2.981, 2.8759, 2.79731, 2.686579, 2.5689211, 2.54002899, 2.447026091, 2.4263234819, 2.50669113371, 2.490022020339, 2.4820198183051 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_atr:outReal';
}
{
  my $outReal = ta_natr($H, $L, $C, $optInTimePeriod);
  my $outReal_expected = [ 1.45435917451334, 1.40335724393696, 1.37190289357528, 1.32005650550315, 1.26504215295218, 1.23795154985866, 1.20075866872761, 1.19647097090587, 1.2469859385683, 1.24663163129018, 1.23067226215049 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_natr:outReal';
}
{
  my $outReal = ta_trange($H, $L, $C);
  my $outReal_expected = [ 2.12, 2.69999999999999, 2.34, 1.83000000000001, 1.87, 1.34999999999999, 2.07000000000002, 12.5, 1.97999999999999, 1.05000000000001, 1.93000000000001, 2.09, 1.69, 1.50999999999999, 2.28, 1.61000000000001, 2.24000000000001, 3.22999999999999, 2.34, 2.41 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_trange:outReal';
}
{
  my $outReal = ta_adx($H, $L, $C, $optInTimePeriod);
  my $outReal_expected = [ 53.0964552554776, 49.3816167352615 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_adx:outReal';
}
{
  my $outReal = ta_adxr($H, $L, $C, 2); # $optInTimePeriod = 2
  my $outReal_expected = [ 28.0261388158044, 31.5074209848252, 46.756833383814, 62.5929272156769, 77.853470007474, 86.4857349912282, 88.8842615269908, 90.9388537555131, 77.5113959911941, 48.8608081813996, 33.3382081048099, 37.4031139437997, 34.8480539552525, 36.474320883812, 57.8992047936763, 75.9775776405214, 63.0006881731016 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_adxr:outReal';
}
{
  my $outReal = ta_apo($R, $optInFastPeriod, $optInSlowPeriod, $optInMAType);
  my $outReal_expected = [ -7.1393333333333, 1.66199999999998, 1.31400000000002, 0.187333333333356, 2.51933333333329, -14.148, -21.0413333333333, -17.4226666666667, -3.56999999999999, 4.44200000000001, 2.60333333333335, 1.78000000000003, -0.985333333333301, -4.94200000000001, -0.701999999999998, 4.90933333333334, 8.12200000000001 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_apo:outReal';
}
{
  my ($outAroonDown, $outAroonUp) = ta_aroon($H, $L, $optInTimePeriod);
  my $outAroonDown_expected = [ 30, 20, 10, 0, 0, 0, 0, 0, 100, 100, 90 ] ;
  delta_ok pdl2aref($outAroonDown), $outAroonDown_expected, 'ta_aroon:outAroonDown';
  my $outAroonUp_expected = [ 80, 70, 60, 50, 40, 30, 20, 10, 0, 20, 10 ] ;
  delta_ok pdl2aref($outAroonUp), $outAroonUp_expected, 'ta_aroon:outAroonUp';
}
{
  my $outReal = ta_aroonosc($H, $L, $optInTimePeriod);
  my $outReal_expected = [ 50, 50, 50, 50, 40, 30, 20, 10, -100, -80, -80 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_aroonosc:outReal';
}
{
  my $outReal = ta_bop($O, $H, $L, $C);
  my $outReal_expected = [ 0.136904761904755, 0.142857142857135, -0.130718954248359, 0.508547008547007, 0.322404371584699, -0.163934426229499, 0.366666666666668, 0.830917874396126, 0.233716475095785, 0.257575757575754, 0.4952380952381, 0.0414507772020789, -0.210526315789472, -0.100591715976324, -0.165562913907286, 0.350993377483447, -0.275862068965518, -0.67857142857143, 0.364161849710999, -0.394618834080717, 0.709543568464735 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_bop:outReal';
}
{
  my $outReal = ta_cci($H, $L, $C, $optInTimePeriod);
  my $outReal_expected = [ 154.297747046543, 110.083904133574, 86.1683475687239, 62.0346217690301, 45.2366686638706, 31.8312957266047, 47.3416771325048, 22.3083340060816, -120.42502951594, -260.002130946674, -185.998441259191, -83.6047774158522 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_cci:outReal';
}
{
  my $outReal = ta_cmo($R, $optInTimePeriod);
  my $outReal_expected = [ -59.6831806181084, -63.9894576224164, -39.329442628103, -40.0562669724564, -40.0669038564181, -37.1862679088161, -40.5053065533328, -46.465032832205, -28.4554814287301, -19.5399430263826, -6.51590497565388 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_cmo:outReal';
}
{
  my $outReal = ta_dx($H, $L, $C, $optInTimePeriod);
  my $outReal_expected = [ 73.1902626521459, 74.7480278007463, 67.97037040573, 63.1041803310983, 60.4641601373122, 63.0094940016137, 59.5086196343193, 45.8453164477071, 15.3038993314749, 7.82022181262844, 15.9480700533166 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_dx:outReal';
}
{
  my ($outMACD, $outMACDSignal, $outMACDHist) = ta_macd($R, $optInFastPeriod, $optInSlowPeriod, $optInSignalPeriod);
  my $outMACD_expected = [ 0.454518518518512, -15.3596625514403, -18.2306666666667, -19.4374083219021, -8.49614906264293, -4.6767249911091, -2.79535853274399, -1.04608631249749, -2.1582288607766, -5.21243190244383, -0.592825488937933, 2.82549271160514, 6.54056504248024 ] ;
  delta_ok pdl2aref($outMACD), $outMACD_expected, 'ta_macd:outMACD';
  my $outMACDSignal_expected = [ -4.5648, -8.16308751714677, -11.5189472336534, -14.1584342630696, -12.2710058629274, -9.73957890565463, -7.42483878135109, -5.29858795839989, -4.25180159252546, -4.57201169583158, -3.2456162935337, -1.22191329182075, 1.36557948627958 ] ;
  delta_ok pdl2aref($outMACDSignal), $outMACDSignal_expected, 'ta_macd:outMACDSignal';
  my $outMACDHist_expected = [ 5.01931851851851, -7.19657503429353, -6.71171943301326, -5.27897405883249, 3.77485680028447, 5.06285391454554, 4.6294802486071, 4.2525016459024, 2.09357273174886, -0.640420206612244, 2.65279080459577, 4.04740600342589, 5.17498555620066 ] ;
  delta_ok pdl2aref($outMACDHist), $outMACDHist_expected, 'ta_macd:outMACDHist';
}
{
  my ($outMACD, $outMACDSignal, $outMACDHist) = ta_macdext($R, $optInFastPeriod, $optInFastMAType, $optInSlowPeriod, $optInSlowMAType, $optInSignalPeriod, $optInSignalMAType);
  my $outMACD_expected = [ -1.14212315195857, 2.9546256160138, 2.2742085386713, 1.6725139573349, -1.25782868088834, -4.45505400474059, 0.555870887308686, 3.94862362910288, 7.04843009858985 ] ;
  delta_ok pdl2aref($outMACD), $outMACD_expected, 'ta_macdext:outMACD';
  my $outMACDSignal_expected = [ -8.57758908702943, -2.40666962466202, 0.553651231274974, 1.72624765621523, 0.613463079522396, -1.86526544119353, -0.470829684100682, 2.14786001687173, 5.2339216572668 ] ;
  delta_ok pdl2aref($outMACDSignal), $outMACDSignal_expected, 'ta_macdext:outMACDSignal';
  my $outMACDHist_expected = [ 7.43546593507086, 5.36129524067582, 1.72055730739633, -0.053733698880329, -1.87129176041073, -2.58978856354706, 1.02670057140937, 1.80076361223115, 1.81450844132304 ] ;
  delta_ok pdl2aref($outMACDHist), $outMACDHist_expected, 'ta_macdext:outMACDHist';
}
{
  my ($outMACD, $outMACDSignal, $outMACDHist) = ta_macdfix($C4, $optInSignalPeriod);
  my $outMACDSignal_expected = [ -0.199686999088539, 0.0476828940266993, 0.343804419650472, 0.639871089264245, 0.884444238369297, 1.06994939908844, 1.19462224757543, 1.3258610054873, 1.41048436329991, 1.43372007643704, 1.37332216979393, 1.23669268446074, 1.11472712103412, 0.785872458089467, 0.410773708469849, -0.00631602478944033, -0.41554653038608, -0.784622289011508, -1.07414321819066, -1.27463634736065, -1.36764535113442, -1.17586163006575, -0.852992542646905, -0.469890094298811, -0.0846225558841095, 0.246007366508261, 0.51153843782278, 0.708868020356136, 0.905000696152532, 1.04691679250477, 1.12031253457206, 1.10356110424255, 1.00473835204592, 0.915411071552579, 0.614662732630921, 0.263723468692463, -0.13262670108141, -0.524069827998604, -0.877899438518693, -1.15435709280441, -1.34365906732424, -1.42708068770423, -1.22708235060544, -0.897173029359936, -0.508034660778853, -0.117589916433876, 0.217483025110138, 0.486829357861891, 0.687437359375516, 0.886389248714964, 1.03073163828829, 1.10621734157839, 1.09126785097669, 0.994000264187091, 0.906016596781302 ] ;
  delta_ok pdl2aref($outMACDSignal), $outMACDSignal_expected, 'ta_macdfix:outMACDSignal';
  my $outMACDHist_expected = [ 0.286177029687491, 0.494739786230476, 0.592243051247546, 0.592133339227546, 0.489146298210103, 0.371010321438292, 0.249345696973971, 0.262477515823744, 0.169246715625224, 0.0464714262742632, -0.120795813286228, -0.27325897066638, -0.243931126853248, -0.657709325889296, -0.750197499239237, -0.834179466518579, -0.818461011193279, -0.738151517250856, -0.579041858358305, -0.400986258339976, -0.18601800754755, 0.383567442137345, 0.645738174837692, 0.766204896696187, 0.770535076829404, 0.661259844784742, 0.531062142629038, 0.394659165066712, 0.392265351592792, 0.283832192704481, 0.146791484134582, -0.0335028606590217, -0.197645504393266, -0.178654560986682, -0.601496677843317, -0.701878527876916, -0.792700339547746, -0.782886253834389, -0.707659221040177, -0.552915308571426, -0.37860394903967, -0.166843240759987, 0.399996674197591, 0.659818642491006, 0.778276737162165, 0.780889488689955, 0.670145883088028, 0.538692665503506, 0.40121600302725, 0.397903778678894, 0.288684779146658, 0.150971406580203, -0.0298989812034125, -0.194535173579195, -0.175967334811577 ] ;
  delta_ok pdl2aref($outMACDHist), $outMACDHist_expected, 'ta_macdfix:outMACDHist';
}
{
  my $outReal = ta_mfi($H, $L, $C, $V, $optInTimePeriod);
  my $outReal_expected = [ 75.2243213002614, 74.7917536498411, 75.2747454022305, 76.7416424101864, 70.4436944435767, 70.2552410459838, 62.6443789677251, 51.4452270435298, 28.111399111212, 29.3481106034256, 28.2094559327123 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_mfi:outReal';
}
{
  my $outReal = ta_minus_di($H, $L, $C, $optInTimePeriod);
  my $outReal_expected = [ 9.32278903987531, 8.63526247010009, 10.4662226822039, 11.5688864381736, 11.801258480106, 10.6660452838831, 11.0944184462794, 14.6399328296113, 25.0354250863579, 26.2721654666971, 23.6137761604961 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_minus_di:outReal';
}
{
  my $outReal = ta_minus_dm($H, $L, $optInTimePeriod);
  my $outReal_expected = [ 2.79000000000002, 2.51100000000002, 2.25990000000002, 2.68391000000002, 2.86551900000001, 2.8089671, 2.52807039, 2.54526335100001, 3.35073701590001, 5.96566331431, 6.24909698287899, 5.62418728459109 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_minus_dm:outReal';
}
{
  my $outReal = ta_mom($R, $optInTimePeriod);
  my $outReal_expected = [ -76.86, -83.17, -44.18, -41.3, -37.66, -51.2, -44.61, -60.51, -52.34, 3.25, 22.31 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_mom:outReal';
}
{
  my $outReal = ta_plus_di($H, $L, $C, $optInTimePeriod);
  my $outReal_expected = [ 60.2249944308309, 59.7575141571076, 54.8871567654075, 51.1422095179066, 47.8977817886375, 47.0030511288474, 43.7044960193913, 39.4271648590852, 34.0828221263183, 30.7298494726192, 32.5747638896253 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_plus_di:outReal';
}
{
  my $outReal = ta_plus_dm($H, $L, $optInTimePeriod);
  my $outReal_expected = [ 17.89, 16.221, 15.6389, 14.07501, 12.667509, 11.4007581, 11.14068229, 10.026614061, 9.0239526549, 8.12155738941, 7.309401650469, 7.75846148542211 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_plus_dm:outReal';
}
{
  my $outReal = ta_ppo($R, $optInFastPeriod, $optInSlowPeriod, $optInMAType);
  my $outReal_expected = [ -1.41935244446977, 0.330138769521288, 0.262204271894261, 0.0372907042634866, 0.499002391355723, -2.88488595443072, -4.42508398153808, -3.78858495612553, -0.784305110066346, 0.980448419407671, 0.572937681106266, 0.389820857605893, -0.215784379138111, -1.09268916803505, -0.154955941302159, 1.07641776820182, 1.75688810311461 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_ppo:outReal';
}
{
  my $outReal = ta_roc($R, $optInTimePeriod);
  my $outReal_expected = [ -14.5416706082679, -15.9635316698656, -8.78889154133843, -8.28817981135862, -7.61361798479702, -10.0331171248849, -8.94848752306827, -11.9926272395751, -10.2867475088933, 0.706521739130435, 4.93922823175187 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_roc:outReal';
}
{
  my $outReal = ta_rocp($R, $optInTimePeriod);
  my $outReal_expected = [ -0.145416706082679, -0.159635316698656, -0.0878889154133843, -0.0828817981135862, -0.0761361798479702, -0.100331171248849, -0.0894848752306827, -0.119926272395751, -0.102867475088933, 0.00706521739130435, 0.0493922823175187 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_rocp:outReal';
}
{
  my $outReal = ta_rocr($R, $optInTimePeriod);
  my $outReal_expected = [ 0.854583293917321, 0.840364683301344, 0.912111084586616, 0.917118201886414, 0.92386382015203, 0.899668828751151, 0.910515124769317, 0.880073727604249, 0.897132524911067, 1.0070652173913, 1.04939228231752 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_rocr:outReal';
}
{
  my $outReal = ta_rocr100($R, $optInTimePeriod);
  my $outReal_expected = [ 85.4583293917321, 84.0364683301344, 91.2111084586616, 91.7118201886414, 92.386382015203, 89.9668828751151, 91.0515124769317, 88.0073727604249, 89.7132524911067, 100.70652173913, 104.939228231752 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_rocr100:outReal';
}
{
  my $outReal = ta_rsi($R, $optInTimePeriod);
  my $outReal_expected = [ 20.1584096909458, 18.0052711887918, 30.3352786859485, 29.9718665137718, 29.9665480717909, 31.406866045592, 29.7473467233336, 26.7674835838975, 35.7722592856349, 40.2300284868087, 46.7420475121731 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_rsi:outReal';
}
{
  my ($outSlowK, $outSlowD) = ta_stoch($H, $L, $C, $optInFastK_Period, $optInSlowK_Period, $optInSlowK_MAType, $optInSlowD_Period, $optInSlowD_MAType);
  my $outSlowK_expected = [ 81.0437263865301, 76.9625411593667, 76.2688213239323, 75.608672844533, 45.2588818768119, 29.6358103651575, 16.5050217469957, 42.3015906280992, 36.1327447725332, 22.9959498510553, 24.1059196578248, 19.9989868154192, 33.6587994014004 ] ;
  delta_ok pdl2aref($outSlowK), $outSlowK_expected, 'ta_stoch:outSlowK';
  my $outSlowD_expected = [ 78.1714807405571, 80.8321687192035, 78.091696289943, 76.280011775944, 65.7121253484257, 50.1677883621675, 30.4665713296551, 29.4808075800841, 31.6464523825427, 33.8100950838959, 27.7448714271378, 22.3669521080998, 25.9212352915481 ] ;
  delta_ok pdl2aref($outSlowD), $outSlowD_expected, 'ta_stoch:outSlowD';
}
{
  my ($outFastK, $outFastD) = ta_stochf($H, $L, $C, $optInFastK_Period, $optInFastD_Period, $optInFastD_MAType);
  my $outFastK_expected = [ 88.5032537960955, 100, 77.5972141613464, 72.8813559322032, 75.5751014884979, 74.9485243651338, 14.9090909090908, 14.0127388535032, 3.37423312883391, 68.0981595092026, 29.9638989169671, 9.85915492957749, 25.2158894645943, 15.8920539730136, 47.3186119873817 ] ;
  delta_ok pdl2aref($outFastK), $outFastK_expected, 'ta_stochf:outFastK';
  my $outFastD_expected = [ 75.7772957339119, 91.2870571221981, 86.8824827133558, 78.9730826865504, 75.0142050818744, 74.8128553341001, 45.033237157673, 24.4674871239708, 8.84287800043314, 37.5092806064631, 38.2437081496901, 26.2672370219782, 20.8883128616508, 17.9945159629678, 33.1593055621277 ] ;
  delta_ok pdl2aref($outFastD), $outFastD_expected, 'ta_stochf:outFastD';
}
{
  my ($outFastK, $outFastD) = ta_stochrsi($R, $optInTimePeriod, $optInFastK_Period, $optInFastD_Period, $optInFastD_MAType);
  my $outFastK_expected = [ 0, 0, 100, 100, 100 ] ;
  delta_ok pdl2aref($outFastK), $outFastK_expected, 'ta_stochrsi:outFastK';
  my $outFastD_expected = [ 49.5015809814084, 16.6666666666667, 50, 83.3333333333333, 100 ] ;
  delta_ok pdl2aref($outFastD), $outFastD_expected, 'ta_stochrsi:outFastD';
}
{
  my $outReal = ta_trix($C4, $optInTimePeriod);
  my $outReal_expected = [ -0.207462548764239, -0.170851835128693, -0.118873721201738, -0.060547157360441, -0.00357372053386226, 0.0445530763115138, 0.0821636355564264, 0.108831383904315, 0.132870725094019, 0.14833084430827, 0.153817223698462, 0.147114977013525, 0.129624204391631, 0.11283321607358, 0.0708399917603024, 0.023050185464132, -0.0293748902323761, -0.0800017690520715, -0.124562591512822, -0.158077048309486, -0.179270331231725, -0.185974720887523, -0.156959492728344, -0.110708317396878, -0.0566376790593586, -0.00276296453303311, 0.0431719798243561, 0.0792926092452687, 0.105006430121922, 0.128489807237836, 0.143691200477238, 0.149134444612698, 0.142541391032865, 0.125261691565884, 0.108741689043013, 0.0670617372705987, 0.0195987681707987, -0.0325002233890247, -0.0828127462434946, -0.127077312292127, -0.160317327475316, -0.181259056887421, -0.187734905764583, -0.158515730242905, -0.1120807744302, -0.057844327899594, -0.00382016920266484, 0.0422491754409384, 0.078489974263581, 0.104310597768529, 0.127888019442653, 0.143172124542608, 0.148687840321916, 0.14215807977275, 0.124933399958612, 0.108460840783541 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_trix:outReal';
}
{
  my $outReal = ta_ultosc($H, $L, $C, $optInTimePeriod1, $optInTimePeriod2, $optInTimePeriod3);
  my $outReal_expected = [ 71.4670126365939, 69.8785619211737, 62.0799180653119, 46.8928212095913, 37.1341239426742, 25.5887973101314, 46.8219434330507, 43.0284627561856, 37.9574449773307, 31.142955115107, 36.7051921746329, 52.5700452307303 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_ultosc:outReal';
}
{
  my $outReal = ta_willr($H, $L, $C, $optInTimePeriod);
  my $outReal_expected = [ -22.8697086311161, -19.8460692688291, -20.065970313359, -25.7284222100055, -29.3673824724318, -35.9191655801827, -23.0040595399188, -32.8757721345231, -95.3871499176277, -78.078078078078, -85.9416445623342, -60.2122015915119 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_willr:outReal';
}
{
  my $outReal = ta_ht_dcperiod($C4);
  my $outReal_expected = [ 15.6466960154498, 17.033953957877, 18.2723103167042, 19.3462547446721, 20.2807218038997, 20.9313964758508, 21.2155477198228, 21.3298207147128, 21.4655750520829, 21.6101765487993, 21.6386040699251, 21.537640490767, 21.4583465683353, 21.5708832037787, 22.0282686187216, 22.7276333635028, 23.3111837908419, 23.9980327536738, 24.5029394351883, 24.4575151700803, 24.0864457628828, 23.6622483380614, 23.2349620695232, 22.7631574720633, 22.3503760197933, 22.0234334174619, 21.5578503006629, 20.9313669770894, 20.3383455064104, 19.9089648805549, 19.6249992265221, 19.3952294747975, 19.180344323665, 19.0252674124952, 18.9898266874118, 19.1331641120418, 19.4359497293984, 19.7609230797922, 20.19466988597, 20.6245420064013, 20.7954112285787, 20.774412853442, 20.6984782995913, 20.5845809400283, 20.4064641864729, 20.2647403330806, 20.1996890952193, 19.9951597499889, 19.6117374071078, 19.2331721444414, 18.9887410548474, 18.8604685535939 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_ht_dcperiod:outReal';
}
{
  my $outReal = ta_ht_dcphase($C4);
  my $outReal_expected = [ 204.489193756159, 223.782584377016, 233.63758585607, 250.385254860004, 251.770919485859, 259.514958360179, 266.789484603829, 273.379974079016, 279.634140618924, 299.386258228814, -33.5310310778445, 1.34163127546753, 24.7403144674207, 52.9589902815417, 73.8515379709615, 95.1406871657446, 118.423519134561, 134.909647060545, 162.052064393335, 189.641416506918, 207.178862460945 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_ht_dcphase:outReal';
}
{
  my ($outInPhase, $outQuadrature) = ta_ht_phasor($C4);
  my $outInPhase_expected = [ 1.46403485840663, 2.96551329453906, 6.98868766554089, 9.42381685623212, 7.46052655953471, 5.29333252946677, 1.66346162992684, -0.477873184322623, -1.13084675943775, -0.226754219914489, 0.192133690393332, -1.21738966251765, -2.41495372784774, -4.43387546824602, -3.49038366986958, -6.18748181552622, -8.28170657191444, -6.24486694932367, -5.65902046947712, -2.7780152159249, -0.224595014272679, 2.40262914995973, 4.59408417356308, 9.79625512704051, 11.9638167930026, 8.69697678077041, 5.78585437033688, 1.72401114589291, -0.477633924325097, -1.09934668406548, -0.214128740554758, 0.177997954081243, -1.1146420218414, -2.18932215466181, -3.9988788425941, -3.15608635184809, -5.62612980091531, -7.52990730889713, -5.62435148812669, -4.99609693956127, -2.41495220848728, -0.195718125197405, 2.08437478979468, 4.02828331957148, 8.79601201405254, 10.9445522197451, 8.04278562740512, 5.39172733911624, 1.61937560326374, -0.452093845337262, -1.04961272744669, -0.206178206942518 ] ;
  delta_ok pdl2aref($outInPhase), $outInPhase_expected, 'ta_ht_phasor:outInPhase';
  my $outQuadrature_expected = [ 5.40336023641963, 7.82776527506655, 8.61781985175749, 0.611622337512332, -5.84759770330489, -9.08974898885992, -9.40518801927729, -5.02567472323596, -1.03857732013208, 0.805036736864015, -2.07815045950858, -3.75149988650516, -5.19796329473302, -3.06830809902378, -3.25220390072599, -6.9576341726626, 0.296677042995411, 4.34360708862944, 6.90895639043035, 10.7358759644087, 10.7287335980849, 10.2481467676987, 12.0812756781249, 10.7103709821664, -1.52159386001755, -8.72559572245702, -10.8616471814661, -9.98108107663345, -4.93203520456349, -1.01086627754763, 0.709265012235521, -1.68958049641003, -3.06404789725122, -4.25070901409302, -2.52476275438513, -2.68869471602838, -5.57763786489681, 0.313518547172761, 3.61805702897433, 5.4993768895031, 8.35357925485057, 8.46635074158659, 8.31735126425758, 10.1502091371145, 9.35609532506087, -0.979320755415299, -7.40798143336436, -9.51533639204499, -8.92574315948693, -4.48460621850696, -0.926459250657231, 0.660989018561203 ] ;
  delta_ok pdl2aref($outQuadrature), $outQuadrature_expected, 'ta_ht_phasor:outQuadrature';
}
{
  my ($outSine, $outLeadSine) = ta_ht_sine($C4);
  my $outSine_expected = [ -0.414521612459407, -0.691923755687888, -0.805282905116331, -0.941971092714203, -0.949813403358635, -0.983302450763599, -0.998430502541886, -0.998260495609668, -0.985896490173352, -0.871331524001826, -0.552388532298845, 0.0234137433278741, 0.418506215008153, 0.798204553633092, 0.960544251681702, 0.995977688212469, 0.87945326152357, 0.708220977992021, 0.30815264652852, -0.167481434668184, -0.456769772781069 ] ;
  delta_ok pdl2aref($outSine), $outSine_expected, 'ta_ht_sine:outSine';
  my $outLeadSine_expected = [ -0.936606121889064, -0.999774271632086, -0.988658073579491, -0.903445649937437, -0.89281454669418, -0.823978287368697, -0.745598314688763, -0.664187541178422, -0.57879536242323, -0.269150816972036, 0.198837183393694, 0.723468952133745, 0.9381328133561, 0.990367428862536, 0.875872984648103, 0.640904686722947, 0.285294965025282, 0.00157695562875339, -0.454799965442866, -0.815546318682707, -0.952016550945869 ] ;
  delta_ok pdl2aref($outLeadSine), $outLeadSine_expected, 'ta_ht_sine:outLeadSine';
}
{
  my $outInteger = ta_ht_trendmode($C4);
  my $outInteger_expected = [ 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0 ] ;
  delta_ok pdl2aref($outInteger), $outInteger_expected, 'ta_ht_trendmode:outInteger';
}
{
  my $outReal = ta_ad($H, $L, $C, $V);
  my $outReal_expected = [ 3264471.42857138, 5112280.95238085, 5684935.85434166, 9036879.44408529, 10089275.6189487, 8816619.88124374, 9348573.21457702, 16520873.214577, 10519675.8965693, 12091918.3208116, 14107258.3208116, 13156271.2741795, 10439119.5995384, 8611692.38060356, 5970695.02961013, 8582429.46669627, 6361265.32876522, 3633590.32876522, 6126537.14957454, 5974771.67872254, 7883629.3550711 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_ad:outReal';
}
{
  my $outReal = ta_adosc($H, $L, $C, $V, $optInFastPeriod, $optInSlowPeriod);
  my $outReal_expected = [ 1958766.28941616, 1137378.5489301, 879414.929059824, 3017430.80707536, 821606.16128171, 675146.790012412, 1164347.64306091, 697319.340847455, -467141.804713107, -1231244.06454639, -2007767.22494128, -863579.287124484, -1157796.8386701, -1875115.33042022, -786844.870445978, -420741.06508104, 390399.242689807 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_adosc:outReal';
}
{
  my $outReal = ta_obv($R, $V);
  my $outReal_expected = [ 3608100, -272300, -4444500, -8616600, -11579500, -7697900, -12257500, -5085200, 7445300, 2998100, -360800, -3184500, 432600, -2565800, -5657200, -2286500, -5475300, -9112200, -5488000, -2411300, 482000 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_obv:outReal';
}
{
  my $outInteger = ta_cdl2crows($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdl3blackcrows($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdl3inside($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdl3linestrike($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdl3outside($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdl3starsinsouth($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdl3whitesoldiers($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlabandonedbaby($O, $H, $L, $C, $optInPenetration);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdladvanceblock($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlbelthold($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlbreakaway($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlclosingmarubozu($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlconcealbabyswall($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlcounterattack($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdldarkcloudcover($O, $H, $L, $C, $optInPenetration);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdldoji($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 100, 0, 100, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdldojistar($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdldragonflydoji($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlengulfing($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdleveningdojistar($O, $H, $L, $C, $optInPenetration);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdleveningstar($O, $H, $L, $C, $optInPenetration);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlgapsidesidewhite($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlgravestonedoji($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlhammer($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlhangingman($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, -100, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlharami($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlharamicross($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlhighwave($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlhikkake($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlhikkakemod($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlhomingpigeon($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlidentical3crows($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlinneck($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlinvertedhammer($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 100, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlkicking($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlkickingbylength($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlladderbottom($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdllongleggeddoji($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 100, 0, 100, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdllongline($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, -100, 0, 0, 100 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlmarubozu($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlmatchinglow($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlmathold($O, $H, $L, $C, $optInPenetration);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlmorningdojistar($O, $H, $L, $C, $optInPenetration);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlmorningstar($O, $H, $L, $C, $optInPenetration);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlonneck($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlpiercing($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlrickshawman($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 100, 0, 100, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlrisefall3methods($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlseparatinglines($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlshootingstar($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlshortline($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlspinningtop($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 100, 0, -100, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlstalledpattern($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlsticksandwich($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdltakuri($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdltasukigap($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlthrusting($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdltristar($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlunique3river($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlupsidegap2crows($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;
}
{
  my $outInteger = ta_cdlxsidegap3methods($O, $H, $L, $C);
  my $outInteger_expected = [ 'BAD', 'BAD', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ] ;
  is_deeply [$outInteger->list], $outInteger_expected;}
{
  my $outReal = ta_beta($R0, $R1, $optInTimePeriod);
  my $outReal_expected = [ 0.823393259155231, 0.832452072832542, 0.78526807746517, 0.802936945206642, 0.784701325782935, 0.798223028667828, 0.796773834663546, 0.794365265306995, 0.783705690422555, 0.492264254990801, 0.455290504404646 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_beta:outReal';
}
{
  my $outReal = ta_correl($R0, $R1, $optInTimePeriod);
  my $outReal_expected = [ 0.981821862722751, 0.991167146552979, 0.992447233915642, 0.987896718001133, 0.986516769103234, 0.983972010679007, 0.981122371002217, 0.979013447121549, 0.966431155410867, 0.744087366256691, 0.75077582799486, 0.873978317377307 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_correl:outReal';
}
{
  my $outReal = ta_linearreg($R, $optInTimePeriod);
  my $outReal_expected = [ 484.285636363636, 471.813818181818, 457.236363636364, 450.672, 445.123272727273, 441.088363636364, 442.977454545455, 443.504909090909, 444.267454545455, 453.471090909091, 458.109818181818, 464.622909090909 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_linearreg:outReal';
}
{
  my $outReal = ta_linearreg_angle($R, $optInTimePeriod);
  my $outReal_expected = [ -76.2940137935796, -79.0400411222334, -81.3261415428359, -81.9064033940186, -82.2493869776483, -82.3111654806503, -80.2987206949964, -78.0891954554062, -72.7814925202904, -1.05551191031817, 43.2340603455907, 62.1386686438628 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_linearreg_angle:outReal';
}
{
  my $outReal = ta_linearreg_intercept($R, $optInTimePeriod);
  my $outReal_expected = [ 521.188363636364, 518.288181818182, 516.231636363636, 513.96, 511.248727272727, 507.751636363636, 495.622545454545, 486.173090909091, 473.308545454545, 453.636909090909, 449.648181818182, 447.597090909091 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_linearreg_intercept:outReal';
}
{
  my $outReal = ta_linearreg_slope($R, $optInTimePeriod);
  my $outReal_expected = [ -4.10030303030303, -5.16381818181818, -6.5550303030303, -7.032, -7.34727272727273, -7.4070303030303, -5.84945454545454, -4.74090909090909, -3.22678787878788, -0.0184242424242388, 0.940181818181821, 1.89175757575758 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_linearreg_slope:outReal';
}
{
  my $outReal = ta_stddev($R, $optInTimePeriod, $optInNbDev);
  my $outReal_expected = [ 17.3979568053263, 20.9180025098, 25.0694946897619, 25.7536254535163, 26.1674126348021, 26.2382689977826, 23.597939740579, 21.8006102896226, 18.0026241420522, 6.85689317402568, 7.22201557738558, 9.33736793748645 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_stddev:outReal';
}
{
  my $outReal = ta_tsf($R, $optInTimePeriod);
  my $outReal_expected = [ 480.185333333333, 466.65, 450.681333333333, 443.64, 437.776, 433.681333333333, 437.128, 438.764, 441.040666666667, 453.452666666667, 459.05, 466.514666666667 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_tsf:outReal';
}
{
  my $outReal = ta_var($R, $optInTimePeriod, $optInNbDev);
  my $outReal_expected = [ 302.688901, 437.562829, 628.479564, 663.249224, 684.733484, 688.44676, 556.86276, 475.266609, 324.094476, 47.016984, 52.157509, 87.18644 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_var:outReal';
}
{
  my $outReal = ta_avgprice($O, $H, $L, $C);
  my $outReal_expected = [ 192.4425, 194.125, 192.6175, 191.7325, 192.28, 193.8, 194.325, 195.1325, 205.04, 204.1175, 204.6825, 205.0725, 204.4025, 203.82, 203.455, 204.755, 204.1425, 203.59, 200.565, 199.9875, 200.855 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_avgprice:outReal';
}
{
  my $outReal = ta_medprice($H, $L);
  my $outReal_expected = [ 192.12, 193.95, 192.515, 191.56, 192.265, 193.85, 194.4, 195.045, 205.97, 204.07, 204.655, 205.255, 204.685, 204.035, 203.715, 204.595, 204.295, 203.63, 200.425, 199.795, 200.885 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_medprice:outReal';
}
{
  my $outReal = ta_typprice($H, $L, $C);
  my $outReal_expected = [ 192.373333333333, 194.116666666667, 192.55, 191.873333333333, 192.373333333333, 193.783333333333, 194.423333333333, 195.39, 205.553333333333, 204.186666666667, 204.76, 205.146666666667, 204.423333333333, 203.863333333333, 203.5, 204.79, 204.126666666667, 203.35, 200.623333333333, 199.776666666667, 201.15 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_typprice:outReal';
}
{
  my $outReal = ta_wclprice($H, $L, $C);
  my $outReal_expected = [ 192.5, 194.2, 192.5675, 192.03, 192.4275, 193.75, 194.435, 195.5625, 205.345, 204.245, 204.8125, 205.0925, 204.2925, 203.7775, 203.3925, 204.8875, 204.0425, 203.21, 200.7225, 199.7675, 201.2825 ] ;
  delta_ok pdl2aref($outReal), $outReal_expected, 'ta_wclprice:outReal';
}

done_testing;