package CPAN::Patches;

=head1 NAME

CPAN::Patches - patch CPAN distributions

=head1 SYNOPSIS

    cd Some-Distribution
    cpan-patches list
    cpan-patches patch
    cpan-patches --patch-set $HOME/cpan-patches-set list
    cpan-patches --patch-set $HOME/cpan-patches-set patch

=head1 DESCRIPTION

This module allows to apply custom patches to the CPAN distributions.

See L</patch> and L</update_debian> for a detail description how.

See L<http://github.com/jozef/CPAN-Patches-Example-Set> for example generated
Debian patches set folder.

=cut

use warnings;
use strict;

our $VERSION = '0.02';

use Moose;
use CPAN::Patches::SPc;
use Carp;
use IO::Any;
use JSON::Util;
use YAML::Syck;
use File::chdir;
use Scalar::Util 'blessed';
use Module::Pluggable require => 1;

=head1 PROPERTIES

=head2 patch_set_location

A folder where are the distribution patches located. Default is
F<< Sys::Path->sharedstatedir/cpan-patches/set >> which is
F</var/lib/cpan-patches/set> on Linux.

=head2 verbose

Turns on/off some verbose output. By default it is on.

=cut

has 'patch_set_location' => (
    is      => 'rw',
    isa     => 'Str',
    lazy    => 1,
    default => sub { File::Spec->catdir(CPAN::Patches::SPc->sharedstatedir, 'cpan-patches', 'set') }
);
has 'verbose' => ( is => 'rw', isa => 'Int', default => 1 );

=head1 METHODS

=head2 new()

Object constructor.

=head2 BUILD

All plugins (Moose roles) from C<CPAN::Patches::Plugin::*> will be loaded.

=cut

sub BUILD {
	my $self = shift;
	
	foreach my $plugin ($self->plugins) {
		$plugin->meta->apply($self);
	}
};

=head2 patch

Apply all patches that are listed in F<.../module-name/patches/series>.

=cut

sub patch {
    my $self = shift;
    my $path = shift || '.';
    
    $self = $self->new()
        if not blessed $self;
    
    local $CWD = $path;
	my $name = $self->clean_meta_name();
 
    foreach my $patch_filename ($self->get_patch_series) {
        print 'patching ', $name,' with ', $patch_filename, "\n"
            if $self->verbose;
        system('cat '.$patch_filename.' | patch -p1');
    }
    
    return;
}

=head1 cpan-patch CMD

=head2 cmd_list

Print out list of all patches files.

=cut

sub cmd_list {
    my $self = shift;
	foreach my $patch_filename ($self->get_patch_series) {
		print $patch_filename, "\n";
	}
}

=head2 cmd_patch

Apply all patches to the current CPAN distribution.

=cut

sub cmd_patch {
	shift->patch();
}


=head1 INTERNAL METHODS

=head2 get_patch_series($module_name)

Return an array of patches filenames for given C<$module_name>.

=cut

sub get_patch_series {
    my $self = shift;
    my $name = shift || $self->clean_meta_name;
    
    my $patches_folder  = File::Spec->catdir($self->patch_set_location, $name, 'patches');
    my $series_filename = File::Spec->catfile($patches_folder, 'series');

    return if not -r $series_filename;
    
    return
        map  { File::Spec->catfile($patches_folder, $_) }
        map  { s/^\s*//;$_; }
        map  { s/\s*$//;$_; }
        map  { split "\n" }
        eval { IO::Any->slurp([$series_filename]) };
}

=head2 clean_meta_name($name)

Returns lowercased :: by - substituted and trimmed module name.

=cut

sub clean_meta_name {
    my $self = shift;
    my $name = shift || $self->read_meta->{'name'};
    
    $name =~ s/::/-/xmsg;
    $name =~ s/\s*$//;
    $name =~ s/^\s*//;
    $name = lc $name;

    return $name;    
}

=head2 read_meta([$path])

Reads a F<META.yml> or F<META.json> from C<$path>. If C<$path> is not provided
than tries to read from current folder.

=cut

sub read_meta {
    my $self = shift;
    my $path = shift || '.';
    
    my $yml  = File::Spec->catfile($path, 'META.yml');
    my $json = File::Spec->catfile($path, 'META.json');
    if (-f $json) {
        my $meta = eval { JSON::Util->decode([$json]) };
        return $meta
            if $meta;
    }
    if (-f $yml) {
        my $meta = eval { YAML::Syck::LoadFile($yml) };
        return $meta
            if $meta;
    }
    croak 'failed to read meta file';
}

__PACKAGE__->meta->make_immutable;

1;


__END__

=head1 AUTHOR

jozef@kutej.net, C<< <jkutej at cpan.org> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-cpan-patches at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=CPAN-Patches>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.




=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc CPAN::Patches


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=CPAN-Patches>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/CPAN-Patches>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/CPAN-Patches>

=item * Search CPAN

L<http://search.cpan.org/dist/CPAN-Patches/>

=back


=head1 ACKNOWLEDGEMENTS


=head1 LICENSE AND COPYRIGHT

This program is free software; you can redistribute it and/or modify it
under the terms of either: the GNU General Public License as published
by the Free Software Foundation; or the Artistic License.

See http://dev.perl.org/licenses/ for more information.


=cut

1; # End of CPAN::Patches
