#include <stdlib.h>
#include <time.h>

#include <arch/neuron.h>
#include <nets/backprop.h>
#include <arch/layer.h>
#include <arch/hidden_layers.h>
#include <utils/transfer.h>

backprop::backprop(real _alpha, int inputs, int outputs, 
                                            int num_hidden_layers, ...) {
    time_t t;
    srand((rand())*(unsigned)time(&t));

     input = new layer("input",  inputs, 1.0);
    output = new layer("output", outputs);
    hidden = new hidden_layers(num_hidden_layers, 
                           (int *)((&num_hidden_layers)+1), SIGMOID, 1.0);

     input->set_transfer_function(SUM);  // input units broadcast ...
    hidden->set_transfer_function(SIGMOID);
    output->set_transfer_function(SIGMOID);

    output->dendrites_touch(hidden->query_head_layer());
    hidden->query_tail_layer()->dendrites_touch(input);

    output->reinitialize_weights_with(0.1, -0.1);
     input->reinitialize_weights_with(0.1, -0.1);
    hidden->reinitialize_weights_with(0.1, -0.1);

    alpha = _alpha;
    num_outputs = outputs;
    reset_nmse();
}

void backprop::set_transfer_function_for_output(int type) {
    output->set_transfer_function(type);
}

void backprop::set_transfer_function_for_hidden(int type) {
    hidden->set_transfer_function(type);
}

void backprop::reset_nmse() {
    error         = 0.0;
    error_counter = 0;
}

real backprop::query_nmse() {
    return (error/((real)num_outputs))/error_counter;
}

void backprop::set_input(real *inputs) {
    input->set_input(inputs);
}

real *backprop::query_output() {
#if DEBUG || PHASE_DEBUG
    fprintf(stderr, "\n==---..______..-----........_______...-+= Feed Forward\n");
#endif
    return output->query_output();
}

void backprop::perform_weight_change(neuron *n) {
    n->dendrites->start_foreach();
    while(n->dendrites->has_a_weight()) {
#if (DEBUG || WEIGHT_DEBUG)
        fprintf(stderr, "DeltaW = %f*%f*%f = %f for [%s].\n", 
            alpha,
            n->query_delta(),
            n->dendrites->query_current_fireing_output(),
            alpha * n->query_delta() 
                  * n->dendrites->query_current_fireing_output(),
            n->dendrites->query_current_name()
        );
#endif
        n->dendrites->change_current_weight_by(
            alpha * n->query_delta() 
                  * n->dendrites->query_current_fireing_output()
        );
        n->dendrites->next_weight();
    }
}

void backprop::perform_weight_change(layer *l) {
    l->start_foreach();
    while(l->has_a_neuron()) {
        perform_weight_change(l->query_current_neuron());
        l->next_neuron();
    }
}

void backprop::change_weights() {
    perform_weight_change(output);

    hidden->start_foreach();
    while(hidden->has_a_layer()) {
        perform_weight_change(hidden->query_current_layer());
        hidden->next_layer();
    }
}

void backprop::backpropagate_for(layer *l) {
    real *delta_in = l->weighted_sums_of_deltas_from_above();
    int counter  = 0;

    l->start_foreach();
    while(l->has_a_neuron()) {
#if (DEBUG || ERROR_DEBUG)
        fprintf(stderr, "deltaj = %1.4f * %1.4f = %1.4f [%s].\n",
            delta_in[counter],
            l->query_current_neuron()->query_output_dot(),
            delta_in[counter] * l->query_current_neuron()->query_output_dot(),
            l->query_current_neuron()->query_name()
        );
#endif
        l->query_current_neuron()->set_delta(
            delta_in[counter] * l->query_current_neuron()->query_output_dot()
        );
        l->next_neuron();
        counter++;
    }
}

void backprop::test_for(real *targets) {
#if DEBUG || PHASE_DEBUG
    fprintf(stderr, "\n==---..______..-----........_______...-+= Backpropagate \n");
#endif
    backpropagate_based_on(targets);
}

void backprop::train_on(real *targets) {
    test_for(targets);
#if DEBUG || PHASE_DEBUG
    fprintf(stderr, "\n==---..______..-----........_______...-+= Change Weights\n");
#endif
    change_weights();
}

void backprop::backpropagate_based_on(real *targets) {
    real difference;
    real *outputs = output->query_output();
    int counter = 0;

    output->start_foreach();
    while(output->has_a_neuron()) {
        difference = (targets[counter] - outputs[counter]);
#if (DEBUG || ERROR_DEBUG)
        fprintf(stderr, "deltak = (%1.4f-%1.4f=%1.4f) * %1.4f = %1.4f [%s].\n",
            targets[counter],
            outputs[counter],
            difference,
            output->query_current_neuron()->query_output_dot(),
            difference * output->query_current_neuron()->query_output_dot(),
            output->query_current_neuron()->query_name()
        );
#endif
        output->query_current_neuron()->set_delta(
            difference * output->query_current_neuron()->query_output_dot()
        );
        error += (difference*difference);
        error_counter++;
        output->next_neuron();
        counter++;
    }

    hidden->start_foreach();
    while(hidden->has_a_layer()) {
        backpropagate_for(hidden->query_current_layer());
        hidden->next_layer();
    }
}

void backprop::save_net(char *filename) {
    FILE *F = fopen(filename, "w");

    hidden->save_weights(F);
    output->save_weights(F);

    fclose(F);
}

void backprop::restore_net(char *filename) {
    FILE *F = fopen(filename, "r");

    hidden->restore_weights(F);
    output->restore_weights(F);

    fclose(F);
}
