package Dataninja::Config;
use Moose;
use YAML::XS 'LoadFile';
use Hash::Merge qw/merge/;

=head1 DESCRIPTION

Dataninja::Config - Dataninja configuration through YAML files

This is the class you use to grab information from config files, such as
the channels that dataninja connects to, Dataninja's nick, database
information, that list goes on.

There are three attributes: there are three configurations:  default,
site, and secret. In the end, the configurations are merged (secret having the
most precendence, and default having the least).

=head1 ATTRIBUTES

=head2 default_config

(Str) File for the default configuration for Dataninja (for things such
as database name and host). If no file is specified, it defaults
to etc/config.yml.

=head2 site_config

(Str) File for site-specific information. This includes things like IRC
networks, channels, command prefixes, and any  custom data you like.
The file defaults to etc/site_config.yml. See etc/example_site_config.yml
for details.

=head2 secret_config

(Str) File for storing secret information such as passwords. This file defaults
to etc/secret_config.

=head2 main

(HashRef) The object that has all the configuation information for the main 
configuration, such as database information.

=head2 site

(HashRef) The object that has all the site-specific configuation information,
such as external applicaiton credentials, channels, the nick of the bot, etc.

=cut

has default_config => (
    is  => 'rw',
    isa => 'Str',
    default => "$ENV{HOME}/.dataninja/config.yml",
);

has site_config => (
    is  => 'rw',
    isa => 'Str',
    default =>  "$ENV{HOME}/.dataninja/site_config.yml",
);

has secret_config => (
    is  => 'rw',
    isa => 'Str',
    default => "$ENV{HOME}/.dataninja/secret_config.yml",
);

has main => (
    is  => 'rw',
    isa => 'HashRef',
    default => sub { +{} },
);

has site => (
    is  => 'rw',
    isa => 'HashRef',
    default => sub { +{} },
);

sub BUILD {
    my $self = shift;
    my $default_config = LoadFile($self->default_config);
    my $site_config    = LoadFile($self->site_config);
    my $secret_config  = LoadFile($self->secret_config);

    Hash::Merge::set_behavior('RIGHT_PRECEDENT');

    my $result = merge($default_config, $site_config);
    $result    = merge($result        , $secret_config);

    $self->main($result->{Main});
    $self->site($result->{Site});
}

__PACKAGE__->meta->make_immutable;
no Moose;

1;
