package Kwiki::PoweredBy;
use Kwiki::Plugin -Base;
use mixin 'Kwiki::Installer';
our $VERSION = '0.10';

const class_id => 'powered_by';
const css_file => 'powered_by.css';

sub register {
    my $registry = shift;
    $registry->add(preload => $self->class_id);
    $registry->add(widget => 'powered_by', 
                   template => 'powered_by.html',
                  );
}

__DATA__

=head1 NAME 

Kwiki::PoweredBy - Kwiki Powered-By Plugin

=head1 SYNOPSIS

    > kwiki -add Kwiki::PoweredBy

=head1 DESCRIPTION

Provides a "Powered by Kwiki" logo for your Kwiki site.

=head1 AUTHOR

Brian Ingerson <ingy@cpan.org>

=head1 COPYRIGHT

Copyright (c) 2005. Brian Ingerson. All rights reserved.

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.

See http://www.perl.com/perl/misc/Artistic.html

=cut
__css/powered_by.css__
#powered_by img
{
  width: 100%;
  margin-left: auto;
  margin-right:auto;
  border: none;
}

#powered_by 
{
  width: 120px;
  margin-left: auto;
  margin-right:auto; 
}
__template/tt2/powered_by.html__
<div id="powered_by">
<a href="http://www.kwiki.org"><img src="plugin/powered_by/powered_by_kwiki.png" title="Powered By Kwiki" /></a>
</div>
__plugin/powered_by/powered_by_kwiki.png__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==
