package Prty::ContentProcessor;
use base qw/Prty::Hash/;

use strict;
use warnings;

our $VERSION = 1.098;

use Prty::Option;
use Prty::Path;
use Time::HiRes ();
use Prty::Perl;
use Prty::Hash;
use Prty::DestinationTree;
use Prty::Section::Parser;
use Prty::Section::Object;
use Prty::Terminal;
use Prty::Formatter;
use Prty::PersistentHash;

# -----------------------------------------------------------------------------

=encoding utf8

=head1 NAME

Prty::ContentProcessor - Prozessor für Abschnitts-Dateien

=head1 BASE CLASS

L<Prty::Hash>

=head1 SYNOPSIS

    use Prty::ContentProcessor;
    
    $cop = Prty::ContentProcessor->new('.mytool');
    $cop->registerType('MyTool::Program::Shell','.mprg','[Program]',Language=>'Shell');
    ...
    $cop->load(@paths)->generate($dir)->test;

=head1 DESCRIPTION

Ein Objekt der Klasse repräsentiert einen Prozessor für
Entitäts-Dateien. Die Dateien bestehen aus einer Folge von
Abschnitten, die von einem Abschnitts-Parser (s. Klasse
Prty::Section::Parser) geparsed und zu Abschnitts-Objekten
instantiiert werden.

Der Prozessor delegiert die Verarbeitung der Abschnitts-Objekte an
die per registerType() registierten Entitäts-Klassen
(Plugin-Schnittstelle). Diese bauen aus den Abschnitts-Objekten
Entitäts-Strukturen auf, aus denen die Ausgabedateien generiert
werden.

=head2 Ausgaben

Der Umfang an Ausgaben wird mit der Konstruktor-Option
-verbosity=>$level eingestellt. Default ist 1.

Die Methode msg() schreibt eine Ausgabe nach STDERR. Der erste
Parameter gibt den Verbosity-Level an. Ist dieser größer als der
eingestellte Verbosity-Level, unterbleibt die Ausgabe. Ist als
Level 0 angegeben, erfolgt die Ausgabe I<immer>:

    $cop->msg(0,...);

=head1 EXAMPLES

Alle Entitäts-Definitionen im Storage in eine einzige Datei $file
schreiben (z.B. für Refactoring):

    $cop->load->join($file);

Alle Entitäts-Definitionen im Storage in eine Verzeichnisstruktur
mit Wurzelverzeichnis $dir übertragen (z.B. um die
Dateien einzeln zu editieren):

    $cop->fetch->($dir);

Alle Entitäten im Storage vom Typ $type auflisten:

    $cop->load;
    for my $ent ($cop->entities($type)) {
        $cop->msg(0,$ent->name);
    }

=head1 METHODS

=head2 Konstruktor

=head3 new() - Instantiiere ContentProcessor

=head4 Synopsis

    $cop = $class->new($storageDir,@opt);

=head4 Description

Instantiiere ein ContentProcessor-Objekt und liefere eine Referenz
auf dieses Objekt zurück.

=head4 Arguments

=over 4

=item $storage

Der Name des Storage-Verzeichnisses ohne Verzeichnisanteil,
z.B. '.storage'.

=back

=head4 Options

=over 4

=item -extensions => \@extensions Liste der Datei-Erweiterungen der

Dateien, die der ContentProcessor verarbeitet. Hinzu kommen die
Erweiterungen, die die per registerType() hinzugefügten
Entitäts-Typen definieren. Die Extension wird ohne Punkt
angegeben.

=item -init => $bool (Default: 0)

Erzeuge und initialisiere Storage.

=item -verbosity => 0|1|2 (Default: 1)

Umfang der Laufzeit-Meldungen.

=back

=head4 Returns

ContentProcessor-Objekt

=cut

# -----------------------------------------------------------------------------

sub new {
    my $class = shift;
    my $storage = shift;

    # Optionen

    my $extensions = [];
    my $init = 0;
    my $verbosity = 1;

    Prty::Option->extract(\@_,
        -extensions=>\$extensions,
        -init=>\$init,
        -verbosity=>\$verbosity,
    );

    # Prüfen, ob Storage-Verzeichnis existert

    if (!-d $storage) {
        if ($init) {
            # Storage-Verzeichnis erzeugen
    
            Prty::Path->mkdir($storage);
            Prty::Path->mkdir("$storage/db");
            Prty::Path->mkdir("$storage/def");
            Prty::Path->mkdir("$storage/pure");
        }
        else {
            die "ERROR: Directory $storage does not exist\n";
        }
    }
    
    # Objekt instantiieren

    my $self = $class->SUPER::new(
        t0=>scalar Time::HiRes::gettimeofday,
        verbosity=>$verbosity,
        storage=>$storage,
        baseExtensionA=>$extensions,
        pluginA=>[],
        fileA=>[],
        parsedSections=>0,
        parsedLines=>0,
        parsedChars=>0,
        parsedBytes=>0,
        # memoize
        entityA=>undef,
        entityTypeA=>undef,
        extensionRegex=>undef,
        stateDb=>undef,
        typeH=>undef,
    );

    return $self;
}

# -----------------------------------------------------------------------------

=head2 Plugins

=head3 registerType() - Registriere Entitäts-Typ

=head4 Synopsis

    $cop->registerType($pluginClass,$extension,$sectionType,@keyVal);
    $cop->registerType($pluginClass,$extension); # universelles Plugin

=head4 Description

Registriere Plugin-Klasse $pluginClass für Abschnitts-Objekte mit
Identifier $sectionType und den Eigenschaften @keyVal. Die
Plugin-Klasse wird automatisch geladen, falls sie noch nicht
vorhanden ist (sie kann für Tests also auch "inline" definiert
werden).

Die Plugin-Definition wird intern auf einem Hash-Objekt
gespeichert, das vom ContentProcessor mit den instantiierten
Entitten verknüpft wird.

Ein I<Universelles Plugin> kann definiert werden, indem nur
$pluginClass und $extension angegeben werden. An diese
Plugin-Klasse werden alle (Haupt-)Abschnitts-Objekte delegiert, für
die kein Plugin definiert ist. Logischerweise kann es
höchstens ein Universelles Plugin geben. Für ein Universelles
Plugin findet keine Attribut-Validierung in der der
Basisklassenmethode create() statt.

=head4 Arguments

=over 4

=item $pluginClass

Name der Plugin-Klasse, z.B. 'Program::Shell'.

=item $extension

Datei-Erweiterung für Dateien dieses Typs, ohne Punkt, z.B. 'prg'.

=item $sectionType

Abschnitts-Bezeichner ohne Klammerung, z.B. 'Program'.

=item @keyVal

Abschnitts-Attribute, die über den Abschnitts-Bezeichner hinaus
den Dateityp kennzeichnen, z.B. Language=>'Shell'.

=back

=head4 Returns

nichts

=cut

# -----------------------------------------------------------------------------

sub registerType {
    my ($self,$pluginClass) = splice @_,0,2;
    # @_: $extension,$sectionType,@keyVal

    Prty::Perl->loadClass($pluginClass);
    push @{$self->{'pluginA'}},Prty::Hash->new(
        class=>$pluginClass,
        extension=>shift,
        sectionType=>shift,
        keyValA=>\@_,
    );
        
    return;
}

# -----------------------------------------------------------------------------

=head2 Operationen

=head3 commit() - Übertrage Veränderungen in den Storage

=head4 Synopsis

    $cop = $cop->commit;

=head4 Description

Vergleiche die Entitäten gegen den Storage und übertrage die
Veränderungen dorthin. Geänderte Entitäten werden in der in der
Datenbank als geändert gekennzeichnet.

=head4 Returns

ContentProcessor-Objekt (für Method-Chaining)

=cut

# -----------------------------------------------------------------------------

sub commit {
    my $self = shift;

    $self->msg(1,'%T ==commit==');

    # Gleiche Eingabedateien mit Storage-Dateien ab

    my $defDir = $self->storage('def');

    my $dt = Prty::DestinationTree->new($defDir,
        -quiet=>0,
        -language=>'en',
        -outHandle=>\*STDERR,
    );
    $dt->addDir($defDir);

    for my $ent (@{$self->entities}) {
        $dt->addFile($ent->entityFile($defDir),$ent->sourceRef,
            -encoding=>'utf-8',
            -onUpdate=>sub {
                $self->state($ent,1);
            },
        );
    }
    $self->stateDb->sync;

    # Lösche überzählige Storage-Dateien (nach Rückfrage)

    $dt->cleanup(1,$self->getRef('t0'));
    $dt->close;

    return $self;
}
    

# -----------------------------------------------------------------------------

=head3 generate() - Generiere Ausgabe-Dateien

=head4 Synopsis

    $cop = $cop->generate($dir);

=head4 Description

Generiere in Zielverzeichnis $dir die Ausgabe-Dateien aller
Entitäten.

=head4 Returns

ContentProcessor-Objekt (für Method-Chaining)

=cut

# -----------------------------------------------------------------------------

sub generate {
    my ($self,$dir) = @_;
    # @_: @opt

    $self->msg(1,'%T ==generate==');

    # Argumente und Optionen

    my $force = 0;

    Prty::Option->extract(\@_,
        -force=>\$force,
    );

    my $pureDir = $self->storage('pure');
    my $dt = Prty::DestinationTree->new($dir,$pureDir,
        -quiet=>0,
        -language=>'en',
        -outHandle=>\*STDERR,
    );
    $dt->addDir($dir);
    $dt->addDir($pureDir);

    for my $ent (@{$self->entities}) {
        my $codeWritten = 0;
        for my $fil ($ent->files) {
            my $written = $dt->addFile($dir.'/'.$fil->name,
                -encoding=>'utf-8',
                -generate=>$self->state($ent) == 1 || $force,
                -onGenerate=>sub {
                    return $fil->generate;
                },
            );
            if ($written && $fil->isCode) {
                $codeWritten++;
            }
        }

        if ($ent->testable) {
            my $written = $dt->addFile($pureDir.'/'.$ent->entityId.'.pure',
                -encoding=>'utf-8',
                -generate=>$codeWritten,
                -onGenerate=>sub {
                    return $ent->pureCode;
                },
            );
            if ($written) {
                $self->state($ent,2);
                next;
            }
        }
    
        $self->state($ent,0);
    }
    $self->stateDb->sync;

    # Lösche überzählige Dateien (ohne Rückfrage)

    $dt->cleanup;
    $dt->close;

    return $self;
}
    

# -----------------------------------------------------------------------------

=head3 load() - Lade Entitäts-Definitionen

=head4 Synopsis

    $cop = $cop->load;
    $cop = $cop->load(@paths);

=head4 Description

Lade die Entitäts-Dateien der Pfade @paths. Ist @path leer, also
kein Path angegeben, werden die Entitäts-Dateien aus dem Storage
geladen.

Die Methode kann beliebig oft aufgerufen werden, aber nur der
erste Aufruf lädt, alle weiteren Aufrufe sind Null-Operationen.

=head4 Arguments

=over 4

=item @paths

Liste der Verzeichnisse und Dateien. Pfad '-' bedeutet STDIN.

=back

=head4 Returns

ContentProcessor-Objekt (für Method-Chaining)

=cut

# -----------------------------------------------------------------------------

sub load {
    my ($self,@paths) = @_;

    $self->memoize('entityA',sub {
        $self->msg(2,'%T ==find==');

        if (!@paths) {
            @paths = ($self->storage('def'));
        }

        # Ermittele die zu verarbeitenden Dateien

        my @files;
        for my $path (@paths) {
            push @files,-d $path? $self->findFiles($path): $path;
        }

        my $n = scalar @files;
        if ($n > 1) {
            $self->msg(2,'Files: %N',$n);
        }

        $self->msg(2,'%T ==parse==');

        # Instantiiere Parser

        my $par = Prty::Section::Parser->new(
            encoding=>'utf-8',
        );

        # Parse Dateien zu Entitäten

        my @entities;
        for my $file (@files) {
            $par->parse($file,sub {
                my $sec = Prty::Section::Object->new(@_);
                # $sec->removeEofMarker;

                if ($sec->brackets eq '[]') {
                    # Wandele Abschnitts-Objekt in Entität

                    my $plg = $self->plugin($sec);
                    if (!$plg) {
                        # Fehler: Für jeden Haupt-Abschnitt muss ein
                        # Plugin definiert worden sein

                        $sec->error(
                            q{COP-00001: Missing plugin for section},
                             Section=>$sec->fullType,
                        );
                    }
                    push @entities,$plg->class->create($sec,$self,$plg);
                }
                elsif (@entities) {
                    # Füge weiteren Abschnitt zu Entität hinzu

                    my $ent = $entities[-1];
                    $ent->addSubSection($sec);
                    $ent->transferSource($sec);
                }
                else {
                    # Fehler: Erster Abschnitt ist kein []-Abschnitt

                    $sec->error(
                        q{COP-00002: First section must be a []-section},
                         Section=>$sec->fullType,
                    );
                }

                # Abschnittsobjekt gegen unabsichtliche Erweiterungen sperren
                $sec->lockKeys;

                return;
            });
        }

        # Statistische Daten sichern

        $self->set(parsedSections=>$par->get('parsedSections'));
        $self->set(parsedLines=>$par->get('parsedLines'));
        $self->set(parsedChars=>$par->get('parsedChars'));
        $self->set(parsedBytes=>$par->get('parsedBytes'));

        return \@entities;
    });
    
    return $self;
}

# -----------------------------------------------------------------------------

=head3 fetch() - Erzeuge Verzeichnisstruktur mit Entitäts-Definitionen

=head4 Synopsis

    $cop = $cop->fetch($dir,$layout);

=head4 Description

Übertrage alle Entitäts-Definitionen in Verzeichnis $dir (oder
STDOUT, s.u.) gemäß dem Layout $layout. Per Differenzbildung wird
dabei ein konsistenter Stand hergestellt. Existiert Verzeichnis
$dir nicht, wird es angelegt. Andernfalls wird eine Rückfrage
gestellt, ob das Verzeichnis überschrieben werden soll (siehe
auch Option --overwrite).

Wird als Verzeichnis ein Bindestrich (-) angegeben, werden die Daten
nach STDOUT geschrieben.

Die Methode bezieht die zu schreibenden Dateien von der Methode
L</fetchFiles>(), an die der Parameter $layout weiter gereicht
wird. Die Methode kann in abgeleiteten Klassen überschrieben
werden, um andere Strukturen zu generieren.

=head4 Arguments

=over 4

=item $dir

Verzeichnis, in das die Entitäts-Definitionen kopiert werden.

=item $layout

Bezeichnung für das Verzeichnis-Layout. Wird von der von
fetchFiles() der Basisklasse nicht genutzt und daher hier nicht
dokumentiert. Siehe Dokumentation bei den Subklassen.

=back

=head4 Options

=over 4

=item -overwrite => $bool (Default: 0)

Stelle keine Rückfrage, wenn Verzeichnis $dir bereits existiert.

=back

=head4 Returns

ContentProcessor-Objekt (für Method-Chaining)

=cut

# -----------------------------------------------------------------------------

sub fetch {
    my ($self,$dir,$layout) = @_;

    $self->msg(2,'%T ==fetch==');

    # Optionen

    my $overwrite = 0;

    Prty::Option->extract(\@_,
        -overwrite=>\$overwrite,
    );

    if ($dir eq '-') {
        for my $e ($self->fetchFiles($layout)) {
            print ref $e->[1]? ${$e->[1]}: $e->[1];
        }
        return $self;
    }
    elsif (-d $dir && !$overwrite) {
        my $answ = Prty::Terminal->askUser(
            "Overwrite files in directory '$dir'?",
            -values=>'y/n',
            -default=>'y',
            -outHandle=>\*STDERR,
            -timer=>$self->getRef('t0'),
        );
        if ($answ ne 'y') {
            return $self;
        }
    }
    
    my $dt = Prty::DestinationTree->new($dir,
        # -files=>scalar $self->findFiles($dir),
        -quiet=>0,
        -language=>'en',
        -outHandle=>\*STDERR,
    );
    $dt->addDir($dir);

    for my $e ($self->fetchFiles($layout)) {
        $dt->addFile("$dir/$e->[0]",$e->[1],
            -encoding=>'utf-8',
        );
    }

    # Lösche überzählige Storage-Dateien (nach Rückfrage)

    $dt->cleanup(1,$self->getRef('t0'));
    $dt->close;

    return $self;
}
    

# -----------------------------------------------------------------------------

=head3 test() - Teste geänderten Code

=head4 Synopsis

    $cop = $cop->test;

=head4 Returns

ContentProcessor-Objekt (für Method-Chaining)

=cut

# -----------------------------------------------------------------------------

sub test {
    my $self = shift;

    $self->msg(1,'%T ==test==');
    
    return $self;
}

# -----------------------------------------------------------------------------

=head2 Entitäten

=head3 entities() - Liefere Entities

=head4 Synopsis

    @entities | $entityA = $cop->entities;
    @entities | $entityA = $cop->entities($type);

=head4 Description

Liefere die Liste aller geladenen Entities oder aller
geladenen Entities vom Typ $type. Bei der Abfrage der Entities
eines Typs werden die Entities nach Name sortiert geliefert.

=head4 Arguments

=over 4

=item $type

Abschnitts-Typ.

=back

=head4 Returns

Liste von Entitäten. Im Skalarkontext eine Referenz auf die Liste.

=cut

# -----------------------------------------------------------------------------

sub entities {
    my ($self,$type) = @_;

    if (!$type) {
        my $entityA = $self->{'entityA'};
        return wantarray? @$entityA: $entityA;
    }

    my $h = $self->memoize('typeH',sub {
        my ($self,$key) = @_;

        my $typeA = $self->entityTypes;
    
        # Hash mit allen Abschnittstypen aufbauen
    
        my %h;
        for my $type (@$typeA) {
            $h{$type} ||= [];
        }

        # Entitäten zuordnen
    
        for my $ent (@{$self->{'entityA'}}) {
            push @{$h{$ent->type}},$ent;
        }

        # Entitäten nach Name sortieren
    
        for my $type (@$typeA) {
            @{$h{$type}} = sort {$a->name cmp $b->name} @{$h{$type}};
        }

        return \%h;
    });

    my $a = $h->{$type} || $self->throw;    
    return wantarray? @$a: $a;
}

# -----------------------------------------------------------------------------

=head2 Dateien

=head3 fetchFiles() - Liste der Dateien für fetch

=head4 Synopsis

    @files = $cop->fetchFiles;
    @files = $cop->fetchFiles($layout);

=head4 Description

Liefere die Liste der Dateien, die von der Methode L</fetch>()
geschrieben werden. Jede Datei wird durch ein zweielementiges
Array repräsentiert, bestehend aus einem Datei-Pfad und dem
Datei-Inhalt. Der Datei-Inhalt kann als String oder String-Referenz
angegeben sein.

Diese (Basisklassen-)Methode liefert für jede Entität die
Datei-Definiton

    [$ent->entityFile, $ent->sourceRef]

Damit erzeugt die Methode fetch() die gleiche Struktur wie
der ContentProcessor im Storage-Verzeichnis def.

Die Methode kann in abgeleiteten Klassen überschrieben werden, um
die Verzeichnisstruktur zu ändern und/oder den Inhalt der Dateien
anders zusammenzustellen (z.B. mehrere Entity-Definitionen in
einer Datei zusammenzufassen). In abgeleiteten Klassen können
verschiedene Layouts können durch das Argument $layout
unterschieden werden.

=head4 Returns

Array mit zweielementigen Arrays

=cut

# -----------------------------------------------------------------------------

sub fetchFiles {
    my ($self,$layout) = @_;

    my @files;
    for my $ent (@{$self->entities}) {
        push @files,[$ent->entityFile,$ent->sourceRef];
    }
    
    return @files;
}

# -----------------------------------------------------------------------------

=head2 Statistik

=head3 info() - Informationszeile

=head4 Synopsis

    $str = $cop->info;

=head4 Description

Liefere eine Informationszeile mit statistischen Informationen, die
am Ende der Verarbeitung ausgegeben werden kann.

=head4 Returns

Zeichenkette

=cut

# -----------------------------------------------------------------------------

sub info {
    my $self = shift;

    my $entityCount = @{$self->get('entityA')};
    
    return sprintf '%.3f sec; Entities: %s; Sections: %s; Lines: %s'.
            '; Bytes: %s; Chars: %s',
        Time::HiRes::gettimeofday-$self->get('t0'),
        Prty::Formatter->readableNumber($entityCount,','),
        Prty::Formatter->readableNumber($self->get('parsedSections'),','),
        Prty::Formatter->readableNumber($self->get('parsedLines'),','),
        Prty::Formatter->readableNumber($self->get('parsedBytes'),','),
        Prty::Formatter->readableNumber($self->get('parsedChars'),',');
}

# -----------------------------------------------------------------------------

=head2 Intern

=head3 entityTypes() - Liste der Abschnitts-Typen

=head4 Synopsis

    @types | $typeA = $cop->entityTypes;

=head4 Description

Liefere die Liste der Abschnitts-Typen, die per registerType()
registriert wurden.

=head4 Returns

Liste von Abschnitts-Typen. Im Skalarkontext eine Referenz auf die
Liste.

=cut

# -----------------------------------------------------------------------------

sub entityTypes {
    my $self = shift;

    my $a = $self->memoize('entityTypeA',sub {
        my ($self,$key) = @_;

        my %h;
        for my $plg (@{$self->{'pluginA'}}) {
            $h{$plg->sectionType}++;
        }

        return [sort keys %h];
    });

    return wantarray? @$a: $a;
}

# -----------------------------------------------------------------------------

=head3 extensionRegex() - Regex zum Auffinden von Eingabe-Dateien

=head4 Synopsis

    $regex = $cop->extensionRegex;

=head4 Description

Liefere den regulären Ausdruck, der die Dateinamen matcht, die vom
ContentProcessor verarbeitet werden. Der Regex wird genutzt, wenn
ein I<Verzeichnis> nach Eingabe-Dateien durchsucht wird.

=head4 Returns

kopilierter Regex

=cut

# -----------------------------------------------------------------------------

sub extensionRegex {
    my $self = shift;

    return $self->memoize('extensionRegex',sub {
        my ($self,$key) = @_;
        
        # Dateinamen-Erweiterungen ermitteln. Verschiedene
        # Plugin-Klassen können identische Datei-Erweiterungen haben,
        # deswegen filtern wir über einen Hash.

        my %extension;
        for my $k (@{$self->{'baseExtensionA'}}) { # Basis-Extensions
            $extension{$k}++;
        }
        for my $plg (@{$self->{'pluginA'}}) { # Plugin-Extensions
            $extension{$plg->extension}++;
        }        

        # Regex erstellen

        my $regex;
        for (sort keys %extension) {
            my $ext = $_;
            if ($regex) {
                $regex .= '|';
            }
            $regex .= $ext;
        }

        return qr/\.($regex)$/;
    });
}

# -----------------------------------------------------------------------------

=head3 findFiles() - Finde ContentProcessor-Dateien in Verzeichnis

=head4 Synopsis

    @files | $fileA = $cop->findFiles($dir);

=head4 Arguments

=over 4

=item $dir

Das Verzeichnis, das nach Dateien durchsucht wird

=back

=head4 Returns

Liste der Datei-Pfade. Im Skalarkontext eine Referenz auf die
Liste.

=cut

# -----------------------------------------------------------------------------

sub findFiles {
    my ($self,$dir) = @_;

    my @files = Prty::Path->find($dir,
        -type=>'f',
        -pattern=>$self->extensionRegex,
        -sloppy=>1,
    );
    
    return wantarray? @files: \@files;
}

# -----------------------------------------------------------------------------

=head3 msg() - Gib Information aus

=head4 Synopsis

    $cop->msg($level,$msg);

=head4 Arguments

=over 4

=item $msg

Text, der ausgegeben werden soll.

=back

=head4 Returns

nichts

=cut

# -----------------------------------------------------------------------------

sub msg {
    my ($self,$level,$msg) = splice @_,0,3;
    # @_

    if ($level > $self->{'verbosity'}) {
        # Keine Ausgabe, wenn der Meldungslevel höher ist
        # als der vorgegebene Verbosity-Level.
        return;
    }
    
    $msg =~ s/%T/sprintf '%.3f',
        Time::HiRes::gettimeofday-$self->get('t0')/e;

    $msg =~ s/%N/Prty::Formatter->readableNumber(shift,',')/eg;

    printf STDERR "$msg\n",@_;

    return;
}

# -----------------------------------------------------------------------------

=head3 plugin() - Ermittele Plugin zu Abschnitts-Objekt

=head4 Synopsis

    $plg = $cop->plugin($sec);

=head4 Description

Ermittele das Plugin zu Abschnitts-Objekt $sec. Existiert
kein Plugin zu dem Abschnitts-Objekt, liefere C<undef>.

=head4 Arguments

=over 4

=item $sec

Abschnitts-Objekt

=back

=head4 Returns

Plugin-Objekt

=cut

# -----------------------------------------------------------------------------

my %Plugins; # Section-Type => Liste der Plugins
my $Plugin;  # Universelles Plugin (es sollte höchstens eins definiert sein)
    
sub plugin {
    my ($self,$sec) = @_;

    if (!%Plugins) {
        # Indiziere Plugins nach Section-Type

        for my $plg (@{$self->{'pluginA'}}) {
            if (my $sectionType = $plg->sectionType) {
                # Normales Plugin
                push @{$Plugins{$sectionType}},$plg;
            }
            else {
                # Universelles Plugin
                $Plugin = $plg;
            }
        }
    }
        
    # Prüfe Section gemäß Plugin-Kriterien

    if (my $pluginA = $Plugins{$sec->type}) {
        for my $plg (@$pluginA) {
            my $ok = 1;
            my $a = $plg->keyValA;
            for (my $i = 0; $i < @$a; $i += 2) {
                if ($sec->get($a->[$i]) ne $a->[$i+1]) {
                    $ok = 0;
                    last;
                }
            }
            if ($ok) {
                return $plg;
            }
        }
    }
    
    # Kein Plugin zum SectionType gefunden. Wir liefern das
    # universelle Plugin, sofern existent, oder undef

    return $Plugin? $Plugin: undef;
}

# -----------------------------------------------------------------------------

=head3 state() - Liefere/Setze persistenten Entitäts-Status

=head4 Synopsis

    $state = $cop->state($ent);
    $state = $cop->state($ent,$state);

=head4 Description

Liefere/Setze den persistenten Status der Entität $ent. Der
Entitäts-Status ist persistent und bleibt daher über
Programmaufrufe hinweg erhalten.

Eine Entität besitzt einen von vier Status:

=over 4

=item 0

Nichts zu tun. Die Entität wurde nicht geändert.

=item 1

Die Entitäts-Datei wurde geändert. Die Ausgabe-Dateien der
Entität müssen neu generiert werden.

=item 2

Der Code der Entität hat sich geändert. Die Entität und alle
abhängigen Entitäten müssen getestet werden.

=item 3

Nur die Entität selbst muss getestet werden. Die Entität
selbst wurde nicht geändert, hängt aber von einer Entität ab,
die geändert wurde, oder ihre Testdateien oder Testdaten
wurden geändert, was keinen Test der abhängigen Entitäten
erfordert.

=back

Ohne Parameter aufgerufen, liefert die Methode den aktuellen
Zustand der Entität. Mit Parameter gerufen, setzt die Methode den
Zustand, wobei dieser persistent gespeichert wird.

=cut

# -----------------------------------------------------------------------------

sub state {
    my ($self,$ent) = splice @_,0,2;
    # @_: $state

    my $h = $self->stateDb;
    my $entityId = $ent->entityId;

    if (@_) {
        my $state = shift;
        $h->set($entityId=>$state);
        return $state;
    }

    return $h->get($entityId);
}

# -----------------------------------------------------------------------------

=head3 stateDb() - Persistenter Hash für Entitäts-Status

=head4 Synopsis

    $h = $cop->stateDb;

=head4 Description

Liefere eine Referenz auf den persistenten Hash, der den Status
von Entitäten speichert.

=cut

# -----------------------------------------------------------------------------

sub stateDb {
    my $self = shift;

    return $self->memoize('stateDb',sub {
        my ($self,$key) = @_;
        
        my $file = $self->storage('db/entity-state.db');
        return Prty::PersistentHash->new($file,'rw');
    });
}

# -----------------------------------------------------------------------------

=head3 storage() - Pfad zum oder innerhalb des Storage

=head4 Synopsis

    $path = $cop->storage;
    $path = $cop->storage($subPath);

=head4 Description

Liefere den Pfad des Storage, ggf. ergänzt um den Sub-Pfad
$subPath.

=head4 Arguments

=over 4

=item $subPath

Ein Sub-Pfad innerhalb des Storage.

=back

=head4 Returns

Pfad

=cut

# -----------------------------------------------------------------------------

sub storage {
    my $self = shift;
    # @_: $subPath

    my $path = $self->{'storage'};
    if (@_) {
        $path .= '/'.shift;
    }

    return $path;
}

# -----------------------------------------------------------------------------

=head1 VERSION

1.098

=head1 AUTHOR

Frank Seitz, L<http://fseitz.de/>

=head1 COPYRIGHT

Copyright (C) 2016 Frank Seitz

=head1 LICENSE

This code is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut

# -----------------------------------------------------------------------------

1;

# eof
