package Prty::JQueryUI::Form::ViewEdit;
use base qw/Prty::Hash/;

use strict;
use warnings;
use utf8;

our $VERSION = 1.085;

use Prty::Unindent;
use Prty::Html::Widget::Button;
use Prty::Html::Widget::CheckBox;
use Prty::Html::Form::Layout;

# -----------------------------------------------------------------------------

=encoding utf8

=head1 NAME

Prty::JQueryUI::Form::ViewEdit - Formular zum Ansehen und Bearbeiten

=head1 BASE CLASS

L<Prty::Hash>

=head1 ATTRIBUTES

=over 4

=item action => $url (Default: undef)

URL, an den die Daten bei bei Betätigung des Save- und
Delete-Button geschickt werden.

=item id (Default: undef)

Die DOM-Id des Formulars.

=item instantiate => $bool (Default: 0)

Füge die Plugin-Instantiierung beim Aufruf von html()
zum HTML-Code hinzu.

=item layout => $html (Default: '')

Der HTML-Code des Layouts. In das Layout wird der HTML-Code der
Widgets eingesetzt.

=item state => 'update'|'insert' (Default: 'update')

Anfänglicher Zusatand des Formulars:

=over 4

=item 'update'

Der Save- und der Delete-Button werden im Edit-Modus freigeschaltet.

=item 'insert'

Nur der Save-Button wird im Edit-Modus freigeschaltet.

=back

=item widgets => \@widgets (Default: [])

Liste der Widgets, die in das Layout eingesetzt werden.

=back

=head1 METHODS

=head2 Konstruktor

=head3 new() - Instantiiere Objekt

=head4 Synopsis

    $e = $class->new(@keyVal);

=head4 Description

Instantiiere ein Formular-Objekt und liefere eine Referenz auf
dieses Objekt zurück.

=cut

# -----------------------------------------------------------------------------

sub new {
    my $class = shift;
    # @_: @keyVal

    my $self = $class->SUPER::new(
        action=>undef,
        id=>undef,
        instantiate=>0,
        layout=>'',
        state=>'update',
        widgets=>[],
    );
    $self->set(@_);

    return $self;
}

# -----------------------------------------------------------------------------

=head2 Klassenmethoden

=head3 pluginCode() - JavaScript-Code des Plugin

=head4 Synopsis

    $javascript = $e->pluginCode;

=cut

# -----------------------------------------------------------------------------

sub pluginCode {
    my $this = shift;

    return Prty::Unindent->hereDoc(q~
    $.widget('prty.viewEditForm',{
        options: {
            action: null,
            state: null,
        },
        widgets: null,
        _create: function () {
            this.element.addClass('viewEditForm');

            this._on(this.element,{
                'click .editCheckbox': function (event) {
                    this.render();
                },
                'click .saveButton': function (event) {
                    this.execute('save');
                },
                'click .deleteButton': function (event) {
                    this.execute('delete');
                },
            });

            var editCheckbox = $('.editCheckbox',this.element).get(0);
            this.widgets = $(':input',this.element).filter(function (i) {
                return this != editCheckbox;
            });

            this.render();
        },
        render: function () {
            var edit = $('.editCheckbox',this.element).is(':checked');

            if (edit) {
                // Widgets enablen

                this.widgets.prop('disabled',false);
                this.widgets.removeClass('disabled');
                this.widgets.addClass('enabled');

                if (this.options.state == 'insert') {
                    // Im Insert-State wird die Delete-Operation
                    // nicht angeboten

                    var $deleteButton = $('.deleteButton');
                    $deleteButton.prop('disabled',true);
                    $deleteButton.removeClass('enabled');
                    $deleteButton.addClass('disabled');
                }
            }
            else {
                // Widgets disablen

                this.widgets.prop('disabled',true);
                this.widgets.removeClass('enabled');
                this.widgets.addClass('disabled');
            }
        },
        execute: function (op) {
            var data = $(':input',this.element).serialize();
            alert(op+': '+this.options.action+' '+data);
        },
    });
    ~);
}

# -----------------------------------------------------------------------------

=head2 Objektmethoden

=head3 html() - Generiere HTML

=head4 Synopsis

    $html = $e->html($h);
    $html = $class->html($h,@keyVal);

=head4 Description

Generiere den HTML-Code des Formular-Objekts und liefere diesen
zurück. Als Klassenmethode gerufen, wird das Objekt intern erzeugt
und mit den Attributen @keyVal instantiiert.

=cut

# -----------------------------------------------------------------------------

sub html {
    my $this = shift;
    my $h = shift;

    my $self = ref $this? $this: $this->new(@_);

    my ($id,$instantiate,$layout,$state,$widgetA) = $self->get(qw/id
        instantiate layout state widgets/);

    # Liste der Widgets kopieren und Save-, Delete-, Edit-Widgets hinzufügen

    my @widgets = @$widgetA;
    push @widgets,Prty::Html::Widget::Button->new(
        class=>'saveButton',
        name=>'save',
        content=>'Speichern',
    );
    push @widgets,Prty::Html::Widget::Button->new(
        class=>'deleteButton',
        name=>'delete',
        content=>'Löschen',
    );
    push @widgets,Prty::Html::Widget::CheckBox->new(
        class=>'editCheckbox',
        label=>'Bearbeiten',
        name=>'edit',
    );

    my $html = $h->tag('div',
        id=>$id,
        Prty::Html::Form::Layout->html($h,
            layout=>$layout,
            widgets=>\@widgets,
        ),
    );

    if ($instantiate) {
        $html .= $h->tag('script',
            $self->instantiate,
        );
    }

    return $html;
}

# -----------------------------------------------------------------------------

=head3 instantiate() - Generiere JavaScript-Code zum Instantiieren des Widget

=head4 Synopsis

    $javaScript = $e->instantiate;

=cut

# -----------------------------------------------------------------------------

sub instantiate {
    my $self = shift;

    my ($action,$id,$state) = $self->get(qw/action id state/);

    my @att;
    if ($state) {
        push @att,"state: '$state'";
    }
    if ($action) {
        push @att,"action: '$action'";
    }

    return sprintf q|$('#%s').viewEditForm({%s});|,$id,
        "\n    ".join(",\n    ",@att)."\n";
}

# -----------------------------------------------------------------------------

=head1 VERSION

1.085

=head1 AUTHOR

Frank Seitz, L<http://fseitz.de/>

=head1 COPYRIGHT

Copyright (C) 2016 Frank Seitz

=head1 LICENSE

This code is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut

# -----------------------------------------------------------------------------

1;

# eof
