package CXC::DS9::Utils;

# ABSTRACT: Things of Utility with DS9

use strict;
use warnings;

use 5.010;

use Exporter 'import';

our @EXPORT_OK = qw[ display_events ];

use Proc::Simple;

use Types::Standard -types;
use Types::PDL -all;
use Type::Params qw[ compile_named ];
use Image::DS9;
use Image::DS9::Constants;

our $VERSION = '0.01';


sub _croak {
    require Carp;
    Carp::croak( @_ );
}

my $WCS_type = Dict [
    TCTYP => Str,
    TCRPX => Num,
    TCRVL => Num,
    TCDLT => Num,
    TCROT => Num,
  ] | Dict [
    CTYPE => Str,
    CRPIX => Num,
    CRVAL => Num,
    CDELT => Num,
    CROTA => Num,
  ];

my $p1d = Piddle1D [ empty => 0, null => 0 ];

#pod =sub display_events
#pod
#pod   display_events( %args );
#pod
#pod =over
#pod
#pod =item x        => 1D piddle
#pod
#pod =item y        => 1D piddle
#pod
#pod =item wt       => Optional [ Undef | 1D piddle ],
#pod
#pod =item center   => Tuple( Num, Num ) | 1D piddle
#pod
#pod =item radius   => Num,
#pod
#pod =item pix_size => Optional [Num],
#pod
#pod =item ds9      => Optional [ Undef | Str | InstanceOf ['Image::DS9'] ],
#pod
#pod =item xwcs     => Optional [ Undef | WCS hash ],
#pod
#pod =item ywcs     => Optional [ Undef | WCS hash ],
#pod
#pod =item kill_on_destroy => Optional [Bool],
#pod
#pod =back
#pod
#pod =cut



sub display_events {

    state $check = compile_named(
        x        => $p1d,
        y        => $p1d,
        wt       => Optional [ Undef | $p1d ],
        center   => $p1d->plus_coercions( PiddleFromAny ),
        radius   => Num,
        pix_size => Optional [Num],
        ds9      => Optional [ Undef | Str | InstanceOf ['Image::DS9'] ],
        xwcs            => Optional [ Undef | $WCS_type ],
        ywcs            => Optional [ Undef | $WCS_type ],
        kill_on_destroy => Optional [Bool],
    );


    my $opt = $check->( @_ );

    _croak( "center must have exactly two elements" )
      unless 2 == $opt->{center}->nelem;

    $opt->{pix_size} //= 1;

    _croak( "x and y piddles must have the same shape" )
      unless $opt->{x}->nelem == $opt->{y}->nelem;

    _croak( "x, y, and wt piddles must have the same shape" )
      if defined $opt->{wt} && $opt->{wt}->nelem != $opt->{y}->nelem;

    my $ds9 = do {

        if ( !ref $opt->{ds9} ) {

            require CXC::DS9::Server;

            CXC::DS9::Server->new( {
                ( defined $opt->{ds9} ? ( Server => $opt->{ds9} ) : () ),
                kill_on_destroy => $opt->{kill_on_destroy},
            } );
        }

        else {

            $opt->{ds9};
        }
    };

    $ds9->raise();

    #  $ds9->frame( _clear, _all );
    $ds9->Set( 'frame clear all' );

    my ( $xc, $yc ) = $opt->{center}->list;

    require CXC::Region2D::Box;
    require Img2D::Events;

    my $box = CXC::Region2D::Box->square_c( $xc, $yc, 2 * $opt->{radius} );
    my $img = Img2D::Events->new(
        $opt->{x},
        $opt->{y},
        $opt->{wt},
        {
            box   => $box,
            block => $opt->{pix_size},
            ( defined $opt->{xwcs} ? ( xwcs => $opt->{xwcs} ) : () ),
            ( defined $opt->{ywcs} ? ( ywcs => $opt->{ywcs} ) : () ),
        } );

    # display image
    $ds9->array( $img->image );
    $ds9->wcs( Image::DS9::Constants::_replace(), $img->get_WCS );
    $ds9->zoom( 0 );

    return $ds9;
}

1;

#
# This file is part of CXC-DS9
#
# This software is Copyright (c) 2017 by Smithsonian Astrophysical Observatory.
#
# This is free software, licensed under:
#
#   The GNU General Public License, Version 3, June 2007
#

=pod

=head1 NAME

CXC::DS9::Utils - Things of Utility with DS9

=head1 VERSION

version 0.01

=head1 SYNOPSIS

=head1 SUBROUTINES

=head2 display_events

  display_events( %args );

=over

=item x        => 1D piddle

=item y        => 1D piddle

=item wt       => Optional [ Undef | 1D piddle ],

=item center   => Tuple( Num, Num ) | 1D piddle

=item radius   => Num,

=item pix_size => Optional [Num],

=item ds9      => Optional [ Undef | Str | InstanceOf ['Image::DS9'] ],

=item xwcs     => Optional [ Undef | WCS hash ],

=item ywcs     => Optional [ Undef | WCS hash ],

=item kill_on_destroy => Optional [Bool],

=back

=head1 AUTHOR

Diab Jerius <djerius@cfa.harvard.edu>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2017 by Smithsonian Astrophysical Observatory.

This is free software, licensed under:

  The GNU General Public License, Version 3, June 2007

=cut

__END__


#pod =head1 SYNOPSIS
#pod
#pod
#pod =head1 SEE ALSO
#pod
