package CXC::DS9::Server;

use 5.010;

# ABSTRACT: Image::DS9 subclass with controlled lifetime of server

use strict;
use warnings;

our $VERSION = '0.01';

use parent 'Image::DS9';
use Proc::Simple;

sub _croak {
    require Carp;
    Carp::croak( @_ );
}

#pod =method new
#pod
#pod   $server = CXC::DS9::Server->new( \%attrs );
#pod
#pod Connect to an existing C<ds9> process or create a new one.  It accepts
#pod the same arguments as does L<Image::DS9/new>, with the following additional
#pod ones:
#pod
#pod =over
#pod
#pod =item C<kill_on_destroy> => I<Boolean>
#pod
#pod If true, the C<ds9> process will be terminated when the object is destroyed.
#pod
#pod =back
#pod
#pod =cut

sub new {

    my ( $class, $attr ) = @_;

    my %attr = %{ $attr // {} };

    my $kill_on_destroy = delete $attr{kill_on_destroy};

    my $self = $class->SUPER::new( \%attr );

    $self->{_kill_on_destroy} = !!$kill_on_destroy;

    unless ( $self->nservers ) {

        $self->{_process} = Proc::Simple->new;
        $self->{_process}->kill_on_destroy( $kill_on_destroy );

        my @cmd = (
            'ds9',
            (
                defined $self->{Server}
                ? ( -title => $self->{Server} )
                : ()
            ),
        );


        $self->{_process}->start( @cmd )
          or _croak( "error running @cmd\n" );

        $self->wait() or _croak( "error connecting to ds9 (@cmd)\n " );
    }

    return $self;
}

sub DESTROY {

    my $self = shift;

    $self->quit
      if $self->{_kill_on_destroy};

    # note that if we had to start up a bespoke ds9 above, the
    # Proc::Simple object will also kill the process upon destruction.


    $self->SUPER::DESTROY;
}

1;

#
# This file is part of CXC-DS9
#
# This software is Copyright (c) 2017 by Smithsonian Astrophysical Observatory.
#
# This is free software, licensed under:
#
#   The GNU General Public License, Version 3, June 2007
#

=pod

=head1 NAME

CXC::DS9::Server - Image::DS9 subclass with controlled lifetime of server

=head1 VERSION

version 0.01

=head1 SYNOPSIS

  use CXC::DS9::Server;

  {
    # attach to an existing ds9 process or attach to a new one.
    my $server = CXC::DS9::Server->new( { kill_on_destroy => 1 } );
  }

  # now the ds9 process is no more

=head1 DESCRIPTION

B<CXC::DS9::Server> is a subclass of L<Image::DS9> which takes care of
starting a B<ds9> process if there is no existent one it can connect to.

It has the added feature of optionally terminating the B<ds9> process
when the B<CXC::DS9::Server> object is destroyed.

=head1 METHODS

=head2 new

  $server = CXC::DS9::Server->new( \%attrs );

Connect to an existing C<ds9> process or create a new one.  It accepts
the same arguments as does L<Image::DS9/new>, with the following additional
ones:

=over

=item C<kill_on_destroy> => I<Boolean>

If true, the C<ds9> process will be terminated when the object is destroyed.

=back

=head1 SEE ALSO

Please see those modules/websites for more information related to this module.

=over 4

=item *

L<CXC::DS9::Utils|CXC::DS9::Utils>

=back

=head1 AUTHOR

Diab Jerius <djerius@cfa.harvard.edu>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2017 by Smithsonian Astrophysical Observatory.

This is free software, licensed under:

  The GNU General Public License, Version 3, June 2007

=cut

__END__


#pod =head1 SYNOPSIS
#pod
#pod   use CXC::DS9::Server;
#pod
#pod   {
#pod     # attach to an existing ds9 process or attach to a new one.
#pod     my $server = CXC::DS9::Server->new( { kill_on_destroy => 1 } );
#pod   }
#pod
#pod   # now the ds9 process is no more
#pod
#pod
#pod =head1 DESCRIPTION
#pod
#pod B<CXC::DS9::Server> is a subclass of L<Image::DS9> which takes care of
#pod starting a B<ds9> process if there is no existent one it can connect to.
#pod
#pod It has the added feature of optionally terminating the B<ds9> process
#pod when the B<CXC::DS9::Server> object is destroyed.
#pod
#pod
#pod =head1 SEE ALSO
