#
# Module Parse::Eyapp::Grammar
#
# (c) Copyright 1998-2001 Francois Desarmenien, all rights reserved.
# (c) Copyright 2006-2008 Casiano Rodriguez-Leon, all rights reserved.
# 
package Parse::Eyapp::Grammar;
@ISA=qw( Parse::Eyapp::Options );

require 5.004;

use Carp;
use strict;
use Parse::Eyapp::Options;
use Parse::Eyapp::Parse;
use Scalar::Util qw{reftype};

###############
# Constructor #
###############
sub new {
    my($class)=shift;
    my($values);

    my($self)=$class->SUPER::new(@_);

    my($parser)=new Parse::Eyapp::Parse;

        defined($self->Option('input'))
    or  croak "No input grammar";

    $values = $parser->Parse($self->Option('input'), 
                             $self->Option('firstline'), # Line where the grammar source starts
                             $self->Option('inputfile'),  # The file or program containing the grammar
                             #$self->Option('prefixname'),  # yyprefix
                             #$self->Option('buildingtree')  # If building AST
                            );

    undef($parser);

    $$self{GRAMMAR}=_ReduceGrammar($values);

        ref($class)
    and $class=ref($class);

    bless($self, $class);

    my $ns = $self->{GRAMMAR}{NAMINGSCHEME} ;
    if ($ns && reftype($ns) eq 'ARRAY') {
      $ns = eval "sub { $ns->[0]; }; ";
      warn "Error in \%namingscheme directive $@" if $@;
      $ns = $ns->($self);
    }
    $ns ||= \&give_default_name;
    $self->{GRAMMAR}{NAMINGSCHEME} = $ns; # added to allow programmable production naming schemes (%name)

    $self;
}

###########
# Methods #
###########
##########################
# Method To View Grammar #
##########################
sub ShowRules {
    my($self)=shift;
    my($rules)=$$self{GRAMMAR}{RULES};
    my($ruleno)=-1;
    my($text);

    for (@$rules) {
        my($lhs,$rhs)=@$_;

        $text.=++$ruleno.":\t".$lhs." -> ";
        if(@$rhs) {
            $text.=join(' ',map { $_ eq chr(0) ? '$end' : $_ } @$rhs);
        }
        else {
            $text.="/* empty */";
        }
        $text.="\n";
    }
    $text;
}

sub give_default_name {
  my ($self, $index, $lhs) = @_;

  my $name = "$lhs"."_$index";
  return $name;
}

sub give_lhs_name {
  my ($self, $index, $lhs, $rhs) = @_;

  my $name = $lhs;
  return $name;
}

sub give_token_name {
  my ($self, $index, $lhs, $rhs) = @_;

  my @rhs = @$rhs;
  $rhs = '';

  unless (@rhs) { # Empty RHS
    return $lhs.'_is_empty';
  }

  my $names = $self->{GRAMMAR}{TOKENNAMES} || {};
  for (@rhs) {
    if ($self->is_token($_)) { 
      s/^'(.*)'$/$1/;
      my $name = $names->{$_} || '';
      unless ($name) {
        $name = $_ if /^\w+$/;
      }
      $rhs .= "_$name" if $name;
    }
  }

  unless ($rhs) { # no 'word' tokens in the RHS
    for (@rhs) {
      $rhs .= "_$_" if /^\w+$/;
    }
  }

  # check if another production with such name exists?
  my $name = $lhs.'_is'.$rhs;
  return $name;
}

sub give_rhs_name {
  my ($self, $index, $lhs, $rhs) = @_;

  my @rhs = @$rhs;
  $rhs = '';

  unless (@rhs) { # Empty RHS
    return $lhs.'_is_empty';
  }

  my $names = $self->{GRAMMAR}{TOKENNAMES} || {};
  for (@rhs) {
    if ($self->is_token($_)) { 
      s/^'(.*)'$/$1/;
      my $name = $names->{$_} || '';
      unless ($name) {
        $name = $_ if /^\w+$/;
      }
      $rhs .= "_$name" if $name;
    }
    else {
      s/\W/_/g;
      $rhs .= "_$_";
    }
  }

  # check if another production with such name exists?
  my $name = $lhs.'_is'.$rhs;
  return $name;
}

sub classname {
  my ($self, $name, $index, $lhs, $rhs) = @_;

  $name = $name->[0];

  unless (defined($name)) {
    if ($lhs =~ /\$start/) {
      $name = "_SUPERSTART"
    }
    elsif ($lhs =~ /\@(\d+)-(\d+)/) {
      $name = "_CODE" 
    }
    elsif ($lhs =~ /PAREN-(\d+)/) {
      $name = "_PAREN" 
    }
    elsif ($lhs =~ /STAR-(\d+)/) {
      $name = "_STAR_LIST"
    }
    elsif ($lhs =~ /PLUS-(\d+)/) {
      $name = "_PLUS_LIST"
    }
    elsif ($lhs =~ /OPTIONAL-(\d+)/) {
      $name = "_OPTIONAL"
    }
  }

  my $naming_scheme = $self->{GRAMMAR}{NAMINGSCHEME};
  $name = $naming_scheme->($self, $index, $lhs, $rhs) unless $name;

  return $name;
}

# Added by Casiano
#####################################
# Method To Return the Grammar Rules#
#####################################
sub Rules { # TODO: find proper names
    my($self)=shift;
    my($rules)=$$self{GRAMMAR}{RULES};
    my($text) = "[\n";
    my $packages = ' qw{TERMINAL _OPTIONAL _STAR_LIST _PLUS_LIST ';

    my $index = 0;
    for (@$rules) {
        my($lhs,$rhs,$prec,$name)=@$_;

        my $bypass = $name->[2];
        $bypass = $self->Bypass unless defined($bypass);
        # find an acceptable perl identifier as name
        $name = $self->classname($name, $index, $lhs, $rhs);
        $packages .= "\n".(" "x9).$name unless $packages =~ m{\b$name\b};

        $text.= "  [ $name => '$lhs', [ ";
        $text.=join(', ',map { $_ eq chr(0) ? "'\$end'" : $_ =~ m{^'} ? $_ : "'$_'" } @$rhs);
        $text.=" ], $bypass ],\n";
        $index++;
    }
    $text .= ']';
    $packages .= '} ';
    return ($text, $packages);
}

# Added by Casiano
#####################################
# Method To Return the Grammar Terms#
#####################################
sub Terms {
    my($self)=shift;
    my(@terms)= sort(keys(%{$$self{GRAMMAR}{TERM}}));
    my %semantic = %{$self->{GRAMMAR}{SEMANTIC}};

    my $text = "{ ";
    $text .= join(",\n\t",
                         # Warning! bug. Before: map { $_ eq chr(0) ? "'\$end' => 0" : "$_ => $semantic{$_}"} @terms);
                         map { $_ eq chr(0) ? "'' => { ISSEMANTIC => 0 }" : "$_ => { ISSEMANTIC => $semantic{$_} }"} @terms); 
    $text .= ",\n\terror => { ISSEMANTIC => 0 },\n}";
}

#####################################
# Method To Return the Bypass Option#
#####################################
sub Bypass {
  my($self)=shift;
    
  return  $$self{GRAMMAR}{BYPASS}
}

#####################################
# Method To Return the Prefix Option#
#####################################
sub Prefix {
  my($self)=shift;
    
  return  $$self{GRAMMAR}{PREFIX}
}


sub Buildingtree {
  my($self)=shift;
    
  return  $$self{GRAMMAR}{BUILDINGTREE}
}


sub is_token {
  my($self)=shift;

  exists($self->{GRAMMAR}{TERM}{$_[0]})
}

#####################################
# Method To Return the ACCESSORS
#####################################
sub Accessors {
  my($self)=shift;
    
  return  $$self{GRAMMAR}{ACCESSORS}
}

###########################
# Method To View Warnings #
###########################
sub Warnings {
    my($self)=shift;
    my($text);
    my($grammar)=$$self{GRAMMAR};

        exists($$grammar{UUTERM})
    and    do {
            $text="Unused terminals:\n\n";
            for (@{$$grammar{UUTERM}}) {
                $text.="\t$$_[0], declared line $$_[1]\n";    
            }
        $text.="\n";
        };
        exists($$grammar{UUNTERM})
    and    do {
            $text.="Useless non-terminals:\n\n";
            for (@{$$grammar{UUNTERM}}) {
                $text.="\t$$_[0], declared line $$_[1]\n";    
            }
        $text.="\n";
        };
        exists($$grammar{UURULES})
    and    do {
            $text.="Useless rules:\n\n";
            for (@{$$grammar{UURULES}}) {
                $text.="\t$$_[0] -> ".join(' ',@{$$_[1]})."\n";    
            }
        $text.="\n";
        };
    $text;
}

######################################
# Method to get summary about parser #
######################################
sub Summary {
    my($self)=shift;
    my($text);

    $text ="Number of rules         : ".
            scalar(@{$$self{GRAMMAR}{RULES}})."\n";
    $text.="Number of terminals     : ".
            scalar(keys(%{$$self{GRAMMAR}{TERM}}))."\n";
    $text.="Number of non-terminals : ".
            scalar(keys(%{$$self{GRAMMAR}{NTERM}}))."\n";
    $text;
}

###############################
# Method to Ouput rules table #
###############################
sub RulesTable {
    my($self)=shift;
    my($inputfile)=$self->Option('inputfile');
    my($linenums)=$self->Option('linenumbers');
    my($rules)=$$self{GRAMMAR}{RULES};
    my $ruleno = 0;
    my($text);

        defined($inputfile)
    or  $inputfile = 'unknown';

    $text="[\n\t";

    $text.=join(",\n\t",
                map {
                    my($lhs,$rhs,$rname,$code)=@$_[0,1,3,4];
                    my($len)=scalar(@$rhs);

                    my($text);

                    $rname = $self->classname($rname, $ruleno, $lhs, $rhs);

                    $ruleno++;
                    $text.="[#Rule $rname\n\t\t '$lhs', $len,";
                    if($code) {
                        $text.= "\nsub {".
                                (  $linenums
                                 ? qq(\n#line $$code[1] "$inputfile"\n)
                                 : " ").
                                "$$code[0]}";
                    }
                    else {
                        $text.=' undef';
                    }
                    $text.="\n$Parse::Eyapp::Output::pattern\n\t]";

                    $text;
                } @$rules);

    $text.="\n]";

    $text;
}

###############################################################
# Method to produce conflict information
# to be used by dynamic conflict solving, e.g. 
# YYSetReduce(token, 'rule_name')
# YYSetShift(token)
# 
# Rules:
# ------
# 0:	$start -> s $end
# 1:	s -> /* empty */
# 2:	s -> s ws
# 3:	s -> s ns
# 4:	ns -> /* empty */
# 5:	ns -> ns NUM
# 6:	ws -> /* empty */
# 7:	ws -> ws ID
# 
#   DB<3> x $self->{CONFLICTS}{FORCED}
# 0  HASH(0x88e2998)
#    'DETAIL' => HASH(0x88a7d0c)
#       1 => HASH(0x88dbe98)
#          'LIST' => ARRAY(0x88dbeb0)
#             0  ARRAY(0x88dbec8)
#                0  'NUM'
#                1  '-6' 3 
#             1  ARRAY(0x88dbab4)
#                0  'ID'
#                1  '-6'
#             2  ARRAY(0x88dba90)
#                0  "\c@"
#                1  '-4'
#             3  ARRAY(0x88dbf58)
#                0  "\c@"
#                1  '-6'
#          'TOTAL' => ARRAY(0x88e235c)
#             0  1
#             1  3
#       2 => HASH(0x88dbfac)
#          'LIST' => ARRAY(0x88dbfd0)
#             0  ARRAY(0x88dbfe8)
#                0  'NUM'
#                1  '-3'
#          'TOTAL' => ARRAY(0x88dbd3c)
#             0  1
#       4 => HASH(0x88dbdd8)
#          'LIST' => ARRAY(0x88dbe38)
#             0  ARRAY(0x88dbe50)
#                0  'ID'
#                1  '-2'
#          'TOTAL' => ARRAY(0x88dbdb4)
#             0  1
#    'TOTAL' => ARRAY(0x88e238c)
#       0  3
#       1  3
# 
###############################################################
sub _Conflicts {
  my $self = shift;
  my $conflicts = $self->{CONFLICTS}{FORCED}{DETAIL};

  # Just while developping
  require Data::Dumper;
  Data::Dumper->import;
  local $Data::Dumper::Indent = 0;
  local $_ = Dumper($conflicts);

  s/^\$VAR\d*\s*=\s*//;
  $_
}

################################
# Methods to get HEAD and TAIL #
################################
sub Head {
    my($self)=shift;
    my($inputfile)=$self->Option('inputfile');
    my($linenums)=$self->Option('linenumbers');
    my($text);

        $$self{GRAMMAR}{HEAD}[0]
    or  return '';

        defined($inputfile)
    or  $inputfile = 'unkown';

    for (@{$$self{GRAMMAR}{HEAD}}) {
            $linenums
        and $text.=qq(#line $$_[1] "$inputfile"\n);
        $text.=$$_[0];
    }
    $text
}

sub Tail {
    my($self)=shift;
    my($inputfile)=$self->Option('inputfile');
    my($linenums)=$self->Option('linenumbers');
    my($text);

        $$self{GRAMMAR}{TAIL}[0]
    or  return '';

        defined($inputfile)
    or  $inputfile = 'unkown';

        $linenums
    and $text=qq(#line $$self{GRAMMAR}{TAIL}[1] "$inputfile"\n);
    $text.=$$self{GRAMMAR}{TAIL}[0];

    $text
}


#################
# Private Stuff #
#################

sub _UsefulRules {
    my($rules,$nterm) = @_;
    my($ufrules,$ufnterm);
    my($done);

    $ufrules=pack('b'.@$rules);
    $ufnterm={};

    vec($ufrules,0,1)=1;    #start rules IS always useful

    RULE:
    for (1..$#$rules) { # Ignore start rule
        for my $sym (@{$$rules[$_][1]}) {
                exists($$nterm{$sym})
            and next RULE;
        }
        vec($ufrules,$_,1)=1;
        ++$$ufnterm{$$rules[$_][0]};
    }

    do {
        $done=1;

        RULE:
        for (grep { vec($ufrules,$_,1) == 0 } 1..$#$rules) {
            for my $sym (@{$$rules[$_][1]}) {
                    exists($$nterm{$sym})
                and not exists($$ufnterm{$sym})
                and next RULE;
            }
            vec($ufrules,$_,1)=1;
                exists($$ufnterm{$$rules[$_][0]})
            or  do {
                $done=0;
                ++$$ufnterm{$$rules[$_][0]};
            };
        }

    }until($done);

    ($ufrules,$ufnterm)

}#_UsefulRules

sub _Reachable {
    my($rules,$nterm,$term,$ufrules,$ufnterm)=@_;
    my($reachable);
    my(@fifo)=( 0 );

    $reachable={ '$start' => 1 }; #$start is always reachable

    while(@fifo) {
        my($ruleno)=shift(@fifo);

        for my $sym (@{$$rules[$ruleno][1]}) {

                exists($$term{$sym})
            and do {
                ++$$reachable{$sym};
                next;
            };

                (   not exists($$ufnterm{$sym})
                 or exists($$reachable{$sym}) )
            and next;

            ++$$reachable{$sym};
            push(@fifo, grep { vec($ufrules,$_,1) } @{$$nterm{$sym}});
        }
    }

    $reachable

}#_Reachable

sub _SetNullable {
    my($rules,$term,$nullable) = @_;
    my(@nrules);
    my($done);

    RULE:
    for (@$rules) {
        my($lhs,$rhs)=@$_;

            exists($$nullable{$lhs})
        and next;

        for (@$rhs) {
                exists($$term{$_})
            and next RULE;
        }
        push(@nrules,[$lhs,$rhs]);
    }

    do {
        $done=1;

        RULE:
        for (@nrules) {
            my($lhs,$rhs)=@$_;

                    exists($$nullable{$lhs})
                and next;

                for (@$rhs) {
                        exists($$nullable{$_})
                    or  next RULE;
                }
            $done=0;
            ++$$nullable{$lhs};
        }

    }until($done);
}

sub _ReduceGrammar {
    my($values)=@_;
    my($ufrules,$ufnterm,$reachable);

    my($grammar)= bless { 
                   HEAD => $values->{HEAD},
                   TAIL => $values->{TAIL},
                   EXPECT => $values->{EXPECT},
                   # Casiano modifications
                   SEMANTIC       => $values->{SEMANTIC}, # added to simplify AST
                   BYPASS         => $values->{BYPASS},   # added to simplify AST
                   BUILDINGTREE   => $values->{BUILDINGTREE},   # influences the semantic of lists * + ?
                   ACCESSORS      => $values->{ACCESSORS}, # getter-setter for %tree and %metatree
                   PREFIX         => $values->{PREFIX},   # yyprefix
                   NAMINGSCHEME   => $values->{NAMINGSCHEME}, # added to allow programmable production naming schemes (%name)
                   NOCOMPACT      => $values->{NOCOMPACT}, # Do not compact action tables. No DEFAULT field for "STATES"
                   TOKENNAMES     => {},
                 }, __PACKAGE__;

    my($rules,$nterm,$term) =  @$values {'RULES', 'NTERM', 'TERM'};

    ($ufrules,$ufnterm) = _UsefulRules($rules,$nterm);

        exists($$ufnterm{$values->{START}})
    or  die "*Fatal* Start symbol $values->{START} derives nothing, at eof\n";

    $reachable = _Reachable($rules,$nterm,$term,$ufrules,$ufnterm);

    $$grammar{TERM}{chr(0)}=undef;
    for my $sym (keys %$term) {
            (   exists($$reachable{$sym})
             or exists($values->{PREC}{$sym}) )
        and do {
            $$grammar{TERM}{$sym}
                = defined($$term{$sym}[0]) ? $$term{$sym} : undef;
            next;
        };
        push(@{$$grammar{UUTERM}},[ $sym, $values->{SYMS}{$sym} ]);
    }

    $$grammar{NTERM}{'$start'}=[];
    for my $sym (keys %$nterm) {
            exists($$reachable{$sym})
        and do {
                exists($values->{NULL}{$sym})
            and ++$$grammar{NULLABLE}{$sym};
            $$grammar{NTERM}{$sym}=[];
            next;
        };
        push(@{$$grammar{UUNTERM}},[ $sym, $values->{SYMS}{$sym} ]);
    }

    for my $ruleno (0..$#$rules) {
            vec($ufrules,$ruleno,1)
        and exists($$grammar{NTERM}{$$rules[$ruleno][0]})
        and do {
            push(@{$$grammar{RULES}},$$rules[$ruleno]);
            push(@{$$grammar{NTERM}{$$rules[$ruleno][0]}},$#{$$grammar{RULES}});
            next;
        };
        push(@{$$grammar{UURULES}},[ @{$$rules[$ruleno]}[0,1] ]);
    }

    _SetNullable(@$grammar{'RULES', 'TERM', 'NULLABLE'});

    $grammar;
}#_ReduceGrammar

sub tokennames {
  my $self = shift;

  my $grammar = $self->{GRAMMAR};
  $grammar->{TOKENNAMES} = { (%{$grammar->{TOKENNAMES}}, @_) } if (@_);
  $grammar->{TOKENNAMES}
}

1;

