package PDK::Device::Base;

use v5.30;
use strict;
use warnings;

use Moose::Role;
use Carp qw'croak';
use Expect;
use namespace::autoclean;

#------------------------------------------------------------------------------
# TODO: 识别普通模式和配置下发模式，需要拦截普通模式进入到配置模式
#------------------------------------------------------------------------------

#------------------------------------------------------------------------------
# 继承 PDK::Device::Role 必须实现的方法
#------------------------------------------------------------------------------
requires 'errCodes';     # 必须实现的错误代码方法
requires 'waitfor';      # 必须实现的等待方法
requires 'getConfig';    # 必须实现的配置获取方法

#------------------------------------------------------------------------------
# 定义 PDK::Device::Role 基础属性
#------------------------------------------------------------------------------
has exp => (
  is       => 'ro',    # 只读属性
  required => 0,       # 非必需
);

has host => (
  is       => 'ro',    # 只读属性
  required => 0,       # 非必需
);

has port => (
  is       => 'ro',    # 只读属性
  required => 0,       # 非必需
  default  => '',      # 默认值为空字符串
);

has proto => (
  is       => 'ro',     # 只读属性
  required => 0,        # 非必需
  default  => 'ssh',    # 默认协议为 SSH
);

has prompt => (
  is       => 'ro',              # 只读属性
  required => 1,                 # 必需
  default  => '\S+[#>]\s*\z',    # 默认提示符
);

has enPrompt => (
  is       => 'ro',              # 只读属性
  required => 0,                 # 非必需
  default  => '',                # 默认值为空字符串
);

has enCommand => (
  is       => 'ro',              # 只读属性
  required => 0,                 # 非必需
  default  => '',                # 默认值为空字符串
);

has username => (
  is       => 'ro',              # 只读属性
  required => 0,                 # 非必需
  default  => '',                # 默认用户名为 admin
);

has password => (
  is       => 'ro',              # 只读属性
  required => 0,                 # 非必需
  default  => '',                # 默认值为空字符串
);

has enPassword => (
  is       => 'ro',              # 只读属性
  required => 0,                 # 非必需
);

has passphrase => (
  is       => 'ro',              # 只读属性
  required => 0,                 # 非必需
  default  => '',                # 默认值为空字符串
);

has mode => (
  is       => 'ro',              # 只读属性
  required => 0,                 # 非必需
  default  => 'normal',          # 默认模式为正常
);

has catchError => (
  is       => 'ro',              # 只读属性
  required => 0,                 # 非必需
  default  => 1,                 # 默认启用错误捕获
);

#------------------------------------------------------------------------------
# 状态和模式属性
#------------------------------------------------------------------------------
has enabled => (
  is       => 'rw',              # 可读可写属性
  required => 0,                 # 非必需
  default  => 0,                 # 默认值为 0
);

has status => (
  is       => 'rw',              # 可读可写属性
  required => 0,                 # 非必需
  default  => 0,                 # 默认值为 0
);

#------------------------------------------------------------------------------
# 日期属性：获取当前日期并格式化为 YYYY-MM
#------------------------------------------------------------------------------
has month => (
  is       => 'rw',              # 属性可读可写
  required => 0,                 # 非必需属性
  default  => sub {
    my $month = `date +%Y-%m`;    # 执行命令获取当前年月
    chomp($month);                # 去除换行符
    return $month;                # 返回格式化后的日期
  },
);

#------------------------------------------------------------------------------
# 日期属性：获取当前日期并格式化为 YYYY-MM-DD
#------------------------------------------------------------------------------
has date => (
  is       => 'rw',    # 属性可读可写
  required => 0,       # 非必需属性
  default  => sub {
    my $date = `date +%Y-%m-%d`;    # 执行命令获取当前日期
    chomp($date);                   # 去除换行符
    return $date;                   # 返回格式化后的日期
  },
);

#------------------------------------------------------------------------------
# 工作目录属性：指定配置文件的根目录
#------------------------------------------------------------------------------
has workdir => (
  is       => 'rw',                                         # 属性可读可写
  required => 0,                                            # 非必需属性
                                                            # 默认值为 PDK_CONFIG_HOME 环境变量或用户家目录
  default  => sub { $ENV{PDK_CONFIG_HOME} // glob("~") },
);

#------------------------------------------------------------------------------
# 调试开关：自动记录登录交互到本地日志文件
#------------------------------------------------------------------------------
has debug => (
  is       => 'rw',                                         # 可读可写属性
  required => 0,                                            # 非必需
  default  => 0,                                            # 默认关闭调试
);

#------------------------------------------------------------------------------
# 定义 PDK::Device::Role 基类方法
#------------------------------------------------------------------------------

#------------------------------------------------------------------------------
# login: 尝试登录设备
# 描述：此方法检查是否已登录设备，若未登录则进行连接尝试。
# 返回：一个包含登录状态的哈希引用，成功时为 { success => 1 }，失败时包含失败原因
#------------------------------------------------------------------------------
sub login {
  my $self = shift;

  # 已登录设备则跳过后续逻辑
  return {success => 1} if $self->{status} == 1;

  # 尝试连接，如果没有已有的 Expect 对象
  eval {
    if (!$self->{exp}) {
      say "[debug] 正在初始化 Expect 对象并登录设备 $self->{host} !" if $self->{debug};    # 打印调试信息
      $self->connect();                                                        # 调用连接方法
    }
    else {
      croak "执行[login/尝试连接设备]，连接 $self->{host} 异常: 已经初始化 Expect 对象，无法再次初始化！";    # 已经存在 Expect 对象
    }
  };

  # 检查是否发生了异常
  if ($@) {
    chomp($@);                                                                   # 去掉异常信息末尾的换行符

    # 根据不同的错误类型进行处理
    if ($@ =~ /RSA modulus too small/mi) {
      eval {
        $self->connect('-v -1');    # 尝试使用 ssh v1 版本的加密参数连接
      };
      if ($@) {
        chomp($@);
        return {success => 0, reason => $@};    # 返回错误原因
      }
    }
    elsif ($@ =~ /Selected cipher type <unknown> not supported by server/mi) {
      eval {
        $self->connect('-c des');               # 尝试使用 DES 加密连接
      };
      if ($@) {
        chomp($@);
        return {success => 0, reason => $@};    # 返回错误原因
      }
    }
    elsif ($@ =~ /no matching key exchange method found./mi) {
      eval {
        $self->connect('-c des');               # 尝试使用 DES 加密连接
      };
      if ($@) {
        chomp($@);
        return {success => 0, reason => $@};    # 返回错误原因
      }
    }
    elsif ($@ =~ /Connection refused/mi) {
      if ($self->{debug}) {
        say "[debug] SSH会话($self->{host})异常, 尝试（仅支持默认端口自动切换）切换 [telnet] 登录！";
      }
      eval {
        $self->{proto} = 'telnet';              # 更改协议为 Telnet
        $self->connect();                       # 再次尝试连接
      };
      if ($@) {
        chomp($@);
        return {success => 0, reason => $@};    # 返回错误原因
      }
    }
    elsif ($@ =~ /IDENTIFICATION CHANGED/mi) {
      if ($self->{debug}) {
        my $msg = "捕捉到异常：" . $@;
        say "[debug] 尝试刷新SSH密钥-> /usr/bin/ssh-keygen -R $self->{host} , $msg";
      }

      # 处理身份验证更改，更新已知主机信息
      system("/usr/bin/ssh-keygen -R $self->{host}");    # 移除已知主机信息
      eval {
        $self->connect();                                # 再次尝试连接
      };
      if ($@) {
        chomp($@);
        return {success => 0, reason => $@};             # 返回错误原因
      }
    }
    else {
      return {success => 0, reason => $@};               # 处理未识别的错误
    }
  }

  # 打印登录提示
  say "\n[debug] 成功登录网络设备 $self->{host};" if $self->{debug};    # 打印调试信息

  return {success => 1};                                        # 返回成功
}

#------------------------------------------------------------------------------
# connect: 建立与设备的连接
# 描述：此方法使用用户名和密码连接到网络设备，并处理登录过程中的各种提示。
# 参数：
#   - $args: 可选的连接参数，默认为空字符串
# 返回：连接状态，1 表示成功，-1 表示失败
#------------------------------------------------------------------------------
sub connect {
  my ($self, $args) = @_;

  # 设置默认连接参数
  $args //= "";

  # 从环境变量加载用户信息和主机信息
  my $username = $self->{username} || $ENV{PDK_USERNAME};    # 用户名
  my $password = $self->{password} || $ENV{PDK_PASSWORD};    # 密码
  my $debug    = $self->{debug};                             # DEBUG级别
  my $prompt   = $self->{prompt};                            # 普通提示符
  my $enPrompt = $self->{enPrompt};                          # 增强提示符（enable prompt）

  # 验证用户名和密码
  croak("请正确提供设备登录所需账户密码凭证，或设置对象的环境变量！") unless $username && $password;

  # 如果没定义调试级别，从环境变量加载调试级别
  $debug = $ENV{PDK_DEBUG} if $debug == 0 && $ENV{PDK_DEBUG};

  # 创建 Expect 对象并配置
  my $exp = Expect->new();                                   # 初始化一个新的 Expect 对象
  $exp->raw_pty(1);                                          # 设置为原始伪终端模式
  $exp->restart_timeout_upon_receive(1);                     # 接收时重启超时
  $exp->debug(0);                                            # 禁用调试输出
  $exp->log_stdout(0);                                       # 禁用标准输出日志

  # 将 Expect 对象存储在当前对象的 exp 属性中
  $self->{exp} = $exp;

  # 如果启用调试，调用 _debug 方法进行调试信息输出
  if ($debug) {
    if ($debug == 3) {

      # 从环境变量加载并更新当前用户名和密码、调试级别
      $self->{username} = $username if $username ne $self->{username};
      $self->{username} = $username if $username ne $self->{username};
    }

    # 从环境变量加载并更新当前调试级别
    $self->{debug} = $debug if $debug ne $self->{debug};

    # 开启调试信息打印和记录
    $self->_debug($debug);
  }

  # 执行连接命令
  my $command = $self->_spawn_command($args);

  # 尝试连接设备，失败时抛出错误，包含命令名称和错误信息
  $exp->spawn($command) or croak "执行[connect/连接脚本准备阶段]，Cannot spawn $command: $!";

  # 处理连接过程中的期望
  my @ret = $exp->expect(
    15,    # 设置等待时间为15秒
    [
      qr/to continue conne/mi => sub {    # 检测到继续连接的提示
        $exp->send("yes\n");              # 发送“yes”以继续连接
        exp_continue;                     # 继续处理其他匹配项
      }
    ],
    [
      qr/assword:\s*$/mi => sub {         # 检测到密码提示
        $exp->send("$password\n");        # 发送密码
      }
    ],
    [
      qr/(name|ogin|user):\s*$/mi => sub {    # 检测到用户名提示
        $exp->send("$username\n");            # 发送用户名
        exp_continue;                         # 继续处理其他匹配项
      }
    ],
    [
      qr/REMOTE HOST IDENTIFICATION HAS CHANGED!/mi => sub {    # 检测到身份验证变化的提示
        croak("IDENTIFICATION CHANGED!");                       # 抛出错误
      }
    ],
    [
      eof => sub {                                              # 处理意外的会话关闭
        croak("执行[connect/尝试登录设备阶段]，与设备 $self->{host} 会话丢失，连接被意外关闭！具体原因：\n" . $exp->before());
      }
    ],
    [
      timeout => sub {                                          # 处理会话超时
        croak("执行[connect/尝试登录设备阶段]，与设备 $self->{host} 会话超时，请检查网络连接或服务器状态！具体原因：\n" . $exp->before());
      }
    ]
  );

  # 检查连接结果
  croak($ret[3]) if defined $ret[1];

  # 验证登录状态
  @ret = $exp->expect(
    10,    # 设置等待时间为10秒
    [
      qr/sername|assword:\s*$/mi => sub {    # 检测到用户名或密码提示
        $self->{status} = -1;                       # 设置状态为-1表示登录失败
        croak("username or password is wrong!");    # 抛出错误
      }
    ],
    [
      qr/$prompt/m => sub {                         # 检测到预期的提示符
        $self->{status} = 1;                        # 设置状态为1表示登录成功
      }
    ],
    [
      eof => sub {                                  # 处理意外的会话关闭
        croak("执行[connect/验证登录状态]，与设备 $self->{host} 会话丢失，连接被意外关闭！具体原因：\n" . $exp->before());
      }
    ],
    [
      timeout => sub {                              # 处理会话超时
        croak("执行[connect/验证登录状态]，与设备 $self->{host} 会话超时，请检查网络连接或服务器状态！具体原因：\n" . $exp->before());
      }
    ]
  );

  # 如果增强提示符不为空且当前匹配不等于增强提示符，调用 enable 方法
  if ($enPrompt && $exp->match() =~ /$enPrompt/m) {
    say "\n[debug] 尝试切换到特权模式;" if $self->{debug};

    # 尝试切换到特权模式
    eval { $self->{enabled} = $self->enable(); };
    if ($@ || $self->{enabled} == 0) {
      croak "username or enPassword is wrong!";    # 切换特权模式失败，抛出错误信息
    }
  }

  return $self->{status};                          # 返回执行结果
}

#------------------------------------------------------------------------------
# send: 发送命令到设备
# 说明：没有直接拼接回车键 ("$command\n")，比如山石防火墙交互命令，只需要输入 [Y] 即可
# 描述：此方法将指定的命令发送到设备，并在调试模式下打印发送的命令。
# 参数：
#   - $command: 要发送的命令
# 返回：无
#------------------------------------------------------------------------------
sub send {
  my ($self, $command) = @_;

  # 获取存储在对象中的 Expect 对象
  my $exp = $self->{exp};

  # 如果开启调试模式，打印发送的命令
  if ($self->{debug}) {
    my $cmd = $command;
    chomp($cmd);
    say "\n[debug] send command: ($cmd);";    # 打印发送的命令
  }

  $exp->send($command);                       # 将命令发送到 Expect 对象
}

#------------------------------------------------------------------------------
# enable: 切换到特权模式
# 描述：此方法通过发送特权命令和密码来切换设备到特权模式。
# 参数：无
# 返回：
#   - 1: 登录成功
#   - 0: 登录失败
#------------------------------------------------------------------------------
sub enable {
  my $self = shift;                           # 获取对象实例

  # 从对象中检索必要的参数
  my $username  = $self->{username};          # 用户名
  my $enPasswd  = $self->{enPassword};        # 特权密码
  my $enCommand = $self->{enCommand};         # 特权模式命令
  my $prompt    = $self->{prompt};            # 提示符

  # 支持从环境变量加载特权密码，使用 password 兜底
  $enPasswd ||= $ENV{PDK_ENPASSWORD} || $self->{password};

  # 获取 Expect 对象，发送命令以切换到特权模式
  my $exp = $self->{exp};
  $exp->send("$enCommand\n");

  # 处理输入密码的期望
  my @ret = $exp->expect(
    10,
    [
      qr/assword:\s*$/mi => sub {
        $exp->send("$enPasswd\n");    # 输入特权密码
      }
    ],
    [
      qr/(ername|ogin|user):\s*$/mi => sub {
        $exp->send("$username\n");    # 输入用户名并继续监听
        exp_continue;
      }
    ],
    [
      eof => sub {
        croak("执行[enable/尝试切换特权模式]，与设备 $self->{host} 会话丢失，连接被意外关闭！具体原因：\n" . $exp->before());
      }
    ],
    [
      timeout => sub {
        croak("执行[enable/尝试切换特权模式]，与设备 $self->{host} 会话超时，请检查网络连接或服务器状态！具体原因：\n" . $exp->before());
      }
    ],
  );

  # 检查是否出现问题
  return 0 if defined $ret[1];    # 返回失败

  # 观察是否正常登录设备
  @ret = $exp->expect(
    10,                           # 设置等待时间为10秒
    [
      qr/sername|assword:\s*$/mi => sub {    # 检测到用户名或密码提示
        $self->{enabled} = -1;                        # 登录失败，设置状态
        croak("username or enPassword is wrong!");    # 抛出错误
      }
    ],
    [
      qr/(\^|Bad secrets|Permission denied|invalid)/mi => sub {    # 检测到错误提示
        $self->{enabled} = -1;                                     # 登录失败，设置状态
        croak("username or enPassword is wrong!");                 # 抛出错误
      }
    ],
    [
      qr/$prompt/m => sub {                                        # 检测到预期的提示符，表示登录成功
        $self->{enabled} = 1;                                      # 登录成功，设置状态
      }
    ],
    [
      eof => sub {                                                 # 处理意外的会话关闭
        croak("执行[enable/检查是否成功切换特权模式]，与设备 $self->{host} 会话丢失，连接被意外关闭！具体原因：\n" . $exp->before());
      }
    ],
    [
      timeout => sub {                                             # 处理会话超时
        croak("执行[enable/检查是否成功切换特权模式]，与设备 $self->{host} 会话超时，请检查网络连接或服务器状态！具体原因：\n" . $exp->before());
      }
    ],
  );

  # 返回操作结果
  return $self->{enabled};
}

#------------------------------------------------------------------------------
# execCommands: 执行一系列命令并处理结果
# 描述：此方法检查是否已登录设备，发送命令并收集执行结果。
# 参数：
#   $commands - 命令列表，数组参考
# 返回：包含执行结果和状态的哈希引用
#------------------------------------------------------------------------------
sub execCommands {
  my ($self, $commands) = @_;

  # 配置下发前检查是否已经登录设备
  if (not defined $self->{exp} and $self->{status} == 0) {
    my $login = $self->login();
    if ($login->{success} == 0) {
      my $snapshot = "执行[execCommands/下发配置前自动登录设备]，尝试（首次）登录设备失败。";
      if (my $exp = $self->{exp}) {
        $snapshot .= "，相关异常：\n" . $exp->before() . $exp->match() . $exp->after();
      }
      return {success => 0, failCommand => join(", ", @{$commands}), snapshot => $snapshot, reason => $login->{reason}};
    }
  }
  elsif ($self->{exp} and $self->{status} == -1) {
    my $exp      = $self->{exp};
    my $snapshot = "先前捕捉到的交互信息：" . $exp->before() . $exp->match() . $exp->after();
    my $reason   = "执行[execCommands/下发配置前自动登录设备]，尝试（非首次）登录设备失败。";
    return {success => 0, failCommand => join(", ", @{$commands}), snapshot => $snapshot, reason => $reason};
  }

  # 初始化变量并提取模块定义的错误码
  my $result = $self->{exp} ? $self->{exp}->match() . $self->{exp}->after() : "";    # 获取初始匹配结果
  my $errors = $self->errCodes();                                                    # 获取错误码定义

  # 检查环境变量 PDK_CATCH_ERROR 是否为有效数字，并赋值
  if ($ENV{PDK_CATCH_ERROR} =~ /^\d+$/) {

    # 仅允许值为1，其他情况设为0
    $self->{catchError} = ($ENV{PDK_CATCH_ERROR} == 1) ? 1 : 0;
  }

  # 遍历命令
  for my $cmd (@{$commands}) {

    # 自动跳过注释行和空白行
    next if $cmd =~ /^\s*$/;     # 跳过空行
    next if $cmd =~ /^[#!;]/;    # 跳过注释行

    # 用于存储命令执行结果
    my $buff = "";

    # 发送命令并尝试自动交互
    $self->send("$cmd\n");       # 发送命令
    eval {
      $buff = $self->waitfor();    # 等待命令执行完成
    };

    # 处理异常
    if ($@) {
      chomp($@);
      my $snapshot = $result . $buff;
      my $reason   = "执行[execCommands/等待脚本回显自动交互]，捕捉到异常: \n" . $@;
      return {success => 0, failCommand => $cmd, reason => $reason, snapshot => $snapshot};
    }

    # 拦截脚本下发期间异常
    if ($self->{catchError}) {
      for my $error (@{$errors}) {
        if ($buff =~ /$error/i) {
          my $snapshot = "执行[execCommands/异常码字典拦截]，捕捉到异常: \n" . $result . $buff;
          return {success => 0, failCommand => $cmd, reason => $error, snapshot => $snapshot};
        }
      }
    }

    # 如无异常，则拼接脚本
    $result .= $buff;    # 追加结果
  }

  return {success => 1, result => $result};    # 返回成功及结果
}

#------------------------------------------------------------------------------
# write_file: 将配置写入指定文件
# 描述：此方法将配置内容写入以当前日期命名的目录中的文件。
# 参数：
#   $config - 要写入的配置内容
#   $name   - 可选，文件名（默认为主机名加 ".cfg"）
# 返回：无
#------------------------------------------------------------------------------
sub write_file {
  my ($self, $config, $name) = @_;

  # 早期异常拦截，确保配置信息非空
  croak("必须提供非空配置信息") unless $config;

  # 默认文件名为主机名加 ".cfg"
  $name //= $self->{host} . ".cfg";

  # 拼接工作目录
  my $workdir = "$self->{workdir}/$self->{month}/$self->{date}";

  # 打印调试信息
  if ($self->{debug}) {
    say "\n[debug] 准备将配置文件写入工作目录: ($workdir)";
  }

  # 创建日期目录（如果不存在）
  use File::Path qw(make_path);
  make_path($workdir) unless -d $workdir;    # 仅在目录不存在时创建

  # 拼接完整的文件路径
  my $filename = "$workdir/$name";

  # 将配置写入文本文件
  open(my $fh, '>', $filename) or croak "无法打开文件 $filename 进行写入: $!";    # 打开文件失败时抛出异常
  print $fh $config            or croak "写入文件 $filename 失败: $!";        # 写入文件失败时抛出异常
  close($fh)                   or croak "关闭文件句柄 $filename 失败: $!";      # 关闭文件失败时抛出异常

  # 打印调试信息
  if ($self->{debug}) {
    say "[debug] 已将配置文件写入文本文件: $filename !";
  }

  return {success => 1};                                                # 返回成功状态
}

#------------------------------------------------------------------------------
# _spawn_command: 生成会话连接命令
# 描述：根据用户名、主机、端口和协议动态生成连接命令。
# 说明：设备登录 connect() 已检测并拦截无效账户密码，此处无需重复检测。
# 参数：
#   $args - 可选，额外的命令行参数
# 返回：生成的连接命令字符串
#------------------------------------------------------------------------------
sub _spawn_command {
  my ($self, $args) = @_;

  # 初始化变量：获取用户名、主机、端口和协议
  my $user  = $self->{username};    # 用户名
  my $host  = $self->{host};        # 主机
  my $port  = $self->{port};        # 端口
  my $proto = $self->{proto};       # 协议

  # 支持从环境变量加载用户名
  $user ||= $ENV{PDK_USERNAME};

  # 动态生成会话连接参数
  $args //= "";                     # 如果 $args 未定义，则初始化为空字符串
  my $command;

  # 根据协议和端口生成不同的命令
  if ($port) {
    if ($proto =~ /telnet/i) {
      $command = qq{$proto $args -l $user $host $port};    # telnet命令
    }
    elsif ($proto =~ /ssh/i) {
      $command = qq{$proto $args -l $user $host -p $port};    # ssh命令
    }
  }
  else {
    $command = qq{$proto $args -l $user $host};               # 没有端口时的命令
  }

  # 打印日志
  say "[debug] 已生成登录设备的脚本: $command" if $self->{debug};

  return $command;                                            # 返回生成的命令
}

#------------------------------------------------------------------------------
# _debug: 启用调试模式并记录日志
# 描述：此方法根据指定的调试级别配置日志记录，并创建相应的日志目录。
# 参数：
#   $level - 可选，调试级别（默认为 1），范围为 1 到 3
# 说明：[1(基础的调试信息打印), 2(调试信息打印到回显), 3(记录更加详细的排错信息)]
# 返回：无
#------------------------------------------------------------------------------
sub _debug {
  my ($self, $level) = @_;

  # 修正 debug 级别，限制在 1 到 3 之间
  $level //= 1;                # 如果未提供级别，默认为 1
  $level = 3 if $level > 3;    # 最大级别为 3

  # 拼接工作目录
  my $workdir = "$self->{workdir}/debug/$self->{date}";

  # 检查日期目录是否存在，如果不存在则创建
  use File::Path qw(make_path);
  make_path($workdir) unless -d $workdir;    # 创建目录，若已存在则不执行

  # 获取执行上下文对象
  my $exp = $self->{exp};

  # 根据调试级别配置日志记录
  if ($level == 2) {
    say '[debug] 当前 debug 级别将打开日志记录功能，并同步脚本执行回显到控制台！';
    $exp->log_stdout(1);    # 启用标准输出日志
  }
  elsif ($level == 3) {
    say '[debug] 当前 debug 级别将打开日志记录功能，观察更详细的 Expect 信息！';
    $exp->log_stdout(1);    # 启用标准输出日志
    $exp->debug($level);    # 开启更高级别的调试日志
  }

  # 设置日志文件路径
  $exp->log_file("$workdir/$self->{host}.log");    # 指定日志文件
}

1;                                                 # 返回真值，以表明模块加载成功
