package Firewall::Policy::Designer::Hillstone;

#------------------------------------------------------------------------------
# 加载扩展模块
#------------------------------------------------------------------------------
use Moose;
use namespace::autoclean;

#------------------------------------------------------------------------------
# 加载项目模块
#------------------------------------------------------------------------------
use Firewall::Utils::Ip;

#------------------------------------------------------------------------------
# 继承 Firewall::Policy::Designer::Role 方法和属性
#------------------------------------------------------------------------------
with 'Firewall::Policy::Designer::Role';

#------------------------------------------------------------------------------
# 山石防火墙策略设计相关模块
#------------------------------------------------------------------------------
sub createRule {
  my $self     = shift;
  my $action   = $self->searcherReportFwInfo->action->{new};
  my $mappings = $self->checkAndCreateAddrOrSrvOrNat($action);
  my $scheduleName;
  my $schedule = $self->searcherReportFwInfo->schedule;
  $scheduleName = $self->createSchedule($schedule) if $schedule->{enddate} ne 'always';
  my ($fromZone, $toZone) = ($self->searcherReportFwInfo->fromZone, $self->searcherReportFwInfo->toZone);

  # 初始化策略数组对象
  my @commands;
  push @commands, "rule top";
  push @commands, "action permit";
  push @commands, "src-zone $fromZone" if $fromZone ne 'any';
  push @commands, "dst-zone $toZone"   if $toZone ne 'any';

  # 添加策略成员对象
  for my $element (keys %{$mappings}) {
    if ($element eq 'src') {
      for my $host (@{$mappings->{$element}}) {
        push @commands, "src-addr $host";
      }
    }
    elsif ($element eq 'dst') {
      for my $host (@{$mappings->{$element}}) {
        push @commands, "dst-addr $host";
      }
    }
    elsif ($element eq 'srv') {
      for my $srv (@{$mappings->{$element}}) {
        push @commands, "service $srv";
      }
    }
  }
  push @commands, "schedule $scheduleName" if defined $scheduleName;
  push @commands, "exit";
  $self->addCommands(@commands);
}

#------------------------------------------------------------------------------
# 修改防火墙策略
#------------------------------------------------------------------------------
sub modifyRule {
  my $self    = shift;
  my $nameMap = $self->checkAndCreateAddrOrSrvOrNat($self->searcherReportFwInfo->action->{add});
  if (my $param = $self->searcherReportFwInfo->action->{new}) {
    for my $type (keys %{$param}) {
      if ($type eq 'natDst' or $type eq 'natSrc') {
        $self->createNat($param->{$type}, $type);
      }
    }
  }

  # 初始化防火墙策略数组
  my $policyId = $self->{searcherReportFwInfo}{ruleObj}{policyId};
  my @commands;
  push @commands, "rule $policyId";
  for my $type (keys %{$nameMap}) {
    if ($type eq 'src') {
      for my $host (@{$nameMap->{$type}}) {
        push @commands, "src-addr $host";
      }
    }
    elsif ($type eq 'dst') {
      for my $host (@{$nameMap->{$type}}) {
        push @commands, "dst-addr $host";
      }
    }
    elsif ($type eq 'srv') {
      for my $srv (@{$nameMap->{$type}}) {
        push @commands, "service $srv";
      }
    }
  }
  push @commands, "exit";
  $self->addCommands(@commands);
}

#------------------------------------------------------------------------------
# 创建静态地址转换关系
#------------------------------------------------------------------------------
sub createStaticNat {
  my ($self, $natInfo) = @_;
  my $natIp = $natInfo->{natInfo}{natIp};

  my @commands;
  push @commands, "nat";
  push @commands, "bnatrule virtual ip $natIp real ip $natInfo->{realIp}";
  push @commands, "exit";
  $self->addCommands(@commands);
}

#------------------------------------------------------------------------------
# 创建动态地址转换关系
#------------------------------------------------------------------------------
sub createDyNat {
  my ($self, $param) = @_;

  my @commands;
  for my $type (keys %{$param}) {
    for my $natIps (values %{$param->{$type}}) {
      my $natInfo = (values %{$natIps})[0]->{natInfo};
      if ($type eq 'natSrc') {
        if (not defined $natInfo->{natIp}) {
          push @commands, "nat";
          for my $natIp (keys %{$natIps}) {
            push @commands, "snatrule from $natIp to any service any trans-to eif-ip mode dynamicport";
          }
        }
        else {
          push @commands, "nat";
          for my $natIp (keys %{$natIps}) {
            my $nat = $natInfo->{natIp};
            push @commands, "snatrule from $natIp to any service any trans-to $nat mode dynamicport";
          }
        }
      }
      elsif ($type eq 'natDst') {
        push @commands, "nat";
        for my $natIp (keys %{$natIps}) {
          my $nat = $natInfo->{natIp};
          push @commands, "dnatrule from any to $nat service any trans-to $natIp";
        }
      }
    }
  }
  push @commands, qq{exit};
  $self->addCommands(@commands);
}

#------------------------------------------------------------------------------
# 创建源地址对象
#------------------------------------------------------------------------------
sub createSrc {
  my ($self, $addr) = @_;
  return $self->createAddress($addr);
}

#------------------------------------------------------------------------------
# 创建目的地址对象
#------------------------------------------------------------------------------
sub createDst {
  my ($self, $addr) = @_;
  if (my $param = $self->{searcherReportFwInfo}{action}{new}) {
    for my $type (keys %{$param}) {
      if ($type eq 'natDst') {
        for (keys %{$param->{$type}}) {
          if ($addr eq $_) {
            return "vip_$addr";
          }
        }
      }
    }
  }
  return $self->createAddress($addr);
}

#------------------------------------------------------------------------------
# 创建端口对象
#------------------------------------------------------------------------------
sub createSrv {
  my ($self, $srv) = @_;
  return $self->createService($srv);
}

#------------------------------------------------------------------------------
# 创建地址对象
#------------------------------------------------------------------------------
sub createAddress {
  my ($self,        $addr) = @_;
  my ($ip,          $mask) = split('/', $addr);
  my ($addressName, $ipString, $ipMin, $ipMax);

  if (not defined $mask) {
    if ($ip =~ /(\d+\.)(\d+\.)(\d+\.)(\d+)-(\d+)/) {
      $addressName = "RANGE_$ip";
      ($ipMin, $ipMax) = ($1 . $2 . $3 . $4, $1 . $2 . $3 . $5);
    }
  }
  elsif ($mask == 32) {
    $ipString    = $ip;
    $addressName = "HOST_$ip";
  }
  elsif ($mask == 0) {
    return 'any';
  }
  else {
    $ipString    = Firewall::Utils::Ip->new->getNetIpFromIpMask($ip, $mask);
    $addressName = "NET_$ipString/$mask";
  }

  my @commands;
  push @commands, "address $addressName";
  push @commands, "ip $ipString/$mask"  if defined $mask;
  push @commands, "range $ipMin $ipMax" if defined $ipMin;
  push @commands, "exit";
  $self->addCommands(@commands);
  return $addressName;
}

#------------------------------------------------------------------------------
# 创建服务端口
#------------------------------------------------------------------------------
sub createService {
  my ($self,     $srv)  = @_;
  my ($protocol, $port) = split('/', $srv);
  $protocol = lc $protocol;
  return if $protocol ne 'tcp' and $protocol ne 'udp';

  my ($serviceName, $dstPort);
  if ($port =~ /^(?<portMin>\d+)\-(?<portMax>\d+)$/o) {
    $serviceName = uc($protocol) . "_" . $+{portMin} . "_" . $+{portMax};
    $dstPort     = $+{portMin} . "-" . $+{portMax};
  }
  elsif ($port =~ /^\d+$/o) {
    $serviceName = uc($protocol) . "_" . $port;
    $dstPort     = $port;
  }
  else {
    confess "ERROR: $port is not a port";
  }

  my @commands;
  push @commands, "service $serviceName";
  push @commands, "$protocol dst-port $dstPort";
  push @commands, "exit";
  $self->addCommands(@commands);
  return $serviceName;
}

#------------------------------------------------------------------------------
# 创建计划任务
#------------------------------------------------------------------------------
sub createSchedule {
  my ($self, $schedule) = @_;
  my @commands;
  my ($syear, $smon, $sday, $shh, $smm) = split('[ :-]', $schedule->{startdate}) if defined $schedule->{startdate};
  my ($year,  $mon,  $day,  $hh,  $mm)  = split('[ :-]', $schedule->{enddate});

  push @commands, "schedule \"$year-$mon-$day\"";
  if (defined $schedule->{startdate}) {
    push @commands, "absolute start $smon/$sday/$syear $shh:$smm end $mon/$day/$year $hh:$mm";

  }
  else {
    push @commands, "absolute end $mon/$day/$year $hh:$mm";
  }
  push @commands, "exit";
  $self->addCommands(@commands);
  return "$year-$mon-$day";
}

__PACKAGE__->meta->make_immutable;
1;
