package Firewall::Policy::Designer;

# ABSTRACT: turns baubles into trinkets

#------------------------------------------------------------------------------
# 加载系统模块，辅助构造函数功能和属性
#------------------------------------------------------------------------------
use Moose;
use namespace::autoclean;

# no warnings 'uninitialized';
use Try::Tiny;
use Time::HiRes;

#------------------------------------------------------------------------------
# 加载项目模块
#------------------------------------------------------------------------------
use Firewall::Policy::Searcher::Report;

#------------------------------------------------------------------------------
# Firewall::Policy::Designer 模块方法属性
#------------------------------------------------------------------------------
has dbi => (is => 'ro', does => 'Firewall::DBI::Role', required => 1,);

has searcherReport => (is => 'ro', isa => 'Firewall::Policy::Searcher::Report', required => 1,);

has designerReport => (is => 'ro', isa => 'ArrayRef[HashRef[Ref]]', default => sub { [] },);

#------------------------------------------------------------------------------
# 生成防火墙策略
#------------------------------------------------------------------------------
sub design {
  my $self = shift;
  my %index;

  for my $report (@{$self->{searcherReport}{FwInfos}}) {
    my $fwId        = $report->{fwId};
    my $vendor      = ucfirst lc $report->{fwType};
    my $fwName      = $report->{fwName};
    my $policyState = $self->getPolicyState($report->type, $report->exist);
    my $newCommands = $self->generateConfig($vendor, $report);
    if ($policyState eq 'allExist' and $newCommands->{content} ne '') {
      $policyState = 'partExist';
    }

    # 判断是否已有防火墙策略
    my $existCommands = {};
    if (defined $report->{exist}) {
      @{$existCommands}{/ src dst srv /} = @{$report->{exist}}{/ src dst srv /};
      $existCommands->{content} = $report->{ruleObj}{content};
    }

    # 输出策略设计报告
    if (not defined $index{$fwId}) {
      push @{$self->{designerReport}},
        {
        fwId           => $fwId,
        fwName         => $fwName,
        fwType         => $vendor,
        policyState    => $policyState,
        policyContents => [{
          src   => $report->{srcMap},
          dst   => $report->{dstMap},
          srv   => $report->{srvMap},
          new   => $newCommands,
          exist => $existCommands,
        }]
        };
      $index{$fwId} = @{$self->designerReport} - 1;
    }
    else {
      if (
        (
          $self->designerReport->[$index{$fwId}]{policyState} eq 'allExist'
          and ($policyState eq 'nonExist' or $policyState eq 'partExist')
        )
        or (
          (
               $self->designerReport->[$index{$fwId}]{policyState} eq 'nonExist'
            or $self->designerReport->[$index{$fwId}]{policyState} eq 'partExist'
          )
          and $policyState eq 'allExist'
        )
        )
      {
        $self->designerReport->[$index{$fwId}]->{policyState} = 'partExist';
      }

      for (keys %{$report->srcMap}) {
        $self->designerReport->[$index{$fwId}]->{policyContents}->[0]->{src}{$_} = $report->srcMap->{$_};
      }
      for (keys %{$report->dstMap}) {
        $self->designerReport->[$index{$fwId}]->{policyContents}->[0]->{dst}{$_} = $report->dstMap->{$_};
      }
      for (keys %{$report->srvMap}) {
        $self->designerReport->[$index{$fwId}]->{policyContents}->[0]->{srv}{$_} = $report->srvMap->{$_};
      }

      $self->designerReport->[$index{$fwId}]->{policyContents}->[0]->{new}{content} .= $newCommands->{content};
      my $newConfig = $self->designerReport->[$index{$fwId}]->{policyContents}->[0]->{exist}{content};
      $newConfig = '' unless (defined $newConfig);

      # FIXME 优化已存在策略显示
      if (defined $existCommands->{content} and $newConfig !~ /\Q$existCommands->{content}\E/) {
        $self->designerReport->[$index{$fwId}]->{policyContents}->[0]->{exist}{content} .= $existCommands->{content};
      }
    }
  }
  return $self->designerReport;
}

#------------------------------------------------------------------------------
# 生成防火墙策略
#------------------------------------------------------------------------------
sub generateConfig {
  my ($self, $vendor, $searcherReportFwInfo) = @_;
  my $config;
  my $meta;
  my $plugin = __PACKAGE__ . "::$vendor";

  # 动态加载防火墙插件
  eval("use $plugin; \$meta = $plugin->new(searcherReportFwInfo=>\$searcherReportFwInfo, dbi=>\$self->dbi)");

  # use Firewall::Policy::Designer::Huawei;
  # $meta = Firewall::Policy::Designer::Huawei->new(searcherReportFwInfo => $searcherReportFwInfo, dbi => $self->dbi);
  confess $@ if !!$@;
  $config->{content} = $meta->design;

  # 检查是否已有策略，如需新增或修改策略，明确具体元素信息
  if ($searcherReportFwInfo->type ne 'ignore') {
    for my $action (keys %{$searcherReportFwInfo->action}) {
      for my $element (qw/ src dst srv /) {
        my $info = $searcherReportFwInfo->{action}{$action}{$element};
        if (defined $info) {
          $config->{$element} = join(',', keys %{$info});
        }
        else {
          my $map = $element . 'Map';
          $config->{$element} = join(',', keys %{$searcherReportFwInfo->$map});
        }
      }
    }
  }
  return $config;
}

#------------------------------------------------------------------------------
# 查询策略状态
#------------------------------------------------------------------------------
sub getPolicyState {
  my ($self, $type, $exist) = @_;
  my %mapping = (ignore => 'allExist', modify => 'partExist', new => 'nonExist');
  confess "ERROR: $type correlative policyState dose not exist.\n" unless (defined $mapping{$type});

  my $result = $mapping{$type};
  if ($type eq 'modify' and not defined $exist) {
    $result = 'nonExist';
  }
  return $result;
}

__PACKAGE__->meta->make_immutable;
1;
