package Firewall::Config::Content::Static;

#------------------------------------------------------------------------------
# 加载扩展模块
#------------------------------------------------------------------------------
use Moose;
use namespace::autoclean;
use Encode;
use Digest::MD5;
use Firewall::Utils::Date;

#------------------------------------------------------------------------------
# 定义 Firewall::Config::Content::Static 方法属性
#------------------------------------------------------------------------------
# 解析配置文件为字符串数组
has config => (is => 'ro', isa => 'ArrayRef[Str]', required => 1,);

#------------------------------------------------------------------------------
# confContent 加载配置的详细信息 | 定义的懒加载
#------------------------------------------------------------------------------
has confContent => (is => 'ro', isa => 'Str', lazy => 1, builder => '_buildConfContent',);

#------------------------------------------------------------------------------
# cursor 配置解析游标，代表配置行数 | 缺省为0，与数组缺省值对应
#------------------------------------------------------------------------------
has cursor => (is => 'ro', isa => 'Int', default => 1,);

#------------------------------------------------------------------------------
# 继承 Firewall::Config::Content::Role 方法属性并具体实现其约束方法
#------------------------------------------------------------------------------
with 'Firewall::Config::Content::Role';

#------------------------------------------------------------------------------
# 改写 confSign 属性并设置懒加载，提供构建方法
#------------------------------------------------------------------------------
has '+confSign' => (required => 0, lazy => 1, builder => '_buildConfSign',);

# 配置解析时间，多次解析相同配置，该属性可以明确什么时候执行了解析动作
has '+timestamp' => (isa => 'Str', default => sub { Firewall::Utils::Date->new->getLocalDate });

#------------------------------------------------------------------------------
# _buildConfSign 通过HASH判断文件是否发生变化
#------------------------------------------------------------------------------
sub _buildConfSign {
  my $self = shift;
  return Digest::MD5::md5_hex(encode_utf8 $self->confContent);
}

#------------------------------------------------------------------------------
# _buildConfContent 拼接 @config 为字符串
#------------------------------------------------------------------------------
sub _buildConfContent {
  my $self    = shift;
  my $content = join("\n", $self->config->@*);
  return $content;
}

#------------------------------------------------------------------------------
# _buildLineParsedFlags 生成配置解析标志位，为每行配置设置初始状态为0
#------------------------------------------------------------------------------
sub _buildLineParsedFlags {
  my $self = shift;
  return [map {0} (0 .. $self->config->$#*)];
}

#------------------------------------------------------------------------------
# goToHead 跳转 Head 头部函数，引入游标的概念
#------------------------------------------------------------------------------
sub goToHead {
  my $self = shift;
  $self->{cursor} = 0;
}

#------------------------------------------------------------------------------
# 跳转 nextLine 函数，跳转下一行
#------------------------------------------------------------------------------
sub nextLine {
  my $self = shift;

  # 游标边界
  if ($self->cursor < $self->config->$#*) {

    # 游标自增
    $self->{cursor}++;
  }
}

#------------------------------------------------------------------------------
# 获取上一行的配置信息
#------------------------------------------------------------------------------
sub prevLine {
  my $self = shift;

  # 游标边界条件处理 | 游标大于 0 才有上一行的概念
  if ($self->cursor > 0) {
    $self->{cursor}--;
  }
}

#------------------------------------------------------------------------------
# nextUnParsedLine 获取后续非解析配置 | 用于 while 循环体
#------------------------------------------------------------------------------
sub nextUnParsedLine {
  my $self = shift;
  my $result;

  # 未到达配置底部且已解析当前行配置，则自动跳过本次解析
  # 自顶向下解析 => 保证顺序解析 | cursor
  # 保证只解析从未解析的条目 => ParseFlag 与 cursor 绑定 对应同一行配置
  while ($self->cursor < $self->config->$#* and $self->getParseFlag == 1) {
    $self->{cursor}++;
  }

  # 获取下一行未解析配置
  if ($self->cursor < $self->config->$#*) {

    # 为当前行设置 flag（代表已经解析），同时游标自增
    $self->setParseFlag;
    $self->{cursor}++;

    # 下一行配置 | 正常情况下是没有解析过的
    $result = $self->config->[$self->cursor];

    # 参数检查：确保找到未解析的非空配置行
    while (not defined $result or $result =~ /^\s*$/) {

      # 即便是空行也打上已解析的标记，游标向下走
      $self->setParseFlag;
      $self->{cursor}++;

      # 参数检查：将游标移动到未解析的配置行
      while ($self->cursor < $self->config->$#* and $self->getParseFlag == 1) {
        $self->{cursor}++;
      }

      # 获取非空行且未解析的配置信息
      if ($self->cursor <= $self->config->$#*) {
        $result = $self->config->[$self->cursor];
      }
      else {
        return;
      }
    }
  }

  # 去除首尾空白字符串
  chomp $result if defined $result;

  # 将字符串解码为 utf8 返回 | 从右往左运算
  return $result;
}

#------------------------------------------------------------------------------
# 回退游标位，返回上一个游标号
#------------------------------------------------------------------------------
sub backtrack {
  my $self = shift;
  if ($self->cursor > 0) {

    # 将游标跳上一级
    $self->{cursor}--;

    # 同时设置上一级游标未解析状态
    $self->setParseFlag(0);
    return 1;
  }
}

#------------------------------------------------------------------------------
# ignore 忽略逻辑，先调转到上一行游标同时解析到下一行游标
#------------------------------------------------------------------------------
sub ignore {
  my $self = shift;
  $self->backtrack and $self->nextLine;
}

#------------------------------------------------------------------------------
# 获取所有未解析过的配置行
#------------------------------------------------------------------------------
sub getUnParsedLines {
  my $self = shift;

  # 过滤未解析的配置项并打包
  my $unParsedLines = join('', map { $self->config->[$_] } grep { $self->lineParsedFlags->[$_] == 0 } (0 .. $self->config->$#*));

  # 返回计算结果
  return $unParsedLines;
}

#------------------------------------------------------------------------------
# 获取配置数组解析状态码
#------------------------------------------------------------------------------
sub getParseFlag {
  my $self = shift;
  if ($self->cursor >= 0 and $self->cursor <= $self->config->$#*) {
    return $self->lineParsedFlags->[$self->cursor];
  }
}

#------------------------------------------------------------------------------
# 设置解析状态标志位 | 与游标实际位置绑定
#------------------------------------------------------------------------------
sub setParseFlag {
  my ($self, $flag) = @_;
  if ($self->cursor >= 0 and $self->cursor <= $self->config->$#*) {
    $self->lineParsedFlags->[$self->cursor] = $flag // 1;
  }
}

__PACKAGE__->meta->make_immutable;
1;
