
package Continuity::Mapper;

use strict;
use Coro::Cont;

=head1 NAME

Continuity::Mapper - Map a request onto a continuation

=head1 DESCRIPTION

This is the continuation dictionary and mapper. Given an HTTP::Request it
returns a continuation.

=head1 METHODS

=over

=item $server = new Continuity::Mapper(...)

Create a new continuation mapper

=cut

sub new {
  my $this = shift;
  my $class = ref($this) || $this;
  my $self = {};
  $self = {%$self, @_};
  bless $self, $class;

  # if new_cont_sub is undef then the default mapper will use &::main, btw
  $self->set_cont_maker_sub($self->{new_cont_sub});

  return $self;
}

sub debug {
  my ($self, $level, $msg) = @_;
  if($level >= $self->{debug}) {
   print STDERR "$msg\n";
  }
}

# Sessions are currently generated by starting with zero and going up from
# there... not a very good way!
my $sessionIdCounter;

sub getSession {
  my ($self, $request) = @_;
  #print "Headers: " . $request->as_string();

  my $cookieHeader = $request->header('Cookie');
  $self->debug(3, "Cookies: $cookieHeader");
  
  $self->debug(3, "sessionIdCounter: $sessionIdCounter");
  if($cookieHeader =~ /sessionid=(\d+)/) {
    $self->debug(3, "Found sessionId!");
    return $1;
  }
  return $sessionIdCounter++;
}

sub map {
  my ($self, $request, $conn) = @_;
  my $c;
  my $sessionId = $self->getSession($request);
  if(defined $self->{continuations}{$sessionId}) {
    $c = $self->{continuations}{$sessionId};
  } else {
    # In this case, lets create a new continuation
    $c = $self->{mkNewCont}->($request);
    # And we call it one time to let it do some initialization
    $c->($self);
    # And we stash it away!
    $self->{continuations}{$sessionId} = $c;
  }

  # And send our session cookie
  # Perhaps instead we should be adding this to a list of headers to be sent
  print $conn "Set-Cookie: sessionid=$sessionId\r\n";
  return $c;
}

sub set_cont_maker_sub {
  my ($self, $sub) = @_;
  if(!$sub) {
    $sub = \&::main;
  }
  $self->{mkNewCont} = $self->mkContMaker($sub);
}

# Take a sub ref and give back a continuation. Just a shortcut
sub mkcont {
  my ($self, $func) = @_;
  my $cont = csub { $func->(@_) };
  return $cont;
}

sub mkContMaker {
  my ($self, $sub) = @_;
  my $mkNewCont = sub {
    $self->mkcont($sub)
  };
  return $mkNewCont;
}

=back

=head1 SEE ALSO

L<Continuity>

=head1 AUTHOR

  Brock Wilcox <awwaiid@thelackthereof.org>
  http://thelackthereof.org/

=head1 COPYRIGHT

  Copyright (c) 2004-2006 Brock Wilcox <awwaiid@thelackthereof.org>. All rights
  reserved.  This program is free software; you can redistribute it and/or
  modify it under the same terms as Perl itself.

=cut

1;

