package Statocles::Image;
# ABSTRACT: A reference to an image
$Statocles::Image::VERSION = '0.061';

use Statocles::Base 'Class';
use Scalar::Util qw( blessed );


has src => (
    is => 'rw',
    isa => Str,
    required => 1,
    coerce => sub {
        my ( $href ) = @_;
        if ( blessed $href && $href->isa( 'Path::Tiny' ) ) {
            return $href->stringify;
        }
        return $href;
    },
);


has alt => (
    is => 'rw',
    isa => Str,
    default => sub { '' },
);


has width => (
    is => 'rw',
    isa => Int,
);


has height => (
    is => 'rw',
    isa => Int,
);


has role => (
    is => 'rw',
    isa => Maybe[Str],
    lazy => 1,
    default => sub {
        return !$_[0]->alt ? 'presentation' : undef;
    },
);

1;

__END__

=pod

=head1 NAME

Statocles::Image - A reference to an image

=head1 VERSION

version 0.061

=head1 SYNOPSIS

    my $img = Statocles::Image->new(
        src     => '/path/to/image.jpg',
        alt     => 'Alternative text',
    );

=head1 DESCRIPTION

This class holds a link to an image, and the attributes required to
render its markup. This is used by L<documents|Statocles::Document/images>
to associate images with the content.

=head1 ATTRIBUTES

=head2 src

The source URL of the image. Required.

=head2 alt

The text to display if the image cannot be fetched or rendered. This is also
the text to use for non-visual media.

If missing, the image is presentational only, not content.

=head2 width

The width of the image, in pixels.

=head2 height

The height of the image, in pixels.

=head2 role

The L<ARIA|http://www.w3.org/TR/wai-aria/> role for this image. If the L</alt>
attribute is empty, this attribute defaults to C<"presentation">.

=head1 AUTHOR

Doug Bell <preaction@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by Doug Bell.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
