require 5.6.1;

use strict;
use warnings;

use ExtUtils::MakeMaker;

# Usage: perl Makefile.PL [options]
# Options:
#  -f file   Use file for default configuration values (default is config.log)
#  -h        Print help
#  -k        Keep same values as config.log (or file specified with -f)

## Note: The version number for the entire release is specified here and
## propagated.  Be sure to edit for every release.
my $Version = "0.003023";

use vars qw($opt_f $opt_h $opt_k);
my $OUTPUT_CFG_FILE = $opt_f = 'config.log';

use Getopt::Std;
usage() if ! getopts("f:hk") || $opt_h;

##### Do all user prompts first
my %CFG_LIST =
    (taint => {default => 'No',
	       desc => 'Run perl tainted (not required for safe operation)',
	       checkfail => \&yesno,
	       ,},
     defaultcss => {
	 default => 'None',
	 desc => 'URL for default cascading style sheet (or "none")',
	 explain => <<'EOS',
Generated documents need a style sheet to look good.  It is recommended
to serve a local copy of lib/Text/Restructured/default.css as an http URL.
You can also specify 'None', in which case the default stylesheet will
be embedded within every document.
EOS
	 checkfail => \&isurl,
         ,},
     docurl => {
	 default => 'None',
	 desc => 'URL where documentation will be installed (or "none")',
	 checkfail => \&isurl,
         ,},
     );
my @CFG_LIST = qw(defaultcss taint docurl);
warn "\@CFG_LIST and \%CFG_LIST have different number of elements"
    if @CFG_LIST != keys %CFG_LIST;

my %CONFIG;			# Our final configuration
my %DEFAULTS;			# Default values

# First read the config file if it exists
if (-f $opt_f) {
    open CF, $opt_f or die "Cannot open $opt_f";
    my %cfg = eval(join('',<CF>));
    @DEFAULTS{keys %cfg} = values %cfg;
    close CF;
}
else {
    # Set the defaults from %CFG_LIST
    @DEFAULTS{keys %CFG_LIST} = map($_->{default}, values %CFG_LIST);
}

# Do the user prompts
while (! $opt_k) {
    foreach my $cfg_item (@CFG_LIST) {
	my $message = $CFG_LIST{$cfg_item}{desc};
	$message = "$CFG_LIST{$cfg_item}{explain}\n$message"
	    if defined $CFG_LIST{$cfg_item}{explain};
	while (1) {
	    my $val = prompt ($message, $DEFAULTS{$cfg_item});
	    $val =~ s/^\s*(.*?)\s*$/$1/;
	    $CONFIG{$cfg_item} = $val;
	    last unless 
		defined $CFG_LIST{$cfg_item}{checkfail} &&
		&{$CFG_LIST{$cfg_item}{checkfail}}($CONFIG{$cfg_item});
	    $message = $CFG_LIST{$cfg_item}{desc};
	}
	$DEFAULTS{$cfg_item} = $CONFIG{$cfg_item};
    }

    print "\n";
    printsummary();

    my $okay = prompt("Does this look right?");
    last if ($okay !~ m/^[n0]/i);
}
if ($opt_k) {
    @CONFIG{@CFG_LIST} = @DEFAULTS{@CFG_LIST};
    printsummary();
}

# Add the prest version to the config
$CONFIG{version} = $Version;
# Add the perl executable to the config
$CONFIG{perl} = $^X;

# Write the configuration file (after saving the old one)
rename "$OUTPUT_CFG_FILE", "$OUTPUT_CFG_FILE.bak";
open (CL, ">$OUTPUT_CFG_FILE") or die "Cannot write to $OUTPUT_CFG_FILE.";
print CL map(qq('$_'=>'$CONFIG{$_}',\n), sort keys %CONFIG);
close CL;

##### Figure out what version of make to use.  We *require* GNU make.
my @path = split /:/, $ENV{PATH};
my ($make) = grep -x $_, map("$_/gmake", @path), map("$_/make", @path);

##### Now construct the common makefile
chomp (my $base = `pwd`);
my $Perl    = $^X;
my $taint_flag = $CONFIG{taint} =~ /^y/i ? ' -T' : '';

my $common_mk = << 'EOS';
#################### Common.mk ####################
# This file was automatically generated by Makefile.PL

TEST_PYs := $(wildcard test_*.py)
RST01s := $(patsubst test_%.py,%01.rst,$(TEST_PYs))
RSTs := $(wildcard *.rst)
CHKs := $(RSTs:.rst=.chk)
SCRIPTDIR = ${base}/blib/script
PREST = $(SCRIPTDIR)/prest
LIBDIR = ${base}/blib/lib/Text
PMFILES := $(LIBDIR)/Restructured.pm $(wildcard $(LIBDIR)/Restructured/*.pm)
# Writer-specific variables
DOM_FLAGS = -W nobackn

PRESTCMD = $(PREST) $(GLOBAL_FLAGS) -D align=0
TESTBIN = ${base}/tbin
EXTRACT_TESTS = $(TESTBIN)/extract_tests
DIFFRE = $(TESTBIN)/diffre
PERL = ${Perl}
DEBUG_FLAGS = -D no_line_directives

# If the suite has makefile definitions, use them
-include ../Suite.mak
# If the local directory has makefile definitions, use them
-include Dir.mak

ifdef COVER
  COVER_FLAG = -MDevel::Cover=-db,../../cover_db,-silent,1,-summary,0
endif
PERL_FLAGS += -I${base}/blib/lib${taint_flag} $(COVER_FLAG)

# Some global flags if not overridden
GLOBAL_FLAGS ?= -D source="test data" -D xformoff='.*'

rsts:	$(RST01s)

%01.rst:	test_%.py $(EXTRACT_TESTS)
	$(PERL) $(EXTRACT_TESTS) $(EXTRACT_TEST_FLAGS) $<

test:	$(CHKs)

${\makerules('dom')}

${\makerules('html')}

${\makerules('latex', 'tex')}

${\makerules('xml')}

${\makerules('xref')}

##^^^^^^^^^^^^^^^^^^ Common.mk ^^^^^^^^^^^^^^^^^^**
EOS
# Substitute in for ${perlvar} references
$common_mk =~ s/(\$ \{ [^\}]+ \})/eval $1/gex;
open MK, ">t/Common.mk" or die "Cannot write to t/Common.mk";
print MK $common_mk;
close MK;

#### Write out all the .t files
# Create generic .t file
my $generic_t = << 'EOS';
# -*-perl-*-
#################### Generic .t file ####################
# This file was automatically generated by Makefile.PL

use strict;
use warnings;

use Test::More;

my $base = '${base}';

my ($mydir, $myname) = $0 =~ m!(?:(.*)/)?(.*)\.t!;
chdir $mydir if defined $mydir;
die "Error: No init directory for this test\n" unless -d "$myname.init";

# First create the subdirectory for doing testing
system "rm -rf $myname.dir" if -d "$myname.dir";
system "cp -r $myname.init $myname.dir";

chdir "$myname.dir";

# Check to see if we need to skip all tests
if (-f "skip.pl") {
    chomp (my $error = `$^X -I$base/blib/lib skip.pl 2>&1`);
    plan(skip_all => $error) if $?;
}

# Now unpack the tests if needed
if (<test_*.py>) {
    system "${make} -s -f ../../Common.mk rsts";
}

# Figure out what tests to do
my @tests = -f "tests.pl" ? do "tests.pl" : <*.rst>;
plan tests => 0+@tests;

# Fix PERL5LIB to keep only blib directories
$ENV{PERL5LIB} = join ':',grep(/\bblib\b/, split(/:/, $ENV{PERL5LIB}));

# Do the tests
TEST:
foreach my $test (@tests) {
    my ($base_test) = $test =~ /(.*)\.rst/;
    if (-f "$base_test.skip.pl") {
	# Check whether we need to skip this file
	chomp (my $error = `$^X -I$base/blib/lib $base_test.skip.pl 2>&1`);
      SKIP:
	{
	    skip($error, 1) if $?;
	}
	next TEST if $?
    }
    my $make_errs = `${make} -s -f ../../Common.mk $base_test.chk 2>&1`;
    # Fake around perl 5.8.8 non-disablable warning
    $make_errs =~ s/^Wide character in print.*\n//gm;
    # Remove make messages
    $make_errs =~ s/^g?make\[\d\]: (Enter|Leav)ing directory.*\n//gm;
    my $errs = -f "$base_test.stderr" ? `cat $base_test.stderr` : '';
    if ($errs) {
	# Convert search characters
	$errs =~ s/([.\\\[\]{}()+*$@])/\\$1/g;
	# Fix line numbers,  paths
	$errs =~ s/(\\\(eval \d+\\\))/$1(\\[.*\\])?/g;
	$errs =~ s!(^| )/\S+/!$1/\\S+/!mg;
	$errs =~ s/(line|eval) (\d+)/$1 \\d+/g;
	like ($make_errs . `cat $base_test.chk 2>&1`, qr/$errs/, $test);
    }
    else {
	is ($make_errs . `cat $base_test.chk 2>&1`, $errs, $test);
    }
}
##^^^^^^^^^^^^^^^^^^ Generic .t file ^^^^^^^^^^^^^^^^^^##
EOS

# Substitute in for ${perlvar} references
$generic_t =~ s/(\$ \{ [^\}]+ \})/eval $1/gex;

my @TESTS;
my @generic_t_dirs = grep -d $_, <t/*>;
# Create generic .t files
foreach my $dir (@generic_t_dirs) {
    opendir DIR, $dir;
    my @need_t = grep s/\.init$/.t/, readdir(DIR);
    closedir DIR;
    foreach my $t (@need_t) {
	open T, ">$dir/$t";
	print T $generic_t;
	close T;
	push @TESTS, "$dir/$t";
    }
}

# Make sure PrestConfig.pm gets rebuilt
unlink "lib/Text/Restructured/PrestConfig.pm";

#### Finally, create the Makefile
# Get list of perl modules in lib subdirectory
chomp (my @pm_files = `find lib -type d -name .svn -prune , -type d -name CVS -prune , -type f -and -not -name '*~' -and -not -name '*.PL'`);
my %pm_files;
@pm_files{@pm_files} = map "blib/$_", @pm_files;

# See lib/ExtUtils/MakeMaker.pm for details of how to influence
# the contents of the Makefile that is written.
WriteMakefile(
    NAME           => 'Text::Restructured',
    AUTHOR         => 'Mark Nodine <mnodine@alum.mit.edu>',
    ABSTRACT       => 'Perl implementation of reStructuredText parser',
    VERSION        => $Version,
    EXE_FILES      => [qw(prest)],
    MAN1PODS       => {},
    MAN3PODS       => {},
    PREREQ_PM      => { 'Text::ASCIIMathML'=>0 }, # e.g., Module::Name => 1.1
    PL_FILES       => {'lib/Text/Restructured/PrestConfig.pm.PL' =>
		       "lib/Text/Restructured/PrestConfig.pm"},
    PM             => {'lib/Text/Restructured/PrestConfig.pm' =>
		       'blib/lib/Text/Restructured/PrestConfig.pm',
		       %pm_files},
#    FIXIN          => 'cp',
    test           => { TESTS => join ' ',@TESTS },
    clean          => { FILES =>(join(' ', (@TESTS, map(/(.*)\.t$/ && "$1.dir",
							@TESTS))) .
				 q[ t/Common.mk lib/Text/Restructured/PrestConfig.pm]) },
    dist           => { COMPRESS => 'gzip', SUFFIX => '.gz' },
    realclean      => { FILES => q(config.log config.log.bak) },
);

sub MY::libscan {
    my ($self, $path) = @_;
    return $path !~ /\.svn/ && $path;
}

sub MY::postamble {
    return <<'MAKE_FRAG';
FIXIN = $(PERLRUNINST) insertperl.pl

.PHONY: doc
doc ::
	cd doc/src; $(MAKE)
MAKE_FRAG
}

#### Random support subroutines

# Returns the list of writer-specific make rules for a given target, with
# the optional suffix (defaults to writer name)
sub makerules {
    my ($writer, $suffix) = @_;

    (my $Writer = $writer) =~ tr/a-z/A-Z/;
    $suffix = $writer unless defined $suffix;
    my $prest_cmd = '$(PRESTCMD) $(${Writer}_FLAGS) $(SUITE_FLAGS) $(DIR_FLAGS) $(RST_FLAG_$(*)) $*.rst $(POSTPROCESS_$(*))';
    # Substitute in for ${perlvar} references
    $prest_cmd =~ s/(\$ \{ [^\}]+ \})/eval $1/gex;
    my $perl_flags = '$(PERL_FLAG_$(*)) $(PERL_FLAGS)';
    my $rules = << 'EOS';
# Location of ${writer} writer
${Writer}WRT = $(LIBDIR)/Restructured/Writer/${writer}.wrt
# Flags to invoke ${writer} writer
${Writer}_FLAGS += -w ${writer}
# Program to create regular expression version for ${writer} writer
RE${Writer} = $(TESTBIN)/re${suffix}

.PRECIOUS:	%.${suffix}c %.${suffix}re

# First diff test is vis-a-vis .my${suffix}
# Build ${suffix}c from my${suffix}
%.${suffix}c:	%.my${suffix}
	@mv $< $@; touch $@

# Otherwise can do diff test vis-a-vis .${suffix}
%.${suffix}c:	%.${suffix}
	@mv $< $@

# Can do re${suffix} test by building ${suffix}re from .${suffix} if .r present
%.${suffix}re: %.my${suffix} %.r $(RE${Writer})
	@perl $(RE${Writer}) $< > $@

%.${suffix}re: %.${suffix} %.r $(RE${Writer})
	@perl $(RE${Writer}) $< > $@

# Generate ${writer} output on STDOUT
%.${suffix}o:	%.rst $(PREST) $(${Writer}WRT) $(PMFILES)
	@$(PERL) ${perl_flags} ${prest_cmd}

# Invoke perl with debugger for ${writer} generation
%.${suffix}d:	%.rst $(PREST) $(${Writer}WRT) $(PMFILES)
	@$(PERL) ${perl_flags} -d ${prest_cmd} $(DEBUG_FLAGS)

# Checking ${writer} with regular expression comparison
%.chk:	%.rst %.${suffix}re $(PREST) $(${Writer}WRT) $(PMFILES)
	@$(PERL) ${perl_flags} ${prest_cmd} | $(PERL) $(DIFFRE) $*.${suffix}re - > $@

# Checking ${writer} with straight diff
%.chk:	%.rst %.${suffix}c $(PREST) $(${Writer}WRT) $(PMFILES)
	@$(PERL) ${perl_flags} ${prest_cmd} | diff $*.${suffix}c - > $@
EOS
    # Substitute in for ${perlvar} references
    $rules =~ s/(\$ \{ [^\}]+ \})/eval $1/gex;
    return $rules;
}

# Prints a summary of the configuration
sub printsummary {
    print "Here is the summary of the configuration:\n";
    foreach my $cfg_item (@CFG_LIST) {
	print "  $CFG_LIST{$cfg_item}{desc}: $CONFIG{$cfg_item}\n";
    }
}

# This subroutine extracts usage information
sub usage {
  my($what,$end) = @_;
  $what = "Usage" if ! $what;
  my $print;
  if (open(ME,$0) == 1) {
    while (<ME>) {
      $print = 1 if /^# $what/o;
      $print = 0 if ! /^#/o || ($end && /^# $end/o);
      print substr($_,2) if $print;
    }
    close(ME);
  }
  else {
    print STDERR "Usage not available.\n";
  }
  exit;
}

# Checks for valid URL or "none"
sub isurl {
    my $fail = $_[0] !~ /^\w+:|none/i;
    print STDERR "Must be either a URL reference or 'None'\n"
	if $fail;
    return $fail;
}

# Checks for yes or no answer
sub yesno {
    my $fail = !($_[0] =~ s/^y.*/Yes/i || $_[0] =~ s/^n.*/No/i);
    print "Must be 'yes' or 'no'\n" if $fail;
    return $fail;
}
