## -*- Mode: CPerl -*-

## File: DTA::TokWrap::Processor::tokenize::http.pm
## Author: Bryan Jurish <moocow@cpan.org>
## Description: DTA tokenizer wrappers: tokenizer: external server via http

package DTA::TokWrap::Processor::tokenize::http;

use DTA::TokWrap::Version;  ##-- imports $VERSION, $RCDIR
use DTA::TokWrap::Base;
use DTA::TokWrap::Utils qw(:progs :slurp :time);
use DTA::TokWrap::Processor;

use LWP::UserAgent;

use Encode qw(encode decode);
use Carp;
use strict;

##==============================================================================
## Constants
##==============================================================================
our @ISA = qw(DTA::TokWrap::Processor::tokenize);

##==============================================================================
## Constructors etc.
##==============================================================================

## $tz = CLASS_OR_OBJ->new(%args)
## %defaults = CLASS->defaults()
##  + static class-dependent defaults
##  + %args, %defaults, %$tz:
##     serverurl => $url,      ##-- server url (default='http://194.95.188.28/dtaos/tok-cgi.perl' [kaskade])
##     txtparam  => $param,    ##-- text query parameter (default='t')
##     timeout   => $secs,     ##-- user agent timeout (default=300)
##     ua => $ua,              ##-- low-level underlying LWP::UserAgent
sub defaults {
  my $that = shift;
  return (
	  $that->SUPER::defaults(),
	  #serverurl => 'http://194.95.188.28/dtaos/tok-cgi.perl', ##-- kaskade1
          #serverurl => 'http://194.95.188.59/dtaos/tok-cgi.perl', ##-- kaskade2
          #serverurl => 'http://kaskade.dwds.de/dtaos/tok-cgi.perl',
	  serverurl => 'http://kaskade.dwds.de/waste/tokenize.fcgi?m=dta&O=mr,loc',
	  #serverpars => {},
	  txtparam  => 't',
	  timeout   => 300,
	  @_,
	 );
}

## $tz = $tz->init()
sub init {
  my $tz = shift;

  ##-- instantiate LWP::UserAgent
  if (!defined($tz->{ua})) {
      $tz->{ua} = LWP::UserAgent->new(timeout=>$tz->{timeout})
	or $tz->logconfess("init(): could not create LWP::UserAgent: $!");
  }

  return $tz;
}

##==============================================================================
## Methods
##==============================================================================

## $doc_or_undef = $CLASS_OR_OBJECT->tokenize($doc)
## + $doc is a DTA::TokWrap::Document object
## + %$doc keys:
##    txtfile => $txtfile,    ##-- (input) serialized text file
##    tokdata0 => $tokdata,   ##-- (output) tokenizer output data (string)
##    tokenize0_stamp  => $f, ##-- (output) timestamp of operation end
##    tokdata0_stamp => $f,   ##-- (output) timestamp of operation end
## + may implicitly call $doc->mkbx() and/or $doc->saveTxtFile()
sub tokenize {
  my ($tz,$doc) = @_;
  $doc->setLogContext();

  ##-- log, stamp
  $tz = $tz->new if (!ref($tz));
  $tz->vlog($tz->{traceLevel},"tokenize()");
  $doc->{tokenize0_stamp0} = timestamp();

  ##-- sanity check(s)
  $tz->logconfess("tokenize(): no LWP::UserAgent (ua) defined")
    if (!$tz->{ua});
  $tz->logconfess("tokenize(): no URL (serverurl) defined")
    if (!$tz->{serverurl});
  $tz->logconfess("tokenize(): no query paramater (txtparm) defined")
    if (!$tz->{txtparam});
  $tz->logconfess("tokenize(): no .txt file defined")
    if (!defined($doc->{txtfile}));
  $tz->logconfess("tokenize(): .txt file '$doc->{txtfile}' not readable")
    if (!-r $doc->{txtfile});

  ##-- post the tokenization request
  $tz->vlog($tz->{traceLevel},"tokenize(): url=$tz->{serverurl}?$tz->{txtparam}=...");
  $doc->{tokdata0} = '';
  slurp_file($doc->{txtfile},\$doc->{txtdata}) if (!defined($doc->{txtdata}));
  ##
  ## Wed, 15 May 2013 10:39:23 +0200 moocow
  ## + use multipart/form-data to avoid implicit LF->CR+LF conversion by LWP::UserAgent (HTTP::Request::Common::POST() v6.03 / debian wheezy)
  ## + fixes goofy byte-offset problems using HTTP tokenizer
  my $rsp = $tz->{ua}->post($tz->{serverurl}, { $tz->{txtparam}=>$doc->{txtdata} }, 'Content-Type'=>'multipart/form-data');
  $tz->vlog($tz->{traceLevel},"tokenize(): response: ", $rsp ? $rsp->status_line : '(no response)');
  $tz->logconfess("tokenize(): error from server $tz->{serverurl}: ", ($rsp ? $rsp->as_string : '(no response)'))
    if (!$rsp || !$rsp->is_success);

  ##-- finalize
  $doc->{tokdata0} = $rsp->content;
  $doc->{ntoks} = $tz->nTokens(\$doc->{tokdata0});
  $doc->{tokfile0_stamp} = $doc->{tokenize0_stamp} = $doc->{tokdata0_stamp} = timestamp(); ##-- stamp
  return $doc;
}


1; ##-- be happy

__END__

##========================================================================
## POD DOCUMENTATION, auto-generated by podextract.perl, edited

##========================================================================
## NAME
=pod

=head1 NAME

DTA::TokWrap::Processor::tokenize::http - DTA tokenizer wrappers: http: external tokenizer via http (hack)

=cut

##========================================================================
## SYNOPSIS
=pod

=head1 SYNOPSIS

 use DTA::TokWrap::Processor::tokenize::http;
 
 $tz = DTA::TokWrap::Processor::tokenize::http->new(%args);
 $doc_or_undef = $tz->tokenize($doc);

=cut

##========================================================================
## DESCRIPTION
=pod

=head1 DESCRIPTION

This class is currently just a wrapper for the band-aid CGI script
at http://kaskade.dwds.de/~moocow/opensearch/tok-cgi.perl .

Most users should use the high-level
L<DTA::TokWrap|DTA::TokWrap> wrapper class
instead of using this module directly.

=cut

##----------------------------------------------------------------
## DESCRIPTION: DTA::TokWrap::Processor::tokenize::http: Constants
=pod

=head2 Constants

=over 4

=item @ISA

DTA::TokWrap::Processor::tokenize::http::tomasotath
inherits from
L<DTA::TokWrap::Processor::tokenize::http|DTA::TokWrap::Processor::tokenize::http>.

=back

=cut

##----------------------------------------------------------------
## DESCRIPTION: DTA::TokWrap::Processor::tokenize::http: Constructors etc.
=pod

=head2 Constructors etc.

=over 4

=item new

 $tz = $CLASS_OR_OBJ->new(%args);


%args, %$tz: (none yet)

 serverurl => $url,      ##-- server url (default='http://kaskade.dwds.de/~moocow/opensearch/tok-cgi.perl')
 txtparam  => $param,    ##-- text query parameter (default='t')
 timeout   => $secs,     ##-- user agent timeout (default=300)
 ua => $ua,              ##-- low-level underlying LWP::UserAgent

=item defaults

 %defaults = CLASS->defaults();

Static class-dependent defaults.

=back

=cut

##----------------------------------------------------------------
## DESCRIPTION: DTA::TokWrap::Processor::tokenize::http: Methods
=pod

=head2 Methods

=over 4

=item tokenize

 $doc_or_undef = $CLASS_OR_OBJECT->tokenize($doc);

See L<DTA::TokWrap::Processor::tokenize::http::tokenize::http()|DTA::TokWrap::Processor::tokenize::http/tokenize>.

=back

=cut

##========================================================================
## END POD DOCUMENTATION, auto-generated by podextract.perl

##======================================================================
## See Also
##======================================================================

=pod

=head1 SEE ALSO

L<DTA::TokWrap::Intro(3pm)|DTA::TokWrap::Intro>,
L<dta-tokwrap.perl(1)|dta-tokwrap.perl>,
...

=cut

##======================================================================
## See Also
##======================================================================

=pod

=head1 SEE ALSO

L<DTA::TokWrap::Intro(3pm)|DTA::TokWrap::Intro>,
L<dta-tokwrap.perl(1)|dta-tokwrap.perl>,
...

=cut

##======================================================================
## Footer
##======================================================================

=pod

=head1 AUTHOR

Bryan Jurish E<lt>moocow@cpan.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2009-2018 by Bryan Jurish

This package is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.14.2 or,
at your option, any later version of Perl 5 you may have available.

=cut


