#########################################################################################
# Package        HiPi::GPIO::Pin
# Description:   Pin
# Copyright    : Copyright (c) 2013-2017 Mark Dootson
# License      : This is free software; you can redistribute it and/or modify it under
#                the same terms as the Perl 5 programming language system itself.
#########################################################################################
package HiPi::GPIO::Pin;

#########################################################################################

use strict;
use warnings;
use parent qw( HiPi::Pin );
use Carp;
use HiPi qw( :rpi );

our $VERSION ='0.61';

__PACKAGE__->create_accessors( qw( gpio ) );

sub _open {
    my ($class, %params) = @_;
    defined($params{pinid}) or croak q(pinid not defined in parameters);
    
    require HiPi::GPIO;
    
    my $self = $class->SUPER::_open(%params);
    return $self;
}

sub _do_getvalue {
    my($self) = @_;
    return HiPi::GPIO::xs_gpio_read( $self->pinid );
}

sub _do_setvalue {
    my( $self, $newval) = @_;
    return HiPi::GPIO::xs_gpio_write( $self->pinid, $newval );
}

sub _do_getmode {
    my $self = shift;
    return HiPi::GPIO::xs_gpio_get_mode( $self->pinid );
}

sub _do_setmode {
    my ($self, $newmode) = @_;
    return HiPi::GPIO::xs_gpio_set_mode( $self->pinid, $newmode );
}

sub _do_getinterrupt {
    my $self = shift;
    my $fh = _open_file( $self->pinroot . '/edge' );
    my $result = _read_fh_bytes( $fh, 16);
    close($fh);
    
    if($result eq 'rising') {
        return RPI_INT_RISE;
    } elsif($result eq 'falling') {
        return RPI_INT_FALL;
    } elsif($result eq 'both') {
        return RPI_INT_BOTH;
    } else {
        return RPI_INT_NONE;
    }
}

sub _do_setinterrupt {
    my ($self, $newedge) = @_;
    
    $newedge ||= RPI_INT_NONE;
    
    my $stredge = 'none';
    
    $newedge = RPI_INT_FALL if $newedge eq 'falling';
    $newedge = RPI_INT_RISE if $newedge eq 'rising';
    $newedge = RPI_INT_BOTH if $newedge eq 'both';
    $newedge = RPI_INT_NONE if $newedge eq 'none';
    
    if ( $newedge == RPI_INT_AFALL || $newedge == RPI_INT_FALL || $newedge == RPI_INT_LOW  ) {
        $stredge = 'falling';
    } elsif( $newedge == RPI_INT_ARISE || $newedge == RPI_INT_RISE || $newedge == RPI_INT_HIGH  ) {
        $stredge = 'rising';
    } elsif( $newedge == RPI_INT_BOTH ) {
        $stredge = 'both';
    } else {
        $stredge = 'none';
    }
    
    my $fh = _open_file( $self->pinroot . '/edge' );
    _write_fh( $fh, $stredge);
    close($fh);
    return $newedge;
}

sub _do_setpud {
    my($self, $pudval) = @_;
    my $pudchars = 'error';
    
    if ( $pudval == RPI_PUD_OFF ) {
        $pudchars = 'pn';
    } elsif ($pudval == RPI_PUD_UP ) {
        $pudchars = 'pu';
    } elsif ($pudval == RPI_PUD_DOWN ) {
        $pudchars = 'pd';
    } else {
        croak(qq(Incorrect PUD setting $pudval));
    }
    
    require HiPi::GPIO;
    
    my $hipigpio = HiPi::GPIO->new;
    $hipigpio->set_pin_pud($self->pinid, $pudval);
}


sub _do_activelow {
    my($self, $newval) = @_;
    
    my $fh = _open_file( $self->pinroot . '/active_low' );
    my $result;
    if(defined($newval)) {
        _write_fh( $fh, $newval);
        $result = $newval;
    } else {
        $result = _read_fh_bytes( $fh, 1);
    }
    close($fh);
    return $result;
} 

sub DESTROY {
    my $self = shift;
    close($self->valfh);
}

1;
