---
title: Design
#nav_title: About
nav_menu: default-nav
change_notes: change-log-data
layout: default
---

# Design of the Perl6 modules

## Introduction

Despite some thinking about the setup, a lot of changes already happened. See the history on the Home page. In time, the following notes crystallized in the 'making of' the set of perl6 packages. First what became clear is that there was a need to separate the modules in several packages like Gtk has done. There is glib, gobject, gdk, gtk, cairo, pango, etc. It was important to keep the same separation among the perl6 packages.
All these libraries come from **Gnome** and the Gtk version to adhere to was 3. So the perl6 packages became **Gnome::Glib**, **Gnome::GObject**, **Gnome::Gdk3** and **Gnome::Gtk3**. Note that gdk is also following the same version as gtk.
Now it is possible to extend the lot with new packages for cairo and pango like **Gnome::Cairo** and **Gnome::Pango**. These packages are not yet available so every use of the subs from these libraries are not yet supported by the perl6 libraries.

An extension to the **Gnome::Gtk3** package is available to handle *XML* files generated by *Glade*, a user interface designer program. It is called **Gnome::Gtk3::Glade**. All things can be programmed using **Gnome::Gtk3** and **XML** alone but there are helpful ways to connect to signals for example.

A package is added to make the native libraries available to the other packages. This package is called **Gnome::N**. There is also a bit of debugging added to that module. This may change later to be defined in another package. E.g. *Gnome::T* for test user interface purposes. At the moment UI testing is possible in **Gnome::Gtk3::Glade**.

## The notes comprising the design

I want to follow the interface of the classes in **Gtk**, **Gdk** and **Glib** as closely as possible by keeping the names of the native functions the same as provided with the following exceptions;

* Native objects are wrapped into Perl6 classes.

* The native subroutines are defined in their corresponding Perl6 classes. They are defined in such a way that they have become methods in those classes.

* Many subs also have as their first argument the native object. Because this object is held in the class, it is automatically inserted when the sub is called. E.g. a definition like the following in the `Gnome::Gtk3::Button` class
  ```
  sub gtk_button_set_label ( N-GObject $widget, Str $label )
    is native(&gtk-lib)
    { * }
  ```
  can be used as
  ```
  my Gnome::Gtk3::Button $button .= new(:empty);
  $button.gtk_button_set_label('Start Program');
  ```

* Classes can use the methods of inherited classes. E.g. The **Gnome::Gtk3::Button** class inherits **Gnome::Gtk3::Bin** and **Gnome::Gtk3::Bin** inherits **Gnome::Gtk3::Container** etc. Therefore, a method like `gtk_widget_set_tooltip_text()` defined in **Gnome::Gtk3::Widget** can be used.
  ```
  $button.gtk_widget_set_tooltip_text('When pressed, program will start');
  ```

* Classes can use methods from their interfaces. E.g. All widgets like **Gnome::Gtk3::Button** use the **Gnome::Gtk3::Buildable** interface.
  ```
  my Str $button-name = $button.gtk_buildable_get_name();
  ```

* The names are sometimes long and prefixed with words which are also used in the class path name. Therefore, those names can be shortened by removing those prefixes. An example method defined in **Gnome::Gtk3::Button** class is `gtk_button_get_label()`. This can be shortened to `get_label()`.
  ```
  my Str $button-label = $button.get_label;
  ```
  In the documentation this will be shown with brackets around the part that can be left out. In this case it is shown as `[gtk_button_] get_label`.  Note that there is still a chance that a different method is found than the one you had in mind. The subs `gtk_widget_get_name()` and `gtk_buildable_get_name()` have the same short version n.l. `get_name()`. So it is important to know what the search path is as follows;
    * search in the current class
    * search in their interfaces
    * search in parent class
    * search in parent class interfaces
    * etc.
  So it follows that the `gtk_buildable_get_name()` has a higher priority than `gtk_widget_get_name()`.

* Names can not be shortened too much. E.g. `gtk_button_new()` and `gtk_label_new()` yield the name *new* which is a perl method from class **Mu**. I am thinking about chopping off the `g_`, `gdk_` and `gtk_` prefixes, but for now, at least one underscore ('\_') must be left in the name.

* All the method names are written with an underscore. Following a perl6 tradition, dashed versions are also possible.
  ```
  my Str $button-label1 = $button.gtk-button-get-label;
  my Str $button-label2 = $button.get-label;
  ```

* Not all native subs or even classes will be implemented or implemented much later because of the following reasons;
  * Many subs and classes in **GTK+** are deprecated. It seems logical to not implement them because there is no history of the Perl6 packages to support.
  * The original idea was to have the interface build by the glade interface designer. This lib was in the *GTK::Glade*(now **Gnome::Gtk3::Glade**) project before refactoring. Therefore a **Gnome::Gtk3::Button** does not have to have all subs to create a button because the **Gnome::Gtk3::Builder** module will do that. On the other hand a **Gnome::Gtk3::ListBox** is a widget which is changed dynamically most of the time and therefore need more subs to manipulate the widget and its contents.
  * The need to implement classes like **Gnome::Gtk3::Assistant**, **Gnome::Gtk3::Plug** or **Gnome::Gtk3::ScrolledWindow** is on a low priority because these can all be instantiated by **Gnome::Gtk3::Builder** using your Glade design.

* There are native subroutines which need a native object as an argument. The `gtk_grid_attach()` in **Gnome::Gtk3::Grid** is an example of such a routine.
  The declaration of the `gtk_grid_attach()` native sub;
  ```
  sub gtk_grid_attach (
    N-GObject $grid, N-GObject $child,
    int32 $x, int32 $y, int32 $width, int32 $height
  ) is native(&gtk-lib)
    { * }
  ```

  The `get-native-gobject()` method is defined in **Gnome::GObject::Object** to return the native object so we can use the gtk_grid_attach as follows. For other types `get-native-gboxed()` is needed.
  ```
  my Gnome::Gtk3::Label $label .= new(:label('my label'));
  my Gnome::Gtk3::Grid $grid .= new;
  $grid.gtk_grid_attach( $label.get-native-gobject(), 0, 0, 1, 1);
  ```
  However, the signatures of all subroutines are checked against the arguments provided, so it is possible to insert the native object hidden in the object when a perl6 object is noticed. So the example becomes more simple;
  ```
  my Gnome::Gtk3::Grid $grid .= new(:empty);
  my Gnome::Gtk3::Label $label .= new(:label('server name'));
  $grid.gtk-grid-attach( $label, 0, 0, 1, 1);
  ```

* Sometimes I had to stray away from the native function names because of the way the sub must be defined in perl6. Causes can be;
  * Returning different types of values. E.g. `g_slist_nth_data()`, found in **Gnome::Glib**, can return several types of data. This is solved using several subs linking to the same native sub (using `is symbol()`). In this library, the methods `g_slist_nth_data_str()` and `g_slist_nth_data_gobject()` are added. This can be extended for other native types like integer or float.

    ```
    sub g_slist_nth_data_str ( N-GSList $list, uint32 $n --> Str )
      is native(&gtk-lib)
      is symbol('g_slist_nth_data')
      { * }

    sub g_slist_nth_data_gobject ( N-GSList $list, uint32 $n --> N-GObject )
      is native(&gtk-lib)
      is symbol('g_slist_nth_data')
      { * }
    ```
  * Variable argument lists where I had to choose for the extra arguments. E.g. in the **Gnome::Gtk3::FileChooserDialog** the native sub `gtk_file_chooser_dialog_new()` has a way to extend it with a number of buttons on the dialog. I had to fix that list to a known number of arguments and renamed the sub `gtk_file_chooser_dialog_new_two_buttons()`.
    **NOTE** This is now changed; It is possible to implement variable argument lists with the newest perl6 version (about July 2019). The above sub is therefore deprecated and `gtk_file_chooser_dialog_new()` can be used.

  * Callback handlers in many cases can have different signatures. When used in a subroutine definition the subroutine must be declared differently every time another type of handler is used. This happens mainly when connecting a signal where a callback handler is provided. To make things easier, the method `register-signal()` defined in **Gnome::GObject::Object**, is created for this purpose. At the moment only the most common types of signals can be processed.
    **NOTE** Also this is changed; Now all types of signals can be processed, although some native objects provided to the signal handler might not yet possible to wrap in a perl6 class because the class is not implemented.

## Implementation details
* The native objects wrapped in perl6 classes are mostly not visible to the user, but if they do, their types always start wit *N-*. E.g. **N-GObject**, **N-GValue**, etc. **_This is not yet done everywhere**. A method `get-native-gobject()` was implemented to get the native object with a minimum effort.

* The `FALLBACK()` method defined in **Gnome::GObject::Object** is called if a method is not found. This makes it possible to search for the defined native subroutines in the class and inherited classes. It calls the `_fallback()` method, which starts with the class at the bottom and working its way up until the subroutine is found. That process is calling `callsame()` when a sub is not found yet. The resulting subroutine address is returned and processed with the `test-call()` functions from **Gnome::N::X**. Thrown exceptions are handled by the function `test-catch-exception()` from the same module.

* Interface modules like e.g. **Gnome::Gtk3::FileChooser**, have methods like `_file_chooser_interface()` which is called by the interface using modules from their `_fallback()` method. For the mentioned example this is **Gnome::Gtk3::FileChooserDialog**. All methods defined by that interface can be used by the interface using module.

* All classes deriving from **Gnome::GObject::Object** know about the `:widget(…)` named attribute when instantiating a widget class. This is used when the result of another native sub returns a **N-GObject**.

* The same classes also recognize the named argument `:build-id(…)` which is used to get a **N-GObject** from a **Gnome::Gtk3::Builder** object. It does something like `$builder.gtk_builder_get_object(…)`. A builder must be initialized and loaded with a GUI description before to be useful. For this, see also **Gnome::Gtk3::Glade**. This option works for all child classes too if those classes are managed by **Gnome::Gtk3::Builder**.

  An example to see both named arguments in use is when cleaning a list box;
  ```
  # instantiate a list box using the :build-id argument
  my Gnome::Gtk3::ListBox $list-box .= new(:build-id<someListBox>);
  loop {
    # Keep the index 0, entries will shift up after removal
    my $nw = $list-box.get-row-at-index(0);
    last unless $nw.defined;

    # Instantiate a container object using the :widget argument
    my Gnome::Gtk3::ListBoxRow $row .= new(:widget($nw));
    $row.gtk-widget-destroy;
  }
  ```

* The C functions can only return simple values like **int32**, **num64** etc. When a structure must be returned, it is returned in a structure using a pointer to that structure. Perl users are used to be able to return all sorts of types. To provide this behavior, the native sub is wrapped in another sub which can return the result and directly assign to some variable. **_Many subs stil need to be converted to show this behavior!_**

  So the definition of the sub is changed like so;
  ```
  sub gtk_range_get_range_rect ( N-GObject $range --> GdkRectangle ) {
    _gtk_range_get_range_rect( $range, my GdkRectangle $rectangle .= new);
    $rectangle
  }

  sub _gtk_range_get_range_rect (
    N-GObject $range, GdkRectangle $rectangle
  ) is native(&gtk-lib)
    is symbol('gtk_range_get_range_rect')
    { * }
  ```
  Now we can do
  ```
  my GdkRectangle $rectangle = $range.get-range-rect();
  ```
* The same situation arises when a native sub wants to return more than one value. In C code one must give a pointer to a location wherein the value can be returned. These arguments must be writable locations. Again this is solved by creating a wrapper around the native sub, the arguments can be provided locally and after the call, the wrapper returns a list of values. **also here, many subs stil need to be converted to show this behavior!**

  An example from `Gnome::Gdk3::Window`;
  ```
  sub gdk_window_get_position ( N-GObject $window --> List ) {
    _gdk_window_get_position( $window, my int32 $x, my int32 $y);
    ( $x, $y)
  }

  sub _gdk_window_get_position (
    N-GObject $window, int32 $x is rw, int32 $y is rw
  ) is native(&gdk-lib)
    is symbol('gdk_window_get_position')
    { * }
  ```
  To use it one can write the following
  ```
  my Int ( $x, $y) = $w.get-position;
  ```

* There is no Boolean type in C. All Booleans are integers and only 0 (False) or 1 (True) is used. Also here, to use Perl6 Booleans, the native sub must be wrapped into another sub to transform the variables. **_Not sure to implement this! because of its overhead of a 2nd call_**

# Errors and crashes

When an exception is thrown, the library captures it to process and create another exception to be (re)thrown. Sometimes it behaves odd and most of the errors end up in **MoarVM panic: Internal error: Unwound entire stack and missed handler**. I found that these occasions are always happening within callback handlers where perl6 code is run from C. This needs a rewrite to handle things more elegantly.

Other times it ends in just a plain crash. Some of the crashes are happening within GTK and cannot be captured by Perl6. One of those moments are the use of GTK calls without initializing GTK with `gtk_init()`.

A few measures are implemented to help a bit to prevent problems;

* The failure to initialize GTK on time (in most cases) is solved by using an initialization flag which is checked in module **Gnome::GObject::Object** which almost all modules inherit from. This way, the user never has to initialize GTK.
* To see the original exception and messages leading up to the crash a debug flag in the class **Gnome::N** can be set to show these messages which might help solving your problems. To use it write;
  ```
  use Gnome::N::X;
  Gnome::N::debug(:on);
  ...
  Gnome::N::debug(:!on);
  ```
  or
  ```
  use Gnome::N::X;
  Gnome::N::debug(:on);
  ...
  Gnome::N::debug(:off);
  ```

# Class hierargy
Below there is a table of the object hierarchy taken from [the developers page](https://developer.gnome.org/gtk3/3.24/ch02.html) and is used here to show what is implemented and what is deprecated. Every perl6 class is in the Gnome:: name space. Also prefixes and module path names are removed from the perl6 modules. so Object is really Gnome::GObject::Object and Window is for Gnome::Gtk3::Window. Other modules from Glib and Gdk3 are not displayed here. `├─✗` in front of a Gtk module means that it is deprecated or not implemented for other reasons.

```
Tree of Gtk C structures                              Perl 6 module
----------------------------------------------------- ------------------------
GObject                                               Object
├── GInitiallyUnowned                                 InitiallyUnowned
│   ├── GtkWidget                                     Widget
│   │   ├── GtkContainer                              Container
│   │   │   ├── GtkBin                                Bin
│   │   │   │   ├── GtkWindow                         Window
│   │   │   │   │   ├── GtkDialog                     Dialog
│   │   │   │   │   │   ├── GtkAboutDialog            AboutDialog
│   │   │   │   │   │   ├── GtkAppChooserDialog
│   │   │   │   │   │   ├── GtkColorChooserDialog     ColorChooserDialog
│   │   │   │   │   │   ├─✗ GtkColorSelectionDialog
│   │   │   │   │   │   ├── GtkFileChooserDialog      FileChooserDialog
│   │   │   │   │   │   ├── GtkFontChooserDialog
│   │   │   │   │   │   ├─✗ GtkFontSelectionDialog
│   │   │   │   │   │   ├── GtkMessageDialog
│   │   │   │   │   │   ├── GtkPageSetupUnixDialog
│   │   │   │   │   │   ├── GtkPrintUnixDialog
│   │   │   │   │   │   ╰── GtkRecentChooserDialog
│   │   │   │   │   ├── GtkApplicationWindow
│   │   │   │   │   ├── GtkAssistant
│   │   │   │   │   ├── GtkOffscreenWindow
│   │   │   │   │   ├── GtkPlug
│   │   │   │   │   ╰── GtkShortcutsWindow
│   │   │   │   ├── GtkActionBar
│   │   │   │   ├─✗ GtkAlignment
│   │   │   │   ├── GtkComboBox                       ComboBox
│   │   │   │   │   ├── GtkAppChooserButton
│   │   │   │   │   ╰── GtkComboBoxText               ComboBoxText
│   │   │   │   ├── GtkFrame
│   │   │   │   │   ╰── GtkAspectFrame
│   │   │   │   ├── GtkButton                         Button
│   │   │   │   │   ├── GtkToggleButton               ToggleButton
│   │   │   │   │   │   ├── GtkCheckButton            CheckButton
│   │   │   │   │   │   │   ╰── GtkRadioButton        RadioButton
│   │   │   │   │   │   ╰── GtkMenuButton             MenuButton
│   │   │   │   │   ├── GtkColorButton                ColorButton
│   │   │   │   │   ├── GtkFontButton
│   │   │   │   │   ├── GtkLinkButton
│   │   │   │   │   ├── GtkLockButton
│   │   │   │   │   ├── GtkModelButton
│   │   │   │   │   ╰── GtkScaleButton
│   │   │   │   │       ╰── GtkVolumeButton
│   │   │   │   ├── GtkMenuItem                       MenuItem
│   │   │   │   │   ├── GtkCheckMenuItem
│   │   │   │   │   │   ╰── GtkRadioMenuItem
│   │   │   │   │   ├─✗ GtkImageMenuItem
│   │   │   │   │   ├── GtkSeparatorMenuItem
│   │   │   │   │   ╰─✗ GtkTearoffMenuItem
│   │   │   │   ├── GtkEventBox
│   │   │   │   ├── GtkExpander
│   │   │   │   ├── GtkFlowBoxChild
│   │   │   │   ├── GtkHandleBox
│   │   │   │   ├── GtkListBoxRow                     ListBoxRow
│   │   │   │   ├── GtkToolItem
│   │   │   │   │   ├── GtkToolButton
│   │   │   │   │   │   ├── GtkMenuToolButton
│   │   │   │   │   │   ╰── GtkToggleToolButton
│   │   │   │   │   │       ╰── GtkRadioToolButton
│   │   │   │   │   ╰── GtkSeparatorToolItem
│   │   │   │   ├── GtkOverlay
│   │   │   │   ├── GtkScrolledWindow
│   │   │   │   │   ╰── GtkPlacesSidebar
│   │   │   │   ├── GtkPopover
│   │   │   │   │   ╰── GtkPopoverMenu
│   │   │   │   ├── GtkRevealer
│   │   │   │   ├── GtkSearchBar
│   │   │   │   ├── GtkStackSidebar
│   │   │   │   ╰── GtkViewport
│   │   │   ├── GtkBox                                Box
│   │   │   │   ├── GtkAppChooserWidget
│   │   │   │   ├── GtkButtonBox
│   │   │   │   │   ├─✗ GtkHButtonBox
│   │   │   │   │   ╰─✗ GtkVButtonBox
│   │   │   │   ├── GtkColorChooserWidget             ColorChooserWidget
│   │   │   │   ├─✗ GtkColorSelection
│   │   │   │   ├── GtkFileChooserButton
│   │   │   │   ├── GtkFileChooserWidget
│   │   │   │   ├── GtkFontChooserWidget
│   │   │   │   ├─✗ GtkFontSelection
│   │   │   │   ├─✗ GtkHBox
│   │   │   │   ├── GtkInfoBar
│   │   │   │   ├── GtkRecentChooserWidget
│   │   │   │   ├── GtkShortcutsSection
│   │   │   │   ├── GtkShortcutsGroup
│   │   │   │   ├── GtkShortcutsShortcut
│   │   │   │   ├── GtkStackSwitcher
│   │   │   │   ├── GtkStatusbar
│   │   │   │   ╰─✗ GtkVBox
│   │   │   ├── GtkFixed
│   │   │   ├── GtkFlowBox
│   │   │   ├── GtkGrid                               Grid
│   │   │   ├── GtkHeaderBar
│   │   │   ├── GtkPaned                              Paned
│   │   │   │   ├─✗ GtkHPaned                         
│   │   │   │   ╰─✗ GtkVPaned
│   │   │   ├── GtkIconView
│   │   │   ├── GtkLayout
│   │   │   ├── GtkListBox                            ListBox
│   │   │   ├── GtkMenuShell                          MenuShell
│   │   │   │   ├── GtkMenuBar                        MenuBar
│   │   │   │   ╰── GtkMenu                           Menu
│   │   │   │       ╰── GtkRecentChooserMenu
│   │   │   ├── GtkNotebook
│   │   │   ├── GtkSocket
│   │   │   ├── GtkStack
│   │   │   ├─✗ GtkTable
│   │   │   ├── GtkTextView                           TextView
│   │   │   ├── GtkToolbar
│   │   │   ├── GtkToolItemGroup
│   │   │   ├── GtkToolPalette
│   │   │   ╰── GtkTreeView                           TreeView
│   │   ├─✗ GtkMisc                                   Misc (Keep hierarchy)
│   │   │   ├── GtkLabel                              Label
│   │   │   │   ╰── GtkAccelLabel
│   │   │   ├─✗ GtkArrow
│   │   │   ╰── GtkImage                              Image
│   │   ├── GtkCalendar
│   │   ├── GtkCellView
│   │   ├── GtkDrawingArea
│   │   ├── GtkEntry                                  Entry
│   │   │   ├── GtkSearchEntry                        SearchEntry
│   │   │   ╰── GtkSpinButton
│   │   ├── GtkGLArea
│   │   ├── GtkRange                                  Range
│   │   │   ├── GtkScale                              Scale
│   │   │   │   ├─✗ GtkHScale
│   │   │   │   ╰─✗ GtkVScale
│   │   │   ╰── GtkScrollbar
│   │   │       ├─✗ GtkHScrollbar
│   │   │       ╰─✗ GtkVScrollbar
│   │   ├── GtkSeparator
│   │   │   ├─✗ GtkHSeparator
│   │   │   ╰─✗ GtkVSeparator
│   │   ├─✗ GtkHSV
│   │   ├── GtkInvisible
│   │   ├── GtkProgressBar
│   │   ├── GtkSpinner
│   │   ├── GtkSwitch
│   │   ╰── GtkLevelBar                               LevelBar
│   ├── GtkAdjustment
│   ├── GtkCellArea
│   │   ╰── GtkCellAreaBox
│   ├── GtkCellRenderer                               CellRenderer
│   │   ├── GtkCellRendererText                       CellRendererText
│   │   │   ├── GtkCellRendererAccel
│   │   │   ├── GtkCellRendererCombo
│   │   │   ╰── GtkCellRendererSpin
│   │   ├── GtkCellRendererPixbuf
│   │   ├── GtkCellRendererProgress
│   │   ├── GtkCellRendererSpinner
│   │   ╰── GtkCellRendererToggle
│   ├── GtkFileFilter                                 FileFilter
│   ├── GtkTreeViewColumn                             TreeViewColumn
│   ╰── GtkRecentFilter
├── GtkAccelGroup
├── GtkAccelMap
├── AtkObject
│   ╰── GtkAccessible
├─✗ GtkAction
│   ├─✗ GtkToggleAction
│   │   ╰─✗ GtkRadioAction
│   ╰─✗ GtkRecentAction
├─✗ GtkActionGroup
├─✗ GApplication                                      Gnome::Gio not implemented
│   ╰─✗ GtkApplication                                Depends on Gio
├── GtkBuilder                                        Builder
├── GtkCellAreaContext
├── GtkClipboard
├── GtkCssProvider                                    CssProvider
├── GtkEntryBuffer
├── GtkEntryCompletion
├── GtkEventController
│   ├── GtkEventControllerKey
│   ├── GtkEventControllerMotion
│   ├── GtkEventControllerScroll
│   ├── GtkGesture
│   │   ├── GtkGestureSingle
│   │   │   ├── GtkGestureDrag
│   │   │   │   ╰── GtkGesturePan
│   │   │   ├── GtkGestureLongPress
│   │   │   ├── GtkGestureMultiPress
│   │   │   ├── GtkGestureStylus
│   │   │   ╰── GtkGestureSwipe
│   │   ├── GtkGestureRotate
│   │   ╰── GtkGestureZoom
│   ╰── GtkPadController
├── GtkIconFactory
├── GtkIconTheme
├── GtkIMContext
│   ├── GtkIMContextSimple
│   ╰── GtkIMMulticontext
├── GtkListStore                                      ListStore
├── GMountOperation
│   ╰── GtkMountOperation
├── GEmblemedIcon
│   ╰─✗ GtkNumerableIcon
├── GtkPageSetup
├── GtkPrinter
├── GtkPrintContext
├── GtkPrintJob
├── GtkPrintOperation
├── GtkPrintSettings
├── GtkRcStyle
├── GtkRecentManager
├── GtkSettings
├── GtkSizeGroup
├─✗ GtkStatusIcon
├─✗ GtkStyle
├── GtkStyleContext                                   StyleContext
├── GtkTextBuffer                                     TextBuffer
├── GtkTextChildAnchor
├── GtkTextMark
├── GtkTextTag                                        TextTag
├── GtkTextTagTable                                   TextTagTable
├─✗ GtkThemingEngine
├── GtkTreeModelFilter
├── GtkTreeModelSort
├── GtkTreeSelection
├── GtkTreeStore
├─✗ GtkUIManager
├── GtkWindowGroup
├── GtkTooltip
╰── GtkPrintBackend
GInterface
├── GtkBuildable                                      Buildable
├── GtkActionable
├─✗ GtkActivatable
├── GtkAppChooser
├── GtkCellLayout
├── GtkCellEditable
├── GtkOrientable                                     Orientable
├── GtkColorChooser                                   ColorChooser
├── GtkStyleProvider                                  StyleProvider
├── GtkEditable
├── GtkFileChooser                                    FileChooser
├── GtkFontChooser
├── GtkScrollable
├── GtkTreeModel                                      TreeModel
├── GtkTreeDragSource
├── GtkTreeDragDest
├── GtkTreeSortable
├── GtkPrintOperationPreview
├── GtkRecentChooser
╰── GtkToolShell
GBoxed
├── GtkPaperSize
├── GtkTextIter                                       TextIter
├── GtkSelectionData
├── GtkRequisition
├── GtkBorder                                         Border
├── GtkTreeIter                                       TreeIter
├── GtkCssSection                                     CssSection
├── GtkTreePath                                       TreePath
├   GtkTreeRowReference                               TreeRowReference
│                                                     Extracted from TreeModel
├── GtkIconSet
╰── GtkTargetList
```
