package Prty::Html::Form::Layout;
use base qw/Prty::Hash/;

use strict;
use warnings;

our $VERSION = 1.06;

use Prty::Template;

# -----------------------------------------------------------------------------

=encoding utf8

=head1 NAME

Prty::Html::Form::Layout - HTML-Formular mit freiem Layout

=head1 BASE CLASS

L<Prty::Hash>

=head1 DESCRIPTION

Die Klasse dient der Erzeugung von HTML-Formularen mit freiem
Layout. D.h. der HTML-Code "um die Widgets herum" wird von der
Klasse nicht vorgegeben, sondern als Objektattribut gesetzt,
ebenso wie die Liste der Widgets. Die Klasse setzt die Widgets in
das Layout ein.

Für jedes Widget enthält das Layout einen Platzhalter, der sich
aus dem Namen des Widget herleitet, indem der Widget-Name in
Großbuchstaben gewandelt und um zwei Unterstriche vorne und hinten
ergänzt wird.

Beispiel:

    Widget-Name  Platzhalter
    -----------  -----------
    vorname      __VORNAME__

=head1 ATTRIBUTES

=over 4

=item form => \@keyVal (Default: undef)

Eigenschaften des C<form>-Tag. Ist das Attribut nicht gesetzt,
wird kein C<form>-Tag erzeugt.

=item layout => $html (Default: '')

Der HTML-Code des Layouts. In das Layout wird der HTML-Code der
Widgets eingesetzt.

=item widgets => \@widgets (Default: [])

Liste der Widgets, die in das Layout eingesetzt werden.

=back

=head1 EXAMPLE

Der Code

    Prty::Html::Form::Layout->html($h,
        layout=>Prty::Html::Table::Simple->html($h,
            class=>'form',
            rows=>[
                [['Vorname:'],['__VORNAME__']],
                [['Nachname:'],['__NACHNAME__']],
                [[''],['__AKTION__']],
            ],
        ),
        widgets=>[
            Prty::Html::Widget::TextField->new(
                name=>'vorname',
                value=>'Linus',
            ),
            Prty::Html::Widget::TextField->new(
                name=>'nachname',
                value=>'Seitz',
            ),
            Prty::Html::Widget::Button->new(
                id=>'speichern',
                name=>'aktion',
                value=>'speichern',
                content=>'Speichern',
            ),
        ],
    );

erzeugt

    <table class="form" cellspacing="0">
    <tbody>
      <tr>
        <td>Vorname:</td>
        <td><input type="text" name="vorname" value="Linus" /></td>
      </tr>
      <tr>
        <td>Nachname:</td>
        <td><input type="text" name="nachname" value="Seitz" /></td>
      </tr>
      <tr>
        <td></td>
        <td><button id="speichern" name="aktion" type="button"
          value="speichern">Speichern</button></td>
      </tr>
    </tbody>
    </table>

=head1 METHODS

=head2 Konstruktor

=head3 new() - Instanziiere Objekt

=head4 Synopsis

    $e = $class->new(@attVal);

=head4 Description

Instanziiere ein Formular-Objekt und liefere eine Referenz auf
dieses Objekt zurück.

=cut

# -----------------------------------------------------------------------------

sub new {
    my $class = shift;
    # @_: @attVal

    my $self = $class->SUPER::new(
        form=>undef,
        layout=>'',
        widgets=>[],
    );
    $self->set(@_);

    return $self;
}

# -----------------------------------------------------------------------------

=head2 Objektmethoden

=head3 html() - Generiere HTML

=head4 Synopsis

    $html = $e->html($h);
    $html = $class->html($h,@attVal);

=head4 Description

Generiere den HTML-Code des Formular-Objekts und liefere diesen
zurück. Als Klassenmethode gerufen, wird das Objekt intern erzeugt
und mit den Attributen @attVal instanziiert.

=cut

# -----------------------------------------------------------------------------

sub html {
    my $this = shift;
    my $h = shift;

    my $self = ref $this? $this: $this->new(@_);

    my ($formA,$layout,$widgetA) = $self->get(qw/form layout widgets/);

    my $hidden = '';
    my @keyVal;
    for my $w (@$widgetA) {
        if ($w->hidden) {
            $hidden .= $w->html($h);
            next;
        }
        push @keyVal,sprintf('__%s__',uc $w->name),$w->html($h);
    }

    if ($formA) {
        $layout = $h->tag('form',
            @$formA,
            $layout
        );
    }

    my $tpl = Prty::Template->new('text',\$layout,
        singleReplace=>1,
    );
    $tpl->replace(@keyVal);
    $tpl->removeOptional;
        
    return $hidden.$tpl->asStringNL;
}

# -----------------------------------------------------------------------------

=head1 VERSION

1.06

=head1 AUTHOR

Frank Seitz, L<http://fseitz.de/>

=head1 COPYRIGHT

Copyright (C) 2016 Frank Seitz

=head1 LICENSE

This code is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut

# -----------------------------------------------------------------------------

1;

# eof
