package Firewall::Policy::Designer::Srx;

#------------------------------------------------------------------------------
# 加载扩展模块
#------------------------------------------------------------------------------
use Moose;
use namespace::autoclean;

#------------------------------------------------------------------------------
# 加载项目模块
#------------------------------------------------------------------------------
use Firewall::Utils::Date;

#------------------------------------------------------------------------------
# 继承 Firewall::Policy::Designer::Role 方法和属性
#------------------------------------------------------------------------------
with 'Firewall::Policy::Designer::Role';

#------------------------------------------------------------------------------
# JUNIPER防火墙策略设计相关模块
#------------------------------------------------------------------------------
sub createRule {
  my $self = shift;

  # 先检查涉及到的 addr or srv 在防火墙上有没有已经存在的名字，没有就需要创建
  my $action   = $self->searcherReportFwInfo->action;
  my $mappings = $self->checkAndCreateAddrOrSrvOrNat($action->{new});
  my ($fromZone, $toZone) = ($self->searcherReportFwInfo->fromZone, $self->searcherReportFwInfo->toZone);

  my @commands;
  my $randNum  = $$ || sprintf('%05d', int(rand(99999)));
  my $ruleName = 'p_' . Firewall::Utils::Date->new->getFormatedDate('yyyymmdd_hhmiss') . "_$randNum";
  for my $type (keys %{$mappings}) {
    push @commands,
      map {"set security policies from-zone $fromZone to-zone $toZone policy $ruleName match source-address $_"}
      @{$mappings->{$type}}
      if $type eq 'src';
    push @commands,
      map {"set security policies from-zone $fromZone to-zone $toZone policy $ruleName match destination-address $_"}
      @{$mappings->{$type}}
      if $type eq 'dst';
    push @commands,
      map {"set security policies from-zone $fromZone to-zone $toZone policy $ruleName match application $_"}
      @{$mappings->{$type}}
      if $type eq 'srv';
  }
  push @commands, "set security policies from-zone $fromZone to-zone $toZone policy $ruleName then permit";
  $self->addCommands(@commands);
}

#------------------------------------------------------------------------------
# JUNIPER防火墙策略调整函数
#------------------------------------------------------------------------------
sub modifyRule {
  my $self = shift;

  # 先检查涉及到的 addr or srv 在防火墙上有没有已经存在的名字，没有就需要创建
  my $mappings = $self->checkAndCreateAddrOrSrvOrNat($self->searcherReportFwInfo->action->{add});
  if (my $action = $self->searcherReportFwInfo->action->{new}) {
    $self->checkAndCreateAddrOrSrvOrNat($action);
  }
  my ($fromZone, $toZone) = ($self->searcherReportFwInfo->fromZone, $self->searcherReportFwInfo->toZone);
  my $ruleName = $self->searcherReportFwInfo->ruleObj->ruleName;

  my @commands;
  for my $type (keys %{$mappings}) {
    push @commands,
      map {"set security policies from-zone $fromZone to-zone $toZone policy $ruleName match source-address $_"}
      @{$mappings->{$type}}
      if $type eq 'src';
    push @commands,
      map {"set security policies from-zone $fromZone to-zone $toZone policy $ruleName match destination-address $_"}
      @{$mappings->{$type}}
      if $type eq 'dst';
    push @commands,
      map {"set security policies from-zone $fromZone to-zone $toZone policy $ruleName match application $_"}
      @{$mappings->{$type}}
      if $type eq 'srv';
  }
  $self->addCommands(@commands);
}

#------------------------------------------------------------------------------
# 静态地址转换
#------------------------------------------------------------------------------
sub createStaticNat {
  my ($self, $natInfo) = @_;

  # set security nat static rule-set dmz_static_nat rule host_172_29_16_1 match destination-address 172.29.16.1/32
  # set security nat static rule-set dmz_static_nat rule host_172_29_16_1 then static-nat prefix 10.11.100.97/32

  my $natName = 'host_' . join('_', (split(/[\.\/]/, $natInfo->{natInfo}{natIp}))[0, 1, 2, 3]);
  my @commands;
  my $realIp  = $natInfo->{realIp};
  my $ruleSet = $natInfo->{ruleSet};
  my $natIp   = $natInfo->{natInfo}{natIp};
  if (defined $ruleSet) {
    push @commands, "set security nat static rule-set $ruleSet rule $natName match destination-address $natIp";
    push @commands, "set security nat static rule-set $ruleSet rule $natName then static-nat prefix $realIp";
    $self->addCommands(@commands);
  }
  else {
    push @commands, "set security nat static rule-set static-nat rule $natName match destination-address $natIp";
    push @commands, "set security nat static rule-set static-nat rule $natName then static-nat prefix $realIp";
    $self->addCommands(@commands);
  }
}

#------------------------------------------------------------------------------
# 动态地址转换
#------------------------------------------------------------------------------
sub createDyNat {
  my ($self, $param) = @_;
  my @commands;
  my $fromZone = $self->searcherReportFwInfo->fromZone;
  my $toZone   = $self->searcherReportFwInfo->toZone;
  my $srvMap   = $self->searcherReportFwInfo->srvMap;

  for my $type (keys %{$param}) {
    for my $natIps (values %{$param->{$type}}) {
      my ($natDirection, $ruleSet, $ruleName, $poolName, $natInfo);
      $natDirection = 'source'      if $type eq 'natSrc';
      $natDirection = 'destination' if $type eq 'natDst';
      $natInfo      = (values %{$natIps})[0]->{natInfo};
      $ruleSet      = (values %{$natIps})[0]->{ruleSet};
      my $randNum = sprintf('%05d', int(rand(99999)));
      $ruleName = "rule_$randNum";
      if (not defined $ruleSet) {
        $ruleSet = $fromZone . '_' . $toZone;
        push @commands, "set security nat $natDirection rule-set $ruleSet from zone $fromZone";
        push @commands, "set security nat $natDirection rule-set $ruleSet to zone $toZone";
      }
      else {
        while (defined $self->searcherReportFwInfo->parser->getDynamicNat($ruleSet, $ruleName)) {
          $randNum  = sprintf('%05d', int(rand(99999)));
          $ruleName = "rule_$randNum";
        }
      }

      if ($type eq 'natSrc') {
        $poolName = $self->getOrCreatePool($type, $natIps);
        for my $natIp (keys %{$natIps}) {
          push @commands, "set security nat source rule-set $ruleSet rule $ruleName match source-address $natIp";
        }
        if (defined $natInfo->{natOption} and $natInfo->{natOption} =~ /d/) {
          for my $dstIp (keys %{$self->searcherReportFwInfo->dstMap}) {
            push @commands, "set security nat source rule-set $ruleSet rule $ruleName match destination-address $dstIp"
              if $dstIp ne '0.0.0.0/0';
          }
        }
        push @commands, "set security nat source rule-set $ruleSet rule $ruleName then source-nat $poolName";
      }
      elsif ($type eq 'natDst') {

        # 需要做端口NAT，可能存在多端口情况，每个端口都需要一套NAT策略
        if (defined $natInfo->{natOption} and $natInfo->{natOption} =~ /p/) {
          for my $portInfo (values %{$self->searcherReportFwInfo->srvMap}) {
            $poolName = $self->getOrCreatePool($type, $natIps, $portInfo);
            $randNum  = sprintf('%05d', int(rand(99999)));
            $ruleName = "rule_$randNum";
            while (defined $self->searcherReportFwInfo->parser->getDynamicNat($ruleSet, $ruleName)) {
              $randNum  = sprintf('%05d', int(rand(99999)));
              $ruleName = "rule_$randNum";
            }

            if (defined $natInfo->{natOption} and $natInfo->{natOption} =~ /s/) {
              for my $srcIp (keys %{$self->searcherReportFwInfo->srcMap}) {
                push @commands, "set security nat destination rule-set $ruleSet rule $ruleName match source-address $srcIp"
                  if $srcIp ne '0.0.0.0/0';
              }
            }
            push @commands,
              "set security nat destination rule-set $ruleSet rule $ruleName match destination-address $natInfo->{natIp}";
            my $natPort = $portInfo->{origin};
            $natPort = $portInfo->{natPort} if defined $portInfo->{natPort};
            my ($pro, $port) = split('/', $natPort);
            push @commands, "set security nat destination rule-set $ruleSet rule $ruleName match destination-port $port";
            push @commands, "set security nat destination rule-set $ruleSet rule $ruleName then destination-nat $poolName";
          }
        }
        else {
          $poolName = $self->getOrCreatePool($type, $natIps);
          $randNum  = sprintf('%05d', int(rand(99999)));
          $ruleName = "rule_$randNum";
          while (defined $self->searcherReportFwInfo->parser->getDynamicNat($ruleSet, $ruleName)) {
            $randNum  = sprintf('%05d', int(rand(99999)));
            $ruleName = "rule_$randNum";
          }

          if (defined $natInfo->{natOption} and $natInfo->{natOption} =~ /s/) {
            for my $srcIp (keys %{$self->searcherReportFwInfo->srcMap}) {
              push @commands, "set security nat destination rule-set $ruleSet rule $ruleName match source-address $srcIp"
                if $srcIp ne '0.0.0.0/0';
            }
          }
          push @commands,
            "set security nat destination rule-set $ruleSet rule $ruleName match destination-address $natInfo->{natIp}";
          push @commands, "set security nat destination rule-set $ruleSet rule $ruleName then destination-nat $poolName";
        }
      }
    }
  }
  $self->addCommands(@commands) if @commands != 0;
}

#------------------------------------------------------------------------------
# 查询地址转换池
#------------------------------------------------------------------------------
sub getOrCreatePool {
  my ($self, $type, $natIps, $portInfo) = @_;
  my $poolName;
  my $natInfo = (values %{$natIps})[0]->{natInfo};
  my $poolIp;
  if ($type eq 'natSrc') {
    if (defined $natInfo->{natIp}) {
      $poolIp = $natInfo->{natIp};
    }
    else {
      return "interface";
    }
  }
  $poolIp = $natInfo->{natIp} if $type eq 'natSrc';
  my $natPools = $self->searcherReportFwInfo->parser->elements->natPool;

  # print dumper $self->searcherReportFwInfo->srvMap;exit;
  $poolIp = $natInfo->{natIp}                if $type eq 'natSrc';
  $poolIp = (values %{$natIps})[0]->{realIp} if $type eq 'natDst';
  $poolIp =~ s/$/\/32/ if $poolIp !~ /\//;
  my ($ip, $mask) = split('/', $poolIp);
  my $poolRange = Firewall::Utils::Ip->new->getRangeFromIpMask($ip, $mask);
  my @commands;
  if ($type eq 'natSrc') {
    for my $pool (values %{$natPools}) {
      if ($pool->natDirection eq 'source' and $pool->poolRange->isEqual($poolRange)) {
        return "pool " . $pool->poolName;
      }
    }

    #没找到已有pool,新建
    # set security nat source pool srcnat_pool_3 address 202.69.21.46/32 to 202.69.21.46/32
    my $randNum = sprintf('%05d', int(rand(99999)));
    $poolName = "srcnat_pool_$randNum";
    while (defined $self->searcherReportFwInfo->parser->getNatPool($poolName)) {
      $randNum  = sprintf('%05d', int(rand(99999)));
      $poolName = "srcnat_pool_$randNum";
    }
    push @commands, "set security nat source pool $poolName address $poolIp to $poolIp";
  }
  elsif ($type eq 'natDst') {
    for my $pool (values %{$natPools}) {
      if ($pool->natDirection eq 'destination' and $pool->poolRange->isEqual($poolRange)) {
        if (defined $portInfo) {
          if (defined $pool->poolPort and $pool->poolPort eq $portInfo->{origin}) {
            return "pool " . $pool->poolName;
          }
        }
        else {
          return "pool " . $pool->poolName;
        }
      }
    }

    # 没有找到pool,需要新建
    # set security nat destination pool dstnat_pool_23 address 172.28.32.75/32
    # set security nat destination pool dstnat_pool_23 address port 80
    my $randNum = sprintf('%05d', int(rand(99999)));
    $poolName = "dstnat_pool_$randNum";
    while (defined $self->searcherReportFwInfo->parser->getNatPool($poolName)) {
      $randNum  = sprintf('%05d', int(rand(99999)));
      $poolName = "dstnat_pool_$randNum";
    }
    push @commands, "set security nat destination pool $poolName address $poolIp";
    if (defined $portInfo) {
      my ($pro, $port) = split('/', $portInfo->{origin});
      push @commands, "set security nat destination pool $poolName address port $port";
    }
  }
  $self->addCommands(@commands) if @commands != 0;
  return "pool " . $poolName;
}

#------------------------------------------------------------------------------
# 创建源地址对象
#------------------------------------------------------------------------------
sub createSrc {
  my ($self, $addr) = @_;
  return $self->createAddress($self->searcherReportFwInfo->fromZone, $addr);
}

#------------------------------------------------------------------------------
# 创建目的地址对象
#------------------------------------------------------------------------------
sub createDst {
  my ($self, $addr) = @_;
  return $self->createAddress($self->searcherReportFwInfo->toZone, $addr);
}

#------------------------------------------------------------------------------
# 创建服务端口对象
#------------------------------------------------------------------------------
sub createSrv {
  my ($self, $srv) = @_;
  return $self->createService($srv);
}

#------------------------------------------------------------------------------
# 创建地址函数
#------------------------------------------------------------------------------
sub createAddress {
  my ($self, $zoneName, $addr) = @_;

  # set security zones security-zone l2-untrust address-book address host_10.35.194.90 10.35.194.90/32
  my ($ip, $mask) = split('/', $addr);
  my ($addressName, $ipString, $ipMin, $ipMax);
  if (not defined $mask) {
    if ($ip =~ /(\d+\.)(\d+\.)(\d+\.)(\d+)-(\d+)/) {
      ($ipMin, $ipMax) = ($1 . $2 . $3 . $4, $1 . $2 . $3 . $5);
      $addressName = "Range_$ip";
    }
  }
  elsif ($mask == 32) {
    $ipString    = $ip;
    $addressName = "Host_$ip";
  }
  elsif ($mask == 0) {
    return 'any';
  }
  else {
    $ipString    = Firewall::Utils::Ip->new->getNetIpFromIpMask($ip, $mask);
    $addressName = "Net_$ipString/$mask";
  }
  my $command = "set security zones security-zone $zoneName address-book address $addressName $ipString/$mask" if defined $mask;
  $command = "set security zones security-zone $zoneName address-book address $addressName range-address $ipMin to $ipMax"
    if defined $ipMin;
  $self->addCommands($command);
  return $addressName;
}

#------------------------------------------------------------------------------
# 创建服务端口函数
#------------------------------------------------------------------------------
sub createService {
  my ($self,     $srv)  = @_;
  my ($protocol, $port) = split('/', $srv);
  $protocol = lc $protocol;
  return if $protocol ne 'tcp' and $protocol ne 'udp';

  my ($serviceName, $dstPort);
  if ($port =~ /^(?<portMin>\d+)\-(?<portMax>\d+)$/o) {
    $serviceName = uc($protocol) . "_" . $+{portMin} . "-" . $+{portMax};
    $dstPort     = $+{portMin} . "-" . $+{portMax};
  }
  elsif ($port =~ /^\d+$/o) {
    $serviceName = uc($protocol) . "_" . $port;
    $dstPort     = $port . "-" . $port;
  }
  else {
    confess "ERROR: $port is not a port";
  }
  my @commands;
  push @commands, "set applications application $serviceName term $serviceName protocol $protocol";
  push @commands, "set applications application $serviceName term $serviceName source-port 0-65535";
  push @commands, "set applications application $serviceName term $serviceName destination-port $dstPort";

  # my $command = qq{set service "$serviceName" protocol $protocol src-port 0-65535 dst-port $dstPort};
  $self->addCommands(@commands);
  return $serviceName;
}

__PACKAGE__->meta->make_immutable;
1;
