#!/usr/bin/perl -T -w

use strict;
use warnings;

use Test::More;
use Asm::X86 qw(
	@regs8_intel @regs16_intel @segregs_intel @regs32_intel @regs64_intel
	@regs_mm_intel @regs_intel @regs_fpu_intel @regs_opmask_intel

	is_reg_intel is_reg8_intel is_reg16_intel is_reg32_intel is_reg64_intel
	is_reg_mm_intel is_segreg_intel is_reg_fpu_intel is_reg_opmask_intel
	is_addressable32_intel is_r32_in64_intel

	is_reg_att is_reg8_att is_reg16_att is_reg32_att is_reg64_att
	is_reg_mm_att is_segreg_att is_reg_fpu_att is_reg_opmask_att
	is_addressable32_att is_r32_in64_att

	is_reg is_reg8 is_reg16 is_reg32 is_reg64
	is_reg_mm is_segreg is_reg_fpu is_reg_opmask
	is_addressable32 is_r32_in64
);

my @addressable32 = (
		'eax',	'ebx', 'ecx', 'edx', 'esi', 'edi', 'esp', 'ebp',
);

my @r32_in64 = (
		'r8d', 'r8l', 'r9d', 'r9l', 'r10d', 'r10l', 'r11d', 'r11l',
		'r12d', 'r12l', 'r13d', 'r13l', 'r14d', 'r14l', 'r15d', 'r15l',
);

my @invalid_regs = ('axmm6', 'cax', 'abx', 'dal', 'ald', 'rsid',
	'eabx', 'ceax', 'ebxc', 'amm1', 'mm30', 'r15db',
	'ar15d', 'ads', 'esx', 'ast0',
	'st5b', 'k02', 'cal', 'dax', 'reax', 'amm0', 'xmm',
	'ymm', 'zmm', 'k12', 'axh', 'eaxl', 'r10ld', 'mm0l',
	'xmm0d', 'ymm0l', 'zmm0d', 'k3l', '[eax', 'eax]'
);

# Test::More:
plan tests => 9
	+ (@regs8_intel + 1) * 33
	+ (@regs16_intel + 1) * 31 + 22
	+ (@regs32_intel + 1) * 29 + 22 + 22
	+ (@regs64_intel + 1) * 33
	+ (@regs_mm_intel + 4) * 33
	+ (@regs_fpu_intel + 1) * 33
	+ (@segregs_intel + 1) * 33
	+ (@regs_opmask_intel + 1) * 33
	+ @addressable32 * 33
	+ @r32_in64 * 33
	+ @invalid_regs * 33
	;

cmp_ok ( $#regs8_intel,   '>', 0, 'Non-empty 8-bit register list' );
cmp_ok ( $#regs16_intel,  '>', 0, 'Non-empty 16-bit register list' );
cmp_ok ( $#segregs_intel, '>', 0, 'Non-empty segment register list' );
cmp_ok ( $#regs32_intel,  '>', 0, 'Non-empty 32-bit register list' );
cmp_ok ( $#regs64_intel,  '>', 0, 'Non-empty 64-bit register list' );
cmp_ok ( $#regs_mm_intel, '>', 0, 'Non-empty multimedia register list' );
cmp_ok ( $#regs_fpu_intel,'>', 0, 'Non-empty FPU register list' );
cmp_ok ( $#regs_opmask_intel,'>', 0, 'Non-empty opmask register list' );
cmp_ok ( $#regs_intel,    '>', 0, 'Non-empty register list' );

foreach ((@regs8_intel, 'al')) {

	is ( is_reg_intel ($_), 1, "'$_' is a valid Intel-syntax register" );
	is ( is_reg8_intel ($_), 1, "'$_' is a valid Intel-syntax 8-bit register" );
	is ( is_reg16_intel ($_), 0, "'$_' is not a valid Intel-syntax 16-bit register" );
	is ( is_reg32_intel ($_), 0, "'$_' is not a valid Intel-syntax 32-bit register" );
	is ( is_reg64_intel ($_), 0, "'$_' is not a valid Intel-syntax 64-bit register" );
	is ( is_reg_mm_intel ($_), 0, "'$_' is not a valid Intel-syntax multimedia register" );
	is ( is_segreg_intel ($_), 0, "'$_' is not a valid Intel-syntax segment register" );
	is ( is_reg_fpu_intel ($_), 0, "'$_' is not a valid Intel-syntax FPU register" );
	is ( is_reg_opmask_intel ($_), 0, "'$_' is not a valid Intel-syntax opmask register" );
	is ( is_addressable32_intel ($_), 0, "'$_' is not a valid Intel-syntax 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64_intel ($_), 0, "'$_' is not a valid Intel-syntax 32-in-64-bit register" );

	is ( is_reg_att ($_), 0, "'$_' is a valid AT&T-syntax register" );
	is ( is_reg8_att ($_), 0, "'$_' is a valid AT&T-syntax 8-bit register" );
	is ( is_reg16_att ($_), 0, "'$_' is not a valid AT&T-syntax 16-bit register" );
	is ( is_reg32_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-bit register" );
	is ( is_reg64_att ($_), 0, "'$_' is not a valid AT&T-syntax 64-bit register" );
	is ( is_reg_mm_att ($_), 0, "'$_' is not a valid AT&T-syntax multimedia register" );
	is ( is_segreg_att ($_), 0, "'$_' is not a valid AT&T-syntax segment register" );
	is ( is_reg_fpu_att ($_), 0, "'$_' is not a valid AT&T-syntax FPU register" );
	is ( is_reg_opmask_att ($_), 0, "'$_' is not a valid AT&T-syntax opmask register" );
	is ( is_addressable32_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-in-64-bit register" );

	is ( is_reg ($_), 1, "'$_' is a valid register" );
	is ( is_reg8 ($_), 1, "'$_' is a valid 8-bit register" );
	is ( is_reg16 ($_), 0, "'$_' is not a valid 16-bit register" );
	is ( is_reg32 ($_), 0, "'$_' is not a valid 32-bit register" );
	is ( is_reg64 ($_), 0, "'$_' is not a valid 64-bit register" );
	is ( is_reg_mm ($_), 0, "'$_' is not a valid multimedia register" );
	is ( is_segreg ($_), 0, "'$_' is not a valid segment register" );
	is ( is_reg_fpu ($_), 0, "'$_' is not a valid FPU register" );
	is ( is_reg_opmask ($_), 0, "'$_' is not a valid opmask register" );
	is ( is_addressable32 ($_), 0, "'$_' is not a valid 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64 ($_), 0, "'$_' is not a valid 32-in-64-bit register" );
}

foreach ((@regs16_intel, 'ax')) {

	is ( is_reg_intel ($_), 1, "'$_' is a valid Intel-syntax register" );
	is ( is_reg8_intel ($_), 0, "'$_' is not a valid Intel-syntax 8-bit register" );
	is ( is_reg16_intel ($_), 1, "'$_' is a valid Intel-syntax 16-bit register" );
	is ( is_reg32_intel ($_), 0, "'$_' is not a valid Intel-syntax 32-bit register" );
	is ( is_reg64_intel ($_), 0, "'$_' is not a valid Intel-syntax 64-bit register" );
	is ( is_reg_mm_intel ($_), 0, "'$_' is not a valid Intel-syntax multimedia register" );
	# NOTE: some 16-bit registers are segment registers
	is ( is_reg_fpu_intel ($_), 0, "'$_' is not a valid Intel-syntax FPU register" );
	is ( is_reg_opmask_intel ($_), 0, "'$_' is not a valid Intel-syntax opmask register" );
	is ( is_addressable32_intel ($_), 0, "'$_' is not a valid Intel-syntax 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64_intel ($_), 0, "'$_' is not a valid Intel-syntax 32-in-64-bit register" );

	is ( is_reg_att ($_), 0, "'$_' is a valid AT&T-syntax register" );
	is ( is_reg8_att ($_), 0, "'$_' is a valid AT&T-syntax 8-bit register" );
	is ( is_reg16_att ($_), 0, "'$_' is not a valid AT&T-syntax 16-bit register" );
	is ( is_reg32_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-bit register" );
	is ( is_reg64_att ($_), 0, "'$_' is not a valid AT&T-syntax 64-bit register" );
	is ( is_reg_mm_att ($_), 0, "'$_' is not a valid AT&T-syntax multimedia register" );
	is ( is_segreg_att ($_), 0, "'$_' is not a valid AT&T-syntax segment register" );
	is ( is_reg_fpu_att ($_), 0, "'$_' is not a valid AT&T-syntax FPU register" );
	is ( is_reg_opmask_att ($_), 0, "'$_' is not a valid AT&T-syntax opmask register" );
	is ( is_addressable32_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-in-64-bit register" );

	is ( is_reg ($_), 1, "'$_' is a valid register" );
	is ( is_reg8 ($_), 0, "'$_' is a valid 8-bit register" );
	is ( is_reg16 ($_), 1, "'$_' is not a valid 16-bit register" );
	is ( is_reg32 ($_), 0, "'$_' is not a valid 32-bit register" );
	is ( is_reg64 ($_), 0, "'$_' is not a valid 64-bit register" );
	# NOTE: some 16-bit registers are segment registers
	is ( is_reg_mm ($_), 0, "'$_' is not a valid multimedia register" );
	is ( is_reg_fpu ($_), 0, "'$_' is not a valid FPU register" );
	is ( is_reg_opmask ($_), 0, "'$_' is not a valid opmask register" );
	is ( is_addressable32 ($_), 0, "'$_' is not a valid 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64 ($_), 0, "'$_' is not a valid 32-in-64-bit register" );
}

is ( is_reg_intel ('cs'), 1, "'cs' is a valid Intel-syntax register" );
is ( is_reg8_intel ('cs'), 0, "'cs' is a valid Intel-syntax 8-bit register" );
is ( is_reg16_intel ('cs'), 1, "'cs' is not a valid Intel-syntax 16-bit register" );
is ( is_reg32_intel ('cs'), 0, "'cs' is not a valid Intel-syntax 32-bit register" );
is ( is_reg64_intel ('cs'), 0, "'cs' is not a valid Intel-syntax 64-bit register" );
is ( is_reg_mm_intel ('cs'), 0, "'cs' is not a valid Intel-syntax multimedia register" );
is ( is_segreg_intel ('cs'), 1, "'cs' is not a valid Intel-syntax segment register" );
is ( is_reg_fpu_intel ('cs'), 0, "'cs' is not a valid Intel-syntax FPU register" );
is ( is_reg_opmask_intel ('cs'), 0, "'cs' is not a valid Intel-syntax opmask register" );
is ( is_addressable32_intel ('cs'), 0, "'cs' is not a valid Intel-syntax 32-bit register which can be used for 32-bit addressing" );
is ( is_r32_in64_intel ('cs'), 0, "'cs' is not a valid Intel-syntax 32-in-64-bit register" );

is ( is_reg ('cs'), 1, "'cs' is a valid register" );
is ( is_reg8 ('cs'), 0, "'cs' is a valid 8-bit register" );
is ( is_reg16 ('cs'), 1, "'cs' is not a valid 16-bit register" );
is ( is_reg32 ('cs'), 0, "'cs' is not a valid 32-bit register" );
is ( is_reg64 ('cs'), 0, "'cs' is not a valid 64-bit register" );
is ( is_reg_mm ('cs'), 0, "'cs' is not a valid multimedia register" );
is ( is_segreg ('cs'), 1, "'cs' is not a valid segment register" );
is ( is_reg_fpu ('cs'), 0, "'cs' is not a valid FPU register" );
is ( is_reg_opmask ('cs'), 0, "'cs' is not a valid opmask register" );
is ( is_addressable32 ('cs'), 0, "'cs' is not a valid 32-bit register which can be used for 32-bit addressing" );
is ( is_r32_in64 ('cs'), 0, "'cs' is not a valid 32-in-64-bit register" );

foreach ((@regs32_intel, 'eax')) {

	is ( is_reg_intel ($_), 1, "'$_' is a valid Intel-syntax register" );
	is ( is_reg8_intel ($_), 0, "'$_' is not a valid Intel-syntax 8-bit register" );
	is ( is_reg16_intel ($_), 0, "'$_' is not a valid Intel-syntax 16-bit register" );
	is ( is_reg32_intel ($_), 1, "'$_' is a valid Intel-syntax 32-bit register" );
	is ( is_reg64_intel ($_), 0, "'$_' is not a valid Intel-syntax 64-bit register" );
	is ( is_reg_mm_intel ($_), 0, "'$_' is not a valid Intel-syntax multimedia register" );
	is ( is_segreg_intel ($_), 0, "'$_' is not a valid Intel-syntax segment register" );
	is ( is_reg_fpu_intel ($_), 0, "'$_' is not a valid Intel-syntax FPU register" );
	is ( is_reg_opmask_intel ($_), 0, "'$_' is not a valid Intel-syntax opmask register" );
	# NOTE: most 32-bit general-purpose registers can be used for addressing
	# NOTE: some 32-bit registers are parts of 64-bit registers

	is ( is_reg_att ($_), 0, "'$_' is a valid AT&T-syntax register" );
	is ( is_reg8_att ($_), 0, "'$_' is a valid AT&T-syntax 8-bit register" );
	is ( is_reg16_att ($_), 0, "'$_' is not a valid AT&T-syntax 16-bit register" );
	is ( is_reg32_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-bit register" );
	is ( is_reg64_att ($_), 0, "'$_' is not a valid AT&T-syntax 64-bit register" );
	is ( is_reg_mm_att ($_), 0, "'$_' is not a valid AT&T-syntax multimedia register" );
	is ( is_segreg_att ($_), 0, "'$_' is not a valid AT&T-syntax segment register" );
	is ( is_reg_fpu_att ($_), 0, "'$_' is not a valid AT&T-syntax FPU register" );
	is ( is_reg_opmask_att ($_), 0, "'$_' is not a valid AT&T-syntax opmask register" );
	is ( is_addressable32_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-in-64-bit register" );

	is ( is_reg ($_), 1, "'$_' is a valid register" );
	is ( is_reg8 ($_), 0, "'$_' is a valid 8-bit register" );
	is ( is_reg16 ($_), 0, "'$_' is not a valid 16-bit register" );
	is ( is_reg32 ($_), 1, "'$_' is not a valid 32-bit register" );
	is ( is_reg64 ($_), 0, "'$_' is not a valid 64-bit register" );
	is ( is_reg_mm ($_), 0, "'$_' is not a valid multimedia register" );
	is ( is_segreg ($_), 0, "'$_' is not a valid segment register" );
	is ( is_reg_fpu ($_), 0, "'$_' is not a valid FPU register" );
	is ( is_reg_opmask ($_), 0, "'$_' is not a valid opmask register" );
	# NOTE: most 32-bit general-purpose registers can be used for addressing
	# NOTE: some 32-bit registers are parts of 64-bit registers
}

is ( is_reg_intel ('eax'), 1, "'eax' is a valid Intel-syntax register" );
is ( is_reg8_intel ('eax'), 0, "'eax' is not a valid Intel-syntax 8-bit register" );
is ( is_reg16_intel ('eax'), 0, "'eax' is not a valid Intel-syntax 16-bit register" );
is ( is_reg32_intel ('eax'), 1, "'eax' is not a valid Intel-syntax 32-bit register" );
is ( is_reg64_intel ('eax'), 0, "'eax' is a valid Intel-syntax 64-bit register" );
is ( is_reg_mm_intel ('eax'), 0, "'eax' is not a valid Intel-syntax multimedia register" );
is ( is_segreg_intel ('eax'), 0, "'eax' is not a valid Intel-syntax segment register" );
is ( is_reg_fpu_intel ('eax'), 0, "'eax' is not a valid Intel-syntax FPU register" );
is ( is_reg_opmask_intel ('eax'), 0, "'eax' is not a valid Intel-syntax opmask register" );
is ( is_addressable32_intel ('eax'), 1, "'eax' is not a valid Intel-syntax 32-bit register which can be used for 32-bit addressing" );
is ( is_r32_in64_intel ('eax'), 0, "'eax' is not a valid Intel-syntax 32-in-64-bit register" );

is ( is_reg ('eax'), 1, "'eax' is a valid register" );
is ( is_reg8 ('eax'), 0, "'eax' is a valid 8-bit register" );
is ( is_reg16 ('eax'), 0, "'eax' is not a valid 16-bit register" );
is ( is_reg32 ('eax'), 1, "'eax' is not a valid 32-bit register" );
is ( is_reg64 ('eax'), 0, "'eax' is not a valid 64-bit register" );
is ( is_reg_mm ('eax'), 0, "'eax' is not a valid multimedia register" );
is ( is_segreg ('eax'), 0, "'eax' is not a valid segment register" );
is ( is_reg_fpu ('eax'), 0, "'eax' is not a valid FPU register" );
is ( is_reg_opmask ('eax'), 0, "'eax' is not a valid opmask register" );
is ( is_addressable32 ('eax'), 1, "'eax' is not a valid 32-bit register which can be used for 32-bit addressing" );
is ( is_r32_in64 ('eax'), 0, "'eax' is not a valid 32-in-64-bit register" );

is ( is_reg_intel ('r9d'), 1, "'r9d' is a valid Intel-syntax register" );
is ( is_reg8_intel ('r9d'), 0, "'r9d' is not a valid Intel-syntax 8-bit register" );
is ( is_reg16_intel ('r9d'), 0, "'r9d' is not a valid Intel-syntax 16-bit register" );
is ( is_reg32_intel ('r9d'), 1, "'r9d' is not a valid Intel-syntax 32-bit register" );
is ( is_reg64_intel ('r9d'), 0, "'r9d' is a valid Intel-syntax 64-bit register" );
is ( is_reg_mm_intel ('r9d'), 0, "'r9d' is not a valid Intel-syntax multimedia register" );
is ( is_segreg_intel ('r9d'), 0, "'r9d' is not a valid Intel-syntax segment register" );
is ( is_reg_fpu_intel ('r9d'), 0, "'r9d' is not a valid Intel-syntax FPU register" );
is ( is_reg_opmask_intel ('r9d'), 0, "'r9d' is not a valid Intel-syntax opmask register" );
is ( is_addressable32_intel ('r9d'), 0, "'r9d' is not a valid Intel-syntax 32-bit register which can be used for 32-bit addressing" );
is ( is_r32_in64_intel ('r9d'), 1, "'r9d' is not a valid Intel-syntax 32-in-64-bit register" );

is ( is_reg ('r9d'), 1, "'r9d' is a valid register" );
is ( is_reg8 ('r9d'), 0, "'r9d' is a valid 8-bit register" );
is ( is_reg16 ('r9d'), 0, "'r9d' is not a valid 16-bit register" );
is ( is_reg32 ('r9d'), 1, "'r9d' is not a valid 32-bit register" );
is ( is_reg64 ('r9d'), 0, "'r9d' is not a valid 64-bit register" );
is ( is_reg_mm ('r9d'), 0, "'r9d' is not a valid multimedia register" );
is ( is_segreg ('r9d'), 0, "'r9d' is not a valid segment register" );
is ( is_reg_fpu ('r9d'), 0, "'r9d' is not a valid FPU register" );
is ( is_reg_opmask ('r9d'), 0, "'r9d' is not a valid opmask register" );
is ( is_addressable32 ('r9d'), 0, "'r9d' is not a valid 32-bit register which can be used for 32-bit addressing" );
is ( is_r32_in64 ('r9d'), 1, "'r9d' is not a valid 32-in-64-bit register" );

foreach ((@regs64_intel, 'rax')) {

	is ( is_reg_intel ($_), 1, "'$_' is a valid Intel-syntax register" );
	is ( is_reg8_intel ($_), 0, "'$_' is not a valid Intel-syntax 8-bit register" );
	is ( is_reg16_intel ($_), 0, "'$_' is not a valid Intel-syntax 16-bit register" );
	is ( is_reg32_intel ($_), 0, "'$_' is not a valid Intel-syntax 32-bit register" );
	is ( is_reg64_intel ($_), 1, "'$_' is a valid Intel-syntax 64-bit register" );
	is ( is_reg_mm_intel ($_), 0, "'$_' is not a valid Intel-syntax multimedia register" );
	is ( is_segreg_intel ($_), 0, "'$_' is not a valid Intel-syntax segment register" );
	is ( is_reg_fpu_intel ($_), 0, "'$_' is not a valid Intel-syntax FPU register" );
	is ( is_reg_opmask_intel ($_), 0, "'$_' is not a valid Intel-syntax opmask register" );
	is ( is_addressable32_intel ($_), 0, "'$_' is not a valid Intel-syntax 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64_intel ($_), 0, "'$_' is not a valid Intel-syntax 32-in-64-bit register" );

	is ( is_reg_att ($_), 0, "'$_' is a valid AT&T-syntax register" );
	is ( is_reg8_att ($_), 0, "'$_' is a valid AT&T-syntax 8-bit register" );
	is ( is_reg16_att ($_), 0, "'$_' is not a valid AT&T-syntax 16-bit register" );
	is ( is_reg32_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-bit register" );
	is ( is_reg64_att ($_), 0, "'$_' is not a valid AT&T-syntax 64-bit register" );
	is ( is_reg_mm_att ($_), 0, "'$_' is not a valid AT&T-syntax multimedia register" );
	is ( is_segreg_att ($_), 0, "'$_' is not a valid AT&T-syntax segment register" );
	is ( is_reg_fpu_att ($_), 0, "'$_' is not a valid AT&T-syntax FPU register" );
	is ( is_reg_opmask_att ($_), 0, "'$_' is not a valid AT&T-syntax opmask register" );
	is ( is_addressable32_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-in-64-bit register" );

	is ( is_reg ($_), 1, "'$_' is a valid register" );
	is ( is_reg8 ($_), 0, "'$_' is a valid 8-bit register" );
	is ( is_reg16 ($_), 0, "'$_' is not a valid 16-bit register" );
	is ( is_reg32 ($_), 0, "'$_' is not a valid 32-bit register" );
	is ( is_reg64 ($_), 1, "'$_' is not a valid 64-bit register" );
	is ( is_reg_mm ($_), 0, "'$_' is not a valid multimedia register" );
	is ( is_segreg ($_), 0, "'$_' is not a valid segment register" );
	is ( is_reg_fpu ($_), 0, "'$_' is not a valid FPU register" );
	is ( is_reg_opmask ($_), 0, "'$_' is not a valid opmask register" );
	is ( is_addressable32 ($_), 0, "'$_' is not a valid 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64 ($_), 0, "'$_' is not a valid 32-in-64-bit register" );
}

foreach ((@regs_mm_intel, 'mm0', 'xmm0', 'ymm0', 'zmm0')) {

	is ( is_reg_intel ($_), 1, "'$_' is a valid Intel-syntax register" );
	is ( is_reg8_intel ($_), 0, "'$_' is not a valid Intel-syntax 8-bit register" );
	is ( is_reg16_intel ($_), 0, "'$_' is not a valid Intel-syntax 16-bit register" );
	is ( is_reg32_intel ($_), 0, "'$_' is not a valid Intel-syntax 32-bit register" );
	is ( is_reg64_intel ($_), 0, "'$_' is not a valid Intel-syntax 64-bit register" );
	is ( is_reg_mm_intel ($_), 1, "'$_' is a valid Intel-syntax multimedia register" );
	is ( is_segreg_intel ($_), 0, "'$_' is not a valid Intel-syntax segment register" );
	is ( is_reg_fpu_intel ($_), 0, "'$_' is not a valid Intel-syntax FPU register" );
	is ( is_reg_opmask_intel ($_), 0, "'$_' is not a valid Intel-syntax opmask register" );
	is ( is_addressable32_intel ($_), 0, "'$_' is not a valid Intel-syntax 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64_intel ($_), 0, "'$_' is not a valid Intel-syntax 32-in-64-bit register" );

	is ( is_reg_att ($_), 0, "'$_' is a valid AT&T-syntax register" );
	is ( is_reg8_att ($_), 0, "'$_' is a valid AT&T-syntax 8-bit register" );
	is ( is_reg16_att ($_), 0, "'$_' is not a valid AT&T-syntax 16-bit register" );
	is ( is_reg32_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-bit register" );
	is ( is_reg64_att ($_), 0, "'$_' is not a valid AT&T-syntax 64-bit register" );
	is ( is_reg_mm_att ($_), 0, "'$_' is not a valid AT&T-syntax multimedia register" );
	is ( is_segreg_att ($_), 0, "'$_' is not a valid AT&T-syntax segment register" );
	is ( is_reg_fpu_att ($_), 0, "'$_' is not a valid AT&T-syntax FPU register" );
	is ( is_reg_opmask_att ($_), 0, "'$_' is not a valid AT&T-syntax opmask register" );
	is ( is_addressable32_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-in-64-bit register" );

	is ( is_reg ($_), 1, "'$_' is a valid register" );
	is ( is_reg8 ($_), 0, "'$_' is a valid 8-bit register" );
	is ( is_reg16 ($_), 0, "'$_' is not a valid 16-bit register" );
	is ( is_reg32 ($_), 0, "'$_' is not a valid 32-bit register" );
	is ( is_reg64 ($_), 0, "'$_' is not a valid 64-bit register" );
	is ( is_reg_mm ($_), 1, "'$_' is not a valid multimedia register" );
	is ( is_segreg ($_), 0, "'$_' is not a valid segment register" );
	is ( is_reg_fpu ($_), 0, "'$_' is not a valid FPU register" );
	is ( is_reg_opmask ($_), 0, "'$_' is not a valid opmask register" );
	is ( is_addressable32 ($_), 0, "'$_' is not a valid 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64 ($_), 0, "'$_' is not a valid 32-in-64-bit register" );
}

foreach ((@regs_fpu_intel, 'st0')) {

	is ( is_reg_intel ($_), 1, "'$_' is a valid Intel-syntax register" );
	is ( is_reg8_intel ($_), 0, "'$_' is not a valid Intel-syntax 8-bit register" );
	is ( is_reg16_intel ($_), 0, "'$_' is not a valid Intel-syntax 16-bit register" );
	is ( is_reg32_intel ($_), 0, "'$_' is not a valid Intel-syntax 32-bit register" );
	is ( is_reg64_intel ($_), 0, "'$_' is not a valid Intel-syntax 64-bit register" );
	is ( is_reg_mm_intel ($_), 0, "'$_' is not a valid Intel-syntax multimedia register" );
	is ( is_segreg_intel ($_), 0, "'$_' is not a valid Intel-syntax segment register" );
	is ( is_reg_fpu_intel ($_), 1, "'$_' is a valid Intel-syntax FPU register" );
	is ( is_reg_opmask_intel ($_), 0, "'$_' is not a valid Intel-syntax opmask register" );
	is ( is_addressable32_intel ($_), 0, "'$_' is not a valid Intel-syntax 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64_intel ($_), 0, "'$_' is not a valid Intel-syntax 32-in-64-bit register" );

	is ( is_reg_att ($_), 0, "'$_' is a valid AT&T-syntax register" );
	is ( is_reg8_att ($_), 0, "'$_' is a valid AT&T-syntax 8-bit register" );
	is ( is_reg16_att ($_), 0, "'$_' is not a valid AT&T-syntax 16-bit register" );
	is ( is_reg32_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-bit register" );
	is ( is_reg64_att ($_), 0, "'$_' is not a valid AT&T-syntax 64-bit register" );
	is ( is_reg_mm_att ($_), 0, "'$_' is not a valid AT&T-syntax multimedia register" );
	is ( is_segreg_att ($_), 0, "'$_' is not a valid AT&T-syntax segment register" );
	is ( is_reg_fpu_att ($_), 0, "'$_' is not a valid AT&T-syntax FPU register" );
	is ( is_reg_opmask_att ($_), 0, "'$_' is not a valid AT&T-syntax opmask register" );
	is ( is_addressable32_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-in-64-bit register" );

	is ( is_reg ($_), 1, "'$_' is a valid register" );
	is ( is_reg8 ($_), 0, "'$_' is a valid 8-bit register" );
	is ( is_reg16 ($_), 0, "'$_' is not a valid 16-bit register" );
	is ( is_reg32 ($_), 0, "'$_' is not a valid 32-bit register" );
	is ( is_reg64 ($_), 0, "'$_' is not a valid 64-bit register" );
	is ( is_reg_mm ($_), 0, "'$_' is not a valid multimedia register" );
	is ( is_segreg ($_), 0, "'$_' is not a valid segment register" );
	is ( is_reg_fpu ($_), 1, "'$_' is not a valid FPU register" );
	is ( is_reg_opmask ($_), 0, "'$_' is not a valid opmask register" );
	is ( is_addressable32 ($_), 0, "'$_' is not a valid 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64 ($_), 0, "'$_' is not a valid 32-in-64-bit register" );
}

foreach ((@segregs_intel, 'cs')) {

	is ( is_reg_intel ($_), 1, "'$_' is a valid Intel-syntax register" );
	is ( is_reg8_intel ($_), 0, "'$_' is not a valid Intel-syntax 8-bit register" );
	is ( is_reg16_intel ($_), 1, "'$_' is not a valid Intel-syntax 16-bit register" );
	is ( is_reg32_intel ($_), 0, "'$_' is not a valid Intel-syntax 32-bit register" );
	is ( is_reg64_intel ($_), 0, "'$_' is not a valid Intel-syntax 64-bit register" );
	is ( is_reg_mm_intel ($_), 0, "'$_' is not a valid Intel-syntax multimedia register" );
	is ( is_segreg_intel ($_), 1, "'$_' is a valid Intel-syntax segment register" );
	is ( is_reg_fpu_intel ($_), 0, "'$_' is not a valid Intel-syntax FPU register" );
	is ( is_reg_opmask_intel ($_), 0, "'$_' is not a valid Intel-syntax opmask register" );
	is ( is_addressable32_intel ($_), 0, "'$_' is not a valid Intel-syntax 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64_intel ($_), 0, "'$_' is not a valid Intel-syntax 32-in-64-bit register" );

	is ( is_reg_att ($_), 0, "'$_' is a valid AT&T-syntax register" );
	is ( is_reg8_att ($_), 0, "'$_' is a valid AT&T-syntax 8-bit register" );
	is ( is_reg16_att ($_), 0, "'$_' is not a valid AT&T-syntax 16-bit register" );
	is ( is_reg32_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-bit register" );
	is ( is_reg64_att ($_), 0, "'$_' is not a valid AT&T-syntax 64-bit register" );
	is ( is_reg_mm_att ($_), 0, "'$_' is not a valid AT&T-syntax multimedia register" );
	is ( is_segreg_att ($_), 0, "'$_' is not a valid AT&T-syntax segment register" );
	is ( is_reg_fpu_att ($_), 0, "'$_' is not a valid AT&T-syntax FPU register" );
	is ( is_reg_opmask_att ($_), 0, "'$_' is not a valid AT&T-syntax opmask register" );
	is ( is_addressable32_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-in-64-bit register" );

	is ( is_reg ($_), 1, "'$_' is a valid register" );
	is ( is_reg8 ($_), 0, "'$_' is a valid 8-bit register" );
	is ( is_reg16 ($_), 1, "'$_' is not a valid 16-bit register" );
	is ( is_reg32 ($_), 0, "'$_' is not a valid 32-bit register" );
	is ( is_reg64 ($_), 0, "'$_' is not a valid 64-bit register" );
	is ( is_reg_mm ($_), 0, "'$_' is not a valid multimedia register" );
	is ( is_segreg ($_), 1, "'$_' is not a valid segment register" );
	is ( is_reg_fpu ($_), 0, "'$_' is not a valid FPU register" );
	is ( is_reg_opmask ($_), 0, "'$_' is not a valid opmask register" );
	is ( is_addressable32 ($_), 0, "'$_' is not a valid 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64 ($_), 0, "'$_' is not a valid 32-in-64-bit register" );
}

foreach ((@regs_opmask_intel, 'k0')) {

	is ( is_reg_intel ($_), 1, "'$_' is a valid Intel-syntax register" );
	is ( is_reg8_intel ($_), 0, "'$_' is not a valid Intel-syntax 8-bit register" );
	is ( is_reg16_intel ($_), 0, "'$_' is not a valid Intel-syntax 16-bit register" );
	is ( is_reg32_intel ($_), 0, "'$_' is not a valid Intel-syntax 32-bit register" );
	is ( is_reg64_intel ($_), 0, "'$_' is not a valid Intel-syntax 64-bit register" );
	is ( is_reg_mm_intel ($_), 0, "'$_' is not a valid Intel-syntax multimedia register" );
	is ( is_segreg_intel ($_), 0, "'$_' is not a valid Intel-syntax segment register" );
	is ( is_reg_fpu_intel ($_), 0, "'$_' is not a valid Intel-syntax FPU register" );
	is ( is_reg_opmask_intel ($_), 1, "'$_' is a valid Intel-syntax opmask register" );
	is ( is_addressable32_intel ($_), 0, "'$_' is not a valid Intel-syntax 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64_intel ($_), 0, "'$_' is not a valid Intel-syntax 32-in-64-bit register" );

	is ( is_reg_att ($_), 0, "'$_' is a valid AT&T-syntax register" );
	is ( is_reg8_att ($_), 0, "'$_' is a valid AT&T-syntax 8-bit register" );
	is ( is_reg16_att ($_), 0, "'$_' is not a valid AT&T-syntax 16-bit register" );
	is ( is_reg32_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-bit register" );
	is ( is_reg64_att ($_), 0, "'$_' is not a valid AT&T-syntax 64-bit register" );
	is ( is_reg_mm_att ($_), 0, "'$_' is not a valid AT&T-syntax multimedia register" );
	is ( is_segreg_att ($_), 0, "'$_' is not a valid AT&T-syntax segment register" );
	is ( is_reg_fpu_att ($_), 0, "'$_' is not a valid AT&T-syntax FPU register" );
	is ( is_reg_opmask_att ($_), 0, "'$_' is not a valid AT&T-syntax opmask register" );
	is ( is_addressable32_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-in-64-bit register" );

	is ( is_reg ($_), 1, "'$_' is a valid register" );
	is ( is_reg8 ($_), 0, "'$_' is a valid 8-bit register" );
	is ( is_reg16 ($_), 0, "'$_' is not a valid 16-bit register" );
	is ( is_reg32 ($_), 0, "'$_' is not a valid 32-bit register" );
	is ( is_reg64 ($_), 0, "'$_' is not a valid 64-bit register" );
	is ( is_reg_mm ($_), 0, "'$_' is not a valid multimedia register" );
	is ( is_segreg ($_), 0, "'$_' is not a valid segment register" );
	is ( is_reg_fpu ($_), 0, "'$_' is not a valid FPU register" );
	is ( is_reg_opmask ($_), 1, "'$_' is not a valid opmask register" );
	is ( is_addressable32 ($_), 0, "'$_' is not a valid 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64 ($_), 0, "'$_' is not a valid 32-in-64-bit register" );
}

foreach (@addressable32) {

	is ( is_reg_intel ($_), 1, "'$_' is a valid Intel-syntax register" );
	is ( is_reg8_intel ($_), 0, "'$_' is not a valid Intel-syntax 8-bit register" );
	is ( is_reg16_intel ($_), 0, "'$_' is not a valid Intel-syntax 16-bit register" );
	is ( is_reg32_intel ($_), 1, "'$_' is a valid Intel-syntax 32-bit register" );
	is ( is_reg64_intel ($_), 0, "'$_' is not a valid Intel-syntax 64-bit register" );
	is ( is_reg_mm_intel ($_), 0, "'$_' is not a valid Intel-syntax multimedia register" );
	is ( is_segreg_intel ($_), 0, "'$_' is not a valid Intel-syntax segment register" );
	is ( is_reg_fpu_intel ($_), 0, "'$_' is not a valid Intel-syntax FPU register" );
	is ( is_reg_opmask_intel ($_), 0, "'$_' is not a valid Intel-syntax opmask register" );
	is ( is_addressable32_intel ($_), 1, "'$_' is a valid Intel-syntax 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64_intel ($_), 0, "'$_' is not a valid Intel-syntax 32-in-64-bit register" );

	is ( is_reg_att ($_), 0, "'$_' is a valid AT&T-syntax register" );
	is ( is_reg8_att ($_), 0, "'$_' is a valid AT&T-syntax 8-bit register" );
	is ( is_reg16_att ($_), 0, "'$_' is not a valid AT&T-syntax 16-bit register" );
	is ( is_reg32_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-bit register" );
	is ( is_reg64_att ($_), 0, "'$_' is not a valid AT&T-syntax 64-bit register" );
	is ( is_reg_mm_att ($_), 0, "'$_' is not a valid AT&T-syntax multimedia register" );
	is ( is_segreg_att ($_), 0, "'$_' is not a valid AT&T-syntax segment register" );
	is ( is_reg_fpu_att ($_), 0, "'$_' is not a valid AT&T-syntax FPU register" );
	is ( is_reg_opmask_att ($_), 0, "'$_' is not a valid AT&T-syntax opmask register" );
	is ( is_addressable32_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-in-64-bit register" );

	is ( is_reg ($_), 1, "'$_' is a valid register" );
	is ( is_reg8 ($_), 0, "'$_' is a valid 8-bit register" );
	is ( is_reg16 ($_), 0, "'$_' is not a valid 16-bit register" );
	is ( is_reg32 ($_), 1, "'$_' is not a valid 32-bit register" );
	is ( is_reg64 ($_), 0, "'$_' is not a valid 64-bit register" );
	is ( is_reg_mm ($_), 0, "'$_' is not a valid multimedia register" );
	is ( is_segreg ($_), 0, "'$_' is not a valid segment register" );
	is ( is_reg_fpu ($_), 0, "'$_' is not a valid FPU register" );
	is ( is_reg_opmask ($_), 0, "'$_' is not a valid opmask register" );
	is ( is_addressable32 ($_), 1, "'$_' is not a valid 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64 ($_), 0, "'$_' is not a valid 32-in-64-bit register" );
}

foreach (@r32_in64) {

	is ( is_reg_intel ($_), 1, "'$_' is a valid Intel-syntax register" );
	is ( is_reg8_intel ($_), 0, "'$_' is not a valid Intel-syntax 8-bit register" );
	is ( is_reg16_intel ($_), 0, "'$_' is not a valid Intel-syntax 16-bit register" );
	is ( is_reg32_intel ($_), 1, "'$_' is a valid Intel-syntax 32-bit register" );
	is ( is_reg64_intel ($_), 0, "'$_' is not a valid Intel-syntax 64-bit register" );
	is ( is_reg_mm_intel ($_), 0, "'$_' is not a valid Intel-syntax multimedia register" );
	is ( is_segreg_intel ($_), 0, "'$_' is not a valid Intel-syntax segment register" );
	is ( is_reg_fpu_intel ($_), 0, "'$_' is not a valid Intel-syntax FPU register" );
	is ( is_reg_opmask_intel ($_), 0, "'$_' is not a valid Intel-syntax opmask register" );
	is ( is_addressable32_intel ($_), 0, "'$_' is a valid Intel-syntax 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64_intel ($_), 1, "'$_' is a valid Intel-syntax 32-in-64-bit register" );

	is ( is_reg_att ($_), 0, "'$_' is a valid AT&T-syntax register" );
	is ( is_reg8_att ($_), 0, "'$_' is a valid AT&T-syntax 8-bit register" );
	is ( is_reg16_att ($_), 0, "'$_' is not a valid AT&T-syntax 16-bit register" );
	is ( is_reg32_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-bit register" );
	is ( is_reg64_att ($_), 0, "'$_' is not a valid AT&T-syntax 64-bit register" );
	is ( is_reg_mm_att ($_), 0, "'$_' is not a valid AT&T-syntax multimedia register" );
	is ( is_segreg_att ($_), 0, "'$_' is not a valid AT&T-syntax segment register" );
	is ( is_reg_fpu_att ($_), 0, "'$_' is not a valid AT&T-syntax FPU register" );
	is ( is_reg_opmask_att ($_), 0, "'$_' is not a valid AT&T-syntax opmask register" );
	is ( is_addressable32_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-in-64-bit register" );

	is ( is_reg ($_), 1, "'$_' is a valid register" );
	is ( is_reg8 ($_), 0, "'$_' is a valid 8-bit register" );
	is ( is_reg16 ($_), 0, "'$_' is not a valid 16-bit register" );
	is ( is_reg32 ($_), 1, "'$_' is not a valid 32-bit register" );
	is ( is_reg64 ($_), 0, "'$_' is not a valid 64-bit register" );
	is ( is_reg_mm ($_), 0, "'$_' is not a valid multimedia register" );
	is ( is_segreg ($_), 0, "'$_' is not a valid segment register" );
	is ( is_reg_fpu ($_), 0, "'$_' is not a valid FPU register" );
	is ( is_reg_opmask ($_), 0, "'$_' is not a valid opmask register" );
	is ( is_addressable32 ($_), 0, "'$_' is not a valid 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64 ($_), 1, "'$_' is not a valid 32-in-64-bit register" );
}

foreach (@invalid_regs) {

	is ( is_reg_intel ($_), 0, "'$_' is not a valid Intel-syntax register" );
	is ( is_reg8_intel ($_), 0, "'$_' is not a valid Intel-syntax 8-bit register" );
	is ( is_reg16_intel ($_), 0, "'$_' is not a valid Intel-syntax 16-bit register" );
	is ( is_reg32_intel ($_), 0, "'$_' is not a valid Intel-syntax 32-bit register" );
	is ( is_reg64_intel ($_), 0, "'$_' is not a valid Intel-syntax 64-bit register" );
	is ( is_reg_mm_intel ($_), 0, "'$_' is not a valid Intel-syntax multimedia register" );
	is ( is_segreg_intel ($_), 0, "'$_' is not a valid Intel-syntax segment register" );
	is ( is_reg_fpu_intel ($_), 0, "'$_' is not a valid Intel-syntax FPU register" );
	is ( is_reg_opmask_intel ($_), 0, "'$_' is not a valid Intel-syntax opmask register" );
	is ( is_addressable32_intel ($_), 0, "'$_' is not a valid Intel-syntax 32-bit register which can be used for addressing" );
	is ( is_r32_in64_intel ($_), 0, "'$_' is not a valid Intel-syntax 32-in-64-bit register" );

	is ( is_reg_att ($_), 0, "'$_' is a valid AT&T-syntax register" );
	is ( is_reg8_att ($_), 0, "'$_' is a valid AT&T-syntax 8-bit register" );
	is ( is_reg16_att ($_), 0, "'$_' is not a valid AT&T-syntax 16-bit register" );
	is ( is_reg32_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-bit register" );
	is ( is_reg64_att ($_), 0, "'$_' is not a valid AT&T-syntax 64-bit register" );
	is ( is_reg_mm_att ($_), 0, "'$_' is not a valid AT&T-syntax multimedia register" );
	is ( is_segreg_att ($_), 0, "'$_' is not a valid AT&T-syntax segment register" );
	is ( is_reg_fpu_att ($_), 0, "'$_' is not a valid AT&T-syntax FPU register" );
	is ( is_reg_opmask_att ($_), 0, "'$_' is not a valid AT&T-syntax opmask register" );
	is ( is_addressable32_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64_att ($_), 0, "'$_' is not a valid AT&T-syntax 32-in-64-bit register" );

	is ( is_reg ($_), 0, "'$_' is not a valid register" );
	is ( is_reg8 ($_), 0, "'$_' is not a valid 8-bit register" );
	is ( is_reg16 ($_), 0, "'$_' is not a valid 16-bit register" );
	is ( is_reg32 ($_), 0, "'$_' is not a valid 32-bit register" );
	is ( is_reg64 ($_), 0, "'$_' is not a valid 64-bit register" );
	is ( is_reg_mm ($_), 0, "'$_' is not a valid multimedia register" );
	is ( is_segreg ($_), 0, "'$_' is not a valid segment register" );
	is ( is_reg_fpu ($_), 0, "'$_' is not a valid FPU register" );
	is ( is_reg_opmask ($_), 0, "'$_' is not a valid opmask register" );
	is ( is_addressable32 ($_), 0, "'$_' is not a valid 32-bit register which can be used for 32-bit addressing" );
	is ( is_r32_in64 ($_), 0, "'$_' is not a valid 32-in-64-bit register" );
}
