package CPANPLUS::Module;

use strict;
use vars qw[@ISA];


use CPANPLUS::Dist;
use CPANPLUS::Error;
use CPANPLUS::Module::Signature;
use CPANPLUS::Module::Checksums;
use CPANPLUS::Internals::Constants;

use FileHandle;

use Locale::Maketext::Simple    Class => 'CPANPLUS', Style => 'gettext';
use IPC::Cmd                    qw[can_run run];
use File::Find                  qw[find];
use Params::Check               qw[check];
use File::Basename              qw[dirname];
use Module::Load::Conditional   qw[can_load check_install];

$Params::Check::VERBOSE = 1;

@ISA = qw[ CPANPLUS::Module::Signature CPANPLUS::Module::Checksums];

my $tmpl = {
    module      => { default => '', required => 1 },    # full module name
    version     => { default => '0.0' },                # version number
    path        => { default => '', required => 1 },    # extended path on the
                                                        # cpan mirror, like
                                                        # /author/id/K/KA/KANE
    comment     => { default => ''},                    # comment on module
    package     => { default => '', required => 1 },    # package name, like
                                                        # 'bar-baz-1.03.tgz'
    description => { default => '' },                   # description of the
                                                        # module
    dslip       => { default => EMPTY_DSLIP },          # dslip information
    _id         => { required => 1 },                   # id of the Internals
                                                        # parent object
    _status     => { no_override => 1 },                # stores status object
    author      => { default => '', required => 1,
                     allow => IS_AUTHOBJ },             # module author
    mtime       => { default => '' },
};

### some of these will be resolved by wrapper functions that
### do Clever Things to find the actual value, so don't create
### an autogenerated sub for that just here, take an alternate
### name to allow for a wrapper
{   my %rename = (
        dslip   => '_dslip'
    );

    ### autogenerate accessors ###
    for my $key ( keys %$tmpl ) {
        no strict 'refs';
      
        my $sub = $rename{$key} || $key;
      
        *{__PACKAGE__."::$sub"} = sub {
            $_[0]->{$key} = $_[1] if @_ > 1;
            return $_[0]->{$key};
        }
    }
}


### *name is an alias, include it explicitly
sub accessors { return ('name', keys %$tmpl) };

sub dslip {
    my $self    = shift;   

    ### if this module has relevant dslip info, return it
    return $self->_dslip if $self->_dslip ne EMPTY_DSLIP;

    ### if not, look at other modules in the same package,
    ### see if *they* have any dslip info
    for my $mod ( $self->contains ) {
        return $mod->_dslip if $mod->_dslip ne EMPTY_DSLIP;
    }
    
    ### ok, really no dslip info found, return the default
    return EMPTY_DSLIP;
}


### Alias ->name to ->module, for human beings.
*name = *module;

sub parent {
    my $self = shift;
    my $obj  = CPANPLUS::Internals->_retrieve_id( $self->_id );

    return $obj;
}


sub new {
    my($class, %hash) = @_;

    ### don't check the template for sanity
    ### -- we know it's good and saves a lot of performance
    local $Params::Check::SANITY_CHECK_TEMPLATE = 0;

    my $object  = check( $tmpl, \%hash ) or return;

    bless $object, $class;

    return $object;
}

### only create status objects when they're actually asked for
sub status {
    my $self = shift;
    return $self->_status if $self->_status;
    
    my $acc = Object::Accessor->new;
    $acc->mk_accessors( qw[ installer_type dist_cpan dist prereqs
                            signature extract fetch readme uninstall
                            created installed prepared checksums files
                            checksum_ok checksum_value _fetch_from
                            configure_requires
                        ] );

    ### create an alias from 'requires' to 'prereqs', so it's more in
    ### line with 'configure_requires';
    $acc->mk_aliases( requires => 'prereqs' );

    $self->_status( $acc );

    return $self->_status;
}


### flush the cache of this object ###
sub _flush {
    my $self = shift;
    $self->status->mk_flush;
    return 1;
}

{   ### fetches the test reports for a certain module ###
    my %map = (
        name        => 0,
        version     => 1,
        extension   => 2,
    );        
    
    while ( my($type, $index) = each %map ) {
        my $name    = 'package_' . $type;
        
        no strict 'refs';
        *$name = sub {
            my $self = shift;
            my $val  = shift || $self->package;
            my @res  = $self->parent->_split_package_string( package => $val );
     
            ### return the corresponding index from the result
            return $res[$index] if @res;
            return;
        };
    }        

    sub package_is_perl_core {
        my $self = shift;
        my $cb   = $self->parent;

        ### check if the package looks like a perl core package
        return 1 if $self->package_name eq PERL_CORE;

        ### address #44562: ::Module->package_is_perl_code : problem comparing 
        ### version strings -- use $cb->_vcmp to avoid warnings when version 
        ### have _ in them

        my $core = $self->module_is_supplied_with_perl_core;
        ### ok, so it's found in the core, BUT it could be dual-lifed
        if ($core) {
            ### if the package is newer than installed, then it's dual-lifed
            return if $cb->_vcmp($self->version, $self->installed_version) > 0;
            
            ### if the package is newer or equal to the corelist, 
            ### then it's dual-lifed
            return if $cb->_vcmp( $self->version, $core ) >= 0; 

            ### otherwise, it's older than corelist, thus unsuitable.
            return 1;
        }

        ### not in corelist, not a perl core package.
        return;
    }

    sub module_is_supplied_with_perl_core {
        my $self = shift;
        my $ver  = shift || $];

        ### allow it to be called as a package function as well like:
        ###   CPANPLUS::Module::module_is_supplied_with_perl_core('Config')
        ### so that we can check the status of modules that aren't released
        ### to CPAN, but are part of the core.
        my $name = ref $self ? $self->module : $self;

        ### check Module::CoreList to see if it's a core package
        require Module::CoreList;
        
        ### Address #41157: Module::module_is_supplied_with_perl_core() 
        ### broken for perl 5.10: Module::CoreList's version key for the 
        ### hash has a different number of trailing zero than $] aka
        ### $PERL_VERSION.
        my $core = $Module::CoreList::version{ 0+$ver }->{ $name };

        return $core;
    }

    ### make sure Bundle-Foo also gets flagged as bundle
    sub is_bundle {
        my $self = shift;
        
        ### cpan'd bundle
        return 1 if $self->module =~ /^bundle(?:-|::)/i;
    
        ### autobundle
        return 1 if $self->is_autobundle;
    
        ### neither
        return;
    }

    ### full path to a generated autobundle
    sub is_autobundle {
        my $self    = shift;
        my $conf    = $self->parent->configure_object;
        my $prefix  = $conf->_get_build('autobundle_prefix');

        return 1 if $self->module eq $prefix;
        return;
    }

    sub is_third_party {
        my $self = shift;
        
        return unless can_load( modules => { 'Module::ThirdParty' => 0 } );
        
        return Module::ThirdParty::is_3rd_party( $self->name );
    }

    sub third_party_information {
        my $self = shift;

        return unless $self->is_third_party; 

        return Module::ThirdParty::module_information( $self->name );
    }
}

{   ### accessors dont change during run time, so only compute once
    my @acc = grep !/status/, __PACKAGE__->accessors();
    
    sub clone {
        my $self = shift;
    
        ### clone the object ###
        my %data = map { $_ => $self->$_ } @acc;
    
        my $obj = CPANPLUS::Module::Fake->new( %data );
    
        return $obj;
    }
}

sub fetch {
    my $self = shift;
    my $cb   = $self->parent;

    ### custom args
    my %args            = ( module => $self );

    ### if a custom fetch location got specified before, add that here
    $args{fetch_from}   = $self->status->_fetch_from 
                            if $self->status->_fetch_from;

    my $where = $cb->_fetch( @_, %args ) or return;

    ### do an md5 check ###
    if( !$self->status->_fetch_from and 
        $cb->configure_object->get_conf('md5') and
        $self->package ne CHECKSUMS
    ) {
        unless( $self->_validate_checksum ) {
            error( loc( "Checksum error for '%1' -- will not trust package",
                        $self->package) );
            return;
        }
    }

    return $where;
}

sub extract {
    my $self = shift;
    my $cb   = $self->parent;

    unless( $self->status->fetch ) {
        error( loc( "You have not fetched '%1' yet -- cannot extract",
                    $self->module) );
        return;
    }
    
    ### can't extract these, so just use the basedir for the file
    if( $self->is_autobundle ) {
    
        ### this is expected to be set after an extract call
        $self->get_installer_type;
    
        return $self->status->extract( dirname( $self->status->fetch ) );
    }
    
    return $cb->_extract( @_, module => $self );
}

sub get_installer_type {
    my $self = shift;
    my $cb   = $self->parent;
    my $conf = $cb->configure_object;
    my %hash = @_;

    my ($prefer_makefile,$verbose);
    my $tmpl = {
        prefer_makefile => { default => $conf->get_conf('prefer_makefile'),
                             store   => \$prefer_makefile, allow => BOOLEANS },
        verbose         => { default => $conf->get_conf('verbose'),
                             store   => \$verbose },                             
    };

    check( $tmpl, \%hash ) or return;

    my $type;
    
    ### autobundles use their own installer, so return that
    if( $self->is_autobundle ) {
        $type = INSTALLER_AUTOBUNDLE;        

    } else {
        my $extract = $self->status->extract();
        unless( $extract ) {
            error(loc(
                "Cannot determine installer type of unextracted module '%1'",
                $self->module
            ));
            return;
        }
    
        ### check if it's a makemaker or a module::build type dist ###
        my $found_build     = -e BUILD_PL->( $extract );
        my $found_makefile  = -e MAKEFILE_PL->( $extract );
    
        $type = INSTALLER_BUILD if !$prefer_makefile &&  $found_build;
        $type = INSTALLER_BUILD if  $found_build     && !$found_makefile;
        $type = INSTALLER_MM    if  $prefer_makefile &&  $found_makefile;
        $type = INSTALLER_MM    if  $found_makefile  && !$found_build;
    }

    ### ok, so it's a 'build' installer, but you don't /have/ module build
    ### XXX duplicated from CPANPLUS::Selfupdate. fix somehow?
    if( $type and $type eq INSTALLER_BUILD and (
        not CPANPLUS::Dist->has_dist_type( INSTALLER_BUILD )
        or not $cb->module_tree( INSTALLER_BUILD )
                    ->is_uptodate( version => '0.24' )
    ) ) {
    
        ### XXX this is for recording purposes only. We *have* to install
        ### these before even creating a dist object, or we'll get an error
        ### saying 'no such dist type';
        ### XXX duplicated from CPANPLUS::Selfupdate. fix somehow?
        my $href = $self->status->configure_requires || {};
        my $deps = { INSTALLER_BUILD, '0.24', %$href };
        
        $self->status->configure_requires( $deps );
        
        msg(loc("This module requires '%1' and '%2' to be installed first. ".
                "Adding these modules to your prerequisites list",
                 'Module::Build', INSTALLER_BUILD
        ), $verbose );                 


    ### ok, actually we found neither ###
    } elsif ( !$type ) {
        error( loc( "Unable to find '%1' or '%2' for '%3'; ".
                    "Will default to '%4' but might be unable ".
                    "to install!", BUILD_PL->(), MAKEFILE_PL->(),
                    $self->module, INSTALLER_MM ) );
        $type = INSTALLER_MM;
    }

    return $self->status->installer_type( $type ) if $type;
    return;
}

sub dist {
    my $self = shift;
    my $cb   = $self->parent;
    my $conf = $cb->configure_object;
    my %hash = @_;

    ### have you determined your installer type yet? if not, do it here,
    ### we need the info
    $self->get_installer_type unless $self->status->installer_type;

    my($type,$args,$target);
    my $tmpl = {
        format  => { default => $conf->get_conf('dist_type') ||
                                $self->status->installer_type,
                     store   => \$type },
        target  => { default => TARGET_CREATE, store => \$target },                     
        args    => { default => {}, store => \$args },
    };

    check( $tmpl, \%hash ) or return;

    ### ok, check for $type. Do we have it?
    unless( CPANPLUS::Dist->has_dist_type( $type ) ) {

        ### ok, we don't have it. Is it C::D::Build? if so we can install the
        ### whole thing now
        ### XXX we _could_ do this for any type we dont have actually...
        if( $type eq INSTALLER_BUILD ) {
            msg(loc("Bootstrapping installer '%1'", $type));
        
            ### don't propagate the format, it's the one we're trying to
            ### bootstrap, so it'll be an infinite loop if we do
        
            $cb->module_tree( $type )->install( target => $target, %$args ) or
                do {
                    error(loc("Could not bootstrap installer '%1' -- ".
                              "can not continue", $type));
                    return;                          
                };
        
            ### re-scan for available modules now
            CPANPLUS::Dist->rescan_dist_types;
            
            unless( CPANPLUS::Dist->has_dist_type( $type ) ) {
                error(loc("Newly installed installer type '%1' should be ".
                          "available, but is not! -- aborting", $type));
                return;
            } else {
                msg(loc("Installer '%1' succesfully bootstrapped", $type));
            }
            
        ### some other plugin you dont have. Abort
        } else {
            error(loc("Installer type '%1' not found. Please verify your ".
                      "installation -- aborting", $type ));
            return;
        }            
    }

    ### make sure we don't overwrite it, just in case we came 
    ### back from a ->save_state. This allows restoration to
    ### work correctly    
    my( $dist, $dist_cpan );
    
    unless( $dist = $self->status->dist ) {
        $dist = $type->new( module => $self ) or return;
        $self->status->dist( $dist );
    }
    
    unless( $dist_cpan = $self->status->dist_cpan ) {
        
        $dist_cpan = $type eq $self->status->installer_type
                        ? $self->status->dist
                        : $self->status->installer_type->new( module => $self );           


        $self->status->dist_cpan(   $dist_cpan );
    }
    
    
    DIST: {
        ### first prepare the dist
        $dist->prepare( %$args ) or return;
        $self->status->prepared(1);

        ### you just wanted us to prepare?
        last DIST if $target eq TARGET_PREPARE;

        $dist->create( %$args ) or return;
        $self->status->created(1);
    }

    return $dist;
}

sub prepare { 
    my $self = shift;
    return $self->install( @_, target => TARGET_PREPARE );
}

sub create { 
    my $self = shift;
    return $self->install( @_, target => TARGET_CREATE );
}

sub test {
    my $self = shift;
    return $self->install( @_, target => TARGET_CREATE, skiptest => 0 );
}

sub install {
    my $self = shift;
    my $cb   = $self->parent;
    my $conf = $cb->configure_object;
    my %hash = @_;

    my $args; my $target; my $format;
    {   ### so we can use the rest of the args to the create calls etc ###
        local $Params::Check::NO_DUPLICATES = 1;
        local $Params::Check::ALLOW_UNKNOWN = 1;

        ### targets 'dist' and 'test' are now completely ignored ###
        my $tmpl = {
                        ### match this allow list with Dist->_resolve_prereqs
            target     => { default => TARGET_INSTALL, store => \$target,
                            allow   => [TARGET_PREPARE, TARGET_CREATE,
                                        TARGET_INSTALL] },
            force      => { default => $conf->get_conf('force'), },
            verbose    => { default => $conf->get_conf('verbose'), },
            format     => { default => $conf->get_conf('dist_type'),
                                store => \$format },
        };

        $args = check( $tmpl, \%hash ) or return;
    }


    ### if this target isn't 'install', we will need to at least 'create' 
    ### every prereq, so it can build
    ### XXX prereq_target of 'prepare' will do weird things here, and is
    ### not supported.
    $args->{'prereq_target'} ||= TARGET_CREATE if $target ne TARGET_INSTALL;

    ### check if it's already upto date ###
    if( $target eq TARGET_INSTALL and !$args->{'force'} and
        !$self->package_is_perl_core() and         # separate rules apply
        ( $self->status->installed() or $self->is_uptodate ) and
        !INSTALL_VIA_PACKAGE_MANAGER->($format)
    ) {
        msg(loc("Module '%1' already up to date, won't install without force",
                $self->module), $args->{'verbose'} );
        return $self->status->installed(1);
    }

    # if it's a non-installable core package, abort the install.
    if( $self->package_is_perl_core() ) {
        # if the installed is newer, say so.
        if( $self->installed_version > $self->version ) {
            error(loc("The core Perl %1 module '%2' (%3) is more ".
                      "recent than the latest release on CPAN (%4). ".
                      "Aborting install.",
                      $], $self->module, $self->installed_version,
                      $self->version ) );
        # if the installed matches, say so.
        } elsif( $self->installed_version == $self->version ) {
            error(loc("The core Perl %1 module '%2' (%3) can only ".
                      "be installed by Perl itself. ".
                      "Aborting install.",
                      $], $self->module, $self->installed_version ) );
        # otherwise, the installed is older; say so.
        } else {
            error(loc("The core Perl %1 module '%2' can only be ".
                      "upgraded from %3 to %4 by Perl itself (%5). ".
                      "Aborting install.",
                      $], $self->module, $self->installed_version,
                      $self->version, $self->package ) );
        }
        return;
    
    ### it might be a known 3rd party module
    } elsif ( $self->is_third_party ) {
        my $info = $self->third_party_information;
        error(loc(
            "%1 is a known third-party module.\n\n".
            "As it isn't available on the CPAN, CPANPLUS can't install " .
            "it automatically. Therefore you need to install it manually " .
            "before proceeding.\n\n".
            "%2 is part of %3, published by %4, and should be available ".
            "for download at the following address:\n\t%5",
            $self->name, $self->name, $info->{name}, $info->{author},
            $info->{url}
        ));
        
        return;
    }

    ### fetch it if need be ###
    unless( $self->status->fetch ) {
        my $params;
        for (qw[prefer_bin fetchdir]) {
            $params->{$_} = $args->{$_} if exists $args->{$_};
        }
        for (qw[force verbose]) {
            $params->{$_} = $args->{$_} if defined $args->{$_};
        }
        $self->fetch( %$params ) or return;
    }

    ### extract it if need be ###
    unless( $self->status->extract ) {
        my $params;
        for (qw[prefer_bin extractdir]) {
            $params->{$_} = $args->{$_} if exists $args->{$_};
        }
        for (qw[force verbose]) {
            $params->{$_} = $args->{$_} if defined $args->{$_};
        }
        $self->extract( %$params ) or return;
    }

    $format ||= $self->status->installer_type;

    unless( $format ) {
        error( loc( "Don't know what installer to use; " .
                    "Couldn't find either '%1' or '%2' in the extraction " .
                    "directory '%3' -- will be unable to install",
                    BUILD_PL->(), MAKEFILE_PL->(), $self->status->extract ) );

        $self->status->installed(0);
        return;
    }


    ### do SIGNATURE checks? ###
    ### XXX check status and not recheck EVERY time?
    if( $conf->get_conf('signature') ) {
        unless( $self->check_signature( verbose => $args->{verbose} ) ) {
            error( loc( "Signature check failed for module '%1' ".
                        "-- Not trusting this module, aborting install",
                        $self->module ) );
            $self->status->signature(0);
            
            ### send out test report on broken sig
            if( $conf->get_conf('cpantest') ) {
                $cb->_send_report( 
                    module  => $self,
                    failed  => 1,
                    buffer  => CPANPLUS::Error->stack_as_string,
                    verbose => $args->{verbose},
                    force   => $args->{force},
                ) or error(loc("Failed to send test report for '%1'",
                     $self->module ) );
            }  
            
            return;

        } else {
            ### signature OK ###
            $self->status->signature(1);
        }
    }

    ### a target of 'create' basically means not to run make test ###
    ### eh, no it /doesn't/.. skiptest => 1 means skiptest => 1.
    #$args->{'skiptest'} = 1 if $target eq 'create';

    ### bundle rules apply ###
    if( $self->is_bundle ) {
        ### check what we need to install ###
        my @prereqs = $self->bundle_modules();
        unless( @prereqs ) {
            error( loc( "Bundle '%1' does not specify any modules to install",
                        $self->module ) );

            ### XXX mark an error here? ###
        }
    }

    my $dist = $self->dist( format  => $format, 
                            target  => $target, 
                            args    => $args );
    unless( $dist ) {
        error( loc( "Unable to create a new distribution object for '%1' " .
                    "-- cannot continue", $self->module ) );
        return;
    }

    return 1 if $target ne TARGET_INSTALL;

    my $ok = $dist->install( %$args ) ? 1 : 0;

    $self->status->installed($ok);

    return 1 if $ok;
    return;
}

sub bundle_modules {
    my $self = shift;
    my $cb   = $self->parent;

    unless( $self->is_bundle ) {
        error( loc("'%1' is not a bundle", $self->module ) );
        return;
    }

    my @files;
    
    ### autobundles are special files generated by CPANPLUS. If we can
    ### read the file, we can determine the prereqs
    if( $self->is_autobundle ) {
        my $where;
        unless( $where = $self->status->fetch ) {
            error(loc("Don't know where '%1' was fetched to", $self->package));
            return;
        }
        
        push @files, $where
    
    ### regular bundle::* upload
    } else {    
        my $dir;
        unless( $dir = $self->status->extract ) {
            error(loc("Don't know where '%1' was extracted to", $self->module));
            return;
        }

        find( {
            wanted   => sub { push @files, File::Spec->rel2abs($_) if /\.pm/i },
            no_chdir => 1,
        }, $dir );
    }

    my $prereqs = {}; my @list; my $seen = {};
    for my $file ( @files ) {
        my $fh = FileHandle->new($file)
                    or( error(loc("Could not open '%1' for reading: %2",
                        $file,$!)), next );

        my $flag;
        while( local $_ = <$fh> ) {
            ### quick hack to read past the header of the file ###
            last if $flag && m|^=head|i;

            ### from perldoc cpan:
            ### =head1 CONTENTS
            ### In this pod section each line obeys the format
            ### Module_Name [Version_String] [- optional text]
            $flag = 1 if m|^=head1 CONTENTS|i;

            if ($flag && /^(?!=)(\S+)\s*(\S+)?/) {
                my $module  = $1;
                my $version = $cb->_version_to_number( version => $2 );

                my $obj = $cb->module_tree($module);

                unless( $obj ) {
                    error(loc("Cannot find bundled module '%1'", $module),
                          loc("-- it does not seem to exist") );
                    next;
                }

                ### make sure we list no duplicates ###
                unless( $seen->{ $obj->module }++ ) {
                    push @list, $obj;
                    $prereqs->{ $module } =
                        $cb->_version_to_number( version => $version );
                }
            }
        }
    }

    ### store the prereqs we just found ###
    $self->status->prereqs( $prereqs );

    return @list;
}

sub readme {
    my $self = shift;
    my $conf = $self->parent->configure_object;    

    ### did we already dl the readme once? ###
    return $self->status->readme() if $self->status->readme();

    ### this should be core ###
    return unless can_load( modules     => { FileHandle => '0.0' },
                            verbose     => 1,
                        );

    ### get a clone of the current object, with a fresh status ###
    my $obj  = $self->clone or return;

    ### munge the package name
    my $pkg = README->( $obj );
    $obj->package($pkg);

    my $file;
    {   ### disable checksum fetches on readme downloads
        
        my $tmp = $conf->get_conf( 'md5' );
        $conf->set_conf( md5 => 0 );
        
        $file = $obj->fetch;

        $conf->set_conf( md5 => $tmp );

        return unless $file;
    }

    ### read the file into a scalar, to store in the original object ###
    my $fh = new FileHandle;
    unless( $fh->open($file) ) {
        error( loc( "Could not open file '%1': %2", $file, $! ) );
        return;
    }

    my $in = do{ local $/; <$fh> };
    $fh->close;

    return $self->status->readme( $in );
}

### uptodate/installed functions
{   my $map = {             # hashkey,      alternate rv
        installed_version   => ['version',  0 ],
        installed_file      => ['file',     ''],
        installed_dir       => ['dir',      ''],
        is_uptodate         => ['uptodate', 0 ],
    };

    while( my($method, $aref) = each %$map ) {
        my($key,$alt_rv) = @$aref;

        no strict 'refs';
        *$method = sub {
            ### never use the @INC hooks to find installed versions of
            ### modules -- they're just there in case they're not on the
            ### perl install, but the user shouldn't trust them for *other*
            ### modules!
            ### XXX CPANPLUS::inc is now obsolete, so this should not
            ### be needed anymore
            #local @INC = CPANPLUS::inc->original_inc;

            my $self = shift;
            
            ### make sure check_install is not looking in %INC, as
            ### that may contain some of our sneakily loaded modules
            ### that aren't installed as such. -- kane
            local $Module::Load::Conditional::CHECK_INC_HASH = 0;
            my $href = check_install(
                            module  => $self->module,
                            version => $self->version,
                            @_,
                        );

            return $href->{$key} || $alt_rv;
        }
    }
}



sub details {
    my $self = shift;
    my $conf = $self->parent->configure_object();
    my $cb   = $self->parent;
    my %hash = @_;

    my $res = {
        Author              => loc("%1 (%2)",   $self->author->author(),
                                                $self->author->email() ),
        Package             => $self->package,
        Description         => $self->description     || loc('None given'),
        'Version on CPAN'   => $self->version,
    };

    ### check if we have the module installed
    ### if so, add version have and version on cpan
    $res->{'Version Installed'} = $self->installed_version
                                    if $self->installed_version;
    $res->{'Installed File'} = $self->installed_file if $self->installed_file;

    my $i = 0;
    for my $item( split '', $self->dslip ) {
        $res->{ $cb->_dslip_defs->[$i]->[0] } =
                $cb->_dslip_defs->[$i]->[1]->{$item} || loc('Unknown');
        $i++;
    }

    return $res;
}

sub contains {
    my $self = shift;
    my $cb   = $self->parent;
    my $pkg  = $self->package;

    my @mods = $cb->search( type => 'package', allow => [qr/^$pkg$/] );
    
    return @mods;
}

sub fetch_report {
    my $self    = shift;
    my $cb      = $self->parent;

    return $cb->_query_report( @_, module => $self );
}

sub uninstall {
    my $self = shift;
    my $conf = $self->parent->configure_object();
    my %hash = @_;

    my ($type,$verbose);
    my $tmpl = {
        type    => { default => 'all', allow => [qw|man prog all|],
                        store => \$type },
        verbose => { default => $conf->get_conf('verbose'),
                        store => \$verbose },
        force   => { default => $conf->get_conf('force') },
    };

    ### XXX add a warning here if your default install dist isn't
    ### makefile or build -- that means you are using a package manager
    ### and this will not do what you think!

    my $args = check( $tmpl, \%hash ) or return;

    if( $conf->get_conf('dist_type') and (
        ($conf->get_conf('dist_type') ne INSTALLER_BUILD) or
        ($conf->get_conf('dist_type') ne INSTALLER_MM))
    ) {
        msg(loc("You have a default installer type set (%1) ".
                "-- you should probably use that package manager to " .
                "uninstall modules", $conf->get_conf('dist_type')), $verbose);
    }

    ### check if we even have the module installed -- no point in continuing
    ### otherwise
    unless( $self->installed_version ) {
        error( loc( "Module '%1' is not installed, so cannot uninstall",
                    $self->module ) );
        return;
    }

                                                ### nothing to uninstall ###
    my $files   = $self->files( type => $type )             or return;
    my $dirs    = $self->directory_tree( type => $type )    or return;
    my $sudo    = $conf->get_program('sudo');

    ### just in case there's no file; M::B doensn't provide .packlists yet ###
    my $pack    = $self->packlist;
    $pack       = $pack->[0]->packlist_file() if $pack;

    ### first remove the files, then the dirs if they are empty ###
    my $flag = 0;
    for my $file( @$files, $pack ) {
        next unless defined $file && -f $file;

        msg(loc("Unlinking '%1'", $file), $verbose);

        my @cmd = ($^X, "-eunlink+q[$file]");
        unshift @cmd, $sudo if $sudo;

        my $buffer;
        unless ( run(   command => \@cmd,
                        verbose => $verbose,
                        buffer  => \$buffer )
        ) {
            error(loc("Failed to unlink '%1': '%2'",$file, $buffer));
            $flag++;
        }
    }

    for my $dir ( sort @$dirs ) {
        local *DIR;
        opendir DIR, $dir or next;
        my @count = readdir(DIR);
        close DIR;

        next unless @count == 2;    # . and ..

        msg(loc("Removing '%1'", $dir), $verbose);

        ### this fails on my win2k machines.. it indeed leaves the
        ### dir, but it's not a critical error, since the files have
        ### been removed. --kane
        #unless( rmdir $dir ) {
        #    error( loc( "Could not remove '%1': %2", $dir, $! ) )
        #        unless $^O eq 'MSWin32';
        #}
        
        my @cmd = ($^X, "-e", "rmdir q[$dir]");
        unshift @cmd, $sudo if $sudo;
        
        my $buffer;
        unless ( run(   command => \@cmd,
                        verbose => $verbose,
                        buffer  => \$buffer )
        ) {
            error(loc("Failed to rmdir '%1': %2",$dir,$buffer));
            $flag++;
        }
    }

    $self->status->uninstall(!$flag);
    $self->status->installed( $flag ? 1 : undef);

    return !$flag;
}

sub distributions {
    my $self = shift;
    my %hash = @_;

    my @list = $self->author->distributions( %hash, module => $self ) or return;

    ### it's another release then by the same author ###
    return grep { $_->package_name eq $self->package_name } @list;
}

for my $sub (qw[files directory_tree packlist validate]) {
    no strict 'refs';
    *$sub = sub {
        return shift->_extutils_installed( @_, method => $sub );
    }
}

### generic method to call an ExtUtils::Installed method ###
sub _extutils_installed {
    my $self = shift;
    my $cb   = $self->parent;
    my $conf = $cb->configure_object;
    my $home = $cb->_home_dir;          # may be needed to fix up prefixes
    my %hash = @_;

    my ($verbose,$type,$method);
    my $tmpl = {
        verbose => {    default     => $conf->get_conf('verbose'),
                        store       => \$verbose, },
        type    => {    default     => 'all',
                        allow       => [qw|prog man all|],
                        store       => \$type, },
        method  => {    required    => 1,
                        store       => \$method,
                        allow       => [qw|files directory_tree packlist
                                        validate|],
                    },
    };

    my $args = check( $tmpl, \%hash ) or return;

    ### old versions of cygwin + perl < 5.8 are buggy here. bail out if we
    ### find we're being used by them
    {   my $err = ON_OLD_CYGWIN;
        if($err) { error($err); return };
    }

    return unless can_load(
                        modules     => { 'ExtUtils::Installed' => '0.0' },
                        verbose     => $verbose,
                    );

    my @config_names = (
        ### lib
        {   lib     => 'privlib',       # perl-only
            arch    => 'archlib',       # compiled code
            prefix  => 'prefix',        # prefix to both
        },
        ### site
        {   lib      => 'sitelib',
            arch     => 'sitearch',
            prefix   => 'siteprefix',
        },
        ### vendor
        {   lib     => 'vendorlib',
            arch    => 'vendorarch',
            prefix  => 'vendorprefix',
        },
    );

    ### search in your regular @INC, and anything you added to your config.
    ### this lets EU::Installed find .packlists that are *not* in the standard
    ### compiled in @INC path. Requires EU::I 1.42 or up. this addresses #33438
    ### make sure the archname path is also added, as that's where the .packlist
    ### files are written
    my @libs;
    for my $lib ( @{ $conf->get_conf('lib') } ) {
        require Config;
  
        ### and just the standard dir
        push @libs, $lib;
  
        ### figure out what an MM prefix expands to. Basically, it's the
        ### site install target from %Config, ie: /opt/lib/perl5/site_perl/5.8.8 
        ### minus the site wide prefix, ie: /opt
        ### this lets users add the dir they have set as their EU::MM PREFIX
        ### to our 'lib' config and it Just Works
        ### the arch specific dir, ie:
        ### /opt/lib/perl5/site_perl/5.8.8/darwin-2level        
        ### XXX is this the right thing to do?
        
        ### we add all 6 dir combos for prefixes:
        ### /foo/lib
        ### /foo/lib/arch
        ### /foo/site/lib
        ### /foo/site/lib/arch
        ### /foo/vendor/lib
        ### /foo/vendor/lib/arch
        for my $href ( @config_names ) {
            for my $key ( qw[lib arch] ) {
            
                ### look up the config value -- use EXP for the EXPANDED
                ### version, so no ~ etc are found in there
                my $dir     = $Config::Config{ $href->{ $key } .'exp' } or next;
                my $prefix  = $Config::Config{ $href->{prefix} };

                ### prefix may be relative to home, and contain a ~
                ### if so, fix it up.
                $prefix     =~ s/^~/$home/;

                ### remove the prefix from it, so we can append to our $lib
                $dir        =~ s/^\Q$prefix\E//;
                
                ### do the appending
                push @libs, File::Spec->catdir( $lib, $dir );
                
            }
        }
    }        

    my $inst;    
    unless( $inst = ExtUtils::Installed->new( extra_libs => \@libs ) ) {
        error( loc("Could not create an '%1' object", 'ExtUtils::Installed' ) );

        ### in case it's being used directly... ###
        return;
    }


    {   ### EU::Installed can die =/
        my @files;
        eval { @files = $inst->$method( $self->module, $type ) };

        if( $@ ) {
            chomp $@;
            error( loc("Could not get '%1' for '%2': %3",
                        $method, $self->module, $@ ) );
            return;
        }

        return wantarray ? @files : \@files;
    }
}

sub add_to_includepath {
    my $self = shift;
    my $cb   = $self->parent;
    
    if( my $dir = $self->status->extract ) {
        
            $cb->_add_to_includepath(
                    directories => [
                        File::Spec->catdir(BLIB->($dir), LIB),
                        File::Spec->catdir(BLIB->($dir), ARCH),
                        BLIB->($dir),
                    ]
            ) or return;
        
    } else {
        error(loc(  "No extract dir registered for '%1' -- can not add ".
                    "add builddir to search path!", $self->module ));
        return;
    }

    return 1;

}

### make sure we're always running 'perl Build.PL' and friends
### against the highest version of module::build available
sub best_path_to_module_build {
    my $self = shift;

    ### Since M::B will actually shell out and run the Build.PL, we must
    ### make sure it refinds the proper version of M::B in the path.
    ### that may be either in our cp::inc or in site_perl, or even a
    ### new M::B being installed.
    ### don't add anything else here, as that might screw up prereq checks

    ### XXX this might be needed for Dist::MM too, if a makefile.pl is
    ###	masquerading as a Build.PL

    ### did we find the most recent module::build in our installer path?

    ### XXX can't do changes to @INC, they're being ignored by
    ### new_from_context when writing a Build script. see ticket:
    ### #8826 Module::Build ignores changes to @INC when writing Build
    ### from new_from_context
    ### XXX applied schwern's patches (as seen on CPANPLUS::Devel 10/12/04)
    ### and upped the version to 0.26061 of the bundled version, and things
    ### work again

    ### this functionality is now obsolete -- prereqs should be installed
    ### and we no longer use the CPANPLUS::inc magic.. so comment this out.
#     require Module::Build;
#     if( CPANPLUS::inc->path_to('Module::Build') and (
#         CPANPLUS::inc->path_to('Module::Build') eq
#         CPANPLUS::inc->installer_path )
#     ) {
# 
#         ### if the module being installed is *not* Module::Build
#         ### itself -- as that would undoubtedly be newer -- add
#         ### the path to the installers to @INC
#         ### if it IS module::build itself, add 'lib' to its path,
#         ### as the Build.PL would do as well, but the API doesn't.
#         ### this makes self updates possible
#         return $self->module eq 'Module::Build'
#                         ? 'lib'
#                         : CPANPLUS::inc->installer_path;
#     }

    ### otherwise, the path was found through a 'normal' way of
    ### scanning @INC.
    return;
}

# Local variables:
# c-indentation-style: bsd
# c-basic-offset: 4
# indent-tabs-mode: nil
# End:
# vim: expandtab shiftwidth=4:

1;

__END__

todo:
reports();
