#!perl

use strict;
use warnings;
$|=1;
use Test::More tests => 1;
use DBI;
#use DBD::SQLite;
use File::Spec;
use File::Path;
use File::Basename;

# rebuild cpanstats db

my $f = File::Spec->catfile('t','_DBDIR','test.db');
unlink $f if -f $f;
mkpath( dirname($f) );

my $dbh = DBI->connect("dbi:SQLite:dbname=$f", '', '', {AutoCommit=>1});
$dbh->do(q{
  CREATE TABLE cpanstats (
                          id            INTEGER PRIMARY KEY,
                          guid          TEXT,
                          state         TEXT,
                          postdate      TEXT,
                          tester        TEXT,
                          dist          TEXT,
                          version       TEXT,
                          platform      TEXT,
                          perl          TEXT,
                          osname        TEXT,
                          osvers        TEXT,
                          fulldate      TEXT
  )
});

while(<DATA>){
  chomp;
  $dbh->do('INSERT INTO cpanstats ( id, guid, state, postdate, tester, dist, version, platform, perl, osname, osvers, fulldate ) VALUES ( ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ? )', {}, split(/\|/,$_) );
}

$dbh->do(q{ CREATE INDEX distverstate ON cpanstats (dist, version, state) });
$dbh->do(q{ CREATE INDEX ixdate ON cpanstats (postdate) });
$dbh->do(q{ CREATE INDEX ixperl ON cpanstats (perl) });
$dbh->do(q{ CREATE INDEX ixplat ON cpanstats (platform) });

my ($ct) = $dbh->selectrow_array('select count(*) from cpanstats');

$dbh->disconnect;

is($ct, 14034, "row count for cpanstats");

#select * from cpanstats where state='cpan' and dist in (SELECT dist from ixlatest where author='DCANTRELL')
# sqlite> select * from cpanstats where postdate=200901 order by dist limit 20;
# id|guid|state|postdate|tester|dist|version|platform|perl|osname|osvers|date
__DATA__
3031|3031-b19f-3f77-b713-d32bba55d77f|pass|199910|schinder@pobox.com|Statistics-ChiSquare|0.2|PA-RISC1.1|5.4.4|hpux|10.20|199910191643
3032|3032-b19f-3f77-b713-d32bba55d77f|pass|199910|schinder@pobox.com|Statistics-ChiSquare|0.2|sun4-solaris|5.5.3|solaris|2.7|199910191644
3121|3121-b19f-3f77-b713-d32bba55d77f|pass|199910|schinder@pobox.com|Statistics-ChiSquare|0.2|ppc-linux|5.5.3|linux|2.2.10|199910221334
3147|3147-b19f-3f77-b713-d32bba55d77f|pass|199910|ljmoore@freespace.net (Larry Moore)|Statistics-ChiSquare|0.2|MacOS_CFM68K|5.4.0|MacOS|7.6|199910230929
18566|18566-b19f-3f77-b713-d32bba55d77f|pass|200102|a.calpini@alos.it ("alos.it/a.calpini")|Data-Compare|0.01|MSWin32-x86-multi-thread|5.6.0|MSWin32|4.0|200102091428
18584|18584-b19f-3f77-b713-d32bba55d77f|pass|200102|schinder@pobox.com|Data-Hexdumper|0.01|ppc-linux|5.6.0|linux|2.2.14|200102091544
18585|18585-b19f-3f77-b713-d32bba55d77f|pass|200102|schinder@pobox.com|Data-Hexdumper|0.01|sun4-solaris|5.6.0|solaris|2.7|200102091544
18586|18586-b19f-3f77-b713-d32bba55d77f|pass|200102|schinder@pobox.com|Data-Hexdumper|0.01|PA-RISC1.1|5.6.0|hpux|10.20|200102091544
18661|18661-b19f-3f77-b713-d32bba55d77f|pass|200102|a.calpini@alos.it ("alos.it/a.calpini")|Data-Hexdumper|0.01|MSWin32-x86-multi-thread|5.6.0|MSWin32|4.0|200102121237
19470|19470-b19f-3f77-b713-d32bba55d77f|pass|200103|a.calpini@alos.it ("alos.it/a.calpini")|Statistics-ChiSquare|0.2|MSWin32-x86-multi-thread|5.6.0|MSWin32|4.0|200103011637
21002|21002-b19f-3f77-b713-d32bba55d77f|unknown|200104|cpantest@quaday.com (Tim Ayers)|Pony|1.01|sparc-openbsd|5.6.0|openbsd|2.8|200104050832
21167|21167-b19f-3f77-b713-d32bba55d77f|pass|200104|cpantest@quaday.com (Tim Ayers)|Tie-Scalar-Decay|1.0|sparc-openbsd|5.6.0|openbsd|2.8|200104110838
22016|22016-b19f-3f77-b713-d32bba55d77f|pass|200104|kriegjcb@CMC01SE.rz.ruhr-uni-bochum.de|Tie-Scalar-Decay|1.1|i686-linux|5.7.1|linux|2.2.16|200104251735
22040|22040-b19f-3f77-b713-d32bba55d77f|unknown|200104|kriegjcb@sunu991.ruhr-uni-bochum.de|Data-Compare|0.02|sun4-solaris|5.6.1|solaris|2.8|200104260801
22045|22045-b19f-3f77-b713-d32bba55d77f|pass|200104|Jost.Krieger@ruhr-uni-bochum.de|Data-Compare|0.02|sun4-solaris|5.6.1|solaris|2.8|200104260812
22158|22158-b19f-3f77-b713-d32bba55d77f|pass|200104|cpantest@quaday.com (Tim Ayers)|Data-Compare|0.02|sparc-openbsd|5.6.0|openbsd|2.8|200104270739
22176|22176-b19f-3f77-b713-d32bba55d77f|pass|200104|kriegjcb@CMC01SE.rz.ruhr-uni-bochum.de|Data-Compare|0.02|i686-linux|5.7.1|linux|2.2.16|200104271607
22192|22192-b19f-3f77-b713-d32bba55d77f|pass|200104|schinder@pobox.com|Tie-Scalar-Decay|1.1|sun4-solaris|5.6.1|solaris|2.7|200104271416
22193|22193-b19f-3f77-b713-d32bba55d77f|pass|200104|schinder@pobox.com|Tie-Scalar-Decay|1.1|PA-RISC1.1|5.6.1|hpux|10.20|200104271416
22194|22194-b19f-3f77-b713-d32bba55d77f|fail|200104|schinder@pobox.com|Tie-Scalar-Decay|1.1|ppc-linux|5.6.1|linux|2.4.2|200104271420
23486|23486-b19f-3f77-b713-d32bba55d77f|pass|200105|Jost.Krieger@ruhr-uni-bochum.de|Tie-Scalar-Decay|1.1.1|sun4-solaris|5.6.1|solaris|2.8|200105221052
23554|23554-b19f-3f77-b713-d32bba55d77f|pass|200105|cpantest@quaday.com (Tim Ayers)|Tie-Scalar-Decay|1.1.1|OpenBSD.sparc-openbsd|5.6.1|openbsd|2.8|200105230932
24779|24779-b19f-3f77-b713-d32bba55d77f|unknown|200106|code@leegoddard.com (Lee Goddard)|Acme-Pony|1.1.1|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200106202115
24988|24988-b19f-3f77-b713-d32bba55d77f|pass|200106|Jost.Krieger@ruhr-uni-bochum.de|Tie-Hash-Rank|1.0.1|sun4-solaris|5.6.1|solaris|2.8|200106281517
25189|25189-b19f-3f77-b713-d32bba55d77f|unknown|200107|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.1|sun4-solaris|5.6.1|solaris|2.8|200107091033
27937|27937-b19f-3f77-b713-d32bba55d77f|unknown|200110|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.2|sun4-solaris|5.6.1|solaris|2.8|200110101407
29413|29413-b19f-3f77-b713-d32bba55d77f|pass|200111|gellyfish@gellyfish.com (Jonathan Stowe)|Tie-Hash-Transactional|1.0|i686-linux|5.6.1|linux|2.2.16-22|200111112039
30073|30073-b19f-3f77-b713-d32bba55d77f|pass|200111|ethan@r220-1.rz.RWTH-Aachen.DE (Tassilo von Parseval)|Statistics-ChiSquare|0.3|i686-linux-thread|5.7.2|linux|2.4.13|200111240806
30092|30092-b19f-3f77-b713-d32bba55d77f|pass|200111|riechert@pobox.com (Andreas Marcel Riechert)|Statistics-ChiSquare|0.3|i686-linux|5.6.1|linux|2.2.14-5.0|200111241833
30240|30240-b19f-3f77-b713-d32bba55d77f|pass|200111|randy@theoryx5.uwinnipeg.ca (Randy Kobes)|Statistics-ChiSquare|0.3|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200111252029
31844|31844-b19f-3f77-b713-d32bba55d77f|unknown|200201|riechert@pobox.com (Andreas Marcel Riechert)|Acme-Pony|1.1.2|i686-linux|5.6.1|linux|2.2.14-5.0|200201051646
32918|32918-b19f-3f77-b713-d32bba55d77f|pass|200201|easmith@beatrice.rutgers.edu (Ed Allen Smith)|Data-Compare|0.02|IP30-R12000-irix|5.6.0|irix|6.5|200201231304
36386|36386-b19f-3f77-b713-d32bba55d77f|pass|200204|autrijus@www.autrijus.org (Autrijus Tang)|Data-Compare|0.02|i386-freebsd|5.7.3 patch 15354|freebsd|4.5-release|200204011435
36442|36442-b19f-3f77-b713-d32bba55d77f|unknown|200204|autrijus@www.autrijus.org (Autrijus Tang)|Acme-Pony|1.1.2|i386-freebsd|5.7.3 patch 15354|freebsd|4.5-release|200204021614
48917|48917-b19f-3f77-b713-d32bba55d77f|pass|200207|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Statistics-ChiSquare|0.3|sun4-solaris|5.8.0|solaris|2.8|200207310932
48959|48959-b19f-3f77-b713-d32bba55d77f|pass|200207|perl_test@logicalchaos.org ((Automated Perl Test Account))|Statistics-ChiSquare|0.3|i386-linux|5.6.1|linux|2.4.8-11mdkenterprise|200207310518
50072|50072-b19f-3f77-b713-d32bba55d77f|pass|200208|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Hexdumper|0.01|sun4-solaris|5.8.0|solaris|2.8|200208051136
50073|50073-b19f-3f77-b713-d32bba55d77f|pass|200208|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Tie-Hash-Transactional|1.0|sun4-solaris|5.8.0|solaris|2.8|200208051138
50115|50115-b19f-3f77-b713-d32bba55d77f|pass|200208|perl_test@logicalchaos.org ((Automated Perl Test Account))|Data-Hexdumper|0.01|i386-linux|5.6.1|linux|2.4.8-11mdkenterprise|200208050731
50116|50116-b19f-3f77-b713-d32bba55d77f|pass|200208|perl_test@logicalchaos.org ((Automated Perl Test Account))|Tie-Hash-Transactional|1.0|i386-linux|5.6.1|linux|2.4.8-11mdkenterprise|200208050731
50564|50564-b19f-3f77-b713-d32bba55d77f|pass|200208|perl_test@logicalchaos.org ((Automated Perl Test Account))|Data-Compare|0.02|i386-linux|5.6.1|linux|2.4.8-11mdkenterprise|200208060203
53617|53617-b19f-3f77-b713-d32bba55d77f|pass|200208|alian@cpan.org|Data-Compare|0.02|i386-netbsd|5.8.0|netbsd|1.5.2|200208141515
54328|54328-b19f-3f77-b713-d32bba55d77f|pass|200208|smoke@nambit.net (Dans Testorama)|Data-Compare|0.02|i386-freebsd|5.6.1|freebsd|4.2-release|200208162340
54333|54333-b19f-3f77-b713-d32bba55d77f|pass|200208|smoke@nambit.net (Dans Testorama)|Data-Compare|0.02|i386-freebsd-thread-multi|5.8.0|freebsd|4.2-release|200208162342
54350|54350-b19f-3f77-b713-d32bba55d77f|unknown|200208|smoke@nambit.net (Dans Testorama)|Acme-Pony|1.1.2|i386-freebsd|5.5.3|freebsd|4.0-current|200208170011
54351|54351-b19f-3f77-b713-d32bba55d77f|unknown|200208|smoke@nambit.net (Dans Testorama)|Acme-Pony|1.1.2|i386-freebsd|5.6.1|freebsd|4.2-release|200208170011
54352|54352-b19f-3f77-b713-d32bba55d77f|unknown|200208|smoke@nambit.net (Dans Testorama)|Acme-Pony|1.1.2|i386-freebsd-thread-multi|5.8.0|freebsd|4.2-release|200208170012
54628|54628-b19f-3f77-b713-d32bba55d77f|pass|200208|smoke@nambit.net (Dans Testorama)|Statistics-ChiSquare|0.3|i386-freebsd|5.5.3|freebsd|4.0-current|200208170322
54629|54629-b19f-3f77-b713-d32bba55d77f|pass|200208|smoke@nambit.net (Dans Testorama)|Statistics-ChiSquare|0.3|i386-freebsd|5.6.1|freebsd|4.2-release|200208170322
54630|54630-b19f-3f77-b713-d32bba55d77f|pass|200208|smoke@nambit.net (Dans Testorama)|Statistics-ChiSquare|0.3|i386-freebsd-thread-multi|5.8.0|freebsd|4.2-release|200208170322
55844|55844-b19f-3f77-b713-d32bba55d77f|pass|200208|alian@cpan.org|Data-Hexdumper|0.01|i386-netbsd|5.8.0|netbsd|1.5.2|200208190038
56068|56068-b19f-3f77-b713-d32bba55d77f|pass|200208|perl_test@logicalchaos.org ((Automated Perl Test Account))|Data-Compare|0.02|i386-linux|5.6.1|linux|2.4.16-6mdksmp|200208191423
56910|56910-b19f-3f77-b713-d32bba55d77f|pass|200208|alian@cpan.org (alian)|Data-Compare|0.02|OpenBSD.i386-openbsd|5.8.0|openbsd|3.0|200208221001
56921|56921-b19f-3f77-b713-d32bba55d77f|pass|200208|alian@cpan.org (alian)|Data-Hexdumper|0.01|OpenBSD.i386-openbsd|5.8.0|openbsd|3.0|200208221028
58539|58539-b19f-3f77-b713-d32bba55d77f|pass|200208|alian@cpan.org (alian)|Data-Compare|0.02|i386-freebsd|5.8.0|freebsd|4.6.2-release|200208301816
58554|58554-b19f-3f77-b713-d32bba55d77f|pass|200208|alian@cpan.org (alian)|Data-Hexdumper|0.01|i386-freebsd|5.8.0|freebsd|4.6.2-release|200208301832
60246|60246-b19f-3f77-b713-d32bba55d77f|unknown|200209|perl_test@logicalchaos.org ((Automated Perl Test Account))|Acme-Licence|1.0|i386-linux|5.6.1|linux|2.4.16-6mdksmp|200209190454
60251|60251-b19f-3f77-b713-d32bba55d77f|unknown|200209|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Acme-Licence|1.0|sun4-solaris|5.8.0|solaris|2.8|200209191322
60401|60401-b19f-3f77-b713-d32bba55d77f|pass|200209|perl_test@logicalchaos.org ((Automated Perl Test Account))|Data-Hexdumper|1.0.1|i386-linux|5.6.1|linux|2.4.16-6mdksmp|200209200810
60428|60428-b19f-3f77-b713-d32bba55d77f|pass|200209|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Hexdumper|1.0.1|sun4-solaris|5.8.0|solaris|2.8|200209201635
61369|61369-b19f-3f77-b713-d32bba55d77f|pass|200209|root@thesafrans.com ((root))|Statistics-ChiSquare|0.3|i686-linux|5.8.0|linux|2.4.7-10|200209290847
61871|61871-b19f-3f77-b713-d32bba55d77f|pass|200209|jlatour@www.arens.nl (Jeroen Latour)|Statistics-ChiSquare|0.3|i586-linux|5.8.0|linux|2.2.16c32_iii|200209301534
61914|61914-b19f-3f77-b713-d32bba55d77f|pass|200209|jlatour@www.arens.nl (Jeroen Latour)|Data-Compare|0.02|i586-linux|5.8.0|linux|2.2.16c32_iii|200209301558
61923|61923-b19f-3f77-b713-d32bba55d77f|pass|200209|jlatour@www.arens.nl (Jeroen Latour)|Data-Hexdumper|1.0.1|i586-linux|5.8.0|linux|2.2.16c32_iii|200209301602
62459|62459-b19f-3f77-b713-d32bba55d77f|unknown|200210|jlatour@www.arens.nl (Jeroen Latour)|Acme-Pony|1.1.2|i586-linux|5.8.0|linux|2.2.16c32_iii|200210012348
62470|62470-b19f-3f77-b713-d32bba55d77f|unknown|200210|jlatour@www.arens.nl (Jeroen Latour)|Acme-Licence|1.0|i586-linux|5.8.0|linux|2.2.16c32_iii|200210020052
62479|62479-b19f-3f77-b713-d32bba55d77f|unknown|200210|perl_test@logicalchaos.org ((Automated Perl Test Account))|Acme-Pony|1.1.2|i386-linux|5.6.1|linux|2.4.16-6mdksmp|200210011738
64602|64602-b19f-3f77-b713-d32bba55d77f|unknown|200210|root@godzilla.uni.to (root)|Acme-Pony|1.1.2|i386-linux|5.6.1|linux|2.4.17-0.13smp|200210161825
68081|68081-b19f-3f77-b713-d32bba55d77f|pass|200211|alian@cpan.org|Statistics-ChiSquare|0.3|i386-netbsd|5.8.0|netbsd|1.5.2|200211151932
68422|68422-b19f-3f77-b713-d32bba55d77f|pass|200211|alian@cpan.org (Alian - CPANPLUS/auto)|Statistics-ChiSquare|0.3|OpenBSD.i386-openbsd|5.8.0|openbsd|3.0|200211152330
70145|70145-b19f-3f77-b713-d32bba55d77f|pass|200211|alian@cpan.org (Alian - CPANPLUS/auto)|Data-Hexdumper|1.0.1|i386-netbsd|5.8.0|netbsd|1.5.2|200211210114
70469|70469-b19f-3f77-b713-d32bba55d77f|pass|200211|alian@cpan.org|Data-Compare|0.02|sparc-netbsd|5.6.1|netbsd|1.5.1|200211211430
70495|70495-b19f-3f77-b713-d32bba55d77f|pass|200211|alian@cpan.org (Alian - CPANPLUS/auto)|Statistics-ChiSquare|0.3|i386-freebsd|5.8.0|freebsd|4.6.2-release|200211211354
70505|70505-b19f-3f77-b713-d32bba55d77f|pass|200211|alian@cpan.org|Data-Hexdumper|1.0.1|sparc-netbsd|5.6.1|netbsd|1.5.1|200211211512
78607|78607-b19f-3f77-b713-d32bba55d77f|pass|200302|alian@cpan.org ((Cpan Tester - CPAN++ Automate ))|Data-Compare|0.02|i386-netbsd-thread-multi|5.8.0|netbsd|1.6|200302110230
78663|78663-b19f-3f77-b713-d32bba55d77f|pass|200302|alian@cpan.org ((CPAN Tester = CPAN++ automate))|Data-Compare|0.02|ppc-linux|5.8.0|linux|2.4.18-0.9a|200302110322
78789|78789-b19f-3f77-b713-d32bba55d77f|pass|200302|alian@cpan.org ((CPAN Tester''CPAN++ automate))|Data-Compare|0.02|sun4-solaris|5.8.0|solaris|2.7|200302110526
79039|79039-b19f-3f77-b713-d32bba55d77f|pass|200302|alian@cpan.org ((CPAN Tester_CPAN++ automate))|Data-Compare|0.02|ppc-linux|5.8.0|linux|2.2.16|200302110959
79143|79143-b19f-3f77-b713-d32bba55d77f|pass|200302|alian@cpan.org|Data-Compare|0.02|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200302111124
79430|79430-b19f-3f77-b713-d32bba55d77f|pass|200302|alian@cpan.org ((alian))|Data-Compare|0.02|sparc-linux|5.6.1|linux|2.4.18|200302120456
79656|79656-b19f-3f77-b713-d32bba55d77f|pass|200302|alian@cpan.org ((CPAN Tester / CPAN++ automate))|Data-Compare|0.02|sparc-netbsd|5.8.0|netbsd|1.5.3|200302130551
86049|86049-b19f-3f77-b713-d32bba55d77f|pass|200303|root@localhost.localdomain ((root))|Data-Compare|0.02|i386-linux|5.8.0|linux|2.4.18-11smp|200303300513
90155|90155-b19f-3f77-b713-d32bba55d77f|pass|200305|alian@cpan.org ((alian))|Data-Hexdumper|1.0.1|i686-linux-thread-multi-64all-ld|5.8.0|linux|2.4.20|200305131230
91242|91242-b19f-3f77-b713-d32bba55d77f|pass|200305|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Bryar|1.0|sun4-solaris|5.8.0|solaris|2.8|200305242112
91335|91335-b19f-3f77-b713-d32bba55d77f|pass|200305|cpantester@calaquendi.net ((Jeroen Latour))|Bryar|1.0|i586-linux|5.8.0|linux|2.2.16c32_iii|200305260746
91402|91402-b19f-3f77-b713-d32bba55d77f|fail|200305|crazyinsomniac@yahoo.com (DH)|Bryar|1.0|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200305270008
92820|92820-b19f-3f77-b713-d32bba55d77f|pass|200306|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Bryar|1.2|sun4-solaris|5.8.0|solaris|2.8|200306150155
94030|94030-b19f-3f77-b713-d32bba55d77f|pass|200306|cpantester@calaquendi.net ((Jeroen Latour))|NestedMap|1.0|i586-linux|5.8.0|linux|2.2.16c32_iii|200306261252
94034|94034-b19f-3f77-b713-d32bba55d77f|pass|200306|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|NestedMap|1.0|sun4-solaris|5.8.0|solaris|2.8|200306261239
94252|94252-b19f-3f77-b713-d32bba55d77f|pass|200306|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|File-Find-Rule-Permissions|1.0|sun4-solaris|5.8.0|solaris|2.8|200306300842
94315|94315-b19f-3f77-b713-d32bba55d77f|pass|200306|cpantester@calaquendi.net ((Jeroen Latour))|File-Find-Rule-Permissions|1.0|i586-linux|5.8.0|linux|2.2.16c32_iii|200306302104
95944|95944-b19f-3f77-b713-d32bba55d77f|pass|200307|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|File-Find-Rule-Permissions|1.1|sun4-solaris|5.8.0|solaris|2.8|200307230034
97303|97303-b19f-3f77-b713-d32bba55d77f|pass|200308|cpantester@calaquendi.net ((Jeroen Latour))|File-Find-Rule-Permissions|1.1|i586-linux|5.8.0|linux|2.2.16c32_iii|200308061042
97862|97862-b19f-3f77-b713-d32bba55d77f|pass|200308|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Hexdumper|1.1|sun4-solaris|5.8.0|solaris|2.8|200308101707
98649|98649-b19f-3f77-b713-d32bba55d77f|fail|200308|cpantester@calaquendi.net ((Jeroen Latour))|Bryar|2.0|i586-linux|5.8.0|linux|2.2.16c32_iii|200308161636
98691|98691-b19f-3f77-b713-d32bba55d77f|fail|200308|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Bryar|2.0|sun4-solaris|5.8.0|solaris|2.8|200308161658
99060|99060-b19f-3f77-b713-d32bba55d77f|pass|200308|root@dropbear ((root))|Data-Hexdumper|1.1|i586-linux|5.8.0|linux|2.4.20-22em|200308221803
99725|99725-b19f-3f77-b713-d32bba55d77f|pass|200308|cpantester@calaquendi.net ((Jeroen Latour))|Tie-Hash-Longest|1.0.reupload-again-because-im-stupid|i586-linux|5.8.0|linux|2.2.16c32_iii|200308290931
99860|99860-b19f-3f77-b713-d32bba55d77f|pass|200308|crazyinsomniac@yahoo.com|Tie-Hash-Longest|1.0.reupload-again-because-im-stupid|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200308310346
100133|100133-b19f-3f77-b713-d32bba55d77f|pass|200309|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Tie-Hash-Longest|1.0.reupload-again-because-im-stupid|sun4-solaris|5.8.0|solaris|2.8|200309011741
100444|100444-b19f-3f77-b713-d32bba55d77f|pass|200309|jlatour@calaquendi.net|Acme-Scurvy-Whoreson-BilgeRat|1.0|i586-linux|5.8.0|linux|2.2.16c32_iii|200309030257
100450|100450-b19f-3f77-b713-d32bba55d77f|pass|200309|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Acme-Scurvy-Whoreson-BilgeRat|1.0|sun4-solaris|5.8.0|solaris|2.8|200309030244
103212|103212-b19f-3f77-b713-d32bba55d77f|pass|200309|Perl_Test@LogicalChaos.org|Data-Hexdumper|1.1|i386-linux|5.8.0|linux|2.4.20-2mdkenterprise|200309280407
104500|104500-b19f-3f77-b713-d32bba55d77f|unknown|200310|kriegjcb@mi.ruhr-uni-bochum.de ((Jost Krieger))|Acme-Pony|1.1.2|sun4-solaris|5.8.1|solaris|2.8|200310061309
105242|105242-b19f-3f77-b713-d32bba55d77f|pass|200310|webmaster@thr.msu.edu|Data-Compare|0.02|i386-freebsd|5.6.1|freebsd|5.1-current|200310091009
105391|105391-b19f-3f77-b713-d32bba55d77f|pass|200310|webmaster@thr.msu.edu|Data-Hexdumper|1.1|i386-freebsd|5.6.1|freebsd|5.1-current|200310091842
105564|105564-b19f-3f77-b713-d32bba55d77f|pass|200310|webmaster@thr.msu.edu|File-Find-Rule-Permissions|1.1|i386-freebsd|5.6.1|freebsd|5.1-current|200310100200
107669|107669-b19f-3f77-b713-d32bba55d77f|pass|200310|webmaster@thr.msu.edu|Tie-Hash-Longest|1.0.reupload-again-because-im-stupid|i386-freebsd|5.6.1|freebsd|5.1-current|200310191312
107670|107670-b19f-3f77-b713-d32bba55d77f|pass|200310|webmaster@thr.msu.edu|Tie-Hash-Transactional|1.0|i386-freebsd|5.6.1|freebsd|5.1-current|200310191312
107671|107671-b19f-3f77-b713-d32bba55d77f|pass|200310|webmaster@thr.msu.edu|Tie-Hash-Rank|1.0.1|i386-freebsd|5.6.1|freebsd|5.1-current|200310191312
107672|107672-b19f-3f77-b713-d32bba55d77f|pass|200310|webmaster@thr.msu.edu|Tie-Scalar-Decay|1.1.1|i386-freebsd|5.6.1|freebsd|5.1-current|200310191313
107712|107712-b19f-3f77-b713-d32bba55d77f|pass|200310|Perl_Test@LogicalChaos.org|Tie-Hash-Rank|1.0.1|i386-linux|5.8.0|linux|2.4.20-2mdkenterprise|200310191223
107713|107713-b19f-3f77-b713-d32bba55d77f|pass|200310|Perl_Test@LogicalChaos.org|Tie-Scalar-Decay|1.1.1|i386-linux|5.8.0|linux|2.4.20-2mdkenterprise|200310191223
107906|107906-b19f-3f77-b713-d32bba55d77f|pass|200310|nothingmuch@woobling.org|Games-Dice-Advanced|1.0|darwin|5.8.1|darwin|6.8|200310210201
108039|108039-b19f-3f77-b713-d32bba55d77f|pass|200310|webmaster@thr.msu.edu|Games-Dice-Advanced|1.0|i386-freebsd|5.6.1|freebsd|5.1-current|200310211714
108140|108140-b19f-3f77-b713-d32bba55d77f|pass|200310|Perl_Test@LogicalChaos.org|Games-Dice-Advanced|1.0|i386-linux|5.8.0|linux|2.4.20-2mdkenterprise|200310220104
108339|108339-b19f-3f77-b713-d32bba55d77f|pass|200310|barbie@missbarbell.co.uk (Barbie)|Games-Dice-Advanced|1.0|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200310222256
108348|108348-b19f-3f77-b713-d32bba55d77f|pass|200310|nothingmuch@woobling.org|Net-Random|1.0|darwin|5.8.1|darwin|6.8|200310230107
108357|108357-b19f-3f77-b713-d32bba55d77f|pass|200310|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Net-Random|1.0|sun4-solaris|5.8.1|solaris|2.8|200310230036
108976|108976-b19f-3f77-b713-d32bba55d77f|fail|200310|barbie@missbarbell.co.uk (Barbie)|Net-Random|1.0|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200310280741
111603|111603-b19f-3f77-b713-d32bba55d77f|pass|200311|nothingmuch@woobling.org|Sub-WrapPackages|1.0|darwin|5.8.1|darwin|6.8|200311152328
111628|111628-b19f-3f77-b713-d32bba55d77f|pass|200311|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Sub-WrapPackages|1.0|sun4-solaris|5.8.1|solaris|2.8|200311161149
111640|111640-b19f-3f77-b713-d32bba55d77f|pass|200311|nothingmuch@woobling.org|Statistics-ChiSquare|0.5|darwin|5.8.1|darwin|6.8|200311161944
111706|111706-b19f-3f77-b713-d32bba55d77f|pass|200311|nothingmuch@woobling.org|Statistics-ChiSquare|0.5|powerpc-linux|5.8.0|linux|2.4.23-pre5|200311172329
111711|111711-b19f-3f77-b713-d32bba55d77f|pass|200311|nothingmuch@woobling.org|Tie-Hash-Longest|1.1|darwin|5.8.1|darwin|6.8|200311180043
111714|111714-b19f-3f77-b713-d32bba55d77f|pass|200311|nothingmuch@woobling.org|Tie-Hash-Longest|1.1|powerpc-linux|5.8.0|linux|2.4.23-pre5|200311180020
111767|111767-b19f-3f77-b713-d32bba55d77f|pass|200311|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Statistics-ChiSquare|0.5|sun4-solaris|5.8.1|solaris|2.8|200311181021
111776|111776-b19f-3f77-b713-d32bba55d77f|pass|200311|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Tie-Hash-Longest|1.1|sun4-solaris|5.8.1|solaris|2.8|200311181049
111919|111919-b19f-3f77-b713-d32bba55d77f|pass|200311|webmaster@thr.msu.edu|Sub-WrapPackages|1.0|i386-freebsd|5.8.2|freebsd|5.1-current|200311191538
111932|111932-b19f-3f77-b713-d32bba55d77f|pass|200311|webmaster@thr.msu.edu|Statistics-ChiSquare|0.5|i386-freebsd|5.8.2|freebsd|5.1-current|200311191552
111946|111946-b19f-3f77-b713-d32bba55d77f|pass|200311|webmaster@thr.msu.edu|Tie-Hash-Longest|1.1|i386-freebsd|5.8.2|freebsd|5.1-current|200311191635
111994|111994-b19f-3f77-b713-d32bba55d77f|pass|200311|barbie@missbarbell.co.uk (Barbie)|Tie-Hash-Longest|1.1|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200311200824
112009|112009-b19f-3f77-b713-d32bba55d77f|pass|200311|barbie@missbarbell.co.uk (Barbie)|Statistics-ChiSquare|0.5|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200311200824
112018|112018-b19f-3f77-b713-d32bba55d77f|pass|200311|barbie@missbarbell.co.uk (Barbie)|Sub-WrapPackages|1.0|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200311200824
112080|112080-b19f-3f77-b713-d32bba55d77f|pass|200311|nothingmuch@woobling.org|Statistics-SerialCorrelation|1.0|darwin|5.8.1|darwin|6.8|200311211439
112103|112103-b19f-3f77-b713-d32bba55d77f|pass|200311|nothingmuch@woobling.org|Statistics-SerialCorrelation|1.0|powerpc-linux|5.8.0|linux|2.4.23-pre5|200311211803
112146|112146-b19f-3f77-b713-d32bba55d77f|pass|200311|webmaster@thr.msu.edu|Statistics-SerialCorrelation|1.0|i386-freebsd|5.8.2|freebsd|5.1-current|200311220119
112172|112172-b19f-3f77-b713-d32bba55d77f|pass|200311|Perl_Test@LogicalChaos.org|Sub-WrapPackages|1.0|i386-linux|5.8.0|linux|2.4.20-2mdkenterprise|200311220236
112185|112185-b19f-3f77-b713-d32bba55d77f|pass|200311|Perl_Test@LogicalChaos.org|Statistics-ChiSquare|0.5|i386-linux|5.8.0|linux|2.4.20-2mdkenterprise|200311220248
112197|112197-b19f-3f77-b713-d32bba55d77f|pass|200311|Perl_Test@LogicalChaos.org|Tie-Hash-Longest|1.1|i386-linux|5.8.0|linux|2.4.20-2mdkenterprise|200311220323
112240|112240-b19f-3f77-b713-d32bba55d77f|pass|200311|Perl_Test@LogicalChaos.org|Statistics-SerialCorrelation|1.0|i386-linux|5.8.0|linux|2.4.20-2mdkenterprise|200311220439
112272|112272-b19f-3f77-b713-d32bba55d77f|pass|200311|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Statistics-SerialCorrelation|1.0|sun4-solaris|5.8.1|solaris|2.8|200311221038
112788|112788-b19f-3f77-b713-d32bba55d77f|pass|200311|barbie@missbarbell.co.uk (Barbie)|Statistics-SerialCorrelation|1.0|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200311270726
113523|113523-b19f-3f77-b713-d32bba55d77f|pass|200312|root@uk-wrk-0017 ((root))|Sub-WrapPackages|1.0|i386-linux|5.8.1|linux|2.4.22-xfs+ti1211|200312021956
114594|114594-b19f-3f77-b713-d32bba55d77f|fail|200312|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Bryar|2.1|sun4-solaris|5.8.1|solaris|2.8|200312121042
115002|115002-b19f-3f77-b713-d32bba55d77f|pass|200312|barbie@missbarbell.co.uk (Barbie)|Bryar|2.1|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200312152024
115405|115405-b19f-3f77-b713-d32bba55d77f|pass|200312|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|File-Find-Rule-Permissions|1.2|sun4-solaris|5.8.1|solaris|2.8|200312191034
115506|115506-b19f-3f77-b713-d32bba55d77f|pass|200312|webmaster@thr.msu.edu|File-Find-Rule-Permissions|1.2|i386-freebsd|5.6.1|freebsd|5.2-beta|200312200410
115565|115565-b19f-3f77-b713-d32bba55d77f|na|200312|barbie@missbarbell.co.uk (Barbie)|File-Find-Rule-Permissions|1.2|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200312201228
115890|115890-b19f-3f77-b713-d32bba55d77f|pass|200312|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Data-Compare|0.03|sun4-solaris|5.8.1|solaris|2.8|200312241039
115892|115892-b19f-3f77-b713-d32bba55d77f|pass|200312|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Data-Compare|0.04|sun4-solaris|5.8.1|solaris|2.8|200312241040
115938|115938-b19f-3f77-b713-d32bba55d77f|pass|200312|barbie@missbarbell.co.uk (Barbie)|Data-Compare|0.04|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200312242048
115941|115941-b19f-3f77-b713-d32bba55d77f|pass|200312|barbie@missbarbell.co.uk (Barbie)|Data-Compare|0.03|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200312242048
115967|115967-b19f-3f77-b713-d32bba55d77f|pass|200312|lithron@bellsouth.net|Data-Compare|0.04|cygwin-thread-multi-64int|5.8.2|cygwin|1.5.5(0.9432)|200312250300
115981|115981-b19f-3f77-b713-d32bba55d77f|pass|200312|webmaster@thr.msu.edu|Data-Compare|0.03|i386-freebsd|5.8.2|freebsd|5.2-current|200312251145
115983|115983-b19f-3f77-b713-d32bba55d77f|pass|200312|webmaster@thr.msu.edu|Data-Compare|0.04|i386-freebsd|5.8.2|freebsd|5.2-current|200312251146
116371|116371-b19f-3f77-b713-d32bba55d77f|pass|200312|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Data-Compare|0.05|sun4-solaris|5.8.1|solaris|2.8|200312311027
116578|116578-b19f-3f77-b713-d32bba55d77f|pass|200401|webmaster@thr.msu.edu|Data-Compare|0.05|i386-freebsd|5.8.2|freebsd|5.2-current|200401021841
117144|117144-b19f-3f77-b713-d32bba55d77f|pass|200401|barbie@missbarbell.co.uk (Barbie)|Data-Compare|0.05|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200401062214
117236|117236-b19f-3f77-b713-d32bba55d77f|pass|200401|Jost.Krieger@ruhr-uni-bochum.de|Data-Compare|0.06|sun4-solaris|5.8.1|solaris|2.8|200401070853
117337|117337-b19f-3f77-b713-d32bba55d77f|pass|200401|jlatour@calaquendi.net|Data-Compare|0.06|i586-linux|5.8.2|linux|2.2.16c37_iii|200401080005
117380|117380-b19f-3f77-b713-d32bba55d77f|pass|200401|webmaster@thr.msu.edu|Data-Compare|0.06|i386-freebsd|5.8.2|freebsd|5.2-current|200401080041
117485|117485-b19f-3f77-b713-d32bba55d77f|pass|200401|jlatour@calaquendi.net|Data-Compare|0.07|i586-linux|5.8.2|linux|2.2.16c37_iii|200401081057
117505|117505-b19f-3f77-b713-d32bba55d77f|pass|200401|Jost.Krieger@ruhr-uni-bochum.de|Data-Compare|0.07|sun4-solaris|5.8.1|solaris|2.8|200401081034
117607|117607-b19f-3f77-b713-d32bba55d77f|pass|200401|webmaster@thr.msu.edu|Data-Compare|0.07|i386-freebsd|5.8.2|freebsd|5.2-current|200401090408
117898|117898-b19f-3f77-b713-d32bba55d77f|pass|200401|barbie@missbarbell.co.uk (Barbie)|Data-Compare|0.07|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200401100851
117907|117907-b19f-3f77-b713-d32bba55d77f|pass|200401|barbie@missbarbell.co.uk (Barbie)|Data-Compare|0.06|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200401100851
119387|119387-b19f-3f77-b713-d32bba55d77f|pass|200401|Jost.Krieger@ruhr-uni-bochum.de|Data-Compare|0.08|sun4-solaris|5.8.1|solaris|2.8|200401210824
119444|119444-b19f-3f77-b713-d32bba55d77f|pass|200401|steve@gandalf.workpc.tds.net ((root))|Data-Compare|0.08|i386-linux|5.8.1|linux|2.4.21-2.elsmp|200401211242
119557|119557-b19f-3f77-b713-d32bba55d77f|pass|200401|barbie@missbarbell.co.uk (Barbie)|Data-Compare|0.08|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200401222135
119724|119724-b19f-3f77-b713-d32bba55d77f|pass|200401|steve@gandalf.workpc.tds.net ((root))|Bryar|2.3|i386-linux|5.8.1|linux|2.4.21-2.elsmp|200401231111
119747|119747-b19f-3f77-b713-d32bba55d77f|pass|200401|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Bryar|2.2|sun4-solaris|5.8.1|solaris|2.8|200401241057
119757|119757-b19f-3f77-b713-d32bba55d77f|pass|200401|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Statistics-SerialCorrelation|1.1|sun4-solaris|5.8.1|solaris|2.8|200401241132
119881|119881-b19f-3f77-b713-d32bba55d77f|pass|200401|webmaster@thr.msu.edu|Statistics-SerialCorrelation|1.1|i386-freebsd|5.8.2|freebsd|5.2-current|200401260203
120817|120817-b19f-3f77-b713-d32bba55d77f|pass|200402|barbie@missbarbell.co.uk (Barbie)|Bryar|2.2|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200402030754
120850|120850-b19f-3f77-b713-d32bba55d77f|pass|200402|barbie@missbarbell.co.uk (Barbie)|Bryar|2.3|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200402030754
120895|120895-b19f-3f77-b713-d32bba55d77f|pass|200402|barbie@missbarbell.co.uk (Barbie)|Statistics-SerialCorrelation|1.1|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200402030755
121670|121670-b19f-3f77-b713-d32bba55d77f|pass|200402|nothingmuch@woobling.org|Bryar|2.4|ppc-darwin-thread-multi|5.8.3|darwin|7.2.0|200402081917
121763|121763-b19f-3f77-b713-d32bba55d77f|pass|200402|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Bryar|2.5|sun4-solaris|5.8.1|solaris|2.8|200402091052
122036|122036-b19f-3f77-b713-d32bba55d77f|pass|200402|nothingmuch@woobling.org|Bryar|2.6|ppc-darwin-thread-multi|5.8.3|darwin|7.2.0|200402111538
122175|122175-b19f-3f77-b713-d32bba55d77f|pass|200402|barbie@missbarbell.co.uk (Barbie)|Bryar|2.5|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200402120813
122192|122192-b19f-3f77-b713-d32bba55d77f|pass|200402|barbie@missbarbell.co.uk (Barbie)|Bryar|2.4|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200402120813
122308|122308-b19f-3f77-b713-d32bba55d77f|pass|200402|barbie@missbarbell.co.uk (Barbie)|Bryar|2.6|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200402130742
123333|123333-b19f-3f77-b713-d32bba55d77f|pass|200402|nothingmuch@woobling.org|Data-Hexdumper|1.2|ppc-darwin-thread-multi|5.8.3|darwin|7.2.0|200402201702
123394|123394-b19f-3f77-b713-d32bba55d77f|pass|200402|cpansmoke@alternation.net|Data-Hexdumper|1.2|i586-linux|5.8.0|linux|2.4.22-4tr|200402210507
123399|123399-b19f-3f77-b713-d32bba55d77f|pass|200402|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Data-Hexdumper|1.2|sun4-solaris|5.8.1|solaris|2.8|200402211022
123609|123609-b19f-3f77-b713-d32bba55d77f|pass|200402|webmaster@thr.msu.edu|Data-Hexdumper|1.2|i386-freebsd|5.8.2|freebsd|5.2-current|200402221849
123735|123735-b19f-3f77-b713-d32bba55d77f|pass|200402|nothingmuch@woobling.org|Data-Compare|0.09|ppc-darwin-thread-multi|5.8.3|darwin|7.2.0|200402240107
123796|123796-b19f-3f77-b713-d32bba55d77f|pass|200402|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Data-Compare|0.09|sun4-solaris|5.8.1|solaris|2.8|200402241052
123817|123817-b19f-3f77-b713-d32bba55d77f|pass|200402|cpansmoke@alternation.net|Data-Compare|0.09|i586-linux|5.8.0|linux|2.4.22-4tr|200402240807
123896|123896-b19f-3f77-b713-d32bba55d77f|pass|200402|webmaster@thr.msu.edu|Data-Compare|0.09|i386-freebsd|5.8.2|freebsd|5.2-current|200402241850
124187|124187-b19f-3f77-b713-d32bba55d77f|pass|200402|barbie@missbarbell.co.uk (Barbie)|Data-Hexdumper|1.2|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200402260808
124420|124420-b19f-3f77-b713-d32bba55d77f|pass|200402|barbie@missbarbell.co.uk (Barbie)|Data-Compare|0.09|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200402270721
127016|127016-b19f-3f77-b713-d32bba55d77f|pass|200403|nothingmuch@woobling.org|Data-Compare|0.1|ppc-darwin-thread-multi|5.8.3|darwin|7.2.0|200403131335
127052|127052-b19f-3f77-b713-d32bba55d77f|pass|200403|cpansmoke@alternation.net|Data-Compare|0.1|i586-linux|5.8.0|linux|2.4.22-4tr|200403131807
127114|127114-b19f-3f77-b713-d32bba55d77f|pass|200403|kriegjcb@sunu991.rz.ruhr-uni-bochum.de|Data-Compare|0.1|sun4-solaris|5.8.1|solaris|2.8|200403141030
127238|127238-b19f-3f77-b713-d32bba55d77f|pass|200403|webmaster@thr.msu.edu|Data-Compare|0.1|i386-freebsd|5.8.2|freebsd|5.2-current|200403141830
128050|128050-b19f-3f77-b713-d32bba55d77f|pass|200403|barbie@missbarbell.co.uk (Barbie)|Data-Compare|0.1|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200403180756
129983|129983-b19f-3f77-b713-d32bba55d77f|pass|200403|Jost.Krieger@ruhr-uni-bochum.de|Data-Compare|0.1|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200403291616
132100|132100-b19f-3f77-b713-d32bba55d77f|pass|200404|cpansmoke@alternation.net|Bryar|2.6|i586-linux|5.8.0|linux|2.4.22-4tr|200404090008
132861|132861-b19f-3f77-b713-d32bba55d77f|pass|200404|kriegjcb@sunu991.rz.ruhr-uni-bochum.de|Bryar|2.6|sun4-solaris|5.8.1|solaris|2.8|200404150923
142676|142676-b19f-3f77-b713-d32bba55d77f|pass|200406|Jost.Krieger@ruhr-uni-bochum.de|Data-Compare|0.11|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200406030737
142964|142964-b19f-3f77-b713-d32bba55d77f|pass|200406|rrwo@cpan.org|Data-Compare|0.11|MSWin32-x86-multi-thread|5.8.3|MSWin32|4.0|200406040942
146822|146822-b19f-3f77-b713-d32bba55d77f|pass|200407|cpansmoke@alternation.net|Data-Compare|0.11|i586-linux|5.8.0|linux|2.4.22-4tr|200407050908
147721|147721-b19f-3f77-b713-d32bba55d77f|pass|200407|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Statistics-ChiSquare|0.5|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200407080346
148267|148267-b19f-3f77-b713-d32bba55d77f|pass|200407|rrwo@cpan.org|Class-CanBeA|1.0|MSWin32-x86-multi-thread|5.8.3|MSWin32|4.0|200407110109
148277|148277-b19f-3f77-b713-d32bba55d77f|pass|200407|rrwo@cpan.org|Class-CanBeA|1.0|MSWin32-x86-multi-thread|5.8.4|MSWin32|4.0|200407110250
148287|148287-b19f-3f77-b713-d32bba55d77f|pass|200407|cpansmoke@alternation.net|Class-CanBeA|1.0|i586-linux|5.8.0|linux|2.4.22-4tr|200407110408
148986|148986-b19f-3f77-b713-d32bba55d77f|pass|200407|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Class-CanBeA|1.0|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200407140502
151471|151471-b19f-3f77-b713-d32bba55d77f|pass|200407|webmaster@thr.msu.edu|Data-Compare|0.11|i386-freebsd-64int|5.8.4|freebsd|5.2-current|200407301504
154082|154082-b19f-3f77-b713-d32bba55d77f|pass|200408|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.0|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200408182140
154114|154114-b19f-3f77-b713-d32bba55d77f|pass|200408|cpantester@thr.msu.edu|Number-Phone|1.0|i386-freebsd-64int|5.8.5|freebsd|5.2-current|200408190127
154121|154121-b19f-3f77-b713-d32bba55d77f|pass|200408|rrwo@cpan.org|Number-Phone|1.0|MSWin32-x86-multi-thread|5.8.4|MSWin32|4.0|200408190341
154183|154183-b19f-3f77-b713-d32bba55d77f|pass|200408|martin.thurn@ngc.com ("Thurn, Martin")|Number-Phone|1.0|sun4-solaris|5.9.1|solaris|2.8|200408190754
155673|155673-b19f-3f77-b713-d32bba55d77f|unknown|200409|jim@jimandkoka.com|Acme-Pony|1.1.2|darwin-2level|5.8.4|darwin|7.3.0|200409021509
156532|156532-b19f-3f77-b713-d32bba55d77f|pass|200409|Jost.Krieger+perl@rub.de|Net-Random|1.1|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200409071326
156605|156605-b19f-3f77-b713-d32bba55d77f|pass|200409|alian@cpan.org ((CPAN Tester + CPAN++ automate))|Net-Random|1.1|ppc-linux|5.8.4|linux|2.4.19-4a|200409072209
156606|156606-b19f-3f77-b713-d32bba55d77f|pass|200409|alian@cpan.org ((CPAN Tester. CPAN++ automate))|Net-Random|1.1|OpenBSD.i386-openbsd|5.8.4|openbsd|3.5|200409080011
156917|156917-b19f-3f77-b713-d32bba55d77f|fail|200409|alian@cpan.org ((Cpan Tester - CPAN++ Automate ))|Net-Random|1.1|i386-netbsd-64int|5.8.4|netbsd|1.6|200409090006
157366|157366-b19f-3f77-b713-d32bba55d77f|pass|200409|barbie@missbarbell.co.uk (Barbie)|Net-Random|1.1|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200409110708
158211|158211-b19f-3f77-b713-d32bba55d77f|pass|200409|Jost.Krieger+perl@rub.de|Number-Phone|1.1|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200409160411
158232|158232-b19f-3f77-b713-d32bba55d77f|pass|200409|alian@cpan.org ((Cpan Tester - CPAN++ Automate ))|Number-Phone|1.1|i386-netbsd-64int|5.8.4|netbsd|1.6|200409161150
158233|158233-b19f-3f77-b713-d32bba55d77f|pass|200409|alian@cpan.org ((CPAN Tester. CPAN++ automate))|Number-Phone|1.1|OpenBSD.i386-openbsd|5.8.4|openbsd|3.5|200409161349
158237|158237-b19f-3f77-b713-d32bba55d77f|pass|200409|alian@cpan.org ((alian))|Number-Phone|1.1|sparc64-linux|5.8.5|linux|2.4.18|200409161204
158254|158254-b19f-3f77-b713-d32bba55d77f|pass|200409|martin.thurn@ngc.com ("Thurn, Martin")|Number-Phone|1.1|sun4-solaris|5.9.1|solaris|2.8|200409160520
158727|158727-b19f-3f77-b713-d32bba55d77f|pass|200409|cpantester@thr.msu.edu|Net-Random|1.1|i386-freebsd-64int|5.8.5|freebsd|5.2-current|200409171223
158921|158921-b19f-3f77-b713-d32bba55d77f|pass|200409|cpantester@thr.msu.edu|Number-Phone|1.1|i386-freebsd-64int|5.8.5|freebsd|5.2-current|200409180135
159531|159531-b19f-3f77-b713-d32bba55d77f|pass|200409|alian@cpan.org ((CPAN Tester + CPAN++ automate))|Number-Phone|1.1|ppc-linux|5.8.4|linux|2.4.19-4a|200409210855
159704|159704-b19f-3f77-b713-d32bba55d77f|pass|200409|barbie@missbarbell.co.uk (Barbie)|Number-Phone|1.1|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200409220830
160038|160038-b19f-3f77-b713-d32bba55d77f|pass|200409|Jost.Krieger+perl@rub.de|Number-Phone|1.2|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200409230357
160051|160051-b19f-3f77-b713-d32bba55d77f|pass|200409|alian@cpan.org ((CPAN Tester + CPAN++ automate))|Number-Phone|1.2|ppc-linux|5.8.4|linux|2.4.19-4a|200409230919
160093|160093-b19f-3f77-b713-d32bba55d77f|pass|200409|martin.thurn@ngc.com ("Thurn, Martin")|Number-Phone|1.2|sun4-solaris|5.9.1|solaris|2.8|200409230617
162192|162192-b19f-3f77-b713-d32bba55d77f|pass|200410|Jost.Krieger+perl@rub.de|Number-Phone|1.2001|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200410132331
162785|162785-b19f-3f77-b713-d32bba55d77f|pass|200410|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Bryar|2.8_01|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200410190938
162948|162948-b19f-3f77-b713-d32bba55d77f|pass|200410|Jost.Krieger+perl@rub.de|Number-Phone|1.3|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200410210359
163103|163103-b19f-3f77-b713-d32bba55d77f|pass|200410|Jost.Krieger+perl@rub.de|Bryar|2.6|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200410230659
163394|163394-b19f-3f77-b713-d32bba55d77f|pass|200410|Jost.Krieger+perl@rub.de|Acme-Scurvy-Whoreson-BilgeRat|1.1|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200410260359
163426|163426-b19f-3f77-b713-d32bba55d77f|pass|200410|shlomif@iglu.org.il (Shlomi Fish)|Number-Phone|1.3|i686-linux|5.8.5|linux|2.6.8.1-10mdksmp|200410261443
163536|163536-b19f-3f77-b713-d32bba55d77f|pass|200410|barbie@missbarbell.co.uk|Number-Phone|1.2|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200410270912
164481|164481-b19f-3f77-b713-d32bba55d77f|pass|200411|cpantester@thr.msu.edu|Number-Phone|1.2|i386-freebsd-64int|5.8.5|freebsd|5.2-current|200411021233
164948|164948-b19f-3f77-b713-d32bba55d77f|pass|200411|cpantester@thr.msu.edu|Number-Phone|1.2001|i386-freebsd-64int|5.8.5|freebsd|5.2-current|200411031032
165117|165117-b19f-3f77-b713-d32bba55d77f|pass|200411|corion+cpansmoker@corion.net|Acme-Scurvy-Whoreson-BilgeRat|1.1|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200411031909
165645|165645-b19f-3f77-b713-d32bba55d77f|pass|200411|saper@cpan.org|Data-Compare|0.11|i386-linux|5.8.1|linux|2.4.18-23mdksmp|200411071232
165739|165739-b19f-3f77-b713-d32bba55d77f|pass|200411|corion-cpansmoker@corion.net|Data-Transactional|0.1|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200411071010
165993|165993-b19f-3f77-b713-d32bba55d77f|pass|200411|cpantester@thr.msu.edu|Data-Transactional|0.1|i386-freebsd-64int|5.8.5|freebsd|5.2-current|200411090940
166096|166096-b19f-3f77-b713-d32bba55d77f|pass|200411|saper@cpan.org|Data-Compare|0.13|i386-linux|5.8.1|linux|2.4.18-23mdksmp|200411110826
166129|166129-b19f-3f77-b713-d32bba55d77f|pass|200411|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Compare|0.12|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200411100845
166130|166130-b19f-3f77-b713-d32bba55d77f|pass|200411|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Compare|0.13|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200411100846
166207|166207-b19f-3f77-b713-d32bba55d77f|pass|200411|barbie@missbarbell.co.uk|Number-Phone|1.2001|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200411101259
166420|166420-b19f-3f77-b713-d32bba55d77f|pass|200411|rrwo@cpan.org|Data-Compare|0.13|MSWin32-x86-multi-thread|5.8.4|MSWin32|4.0|200411110047
166421|166421-b19f-3f77-b713-d32bba55d77f|pass|200411|rrwo@cpan.org|Data-Compare|0.12|MSWin32-x86-multi-thread|5.8.4|MSWin32|4.0|200411110048
166465|166465-b19f-3f77-b713-d32bba55d77f|pass|200411|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.3001|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200411111217
166521|166521-b19f-3f77-b713-d32bba55d77f|pass|200411|barbie@missbarbell.co.uk|Number-Phone|1.3|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200411111855
166525|166525-b19f-3f77-b713-d32bba55d77f|fail|200411|barbie@missbarbell.co.uk|Bryar|2.8_01|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200411111855
166546|166546-b19f-3f77-b713-d32bba55d77f|fail|200411|barbie@missbarbell.co.uk|Data-Compare|0.13|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200411111855
166588|166588-b19f-3f77-b713-d32bba55d77f|pass|200411|cpantester@thr.msu.edu|Data-Compare|0.12|i386-freebsd-64int|5.8.5|freebsd|5.2-current|200411111416
166590|166590-b19f-3f77-b713-d32bba55d77f|pass|200411|cpantester@thr.msu.edu|Data-Compare|0.13|i386-freebsd-64int|5.8.5|freebsd|5.2-current|200411111416
166610|166610-b19f-3f77-b713-d32bba55d77f|pass|200411|corion-cpansmoker@corion.net|Number-Phone|1.3001|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200411112036
166652|166652-b19f-3f77-b713-d32bba55d77f|pass|200411|cpantester@thr.msu.edu|Number-Phone|1.3001|i386-freebsd-64int|5.8.5|freebsd|5.2-current|200411111530
166717|166717-b19f-3f77-b713-d32bba55d77f|pass|200411|Jost.Krieger+cpanplus@rub.de|Data-Transactional|0.1|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200411120929
167036|167036-b19f-3f77-b713-d32bba55d77f|pass|200411|shlomif@iglu.org.il (Shlomi Fish)|Number-Phone|1.3001|i686-linux|5.8.5|linux|2.6.8.1-10mdksmp|200411132058
169429|169429-b19f-3f77-b713-d32bba55d77f|pass|200411|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.3002|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200411261633
169442|169442-b19f-3f77-b713-d32bba55d77f|pass|200411|nothingmuch@woobling.org|Number-Phone|1.3002|i686-linux|5.9.1|linux|2.6.8.1|200411262046
169444|169444-b19f-3f77-b713-d32bba55d77f|pass|200411|nothingmuch@woobling.org|Number-Phone|1.3002|i386-linux-thread|5.8.4|linux|2.6.7-gentoo-r11|200411262047
169521|169521-b19f-3f77-b713-d32bba55d77f|fail|200411|nothingmuch@woobling.org|Number-Phone|1.3002|i686-linux|5.5.4|linux|2.6.8.1|200411262045
169522|169522-b19f-3f77-b713-d32bba55d77f|pass|200411|nothingmuch@woobling.org|Number-Phone|1.3002|i686-linux|5.6.2|linux|2.6.8.1|200411262046
169524|169524-b19f-3f77-b713-d32bba55d77f|pass|200411|nothingmuch@woobling.org|Number-Phone|1.3002|i686-linux-thread-multi|5.9.1|linux|2.6.8.1|200411262047
169526|169526-b19f-3f77-b713-d32bba55d77f|pass|200411|nothingmuch@woobling.org|Number-Phone|1.3002|i686-linux|5.8.5|linux|2.6.8.1|200411262048
169528|169528-b19f-3f77-b713-d32bba55d77f|pass|200411|nothingmuch@woobling.org|Number-Phone|1.3002|i686-linux-thread-multi|5.8.5|linux|2.6.8.1|200411262048
169559|169559-b19f-3f77-b713-d32bba55d77f|pass|200411|corion-cpansmoker@corion.net|Number-Phone|1.3002|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200411270047
169740|169740-b19f-3f77-b713-d32bba55d77f|pass|200411|saper@cpan.org|Number-Phone|1.3002|i386-linux|5.8.1|linux|2.4.18-23mdksmp|200411271546
169753|169753-b19f-3f77-b713-d32bba55d77f|pass|200411|saper@cpan.org|Number-Phone|1.3002|darwin|5.6.0|darwin|6.0|200411271602
172027|172027-b19f-3f77-b713-d32bba55d77f|pass|200412|Jost.Krieger+cpanplus@rub.de|Number-Phone|1.3003|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200412060512
172039|172039-b19f-3f77-b713-d32bba55d77f|fail|200412|nothingmuch@woobling.org|Number-Phone|1.3003|i686-linux|5.5.4|linux|2.6.8.1|200412061054
172045|172045-b19f-3f77-b713-d32bba55d77f|pass|200412|nothingmuch@woobling.org|Number-Phone|1.3003|i686-linux|5.6.2|linux|2.6.8.1|200412061106
172055|172055-b19f-3f77-b713-d32bba55d77f|pass|200412|nothingmuch@woobling.org|Number-Phone|1.3003|i686-linux|5.9.2 patch 23609|linux|2.6.8.1|200412061115
172072|172072-b19f-3f77-b713-d32bba55d77f|pass|200412|nothingmuch@woobling.org|Number-Phone|1.3003|i686-linux-thread-multi|5.9.2 patch 23609|linux|2.6.8.1|200412061138
172080|172080-b19f-3f77-b713-d32bba55d77f|pass|200412|nothingmuch@woobling.org|Number-Phone|1.3003|i686-linux|5.8.6|linux|2.6.8.1|200412061149
172095|172095-b19f-3f77-b713-d32bba55d77f|pass|200412|nothingmuch@woobling.org|Number-Phone|1.3003|i686-linux-thread-multi|5.8.6|linux|2.6.8.1|200412061203
172107|172107-b19f-3f77-b713-d32bba55d77f|pass|200412|saper@cpan.org|Number-Phone|1.3003|i386-linux|5.8.1|linux|2.4.18-23mdksmp|200412061110
172118|172118-b19f-3f77-b713-d32bba55d77f|pass|200412|nothingmuch@woobling.org|Number-Phone|1.3003|i386-linux-thread|5.8.4|linux|2.6.7-gentoo-r11|200412061222
172178|172178-b19f-3f77-b713-d32bba55d77f|pass|200412|corion-cpansmoker@corion.net|Number-Phone|1.3003|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200412061934
172666|172666-b19f-3f77-b713-d32bba55d77f|pass|200412|foo@bar.com|Acme-Pony|1.1.2|darwin-thread-multi-2level|5.8.1 RC3|darwin|7.0|200412081422
172735|172735-b19f-3f77-b713-d32bba55d77f|fail|200412|nothingmuch@woobling.org|Data-Compare|0.13|i686-linux|5.6.2|linux|2.6.8.1|200412090318
172774|172774-b19f-3f77-b713-d32bba55d77f|pass|200412|nothingmuch@woobling.org|Data-Compare|0.13|i686-linux|5.8.6|linux|2.6.8.1|200412090439
172783|172783-b19f-3f77-b713-d32bba55d77f|pass|200412|nothingmuch@woobling.org|Data-Compare|0.13|i686-linux-thread-multi|5.8.6|linux|2.6.8.1|200412090454
174147|174147-b19f-3f77-b713-d32bba55d77f|pass|200412|nothingmuch@woobling.org|Sub-WrapPackages|1.0|i686-linux|5.6.2|linux|2.6.8.1|200412151759
174466|174466-b19f-3f77-b713-d32bba55d77f|pass|200412|nothingmuch@woobling.org|Sub-WrapPackages|1.0|i686-linux|5.9.2 patch 23656|linux|2.6.8.1|200412170002
174476|174476-b19f-3f77-b713-d32bba55d77f|pass|200412|nothingmuch@woobling.org|Sub-WrapPackages|1.0|i686-linux-thread-multi|5.9.2 patch 23656|linux|2.6.8.1|200412170021
174501|174501-b19f-3f77-b713-d32bba55d77f|pass|200412|nothingmuch@woobling.org|Sub-WrapPackages|1.0|i686-linux-thread-multi|5.8.6|linux|2.6.8.1|200412170049
174828|174828-b19f-3f77-b713-d32bba55d77f|pass|200412|saper@cpan.org|Sub-WrapPackages|1.0|darwin|5.6.0|darwin|6.0|200412190038
175191|175191-b19f-3f77-b713-d32bba55d77f|pass|200412|nothingmuch@woobling.org|Class-CanBeA|1.1|i686-linux|5.6.2|linux|2.6.8.1|200412211016
175207|175207-b19f-3f77-b713-d32bba55d77f|pass|200412|nothingmuch@woobling.org|Class-CanBeA|1.1|i686-linux|5.9.2 patch 23662|linux|2.6.8.1|200412211029
175225|175225-b19f-3f77-b713-d32bba55d77f|pass|200412|nothingmuch@woobling.org|Class-CanBeA|1.1|i686-linux-thread-multi|5.9.2 patch 23662|linux|2.6.8.1|200412211043
175238|175238-b19f-3f77-b713-d32bba55d77f|pass|200412|nothingmuch@woobling.org|Class-CanBeA|1.1|i386-linux-thread|5.8.4|linux|2.6.7-gentoo-r11|200412211101
175264|175264-b19f-3f77-b713-d32bba55d77f|pass|200412|nothingmuch@woobling.org|Class-CanBeA|1.1|i686-linux|5.8.6|linux|2.6.8.1|200412211123
175282|175282-b19f-3f77-b713-d32bba55d77f|pass|200412|nothingmuch@woobling.org|Class-CanBeA|1.1|i686-linux-thread-multi|5.8.6|linux|2.6.8.1|200412211139
175392|175392-b19f-3f77-b713-d32bba55d77f|pass|200412|rrwo@cpan.org|Class-CanBeA|1.1|MSWin32-x86-multi-thread|5.8.4|MSWin32|4.0|200412211514
175604|175604-b19f-3f77-b713-d32bba55d77f|pass|200412|Jost.Krieger@ruhr-uni-bochum.de|Class-CanBeA|1.1|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200412221554
175780|175780-b19f-3f77-b713-d32bba55d77f|pass|200412|saper@cpan.org|Class-CanBeA|1.1|i386-linux|5.8.1|linux|2.4.18-23mdksmp|200412230008
176594|176594-b19f-3f77-b713-d32bba55d77f|pass|200501|rrwo@cpan.org|Data-Compare|0.13|MSWin32-x86-multi-thread|5.8.4|MSWin32|4.0|200501011906
176599|176599-b19f-3f77-b713-d32bba55d77f|pass|200501|rrwo@cpan.org|Data-Compare|0.13|MSWin32-x86-multi-thread|5.8.4|MSWin32|4.0|200501011908
177521|177521-b19f-3f77-b713-d32bba55d77f|pass|200501|cpantester@thr.msu.edu|Number-Phone|1.3003|i386-freebsd-64int|5.8.5|freebsd|5.2-current|200501080223
177682|177682-b19f-3f77-b713-d32bba55d77f|pass|200501|cpantester@thr.msu.edu|Sub-WrapPackages|1.0|i386-freebsd-64int|5.8.5|freebsd|5.2-current|200501080846
177830|177830-b19f-3f77-b713-d32bba55d77f|pass|200501|cpantester@thr.msu.edu|Class-CanBeA|1.1|i386-freebsd-64int|5.8.5|freebsd|5.2-current|200501090029
178996|178996-b19f-3f77-b713-d32bba55d77f|pass|200501|saper@cpan.org|Data-Compare|0.13|i386-linux|5.8.5|linux|2.4.19-44mdkenterprise|200501162222
179170|179170-b19f-3f77-b713-d32bba55d77f|pass|200501|root@fisharerojo.org ((Charlie &))|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-openbsd|5.8.5|openbsd|3.6|200501180850
179209|179209-b19f-3f77-b713-d32bba55d77f|pass|200501|shlomif@iglu.org.il (Shlomi Fish)|Sub-WrapPackages|1.0|i386-linux|5.8.5|linux|2.4.19-44mdkenterprise|200501181930
179679|179679-b19f-3f77-b713-d32bba55d77f|pass|200501|rrwo@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|MSWin32-x86-multi-thread|5.8.6|MSWin32|4.0|200501201640
180545|180545-b19f-3f77-b713-d32bba55d77f|pass|200501|saperghis@aacom.fr|Data-Compare|0.13|i386-linux|5.8.3|linux|2.6.0-1mdkenterprise|200501241408
180718|180718-b19f-3f77-b713-d32bba55d77f|pass|200501|root@fisharerojo.org ((Charlie &))|Bryar|2.6|i386-openbsd|5.8.5|openbsd|3.6|200501250751
181235|181235-b19f-3f77-b713-d32bba55d77f|fail|200501|corion-cpansmoker@corion.net|Palm-SMS|0.02|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200501290957
181709|181709-b19f-3f77-b713-d32bba55d77f|pass|200502|cpansmoke@alternation.net|Sub-WrapPackages|1.0|i586-linux|5.8.3|linux|2.4.21-13trfirewallsmp|200502020612
181897|181897-b19f-3f77-b713-d32bba55d77f|pass|200502|saper@cpan.org|Sub-WrapPackages|1.0|i386-linux|5.8.5|linux|2.4.19-44mdkenterprise|200502022306
182231|182231-b19f-3f77-b713-d32bba55d77f|pass|200502|cpantester@thr.msu.edu|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-freebsd-64int|5.8.6|freebsd|5.3-stable|200502031858
185115|185115-b19f-3f77-b713-d32bba55d77f|pass|200502|rrwo@cpan.org|Data-Compare|0.13|cygwin-thread-multi-64int|5.8.6|cygwin|1.5.12(0.11642)|200502182335
189095|189095-b19f-3f77-b713-d32bba55d77f|pass|200503|saper@cpan.org|Data-Hexdumper|1.2|i386-linux|5.8.5|linux|2.4.19-44mdkenterprise|200503100108
196329|196329-b19f-3f77-b713-d32bba55d77f|pass|200504|leeym@utopia.leeym.com ((Yen-Ming Lee))|Data-Compare|0.13|i386-freebsd-64int|5.8.6|freebsd|5.4-prerelease|200504032333
196772|196772-b19f-3f77-b713-d32bba55d77f|pass|200504|cpan@calaquendi.net|Data-Compare|0.13|i386-freebsd-64int|5.8.6|freebsd|4.10-release-p3|200504042125
197527|197527-b19f-3f77-b713-d32bba55d77f|pass|200504|leeym@utopia.leeym.com ((Yen-Ming Lee))|Number-Phone|1.3003|i386-freebsd-64int|5.8.6|freebsd|5.4-prerelease|200504081244
197600|197600-b19f-3f77-b713-d32bba55d77f|pass|200504|rrwo@cpan.org|Number-Phone|1.3003|cygwin-thread-multi-64int|5.8.6|cygwin|1.5.12(0.11642)|200504081254
197676|197676-b19f-3f77-b713-d32bba55d77f|pass|200504|saper@cpan.org|Number-Phone|1.3003|i386-linux|5.8.5|linux|2.4.19-44mdkenterprise|200504082105
197683|197683-b19f-3f77-b713-d32bba55d77f|pass|200504|leeym@utopia.leeym.com ((Yen-Ming Lee))|Number-Phone|1.3003|i386-freebsd-64int|5.8.6|freebsd|5.4-prerelease|200504090406
198129|198129-b19f-3f77-b713-d32bba55d77f|pass|200504|saper@cpan.org|Number-Phone|1.3003|sun4-solaris|5.8.5|solaris|2.9|200504111111
198209|198209-b19f-3f77-b713-d32bba55d77f|pass|200504|corion-cpanplus@corion.net|Number-Phone|1.3003|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200504111758
198491|198491-b19f-3f77-b713-d32bba55d77f|pass|200504|leeym@utopia.leeym.com ((Yen-Ming Lee))|Number-Phone|1.3004|i386-freebsd-64int|5.8.6|freebsd|5.4-prerelease|200504131128
198591|198591-b19f-3f77-b713-d32bba55d77f|pass|200504|corion-cpanplus@corion.net|Number-Phone|1.3004|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200504132021
208372|208372-b19f-3f77-b713-d32bba55d77f|pass|200505|saper@cpan.org|Number-Phone|1.4|sun4-solaris|5.8.5|solaris|2.9|200505191028
208402|208402-b19f-3f77-b713-d32bba55d77f|pass|200505|imacat@mail.imacat.idv.tw|Number-Phone|1.4|cygwin-thread-multi-64int|5.8.6|cygwin|1.5.12(0.11642)|200505192028
208414|208414-b19f-3f77-b713-d32bba55d77f|pass|200505|imacat@mail.imacat.idv.tw|Number-Phone|1.4|MSWin32-x86-multi-thread|5.8.6|MSWin32|4.0|200505192055
208419|208419-b19f-3f77-b713-d32bba55d77f|pass|200505|imacat@mail.imacat.idv.tw|Number-Phone|1.4|i686-linux-thread-multi-64int-ld|5.8.6|linux|2.4.30|200505192057
208639|208639-b19f-3f77-b713-d32bba55d77f|pass|200505|Jost.Krieger@ruhr-uni-bochum.de|Number-Phone|1.4|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200505200649
208931|208931-b19f-3f77-b713-d32bba55d77f|pass|200505|corion-cpanplus@corion.net|Number-Phone|1.4|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200505202142
209198|209198-b19f-3f77-b713-d32bba55d77f|pass|200505|saper@cpan.org|Number-Phone|1.4|i386-linux|5.8.5|linux|2.4.19-44mdkenterprise|200505220037
211199|211199-b19f-3f77-b713-d32bba55d77f|pass|200505|root@houseabsolute.urth.org ((root))|Data-Compare|0.13|i386-linux|5.8.4|linux|2.4.27-ti1211|200505310008
211443|211443-b19f-3f77-b713-d32bba55d77f|fail|200506|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Palm-SMS|0.01|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200506020717
214964|214964-b19f-3f77-b713-d32bba55d77f|pass|200506|saper@cpan.org|Number-Phone|1.4|darwin|5.6.0|darwin|6.0|200506120128
219764|219764-b19f-3f77-b713-d32bba55d77f|pass|200506|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.4001|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200506281457
220045|220045-b19f-3f77-b713-d32bba55d77f|pass|200506|imacat@mail.imacat.idv.tw|Number-Phone|1.4001|i686-linux-thread-multi-64int-ld|5.8.7|linux|2.4.30|200506291710
220096|220096-b19f-3f77-b713-d32bba55d77f|pass|200506|saper@cpan.org|Number-Phone|1.4001|sun4-solaris|5.8.5|solaris|2.9|200506291213
220127|220127-b19f-3f77-b713-d32bba55d77f|pass|200506|imacat@mail.imacat.idv.tw|Number-Phone|1.4001|cygwin-thread-multi-64int|5.8.6|cygwin|1.5.12(0.11642)|200506292034
220138|220138-b19f-3f77-b713-d32bba55d77f|pass|200506|imacat@mail.imacat.idv.tw|Number-Phone|1.4001|MSWin32-x86-multi-thread|5.8.7|MSWin32|5.0|200506292041
220280|220280-b19f-3f77-b713-d32bba55d77f|pass|200506|corion-cpanplus@corion.net|Number-Phone|1.4001|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200506300845
220308|220308-b19f-3f77-b713-d32bba55d77f|pass|200506|corion-cpanplus@corion.net|Data-Compare|0.13|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200506300919
220438|220438-b19f-3f77-b713-d32bba55d77f|pass|200506|imacat@mail.imacat.idv.tw|Data-Compare|0.13|i686-linux-thread-multi-64int-ld|5.8.7|linux|2.4.30|200506302053
220485|220485-b19f-3f77-b713-d32bba55d77f|pass|200506|imacat@mail.imacat.idv.tw|Data-Compare|0.13|cygwin-thread-multi-64int|5.8.6|cygwin|1.5.12(0.11642)|200506302147
220553|220553-b19f-3f77-b713-d32bba55d77f|pass|200507|saper@cpan.org|Number-Phone|1.4001|i386-linux|5.8.5|linux|2.4.19-44mdkenterprise|200507010001
220664|220664-b19f-3f77-b713-d32bba55d77f|pass|200507|root@berthoudwireless.net ((root))|Data-Compare|0.13|i686-linux|5.8.7|linux|2.6.12-gentoo-r2|200507010103
223927|223927-b19f-3f77-b713-d32bba55d77f|pass|200507|saper@cpan.org|Data-Compare|0.13|sun4-solaris|5.8.5|solaris|2.9|200507130954
226474|226474-b19f-3f77-b713-d32bba55d77f|pass|200507|saper@cpan.org|Number-Phone|1.4001|darwin|5.6.0|darwin|6.0|200507210008
228104|228104-b19f-3f77-b713-d32bba55d77f|pass|200507|imacat@mail.imacat.idv.tw|Data-Compare|0.13|cygwin-thread-multi-64int|5.8.7|cygwin|1.5.18(0.13242)|200507272149
229324|229324-b19f-3f77-b713-d32bba55d77f|pass|200508|leeym@utopia.leeym.com ((Yen-Ming Lee))|Data-Compare|0.13|i386-freebsd-64int|5.8.7|freebsd|5.4-stable|200508011634
230507|230507-b19f-3f77-b713-d32bba55d77f|pass|200508|imacat@mail.imacat.idv.tw|Number-Phone|1.4002|cygwin-thread-multi-64int|5.8.7|cygwin|1.5.18(0.13242)|200508032142
230641|230641-b19f-3f77-b713-d32bba55d77f|pass|200508|imacat@mail.imacat.idv.tw|Number-Phone|1.4002|i686-linux-thread-multi-64int-ld|5.8.7|linux|2.4.30|200508042018
230643|230643-b19f-3f77-b713-d32bba55d77f|pass|200508|imacat@mail.imacat.idv.tw|Number-Phone|1.4002|MSWin32-x86-multi-thread|5.8.7|MSWin32|5.0|200508042019
231459|231459-b19f-3f77-b713-d32bba55d77f|pass|200508|corion-cpanplus@corion.net|Number-Phone|1.4002|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200508072159
232232|232232-b19f-3f77-b713-d32bba55d77f|pass|200508|saper@cpan.org|Number-Phone|1.4002|sun4-solaris|5.8.5|solaris|2.9|200508091108
233794|233794-b19f-3f77-b713-d32bba55d77f|pass|200508|leeym@utopia.leeym.com ((Yen-Ming Lee))|Data-Hexdumper|1.2|i386-freebsd-64int|5.8.7|freebsd|5.4-stable|200508141701
234470|234470-b19f-3f77-b713-d32bba55d77f|pass|200508|imacat@mail.imacat.idv.tw|Data-Hexdumper|1.2|i686-linux-thread-multi-64int-ld|5.8.7|linux|2.4.30|200508170017
236166|236166-b19f-3f77-b713-d32bba55d77f|pass|200508|gwood@email.mot.com ((George W Wood))|Data-Compare|0.13|i686-linux|5.8.6|linux|2.4.21-20.elsmp|200508220342
236819|236819-b19f-3f77-b713-d32bba55d77f|pass|200508|leeym@utopia.leeym.com ((Yen-Ming Lee))|Number-Phone|1.4003|i386-freebsd-64int|5.8.7|freebsd|5.4-stable|200508241729
236907|236907-b19f-3f77-b713-d32bba55d77f|pass|200508|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.4003|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200508241616
236968|236968-b19f-3f77-b713-d32bba55d77f|pass|200508|saper@cpan.org|Number-Phone|1.4003|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200508242102
237109|237109-b19f-3f77-b713-d32bba55d77f|pass|200508|imacat@mail.imacat.idv.tw|Number-Phone|1.4003|cygwin-thread-multi-64int|5.8.7|cygwin|1.5.18(0.13242)|200508250747
237124|237124-b19f-3f77-b713-d32bba55d77f|pass|200508|imacat@mail.imacat.idv.tw|Number-Phone|1.4003|i686-linux-thread-multi-64int-ld|5.8.7|linux|2.4.30|200508250805
237270|237270-b19f-3f77-b713-d32bba55d77f|pass|200508|rrwo@cpan.org|Number-Phone|1.4003|MSWin32-x86-multi-thread|5.8.7|MSWin32|5.1|200508251439
244310|244310-b19f-3f77-b713-d32bba55d77f|pass|200509|meneldor@metallibrary.ru|Data-Hexdumper|1.2|i386-linux|5.8.3|linux|2.4.21-9.elsmp|200509161951
244808|244808-b19f-3f77-b713-d32bba55d77f|pass|200509|ultradm@cpan.org|Data-Compare|0.13|i686-linux-thread|5.8.6|linux|2.6.11-gentoo-r6|200509172039
245054|245054-b19f-3f77-b713-d32bba55d77f|pass|200509|corion-cpanplus@corion.net|Data-Hexdumper|1.2|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200509172204
245305|245305-b19f-3f77-b713-d32bba55d77f|pass|200509|njh@despammed.com|Data-Hexdumper|1.2|ppc-linux|5.8.3|linux|2.6.6-1.1326smp|200509171902
246450|246450-b19f-3f77-b713-d32bba55d77f|pass|200509|david@landgren.net|Data-Hexdumper|1.2|i386-freebsd|5.8.1|freebsd|5.4-stable|200509191213
247422|247422-b19f-3f77-b713-d32bba55d77f|pass|200509|njh@despammed.com|Data-Compare|0.13|i386-linux|5.8.7|linux|2.4.27-ti1211|200509201402
258137|258137-b19f-3f77-b713-d32bba55d77f|pass|200510|leeym@utopia.leeym.com ((Yen-Ming Lee))|Number-Phone|1.4004|i386-freebsd-64int|5.8.7|freebsd|5.4-stable|200510121351
258161|258161-b19f-3f77-b713-d32bba55d77f|pass|200510|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.4004|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200510120819
258306|258306-b19f-3f77-b713-d32bba55d77f|pass|200510|corion-cpanplus@corion.net|Number-Phone|1.4004|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200510122021
258774|258774-b19f-3f77-b713-d32bba55d77f|pass|200510|njh@despammed.com|Number-Phone|1.4004|i386-linux|5.8.5|linux|2.4.21-25.elsmp|200510131058
258922|258922-b19f-3f77-b713-d32bba55d77f|pass|200510|saper@cpan.org|Number-Phone|1.4004|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200510132040
259968|259968-b19f-3f77-b713-d32bba55d77f|pass|200510|saper@cpan.org|Number-Phone|1.5|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200510161745
260021|260021-b19f-3f77-b713-d32bba55d77f|pass|200510|imacat@mail.imacat.idv.tw|Number-Phone|1.5|i686-linux-thread-multi-64int-ld|5.8.7|linux|2.4.30|200510170414
260029|260029-b19f-3f77-b713-d32bba55d77f|pass|200510|imacat@mail.imacat.idv.tw|Number-Phone|1.5|cygwin-thread-multi-64int|5.8.7|cygwin|1.5.18(0.13242)|200510170421
260051|260051-b19f-3f77-b713-d32bba55d77f|pass|200510|leeym@utopia.leeym.com ((Yen-Ming Lee))|Number-Phone|1.5|i386-freebsd-64int|5.8.7|freebsd|5.4-stable|200510170844
260191|260191-b19f-3f77-b713-d32bba55d77f|pass|200510|imacat@mail.imacat.idv.tw|Number-Phone|1.4004|cygwin-thread-multi-64int|5.8.7|cygwin|1.5.18(0.13242)|200510180825
260587|260587-b19f-3f77-b713-d32bba55d77f|pass|200510|corion-cpanplus@corion.net|Number-Phone|1.5|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200510192027
260898|260898-b19f-3f77-b713-d32bba55d77f|pass|200510|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.5|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200510202001
261368|261368-b19f-3f77-b713-d32bba55d77f|pass|200510|njh@despammed.com|Number-Phone|1.5|i386-linux|5.8.6|linux|2.4.21-27.0.2.elsmp|200510221455
261541|261541-b19f-3f77-b713-d32bba55d77f|pass|200510|njh@despammed.com|Number-Phone|1.5|ppc-linux|5.8.3|linux|2.6.6-1.1326smp|200510221844
262207|262207-b19f-3f77-b713-d32bba55d77f|pass|200510|njh@despammed.com|Number-Phone|1.5|i486-linux-gnu|5.8.7|linux|2.6.14-rc4|200510251548
264405|264405-b19f-3f77-b713-d32bba55d77f|pass|200511|leeym@utopia.leeym.com ((Yen-Ming Lee))|Data-Compare|0.13|i386-freebsd-64int|5.8.7|freebsd|5.4-stable|200511010235
265861|265861-b19f-3f77-b713-d32bba55d77f|pass|200511|njh@despammed.com|Data-Compare|0.13|ppc-linux|5.8.3|linux|2.6.6-1.1326smp|200511051614
267538|267538-b19f-3f77-b713-d32bba55d77f|pass|200511|leeym@utopia.leeym.com ((Yen-Ming Lee))|Data-Compare|0.13|i386-freebsd-64int|5.8.7|freebsd|5.4-stable|200511120225
269770|269770-b19f-3f77-b713-d32bba55d77f|pass|200511|postmaster@amsfep18-int.chello.nl|Data-Compare|0.13|MSWin32-x86-multi-thread|5.8.4|MSWin32|4.0|200511201354
275357|275357-b19f-3f77-b713-d32bba55d77f|pass|200512|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Net-Random|1.2|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200512140828
275393|275393-b19f-3f77-b713-d32bba55d77f|pass|200512|leeym@utopia.leeym.com ((Yen-Ming Lee))|Net-Random|1.2|i386-freebsd-64int|5.8.7|freebsd|6.0-stable|200512141740
275676|275676-b19f-3f77-b713-d32bba55d77f|pass|200512|njh@despammed.com|Net-Random|1.2|i386-linux|5.8.6|linux|2.6.9-22.18.bz155725.elsmp|200512150658
275830|275830-b19f-3f77-b713-d32bba55d77f|pass|200512|corion-cpanplus@corion.net|Net-Random|1.2|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200512151939
279001|279001-b19f-3f77-b713-d32bba55d77f|pass|200512|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Class-CanBeA|1.2|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200512291152
279781|279781-b19f-3f77-b713-d32bba55d77f|pass|200601|corion-cpanplus@corion.net|Class-CanBeA|1.2|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200601011841
281710|281710-b19f-3f77-b713-d32bba55d77f|pass|200601|saper@cpan.org|Class-CanBeA|1.2|i686-linux|5.6.2|linux|2.6.8.1-10mdk|200601110339
281711|281711-b19f-3f77-b713-d32bba55d77f|pass|200601|saper@cpan.org|Class-CanBeA|1.2|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200601110339
286555|286555-b19f-3f77-b713-d32bba55d77f|pass|200601|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|File-Overwrite|1.0|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200601301202
287234|287234-b19f-3f77-b713-d32bba55d77f|pass|200602|leeym@utopia.leeym.com ((Yen-Ming Lee))|File-Overwrite|1.0|i386-freebsd-64int|5.8.7|freebsd|6.0-stable|200602012357
289121|289121-b19f-3f77-b713-d32bba55d77f|pass|200602|gam3-cpanplus@gam3.net|Data-Compare|0.13|i486-linux-gnu|5.8.7|linux|2.6.14-rc4|200602101344
300432|300432-b19f-3f77-b713-d32bba55d77f|pass|200603|leeym@utopia.leeym.com ((Yen-Ming Lee))|Data-Compare|0.13|i386-freebsd-64int|5.8.8|freebsd|6.1-prerelease|200603200436
305984|305984-b19f-3f77-b713-d32bba55d77f|pass|200604|edepagne@184dhcp145.pl.eso.org ((root))|Statistics-ChiSquare|0.5|i386-linux|5.8.7|linux|2.6.3-28mdk-i686-up-4gb|200604080602
308015|308015-b19f-3f77-b713-d32bba55d77f|pass|200604|root@chorny.net|Statistics-ChiSquare|0.5|i686-linux|5.8.8|linux|2.4.20-8smp|200604180521
312967|312967-b19f-3f77-b713-d32bba55d77f|pass|200605|leeym@utopia.leeym.com ((Yen-Ming Lee))|Data-Compare|0.13|i386-freebsd-64int|5.8.8|freebsd|6.1-prerelease|200605092239
314112|314112-b19f-3f77-b713-d32bba55d77f|pass|200605|Jost.Krieger+perl@rub.de|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200605151414
326166|326166-b19f-3f77-b713-d32bba55d77f|pass|200606|root@webigail.pooq.com ((root))|Data-Compare|0.13|i386-linux|5.8.8|linux|2.6.12-12mdksmp|200606301715
326827|326827-b19f-3f77-b713-d32bba55d77f|pass|200607|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Tie-STDOUT|1.01|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200607041255
326894|326894-b19f-3f77-b713-d32bba55d77f|pass|200607|imacat@mail.imacat.idv.tw|Tie-STDOUT|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607050456
326896|326896-b19f-3f77-b713-d32bba55d77f|pass|200607|imacat@mail.imacat.idv.tw|Tie-STDOUT|1.01|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607050458
327028|327028-b19f-3f77-b713-d32bba55d77f|pass|200607|imacat@mail.imacat.idv.tw|Tie-STDOUT|1.02|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607060241
327156|327156-b19f-3f77-b713-d32bba55d77f|pass|200607|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Tie-STDOUT|1.02|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200607061747
330218|330218-b19f-3f77-b713-d32bba55d77f|pass|200607|imacat@mail.imacat.idv.tw|Data-Compare|0.13|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607190322
331278|331278-b19f-3f77-b713-d32bba55d77f|pass|200607|imacat@mail.imacat.idv.tw|Data-Hexdumper|1.2|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607200146
331975|331975-b19f-3f77-b713-d32bba55d77f|pass|200607|imacat@mail.imacat.idv.tw|Number-Phone|1.5|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607202026
332024|332024-b19f-3f77-b713-d32bba55d77f|pass|200607|imacat@mail.imacat.idv.tw|Number-Phone|1.4004|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607201820
333143|333143-b19f-3f77-b713-d32bba55d77f|pass|200607|imacat@mail.imacat.idv.tw|Net-Random|1.2|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607220710
333371|333371-b19f-3f77-b713-d32bba55d77f|pass|200607|imacat@mail.imacat.idv.tw|Class-CanBeA|1.2|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607221801
334134|334134-b19f-3f77-b713-d32bba55d77f|pass|200607|imacat@mail.imacat.idv.tw|File-Overwrite|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607231648
335574|335574-b19f-3f77-b713-d32bba55d77f|pass|200607|imacat@mail.imacat.idv.tw|Data-Compare|0.13|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607251327
336203|336203-b19f-3f77-b713-d32bba55d77f|pass|200607|imacat@mail.imacat.idv.tw|Number-Phone|1.5|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607260008
336692|336692-b19f-3f77-b713-d32bba55d77f|pass|200607|imacat@mail.imacat.idv.tw|Net-Random|1.2|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607261742
336789|336789-b19f-3f77-b713-d32bba55d77f|pass|200607|imacat@mail.imacat.idv.tw|Class-CanBeA|1.2|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607262132
337018|337018-b19f-3f77-b713-d32bba55d77f|pass|200607|imacat@mail.imacat.idv.tw|File-Overwrite|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607270205
337872|337872-b19f-3f77-b713-d32bba55d77f|pass|200607|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Sub-WrapPackages|1.1|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200607271444
338614|338614-b19f-3f77-b713-d32bba55d77f|pass|200607|imacat@mail.imacat.idv.tw|Tie-STDOUT|1.02|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607281708
338982|338982-b19f-3f77-b713-d32bba55d77f|pass|200607|imacat@mail.imacat.idv.tw|Sub-WrapPackages|1.1|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607291323
340236|340236-b19f-3f77-b713-d32bba55d77f|pass|200608|imacat@mail.imacat.idv.tw|Sub-WrapPackages|1.2|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200608030100
340401|340401-b19f-3f77-b713-d32bba55d77f|pass|200608|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Sub-WrapPackages|1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200608031203
340715|340715-b19f-3f77-b713-d32bba55d77f|pass|200608|jworkman@pimpworks.org (Jeff Workman)|Acme-Licence|1.0|i386-netbsd|5.8.3|netbsd|1.6.2|200608031600
341364|341364-b19f-3f77-b713-d32bba55d77f|pass|200608|corion-cpanplus@corion.net|Sub-WrapPackages|1.2|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200608050855
342214|342214-b19f-3f77-b713-d32bba55d77f|pass|200608|cogent@cpan.org|Data-Compare|0.13|darwin-thread-multi-64int-2level|5.8.8|darwin|8.7.0|200608100437
344460|344460-b19f-3f77-b713-d32bba55d77f|pass|200608|imacat@mail.imacat.idv.tw|Number-Phone|1.51|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200608240204
344713|344713-b19f-3f77-b713-d32bba55d77f|pass|200608|imacat@mail.imacat.idv.tw|Number-Phone|1.52|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200608250509
344714|344714-b19f-3f77-b713-d32bba55d77f|pass|200608|imacat@mail.imacat.idv.tw|Number-Phone|1.54|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200608250524
344715|344715-b19f-3f77-b713-d32bba55d77f|fail|200608|imacat@mail.imacat.idv.tw|Number-Phone-UK-DetailedLocations|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200608250524
344716|344716-b19f-3f77-b713-d32bba55d77f|fail|200608|imacat@mail.imacat.idv.tw|Number-Phone|1.53|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200608250527
344828|344828-b19f-3f77-b713-d32bba55d77f|pass|200608|imacat@mail.imacat.idv.tw|Number-Phone|1.55|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200608260505
344829|344829-b19f-3f77-b713-d32bba55d77f|pass|200608|imacat@mail.imacat.idv.tw|Number-Phone-UK-DetailedLocations|1.2|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200608260515
348551|348551-b19f-3f77-b713-d32bba55d77f|pass|200609|imacat@mail.imacat.idv.tw|Games-Dice-Advanced|1.1|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200609140041
348707|348707-b19f-3f77-b713-d32bba55d77f|pass|200609|chris@bingosnet.co.uk|Games-Dice-Advanced|1.1|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200609140529
349313|349313-b19f-3f77-b713-d32bba55d77f|pass|200609|alexchorny@gmail.com (Alexandr Ciornii)|Games-Dice-Advanced|1.1|i686-linux|5.8.8|linux|2.4.20-8smp|200609141102
351783|351783-b19f-3f77-b713-d32bba55d77f|pass|200609|gabor@pti.co.il ("Gabor Szabo")|Data-Compare|0.13|i486-linux-gnu|5.8.7|linux|2.6.10|200609251531
353009|353009-b19f-3f77-b713-d32bba55d77f|pass|200609|mark.gardner@bms.com|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.8.8|linux|2.4.21-37.0.1.el|200609281424
356781|356781-b19f-3f77-b713-d32bba55d77f|pass|200610|chris@bingosnet.co.uk|Data-Compare|0.13|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200610082338
357072|357072-b19f-3f77-b713-d32bba55d77f|pass|200610|chris@bingosnet.co.uk|Data-Compare|0.13|i386-freebsd-64int|5.8.5|freebsd|5.3-release|200610091142
358244|358244-b19f-3f77-b713-d32bba55d77f|pass|200610|chris@bingosnet.co.uk|Data-Compare|0.13|i686-linux-thread-multi-64int-ld|5.8.7|linux|2.4.20-8|200610111335
358966|358966-b19f-3f77-b713-d32bba55d77f|pass|200610|chris@bingosnet.co.uk|Data-Compare|0.13|alpha-freebsd|5.8.7|freebsd|6.0-release|200610121131
359236|359236-b19f-3f77-b713-d32bba55d77f|pass|200610|chris@bingosnet.co.uk|Data-Compare|0.13|cygwin-thread-multi-64int|5.8.7|cygwin|1.5.18(0.13242)|200610121735
361330|361330-b19f-3f77-b713-d32bba55d77f|pass|200610|chris@bingosnet.co.uk|Data-Compare|0.13|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200610181038
365977|365977-b19f-3f77-b713-d32bba55d77f|pass|200610|chris@bingosnet.co.uk|Data-Compare|0.13|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.18(0.13242)|200610301814
366628|366628-b19f-3f77-b713-d32bba55d77f|pass|200610|chris@bingosnet.co.uk|Data-Compare|0.13|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200610311115
367382|367382-b19f-3f77-b713-d32bba55d77f|pass|200611|andreas.koenig.gmwojprw@franz.ak.mind.de ((Andreas J. Koenig))|Data-Compare|0.14|i686-linux-64int|5.9.5|linux|2.6.17-2-k7|200611012205
367439|367439-b19f-3f77-b713-d32bba55d77f|pass|200611|chris@bingosnet.co.uk|Data-Compare|0.14|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200611020324
367783|367783-b19f-3f77-b713-d32bba55d77f|pass|200611|imacat@mail.imacat.idv.tw|Data-Compare|0.14|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200611030258
367810|367810-b19f-3f77-b713-d32bba55d77f|pass|200611|alexchorny@gmail.com|Data-Compare|0.14|MSWin32-x86-multi-thread|5.8.7|MSWin32|5.0|200611020952
367872|367872-b19f-3f77-b713-d32bba55d77f|pass|200611|chris@bingosnet.co.uk|Data-Compare|0.14|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200611022046
368276|368276-b19f-3f77-b713-d32bba55d77f|pass|200611|fayland@gmail.com|Data-Compare|0.14|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200611030209
368354|368354-b19f-3f77-b713-d32bba55d77f|pass|200611|chris@bingosnet.co.uk|Data-Compare|0.14|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.18(0.13242)|200611031157
368655|368655-b19f-3f77-b713-d32bba55d77f|pass|200611|chris@bingosnet.co.uk|Data-Compare|0.14|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200611031750
370342|370342-b19f-3f77-b713-d32bba55d77f|pass|200611|davidp@preshweb.co.uk ("David Precious")|Data-Compare|0.14|i486-linux|5.8.7|linux|2.4.31|200611081605
370768|370768-b19f-3f77-b713-d32bba55d77f|pass|200611|bladebrown@mail.ru|Data-Hexdumper|1.2|cygwin-thread-multi-64int|5.8.7|cygwin|1.5.18(0.13242)|200611110950
370898|370898-b19f-3f77-b713-d32bba55d77f|pass|200611|chris@bingosnet.co.uk|Bryar|2.6|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200611112020
371195|371195-b19f-3f77-b713-d32bba55d77f|pass|200611|chris@bingosnet.co.uk|Bryar|2.6|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200611121102
371918|371918-b19f-3f77-b713-d32bba55d77f|fail|200611|alexchorny@gmail.com|Bryar|2.6|MSWin32-x86-multi-thread|5.8.7|MSWin32|5.0|200611121255
372376|372376-b19f-3f77-b713-d32bba55d77f|pass|200611|chris@bingosnet.co.uk|Bryar|2.6|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.18(0.13242)|200611132232
373000|373000-b19f-3f77-b713-d32bba55d77f|pass|200611|chris@bingosnet.co.uk|Bryar|2.6|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.21(0.15642)|200611141002
373148|373148-b19f-3f77-b713-d32bba55d77f|pass|200611|chris@bingosnet.co.uk|Bryar|2.6|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200611140821
381086|381086-b19f-3f77-b713-d32bba55d77f|pass|200612|gooba42@gmail.com|Games-Dice-Advanced|1.1|cygwin-thread-multi-64int|5.8.7|cygwin|1.5.18(0.13242)|200612010507
382411|382411-b19f-3f77-b713-d32bba55d77f|pass|200612|cogent@cpan.org|Data-Compare|0.14|darwin-thread-multi-64int-2level|5.8.8|darwin|8.7.0|200612030522
383534|383534-b19f-3f77-b713-d32bba55d77f|unknown|200612|srezic@cpan.org|Acme-Pony|1.1.2|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200612052257
385883|385883-b19f-3f77-b713-d32bba55d77f|pass|200612|chris@bingosnet.co.uk|Number-Phone|1.56|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200612130007
386086|386086-b19f-3f77-b713-d32bba55d77f|pass|200612|chris@bingosnet.co.uk|Number-Phone|1.56|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200612132153
386324|386324-b19f-3f77-b713-d32bba55d77f|pass|200612|chris@bingosnet.co.uk|Number-Phone|1.56|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.21(0.15642)|200612141113
386332|386332-b19f-3f77-b713-d32bba55d77f|pass|200612|chris@bingosnet.co.uk|Number-Phone|1.56|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200612141052
386341|386341-b19f-3f77-b713-d32bba55d77f|pass|200612|chris@bingosnet.co.uk|Number-Phone|1.56|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200612141125
386581|386581-b19f-3f77-b713-d32bba55d77f|pass|200612|corion-cpanplus@corion.net|Number-Phone|1.56|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200612142241
387488|387488-b19f-3f77-b713-d32bba55d77f|pass|200612|chris@bingosnet.co.uk|Number-Phone|1.56|sun4-solaris-64int|5.8.8|solaris|2.7|200612161308
388186|388186-b19f-3f77-b713-d32bba55d77f|pass|200612|chris@bingosnet.co.uk|Number-Phone|1.56|alpha-freebsd|5.8.7|freebsd|6.0-release|200612181024
393599|393599-b19f-3f77-b713-d32bba55d77f|pass|200701|saper@cpan.org|Sub-WrapPackages|1.2|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200701021141
393934|393934-b19f-3f77-b713-d32bba55d77f|pass|200701|autarch@urth.org|Data-Compare|0.14|i486-linux-gnu|5.8.8|linux|2.6.15.7|200701030556
394575|394575-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|Data-Compare|0.14|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200701041619
395005|395005-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|Data-Compare|0.14|sun4-solaris-64int|5.8.8|solaris|2.7|200701060943
395212|395212-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|Data-Compare|0.14|alpha-freebsd|5.8.7|freebsd|6.0-release|200701061407
395489|395489-b19f-3f77-b713-d32bba55d77f|pass|200701|njh@despammed.com|Data-Compare|0.14|i486-linux-gnu|5.8.7|linux|2.6.10|200701070017
395731|395731-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|Data-Compare|0.14|i386-pc-solaris2.11|5.8.7|solaris|2.11|200701071231
396008|396008-b19f-3f77-b713-d32bba55d77f|pass|200701|claco@chrislaco.com|Data-Compare|0.14|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200701080049
396170|396170-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|Data-Compare|0.14|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200701081709
396447|396447-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|Data-Compare|0.14|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200701090751
398116|398116-b19f-3f77-b713-d32bba55d77f|pass|200701|saper@cpan.org|Data-Compare|0.14|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200701111152
400138|400138-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|Data-Compare|0.14|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.21(0.15642)|200701161135
401271|401271-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|Data-Compare|0.14|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.23(0.15642)|200701171904
401652|401652-b19f-3f77-b713-d32bba55d77f|pass|200701|james@mastros.biz|Number-Phone|1.56|i486-linux-gnu|5.8.8|linux|2.6.18.3|200701180143
402258|402258-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|Data-Compare|0.14|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200701191454
402576|402576-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|Bryar|3.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200701210243
402577|402577-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|Bryar|3.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200701210247
402614|402614-b19f-3f77-b713-d32bba55d77f|fail|200701|chris@bingosnet.co.uk|Net-Random|1.3|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200701211110
402616|402616-b19f-3f77-b713-d32bba55d77f|fail|200701|chris@bingosnet.co.uk|Net-Random|1.3|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200701211114
402617|402617-b19f-3f77-b713-d32bba55d77f|fail|200701|chris@bingosnet.co.uk|Net-Random|1.3|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200701211115
402745|402745-b19f-3f77-b713-d32bba55d77f|fail|200701|chris@bingosnet.co.uk|Bryar|3.0|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.23(0.15642)|200701212346
402746|402746-b19f-3f77-b713-d32bba55d77f|fail|200701|chris@bingosnet.co.uk|Net-Random|1.3|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.23(0.15642)|200701212348
402878|402878-b19f-3f77-b713-d32bba55d77f|fail|200701|saper@cpan.org|Net-Random|1.3|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200701220934
403051|403051-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|Net-Random|1.3|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200701221027
403054|403054-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|Data-Compare|0.14|alpha-freebsd|5.8.7|freebsd|6.0-release|200701221135
403141|403141-b19f-3f77-b713-d32bba55d77f|pass|200701|imacat@mail.imacat.idv.tw|Net-Random|1.3|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200701222020
403217|403217-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|Net-Random|1.3|alpha-freebsd|5.8.7|freebsd|6.0-release|200701221247
403247|403247-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|Data-Compare|0.14|i686-linux-thread-multi-64int-ld|5.9.4|linux|2.4.26|200701221245
403469|403469-b19f-3f77-b713-d32bba55d77f|fail|200701|chris@bingosnet.co.uk|Net-Random|1.3|i686-linux-thread-multi-64int-ld|5.9.4|linux|2.4.26|200701221428
403807|403807-b19f-3f77-b713-d32bba55d77f|fail|200701|corion-cpanplus@corion.net|Bryar|3.0|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200701221903
403808|403808-b19f-3f77-b713-d32bba55d77f|fail|200701|corion-cpanplus@corion.net|Net-Random|1.3|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200701221903
404101|404101-b19f-3f77-b713-d32bba55d77f|fail|200701|chris@bingosnet.co.uk|Net-Random|1.3|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200701231253
404652|404652-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|Bryar|3.0|i386-openbsd|5.8.8|openbsd|4.0|200701241445
404654|404654-b19f-3f77-b713-d32bba55d77f|fail|200701|chris@bingosnet.co.uk|Net-Random|1.3|i386-openbsd|5.8.8|openbsd|4.0|200701241445
404727|404727-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|Bryar|3.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200701241624
404728|404728-b19f-3f77-b713-d32bba55d77f|fail|200701|chris@bingosnet.co.uk|Net-Random|1.3|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200701241624
404938|404938-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|Bryar|3.0|i386-pc-solaris2.11|5.8.7|solaris|2.11|200701242140
404939|404939-b19f-3f77-b713-d32bba55d77f|fail|200701|chris@bingosnet.co.uk|Net-Random|1.3|i386-pc-solaris2.11|5.8.7|solaris|2.11|200701242143
405024|405024-b19f-3f77-b713-d32bba55d77f|fail|200701|bricas@cpan.org|Net-Random|1.3|i486-linux-gnu|5.8.8|linux|2.6.15.7|200701241254
405090|405090-b19f-3f77-b713-d32bba55d77f|fail|200701|bricas@cpan.org|Bryar|3.0|i486-linux-gnu|5.8.8|linux|2.6.15.7|200701241253
406240|406240-b19f-3f77-b713-d32bba55d77f|pass|200701|corion-cpanplus@corion.net|XML-Tiny|1.0|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200701271603
406297|406297-b19f-3f77-b713-d32bba55d77f|pass|200701|imacat@mail.imacat.idv.tw|XML-Tiny|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200701280232
406369|406369-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200701281727
406372|406372-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200701281730
406388|406388-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200701281814
406449|406449-b19f-3f77-b713-d32bba55d77f|pass|200701|saper@cpan.org|Number-Phone|1.56|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200701282030
406568|406568-b19f-3f77-b713-d32bba55d77f|pass|200701|saper@cpan.org|XML-Tiny|1.0|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200701291002
406701|406701-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200701291118
406775|406775-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.0|i686-linux-thread-multi-64int-ld|5.9.4|linux|2.4.26|200701291146
407142|407142-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.0|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.23(0.15642)|200701291804
407210|407210-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200701291938
407352|407352-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.0|i386-pc-solaris2.11|5.8.7|solaris|2.11|200701292244
407472|407472-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.01|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200701300837
407503|407503-b19f-3f77-b713-d32bba55d77f|pass|200701|saper@cpan.org|XML-Tiny|1.01|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200701301003
407507|407507-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.01|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200701300910
407521|407521-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.01|i386-pc-solaris2.11|5.8.7|solaris|2.11|200701300936
407580|407580-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.0|alpha-freebsd|5.8.7|freebsd|6.0-release|200701301052
407581|407581-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.01|alpha-freebsd|5.8.7|freebsd|6.0-release|200701301053
407664|407664-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.01|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200701301210
407864|407864-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.01|i686-linux-thread-multi-64int-ld|5.9.4|linux|2.4.26|200701301354
408226|408226-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.02|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200701310829
408246|408246-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.02|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200701310844
408304|408304-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.02|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200701310925
408321|408321-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.0|i386-openbsd|5.8.8|openbsd|4.0|200701310943
408322|408322-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.01|i386-openbsd|5.8.8|openbsd|4.0|200701310944
408323|408323-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.02|i386-openbsd|5.8.8|openbsd|4.0|200701310944
408325|408325-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.02|i686-linux-thread-multi-64int-ld|5.9.4|linux|2.4.26|200701310944
408457|408457-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|Data-Compare|0.14|i386-openbsd|5.8.8|openbsd|4.0|200701311200
408469|408469-b19f-3f77-b713-d32bba55d77f|pass|200701|imacat@mail.imacat.idv.tw|XML-Tiny|1.01|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200701312008
408518|408518-b19f-3f77-b713-d32bba55d77f|pass|200701|saper@cpan.org|XML-Tiny|1.02|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200701311345
408519|408519-b19f-3f77-b713-d32bba55d77f|pass|200701|saper@cpan.org|XML-Tiny|1.02|i686-linux|5.4.4|linux|2.6.8.1-10mdk|200701311346
408662|408662-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200701311611
408663|408663-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.01|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200701311611
408664|408664-b19f-3f77-b713-d32bba55d77f|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.02|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200701311611
408851|408851-b19f-3f77-b713-d32bba55d77f|pass|200701|castaway@desert-island.dynodns.net|Data-Compare|0.14|i686-linux-thread|5.8.7|linux|2.6.15.1|200701311544
409022|409022-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.01|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200702010958
409023|409023-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.02|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200702010958
409131|409131-b19f-3f77-b713-d32bba55d77f|pass|200702|imacat@mail.imacat.idv.tw|XML-Tiny|1.02|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200702012013
409233|409233-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.02|i386-pc-solaris2.11|5.8.7|solaris|2.11|200702011557
409389|409389-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.02|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200702011906
409390|409390-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.01|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200702011906
409399|409399-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.01|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.23(0.15642)|200702012025
409400|409400-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.02|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.23(0.15642)|200702012025
409816|409816-b19f-3f77-b713-d32bba55d77f|pass|200702|jrockway@cpan.org|Data-Compare|0.14|i486-linux-gnu|5.8.8|linux|2.6.18.3|200702021305
409919|409919-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.03|i686-linux-thread-multi-64int-ld|5.9.4|linux|2.4.26|200702030740
409976|409976-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.03|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200702031408
409980|409980-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.03|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200702031502
410113|410113-b19f-3f77-b713-d32bba55d77f|pass|200702|smueller@cpan.org ("Steffen Mueller")|XML-Tiny|1.02|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200702032114
410364|410364-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.03|i386-openbsd|5.8.8|openbsd|4.0|200702041215
410380|410380-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.03|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200702041245
410746|410746-b19f-3f77-b713-d32bba55d77f|pass|200702|saper@cpan.org|XML-Tiny|1.03|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200702051015
410765|410765-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.03|i386-pc-solaris2.11|5.8.7|solaris|2.11|200702050953
410944|410944-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.02|alpha-freebsd|5.8.7|freebsd|6.0-release|200702051437
410945|410945-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.03|alpha-freebsd|5.8.7|freebsd|6.0-release|200702051437
411263|411263-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.03|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200702060952
411314|411314-b19f-3f77-b713-d32bba55d77f|pass|200702|imacat@mail.imacat.idv.tw|XML-Tiny|1.03|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.29|200702062123
411349|411349-b19f-3f77-b713-d32bba55d77f|pass|200702|imacat@mail.imacat.idv.tw|XML-Tiny|1.03|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.29|200702062145
411552|411552-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.03|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.23(0.15642)|200702062002
411568|411568-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.03|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200702062036
411918|411918-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.03|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200702071720
413290|413290-b19f-3f77-b713-d32bba55d77f|pass|200702|david@cantrell.org.uk|Tie-Scalar-Decay|1.1.1|i686-linux|5.8.8|linux|2.4.27-3-686|200702112331
413292|413292-b19f-3f77-b713-d32bba55d77f|pass|200702|david@cantrell.org.uk|Tie-Hash-Rank|1.0.1|i686-linux|5.8.8|linux|2.4.27-3-686|200702112335
413328|413328-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.04|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200702120203
413330|413330-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.04|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200702120204
413341|413341-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.04|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200702120314
413385|413385-b19f-3f77-b713-d32bba55d77f|pass|200702|saper@cpan.org|XML-Tiny|1.04|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200702121120
413428|413428-b19f-3f77-b713-d32bba55d77f|pass|200702|saper@cpan.org|XML-Tiny|1.04|i686-linux|5.4.4|linux|2.6.8.1-10mdk|200702121203
413448|413448-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.04|alpha-freebsd|5.8.7|freebsd|6.0-release|200702121122
413729|413729-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.04|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200702121534
413903|413903-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.04|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.23(0.15642)|200702121813
414124|414124-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|Data-Compare|0.14|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200702122206
414424|414424-b19f-3f77-b713-d32bba55d77f|pass|200702|imacat@mail.imacat.idv.tw|XML-Tiny|1.04|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.29|200702132233
414887|414887-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.04|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200702140850
415005|415005-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.04|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200702140853
415088|415088-b19f-3f77-b713-d32bba55d77f|pass|200702|saper@cpan.org|XML-Tiny|1.04|i686-linux|5.6.2|linux|2.6.8.1-10mdk|200702141401
415282|415282-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.04|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200702142048
415613|415613-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.04|i386-openbsd|5.8.8|openbsd|4.0|200702151026
416151|416151-b19f-3f77-b713-d32bba55d77f|pass|200702|david@cantrell.org.uk|Number-Phone|1.56|i686-linux|5.6.2|linux|2.4.27-3-686|200702160107
416160|416160-b19f-3f77-b713-d32bba55d77f|pass|200702|david@cantrell.org.uk|Number-Phone-UK-DetailedLocations|1.2|i686-linux|5.6.2|linux|2.4.27-3-686|200702160110
416168|416168-b19f-3f77-b713-d32bba55d77f|pass|200702|david@cantrell.org.uk|Number-Phone|1.56|i686-linux|5.6.2|linux|2.4.27-3-686|200702160110
416390|416390-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.04|i386-pc-solaris2.11|5.8.7|solaris|2.11|200702161340
420805|420805-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|Data-Compare|0.14|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200702241548
421144|421144-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|Number-Phone|1.5601|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200702242015
421152|421152-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|Number-Phone|1.5601|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200702242018
421157|421157-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|Number-Phone|1.5601|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200702242023
421523|421523-b19f-3f77-b713-d32bba55d77f|pass|200702|david@cantrell.org.uk|Data-Compare|0.14|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200702251914
421525|421525-b19f-3f77-b713-d32bba55d77f|fail|200702|david@cantrell.org.uk|Data-Compare|0.14|i686-linux|5.6.2|linux|2.4.27-3-686|200702251914
421552|421552-b19f-3f77-b713-d32bba55d77f|pass|200702|david@cantrell.org.uk|Number-Phone|1.5601|i686-linux|5.6.2|linux|2.4.27-3-686|200702251920
421581|421581-b19f-3f77-b713-d32bba55d77f|pass|200702|david@cantrell.org.uk|Number-Phone|1.5601|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200702251927
421634|421634-b19f-3f77-b713-d32bba55d77f|pass|200702|david@cantrell.org.uk|Data-Compare|0.14|i686-linux|5.8.8|linux|2.4.27-3-686|200702251941
421661|421661-b19f-3f77-b713-d32bba55d77f|pass|200702|david@cantrell.org.uk|Number-Phone|1.5601|i686-linux|5.8.8|linux|2.4.27-3-686|200702251947
422037|422037-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|Number-Phone|1.5601|alpha-freebsd|5.8.7|freebsd|6.0-release|200702261104
422104|422104-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|Number-Phone|1.5601|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200702261121
422151|422151-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|Number-Phone|1.5601|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200702261228
422169|422169-b19f-3f77-b713-d32bba55d77f|pass|200702|imacat@mail.imacat.idv.tw|Number-Phone|1.5601|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.29|200702262037
422879|422879-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|Number-Phone|1.5601|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200702270133
423054|423054-b19f-3f77-b713-d32bba55d77f|pass|200702|saper@cpan.org|Data-Compare|0.15|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200702271007
423090|423090-b19f-3f77-b713-d32bba55d77f|pass|200702|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Compare|0.15|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200702270927
423091|423091-b19f-3f77-b713-d32bba55d77f|pass|200702|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.5601|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200702270928
423199|423199-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|Number-Phone|1.5601|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200702270958
423480|423480-b19f-3f77-b713-d32bba55d77f|pass|200702|david@cantrell.org.uk|Data-Compare|0.15|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200702271507
423670|423670-b19f-3f77-b713-d32bba55d77f|pass|200702|david@cantrell.org.uk|Data-Compare|0.15|i686-linux|5.8.8|linux|2.4.27-3-686|200702271636
424064|424064-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.05|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200702280418
424067|424067-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.05|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200702280420
424100|424100-b19f-3f77-b713-d32bba55d77f|pass|200702|corion-cpanplus@corion.net|Data-Compare|0.16|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200702280837
424204|424204-b19f-3f77-b713-d32bba55d77f|pass|200702|Jost.Krieger@ruhr-uni-bochum.de|Data-Compare|0.16|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200702281107
424249|424249-b19f-3f77-b713-d32bba55d77f|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.05|i386-pc-solaris2.11|5.8.7|solaris|2.11|200702281202
424258|424258-b19f-3f77-b713-d32bba55d77f|fail|200702|Jost.Krieger@ruhr-uni-bochum.de|Net-Random|1.3|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200702281235
424590|424590-b19f-3f77-b713-d32bba55d77f|pass|200702|david@cantrell.org.uk|XML-Tiny|1.05|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200702281549
424641|424641-b19f-3f77-b713-d32bba55d77f|pass|200702|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.6.2|linux|2.4.27-3-686|200702281449
424664|424664-b19f-3f77-b713-d32bba55d77f|pass|200702|david@cantrell.org.uk|Data-Compare|0.16|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200702281502
424839|424839-b19f-3f77-b713-d32bba55d77f|pass|200702|david@cantrell.org.uk|XML-Tiny|1.05|i686-linux|5.6.2|linux|2.4.27-3-686|200702281459
425056|425056-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.05|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200703010144
425143|425143-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Net-Random|1.4|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200703010319
425147|425147-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Net-Random|1.4|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200703010323
425457|425457-b19f-3f77-b713-d32bba55d77f|pass|200703|saper@cpan.org|Data-Compare|0.16|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200703011023
425610|425610-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.8.8|linux|2.4.27-3-686|200703011040
425615|425615-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|XML-Tiny|1.05|i686-linux|5.8.8|linux|2.4.27-3-686|200703011042
425718|425718-b19f-3f77-b713-d32bba55d77f|pass|200703|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Compare|0.16|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200703011108
425719|425719-b19f-3f77-b713-d32bba55d77f|pass|200703|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Net-Random|1.4|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200703011109
425723|425723-b19f-3f77-b713-d32bba55d77f|pass|200703|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-Tiny|1.05|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200703011109
425770|425770-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703011115
425784|425784-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.6.2|linux|2.4.27-3-686|200703011121
425799|425799-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Net-Random|1.4|i386-openbsd|5.8.8|openbsd|4.0|200703011122
425819|425819-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.05|i386-openbsd|5.8.8|openbsd|4.0|200703011129
425969|425969-b19f-3f77-b713-d32bba55d77f|pass|200703|imacat@mail.imacat.idv.tw|XML-Tiny|1.05|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.29|200703012155
425979|425979-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Net-Random|1.4|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200703011402
426002|426002-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Net-Random|1.4|i686-linux|5.6.2|linux|2.4.27-3-686|200703011412
426092|426092-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.8.8|linux|2.4.27-3-686|200703011503
426155|426155-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Net-Random|1.4|i686-linux|5.8.8|linux|2.4.27-3-686|200703011528
426308|426308-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Net-Random|1.4|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703011415
426482|426482-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Net-Random|1.4|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200703012329
426483|426483-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.05|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200703012335
426651|426651-b19f-3f77-b713-d32bba55d77f|pass|200703|saper@cpan.org|Net-Random|1.4|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200703021008
426712|426712-b19f-3f77-b713-d32bba55d77f|pass|200703|saper@cpan.org|XML-Tiny|1.05|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200703021050
426868|426868-b19f-3f77-b713-d32bba55d77f|pass|200703|imacat@mail.imacat.idv.tw|Net-Random|1.4|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.29|200703022135
426952|426952-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.6.2|linux|2.4.27-3-686|200703021425
426980|426980-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703021435
427087|427087-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.8.8|linux|2.4.27-3-686|200703021536
427212|427212-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Net-Random|1.4|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200703021558
427213|427213-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.05|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200703021559
427431|427431-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Net-Random|1.4|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200703031108
427432|427432-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.05|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200703031108
428389|428389-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.06|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200703050518
428390|428390-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.06|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200703050520
428391|428391-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.06|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200703050521
428489|428489-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Net-Random|1.4|alpha-freebsd|5.8.7|freebsd|6.0-release|200703051004
428490|428490-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.05|alpha-freebsd|5.8.7|freebsd|6.0-release|200703051005
428491|428491-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.06|alpha-freebsd|5.8.7|freebsd|6.0-release|200703051005
428598|428598-b19f-3f77-b713-d32bba55d77f|pass|200703|srezic@iconmobile.com (Slaven Rezic)|Data-Compare|0.16|i686-linux|5.8.7|linux|2.4.20-18.8|200703051233
428962|428962-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|XML-Tiny|1.06|i686-linux|5.8.8|linux|2.4.27-3-686|200703051539
429094|429094-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|XML-Tiny|1.06|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703051734
429161|429161-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Net-Random|1.4|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200703051912
429162|429162-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.05|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200703051912
429163|429163-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.06|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200703051913
429176|429176-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Net-Random|1.4|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.23(0.15642)|200703051926
429179|429179-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.05|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.23(0.15642)|200703051927
429180|429180-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.06|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.23(0.15642)|200703051928
429695|429695-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.06|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200703061105
429805|429805-b19f-3f77-b713-d32bba55d77f|pass|200703|imacat@mail.imacat.idv.tw|XML-Tiny|1.06|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.29|200703062107
430026|430026-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.06|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200703061525
430283|430283-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.05|i686-linux-thread-multi-64int-ld|5.9.4|linux|2.4.26|200703061713
430284|430284-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.06|i686-linux-thread-multi-64int-ld|5.9.4|linux|2.4.26|200703061714
430700|430700-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.06|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200703070516
431402|431402-b19f-3f77-b713-d32bba55d77f|pass|200703|njh@bandsman.co.uk|Data-Compare|0.16|i386-linux|5.8.8|linux|2.6.9-34.elsmp|200703080651
432496|432496-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.06|i386-pc-solaris2.11|5.8.7|solaris|2.11|200703090229
432921|432921-b19f-3f77-b713-d32bba55d77f|pass|200703|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-Tiny|1.06|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200703091413
433143|433143-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.06|i386-openbsd|5.8.8|openbsd|4.0|200703091651
433802|433802-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.06|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200703101135
436498|436498-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200703130946
436583|436583-b19f-3f77-b713-d32bba55d77f|pass|200703|Jost.Krieger@ruhr-uni-bochum.de|Net-Random|1.4|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200703131155
436826|436826-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|i686-linux|5.6.2|linux|2.4.27-3-686|200703131620
436857|436857-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703131628
436923|436923-b19f-3f77-b713-d32bba55d77f|fail|200703|david@cantrell.org.uk|Number-Phone|1.5601|i686-linux|5.8.8|linux|2.4.27-3-686|200703131515
437019|437019-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.23(0.15642)|200703131809
437354|437354-b19f-3f77-b713-d32bba55d77f|pass|200703|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-DoubleEncodedEntities|1.0|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200703141014
437425|437425-b19f-3f77-b713-d32bba55d77f|pass|200703|imacat@mail.imacat.idv.tw|XML-DoubleEncodedEntities|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.29|200703141958
437858|437858-b19f-3f77-b713-d32bba55d77f|fail|200703|chris@bingosnet.co.uk|Number-Phone|1.5601|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200703150458
437859|437859-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Number-Phone|1.57|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200703150506
437965|437965-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Number-Phone|1.57|i386-openbsd|5.8.8|openbsd|4.0|200703150950
437967|437967-b19f-3f77-b713-d32bba55d77f|pass|200703|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.57|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200703151000
437969|437969-b19f-3f77-b713-d32bba55d77f|pass|200703|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone-UK-DetailedLocations|1.3|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200703151001
437970|437970-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Number-Phone-UK-DetailedLocations|1.3|i386-openbsd|5.8.8|openbsd|4.0|200703150952
437971|437971-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-openbsd|5.8.8|openbsd|4.0|200703150953
438217|438217-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Number-Phone|1.57|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200703151453
438220|438220-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200703151453
438278|438278-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Number-Phone|1.57|i686-linux|5.6.2|linux|2.4.27-3-686|200703151537
438279|438279-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.6.2|linux|2.4.27-3-686|200703151539
438390|438390-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Number-Phone|1.57|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703151638
438391|438391-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Number-Phone-UK-DetailedLocations|1.3|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703151638
438441|438441-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Number-Phone|1.57|i686-linux|5.8.8|linux|2.4.27-3-686|200703151714
438442|438442-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.8.8|linux|2.4.27-3-686|200703151716
438515|438515-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Number-Phone|1.57|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200703151958
438516|438516-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Number-Phone-UK-DetailedLocations|1.3|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200703152000
438527|438527-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Number-Phone|1.57|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703151616
439309|439309-b19f-3f77-b713-d32bba55d77f|pass|200703|imacat@mail.imacat.idv.tw|Number-Phone|1.57|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.29|200703170044
439389|439389-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Number-Phone|1.57|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200703170812
439391|439391-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Number-Phone-UK-DetailedLocations|1.3|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200703170813
439392|439392-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200703170813
439763|439763-b19f-3f77-b713-d32bba55d77f|pass|200703|imacat@mail.imacat.idv.tw|Number-Phone-UK-DetailedLocations|1.3|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.29|200703171627
439973|439973-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.8.8|linux|2.4.27-3-686|200703171908
440130|440130-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703171754
440280|440280-b19f-3f77-b713-d32bba55d77f|fail|200703|corion-cpanplus@corion.net|Number-Phone|1.57|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200703172321
440293|440293-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Number-Phone|1.57|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200703172124
440294|440294-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Number-Phone-UK-DetailedLocations|1.3|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200703172126
440295|440295-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200703172126
440390|440390-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Data-Hexdumper|1.2|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200703181013
440713|440713-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703181558
440867|440867-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.8.8|linux|2.4.27-3-686|200703181921
441193|441193-b19f-3f77-b713-d32bba55d77f|pass|200703|jrockway@cpan.org|Data-Compare|0.16|i486-linux-gnu|5.8.8|linux|2.6.18.3|200703181659
441296|441296-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200703191312
441475|441475-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.8.8|linux|2.4.27-3-686|200703192317
441520|441520-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Data-Hexdumper|1.2|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703192340
441568|441568-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Data-Hexdumper|1.2|i686-linux|5.8.8|linux|2.4.27-3-686|200703200007
441785|441785-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Number-Phone|1.57|alpha-freebsd|5.8.7|freebsd|6.0-release|200703201026
441798|441798-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Number-Phone-UK-DetailedLocations|1.3|alpha-freebsd|5.8.7|freebsd|6.0-release|200703201027
441880|441880-b19f-3f77-b713-d32bba55d77f|pass|200703|imacat@mail.imacat.idv.tw|Data-Hexdumper|1.2|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.29|200703201956
441964|441964-b19f-3f77-b713-d32bba55d77f|pass|200703|davidp@preshweb.co.uk ("David Precious")|Data-Compare|0.16|i486-linux|5.8.7|linux|2.4.31|200703201224
442206|442206-b19f-3f77-b713-d32bba55d77f|pass|200703|corion-cpanplus@corion.net|Number-Phone|1.57|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200703202030
442207|442207-b19f-3f77-b713-d32bba55d77f|pass|200703|corion-cpanplus@corion.net|Number-Phone-UK-DetailedLocations|1.3|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200703202030
442408|442408-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Number-Phone|1.57|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200703210008
442410|442410-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Number-Phone-UK-DetailedLocations|1.3|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200703210010
442474|442474-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200703210214
442479|442479-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200703210218
442482|442482-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200703210224
442587|442587-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Data-Hexdumper|1.2|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200703210706
442624|442624-b19f-3f77-b713-d32bba55d77f|pass|200703|wyant@cpan.org|XML-DoubleEncodedEntities|1.0|darwin-2level|5.8.8|darwin|7.9.0|200703210259
442968|442968-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703211540
442976|442976-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|i686-linux|5.6.2|linux|2.4.27-3-686|200703211618
443027|443027-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|i686-linux|5.8.8|linux|2.4.27-3-686|200703211731
443143|443143-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Number-Phone|1.57|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200703212141
443144|443144-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200703212142
443425|443425-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Data-Hexdumper|1.2|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200703220318
443561|443561-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200703220515
444751|444751-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Data-Hexdumper|1.2|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200703232117
444790|444790-b19f-3f77-b713-d32bba55d77f|pass|200703|dither@gmail.com|Data-Compare|0.16|i386-freebsd-64int|5.8.8|freebsd|5.4-release|200703231735
444941|444941-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Data-Hexdumper|1.2|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200703240104
445126|445126-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Data-Hexdumper|1.2|i386-openbsd|5.8.8|openbsd|4.0|200703240725
445864|445864-b19f-3f77-b713-d32bba55d77f|pass|200703|chris@bingosnet.co.uk|Data-Hexdumper|1.2|i386-pc-solaris2.11|5.8.7|solaris|2.11|200703242336
447207|447207-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Data-Hexdumper|1.2|i686-linux|5.6.2|linux|2.4.27-3-686|200703261737
447526|447526-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Data-Hexdumper|1.2|i686-linux|5.8.8|linux|2.4.27-3-686|200703262159
447776|447776-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|i686-linux|5.6.2|linux|2.4.27-3-686|200703271458
447879|447879-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703271609
449979|449979-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.6.2|linux|2.4.27-3-686|200703301451
450622|450622-b19f-3f77-b713-d32bba55d77f|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703302101
454327|454327-b19f-3f77-b713-d32bba55d77f|pass|200704|chris@bingosnet.co.uk|Acme-Pony|1.1.2|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200704051030
454539|454539-b19f-3f77-b713-d32bba55d77f|pass|200704|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.6.2|linux|2.4.27-3-686|200704051512
454755|454755-b19f-3f77-b713-d32bba55d77f|pass|200704|david@cantrell.org.uk|Data-Compare|0.16|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200704051622
458609|458609-b19f-3f77-b713-d32bba55d77f|pass|200704|tom@eborcom.com ("Tom Hukins")|Sub-WrapPackages|1.2|i686-linux|5.9.5 patch 30836|linux|2.6.17-11-generic|200704121137
458708|458708-b19f-3f77-b713-d32bba55d77f|pass|200704|tom@eborcom.com ("Tom Hukins")|XML-Tiny|1.06|i686-linux|5.9.5 patch 30836|linux|2.6.17-11-generic|200704121420
459149|459149-b19f-3f77-b713-d32bba55d77f|pass|200704|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Net-Random|2.0|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200704130952
459445|459445-b19f-3f77-b713-d32bba55d77f|pass|200704|chris@bingosnet.co.uk|Net-Random|2.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200704131703
459556|459556-b19f-3f77-b713-d32bba55d77f|pass|200704|chris@bingosnet.co.uk|Net-Random|2.0|i386-pc-solaris2.11|5.8.7|solaris|2.11|200704131846
459577|459577-b19f-3f77-b713-d32bba55d77f|pass|200704|chris@bingosnet.co.uk|Net-Random|2.0|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.23(0.15642)|200704131927
460125|460125-b19f-3f77-b713-d32bba55d77f|pass|200704|imacat@mail.imacat.idv.tw|Net-Random|2.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.29|200704142351
460308|460308-b19f-3f77-b713-d32bba55d77f|pass|200704|david@cantrell.org.uk|Net-Random|2.0|i686-linux|5.6.2|linux|2.4.27-3-686|200704142209
460619|460619-b19f-3f77-b713-d32bba55d77f|pass|200704|david@cantrell.org.uk|Net-Random|2.0|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200704151540
461008|461008-b19f-3f77-b713-d32bba55d77f|pass|200704|chris@bingosnet.co.uk|Net-Random|2.0|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200704161042
461251|461251-b19f-3f77-b713-d32bba55d77f|pass|200704|chris@bingosnet.co.uk|Net-Random|2.0|alpha-linux-thread-multi|5.8.8|linux|2.6.16.19|200704161338
462048|462048-b19f-3f77-b713-d32bba55d77f|pass|200704|saper@cpan.org|Net-Random|2.0|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200704171032
462055|462055-b19f-3f77-b713-d32bba55d77f|pass|200704|chris@bingosnet.co.uk|Net-Random|2.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200704170903
462970|462970-b19f-3f77-b713-d32bba55d77f|pass|200704|alexchorny@gmail.com (Alexandr Ciornii)|Data-Compare|0.16|i686-linux|5.8.8|linux|2.4.20-8smp|200704180113
463006|463006-b19f-3f77-b713-d32bba55d77f|pass|200704|chris@bingosnet.co.uk|Net-Random|2.0|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.21(0.15642)|200704181224
463921|463921-b19f-3f77-b713-d32bba55d77f|pass|200704|chris@bingosnet.co.uk|Net-Random|2.0|i386-openbsd|5.8.8|openbsd|4.0|200704190037
465406|465406-b19f-3f77-b713-d32bba55d77f|pass|200704|david@cantrell.org.uk|Net-Random|2.0|i686-linux|5.8.3|linux|2.4.22-physmem-patch3-5um|200704201622
466836|466836-b19f-3f77-b713-d32bba55d77f|pass|200704|david@cantrell.org.uk|File-Overwrite|1.0|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200704231400
468218|468218-b19f-3f77-b713-d32bba55d77f|pass|200704|david@cantrell.org.uk|Class-CanBeA|1.2|i686-linux|5.6.2|linux|2.4.27-3-686|200704251738
468223|468223-b19f-3f77-b713-d32bba55d77f|pass|200704|david@cantrell.org.uk|Class-CanBeA|1.2|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200704251844
474057|474057-b19f-3f77-b713-d32bba55d77f|pass|200705|xdg@hyperbolic.net (DAGOLDEN)|Net-Random|2.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200705040829
475133|475133-b19f-3f77-b713-d32bba55d77f|pass|200705|chaddai.fouche@gmail.com ("Chaddai Fouche")|Data-Compare|0.16|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200705052338
475141|475141-b19f-3f77-b713-d32bba55d77f|pass|200705|chaddai.fouche@gmail.com ("Chaddai Fouche")|Data-Compare|0.16|MSWin32-x86-multi-thread|5.9.5|MSWin32|4.0|200705052344
475387|475387-b19f-3f77-b713-d32bba55d77f|pass|200705|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.16|i686-linux-64int|5.9.5|linux|2.6.18-4-k7|200705061119
475972|475972-b19f-3f77-b713-d32bba55d77f|pass|200705|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.16|i686-linux-64int|5.8.8 patch 30961|linux|2.6.18-4-k7|200705070015
476806|476806-b19f-3f77-b713-d32bba55d77f|fail|200705|bsmith@cpan.org|Bryar|3.0|i486-linux-gnu|5.9.5 patch 31148|linux|2.6.17.13.vtrl22.01|200705061925
476985|476985-b19f-3f77-b713-d32bba55d77f|pass|200705|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.6.2|linux|2.4.27-3-686|200705071450
477045|477045-b19f-3f77-b713-d32bba55d77f|pass|200705|david@cantrell.org.uk|Data-Compare|0.16|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200705071614
478961|478961-b19f-3f77-b713-d32bba55d77f|fail|200705|bsmith@cpan.org|Bryar|3.0|i486-linux-gnu|5.9.5 patch 31148|linux|2.6.17.13.vtrl22.01|200705091231
483520|483520-b19f-3f77-b713-d32bba55d77f|pass|200705|bsmith@cpan.org|Data-Compare|0.16|i486-linux-gnu|5.9.5 patch 31148|linux|2.6.17.13.vtrl22.01|200705131130
484000|484000-b19f-3f77-b713-d32bba55d77f|pass|200705|bsmith@cpan.org|Data-Hexdumper|1.2|i486-linux-gnu|5.9.5 patch 31148|linux|2.6.17.13.vtrl22.01|200705131435
484013|484013-b19f-3f77-b713-d32bba55d77f|fail|200705|bsmith@cpan.org|Data-Transactional|0.1|i486-linux-gnu|5.9.5 patch 31148|linux|2.6.17.13.vtrl22.01|200705131441
498580|498580-b19f-3f77-b713-d32bba55d77f|pass|200705|tom@eborcom.com ("Tom Hukins")|Data-Compare|0.16|i686-linux|5.9.5 patch 30836|linux|2.6.17-11-generic|200705301013
499954|499954-b19f-3f77-b713-d32bba55d77f|pass|200705|stro@cpan.org|Number-Phone-UK-DetailedLocations|1.2|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200705311327
499957|499957-b19f-3f77-b713-d32bba55d77f|pass|200705|stro@cpan.org|Number-Phone-UK-DetailedLocations|1.3|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200705311330
500760|500760-b19f-3f77-b713-d32bba55d77f|pass|200706|chris@bingosnet.co.uk|Number-Phone|1.58|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200706020634
500763|500763-b19f-3f77-b713-d32bba55d77f|pass|200706|chris@bingosnet.co.uk|Number-Phone|1.58|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200706020637
501025|501025-b19f-3f77-b713-d32bba55d77f|pass|200706|david@cantrell.org.uk|Number-Phone|1.58|i686-linux|5.6.2|linux|2.4.27-3-686|200706031142
501028|501028-b19f-3f77-b713-d32bba55d77f|pass|200706|david@cantrell.org.uk|Number-Phone|1.58|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200706031245
501042|501042-b19f-3f77-b713-d32bba55d77f|pass|200706|imacat@mail.imacat.idv.tw|Number-Phone|1.58|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200706031911
501405|501405-b19f-3f77-b713-d32bba55d77f|pass|200706|saper@cpan.org|Number-Phone|1.58|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200706041055
501492|501492-b19f-3f77-b713-d32bba55d77f|pass|200706|stro@cpan.org|Number-Phone|1.58|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200706031924
501643|501643-b19f-3f77-b713-d32bba55d77f|pass|200706|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.58|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200706041402
501841|501841-b19f-3f77-b713-d32bba55d77f|pass|200706|chris@bingosnet.co.uk|Number-Phone|1.58|i386-pc-solaris2.11|5.8.7|solaris|2.11|200706041812
502234|502234-b19f-3f77-b713-d32bba55d77f|pass|200706|chris@bingosnet.co.uk|Number-Phone|1.58|alpha-linux-thread-multi|5.8.8|linux|2.6.16.19|200706050932
502780|502780-b19f-3f77-b713-d32bba55d77f|pass|200706|chris@bingosnet.co.uk|Number-Phone|1.58|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200706060804
503556|503556-b19f-3f77-b713-d32bba55d77f|pass|200706|chris@bingosnet.co.uk|Number-Phone|1.58|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200706061615
503784|503784-b19f-3f77-b713-d32bba55d77f|pass|200706|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.9.5 patch 31335|linux|2.4.27-3-686|200706062047
504111|504111-b19f-3f77-b713-d32bba55d77f|pass|200706|chris@bingosnet.co.uk|Number-Phone|1.58|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200706071044
504704|504704-b19f-3f77-b713-d32bba55d77f|pass|200706|chris@bingosnet.co.uk|Number-Phone|1.58|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200706071858
519559|519559-b19f-3f77-b713-d32bba55d77f|pass|200706|david@cantrell.org.uk|NestedMap|1.0|i686-linux|5.8.8|linux|2.4.27-3-686|200706261547
527714|527714-b19f-3f77-b713-d32bba55d77f|pass|200707|stro@cpan.org|Tie-Hash-Rank|1.0.1|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200707041839
527792|527792-b19f-3f77-b713-d32bba55d77f|pass|200707|stro@cpan.org|Data-Compare|0.16|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200707041757
528095|528095-b19f-3f77-b713-d32bba55d77f|pass|200707|stro@cpan.org|Acme-Pony|1.1.2|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200707041749
532760|532760-b19f-3f77-b713-d32bba55d77f|pass|200707|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.8.8|linux|2.4.27-3-686|200707101103
532768|532768-b19f-3f77-b713-d32bba55d77f|pass|200707|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.9.5 patch 31416|linux|2.4.27-3-686|200707101104
536645|536645-b19f-3f77-b713-d32bba55d77f|pass|200707|david@cantrell.org.uk|Class-CanBeA|1.2|i686-linux|5.9.5|linux|2.4.27-3-686|200707142021
537892|537892-b19f-3f77-b713-d32bba55d77f|pass|200707|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.16|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200707152130
538989|538989-b19f-3f77-b713-d32bba55d77f|pass|200707|overlordq@gmail.com|Data-Compare|0.16|i486-linux-gnu|5.8.8|linux|2.6.18.3|200707161626
541445|541445-b19f-3f77-b713-d32bba55d77f|pass|200707|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|i686-linux|5.6.2|linux|2.4.27-3-686|200707182344
541446|541446-b19f-3f77-b713-d32bba55d77f|pass|200707|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|i686-linux|5.8.8|linux|2.4.27-3-686|200707182344
541447|541447-b19f-3f77-b713-d32bba55d77f|pass|200707|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|i686-linux|5.9.5|linux|2.4.27-3-686|200707182345
541450|541450-b19f-3f77-b713-d32bba55d77f|pass|200707|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200707190052
546402|546402-b19f-3f77-b713-d32bba55d77f|pass|200707|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.16|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200707250634
546525|546525-b19f-3f77-b713-d32bba55d77f|pass|200707|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.06|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200707250749
551041|551041-b19f-3f77-b713-d32bba55d77f|pass|200707|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1|i686-linux|5.9.5|linux|2.4.27-3-686|200707301143
551134|551134-b19f-3f77-b713-d32bba55d77f|pass|200707|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1|i686-linux|5.8.8|linux|2.4.27-3-686|200707301418
551151|551151-b19f-3f77-b713-d32bba55d77f|pass|200707|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200707301537
551157|551157-b19f-3f77-b713-d32bba55d77f|fail|200707|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1|i686-linux|5.6.2|linux|2.4.27-3-686|200707301412
551158|551158-b19f-3f77-b713-d32bba55d77f|pass|200707|david@cantrell.org.uk|Data-Hexdumper|1.2|i686-linux|5.6.2|linux|2.4.27-3-686|200707301408
551428|551428-b19f-3f77-b713-d32bba55d77f|pass|200707|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1.1|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200707310541
551444|551444-b19f-3f77-b713-d32bba55d77f|pass|200707|andreas.koenig.gmwojprw@franz.ak.mind.de|Palm-TreoPhoneCallDB|1.1|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200707310704
551498|551498-b19f-3f77-b713-d32bba55d77f|pass|200707|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Palm-TreoPhoneCallDB|1.1|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200707310826
551733|551733-b19f-3f77-b713-d32bba55d77f|pass|200707|ferreira@cpan.org ("Adriano Ferreira")|Data-Compare|0.16|cygwin|5.9.5|cygwin|1.5.24(0.15642)|200707311148
551763|551763-b19f-3f77-b713-d32bba55d77f|pass|200707|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1.1|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200707311721
551794|551794-b19f-3f77-b713-d32bba55d77f|pass|200707|tom@eborcom.com ("Tom Hukins")|Palm-TreoPhoneCallDB|1.1|i686-linux|5.9.5 patch 31576|linux|2.6.17-11-generic|200707311337
551801|551801-b19f-3f77-b713-d32bba55d77f|pass|200707|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1.1|i686-linux|5.8.8|linux|2.4.27-3-686|200707311725
551813|551813-b19f-3f77-b713-d32bba55d77f|pass|200707|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1.1|i686-linux|5.6.2|linux|2.4.27-3-686|200707311818
551815|551815-b19f-3f77-b713-d32bba55d77f|pass|200707|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1.1|i686-linux|5.9.5|linux|2.4.27-3-686|200707311828
552449|552449-b19f-3f77-b713-d32bba55d77f|pass|200708|imacat@mail.imacat.idv.tw|Palm-TreoPhoneCallDB|1.1|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200708020036
552542|552542-b19f-3f77-b713-d32bba55d77f|pass|200708|stro@cpan.org|Palm-TreoPhoneCallDB|1.1|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200708011914
553871|553871-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200708032142
553873|553873-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200708032144
554693|554693-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1.1|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200708041837
554699|554699-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200708041840
555029|555029-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200708050148
555030|555030-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200708050150
555430|555430-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1.1|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200708051356
555431|555431-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200708051400
555684|555684-b19f-3f77-b713-d32bba55d77f|fail|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.16|i686-linux-64int|5.8.3|linux|2.4.22-rc2-ac2|200708051931
555827|555827-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1.1|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200708052212
555828|555828-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200708052214
556260|556260-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1.1|alpha-linux-thread-multi|5.8.8|linux|2.6.16.19|200708061125
556303|556303-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1.1|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200708061143
558672|558672-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200708072108
558814|558814-b19f-3f77-b713-d32bba55d77f|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200708080159
558828|558828-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|XML-Tiny|1.06|i686-linux|5.8.3|linux|2.4.22-physmem-patch3-5um|200708071720
558868|558868-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200708080137
558881|558881-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200708080141
559090|559090-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200708080854
559174|559174-b19f-3f77-b713-d32bba55d77f|pass|200708|saper@cpan.org|Data-Compare|0.17|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200708081011
559259|559259-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Data-Compare|0.17|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200708081147
559501|559501-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Data-Compare|0.17|i686-linux|5.8.8|linux|2.4.27-3-686|200708081457
559543|559543-b19f-3f77-b713-d32bba55d77f|pass|200708|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Compare|0.17|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200708081416
559576|559576-b19f-3f77-b713-d32bba55d77f|pass|200708|tom@eborcom.com ("Tom Hukins")|Data-Compare|0.17|i686-linux|5.9.5 patch 31576|linux|2.6.17-11-generic|200708080757
559775|559775-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200708081635
559788|559788-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Data-Compare|0.17|i686-linux|5.6.2|linux|2.4.27-3-686|200708081706
559811|559811-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Data-Compare|0.17|i686-linux|5.9.5|linux|2.4.27-3-686|200708081751
559866|559866-b19f-3f77-b713-d32bba55d77f|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-64int|5.9.4|linux|2.6.17-2-k7|200708081955
559868|559868-b19f-3f77-b713-d32bba55d77f|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-64int|5.9.3|linux|2.6.14|200708081956
559872|559872-b19f-3f77-b713-d32bba55d77f|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-64int|5.8.8 patch 30961|linux|2.6.18-4-k7|200708081958
559876|559876-b19f-3f77-b713-d32bba55d77f|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200708081959
559878|559878-b19f-3f77-b713-d32bba55d77f|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200708082000
559880|559880-b19f-3f77-b713-d32bba55d77f|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-thread-multi-64int|5.8.4|linux|2.6.18-4-k7|200708082001
559882|559882-b19f-3f77-b713-d32bba55d77f|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-thread-multi-64int|5.8.3|linux|2.6.18-4-k7|200708082002
559883|559883-b19f-3f77-b713-d32bba55d77f|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200708082003
559905|559905-b19f-3f77-b713-d32bba55d77f|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-64int|5.9.1 patch 21538|linux|2.4.22-rc2-ac2|200708081957
559906|559906-b19f-3f77-b713-d32bba55d77f|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-thread-multi-64int|5.9.2|linux|2.6.18-4-k7|200708081956
559907|559907-b19f-3f77-b713-d32bba55d77f|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200708081958
560131|560131-b19f-3f77-b713-d32bba55d77f|pass|200708|feldt@nhn.ou.edu|Data-Compare|0.17|sun4-solaris-thread-multi|5.8.8|solaris|2.8|200708081419
560273|560273-b19f-3f77-b713-d32bba55d77f|pass|200708|g.grigelionis@computer.org|Data-Compare|0.17|i586-linux-thread-multi|5.8.8|linux|2.6.16|200708081741
561378|561378-b19f-3f77-b713-d32bba55d77f|pass|200708|stro@cpan.org|Data-Compare|0.17|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200708091302
561436|561436-b19f-3f77-b713-d32bba55d77f|pass|200708|imacat@mail.imacat.idv.tw|Data-Compare|0.17|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200708092112
561571|561571-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200708091651
561620|561620-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|File-Find-Rule-Permissions|1.2|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200708091620
561639|561639-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|XML-Tiny|1.06|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200708091626
561654|561654-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|File-Find-Rule-Permissions|1.2|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200708091612
561872|561872-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200708092054
562017|562017-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Data-Compare|0.17|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200708092047
562255|562255-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Tie-Scalar-Decay|1.1.1|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200708092050
563659|563659-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Data-Compare|0.17|darwin-2level|5.9.5|darwin|8.10.1|200708101841
563736|563736-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i386-pc-solaris2.11|5.8.7|solaris|2.11|200708101903
564951|564951-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200708111221
565096|565096-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200708111422
566789|566789-b19f-3f77-b713-d32bba55d77f|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.06|i686-linux-thread-multi-64int|5.8.8 patch 31223|linux|2.6.22-1-k7|200708121117
567459|567459-b19f-3f77-b713-d32bba55d77f|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-thread-multi-64int|5.8.8 patch 31223|linux|2.6.22-1-k7|200708121328
569962|569962-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.9.5 patch 31701|netbsd|3.1|200708131004
570482|570482-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|alpha-linux-thread-multi|5.8.8|linux|2.6.16.19|200708131205
571994|571994-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i386-freebsd-thread-multi-64int|5.9.5|freebsd|6.2-release|200708131852
573512|573512-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|alpha-linux-thread-multi|5.9.5 patch 31711|linux|2.6.16.19|200708141248
574134|574134-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200708141634
577657|577657-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Number-Phone|1.58|darwin-2level|5.9.5|darwin|8.10.1|200708160846
577933|577933-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Number-Phone|1.58|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200708161122
578018|578018-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Number-Phone|1.58|i686-linux|5.8.8|linux|2.4.27-3-686|200708161225
578081|578081-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Number-Phone|1.58|i686-linux|5.9.5|linux|2.4.27-3-686|200708161239
578089|578089-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Number-Phone|1.58|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200708161137
581816|581816-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Number-Phone|1.58|i686-linux-thread-multi-64int|5.9.5|linux|2.6.16.38|200708172240
582029|582029-b19f-3f77-b713-d32bba55d77f|pass|200708|tony@develop-help.com|Number-Phone|1.58|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200708181009
582646|582646-b19f-3f77-b713-d32bba55d77f|fail|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200708190002
582647|582647-b19f-3f77-b713-d32bba55d77f|fail|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200708190003
582649|582649-b19f-3f77-b713-d32bba55d77f|fail|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200708190004
582650|582650-b19f-3f77-b713-d32bba55d77f|fail|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200708190005
582652|582652-b19f-3f77-b713-d32bba55d77f|fail|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200708190006
582653|582653-b19f-3f77-b713-d32bba55d77f|fail|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200708190007
582654|582654-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200708190009
582655|582655-b19f-3f77-b713-d32bba55d77f|fail|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200708190010
583002|583002-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Number-Phone|1.58|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200708190601
583066|583066-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200708190837
583126|583126-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200708191235
583431|583431-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Number-Phone|1.58|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200708191551
583694|583694-b19f-3f77-b713-d32bba55d77f|fail|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200708191732
583699|583699-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200708191734
583850|583850-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200708191937
584790|584790-b19f-3f77-b713-d32bba55d77f|pass|200708|imacat@mail.imacat.idv.tw|CPAN-FindDependencies|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200708201327
584903|584903-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200708200824
584909|584909-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200708200827
584915|584915-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200708200829
584924|584924-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200708200832
584932|584932-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200708200834
584938|584938-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200708200837
584941|584941-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200708200841
584952|584952-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Number-Phone|1.58|i386-freebsd-thread-multi-64int|5.9.5|freebsd|6.2-release|200708200342
585019|585019-b19f-3f77-b713-d32bba55d77f|pass|200708|saper@cpan.org|CPAN-FindDependencies|1.02|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200708201042
585127|585127-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i386-freebsd-thread-multi-64int|5.9.5|freebsd|6.2-release|200708200543
585132|585132-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i386-freebsd-thread-multi-64int|5.9.5|freebsd|6.2-release|200708200545
585137|585137-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i386-freebsd-thread-multi-64int|5.9.5|freebsd|6.2-release|200708200547
585170|585170-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Number-Phone|1.58|alpha-linux-thread-multi|5.8.8|linux|2.6.16.19|200708201137
585345|585345-b19f-3f77-b713-d32bba55d77f|pass|200708|imacat@mail.imacat.idv.tw|CPAN-FindDependencies|1.01|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200708201950
585388|585388-b19f-3f77-b713-d32bba55d77f|fail|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|alpha-linux-thread-multi|5.8.8|linux|2.6.16.19|200708201318
585397|585397-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|alpha-linux-thread-multi|5.8.8|linux|2.6.16.19|200708201321
585398|585398-b19f-3f77-b713-d32bba55d77f|pass|200708|imacat@mail.imacat.idv.tw|CPAN-FindDependencies|1.02|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200708202021
585401|585401-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|alpha-linux-thread-multi|5.8.8|linux|2.6.16.19|200708201324
585417|585417-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200708201331
585420|585420-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200708201334
585423|585423-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200708201338
585425|585425-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200708201341
585428|585428-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200708201344
585430|585430-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200708201348
585431|585431-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200708201350
585436|585436-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200708201353
585809|585809-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200708201719
585816|585816-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200708201722
585822|585822-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200708201724
585836|585836-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200708201727
585840|585840-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200708201729
585844|585844-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200708201731
585847|585847-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200708201733
585854|585854-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200708201736
586798|586798-b19f-3f77-b713-d32bba55d77f|fail|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200708210028
586800|586800-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200708210031
586802|586802-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200708210034
587051|587051-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200708210635
587052|587052-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200708210635
587053|587053-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200708210636
587054|587054-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200708210636
587055|587055-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200708210636
587056|587056-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200708210636
587057|587057-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200708210636
587058|587058-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200708210637
587059|587059-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200708210637
587240|587240-b19f-3f77-b713-d32bba55d77f|pass|200708|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPAN-FindDependencies|1.02|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200708210843
587241|587241-b19f-3f77-b713-d32bba55d77f|pass|200708|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Hexdumper|1.3|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200708210844
587255|587255-b19f-3f77-b713-d32bba55d77f|pass|200708|saper@cpan.org|Data-Hexdumper|1.3|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200708211047
587301|587301-b19f-3f77-b713-d32bba55d77f|pass|200708|plu@pqpq.de|Data-Compare|0.17|i386-freebsd|5.8.8|freebsd|6.1-release-p6|200708211044
587322|587322-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|alpha-linux-thread-multi|5.9.5 patch 31739|linux|2.6.16.19|200708211012
587418|587418-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Data-Compare|0.17|i386-freebsd|5.9.5|freebsd|6.2-release|200708211028
587508|587508-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Number-Phone|1.58|alpha-linux-thread-multi|5.9.5 patch 31739|linux|2.6.16.19|200708211103
587652|587652-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|alpha-linux-thread-multi|5.9.5 patch 31739|linux|2.6.16.19|200708211244
587655|587655-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|alpha-linux-thread-multi|5.9.5 patch 31739|linux|2.6.16.19|200708211245
587656|587656-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|alpha-linux-thread-multi|5.9.5 patch 31739|linux|2.6.16.19|200708211247
587657|587657-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|alpha-linux-thread-multi|5.9.5 patch 31739|linux|2.6.16.19|200708211247
587803|587803-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Data-Hexdumper|1.3|i686-linux|5.8.8|linux|2.4.27-3-686|200708211459
587851|587851-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Data-Hexdumper|1.3|i686-linux|5.6.2|linux|2.4.27-3-686|200708211525
587855|587855-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Data-Hexdumper|1.3|i686-linux|5.9.5|linux|2.4.27-3-686|200708211529
587856|587856-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Number-Phone|1.58|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200708211534
587942|587942-b19f-3f77-b713-d32bba55d77f|pass|200708|plu@pqpq.de|Number-Phone|1.58|i386-freebsd|5.8.8|freebsd|6.1-release-p6|200708211120
588007|588007-b19f-3f77-b713-d32bba55d77f|pass|200708|plu@pqpq.de|Data-Compare|0.17|i386-freebsd|5.8.8|freebsd|6.1-release-p6|200708211358
588093|588093-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Data-Hexdumper|1.3|darwin-2level|5.9.5|darwin|8.10.1|200708211615
588147|588147-b19f-3f77-b713-d32bba55d77f|pass|200708|plu@pqpq.de|CPAN-FindDependencies|1.01|i386-freebsd|5.8.8|freebsd|6.1-release-p6|200708211533
588180|588180-b19f-3f77-b713-d32bba55d77f|pass|200708|plu@pqpq.de|Number-Phone|1.58|i386-freebsd|5.8.8|freebsd|6.1-release-p6|200708211424
588194|588194-b19f-3f77-b713-d32bba55d77f|pass|200708|tom@eborcom.com ("Tom Hukins")|Data-Hexdumper|1.3|i686-linux|5.9.5 patch 31576|linux|2.6.17-11-generic|200708211135
588231|588231-b19f-3f77-b713-d32bba55d77f|fail|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200708211653
588253|588253-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200708211656
588256|588256-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200708211658
588258|588258-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200708211658
588287|588287-b19f-3f77-b713-d32bba55d77f|pass|200708|plu@pqpq.de|CPAN-FindDependencies|1.02|i386-freebsd|5.8.8|freebsd|6.1-release-p6|200708211533
588363|588363-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Data-Hexdumper|1.3|i386-freebsd|5.9.5|freebsd|6.2-release|200708211830
588480|588480-b19f-3f77-b713-d32bba55d77f|fail|200708|plu@pqpq.de|CPAN-FindDependencies|1.0|i386-freebsd|5.8.8|freebsd|6.1-release-p6|200708211532
588481|588481-b19f-3f77-b713-d32bba55d77f|pass|200708|plu@pqpq.de|Data-Hexdumper|1.3|i386-freebsd|5.8.8|freebsd|6.1-release-p6|200708211534
588709|588709-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux|5.6.2|linux|2.4.27-3-686|200708212106
588724|588724-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Bryar|3.0|i686-linux|5.6.2|linux|2.4.27-3-686|200708212109
588730|588730-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Data-Hexdumper|1.0.1|i686-linux|5.6.2|linux|2.4.27-3-686|200708212110
588735|588735-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|File-Overwrite|1.0|i686-linux|5.6.2|linux|2.4.27-3-686|200708212110
588736|588736-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Data-Transactional|0.1|i686-linux|5.6.2|linux|2.4.27-3-686|200708212110
588738|588738-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|File-Find-Rule-Permissions|1.2|i686-linux|5.6.2|linux|2.4.27-3-686|200708212110
588739|588739-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Games-Dice-Advanced|1.1|i686-linux|5.6.2|linux|2.4.27-3-686|200708212111
588740|588740-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|NestedMap|1.0|i686-linux|5.6.2|linux|2.4.27-3-686|200708212111
588744|588744-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Net-Random|2.0|i686-linux|5.6.2|linux|2.4.27-3-686|200708212113
588751|588751-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.6.2|linux|2.4.27-3-686|200708212116
588764|588764-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Statistics-ChiSquare|0.5|i686-linux|5.6.2|linux|2.4.27-3-686|200708212117
588765|588765-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Sub-WrapPackages|1.2|i686-linux|5.6.2|linux|2.4.27-3-686|200708212118
588766|588766-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Statistics-SerialCorrelation|1.1|i686-linux|5.6.2|linux|2.4.27-3-686|200708212117
588767|588767-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Tie-Hash-Longest|1.1|i686-linux|5.6.2|linux|2.4.27-3-686|200708212118
588768|588768-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Tie-Hash-Rank|1.0.1|i686-linux|5.6.2|linux|2.4.27-3-686|200708212118
588819|588819-b19f-3f77-b713-d32bba55d77f|fail|200708|david@cantrell.org.uk|Tie-STDOUT|1.02|i686-linux|5.6.2|linux|2.4.27-3-686|200708212145
588820|588820-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Tie-Scalar-Decay|1.1.1|i686-linux|5.6.2|linux|2.4.27-3-686|200708212146
588822|588822-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Tie-Scalar-Decay|1.1|i686-linux|5.6.2|linux|2.4.27-3-686|200708212146
588824|588824-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|XML-Tiny|1.06|i686-linux|5.6.2|linux|2.4.27-3-686|200708212146
588886|588886-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Number-Phone|1.58|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200708212205
589058|589058-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200708212319
589059|589059-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200708212319
589061|589061-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200708212320
589063|589063-b19f-3f77-b713-d32bba55d77f|unknown|200708|david@cantrell.org.uk|Acme-Licence|1.0|darwin-2level|5.9.5|darwin|8.10.1|200708212318
589066|589066-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Acme-Scurvy-Whoreson-BilgeRat|1.1|darwin-2level|5.9.5|darwin|8.10.1|200708212318
589067|589067-b19f-3f77-b713-d32bba55d77f|unknown|200708|david@cantrell.org.uk|Acme-Pony|1.1.2|darwin-2level|5.9.5|darwin|8.10.1|200708212318
589072|589072-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200708212320
589087|589087-b19f-3f77-b713-d32bba55d77f|fail|200708|david@cantrell.org.uk|Bryar|3.0|darwin-2level|5.9.5|darwin|8.10.1|200708212323
589089|589089-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Class-CanBeA|1.2|darwin-2level|5.9.5|darwin|8.10.1|200708212323
589093|589093-b19f-3f77-b713-d32bba55d77f|fail|200708|david@cantrell.org.uk|Data-Transactional|0.1|darwin-2level|5.9.5|darwin|8.10.1|200708212323
589095|589095-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|File-Overwrite|1.0|darwin-2level|5.9.5|darwin|8.10.1|200708212324
589096|589096-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|File-Find-Rule-Permissions|1.2|darwin-2level|5.9.5|darwin|8.10.1|200708212323
589099|589099-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Net-Random|2.0|darwin-2level|5.9.5|darwin|8.10.1|200708212324
589100|589100-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Games-Dice-Advanced|1.1|darwin-2level|5.9.5|darwin|8.10.1|200708212324
589102|589102-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|NestedMap|1.0|darwin-2level|5.9.5|darwin|8.10.1|200708212324
589105|589105-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Number-Phone-UK-DetailedLocations|1.3|darwin-2level|5.9.5|darwin|8.10.1|200708212327
589106|589106-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1.1|darwin-2level|5.9.5|darwin|8.10.1|200708212328
589107|589107-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1|darwin-2level|5.9.5|darwin|8.10.1|200708212328
589110|589110-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Statistics-ChiSquare|0.5|darwin-2level|5.9.5|darwin|8.10.1|200708212328
589112|589112-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Statistics-SerialCorrelation|1.1|darwin-2level|5.9.5|darwin|8.10.1|200708212328
589114|589114-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Tie-Hash-Longest|1.1|darwin-2level|5.9.5|darwin|8.10.1|200708212328
589115|589115-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Sub-WrapPackages|1.2|darwin-2level|5.9.5|darwin|8.10.1|200708212328
589116|589116-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Tie-STDOUT|1.02|darwin-2level|5.9.5|darwin|8.10.1|200708212328
589117|589117-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Tie-Hash-Rank|1.0.1|darwin-2level|5.9.5|darwin|8.10.1|200708212328
589119|589119-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Tie-Scalar-Decay|1.1.1|darwin-2level|5.9.5|darwin|8.10.1|200708212329
589120|589120-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|darwin-2level|5.9.5|darwin|8.10.1|200708212329
589121|589121-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Tie-Scalar-Decay|1.1|darwin-2level|5.9.5|darwin|8.10.1|200708212329
589122|589122-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|XML-Tiny|1.06|darwin-2level|5.9.5|darwin|8.10.1|200708212329
589147|589147-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Data-Hexdumper|1.2|darwin-2level|5.9.5|darwin|8.10.1|200708212323
589149|589149-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Data-Hexdumper|1.0.1|darwin-2level|5.9.5|darwin|8.10.1|200708212323
589707|589707-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Data-Hexdumper|1.3|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200708220732
589902|589902-b19f-3f77-b713-d32bba55d77f|pass|200708|stro@cpan.org|Data-Hexdumper|1.3|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200708221013
589905|589905-b19f-3f77-b713-d32bba55d77f|pass|200708|stro@cpan.org|CPAN-FindDependencies|1.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200708220951
589906|589906-b19f-3f77-b713-d32bba55d77f|pass|200708|stro@cpan.org|CPAN-FindDependencies|1.02|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200708220952
589960|589960-b19f-3f77-b713-d32bba55d77f|pass|200708|stro@cpan.org|CPAN-FindDependencies|1.01|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200708220951
590091|590091-b19f-3f77-b713-d32bba55d77f|pass|200708|chaddai.fouche@gmail.com ("Chaddai Fouche")|Data-Compare|0.17|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200708221618
590117|590117-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i386-pc-solaris2.11|5.8.7|solaris|2.11|200708221558
590120|590120-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i386-pc-solaris2.11|5.8.7|solaris|2.11|200708221600
590122|590122-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i386-pc-solaris2.11|5.8.7|solaris|2.11|200708221601
590124|590124-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i386-pc-solaris2.11|5.8.7|solaris|2.11|200708221601
590675|590675-b19f-3f77-b713-d32bba55d77f|pass|200708|imacat@mail.imacat.idv.tw|Data-Hexdumper|1.3|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200708230522
591020|591020-b19f-3f77-b713-d32bba55d77f|pass|200708|tony@develop-help.com|Data-Compare|0.17|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200708230944
591446|591446-b19f-3f77-b713-d32bba55d77f|pass|200708|bob@randomness.org.uk ("bob walker")|Bryar|3.0|i86pc-solaris|5.8.8|solaris|2.10|200708231015
591701|591701-b19f-3f77-b713-d32bba55d77f|pass|200708|cpan@pqpq.de|CPAN-FindDependencies|1.02|i386-freebsd-thread-multi|5.9.5 patch 31749|freebsd|6.1-release-p6|200708231335
591731|591731-b19f-3f77-b713-d32bba55d77f|pass|200708|cpan@pqpq.de|Data-Hexdumper|1.3|i386-freebsd-thread-multi|5.9.5 patch 31749|freebsd|6.1-release-p6|200708231335
591734|591734-b19f-3f77-b713-d32bba55d77f|pass|200708|cpan@pqpq.de|CPAN-FindDependencies|1.0|i386-freebsd-thread-multi|5.9.5 patch 31749|freebsd|6.1-release-p6|200708231334
591735|591735-b19f-3f77-b713-d32bba55d77f|pass|200708|cpan@pqpq.de|CPAN-FindDependencies|1.01|i386-freebsd-thread-multi|5.9.5 patch 31749|freebsd|6.1-release-p6|200708231335
591769|591769-b19f-3f77-b713-d32bba55d77f|pass|200708|cpan@pqpq.de|Data-Compare|0.17|i386-freebsd-thread-multi|5.9.5 patch 31749|freebsd|6.1-release-p6|200708231210
592026|592026-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200708231648
592027|592027-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200708231649
592028|592028-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200708231651
592029|592029-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200708231651
592644|592644-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200708232350
592919|592919-b19f-3f77-b713-d32bba55d77f|pass|200708|cogent@cpan.org|Data-Compare|0.17|darwin-thread-multi-64int-2level|5.8.8|darwin|8.7.0|200708231440
593087|593087-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200708240501
593443|593443-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200708241048
593448|593448-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200708241050
593461|593461-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200708241047
593475|593475-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200708241050
594276|594276-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200708250644
594281|594281-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200708250648
594282|594282-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200708250649
594391|594391-b19f-3f77-b713-d32bba55d77f|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.02|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200708250940
594684|594684-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200708251155
594685|594685-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200708251156
594686|594686-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200708251156
595047|595047-b19f-3f77-b713-d32bba55d77f|pass|200708|srezic@cpan.org|CPAN-FindDependencies|1.02|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200708251106
595064|595064-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i686-linux-thread-multi-64int|5.9.5|linux|2.6.16.38|200708251908
595274|595274-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i686-linux-thread-multi-64int|5.9.5|linux|2.6.16.38|200708260006
595275|595275-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i686-linux-thread-multi-64int|5.9.5|linux|2.6.16.38|200708260007
595276|595276-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i686-linux-thread-multi-64int|5.9.5|linux|2.6.16.38|200708260007
595889|595889-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200708262118
595890|595890-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200708262118
597202|597202-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200708281836
597204|597204-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200708281837
597205|597205-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200708281837
597207|597207-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200708281837
597208|597208-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200708281838
597209|597209-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200708281838
597210|597210-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200708281838
597211|597211-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200708281839
597213|597213-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200708281839
597630|597630-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Transactional|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200708290537
597669|597669-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Transactional|1.0|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200708290641
597670|597670-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Transactional|1.0|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200708290642
597671|597671-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Transactional|1.0|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200708290643
597672|597672-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Transactional|1.0|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200708290643
597673|597673-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Transactional|1.0|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200708290644
597674|597674-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Transactional|1.0|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200708290644
597675|597675-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Transactional|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200708290645
597676|597676-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|Data-Transactional|1.0|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200708290646
597713|597713-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Tie-STDOUT|1.03|darwin-2level|5.9.5|darwin|8.10.1|200708290946
597716|597716-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Tie-STDOUT|1.03|i686-linux|5.8.8|linux|2.4.27-3-686|200708290951
597730|597730-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Tie-STDOUT|1.03|i686-linux|5.9.5|linux|2.4.27-3-686|200708290959
597780|597780-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Data-Transactional|1.0|i686-linux|5.6.2|linux|2.4.27-3-686|200708291105
597790|597790-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Data-Transactional|1.0|darwin-2level|5.9.5|darwin|8.10.1|200708291109
597795|597795-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Data-Transactional|1.0|i686-linux|5.8.8|linux|2.4.27-3-686|200708291110
597813|597813-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Data-Transactional|1.0|i686-linux|5.9.5|linux|2.4.27-3-686|200708291219
597905|597905-b19f-3f77-b713-d32bba55d77f|pass|200708|stro@cpan.org|Data-Transactional|1.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200708291330
597961|597961-b19f-3f77-b713-d32bba55d77f|pass|200708|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Transactional|1.0|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200708291427
597962|597962-b19f-3f77-b713-d32bba55d77f|pass|200708|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Tie-STDOUT|1.03|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200708291427
598003|598003-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Tie-STDOUT|1.03|i386-freebsd|5.9.5|freebsd|6.2-release|200708290937
598061|598061-b19f-3f77-b713-d32bba55d77f|pass|200708|imacat@mail.imacat.idv.tw|Tie-STDOUT|1.03|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200708292348
598091|598091-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Data-Transactional|1.0|i386-freebsd|5.9.5|freebsd|6.2-release|200708291141
598406|598406-b19f-3f77-b713-d32bba55d77f|pass|200708|imacat@mail.imacat.idv.tw|Data-Transactional|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200708301923
598875|598875-b19f-3f77-b713-d32bba55d77f|pass|200708|tony@develop-help.com|Data-Transactional|1.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200708311047
598876|598876-b19f-3f77-b713-d32bba55d77f|na|200708|tony@develop-help.com|Tie-STDOUT|1.03|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200708311048
598901|598901-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200708310354
598903|598903-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200708310355
598905|598905-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200708310356
598907|598907-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200708310358
598909|598909-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200708310359
598911|598911-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200708310400
598913|598913-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200708310401
598915|598915-b19f-3f77-b713-d32bba55d77f|pass|200708|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200708310402
599214|599214-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Tie-STDOUT|1.03|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200708311420
599278|599278-b19f-3f77-b713-d32bba55d77f|pass|200708|david@cantrell.org.uk|Data-Transactional|1.0|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200708311520
599522|599522-b19f-3f77-b713-d32bba55d77f|pass|200709|cpan@pqpq.de|Data-Transactional|1.0|i386-freebsd-thread-multi|5.9.5 patch 31749|freebsd|6.1-release-p6|200709010933
599523|599523-b19f-3f77-b713-d32bba55d77f|pass|200709|cpan@pqpq.de|Tie-STDOUT|1.03|i386-freebsd-thread-multi|5.9.5 patch 31749|freebsd|6.1-release-p6|200709010933
599621|599621-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|i386-freebsd|5.9.5|freebsd|6.2-release|200709010951
599788|599788-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Data-Transactional|1.0|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200709011328
599789|599789-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Tie-STDOUT|1.03|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200709011328
599841|599841-b19f-3f77-b713-d32bba55d77f|pass|200709|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200709011415
599842|599842-b19f-3f77-b713-d32bba55d77f|pass|200709|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.9.4|linux|2.6.17-2-k7|200709011417
599847|599847-b19f-3f77-b713-d32bba55d77f|pass|200709|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200709011421
599849|599849-b19f-3f77-b713-d32bba55d77f|pass|200709|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200709011422
599850|599850-b19f-3f77-b713-d32bba55d77f|pass|200709|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200709011423
599851|599851-b19f-3f77-b713-d32bba55d77f|pass|200709|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int|5.8.4|linux|2.6.18-4-k7|200709011424
599852|599852-b19f-3f77-b713-d32bba55d77f|pass|200709|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int|5.8.3|linux|2.6.18-4-k7|200709011425
599853|599853-b19f-3f77-b713-d32bba55d77f|pass|200709|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200709011426
599857|599857-b19f-3f77-b713-d32bba55d77f|pass|200709|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.9.3|linux|2.6.14|200709011418
599858|599858-b19f-3f77-b713-d32bba55d77f|pass|200709|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int|5.8.8 patch 31223|linux|2.6.22-1-k7|200709011420
600229|600229-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200709011840
600617|600617-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Data-Transactional|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200709012018
600618|600618-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Data-Compare|0.17|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200709012018
600620|600620-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Tie-STDOUT|1.03|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200709012019
600828|600828-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200709012036
601889|601889-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Acme-Licence|1.0|i386-netbsd-thread-multi-64int|5.9.5|netbsd|3.1|200709021301
602062|602062-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Acme-Pony|1.1.2|i386-netbsd-thread-multi-64int|5.9.5|netbsd|3.1|200709021344
602128|602128-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-netbsd-thread-multi-64int|5.9.5|netbsd|3.1|200709021403
602983|602983-b19f-3f77-b713-d32bba55d77f|pass|200709|bob@randomness.org.uk ("bob walker")|Tie-STDOUT|1.03|i86pc-solaris|5.8.8|solaris|2.10|200709021453
603004|603004-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200709020706
603202|603202-b19f-3f77-b713-d32bba55d77f|pass|200709|jrockway@cpan.org|Data-Compare|0.17|i486-linux-gnu|5.8.8|linux|2.6.18.3|200709011804
603343|603343-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Data-Transactional|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200709021923
603344|603344-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200709021923
603532|603532-b19f-3f77-b713-d32bba55d77f|pass|200709|corion-cpanplus@corion.net|Data-Compare|0.17|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200709022206
603534|603534-b19f-3f77-b713-d32bba55d77f|pass|200709|corion-cpanplus@corion.net|Data-Transactional|1.0|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200709022207
603536|603536-b19f-3f77-b713-d32bba55d77f|fail|200709|corion-cpanplus@corion.net|Tie-STDOUT|1.03|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200709022208
603917|603917-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200709030032
604525|604525-b19f-3f77-b713-d32bba55d77f|pass|200709|cpan@pqpq.de|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi|5.9.5 patch 31749|freebsd|6.1-release-p6|200709031103
604902|604902-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Data-Transactional|1.0|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200709031349
604903|604903-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200709031350
604907|604907-b19f-3f77-b713-d32bba55d77f|pass|200709|saper@cpan.org|Tie-STDOUT|1.03|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200709031450
605243|605243-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Data-Transactional|1.0|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200709031624
605244|605244-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200709031624
605308|605308-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Data-Transactional|1.0|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200709031812
605309|605309-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Tie-STDOUT|1.03|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200709031813
606619|606619-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200709040455
606840|606840-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Data-Transactional|1.0|i386-pc-solaris2.11|5.8.7|solaris|2.11|200709040750
606845|606845-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i386-pc-solaris2.11|5.8.7|solaris|2.11|200709040750
607296|607296-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Data-Transactional|1.0|alpha-linux-thread-multi|5.8.8|linux|2.6.16.19|200709041124
607298|607298-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Tie-STDOUT|1.03|alpha-linux-thread-multi|5.8.8|linux|2.6.16.19|200709041124
607602|607602-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-pc-solaris2.11|5.8.7|solaris|2.11|200709041352
607854|607854-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Data-Transactional|1.0|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200709041631
607857|607857-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200709041631
607951|607951-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|alpha-linux-thread-multi|5.8.8|linux|2.6.16.19|200709041750
609071|609071-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Data-Transactional|1.0|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200709050035
609076|609076-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200709050036
609166|609166-b19f-3f77-b713-d32bba55d77f|pass|200709|cpan@pqpq.de|Tie-STDOUT|1.03|i86pc-solaris|5.9.5 patch 31788|solaris|2.10|200709042059
609202|609202-b19f-3f77-b713-d32bba55d77f|pass|200709|cpan@pqpq.de|Data-Transactional|1.0|i86pc-solaris|5.9.5 patch 31788|solaris|2.10|200709042059
609215|609215-b19f-3f77-b713-d32bba55d77f|pass|200709|cpan@pqpq.de|Data-Compare|0.17|i86pc-solaris|5.9.5 patch 31788|solaris|2.10|200709042059
610943|610943-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200709052027
611293|611293-b19f-3f77-b713-d32bba55d77f|pass|200709|perlgeek@perlgeek.net|Tie-STDOUT|1.03|i686-linux|5.8.8|linux|2.6.9-42.0.3.elsmp|200709060223
611964|611964-b19f-3f77-b713-d32bba55d77f|pass|200709|davidp@preshweb.co.uk ("David Precious")|Data-Compare|0.17|i486-linux|5.8.7|linux|2.4.31|200709061158
612850|612850-b19f-3f77-b713-d32bba55d77f|pass|200709|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.06|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200709062106
616595|616595-b19f-3f77-b713-d32bba55d77f|pass|200709|cpan@0x64.nl|XML-Tiny|1.06|x86_64-linux-gnu|5.8.8|linux|2.6.18-1-amd64|200709081211
617035|617035-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|Data-Compare|0.02|i686-linux|5.6.2|linux|2.4.27-3-686|200709090041
617180|617180-b19f-3f77-b713-d32bba55d77f|pass|200709|andy@hexten.net|Data-Hexdumper|1.3|i486-linux-gnu|5.8.8|linux|2.6.15.7|200709090338
617430|617430-b19f-3f77-b713-d32bba55d77f|pass|200709|smueller@cpan.org ("Steffen Mueller")|XML-Tiny|1.06|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200709091052
620597|620597-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|Data-Compare|0.02|darwin-2level|5.9.5|darwin|8.10.1|200709102055
622464|622464-b19f-3f77-b713-d32bba55d77f|pass|200709|kamensky@cpan.org ("Moshe Kamensky")|Data-Compare|0.17|sun4-solaris|5.8.8|solaris|2.8|200709111136
623174|623174-b19f-3f77-b713-d32bba55d77f|pass|200709|andy@hexten.net|Data-Hexdumper|1.3|i486-linux-gnu|5.8.8|linux|2.6.15.7|200709112108
625331|625331-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|Data-Compare|0.17|sun4-solaris|5.9.5|solaris|2.9|200709121723
631722|631722-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Data-Compare|0.17|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200709151723
632261|632261-b19f-3f77-b713-d32bba55d77f|pass|200709|gabor@pti.co.il ("Gabor Szabo")|XML-DoubleEncodedEntities|1.0|i686-linux|5.8.8|linux|2.6.20-16-generic|200709151533
632453|632453-b19f-3f77-b713-d32bba55d77f|pass|200709|gabor@pti.co.il ("Gabor Szabo")|Data-Compare|0.17|i686-linux|5.8.8|linux|2.6.20-16-generic|200709151221
634970|634970-b19f-3f77-b713-d32bba55d77f|pass|200709|andy@hexten.net|Data-Hexdumper|1.3|i486-linux-gnu|5.8.8|linux|2.6.15.7|200709160900
636217|636217-b19f-3f77-b713-d32bba55d77f|pass|200709|cpan@pqpq.de|Data-Compare|0.17|i386-freebsd|5.10.0 patch 31806|freebsd|6.1-release-p6|200709162224
644503|644503-b19f-3f77-b713-d32bba55d77f|pass|200709|srezic@cpan.org|CPAN-FindDependencies|1.02|amd64-freebsd|5.10.0 patch 31894|freebsd|6.2-release|200709200002
645430|645430-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Data-Compare|0.17|i686-linux-thread-multi-64int|5.10.0 patch 31906|linux|2.6.16.38|200709201232
647235|647235-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Data-Compare|0.17|darwin-thread-multi-2level|5.8.6|darwin|8.10.0|200709210757
649448|649448-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Data-Compare|0.17|darwin-thread-multi-2level|5.8.5|darwin|8.10.0|200709220425
649692|649692-b19f-3f77-b713-d32bba55d77f|pass|200709|srezic@cpan.org|CPAN-FindDependencies|1.02|i386-freebsd|5.10.0 patch 31938|freebsd|6.1-release|200709220244
650269|650269-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200709221114
651644|651644-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Data-Compare|0.17|darwin-thread-multi-2level|5.8.4|darwin|8.10.0|200709230013
653290|653290-b19f-3f77-b713-d32bba55d77f|unknown|200709|david@cantrell.org.uk|Acme-Licence|1.0|i386-freebsd|5.9.5|freebsd|6.2-release|200709231412
653303|653303-b19f-3f77-b713-d32bba55d77f|unknown|200709|david@cantrell.org.uk|Acme-Pony|1.1.2|i386-freebsd|5.9.5|freebsd|6.2-release|200709231415
653315|653315-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-freebsd|5.9.5|freebsd|6.2-release|200709231416
653355|653355-b19f-3f77-b713-d32bba55d77f|fail|200709|david@cantrell.org.uk|Bryar|3.0|i386-freebsd|5.9.5|freebsd|6.2-release|200709231422
653360|653360-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|Class-CanBeA|1.2|i386-freebsd|5.9.5|freebsd|6.2-release|200709231423
653373|653373-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|Data-Hexdumper|1.0.1|i386-freebsd|5.9.5|freebsd|6.2-release|200709231425
653383|653383-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|Data-Hexdumper|1.2|i386-freebsd|5.9.5|freebsd|6.2-release|200709231426
653410|653410-b19f-3f77-b713-d32bba55d77f|fail|200709|david@cantrell.org.uk|Data-Transactional|0.1|i386-freebsd|5.9.5|freebsd|6.2-release|200709231428
653422|653422-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|File-Find-Rule-Permissions|1.2|i386-freebsd|5.9.5|freebsd|6.2-release|200709231429
653424|653424-b19f-3f77-b713-d32bba55d77f|pass|200709|chris@bingosnet.co.uk|Acme-Pony|1.1.2|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200709231454
653426|653426-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|File-Overwrite|1.0|i386-freebsd|5.9.5|freebsd|6.2-release|200709231431
653429|653429-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|Games-Dice-Advanced|1.1|i386-freebsd|5.9.5|freebsd|6.2-release|200709231432
653434|653434-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|NestedMap|1.0|i386-freebsd|5.9.5|freebsd|6.2-release|200709231434
653448|653448-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|Net-Random|2.0|i386-freebsd|5.9.5|freebsd|6.2-release|200709231438
653454|653454-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|Number-Phone|1.58|i386-freebsd|5.9.5|freebsd|6.2-release|200709231439
653465|653465-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|Number-Phone-UK-DetailedLocations|1.3|i386-freebsd|5.9.5|freebsd|6.2-release|200709231442
653473|653473-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1.1|i386-freebsd|5.9.5|freebsd|6.2-release|200709231444
653479|653479-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1|i386-freebsd|5.9.5|freebsd|6.2-release|200709231446
653484|653484-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|Statistics-ChiSquare|0.5|i386-freebsd|5.9.5|freebsd|6.2-release|200709231449
653494|653494-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|Statistics-SerialCorrelation|1.1|i386-freebsd|5.9.5|freebsd|6.2-release|200709231450
653506|653506-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|Tie-Hash-Longest|1.1|i386-freebsd|5.9.5|freebsd|6.2-release|200709231453
653512|653512-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|Tie-Hash-Rank|1.0.1|i386-freebsd|5.9.5|freebsd|6.2-release|200709231454
653518|653518-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|Tie-STDOUT|1.02|i386-freebsd|5.9.5|freebsd|6.2-release|200709231456
653529|653529-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|Tie-Scalar-Decay|1.1.1|i386-freebsd|5.9.5|freebsd|6.2-release|200709231457
653534|653534-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|Tie-Scalar-Decay|1.1|i386-freebsd|5.9.5|freebsd|6.2-release|200709231459
653538|653538-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|XML-Tiny|1.06|i386-freebsd|5.9.5|freebsd|6.2-release|200709231500
653652|653652-b19f-3f77-b713-d32bba55d77f|pass|200709|david@cantrell.org.uk|Sub-WrapPackages|1.2|i386-freebsd|5.9.5|freebsd|6.2-release|200709231452
663263|663263-b19f-3f77-b713-d32bba55d77f|pass|200710|imacat@mail.imacat.idv.tw|Devel-CheckOS|0.9|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200710012251
663600|663600-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|0.92|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200710012338
663601|663601-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|0.92|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200710012335
663602|663602-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|0.92|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710012334
663613|663613-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|0.92|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200710012336
663622|663622-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|0.92|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200710012337
663623|663623-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|0.92|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200710012337
663624|663624-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|0.92|i686-linux-thread-multi-64int|5.8.8 patch 31223|linux|2.6.22-1-k7|200710012336
664271|664271-b19f-3f77-b713-d32bba55d77f|pass|200710|cpan@pqpq.de|Devel-CheckOS|0.92|i386-freebsd|5.10.0 patch 31806|freebsd|6.1-release-p6|200710020911
664272|664272-b19f-3f77-b713-d32bba55d77f|pass|200710|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200710020710
664273|664273-b19f-3f77-b713-d32bba55d77f|pass|200710|cpan@pqpq.de|Devel-CheckOS|0.91|i386-freebsd|5.10.0 patch 31806|freebsd|6.1-release-p6|200710020911
664275|664275-b19f-3f77-b713-d32bba55d77f|pass|200710|cpan@pqpq.de|Devel-CheckOS|0.9|i386-freebsd|5.10.0 patch 31806|freebsd|6.1-release-p6|200710020911
664608|664608-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.91|darwin-2level|5.9.5|darwin|8.10.1|200710021141
664666|664666-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.92|darwin-2level|5.9.5|darwin|8.10.1|200710021152
664742|664742-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|Data-Compare|0.17|i686-linux|5.10.0 patch 31937|linux|2.6.18-4-486|200710021313
664771|664771-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.91|sun4-solaris|5.9.5|solaris|2.9|200710020627
664779|664779-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.91|i386-freebsd|5.9.5|freebsd|6.2-release|200710021131
664789|664789-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.91|i686-linux|5.8.8|linux|2.4.27-3-686|200710021233
664790|664790-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.91|i686-linux|5.9.5|linux|2.4.27-3-686|200710021234
665010|665010-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.92|i686-linux|5.9.5|linux|2.4.27-3-686|200710021358
665011|665011-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.91|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200710021258
665091|665091-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.92|i686-linux|5.8.8|linux|2.4.27-3-686|200710021421
665092|665092-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.91|i686-linux|5.6.2|linux|2.4.27-3-686|200710021421
665097|665097-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.92|i386-freebsd|5.9.5|freebsd|6.2-release|200710021324
665102|665102-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.92|sun4-solaris|5.9.5|solaris|2.9|200710020824
665182|665182-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.92|i686-linux|5.6.2|linux|2.4.27-3-686|200710021445
665351|665351-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|Devel-CheckOS|0.92|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200710020837
665385|665385-b19f-3f77-b713-d32bba55d77f|pass|200710|saper@cpan.org|Devel-CheckOS|0.92|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200710021621
665491|665491-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|Devel-CheckOS|0.92|amd64-freebsd|5.10.0 patch 31995|freebsd|6.2-release|200710020837
665546|665546-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.92|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200710021501
665823|665823-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.0|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200710021809
665827|665827-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.0|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200710021807
665828|665828-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.0|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200710021808
665829|665829-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.0|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200710021808
665831|665831-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.0|i686-linux-thread-multi-64int|5.8.8 patch 31223|linux|2.6.22-1-k7|200710021806
665832|665832-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.0|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200710021805
665833|665833-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.0|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710021804
666004|666004-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux|5.10.0|linux|2.6.22-1-k7|200710021710
666037|666037-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.9|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710021839
666053|666053-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.9|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710021846
666078|666078-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.9|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710021853
666102|666102-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|File-Find-Rule-Permissions|1.3|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710021958
666107|666107-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|File-Find-Rule-Permissions|1.3|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200710021959
666113|666113-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.9|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710021901
666149|666149-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.9|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710021908
666179|666179-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.9|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710021915
666199|666199-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|File-Find-Rule-Permissions|1.3|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200710022002
666204|666204-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.9|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710021922
666222|666222-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200710022000
666228|666228-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200710022001
666229|666229-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200710022001
666230|666230-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi-64int|5.8.8 patch 31223|linux|2.6.22-1-k7|200710022000
666239|666239-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.9|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710021929
666247|666247-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.9|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710021931
666540|666540-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.06|i686-linux|5.10.0|linux|2.6.22-1-k7|200710022105
667743|667743-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.0|i686-linux|5.10.0|linux|2.6.22-1-k7|200710031215
667823|667823-b19f-3f77-b713-d32bba55d77f|pass|200710|imacat@mail.imacat.idv.tw|Devel-CheckOS|0.91|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200710031934
667829|667829-b19f-3f77-b713-d32bba55d77f|pass|200710|imacat@mail.imacat.idv.tw|Devel-CheckOS|0.92|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200710031944
667886|667886-b19f-3f77-b713-d32bba55d77f|pass|200710|saper@cpan.org|File-Find-Rule-Permissions|1.3|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200710031453
668899|668899-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.0|i686-linux|5.8.8|linux|2.4.27-3-686|200710032339
668904|668904-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|File-Find-Rule-Permissions|1.3|i686-linux|5.8.8|linux|2.4.27-3-686|200710032341
668977|668977-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.0|darwin-2level|5.9.5|darwin|8.10.1|200710040024
668982|668982-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.0|sun4-solaris|5.9.5|solaris|2.9|200710040024
668983|668983-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|File-Find-Rule-Permissions|1.3|darwin-2level|5.9.5|darwin|8.10.1|200710040024
668985|668985-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.0|i386-freebsd|5.9.5|freebsd|6.2-release|200710040025
668989|668989-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|File-Find-Rule-Permissions|1.3|sun4-solaris|5.9.5|solaris|2.9|200710040027
668990|668990-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|File-Find-Rule-Permissions|1.3|i386-freebsd|5.9.5|freebsd|6.2-release|200710040028
669006|669006-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.0|i686-linux|5.9.5|linux|2.4.27-3-686|200710040037
669009|669009-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|File-Find-Rule-Permissions|1.3|i686-linux|5.9.5|linux|2.4.27-3-686|200710040038
669010|669010-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.0|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200710040038
669019|669019-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|File-Find-Rule-Permissions|1.3|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200710040043
669023|669023-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.0|i686-linux|5.6.2|linux|2.4.27-3-686|200710040045
669026|669026-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|File-Find-Rule-Permissions|1.3|i686-linux|5.6.2|linux|2.4.27-3-686|200710040046
669230|669230-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.10.0|linux|2.6.22-1-k7|200710040801
669588|669588-b19f-3f77-b713-d32bba55d77f|pass|200710|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200710042126
669592|669592-b19f-3f77-b713-d32bba55d77f|pass|200710|imacat@mail.imacat.idv.tw|File-Find-Rule-Permissions|1.3|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200710042127
669812|669812-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.91|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710041746
669813|669813-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.91|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710041744
669814|669814-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.91|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710041748
669815|669815-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.91|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710041750
669816|669816-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.91|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710041752
669817|669817-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.91|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710041754
669818|669818-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.91|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710041756
669820|669820-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.91|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710041758
669822|669822-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.91|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710041800
670143|670143-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.92|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710042349
670144|670144-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.92|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710042352
670145|670145-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.92|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710042353
670146|670146-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.92|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710042355
670148|670148-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.92|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710042358
670150|670150-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.92|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710050004
670153|670153-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.92|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710050007
670155|670155-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.92|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710050009
670157|670157-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.92|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710050012
670386|670386-b19f-3f77-b713-d32bba55d77f|pass|200710|alexchorny@gmail.com|File-Find-Rule-Permissions|1.3|i686-linux|5.10.0 patch 32015|linux|2.4.20-8smp|200710040629
670692|670692-b19f-3f77-b713-d32bba55d77f|pass|200710|alexchorny@gmail.com|Devel-CheckOS|1.0|i686-linux|5.10.0 patch 32015|linux|2.4.20-8smp|200710040629
670971|670971-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.06|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200710051147
671194|671194-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.0|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710051318
671198|671198-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.0|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710051320
671199|671199-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.0|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710051323
671200|671200-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.0|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710051325
671201|671201-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.0|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710051327
671202|671202-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.0|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710051329
671215|671215-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.0|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710051331
671220|671220-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.0|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710051333
671223|671223-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710051335
671243|671243-b19f-3f77-b713-d32bba55d77f|pass|200710|rlane10@wi.rr.com|Number-Phone|1.58|i386-linux|5.8.8|linux|2.6.9-34.elsmp|200710050721
671301|671301-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|File-Find-Rule-Permissions|1.3|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710051410
671306|671306-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|File-Find-Rule-Permissions|1.3|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710051413
671308|671308-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|File-Find-Rule-Permissions|1.3|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710051415
671312|671312-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|File-Find-Rule-Permissions|1.3|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710051417
671318|671318-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|File-Find-Rule-Permissions|1.3|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710051420
671322|671322-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|File-Find-Rule-Permissions|1.3|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710051422
671325|671325-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|File-Find-Rule-Permissions|1.3|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710051425
671326|671326-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|File-Find-Rule-Permissions|1.3|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710051427
671328|671328-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|File-Find-Rule-Permissions|1.3|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710051429
671367|671367-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.1|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710051606
671369|671369-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.1|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200710051607
671370|671370-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.1|i686-linux-thread-multi-64int|5.8.8 patch 31223|linux|2.6.22-1-k7|200710051608
671371|671371-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.1|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200710051608
671374|671374-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.1|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200710051609
671377|671377-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.1|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200710051609
671400|671400-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.1|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200710051610
671477|671477-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710051757
673261|673261-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200710061431
674252|674252-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200710061335
674519|674519-b19f-3f77-b713-d32bba55d77f|fail|200710|icerider70@gmail.com ("ICERIDER")|Devel-CheckOS|1.1|i486-linux-gnu|5.8.8|linux|2.6.18.3|200710061502
674615|674615-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.1|darwin-2level|5.9.5|darwin|8.10.1|200710062344
674617|674617-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.1|sun4-solaris|5.9.5|solaris|2.9|200710061745
674669|674669-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.1|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200710070004
674672|674672-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.1|i686-linux|5.9.5|linux|2.4.27-3-686|200710070004
674674|674674-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.1|i386-freebsd|5.9.5|freebsd|6.2-release|200710070005
674727|674727-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.1|i686-linux|5.6.2|linux|2.4.27-3-686|200710070039
674728|674728-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.1|i686-linux|5.8.8|linux|2.4.27-3-686|200710070039
675923|675923-b19f-3f77-b713-d32bba55d77f|pass|200710|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.1|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200710080206
676370|676370-b19f-3f77-b713-d32bba55d77f|pass|200710|alexchorny@gmail.com|Devel-CheckOS|1.1|i686-linux|5.10.0 patch 32015|linux|2.4.20-8smp|200710071351
676967|676967-b19f-3f77-b713-d32bba55d77f|pass|200710|saper@cpan.org|Devel-CheckOS|1.1|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200710081119
677780|677780-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|Data-Compare|0.17|amd64-freebsd|5.10.0 patch 32039|freebsd|6.2-release|200710082304
677917|677917-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|Data-Compare|0.17|amd64-freebsd|5.6.2|freebsd|6.2-release|200710082309
679862|679862-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.1|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710101039
679863|679863-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.1|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710101041
679870|679870-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.1|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710101043
679875|679875-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.1|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710101045
679883|679883-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.1|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710101048
679889|679889-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.1|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710101050
679894|679894-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.1|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710101052
679895|679895-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.1|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710101054
679896|679896-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.1|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710101056
680299|680299-b19f-3f77-b713-d32bba55d77f|pass|200710|david@valuesofn.com|Data-Compare|0.17|darwin-2level|5.8.8|darwin|8.8.2|200710101043
681705|681705-b19f-3f77-b713-d32bba55d77f|pass|200710|icerider70@gmail.com ("ICERIDER")|Devel-CheckOS|1.1|i486-linux-gnu|5.8.8|linux|2.6.18.3|200710111716
682486|682486-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710121126
683801|683801-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|XML-Tiny|1.06|amd64-freebsd|5.10.0 patch 32096|freebsd|6.2-release|200710131421
683802|683802-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|XML-Tiny|1.06|amd64-freebsd|5.6.2|freebsd|6.2-release|200710131422
683803|683803-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|XML-Tiny|1.06|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200710131421
683804|683804-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|XML-Tiny|1.06|amd64-freebsd|5.5.5|freebsd|6.2-release|200710131422
685543|685543-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|Devel-CheckOS|1.1|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200710141046
685544|685544-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|Devel-CheckOS|1.1|amd64-freebsd|5.10.0 patch 32096|freebsd|6.2-release|200710141045
685545|685545-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|Devel-CheckOS|1.1|amd64-freebsd|5.6.2|freebsd|6.2-release|200710141046
686176|686176-b19f-3f77-b713-d32bba55d77f|unknown|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-AssertLib|0.1|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710141723
686178|686178-b19f-3f77-b713-d32bba55d77f|unknown|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-AssertLib|0.1|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200710141723
686180|686180-b19f-3f77-b713-d32bba55d77f|unknown|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-AssertLib|0.1|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200710141724
686183|686183-b19f-3f77-b713-d32bba55d77f|unknown|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-AssertLib|0.1|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200710141724
686207|686207-b19f-3f77-b713-d32bba55d77f|unknown|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-AssertLib|0.1|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200710141723
686209|686209-b19f-3f77-b713-d32bba55d77f|unknown|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-AssertLib|0.1|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200710141723
686210|686210-b19f-3f77-b713-d32bba55d77f|unknown|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-AssertLib|0.1|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200710141724
686293|686293-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.2|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710141815
686295|686295-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.2|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200710141815
686300|686300-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.2|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200710141815
686304|686304-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.2|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200710141816
686308|686308-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.2|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200710141816
686373|686373-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.2|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200710141816
686378|686378-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.2|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200710141815
686488|686488-b19f-3f77-b713-d32bba55d77f|unknown|200710|srezic@cpan.org|Devel-AssertLib|0.1|amd64-freebsd|5.10.0 patch 32096|freebsd|6.2-release|200710141935
686622|686622-b19f-3f77-b713-d32bba55d77f|unknown|200710|srezic@cpan.org|Devel-AssertLib|0.1|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200710142014
686730|686730-b19f-3f77-b713-d32bba55d77f|unknown|200710|srezic@cpan.org|Devel-AssertLib|0.1|amd64-freebsd|5.6.2|freebsd|6.2-release|200710142111
686818|686818-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|Devel-CheckOS|1.2|amd64-freebsd|5.10.0 patch 32096|freebsd|6.2-release|200710150044
686938|686938-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|Devel-CheckOS|1.2|amd64-freebsd|5.6.2|freebsd|6.2-release|200710150113
686952|686952-b19f-3f77-b713-d32bba55d77f|unknown|200710|srezic@cpan.org|Devel-AssertLib|0.1|amd64-freebsd|5.5.5|freebsd|6.2-release|200710150117
687066|687066-b19f-3f77-b713-d32bba55d77f|pass|200710|andy@hexten.net|Data-Hexdumper|1.3|i86pc-solaris-64int|5.8.4|solaris|2.11|200710150226
687689|687689-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|alpha-linux-thread-multi|5.8.8|linux|2.6.16.19|200710151304
687691|687691-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|alpha-linux-thread-multi|5.8.8|linux|2.6.16.19|200710151305
687712|687712-b19f-3f77-b713-d32bba55d77f|pass|200710|xdg@hyperbolic.net (DAGOLDEN)|Devel-CheckOS|1.2|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200710150820
687716|687716-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200710151333
687717|687717-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200710151334
688024|688024-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.2|i686-linux|5.9.5|linux|2.4.27-3-686|200710151516
688025|688025-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.2|i686-linux|5.8.8|linux|2.4.27-3-686|200710151516
688026|688026-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.2|darwin-2level|5.9.5|darwin|8.10.1|200710151508
688030|688030-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.2|i686-linux|5.6.2|linux|2.4.27-3-686|200710151507
688096|688096-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.2|sun4-solaris|5.9.5|solaris|2.9|200710150937
688180|688180-b19f-3f77-b713-d32bba55d77f|pass|200710|saper@cpan.org|Devel-AssertLib|0.1|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200710151711
688233|688233-b19f-3f77-b713-d32bba55d77f|pass|200710|imacat@mail.imacat.idv.tw|Devel-AssertLib|0.1|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200710152339
688356|688356-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.2|i386-freebsd|5.9.5|freebsd|6.2-release|200710151316
688419|688419-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.2|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200710151750
689795|689795-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200710160742
689801|689801-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200710160743
690422|690422-b19f-3f77-b713-d32bba55d77f|pass|200710|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.2|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200710162118
690817|690817-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i686-linux-thread-multi-64int|5.10.0 patch 31906|linux|2.6.16.38|200710161710
690828|690828-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i686-linux-thread-multi-64int|5.10.0 patch 31906|linux|2.6.16.38|200710161711
691859|691859-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710170645
691860|691860-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710170645
691861|691861-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710170645
691862|691862-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710170646
691863|691863-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710170646
691864|691864-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710170646
691865|691865-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710170646
691866|691866-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710170647
691867|691867-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710170647
691868|691868-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710170647
691869|691869-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710170648
691870|691870-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710170649
691871|691871-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710170649
691872|691872-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710170650
691873|691873-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710170650
691874|691874-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710170651
691875|691875-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710170651
691876|691876-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710170652
692213|692213-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200710171009
692214|692214-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200710171010
692388|692388-b19f-3f77-b713-d32bba55d77f|pass|200710|cpan@pqpq.de|Devel-CheckOS|1.2|i386-freebsd|5.10.0 patch 31806|freebsd|6.1-release-p6|200710171254
692389|692389-b19f-3f77-b713-d32bba55d77f|pass|200710|cpan@pqpq.de|Devel-AssertLib|0.1|i386-freebsd|5.10.0 patch 31806|freebsd|6.1-release-p6|200710171254
693524|693524-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710180053
693559|693559-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200710180055
693630|693630-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200710180016
693641|693641-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200710180017
693720|693720-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd|5.10.0 patch 31938|freebsd|6.1-release|200710180126
693798|693798-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200710180126
693802|693802-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd|5.6.2|freebsd|6.1-release|200710180127
695204|695204-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200710181755
695210|695210-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200710181758
695352|695352-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|sun4-solaris|5.9.5|solaris|2.9|200710181429
695818|695818-b19f-3f77-b713-d32bba55d77f|pass|200710|xdg@hyperbolic.net (DAGOLDEN)|Devel-CheckLib|0.2|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200710181841
695827|695827-b19f-3f77-b713-d32bba55d77f|pass|200710|xdg@hyperbolic.net (DAGOLDEN)|Devel-CheckLib|0.2|i686-linux-64int-ld|5.8.8|linux|2.6.17-10-generic|200710181845
695959|695959-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.2|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710190300
696149|696149-b19f-3f77-b713-d32bba55d77f|pass|200710|cpan@pqpq.de|XML-DoubleEncodedEntities|1.0|i386-freebsd|5.10.0 patch 31806|freebsd|6.1-release-p6|200710190702
696435|696435-b19f-3f77-b713-d32bba55d77f|pass|200710|cpan@pqpq.de|Devel-CheckLib|0.2|i386-freebsd|5.10.0 patch 31806|freebsd|6.1-release-p6|200710190814
696830|696830-b19f-3f77-b713-d32bba55d77f|pass|200710|saper@cpan.org|Devel-CheckLib|0.2|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200710191054
696970|696970-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200710191005
696972|696972-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200710191005
696974|696974-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200710191006
697028|697028-b19f-3f77-b713-d32bba55d77f|pass|200710|icerider70@gmail.com ("ICERIDER")|Devel-CheckOS|1.2|i486-linux-gnu|5.8.8|linux|2.6.18.3|200710190612
697605|697605-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200710191626
697844|697844-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200710191859
697845|697845-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200710191900
697846|697846-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200710191900
698349|698349-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200710200033
698350|698350-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200710200035
698351|698351-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200710200037
698796|698796-b19f-3f77-b713-d32bba55d77f|fail|200710|mr.muskrat@perlgeek.net ("Mr. Muskrat")|Devel-CheckLib|0.2|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200710200031
698896|698896-b19f-3f77-b713-d32bba55d77f|unknown|200710|mr.muskrat@perlgeek.net ("Mr. Muskrat")|Devel-AssertLib|0.1|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200710200030
698912|698912-b19f-3f77-b713-d32bba55d77f|pass|200710|mr.muskrat@perlgeek.net ("Mr. Muskrat")|Devel-CheckOS|1.2|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200710200032
699054|699054-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710201144
699218|699218-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200710200134
699476|699476-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.2|darwin-2level|5.9.5|darwin|8.10.1|200710201531
699498|699498-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.2|i686-linux|5.9.5|linux|2.4.27-3-686|200710201535
699502|699502-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.2|i686-linux|5.8.8|linux|2.4.27-3-686|200710201535
699537|699537-b19f-3f77-b713-d32bba55d77f|pass|200710|imacat@mail.imacat.idv.tw|Devel-CheckLib|0.2|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200710202240
699565|699565-b19f-3f77-b713-d32bba55d77f|pass|200710|icerider70@gmail.com ("ICERIDER")|XML-DoubleEncodedEntities|1.0|i486-linux-gnu|5.8.8|linux|2.6.18.3|200710200957
699613|699613-b19f-3f77-b713-d32bba55d77f|pass|200710|mmusgrove@cpan.org|Devel-CheckOS|1.2|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200710200902
699617|699617-b19f-3f77-b713-d32bba55d77f|pass|200710|mmusgrove@cpan.org|Devel-CheckLib|0.2|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200710200937
699640|699640-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.2|i686-linux|5.6.2|linux|2.4.27-3-686|200710201631
699764|699764-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.2|i386-freebsd|5.9.5|freebsd|6.2-release|200710201454
699784|699784-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.2|sun4-solaris|5.9.5|solaris|2.9|200710201118
699895|699895-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.2|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200710201649
700529|700529-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200710202034
700532|700532-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200710202035
700533|700533-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200710202036
700907|700907-b19f-3f77-b713-d32bba55d77f|pass|200710|mmusgrove@cpan.org|XML-DoubleEncodedEntities|1.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200710201315
701122|701122-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710202334
701123|701123-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710202335
701125|701125-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710202336
701127|701127-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710202336
701130|701130-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710202337
701131|701131-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710202337
701133|701133-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710202338
701134|701134-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710202339
701136|701136-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710202339
701604|701604-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200710210601
702100|702100-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i386-pc-solaris2.11|5.8.7|solaris|2.11|200710211441
702101|702101-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i386-pc-solaris2.11|5.8.7|solaris|2.11|200710211441
702102|702102-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i386-pc-solaris2.11|5.8.7|solaris|2.11|200710211442
702158|702158-b19f-3f77-b713-d32bba55d77f|pass|200710|bob@randomness.org.uk ("bob walker")|Devel-CheckLib|0.2|i86pc-solaris|5.8.8|solaris|2.10|200710211502
702598|702598-b19f-3f77-b713-d32bba55d77f|fail|200710|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-AssertLib|0.1|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200710211728
702757|702757-b19f-3f77-b713-d32bba55d77f|fail|200710|srezic@cpan.org|Devel-CheckLib|0.2|i386-freebsd|5.10.0 patch 31938|freebsd|6.1-release|200710212020
702758|702758-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|Devel-CheckLib|0.2|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200710212020
702759|702759-b19f-3f77-b713-d32bba55d77f|fail|200710|srezic@cpan.org|Devel-CheckLib|0.2|i386-freebsd|5.6.2|freebsd|6.1-release|200710212021
702855|702855-b19f-3f77-b713-d32bba55d77f|fail|200710|srezic@cpan.org|Devel-CheckLib|0.2|i386-freebsd|5.5.5|freebsd|6.1-release|200710212111
703302|703302-b19f-3f77-b713-d32bba55d77f|pass|200710|bob@randomness.org.uk|Devel-CheckOS|1.2|i386-openbsd|5.8.8|openbsd|4.2|200710212258
703306|703306-b19f-3f77-b713-d32bba55d77f|pass|200710|bob@randomness.org.uk|Devel-CheckLib|0.2|i386-openbsd|5.8.8|openbsd|4.2|200710212259
704075|704075-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200710221130
704614|704614-b19f-3f77-b713-d32bba55d77f|pass|200710|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200710221350
705556|705556-b19f-3f77-b713-d32bba55d77f|pass|200710|andy@hexten.net|Data-Compare|0.17|i486-linux-gnu|5.8.8|linux|2.6.15.7|200710222023
706080|706080-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200710222325
707010|707010-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.2|i686-linux-thread-multi|5.10.0|linux|2.6.22-1-k7|200710230655
707013|707013-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.06|i686-linux-thread-multi|5.10.0|linux|2.6.22-1-k7|200710230655
707652|707652-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.10.0|linux|2.6.22-1-k7|200710230718
708218|708218-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-thread-multi|5.10.0|linux|2.6.22-1-k7|200710230351
708858|708858-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200710231428
708917|708917-b19f-3f77-b713-d32bba55d77f|pass|200710|Jost.Krieger@ruhr-uni-bochum.de|Devel-CheckOS|1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200710231407
708922|708922-b19f-3f77-b713-d32bba55d77f|pass|200710|Jost.Krieger@ruhr-uni-bochum.de|File-Find-Rule-Permissions|1.3|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200710231407
709511|709511-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200710231759
710694|710694-b19f-3f77-b713-d32bba55d77f|fail|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.2|200710241103
710738|710738-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.2|200710241111
711237|711237-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.07|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710241518
711238|711238-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.07|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200710241519
711240|711240-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.07|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200710241519
711243|711243-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.07|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200710241520
711245|711245-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.07|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200710241519
711247|711247-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.07|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200710241519
711778|711778-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|XML-Tiny|1.07|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200710242156
711936|711936-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710242158
711937|711937-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710242158
711939|711939-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710242158
711940|711940-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710242159
711941|711941-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710242159
711942|711942-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710242200
711943|711943-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710242200
711944|711944-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710242200
711945|711945-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710242201
712027|712027-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|XML-Tiny|1.07|i386-freebsd|5.10.0 patch 32171|freebsd|6.1-release|200710242237
712321|712321-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|XML-Tiny|1.07|i386-freebsd|5.5.5|freebsd|6.1-release|200710250037
712334|712334-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|XML-Tiny|1.07|i386-freebsd|5.6.2|freebsd|6.1-release|200710242345
712381|712381-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-thread-multi-64int|5.10.0|linux|2.6.22-1-k7|200710250128
713464|713464-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.2|i686-linux-thread-multi-64int|5.10.0|linux|2.6.22-1-k7|200710250735
713625|713625-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.07|i686-linux-thread-multi-64int|5.10.0|linux|2.6.22-1-k7|200710250848
713802|713802-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int|5.10.0|linux|2.6.22-1-k7|200710250829
714082|714082-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200710251121
714083|714083-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200710251121
714227|714227-b19f-3f77-b713-d32bba55d77f|pass|200710|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-Tiny|1.07|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200710251225
714305|714305-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.07|darwin-2level|5.9.5|darwin|8.10.1|200710251427
714319|714319-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.07|i686-linux|5.6.2|linux|2.4.27-3-686|200710251428
714330|714330-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.07|i686-linux|5.9.5|linux|2.4.27-3-686|200710251430
714400|714400-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.07|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200710251353
714407|714407-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.07|sun4-solaris|5.9.5|solaris|2.9|200710250851
714443|714443-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.07|i686-linux|5.8.8|linux|2.4.27-3-686|200710251513
714470|714470-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.07|i386-freebsd|5.9.5|freebsd|6.2-release|200710251113
714489|714489-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.08|i686-linux-thread-multi-64int|5.10.0|linux|2.6.22-1-k7|200710251631
714490|714490-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.08|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200710251633
714492|714492-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.08|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200710251633
714493|714493-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.08|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200710251634
714498|714498-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.08|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200710251632
714500|714500-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.08|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200710251632
714524|714524-b19f-3f77-b713-d32bba55d77f|pass|200710|andy@hexten.net|Data-Hexdumper|1.3|powerpc-linux-gnu|5.8.8|linux|2.6.17-10-powerpc64-smp|200710251645
714870|714870-b19f-3f77-b713-d32bba55d77f|pass|200710|imacat@mail.imacat.idv.tw|XML-Tiny|1.07|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200710260119
714943|714943-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|XML-Tiny|1.08|amd64-freebsd|5.10.0 patch 32096|freebsd|6.2-release|200710252029
715122|715122-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|XML-Tiny|1.08|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200710252155
715413|715413-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710252223
715420|715420-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710252224
715423|715423-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710252225
715425|715425-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710252225
715427|715427-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710252226
715428|715428-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710252227
715432|715432-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710252227
715434|715434-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710252228
715435|715435-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710252229
715463|715463-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|XML-Tiny|1.08|amd64-freebsd|5.6.2|freebsd|6.2-release|200710252317
715653|715653-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|XML-Tiny|1.08|amd64-freebsd|5.5.5|freebsd|6.2-release|200710260039
715764|715764-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|Data-Compare|0.17|amd64-freebsd|5.10.0 patch 32096|freebsd|6.2-release|200710260041
715840|715840-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.08|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710260247
715852|715852-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200710260233
715853|715853-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200710260233
716103|716103-b19f-3f77-b713-d32bba55d77f|pass|200710|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-Tiny|1.08|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200710260752
716290|716290-b19f-3f77-b713-d32bba55d77f|pass|200710|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200710260935
716541|716541-b19f-3f77-b713-d32bba55d77f|pass|200710|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200710261317
716957|716957-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.08|darwin-2level|5.9.5|darwin|8.10.1|200710261635
716962|716962-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.08|i686-linux|5.9.5|linux|2.4.27-3-686|200710261636
716977|716977-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.08|i686-linux|5.8.8|linux|2.4.27-3-686|200710261638
716993|716993-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.08|i386-freebsd|5.9.5|freebsd|6.2-release|200710261351
717087|717087-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.08|i686-linux|5.6.2|linux|2.4.27-3-686|200710261725
717191|717191-b19f-3f77-b713-d32bba55d77f|pass|200710|imacat@mail.imacat.idv.tw|XML-Tiny|1.08|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200710270055
717424|717424-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.08|sun4-solaris|5.9.5|solaris|2.9|200710261445
717447|717447-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200710262045
717448|717448-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200710262046
717474|717474-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200710262133
717475|717475-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200710262134
717581|717581-b19f-3f77-b713-d32bba55d77f|pass|200710|icerider70@gmail.com ("ICERIDER")|XML-Tiny|1.07|i486-linux-gnu|5.8.8|linux|2.6.18.3|200710261644
718615|718615-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200710271354
718617|718617-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200710271354
719002|719002-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200710270712
719003|719003-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200710270712
719671|719671-b19f-3f77-b713-d32bba55d77f|pass|200710|mmusgrove@cpan.org|XML-Tiny|1.08|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200710280102
719929|719929-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200710280342
719930|719930-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200710280342
721013|721013-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i386-netbsd-thread-multi-64int|5.10.0 patch 32199|netbsd|3.1|200710281256
721014|721014-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i386-netbsd-thread-multi-64int|5.10.0 patch 32199|netbsd|3.1|200710281256
721312|721312-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i386-pc-solaris2.11|5.8.7|solaris|2.11|200710281527
721313|721313-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i386-pc-solaris2.11|5.8.7|solaris|2.11|200710281528
722085|722085-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.10.0 patch 32199|netbsd|3.1|200710281954
722872|722872-b19f-3f77-b713-d32bba55d77f|pass|200710|mmusgrove@cpan.org|Data-Compare|0.17|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200710290333
723241|723241-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200710291142
723243|723243-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200710291142
723796|723796-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.09|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710291858
723799|723799-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.09|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200710291859
723801|723801-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.09|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200710291900
723806|723806-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.09|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200710291901
723808|723808-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.09|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200710291902
723809|723809-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.09|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200710291903
724183|724183-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200710292135
724187|724187-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200710292136
724259|724259-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|XML-Tiny|1.09|amd64-freebsd|5.10.0 patch 32096|freebsd|6.2-release|200710292255
724288|724288-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|XML-Tiny|1.09|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200710292317
724381|724381-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.10|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710292352
724384|724384-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|XML-Tiny|1.09|amd64-freebsd|5.6.2|freebsd|6.2-release|200710292347
724386|724386-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.10|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200710292353
724392|724392-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.10|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200710292353
724403|724403-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.10|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200710292353
724404|724404-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.10|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200710292353
724405|724405-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.10|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200710292354
724510|724510-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|XML-Tiny|1.09|amd64-freebsd|5.5.5|freebsd|6.2-release|200710300021
724537|724537-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|XML-Tiny|1.10|amd64-freebsd|5.5.5|freebsd|6.2-release|200710300027
724568|724568-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|XML-Tiny|1.10|amd64-freebsd|5.6.2|freebsd|6.2-release|200710300026
724572|724572-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|XML-Tiny|1.10|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200710300026
724585|724585-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|XML-Tiny|1.10|amd64-freebsd|5.10.0 patch 32096|freebsd|6.2-release|200710300025
724754|724754-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.09|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710300127
724755|724755-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.09|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710300128
724756|724756-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.09|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710300128
724757|724757-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.09|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710300128
724758|724758-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.09|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710300129
724759|724759-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.09|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710300129
724761|724761-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.09|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710300129
724762|724762-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.09|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710300130
724767|724767-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.09|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710300130
725114|725114-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.10|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710300556
725116|725116-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.10|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710300556
725117|725117-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.10|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710300557
725118|725118-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.10|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710300557
725119|725119-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.10|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710300557
725120|725120-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.10|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710300558
725123|725123-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.10|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710300558
725125|725125-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.10|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710300558
725127|725127-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.10|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710300559
725167|725167-b19f-3f77-b713-d32bba55d77f|pass|200710|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-Tiny|1.10|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200710300816
725313|725313-b19f-3f77-b713-d32bba55d77f|pass|200710|cpan@0x64.nl|Bryar|3.0|x86_64-linux-gnu|5.8.8|linux|2.6.17-2-vserver-amd64|200710300124
725387|725387-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200710301051
725389|725389-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.09|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200710301051
725391|725391-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.10|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200710301052
725392|725392-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200710301051
725433|725433-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.09|i686-linux|5.8.8|linux|2.4.27-3-686|200710301118
725438|725438-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.09|darwin-2level|5.9.5|darwin|8.10.1|200710301119
725448|725448-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.09|i686-linux|5.9.5|linux|2.4.27-3-686|200710301124
725561|725561-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.10|i686-linux|5.9.5|linux|2.4.27-3-686|200710301228
725576|725576-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.09|i686-linux|5.6.2|linux|2.4.27-3-686|200710301230
725595|725595-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.10|i686-linux|5.8.8|linux|2.4.27-3-686|200710301232
725596|725596-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.10|darwin-2level|5.9.5|darwin|8.10.1|200710301232
725706|725706-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.09|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200710301302
725747|725747-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.10|i686-linux|5.6.2|linux|2.4.27-3-686|200710301249
725750|725750-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.09|sun4-solaris|5.9.5|solaris|2.9|200710300813
725806|725806-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.09|i386-freebsd|5.9.5|freebsd|6.2-release|200710301029
725849|725849-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.10|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200710301411
725881|725881-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.10|sun4-solaris|5.9.5|solaris|2.9|200710300937
725967|725967-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710301630
725968|725968-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200710301631
725969|725969-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200710301631
725970|725970-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200710301632
725978|725978-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200710301631
725980|725980-b19f-3f77-b713-d32bba55d77f|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200710301631
726169|726169-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i686-linux-thread-multi-64int-ld|5.10.0 patch 32202|linux|2.6.16.38|200710301718
726170|726170-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i686-linux-thread-multi-64int-ld|5.10.0 patch 32202|linux|2.6.16.38|200710301718
726171|726171-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.09|i686-linux-thread-multi-64int-ld|5.10.0 patch 32202|linux|2.6.16.38|200710301718
726172|726172-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.10|i686-linux-thread-multi-64int-ld|5.10.0 patch 32202|linux|2.6.16.38|200710301719
726508|726508-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|Devel-CheckLib|0.3|amd64-freebsd|5.10.0 patch 32200|freebsd|6.2-release|200710302036
726675|726675-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|XML-Tiny|1.10|amd64-freebsd|5.10.0 patch 32200|freebsd|6.2-release|200710302055
726727|726727-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|Devel-CheckLib|0.3|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200710302109
726791|726791-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|Devel-CheckLib|0.3|amd64-freebsd|5.6.2|freebsd|6.2-release|200710302200
726828|726828-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|XML-Tiny|1.10|i386-freebsd|5.9.5|freebsd|6.2-release|200710301059
726836|726836-b19f-3f77-b713-d32bba55d77f|pass|200710|srezic@cpan.org|Devel-CheckLib|0.3|amd64-freebsd|5.5.5|freebsd|6.2-release|200710302227
726852|726852-b19f-3f77-b713-d32bba55d77f|fail|200710|srezic@cpan.org|XML-Tiny|1.10|amd64-freebsd|5.5.5|freebsd|6.2-release|200710302232
727114|727114-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710310013
727115|727115-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710310014
727116|727116-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710310014
727117|727117-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710310014
727118|727118-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710310015
727120|727120-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710310015
727121|727121-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710310015
727122|727122-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710310016
727123|727123-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710310016
727192|727192-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Data-Compare|0.17|i686-linux-thread-multi-64int-ld|5.10.0 patch 32202|linux|2.6.16.38|200710310050
727341|727341-b19f-3f77-b713-d32bba55d77f|pass|200710|mmusgrove@cpan.org|XML-Tiny|1.10|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200710310337
727446|727446-b19f-3f77-b713-d32bba55d77f|pass|200710|mmusgrove@cpan.org|XML-Tiny|1.10|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200710310008
727554|727554-b19f-3f77-b713-d32bba55d77f|pass|200710|mmusgrove@cpan.org|Data-Compare|0.17|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200710310116
727762|727762-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.3|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200710311018
727763|727763-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200710311019
727764|727764-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.09|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200710311019
727765|727765-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.10|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200710311020
727781|727781-b19f-3f77-b713-d32bba55d77f|pass|200710|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckLib|0.3|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200710311052
727820|727820-b19f-3f77-b713-d32bba55d77f|pass|200710|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200710311203
728112|728112-b19f-3f77-b713-d32bba55d77f|pass|200711|imacat@mail.imacat.idv.tw|XML-Tiny|1.09|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200711010038
728139|728139-b19f-3f77-b713-d32bba55d77f|pass|200711|imacat@mail.imacat.idv.tw|XML-Tiny|1.10|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200711010058
728217|728217-b19f-3f77-b713-d32bba55d77f|pass|200711|imacat@mail.imacat.idv.tw|Devel-CheckLib|0.3|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200711010151
728275|728275-b19f-3f77-b713-d32bba55d77f|pass|200710|saper@cpan.org|XML-Tiny|1.08|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200710311957
728443|728443-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.3|i686-linux|5.9.5|linux|2.4.27-3-686|200710312032
728512|728512-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.3|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200710312101
728514|728514-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200710312101
728517|728517-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.09|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200710312102
728518|728518-b19f-3f77-b713-d32bba55d77f|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.10|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200710312102
728552|728552-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.3|i686-linux|5.6.2|linux|2.4.27-3-686|200710312115
728555|728555-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.3|darwin-2level|5.9.5|darwin|8.10.1|200710312116
728567|728567-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.3|i686-linux|5.8.8|linux|2.4.27-3-686|200710312117
728568|728568-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.3|i386-freebsd|5.9.5|freebsd|6.2-release|200710312115
728604|728604-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.3|sun4-solaris|5.9.5|solaris|2.9|200710311636
728605|728605-b19f-3f77-b713-d32bba55d77f|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.3|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200710312136
728884|728884-b19f-3f77-b713-d32bba55d77f|pass|200711|srezic@cpan.org|Data-Compare|0.17|amd64-freebsd|5.10.0 patch 32200|freebsd|6.2-release|200711010041
729616|729616-b19f-3f77-b713-d32bba55d77f|pass|200711|icerider70@gmail.com ("ICERIDER")|XML-Tiny|1.10|i486-linux-gnu|5.8.8|linux|2.6.18.3|200711010424
730066|730066-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200711010550
730773|730773-b19f-3f77-b713-d32bba55d77f|pass|200711|icerider70@gmail.com ("ICERIDER")|Devel-CheckLib|0.3|i486-linux-gnu|5.8.8|linux|2.6.18.3|200711010651
731697|731697-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200711011807
731701|731701-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200711011807
731719|731719-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckLib|0.3|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200711011714
731720|731720-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.08|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200711011714
731721|731721-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.09|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200711011715
731722|731722-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.10|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200711011715
731730|731730-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.9.4|linux|2.6.17-2-k7|200711011808
731782|731782-b19f-3f77-b713-d32bba55d77f|fail|200711|david@cantrell.org.uk|Palm-SMS|0.02|darwin-2level|5.9.5|darwin|8.10.1|200711011729
732214|732214-b19f-3f77-b713-d32bba55d77f|pass|200711|srezic@cpan.org|XML-Tiny|1.11|amd64-freebsd|5.6.2|freebsd|6.2-release|200711012140
732229|732229-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200711012145
732356|732356-b19f-3f77-b713-d32bba55d77f|pass|200711|srezic@cpan.org|XML-Tiny|1.11|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200711012145
732438|732438-b19f-3f77-b713-d32bba55d77f|pass|200711|srezic@cpan.org|XML-Tiny|1.11|amd64-freebsd|5.5.5|freebsd|6.2-release|200711012223
732646|732646-b19f-3f77-b713-d32bba55d77f|pass|200711|srezic@cpan.org|XML-Tiny|1.11|amd64-freebsd|5.10.0 patch 32200|freebsd|6.2-release|200711012233
732819|732819-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0_rc3|200711012325
732820|732820-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.08|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0_rc3|200711012325
732821|732821-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.09|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0_rc3|200711012326
732822|732822-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.10|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0_rc3|200711012326
733089|733089-b19f-3f77-b713-d32bba55d77f|pass|200711|mmusgrove@cpan.org|Devel-CheckLib|0.3|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711020241
733397|733397-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0_rc3|200711020511
733409|733409-b19f-3f77-b713-d32bba55d77f|pass|200711|mmusgrove@cpan.org|XML-Tiny|1.11|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711020026
733855|733855-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|XML-Tiny|1.11|i686-linux|5.6.2|linux|2.4.27-3-686|200711021208
733946|733946-b19f-3f77-b713-d32bba55d77f|pass|200711|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-Tiny|1.11|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200711021252
734213|734213-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200711021542
734243|734243-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|XML-Tiny|1.11|darwin-2level|5.9.5|darwin|8.10.1|200711021558
734247|734247-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|XML-Tiny|1.11|i686-linux|5.9.5|linux|2.4.27-3-686|200711021559
734259|734259-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|XML-Tiny|1.11|i386-freebsd|5.9.5|freebsd|6.2-release|200711021552
734350|734350-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckLib|0.3|MSWin32-x86-multi-thread|5.10.0 patchlevel 32202|MSWin32|5.2|200711021633
734384|734384-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|XML-Tiny|1.11|i686-linux|5.8.8|linux|2.4.27-3-686|200711021559
734440|734440-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200711021640
734442|734442-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.09|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200711021640
734443|734443-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.10|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200711021641
734445|734445-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200711021641
734649|734649-b19f-3f77-b713-d32bba55d77f|pass|200711|andy@hexten.net|Devel-CheckOS|1.2|i486-linux-gnu|5.8.8|linux|2.6.15.7|200711021844
734964|734964-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200711022109
734969|734969-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200711022109
734970|734970-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200711022110
734983|734983-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200711022111
734990|734990-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200711022111
734991|734991-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200711022111
734996|734996-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200711022112
735000|735000-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200711022110
735001|735001-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200711022110
735073|735073-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|XML-Tiny|1.11|sun4-solaris|5.9.5|solaris|2.9|200711021627
735204|735204-b19f-3f77-b713-d32bba55d77f|pass|200711|cpan@0x64.nl|XML-Tiny|1.11|x86_64-linux-gnu|5.8.8|linux|2.6.17-2-vserver-amd64|200711022008
735291|735291-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|XML-Tiny|1.11|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200711022208
735637|735637-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Data-Compare|0.02|i386-freebsd|5.9.5|freebsd|6.2-release|200711030027
735719|735719-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckLib|0.3|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200711030139
735720|735720-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.09|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200711030140
735721|735721-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.10|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200711030140
735724|735724-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200711030141
735927|735927-b19f-3f77-b713-d32bba55d77f|pass|200711|mmusgrove@cpan.org|Devel-CheckLib|0.3|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711030313
736341|736341-b19f-3f77-b713-d32bba55d77f|pass|200711|srezic@cpan.org|CPAN-FindDependencies|1.02|amd64-freebsd|5.10.0 patch 32200|freebsd|6.2-release|200711031033
736770|736770-b19f-3f77-b713-d32bba55d77f|pass|200711|alexchorny@gmail.com|Devel-CheckLib|0.3|i686-linux|5.10.0 patch 32015|linux|2.4.20-8smp|200711030412
736772|736772-b19f-3f77-b713-d32bba55d77f|pass|200711|alexchorny@gmail.com|XML-Tiny|1.11|i686-linux|5.10.0 patch 32015|linux|2.4.20-8smp|200711030412
736795|736795-b19f-3f77-b713-d32bba55d77f|pass|200711|mmusgrove@cpan.org|XML-Tiny|1.11|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711030733
737084|737084-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200711030337
737086|737086-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.09|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200711030337
737087|737087-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.10|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200711030338
737089|737089-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200711030338
737280|737280-b19f-3f77-b713-d32bba55d77f|pass|200711|imacat@mail.imacat.idv.tw|XML-Tiny|1.11|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200711032258
738907|738907-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200711040155
738908|738908-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.09|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200711040155
738909|738909-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.10|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200711040155
738910|738910-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200711040156
739569|739569-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i386-pc-solaris2.11|5.8.7|solaris|2.11|200711041250
739570|739570-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.09|i386-pc-solaris2.11|5.8.7|solaris|2.11|200711041250
739571|739571-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.10|i386-pc-solaris2.11|5.8.7|solaris|2.11|200711041251
739572|739572-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-pc-solaris2.11|5.8.7|solaris|2.11|200711041251
740713|740713-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200711050044
740714|740714-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.09|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200711050044
740715|740715-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.10|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200711050044
740716|740716-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200711050045
741522|741522-b19f-3f77-b713-d32bba55d77f|pass|200711|cpan@pqpq.de|XML-Tiny|1.11|i386-freebsd|5.10.0 patch 31806|freebsd|6.1-release-p6|200711050851
741523|741523-b19f-3f77-b713-d32bba55d77f|pass|200711|cpan@pqpq.de|XML-Tiny|1.10|i386-freebsd|5.10.0 patch 31806|freebsd|6.1-release-p6|200711050851
741524|741524-b19f-3f77-b713-d32bba55d77f|pass|200711|cpan@pqpq.de|XML-Tiny|1.09|i386-freebsd|5.10.0 patch 31806|freebsd|6.1-release-p6|200711050851
741525|741525-b19f-3f77-b713-d32bba55d77f|pass|200711|cpan@pqpq.de|Devel-CheckLib|0.3|i386-freebsd|5.10.0 patch 31806|freebsd|6.1-release-p6|200711050851
741809|741809-b19f-3f77-b713-d32bba55d77f|pass|200711|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200711051058
742656|742656-b19f-3f77-b713-d32bba55d77f|pass|200711|andy@hexten.net|Data-Hexdumper|1.3|darwin-2level|5.9.5|darwin|9.0.0|200711051831
743026|743026-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.10|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200711052057
743028|743028-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200711052058
743035|743035-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.09|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200711052057
743052|743052-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200711052056
743914|743914-b19f-3f77-b713-d32bba55d77f|pass|200711|mmusgrove@cpan.org ("Matthew Musgrove")|Data-Compare|0.17|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711060535
745365|745365-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Data-Compare|0.17|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200711061836
746713|746713-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200711070246
746714|746714-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200711070246
747954|747954-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.10.0 patch 32202|linux|2.6.16.38|200711071243
748739|748739-b19f-3f77-b713-d32bba55d77f|pass|200711|andy@hexten.net|Data-Compare|0.17|i486-linux-gnu|5.8.8|linux|2.6.15.7|200711071835
749141|749141-b19f-3f77-b713-d32bba55d77f|pass|200711|bob@randomness.org.uk ("bob walker")|Devel-CheckOS|1.2|i86pc-solaris|5.8.8|solaris|2.10|200711072025
749292|749292-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.3|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200711072209
749304|749304-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.3|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200711072211
749306|749306-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.3|i686-linux-64int|5.9.4|linux|2.6.17-2-k7|200711072212
749311|749311-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.3|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200711072214
749312|749312-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.3|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200711072215
749316|749316-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.3|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200711072216
749324|749324-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.3|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200711072218
749330|749330-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.3|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200711072219
749605|749605-b19f-3f77-b713-d32bba55d77f|pass|200711|bob@randomness.org.uk ("bob walker")|CPAN-FindDependencies|1.02|i86pc-solaris|5.8.8|solaris|2.10|200711072344
749902|749902-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.41|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200711080739
749905|749905-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.41|i686-linux-64int|5.9.4|linux|2.6.17-2-k7|200711080739
749907|749907-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.41|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200711080740
749909|749909-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.41|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200711080741
749910|749910-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.41|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200711080741
749912|749912-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.41|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200711080742
749914|749914-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.41|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200711080743
749917|749917-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200711080643
749954|749954-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.41|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200711080738
750004|750004-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.1|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200711080815
750021|750021-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.1|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200711080824
750025|750025-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.1|i686-linux-64int|5.9.4|linux|2.6.17-2-k7|200711080830
750026|750026-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.1|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200711080834
750042|750042-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.1|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200711080840
750060|750060-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.1|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200711080848
750106|750106-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.1|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200711080859
750197|750197-b19f-3f77-b713-d32bba55d77f|pass|200711|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPAN-FindDependencies|1.1|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200711080817
750202|750202-b19f-3f77-b713-d32bba55d77f|pass|200711|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckOS|1.41|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200711080818
750450|750450-b19f-3f77-b713-d32bba55d77f|pass|200711|bob@randomness.org.uk ("bob walker")|CPAN-FindDependencies|1.1|i86pc-solaris|5.8.8|solaris|2.10|200711081027
751024|751024-b19f-3f77-b713-d32bba55d77f|pass|200711|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200711081552
751025|751025-b19f-3f77-b713-d32bba55d77f|pass|200711|andy@hexten.net|Data-Hexdumper|1.3|darwin-thread-multi-2level|5.10.0 patch 32245|darwin|9.0.0|200711081554
751606|751606-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.3|darwin-2level|5.9.5|darwin|8.10.1|200711081833
751640|751640-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.3|i686-linux|5.9.5|linux|2.4.27-3-686|200711081842
751643|751643-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.3|i686-linux|5.8.8|linux|2.4.27-3-686|200711081843
751647|751647-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.3|i686-linux|5.6.2|linux|2.4.27-3-686|200711081843
751663|751663-b19f-3f77-b713-d32bba55d77f|pass|200711|srezic@cpan.org|CPAN-FindDependencies|1.1|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200711081939
751724|751724-b19f-3f77-b713-d32bba55d77f|pass|200711|srezic@cpan.org|Devel-CheckOS|1.41|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200711081941
751804|751804-b19f-3f77-b713-d32bba55d77f|pass|200711|srezic@cpan.org|Devel-CheckOS|1.41|amd64-freebsd|5.10.0 patch 32200|freebsd|6.2-release|200711082040
751901|751901-b19f-3f77-b713-d32bba55d77f|pass|200711|srezic@cpan.org|CPAN-FindDependencies|1.1|amd64-freebsd|5.10.0 patch 32200|freebsd|6.2-release|200711082039
752045|752045-b19f-3f77-b713-d32bba55d77f|pass|200711|srezic@cpan.org|Devel-CheckOS|1.41|amd64-freebsd|5.6.2|freebsd|6.2-release|200711082125
752282|752282-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.4|darwin-2level|5.9.5|darwin|8.10.1|200711090012
752289|752289-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.41|darwin-2level|5.9.5|darwin|8.10.1|200711090014
752314|752314-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200711090031
752315|752315-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200711090032
752317|752317-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200711090033
752318|752318-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200711090034
752322|752322-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.4|i686-linux|5.8.8|linux|2.4.27-3-686|200711090038
752326|752326-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.4|i686-linux|5.9.5|linux|2.4.27-3-686|200711090039
752338|752338-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.41|i686-linux|5.8.8|linux|2.4.27-3-686|200711090045
752342|752342-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.41|i686-linux|5.9.5|linux|2.4.27-3-686|200711090046
752367|752367-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.4|i686-linux|5.6.2|linux|2.4.27-3-686|200711090058
752369|752369-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.41|i686-linux|5.6.2|linux|2.4.27-3-686|200711090100
752699|752699-b19f-3f77-b713-d32bba55d77f|fail|200711|mmusgrove@cpan.org ("Matthew Musgrove")|Devel-CheckOS|1.3|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711090104
752721|752721-b19f-3f77-b713-d32bba55d77f|fail|200711|mmusgrove@cpan.org ("Matthew Musgrove")|Devel-CheckOS|1.4|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711090112
752722|752722-b19f-3f77-b713-d32bba55d77f|fail|200711|mmusgrove@cpan.org ("Matthew Musgrove")|Devel-CheckOS|1.41|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711090113
752727|752727-b19f-3f77-b713-d32bba55d77f|pass|200711|mmusgrove@cpan.org ("Matthew Musgrove")|CPAN-FindDependencies|1.1|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711090120
753143|753143-b19f-3f77-b713-d32bba55d77f|fail|200711|mmusgrove@cpan.org|Devel-CheckOS|1.3|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711090606
753276|753276-b19f-3f77-b713-d32bba55d77f|na|200711|mmusgrove@cpan.org|Devel-CheckOS|1.4|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711090711
753277|753277-b19f-3f77-b713-d32bba55d77f|fail|200711|mmusgrove@cpan.org|Devel-CheckOS|1.41|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711090713
753292|753292-b19f-3f77-b713-d32bba55d77f|pass|200711|mmusgrove@cpan.org|CPAN-FindDependencies|1.1|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711090726
754375|754375-b19f-3f77-b713-d32bba55d77f|pass|200711|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.3|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200711100554
754643|754643-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200711100042
754646|754646-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200711100044
754648|754648-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200711100046
754981|754981-b19f-3f77-b713-d32bba55d77f|pass|200711|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.41|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200711101304
754998|754998-b19f-3f77-b713-d32bba55d77f|pass|200711|imacat@mail.imacat.idv.tw|CPAN-FindDependencies|1.1|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200711101311
755487|755487-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.3|sun4-solaris|5.9.5|solaris|2.9|200711100348
755495|755495-b19f-3f77-b713-d32bba55d77f|fail|200711|david@cantrell.org.uk|Devel-CheckOS|1.3|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200711100951
755563|755563-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0_rc3|200711101013
755568|755568-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0_rc3|200711101014
755573|755573-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0_rc3|200711101015
758030|758030-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|darwin-thread-multi-2level|5.8.1|darwin|8.10.0|200711110901
758033|758033-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|darwin-thread-multi-2level|5.8.2|darwin|8.10.0|200711110907
758036|758036-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|darwin-thread-multi-2level|5.8.3|darwin|8.10.0|200711110912
758039|758039-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|darwin-thread-multi-2level|5.8.4|darwin|8.10.0|200711110918
758050|758050-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|darwin-thread-multi-2level|5.8.5|darwin|8.10.0|200711110923
758056|758056-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|darwin-thread-multi-2level|5.8.6|darwin|8.10.0|200711110928
758060|758060-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|darwin-thread-multi-2level|5.8.7|darwin|8.10.0|200711110932
758061|758061-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|darwin-thread-multi-2level|5.8.8|darwin|8.10.0|200711110937
758062|758062-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|darwin-thread-multi|5.6.2|darwin|8.10.0|200711110938
758063|758063-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|darwin-thread-multi-2level|5.8.1|darwin|8.10.0|200711110940
758065|758065-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|darwin-thread-multi-2level|5.8.2|darwin|8.10.0|200711110941
758066|758066-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|darwin-thread-multi-2level|5.8.3|darwin|8.10.0|200711110942
758068|758068-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|darwin-thread-multi-2level|5.8.4|darwin|8.10.0|200711110943
758070|758070-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|darwin-thread-multi-2level|5.8.5|darwin|8.10.0|200711110944
758072|758072-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|darwin-thread-multi-2level|5.8.6|darwin|8.10.0|200711110946
758073|758073-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|darwin-thread-multi-2level|5.8.7|darwin|8.10.0|200711110947
758074|758074-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|darwin-thread-multi-2level|5.8.8|darwin|8.10.0|200711110948
758077|758077-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|darwin-thread-multi|5.6.2|darwin|8.10.0|200711110952
758078|758078-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|darwin-thread-multi-2level|5.8.1|darwin|8.10.0|200711110953
758081|758081-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|darwin-thread-multi-2level|5.8.2|darwin|8.10.0|200711110955
758082|758082-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|darwin-thread-multi-2level|5.8.3|darwin|8.10.0|200711110956
758083|758083-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|darwin-thread-multi-2level|5.8.4|darwin|8.10.0|200711110958
758084|758084-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|darwin-thread-multi-2level|5.8.5|darwin|8.10.0|200711110959
758085|758085-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|darwin-thread-multi-2level|5.8.6|darwin|8.10.0|200711111000
758087|758087-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|darwin-thread-multi-2level|5.8.7|darwin|8.10.0|200711111001
758088|758088-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|darwin-thread-multi-2level|5.8.8|darwin|8.10.0|200711111002
758230|758230-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200711111148
758232|758232-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200711111149
758233|758233-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200711111149
758235|758235-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200711111151
758239|758239-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200711111152
758242|758242-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200711111153
758243|758243-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200711111154
758246|758246-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200711111155
758247|758247-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200711111156
758365|758365-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200711111346
758366|758366-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200711111347
758367|758367-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200711111348
758725|758725-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200711111840
758726|758726-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200711111840
758727|758727-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200711111841
758728|758728-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200711111841
758729|758729-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200711111842
758730|758730-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200711111842
758731|758731-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200711111843
758732|758732-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200711111844
758735|758735-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200711111844
758765|758765-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200711111859
758779|758779-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200711111903
758783|758783-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200711111907
758798|758798-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200711111911
758810|758810-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200711111914
758828|758828-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200711111918
758842|758842-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200711111922
758852|758852-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200711111925
758997|758997-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-64int|5.8.8 patch 32273|linux|2.6.22-1-k7|200711112053
759211|759211-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 32273|linux|2.6.22-1-k7|200711112249
759785|759785-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.41|sun4-solaris|5.9.5|solaris|2.9|200711120107
759793|759793-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.41|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200711120012
759916|759916-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 32273|linux|2.6.22-1-k7|200711120108
760130|760130-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux-64int|5.8.8 patch 32273|linux|2.6.22-1-k7|200711120257
761118|761118-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200711120944
761122|761122-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200711120944
761123|761123-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200711120945
761727|761727-b19f-3f77-b713-d32bba55d77f|pass|200711|Jost.Krieger@ruhr-uni-bochum.de|Devel-CheckOS|1.41|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200711121152
761946|761946-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200711121258
761955|761955-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200711121300
761962|761962-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200711121301
762712|762712-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.10.0 patch 32280|netbsd|3.1|200711121713
764083|764083-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.41|i386-freebsd|5.9.5|freebsd|6.2-release|200711122250
764588|764588-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200711130237
764591|764591-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200711130237
764854|764854-b19f-3f77-b713-d32bba55d77f|pass|200711|icerider70@gmail.com ("ICERIDER")|Data-Compare|0.17|i486-linux-gnu|5.8.8|linux|2.6.22-3-amd64|200711130039
765192|765192-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|i386-pc-solaris2.11|5.8.7|solaris|2.11|200711131007
765193|765193-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i386-pc-solaris2.11|5.8.7|solaris|2.11|200711131008
765194|765194-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|i386-pc-solaris2.11|5.8.7|solaris|2.11|200711131009
765970|765970-b19f-3f77-b713-d32bba55d77f|pass|200711|andy@hexten.net|Data-Hexdumper|1.3|i386-linux|5.8.5|linux|2.6.9-42.elsmp|200711131828
766337|766337-b19f-3f77-b713-d32bba55d77f|pass|200711|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|amd64-freebsd|5.10.0 patch 32200|freebsd|6.2-release|200711132151
766737|766737-b19f-3f77-b713-d32bba55d77f|pass|200711|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200711140001
766767|766767-b19f-3f77-b713-d32bba55d77f|pass|200711|icerider70@gmail.com ("ICERIDER")|XML-DoubleEncodedEntities|1.0|i486-linux-gnu|5.8.8|linux|2.6.22-3-amd64|200711140002
767072|767072-b19f-3f77-b713-d32bba55d77f|pass|200711|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|amd64-freebsd|5.6.2|freebsd|6.2-release|200711140233
767176|767176-b19f-3f77-b713-d32bba55d77f|pass|200711|bob@randomness.org.uk ("bob walker")|XML-Tiny|1.11|i86pc-solaris|5.8.8|solaris|2.10|200711140157
768950|768950-b19f-3f77-b713-d32bba55d77f|pass|200711|andy@hexten.net|Data-Hexdumper|1.3|i486-linux-gnu|5.8.8|linux|2.6.22-3-amd64|200711142150
769960|769960-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200711150217
769974|769974-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200711150218
770001|770001-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200711150224
770227|770227-b19f-3f77-b713-d32bba55d77f|pass|200711|andy@hexten.net|Data-Hexdumper|1.3|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711150336
771720|771720-b19f-3f77-b713-d32bba55d77f|pass|200711|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200711151421
773882|773882-b19f-3f77-b713-d32bba55d77f|pass|200711|andy@hexten.net|Devel-CheckOS|1.41|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200711160111
774066|774066-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Data-Compare|0.17|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200711160119
775025|775025-b19f-3f77-b713-d32bba55d77f|pass|200711|andy@hexten.net|Devel-CheckOS|1.41|i486-linux-gnu|5.8.8|linux|2.6.15.7|200711161355
776151|776151-b19f-3f77-b713-d32bba55d77f|pass|200711|andy@hexten.net|Data-Hexdumper|1.3|i686-linux-thread-multi|5.10.0 patch 32330|linux|2.6.22-14-generic|200711161734
779725|779725-b19f-3f77-b713-d32bba55d77f|pass|200711|srezic@cpan.org|CPAN-FindDependencies|1.1|amd64-freebsd|5.10.0 patch 32367|freebsd|6.2-release|200711172259
779882|779882-b19f-3f77-b713-d32bba55d77f|pass|200711|mmusgrove@cpan.org ("Matthew Musgrove")|XML-DoubleEncodedEntities|1.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711171714
780203|780203-b19f-3f77-b713-d32bba55d77f|pass|200711|srezic@cpan.org|Data-Compare|0.17|amd64-freebsd|5.10.0 patch 32367|freebsd|6.2-release|200711180053
780237|780237-b19f-3f77-b713-d32bba55d77f|pass|200711|mmusgrove@cpan.org|XML-DoubleEncodedEntities|1.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711171759
782724|782724-b19f-3f77-b713-d32bba55d77f|pass|200711|srezic@cpan.org|CPAN-FindDependencies|1.1|amd64-freebsd-thread-multi|5.10.0 patch 32367|freebsd|6.2-release|200711181939
783177|783177-b19f-3f77-b713-d32bba55d77f|pass|200711|srezic@cpan.org|Data-Compare|0.17|amd64-freebsd-thread-multi|5.10.0 patch 32367|freebsd|6.2-release|200711182121
783899|783899-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Data-Compare|0.17|i686-linux-thread-multi-64int|5.10.0|linux|2.6.16.38|200711190054
784863|784863-b19f-3f77-b713-d32bba55d77f|pass|200711|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200711191100
790887|790887-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200711202149
790891|790891-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200711202150
790893|790893-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200711202150
790896|790896-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200711202151
790898|790898-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200711202151
790899|790899-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200711202152
790900|790900-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200711202153
791726|791726-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-64int|5.8.8 patch 32389|linux|2.6.22-1-k7|200711210525
791728|791728-b19f-3f77-b713-d32bba55d77f|pass|200711|icerider70@gmail.com ("ICERIDER")|Devel-CheckOS|1.42|i486-linux-gnu|5.8.8|linux|2.6.22-3-amd64|200711210436
792257|792257-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200711211005
792263|792263-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200711211005
792268|792268-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200711211006
792275|792275-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200711211006
792278|792278-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200711211007
792281|792281-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200711211008
792287|792287-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200711211008
792289|792289-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200711211009
792294|792294-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200711211009
792332|792332-b19f-3f77-b713-d32bba55d77f|pass|200711|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckOS|1.42|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200711211015
792411|792411-b19f-3f77-b713-d32bba55d77f|pass|200711|srezic@cpan.org|Devel-CheckOS|1.42|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200711211155
792711|792711-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200711211315
792839|792839-b19f-3f77-b713-d32bba55d77f|pass|200711|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200711211437
793079|793079-b19f-3f77-b713-d32bba55d77f|pass|200711|Jost.Krieger@ruhr-uni-bochum.de|Devel-CheckOS|1.42|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200711211551
793685|793685-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0_rc3|200711211914
793833|793833-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.42|darwin-2level|5.9.5|darwin|8.10.1|200711211958
793845|793845-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.42|i686-linux|5.8.8|linux|2.4.27-3-686|200711212002
793890|793890-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.42|i686-linux|5.6.2|linux|2.4.27-3-686|200711212013
793933|793933-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.42|i686-linux|5.9.5|linux|2.4.27-3-686|200711212018
793969|793969-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.42|i386-freebsd|5.9.5|freebsd|6.2-release|200711212013
794720|794720-b19f-3f77-b713-d32bba55d77f|fail|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.0|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200711220351
794722|794722-b19f-3f77-b713-d32bba55d77f|fail|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.0|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200711220351
794724|794724-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.0|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200711220352
794732|794732-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.0|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200711220354
794757|794757-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.0|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200711220352
794758|794758-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.0|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200711220353
794759|794759-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.0|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200711220353
794883|794883-b19f-3f77-b713-d32bba55d77f|fail|200711|srezic@cpan.org|Bryar|3.0|amd64-freebsd|5.10.0 patch 32367|freebsd|6.2-release|200711220756
794928|794928-b19f-3f77-b713-d32bba55d77f|na|200711|srezic@cpan.org|Bryar|3.0|amd64-freebsd|5.5.5|freebsd|6.2-release|200711220815
794967|794967-b19f-3f77-b713-d32bba55d77f|pass|200711|srezic@cpan.org|Bryar|3.0|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200711220802
794968|794968-b19f-3f77-b713-d32bba55d77f|pass|200711|srezic@cpan.org|Bryar|3.0|amd64-freebsd|5.6.2|freebsd|6.2-release|200711220809
795134|795134-b19f-3f77-b713-d32bba55d77f|fail|200711|srezic@cpan.org|Bryar|3.0|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200711221015
796071|796071-b19f-3f77-b713-d32bba55d77f|fail|200711|icerider70@gmail.com ("ICERIDER")|Bryar|3.0|i486-linux-gnu|5.8.8|linux|2.6.22-3-amd64|200711220952
796188|796188-b19f-3f77-b713-d32bba55d77f|fail|200711|icerider70@gmail.com ("ICERIDER")|Bryar|3.0|i486-linux-gnu|5.8.8|linux|2.6.22-3-amd64|200711220952
796262|796262-b19f-3f77-b713-d32bba55d77f|pass|200711|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.42|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200711230041
796506|796506-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Bryar|3.0|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200711221727
796514|796514-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Bryar|3.0|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200711221732
796522|796522-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Bryar|3.0|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200711221737
796553|796553-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Bryar|3.0|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200711221742
796572|796572-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Bryar|3.0|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200711221748
796574|796574-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Bryar|3.0|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200711221753
796578|796578-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Bryar|3.0|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200711221758
796586|796586-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Bryar|3.0|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200711221803
796609|796609-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0_rc3|200711221805
796624|796624-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Bryar|3.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200711221809
796632|796632-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.42|sun4-solaris|5.9.5|solaris|2.9|200711221409
796633|796633-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.42|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200711222008
796792|796792-b19f-3f77-b713-d32bba55d77f|pass|200711|taro-nishino@sam.hi-ho.ne.jp (Taro Nishino)|Devel-CheckOS|1.42|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200711230101
796943|796943-b19f-3f77-b713-d32bba55d77f|pass|200711|taro-nishino@sam.hi-ho.ne.jp (Taro Nishino)|Devel-CheckOS|1.42|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200711230101
797187|797187-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Class-CanBeA|1.2|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230002
797200|797200-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|CPAN-FindDependencies|1.1|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230006
797203|797203-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Data-Compare|0.17|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230006
797204|797204-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Data-Transactional|1.0|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230007
797205|797205-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Data-Hexdumper|1.3|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230006
797207|797207-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Devel-CheckLib|0.3|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230007
797208|797208-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.42|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230007
797226|797226-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|File-Overwrite|1.0|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230021
797227|797227-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Games-Dice-Advanced|1.1|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230021
797228|797228-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|NestedMap|1.0|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230022
797240|797240-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Net-Random|2.0|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230026
797246|797246-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Number-Phone|1.58|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230028
797251|797251-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230029
797265|797265-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230031
797266|797266-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230031
797268|797268-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Tie-STDOUT|1.03|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230031
797269|797269-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230031
797270|797270-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230031
797271|797271-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|XML-Tiny|1.11|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230032
797770|797770-b19f-3f77-b713-d32bba55d77f|fail|200711|chris@bingosnet.co.uk|Bryar|3.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0_rc3|200711230741
798215|798215-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Bryar|3.0|i686-linux|5.8.8|linux|2.4.27-3-686|200711231110
798251|798251-b19f-3f77-b713-d32bba55d77f|fail|200711|david@cantrell.org.uk|Bryar|3.0|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711231131
798475|798475-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200711231503
799278|799278-b19f-3f77-b713-d32bba55d77f|fail|200711|david@cantrell.org.uk|Bryar|3.0|sun4-solaris|5.9.5|solaris|2.9|200711231241
799280|799280-b19f-3f77-b713-d32bba55d77f|fail|200711|david@cantrell.org.uk|Bryar|3.0|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200711231841
799737|799737-b19f-3f77-b713-d32bba55d77f|fail|200711|david@cantrell.org.uk|Bryar|3.0|darwin-thread-multi-2level|5.10.0 patch 32468|darwin|8.10.1|200711232131
799879|799879-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200711232234
800041|800041-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|CPAN-FindDependencies|1.1|sun4-solaris-thread-multi|5.10.0 patch 32468|solaris|2.9|200711231702
800120|800120-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.42|sun4-solaris-thread-multi|5.10.0 patch 32468|solaris|2.9|200711231717
800122|800122-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|File-Find-Rule-Permissions|1.3|sun4-solaris-thread-multi|5.10.0 patch 32468|solaris|2.9|200711231717
800244|800244-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200711240055
800714|800714-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Bryar|3.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200711240531
802110|802110-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Bryar|3.0|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200711241858
802164|802164-b19f-3f77-b713-d32bba55d77f|pass|200711|andy@hexten.net|Data-Hexdumper|1.3|i686-linux-thread-multi|5.10.0 patch 32478|linux|2.6.18-5-686|200711241915
804772|804772-b19f-3f77-b713-d32bba55d77f|pass|200711|andy@hexten.net|Data-Hexdumper|1.3|i386-freebsd-thread-multi|5.10.0 patch 32487|freebsd|6.2-release|200711251905
805243|805243-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-pc-solaris2.11|5.8.7|solaris|2.11|200711252121
805468|805468-b19f-3f77-b713-d32bba55d77f|pass|200711|andy@hexten.net|Data-Hexdumper|1.3|ppc-linux-thread-multi|5.10.0 patch 32376|linux|2.6.20-16-powerpc|200711252222
805729|805729-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Tie-STDOUT|1.03|alpha-netbsd|5.10.0 patch 31856|netbsd|2.1.0_stable|200711260042
805754|805754-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|alpha-netbsd|5.10.0 patch 31856|netbsd|2.1.0_stable|200711260050
805787|805787-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Net-Random|2.0|alpha-netbsd|5.10.0 patch 31856|netbsd|2.1.0_stable|200711260103
806233|806233-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Bryar|3.0|i386-pc-solaris2.11|5.8.7|solaris|2.11|200711260744
807683|807683-b19f-3f77-b713-d32bba55d77f|pass|200711|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200711261548
807867|807867-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200711261640
807929|807929-b19f-3f77-b713-d32bba55d77f|pass|200711|stro@cpan.org|Devel-CheckOS|1.42|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200711261758
809202|809202-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Data-Hexdumper|1.3|darwin-thread-multi-2level|5.10.0 patch 32468|darwin|8.10.1|200711262249
810317|810317-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Bryar|3.0|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200711270828
811314|811314-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200711271352
811999|811999-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200711271804
813087|813087-b19f-3f77-b713-d32bba55d77f|pass|200711|andy@hexten.net|Data-Hexdumper|1.3|i486-linux-gnu|5.8.8|linux|2.6.15.7|200711280142
813495|813495-b19f-3f77-b713-d32bba55d77f|fail|200711|chris@bingosnet.co.uk|Bryar|3.0|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200711280649
813940|813940-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux-thread-multi|5.10.0|linux|2.6.22-1-k7|200711281215
814046|814046-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-thread-multi|5.10.0|linux|2.6.22-1-k7|200711281320
814406|814406-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200711281443
814407|814407-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|darwin-thread-multi|5.6.2|darwin|8.10.0|200711281444
814413|814413-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|darwin-thread-multi-2level|5.8.1|darwin|8.10.0|200711281445
814415|814415-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|darwin-thread-multi-2level|5.8.2|darwin|8.10.0|200711281446
814417|814417-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|darwin-thread-multi-2level|5.8.3|darwin|8.10.0|200711281447
814420|814420-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|darwin-thread-multi-2level|5.8.4|darwin|8.10.0|200711281448
814424|814424-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|darwin-thread-multi-2level|5.8.5|darwin|8.10.0|200711281449
814427|814427-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|darwin-thread-multi-2level|5.8.6|darwin|8.10.0|200711281450
814429|814429-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|darwin-thread-multi-2level|5.8.7|darwin|8.10.0|200711281451
814433|814433-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|darwin-thread-multi-2level|5.8.8|darwin|8.10.0|200711281452
815957|815957-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi|5.10.0|linux|2.6.22-1-k7|200711290131
816430|816430-b19f-3f77-b713-d32bba55d77f|fail|200711|chris@bingosnet.co.uk|Bryar|3.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200711290454
818090|818090-b19f-3f77-b713-d32bba55d77f|pass|200711|srezic@cpan.org|Data-Compare|0.17|i386-freebsd|5.10.0 patch 32478|freebsd|6.1-release|200711291851
818305|818305-b19f-3f77-b713-d32bba55d77f|pass|200711|srezic@cpan.org|Data-Compare|0.17|i386-freebsd|5.6.2|freebsd|6.1-release|200711292028
819571|819571-b19f-3f77-b713-d32bba55d77f|pass|200711|alexchorny@gmail.com|Data-Hexdumper|1.3|i686-linux|5.10.0 patch 32015|linux|2.4.20-8smp|200711300129
819990|819990-b19f-3f77-b713-d32bba55d77f|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.2|200711301328
820335|820335-b19f-3f77-b713-d32bba55d77f|pass|200711|alexchorny@gmail.com|Data-Compare|0.17|i686-linux|5.10.0 patch 32015|linux|2.4.20-8smp|200711300740
820981|820981-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux|5.10.0|linux|2.6.22-1-k7|200711302032
821716|821716-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux|5.10.0|linux|2.6.22-1-k7|200711302146
821881|821881-b19f-3f77-b713-d32bba55d77f|pass|200711|david@cantrell.org.uk|Data-Compare|0.17|darwin-thread-multi-2level|5.10.0 patch 32468|darwin|8.10.1|200711302135
822556|822556-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.10.0|linux|2.6.22-1-k7|200711302357
822812|822812-b19f-3f77-b713-d32bba55d77f|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux|5.10.0|linux|2.6.22-1-k7|200711302254
823787|823787-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200712010557
825915|825915-b19f-3f77-b713-d32bba55d77f|pass|200712|david@cantrell.org.uk|Data-Compare|0.17|alpha-netbsd|5.10.0 patch 31856|netbsd|2.1.0_stable|200712012335
825953|825953-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200712020017
826485|826485-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0_rc3|200712020732
828013|828013-b19f-3f77-b713-d32bba55d77f|pass|200712|david@cantrell.org.uk|Data-Compare|0.17|sun4-solaris-thread-multi|5.10.0 patch 32468|solaris|2.9|200712021707
828319|828319-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0_rc3|200712030142
829840|829840-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.99_01|i686-linux|5.10.0|linux|2.6.22-1-k7|200712031905
829850|829850-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.99_01|i686-linux-thread-multi|5.10.0|linux|2.6.22-1-k7|200712031910
829873|829873-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.99_01|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200712031916
829878|829878-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.99_01|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200712031921
829881|829881-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.99_01|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200712031926
829889|829889-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.99_01|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200712031932
829908|829908-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.99_01|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200712031938
829926|829926-b19f-3f77-b713-d32bba55d77f|pass|200712|srezic@cpan.org|Data-Compare|0.17|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200712031942
830133|830133-b19f-3f77-b713-d32bba55d77f|pass|200712|srezic@cpan.org|CPAN-FindDependencies|1.99_01|amd64-freebsd|5.10.0 patch 32559|freebsd|6.2-release|200712032137
830134|830134-b19f-3f77-b713-d32bba55d77f|pass|200712|srezic@cpan.org|CPAN-FindDependencies|1.99_01|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200712032135
830173|830173-b19f-3f77-b713-d32bba55d77f|pass|200712|xdg@hyperbolic.net (DAGOLDEN)|CPAN-FindDependencies|1.99_01|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200712031555
830197|830197-b19f-3f77-b713-d32bba55d77f|pass|200712|srezic@cpan.org|CPAN-FindDependencies|1.99_01|amd64-freebsd-thread-multi|5.10.0 patch 32367|freebsd|6.2-release|200712032138
831557|831557-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200712040554
831803|831803-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200712040900
831868|831868-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200712040923
831873|831873-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200712040927
831878|831878-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200712040930
831888|831888-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200712040934
831893|831893-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200712040938
831897|831897-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200712040942
831902|831902-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200712040946
831907|831907-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200712040949
831993|831993-b19f-3f77-b713-d32bba55d77f|pass|200712|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPAN-FindDependencies|1.99_01|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200712041045
832037|832037-b19f-3f77-b713-d32bba55d77f|pass|200712|srezic@cpan.org|Devel-CheckOS|1.42|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200712041214
832173|832173-b19f-3f77-b713-d32bba55d77f|pass|200712|icerider70@gmail.com ("ICERIDER")|CPAN-FindDependencies|1.99_01|i486-linux-gnu|5.8.8|linux|2.6.22-3-amd64|200712040644
832268|832268-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200712041351
832504|832504-b19f-3f77-b713-d32bba55d77f|pass|200712|srezic@cpan.org|CPAN-FindDependencies|1.99_01|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200712041628
832963|832963-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200712040744
834171|834171-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200712050347
834470|834470-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200712050739
834935|834935-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200712050000
835613|835613-b19f-3f77-b713-d32bba55d77f|pass|200712|imacat@mail.imacat.idv.tw|CPAN-FindDependencies|1.99_01|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200712060104
835862|835862-b19f-3f77-b713-d32bba55d77f|pass|200712|srezic@cpan.org|CPAN-FindDependencies|1.99_01|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200712051931
836161|836161-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200712050931
836650|836650-b19f-3f77-b713-d32bba55d77f|pass|200712|david@cantrell.org.uk|Devel-CheckOS|1.42|darwin-thread-multi-2level|5.10.0 patch 32468|darwin|8.10.1|200712052301
836743|836743-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200712051300
837303|837303-b19f-3f77-b713-d32bba55d77f|pass|200712|srezic@cpan.org|Data-Compare|0.17|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200712061010
837491|837491-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i386-pc-solaris2.11|5.8.7|solaris|2.11|200712061102
838315|838315-b19f-3f77-b713-d32bba55d77f|pass|200712|dillman@ria.buffalo.edu|Devel-CheckOS|1.42|sun4-solaris|5.8.8|solaris|2.9|200712061058
838624|838624-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|i386-pc-solaris2.11-thread-multi-64int|5.10.0|solaris|2.11|200712062244
838827|838827-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i386-pc-solaris2.11-thread-multi-64int|5.10.0|solaris|2.11|200712070122
839373|839373-b19f-3f77-b713-d32bba55d77f|pass|200712|notbenh@cpan.org|Devel-CheckOS|1.42|i486-linux-gnu|5.8.8|linux|2.6.15.7|200712061109
839420|839420-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-pc-solaris2.11-thread-multi-64int|5.10.0|solaris|2.11|200712070844
840134|840134-b19f-3f77-b713-d32bba55d77f|pass|200712|andy@hexten.net|Devel-CheckOS|1.42|darwin-thread-multi-2level|5.10.0 patch 32560|darwin|9.1.0|200712071523
840932|840932-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200712072327
840955|840955-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|darwin-thread-multi-2level|5.8.1|darwin|8.10.0|200712072337
840967|840967-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|darwin-thread-multi-2level|5.8.2|darwin|8.10.0|200712072342
840982|840982-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|darwin-thread-multi-2level|5.8.3|darwin|8.10.0|200712072348
841002|841002-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|darwin-thread-multi-2level|5.8.7|darwin|8.10.0|200712080006
841011|841011-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|darwin-thread-multi-2level|5.8.8|darwin|8.10.0|200712080012
841362|841362-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200712080341
841879|841879-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200712080232
841964|841964-b19f-3f77-b713-d32bba55d77f|pass|200712|g.grigelionis@computer.org|Devel-CheckOS|1.42|i586-linux-thread-multi|5.8.8|linux|2.6.16|200712080029
842674|842674-b19f-3f77-b713-d32bba55d77f|pass|200712|david@cantrell.org.uk|Devel-CheckOS|1.42|alpha-netbsd|5.10.0 patch 31856|netbsd|2.1.0_stable|200712082045
842797|842797-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200712082257
843205|843205-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200712090321
843500|843500-b19f-3f77-b713-d32bba55d77f|pass|200712|cpan@pqpq.de|Data-Compare|0.17|amd64-freebsd|5.10.0 patch 32559|freebsd|6.2-release|200712090923
843671|843671-b19f-3f77-b713-d32bba55d77f|pass|200712|cpan@pqpq.de|CPAN-FindDependencies|1.99_01|amd64-freebsd|5.10.0 patch 32559|freebsd|6.2-release|200712091035
844619|844619-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200712091711
846267|846267-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200712100820
846955|846955-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200712101233
847107|847107-b19f-3f77-b713-d32bba55d77f|pass|200712|d@0x64.nl|Devel-CheckOS|1.42|x86_64-linux-gnu|5.8.8|linux|2.6.17-2-vserver-amd64|200712101305
847826|847826-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200712101833
848586|848586-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200712110203
848851|848851-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200712110636
849160|849160-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200712111048
849338|849338-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200712111202
850973|850973-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200712112231
852048|852048-b19f-3f77-b713-d32bba55d77f|pass|200712|paranoid@sarpedon.site5.com ((Mike Bissett))|Data-Compare|0.17|x86_64-linux-gnu|5.8.8|linux|2.6.15.7|200712121210
852788|852788-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200712121443
852849|852849-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200712120434
854890|854890-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200712131119
855361|855361-b19f-3f77-b713-d32bba55d77f|pass|200712|rlane10@wi.rr.com|Devel-CheckOS|1.42|darwin-thread-multi-2level|5.8.8|darwin|8.9.0|200712130738
857132|857132-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.0|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200712141556
857136|857136-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.0|i686-linux|5.10.0|linux|2.6.22-1-k7|200712141557
857139|857139-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.0|i686-linux-thread-multi|5.10.0|linux|2.6.22-1-k7|200712141558
857142|857142-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.0|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200712141559
857143|857143-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.0|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200712141600
857146|857146-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.0|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200712141601
857148|857148-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.0|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200712141602
857151|857151-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.0|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200712141604
857197|857197-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200712141517
857215|857215-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|darwin-thread-multi-2level|5.8.1|darwin|8.10.0|200712141525
857223|857223-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|darwin-thread-multi-2level|5.8.2|darwin|8.10.0|200712141532
857243|857243-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|darwin-thread-multi-2level|5.8.3|darwin|8.10.0|200712141540
857270|857270-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|darwin-thread-multi-2level|5.8.4|darwin|8.10.0|200712141547
857285|857285-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|darwin-thread-multi-2level|5.8.5|darwin|8.10.0|200712141555
857307|857307-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|darwin-thread-multi-2level|5.8.6|darwin|8.10.0|200712141603
857326|857326-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|darwin-thread-multi-2level|5.8.7|darwin|8.10.0|200712141611
857348|857348-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|darwin-thread-multi-2level|5.8.8|darwin|8.10.0|200712141620
857425|857425-b19f-3f77-b713-d32bba55d77f|pass|200712|srezic@cpan.org|CPAN-FindDependencies|2.0|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200712141822
857789|857789-b19f-3f77-b713-d32bba55d77f|pass|200712|srezic@cpan.org|CPAN-FindDependencies|2.0|amd64-freebsd|5.10.0 patch 32559|freebsd|6.2-release|200712142121
857887|857887-b19f-3f77-b713-d32bba55d77f|pass|200712|icerider70@gmail.com ("ICERIDER")|CPAN-FindDependencies|2.0|i486-linux-gnu|5.8.8|linux|2.6.22-3-amd64|200712141449
857999|857999-b19f-3f77-b713-d32bba55d77f|pass|200712|srezic@cpan.org|CPAN-FindDependencies|2.0|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200712142200
858010|858010-b19f-3f77-b713-d32bba55d77f|fail|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200712142141
858011|858011-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200712142142
858012|858012-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200712142144
858013|858013-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200712142145
858014|858014-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200712142146
858016|858016-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200712142147
858018|858018-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200712142149
858020|858020-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200712142150
858175|858175-b19f-3f77-b713-d32bba55d77f|pass|200712|srezic@cpan.org|CPAN-FindDependencies|2.0|amd64-freebsd-thread-multi|5.10.0 patch 32367|freebsd|6.2-release|200712150009
859571|859571-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200712152036
860492|860492-b19f-3f77-b713-d32bba55d77f|pass|200712|overlordq@gmail.com|Net-Random|2.0|i486-linux-gnu|5.8.8|linux|2.6.22.10|200712160142
860548|860548-b19f-3f77-b713-d32bba55d77f|pass|200712|overlordq@gmail.com|Data-Compare|0.17|i486-linux-gnu|5.8.8|linux|2.6.22.10|200712160211
861014|861014-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200712161129
861340|861340-b19f-3f77-b713-d32bba55d77f|pass|200712|imacat@mail.imacat.idv.tw|CPAN-FindDependencies|2.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200712170123
862554|862554-b19f-3f77-b713-d32bba55d77f|pass|200712|andy@hexten.net|Number-Phone|1.58|darwin-thread-multi-2level|5.10.0 patch 32604|darwin|9.1.0|200712170017
862555|862555-b19f-3f77-b713-d32bba55d77f|pass|200712|andy@hexten.net|Class-CanBeA|1.2|darwin-thread-multi-2level|5.10.0 patch 32604|darwin|9.1.0|200712170017
862556|862556-b19f-3f77-b713-d32bba55d77f|pass|200712|andy@hexten.net|Data-Compare|0.17|darwin-thread-multi-2level|5.10.0 patch 32604|darwin|9.1.0|200712170017
862557|862557-b19f-3f77-b713-d32bba55d77f|pass|200712|andy@hexten.net|Data-Transactional|1.0|darwin-thread-multi-2level|5.10.0 patch 32604|darwin|9.1.0|200712170017
862558|862558-b19f-3f77-b713-d32bba55d77f|pass|200712|andy@hexten.net|Net-Random|2.0|darwin-thread-multi-2level|5.10.0 patch 32604|darwin|9.1.0|200712170017
862559|862559-b19f-3f77-b713-d32bba55d77f|pass|200712|andy@hexten.net|Number-Phone-UK-DetailedLocations|1.3|darwin-thread-multi-2level|5.10.0 patch 32604|darwin|9.1.0|200712170018
862561|862561-b19f-3f77-b713-d32bba55d77f|pass|200712|andy@hexten.net|CPAN-FindDependencies|2.0|darwin-thread-multi-2level|5.10.0 patch 32604|darwin|9.1.0|200712170020
862562|862562-b19f-3f77-b713-d32bba55d77f|pass|200712|andy@hexten.net|Devel-CheckLib|0.3|darwin-thread-multi-2level|5.10.0 patch 32604|darwin|9.1.0|200712170020
862563|862563-b19f-3f77-b713-d32bba55d77f|pass|200712|andy@hexten.net|Games-Dice-Advanced|1.1|darwin-thread-multi-2level|5.10.0 patch 32604|darwin|9.1.0|200712170021
862763|862763-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|darwin-thread-multi|5.6.2|darwin|8.10.0|200712170404
863091|863091-b19f-3f77-b713-d32bba55d77f|pass|200712|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPAN-FindDependencies|2.0|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200712170901
863578|863578-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200712170416
864697|864697-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200712180430
865057|865057-b19f-3f77-b713-d32bba55d77f|pass|200712|cpan@pqpq.de|CPAN-FindDependencies|2.0|amd64-freebsd|5.10.0 patch 32559|freebsd|6.2-release|200712180921
865873|865873-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200712181359
866599|866599-b19f-3f77-b713-d32bba55d77f|pass|200712|salvacpan@gmail.com|Data-Compare|0.17|sparc64-netbsd|5.10.0|netbsd|4.0_rc5|200712181435
866762|866762-b19f-3f77-b713-d32bba55d77f|pass|200712|andy@hexten.net|Devel-CheckLib|0.3|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200712181902
866770|866770-b19f-3f77-b713-d32bba55d77f|pass|200712|andy@hexten.net|Devel-CheckLib|0.3|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200712181803
866873|866873-b19f-3f77-b713-d32bba55d77f|pass|200712|andy@hexten.net|Class-CanBeA|1.2|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200712181751
866889|866889-b19f-3f77-b713-d32bba55d77f|pass|200712|andy@hexten.net|Games-Dice-Advanced|1.1|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200712181917
867078|867078-b19f-3f77-b713-d32bba55d77f|pass|200712|andy@hexten.net|Class-CanBeA|1.2|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200712181849
867192|867192-b19f-3f77-b713-d32bba55d77f|pass|200712|andy@hexten.net|Number-Phone|1.58|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200712181931
867193|867193-b19f-3f77-b713-d32bba55d77f|pass|200712|andy@hexten.net|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200712181931
867611|867611-b19f-3f77-b713-d32bba55d77f|pass|200712|andy@hexten.net|Games-Dice-Advanced|1.1|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200712181816
867648|867648-b19f-3f77-b713-d32bba55d77f|pass|200712|andy@hexten.net|Devel-CheckOS|1.42|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200712182105
867785|867785-b19f-3f77-b713-d32bba55d77f|pass|200712|andy@hexten.net|Net-Random|2.0|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200712182123
868027|868027-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|i386-pc-solaris2.11-thread-multi-64int|5.10.0|solaris|2.11|200712182138
868129|868129-b19f-3f77-b713-d32bba55d77f|pass|200712|andy@hexten.net|CPAN-FindDependencies|2.0|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200712182225
868221|868221-b19f-3f77-b713-d32bba55d77f|pass|200712|andy@hexten.net|Data-Compare|0.17|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200712182234
868223|868223-b19f-3f77-b713-d32bba55d77f|pass|200712|andy@hexten.net|Data-Transactional|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200712182235
868857|868857-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i386-pc-solaris2.11-thread-multi-64int|5.10.0|solaris|2.11|200712190230
870729|870729-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200712191657
872660|872660-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200712200418
874234|874234-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux|5.11.0|linux|2.6.22-1-k7|200712202153
874402|874402-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.11.0|linux|2.6.22-1-k7|200712202233
874775|874775-b19f-3f77-b713-d32bba55d77f|pass|200712|bricas@cpan.org ("Brian Cassidy")|Devel-CheckOS|1.42|i486-linux-gnu|5.8.8|linux|2.6.15.7|200712201446
875337|875337-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux|5.11.0|linux|2.6.22-1-k7|200712210125
875730|875730-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200712210230
877394|877394-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.11.0|linux|2.6.22-1-k7|200712210733
877531|877531-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux|5.11.0|linux|2.6.22-1-k7|200712211038
878404|878404-b19f-3f77-b713-d32bba55d77f|pass|200712|bricas@cpan.org ("Brian Cassidy")|Devel-CheckLib|0.3|i486-linux-gnu|5.8.8|linux|2.6.15.7|200712210939
878801|878801-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200712211947
879522|879522-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-64int|5.11.0|linux|2.6.22-1-k7|200712212314
879947|879947-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0|linux|2.6.22-1-k7|200712220025
880232|880232-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200712220044
880595|880595-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-64int|5.11.0|linux|2.6.22-1-k7|200712220224
880734|880734-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux-64int|5.11.0|linux|2.6.22-1-k7|200712220340
880776|880776-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0|linux|2.6.22-1-k7|200712220304
884030|884030-b19f-3f77-b713-d32bba55d77f|pass|200712|david@cantrell.org.uk|Bryar|2.8_02|i686-linux|5.6.2|linux|2.4.27-3-686|200712222041
885997|885997-b19f-3f77-b713-d32bba55d77f|pass|200712|stro@cpan.org|CPAN-FindDependencies|2.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200712230141
888988|888988-b19f-3f77-b713-d32bba55d77f|pass|200712|srezic@cpan.org|Data-Compare|0.17|amd64-freebsd|5.10.0|freebsd|6.2-release|200712241030
889201|889201-b19f-3f77-b713-d32bba55d77f|pass|200712|andy@hexten.net|XML-Tiny|1.11|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200712241127
890820|890820-b19f-3f77-b713-d32bba55d77f|pass|200712|cpan@pqpq.de|Devel-CheckOS|1.42|amd64-freebsd|5.10.0 patch 32559|freebsd|6.2-release|200712251539
891034|891034-b19f-3f77-b713-d32bba55d77f|pass|200712|rlane10@wi.rr.com|Devel-CheckOS|1.42|darwin-2level|5.10.0|darwin|8.11.0|200712250509
892422|892422-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200712260259
894567|894567-b19f-3f77-b713-d32bba55d77f|fail|200712|david@cantrell.org.uk|Bryar|2.8_02|darwin-thread-multi-2level|5.10.0 patch 32468|darwin|8.10.1|200712270027
894603|894603-b19f-3f77-b713-d32bba55d77f|pass|200712|srezic@cpan.org|Data-Compare|0.17|amd64-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200712270134
894628|894628-b19f-3f77-b713-d32bba55d77f|pass|200712|rlane10@wi.rr.com|XML-Tiny|1.11|darwin-2level|5.10.0|darwin|8.11.0|200712261305
894709|894709-b19f-3f77-b713-d32bba55d77f|pass|200712|icerider70@gmail.com ("ICERIDER")|Devel-CheckOS|1.42|i386-linux|5.8.8|linux|2.6.18-53.el5|200712270448
895423|895423-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int|5.11.0|linux|2.6.22-1-k7|200712271029
895613|895613-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-thread-multi-64int|5.11.0|linux|2.6.22-1-k7|200712271144
896014|896014-b19f-3f77-b713-d32bba55d77f|pass|200712|claco@chrislaco.com ("Christopher H. Laco")|Data-Compare|0.17|darwin-thread-multi-2level|5.8.6|darwin|8.0|200712270048
896291|896291-b19f-3f77-b713-d32bba55d77f|pass|200712|icerider70@gmail.com ("ICERIDER")|Data-Compare|0.17|i386-linux|5.8.8|linux|2.6.18-53.el5|200712270727
896293|896293-b19f-3f77-b713-d32bba55d77f|pass|200712|root@alysande.altrion.org ((System Administrator))|Devel-CheckOS|1.42|darwin-2level|5.10.0|darwin|8.11.1|200712271305
896347|896347-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.11.0|linux|2.6.22-1-k7|200712271437
896767|896767-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux-thread-multi-64int|5.11.0|linux|2.6.22-1-k7|200712271414
897406|897406-b19f-3f77-b713-d32bba55d77f|pass|200712|david@cantrell.org.uk|XML-Tiny|1.11|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200712271820
897407|897407-b19f-3f77-b713-d32bba55d77f|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-thread-multi-64int|5.11.0|linux|2.6.22-1-k7|200712271021
897430|897430-b19f-3f77-b713-d32bba55d77f|pass|200712|david@cantrell.org.uk|Data-Compare|0.17|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200712271826
897445|897445-b19f-3f77-b713-d32bba55d77f|pass|200712|david@cantrell.org.uk|XML-Tiny|1.11|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200712271817
897550|897550-b19f-3f77-b713-d32bba55d77f|pass|200712|david@cantrell.org.uk|Net-Random|2.0|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200712271856
897601|897601-b19f-3f77-b713-d32bba55d77f|pass|200712|david@cantrell.org.uk|Net-Random|2.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200712271853
897769|897769-b19f-3f77-b713-d32bba55d77f|pass|200712|david@cantrell.org.uk|Net-Random|2.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200712271354
897863|897863-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200712271233
897915|897915-b19f-3f77-b713-d32bba55d77f|pass|200712|david@cantrell.org.uk|Net-Random|2.0|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200712272028
897993|897993-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200712271258
898157|898157-b19f-3f77-b713-d32bba55d77f|pass|200712|david@cantrell.org.uk|XML-Tiny|1.11|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200712272144
898228|898228-b19f-3f77-b713-d32bba55d77f|pass|200712|david@cantrell.org.uk|Net-Random|2.0|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200712272152
898271|898271-b19f-3f77-b713-d32bba55d77f|pass|200712|david@cantrell.org.uk|Data-Compare|0.17|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200712272143
899962|899962-b19f-3f77-b713-d32bba55d77f|pass|200712|david@cantrell.org.uk|XML-Tiny|1.11|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200712281636
900858|900858-b19f-3f77-b713-d32bba55d77f|pass|200712|imacat@mail.imacat.idv.tw|Data-Compare|0.17|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200712290438
903334|903334-b19f-3f77-b713-d32bba55d77f|pass|200712|rlane10@wi.rr.com|Tie-STDOUT|1.03|darwin-2level|5.10.0|darwin|8.11.0|200712300053
904773|904773-b19f-3f77-b713-d32bba55d77f|pass|200712|srezic@cpan.org|Statistics-ChiSquare|0.5|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200712302026
904802|904802-b19f-3f77-b713-d32bba55d77f|pass|200712|srezic@cpan.org|Statistics-SerialCorrelation|1.1|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200712302034
905218|905218-b19f-3f77-b713-d32bba55d77f|pass|200712|rlane10@wi.rr.com|Devel-CheckOS|1.42|i686-linux|5.10.0|linux|2.6.23.1luckyseven|200712301444
906166|906166-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-pc-solaris2.11-thread-multi-64int|5.10.0|solaris|2.11|200712310730
906374|906374-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200712310854
906487|906487-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200712311007
906696|906696-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200712311151
906698|906698-b19f-3f77-b713-d32bba55d77f|na|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|darwin-thread-multi-2level|5.8.1|darwin|8.10.0|200712311154
906700|906700-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|darwin-thread-multi-2level|5.8.2|darwin|8.10.0|200712311157
906701|906701-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|darwin-thread-multi-2level|5.8.3|darwin|8.10.0|200712311159
906705|906705-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|darwin-thread-multi-2level|5.8.4|darwin|8.10.0|200712311202
906707|906707-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|darwin-thread-multi-2level|5.8.5|darwin|8.10.0|200712311204
906712|906712-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|darwin-thread-multi-2level|5.8.6|darwin|8.10.0|200712311206
906715|906715-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|darwin-thread-multi-2level|5.8.7|darwin|8.10.0|200712311208
906717|906717-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|darwin-thread-multi-2level|5.8.8|darwin|8.10.0|200712311210
907980|907980-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200712310840
908298|908298-b19f-3f77-b713-d32bba55d77f|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200712310949
908891|908891-b19f-3f77-b713-d32bba55d77f|pass|200712|cpantester@zombie.org.uk|Data-Compare|0.17|x86_64-linux-ld|5.11.0 patch 32787|linux|2.6.22-3-amd64|200712311732
910528|910528-b19f-3f77-b713-d32bba55d77f|pass|200712|cpantester@zombie.org.uk|Data-Compare|0.17|x86_64-linux-ld|5.11.0 patch 32790|linux|2.6.22-3-amd64|200712312036
911392|911392-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|0.17|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200801011056
911683|911683-b19f-3f77-b713-d32bba55d77f|pass|200801|cpantester@zombie.org.uk|Data-Compare|0.17|x86_64-linux-ld|5.11.0 patch 32790|linux|2.6.22-3-amd64|200801011209
912997|912997-b19f-3f77-b713-d32bba55d77f|pass|200801|cpantester@hotmail.co.uk|Data-Compare|0.17|x86_64-linux-thread-multi-ld|5.11.0 patch 32792|linux|2.6.22-3-amd64|200801012023
914399|914399-b19f-3f77-b713-d32bba55d77f|pass|200801|cpantester@hotmail.co.uk|Data-Compare|0.17|x86_64-linux-thread-multi-ld|5.11.0 patch 32792|linux|2.6.22-3-amd64|200801020106
914536|914536-b19f-3f77-b713-d32bba55d77f|pass|200801|andy@hexten.net|Data-Hexdumper|1.3|i686-linux-thread-multi|5.10.0|linux|2.6.18-5-686|200801021551
914904|914904-b19f-3f77-b713-d32bba55d77f|pass|200801|andy@hexten.net|Data-Hexdumper|1.3|i686-linux-thread-multi|5.10.0|linux|2.6.20-16-server|200801021710
915075|915075-b19f-3f77-b713-d32bba55d77f|pass|200801|andy@hexten.net|Data-Hexdumper|1.3|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200801021749
915506|915506-b19f-3f77-b713-d32bba55d77f|pass|200801|andy@hexten.net|Statistics-ChiSquare|0.5|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200801022013
915791|915791-b19f-3f77-b713-d32bba55d77f|pass|200801|d@0x64.nl|XML-Tiny|1.11|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200801021048
916082|916082-b19f-3f77-b713-d32bba55d77f|pass|200801|cpantester@hotmail.co.uk|Data-Compare|0.17|x86_64-linux-thread-multi-ld|5.11.0 patch 32805|linux|2.6.22-3-amd64|200801022238
917482|917482-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200801030803
918102|918102-b19f-3f77-b713-d32bba55d77f|pass|200801|cpantester@hotmail.co.uk|Data-Compare|0.17|x86_64-linux-thread-multi-ld|5.11.0 patch 32800|linux|2.6.22-3-amd64|200801021356
918614|918614-b19f-3f77-b713-d32bba55d77f|pass|200801|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200801031055
919916|919916-b19f-3f77-b713-d32bba55d77f|pass|200801|imacat@mail.imacat.idv.tw|Number-Phone|1.58|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200801040155
920490|920490-b19f-3f77-b713-d32bba55d77f|pass|200801|d@0x64.nl|Data-Compare|0.17|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200801032352
921985|921985-b19f-3f77-b713-d32bba55d77f|pass|200801|cpantester@hotmail.co.uk|Data-Compare|0.17|x86_64-linux-thread-multi-ld|5.11.0 patch 32809|linux|2.6.22-3-amd64|200801041016
923011|923011-b19f-3f77-b713-d32bba55d77f|pass|200801|cpantester@hotmail.co.uk|Data-Compare|0.17|x86_64-linux-thread-multi-ld|5.11.0 patch 32824|linux|2.6.22-3-amd64|200801041358
923177|923177-b19f-3f77-b713-d32bba55d77f|pass|200801|bob@randomness.org.uk ("bob walker")|Data-Compare|0.17|i86pc-solaris|5.10.0|solaris|2.10|200801041601
923700|923700-b19f-3f77-b713-d32bba55d77f|pass|200801|cpantester@hotmail.co.uk|Data-Compare|0.17|x86_64-linux-thread-multi-ld|5.11.0 patch 32832|linux|2.6.22-3-amd64|200801041845
923985|923985-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|0.17|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200801042006
924386|924386-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Data-Compare|0.17|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200801042156
924870|924870-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Data-Compare|0.17|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200801050035
926854|926854-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|0.17|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200801052014
927106|927106-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Data-Compare|0.17|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801051404
927629|927629-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|0.17|cygwin-thread-multi-64int|5.11.0 patch 32837|cygwin|1.5.24(0.15642)|200801060155
930316|930316-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|0.17|cygwin-thread-multi-64int|5.11.0 patch 32864|cygwin|1.5.24(0.15642)|200801070135
931226|931226-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200801071157
931657|931657-b19f-3f77-b713-d32bba55d77f|pass|200801|kamensky@cpan.org ("Moshe Kamensky")|Data-Compare|0.17|sun4-solaris|5.10.0|solaris|2.8|200801070928
931974|931974-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|0.17|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200801071713
932623|932623-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|0.17|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200801072025
934822|934822-b19f-3f77-b713-d32bba55d77f|pass|200801|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Compare|0.17|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200801081334
938822|938822-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|0.17|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200801091702
939607|939607-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|0.17|alpha-linux-thread-multi|5.11.0 patch 32908|linux|2.6.18-5-alpha-generic|200801092044
939664|939664-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Bryar|2.6|i686-linux|5.6.2|linux|2.4.27-3-686|200801092100
939750|939750-b19f-3f77-b713-d32bba55d77f|fail|200801|david@cantrell.org.uk|Bryar|3.0|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200801092123
940465|940465-b19f-3f77-b713-d32bba55d77f|pass|200801|bob@randomness.org.uk ("bob walker")|Data-Compare|0.17|i86pc-solaris|5.8.8|solaris|2.10|200801100207
946793|946793-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|0.17|cygwin-thread-multi-64int|5.11.0 patch 32953|cygwin|1.5.24(0.15642)|200801120913
947178|947178-b19f-3f77-b713-d32bba55d77f|pass|200801|cpantester@hotmail.co.uk|Data-Compare|0.17|x86_64-linux-thread-multi-ld|5.11.0 patch 32966|linux|2.6.22-3-amd64|200801121142
953318|953318-b19f-3f77-b713-d32bba55d77f|fail|200801|rjbs@cpan.org|Bryar|3.0|darwin-2level|5.10.0|darwin|9.1.0|200801141030
955700|955700-b19f-3f77-b713-d32bba55d77f|pass|200801|mmusgrove@cpan.org ("Matthew Musgrove")|Data-Compare|0.17|i686-linux-64int-ld|5.8.8|linux|2.6.22-14-generic|200801150051
955762|955762-b19f-3f77-b713-d32bba55d77f|pass|200801|mmusgrove@cpan.org ("Matthew Musgrove")|Data-Compare|0.17|i686-linux-64int-ld|5.10.0|linux|2.6.22-14-generic|200801150101
956019|956019-b19f-3f77-b713-d32bba55d77f|pass|200801|robertl@apnic.net ("Robert Loomans")|Data-Compare|0.17|darwin-thread-multi-2level|5.8.8|darwin|8.11.1|200801151755
958423|958423-b19f-3f77-b713-d32bba55d77f|pass|200801|bob@randomness.org.uk ("bob walker")|Data-Compare|0.17|OpenBSD.i386-openbsd|5.8.8|openbsd|4.2|200801152111
958437|958437-b19f-3f77-b713-d32bba55d77f|pass|200801|bob@randomness.org.uk ("bob walker")|Data-Compare|0.17|OpenBSD.i386-openbsd|5.10.0|openbsd|4.2|200801152114
958862|958862-b19f-3f77-b713-d32bba55d77f|pass|200801|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.11.0|linux|2.6.22-1-k7|200801160016
958863|958863-b19f-3f77-b713-d32bba55d77f|pass|200801|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200801160016
958866|958866-b19f-3f77-b713-d32bba55d77f|pass|200801|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200801160016
958869|958869-b19f-3f77-b713-d32bba55d77f|pass|200801|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200801160017
958871|958871-b19f-3f77-b713-d32bba55d77f|pass|200801|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200801160017
958873|958873-b19f-3f77-b713-d32bba55d77f|pass|200801|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200801160018
958875|958875-b19f-3f77-b713-d32bba55d77f|pass|200801|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200801160019
959222|959222-b19f-3f77-b713-d32bba55d77f|pass|200801|bob@randomness.org.uk ("bob walker")|Data-Compare|1.18|i86pc-solaris|5.8.8|solaris|2.10|200801160124
959225|959225-b19f-3f77-b713-d32bba55d77f|pass|200801|bob@randomness.org.uk ("bob walker")|Data-Compare|1.18|i86pc-solaris|5.10.0|solaris|2.10|200801160125
959422|959422-b19f-3f77-b713-d32bba55d77f|pass|200801|icerider70@gmail.com ("ICERIDER")|Data-Compare|1.18|i386-linux|5.8.8|linux|2.6.18-53.el5|200801160210
960005|960005-b19f-3f77-b713-d32bba55d77f|pass|200801|mmusgrove@cpan.org ("Matthew Musgrove")|CPAN-FindDependencies|1.99_01|i686-linux-64int-ld|5.8.8|linux|2.6.22-14-generic|200801160419
960006|960006-b19f-3f77-b713-d32bba55d77f|pass|200801|mmusgrove@cpan.org ("Matthew Musgrove")|Devel-CheckOS|1.42|i686-linux-64int-ld|5.8.8|linux|2.6.22-14-generic|200801160419
960026|960026-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200801160430
960030|960030-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|darwin-thread-multi|5.6.2|darwin|8.10.0|200801160433
960035|960035-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|darwin-thread-multi-2level|5.8.8|darwin|8.10.0|200801160439
960318|960318-b19f-3f77-b713-d32bba55d77f|pass|200801|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Data-Compare|1.18|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200801161514
960472|960472-b19f-3f77-b713-d32bba55d77f|pass|200801|Jost.Krieger+perl@rub.de|Data-Compare|1.18|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200801160920
960531|960531-b19f-3f77-b713-d32bba55d77f|pass|200801|d@0x64.nl (DEVOGON)|Data-Compare|1.18|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200801160948
961467|961467-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Data-Compare|1.18|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200801161415
961485|961485-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Data-Compare|1.18|i686-linux|5.8.8|linux|2.4.27-3-686|200801161417
961502|961502-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Data-Compare|1.18|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200801161347
961506|961506-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Data-Compare|1.18|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801160820
961509|961509-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Data-Compare|1.18|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200801161420
961510|961510-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Data-Compare|1.18|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200801161420
961520|961520-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Data-Compare|1.18|i686-linux|5.6.2|linux|2.4.27-3-686|200801161421
961562|961562-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|alpha-linux-thread-multi|5.11.0 patch 32981|linux|2.6.18-5-alpha-generic|200801161431
961963|961963-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200801161638
962412|962412-b19f-3f77-b713-d32bba55d77f|pass|200801|cpantester@hotmail.co.uk|Data-Compare|1.18|x86_64-linux-thread-multi-ld|5.11.0 patch 32982|linux|2.6.22-3-amd64|200801161929
962482|962482-b19f-3f77-b713-d32bba55d77f|pass|200801|srezic@cpan.org|Data-Compare|1.18|amd64-freebsd|5.10.0|freebsd|6.2-release|200801162046
963003|963003-b19f-3f77-b713-d32bba55d77f|pass|200801|srezic@cpan.org|Data-Compare|1.18|amd64-freebsd|5.6.2|freebsd|6.2-release|200801162135
963156|963156-b19f-3f77-b713-d32bba55d77f|pass|200801|g.grigelionis@computer.org|Data-Compare|1.18|i586-linux-thread-multi|5.8.8|linux|2.6.16|200801162212
963203|963203-b19f-3f77-b713-d32bba55d77f|pass|200801|srezic@cpan.org|Data-Compare|1.18|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200801162221
963745|963745-b19f-3f77-b713-d32bba55d77f|pass|200801|bob@randomness.org.uk ("bob walker")|Data-Compare|1.18|OpenBSD.i386-openbsd|5.8.8|openbsd|4.2|200801162358
963770|963770-b19f-3f77-b713-d32bba55d77f|pass|200801|bob@randomness.org.uk ("bob walker")|Data-Compare|1.18|OpenBSD.i386-openbsd|5.10.0|openbsd|4.2|200801170006
964012|964012-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200801170031
964037|964037-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200801170036
964040|964040-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200801170039
964071|964071-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200801170059
964084|964084-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200801170102
964087|964087-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200801170105
964091|964091-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200801170108
964106|964106-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200801170112
964125|964125-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200801170115
964127|964127-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200801170118
964635|964635-b19f-3f77-b713-d32bba55d77f|pass|200801|d@0x64.nl (DEVOGON)|Data-Compare|1.18|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200801170756
964967|964967-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200801170952
965354|965354-b19f-3f77-b713-d32bba55d77f|pass|200801|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Compare|1.18|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200801171203
966001|966001-b19f-3f77-b713-d32bba55d77f|pass|200801|imacat@mail.imacat.idv.tw|Data-Compare|1.18|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200801180044
967324|967324-b19f-3f77-b713-d32bba55d77f|pass|200801|salve.nilsen@met.no ("Salve J Nilsen")|Data-Compare|1.18|i386-linux|5.8.8|linux|2.6.9-34.elsmp|200801171735
967365|967365-b19f-3f77-b713-d32bba55d77f|pass|200801|kamensky@cpan.org ("Moshe Kamensky")|Data-Compare|1.18|sun4-solaris|5.10.0|solaris|2.8|200801171332
968007|968007-b19f-3f77-b713-d32bba55d77f|pass|200801|overlordq@gmail.com|Data-Compare|1.18|i486-linux-gnu|5.8.8|linux|2.6.22.10|200801180317
968952|968952-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200801181450
969190|969190-b19f-3f77-b713-d32bba55d77f|pass|200801|d@0x64.nl (DEVOGON)|Data-Compare|1.18|i686-linux|5.10.0|linux|2.6.22-3-486|200801181718
969205|969205-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200801181623
969783|969783-b19f-3f77-b713-d32bba55d77f|pass|200801|dillman@ria.buffalo.edu|Data-Compare|1.18|sun4-solaris|5.8.8|solaris|2.9|200801181347
970950|970950-b19f-3f77-b713-d32bba55d77f|pass|200801|mmusgrove@cpan.org ("Matthew Musgrove")|Devel-CheckOS|1.42|i686-linux-64int-ld|5.10.0|linux|2.6.22-14-generic|200801190046
972266|972266-b19f-3f77-b713-d32bba55d77f|pass|200801|srezic@cpan.org|Data-Compare|1.18|amd64-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200801191057
972674|972674-b19f-3f77-b713-d32bba55d77f|pass|200801|d@0x64.nl (DEVOGON)|Devel-CheckOS|1.42|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200801191419
972689|972689-b19f-3f77-b713-d32bba55d77f|pass|200801|d@0x64.nl (DEVOGON)|Devel-CheckOS|1.42|i686-linux|5.10.0|linux|2.6.22-3-486|200801191423
972692|972692-b19f-3f77-b713-d32bba55d77f|pass|200801|d@0x64.nl (DEVOGON)|Devel-CheckLib|0.3|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200801191423
972693|972693-b19f-3f77-b713-d32bba55d77f|pass|200801|d@0x64.nl (DEVOGON)|Devel-CheckLib|0.3|i686-linux|5.10.0|linux|2.6.22-3-486|200801191425
974401|974401-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|cygwin-thread-multi-64int|5.11.0 patch 33004|cygwin|1.5.24(0.15642)|200801192233
974723|974723-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200801200019
974764|974764-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Class-CanBeA|1.2|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200048
974766|974766-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Data-Hexdumper|1.0.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200049
974769|974769-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Data-Hexdumper|1.2|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200050
974786|974786-b19f-3f77-b713-d32bba55d77f|fail|200801|david@cantrell.org.uk|Net-Random|2.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200104
974787|974787-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Number-Phone|1.58|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200105
974789|974789-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Number-Phone-UK-DetailedLocations|1.3|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200106
974791|974791-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Statistics-ChiSquare|0.5|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200109
974793|974793-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Statistics-SerialCorrelation|1.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200109
974795|974795-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Sub-WrapPackages|1.2|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200110
974797|974797-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Tie-Hash-Longest|1.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200111
974799|974799-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Tie-Hash-Rank|1.0.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200111
974800|974800-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Tie-STDOUT|1.02|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200111
974803|974803-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Tie-Scalar-Decay|1.1.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200112
974806|974806-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Tie-Scalar-Decay|1.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200113
974809|974809-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200113
974811|974811-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|XML-Tiny|1.06|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200114
975159|975159-b19f-3f77-b713-d32bba55d77f|pass|200801|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.18|i686-linux-64int-ld|5.10.0|linux|2.6.22-14-386|200801200355
975983|975983-b19f-3f77-b713-d32bba55d77f|pass|200801|andy@hexten.net|Data-Compare|1.18|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200801201242
976661|976661-b19f-3f77-b713-d32bba55d77f|pass|200801|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux|5.8.8 patch 33008|linux|2.6.22-1-k7|200801201822
977539|977539-b19f-3f77-b713-d32bba55d77f|pass|200801|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.8.8 patch 33008|linux|2.6.22-1-k7|200801202155
977631|977631-b19f-3f77-b713-d32bba55d77f|pass|200801|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.8.8 patch 33008|linux|2.6.22-1-k7|200801202104
977717|977717-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|NestedMap|1.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801202208
977792|977792-b19f-3f77-b713-d32bba55d77f|fail|200801|david@cantrell.org.uk|Bryar|3.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801202224
977962|977962-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Acme-Scurvy-Whoreson-BilgeRat|1.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801202248
978266|978266-b19f-3f77-b713-d32bba55d77f|unknown|200801|david@cantrell.org.uk|Acme-Licence|1.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801202350
978376|978376-b19f-3f77-b713-d32bba55d77f|pass|200801|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux|5.8.8 patch 33008|linux|2.6.22-1-k7|200801210111
978612|978612-b19f-3f77-b713-d32bba55d77f|unknown|200801|david@cantrell.org.uk|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801210109
978852|978852-b19f-3f77-b713-d32bba55d77f|fail|200801|david@cantrell.org.uk|Data-Transactional|0.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801210149
978863|978863-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|File-Find-Rule-Permissions|1.2|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801210151
978893|978893-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|File-Overwrite|1.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801210158
978896|978896-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Games-Dice-Advanced|1.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801210159
978966|978966-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801210217
978970|978970-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801210219
979198|979198-b19f-3f77-b713-d32bba55d77f|pass|200801|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.8.8 patch 33008|linux|2.6.22-1-k7|200801210425
979494|979494-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200801210515
980641|980641-b19f-3f77-b713-d32bba55d77f|fail|200801|d@0x64.nl (DEVOGON)|Bryar|3.0|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200801211628
980947|980947-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|alpha-linux-thread-multi|5.11.0 patch 33022|linux|2.6.18-5-alpha-generic|200801211712
982905|982905-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200801220618
983254|983254-b19f-3f77-b713-d32bba55d77f|pass|200801|fayland@cpan.org|Data-Compare|1.18|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200801220922
983260|983260-b19f-3f77-b713-d32bba55d77f|pass|200801|rlane10@wi.rr.com|Data-Compare|1.18|darwin-2level-ld|5.10.0|darwin|8.11.0|200801220144
987518|987518-b19f-3f77-b713-d32bba55d77f|pass|200801|andy@hexten.net|Data-Compare|1.18|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200801231403
987884|987884-b19f-3f77-b713-d32bba55d77f|fail|200801|david@cantrell.org.uk|Bryar|2.6|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200801231711
988738|988738-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Acme-Scurvy-Whoreson-BilgeRat|1.1|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200801240003
989519|989519-b19f-3f77-b713-d32bba55d77f|pass|200801|callen@kpi-consulting.net|Devel-CheckOS|1.42|i386-linux|5.8.8|linux|2.6.9-42.0.3.elsmp|200801231502
989759|989759-b19f-3f77-b713-d32bba55d77f|pass|200801|stro@cpan.org|Data-Compare|1.18|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200801240511
994236|994236-b19f-3f77-b713-d32bba55d77f|pass|200801|rjbs@cpan.org|Data-Compare|1.18|darwin-2level|5.10.0|darwin|9.1.0|200801261005
994237|994237-b19f-3f77-b713-d32bba55d77f|pass|200801|rjbs@cpan.org|Data-Compare|1.18|darwin-2level|5.10.0|darwin|9.1.0|200801261005
996329|996329-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|CPAN-FindDependencies|2.0|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200801271620
998612|998612-b19f-3f77-b713-d32bba55d77f|pass|200801|bob@randomness.org.uk ("bob walker")|Devel-CheckOS|1.42|i86pc-solaris|5.8.8|solaris|2.10|200801281301
998621|998621-b19f-3f77-b713-d32bba55d77f|pass|200801|bob@randomness.org.uk ("bob walker")|Devel-CheckOS|1.42|i86pc-solaris|5.10.0|solaris|2.10|200801281302
999761|999761-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Devel-CheckOS|1.42|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200801281923
1000713|1000713-b19f-3f77-b713-d32bba55d77f|pass|200801|bob@randomness.org.uk ("bob walker")|Devel-CheckOS|1.42|OpenBSD.i386-openbsd|5.8.8|openbsd|4.2|200801282355
1000716|1000716-b19f-3f77-b713-d32bba55d77f|pass|200801|bob@randomness.org.uk ("bob walker")|Devel-CheckOS|1.42|OpenBSD.i386-openbsd|5.10.0|openbsd|4.2|200801282358
1000819|1000819-b19f-3f77-b713-d32bba55d77f|pass|200801|g.grigelionis@computer.org|Devel-CheckOS|1.42|i386-linux|5.8.0|linux|2.6.9-22.18.bz155725.elsmp|200801282113
1000850|1000850-b19f-3f77-b713-d32bba55d77f|pass|200801|g.grigelionis@computer.org|Devel-CheckOS|1.42|x86_64-linux-thread-multi|5.8.3|linux|2.6.4|200801282117
1001686|1001686-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Devel-CheckOS|1.42|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801290757
1001838|1001838-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Devel-CheckOS|1.42|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200801291312
1001936|1001936-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Devel-CheckOS|1.42|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200801291540
1002105|1002105-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Devel-CheckOS|1.42|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200801291616
1002250|1002250-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Devel-CheckOS|1.42|alpha-linux-thread-multi|5.11.0 patch 33082|linux|2.6.18-5-alpha-generic|200801291819
1002345|1002345-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|Devel-CheckOS|1.42|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200801291841
1002699|1002699-b19f-3f77-b713-d32bba55d77f|pass|200801|cpantester@hotmail.co.uk|Devel-CheckOS|1.42|x86_64-linux-thread-multi-ld|5.11.0 patch 33109|linux|2.6.22-3-amd64|200801292057
1002733|1002733-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200801292109
1006145|1006145-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.11.0 patch 33125|netbsd|4.0|200801310014
1006342|1006342-b19f-3f77-b713-d32bba55d77f|pass|200801|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200801310442
1008067|1008067-b19f-3f77-b713-d32bba55d77f|pass|200801|david@cantrell.org.uk|XML-Tiny|1.11|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801311506
1008614|1008614-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200802010308
1008765|1008765-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200802010622
1009502|1009502-b19f-3f77-b713-d32bba55d77f|pass|200802|ferreira@cpan.org|Devel-CheckOS|1.42|i486-linux-gnu|5.8.8|linux|2.6.15.7|200802011257
1010498|1010498-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200802020418
1010848|1010848-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|cygwin-thread-multi-64int|5.11.0 patch 33163|cygwin|1.5.24(0.15642)|200802021012
1010888|1010888-b19f-3f77-b713-d32bba55d77f|pass|200802|feldt@nhn.ou.edu|Data-Compare|1.18|sun4-solaris-thread-multi|5.8.8|solaris|2.8|200802011342
1011511|1011511-b19f-3f77-b713-d32bba55d77f|pass|200802|andy@hexten.net|Devel-CheckOS|1.42|i486-linux-gnu|5.8.8|linux|2.6.15.7|200802021419
1012282|1012282-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-64int|5.8.8|linux|2.6.14|200802022222
1012807|1012807-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-CanBeA|1.2|i686-linux-64int|5.8.8|linux|2.6.14|200802030414
1013908|1013908-b19f-3f77-b713-d32bba55d77f|pass|200802|rjbs@cpan.org|Data-Compare|1.18|darwin-2level|5.10.0|darwin|9.1.0|200802031115
1014995|1014995-b19f-3f77-b713-d32bba55d77f|pass|200802|claco@chrislaco.com ("Christopher H. Laco")|Data-Compare|1.18|darwin-thread-multi-2level|5.8.6|darwin|8.0|200802031702
1016124|1016124-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i686-linux-thread-multi-64int-ld|5.11.0 patch 33228|linux|2.6.16.38|200802041359
1017394|1017394-b19f-3f77-b713-d32bba55d77f|pass|200802|rlane10@wi.rr.com|Data-Compare|1.18|i686-linux|5.10.0|linux|2.6.23.1luckyseven|200802041337
1017716|1017716-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200802050423
1021750|1021750-b19f-3f77-b713-d32bba55d77f|fail|200802|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Devel-CheckLib|0.3|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200802062240
1027183|1027183-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.11.0|linux|2.6.22-1-k7|200802091719
1030279|1030279-b19f-3f77-b713-d32bba55d77f|pass|200802|autarch@urth.org|Devel-CheckOS|1.42|x86_64-linux-gnu|5.8.8|linux|2.6.15.7|200802101104
1031268|1031268-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.8.8 patch 33243|linux|2.6.22-1-k7|200802110552
1031357|1031357-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-64int|5.8.8 patch 33243|linux|2.6.22-1-k7|200802110628
1033621|1033621-b19f-3f77-b713-d32bba55d77f|pass|200802|autarch@urth.org|Data-Compare|1.18|x86_64-linux-gnu|5.8.8|linux|2.6.15.7|200802111240
1036645|1036645-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.58|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200802131211
1036650|1036650-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.58|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200802131213
1036653|1036653-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.58|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200802131215
1036660|1036660-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.58|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200802131217
1036665|1036665-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.58|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200802131220
1036669|1036669-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.58|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200802131223
1038238|1038238-b19f-3f77-b713-d32bba55d77f|pass|200802|d@0x64.nl (DEVOGON)|Number-Phone|1.58|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200802140759
1038319|1038319-b19f-3f77-b713-d32bba55d77f|pass|200802|d@0x64.nl (DEVOGON)|Number-Phone|1.58|i686-linux|5.10.0|linux|2.6.22-3-486|200802140931
1038655|1038655-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|Number-Phone|1.58|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200802141118
1038698|1038698-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|Number-Phone|1.58|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200802141225
1038987|1038987-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|Number-Phone|1.58|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200802141532
1040103|1040103-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.0|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200802150401
1040104|1040104-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.0|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200802150401
1040105|1040105-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.0|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200802150401
1040106|1040106-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.0|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200802150402
1040107|1040107-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.0|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200802150402
1040108|1040108-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.0|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200802150402
1040520|1040520-b19f-3f77-b713-d32bba55d77f|pass|200802|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.58|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200802151127
1040656|1040656-b19f-3f77-b713-d32bba55d77f|pass|200802|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPU-Emulator-Memory|1.0|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200802151141
1040805|1040805-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i386-netbsd-thread-multi-64int|5.11.0 patch 33314|netbsd|4.0|200802151235
1040850|1040850-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.0|i686-linux|5.6.2|linux|2.4.27-3-686|200802151310
1040931|1040931-b19f-3f77-b713-d32bba55d77f|pass|200802|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckOS|1.42|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200802151346
1041070|1041070-b19f-3f77-b713-d32bba55d77f|pass|200802|cpantester@hotmail.co.uk|Number-Phone|1.58|x86_64-linux-thread-multi-ld|5.11.0 patch 33317|linux|2.6.22-3-amd64|200802151430
1041385|1041385-b19f-3f77-b713-d32bba55d77f|pass|200802|cpantester@hotmail.co.uk|CPU-Emulator-Memory|1.0|x86_64-linux-thread-multi-ld|5.11.0 patch 33317|linux|2.6.22-3-amd64|200802151500
1041451|1041451-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.0|i686-linux|5.8.8|linux|2.4.27-3-686|200802151656
1041529|1041529-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.11.0 patch 33314|netbsd|4.0|200802151718
1041778|1041778-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.11.0 patch 33314|netbsd|4.0|200802151914
1041836|1041836-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.0|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200802151957
1041864|1041864-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200802152026
1041899|1041899-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.0|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200802152026
1041986|1041986-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.0|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200802152119
1042830|1042830-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200802161045
1043635|1043635-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200802161908
1043680|1043680-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200802162012
1043681|1043681-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200802162012
1043682|1043682-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200802162012
1043683|1043683-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200802162012
1043684|1043684-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200802162013
1043685|1043685-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200802162013
1043686|1043686-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200802162013
1043687|1043687-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200802162014
1043688|1043688-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200802162014
1043689|1043689-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200802162014
1043725|1043725-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|CPU-Emulator-Memory|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200802170429
1044143|1044143-b19f-3f77-b713-d32bba55d77f|pass|200802|g.grigelionis@computer.org|Devel-CheckOS|1.42|i586-linux-thread-multi|5.8.5|linux|2.6.8.1|200802162101
1044165|1044165-b19f-3f77-b713-d32bba55d77f|pass|200802|g.grigelionis@computer.org|Data-Compare|1.18|i586-linux-thread-multi|5.8.5|linux|2.6.8.1|200802162129
1044281|1044281-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200802170216
1044955|1044955-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200802171216
1045272|1045272-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|cygwin-thread-multi-64int|5.11.0 patch 33318|cygwin|1.5.24(0.15642)|200802171534
1045402|1045402-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200802171038
1045403|1045403-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Number-Phone|1.58|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200802171639
1045878|1045878-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200802171913
1045909|1045909-b19f-3f77-b713-d32bba55d77f|pass|200802|cpantester@hotmail.co.uk|Devel-CheckOS|1.42|x86_64-linux-thread-multi-ld|5.11.0 patch 33325|linux|2.6.22-3-amd64|200802171932
1046019|1046019-b19f-3f77-b713-d32bba55d77f|pass|200802|cpantester@hotmail.co.uk|Data-Compare|1.18|x86_64-linux-thread-multi-ld|5.11.0 patch 33325|linux|2.6.22-3-amd64|200802171959
1046334|1046334-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200802172212
1046421|1046421-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200802172247
1046511|1046511-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Number-Phone|1.58|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200802180017
1046517|1046517-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Number-Phone|1.58|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200802180019
1046529|1046529-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Number-Phone|1.58|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200802180021
1046827|1046827-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|cygwin-thread-multi-64int|5.11.0 patch 33318|cygwin|1.5.24(0.15642)|200802180336
1047355|1047355-b19f-3f77-b713-d32bba55d77f|pass|200802|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPU-Emulator-Memory|1.0|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200802180921
1047665|1047665-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Number-Phone|1.58|alpha-linux-thread-multi|5.11.0 patch 33325|linux|2.6.18-5-alpha-generic|200802181229
1047818|1047818-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Number-Phone|1.58|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200802181324
1048072|1048072-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|alpha-linux-thread-multi|5.11.0 patch 33325|linux|2.6.18-5-alpha-generic|200802181442
1048201|1048201-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200802181524
1049215|1049215-b19f-3f77-b713-d32bba55d77f|pass|200802|d@0x64.nl (DEVOGON)|CPU-Emulator-Memory|1.0|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200802182238
1049645|1049645-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Data-Compare|1.18|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200802190111
1049667|1049667-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Data-Compare|1.18|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200802190131
1050454|1050454-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200802191305
1050470|1050470-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200802191323
1050471|1050471-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200802191323
1050472|1050472-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200802191324
1050914|1050914-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|alpha-linux-thread-multi|5.11.0 patch 33325|linux|2.6.18-5-alpha-generic|200802191607
1051021|1051021-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Number-Phone|1.58|i686-linux-thread-multi-64int|5.11.0 patch 33334|linux|2.6.16.38|200802191639
1051219|1051219-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i686-linux-thread-multi-64int|5.11.0 patch 33334|linux|2.6.16.38|200802191819
1051837|1051837-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i686-linux-thread-multi-64int|5.11.0 patch 33334|linux|2.6.16.38|200802192329
1052526|1052526-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Number-Phone|1.58|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200802201039
1053541|1053541-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Number-Phone|1.58|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200802201212
1053595|1053595-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200802201220
1053936|1053936-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200802201357
1054546|1054546-b19f-3f77-b713-d32bba55d77f|pass|200802|d@0x64.nl (DEVOGON)|CPU-Emulator-Memory|1.0|i686-linux|5.10.0|linux|2.6.22-3-486|200802201934
1057443|1057443-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200802211259
1059347|1059347-b19f-3f77-b713-d32bba55d77f|pass|200802|d@0x64.nl ("DEVOGON")|Data-Compare|1.18|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200802220038
1060654|1060654-b19f-3f77-b713-d32bba55d77f|pass|200802|stro@cpan.org|Class-CanBeA|1.2|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802211610
1061638|1061638-b19f-3f77-b713-d32bba55d77f|pass|200802|stro@cpan.org|Data-Hexdumper|1.3|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802220802
1061838|1061838-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200802221641
1061925|1061925-b19f-3f77-b713-d32bba55d77f|pass|200802|stro@cpan.org|Devel-AssertLib|0.1|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802221016
1061941|1061941-b19f-3f77-b713-d32bba55d77f|pass|200802|stro@cpan.org|Devel-CheckLib|0.2|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802221018
1062177|1062177-b19f-3f77-b713-d32bba55d77f|pass|200802|d@0x64.nl ("DEVOGON")|XML-DoubleEncodedEntities|1.0|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200802221926
1062313|1062313-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200802221916
1062433|1062433-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|amd64-freebsd|5.10.0|freebsd|6.2-release|200802222043
1062487|1062487-b19f-3f77-b713-d32bba55d77f|unknown|200802|imacat@mail.imacat.idv.tw|Acme-Licence|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230357
1062489|1062489-b19f-3f77-b713-d32bba55d77f|unknown|200802|imacat@mail.imacat.idv.tw|Acme-Pony|1.1.2|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230358
1062491|1062491-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|Acme-Scurvy-Whoreson-BilgeRat|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230358
1062503|1062503-b19f-3f77-b713-d32bba55d77f|fail|200802|imacat@mail.imacat.idv.tw|Bryar|3.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230401
1062507|1062507-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|CPAN-FindDependencies|2.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230402
1062510|1062510-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|CPU-Emulator-Memory|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230403
1062512|1062512-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|Class-CanBeA|1.2|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230403
1062516|1062516-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|Data-Compare|1.18|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230404
1062517|1062517-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|Data-Hexdumper|1.3|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230404
1062521|1062521-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|Data-Transactional|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230405
1062523|1062523-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|Devel-CheckLib|0.3|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230405
1062526|1062526-b19f-3f77-b713-d32bba55d77f|fail|200802|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.42|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230406
1062529|1062529-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|File-Overwrite|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230407
1062532|1062532-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|Games-Dice-Advanced|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230407
1062534|1062534-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|NestedMap|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230408
1062537|1062537-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|Net-Random|2.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230409
1062544|1062544-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|Number-Phone-UK-DetailedLocations|1.3|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230412
1062554|1062554-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|Palm-TreoPhoneCallDB|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230413
1062556|1062556-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|Statistics-ChiSquare|0.5|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230414
1062561|1062561-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|Statistics-SerialCorrelation|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230414
1062567|1062567-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|Sub-WrapPackages|1.2|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230415
1062571|1062571-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|Tie-Hash-Longest|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230415
1062573|1062573-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|Tie-Hash-Rank|1.0.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230416
1062576|1062576-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|Tie-STDOUT|1.03|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230416
1062581|1062581-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|Tie-Scalar-Decay|1.1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230417
1062582|1062582-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|XML-DoubleEncodedEntities|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230417
1062583|1062583-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|XML-Tiny|1.11|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230418
1062858|1062858-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200802222106
1062932|1062932-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|amd64-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200802222236
1062969|1062969-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200802222147
1063197|1063197-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200802222304
1063404|1063404-b19f-3f77-b713-d32bba55d77f|pass|200802|cpantester@hotmail.co.uk|XML-DoubleEncodedEntities|1.0|x86_64-linux-thread-multi-ld|5.11.0 patch 33346|linux|2.6.22-3-amd64|200802221902
1063577|1063577-b19f-3f77-b713-d32bba55d77f|pass|200802|stro@cpan.org|Games-Dice-Advanced|1.1|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802230059
1064682|1064682-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200802230840
1064703|1064703-b19f-3f77-b713-d32bba55d77f|pass|200802|d@0x64.nl (DEVOGON)|XML-DoubleEncodedEntities|1.0|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200802230944
1066771|1066771-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|Tie-Hash-Longest|1.1|amd64-freebsd|5.10.0|freebsd|6.2-release|200802232046
1066779|1066779-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|Tie-Hash-Rank|1.0.1|amd64-freebsd|5.10.0|freebsd|6.2-release|200802232046
1067204|1067204-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|Tie-Scalar-Decay|1.1.1|amd64-freebsd|5.10.0|freebsd|6.2-release|200802232147
1067751|1067751-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|Tie-STDOUT|1.03|amd64-freebsd|5.10.0|freebsd|6.2-release|200802232144
1069310|1069310-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|Data-Hexdumper|1.3|amd64-freebsd|5.10.0|freebsd|6.2-release|200802241047
1070548|1070548-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|Data-Transactional|1.0|amd64-freebsd|5.10.0|freebsd|6.2-release|200802242001
1070873|1070873-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|Data-Transactional|1.0|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200802242125
1071241|1071241-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|Data-Transactional|1.0|amd64-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200802242226
1071744|1071744-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|CPU-Emulator-Memory|1.0|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200802250102
1071782|1071782-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|Data-Hexdumper|1.3|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200802250116
1071902|1071902-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200802250138
1072310|1072310-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200802250340
1073849|1073849-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Data-Compare|1.18|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200802251738
1074148|1074148-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200802251900
1074970|1074970-b19f-3f77-b713-d32bba55d77f|fail|200802|srezic@cpan.org|CPU-Emulator-Memory|1.0|amd64-freebsd|5.5.5|freebsd|6.2-release|200802252312
1075214|1075214-b19f-3f77-b713-d32bba55d77f|fail|200802|srezic@cpan.org|Number-Phone|1.58|amd64-freebsd|5.5.5|freebsd|6.2-release|200802252348
1075256|1075256-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|Number-Phone|1.58|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200802260012
1076381|1076381-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|alpha-linux-thread-multi|5.11.0 patch 33365|linux|2.6.18-5-alpha-generic|200802260931
1076900|1076900-b19f-3f77-b713-d32bba55d77f|pass|200802|stro@cpan.org|XML-Tiny|1.11|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802260921
1077232|1077232-b19f-3f77-b713-d32bba55d77f|pass|200802|andy@hexten.net|Data-Hexdumper|1.3|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200802261643
1077815|1077815-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|alpha-linux-thread-multi|5.11.0 patch 33365|linux|2.6.18-5-alpha-generic|200802261908
1078107|1078107-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|Number-Phone|1.58|amd64-freebsd|5.10.0|freebsd|6.2-release|200802262125
1078409|1078409-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|CPU-Emulator-Memory|1.0|amd64-freebsd|5.10.0|freebsd|6.2-release|200802262304
1078638|1078638-b19f-3f77-b713-d32bba55d77f|pass|200802|andy@hexten.net|Data-Hexdumper|1.3|i386-netbsd|5.8.8|netbsd|3.1|200802262241
1078761|1078761-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|Number-Phone|1.58|amd64-freebsd|5.6.2|freebsd|6.2-release|200802262356
1079212|1079212-b19f-3f77-b713-d32bba55d77f|pass|200802|stro@cpan.org|NestedMap|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802270116
1079566|1079566-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.42|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200802271221
1079567|1079567-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.42|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802271222
1079627|1079627-b19f-3f77-b713-d32bba55d77f|fail|200802|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.42|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200802271247
1081377|1081377-b19f-3f77-b713-d32bba55d77f|pass|200802|stro@cpan.org|Number-Phone|1.58|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802271210
1082191|1082191-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.581|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200802272252
1082196|1082196-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.581|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200802272253
1082201|1082201-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.581|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200802272254
1082205|1082205-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.581|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200802272254
1082208|1082208-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.581|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200802272256
1082209|1082209-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.581|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200802272256
1082741|1082741-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200802280301
1082847|1082847-b19f-3f77-b713-d32bba55d77f|pass|200802|stro@cpan.org|Statistics-ChiSquare|0.5|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802280359
1083055|1083055-b19f-3f77-b713-d32bba55d77f|pass|200802|stro@cpan.org|Statistics-SerialCorrelation|1.1|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802280146
1083328|1083328-b19f-3f77-b713-d32bba55d77f|pass|200802|d@0x64.nl (DEVOGON)|Number-Phone|1.581|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200802280857
1083450|1083450-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|Number-Phone|1.581|amd64-freebsd|5.6.2|freebsd|6.2-release|200802281005
1083451|1083451-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|Number-Phone|1.581|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200802281005
1083456|1083456-b19f-3f77-b713-d32bba55d77f|na|200802|srezic@cpan.org|Number-Phone|1.581|amd64-freebsd|5.5.5|freebsd|6.2-release|200802280849
1083536|1083536-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|Number-Phone|1.581|amd64-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200802281005
1083539|1083539-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|Number-Phone|1.581|amd64-freebsd|5.10.0|freebsd|6.2-release|200802281005
1083913|1083913-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|Number-Phone|1.581|i686-linux|5.8.8|linux|2.4.27-3-686|200802281253
1083955|1083955-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Number-Phone|1.581|i386-pc-solaris2.11-thread-multi-64int|5.10.0|solaris|2.11|200802281338
1083992|1083992-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|Number-Phone|1.581|i686-linux|5.6.2|linux|2.4.27-3-686|200802281357
1084043|1084043-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|Number-Phone|1.581|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200802281426
1084397|1084397-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|Number-Phone|1.581|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200802281624
1084412|1084412-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|Number-Phone|1.581|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200802281612
1084423|1084423-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|Number-Phone|1.581|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200802281043
1084809|1084809-b19f-3f77-b713-d32bba55d77f|pass|200802|d@0x64.nl|Number-Phone|1.581|i686-linux|5.10.0|linux|2.6.22-3-486|200802282044
1085311|1085311-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200802282220
1085315|1085315-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200802282220
1085318|1085318-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200802282221
1085326|1085326-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200802282222
1085331|1085331-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200802282222
1085334|1085334-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200802282223
1085752|1085752-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.001|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200802290020
1085753|1085753-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.001|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200802290020
1085754|1085754-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.001|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200802290021
1085755|1085755-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.001|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200802290021
1085756|1085756-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.001|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200802290022
1085758|1085758-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.001|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200802290022
1085783|1085783-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200802290024
1085784|1085784-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200802290024
1085785|1085785-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200802290025
1085786|1085786-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200802290025
1085810|1085810-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200802290024
1085811|1085811-b19f-3f77-b713-d32bba55d77f|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200802290024
1085818|1085818-b19f-3f77-b713-d32bba55d77f|pass|200802|stro@cpan.org|Tie-Hash-Longest|1.1|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802290133
1085828|1085828-b19f-3f77-b713-d32bba55d77f|pass|200802|stro@cpan.org|Tie-Hash-Rank|1.0.1|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802290135
1085940|1085940-b19f-3f77-b713-d32bba55d77f|pass|200802|stro@cpan.org|Tie-Scalar-Decay|1.1.1|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802290003
1086158|1086158-b19f-3f77-b713-d32bba55d77f|pass|200802|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.581|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200802290102
1086808|1086808-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|CPU-Emulator-Memory|1.1001|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802291357
1086809|1086809-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|Number-Phone|1.581|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802291358
1086909|1086909-b19f-3f77-b713-d32bba55d77f|pass|200802|d@0x64.nl (DEVOGON)|CPU-Emulator-Memory|1.1001|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200802290725
1086987|1086987-b19f-3f77-b713-d32bba55d77f|pass|200802|stro@cpan.org|XML-DoubleEncodedEntities|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802290139
1087243|1087243-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|Devel-CheckOS|1.42|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200802290837
1087254|1087254-b19f-3f77-b713-d32bba55d77f|na|200802|srezic@cpan.org|CPU-Emulator-Memory|1.1001|amd64-freebsd|5.5.5|freebsd|6.2-release|200802290841
1087257|1087257-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|CPU-Emulator-Memory|1.1001|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200802290841
1087392|1087392-b19f-3f77-b713-d32bba55d77f|pass|200802|imacat@mail.imacat.idv.tw|Number-Phone|1.581|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200802291728
1087462|1087462-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|Devel-CheckOS|1.42|amd64-freebsd|5.10.0|freebsd|6.2-release|200802290958
1087479|1087479-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|CPU-Emulator-Memory|1.1001|amd64-freebsd|5.10.0|freebsd|6.2-release|200802291017
1087738|1087738-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|CPAN-FindDependencies|2.0|amd64-freebsd|5.10.0|freebsd|6.2-release|200802291223
1087808|1087808-b19f-3f77-b713-d32bba55d77f|pass|200802|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPU-Emulator-Memory|1.1001|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200802291206
1087820|1087820-b19f-3f77-b713-d32bba55d77f|pass|200802|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.581|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200802291209
1088420|1088420-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.001|i686-linux|5.8.8|linux|2.4.27-3-686|200802291542
1088421|1088421-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.1|i686-linux|5.8.8|linux|2.4.27-3-686|200802291542
1088423|1088423-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.1001|i686-linux|5.8.8|linux|2.4.27-3-686|200802291542
1088438|1088438-b19f-3f77-b713-d32bba55d77f|fail|200802|srezic@cpan.org|Bryar|3.0|amd64-freebsd|5.10.0|freebsd|6.2-release|200802291520
1088488|1088488-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.001|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200802291646
1088490|1088490-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.1|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200802291647
1088491|1088491-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.1001|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200802291647
1088496|1088496-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.001|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200802291626
1088497|1088497-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.001|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200802291651
1088498|1088498-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.1|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200802291628
1088499|1088499-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.1|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200802291652
1088501|1088501-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200802291652
1088502|1088502-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.001|i686-linux|5.6.2|linux|2.4.27-3-686|200802291653
1088503|1088503-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.1|i686-linux|5.6.2|linux|2.4.27-3-686|200802291653
1088504|1088504-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.1001|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200802291629
1088505|1088505-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.1001|i686-linux|5.6.2|linux|2.4.27-3-686|200802291654
1088757|1088757-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|Devel-CheckOS|1.42|amd64-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200802291930
1089198|1089198-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200802292052
1089200|1089200-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200802292052
1089202|1089202-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200802292053
1089203|1089203-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200802292053
1089205|1089205-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200802292054
1089207|1089207-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200802292054
1089209|1089209-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200802292054
1089210|1089210-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200802292055
1089212|1089212-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200802292055
1089213|1089213-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200802292056
1089296|1089296-b19f-3f77-b713-d32bba55d77f|pass|200802|srezic@cpan.org|CPU-Emulator-Memory|1.1001|amd64-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200802292222
1089383|1089383-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200802292310
1089384|1089384-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200802292311
1089386|1089386-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200802292311
1089387|1089387-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|Number-Phone|1.581|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200802292312
1089556|1089556-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200802292338
1089559|1089559-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200802292339
1089561|1089561-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200802292339
1089562|1089562-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200802292339
1089563|1089563-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200802292340
1089565|1089565-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200802292340
1089566|1089566-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200802292340
1089568|1089568-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200802292341
1089569|1089569-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200802292341
1089571|1089571-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200802292342
1089572|1089572-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200802292342
1089573|1089573-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200802292342
1089574|1089574-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200802292343
1089576|1089576-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200802292343
1089578|1089578-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200802292344
1089580|1089580-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200802292344
1089581|1089581-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200802292344
1089583|1089583-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200802292345
1089586|1089586-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200802292345
1089588|1089588-b19f-3f77-b713-d32bba55d77f|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200802292345
1089617|1089617-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.001|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200802292355
1089621|1089621-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.1|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200802292356
1089625|1089625-b19f-3f77-b713-d32bba55d77f|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.1001|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200802292356
1089803|1089803-b19f-3f77-b713-d32bba55d77f|pass|200803|icerider70@gmail.com ("ICERIDER")|Number-Phone|1.581|i386-linux|5.8.8|linux|2.6.18-53.el5|200803010151
1089978|1089978-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|i686-cygwin-thread-multi-64int|5.11.0 patch 33400|cygwin|1.5.24(0.15642)|200803010318
1089979|1089979-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|i686-cygwin-thread-multi-64int|5.11.0 patch 33400|cygwin|1.5.24(0.15642)|200803010319
1089983|1089983-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i686-cygwin-thread-multi-64int|5.11.0 patch 33400|cygwin|1.5.24(0.15642)|200803010319
1091103|1091103-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|CPU-Emulator-Memory|1.1001|amd64-freebsd|5.6.2|freebsd|6.2-release|200803011302
1091197|1091197-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl (DEVOGON)|CPU-Emulator-Memory|1.1001|i686-linux|5.10.0|linux|2.6.22-3-486|200803011427
1091212|1091212-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl (DEVOGON)|Number-Phone|1.581|i686-linux|5.10.0|linux|2.6.22-3-486|200803011443
1091352|1091352-b19f-3f77-b713-d32bba55d77f|pass|200803|icerider70@gmail.com ("ICERIDER")|CPU-Emulator-Memory|1.0|i386-linux|5.8.8|linux|2.6.18-53.el5|200803010851
1091742|1091742-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200803011746
1091891|1091891-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i686-cygwin-thread-multi-64int|5.11.0 patch 33400|cygwin|1.5.24(0.15642)|200803011756
1092328|1092328-b19f-3f77-b713-d32bba55d77f|fail|200803|andy@hexten.net|Devel-CheckOS|1.42|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200803012024
1092354|1092354-b19f-3f77-b713-d32bba55d77f|pass|200803|imacat@mail.imacat.idv.tw|CPU-Emulator-Memory|1.1|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803020435
1092383|1092383-b19f-3f77-b713-d32bba55d77f|pass|200803|imacat@mail.imacat.idv.tw|CPU-Emulator-Memory|1.001|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803020446
1092386|1092386-b19f-3f77-b713-d32bba55d77f|pass|200803|imacat@mail.imacat.idv.tw|CPU-Emulator-Memory|1.1001|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803020446
1092659|1092659-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200803012108
1092959|1092959-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Devel-CheckLib|0.3|amd64-freebsd|5.10.0|freebsd|6.2-release|200803020039
1093139|1093139-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|XML-Tiny|1.11|amd64-freebsd|5.10.0|freebsd|6.2-release|200803020102
1094820|1094820-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|File-Find-Rule-Permissions|1.3|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200803021432
1094966|1094966-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33408|openbsd|4.2|200803021600
1094968|1094968-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33408|openbsd|4.2|200803021600
1094970|1094970-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33408|openbsd|4.2|200803021601
1094978|1094978-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Number-Phone|1.581|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33408|openbsd|4.2|200803021604
1095292|1095292-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl (DEVOGON)|Devel-CheckOS|1.42|i686-linux-thread-multi|5.11.0 patch 33261|linux|2.6.22-3-486|200803021652
1095759|1095759-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl (DEVOGON)|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi|5.11.0 patch 33261|linux|2.6.22-3-486|200803021909
1095788|1095788-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200803021917
1095789|1095789-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200803021917
1095791|1095791-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200803021918
1095803|1095803-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200803021918
1095808|1095808-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200803021919
1095809|1095809-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200803021919
1095811|1095811-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200803021920
1095813|1095813-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200803021920
1095815|1095815-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200803021921
1095821|1095821-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Number-Phone|1.581|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200803021922
1095822|1095822-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Number-Phone|1.581|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200803021922
1095823|1095823-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Number-Phone|1.581|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200803021923
1096373|1096373-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Data-Hexdumper|1.3|amd64-freebsd|5.6.2|freebsd|6.2-release|200803022143
1096416|1096416-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Tie-STDOUT|1.03|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200803022230
1096935|1096935-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|CPU-Emulator-Memory|1.001|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200803021712
1096946|1096946-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|CPU-Emulator-Memory|1.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200803021713
1096963|1096963-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|CPU-Emulator-Memory|1.1001|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200803021714
1096964|1096964-b19f-3f77-b713-d32bba55d77f|pass|200803|icerider70@gmail.com ("ICERIDER")|CPU-Emulator-Memory|1.1001|i386-linux|5.8.8|linux|2.6.18-53.el5|200803021714
1097054|1097054-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|amd64-freebsd|5.10.0|freebsd|6.2-release|200803030000
1097084|1097084-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.42|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33408|openbsd|4.2|200803030214
1098112|1098112-b19f-3f77-b713-d32bba55d77f|pass|200803|stro@cpan.org|Number-Phone|1.581|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200803031135
1098687|1098687-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl ("DEVOGON")|CPU-Emulator-Memory|1.1001|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803031242
1099024|1099024-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|alpha-linux-thread-multi|5.11.0 patch 33411|linux|2.6.18-5-alpha-generic|200803031447
1099026|1099026-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|alpha-linux-thread-multi|5.11.0 patch 33411|linux|2.6.18-5-alpha-generic|200803031448
1099029|1099029-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|alpha-linux-thread-multi|5.11.0 patch 33411|linux|2.6.18-5-alpha-generic|200803031448
1099031|1099031-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Number-Phone|1.581|alpha-linux-thread-multi|5.11.0 patch 33411|linux|2.6.18-5-alpha-generic|200803031450
1099628|1099628-b19f-3f77-b713-d32bba55d77f|pass|200803|stro@cpan.org|CPAN-FindDependencies|2.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803032027
1099664|1099664-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200803040107
1099666|1099666-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200803040108
1099669|1099669-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200803040108
1099674|1099674-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Number-Phone|1.581|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200803040111
1100247|1100247-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Data-Transactional|1.0|amd64-freebsd|5.6.2|freebsd|6.2-release|200803032215
1100248|1100248-b19f-3f77-b713-d32bba55d77f|na|200803|srezic@cpan.org|Tie-STDOUT|1.03|amd64-freebsd|5.6.2|freebsd|6.2-release|200803032215
1100533|1100533-b19f-3f77-b713-d32bba55d77f|pass|200803|stro@cpan.org|Data-Compare|0.17|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803040008
1100537|1100537-b19f-3f77-b713-d32bba55d77f|pass|200803|stro@cpan.org|Data-Compare|1.18|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803040009
1100802|1100802-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.11.0 patch 33423|linux|2.6.22-1-k7|200803032326
1100928|1100928-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux|5.11.0 patch 33423|linux|2.6.22-1-k7|200803040020
1101250|1101250-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux|5.11.0 patch 33423|linux|2.6.22-1-k7|200803040142
1101756|1101756-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.11.0 patch 33423|linux|2.6.22-1-k7|200803040434
1101874|1101874-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.42|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200803041208
1102093|1102093-b19f-3f77-b713-d32bba55d77f|pass|200803|stro@cpan.org|Data-Transactional|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803040643
1103433|1103433-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.11.0 patch 33423|linux|2.6.22-1-k7|200803040929
1103711|1103711-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi|5.8.8 patch 33430|linux|2.6.22-1-k7|200803041016
1104229|1104229-b19f-3f77-b713-d32bba55d77f|pass|200803|Jost.Krieger@ruhr-uni-bochum.de|Devel-CheckOS|1.42|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200803041047
1104296|1104296-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-thread-multi|5.8.8 patch 33430|linux|2.6.22-1-k7|200803041207
1104629|1104629-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux-thread-multi|5.8.8 patch 33430|linux|2.6.22-1-k7|200803041241
1104915|1104915-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.42|alpha-linux-thread-multi|5.11.0 patch 33411|linux|2.6.18-5-alpha-generic|200803041323
1105211|1105211-b19f-3f77-b713-d32bba55d77f|pass|200803|stro@cpan.org|Devel-CheckOS|1.42|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803041613
1105438|1105438-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.8.8|linux|2.6.14|200803041602
1105779|1105779-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi|5.8.8 patch 33430|linux|2.6.22-1-k7|200803041713
1105891|1105891-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.8.8 patch 33430|linux|2.6.22-1-k7|200803041708
1105955|1105955-b19f-3f77-b713-d32bba55d77f|pass|200803|andy@hexten.net|Number-Phone|1.581|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200803041654
1107340|1107340-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-thread-multi-64int|5.10.0 patch 33412|linux|2.6.22-1-k7|200803042222
1107362|1107362-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-thread-multi-64int|5.10.0 patch 33412|linux|2.6.22-1-k7|200803042255
1107366|1107366-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200803042256
1107381|1107381-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-thread-multi|5.8.8 patch 33430|linux|2.6.22-1-k7|200803042257
1107382|1107382-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-64int|5.8.8|linux|2.6.14|200803042257
1107384|1107384-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200803042259
1107386|1107386-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200803042259
1107387|1107387-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200803042300
1107408|1107408-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-64int|5.8.8 patch 33243|linux|2.6.22-1-k7|200803042257
1107423|1107423-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi-64int|5.10.0 patch 33412|linux|2.6.22-1-k7|200803042251
1107614|1107614-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.10.0 patch 33412|linux|2.6.22-1-k7|200803042331
1108058|1108058-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200803042331
1108061|1108061-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200803042332
1108065|1108065-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200803042333
1108080|1108080-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Number-Phone|1.581|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200803042335
1109224|1109224-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int|5.10.0 patch 33412|linux|2.6.22-1-k7|200803050548
1109254|1109254-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200803050457
1109259|1109259-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200803050458
1109263|1109263-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200803050459
1109269|1109269-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200803050500
1109275|1109275-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200803050501
1109279|1109279-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200803050501
1109282|1109282-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200803050502
1109284|1109284-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200803050503
1109288|1109288-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200803050504
1109292|1109292-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200803050504
1109373|1109373-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux|5.11.0 patch 33423|linux|2.6.22-1-k7|200803050617
1109863|1109863-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Devel-CheckLib|0.4|amd64-freebsd|5.6.2|freebsd|6.2-release|200803050722
1109864|1109864-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Devel-CheckLib|0.4|amd64-freebsd|5.10.0|freebsd|6.2-release|200803050722
1109984|1109984-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Devel-CheckLib|0.4|amd64-freebsd|5.5.5|freebsd|6.2-release|200803050836
1110587|1110587-b19f-3f77-b713-d32bba55d77f|pass|200803|imacat@mail.imacat.idv.tw|Devel-CheckLib|0.4|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200803051857
1110726|1110726-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|alpha-linux-thread-multi|5.11.0 patch 33411|linux|2.6.18-5-alpha-generic|200803051124
1110843|1110843-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|alpha-linux-thread-multi|5.11.0 patch 33411|linux|2.6.18-5-alpha-generic|200803051203
1111089|1111089-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200803051404
1111091|1111091-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200803051405
1111095|1111095-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200803051407
1111102|1111102-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200803051408
1111109|1111109-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Number-Phone|1.581|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200803051410
1111240|1111240-b19f-3f77-b713-d32bba55d77f|pass|200803|andy@hexten.net|Devel-CheckLib|0.4|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200803051449
1111292|1111292-b19f-3f77-b713-d32bba55d77f|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckLib|0.4|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200803051454
1111491|1111491-b19f-3f77-b713-d32bba55d77f|pass|200803|andy@hexten.net|Number-Phone|1.581|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200803051556
1111498|1111498-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200803051558
1111809|1111809-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.4|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200803051707
1111850|1111850-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.4|i686-linux|5.6.2|linux|2.4.27-3-686|200803051716
1111851|1111851-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.4|i686-linux|5.8.8|linux|2.4.27-3-686|200803051717
1112005|1112005-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.4|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200803051747
1112021|1112021-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|File-Find-Rule-Permissions|1.3|amd64-freebsd|5.10.0|freebsd|6.2-release|200803051839
1112103|1112103-b19f-3f77-b713-d32bba55d77f|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPU-Emulator-Memory|1.1001|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200803051758
1112105|1112105-b19f-3f77-b713-d32bba55d77f|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckLib|0.4|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200803051759
1112985|1112985-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Devel-CheckLib|0.4|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200803052123
1113234|1113234-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl ("DEVOGON")|Number-Phone|1.581|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803052211
1113289|1113289-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl (DEVOGON)|Devel-CheckLib|0.4|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803052224
1113959|1113959-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux|5.8.8 patch 33436|linux|2.6.22-1-k7|200803060059
1114475|1114475-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.8.8 patch 33436|linux|2.6.22-1-k7|200803060313
1115303|1115303-b19f-3f77-b713-d32bba55d77f|pass|200803|stro@cpan.org|Net-Random|2.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803060814
1115390|1115390-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-64int|5.10.0 patch 33443|linux|2.6.22-1-k7|200803060747
1115424|1115424-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 33443|linux|2.6.22-1-k7|200803060744
1117012|1117012-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.10.0 patch 33443|linux|2.6.22-1-k7|200803061219
1117025|1117025-b19f-3f77-b713-d32bba55d77f|pass|200803|stro@cpan.org|Number-Phone|1.581|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803061327
1117073|1117073-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|i386-netbsd-thread-multi-64int|5.11.0 patch 33445|netbsd|4.0|200803061137
1117075|1117075-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|i386-netbsd-thread-multi-64int|5.11.0 patch 33445|netbsd|4.0|200803061137
1117077|1117077-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.11.0 patch 33445|netbsd|4.0|200803061138
1117079|1117079-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i386-netbsd-thread-multi-64int|5.11.0 patch 33445|netbsd|4.0|200803061138
1117086|1117086-b19f-3f77-b713-d32bba55d77f|pass|200803|stro@cpan.org|Number-Phone-UK-DetailedLocations|1.3|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803061340
1117231|1117231-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.11.0 patch 33445|netbsd|4.0|200803061205
1117687|1117687-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200803061318
1117693|1117693-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200803061319
1117694|1117694-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200803061320
1118024|1118024-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.42|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200803061412
1118753|1118753-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl ("DEVOGON")|Devel-CheckLib|0.4|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803061721
1118838|1118838-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 33443|linux|2.6.22-1-k7|200803061739
1119041|1119041-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl ("DEVOGON")|CPAN-FindDependencies|2.0|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803061805
1119501|1119501-b19f-3f77-b713-d32bba55d77f|pass|200803|imacat@mail.imacat.idv.tw|Devel-CheckLib|0.4|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803070159
1119566|1119566-b19f-3f77-b713-d32bba55d77f|pass|200803|stro@cpan.org|Palm-TreoPhoneCallDB|1.1|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803062004
1119718|1119718-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-64int|5.10.0 patch 33443|linux|2.6.22-1-k7|200803061924
1119800|1119800-b19f-3f77-b713-d32bba55d77f|pass|200803|andy@hexten.net|Devel-CheckLib|0.4|i686-linux-thread-multi|5.10.0|linux|2.6.20-16-server|200803061839
1119955|1119955-b19f-3f77-b713-d32bba55d77f|pass|200803|andy@hexten.net|Data-Compare|1.18|i686-linux-thread-multi|5.10.0|linux|2.6.20-16-server|200803061932
1120096|1120096-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.11.0 patch 33445|netbsd|4.0|200803062000
1120629|1120629-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Devel-CheckOS|1.42|amd64-freebsd|5.6.2|freebsd|6.2-release|200803062319
1120958|1120958-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.11.0 patch 33450|linux|2.6.22-1-k7|200803070001
1121130|1121130-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.4|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200803062341
1121862|1121862-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi|5.11.0 patch 33450|linux|2.6.22-1-k7|200803070301
1122380|1122380-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-thread-multi|5.11.0 patch 33450|linux|2.6.22-1-k7|200803070500
1122639|1122639-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-thread-multi|5.11.0 patch 33450|linux|2.6.22-1-k7|200803070021
1122686|1122686-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi|5.11.0 patch 33450|linux|2.6.22-1-k7|200803070547
1123451|1123451-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200803070919
1123847|1123847-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-thread-multi-64int|5.8.8 patch 33449|linux|2.6.22-1-k7|200803071224
1123868|1123868-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-thread-multi-64int|5.8.8 patch 33449|linux|2.6.22-1-k7|200803071229
1124313|1124313-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi-64int|5.8.8 patch 33449|linux|2.6.22-1-k7|200803071514
1124659|1124659-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.8.8 patch 33449|linux|2.6.22-1-k7|200803071714
1124710|1124710-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int|5.8.8 patch 33449|linux|2.6.22-1-k7|200803071711
1125157|1125157-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl|Devel-CheckLib|0.4|i686-linux|5.10.0|linux|2.6.22-3-486|200803071854
1125298|1125298-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl|Data-Compare|1.18|i686-linux|5.10.0|linux|2.6.22-3-486|200803071916
1126853|1126853-b19f-3f77-b713-d32bba55d77f|fail|200803|d@0x64.nl ("DEVOGON")|Bryar|3.0|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803072259
1126885|1126885-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.4|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200803071606
1126910|1126910-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl ("DEVOGON")|Devel-CheckOS|1.42|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803072306
1126958|1126958-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.4|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200803072218
1127166|1127166-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Devel-CheckLib|0.4|amd64-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200803080007
1127202|1127202-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i686-cygwin-thread-multi-64int|5.11.0 patch 33450|cygwin|1.5.24(0.15642)|200803072324
1127604|1127604-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl|CPU-Emulator-Memory|1.1001|i686-linux|5.10.0|linux|2.6.22-3-486|200803080142
1127792|1127792-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|i686-cygwin-thread-multi-64int|5.11.0 patch 33450|cygwin|1.5.24(0.15642)|200803080054
1128559|1128559-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200803080351
1128897|1128897-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux|5.11.0 patch 33451|linux|2.6.22-1-k7|200803080630
1129275|1129275-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl (DEVOGON)|Devel-CheckLib|0.4|i686-linux|5.10.0|linux|2.6.22-3-486|200803080819
1129286|1129286-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.0|darwin-2level|5.8.8|darwin|8.10.1|200803080814
1129634|1129634-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.11.0 patch 33451|linux|2.6.22-1-k7|200803080853
1129685|1129685-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.11.0 patch 33451|linux|2.6.22-1-k7|200803080905
1130112|1130112-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux|5.11.0 patch 33451|linux|2.6.22-1-k7|200803080921
1131122|1131122-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i686-cygwin-thread-multi-64int|5.11.0 patch 33450|cygwin|1.5.24(0.15642)|200803081156
1131719|1131719-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.11.0 patch 33451|linux|2.6.22-1-k7|200803081236
1132402|1132402-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.11.0 patch 33452|linux|2.6.22-1-k7|200803081702
1132651|1132651-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux|5.11.0 patch 33452|linux|2.6.22-1-k7|200803081738
1132947|1132947-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.11.0 patch 33452|linux|2.6.22-1-k7|200803081626
1133084|1133084-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|darwin-2level|5.8.8|darwin|8.10.1|200803081604
1133252|1133252-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux|5.11.0 patch 33452|linux|2.6.22-1-k7|200803081858
1133258|1133258-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|File-Find-Rule-Permissions|1.3|amd64-freebsd|5.6.2|freebsd|6.2-release|200803081629
1133307|1133307-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Devel-CheckOS|1.42|amd64-freebsd-thread-multi|5.8.8|freebsd|6.2-release|200803081632
1134224|1134224-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.11.0 patch 33452|linux|2.6.22-1-k7|200803082210
1134295|1134295-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl|XML-DoubleEncodedEntities|1.0|i686-linux|5.10.0|linux|2.6.22-3-486|200803082233
1134317|1134317-b19f-3f77-b713-d32bba55d77f|pass|200803|stro@cpan.org|Sub-WrapPackages|1.2|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803082331
1135446|1135446-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200803090046
1135548|1135548-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200803090105
1136200|1136200-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Net-Random|2.0|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200803090407
1137541|1137541-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl ("DEVOGON")|XML-Tiny|1.11|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803091228
1137905|1137905-b19f-3f77-b713-d32bba55d77f|pass|200803|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.4|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200803090836
1137986|1137986-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Data-Compare|1.18|amd64-freebsd-thread-multi|5.8.8|freebsd|6.2-release|200803091335
1138137|1138137-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Devel-CheckLib|0.4|amd64-freebsd-thread-multi|5.8.8|freebsd|6.2-release|200803091353
1138692|1138692-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-64int|5.8.8 patch 33454|linux|2.6.22-1-k7|200803091548
1139225|1139225-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-64int|5.8.8 patch 33454|linux|2.6.22-1-k7|200803091727
1139601|1139601-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.8.8 patch 33454|linux|2.6.22-1-k7|200803091824
1139874|1139874-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Number-Phone-UK-DetailedLocations|1.3|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200803091845
1140020|1140020-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200803091835
1140224|1140224-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200803091919
1140377|1140377-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200803091942
1140430|1140430-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 33454|linux|2.6.22-1-k7|200803092053
1140490|1140490-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|amd64-freebsd|5.6.2|freebsd|6.2-release|200803092059
1140822|1140822-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.8.8 patch 33454|linux|2.6.22-1-k7|200803092158
1141239|1141239-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl (DEVOGON)|Devel-CheckLib|0.4|i686-linux-thread-multi|5.11.0 patch 33261|linux|2.6.22-3-486|200803092254
1141354|1141354-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 33454|linux|2.6.22-1-k7|200803100000
1141393|1141393-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl (DEVOGON)|Data-Compare|1.18|i686-linux-thread-multi|5.11.0 patch 33261|linux|2.6.22-3-486|200803100018
1141448|1141448-b19f-3f77-b713-d32bba55d77f|pass|200803|andy@hexten.net|Devel-CheckLib|0.4|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200803092324
1142886|1142886-b19f-3f77-b713-d32bba55d77f|pass|200803|daxim@cpan.org|XML-DoubleEncodedEntities|1.0|x86_64-linux|5.8.8|linux|2.6.20-gentoo-r8|200803092241
1143297|1143297-b19f-3f77-b713-d32bba55d77f|pass|200803|andy@hexten.net|Devel-CheckLib|0.4|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200803101156
1143388|1143388-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-thread-multi|5.11.0 patch 33457|linux|2.6.22-1-k7|200803101336
1144603|1144603-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|alpha-linux-thread-multi|5.11.0 patch 33456|linux|2.6.18-5-alpha-generic|200803101704
1144632|1144632-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.11.0 patch 33457|linux|2.6.22-1-k7|200803101810
1144753|1144753-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|alpha-linux-thread-multi|5.11.0 patch 33456|linux|2.6.18-5-alpha-generic|200803101728
1145110|1145110-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi|5.11.0 patch 33457|linux|2.6.22-1-k7|200803101835
1146007|1146007-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-thread-multi|5.11.0 patch 33457|linux|2.6.22-1-k7|200803102130
1146153|1146153-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-thread-multi|5.11.0 patch 33457|linux|2.6.22-1-k7|200803102210
1146201|1146201-b19f-3f77-b713-d32bba55d77f|pass|200803|schwern@pobox.com|Devel-CheckLib|0.4|darwin-thread-multi-2level|5.8.8|darwin|8.8.1|200803101116
1146230|1146230-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi|5.11.0 patch 33457|linux|2.6.22-1-k7|200803102214
1147736|1147736-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i686-linux-thread-multi-64int|5.11.0 patch 33467|linux|2.6.16.38|200803110257
1147779|1147779-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|i686-linux-thread-multi-64int|5.11.0 patch 33467|linux|2.6.16.38|200803110307
1147933|1147933-b19f-3f77-b713-d32bba55d77f|fail|200803|imacat@mail.imacat.idv.tw|Palm-SMS|0.02|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200803111159
1148083|1148083-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|darwin-2level|5.8.8|darwin|8.10.1|200803110610
1148832|1148832-b19f-3f77-b713-d32bba55d77f|pass|200803|stro@cpan.org|CPU-Emulator-Memory|1.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200803111132
1148834|1148834-b19f-3f77-b713-d32bba55d77f|pass|200803|stro@cpan.org|CPU-Emulator-Memory|1.001|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200803111133
1148837|1148837-b19f-3f77-b713-d32bba55d77f|pass|200803|stro@cpan.org|CPU-Emulator-Memory|1.1|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200803111133
1148839|1148839-b19f-3f77-b713-d32bba55d77f|pass|200803|stro@cpan.org|CPU-Emulator-Memory|1.1001|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200803111133
1148988|1148988-b19f-3f77-b713-d32bba55d77f|pass|200803|stro@cpan.org|CPU-Emulator-Memory|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803111231
1148993|1148993-b19f-3f77-b713-d32bba55d77f|pass|200803|stro@cpan.org|CPU-Emulator-Memory|1.001|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803111232
1148994|1148994-b19f-3f77-b713-d32bba55d77f|pass|200803|stro@cpan.org|CPU-Emulator-Memory|1.1|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803111233
1148995|1148995-b19f-3f77-b713-d32bba55d77f|pass|200803|stro@cpan.org|CPU-Emulator-Memory|1.1001|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803111234
1149737|1149737-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i686-linux-thread-multi-64int|5.11.0 patch 33467|linux|2.6.16.38|200803111346
1150508|1150508-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl ("DEVOGON")|File-Find-Rule-Permissions|1.3|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803111731
1151369|1151369-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.42|alpha-linux-thread-multi|5.11.0 patch 33456|linux|2.6.18-5-alpha-generic|200803111923
1151818|1151818-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-thread-multi|5.11.0 patch 33470|linux|2.6.22-1-k7|200803112101
1152039|1152039-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi|5.11.0 patch 33470|linux|2.6.22-1-k7|200803112202
1152397|1152397-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-thread-multi|5.11.0 patch 33470|linux|2.6.22-1-k7|200803112317
1152971|1152971-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-thread-multi|5.11.0 patch 33470|linux|2.6.22-1-k7|200803120059
1153370|1153370-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.11.0 patch 33470|linux|2.6.22-1-k7|200803120240
1153517|1153517-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi|5.11.0 patch 33470|linux|2.6.22-1-k7|200803120254
1153896|1153896-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200803120341
1154487|1154487-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-thread-multi-64int|5.11.0 patch 33485|linux|2.6.22-1-k7|200803120848
1155279|1155279-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.11.0 patch 33485|linux|2.6.22-1-k7|200803121135
1155343|1155343-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi-64int|5.11.0 patch 33485|linux|2.6.22-1-k7|200803121113
1155416|1155416-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int|5.11.0 patch 33485|linux|2.6.22-1-k7|200803121030
1155581|1155581-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-thread-multi-64int|5.11.0 patch 33485|linux|2.6.22-1-k7|200803121207
1156349|1156349-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-thread-multi-64int|5.11.0 patch 33485|linux|2.6.22-1-k7|200803121248
1157352|1157352-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 33443|linux|2.6.22-1-k7|200803122103
1157355|1157355-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-thread-multi-64int|5.10.0 patch 33412|linux|2.6.22-1-k7|200803122104
1157358|1157358-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200803122104
1157363|1157363-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-thread-multi|5.8.8 patch 33430|linux|2.6.22-1-k7|200803122105
1157366|1157366-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 33243|linux|2.6.22-1-k7|200803122105
1157369|1157369-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200803122106
1157374|1157374-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8|linux|2.6.14|200803122107
1157385|1157385-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200803122107
1157390|1157390-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200803122107
1157394|1157394-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200803122108
1157434|1157434-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.10.0 patch 33443|linux|2.6.22-1-k7|200803122119
1157436|1157436-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-thread-multi-64int|5.10.0 patch 33412|linux|2.6.22-1-k7|200803122120
1157438|1157438-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200803122120
1157439|1157439-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-thread-multi|5.8.8 patch 33430|linux|2.6.22-1-k7|200803122121
1157441|1157441-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 33243|linux|2.6.22-1-k7|200803122121
1157443|1157443-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200803122122
1157447|1157447-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8|linux|2.6.14|200803122122
1157449|1157449-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200803122123
1157450|1157450-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200803122124
1157454|1157454-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200803122125
1157638|1157638-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|darwin-2level|5.8.8|darwin|8.10.1|200803122209
1157643|1157643-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|darwin-2level|5.8.8|darwin|8.10.1|200803122205
1158128|1158128-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl (DEVOGON)|Devel-CheckLib|0.5|i686-linux-thread-multi|5.11.0 patch 33261|linux|2.6.22-3-486|200803122231
1158271|1158271-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-64int|5.10.0 patch 33443|linux|2.6.22-1-k7|200803122351
1158283|1158283-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int|5.10.0 patch 33412|linux|2.6.22-1-k7|200803122353
1158286|1158286-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|0.9|i686-linux-thread-multi-64int|5.10.0 patch 33412|linux|2.6.22-1-k7|200803122354
1158289|1158289-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|0.9|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200803122355
1158291|1158291-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi|5.8.8 patch 33430|linux|2.6.22-1-k7|200803122355
1158292|1158292-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|0.9|i686-linux-thread-multi|5.8.8 patch 33430|linux|2.6.22-1-k7|200803122356
1158310|1158310-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|0.9|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200803122359
1158312|1158312-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-64int|5.8.8|linux|2.6.14|200803130000
1158313|1158313-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|0.9|i686-linux-64int|5.8.8|linux|2.6.14|200803130001
1158316|1158316-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|0.9|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200803130003
1158400|1158400-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|0.9|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200803130004
1158411|1158411-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-64int|5.8.8 patch 33243|linux|2.6.22-1-k7|200803122357
1158412|1158412-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200803122359
1158422|1158422-b19f-3f77-b713-d32bba55d77f|pass|200803|g.grigelionis@computer.org|Devel-CheckOS|1.43|i586-linux-thread-multi|5.8.8|linux|2.6.16|200803122325
1158434|1158434-b19f-3f77-b713-d32bba55d77f|pass|200803|g.grigelionis@computer.org|Devel-CheckOS|1.43|i386-linux|5.8.0|linux|2.6.9-22.18.bz155725.elsmp|200803130003
1158441|1158441-b19f-3f77-b713-d32bba55d77f|pass|200803|g.grigelionis@computer.org|Devel-CheckOS|1.43|x86_64-linux-thread-multi|5.8.3|linux|2.6.4|200803130011
1158519|1158519-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|0.9|i686-linux-64int|5.8.8 patch 33243|linux|2.6.22-1-k7|200803122357
1158521|1158521-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|0.9|i686-linux-64int|5.10.0 patch 33443|linux|2.6.22-1-k7|200803122352
1158548|1158548-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-thread-multi-64int|5.8.8 patch 33497|linux|2.6.22-1-k7|200803122351
1158832|1158832-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi-64int|5.8.8 patch 33497|linux|2.6.22-1-k7|200803130127
1159673|1159673-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-thread-multi-64int|5.8.8 patch 33497|linux|2.6.22-1-k7|200803130437
1160079|1160079-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.8.8 patch 33497|linux|2.6.22-1-k7|200803130539
1160098|1160098-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int|5.8.8 patch 33497|linux|2.6.22-1-k7|200803130545
1160108|1160108-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-thread-multi-64int|5.8.8 patch 33497|linux|2.6.22-1-k7|200803130546
1160637|1160637-b19f-3f77-b713-d32bba55d77f|pass|200803|hohokus@gmail.com|Devel-CheckOS|1.43|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200803121942
1160646|1160646-b19f-3f77-b713-d32bba55d77f|pass|200803|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|CPU-Emulator-Memory|1.1001|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200803130956
1160656|1160656-b19f-3f77-b713-d32bba55d77f|pass|200803|artur+cpan@niif.spb.su|Devel-CheckOS|1.43|i386-linux|5.8.8|linux|2.6.18-53.1.4.el5.bz358661.1|200803130833
1160961|1160961-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|CPU-Emulator-Z80|0.9|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200803130957
1160963|1160963-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.5|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200803130957
1160965|1160965-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl|CPU-Emulator-Z80|0.9|i686-linux|5.10.0|linux|2.6.22-3-486|200803131104
1160988|1160988-b19f-3f77-b713-d32bba55d77f|pass|200803|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Devel-CheckOS|1.43|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200803130949
1160990|1160990-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.22-3-486|200803131109
1160994|1160994-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl|Devel-CheckLib|0.5|i686-linux|5.10.0|linux|2.6.22-3-486|200803131111
1161055|1161055-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|Devel-CheckOS|1.43|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200803131026
1161078|1161078-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|CPU-Emulator-Z80|0.9|i686-linux|5.8.8|linux|2.4.27-3-686|200803131035
1161086|1161086-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.5|i686-linux|5.8.8|linux|2.4.27-3-686|200803131036
1161091|1161091-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|CPU-Emulator-Z80|0.9|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200803131041
1161093|1161093-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|CPU-Emulator-Z80|0.9|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200803131039
1161099|1161099-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.5|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200803131044
1161521|1161521-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200803131308
1161673|1161673-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux|5.10.0 patch 33501|linux|2.6.22-1-k7|200803131453
1161703|1161703-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.10.0 patch 33501|linux|2.6.22-1-k7|200803131500
1161788|1161788-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.5|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200803131414
1161794|1161794-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|Devel-CheckOS|1.43|i686-linux|5.8.8|linux|2.4.27-3-686|200803131415
1161800|1161800-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200803131418
1161845|1161845-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|Devel-CheckOS|1.43|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200803131424
1161859|1161859-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.5|i686-linux|5.6.2|linux|2.4.27-3-686|200803131427
1161889|1161889-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|Devel-CheckOS|1.43|i686-linux|5.6.2|linux|2.4.27-3-686|200803131432
1161990|1161990-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl (DEVOGON)|Devel-CheckLib|0.5|i686-linux|5.10.0|linux|2.6.22-3-486|200803131558
1162040|1162040-b19f-3f77-b713-d32bba55d77f|fail|200803|d@0x64.nl (DEVOGON)|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.22-3-486|200803131611
1162060|1162060-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200803131508
1162067|1162067-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.22-3-486|200803131616
1162156|1162156-b19f-3f77-b713-d32bba55d77f|pass|200803|justinr@pollackassociates.com|Devel-CheckOS|1.43|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200803130840
1162359|1162359-b19f-3f77-b713-d32bba55d77f|pass|200803|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200803131154
1162362|1162362-b19f-3f77-b713-d32bba55d77f|pass|200803|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|0.9|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200803131155
1162405|1162405-b19f-3f77-b713-d32bba55d77f|pass|200803|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200803131204
1162419|1162419-b19f-3f77-b713-d32bba55d77f|pass|200803|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200803131205
1162522|1162522-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl (DEVOGON)|Devel-CheckOS|1.43|i686-linux-thread-multi|5.11.0 patch 33261|linux|2.6.22-3-486|200803131630
1162536|1162536-b19f-3f77-b713-d32bba55d77f|pass|200803|rlane10@wi.rr.com|Devel-CheckOS|1.43|darwin-2level-ld|5.10.0|darwin|8.11.0|200803130855
1163299|1163299-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|CPU-Emulator-Z80|0.9|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200803131849
1163308|1163308-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.5|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200803131851
1163319|1163319-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|Devel-CheckOS|1.43|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200803131852
1163412|1163412-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.10.0 patch 33501|linux|2.6.22-1-k7|200803132004
1163421|1163421-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.10.0 patch 33501|linux|2.6.22-1-k7|200803132011
1163434|1163434-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl (DEVOGON)|Number-Phone|1.581|i686-linux-thread-multi|5.11.0 patch 33261|linux|2.6.22-3-486|200803131922
1163766|1163766-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Devel-CheckOS|1.43|amd64-freebsd|5.10.0|freebsd|6.2-release|200803132120
1163769|1163769-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Devel-CheckLib|0.5|amd64-freebsd|5.10.0|freebsd|6.2-release|200803132120
1163771|1163771-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Devel-CheckLib|0.5|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200803132121
1163772|1163772-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Devel-CheckOS|1.43|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200803132121
1163818|1163818-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux|5.10.0 patch 33501|linux|2.6.22-1-k7|200803132126
1164045|1164045-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux|5.10.0 patch 33501|linux|2.6.22-1-k7|200803132031
1164132|1164132-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|CPU-Emulator-Z80|0.9|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200803132151
1164362|1164362-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Devel-CheckOS|1.43|amd64-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200803132246
1164411|1164411-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|CPU-Emulator-Z80|0.9|amd64-freebsd-thread-multi|5.8.8|freebsd|6.2-release|200803132303
1164413|1164413-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|CPU-Emulator-Memory|1.1001|amd64-freebsd-thread-multi|5.8.8|freebsd|6.2-release|200803132302
1164476|1164476-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Devel-CheckLib|0.5|amd64-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200803132247
1164479|1164479-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|CPU-Emulator-Z80|0.9|amd64-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200803132250
1164492|1164492-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Devel-CheckOS|1.43|amd64-freebsd-thread-multi|5.8.8|freebsd|6.2-release|200803132301
1164500|1164500-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Devel-CheckLib|0.5|amd64-freebsd-thread-multi|5.8.8|freebsd|6.2-release|200803132301
1164748|1164748-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|CPAN-FindDependencies|2.0|amd64-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200803140009
1164865|1164865-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|CPU-Emulator-Z80|0.9|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200803131853
1164866|1164866-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.5|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200803131854
1164886|1164886-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200803140006
1164887|1164887-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200803140007
1164888|1164888-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200803140008
1164889|1164889-b19f-3f77-b713-d32bba55d77f|pass|200803|david@cantrell.org.uk|Devel-CheckOS|1.43|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200803140009
1164896|1164896-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200803140013
1164899|1164899-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200803140014
1164902|1164902-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200803140015
1164995|1164995-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200803140114
1165001|1165001-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200803140118
1165072|1165072-b19f-3f77-b713-d32bba55d77f|pass|200803|imacat@mail.imacat.idv.tw|CPU-Emulator-Z80|0.9|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200803141048
1165073|1165073-b19f-3f77-b713-d32bba55d77f|pass|200803|imacat@mail.imacat.idv.tw|Devel-CheckLib|0.5|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200803141048
1165074|1165074-b19f-3f77-b713-d32bba55d77f|pass|200803|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.43|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200803141048
1165344|1165344-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200803140603
1165350|1165350-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200803140604
1165353|1165353-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200803140605
1165354|1165354-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200803140606
1165541|1165541-b19f-3f77-b713-d32bba55d77f|pass|200803|imacat@mail.imacat.idv.tw|Devel-CheckLib|0.5|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803141620
1165549|1165549-b19f-3f77-b713-d32bba55d77f|pass|200803|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.43|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803141622
1165579|1165579-b19f-3f77-b713-d32bba55d77f|pass|200803|imacat@mail.imacat.idv.tw|CPU-Emulator-Z80|0.9|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803141634
1165909|1165909-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200803141047
1166648|1166648-b19f-3f77-b713-d32bba55d77f|pass|200803|dillman@ria.buffalo.edu|Devel-CheckOS|1.43|sun4-solaris|5.10.0|solaris|2.9|200803141303
1167659|1167659-b19f-3f77-b713-d32bba55d77f|pass|200803|pjacklam@online.no|Devel-CheckOS|1.43|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200803142333
1167878|1167878-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200803150009
1167910|1167910-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200803150015
1167915|1167915-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200803150016
1167924|1167924-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200803150018
1168085|1168085-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33523|linux|2.6.22-1-k7|200803150144
1168411|1168411-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33523|linux|2.6.22-1-k7|200803150255
1168612|1168612-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|cygwin-thread-multi-64int|5.11.0 patch 33523|cygwin|1.5.24(0.15642)|200803150315
1168614|1168614-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|cygwin-thread-multi-64int|5.11.0 patch 33523|cygwin|1.5.24(0.15642)|200803150316
1168616|1168616-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|cygwin-thread-multi-64int|5.11.0 patch 33523|cygwin|1.5.24(0.15642)|200803150318
1168618|1168618-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|cygwin-thread-multi-64int|5.11.0 patch 33523|cygwin|1.5.24(0.15642)|200803150320
1168671|1168671-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200803150345
1168674|1168674-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200803150347
1168676|1168676-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200803150349
1168679|1168679-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200803150350
1168682|1168682-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200803150352
1168773|1168773-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200803150425
1168780|1168780-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200803150426
1168783|1168783-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200803150428
1168787|1168787-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200803150429
1168791|1168791-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200803150431
1168843|1168843-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200803150452
1168844|1168844-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200803150453
1168847|1168847-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200803150455
1168848|1168848-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200803150456
1168850|1168850-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200803150457
1168852|1168852-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200803150459
1168855|1168855-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200803150500
1168857|1168857-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200803150501
1168862|1168862-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200803150502
1168863|1168863-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200803150504
1169140|1169140-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Devel-CheckOS|1.43|amd64-freebsd|5.6.2|freebsd|6.2-release|200803150824
1169142|1169142-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Devel-CheckLib|0.5|amd64-freebsd|5.6.2|freebsd|6.2-release|200803150824
1170491|1170491-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|cygwin-thread-multi-64int|5.11.0 patch 33523|cygwin|1.5.24(0.15642)|200803151654
1171035|1171035-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl (DEVOGON)|Devel-CheckLib|0.5|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803152243
1171036|1171036-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl (DEVOGON)|Devel-CheckOS|1.43|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803152243
1171715|1171715-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-thread-multi|5.8.8 patch 33536|linux|2.6.22-1-k7|200803160058
1172433|1172433-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200803160609
1172435|1172435-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200803160609
1172436|1172436-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200803160610
1172438|1172438-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200803160611
1172467|1172467-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200803160656
1172895|1172895-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|CPAN-FindDependencies|2.0|amd64-freebsd|5.8.8|freebsd|6.3-stable|200803161138
1173125|1173125-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Data-Compare|1.18|amd64-freebsd|5.8.8|freebsd|6.3-stable|200803161212
1173183|1173183-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Devel-CheckOS|1.43|amd64-freebsd|5.8.8|freebsd|6.3-stable|200803161216
1173817|1173817-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi|5.8.8 patch 33536|linux|2.6.22-1-k7|200803161246
1174405|1174405-b19f-3f77-b713-d32bba55d77f|pass|200803|g.grigelionis@computer.org|Devel-CheckOS|1.43|i586-linux-thread-multi|5.8.5|linux|2.6.8.1|200803151032
1174545|1174545-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-thread-multi|5.8.8 patch 33536|linux|2.6.22-1-k7|200803161404
1174714|1174714-b19f-3f77-b713-d32bba55d77f|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPU-Emulator-Z80|0.9|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200803161337
1174721|1174721-b19f-3f77-b713-d32bba55d77f|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckLib|0.5|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200803161338
1174724|1174724-b19f-3f77-b713-d32bba55d77f|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckOS|1.43|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200803161339
1174757|1174757-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi|5.8.8 patch 33536|linux|2.6.22-1-k7|200803161451
1176590|1176590-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-thread-multi|5.8.8 patch 33536|linux|2.6.22-1-k7|200803161921
1176723|1176723-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.8.8 patch 33536|linux|2.6.22-1-k7|200803161932
1176968|1176968-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200803161952
1176971|1176971-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200803161954
1176974|1176974-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200803161956
1176979|1176979-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200803161958
1177423|1177423-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200803162007
1177430|1177430-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200803162008
1177436|1177436-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200803162011
1177445|1177445-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200803162014
1178306|1178306-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl ("DEVOGON")|CPU-Emulator-Z80|0.9|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803170730
1178312|1178312-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl ("DEVOGON")|Devel-CheckOS|1.43|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803170734
1178317|1178317-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl ("DEVOGON")|Devel-CheckLib|0.5|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803170735
1179297|1179297-b19f-3f77-b713-d32bba55d77f|pass|200803|jrockway@cpan.org|Data-Compare|1.18|x86_64-linux|5.10.0|linux|2.6.22-3-amd64|200803170556
1179476|1179476-b19f-3f77-b713-d32bba55d77f|pass|200803|stro@cpan.org|Devel-CheckOS|1.43|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803171456
1179553|1179553-b19f-3f77-b713-d32bba55d77f|pass|200803|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.581|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200803170934
1180042|1180042-b19f-3f77-b713-d32bba55d77f|pass|200803|stro@cpan.org|Devel-CheckOS|1.43|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200803171718
1180067|1180067-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|alpha-linux-thread-multi|5.11.0 patch 33539|linux|2.6.18-5-alpha-generic|200803171523
1180073|1180073-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|alpha-linux-thread-multi|5.11.0 patch 33539|linux|2.6.18-5-alpha-generic|200803171524
1180076|1180076-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|alpha-linux-thread-multi|5.11.0 patch 33539|linux|2.6.18-5-alpha-generic|200803171525
1180083|1180083-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|alpha-linux-thread-multi|5.11.0 patch 33539|linux|2.6.18-5-alpha-generic|200803171526
1181294|1181294-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-thread-multi-64int|5.10.0 patch 33535|linux|2.6.22-1-k7|200803172316
1181486|1181486-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi-64int|5.10.0 patch 33535|linux|2.6.22-1-k7|200803172255
1182212|1182212-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int|5.10.0 patch 33535|linux|2.6.22-1-k7|200803180154
1182266|1182266-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.10.0 patch 33535|linux|2.6.22-1-k7|200803180159
1182400|1182400-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-thread-multi-64int|5.10.0 patch 33535|linux|2.6.22-1-k7|200803180219
1182464|1182464-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-thread-multi-64int|5.10.0 patch 33535|linux|2.6.22-1-k7|200803180238
1183117|1183117-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200803180413
1183120|1183120-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200803180414
1183122|1183122-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200803180415
1183125|1183125-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200803180416
1183818|1183818-b19f-3f77-b713-d32bba55d77f|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPU-Emulator-Z80|0.9|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200803180856
1183820|1183820-b19f-3f77-b713-d32bba55d77f|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckLib|0.5|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200803180857
1183823|1183823-b19f-3f77-b713-d32bba55d77f|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckOS|1.43|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200803180858
1184296|1184296-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200803181113
1184598|1184598-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|alpha-linux-thread-multi|5.11.0 patch 33539|linux|2.6.18-5-alpha-generic|200803181322
1185299|1185299-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|CPU-Emulator-Memory|1.1001|amd64-freebsd|5.8.8|freebsd|6.3-stable|200803181634
1185301|1185301-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|CPU-Emulator-Z80|0.9|amd64-freebsd|5.8.8|freebsd|6.3-stable|200803181635
1185302|1185302-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Devel-CheckLib|0.5|amd64-freebsd|5.8.8|freebsd|6.3-stable|200803181634
1185643|1185643-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200803181710
1185646|1185646-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200803181711
1185669|1185669-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200803181713
1185676|1185676-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200803181714
1186256|1186256-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Net-Random|2.0|amd64-freebsd|5.6.2|freebsd|6.2-release|200803182000
1186749|1186749-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Number-Phone|1.581|amd64-freebsd-thread-multi|5.8.8|freebsd|6.2-release|200803182152
1187699|1187699-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200803190210
1189947|1189947-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200803191518
1189953|1189953-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200803191519
1189956|1189956-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200803191520
1190402|1190402-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200803191634
1190403|1190403-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200803191635
1190405|1190405-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200803191635
1190469|1190469-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux|5.11.0 patch 33542|linux|2.6.22-1-k7|200803191916
1190666|1190666-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux|5.11.0 patch 33542|linux|2.6.22-1-k7|200803191844
1191004|1191004-b19f-3f77-b713-d32bba55d77f|fail|200803|srezic@cpan.org|Bryar|3.0|amd64-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200803192109
1191974|1191974-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.11.0 patch 33542|linux|2.6.22-1-k7|200803200158
1192362|1192362-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.11.0 patch 33542|linux|2.6.22-1-k7|200803200342
1192598|1192598-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200803200110
1192731|1192731-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux|5.11.0 patch 33542|linux|2.6.22-1-k7|200803200339
1193708|1193708-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.11.0 patch 33542|linux|2.6.22-1-k7|200803200740
1193709|1193709-b19f-3f77-b713-d32bba55d77f|pass|200803|imacat@mail.imacat.idv.tw|Acme-Licence|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803201440
1193710|1193710-b19f-3f77-b713-d32bba55d77f|pass|200803|imacat@mail.imacat.idv.tw|Acme-Pony|1.1.2|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803201440
1193712|1193712-b19f-3f77-b713-d32bba55d77f|pass|200803|imacat@mail.imacat.idv.tw|Acme-Scurvy-Whoreson-BilgeRat|1.1|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803201440
1193714|1193714-b19f-3f77-b713-d32bba55d77f|pass|200803|imacat@mail.imacat.idv.tw|NestedMap|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803201440
1193716|1193716-b19f-3f77-b713-d32bba55d77f|pass|200803|imacat@mail.imacat.idv.tw|Statistics-ChiSquare|0.5|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803201441
1193717|1193717-b19f-3f77-b713-d32bba55d77f|pass|200803|imacat@mail.imacat.idv.tw|Statistics-SerialCorrelation|1.1|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803201441
1193719|1193719-b19f-3f77-b713-d32bba55d77f|pass|200803|imacat@mail.imacat.idv.tw|Tie-Hash-Longest|1.1|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803201441
1193720|1193720-b19f-3f77-b713-d32bba55d77f|pass|200803|imacat@mail.imacat.idv.tw|Tie-Hash-Rank|1.0.1|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803201441
1193725|1193725-b19f-3f77-b713-d32bba55d77f|pass|200803|imacat@mail.imacat.idv.tw|Tie-Scalar-Decay|1.1.1|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803201442
1197084|1197084-b19f-3f77-b713-d32bba55d77f|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Compare|1.18|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200803202243
1197372|1197372-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|i386-freebsd-64int|5.11.0 patch 33543|freebsd|7.0-release|200803201903
1198104|1198104-b19f-3f77-b713-d32bba55d77f|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPU-Emulator-Memory|1.1001|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200803210453
1198112|1198112-b19f-3f77-b713-d32bba55d77f|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPU-Emulator-Z80|0.9|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200803210455
1198152|1198152-b19f-3f77-b713-d32bba55d77f|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckOS|1.43|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200803210504
1198166|1198166-b19f-3f77-b713-d32bba55d77f|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckLib|0.5|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200803210507
1198952|1198952-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|amd64-freebsd-thread-multi|5.8.8|freebsd|6.2-release|200803210013
1198983|1198983-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl ("DEVOGON")|Data-Transactional|1.0|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803210932
1198998|1198998-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl ("DEVOGON")|Tie-STDOUT|1.03|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803210934
1200386|1200386-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|cygwin-thread-multi-64int|5.11.0 patch 33543|cygwin|1.5.24(0.15642)|200803211349
1201926|1201926-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux|5.11.0 patch 33547|linux|2.6.22-1-k7|200803212318
1202128|1202128-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.11.0 patch 33547|linux|2.6.22-1-k7|200803212309
1203286|1203286-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|i86pc-solaris-thread-multi-64int|5.11.0 patch 33544|solaris|2.11|200803221120
1204426|1204426-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200803222136
1204805|1204805-b19f-3f77-b713-d32bba55d77f|pass|200803|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200803230315
1205235|1205235-b19f-3f77-b713-d32bba55d77f|fail|200803|imacat@mail.imacat.idv.tw|Palm-SMS|0.02|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803231512
1205373|1205373-b19f-3f77-b713-d32bba55d77f|pass|200803|imacat@mail.imacat.idv.tw|Bryar|3.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803231644
1207445|1207445-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|i386-pc-solaris2.11-thread-multi-64int|5.10.0|solaris|2.11|200803240109
1208722|1208722-b19f-3f77-b713-d32bba55d77f|pass|200803|pjacklam@online.no|Statistics-ChiSquare|0.5|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200803240946
1209414|1209414-b19f-3f77-b713-d32bba55d77f|pass|200803|pjacklam@online.no|Statistics-SerialCorrelation|1.1|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200803241042
1211329|1211329-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux|5.11.0 patch 33551|linux|2.6.22-1-k7|200803242029
1211522|1211522-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.11.0 patch 33551|linux|2.6.22-1-k7|200803242028
1211793|1211793-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Number-Phone-UK-DetailedLocations|1.3|amd64-freebsd|5.6.2|freebsd|6.2-release|200803242136
1211844|1211844-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.11.0 patch 33551|linux|2.6.22-1-k7|200803241943
1211972|1211972-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.11.0 patch 33551|linux|2.6.22-1-k7|200803242157
1213162|1213162-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux|5.11.0 patch 33551|linux|2.6.22-1-k7|200803250221
1213482|1213482-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux|5.11.0 patch 33551|linux|2.6.22-1-k7|200803250005
1214140|1214140-b19f-3f77-b713-d32bba55d77f|pass|200803|victorg@hacktux.com|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.18-53.1.13.el5|200803250023
1219608|1219608-b19f-3f77-b713-d32bba55d77f|pass|200803|mthurn@cpan.org|Devel-CheckOS|1.43|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803260359
1222775|1222775-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Devel-CheckOS|1.43|i386-freebsd|5.6.2|freebsd|6.1-release|200803270221
1223051|1223051-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33580|linux|2.6.22-1-k7|200803270427
1223335|1223335-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33580|linux|2.6.22-1-k7|200803270607
1223339|1223339-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33580|linux|2.6.22-1-k7|200803270608
1223390|1223390-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.11.0 patch 33580|linux|2.6.22-1-k7|200803270618
1223398|1223398-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.11.0 patch 33580|linux|2.6.22-1-k7|200803270615
1224707|1224707-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33580|linux|2.6.22-1-k7|200803271100
1224842|1224842-b19f-3f77-b713-d32bba55d77f|pass|200803|imacat@mail.imacat.idv.tw|File-Find-Rule-Permissions|1.3|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200803271844
1225443|1225443-b19f-3f77-b713-d32bba55d77f|pass|200803|andy@hexten.net|Devel-CheckLib|0.5|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200803271533
1227809|1227809-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.8.8 patch 33576|linux|2.6.22-1-k7|200803281302
1227848|1227848-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 33576|linux|2.6.22-1-k7|200803281322
1228566|1228566-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 33576|linux|2.6.22-1-k7|200803281513
1228671|1228671-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 33576|linux|2.6.22-1-k7|200803281726
1228876|1228876-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200803281722
1228967|1228967-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200803281737
1229101|1229101-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.8.8 patch 33576|linux|2.6.22-1-k7|200803281854
1229520|1229520-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 33576|linux|2.6.22-1-k7|200803282040
1230271|1230271-b19f-3f77-b713-d32bba55d77f|pass|200803|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.18|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200803290030
1231970|1231970-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Devel-CheckOS|1.43|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200803291342
1232260|1232260-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Devel-CheckOS|1.43|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200803291414
1232395|1232395-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.11.0 patch 33584|netbsd|4.0|200803291444
1232715|1232715-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Data-Compare|1.18|i386-freebsd|5.10.0|freebsd|6.1-release|200803291728
1232969|1232969-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Data-Compare|1.18|i386-freebsd|5.6.2|freebsd|6.1-release|200803291533
1233051|1233051-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.11.0 patch 33584|netbsd|4.0|200803291737
1233578|1233578-b19f-3f77-b713-d32bba55d77f|pass|200803|icerider70@gmail.com ("ICERIDER")|Devel-CheckLib|0.4|i386-linux|5.8.8|linux|2.6.18-53.el5|200803291440
1233672|1233672-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Data-Compare|1.18|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200803292004
1234280|1234280-b19f-3f77-b713-d32bba55d77f|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.581|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200803292355
1234283|1234283-b19f-3f77-b713-d32bba55d77f|fail|200803|david@cantrell.org.uk|Devel-CheckOS|1.43|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200803292256
1234304|1234304-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|cygwin-thread-multi-64int|5.11.0 patch 33592|cygwin|1.5.24(0.15642)|200803292311
1235061|1235061-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 33243|linux|2.6.22-1-k7|200803301032
1235065|1235065-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200803301033
1235068|1235068-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8|linux|2.6.14|200803301034
1235070|1235070-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200803301035
1235073|1235073-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200803301036
1235078|1235078-b19f-3f77-b713-d32bba55d77f|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200803301037
1236117|1236117-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|XML-Tiny|1.11|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200803301809
1236572|1236572-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200803301949
1236770|1236770-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl (DEVOGON)|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.22-3-486|200803302143
1236800|1236800-b19f-3f77-b713-d32bba55d77f|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-DoubleEncodedEntities|1.0|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200803302152
1237579|1237579-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Devel-CheckOS|1.43|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200803302354
1237843|1237843-b19f-3f77-b713-d32bba55d77f|pass|200803|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200803310046
1238161|1238161-b19f-3f77-b713-d32bba55d77f|pass|200803|icerider70@gmail.com ("ICERIDER")|Devel-CheckLib|0.5|i386-linux|5.8.8|linux|2.6.18-53.el5|200803310322
1238163|1238163-b19f-3f77-b713-d32bba55d77f|pass|200803|icerider70@gmail.com ("ICERIDER")|Devel-CheckOS|1.43|i386-linux|5.8.8|linux|2.6.18-53.el5|200803310324
1238202|1238202-b19f-3f77-b713-d32bba55d77f|pass|200803|icerider70@gmail.com ("ICERIDER")|CPU-Emulator-Z80|0.9|i386-linux|5.8.8|linux|2.6.18-53.el5|200803310336
1239152|1239152-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl|XML-Tiny|1.11|i686-linux|5.10.0|linux|2.6.22-3-486|200803311156
1239517|1239517-b19f-3f77-b713-d32bba55d77f|pass|200803|flemming@ipk-gatersleben.de|Data-Compare|1.18|x86_64-linux|5.8.5|linux|2.6.9-55.0.9.elsmp|200803311253
1240050|1240050-b19f-3f77-b713-d32bba55d77f|pass|200803|flemming@ipk-gatersleben.de|File-Find-Rule-Permissions|1.3|x86_64-linux|5.8.5|linux|2.6.9-55.0.9.elsmp|200803311452
1240150|1240150-b19f-3f77-b713-d32bba55d77f|pass|200803|flemming@ipk-gatersleben.de|Devel-CheckOS|1.43|x86_64-linux|5.8.5|linux|2.6.9-55.0.9.elsmp|200803311452
1240236|1240236-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200803311522
1240361|1240361-b19f-3f77-b713-d32bba55d77f|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-Tiny|1.11|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200803311646
1240809|1240809-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|alpha-linux-thread-multi|5.11.0 patch 33609|linux|2.6.18-5-alpha-generic|200803311748
1241129|1241129-b19f-3f77-b713-d32bba55d77f|pass|200803|d@0x64.nl (DEVOGON)|XML-Tiny|1.11|i686-linux-thread-multi|5.11.0 patch 33261|linux|2.6.22-3-486|200803311839
1241188|1241188-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|XML-Tiny|1.11|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200803311943
1241491|1241491-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|CPU-Emulator-Memory|1.1001|i386-freebsd|5.6.2|freebsd|6.1-release|200803312138
1241502|1241502-b19f-3f77-b713-d32bba55d77f|pass|200803|srezic@cpan.org|Devel-CheckLib|0.5|i386-freebsd|5.6.2|freebsd|6.1-release|200803312143
1241637|1241637-b19f-3f77-b713-d32bba55d77f|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.11.0 patch 33609|netbsd|3.1|200803312125
1242439|1242439-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.11.0 patch 33609|netbsd|3.1|200804010218
1242549|1242549-b19f-3f77-b713-d32bba55d77f|pass|200804|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200804010215
1242883|1242883-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 33620|linux|2.6.22-1-k7|200804010613
1243016|1243016-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 33620|linux|2.6.22-1-k7|200804010702
1243595|1243595-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.10.0 patch 33620|linux|2.6.22-1-k7|200804010745
1244664|1244664-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.10.0 patch 33620|linux|2.6.22-1-k7|200804011204
1245283|1245283-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 33620|linux|2.6.22-1-k7|200804011238
1245469|1245469-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.10.0 patch 33620|linux|2.6.22-1-k7|200804010443
1245671|1245671-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int|5.11.0 patch 33623|linux|2.6.16.38|200804011411
1246102|1246102-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|alpha-linux-thread-multi|5.11.0 patch 33609|linux|2.6.18-5-alpha-generic|200804011728
1246605|1246605-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.11.0 patch 33623|linux|2.6.16.38|200804011834
1247239|1247239-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Data-Compare|1.18|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200804020018
1249436|1249436-b19f-3f77-b713-d32bba55d77f|fail|200804|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200804020616
1250046|1250046-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200804021400
1250303|1250303-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200804021720
1250327|1250327-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200804021640
1250845|1250845-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200804021958
1250931|1250931-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Devel-CheckLib|0.5|i386-freebsd|5.5.5|freebsd|6.1-release|200804022228
1251046|1251046-b19f-3f77-b713-d32bba55d77f|pass|200804|g.grigelionis@computer.org|Devel-CheckOS|1.43|x86_64-linux|5.8.8|linux|2.6.5-7.276-smp|200804022200
1251060|1251060-b19f-3f77-b713-d32bba55d77f|na|200804|srezic@cpan.org|CPU-Emulator-Memory|1.1001|i386-freebsd|5.5.5|freebsd|6.1-release|200804022228
1251663|1251663-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200804030054
1252351|1252351-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi-64int|5.10.0 patch 33628|linux|2.6.22-1-k7|200804030743
1252495|1252495-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|XML-Tiny|1.11|i386-freebsd|5.6.2|freebsd|6.1-release|200804030804
1252565|1252565-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|XML-Tiny|1.11|i386-freebsd|5.10.0|freebsd|6.1-release|200804030807
1252690|1252690-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|XML-Tiny|1.11|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200804030846
1252944|1252944-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-thread-multi-64int|5.10.0 patch 33628|linux|2.6.22-1-k7|200804031013
1253193|1253193-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-thread-multi-64int|5.10.0 patch 33628|linux|2.6.22-1-k7|200804031110
1253816|1253816-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33632|openbsd|4.1|200804031219
1254074|1254074-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-thread-multi-64int|5.10.0 patch 33628|linux|2.6.22-1-k7|200804031419
1254640|1254640-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.10.0 patch 33628|linux|2.6.22-1-k7|200804031557
1254832|1254832-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int|5.10.0 patch 33628|linux|2.6.22-1-k7|200804031639
1256353|1256353-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33632|openbsd|4.1|200804032241
1256641|1256641-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-thread-multi-64int|5.8.8 patch 33638|linux|2.6.22-1-k7|200804040035
1256699|1256699-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-thread-multi-64int|5.8.8 patch 33638|linux|2.6.22-1-k7|200804040046
1258103|1258103-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi-64int|5.8.8 patch 33638|linux|2.6.22-1-k7|200804040823
1258262|1258262-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|XML-Tiny|1.11|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200804040844
1258588|1258588-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|XML-Tiny|1.11|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200804040903
1258998|1258998-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int|5.8.8 patch 33638|linux|2.6.22-1-k7|200804041115
1259325|1259325-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-thread-multi-64int|5.8.8 patch 33638|linux|2.6.22-1-k7|200804041211
1259700|1259700-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.8.8 patch 33638|linux|2.6.22-1-k7|200804041104
1260650|1260650-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|cygwin-thread-multi-64int|5.11.0 patch 33643|cygwin|1.5.24(0.15642)|200804041859
1261597|1261597-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Devel-CheckOS|1.43|i386-freebsd|5.10.0|freebsd|6.1-release|200804050039
1261932|1261932-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux|5.11.0 patch 33646|linux|2.6.22-1-k7|200804050156
1262025|1262025-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.11.0 patch 33646|linux|2.6.22-1-k7|200804050243
1262253|1262253-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux|5.11.0 patch 33646|linux|2.6.22-1-k7|200804050337
1262830|1262830-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.11.0 patch 33646|linux|2.6.22-1-k7|200804050707
1262965|1262965-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.11.0 patch 33646|linux|2.6.22-1-k7|200804050734
1263296|1263296-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux|5.11.0 patch 33646|linux|2.6.22-1-k7|200804050833
1264058|1264058-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Devel-CheckLib|0.5|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200804051046
1264059|1264059-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|CPU-Emulator-Memory|1.1001|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200804051047
1264192|1264192-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|CPU-Emulator-Z80|0.9|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200804051048
1264387|1264387-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Number-Phone|1.581|i386-freebsd|5.6.2|freebsd|6.1-release|200804051254
1264466|1264466-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Devel-CheckLib|0.5|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200804051243
1264548|1264548-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|CPU-Emulator-Memory|1.1001|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200804051244
1264550|1264550-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|CPU-Emulator-Z80|0.9|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200804051245
1264740|1264740-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200804051349
1265774|1265774-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Devel-CheckLib|0.5|i386-freebsd|5.10.0|freebsd|6.1-release|200804051816
1265782|1265782-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|CPU-Emulator-Z80|0.9|i386-freebsd|5.10.0|freebsd|6.1-release|200804051819
1265898|1265898-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|CPU-Emulator-Memory|1.1001|i386-freebsd|5.10.0|freebsd|6.1-release|200804051818
1266599|1266599-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux|5.11.0 patch 33648|linux|2.6.22-1-k7|200804052116
1267915|1267915-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux|5.11.0 patch 33648|linux|2.6.22-1-k7|200804060102
1267993|1267993-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux|5.11.0 patch 33648|linux|2.6.22-1-k7|200804060048
1268179|1268179-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.11.0 patch 33648|linux|2.6.22-1-k7|200804060113
1268941|1268941-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.11.0 patch 33648|linux|2.6.22-1-k7|200804060231
1269251|1269251-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.11.0 patch 33648|linux|2.6.22-1-k7|200804060207
1270069|1270069-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|CPU-Emulator-Memory|1.1001|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200804061049
1270100|1270100-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|CPU-Emulator-Z80|0.9|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200804061054
1270807|1270807-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Number-Phone|1.581|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200804061538
1271214|1271214-b19f-3f77-b713-d32bba55d77f|pass|200804|gabor@pti.co.il ("Gabor Szabo")|Data-Compare|1.18|i486-linux-gnu|5.8.8|linux|2.6.15.7|200804062023
1272262|1272262-b19f-3f77-b713-d32bba55d77f|pass|200804|daxim@cpan.org|XML-Tiny|1.11|x86_64-linux|5.8.8|linux|2.6.20-gentoo-r8|200804062307
1272703|1272703-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200804070033
1273196|1273196-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.10.0 patch 33642|linux|2.6.22-1-k7|200804070512
1273374|1273374-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.10.0 patch 33642|linux|2.6.22-1-k7|200804070526
1273441|1273441-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.10.0 patch 33642|linux|2.6.22-1-k7|200804070552
1273961|1273961-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 33642|linux|2.6.22-1-k7|200804070811
1274437|1274437-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Number-Phone|1.581|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200804070943
1274783|1274783-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 33642|linux|2.6.22-1-k7|200804071031
1275261|1275261-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|i686-linux-thread-multi-64int|5.11.0 patch 33655|linux|2.6.21.5-smp|200804071002
1275273|1275273-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 33642|linux|2.6.22-1-k7|200804071154
1275419|1275419-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|alpha-linux-thread-multi|5.11.0 patch 33655|linux|2.6.18-5-alpha-generic|200804071127
1275742|1275742-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.11.0 patch 33655|netbsd|3.1|200804071205
1275956|1275956-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|darwin-thread-multi-64int-2level|5.11.0 patch 33655|darwin|8.10.0|200804071235
1276355|1276355-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200804071621
1276975|1276975-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Bryar|3.0|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200804071919
1280289|1280289-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Bryar|3.0|i386-freebsd|5.6.2|freebsd|6.1-release|200804080705
1282762|1282762-b19f-3f77-b713-d32bba55d77f|pass|200804|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPAN-FindDependencies|2.0|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200804081911
1285009|1285009-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33659|linux|2.6.22-1-k7|200804090442
1285671|1285671-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33659|linux|2.6.22-1-k7|200804090643
1285857|1285857-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.11.0 patch 33659|linux|2.6.22-1-k7|200804090734
1286432|1286432-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33659|linux|2.6.22-1-k7|200804090756
1286479|1286479-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33659|linux|2.6.22-1-k7|200804090922
1286504|1286504-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.11.0 patch 33659|linux|2.6.22-1-k7|200804090219
1286579|1286579-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Devel-CheckLib|0.5|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200804090930
1287189|1287189-b19f-3f77-b713-d32bba55d77f|pass|200804|rafl@debian.org|Data-Compare|1.18|i686-linux-multi|5.10.0|linux|2.6.24-rc5|200804091259
1287547|1287547-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Number-Phone|1.581|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200804091419
1288483|1288483-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200804091852
1289736|1289736-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Number-Phone|1.581|i386-freebsd|5.10.0|freebsd|6.1-release|200804100214
1291176|1291176-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd|5.10.0|freebsd|6.1-release|200804101229
1292857|1292857-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|darwin-thread-multi-64int-2level|5.11.0 patch 33659|darwin|8.10.0|200804102014
1292969|1292969-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Data-Compare|1.18|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200804102153
1294270|1294270-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|File-Find-Rule-Permissions|1.3|i386-freebsd|5.6.2|freebsd|6.1-release|200804110417
1295251|1295251-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|i86pc-solaris-thread-multi-64int|5.11.0 patch 33659|solaris|2.11|200804111006
1296395|1296395-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|CPAN-FindDependencies|2.0|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200804111548
1296444|1296444-b19f-3f77-b713-d32bba55d77f|fail|200804|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Bryar|3.0|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200804111559
1298893|1298893-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|cygwin-thread-multi-64int|5.11.0 patch 33669|cygwin|1.5.24(0.15642)|200804112348
1299784|1299784-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-thread-multi-64int|5.11.0 patch 33669|linux|2.6.22-1-k7|200804120744
1299806|1299806-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi-64int|5.11.0 patch 33669|linux|2.6.22-1-k7|200804120736
1300813|1300813-b19f-3f77-b713-d32bba55d77f|fail|200804|srezic@cpan.org|Bryar|3.0|i386-freebsd|5.10.0|freebsd|6.1-release|200804121203
1302083|1302083-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-thread-multi-64int|5.11.0 patch 33669|linux|2.6.22-1-k7|200804121611
1302530|1302530-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int|5.11.0 patch 33669|linux|2.6.22-1-k7|200804121655
1302619|1302619-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-thread-multi-64int|5.11.0 patch 33669|linux|2.6.22-1-k7|200804121648
1302734|1302734-b19f-3f77-b713-d32bba55d77f|pass|200804|andy@hexten.net|Devel-CheckLib|0.5|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200804121616
1302878|1302878-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.11.0 patch 33669|linux|2.6.22-1-k7|200804121725
1303747|1303747-b19f-3f77-b713-d32bba55d77f|pass|200804|bobtfish@bobtfish.net|Data-Compare|1.18|darwin-thread-multi-2level|5.8.6|darwin|8.0|200804121656
1304958|1304958-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33670|openbsd|4.2|200804122311
1306578|1306578-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Net-Random|2.0|i386-freebsd|5.6.2|freebsd|6.1-release|200804131000
1307348|1307348-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200804131229
1308479|1308479-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Net-Random|2.0|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200804131759
1308506|1308506-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Net-Random|2.0|i386-freebsd|5.10.0|freebsd|6.1-release|200804131809
1309852|1309852-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|i386-freebsd|5.6.2|freebsd|6.1-release|200804132352
1309901|1309901-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200804140004
1309990|1309990-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|i386-freebsd|5.10.0|freebsd|6.1-release|200804140024
1310065|1310065-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200804140032
1310264|1310264-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200804140112
1312796|1312796-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|alpha-linux-thread-multi|5.11.0 patch 33673|linux|2.6.18-5-alpha-generic|200804141247
1314443|1314443-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.11.0 patch 33673|netbsd|3.1|200804142104
1314651|1314651-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Games-Dice-Advanced|1.1|i386-freebsd|5.10.0|freebsd|6.1-release|200804142240
1316284|1316284-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|File-Find-Rule-Permissions|1.3|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200804150951
1318154|1318154-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|darwin-thread-multi-64int-2level|5.11.0 patch 33684|darwin|8.10.0|200804151533
1318181|1318181-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Data-Compare|1.18|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804151633
1319178|1319178-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Devel-CheckOS|1.43|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804152010
1319424|1319424-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Tie-STDOUT|1.03|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200804152133
1319427|1319427-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Data-Transactional|1.0|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200804152135
1320037|1320037-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804160006
1320202|1320202-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33695|linux|2.6.22-1-k7|200804160045
1320688|1320688-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|CPU-Emulator-Z80|0.9|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804160154
1320854|1320854-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|CPU-Emulator-Memory|1.1001|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804160153
1321240|1321240-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|File-Find-Rule-Permissions|1.3|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200804160306
1321718|1321718-b19f-3f77-b713-d32bba55d77f|pass|200804|bobtfish@bobtfish.net|Data-Compare|1.18|i486-linux-gnu|5.8.8|linux|2.6.22-3-amd64|200804160036
1321958|1321958-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.11.0 patch 33695|linux|2.6.22-1-k7|200804160426
1322246|1322246-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Devel-CheckLib|0.5|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804160545
1322346|1322346-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33695|linux|2.6.22-1-k7|200804160609
1322726|1322726-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|XML-Tiny|1.11|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804160707
1322791|1322791-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33695|linux|2.6.22-1-k7|200804160722
1323725|1323725-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Bryar|3.0|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804160935
1323925|1323925-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|CPAN-FindDependencies|2.0|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804160955
1324139|1324139-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33695|linux|2.6.22-1-k7|200804160937
1324238|1324238-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.11.0 patch 33695|linux|2.6.22-1-k7|200804161023
1324470|1324470-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200804161007
1324479|1324479-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200804161008
1324893|1324893-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|File-Find-Rule-Permissions|1.3|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804161208
1325309|1325309-b19f-3f77-b713-d32bba55d77f|pass|200804|daxim@cpan.org|Devel-CheckOS|1.43|x86_64-linux|5.8.8|linux|2.6.20-gentoo-r8|200804161332
1329851|1329851-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Number-Phone|1.581|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804170219
1330384|1330384-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200804170602
1330587|1330587-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Bryar|3.0|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200804170816
1330869|1330869-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Data-Hexdumper|1.3|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804170940
1330877|1330877-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Data-Transactional|1.0|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804170941
1331563|1331563-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804171159
1331870|1331870-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Tie-STDOUT|1.03|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804171256
1333040|1333040-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|XML-Tiny|1.11|i386-freebsd|5.5.5|freebsd|6.1-release|200804171656
1333594|1333594-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Number-Phone-UK-DetailedLocations|1.3|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804171929
1333630|1333630-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Net-Random|2.0|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804171941
1334678|1334678-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Sub-WrapPackages|1.2|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804172336
1334892|1334892-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Number-Phone|1.581|darwin-thread-multi-64int-2level|5.11.0 patch 33684|darwin|8.10.0|200804172324
1334893|1334893-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200804172324
1336996|1336996-b19f-3f77-b713-d32bba55d77f|pass|200804|stro@cpan.org|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.4.27-2-386|200804181300
1337168|1337168-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Games-Dice-Advanced|1.1|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804181149
1338088|1338088-b19f-3f77-b713-d32bba55d77f|pass|200804|stro@cpan.org|Devel-CheckOS|1.43|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200804181535
1338833|1338833-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|File-Overwrite|1.0|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804181627
1340146|1340146-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Palm-SMS|0.02|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804182058
1341403|1341403-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Class-CanBeA|1.2|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804190052
1342227|1342227-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804190341
1343054|1343054-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.11.0 patch 33713|netbsd|3.1|200804191000
1343247|1343247-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.11.0 patch 33713|netbsd|3.1|200804191113
1343738|1343738-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Devel-CheckOS|1.43|cygwin-thread-multi-64int|5.11.0 patch 33713|cygwin|1.5.24(0.15642)|200804191330
1344079|1344079-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|cygwin-thread-multi-64int|5.11.0 patch 33713|cygwin|1.5.24(0.15642)|200804191507
1344109|1344109-b19f-3f77-b713-d32bba55d77f|pass|200804|stro@cpan.org|XML-Tiny|1.11|i686-linux|5.8.8|linux|2.4.27-2-386|200804191416
1344425|1344425-b19f-3f77-b713-d32bba55d77f|pass|200804|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckOS|1.43|sun4-solaris|5.10.0|solaris|2.9|200804191717
1344978|1344978-b19f-3f77-b713-d32bba55d77f|pass|200804|stro@cpan.org|Devel-CheckOS|1.43|i686-linux|5.8.8|linux|2.4.27-2-386|200804191700
1345067|1345067-b19f-3f77-b713-d32bba55d77f|pass|200804|stro@cpan.org|Devel-CheckOS|1.43|i386-freebsd-64int|5.8.8|freebsd|7.0-release|200804192108
1346102|1346102-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Statistics-SerialCorrelation|1.1|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804192337
1346403|1346403-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|darwin-thread-multi-64int-2level|5.11.0 patch 33713|darwin|8.10.0|200804192340
1346430|1346430-b19f-3f77-b713-d32bba55d77f|unknown|200804|srezic@cpan.org|Acme-Licence|1.0|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804200042
1346569|1346569-b19f-3f77-b713-d32bba55d77f|pass|200804|stro@cpan.org|XML-Tiny|1.11|i686-linux|5.10.0|linux|2.4.27-2-386|200804200011
1346627|1346627-b19f-3f77-b713-d32bba55d77f|unknown|200804|srezic@cpan.org|Acme-Pony|1.1.2|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804200043
1347609|1347609-b19f-3f77-b713-d32bba55d77f|pass|200804|cpansmoker@renormalist.net ("Steffen Schwigon")|XML-Tiny|1.11|i686-linux|5.10.0|linux|2.6.18-5-k7|200804200050
1348140|1348140-b19f-3f77-b713-d32bba55d77f|pass|200804|cpansmoker@renormalist.net ("Steffen Schwigon")|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.18-5-k7|200804200841
1348289|1348289-b19f-3f77-b713-d32bba55d77f|pass|200804|cpansmoker@renormalist.net ("Steffen Schwigon")|Data-Compare|1.18|i686-linux|5.10.0|linux|2.6.18-5-k7|200804200911
1348942|1348942-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Statistics-ChiSquare|0.5|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804201142
1349045|1349045-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Tie-Hash-Longest|1.1|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804201117
1349578|1349578-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33714|linux|2.6.22-1-k7|200804201309
1349660|1349660-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33714|linux|2.6.22-1-k7|200804201322
1349824|1349824-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.11.0 patch 33714|linux|2.6.22-1-k7|200804201341
1349960|1349960-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33714|linux|2.6.22-1-k7|200804201306
1350142|1350142-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33714|linux|2.6.22-1-k7|200804201239
1350369|1350369-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|NestedMap|1.0|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804201434
1350385|1350385-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|darwin-2level|5.10.0|darwin|8.11.1|200804201202
1351314|1351314-b19f-3f77-b713-d32bba55d77f|pass|200804|artur+cpan@niif.spb.su|XML-Tiny|1.11|i386-linux|5.8.8|linux|2.6.18-53.1.4.el5.bz358661.1|200804201138
1352831|1352831-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|darwin-2level|5.10.0|darwin|8.11.1|200804201815
1352844|1352844-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int|5.11.0 patch 33714|linux|2.6.16.38|200804202018
1352857|1352857-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.11.0 patch 33714|linux|2.6.22-1-k7|200804201809
1353180|1353180-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.11.0 patch 33714|linux|2.6.16.38|200804202058
1354151|1354151-b19f-3f77-b713-d32bba55d77f|pass|200804|stro@cpan.org|XML-Tiny|1.11|armv5tejl-linux|5.8.8|linux|2.6.17.14|200804210414
1355985|1355985-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33714|openbsd|4.1|200804211355
1356080|1356080-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|alpha-linux-thread-multi|5.11.0 patch 33716|linux|2.6.18-5-alpha-generic|200804211407
1357339|1357339-b19f-3f77-b713-d32bba55d77f|pass|200804|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Compare|1.18|sun4-solaris|5.10.0|solaris|2.9|200804211952
1357838|1357838-b19f-3f77-b713-d32bba55d77f|pass|200804|stro@cpan.org|Data-Compare|1.18|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200804212200
1359285|1359285-b19f-3f77-b713-d32bba55d77f|unknown|200804|stro@cpan.org|Acme-Licence|1.0|i686-linux|5.8.8|linux|2.4.27-2-386|200804220029
1360100|1360100-b19f-3f77-b713-d32bba55d77f|unknown|200804|stro@cpan.org|Acme-Pony|1.1.2|i686-linux|5.8.8|linux|2.4.27-2-386|200804220442
1360220|1360220-b19f-3f77-b713-d32bba55d77f|pass|200804|stro@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux|5.8.8|linux|2.4.27-2-386|200804220549
1360275|1360275-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33714|openbsd|4.1|200804220557
1360430|1360430-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33714|openbsd|4.1|200804220712
1360762|1360762-b19f-3f77-b713-d32bba55d77f|pass|200804|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-Tiny|1.11|sun4-solaris|5.10.0|solaris|2.9|200804221035
1361828|1361828-b19f-3f77-b713-d32bba55d77f|unknown|200804|stro@cpan.org|Acme-Licence|1.0|armv5tejl-linux|5.8.8|linux|2.6.17.14|200804221657
1361841|1361841-b19f-3f77-b713-d32bba55d77f|unknown|200804|stro@cpan.org|Acme-Licence|1.0|i686-linux|5.10.0|linux|2.4.27-2-386|200804221541
1362745|1362745-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Tie-Scalar-Decay|1.1.1|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804221953
1362863|1362863-b19f-3f77-b713-d32bba55d77f|pass|200804|srezic@cpan.org|Tie-Hash-Rank|1.0.1|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804221951
1362883|1362883-b19f-3f77-b713-d32bba55d77f|unknown|200804|stro@cpan.org|Acme-Pony|1.1.2|i686-linux|5.10.0|linux|2.4.27-2-386|200804222000
1362913|1362913-b19f-3f77-b713-d32bba55d77f|unknown|200804|stro@cpan.org|Acme-Licence|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200804222148
1365292|1365292-b19f-3f77-b713-d32bba55d77f|pass|200804|mgardner@usatech.com|Data-Compare|1.18|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200804221340
1365401|1365401-b19f-3f77-b713-d32bba55d77f|pass|200804|stro@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux|5.10.0|linux|2.4.27-2-386|200804231029
1365419|1365419-b19f-3f77-b713-d32bba55d77f|unknown|200804|stro@cpan.org|Acme-Pony|1.1.2|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200804231207
1365571|1365571-b19f-3f77-b713-d32bba55d77f|pass|200804|stro@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200804231245
1365589|1365589-b19f-3f77-b713-d32bba55d77f|pass|200804|rafl@debian.org|XML-Tiny|1.11|i686-linux-multi|5.10.0|linux|2.6.24-rc5|200804230011
1365735|1365735-b19f-3f77-b713-d32bba55d77f|pass|200804|mgardner@usatech.com|Devel-CheckOS|1.43|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200804221350
1366323|1366323-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200804231630
1366439|1366439-b19f-3f77-b713-d32bba55d77f|unknown|200804|stro@cpan.org|Acme-Pony|1.1.2|armv5tejl-linux|5.8.8|linux|2.6.17.14|200804231843
1366955|1366955-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Devel-CheckOS|1.43|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200804231902
1367132|1367132-b19f-3f77-b713-d32bba55d77f|pass|200804|stro@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|armv5tejl-linux|5.8.8|linux|2.6.17.14|200804232203
1368275|1368275-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Acme-Licence|1.0|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240105
1368277|1368277-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Acme-Pony|1.1.2|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240106
1368280|1368280-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Acme-Scurvy-Whoreson-BilgeRat|1.1|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240107
1368324|1368324-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240122
1368325|1368325-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240123
1368334|1368334-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240126
1368335|1368335-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Class-CanBeA|1.2|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240126
1368340|1368340-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240127
1368341|1368341-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Data-Hexdumper|1.3|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240128
1368343|1368343-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Data-Transactional|1.0|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240129
1368347|1368347-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Devel-CheckLib|0.5|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240130
1368352|1368352-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240131
1368354|1368354-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|File-Find-Rule-Permissions|1.3|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240132
1368358|1368358-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|File-Overwrite|1.0|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240132
1368360|1368360-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Games-Dice-Advanced|1.1|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240133
1368362|1368362-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|NestedMap|1.0|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240133
1368371|1368371-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Net-Random|2.0|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240138
1368372|1368372-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Number-Phone|1.581|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240139
1368373|1368373-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Number-Phone-UK-DetailedLocations|1.3|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240141
1368375|1368375-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Statistics-ChiSquare|0.5|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240142
1368376|1368376-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Statistics-SerialCorrelation|1.1|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240142
1368377|1368377-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Sub-WrapPackages|1.2|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240143
1368378|1368378-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Tie-Hash-Longest|1.1|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240143
1368379|1368379-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Tie-Hash-Rank|1.0.1|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240143
1368380|1368380-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Tie-STDOUT|1.03|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240144
1368381|1368381-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Tie-Scalar-Decay|1.1.1|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240144
1368383|1368383-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240145
1368386|1368386-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240145
1371563|1371563-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.10.0 patch 33735|linux|2.6.22-1-k7|200804250734
1372822|1372822-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.10.0 patch 33735|linux|2.6.22-1-k7|200804251054
1373335|1373335-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux|5.10.0 patch 33735|linux|2.6.22-1-k7|200804250926
1373398|1373398-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|i386-freebsd-64int|5.11.0 patch 33741|freebsd|7.0-release|200804251510
1373795|1373795-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200804251645
1375050|1375050-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-64int|5.11.0 patch 33741|freebsd|7.0-release|200804260104
1375153|1375153-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-freebsd-64int|5.11.0 patch 33741|freebsd|7.0-release|200804260154
1375157|1375157-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux|5.10.0 patch 33735|linux|2.6.22-1-k7|200804260235
1375177|1375177-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.10.0 patch 33735|linux|2.6.22-1-k7|200804260242
1375466|1375466-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux|5.10.0 patch 33735|linux|2.6.22-1-k7|200804260413
1377678|1377678-b19f-3f77-b713-d32bba55d77f|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|darwin-thread-multi-64int-2level|5.11.0 patch 33741|darwin|8.10.0|200804262114
1380106|1380106-b19f-3f77-b713-d32bba55d77f|pass|200804|alexchorny@gmail.com|Data-Compare|1.18|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200804271820
1380947|1380947-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi|5.11.0 patch 33760|linux|2.6.22-1-k7|200804281120
1382101|1382101-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-thread-multi|5.11.0 patch 33760|linux|2.6.22-1-k7|200804281840
1382290|1382290-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-thread-multi|5.11.0 patch 33760|linux|2.6.22-1-k7|200804281946
1384214|1384214-b19f-3f77-b713-d32bba55d77f|pass|200804|daxim@cpan.org|CPU-Emulator-Z80|0.9|x86_64-linux|5.8.8|linux|2.6.20-gentoo-r8|200804290214
1384220|1384220-b19f-3f77-b713-d32bba55d77f|pass|200804|daxim@cpan.org|Devel-CheckLib|0.5|x86_64-linux|5.8.8|linux|2.6.20-gentoo-r8|200804290215
1384268|1384268-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.11.0 patch 33760|linux|2.6.22-1-k7|200804290043
1385201|1385201-b19f-3f77-b713-d32bba55d77f|pass|200804|daxim@cpan.org|Bryar|3.0|x86_64-linux|5.8.8|linux|2.6.20-gentoo-r8|200804290723
1385859|1385859-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-thread-multi|5.11.0 patch 33760|linux|2.6.22-1-k7|200804290843
1386203|1386203-b19f-3f77-b713-d32bba55d77f|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi|5.11.0 patch 33760|linux|2.6.22-1-k7|200804291034
1386295|1386295-b19f-3f77-b713-d32bba55d77f|pass|200804|tester@tcool.org ("ishigaki")|Data-Compare|1.18|i386-freebsd-64int|5.8.8|freebsd|5.4-release-p16|200804291212
1391895|1391895-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.3|darwin-thread-multi-64int-2level|5.11.0 patch 33741|darwin|8.10.0|200805011812
1392094|1392094-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux|5.10.0 patch 33755|linux|2.6.22-1-k7|200805011924
1392295|1392295-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.10.0 patch 33755|linux|2.6.22-1-k7|200805012217
1392648|1392648-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.10.0 patch 33755|linux|2.6.22-1-k7|200805012357
1392682|1392682-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.10.0 patch 33755|linux|2.6.22-1-k7|200805012352
1392765|1392765-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux|5.10.0 patch 33755|linux|2.6.22-1-k7|200805012331
1393191|1393191-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux|5.10.0 patch 33755|linux|2.6.22-1-k7|200805020601
1394719|1394719-b19f-3f77-b713-d32bba55d77f|na|200805|srezic@cpan.org|Tie-STDOUT|1.03|i386-freebsd|5.5.5|freebsd|6.1-release|200805021325
1396129|1396129-b19f-3f77-b713-d32bba55d77f|na|200805|srezic@cpan.org|Number-Phone|1.581|i386-freebsd|5.5.5|freebsd|6.1-release|200805021916
1396253|1396253-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.11.0 patch 33776|linux|2.6.22-1-k7|200805021945
1396966|1396966-b19f-3f77-b713-d32bba55d77f|na|200805|srezic@cpan.org|Bryar|3.0|i386-freebsd|5.5.5|freebsd|6.1-release|200805022257
1397502|1397502-b19f-3f77-b713-d32bba55d77f|pass|200805|gam3-cpanp@gam3.net|Devel-CheckOS|1.43|i486-linux-gnu|5.8.8|linux|2.6.22.10|200805020917
1397513|1397513-b19f-3f77-b713-d32bba55d77f|pass|200805|ewhipple@kubuntu-1.americanstaffing.org ((ewhipple))|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.24-16-generic|200805021038
1397923|1397923-b19f-3f77-b713-d32bba55d77f|na|200805|srezic@cpan.org|Data-Transactional|1.0|i386-freebsd|5.5.5|freebsd|6.1-release|200805030108
1398049|1398049-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33776|linux|2.6.22-1-k7|200805030108
1398157|1398157-b19f-3f77-b713-d32bba55d77f|pass|200805|srezic@cpan.org|Data-Hexdumper|1.3|i386-freebsd|5.5.5|freebsd|6.1-release|200805030059
1398349|1398349-b19f-3f77-b713-d32bba55d77f|pass|200805|jrockway@cpan.org|Devel-CheckOS|1.43|x86_64-linux|5.10.0|linux|2.6.22-3-amd64|200805021352
1398975|1398975-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.11.0 patch 33776|linux|2.6.22-1-k7|200805030441
1401073|1401073-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.8.8 patch 33736|linux|2.6.22-1-k7|200805031752
1401597|1401597-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33776|linux|2.6.22-1-k7|200805031236
1402427|1402427-b19f-3f77-b713-d32bba55d77f|pass|200805|d@0x64.nl ("DEVOGON")|Devel-CheckLib|0.5|x86_64-linux-gnu|5.10.0|linux|2.6.24-1-amd64|200805032247
1402547|1402547-b19f-3f77-b713-d32bba55d77f|fail|200805|chris@bingosnet.co.uk|Palm-SMS|0.02|darwin-thread-multi-64int-2level|5.11.0 patch 33782|darwin|8.10.0|200805032158
1402664|1402664-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 33736|linux|2.6.22-1-k7|200805032304
1403160|1403160-b19f-3f77-b713-d32bba55d77f|pass|200805|d@0x64.nl ("DEVOGON")|XML-Tiny|1.11|x86_64-linux-gnu|5.10.0|linux|2.6.24-1-amd64|200805040108
1403333|1403333-b19f-3f77-b713-d32bba55d77f|pass|200805|d@0x64.nl ("DEVOGON")|Devel-CheckOS|1.43|x86_64-linux-gnu|5.10.0|linux|2.6.24-1-amd64|200805040137
1403502|1403502-b19f-3f77-b713-d32bba55d77f|pass|200805|d@0x64.nl ("DEVOGON")|Data-Compare|1.18|x86_64-linux-gnu|5.10.0|linux|2.6.24-1-amd64|200805040200
1404074|1404074-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 33736|linux|2.6.22-1-k7|200805040138
1404127|1404127-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 33736|linux|2.6.22-1-k7|200805040133
1404862|1404862-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.18|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33782|openbsd|4.2|200805041041
1406078|1406078-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.8.8 patch 33736|linux|2.6.22-1-k7|200805041339
1406538|1406538-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 33736|linux|2.6.22-1-k7|200805041535
1407046|1407046-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33776|linux|2.6.22-1-k7|200805041750
1407061|1407061-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200805041858
1407291|1407291-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33776|linux|2.6.22-1-k7|200805041929
1408339|1408339-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33782|openbsd|4.2|200805050643
1409006|1409006-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.11.0 patch 33783|darwin|8.10.0|200805051201
1410617|1410617-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.11.0 patch 33788|netbsd|4.0|200805052144
1410773|1410773-b19f-3f77-b713-d32bba55d77f|pass|200805|frank.wiegand@gmail.com ("Frank Wiegand")|Devel-CheckOS|1.43|i686-linux|5.11.0 patch 33788|linux|2.6.24-17-generic|200805052323
1413440|1413440-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.11.0 patch 33788|netbsd|4.0|200805061317
1414429|1414429-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33788|linux|2.6.22-1-k7|200805061700
1414480|1414480-b19f-3f77-b713-d32bba55d77f|fail|200805|chris@bingosnet.co.uk|Bryar|3.0|darwin-thread-multi-64int-2level|5.11.0 patch 33783|darwin|8.10.0|200805061751
1416098|1416098-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.11.0 patch 33788|linux|2.6.22-1-k7|200805070052
1417178|1417178-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33788|linux|2.6.22-1-k7|200805070124
1417798|1417798-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33788|linux|2.6.22-1-k7|200805070924
1418235|1418235-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33788|linux|2.6.22-1-k7|200805071058
1418752|1418752-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.11.0 patch 33788|linux|2.6.22-1-k7|200805071326
1419366|1419366-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.18|darwin-thread-multi-64int-2level|5.11.0 patch 33783|darwin|8.10.0|200805071751
1420893|1420893-b19f-3f77-b713-d32bba55d77f|pass|200805|srezic@cpan.org|CPAN-FindDependencies|2.0|i386-freebsd|5.10.0|freebsd|6.1-release|200805080836
1422492|1422492-b19f-3f77-b713-d32bba55d77f|pass|200805|artur+cpan@niif.spb.su (ARTO)|Data-Compare|1.18|i386-linux|5.10.0|linux|2.6.18-53.1.6.el5xen|200805081517
1423103|1423103-b19f-3f77-b713-d32bba55d77f|pass|200805|d@0x64.nl|Data-Compare|1.18|i486-linux-gnu|5.8.8|linux|2.6.22.10|200805081919
1423254|1423254-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.18|i86pc-solaris-thread-multi-64int|5.11.0 patch 33789|solaris|2.10|200805082054
1424466|1424466-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.3|darwin-thread-multi-64int-2level|5.11.0 patch 33783|darwin|8.10.0|200805090547
1425874|1425874-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805090915
1426736|1426736-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|Data-Compare|1.18|darwin-2level|5.8.1|darwin|7.9.0|200805091542
1427856|1427856-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux|5.10.0 patch 33787|linux|2.6.22-1-k7|200805092020
1428749|1428749-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux|5.10.0 patch 33787|linux|2.6.22-1-k7|200805092230
1428861|1428861-b19f-3f77-b713-d32bba55d77f|pass|200805|bob@randomness.org.uk ("bob walker")|Devel-CheckOS|1.43|OpenBSD.i386-openbsd|5.8.8|openbsd|4.2|200805092240
1428880|1428880-b19f-3f77-b713-d32bba55d77f|pass|200805|bob@randomness.org.uk ("bob walker")|Devel-CheckOS|1.43|OpenBSD.i386-openbsd|5.10.0|openbsd|4.2|200805092244
1429209|1429209-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.18|darwin-thread-multi-64int-2level|5.11.0 patch 33797|darwin|8.10.0|200805092351
1430158|1430158-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.3|darwin-thread-multi-64int-2level|5.11.0 patch 33797|darwin|8.10.0|200805100357
1431052|1431052-b19f-3f77-b713-d32bba55d77f|pass|200805|frank.wiegand@gmail.com ("Frank Wiegand")|Devel-CheckOS|1.43|i686-linux|5.11.0 patch 33797|linux|2.6.24-17-generic|200805100805
1431653|1431653-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33798|linux|2.6.22-1-k7|200805101021
1432538|1432538-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33798|linux|2.6.22-1-k7|200805101402
1432694|1432694-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33798|linux|2.6.22-1-k7|200805101402
1433056|1433056-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|Devel-CheckOS|1.43|darwin-2level|5.8.1|darwin|7.9.0|200805101404
1433313|1433313-b19f-3f77-b713-d32bba55d77f|pass|200805|icerider70@gmail.com ("ICERIDER")|XML-Tiny|1.11|i386-linux|5.8.8|linux|2.6.18-53.el5|200805100903
1433348|1433348-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.18|cygwin-thread-multi-64int|5.11.0 patch 33796|cygwin|1.5.24(0.15642)|200805101508
1433636|1433636-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.11.0 patch 33798|linux|2.6.22-1-k7|200805101544
1434358|1434358-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805101412
1434661|1434661-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200805101847
1435099|1435099-b19f-3f77-b713-d32bba55d77f|pass|200805|frank.wiegand@gmail.com ("Frank Wiegand")|Data-Compare|1.18|i686-linux|5.11.0 patch 33797|linux|2.6.24-17-generic|200805102320
1436231|1436231-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|XML-Tiny|1.11|darwin-2level|5.8.1|darwin|7.9.0|200805110257
1436370|1436370-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33811|linux|2.6.22-1-k7|200805110336
1437768|1437768-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.11.0 patch 33811|linux|2.6.22-1-k7|200805110214
1438065|1438065-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33811|linux|2.6.22-1-k7|200805110149
1438346|1438346-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33811|linux|2.6.22-1-k7|200805111359
1438484|1438484-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33811|linux|2.6.22-1-k7|200805111438
1438874|1438874-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.18|i686-linux-thread-multi-64int|5.11.0 patch 33816|linux|2.6.21.5-smp|200805111621
1438928|1438928-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.11.0 patch 33811|linux|2.6.22-1-k7|200805111638
1439819|1439819-b19f-3f77-b713-d32bba55d77f|pass|200805|srezic@cpan.org|Data-Transactional|1.0|i386-freebsd|5.6.2|freebsd|6.1-release|200805112200
1439820|1439820-b19f-3f77-b713-d32bba55d77f|na|200805|srezic@cpan.org|Tie-STDOUT|1.03|i386-freebsd|5.6.2|freebsd|6.1-release|200805112201
1441505|1441505-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33798|linux|2.6.22-1-k7|200805120600
1441873|1441873-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.11.0 patch 33798|linux|2.6.22-1-k7|200805120751
1443828|1443828-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.10.0 patch 33787|linux|2.6.22-1-k7|200805121541
1444415|1444415-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.10.0 patch 33787|linux|2.6.22-1-k7|200805121738
1444798|1444798-b19f-3f77-b713-d32bba55d77f|pass|200805|srezic@cpan.org|Data-Compare|1.19|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200805122042
1445168|1445168-b19f-3f77-b713-d32bba55d77f|pass|200805|srezic@cpan.org|Data-Hexdumper|1.3|i386-freebsd|5.6.2|freebsd|6.1-release|200805122155
1445749|1445749-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.10.0 patch 33823|linux|2.6.22-1-k7|200805122232
1445774|1445774-b19f-3f77-b713-d32bba55d77f|pass|200805|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Data-Compare|1.19|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200805130718
1446070|1446070-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.10.0 patch 33823|linux|2.6.22-1-k7|200805130005
1446250|1446250-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200805130109
1446252|1446252-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.8.8 patch 33243|linux|2.6.22-1-k7|200805130110
1446254|1446254-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200805130111
1446260|1446260-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.8.8|linux|2.6.14|200805130112
1446262|1446262-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-thread-multi|5.8.8 patch 33430|linux|2.6.22-1-k7|200805130113
1446263|1446263-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200805130113
1446267|1446267-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200805130114
1446271|1446271-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-thread-multi-64int|5.10.0 patch 33628|linux|2.6.22-1-k7|200805130115
1446273|1446273-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200805130116
1446274|1446274-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200805130216
1446276|1446276-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200805130218
1446279|1446279-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200805130220
1446294|1446294-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.10.0 patch 33443|linux|2.6.22-1-k7|200805130127
1446649|1446649-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200805130443
1446652|1446652-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200805130443
1446655|1446655-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200805130444
1446656|1446656-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200805130444
1446659|1446659-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200805130445
1446665|1446665-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200805130445
1446669|1446669-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200805130446
1446671|1446671-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200805130447
1446674|1446674-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200805130447
1446676|1446676-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200805130448
1447210|1447210-b19f-3f77-b713-d32bba55d77f|pass|200805|d@0x64.nl ("DEVOGON")|Data-Compare|1.19|x86_64-linux-gnu|5.10.0|linux|2.6.18-6-xen-amd64|200805130904
1447307|1447307-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 33823|linux|2.6.22-1-k7|200805130810
1447348|1447348-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Data-Compare|1.19|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200805130716
1447365|1447365-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Data-Compare|1.19|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200805130219
1447405|1447405-b19f-3f77-b713-d32bba55d77f|pass|200805|rafl@debian.org|Data-Compare|1.19|i686-linux-multi|5.10.0|linux|2.6.24-rc5|200805130713
1447428|1447428-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Data-Compare|1.19|i686-linux|5.8.8|linux|2.4.27-3-686|200805130827
1447433|1447433-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Data-Compare|1.19|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200805130829
1447622|1447622-b19f-3f77-b713-d32bba55d77f|pass|200805|artur+cpan@niif.spb.su (ARTO)|Data-Compare|1.19|i386-linux|5.10.0|linux|2.6.18-53.1.6.el5xen|200805130901
1447705|1447705-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|Data-Compare|1.19|darwin-2level|5.8.1|darwin|7.9.0|200805130933
1447804|1447804-b19f-3f77-b713-d32bba55d77f|pass|200805|imacat@mail.imacat.idv.tw|Data-Compare|1.19|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200805131651
1447805|1447805-b19f-3f77-b713-d32bba55d77f|pass|200805|imacat@mail.imacat.idv.tw|Data-Compare|1.19|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200805131651
1447883|1447883-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Data-Compare|1.19|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200805131008
1447892|1447892-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Data-Compare|1.19|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200805130735
1448176|1448176-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805131111
1448508|1448508-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.18|i386-freebsd-64int|5.11.0 patch 33822|freebsd|7.0-release|200805131027
1448683|1448683-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200805131253
1448737|1448737-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Data-Compare|1.19|i686-linux|5.6.2|linux|2.4.27-3-686|200805131306
1448903|1448903-b19f-3f77-b713-d32bba55d77f|pass|200805|hohokus@gmail.com|Data-Compare|1.19|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805130857
1449105|1449105-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|alpha-linux-thread-multi|5.11.0 patch 33822|linux|2.6.18-5-alpha-generic|200805131449
1449658|1449658-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 33823|linux|2.6.22-1-k7|200805131025
1449817|1449817-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int-2level|5.11.0 patch 33822|darwin|8.10.0|200805131730
1449960|1449960-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|CPU-Emulator-Memory|1.1001|darwin-2level|5.8.1|darwin|7.9.0|200805131811
1449999|1449999-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|Devel-CheckLib|0.5|darwin-2level|5.8.1|darwin|7.9.0|200805131821
1450090|1450090-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200805131934
1450191|1450191-b19f-3f77-b713-d32bba55d77f|pass|200805|d@0x64.nl (DEVOGON)|Data-Compare|1.19|i686-linux-thread-multi|5.11.0 patch 33261|linux|2.6.22-3-486|200805131835
1450235|1450235-b19f-3f77-b713-d32bba55d77f|pass|200805|g.grigelionis@computer.org|Data-Compare|1.19|i586-linux-thread-multi|5.8.8|linux|2.6.16|200805131958
1450615|1450615-b19f-3f77-b713-d32bba55d77f|pass|200805|srezic@cpan.org|Data-Compare|1.19|i386-freebsd|5.10.0|freebsd|6.1-release|200805132205
1451204|1451204-b19f-3f77-b713-d32bba55d77f|pass|200805|srezic@cpan.org|Data-Compare|1.19|i386-freebsd|5.6.2|freebsd|6.1-release|200805140013
1451206|1451206-b19f-3f77-b713-d32bba55d77f|pass|200805|srezic@cpan.org|Data-Compare|1.19|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200805140015
1451390|1451390-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33822|openbsd|4.2|200805132353
1451490|1451490-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200805140016
1451562|1451562-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.10.0 patch 33823|linux|2.6.22-1-k7|200805140029
1451906|1451906-b19f-3f77-b713-d32bba55d77f|pass|200805|rlane10@wi.rr.com|Data-Compare|1.19|darwin-2level-ld|5.10.0|darwin|8.11.0|200805131335
1452642|1452642-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33824|linux|2.6.22-1-k7|200805140623
1453081|1453081-b19f-3f77-b713-d32bba55d77f|pass|200805|srezic@cpan.org|Data-Compare|1.19|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200805140836
1453184|1453184-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 33824|linux|2.6.22-1-k7|200805140900
1453262|1453262-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|Number-Phone|1.581|darwin-2level|5.8.1|darwin|7.9.0|200805140826
1453544|1453544-b19f-3f77-b713-d32bba55d77f|pass|200805|claco@chrislaco.com ("Christopher H. Laco")|Data-Compare|1.19|darwin-thread-multi-2level|5.8.6|darwin|8.0|200805140110
1453574|1453574-b19f-3f77-b713-d32bba55d77f|pass|200805|artur+cpan@niif.spb.su (ARTO)|Devel-CheckOS|1.43|i386-linux|5.10.0|linux|2.6.18-53.1.6.el5xen|200805141023
1454231|1454231-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.11.0 patch 33822|darwin|8.10.0|200805141259
1454795|1454795-b19f-3f77-b713-d32bba55d77f|pass|200805|stro@cpan.org|Data-Compare|1.19|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200805141727
1455003|1455003-b19f-3f77-b713-d32bba55d77f|pass|200805|stro@cpan.org|Data-Compare|1.19|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200805141817
1455200|1455200-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|XML-DoubleEncodedEntities|1.0|darwin-2level|5.8.1|darwin|7.9.0|200805141701
1455314|1455314-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33824|linux|2.6.22-1-k7|200805141105
1455526|1455526-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33824|linux|2.6.22-1-k7|200805141019
1456155|1456155-b19f-3f77-b713-d32bba55d77f|pass|200805|d@0x64.nl ("DEVOGON")|Data-Compare|1.19|i686-linux|5.10.0|linux|2.6.22-3-486|200805142209
1456214|1456214-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200805142209
1456357|1456357-b19f-3f77-b713-d32bba55d77f|pass|200805|srezic@cpan.org|Data-Compare|1.19|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200805142248
1456779|1456779-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200805142347
1456839|1456839-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805141907
1457182|1457182-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.11.0 patch 33824|linux|2.6.22-1-k7|200805150105
1458610|1458610-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33824|linux|2.6.22-1-k7|200805150835
1459230|1459230-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200805151214
1460527|1460527-b19f-3f77-b713-d32bba55d77f|pass|200805|stro@cpan.org|Devel-CheckOS|1.43|armv5tejl-linux|5.8.8|linux|2.6.17.14|200805151846
1460623|1460623-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805151656
1460959|1460959-b19f-3f77-b713-d32bba55d77f|pass|200805|dillman@ria.buffalo.edu|Data-Compare|1.19|sun4-solaris|5.10.0|solaris|2.9|200805150915
1461199|1461199-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 33823|linux|2.6.22-1-k7|200805151803
1461793|1461793-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805151626
1461874|1461874-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805151640
1462583|1462583-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 33819|linux|2.6.22-1-k7|200805160026
1462813|1462813-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.8.8 patch 33819|linux|2.6.22-1-k7|200805160218
1463177|1463177-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805160229
1463182|1463182-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|0.9|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805160229
1463203|1463203-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805160233
1463474|1463474-b19f-3f77-b713-d32bba55d77f|fail|200805|chris@bingosnet.co.uk|Bryar|3.0|darwin-thread-multi-64int-2level|5.11.0 patch 33824|darwin|8.10.0|200805160418
1463606|1463606-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int-2level|5.11.0 patch 33824|darwin|8.10.0|200805160448
1464426|1464426-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805160243
1464429|1464429-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200805160143
1465183|1465183-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 33819|linux|2.6.22-1-k7|200805161049
1465966|1465966-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 33832|linux|2.6.22-1-k7|200805161339
1466142|1466142-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33832|linux|2.6.22-1-k7|200805161335
1466168|1466168-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.11.0 patch 33831|netbsd|4.0|200805160737
1466808|1466808-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33832|linux|2.6.22-1-k7|200805161613
1466853|1466853-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33832|linux|2.6.22-1-k7|200805161715
1467732|1467732-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|CPAN-FindDependencies|2.0|IP35-irix|5.8.8|irix|6.5|200805162203
1467738|1467738-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|CPU-Emulator-Memory|1.1001|IP35-irix|5.8.8|irix|6.5|200805162204
1467769|1467769-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Class-CanBeA|1.2|IP35-irix|5.8.8|irix|6.5|200805162209
1467784|1467784-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Data-Compare|1.19|IP35-irix|5.8.8|irix|6.5|200805162211
1467790|1467790-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Data-Hexdumper|1.3|IP35-irix|5.8.8|irix|6.5|200805162212
1467792|1467792-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Data-Transactional|1.0|IP35-irix|5.8.8|irix|6.5|200805162213
1467796|1467796-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Devel-CheckLib|0.5|IP35-irix|5.8.8|irix|6.5|200805162213
1467804|1467804-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Devel-CheckOS|1.43|IP35-irix|5.8.8|irix|6.5|200805162214
1467811|1467811-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|File-Find-Rule-Permissions|1.3|IP35-irix|5.8.8|irix|6.5|200805162215
1467815|1467815-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|File-Overwrite|1.0|IP35-irix|5.8.8|irix|6.5|200805162215
1467819|1467819-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Games-Dice-Advanced|1.1|IP35-irix|5.8.8|irix|6.5|200805162216
1467823|1467823-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|NestedMap|1.0|IP35-irix|5.8.8|irix|6.5|200805162216
1467841|1467841-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Net-Random|2.0|IP35-irix|5.8.8|irix|6.5|200805162219
1467846|1467846-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Number-Phone|1.581|IP35-irix|5.8.8|irix|6.5|200805162220
1467856|1467856-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Number-Phone-UK-DetailedLocations|1.3|IP35-irix|5.8.8|irix|6.5|200805162221
1467869|1467869-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|XML-Tiny|1.11|darwin-thread-multi-64int-2level|5.11.0 patch 33824|darwin|8.10.0|200805162123
1467879|1467879-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1.1|IP35-irix|5.8.8|irix|6.5|200805162225
1467884|1467884-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Statistics-ChiSquare|0.5|IP35-irix|5.8.8|irix|6.5|200805162225
1467885|1467885-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Statistics-SerialCorrelation|1.1|IP35-irix|5.8.8|irix|6.5|200805162226
1467893|1467893-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Sub-WrapPackages|1.2|IP35-irix|5.8.8|irix|6.5|200805162227
1467896|1467896-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Tie-Hash-Longest|1.1|IP35-irix|5.8.8|irix|6.5|200805162227
1467900|1467900-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Tie-Hash-Rank|1.0.1|IP35-irix|5.8.8|irix|6.5|200805162228
1467903|1467903-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Tie-STDOUT|1.03|IP35-irix|5.8.8|irix|6.5|200805162228
1467906|1467906-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Tie-Scalar-Decay|1.1.1|IP35-irix|5.8.8|irix|6.5|200805162229
1467908|1467908-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|IP35-irix|5.8.8|irix|6.5|200805162229
1467909|1467909-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|XML-Tiny|1.11|IP35-irix|5.8.8|irix|6.5|200805162230
1468355|1468355-b19f-3f77-b713-d32bba55d77f|pass|200805|srezic@cpan.org|CPAN-FindDependencies|2.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200805170020
1468495|1468495-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|cygwin-thread-multi-64int|5.11.0 patch 33836|cygwin|1.5.25(0.15642)|200805162356
1468894|1468894-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.11.0 patch 33832|linux|2.6.22-1-k7|200805170117
1469173|1469173-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.581|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805170251
1469324|1469324-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33832|linux|2.6.22-1-k7|200805170336
1469658|1469658-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|Bryar|3.0|darwin-2level|5.8.1|darwin|7.9.0|200805170606
1470250|1470250-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805170326
1470540|1470540-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200805170334
1471030|1471030-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.11.0 patch 33831|netbsd|4.0|200805170512
1473276|1473276-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|cygwin-thread-multi-64int|5.11.0 patch 33836|cygwin|1.5.25(0.15642)|200805171845
1474419|1474419-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.8.8 patch 33819|linux|2.6.22-1-k7|200805172158
1474499|1474499-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 33819|linux|2.6.22-1-k7|200805172209
1474669|1474669-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805171833
1474670|1474670-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|0.9|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805171833
1474676|1474676-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805171834
1475128|1475128-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 33819|linux|2.6.22-1-k7|200805172348
1475216|1475216-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.581|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805180005
1475628|1475628-b19f-3f77-b713-d32bba55d77f|pass|200805|g.grigelionis@computer.org|Data-Compare|1.19|i586-linux-thread-multi|5.8.5|linux|2.6.8.1|200805172247
1475926|1475926-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805180241
1476902|1476902-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200805180007
1477233|1477233-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|File-Find-Rule-Permissions|1.3|darwin-2level|5.8.1|darwin|7.9.0|200805180904
1479601|1479601-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33845|linux|2.6.22-1-k7|200805181231
1480382|1480382-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200805182014
1480561|1480561-b19f-3f77-b713-d32bba55d77f|pass|200805|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Data-Hexdumper|1.4|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200805190443
1480879|1480879-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805181630
1480882|1480882-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33845|linux|2.6.22-1-k7|200805180852
1481334|1481334-b19f-3f77-b713-d32bba55d77f|fail|200805|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805181746
1481748|1481748-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33845|linux|2.6.22-1-k7|200805182258
1482086|1482086-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 33845|linux|2.6.22-1-k7|200805190039
1482184|1482184-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200805190202
1482186|1482186-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200805190202
1482187|1482187-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200805190202
1482189|1482189-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200805190202
1482190|1482190-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200805190203
1482191|1482191-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200805190203
1482193|1482193-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200805190203
1482194|1482194-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200805190204
1482196|1482196-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200805190204
1482197|1482197-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200805190204
1482284|1482284-b19f-3f77-b713-d32bba55d77f|pass|200805|stro@cpan.org|Data-Hexdumper|1.4|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200805190430
1482543|1482543-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200805190422
1482545|1482545-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200805190423
1482546|1482546-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200805190425
1482735|1482735-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805190027
1482909|1482909-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|Data-Transactional|1.0|darwin-2level|5.8.1|darwin|7.9.0|200805190618
1482957|1482957-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|Tie-STDOUT|1.03|darwin-2level|5.8.1|darwin|7.9.0|200805190640
1483249|1483249-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.11.0 patch 33858|linux|2.6.16.38|200805190900
1483386|1483386-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 33443|linux|2.6.22-1-k7|200805191049
1483387|1483387-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-thread-multi-64int|5.10.0 patch 33412|linux|2.6.22-1-k7|200805191050
1483454|1483454-b19f-3f77-b713-d32bba55d77f|pass|200805|imacat@mail.imacat.idv.tw|Data-Hexdumper|1.4|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200805191703
1483455|1483455-b19f-3f77-b713-d32bba55d77f|pass|200805|imacat@mail.imacat.idv.tw|Data-Hexdumper|1.4|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200805191703
1483524|1483524-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Data-Hexdumper|1.4|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200805191018
1483536|1483536-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|Data-Hexdumper|1.4|darwin-2level|5.8.1|darwin|7.9.0|200805191021
1483579|1483579-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Data-Hexdumper|1.4|i686-linux|5.8.8|linux|2.4.27-3-686|200805191026
1483593|1483593-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200805191028
1483622|1483622-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Data-Hexdumper|1.4|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200805190431
1483630|1483630-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Data-Hexdumper|1.4|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200805190718
1483690|1483690-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Data-Hexdumper|1.4|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200805190944
1483703|1483703-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Data-Hexdumper|1.4|IP35-irix|5.8.8|irix|6.5|200805191145
1484127|1484127-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Data-Hexdumper|1.4|i686-linux|5.6.2|linux|2.4.27-3-686|200805191221
1484190|1484190-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|alpha-linux-thread-multi|5.11.0 patch 33858|linux|2.6.18-5-alpha-generic|200805191236
1484331|1484331-b19f-3f77-b713-d32bba55d77f|pass|200805|stro@cpan.org|Data-Hexdumper|1.4|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200805191506
1484424|1484424-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.11.0 patch 33858|linux|2.6.16.38|200805191331
1485485|1485485-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805191154
1486143|1486143-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33858|linux|2.6.22-1-k7|200805191312
1486218|1486218-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805191411
1486232|1486232-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|alpha-linux-thread-multi|5.11.0 patch 33858|linux|2.6.18-5-alpha-generic|200805191914
1486326|1486326-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|Palm-TreoPhoneCallDB|1.1|darwin-2level|5.8.1|darwin|7.9.0|200805191942
1486432|1486432-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.0|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805191502
1486442|1486442-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805191503
1486855|1486855-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805191602
1487259|1487259-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.11.0 patch 33858|linux|2.6.16.38|200805192153
1487680|1487680-b19f-3f77-b713-d32bba55d77f|pass|200805|srezic@cpan.org|Data-Hexdumper|1.4|i386-freebsd|5.6.2|freebsd|6.1-release|200805200038
1488161|1488161-b19f-3f77-b713-d32bba55d77f|pass|200805|feldt@nhn.ou.edu|Data-Compare|1.19|sun4-solaris-thread-multi|5.8.8|solaris|2.8|200805191428
1489311|1489311-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33858|linux|2.6.22-1-k7|200805200546
1489810|1489810-b19f-3f77-b713-d32bba55d77f|pass|200805|srezic@cpan.org|Data-Hexdumper|1.4|i386-freebsd|5.10.0|freebsd|6.1-release|200805200901
1489931|1489931-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 33858|linux|2.6.22-1-k7|200805200241
1490380|1490380-b19f-3f77-b713-d32bba55d77f|pass|200805|srezic@cpan.org|Data-Hexdumper|1.4|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200805201058
1490505|1490505-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33858|linux|2.6.22-1-k7|200805200147
1490736|1490736-b19f-3f77-b713-d32bba55d77f|pass|200805|d@0x64.nl ("DEVOGON")|Data-Hexdumper|1.4|x86_64-linux-gnu|5.10.0|linux|2.6.18-6-xen-amd64|200805201311
1491016|1491016-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33871|openbsd|4.1|200805201248
1491047|1491047-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805200752
1491605|1491605-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805200852
1492814|1492814-b19f-3f77-b713-d32bba55d77f|pass|200805|srezic@cpan.org|Data-Hexdumper|1.4|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200805201725
1493168|1493168-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200805201705
1493397|1493397-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805201231
1493723|1493723-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33858|linux|2.6.22-1-k7|200805201734
1493871|1493871-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|alpha-linux-thread-multi|5.11.0 patch 33858|linux|2.6.18-5-alpha-generic|200805201917
1494038|1494038-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33871|openbsd|4.1|200805202004
1494083|1494083-b19f-3f77-b713-d32bba55d77f|pass|200805|srezic@cpan.org|Data-Hexdumper|1.4|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200805202115
1494478|1494478-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805201620
1494758|1494758-b19f-3f77-b713-d32bba55d77f|pass|200805|srezic@cpan.org|Data-Hexdumper|1.4|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200805202316
1494873|1494873-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33887|linux|2.6.22-1-k7|200805202306
1495109|1495109-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 33887|linux|2.6.22-1-k7|200805202251
1495311|1495311-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33887|linux|2.6.22-1-k7|200805210027
1495328|1495328-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Acme-Scurvy-Whoreson-BilgeRat|1.1|darwin-thread-multi-64int-2level|5.11.0 patch 33852|darwin|8.10.0|200805210132
1495828|1495828-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|Net-Random|2.0|darwin-2level|5.8.1|darwin|7.9.0|200805210434
1496297|1496297-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33871|openbsd|4.1|200805210724
1497266|1497266-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200805211319
1497286|1497286-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200805211325
1497634|1497634-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|Number-Phone-UK-DetailedLocations|1.3|darwin-2level|5.8.1|darwin|7.9.0|200805211454
1497883|1497883-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33887|linux|2.6.22-1-k7|200805210933
1497924|1497924-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33887|linux|2.6.22-1-k7|200805210936
1498842|1498842-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200805212052
1499156|1499156-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 33887|linux|2.6.22-1-k7|200805212132
1499608|1499608-b19f-3f77-b713-d32bba55d77f|pass|200805|d@0x64.nl (DEVOGON)|Data-Hexdumper|1.4|i686-linux-thread-multi|5.11.0 patch 33261|linux|2.6.22-3-486|200805220051
1500298|1500298-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805220037
1500367|1500367-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200805220152
1500524|1500524-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805220119
1501526|1501526-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200805220657
1501834|1501834-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200805220845
1502059|1502059-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200805220843
1502062|1502062-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 33243|linux|2.6.22-1-k7|200805220844
1502068|1502068-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200805220845
1502069|1502069-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8|linux|2.6.14|200805220845
1502073|1502073-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-thread-multi|5.8.8 patch 33430|linux|2.6.22-1-k7|200805220847
1502087|1502087-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200805220851
1502088|1502088-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200805220852
1504694|1504694-b19f-3f77-b713-d32bba55d77f|pass|200805|d@0x64.nl ("DEVOGON")|Devel-CheckOS|1.43|i486-linux-gnu|5.10.0|linux|2.6.24.4|200805222319
1504918|1504918-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.11.0 patch 33852|darwin|8.10.0|200805222259
1505260|1505260-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int-2level|5.11.0 patch 33852|darwin|8.10.0|200805230048
1505550|1505550-b19f-3f77-b713-d32bba55d77f|fail|200805|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805230140
1505637|1505637-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805230230
1506120|1506120-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200805230724
1506208|1506208-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200805230759
1506315|1506315-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|Games-Dice-Advanced|1.1|darwin-2level|5.8.1|darwin|7.9.0|200805230737
1506856|1506856-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 33911|linux|2.6.22-1-k7|200805231023
1506900|1506900-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200805230934
1507112|1507112-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33911|linux|2.6.22-1-k7|200805230941
1507301|1507301-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33911|linux|2.6.22-1-k7|200805231205
1507645|1507645-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33911|linux|2.6.22-1-k7|200805231316
1508364|1508364-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33911|linux|2.6.22-1-k7|200805231302
1508500|1508500-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200805231410
1509961|1509961-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 33911|linux|2.6.22-1-k7|200805231127
1510117|1510117-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200805231809
1510229|1510229-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200805231850
1510416|1510416-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805231459
1510860|1510860-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|Sub-WrapPackages|1.2|darwin-2level|5.8.1|darwin|7.9.0|200805232145
1511003|1511003-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200805232101
1512245|1512245-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805240206
1512408|1512408-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200805240348
1513036|1513036-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33916|linux|2.6.22-1-k7|200805240848
1513111|1513111-b19f-3f77-b713-d32bba55d77f|pass|200805|srezic@cpan.org|File-Find-Rule-Permissions|1.3|i386-freebsd|5.10.0|freebsd|6.1-release|200805240835
1513937|1513937-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 33916|linux|2.6.22-1-k7|200805241100
1515510|1515510-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805241515
1515514|1515514-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33916|linux|2.6.22-1-k7|200805241615
1515633|1515633-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33916|linux|2.6.22-1-k7|200805241635
1515743|1515743-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 33916|linux|2.6.22-1-k7|200805241652
1516065|1516065-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200805241728
1516784|1516784-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200805241904
1516931|1516931-b19f-3f77-b713-d32bba55d77f|fail|200805|srezic@cpan.org|Bryar|3.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200805241940
1517129|1517129-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200805241305
1517199|1517199-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.581|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200805241040
1517200|1517200-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33916|linux|2.6.22-1-k7|200805241039
1517791|1517791-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805241538
1517792|1517792-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805241538
1518444|1518444-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200805250045
1519128|1519128-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|0.9|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200805250431
1519161|1519161-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.0|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200805250425
1519188|1519188-b19f-3f77-b713-d32bba55d77f|fail|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.0|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200805250422
1520092|1520092-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200805251403
1520295|1520295-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805251044
1520358|1520358-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200805251605
1520824|1520824-b19f-3f77-b713-d32bba55d77f|pass|200805|srezic@cpan.org|Data-Transactional|1.0|i386-freebsd|5.10.0|freebsd|6.1-release|200805251949
1520825|1520825-b19f-3f77-b713-d32bba55d77f|pass|200805|srezic@cpan.org|Tie-STDOUT|1.03|i386-freebsd|5.10.0|freebsd|6.1-release|200805251949
1521312|1521312-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi|5.11.0 patch 33919|linux|2.6.22-1-k7|200805252129
1521493|1521493-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-thread-multi|5.11.0 patch 33919|linux|2.6.22-1-k7|200805252213
1521967|1521967-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-thread-multi|5.11.0 patch 33919|linux|2.6.22-1-k7|200805252315
1522401|1522401-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-thread-multi|5.11.0 patch 33919|linux|2.6.22-1-k7|200805260053
1522787|1522787-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200805260024
1523008|1523008-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.11.0 patch 33919|linux|2.6.22-1-k7|200805260301
1523621|1523621-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-thread-multi|5.11.0 patch 33919|linux|2.6.22-1-k7|200805260529
1523732|1523732-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.10.0 patch 33921|linux|2.4.26|200805260503
1525227|1525227-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int-2level|5.10.0 patch 33921|darwin|8.10.0|200805261723
1525775|1525775-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0 patch 33921|linux|2.4.26|200805262215
1526175|1526175-b19f-3f77-b713-d32bba55d77f|unknown|200805|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805262000
1526260|1526260-b19f-3f77-b713-d32bba55d77f|pass|200805|stro@cpan.org|Data-Compare|1.19|i686-linux|5.8.8|linux|2.4.27-2-386|200805270227
1527261|1527261-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.8.8 patch 33935|linux|2.6.22-1-k7|200805270739
1527373|1527373-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 33935|linux|2.6.22-1-k7|200805270839
1528611|1528611-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 33935|linux|2.6.22-1-k7|200805271252
1528936|1528936-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 33935|linux|2.6.22-1-k7|200805271348
1529854|1529854-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 33935|linux|2.6.22-1-k7|200805271519
1529942|1529942-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|Devel-CheckOS|1.43|darwin-2level|5.8.2|darwin|7.9.0|200805271516
1531701|1531701-b19f-3f77-b713-d32bba55d77f|unknown|200805|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805271600
1531753|1531753-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805271604
1531855|1531855-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805271618
1532012|1532012-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 33935|linux|2.6.22-1-k7|200805272231
1533859|1533859-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|Data-Hexdumper|1.4|darwin-2level|5.8.2|darwin|7.9.0|200805280504
1534264|1534264-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200805280623
1534518|1534518-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|Data-Compare|1.19|darwin-2level|5.8.2|darwin|7.9.0|200805280702
1534830|1534830-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Number-Phone|1.6|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200805280835
1534833|1534833-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Number-Phone|1.6|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200805280836
1534837|1534837-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Number-Phone|1.6|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200805280837
1534968|1534968-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.6|i686-linux-64int|5.10.0 patch 33443|linux|2.6.22-1-k7|200805281011
1534970|1534970-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.6|i686-linux-thread-multi-64int|5.10.0 patch 33628|linux|2.6.22-1-k7|200805281011
1534971|1534971-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Number-Phone|1.581|darwin-thread-multi-64int-2level|5.10.0 patch 33921|darwin|8.10.0|200805280911
1534974|1534974-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.6|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200805281011
1534976|1534976-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.6|i686-linux-thread-multi|5.8.8 patch 33430|linux|2.6.22-1-k7|200805281012
1534978|1534978-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.6|i686-linux-64int|5.8.8 patch 33243|linux|2.6.22-1-k7|200805281012
1534979|1534979-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.6|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200805281013
1534981|1534981-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.6|i686-linux-64int|5.8.8|linux|2.6.14|200805281013
1534982|1534982-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.6|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200805281013
1534986|1534986-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.6|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200805281014
1534989|1534989-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.6|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200805281015
1535216|1535216-b19f-3f77-b713-d32bba55d77f|pass|200805|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.6|sun4-solaris|5.10.0|solaris|2.9|200805281141
1535363|1535363-b19f-3f77-b713-d32bba55d77f|pass|200805|imacat@mail.imacat.idv.tw|Number-Phone|1.6|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200805281807
1535406|1535406-b19f-3f77-b713-d32bba55d77f|pass|200805|imacat@mail.imacat.idv.tw|Number-Phone|1.6|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200805281814
1535658|1535658-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Number-Phone|1.6|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200805281052
1535682|1535682-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Number-Phone|1.6|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200805281157
1535806|1535806-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805280721
1536198|1536198-b19f-3f77-b713-d32bba55d77f|pass|200805|stro@cpan.org|Number-Phone|1.6|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200805281549
1536545|1536545-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Number-Phone|1.6|i686-linux|5.6.2|linux|2.4.27-3-686|200805281657
1537012|1537012-b19f-3f77-b713-d32bba55d77f|pass|200805|srezic@cpan.org|Number-Phone|1.6|i386-freebsd|5.10.0|freebsd|6.1-release|200805281511
1537078|1537078-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.10.0 patch 33942|linux|2.4.26|200805281715
1537130|1537130-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Number-Phone|1.6|IP35-irix|5.8.8|irix|6.5|200805281838
1537263|1537263-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Number-Phone|1.6|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200805280739
1537595|1537595-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805281352
1537697|1537697-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 33939|linux|2.6.22-1-k7|200805281949
1537706|1537706-b19f-3f77-b713-d32bba55d77f|pass|200805|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Hexdumper|1.4|sun4-solaris|5.10.0|solaris|2.9|200805282003
1537736|1537736-b19f-3f77-b713-d32bba55d77f|pass|200805|srezic@cpan.org|Number-Phone|1.6|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200805281901
1537752|1537752-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Number-Phone|1.6|i686-linux|5.8.8|linux|2.4.27-3-686|200805282041
1537844|1537844-b19f-3f77-b713-d32bba55d77f|pass|200805|srezic@cpan.org|Data-Hexdumper|1.4|i386-freebsd|5.5.5|freebsd|6.1-release|200805281852
1537953|1537953-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Number-Phone|1.6|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200805281321
1538003|1538003-b19f-3f77-b713-d32bba55d77f|pass|200805|david@cantrell.org.uk|Number-Phone|1.6|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200805282101
1538119|1538119-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805281438
1538401|1538401-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33939|linux|2.6.22-1-k7|200805282009
1539141|1539141-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805281600
1539258|1539258-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805281618
1539381|1539381-b19f-3f77-b713-d32bba55d77f|pass|200805|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Compare|1.19|sun4-solaris|5.10.0|solaris|2.9|200805282235
1540914|1540914-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805281955
1541207|1541207-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33939|linux|2.6.22-1-k7|200805290225
1541712|1541712-b19f-3f77-b713-d32bba55d77f|pass|200805|stro@cpan.org|Number-Phone|1.6|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200805290428
1541900|1541900-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33939|linux|2.6.22-1-k7|200805290306
1542288|1542288-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 33939|linux|2.6.22-1-k7|200805290451
1542404|1542404-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0 patch 33942|linux|2.4.26|200805290404
1542411|1542411-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805290305
1542913|1542913-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33939|linux|2.6.22-1-k7|200805290508
1543246|1543246-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|Devel-CheckOS|1.43|darwin-2level|5.8.3|darwin|7.9.0|200805290544
1543714|1543714-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805290151
1543716|1543716-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|0.9|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805290151
1543723|1543723-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805290152
1544485|1544485-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805290403
1545530|1545530-b19f-3f77-b713-d32bba55d77f|pass|200805|frank.wiegand@gmail.com ("Frank Wiegand")|Number-Phone|1.581|i686-linux|5.11.0 patch 33939|linux|2.6.24-17-generic|200805291306
1546019|1546019-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|Number-Phone|1.6|darwin-2level|5.8.3|darwin|7.9.0|200805291421
1546378|1546378-b19f-3f77-b713-d32bba55d77f|fail|200805|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805291059
1546701|1546701-b19f-3f77-b713-d32bba55d77f|pass|200805|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Number-Phone|1.6|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200805300056
1546823|1546823-b19f-3f77-b713-d32bba55d77f|fail|200805|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805291223
1547548|1547548-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805291502
1547613|1547613-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|Number-Phone|1.6|darwin-2level|5.8.2|darwin|7.9.0|200805292012
1547844|1547844-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.6|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200805292232
1547908|1547908-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805291631
1547923|1547923-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805291632
1548260|1548260-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805291807
1549334|1549334-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805300138
1549677|1549677-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805300301
1550907|1550907-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805300326
1551105|1551105-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33949|linux|2.6.22-1-k7|200805301000
1551379|1551379-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805300436
1551554|1551554-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 33949|linux|2.6.22-1-k7|200805301024
1552861|1552861-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805300915
1553080|1553080-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805301002
1553233|1553233-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|Data-Hexdumper|1.4|darwin-2level|5.8.3|darwin|7.9.0|200805301529
1553452|1553452-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805301607
1553827|1553827-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.10.0 patch 33947|linux|2.4.26|200805301707
1554311|1554311-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|Data-Compare|1.19|darwin-2level|5.8.3|darwin|7.9.0|200805301832
1555790|1555790-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200805302248
1556369|1556369-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33949|linux|2.6.22-1-k7|200805310153
1556382|1556382-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33949|linux|2.6.22-1-k7|200805310149
1557706|1557706-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33949|linux|2.6.22-1-k7|200805310526
1558052|1558052-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0 patch 33947|linux|2.4.26|200805310524
1558204|1558204-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 33949|linux|2.6.22-1-k7|200805310629
1558565|1558565-b19f-3f77-b713-d32bba55d77f|pass|200805|jj@jonallen.info ("JJ")|XML-Tiny|1.11|darwin-2level|5.8.2|darwin|7.9.0|200805310703
1559072|1559072-b19f-3f77-b713-d32bba55d77f|na|200805|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805310424
1559315|1559315-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200805311047
1559566|1559566-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200805311146
1559768|1559768-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200805311229
1560142|1560142-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200805311240
1560523|1560523-b19f-3f77-b713-d32bba55d77f|fail|200805|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805310810
1560668|1560668-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805310821
1561580|1561580-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int-2level|5.10.0 patch 33947|darwin|8.10.0|200805311505
1561739|1561739-b19f-3f77-b713-d32bba55d77f|pass|200805|chris@bingosnet.co.uk|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200805311427
1563597|1563597-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805311346
1563631|1563631-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200805311949
1563925|1563925-b19f-3f77-b713-d32bba55d77f|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200805312016
1564012|1564012-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805311430
1564013|1564013-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805311431
1565202|1565202-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805311633
1565287|1565287-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200805311647
1566031|1566031-b19f-3f77-b713-d32bba55d77f|pass|200805|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200805311827
1566359|1566359-b19f-3f77-b713-d32bba55d77f|unknown|200805|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805311901
1567194|1567194-b19f-3f77-b713-d32bba55d77f|pass|200806|icerider70@gmail.com ("ICERIDER")|Data-Compare|1.19|i386-linux|5.8.8|linux|2.6.18-53.el5|200806010053
1567352|1567352-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806010110
1567593|1567593-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806010135
1567652|1567652-b19f-3f77-b713-d32bba55d77f|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200806010141
1567710|1567710-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200806010147
1567753|1567753-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200806010153
1569849|1569849-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|cygwin-thread-multi-64int|5.10.0 patch 33947|cygwin|1.5.25(0.15642)|200806011153
1570315|1570315-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806010845
1570728|1570728-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806010925
1570773|1570773-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806010933
1571548|1571548-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806011106
1571625|1571625-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Number-Phone|1.6|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200806011615
1571654|1571654-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806011119
1572524|1572524-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200806011401
1572574|1572574-b19f-3f77-b713-d32bba55d77f|fail|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200806011421
1574018|1574018-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806011810
1574265|1574265-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806011854
1574269|1574269-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|0.9|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806011854
1574295|1574295-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806011857
1574313|1574313-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Number-Phone|1.6|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200806020000
1575798|1575798-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806020329
1575812|1575812-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806020331
1575813|1575813-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|0.9|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806020331
1575819|1575819-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806020332
1576599|1576599-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806020112
1578359|1578359-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33986|linux|2.6.22-1-k7|200806021115
1578579|1578579-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33986|linux|2.6.22-1-k7|200806021128
1578637|1578637-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806020544
1578699|1578699-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Number-Phone|1.6|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200806021049
1578853|1578853-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806020604
1578906|1578906-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33986|linux|2.6.22-1-k7|200806021204
1579398|1579398-b19f-3f77-b713-d32bba55d77f|fail|200806|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806020705
1579434|1579434-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200806020709
1580012|1580012-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200806021223
1581408|1581408-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806021328
1581689|1581689-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Number-Phone|1.6|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200806021910
1581836|1581836-b19f-3f77-b713-d32bba55d77f|pass|200806|jj@jonallen.info ("JJ")|XML-Tiny|1.11|darwin-2level|5.8.3|darwin|7.9.0|200806021935
1581849|1581849-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.0|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806021441
1581858|1581858-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806021442
1582149|1582149-b19f-3f77-b713-d32bba55d77f|pass|200806|flemming@ipk-gatersleben.de|Data-Compare|1.19|x86_64-linux|5.8.5|linux|2.6.9-55.0.9.elsmp|200806021423
1582419|1582419-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 33986|linux|2.6.22-1-k7|200806022203
1582436|1582436-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806021604
1582778|1582778-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806021631
1583949|1583949-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.0|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806021819
1583985|1583985-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806021821
1584228|1584228-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33986|linux|2.6.22-1-k7|200806030046
1584692|1584692-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Number-Phone|1.6|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200806030026
1584702|1584702-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806021928
1584989|1584989-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806021949
1585271|1585271-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 33986|linux|2.6.22-1-k7|200806030216
1585349|1585349-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806030023
1587211|1587211-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806030010
1587592|1587592-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806030118
1589241|1589241-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806030433
1589570|1589570-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806030510
1590004|1590004-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806030604
1590985|1590985-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806030830
1591364|1591364-b19f-3f77-b713-d32bba55d77f|pass|200806|icerider70@gmail.com ("ICERIDER")|Data-Hexdumper|1.4|i386-linux|5.8.8|linux|2.6.18-53.el5|200806030838
1591564|1591564-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Number-Phone|1.6|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200806031308
1592681|1592681-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Number-Phone|1.6|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200806031827
1594222|1594222-b19f-3f77-b713-d32bba55d77f|fail|200806|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806031907
1594268|1594268-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806031916
1596106|1596106-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200806040506
1596136|1596136-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806040103
1596418|1596418-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806040139
1596420|1596420-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806040140
1597040|1597040-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806040324
1598207|1598207-b19f-3f77-b713-d32bba55d77f|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806040538
1599062|1599062-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806040715
1599765|1599765-b19f-3f77-b713-d32bba55d77f|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806040824
1599815|1599815-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806040829
1599859|1599859-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806040835
1600101|1600101-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806040852
1602119|1602119-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200806041622
1602224|1602224-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200806041822
1602252|1602252-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200806041824
1602709|1602709-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806041318
1602894|1602894-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200806041850
1603182|1603182-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806041406
1604234|1604234-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806041543
1604457|1604457-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806041611
1604702|1604702-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806041631
1604828|1604828-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806041642
1605376|1605376-b19f-3f77-b713-d32bba55d77f|fail|200806|jj@jonallen.info ("JJ")|CPAN-FindDependencies|2.0|darwin-2level|5.8.1|darwin|7.9.0|200806042232
1606004|1606004-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200806050018
1606511|1606511-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200806050051
1606666|1606666-b19f-3f77-b713-d32bba55d77f|pass|200806|icerider70@gmail.com ("ICERIDER")|Number-Phone|1.6|i386-linux|5.8.8|linux|2.6.18-53.el5|200806041827
1607012|1607012-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200806050157
1607495|1607495-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806050045
1609203|1609203-b19f-3f77-b713-d32bba55d77f|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPU-Emulator-Memory|1.1001|sun4-solaris|5.10.0|solaris|2.9|200806050719
1609209|1609209-b19f-3f77-b713-d32bba55d77f|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPU-Emulator-Z80|0.9|sun4-solaris|5.10.0|solaris|2.9|200806050720
1609274|1609274-b19f-3f77-b713-d32bba55d77f|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckLib|0.5|sun4-solaris|5.10.0|solaris|2.9|200806050731
1609546|1609546-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806050239
1609548|1609548-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|0.9|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806050239
1609553|1609553-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806050240
1610500|1610500-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806050511
1610872|1610872-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200806051117
1612263|1612263-b19f-3f77-b713-d32bba55d77f|pass|200806|rurban@x-ray.at (Reini Urban)|Data-Compare|1.19|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.25(0.15642)|200806051131
1612459|1612459-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200806051454
1613348|1613348-b19f-3f77-b713-d32bba55d77f|fail|200806|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806051119
1613350|1613350-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806051119
1613481|1613481-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806051131
1614296|1614296-b19f-3f77-b713-d32bba55d77f|fail|200806|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806051253
1615657|1615657-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806051522
1616492|1616492-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806051650
1616507|1616507-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806051652
1616957|1616957-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806051741
1617027|1617027-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806051749
1617452|1617452-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|XML-Tiny|1.11|IP35-irix-64int|5.8.8|irix|6.5|200806060027
1617506|1617506-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806051831
1617508|1617508-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806051832
1617612|1617612-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|CPU-Emulator-Memory|1.1001|IP35-irix-64int|5.8.8|irix|6.5|200806060044
1617620|1617620-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|CPU-Emulator-Z80|0.9|IP35-irix-64int|5.8.8|irix|6.5|200806060044
1618807|1618807-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806060122
1619151|1619151-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200806060405
1619228|1619228-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806060216
1619306|1619306-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806060225
1620600|1620600-b19f-3f77-b713-d32bba55d77f|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806060107
1622038|1622038-b19f-3f77-b713-d32bba55d77f|pass|200806|stro@cpan.org|CPU-Emulator-Memory|1.1001|i686-linux|5.8.8|linux|2.4.27-2-386|200806060117
1622052|1622052-b19f-3f77-b713-d32bba55d77f|pass|200806|stro@cpan.org|CPU-Emulator-Z80|0.9|i686-linux|5.8.8|linux|2.4.27-2-386|200806060118
1622201|1622201-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200806061127
1623332|1623332-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806060916
1623843|1623843-b19f-3f77-b713-d32bba55d77f|pass|200806|andy@hexten.net|Data-Hexdumper|1.4|darwin-thread-multi-2level|5.10.0|darwin|9.2.2|200806061504
1624056|1624056-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806061023
1624257|1624257-b19f-3f77-b713-d32bba55d77f|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806061039
1624325|1624325-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806061044
1624400|1624400-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806061052
1625162|1625162-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200806061703
1626149|1626149-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806061413
1626487|1626487-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806061453
1627601|1627601-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Number-Phone|1.6|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200806062213
1628001|1628001-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806061748
1629230|1629230-b19f-3f77-b713-d32bba55d77f|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-DoubleEncodedEntities|1.0|sun4-solaris|5.10.0|solaris|2.9|200806070226
1629294|1629294-b19f-3f77-b713-d32bba55d77f|fail|200806|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806070036
1629367|1629367-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806070044
1629841|1629841-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806070146
1630504|1630504-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806070320
1631286|1631286-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806070101
1631562|1631562-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806070138
1631563|1631563-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806070138
1632128|1632128-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806070307
1632247|1632247-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806070335
1632360|1632360-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806070354
1633163|1633163-b19f-3f77-b713-d32bba55d77f|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806070545
1634343|1634343-b19f-3f77-b713-d32bba55d77f|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806070823
1634390|1634390-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806070827
1634430|1634430-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806070833
1635103|1635103-b19f-3f77-b713-d32bba55d77f|pass|200806|andy@hexten.net|Data-Compare|1.19|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200806071437
1635106|1635106-b19f-3f77-b713-d32bba55d77f|pass|200806|andy@hexten.net|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200806071437
1635216|1635216-b19f-3f77-b713-d32bba55d77f|pass|200806|andy@hexten.net|Data-Compare|1.19|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200806071445
1635218|1635218-b19f-3f77-b713-d32bba55d77f|pass|200806|andy@hexten.net|Data-Hexdumper|1.4|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200806071445
1635354|1635354-b19f-3f77-b713-d32bba55d77f|pass|200806|andy@hexten.net|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.6.20-16-server|200806071455
1635515|1635515-b19f-3f77-b713-d32bba55d77f|pass|200806|andy@hexten.net|Number-Phone|1.6|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200806071505
1635990|1635990-b19f-3f77-b713-d32bba55d77f|pass|200806|andy@hexten.net|Number-Phone|1.6|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200806071524
1636361|1636361-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-freebsd-64int|5.11.0 patch 34005|freebsd|7.0-release|200806071453
1636805|1636805-b19f-3f77-b713-d32bba55d77f|pass|200806|stro@cpan.org|Data-Hexdumper|1.4|i686-linux|5.8.8|linux|2.4.27-2-386|200806071730
1637399|1637399-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806071448
1637953|1637953-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806071623
1638005|1638005-b19f-3f77-b713-d32bba55d77f|pass|200806|stro@cpan.org|Data-Transactional|1.0|i686-linux|5.8.8|linux|2.4.27-2-386|200806072033
1638178|1638178-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806071700
1639376|1639376-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806080352
1639451|1639451-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806080007
1640367|1640367-b19f-3f77-b713-d32bba55d77f|pass|200806|jj@jonallen.info ("JJ")|Devel-CheckLib|0.5|darwin-2level|5.8.2|darwin|7.9.0|200806080749
1641848|1641848-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806080705
1641850|1641850-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|0.9|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806080705
1641862|1641862-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806080707
1642539|1642539-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806080853
1642540|1642540-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|0.9|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806080854
1642550|1642550-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806080855
1643221|1643221-b19f-3f77-b713-d32bba55d77f|pass|200806|jj@jonallen.info ("JJ")|CPU-Emulator-Memory|1.1001|darwin-2level|5.8.3|darwin|7.9.0|200806081536
1643232|1643232-b19f-3f77-b713-d32bba55d77f|pass|200806|jj@jonallen.info ("JJ")|CPU-Emulator-Z80|0.9|darwin-2level|5.8.3|darwin|7.9.0|200806081537
1643318|1643318-b19f-3f77-b713-d32bba55d77f|pass|200806|jj@jonallen.info ("JJ")|Devel-CheckLib|0.5|darwin-2level|5.8.3|darwin|7.9.0|200806081554
1643424|1643424-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806081116
1644483|1644483-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806081431
1645296|1645296-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806081757
1646787|1646787-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-64int|5.11.0 patch 34005|freebsd|7.0-release|200806090359
1646850|1646850-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.10.0 patch 33976|darwin|8.10.0|200806090948
1647408|1647408-b19f-3f77-b713-d32bba55d77f|pass|200806|jj@jonallen.info ("JJ")|CPU-Emulator-Memory|1.1001|darwin-2level|5.8.2|darwin|7.9.0|200806091145
1647420|1647420-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200806091147
1647683|1647683-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806090759
1648404|1648404-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806090946
1648608|1648608-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Number-Phone|1.6|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200806091520
1649407|1649407-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806091208
1649840|1649840-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806091325
1650133|1650133-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Number-Phone|1.6|i386-freebsd-64int|5.11.0 patch 34005|freebsd|7.0-release|200806091233
1650326|1650326-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.0|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806091553
1650336|1650336-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806091554
1651712|1651712-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806091940
1652060|1652060-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806100044
1653013|1653013-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806100317
1653402|1653402-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806100019
1653717|1653717-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806100117
1654962|1654962-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806100426
1655173|1655173-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806100502
1655194|1655194-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.0|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806100505
1655216|1655216-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806100508
1655523|1655523-b19f-3f77-b713-d32bba55d77f|pass|200806|jj@jonallen.info ("JJ")|File-Overwrite|1.0|darwin-2level|5.8.1|darwin|7.9.0|200806101102
1655746|1655746-b19f-3f77-b713-d32bba55d77f|pass|200806|jj@jonallen.info ("JJ")|XML-DoubleEncodedEntities|1.0|darwin-2level|5.8.2|darwin|7.9.0|200806101129
1657381|1657381-b19f-3f77-b713-d32bba55d77f|fail|200806|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806101014
1657448|1657448-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806101021
1658997|1658997-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-freebsd-64int|5.11.0 patch 34005|freebsd|7.0-release|200806100709
1659175|1659175-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806101422
1659193|1659193-b19f-3f77-b713-d32bba55d77f|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPAN-FindDependencies|2.0|sun4-solaris|5.10.0|solaris|2.9|200806102025
1659368|1659368-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|alpha-linux-thread-multi|5.11.0 patch 34031|linux|2.6.18-5-alpha-generic|200806101942
1659428|1659428-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806101451
1659430|1659430-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806101451
1660151|1660151-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806101620
1660908|1660908-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806101750
1661054|1661054-b19f-3f77-b713-d32bba55d77f|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806101804
1661124|1661124-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Compare|1.19|i386-freebsd-64int|5.11.0 patch 34005|freebsd|7.0-release|200806102322
1662159|1662159-b19f-3f77-b713-d32bba55d77f|pass|200806|jj@jonallen.info ("JJ")|XML-DoubleEncodedEntities|1.0|darwin-2level|5.8.3|darwin|7.9.0|200806110209
1662363|1662363-b19f-3f77-b713-d32bba55d77f|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806110130
1662410|1662410-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806110134
1662449|1662449-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806110138
1664439|1664439-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806110439
1664738|1664738-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806110545
1665737|1665737-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806110746
1665964|1665964-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806110814
1666351|1666351-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806110922
1666368|1666368-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200806111417
1666437|1666437-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806110933
1667846|1667846-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806111412
1668478|1668478-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806111638
1669283|1669283-b19f-3f77-b713-d32bba55d77f|pass|200806|andy@hexten.net|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.6.18-8.el5|200806120142
1669424|1669424-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 34037|linux|2.6.22-1-k7|200806120259
1669539|1669539-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806120117
1669540|1669540-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|0.9|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806120117
1669547|1669547-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806120118
1669897|1669897-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34037|linux|2.6.22-1-k7|200806120423
1670596|1670596-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806120003
1671876|1671876-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806120241
1672276|1672276-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34037|linux|2.6.22-1-k7|200806120933
1672311|1672311-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806120336
1672452|1672452-b19f-3f77-b713-d32bba55d77f|fail|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Bryar|3.0|sun4-solaris|5.10.0|solaris|2.9|200806120955
1672725|1672725-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806120443
1673498|1673498-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806120646
1673962|1673962-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806120744
1674081|1674081-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806120801
1674093|1674093-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806120802
1674468|1674468-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806120851
1675737|1675737-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806121133
1676188|1676188-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806121226
1676852|1676852-b19f-3f77-b713-d32bba55d77f|pass|200806|kamensky@cpan.org ("Moshe Kamensky")|Data-Compare|1.19|sun4-solaris|5.10.0|solaris|2.8|200806121006
1677239|1677239-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.8.8 patch 34037|linux|2.6.22-1-k7|200806121957
1677784|1677784-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34037|linux|2.6.22-1-k7|200806122038
1678451|1678451-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806121559
1679107|1679107-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806121659
1680250|1680250-b19f-3f77-b713-d32bba55d77f|fail|200806|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806121900
1680311|1680311-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806121912
1680879|1680879-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806130037
1681135|1681135-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806130111
1682379|1682379-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34037|linux|2.6.22-1-k7|200806130553
1684521|1684521-b19f-3f77-b713-d32bba55d77f|fail|200806|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806130829
1684598|1684598-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806130837
1685330|1685330-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806130950
1686169|1686169-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806131105
1686172|1686172-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806131105
1686674|1686674-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806131150
1686957|1686957-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806131217
1686959|1686959-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806131218
1687989|1687989-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806131500
1688343|1688343-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806131528
1688392|1688392-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 34034|linux|2.6.22-1-k7|200806132133
1688860|1688860-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 34034|linux|2.6.22-1-k7|200806132202
1689558|1689558-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200806132254
1689563|1689563-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|1.0|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200806132255
1689568|1689568-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|1.0|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200806132255
1689573|1689573-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200806132256
1689579|1689579-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|1.0|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200806132257
1689623|1689623-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200806132300
1689644|1689644-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200806132302
1689666|1689666-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200806132304
1689729|1689729-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|1.0|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200806132257
1689733|1689733-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|1.0|i686-linux-64int|5.8.8|linux|2.6.14|200806132259
1689826|1689826-b19f-3f77-b713-d32bba55d77f|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806131713
1690007|1690007-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200806131927
1690730|1690730-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.10.0 patch 34034|linux|2.6.22-1-k7|200806132226
1690743|1690743-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.10.0 patch 34034|linux|2.6.22-1-k7|200806132221
1690801|1690801-b19f-3f77-b713-d32bba55d77f|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806131827
1691351|1691351-b19f-3f77-b713-d32bba55d77f|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806131909
1691414|1691414-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806131913
1691484|1691484-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806131917
1691805|1691805-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 34034|linux|2.6.22-1-k7|200806140110
1693311|1693311-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|1.0|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200806140324
1693934|1693934-b19f-3f77-b713-d32bba55d77f|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806140325
1693991|1693991-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806140331
1694046|1694046-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806140339
1694082|1694082-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806140343
1694324|1694324-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806140001
1694392|1694392-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 34034|linux|2.6.22-1-k7|200806140606
1694436|1694436-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200806140510
1694616|1694616-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806140025
1696842|1696842-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|CPU-Emulator-Z80|1.0|i686-linux|5.8.8|linux|2.4.27-3-686|200806141100
1696860|1696860-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|CPU-Emulator-Z80|1.0|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200806141107
1696874|1696874-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|CPU-Emulator-Z80|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200806141101
1696950|1696950-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|CPU-Emulator-Z80|1.0|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200806141132
1696955|1696955-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|CPU-Emulator-Z80|1.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200806140533
1696965|1696965-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|CPU-Emulator-Z80|1.0|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200806141035
1697410|1697410-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806140818
1697529|1697529-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806140837
1697947|1697947-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806140939
1698033|1698033-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806140950
1698059|1698059-b19f-3f77-b713-d32bba55d77f|pass|200806|jj@jonallen.info ("JJ")|CPU-Emulator-Z80|1.0|darwin-2level|5.8.3|darwin|7.9.0|200806141453
1698529|1698529-b19f-3f77-b713-d32bba55d77f|pass|200806|imacat@mail.imacat.idv.tw|CPU-Emulator-Z80|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200806142300
1698538|1698538-b19f-3f77-b713-d32bba55d77f|pass|200806|imacat@mail.imacat.idv.tw|CPU-Emulator-Z80|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200806142301
1699335|1699335-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int-ld|5.11.0 patch 34053|linux|2.6.21.5-smp|200806141722
1699342|1699342-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-64int-ld|5.11.0 patch 34053|linux|2.6.21.5-smp|200806141723
1699778|1699778-b19f-3f77-b713-d32bba55d77f|pass|200806|stro@cpan.org|XML-Tiny|1.11|i386-freebsd-64int|5.8.8|freebsd|7.0-release|200806142055
1700297|1700297-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806141629
1700782|1700782-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806141752
1701090|1701090-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200806150036
1702144|1702144-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806150218
1702831|1702831-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34053|linux|2.6.22-1-k7|200806151122
1702859|1702859-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34053|linux|2.6.22-1-k7|200806151142
1702893|1702893-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34053|linux|2.6.22-1-k7|200806151145
1703095|1703095-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200806151047
1703430|1703430-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34053|linux|2.6.22-1-k7|200806151344
1703470|1703470-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.4.26|200806151250
1703635|1703635-b19f-3f77-b713-d32bba55d77f|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|File-Find-Rule-Permissions|1.3|sun4-solaris|5.10.0|solaris|2.9|200806151434
1704329|1704329-b19f-3f77-b713-d32bba55d77f|fail|200806|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806151039
1704932|1704932-b19f-3f77-b713-d32bba55d77f|fail|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200806151653
1704995|1704995-b19f-3f77-b713-d32bba55d77f|fail|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200806151702
1705126|1705126-b19f-3f77-b713-d32bba55d77f|fail|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200806151716
1705195|1705195-b19f-3f77-b713-d32bba55d77f|fail|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200806151722
1705271|1705271-b19f-3f77-b713-d32bba55d77f|fail|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200806151727
1705279|1705279-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806151230
1705323|1705323-b19f-3f77-b713-d32bba55d77f|fail|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200806151733
1706395|1706395-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34053|linux|2.6.22-1-k7|200806152106
1706425|1706425-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34053|linux|2.6.22-1-k7|200806152058
1706452|1706452-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.11.0 patch 34053|linux|2.6.21.5-smp|200806151921
1707633|1707633-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200806160103
1707653|1707653-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.0|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806160101
1707663|1707663-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806160102
1707973|1707973-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806160144
1708315|1708315-b19f-3f77-b713-d32bba55d77f|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPU-Emulator-Z80|1.0|sun4-solaris|5.10.0|solaris|2.9|200806160502
1708637|1708637-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|darwin-thread-multi-64int-2level|5.10.0 patch 34034|darwin|8.10.0|200806160805
1708672|1708672-b19f-3f77-b713-d32bba55d77f|pass|200806|andy@hexten.net|Data-Compare|1.19|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200806160903
1708673|1708673-b19f-3f77-b713-d32bba55d77f|pass|200806|andy@hexten.net|Data-Hexdumper|1.4|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200806160905
1709088|1709088-b19f-3f77-b713-d32bba55d77f|pass|200806|icerider70@gmail.com ("ICERIDER")|CPU-Emulator-Z80|1.0|i386-linux|5.8.8|linux|2.6.18-53.el5|200806160648
1709801|1709801-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806161222
1709934|1709934-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200806161651
1710635|1710635-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806161430
1710712|1710712-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806161446
1711100|1711100-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Transactional|1.02|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200806162138
1711102|1711102-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Transactional|1.02|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200806162139
1711104|1711104-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Transactional|1.02|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200806162139
1711107|1711107-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Transactional|1.02|i686-linux-64int|5.8.8|linux|2.6.14|200806162139
1711109|1711109-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Transactional|1.02|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200806162139
1711111|1711111-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Transactional|1.02|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200806162140
1711114|1711114-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Transactional|1.02|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200806162140
1711130|1711130-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806161542
1711213|1711213-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806161554
1711344|1711344-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Transactional|1.02|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200806162139
1711788|1711788-b19f-3f77-b713-d32bba55d77f|pass|200806|jj@jonallen.info ("JJ")|Number-Phone|1.6|darwin-2level|5.8.1|darwin|7.9.0|200806162225
1712009|1712009-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200806162253
1712014|1712014-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200806162255
1712052|1712052-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200806162257
1712140|1712140-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200806162312
1712173|1712173-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200806162314
1712180|1712180-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200806162316
1713291|1713291-b19f-3f77-b713-d32bba55d77f|pass|200806|icerider70@gmail.com ("ICERIDER")|Data-Transactional|1.02|i386-linux|5.8.8|linux|2.6.18-53.el5|200806170216
1713400|1713400-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.02|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806170325
1713550|1713550-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806170343
1713618|1713618-b19f-3f77-b713-d32bba55d77f|pass|200806|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Data-Transactional|1.02|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200806171259
1713683|1713683-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806170407
1713736|1713736-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806170019
1713937|1713937-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.4.26|200806170606
1713951|1713951-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.4.26|200806170608
1714719|1714719-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806170416
1714825|1714825-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200806170905
1714858|1714858-b19f-3f77-b713-d32bba55d77f|pass|200806|imacat@mail.imacat.idv.tw|Data-Transactional|1.02|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200806171643
1714961|1714961-b19f-3f77-b713-d32bba55d77f|pass|200806|imacat@mail.imacat.idv.tw|Data-Transactional|1.02|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200806171700
1714995|1714995-b19f-3f77-b713-d32bba55d77f|pass|200806|stro@cpan.org|Data-Transactional|1.01|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200806171208
1714999|1714999-b19f-3f77-b713-d32bba55d77f|pass|200806|stro@cpan.org|Data-Transactional|1.02|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200806171208
1715113|1715113-b19f-3f77-b713-d32bba55d77f|fail|200806|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806170534
1715183|1715183-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|Data-Transactional|1.01|i686-linux|5.8.8|linux|2.4.27-3-686|200806171047
1715190|1715190-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|Data-Transactional|1.01|i686-linux|5.6.2|linux|2.4.27-3-686|200806171048
1715208|1715208-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|Data-Transactional|1.02|i686-linux|5.8.8|linux|2.4.27-3-686|200806171047
1715210|1715210-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|Data-Transactional|1.02|i686-linux|5.6.2|linux|2.4.27-3-686|200806171048
1715252|1715252-b19f-3f77-b713-d32bba55d77f|pass|200806|jj@jonallen.info ("JJ")|Data-Transactional|1.02|darwin-2level|5.8.2|darwin|7.9.0|200806171056
1715318|1715318-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|Data-Transactional|1.01|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200806171105
1715320|1715320-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|Data-Transactional|1.02|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200806171105
1715342|1715342-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|Data-Transactional|1.01|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200806170509
1715353|1715353-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|Data-Transactional|1.02|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200806170510
1715365|1715365-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|Data-Transactional|1.01|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200806171106
1715367|1715367-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|Data-Transactional|1.02|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200806171107
1715383|1715383-b19f-3f77-b713-d32bba55d77f|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Transactional|1.02|sun4-solaris|5.10.0|solaris|2.9|200806171214
1715448|1715448-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|Data-Transactional|1.01|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200806171059
1715465|1715465-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|Data-Transactional|1.02|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200806171101
1715871|1715871-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806170756
1716180|1716180-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|Data-Compare|1.19|IP35-irix-64int|5.8.8|irix|6.5|200806171455
1716184|1716184-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|Data-Transactional|1.01|IP35-irix-64int|5.8.8|irix|6.5|200806171455
1716186|1716186-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|Data-Transactional|1.02|IP35-irix-64int|5.8.8|irix|6.5|200806171455
1716276|1716276-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806170912
1716633|1716633-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|Data-Transactional|1.01|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200806171400
1716636|1716636-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|Data-Transactional|1.02|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200806171400
1716688|1716688-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806171006
1716761|1716761-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200806170909
1716770|1716770-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200806170910
1716945|1716945-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200806171504
1717130|1717130-b19f-3f77-b713-d32bba55d77f|pass|200806|robertl@apnic.net ("Robert Loomans")|CPAN-FindDependencies|2.0|darwin-2level|5.8.8|darwin|9.2.2|200806171530
1717223|1717223-b19f-3f77-b713-d32bba55d77f|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Tie-STDOUT|1.03|sun4-solaris|5.10.0|solaris|2.9|200806171701
1717464|1717464-b19f-3f77-b713-d32bba55d77f|pass|200806|srezic@cpan.org|CPU-Emulator-Z80|1.0|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200806171732
1718386|1718386-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806171319
1718665|1718665-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 34066|linux|2.6.22-1-k7|200806171931
1719000|1719000-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.8.8 patch 34066|linux|2.6.22-1-k7|200806171958
1719112|1719112-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806171420
1719207|1719207-b19f-3f77-b713-d32bba55d77f|pass|200806|srezic@cpan.org|Data-Transactional|1.02|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200806172032
1719292|1719292-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200806171946
1719316|1719316-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200806171948
1719348|1719348-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200806171950
1719382|1719382-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200806171952
1719408|1719408-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200806171954
1719445|1719445-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200806171956
1719472|1719472-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200806171958
1719487|1719487-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200806172000
1719502|1719502-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200806172001
1719510|1719510-b19f-3f77-b713-d32bba55d77f|pass|200806|srezic@cpan.org|Data-Transactional|1.02|i386-freebsd|5.6.2|freebsd|6.1-release|200806172058
1719534|1719534-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200806172004
1719991|1719991-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200806172058
1720006|1720006-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200806172059
1720028|1720028-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200806172101
1720046|1720046-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34066|linux|2.6.22-1-k7|200806172202
1720049|1720049-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200806172103
1720057|1720057-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200806172104
1720065|1720065-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200806172105
1720087|1720087-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200806172107
1720104|1720104-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200806172108
1720126|1720126-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200806172110
1720153|1720153-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200806172111
1720412|1720412-b19f-3f77-b713-d32bba55d77f|pass|200806|srezic@cpan.org|Data-Transactional|1.02|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200806172156
1720585|1720585-b19f-3f77-b713-d32bba55d77f|na|200806|srezic@cpan.org|Data-Transactional|1.02|i386-freebsd|5.5.5|freebsd|6.1-release|200806172255
1720855|1720855-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|Devel-CheckOS|1.43|IP35-irix-64int|5.8.8|irix|6.5|200806172325
1721186|1721186-b19f-3f77-b713-d32bba55d77f|pass|200806|srezic@cpan.org|Data-Transactional|1.02|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200806180001
1721360|1721360-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806171817
1721943|1721943-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34066|linux|2.6.22-1-k7|200806172057
1722051|1722051-b19f-3f77-b713-d32bba55d77f|pass|200806|srezic@cpan.org|Data-Transactional|1.02|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200806180106
1722175|1722175-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806171926
1722205|1722205-b19f-3f77-b713-d32bba55d77f|pass|200806|srezic@cpan.org|Data-Transactional|1.02|i386-freebsd|5.10.0|freebsd|6.1-release|200806180128
1722423|1722423-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200806180029
1722727|1722727-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int-2level|5.10.0 patch 34065|darwin|8.10.0|200806180144
1722737|1722737-b19f-3f77-b713-d32bba55d77f|pass|200806|srezic@cpan.org|CPU-Emulator-Z80|1.0|i386-freebsd|5.10.0|freebsd|6.1-release|200806180244
1723607|1723607-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806180258
1723848|1723848-b19f-3f77-b713-d32bba55d77f|pass|200806|srezic@cpan.org|CPU-Emulator-Z80|1.0|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200806180539
1723888|1723888-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806180348
1725628|1725628-b19f-3f77-b713-d32bba55d77f|pass|200806|srezic@cpan.org|Number-Phone|1.6|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200806180837
1726053|1726053-b19f-3f77-b713-d32bba55d77f|pass|200806|srezic@cpan.org|CPU-Emulator-Z80|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200806180944
1726116|1726116-b19f-3f77-b713-d32bba55d77f|na|200806|srezic@cpan.org|Number-Phone|1.6|i386-freebsd|5.5.5|freebsd|6.1-release|200806180953
1726167|1726167-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34066|linux|2.6.22-1-k7|200806180942
1726246|1726246-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34066|linux|2.6.22-1-k7|200806180959
1726885|1726885-b19f-3f77-b713-d32bba55d77f|fail|200806|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806180509
1727011|1727011-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806180517
1727998|1727998-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200806181118
1728032|1728032-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200806180257
1728632|1728632-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.4.26|200806181255
1729448|1729448-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806180919
1729848|1729848-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806180951
1729855|1729855-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806180950
1730104|1730104-b19f-3f77-b713-d32bba55d77f|pass|200806|srezic@cpan.org|Number-Phone|1.6|i386-freebsd|5.6.2|freebsd|6.1-release|200806181532
1730602|1730602-b19f-3f77-b713-d32bba55d77f|pass|200806|stro@cpan.org|File-Find-Rule-Permissions|1.3|i686-linux|5.8.8|linux|2.4.27-2-386|200806181452
1730977|1730977-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806181129
1731015|1731015-b19f-3f77-b713-d32bba55d77f|pass|200806|stro@cpan.org|File-Overwrite|1.0|i686-linux|5.8.8|linux|2.4.27-2-386|200806181535
1732000|1732000-b19f-3f77-b713-d32bba55d77f|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Palm-TreoPhoneCallDB|1.1|sun4-solaris|5.10.0|solaris|2.9|200806181923
1732178|1732178-b19f-3f77-b713-d32bba55d77f|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806181340
1733402|1733402-b19f-3f77-b713-d32bba55d77f|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806181558
1733445|1733445-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806181602
1733518|1733518-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806181606
1735264|1735264-b19f-3f77-b713-d32bba55d77f|pass|200806|srezic@cpan.org|CPU-Emulator-Z80|1.0|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200806190000
1735338|1735338-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806181844
1735339|1735339-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.02|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806181844
1736559|1736559-b19f-3f77-b713-d32bba55d77f|pass|200806|stro@cpan.org|Data-Transactional|1.02|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200806190450
1736819|1736819-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806190232
1737043|1737043-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.11.0 patch 34073|netbsd|4.0|200806190418
1737055|1737055-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i386-netbsd-thread-multi-64int|5.11.0 patch 34073|netbsd|4.0|200806190419
1737077|1737077-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.11.0 patch 34073|netbsd|4.0|200806190421
1737080|1737080-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i386-netbsd-thread-multi-64int|5.11.0 patch 34073|netbsd|4.0|200806190421
1737088|1737088-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i386-netbsd-thread-multi-64int|5.11.0 patch 34073|netbsd|4.0|200806190422
1737193|1737193-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806190352
1737404|1737404-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806190037
1737459|1737459-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806190048
1737510|1737510-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806190053
1737535|1737535-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806190058
1738004|1738004-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806190237
1738286|1738286-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Number-Phone|1.6|darwin-thread-multi-64int-2level|5.10.0 patch 34065|darwin|8.10.0|200806190841
1739632|1739632-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806190843
1739702|1739702-b19f-3f77-b713-d32bba55d77f|pass|200806|srezic@cpan.org|Number-Phone|1.6|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200806191453
1740244|1740244-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806191013
1740729|1740729-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.11.0 patch 34073|netbsd|4.0|200806191653
1741329|1741329-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.11.0 patch 34073|netbsd|4.0|200806191749
1741735|1741735-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806191338
1741954|1741954-b19f-3f77-b713-d32bba55d77f|pass|200806|srezic@cpan.org|Tie-STDOUT|1.03|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200806192004
1742423|1742423-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200806192013
1742430|1742430-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200806192014
1742448|1742448-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Compare|1.19|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200806192017
1742449|1742449-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200806192017
1742455|1742455-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200806192019
1742894|1742894-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806191719
1743652|1743652-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806191852
1744687|1744687-b19f-3f77-b713-d32bba55d77f|pass|200806|srezic@cpan.org|Number-Phone-UK-DetailedLocations|1.3|i386-freebsd|5.6.2|freebsd|6.1-release|200806200318
1744728|1744728-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806200133
1745499|1745499-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200806200427
1745508|1745508-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200806200428
1745580|1745580-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806200338
1745643|1745643-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200806200444
1745655|1745655-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200806200446
1745714|1745714-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806200355
1746263|1746263-b19f-3f77-b713-d32bba55d77f|na|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806200113
1746585|1746585-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806200154
1747063|1747063-b19f-3f77-b713-d32bba55d77f|pass|200806|srezic@cpan.org|Number-Phone-UK-DetailedLocations|1.3|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200806200848
1747364|1747364-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34074|linux|2.6.22-1-k7|200806200933
1747482|1747482-b19f-3f77-b713-d32bba55d77f|pass|200806|srezic@cpan.org|Number-Phone-UK-DetailedLocations|1.3|i386-freebsd|5.10.0|freebsd|6.1-release|200806200954
1748185|1748185-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34074|linux|2.6.22-1-k7|200806201012
1748835|1748835-b19f-3f77-b713-d32bba55d77f|pass|200806|srezic@cpan.org|Net-Random|2.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200806201319
1749601|1749601-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806200926
1749788|1749788-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200806201454
1749808|1749808-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200806201455
1749834|1749834-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200806201500
1749849|1749849-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200806201502
1749881|1749881-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806201013
1750206|1750206-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806201137
1750249|1750249-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200806201605
1750719|1750719-b19f-3f77-b713-d32bba55d77f|na|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806201458
1751000|1751000-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.10.0 patch 34065|linux|2.4.26|200806202127
1752441|1752441-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200806210053
1752510|1752510-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806210004
1752784|1752784-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806210050
1753897|1753897-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806210326
1753972|1753972-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806210338
1754219|1754219-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806210029
1755355|1755355-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806210532
1755663|1755663-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806210653
1757045|1757045-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806211146
1757339|1757339-b19f-3f77-b713-d32bba55d77f|na|200806|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806211228
1757376|1757376-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806211233
1757418|1757418-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806211240
1758182|1758182-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Compare|1.19|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34075|openbsd|4.3|200806211908
1758185|1758185-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34075|openbsd|4.3|200806211908
1758189|1758189-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34075|openbsd|4.3|200806211910
1759401|1759401-b19f-3f77-b713-d32bba55d77f|na|200806|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806220030
1759456|1759456-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806220039
1759671|1759671-b19f-3f77-b713-d32bba55d77f|pass|200806|rjbs@cpan.org|Data-Compare|1.19|darwin-2level|5.10.0|darwin|9.1.0|200806220149
1759781|1759781-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Compare|1.19|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200806220346
1759782|1759782-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200806220347
1759787|1759787-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200806220348
1760117|1760117-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Compare|1.19|i86pc-solaris-thread-multi-64int|5.11.0 patch 34076|solaris|2.10|200806220459
1760119|1760119-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i86pc-solaris-thread-multi-64int|5.11.0 patch 34076|solaris|2.10|200806220459
1760130|1760130-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i86pc-solaris-thread-multi-64int|5.11.0 patch 34076|solaris|2.10|200806220500
1761272|1761272-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806220437
1761507|1761507-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806220540
1761508|1761508-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806220540
1762012|1762012-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806220748
1762046|1762046-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34075|openbsd|4.3|200806221116
1762754|1762754-b19f-3f77-b713-d32bba55d77f|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806221051
1763690|1763690-b19f-3f77-b713-d32bba55d77f|pass|200806|andy@hexten.net|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.6.18-5-686|200806221802
1764682|1764682-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806221424
1765012|1765012-b19f-3f77-b713-d32bba55d77f|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806221441
1765096|1765096-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806221446
1765124|1765124-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806221448
1765127|1765127-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806221448
1765208|1765208-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806221453
1765564|1765564-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34074|linux|2.6.22-1-k7|200806222048
1765798|1765798-b19f-3f77-b713-d32bba55d77f|pass|200806|andy@hexten.net|Data-Transactional|1.02|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200806222033
1766324|1766324-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806221605
1766480|1766480-b19f-3f77-b713-d32bba55d77f|pass|200806|andy@hexten.net|Data-Transactional|1.02|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200806222113
1767765|1767765-b19f-3f77-b713-d32bba55d77f|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806221755
1767873|1767873-b19f-3f77-b713-d32bba55d77f|pass|200806|stro@cpan.org|Data-Hexdumper|1.4|i686-linux|5.10.0|linux|2.4.27-2-386|200806230109
1768010|1768010-b19f-3f77-b713-d32bba55d77f|pass|200806|stro@cpan.org|Data-Compare|1.19|i686-linux|5.10.0|linux|2.4.27-2-386|200806230128
1768606|1768606-b19f-3f77-b713-d32bba55d77f|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806221943
1768644|1768644-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806221946
1768673|1768673-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806221950
1769631|1769631-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.11.0 patch 34076|solaris|2.10|200806230341
1770114|1770114-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34074|linux|2.6.22-1-k7|200806230642
1770299|1770299-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34074|linux|2.6.22-1-k7|200806230724
1770586|1770586-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806230204
1770609|1770609-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806230207
1770610|1770610-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.02|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806230207
1770857|1770857-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806230233
1770858|1770858-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806230233
1770886|1770886-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34074|linux|2.6.22-1-k7|200806230728
1770959|1770959-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806230246
1771753|1771753-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806230450
1772060|1772060-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806230550
1772065|1772065-b19f-3f77-b713-d32bba55d77f|pass|200806|stro@cpan.org|Games-Dice-Advanced|1.1|i686-linux|5.8.8|linux|2.4.27-2-386|200806231141
1772583|1772583-b19f-3f77-b713-d32bba55d77f|pass|200806|srezic@cpan.org|Devel-CheckLib|0.5|i486-linux-gnu|5.8.8|linux|2.6.22.10|200806231259
1774401|1774401-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806231217
1775143|1775143-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806231356
1775988|1775988-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Compare|1.19|i86pc-solaris-thread-multi-64int|5.6.2|solaris|2.10|200806232121
1775991|1775991-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i86pc-solaris-thread-multi-64int|5.6.2|solaris|2.10|200806232121
1776015|1776015-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i86pc-solaris-thread-multi-64int|5.6.2|solaris|2.10|200806232123
1776860|1776860-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806231816
1777468|1777468-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200806240003
1777470|1777470-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200806240005
1777653|1777653-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Compare|1.19|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200806240214
1777657|1777657-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200806240214
1777673|1777673-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Compare|1.19|i686-linux|5.11.0 patch 34080|linux|2.6.16.38|200806240116
1777675|1777675-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i686-linux|5.11.0 patch 34080|linux|2.6.16.38|200806240116
1777677|1777677-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200806240217
1777685|1777685-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i686-linux|5.11.0 patch 34080|linux|2.6.16.38|200806240118
1777743|1777743-b19f-3f77-b713-d32bba55d77f|fail|200806|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806240127
1778008|1778008-b19f-3f77-b713-d32bba55d77f|pass|200806|srezic@cpan.org|Games-Dice-Advanced|1.1|i386-freebsd|5.6.2|freebsd|6.1-release|200806240417
1778545|1778545-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806240331
1779184|1779184-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806240047
1779573|1779573-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806240137
1780615|1780615-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806240409
1780753|1780753-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806240429
1781189|1781189-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806240524
1782392|1782392-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806240811
1782638|1782638-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806240847
1782872|1782872-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806240926
1783244|1783244-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806241025
1783498|1783498-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806241059
1785150|1785150-b19f-3f77-b713-d32bba55d77f|fail|200806|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806241944
1785597|1785597-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806250244
1786349|1786349-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806250134
1786732|1786732-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806250710
1786739|1786739-b19f-3f77-b713-d32bba55d77f|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Net-Random|2.0|sun4-solaris|5.10.0|solaris|2.9|200806250908
1786870|1786870-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux|5.11.0 patch 34080|linux|2.6.16.38|200806250724
1787188|1787188-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34085|linux|2.6.22-1-k7|200806250740
1787558|1787558-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34085|linux|2.6.22-1-k7|200806250647
1787890|1787890-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806250520
1788301|1788301-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806250554
1788304|1788304-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806250554
1789082|1789082-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int-2level|5.10.0 patch 34065|darwin|8.10.0|200806251123
1789201|1789201-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806250741
1790185|1790185-b19f-3f77-b713-d32bba55d77f|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806250949
1790585|1790585-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.1|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200806251642
1790586|1790586-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.1|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200806251642
1790589|1790589-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.1|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200806251642
1790595|1790595-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.1|i686-linux-64int|5.8.8|linux|2.6.14|200806251642
1790599|1790599-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.1|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200806251643
1790852|1790852-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.1|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200806251642
1790854|1790854-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.1|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200806251643
1790858|1790858-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.1|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200806251643
1791331|1791331-b19f-3f77-b713-d32bba55d77f|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806251204
1791378|1791378-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806251207
1791409|1791409-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806251211
1793688|1793688-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34085|linux|2.6.22-1-k7|200806252241
1793705|1793705-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34085|linux|2.6.22-1-k7|200806252246
1793794|1793794-b19f-3f77-b713-d32bba55d77f|fail|200806|srezic@cpan.org|DBIx-Class-SingletonRows|0.1|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200806252323
1795338|1795338-b19f-3f77-b713-d32bba55d77f|pass|200806|danl@metrex.net|Data-Compare|1.19|i586-linux-thread-multi|5.10.0|linux|2.6.25|200806260023
1795508|1795508-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34088|linux|2.6.22-1-k7|200806260948
1795580|1795580-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806260400
1795706|1795706-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34088|linux|2.6.22-1-k7|200806261019
1796391|1796391-b19f-3f77-b713-d32bba55d77f|pass|200806|stro@cpan.org|Data-Compare|1.19|i386-freebsd-64int|5.8.8|freebsd|7.0-release|200806260455
1796581|1796581-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd|5.11.0 patch 34085|openbsd|4.1|200806260621
1796835|1796835-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806260800
1797399|1797399-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34088|linux|2.6.22-1-k7|200806261514
1797468|1797468-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34088|linux|2.6.22-1-k7|200806261456
1797741|1797741-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34088|linux|2.6.22-1-k7|200806261530
1798310|1798310-b19f-3f77-b713-d32bba55d77f|fail|200806|david@cantrell.org.uk|DBIx-Class-SingletonRows|0.1|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200806261650
1798878|1798878-b19f-3f77-b713-d32bba55d77f|fail|200806|david@cantrell.org.uk|DBIx-Class-SingletonRows|0.1|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200806261706
1799261|1799261-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.11|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200806261934
1799264|1799264-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.11|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200806261934
1799266|1799266-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.11|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200806261934
1799293|1799293-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34088|linux|2.6.22-1-k7|200806261937
1799302|1799302-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200806261938
1799396|1799396-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|DBIx-Class-SingletonRows|0.1|i686-linux|5.8.8|linux|2.4.27-3-686|200806261703
1799427|1799427-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.11|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200806261935
1799462|1799462-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200806261936
1799463|1799463-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200806261937
1799469|1799469-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.11|i686-linux-64int|5.8.8|linux|2.6.14|200806261936
1800691|1800691-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|DBIx-Class-SingletonRows|0.11|i686-linux|5.10.0|linux|2.6.24-16-generic|200806261801
1801186|1801186-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.24-16-generic|200806261850
1801665|1801665-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806270028
1802752|1802752-b19f-3f77-b713-d32bba55d77f|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|DBIx-Class-SingletonRows|0.1|sun4-solaris|5.10.0|solaris|2.9|200806270638
1802983|1802983-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux|5.10.0|linux|2.6.24-16-generic|200806270136
1802984|1802984-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.02|i686-linux|5.10.0|linux|2.6.24-16-generic|200806270136
1803070|1803070-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|darwin-2level|5.8.8|darwin|8.10.1|200806270744
1803075|1803075-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|darwin-2level|5.8.8|darwin|8.10.1|200806270744
1803651|1803651-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux|5.10.0|linux|2.6.24-16-generic|200806270316
1803652|1803652-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|1.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200806270316
1803824|1803824-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux|5.10.0|linux|2.6.24-16-generic|200806270355
1804043|1804043-b19f-3f77-b713-d32bba55d77f|pass|200806|srezic@cpan.org|DBIx-Class-SingletonRows|0.11|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200806270937
1804246|1804246-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806270522
1804400|1804400-b19f-3f77-b713-d32bba55d77f|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|DBIx-Class-SingletonRows|0.11|sun4-solaris|5.10.0|solaris|2.9|200806271202
1804487|1804487-b19f-3f77-b713-d32bba55d77f|fail|200806|david@cantrell.org.uk|DBIx-Class-SingletonRows|0.1|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200806271132
1804540|1804540-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|DBIx-Class-SingletonRows|0.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200806270546
1804665|1804665-b19f-3f77-b713-d32bba55d77f|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone-UK-DetailedLocations|1.3|sun4-solaris|5.10.0|solaris|2.9|200806271315
1804813|1804813-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806270753
1805948|1805948-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|DBIx-Class-SingletonRows|0.1|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200806271358
1806046|1806046-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux|5.10.0|linux|2.6.24-16-generic|200806271150
1806122|1806122-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|DBIx-Class-SingletonRows|0.11|i686-linux|5.8.8|linux|2.4.27-3-686|200806271432
1806936|1806936-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|DBIx-Class-SingletonRows|0.1|IP35-irix-64int|5.8.8|irix|6.5|200806272105
1806945|1806945-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux|5.10.0|linux|2.6.24-16-generic|200806271506
1807304|1807304-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806271548
1807966|1807966-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806271729
1808105|1808105-b19f-3f77-b713-d32bba55d77f|pass|200806|stro@cpan.org|Devel-CheckOS|1.43|armv5tejl-linux|5.10.0|linux|2.6.17.14|200806280102
1809253|1809253-b19f-3f77-b713-d32bba55d77f|fail|200806|icerider70@gmail.com ("ICERIDER")|DBIx-Class-SingletonRows|0.11|i386-linux|5.8.8|linux|2.6.18-53.el5|200806280206
1810869|1810869-b19f-3f77-b713-d32bba55d77f|fail|200806|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806280426
1810950|1810950-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806280434
1813714|1813714-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806281155
1813849|1813849-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200806281343
1814361|1814361-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806281300
1814366|1814366-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806281300
1814639|1814639-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34090|linux|2.6.22-1-k7|200806281931
1815011|1815011-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34090|linux|2.6.22-1-k7|200806282015
1815274|1815274-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34090|linux|2.6.22-1-k7|200806282048
1815862|1815862-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806281540
1815886|1815886-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34090|linux|2.6.22-1-k7|200806282010
1815954|1815954-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux|5.10.0|linux|2.6.24-16-generic|200806281547
1815995|1815995-b19f-3f77-b713-d32bba55d77f|pass|200806|david@cantrell.org.uk|DBIx-Class-SingletonRows|0.11|IP35-irix-64int|5.8.8|irix|6.5|200806282149
1817322|1817322-b19f-3f77-b713-d32bba55d77f|pass|200806|stro@cpan.org|Palm-SMS|0.02|i686-linux|5.8.8|linux|2.4.27-2-386|200806290122
1817742|1817742-b19f-3f77-b713-d32bba55d77f|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806281911
1817860|1817860-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34090|linux|2.6.22-1-k7|200806290032
1818177|1818177-b19f-3f77-b713-d32bba55d77f|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34090|linux|2.6.22-1-k7|200806290200
1818908|1818908-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200806290106
1820182|1820182-b19f-3f77-b713-d32bba55d77f|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806290329
1820329|1820329-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806290007
1820383|1820383-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806290014
1820882|1820882-b19f-3f77-b713-d32bba55d77f|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Games-Dice-Advanced|1.1|sun4-solaris|5.10.0|solaris|2.9|200806290715
1821606|1821606-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-ld|5.11.0 patch 34090|linux|2.6.21.5-smp|200806290644
1822116|1822116-b19f-3f77-b713-d32bba55d77f|pass|200806|andy@hexten.net|Data-Compare|1.19|darwin-thread-multi-2level|5.10.0|darwin|9.2.2|200806290951
1822309|1822309-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200806290530
1822949|1822949-b19f-3f77-b713-d32bba55d77f|fail|200806|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200806290755
1823789|1823789-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux|5.10.0|linux|2.6.24-16-generic|200806291145
1824422|1824422-b19f-3f77-b713-d32bba55d77f|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Sub-WrapPackages|1.2|sun4-solaris|5.10.0|solaris|2.9|200806291954
1824425|1824425-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux|5.10.0|linux|2.6.24-16-generic|200806291355
1824977|1824977-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200806291536
1826691|1826691-b19f-3f77-b713-d32bba55d77f|pass|200806|andy@hexten.net|Devel-CheckLib|0.5|darwin-thread-multi-2level|5.10.0|darwin|9.2.2|200806300051
1826968|1826968-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200806300042
1827379|1827379-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.10.0|linux|2.6.24-16-generic|200806300225
1828898|1828898-b19f-3f77-b713-d32bba55d77f|pass|200806|stro@cpan.org|XML-Tiny|1.11|armv5tejl-linux|5.10.0|linux|2.6.17.14|200806301547
1829462|1829462-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806301028
1830918|1830918-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806301443
1830919|1830919-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200806301443
1831419|1831419-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806301631
1831425|1831425-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux|5.10.0|linux|2.6.24-16-generic|200806301631
1831539|1831539-b19f-3f77-b713-d32bba55d77f|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806301648
1832513|1832513-b19f-3f77-b713-d32bba55d77f|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-64int|5.11.0 patch 34093|openbsd|4.2|200806302346
1833685|1833685-b19f-3f77-b713-d32bba55d77f|pass|200807|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200807010141
1833952|1833952-b19f-3f77-b713-d32bba55d77f|pass|200807|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200807010218
1834202|1834202-b19f-3f77-b713-d32bba55d77f|pass|200807|stro@cpan.org|Class-CanBeA|1.2|i386-freebsd-64int|5.8.8|freebsd|7.0-release|200807011305
1834458|1834458-b19f-3f77-b713-d32bba55d77f|pass|200807|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux|5.10.0|linux|2.6.24-16-generic|200807010357
1836068|1836068-b19f-3f77-b713-d32bba55d77f|pass|200807|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-thread-multi-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200807010526
1837205|1837205-b19f-3f77-b713-d32bba55d77f|fail|200807|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux|5.10.0|linux|2.6.24-16-generic|200807010935
1837251|1837251-b19f-3f77-b713-d32bba55d77f|pass|200807|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200807010950
1839694|1839694-b19f-3f77-b713-d32bba55d77f|pass|200807|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200807011613
1839907|1839907-b19f-3f77-b713-d32bba55d77f|pass|200807|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200807011646
1840333|1840333-b19f-3f77-b713-d32bba55d77f|pass|200807|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200807011737
1840334|1840334-b19f-3f77-b713-d32bba55d77f|pass|200807|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux|5.10.0|linux|2.6.24-16-generic|200807011737
1840914|1840914-b19f-3f77-b713-d32bba55d77f|pass|200807|stro@cpan.org|Devel-CheckLib|0.5|os2|5.10.0|os2|2.45|200807020331
1842188|1842188-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd|5.11.0 patch 34094|netbsd|4.0|200807021131
1842724|1842724-b19f-3f77-b713-d32bba55d77f|pass|200807|stro@cpan.org|XML-DoubleEncodedEntities|1.0|armv5tejl-linux|5.10.0|linux|2.6.17.14|200807021706
1842760|1842760-b19f-3f77-b713-d32bba55d77f|pass|200807|stro@cpan.org|XML-DoubleEncodedEntities|1.0|armv5tejl-linux|5.8.8|linux|2.6.17.14|200807021715
1842991|1842991-b19f-3f77-b713-d32bba55d77f|pass|200807|stro@cpan.org|Devel-CheckOS|1.43|os2|5.10.0|os2|2.45|200807021803
1843586|1843586-b19f-3f77-b713-d32bba55d77f|pass|200807|andy@hexten.net|Data-Compare|1.19|darwin-thread-multi-2level|5.10.0|darwin|9.2.2|200807022017
1844679|1844679-b19f-3f77-b713-d32bba55d77f|pass|200807|stro@cpan.org|CPAN-FindDependencies|2.0|i386-freebsd-64int|5.8.8|freebsd|7.0-release|200807030836
1845117|1845117-b19f-3f77-b713-d32bba55d77f|pass|200807|stro@cpan.org|CPU-Emulator-Memory|1.1001|i386-freebsd-64int|5.8.8|freebsd|7.0-release|200807031253
1845123|1845123-b19f-3f77-b713-d32bba55d77f|pass|200807|stro@cpan.org|CPU-Emulator-Z80|0.9|i386-freebsd-64int|5.8.8|freebsd|7.0-release|200807031258
1845746|1845746-b19f-3f77-b713-d32bba55d77f|pass|200807|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200807030249
1847050|1847050-b19f-3f77-b713-d32bba55d77f|pass|200807|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200807030911
1847135|1847135-b19f-3f77-b713-d32bba55d77f|pass|200807|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200807030919
1847182|1847182-b19f-3f77-b713-d32bba55d77f|unknown|200807|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200807031854
1847204|1847204-b19f-3f77-b713-d32bba55d77f|pass|200807|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|File-Overwrite|1.0|sun4-solaris|5.10.0|solaris|2.9|200807032050
1849403|1849403-b19f-3f77-b713-d32bba55d77f|unknown|200807|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux|5.10.0|linux|2.6.24-16-generic|200807032210
1849983|1849983-b19f-3f77-b713-d32bba55d77f|pass|200807|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200807031927
1850429|1850429-b19f-3f77-b713-d32bba55d77f|pass|200807|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-thread-multi-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200807040048
1851555|1851555-b19f-3f77-b713-d32bba55d77f|pass|200807|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Class-CanBeA|1.2|sun4-solaris|5.10.0|solaris|2.9|200807040710
1852008|1852008-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|darwin-2level|5.10.0|darwin|8.11.1|200807040854
1852380|1852380-b19f-3f77-b713-d32bba55d77f|pass|200807|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-thread-multi-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200807040421
1852837|1852837-b19f-3f77-b713-d32bba55d77f|pass|200807|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200807040634
1857100|1857100-b19f-3f77-b713-d32bba55d77f|pass|200807|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200807050239
1857594|1857594-b19f-3f77-b713-d32bba55d77f|pass|200807|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200807050403
1857595|1857595-b19f-3f77-b713-d32bba55d77f|pass|200807|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.02|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200807050403
1857796|1857796-b19f-3f77-b713-d32bba55d77f|pass|200807|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200807050437
1858026|1858026-b19f-3f77-b713-d32bba55d77f|pass|200807|srezic@cpan.org|Sub-WrapPackages|1.2|i386-freebsd|5.6.2|freebsd|6.1-release|200807051120
1858459|1858459-b19f-3f77-b713-d32bba55d77f|pass|200807|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200807050641
1858671|1858671-b19f-3f77-b713-d32bba55d77f|pass|200807|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200807050727
1860147|1860147-b19f-3f77-b713-d32bba55d77f|pass|200807|stro@cpan.org|Data-Transactional|1.0|i386-freebsd-64int|5.8.8|freebsd|7.0-release|200807051639
1864049|1864049-b19f-3f77-b713-d32bba55d77f|pass|200807|sanko@cpan.org|Devel-CheckOS|1.43|MSWin32-x86-multi-thread|5.11.0 patch 34101|MSWin32|5.1|200807051408
1865845|1865845-b19f-3f77-b713-d32bba55d77f|pass|200807|postmaster@TOMXP.spectra-direct.com|Data-Compare|1.19|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200807071710
1870236|1870236-b19f-3f77-b713-d32bba55d77f|pass|200807|mgardner@usatech.com|Data-Compare|1.19|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200807081438
1870237|1870237-b19f-3f77-b713-d32bba55d77f|pass|200807|sanko@cpan.org|Devel-CheckOS|1.43|MSWin32-x86-multi-thread|5.11.0 patch 34107|MSWin32|5.1|200807080816
1870763|1870763-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34112|linux|2.6.22-1-k7|200807090026
1871046|1871046-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34112|linux|2.6.22-1-k7|200807090005
1871077|1871077-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34112|linux|2.6.22-1-k7|200807090000
1871298|1871298-b19f-3f77-b713-d32bba55d77f|pass|200807|bob@randomness.org.uk ("bob walker")|Data-Compare|1.19|OpenBSD.i386-openbsd|5.8.8|openbsd|4.2|200807090112
1871307|1871307-b19f-3f77-b713-d32bba55d77f|pass|200807|bob@randomness.org.uk ("bob walker")|Data-Compare|1.19|OpenBSD.i386-openbsd|5.10.0|openbsd|4.2|200807090114
1871711|1871711-b19f-3f77-b713-d32bba55d77f|fail|200807|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Palm-SMS|0.02|sun4-solaris|5.10.0|solaris|2.9|200807090504
1871761|1871761-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34112|linux|2.6.22-1-k7|200807090518
1871953|1871953-b19f-3f77-b713-d32bba55d77f|pass|200807|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Acme-Scurvy-Whoreson-BilgeRat|1.1|sun4-solaris|5.10.0|solaris|2.9|200807090630
1874988|1874988-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34112|linux|2.6.22-1-k7|200807100101
1875310|1875310-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34112|linux|2.6.22-1-k7|200807100240
1877308|1877308-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|OpenBSD.i386-openbsd-64int|5.11.0 patch 34114|openbsd|4.1|200807101527
1879484|1879484-b19f-3f77-b713-d32bba55d77f|pass|200807|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200807110816
1879866|1879866-b19f-3f77-b713-d32bba55d77f|pass|200807|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Palm-Treo680MessagesDB|1.0|sun4-solaris|5.10.0|solaris|2.9|200807111104
1880013|1880013-b19f-3f77-b713-d32bba55d77f|pass|200807|imacat@mail.imacat.idv.tw|Palm-Treo680MessagesDB|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200807111811
1880015|1880015-b19f-3f77-b713-d32bba55d77f|pass|200807|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|i386-freebsd|5.10.0|freebsd|6.1-release|200807111209
1880023|1880023-b19f-3f77-b713-d32bba55d77f|pass|200807|imacat@mail.imacat.idv.tw|Palm-Treo680MessagesDB|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200807111815
1880186|1880186-b19f-3f77-b713-d32bba55d77f|pass|200807|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200807111238
1880257|1880257-b19f-3f77-b713-d32bba55d77f|pass|200807|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|i386-freebsd|5.6.2|freebsd|6.1-release|200807111333
1880431|1880431-b19f-3f77-b713-d32bba55d77f|pass|200807|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200807111434
1880469|1880469-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|OpenBSD.i386-openbsd-thread-multi|5.11.0 patch 34117|openbsd|4.3|200807111214
1880550|1880550-b19f-3f77-b713-d32bba55d77f|pass|200807|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200807111538
1880567|1880567-b19f-3f77-b713-d32bba55d77f|pass|200807|david@cantrell.org.uk|Palm-Treo680MessagesDB|1.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200807110851
1880653|1880653-b19f-3f77-b713-d32bba55d77f|pass|200807|david@cantrell.org.uk|Data-Hexdumper|1.4|IP35-irix-64int|5.8.8|irix|6.5|200807111613
1880661|1880661-b19f-3f77-b713-d32bba55d77f|pass|200807|david@cantrell.org.uk|Palm-Treo680MessagesDB|1.0|IP35-irix-64int|5.8.8|irix|6.5|200807111614
1880688|1880688-b19f-3f77-b713-d32bba55d77f|pass|200807|david@cantrell.org.uk|Palm-Treo680MessagesDB|1.0|i686-linux|5.8.8|linux|2.4.27-3-686|200807110844
1880690|1880690-b19f-3f77-b713-d32bba55d77f|pass|200807|david@cantrell.org.uk|Palm-Treo680MessagesDB|1.0|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200807110844
1880696|1880696-b19f-3f77-b713-d32bba55d77f|pass|200807|david@cantrell.org.uk|Palm-Treo680MessagesDB|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200807111521
1880888|1880888-b19f-3f77-b713-d32bba55d77f|pass|200807|david@cantrell.org.uk|Palm-Treo680MessagesDB|1.0|i686-linux|5.6.2|linux|2.4.27-3-686|200807110924
1880913|1880913-b19f-3f77-b713-d32bba55d77f|pass|200807|david@cantrell.org.uk|Palm-Treo680MessagesDB|1.0|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200807111604
1881263|1881263-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200807111611
1881373|1881373-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-freebsd|5.11.0 patch 34120|freebsd|6.2-release|200807111700
1881381|1881381-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-freebsd|5.11.0 patch 34120|freebsd|6.2-release|200807111706
1881526|1881526-b19f-3f77-b713-d32bba55d77f|pass|200807|david@cantrell.org.uk|Palm-Treo680MessagesDB|1.0|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200807111839
1881589|1881589-b19f-3f77-b713-d32bba55d77f|pass|200807|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Statistics-SerialCorrelation|1.1|sun4-solaris|5.10.0|solaris|2.9|200807112102
1881597|1881597-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200807112010
1881631|1881631-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200807112030
1881979|1881979-b19f-3f77-b713-d32bba55d77f|pass|200807|pjacklam@online.no|Devel-CheckOS|1.43|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.25(0.15642)|200807120016
1882976|1882976-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|i386-freebsd|5.11.0 patch 34120|freebsd|6.2-release|200807120643
1883227|1883227-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200807121146
1883249|1883249-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200807121153
1883279|1883279-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200807121206
1883513|1883513-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200807121315
1883722|1883722-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i86pc-solaris|5.11.0 patch 34131|solaris|2.10|200807121410
1883742|1883742-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i86pc-solaris|5.11.0 patch 34131|solaris|2.10|200807121416
1884017|1884017-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34134|linux|2.6.22-1-k7|200807121625
1884092|1884092-b19f-3f77-b713-d32bba55d77f|pass|200807|sanko@cpan.org|Data-Hexdumper|1.4|MSWin32-x86-multi-thread|5.11.0 patch 34116|MSWin32|5.1|200807120352
1884116|1884116-b19f-3f77-b713-d32bba55d77f|fail|200807|sanko@cpan.org|Palm-Treo680MessagesDB|1.0|MSWin32-x86-multi-thread|5.11.0 patch 34116|MSWin32|5.1|200807120358
1884331|1884331-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34134|linux|2.6.22-1-k7|200807121639
1884411|1884411-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200807121621
1884991|1884991-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34134|linux|2.6.22-1-k7|200807121957
1885642|1885642-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34134|linux|2.6.22-1-k7|200807122151
1886018|1886018-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|i86pc-solaris|5.11.0 patch 34131|solaris|2.10|200807122246
1886250|1886250-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34134|linux|2.6.22-1-k7|200807130030
1886715|1886715-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34134|linux|2.6.22-1-k7|200807130209
1887496|1887496-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200807130705
1887713|1887713-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i86pc-solaris-thread-multi-64int|5.6.2|solaris|2.10|200807131014
1887720|1887720-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i86pc-solaris-thread-multi-64int|5.6.2|solaris|2.10|200807131017
1888851|1888851-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i686-linux-thread-multi-ld|5.11.0 patch 34136|linux|2.6.21.5-smp|200807131751
1888875|1888875-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i686-linux-thread-multi-ld|5.11.0 patch 34136|linux|2.6.21.5-smp|200807131803
1889291|1889291-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200807132039
1889768|1889768-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200807140010
1889801|1889801-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200807140028
1889814|1889814-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200807140047
1889836|1889836-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200807140059
1889837|1889837-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200807140101
1889858|1889858-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200807140115
1890161|1890161-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200807140411
1891058|1891058-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|i686-linux-thread-multi-ld|5.11.0 patch 34136|linux|2.6.21.5-smp|200807141140
1891574|1891574-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200807141638
1891600|1891600-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200807141652
1891730|1891730-b19f-3f77-b713-d32bba55d77f|pass|200807|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Tie-Hash-Longest|1.1|sun4-solaris|5.10.0|solaris|2.9|200807141846
1891733|1891733-b19f-3f77-b713-d32bba55d77f|pass|200807|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Statistics-ChiSquare|0.5|sun4-solaris|5.10.0|solaris|2.9|200807141847
1891774|1891774-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-64int|5.11.0 patch 34134|netbsd|4.0|200807141758
1891813|1891813-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-netbsd-64int|5.11.0 patch 34134|netbsd|4.0|200807141811
1893246|1893246-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200807150513
1893480|1893480-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|OpenBSD.i386-openbsd-thread-multi|5.11.0 patch 34134|openbsd|4.2|200807150707
1893723|1893723-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200807150928
1893919|1893919-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200807151108
1894000|1894000-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-64int|5.11.0 patch 34134|netbsd|4.0|200807151145
1894908|1894908-b19f-3f77-b713-d32bba55d77f|pass|200807|srezic@cpan.org|Sub-WrapPackages|1.2|i386-freebsd|5.10.0|freebsd|6.1-release|200807151806
1897024|1897024-b19f-3f77-b713-d32bba55d77f|pass|200807|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|NestedMap|1.0|sun4-solaris|5.10.0|solaris|2.9|200807160638
1897245|1897245-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200807160506
1897251|1897251-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-freebsd-64int|5.11.0 patch 34144|freebsd|7.0-release|200807160507
1897276|1897276-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200807160516
1897283|1897283-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-freebsd-64int|5.11.0 patch 34144|freebsd|7.0-release|200807160517
1897439|1897439-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34144|linux|2.6.22-1-k7|200807160837
1897771|1897771-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200807160715
1897774|1897774-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-64int|5.11.0 patch 34144|freebsd|7.0-release|200807160715
1897831|1897831-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200807160906
1898169|1898169-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34144|linux|2.6.22-1-k7|200807161054
1899447|1899447-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|OpenBSD.i386-openbsd-thread-multi|5.11.0 patch 34134|openbsd|4.2|200807161430
1899642|1899642-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34144|linux|2.6.22-1-k7|200807161619
1900007|1900007-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34144|linux|2.6.22-1-k7|200807161741
1900014|1900014-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34144|linux|2.6.22-1-k7|200807161731
1901024|1901024-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|i386-freebsd-64int|5.11.0 patch 34144|freebsd|7.0-release|200807161945
1901200|1901200-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200807162036
1901212|1901212-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200807162227
1901431|1901431-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200807162323
1901474|1901474-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200807162328
1901477|1901477-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34144|linux|2.6.22-1-k7|200807170035
1901489|1901489-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200807162333
1901499|1901499-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200807162334
1901510|1901510-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200807162339
1901513|1901513-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200807162341
1901520|1901520-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200807162342
1901535|1901535-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200807162347
1901566|1901566-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200807162352
1901576|1901576-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200807162358
1902813|1902813-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i686-linux-thread-multi-64int|5.10.0|linux|2.4.26|200807170739
1902859|1902859-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i686-linux-thread-multi-64int|5.10.0|linux|2.4.26|200807170747
1903090|1903090-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int|5.10.0|linux|2.4.26|200807170853
1904723|1904723-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i686-linux-thread-multi-ld|5.11.0 patch 34150|linux|2.6.24.5-smp|200807171449
1904735|1904735-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i686-linux-thread-multi-ld|5.11.0 patch 34150|linux|2.6.24.5-smp|200807171453
1904820|1904820-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-64int|5.11.0 patch 34150|netbsd|4.0|200807171506
1904869|1904869-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-netbsd-64int|5.11.0 patch 34150|netbsd|4.0|200807171523
1904936|1904936-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-ld|5.11.0 patch 34150|linux|2.6.24.5-smp|200807171529
1905990|1905990-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200807180620
1907801|1907801-b19f-3f77-b713-d32bba55d77f|unknown|200807|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Acme-Licence|1.0|sun4-solaris|5.10.0|solaris|2.9|200807181658
1909274|1909274-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200807182328
1909836|1909836-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-64int|5.11.0 patch 34151|freebsd|7.0-release|200807190141
1909974|1909974-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200807190317
1910043|1910043-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34151|openbsd|4.3|200807190545
1911641|1911641-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34151|linux|2.6.22-1-k7|200807191421
1912343|1912343-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34151|linux|2.6.22-1-k7|200807191741
1914094|1914094-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34151|linux|2.6.22-1-k7|200807192239
1915717|1915717-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34151|linux|2.6.22-1-k7|200807200325
1916521|1916521-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34151|linux|2.6.22-1-k7|200807200940
1916732|1916732-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34151|linux|2.6.22-1-k7|200807201023
1916950|1916950-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.11.0 patch 34151|solaris|2.10|200807201014
1917372|1917372-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200807201134
1917468|1917468-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200807201154
1918191|1918191-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200807201617
1918224|1918224-b19f-3f77-b713-d32bba55d77f|unknown|200807|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Acme-Pony|1.1.2|sun4-solaris|5.10.0|solaris|2.9|200807201729
1918463|1918463-b19f-3f77-b713-d32bba55d77f|pass|200807|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Tie-Hash-Rank|1.0.1|sun4-solaris|5.10.0|solaris|2.9|200807201834
1918665|1918665-b19f-3f77-b713-d32bba55d77f|pass|200807|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Tie-Scalar-Decay|1.1.1|sun4-solaris|5.10.0|solaris|2.9|200807201949
1918745|1918745-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.6.2|solaris|2.10|200807201910
1918958|1918958-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Acme-Licence|1.0|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807202018
1918963|1918963-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Acme-Pony|1.1.2|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807202019
1918968|1918968-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Acme-Scurvy-Whoreson-BilgeRat|1.1|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807202020
1919527|1919527-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210026
1919530|1919530-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210028
1919544|1919544-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210034
1919550|1919550-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Class-CanBeA|1.2|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210037
1919562|1919562-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210043
1919563|1919563-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210044
1919567|1919567-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Data-Transactional|1.02|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210047
1919574|1919574-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckLib|0.5|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210049
1919576|1919576-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210051
1919615|1919615-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|File-Find-Rule-Permissions|1.3|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210107
1919619|1919619-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|File-Overwrite|1.0|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210109
1919621|1919621-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Games-Dice-Advanced|1.1|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210110
1919622|1919622-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|NestedMap|1.0|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210111
1919632|1919632-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Net-Random|2.0|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210118
1919642|1919642-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Number-Phone|1.6|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210120
1919662|1919662-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Number-Phone-UK-DetailedLocations|1.3|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210132
1919679|1919679-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210141
1919693|1919693-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1.1|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210154
1919695|1919695-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Statistics-ChiSquare|0.5|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210155
1919696|1919696-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Statistics-SerialCorrelation|1.1|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210155
1919699|1919699-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sub-WrapPackages|1.2|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210156
1919701|1919701-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Tie-Hash-Longest|1.1|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210157
1919703|1919703-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Tie-Hash-Rank|1.0.1|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210158
1919707|1919707-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Tie-STDOUT|1.03|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210159
1919712|1919712-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Tie-Scalar-Decay|1.1.1|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210200
1919717|1919717-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210202
1919719|1919719-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210201
1920793|1920793-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200807210841
1920887|1920887-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.11.0 patch 34151|solaris|2.10|200807210931
1921536|1921536-b19f-3f77-b713-d32bba55d77f|pass|200807|andy@hexten.net|Data-Compare|1.19|i86pc-solaris-thread-multi|5.10.0|solaris|2.11|200807211405
1922183|1922183-b19f-3f77-b713-d32bba55d77f|fail|200807|stro@cpan.org|Bryar|3.0|armv5tejl-linux|5.10.0|linux|2.6.17.14|200807211946
1924105|1924105-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-ld|5.11.0 patch 34151|linux|2.6.16.38|200807220155
1924198|1924198-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200807220244
1924342|1924342-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200807220346
1926834|1926834-b19f-3f77-b713-d32bba55d77f|pass|200807|stro@cpan.org|Data-Hexdumper|1.4|armv5tejl-linux|5.10.0|linux|2.6.17.14|200807221251
1926928|1926928-b19f-3f77-b713-d32bba55d77f|pass|200807|stro@cpan.org|Data-Compare|1.19|armv5tejl-linux|5.10.0|linux|2.6.17.14|200807221609
1927030|1927030-b19f-3f77-b713-d32bba55d77f|pass|200807|stro@cpan.org|Data-Hexdumper|1.4|armv5tejl-linux|5.8.8|linux|2.6.17.14|200807221523
1927087|1927087-b19f-3f77-b713-d32bba55d77f|pass|200807|stro@cpan.org|Data-Compare|1.19|armv5tejl-linux|5.8.8|linux|2.6.17.14|200807221639
1927327|1927327-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34153|linux|2.6.22-1-k7|200807222103
1927555|1927555-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34153|linux|2.6.22-1-k7|200807222147
1929231|1929231-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200807230207
1929554|1929554-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34153|linux|2.6.22-1-k7|200807230416
1929713|1929713-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34153|linux|2.6.22-1-k7|200807230450
1930126|1930126-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.11.0 patch 34153|netbsd|4.0|200807230525
1930619|1930619-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200807230704
1930745|1930745-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-64int|5.11.0 patch 34153|openbsd|4.1|200807230724
1930792|1930792-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200807230733
1931384|1931384-b19f-3f77-b713-d32bba55d77f|pass|200807|stro@cpan.org|Tie-Hash-Longest|1.1|i686-linux|5.8.8|linux|2.4.27-2-386|200807230824
1932393|1932393-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34153|linux|2.6.22-1-k7|200807231418
1932727|1932727-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34153|linux|2.6.22-1-k7|200807231536
1938533|1938533-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200807242216
1938537|1938537-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200807242218
1938542|1938542-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200807242218
1938547|1938547-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.8.8|linux|2.6.14|200807242219
1939381|1939381-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-freebsd-thread-multi|5.11.0 patch 34156|freebsd|6.2-release|200807242148
1939657|1939657-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200807250026
1939754|1939754-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200807250038
1939827|1939827-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200807250047
1939856|1939856-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200807250056
1939913|1939913-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200807250107
1939928|1939928-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200807250109
1939959|1939959-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200807250115
1939961|1939961-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200807250116
1940011|1940011-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200807250126
1940530|1940530-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200807250139
1940570|1940570-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200807250150
1940772|1940772-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200807250514
1940799|1940799-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-freebsd-64int|5.11.0 patch 34156|freebsd|7.0-release|200807250125
1941869|1941869-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200807251244
1941928|1941928-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200807251315
1942156|1942156-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.11.0 patch 34156|linux|2.6.21.5-smp|200807251421
1943494|1943494-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|0.001_01|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200807252350
1943498|1943498-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|0.001_01|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200807260004
1943503|1943503-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|0.001_01|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200807260004
1943506|1943506-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|0.001_01|i686-linux-64int|5.8.8|linux|2.6.14|200807260005
1943508|1943508-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|0.001_01|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200807260005
1943509|1943509-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|0.001_01|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200807260006
1943511|1943511-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|0.001_01|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200807260006
1943532|1943532-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|0.001_01|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200807260004
1943533|1943533-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|0.001_01|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200807260003
1943639|1943639-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34156|linux|2.6.22-1-k7|200807260040
1944678|1944678-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34156|linux|2.6.22-1-k7|200807260447
1944791|1944791-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34156|linux|2.6.22-1-k7|200807260424
1945089|1945089-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.11.0 patch 34156|linux|2.6.21.5-smp|200807260451
1945112|1945112-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200807260601
1945113|1945113-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200807260602
1945116|1945116-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200807260603
1945635|1945635-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.11.0 patch 34156|linux|2.6.21.5-smp|200807260730
1945982|1945982-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200807260851
1946153|1946153-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200807260936
1946430|1946430-b19f-3f77-b713-d32bba55d77f|pass|200807|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|IP35-irix-64int|5.8.8|irix|6.5|200807261226
1946495|1946495-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200807261047
1946586|1946586-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200807261115
1946745|1946745-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34156|linux|2.6.22-1-k7|200807261603
1946879|1946879-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34156|linux|2.6.22-1-k7|200807261633
1947584|1947584-b19f-3f77-b713-d32bba55d77f|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|0.001_01|i686-linux|5.6.2|linux|2.4.27-3-686|200807261910
1947597|1947597-b19f-3f77-b713-d32bba55d77f|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|0.001_01|i686-linux|5.8.8|linux|2.4.27-3-686|200807261915
1947656|1947656-b19f-3f77-b713-d32bba55d77f|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|0.001_01|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200807261930
1947672|1947672-b19f-3f77-b713-d32bba55d77f|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|0.001_01|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200807261935
1947857|1947857-b19f-3f77-b713-d32bba55d77f|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|0.001_01|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200807261455
1947861|1947861-b19f-3f77-b713-d32bba55d77f|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|0.001_01|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200807261956
1948026|1948026-b19f-3f77-b713-d32bba55d77f|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|0.001_01|IP35-irix-64int|5.8.8|irix|6.5|200807262244
1948071|1948071-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34156|linux|2.6.22-1-k7|200807262300
1948183|1948183-b19f-3f77-b713-d32bba55d77f|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|0.001_01|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200807262320
1949069|1949069-b19f-3f77-b713-d32bba55d77f|pass|200807|stro@cpan.org|Net-Random|2.0|i686-linux|5.10.0|linux|2.4.27-2-386|200807270954
1950303|1950303-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-dragonfly-thread-multi-64int|5.11.0 patch 34156|dragonfly|2.0.0-release|200807271731
1950420|1950420-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-dragonfly-thread-multi-64int|5.10.0|dragonfly|2.0.0-release|200807271749
1950511|1950511-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200807271712
1950562|1950562-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-dragonfly-thread-multi-64int|5.11.0 patch 34156|dragonfly|2.0.0-release|200807271826
1950606|1950606-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|OpenBSD.i386-openbsd-thread-multi|5.11.0 patch 34160|openbsd|4.2|200807271738
1950656|1950656-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-dragonfly-thread-multi-64int|5.10.0|dragonfly|2.0.0-release|200807271846
1950912|1950912-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200807271901
1950950|1950950-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200807271910
1950981|1950981-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi|5.11.0 patch 34160|openbsd|4.2|200807271922
1951060|1951060-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200807271959
1952306|1952306-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-dragonfly-thread-multi-64int|5.8.8|dragonfly|2.0.0-release|200807280602
1952445|1952445-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-dragonfly-thread-multi-64int|5.8.8|dragonfly|2.0.0-release|200807280711
1952552|1952552-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200807280659
1952593|1952593-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200807280707
1952613|1952613-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200807280712
1952631|1952631-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200807280716
1952647|1952647-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200807280721
1952670|1952670-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200807280726
1952689|1952689-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200807280730
1953802|1953802-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|OpenBSD.i386-openbsd-thread-multi|5.11.0 patch 34160|openbsd|4.2|200807281343
1953922|1953922-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200807281437
1953930|1953930-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200807281439
1954513|1954513-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200807281812
1954596|1954596-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200807281904
1954667|1954667-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-dragonfly-thread-multi-64int|5.11.0 patch 34156|dragonfly|2.0.0-release|200807282035
1955416|1955416-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200807282252
1955426|1955426-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200807282255
1955429|1955429-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|1.0|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200807282358
1955430|1955430-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|1.0|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200807282359
1955431|1955431-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|1.0|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200807282359
1955432|1955432-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|1.0|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200807282359
1955435|1955435-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|1.0|i686-linux-64int|5.8.8|linux|2.6.14|200807282359
1955436|1955436-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200807282257
1955437|1955437-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|1.0|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200807290000
1955439|1955439-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|1.0|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200807290000
1955441|1955441-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|1.0|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200807290000
1955445|1955445-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200807282259
1955448|1955448-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200807282301
1955452|1955452-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200807282303
1955457|1955457-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200807282305
1955474|1955474-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200807282307
1955483|1955483-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|1.0|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200807290007
1955880|1955880-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-dragonfly-thread-multi-64int|5.11.0 patch 34156|dragonfly|2.0.0-release|200807290246
1955968|1955968-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-dragonfly-thread-multi-64int|5.10.0|dragonfly|2.0.0-release|200807290338
1955990|1955990-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200807290252
1955999|1955999-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200807290256
1956268|1956268-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200807290547
1956270|1956270-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200807290547
1956271|1956271-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200807290548
1956531|1956531-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-dragonfly-thread-multi-64int|5.8.8|dragonfly|2.0.0-release|200807290813
1956557|1956557-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34160|linux|2.6.22-1-k7|200807290823
1956909|1956909-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34160|linux|2.6.22-1-k7|200807290951
1956961|1956961-b19f-3f77-b713-d32bba55d77f|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|1.0|i686-linux|5.8.8|linux|2.4.27-3-686|200807290901
1957223|1957223-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34160|linux|2.6.22-1-k7|200807291103
1957283|1957283-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-netbsd-thread-multi-64int|5.11.0 patch 34160|netbsd|4.0|200807291013
1957288|1957288-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-netbsd-thread-multi-64int|5.11.0 patch 34160|netbsd|4.0|200807291014
1957376|1957376-b19f-3f77-b713-d32bba55d77f|pass|200807|imacat@mail.imacat.idv.tw|Sort-MultipleFields|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200807291735
1957406|1957406-b19f-3f77-b713-d32bba55d77f|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|1.0|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200807291045
1957407|1957407-b19f-3f77-b713-d32bba55d77f|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|1.0|i686-linux|5.6.2|linux|2.4.27-3-686|200807291045
1957413|1957413-b19f-3f77-b713-d32bba55d77f|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200807291009
1957470|1957470-b19f-3f77-b713-d32bba55d77f|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|1.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200807290505
1957471|1957471-b19f-3f77-b713-d32bba55d77f|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|1.0|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200807291005
1957476|1957476-b19f-3f77-b713-d32bba55d77f|pass|200807|imacat@mail.imacat.idv.tw|Sort-MultipleFields|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200807291808
1957513|1957513-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.11.0 patch 34160|netbsd|4.0|200807291115
1957650|1957650-b19f-3f77-b713-d32bba55d77f|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|1.0|IP35-irix-64int|5.8.8|irix|6.5|200807291312
1957671|1957671-b19f-3f77-b713-d32bba55d77f|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|1.0|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200807291216
1958030|1958030-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-dragonfly-thread-multi-64int|5.10.0 patch 34161|dragonfly|2.0.0-release|200807291444
1958035|1958035-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-dragonfly-thread-multi-64int|5.10.0 patch 34161|dragonfly|2.0.0-release|200807291445
1958203|1958203-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-dragonfly-thread-multi-64int|5.10.0 patch 34161|dragonfly|2.0.0-release|200807291541
1958905|1958905-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-dragonfly-thread-multi-64int|5.8.8 patch 34096|dragonfly|2.0.0-release|200807291925
1958907|1958907-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-dragonfly-thread-multi-64int|5.8.8 patch 34096|dragonfly|2.0.0-release|200807291926
1958966|1958966-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-dragonfly-thread-multi-64int|5.8.8 patch 34096|dragonfly|2.0.0-release|200807291953
1959723|1959723-b19f-3f77-b713-d32bba55d77f|pass|200807|andy@hexten.net|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200807300117
1959756|1959756-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200807300048
1959797|1959797-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200807300109
1959916|1959916-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200807300151
1959917|1959917-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200807300151
1959918|1959918-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200807300152
1959919|1959919-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200807300152
1959921|1959921-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200807300153
1959922|1959922-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200807300153
1959923|1959923-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200807300153
1959924|1959924-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200807300154
1959925|1959925-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200807300154
1959926|1959926-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200807300154
1960114|1960114-b19f-3f77-b713-d32bba55d77f|pass|200807|jeff@zeroclue.com|Acme-Scurvy-Whoreson-BilgeRat|1.1|darwin-2level|5.10.0|darwin|9.2.0|200807300151
1960233|1960233-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34160|linux|2.6.22-1-k7|200807300451
1960358|1960358-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.11.0 patch 34160|netbsd|4.0|200807300435
1960665|1960665-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200807300600
1960702|1960702-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200807300611
1960894|1960894-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.11.0 patch 34160|netbsd|4.0|200807300708
1961785|1961785-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200807301059
1961787|1961787-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200807301059
1962019|1962019-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34160|linux|2.6.22-1-k7|200807301409
1962097|1962097-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-dragonfly-thread-multi-64int|5.8.8 patch 34096|dragonfly|2.0.0-release|200807301357
1962234|1962234-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-freebsd-64int|5.11.0 patch 34162|freebsd|7.0-release|200807301233
1962240|1962240-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-freebsd-64int|5.11.0 patch 34162|freebsd|7.0-release|200807301234
1962428|1962428-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-64int|5.11.0 patch 34162|freebsd|7.0-release|200807301314
1962664|1962664-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-dragonfly-thread-multi-64int|5.8.8 patch 34096|dragonfly|2.0.0-release|200807301544
1962973|1962973-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-dragonfly-thread-multi-64int|5.10.0 patch 34161|dragonfly|2.0.0-release|200807301640
1963162|1963162-b19f-3f77-b713-d32bba55d77f|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34160|linux|2.6.22-1-k7|200807301740
1963239|1963239-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-freebsd-64int|5.10.0 patch 34161|freebsd|7.0-release|200807301538
1963241|1963241-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-freebsd-64int|5.10.0 patch 34161|freebsd|7.0-release|200807301539
1963368|1963368-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-dragonfly-thread-multi-64int|5.10.0 patch 34161|dragonfly|2.0.0-release|200807301800
1963405|1963405-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-64int|5.10.0 patch 34161|freebsd|7.0-release|200807301617
1964343|1964343-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200807302232
1964347|1964347-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200807302232
1964707|1964707-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34164|openbsd|4.3|200807310048
1964708|1964708-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34164|openbsd|4.3|200807310049
1964821|1964821-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34164|openbsd|4.3|200807310133
1965204|1965204-b19f-3f77-b713-d32bba55d77f|pass|200807|stro@cpan.org|Sort-MultipleFields|1.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200807310506
1965814|1965814-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-freebsd-64int|5.11.0 patch 34162|freebsd|7.0-release|200807310624
1966158|1966158-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200807310826
1966586|1966586-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200807311051
1966756|1966756-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-freebsd-64int|5.10.0 patch 34161|freebsd|7.0-release|200807311155
1966845|1966845-b19f-3f77-b713-d32bba55d77f|pass|200807|stro@cpan.org|Number-Phone|1.6|i686-linux|5.10.0|linux|2.4.27-2-386|200807311319
1966860|1966860-b19f-3f77-b713-d32bba55d77f|pass|200807|stro@cpan.org|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.10.0|linux|2.4.27-2-386|200807311323
1967293|1967293-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34164|openbsd|4.3|200807311603
1967574|1967574-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34164|openbsd|4.3|200807311835
1967656|1967656-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200807311918
1967811|1967811-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.11.0 patch 34166|solaris|2.10|200807312059
1967829|1967829-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200807312046
1968131|1968131-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i86pc-solaris-thread-multi-64int|5.11.0 patch 34166|solaris|2.10|200807312348
1968132|1968132-b19f-3f77-b713-d32bba55d77f|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i86pc-solaris-thread-multi-64int|5.11.0 patch 34166|solaris|2.10|200807312348
1968342|1968342-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200808010101
1968348|1968348-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200808010102
1968695|1968695-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.10.0 patch 34161|linux|2.6.22-1-k7|200808010330
1968845|1968845-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200808010255
1968847|1968847-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200808010255
1968849|1968849-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200808010256
1968851|1968851-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200808010256
1968853|1968853-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200808010256
1968854|1968854-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200808010257
1968856|1968856-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200808010257
1968858|1968858-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200808010257
1968859|1968859-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200808010258
1968861|1968861-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200808010258
1969122|1969122-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.10.0 patch 34161|linux|2.6.22-1-k7|200808010531
1969492|1969492-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 34161|linux|2.6.22-1-k7|200808010701
1969982|1969982-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 34161|linux|2.6.22-1-k7|200808010854
1971018|1971018-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.11.0 patch 34169|solaris|2.10|200808011258
1971041|1971041-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i86pc-solaris-thread-multi-64int|5.6.2|solaris|2.10|200808011303
1971045|1971045-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i86pc-solaris-thread-multi-64int|5.6.2|solaris|2.10|200808011305
1971441|1971441-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200808011451
1971445|1971445-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200808011452
1971643|1971643-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|XML-Tiny|1.11|i86pc-solaris-thread-multi-64int|5.11.0 patch 34166|solaris|2.10|200808011530
1972120|1972120-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i86pc-solaris-thread-multi-64int|5.11.0 patch 34169|solaris|2.10|200808011756
1972121|1972121-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i86pc-solaris-thread-multi-64int|5.11.0 patch 34169|solaris|2.10|200808011757
1972217|1972217-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i86pc-solaris-thread-multi-64int|5.11.0 patch 34166|solaris|2.10|200808011840
1972481|1972481-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|XML-Tiny|1.11|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200808011953
1972895|1972895-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200808012154
1973809|1973809-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 34161|linux|2.6.22-1-k7|200808020555
1974358|1974358-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 34161|linux|2.6.22-1-k7|200808020821
1975026|1975026-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|XML-Tiny|1.11|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200808021053
1975119|1975119-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Sort-MultipleFields|1.0|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200808021219
1975137|1975137-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|XML-Tiny|1.11|i86pc-solaris-thread-multi-64int|5.11.0 patch 34169|solaris|2.10|200808021124
1975297|1975297-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-ld|5.11.0 patch 34169|linux|2.6.16.38|200808021204
1976192|1976192-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200808021702
1976242|1976242-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i686-linux-thread-multi-ld|5.11.0 patch 34169|linux|2.6.16.38|200808021716
1976243|1976243-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200808021717
1976693|1976693-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200808022003
1976706|1976706-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-64int|5.11.0 patch 34169|netbsd|4.0|200808022006
1976985|1976985-b19f-3f77-b713-d32bba55d77f|pass|200808|conrad@fringehead.org|XML-Tiny|1.11|darwin-2level|5.10.0|darwin|9.3.0|200808021334
1976994|1976994-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Sort-MultipleFields|1.0|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808022237
1977069|1977069-b19f-3f77-b713-d32bba55d77f|pass|200808|stro@cpan.org|Palm-SMS|0.02|i686-linux|5.10.0|linux|2.4.27-2-386|200808022054
1977074|1977074-b19f-3f77-b713-d32bba55d77f|pass|200808|stro@cpan.org|Palm-TreoPhoneCallDB|1.1|i686-linux|5.10.0|linux|2.4.27-2-386|200808022055
1977127|1977127-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Sort-MultipleFields|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200808022314
1977128|1977128-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-netbsd-64int|5.11.0 patch 34169|netbsd|4.0|200808022212
1977200|1977200-b19f-3f77-b713-d32bba55d77f|pass|200808|conrad@fringehead.org|XML-Tiny|1.11|i686-linux-thread-multi|5.8.8|linux|2.6.15.1|200808021437
1977216|1977216-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Sort-MultipleFields|1.0|i386-freebsd|5.10.0|freebsd|6.1-release|200808022336
1978598|1978598-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-64int|5.11.0 patch 34169|netbsd|4.0|200808030831
1978664|1978664-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200808030849
1978893|1978893-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-ld|5.11.0 patch 34169|linux|2.6.16.38|200808031003
1978994|1978994-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Sort-MultipleFields|1.0|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200808031131
1979426|1979426-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200808031308
1979701|1979701-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200808031444
1980396|1980396-b19f-3f77-b713-d32bba55d77f|pass|200808|dpavlin@rot13.org ((Dobrica Pavlinusic))|Data-Compare|1.19|i486-linux-gnu|5.10.0|linux|2.6.25.10|200808032133
1980946|1980946-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.11.0 patch 34169|freebsd|7.0-release|200808032245
1981115|1981115-b19f-3f77-b713-d32bba55d77f|na|200808|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|i386-freebsd|5.5.5|freebsd|6.1-release|200808040140
1981178|1981178-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200808040110
1981369|1981369-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200808040136
1981852|1981852-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34169|openbsd|4.1|200808040345
1981982|1981982-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34169|openbsd|4.1|200808040454
1981986|1981986-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-freebsd-thread-multi-64int|5.11.0 patch 34169|freebsd|7.0-release|200808040324
1982322|1982322-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Sort-MultipleFields|1.0|i386-freebsd|5.6.2|freebsd|6.1-release|200808040829
1985957|1985957-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200808050827
1985958|1985958-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200808050827
1986173|1986173-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.11.0 patch 34170|freebsd|6.2-release|200808050950
1986927|1986927-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-19-generic|200808051559
1987031|1987031-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.11.0 patch 34170|linux|2.6.24-19-generic|200808051640
1988273|1988273-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0 patch 34161|linux|2.6.24-19-generic|200808052200
1990402|1990402-b19f-3f77-b713-d32bba55d77f|pass|200808|yanick@cpan.org ("Yanick Champoux")|Data-Compare|1.19|i686-linux|5.10.0|linux|2.6.22-gentoo-r2enkidu|200808060050
1993045|1993045-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Tie-Hash-Rank|1.0.1|i386-freebsd|5.5.5|freebsd|6.1-release|200808061711
1993209|1993209-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Tie-Scalar-Decay|1.1.1|i386-freebsd|5.5.5|freebsd|6.1-release|200808061710
1994194|1994194-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Tie-Scalar-Decay|1.1.1|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808062152
1994372|1994372-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Tie-Hash-Rank|1.0.1|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808062154
1994469|1994469-b19f-3f77-b713-d32bba55d77f|unknown|200808|srezic@cpan.org|Acme-Pony|1.1.2|i386-freebsd|5.5.5|freebsd|6.1-release|200808062231
1994636|1994636-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.11.0 patch 34174|linux|2.6.21.5-smp|200808062134
1996505|1996505-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Tie-Scalar-Decay|1.1.1|i386-freebsd|5.6.2|freebsd|6.1-release|200808070918
1996526|1996526-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Tie-Hash-Rank|1.0.1|i386-freebsd|5.6.2|freebsd|6.1-release|200808070921
1996771|1996771-b19f-3f77-b713-d32bba55d77f|unknown|200808|srezic@cpan.org|Acme-Pony|1.1.2|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808071017
1997654|1997654-b19f-3f77-b713-d32bba55d77f|unknown|200808|srezic@cpan.org|Acme-Pony|1.1.2|i386-freebsd|5.6.2|freebsd|6.1-release|200808071253
1999527|1999527-b19f-3f77-b713-d32bba55d77f|unknown|200808|srezic@cpan.org|Acme-Licence|1.0|i386-freebsd|5.5.5|freebsd|6.1-release|200808071845
2000199|2000199-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-dragonfly-thread-multi-64int|5.11.0 patch 34175|dragonfly|2.0.0-release|200808072155
2000255|2000255-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Tie-Hash-Longest|1.1|i386-freebsd|5.5.5|freebsd|6.1-release|200808072227
2001810|2001810-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Statistics-ChiSquare|0.5|i386-freebsd|5.5.5|freebsd|6.1-release|200808080804
2001827|2001827-b19f-3f77-b713-d32bba55d77f|unknown|200808|srezic@cpan.org|Acme-Licence|1.0|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808080830
2002243|2002243-b19f-3f77-b713-d32bba55d77f|unknown|200808|srezic@cpan.org|Acme-Licence|1.0|i386-freebsd|5.6.2|freebsd|6.1-release|200808080953
2002433|2002433-b19f-3f77-b713-d32bba55d77f|pass|200808|vpit@cpan.org|Devel-CheckLib|0.5|sun4-solaris-thread-multi-64|5.10.0|solaris|2.7|200808080127
2002615|2002615-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|NestedMap|1.0|i386-freebsd|5.5.5|freebsd|6.1-release|200808081055
2002850|2002850-b19f-3f77-b713-d32bba55d77f|pass|200808|vpit@cpan.org|Devel-CheckOS|1.43|sun4-solaris-thread-multi-64|5.10.0|solaris|2.7|200808080125
2003523|2003523-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Tie-Hash-Longest|1.1|i386-freebsd|5.6.2|freebsd|6.1-release|200808081350
2003686|2003686-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Statistics-ChiSquare|0.5|i386-freebsd|5.6.2|freebsd|6.1-release|200808081425
2004412|2004412-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Tie-Hash-Longest|1.1|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808081652
2004545|2004545-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|NestedMap|1.0|i386-freebsd|5.6.2|freebsd|6.1-release|200808081722
2004701|2004701-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Statistics-ChiSquare|0.5|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808081737
2005947|2005947-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|NestedMap|1.0|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808082105
2006352|2006352-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Statistics-SerialCorrelation|1.1|i386-freebsd|5.5.5|freebsd|6.1-release|200808082202
2007141|2007141-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-dragonfly-thread-multi-64int|5.10.0 patch 34161|dragonfly|2.0.0-release|200808090233
2008766|2008766-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Statistics-SerialCorrelation|1.1|i386-freebsd|5.6.2|freebsd|6.1-release|200808091214
2009821|2009821-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Statistics-SerialCorrelation|1.1|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808091353
2010349|2010349-b19f-3f77-b713-d32bba55d77f|pass|200808|stro@cpan.org|Sort-MultipleFields|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200808091937
2011846|2011846-b19f-3f77-b713-d32bba55d77f|pass|200808|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckOS|1.43|i86pc-solaris|5.10.0|solaris|2.11|200808100230
2012066|2012066-b19f-3f77-b713-d32bba55d77f|pass|200808|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Sort-MultipleFields|1.0|i86pc-solaris|5.10.0|solaris|2.11|200808100338
2012237|2012237-b19f-3f77-b713-d32bba55d77f|pass|200808|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-Tiny|1.11|i86pc-solaris|5.10.0|solaris|2.11|200808100415
2012452|2012452-b19f-3f77-b713-d32bba55d77f|pass|200808|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-DoubleEncodedEntities|1.0|i86pc-solaris|5.10.0|solaris|2.11|200808100701
2014850|2014850-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.11.0 patch 34192|netbsd|4.0|200808110207
2016490|2016490-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34197|linux|2.6.22-1-k7|200808111356
2016902|2016902-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34197|linux|2.6.22-1-k7|200808111555
2017203|2017203-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34197|linux|2.6.22-1-k7|200808111703
2017416|2017416-b19f-3f77-b713-d32bba55d77f|pass|200808|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Hexdumper|1.4|i86pc-solaris|5.10.0|solaris|2.11|200808111825
2017451|2017451-b19f-3f77-b713-d32bba55d77f|pass|200808|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Compare|1.19|i86pc-solaris|5.10.0|solaris|2.11|200808111834
2018259|2018259-b19f-3f77-b713-d32bba55d77f|pass|200808|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|DBIx-Class-SingletonRows|0.11|i86pc-solaris|5.10.0|solaris|2.11|200808120046
2018882|2018882-b19f-3f77-b713-d32bba55d77f|pass|200808|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Transactional|1.02|i86pc-solaris|5.10.0|solaris|2.11|200808120433
2019240|2019240-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.11.0 patch 34201|freebsd|7.0-release|200808120403
2019367|2019367-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-ld|5.11.0 patch 34197|linux|2.6.24.5-smp|200808120651
2020015|2020015-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.10.0 patch 34161|freebsd|7.0-release|200808120710
2021100|2021100-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.24.3|200808120312
2021633|2021633-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34197|linux|2.6.22-1-k7|200808121726
2021647|2021647-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34197|linux|2.6.22-1-k7|200808121727
2023871|2023871-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34197|linux|2.6.22-1-k7|200808131503
2027286|2027286-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.11.0 patch 34203|solaris|2.10|200808141452
2028942|2028942-b19f-3f77-b713-d32bba55d77f|pass|200808|brooks-cpan-tester@deseret.com|Devel-CheckOS|1.43|x86_64-linux|5.8.8|linux|2.6.9-023stab044.4-smp|200808141358
2029012|2029012-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net ("Rick Myers")|Data-Hexdumper|1.4|i686-linux|5.8.8|linux|2.6.24.3|200808150140
2029368|2029368-b19f-3f77-b713-d32bba55d77f|pass|200808|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckOS|1.43|i86pc-solaris-64int|5.10.0|solaris|2.11|200808151003
2029752|2029752-b19f-3f77-b713-d32bba55d77f|pass|200808|stro@cpan.org|Statistics-ChiSquare|0.5|i686-linux|5.10.0|linux|2.4.27-2-386|200808151016
2029819|2029819-b19f-3f77-b713-d32bba55d77f|pass|200808|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Sort-MultipleFields|1.0|i86pc-solaris-64int|5.10.0|solaris|2.11|200808151231
2029928|2029928-b19f-3f77-b713-d32bba55d77f|pass|200808|stro@cpan.org|Statistics-SerialCorrelation|1.1|i686-linux|5.10.0|linux|2.4.27-2-386|200808151101
2029960|2029960-b19f-3f77-b713-d32bba55d77f|pass|200808|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-Tiny|1.11|i86pc-solaris-64int|5.10.0|solaris|2.11|200808151305
2030081|2030081-b19f-3f77-b713-d32bba55d77f|pass|200808|stro@cpan.org|Sub-WrapPackages|1.2|i686-linux|5.10.0|linux|2.4.27-2-386|200808151133
2030248|2030248-b19f-3f77-b713-d32bba55d77f|pass|200808|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-DoubleEncodedEntities|1.0|i86pc-solaris-64int|5.10.0|solaris|2.11|200808151443
2032178|2032178-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.43|i686-linux|5.8.8|linux|2.6.24.3|200808150705
2032940|2032940-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-freebsd|5.10.0|freebsd|6.1-release|200808160849
2035176|2035176-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Sub-WrapPackages|1.2|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808161754
2035227|2035227-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net ("Rick Myers")|Number-Phone|1.6|i686-linux|5.8.8|linux|2.6.24.3|200808161251
2035519|2035519-b19f-3f77-b713-d32bba55d77f|pass|200808|JOST@cpan.org ("Josts Smokehouse")|Data-Hexdumper|1.4|i86pc-solaris-64int|5.10.0|solaris|2.11|200808161324
2035534|2035534-b19f-3f77-b713-d32bba55d77f|pass|200808|JOST@cpan.org ("Josts Smokehouse")|Data-Compare|1.19|i86pc-solaris-64int|5.10.0|solaris|2.11|200808161341
2037202|2037202-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Class-CanBeA|1.2|i386-freebsd|5.10.0|freebsd|6.1-release|200808170827
2037266|2037266-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Sub-WrapPackages|1.2|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200808170900
2037620|2037620-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Sub-WrapPackages|1.2|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200808171001
2038600|2038600-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|File-Overwrite|1.0|i386-freebsd|5.10.0|freebsd|6.1-release|200808171540
2038755|2038755-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Palm-SMS|0.02|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808171541
2038973|2038973-b19f-3f77-b713-d32bba55d77f|pass|200808|rafl@debian.org|Devel-CheckLib|0.5|i686-linux-multi|5.10.0|linux|2.6.24-rc5|200808171641
2039107|2039107-b19f-3f77-b713-d32bba55d77f|pass|200808|rafl@debian.org|Devel-CheckOS|1.43|i686-linux-multi|5.10.0|linux|2.6.24-rc5|200808171644
2040114|2040114-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Palm-SMS|0.02|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200808172114
2040460|2040460-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808172321
2041167|2041167-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux|5.11.0 patch 34204|linux|2.6.16.38|200808180203
2041799|2041799-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Palm-SMS|0.02|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200808180838
2042345|2042345-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34204|linux|2.6.22-1-k7|200808181112
2042373|2042373-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34204|linux|2.6.22-1-k7|200808181115
2042962|2042962-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net ("Rick Myers")|Bryar|3.0|i686-linux|5.8.8|linux|2.6.24.3|200808180651
2043275|2043275-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.19|i686-linux|5.8.8|linux|2.6.24.3|200808180740
2043444|2043444-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34204|linux|2.6.22-1-k7|200808181402
2044098|2044098-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34204|linux|2.6.22-1-k7|200808181544
2044616|2044616-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34204|linux|2.6.22-1-k7|200808181713
2045471|2045471-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Number-Phone|1.6|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200808181711
2045989|2045989-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34204|linux|2.6.22-1-k7|200808181920
2046283|2046283-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0 patch 34161|openbsd|4.3|200808181934
2046651|2046651-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Palm-SMS|0.02|i386-freebsd|5.10.0|freebsd|6.1-release|200808181958
2047415|2047415-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Class-CanBeA|1.2|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808182212
2048317|2048317-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200808190533
2048368|2048368-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200808190548
2048381|2048381-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200808190550
2048399|2048399-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200808190553
2048411|2048411-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200808190557
2048450|2048450-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200808190614
2048486|2048486-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200808190632
2048568|2048568-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200808190645
2048588|2048588-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200808190658
2048980|2048980-b19f-3f77-b713-d32bba55d77f|pass|200808|JOST@cpan.org ("Josts Smokehouse")|Data-Transactional|1.02|i86pc-solaris-64int|5.10.0|solaris|2.11|200808182153
2049353|2049353-b19f-3f77-b713-d32bba55d77f|pass|200808|JOST@cpan.org ("Josts Smokehouse")|Number-Phone|1.6|i86pc-solaris-64int|5.10.0|solaris|2.11|200808190300
2049726|2049726-b19f-3f77-b713-d32bba55d77f|pass|200808|rafl@debian.org|Devel-CheckOS|1.43|x86_64-linux|5.10.0|linux|2.6.18-6-xen-amd64|200808182146
2051551|2051551-b19f-3f77-b713-d32bba55d77f|pass|200808|rafl@debian.org|Devel-CheckOS|1.43|x86_64-linux|5.10.0|linux|2.6.18-6-xen-amd64|200808191418
2051862|2051862-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net ("Rick Myers")|Number-Phone|1.6|i686-linux|5.6.2|linux|2.6.24.3|200808190925
2052618|2052618-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|File-Overwrite|1.0|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808191826
2053168|2053168-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Games-Dice-Advanced|1.1|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808191826
2053301|2053301-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|File-Overwrite|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200808191826
2053463|2053463-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Games-Dice-Advanced|1.1|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200808191826
2054987|2054987-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i86pc-solaris-thread-multi-64int|5.6.2|solaris|2.10|200808192316
2055631|2055631-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200808200539
2055923|2055923-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|File-Overwrite|1.0|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200808200749
2055983|2055983-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Games-Dice-Advanced|1.1|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200808200749
2057195|2057195-b19f-3f77-b713-d32bba55d77f|pass|200808|shlomif@cpan.org|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.26.2-desktop-2mnb|200808192108
2057250|2057250-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net ("Rick Myers")|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux|5.8.8|linux|2.6.24.3|200808200655
2057810|2057810-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.10.0 patch 34161|solaris|2.10|200808201411
2059653|2059653-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Class-CanBeA|1.2|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200808202236
2060673|2060673-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34208|linux|2.6.22-1-k7|200808210423
2060880|2060880-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34208|linux|2.6.22-1-k7|200808210513
2061033|2061033-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200808210107
2061156|2061156-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200808210137
2061484|2061484-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200808210739
2061621|2061621-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34208|linux|2.6.22-1-k7|200808210810
2063986|2063986-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34208|linux|2.6.22-1-k7|200808211357
2064102|2064102-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34208|linux|2.6.22-1-k7|200808211516
2064510|2064510-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.20|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200808211743
2064513|2064513-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.20|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200808211743
2064515|2064515-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.20|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200808211743
2064516|2064516-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.20|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200808211744
2064523|2064523-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.20|i686-linux-64int|5.8.8|linux|2.6.14|200808211744
2064530|2064530-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.20|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200808211746
2064533|2064533-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.20|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200808211746
2064554|2064554-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.10.0 patch 34161|netbsd|4.0|200808211645
2064567|2064567-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.20|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200808211745
2064936|2064936-b19f-3f77-b713-d32bba55d77f|pass|200808|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Data-Compare|1.20|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200808220323
2064961|2064961-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Data-Compare|1.20|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200808212029
2065204|2065204-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34208|linux|2.6.22-1-k7|200808212130
2065411|2065411-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.8 patch 34096|netbsd|4.0|200808212048
2066247|2066247-b19f-3f77-b713-d32bba55d77f|pass|200808|david@cantrell.org.uk|Data-Compare|1.20|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200808220134
2066292|2066292-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200808220251
2066294|2066294-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200808220252
2066295|2066295-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200808220253
2066296|2066296-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200808220253
2066298|2066298-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200808220254
2066300|2066300-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200808220255
2066302|2066302-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200808220255
2066304|2066304-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200808220256
2066306|2066306-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200808220257
2066308|2066308-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200808220257
2066367|2066367-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.11.0 patch 34212|linux|2.6.21.5-smp|200808220245
2067198|2067198-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.20|i686-linux-64int|5.11.0 patch 34208|linux|2.6.22-1-k7|200808220857
2067222|2067222-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200808220819
2067427|2067427-b19f-3f77-b713-d32bba55d77f|pass|200808|imacat@mail.imacat.idv.tw|Data-Compare|1.20|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200808221703
2067458|2067458-b19f-3f77-b713-d32bba55d77f|pass|200808|imacat@mail.imacat.idv.tw|Data-Compare|1.20|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200808221717
2068324|2068324-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.10.0 patch 34161|netbsd|4.0|200808221626
2068382|2068382-b19f-3f77-b713-d32bba55d77f|pass|200808|stro@cpan.org|Data-Compare|1.20|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200808221842
2068467|2068467-b19f-3f77-b713-d32bba55d77f|pass|200808|david@cantrell.org.uk|Data-Compare|1.20|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200808221701
2068507|2068507-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200808221719
2068520|2068520-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-64int|5.11.0 patch 34169|netbsd|4.0|200808221726
2068840|2068840-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200808221733
2068887|2068887-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200808221740
2068905|2068905-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.8.8 patch 34096|netbsd|4.0|200808221834
2068971|2068971-b19f-3f77-b713-d32bba55d77f|pass|200808|g.grigelionis@computer.org|Data-Compare|1.20|i586-linux-thread-multi|5.8.5|linux|2.6.8.1|200808222001
2068972|2068972-b19f-3f77-b713-d32bba55d77f|pass|200808|g.grigelionis@computer.org|Data-Compare|1.20|i586-linux-thread-multi|5.8.8|linux|2.6.16|200808222002
2069329|2069329-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Data-Compare|1.20|i386-freebsd|5.10.0|freebsd|6.1-release|200808222136
2069741|2069741-b19f-3f77-b713-d32bba55d77f|pass|200808|shlomif@cpan.org|Data-Compare|1.20|i686-linux|5.10.0|linux|2.6.26.2-desktop-2mnb|200808221504
2069771|2069771-b19f-3f77-b713-d32bba55d77f|pass|200808|artur+cpan@niif.spb.su (ARTO)|Data-Compare|1.20|i386-linux-thread-multi|5.10.0|linux|2.6.18-92.1.6.el5|200808222145
2070163|2070163-b19f-3f77-b713-d32bba55d77f|pass|200808|david@cantrell.org.uk|Data-Compare|1.20|i686-linux|5.6.2|linux|2.4.27-3-686|200808230033
2070440|2070440-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i686-linux-thread-multi-64int-ld|5.11.0 patch 34212|linux|2.6.21.5-smp|200808230235
2070919|2070919-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net ("Rick Myers")|XML-Tiny|1.11|i686-linux|5.10.0|linux|2.6.24.3|200808230317
2071133|2071133-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net ("Rick Myers")|Data-Hexdumper|1.4|i686-linux|5.10.0|linux|2.6.24.3|200808230555
2071338|2071338-b19f-3f77-b713-d32bba55d77f|pass|200808|claco@chrislaco.com|Data-Compare|1.20|darwin-thread-multi-2level|5.8.8|darwin|9.0|200808230018
2071425|2071425-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net ("Rick Myers")|XML-DoubleEncodedEntities|1.0|i686-linux|5.10.0|linux|2.6.24.3|200808230709
2071692|2071692-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200808231339
2072061|2072061-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200808231723
2072209|2072209-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34218|linux|2.6.22-1-k7|200808231936
2072470|2072470-b19f-3f77-b713-d32bba55d77f|pass|200808|rafl@debian.org|Data-Compare|1.20|i686-linux-multi|5.10.0|linux|2.6.24-rc5|200808232020
2072533|2072533-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200808231829
2073155|2073155-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.20|i686-linux-64int|5.11.0 patch 34218|linux|2.6.22-1-k7|200808232148
2073186|2073186-b19f-3f77-b713-d32bba55d77f|fail|200808|srezic@cpan.org|Data-Compare|1.20|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808232151
2073566|2073566-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|NestedMap|1.0|i386-freebsd|5.10.0|freebsd|6.1-release|200808232225
2073732|2073732-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Data-Compare|1.20|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200808232123
2074123|2074123-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Class-CanBeA|1.2|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200808232123
2074801|2074801-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-freebsd-thread-multi-64int|5.11.0 patch 34218|freebsd|7.0-release|200808232205
2074805|2074805-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.11.0 patch 34218|freebsd|7.0-release|200808232206
2075907|2075907-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.43|i686-linux|5.6.2|linux|2.6.24.3|200808240047
2075993|2075993-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34218|linux|2.6.22-1-k7|200808240708
2076432|2076432-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34218|linux|2.6.22-1-k7|200808240834
2076708|2076708-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34218|linux|2.6.22-1-k7|200808240918
2076997|2076997-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34218|linux|2.6.22-1-k7|200808240951
2077246|2077246-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Data-Compare|1.20|i386-freebsd|5.6.2|freebsd|6.1-release|200808241010
2078672|2078672-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|OpenBSD.i386-openbsd-64int|5.11.0 patch 34218|openbsd|4.2|200808241519
2078905|2078905-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200808241606
2079549|2079549-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-19-generic|200808242014
2080095|2080095-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i686-linux-thread-multi-ld|5.11.0 patch 34225|linux|2.6.24-19-generic|200808242229
2081096|2081096-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Data-Compare|1.20|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200808250450
2081180|2081180-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.24.3|200808250311
2082190|2082190-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i686-linux-64int-ld|5.10.0 patch 34161|linux|2.6.24-19-generic|200808251225
2082734|2082734-b19f-3f77-b713-d32bba55d77f|pass|200808|hohokus@gmail.com|Data-Compare|1.20|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200808250932
2082785|2082785-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200808251445
2082915|2082915-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200808251519
2083908|2083908-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.11.0 patch 34225|netbsd|4.0|200808251845
2084703|2084703-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i686-linux-64int-ld|5.8.8 patch 34096|linux|2.6.24-19-generic|200808260127
2085617|2085617-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Statistics-SerialCorrelation|1.1|i386-freebsd|5.10.0|freebsd|6.1-release|200808260842
2085777|2085777-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Statistics-ChiSquare|0.5|i386-freebsd|5.10.0|freebsd|6.1-release|200808260842
2086330|2086330-b19f-3f77-b713-d32bba55d77f|pass|200808|stro@cpan.org|Data-Compare|1.20|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200808261335
2086870|2086870-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net ("Rick Myers")|XML-Tiny|1.11|i686-linux|5.8.8|linux|2.6.24.3|200808260813
2086986|2086986-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net ("Rick Myers")|XML-Tiny|1.11|i686-linux|5.6.2|linux|2.6.24.3|200808260836
2087422|2087422-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34226|linux|2.6.22-1-k7|200808261521
2087755|2087755-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34226|linux|2.6.22-1-k7|200808261456
2087975|2087975-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.20|i686-linux-64int|5.11.0 patch 34226|linux|2.6.22-1-k7|200808261643
2089096|2089096-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.20|i686-linux|5.10.0|linux|2.6.24.3|200808261322
2089169|2089169-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net ("Rick Myers")|XML-DoubleEncodedEntities|1.0|i686-linux|5.8.8|linux|2.6.24.3|200808261335
2089572|2089572-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34226|linux|2.6.22-1-k7|200808262033
2089756|2089756-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-dragonfly-thread-multi-64int|5.8.8|dragonfly|2.0.0-release|200808262039
2089787|2089787-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-dragonfly-thread-multi-64int|5.10.0|dragonfly|2.0.0-release|200808262045
2089835|2089835-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34226|linux|2.6.22-1-k7|200808262040
2090135|2090135-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34226|linux|2.6.22-1-k7|200808262145
2090474|2090474-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.200_500|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200808262312
2090475|2090475-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.200_500|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200808262313
2090477|2090477-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.200_500|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200808262313
2090479|2090479-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.200_500|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200808262314
2090489|2090489-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.200_500|i686-linux-64int|5.8.8|linux|2.6.14|200808262315
2090503|2090503-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.200_500|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200808262318
2090706|2090706-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.200_500|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200808262316
2090707|2090707-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.200_500|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200808262317
2090725|2090725-b19f-3f77-b713-d32bba55d77f|pass|200808|david@cantrell.org.uk|Data-Compare|1.20|i686-linux|5.8.8|linux|2.4.27-3-686|200808262257
2090771|2090771-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Data-Compare|1.200_500|i386-freebsd|5.10.0|freebsd|6.1-release|200808270000
2092079|2092079-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200808270619
2092080|2092080-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200808270619
2092082|2092082-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200808270620
2092083|2092083-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200808270621
2092085|2092085-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200808270621
2092086|2092086-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200808270622
2092089|2092089-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200808270623
2092092|2092092-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200808270624
2092094|2092094-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200808270625
2092097|2092097-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200808270625
2092290|2092290-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.11.0 patch 34225|netbsd|4.0|200808270722
2093120|2093120-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200808271221
2093398|2093398-b19f-3f77-b713-d32bba55d77f|pass|200808|david@cantrell.org.uk|Data-Compare|1.20|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200808270738
2093507|2093507-b19f-3f77-b713-d32bba55d77f|pass|200808|david@cantrell.org.uk|Data-Compare|1.20|IP35-irix-64int|5.8.8|irix|6.5|200808271500
2093624|2093624-b19f-3f77-b713-d32bba55d77f|pass|200808|david@cantrell.org.uk|Data-Compare|1.20|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200808271232
2094114|2094114-b19f-3f77-b713-d32bba55d77f|pass|200808|cpan@herlpacker.co.uk|Data-Compare|1.200_500|i486-linux-gnu|5.8.8|linux|2.6.15.7|200808271419
2094165|2094165-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-dragonfly-thread-multi-64int|5.10.0 patch 34161|dragonfly|2.0.0-release|200808271658
2094167|2094167-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-dragonfly-thread-multi-64int|5.10.0 patch 34161|dragonfly|2.0.0-release|200808271700
2094507|2094507-b19f-3f77-b713-d32bba55d77f|fail|200808|rick@jrmyers.net ("Rick Myers")|Bryar|3.0|i686-linux|5.10.0|linux|2.6.24.3|200808270846
2094847|2094847-b19f-3f77-b713-d32bba55d77f|pass|200808|stro@cpan.org|Devel-CheckLib|0.5|armv5tejl-linux|5.10.0|linux|2.6.17.14|200808272011
2094886|2094886-b19f-3f77-b713-d32bba55d77f|pass|200808|stro@cpan.org|Devel-CheckLib|0.5|armv5tejl-linux|5.8.8|linux|2.6.17.14|200808272014
2094976|2094976-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Data-Compare|1.200_500|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808271919
2094995|2094995-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-dragonfly-64int|5.11.0 patch 34233|dragonfly|2.0.0-release|200808271901
2095008|2095008-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-dragonfly-64int|5.11.0 patch 34233|dragonfly|2.0.0-release|200808271902
2095144|2095144-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net ("Rick Myers")|CPAN-FindDependencies|2.0|i686-linux|5.8.8|linux|2.6.24.3|200808271337
2095165|2095165-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net ("Rick Myers")|CPAN-FindDependencies|2.0|i686-linux|5.10.0|linux|2.6.24.3|200808271338
2095284|2095284-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.20|i686-linux|5.6.2|linux|2.6.24.3|200808271348
2095553|2095553-b19f-3f77-b713-d32bba55d77f|fail|200808|rick@jrmyers.net ("Rick Myers")|CPAN-FindDependencies|2.0|i686-linux|5.6.2|linux|2.6.24.3|200808271432
2095708|2095708-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Memory|1.1001|i686-linux|5.8.8|linux|2.6.24.3|200808271510
2095711|2095711-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Z80|1.0|i686-linux|5.8.8|linux|2.6.24.3|200808271512
2095822|2095822-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Memory|1.1001|i686-linux|5.10.0|linux|2.6.24.3|200808271549
2095833|2095833-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Z80|1.0|i686-linux|5.10.0|linux|2.6.24.3|200808271550
2096531|2096531-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-dragonfly-thread-multi-64int|5.8.8 patch 34096|dragonfly|2.0.0-release|200808280030
2096537|2096537-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-dragonfly-thread-multi-64int|5.8.8 patch 34096|dragonfly|2.0.0-release|200808280032
2096958|2096958-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-freebsd-thread-multi-64int|5.10.0 patch 34161|freebsd|7.0-release|200808280031
2096961|2096961-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-freebsd-thread-multi-64int|5.10.0 patch 34161|freebsd|7.0-release|200808280033
2097113|2097113-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-freebsd-thread-multi|5.11.0 patch 34233|freebsd|7.0-release|200808280141
2097115|2097115-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-freebsd-thread-multi|5.11.0 patch 34233|freebsd|7.0-release|200808280143
2097915|2097915-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-dragonfly-thread-multi-64int|5.10.0 patch 34161|dragonfly|2.0.0-release|200808280949
2098421|2098421-b19f-3f77-b713-d32bba55d77f|pass|200808|cpan@herlpacker.co.uk|Data-Compare|1.20|i486-linux-gnu|5.8.8|linux|2.6.15.7|200808280347
2098456|2098456-b19f-3f77-b713-d32bba55d77f|pass|200808|cpan@herlpacker.co.uk|Devel-CheckOS|1.43|i486-linux-gnu|5.8.8|linux|2.6.15.7|200808281035
2098735|2098735-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-dragonfly-64int|5.11.0 patch 34233|dragonfly|2.0.0-release|200808281333
2098779|2098779-b19f-3f77-b713-d32bba55d77f|pass|200808|stro@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux|5.10.0|linux|2.4.27-2-386|200808281020
2099111|2099111-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200808281117
2099969|2099969-b19f-3f77-b713-d32bba55d77f|pass|200808|stro@cpan.org|Data-Compare|1.20|i686-linux|5.10.0|linux|2.4.27-2-386|200808281411
2100251|2100251-b19f-3f77-b713-d32bba55d77f|pass|200808|david@cantrell.org.uk|Data-Compare|1.200_500|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200808281754
2100283|2100283-b19f-3f77-b713-d32bba55d77f|pass|200808|stro@cpan.org|Data-Compare|1.20|i686-linux|5.8.8|linux|2.4.27-2-386|200808281956
2100457|2100457-b19f-3f77-b713-d32bba55d77f|pass|200808|david@cantrell.org.uk|Data-Compare|1.200_500|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200808281721
2101020|2101020-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Tie-Hash-Longest|1.1|i386-freebsd|5.10.0|freebsd|6.1-release|200808282129
2101100|2101100-b19f-3f77-b713-d32bba55d77f|unknown|200808|srezic@cpan.org|Acme-Pony|1.1.2|i386-freebsd|5.10.0|freebsd|6.1-release|200808282130
2101199|2101199-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Data-Compare|1.200_500|i386-freebsd|5.6.2|freebsd|6.1-release|200808282135
2101531|2101531-b19f-3f77-b713-d32bba55d77f|unknown|200808|srezic@cpan.org|Acme-Licence|1.0|i386-freebsd|5.10.0|freebsd|6.1-release|200808282130
2102011|2102011-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-dragonfly-thread-multi-64int|5.8.8 patch 34096|dragonfly|2.0.0-release|200808282328
2102284|2102284-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.0|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200808290030
2102290|2102290-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.0|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200808290031
2102308|2102308-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.0|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200808290034
2102312|2102312-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.0|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200808290034
2102320|2102320-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.0|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200808290031
2102324|2102324-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.0|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200808290032
2102325|2102325-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.0|i686-linux-64int|5.8.8|linux|2.6.14|200808290033
2102326|2102326-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.0|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200808290033
2102373|2102373-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi|5.11.0 patch 34233|freebsd|7.0-release|200808282034
2102398|2102398-b19f-3f77-b713-d32bba55d77f|pass|200808|stro@cpan.org|Number-Phone|1.6|i686-linux|5.8.8|linux|2.4.27-2-386|200808290126
2102647|2102647-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.10.0 patch 34161|freebsd|7.0-release|200808290017
2102935|2102935-b19f-3f77-b713-d32bba55d77f|pass|200808|david@cantrell.org.uk|Data-Compare|1.20|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200808290055
2103035|2103035-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34234|linux|2.6.22-1-k7|200808290211
2103229|2103229-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34234|linux|2.6.22-1-k7|200808290227
2103434|2103434-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.20|i686-linux-64int|5.11.0 patch 34234|linux|2.6.22-1-k7|200808290312
2103568|2103568-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34234|linux|2.6.22-1-k7|200808290348
2104091|2104091-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34234|linux|2.6.22-1-k7|200808290554
2105206|2105206-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200808290814
2105209|2105209-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200808290815
2105219|2105219-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200808290816
2105234|2105234-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200808290816
2105247|2105247-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200808290817
2105252|2105252-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200808290818
2105258|2105258-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200808290819
2105265|2105265-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200808290820
2105938|2105938-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200808290951
2106681|2106681-b19f-3f77-b713-d32bba55d77f|pass|200808|srezic@cpan.org|Data-Compare|1.200_500|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200808291334
2106692|2106692-b19f-3f77-b713-d32bba55d77f|fail|200808|chris@bingosnet.co.uk|Class-DBI-ClassGenerator|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200808291239
2107403|2107403-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34234|linux|2.6.22-1-k7|200808291506
2108246|2108246-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200808291712
2108287|2108287-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200808291727
2108604|2108604-b19f-3f77-b713-d32bba55d77f|fail|200808|srezic@cpan.org|Class-DBI-ClassGenerator|1.0|i386-freebsd|5.6.2|freebsd|6.1-release|200808292011
2108719|2108719-b19f-3f77-b713-d32bba55d77f|fail|200808|srezic@cpan.org|Class-DBI-ClassGenerator|1.0|i386-freebsd|5.10.0|freebsd|6.1-release|200808292040
2108765|2108765-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i86pc-solaris-thread-multi-64int|5.11.0 patch 34234|solaris|2.10|200808291956
2108996|2108996-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200808292103
2109051|2109051-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0 patch 34161|openbsd|4.3|200808292112
2109463|2109463-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34234|openbsd|4.3|200808292339
2110749|2110749-b19f-3f77-b713-d32bba55d77f|fail|200808|srezic@cpan.org|Class-DBI-ClassGenerator|1.0|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200808300810
2110879|2110879-b19f-3f77-b713-d32bba55d77f|fail|200808|srezic@cpan.org|Class-DBI-ClassGenerator|1.0|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808300841
2110994|2110994-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net ("Rick Myers")|Games-Dice-Advanced|1.1|i686-linux|5.6.2|linux|2.6.24.3|200808300306
2111004|2111004-b19f-3f77-b713-d32bba55d77f|fail|200808|srezic@cpan.org|Class-DBI-ClassGenerator|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200808300904
2111005|2111005-b19f-3f77-b713-d32bba55d77f|fail|200808|srezic@cpan.org|Class-DBI-ClassGenerator|1.0|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200808300904
2112339|2112339-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200808301515
2112373|2112373-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200808301528
2112873|2112873-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i86pc-solaris-thread-multi-64int|5.11.0 patch 34234|solaris|2.10|200808301829
2113254|2113254-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0 patch 34161|openbsd|4.3|200808302103
2113274|2113274-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.11.0 patch 34234|solaris|2.10|200808302114
2113325|2113325-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200808302139
2113593|2113593-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0 patch 34161|openbsd|4.3|200808310000
2113928|2113928-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34234|openbsd|4.3|200808310238
2114152|2114152-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34234|openbsd|4.3|200808310515
2114170|2114170-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200808310534
2114229|2114229-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200808310606
2115011|2115011-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34237|linux|2.6.22-1-k7|200808311239
2115094|2115094-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200808311155
2115120|2115120-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200808311202
2116106|2116106-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200808311420
2116113|2116113-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200808311422
2116364|2116364-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200808311502
2116382|2116382-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200808311504
2116417|2116417-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.20|i686-linux-64int|5.11.0 patch 34237|linux|2.6.22-1-k7|200808311548
2116847|2116847-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34237|linux|2.6.22-1-k7|200808311627
2117292|2117292-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net ("Rick Myers")|Number-Phone|1.6|i686-linux|5.10.0|linux|2.6.24.3|200808311248
2117351|2117351-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34237|linux|2.6.22-1-k7|200808311854
2118245|2118245-b19f-3f77-b713-d32bba55d77f|pass|200808|rick@jrmyers.net ("Rick Myers")|Palm-TreoPhoneCallDB|1.1|i686-linux|5.10.0|linux|2.6.24.3|200808311437
2118585|2118585-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34237|linux|2.6.22-1-k7|200808312124
2118957|2118957-b19f-3f77-b713-d32bba55d77f|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34237|linux|2.6.22-1-k7|200808312218
2119384|2119384-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i686-linux-thread-multi-64int-ld|5.11.0 patch 34237|linux|2.6.16.38|200808312231
2119435|2119435-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i686-linux-thread-multi-64int-ld|5.11.0 patch 34237|linux|2.6.16.38|200808312239
2119735|2119735-b19f-3f77-b713-d32bba55d77f|pass|200808|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200808312337
2119793|2119793-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.11.0 patch 34237|linux|2.6.16.38|200809010033
2120156|2120156-b19f-3f77-b713-d32bba55d77f|pass|200809|stro@cpan.org|Class-DBI-ClassGenerator|1.0|i686-linux|5.10.0|linux|2.4.27-2-386|200809010537
2120442|2120442-b19f-3f77-b713-d32bba55d77f|pass|200809|stro@cpan.org|Data-Compare|1.200_500|i686-linux|5.10.0|linux|2.4.27-2-386|200809010647
2120514|2120514-b19f-3f77-b713-d32bba55d77f|pass|200809|stro@cpan.org|Class-DBI-ClassGenerator|1.0|i686-linux|5.8.8|linux|2.4.27-2-386|200809010655
2120517|2120517-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.0|i686-linux|5.10.0|linux|2.6.24.3|200809010029
2120778|2120778-b19f-3f77-b713-d32bba55d77f|pass|200809|stro@cpan.org|Data-Compare|1.200_500|i686-linux|5.8.8|linux|2.4.27-2-386|200809010855
2121277|2121277-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.200_500|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200809011108
2121300|2121300-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.20|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200809011114
2121647|2121647-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.200_500|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200809011256
2121692|2121692-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.20|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200809011302
2122006|2122006-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.200_500|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200809011452
2122014|2122014-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.20|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200809011454
2122158|2122158-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200809011552
2122165|2122165-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200809011555
2122170|2122170-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200809011558
2122414|2122414-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200809011833
2122416|2122416-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200809011833
2122420|2122420-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200809011833
2122421|2122421-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200809011834
2122424|2122424-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8|linux|2.6.14|200809011834
2122425|2122425-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200809011834
2122429|2122429-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200809011835
2122432|2122432-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200809011835
2122703|2122703-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.200_500|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200809011911
2122729|2122729-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.20|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200809011920
2122923|2122923-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|Data-Compare|1.21|i386-freebsd|5.10.0|freebsd|6.1-release|200809012154
2123235|2123235-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|Data-Compare|1.21|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200809012347
2123274|2123274-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|Data-Compare|1.21|i386-freebsd|5.6.2|freebsd|6.1-release|200809012355
2123275|2123275-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|Data-Compare|1.21|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200809012355
2123451|2123451-b19f-3f77-b713-d32bba55d77f|pass|200809|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Data-Compare|1.21|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200809020814
2123737|2123737-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200809020145
2123739|2123739-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200809020146
2123741|2123741-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200809020147
2123742|2123742-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200809020147
2123744|2123744-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200809020148
2123745|2123745-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200809020149
2123747|2123747-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200809020149
2123748|2123748-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200809020150
2123749|2123749-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200809020151
2123752|2123752-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200809020151
2124180|2124180-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200809020557
2124282|2124282-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200809020753
2124283|2124283-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200809020751
2124285|2124285-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200809020656
2124458|2124458-b19f-3f77-b713-d32bba55d77f|fail|200809|david@cantrell.org.uk|Class-DBI-ClassGenerator|1.0|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200809020816
2124573|2124573-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200809020845
2124579|2124579-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200809020846
2124587|2124587-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200809020846
2124592|2124592-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200809020847
2124706|2124706-b19f-3f77-b713-d32bba55d77f|fail|200809|david@cantrell.org.uk|Class-DBI-ClassGenerator|1.0|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200809020822
2124735|2124735-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200809020928
2124737|2124737-b19f-3f77-b713-d32bba55d77f|pass|200809|david@cantrell.org.uk|Data-Compare|1.21|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200809020828
2124773|2124773-b19f-3f77-b713-d32bba55d77f|pass|200809|imacat@mail.imacat.idv.tw|Data-Compare|1.21|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200809021637
2124774|2124774-b19f-3f77-b713-d32bba55d77f|pass|200809|imacat@mail.imacat.idv.tw|Data-Compare|1.21|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200809021637
2125023|2125023-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200809021104
2125137|2125137-b19f-3f77-b713-d32bba55d77f|pass|200809|rafl@debian.org|Data-Compare|1.21|i686-linux-multi|5.10.0|linux|2.6.24-rc5|200809021255
2125465|2125465-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.200_500|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34238|openbsd|4.1|200809021411
2125471|2125471-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34238|openbsd|4.1|200809021412
2125812|2125812-b19f-3f77-b713-d32bba55d77f|pass|200809|hohokus@gmail.com|Data-Compare|1.21|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200809020949
2126196|2126196-b19f-3f77-b713-d32bba55d77f|pass|200809|artur+cpan@niif.spb.su (ARTO)|Data-Compare|1.21|i386-linux-thread-multi|5.10.0|linux|2.6.18-92.1.6.el5|200809012146
2126563|2126563-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Palm-TreoPhoneCallDB|1.1|i686-linux|5.8.8|linux|2.6.24.3|200809021204
2126630|2126630-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.0|i686-linux|5.8.8|linux|2.6.24.3|200809021214
2126647|2126647-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200809021715
2126719|2126719-b19f-3f77-b713-d32bba55d77f|pass|200809|david@cantrell.org.uk|Data-Compare|1.200_500|i686-linux|5.6.2|linux|2.4.27-3-686|200809021742
2126957|2126957-b19f-3f77-b713-d32bba55d77f|pass|200809|stro@cpan.org|Data-Compare|1.21|i686-linux|5.10.0|linux|2.4.27-2-386|200809022101
2127042|2127042-b19f-3f77-b713-d32bba55d77f|pass|200809|david@cantrell.org.uk|Data-Compare|1.21|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200809021603
2127349|2127349-b19f-3f77-b713-d32bba55d77f|pass|200809|david@cantrell.org.uk|Data-Compare|1.21|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200809021425
2127399|2127399-b19f-3f77-b713-d32bba55d77f|pass|200809|g.grigelionis@computer.org|Data-Compare|1.21|i586-linux-thread-multi|5.8.8|linux|2.6.16|200809022139
2127698|2127698-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34243|linux|2.6.22-1-k7|200809022243
2127723|2127723-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34243|linux|2.6.22-1-k7|200809022254
2127739|2127739-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i86pc-solaris-thread-multi-64int|5.10.0 patch 34161|solaris|2.10|200809022153
2128059|2128059-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34243|linux|2.6.22-1-k7|200809030011
2128162|2128162-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.10.0 patch 34161|solaris|2.10|200809022328
2128314|2128314-b19f-3f77-b713-d32bba55d77f|pass|200809|kamensky@cpan.org ("Moshe Kamensky")|Data-Compare|1.21|sun4-solaris|5.10.0|solaris|2.8|200809021035
2128711|2128711-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34238|openbsd|4.1|200809030050
2128775|2128775-b19f-3f77-b713-d32bba55d77f|pass|200809|david@cantrell.org.uk|Data-Compare|1.21|i686-linux|5.8.8|linux|2.4.27-3-686|200809030101
2129258|2129258-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200809030255
2129836|2129836-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34243|linux|2.6.22-1-k7|200809030817
2130314|2130314-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34243|linux|2.6.22-1-k7|200809030928
2130745|2130745-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34243|linux|2.6.22-1-k7|200809031006
2131867|2131867-b19f-3f77-b713-d32bba55d77f|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Data-Compare|1.21|i86pc-solaris-64int|5.10.0|solaris|2.11|200809031242
2132073|2132073-b19f-3f77-b713-d32bba55d77f|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Class-DBI-ClassGenerator|1.0|i86pc-solaris-64int|5.10.0|solaris|2.11|200809031339
2132570|2132570-b19f-3f77-b713-d32bba55d77f|pass|200809|vpit@cpan.org|Data-Compare|1.200_500|x86_64-linux-thread-multi|5.10.0|linux|2.6.24.2-fuuka.profvince.com|200809031533
2132572|2132572-b19f-3f77-b713-d32bba55d77f|pass|200809|vpit@cpan.org|Data-Compare|1.21|x86_64-linux-thread-multi|5.10.0|linux|2.6.24.2-fuuka.profvince.com|200809031534
2132621|2132621-b19f-3f77-b713-d32bba55d77f|pass|200809|vpit@cpan.org|Devel-CheckOS|1.43|x86_64-linux-thread-multi|5.10.0|linux|2.6.24.2-fuuka.profvince.com|200809031543
2132704|2132704-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200809031335
2132760|2132760-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200809031347
2132788|2132788-b19f-3f77-b713-d32bba55d77f|pass|200809|david@cantrell.org.uk|Data-Compare|1.200_500|i686-linux|5.8.8|linux|2.4.27-3-686|200809031526
2132967|2132967-b19f-3f77-b713-d32bba55d77f|pass|200809|david@cantrell.org.uk|Data-Compare|1.200_500|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200809031015
2133120|2133120-b19f-3f77-b713-d32bba55d77f|fail|200809|david@cantrell.org.uk|Class-DBI-ClassGenerator|1.0|IP35-irix-64int|5.8.8|irix|6.5|200809031755
2133127|2133127-b19f-3f77-b713-d32bba55d77f|pass|200809|david@cantrell.org.uk|Data-Compare|1.200_500|IP35-irix-64int|5.8.8|irix|6.5|200809031756
2134058|2134058-b19f-3f77-b713-d32bba55d77f|fail|200809|srezic@cpan.org|Class-DBI-ClassGenerator|1.01|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200809032134
2134276|2134276-b19f-3f77-b713-d32bba55d77f|fail|200809|david@cantrell.org.uk|Class-DBI-ClassGenerator|1.0|i686-linux|5.8.8|linux|2.4.27-3-686|200809032122
2134304|2134304-b19f-3f77-b713-d32bba55d77f|pass|200809|david@cantrell.org.uk|Data-Compare|1.21|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200809032125
2134327|2134327-b19f-3f77-b713-d32bba55d77f|fail|200809|srezic@cpan.org|Class-DBI-ClassGenerator|1.01|i386-freebsd|5.10.0|freebsd|6.1-release|200809032232
2134426|2134426-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.01|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200809032244
2134428|2134428-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.01|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200809032244
2134429|2134429-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.01|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200809032245
2134431|2134431-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.01|i686-linux-64int|5.8.8|linux|2.6.14|200809032245
2134432|2134432-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.01|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200809032245
2134542|2134542-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.01|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200809032244
2134677|2134677-b19f-3f77-b713-d32bba55d77f|fail|200809|srezic@cpan.org|Class-DBI-ClassGenerator|1.01|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200809032345
2135029|2135029-b19f-3f77-b713-d32bba55d77f|pass|200809|vpit@cpan.org|XML-DoubleEncodedEntities|1.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.24.2-fuuka.profvince.com|200809040049
2135133|2135133-b19f-3f77-b713-d32bba55d77f|pass|200809|david@cantrell.org.uk|Data-Compare|1.21|i686-linux|5.6.2|linux|2.4.27-3-686|200809040028
2136726|2136726-b19f-3f77-b713-d32bba55d77f|fail|200809|david@cantrell.org.uk|Class-DBI-ClassGenerator|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200809040443
2136741|2136741-b19f-3f77-b713-d32bba55d77f|pass|200809|david@cantrell.org.uk|Data-Compare|1.200_500|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200809040445
2136884|2136884-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200809040528
2136914|2136914-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Class-DBI-ClassGenerator|1.01|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200809040912
2136974|2136974-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|Class-DBI-ClassGenerator|1.01|i386-freebsd|5.6.2|freebsd|6.1-release|200809041033
2137123|2137123-b19f-3f77-b713-d32bba55d77f|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Class-DBI-ClassGenerator|1.01|i86pc-solaris-64int|5.10.0|solaris|2.11|200809041109
2137189|2137189-b19f-3f77-b713-d32bba55d77f|pass|200809|claco@chrislaco.com|Data-Compare|1.21|darwin-thread-multi-2level|5.8.8|darwin|9.0|200809031956
2137306|2137306-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|Class-DBI-ClassGenerator|1.01|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200809041141
2137485|2137485-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200809040720
2137803|2137803-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200809041316
2138046|2138046-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i86pc-solaris-thread-multi-64int|5.10.0 patch 34161|solaris|2.10|200809041517
2138691|2138691-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.21|i686-linux|5.10.0|linux|2.6.24.3|200809041233
2138785|2138785-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200809041704
2139060|2139060-b19f-3f77-b713-d32bba55d77f|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckLib|0.5|i86pc-solaris-64int|5.10.0|solaris|2.11|200809042023
2140153|2140153-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 34266|linux|2.6.22-1-k7|200809042353
2140156|2140156-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 34266|linux|2.6.22-1-k7|200809042354
2140181|2140181-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.10.0 patch 34266|linux|2.6.22-1-k7|200809050030
2140954|2140954-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.21.5-smp|200809050032
2140978|2140978-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 34266|linux|2.6.22-1-k7|200809050207
2141182|2141182-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200809050112
2142272|2142272-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 34266|linux|2.6.22-1-k7|200809050642
2142646|2142646-b19f-3f77-b713-d32bba55d77f|pass|200809|JOST@cpan.org ("Josts Smokehouse")|CPAN-FindDependencies|2.0|i86pc-solaris-64int|5.10.0|solaris|2.11|200809050740
2143390|2143390-b19f-3f77-b713-d32bba55d77f|fail|200809|JOST@cpan.org ("Josts Smokehouse")|Bryar|3.0|i86pc-solaris-64int|5.10.0|solaris|2.11|200809050959
2143413|2143413-b19f-3f77-b713-d32bba55d77f|pass|200809|g.grigelionis@computer.org|Data-Compare|1.21|i586-linux-thread-multi|5.8.5|linux|2.6.8.1|200809051001
2143584|2143584-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 34266|linux|2.6.22-1-k7|200809050954
2144940|2144940-b19f-3f77-b713-d32bba55d77f|pass|200809|JOST@cpan.org ("Josts Smokehouse")|File-Find-Rule-Permissions|1.3|i86pc-solaris-64int|5.10.0|solaris|2.11|200809051418
2145002|2145002-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200809051336
2145208|2145208-b19f-3f77-b713-d32bba55d77f|pass|200809|jrockway@cpan.org|Data-Compare|1.21|x86_64-linux|5.10.0|linux|2.6.22-3-amd64|200809050656
2145547|2145547-b19f-3f77-b713-d32bba55d77f|pass|200809|david@cantrell.org.uk|Class-DBI-ClassGenerator|1.01|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200809051514
2145666|2145666-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Palm-SMS|0.02|i686-linux|5.10.0|linux|2.6.24.3|200809051039
2146109|2146109-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Transactional|1.02|i686-linux|5.8.8|linux|2.6.24.3|200809051232
2146110|2146110-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.21|i686-linux|5.8.8|linux|2.6.24.3|200809051232
2146406|2146406-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200809051828
2146859|2146859-b19f-3f77-b713-d32bba55d77f|pass|200809|andy@hexten.net|Data-Compare|1.21|darwin-thread-multi-2level|5.10.0|darwin|9.2.2|200809052042
2146903|2146903-b19f-3f77-b713-d32bba55d77f|pass|200809|andy@hexten.net|Data-Compare|1.21|darwin-thread-multi-2level|5.10.0|darwin|9.2.2|200809052044
2146936|2146936-b19f-3f77-b713-d32bba55d77f|pass|200809|shlomif@cpan.org|Data-Compare|1.21|i686-linux|5.10.0|linux|2.6.26.2-desktop-2mnb|200809052248
2146953|2146953-b19f-3f77-b713-d32bba55d77f|pass|200809|david@cantrell.org.uk|Data-Compare|1.21|IP35-irix-64int|5.8.8|irix|6.5|200809052153
2147039|2147039-b19f-3f77-b713-d32bba55d77f|pass|200809|andy@hexten.net|Data-Compare|1.21|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200809052103
2147815|2147815-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 34279|linux|2.6.22-1-k7|200809060038
2148192|2148192-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200809060001
2148403|2148403-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34279|linux|2.6.22-1-k7|200809060245
2148500|2148500-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34279|linux|2.6.22-1-k7|200809060246
2148806|2148806-b19f-3f77-b713-d32bba55d77f|pass|200809|rjbs@cpan.org|Data-Compare|1.21|darwin-2level|5.10.0|darwin|9.1.0|200809060200
2148848|2148848-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34279|linux|2.6.22-1-k7|200809060205
2148887|2148887-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.6.2|netbsd|4.0|200809060310
2149478|2149478-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34279|linux|2.6.22-1-k7|200809060620
2150047|2150047-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34279|linux|2.6.22-1-k7|200809060919
2150562|2150562-b19f-3f77-b713-d32bba55d77f|pass|200809|dillman@ria.buffalo.edu|Data-Compare|1.21|sun4-solaris|5.10.0|solaris|2.9|200809051516
2151414|2151414-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i686-linux-thread-multi-ld|5.11.0 patch 34282|linux|2.6.21.5-smp|200809061758
2151492|2151492-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Palm-Treo680MessagesDB|1.0|i686-linux|5.10.0|linux|2.6.24.3|200809061318
2152473|2152473-b19f-3f77-b713-d32bba55d77f|fail|200809|rick@jrmyers.net ("Rick Myers")|Net-Random|2.0|i686-linux|5.10.0|linux|2.6.24.3|200809061749
2152821|2152821-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34301|linux|2.6.22-1-k7|200809062350
2152856|2152856-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 34301|linux|2.6.22-1-k7|200809070133
2153132|2153132-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34301|linux|2.6.22-1-k7|200809070233
2154234|2154234-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34301|linux|2.6.22-1-k7|200809070700
2154322|2154322-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34301|linux|2.6.22-1-k7|200809070719
2154565|2154565-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34301|linux|2.6.22-1-k7|200809070831
2155585|2155585-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Rank|1.0.1|i686-linux|5.8.8|linux|2.6.24.3|200809070542
2155658|2155658-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Transactional|1.02|i686-linux|5.10.0|linux|2.6.24.3|200809070555
2156076|2156076-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-ld|5.11.0 patch 34282|linux|2.6.21.5-smp|200809071257
2156378|2156378-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Rank|1.0.1|i686-linux|5.10.0|linux|2.6.24.3|200809070934
2157510|2157510-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34304|linux|2.6.22-1-k7|200809071950
2158290|2158290-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 34304|linux|2.6.22-1-k7|200809072204
2158758|2158758-b19f-3f77-b713-d32bba55d77f|pass|200809|david@cantrell.org.uk|Class-DBI-ClassGenerator|1.01|IP35-irix-64int|5.8.8|irix|6.5|200809072310
2158801|2158801-b19f-3f77-b713-d32bba55d77f|pass|200809|david@cantrell.org.uk|Class-DBI-ClassGenerator|1.01|i686-linux|5.8.8|linux|2.4.27-3-686|200809072214
2159044|2159044-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200809072251
2159070|2159070-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200809072254
2159366|2159366-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Longest|1.1|i686-linux|5.10.0|linux|2.6.24.3|200809071842
2159738|2159738-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200809080100
2159772|2159772-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200809080116
2160285|2160285-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34304|linux|2.6.22-1-k7|200809080652
2160582|2160582-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34304|linux|2.6.22-1-k7|200809080745
2161534|2161534-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34304|linux|2.6.22-1-k7|200809081021
2161756|2161756-b19f-3f77-b713-d32bba55d77f|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Tie-STDOUT|1.03|i86pc-solaris-64int|5.10.0|solaris|2.11|200809081110
2162236|2162236-b19f-3f77-b713-d32bba55d77f|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Palm-TreoPhoneCallDB|1.1|i86pc-solaris-64int|5.10.0|solaris|2.11|200809081327
2162663|2162663-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|NestedMap|1.0|i686-linux|5.10.0|linux|2.6.24.3|200809080915
2164470|2164470-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.10.0 patch 34305|netbsd|4.0|200809081903
2164719|2164719-b19f-3f77-b713-d32bba55d77f|pass|200809|dwheeler@cpan.org|Data-Compare|1.21|darwin-2level|5.10.0|darwin|9.4.0|200809081854
2164829|2164829-b19f-3f77-b713-d32bba55d77f|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Net-Random|2.0|i86pc-solaris-64int|5.10.0|solaris|2.11|200809082106
2165191|2165191-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Longest|1.1|i686-linux|5.8.8|linux|2.6.24.3|200809081551
2165321|2165321-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|Data-Compare|1.21|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809082130
2166664|2166664-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34304|linux|2.6.22-1-k7|200809090041
2166850|2166850-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200809090011
2166888|2166888-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200809090017
2166963|2166963-b19f-3f77-b713-d32bba55d77f|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Number-Phone-UK-DetailedLocations|1.3|i86pc-solaris-64int|5.10.0|solaris|2.11|200809090135
2168102|2168102-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34322|linux|2.6.22-1-k7|200809090453
2168515|2168515-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.8 patch 34313|netbsd|4.0|200809090446
2168549|2168549-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34322|linux|2.6.22-1-k7|200809090530
2168917|2168917-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Palm-SMS|0.02|i686-linux|5.8.8|linux|2.6.24.3|200809090101
2169493|2169493-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34322|linux|2.6.22-1-k7|200809090845
2169887|2169887-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34322|linux|2.6.22-1-k7|200809091001
2170196|2170196-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-STDOUT|1.03|i686-linux|5.10.0|linux|2.6.24.3|200809090442
2170228|2170228-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Games-Dice-Advanced|1.1|i686-linux|5.10.0|linux|2.6.24.3|200809090445
2170535|2170535-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809091035
2170688|2170688-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34322|linux|2.6.22-1-k7|200809091146
2171000|2171000-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34322|linux|2.6.22-1-k7|200809091224
2171365|2171365-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|File-Overwrite|1.0|i686-linux|5.10.0|linux|2.6.24.3|200809090717
2172415|2172415-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Sub-WrapPackages|1.2|i686-linux|5.10.0|linux|2.6.24.3|200809090948
2173132|2173132-b19f-3f77-b713-d32bba55d77f|pass|200809|david@cantrell.org.uk|Class-DBI-ClassGenerator|1.01|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200809091118
2173799|2173799-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 34305|linux|2.6.22-1-k7|200809091905
2173893|2173893-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.10.0 patch 34305|linux|2.6.22-1-k7|200809091923
2174004|2174004-b19f-3f77-b713-d32bba55d77f|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Games-Dice-Advanced|1.1|i86pc-solaris-64int|5.10.0|solaris|2.11|200809091941
2174587|2174587-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 34305|linux|2.6.22-1-k7|200809092124
2174928|2174928-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 34305|linux|2.6.22-1-k7|200809092215
2175725|2175725-b19f-3f77-b713-d32bba55d77f|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Sub-WrapPackages|1.2|i86pc-solaris-64int|5.10.0|solaris|2.11|200809100019
2176322|2176322-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 34305|linux|2.6.22-1-k7|200809100330
2177303|2177303-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.10.0 patch 34305|netbsd|4.0|200809100536
2177586|2177586-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Palm-Treo680MessagesDB|1.0|i686-linux|5.8.8|linux|2.6.24.3|200809100233
2178345|2178345-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 34305|linux|2.6.22-1-k7|200809100908
2179057|2179057-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|Devel-CheckLib|0.5|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809100846
2179468|2179468-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|Devel-CheckOS|1.43|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809100846
2179568|2179568-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|CPU-Emulator-Z80|1.0|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809100845
2179569|2179569-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|CPU-Emulator-Memory|1.1001|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809100845
2179797|2179797-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200809100822
2179849|2179849-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|Sort-MultipleFields|1.0|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809100847
2179889|2179889-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|DBIx-Class-SingletonRows|0.11|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809100845
2179938|2179938-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|Data-Hexdumper|1.4|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809100845
2180325|2180325-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|XML-Tiny|1.11|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809100848
2180393|2180393-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|Data-Transactional|1.02|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809100845
2180813|2180813-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|Number-Phone|1.6|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809101259
2180933|2180933-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.10.0 patch 34325|freebsd|7.0-release|200809101058
2182471|2182471-b19f-3f77-b713-d32bba55d77f|pass|200809|JOST@cpan.org ("Josts Smokehouse")|File-Overwrite|1.0|i86pc-solaris-64int|5.10.0|solaris|2.11|200809101709
2182493|2182493-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.8 patch 34313|netbsd|4.0|200809101525
2182526|2182526-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.8 patch 34327|freebsd|7.0-release|200809101506
2183605|2183605-b19f-3f77-b713-d32bba55d77f|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Class-CanBeA|1.2|i86pc-solaris-64int|5.10.0|solaris|2.11|200809102018
2183883|2183883-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809101945
2184050|2184050-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Statistics-SerialCorrelation|1.1|i686-linux|5.10.0|linux|2.6.24.3|200809101520
2184500|2184500-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|Bryar|3.0|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809102241
2184805|2184805-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|CPAN-FindDependencies|2.0|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809102322
2185551|2185551-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i686-linux-64int-ld|5.10.0|linux|2.6.24.5-smp|200809110110
2186886|2186886-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34327|linux|2.6.22-1-k7|200809110758
2186951|2186951-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 34327|linux|2.6.22-1-k7|200809110813
2187759|2187759-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34327|linux|2.6.22-1-k7|200809110955
2188217|2188217-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809111052
2188457|2188457-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|Number-Phone-UK-DetailedLocations|1.3|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809111110
2188515|2188515-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|Net-Random|2.0|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809111117
2189291|2189291-b19f-3f77-b713-d32bba55d77f|fail|200809|JOST@cpan.org ("Josts Smokehouse")|Palm-SMS|0.02|i86pc-solaris-64int|5.10.0|solaris|2.11|200809111240
2189385|2189385-b19f-3f77-b713-d32bba55d77f|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Acme-Scurvy-Whoreson-BilgeRat|1.1|i86pc-solaris-64int|5.10.0|solaris|2.11|200809111252
2189538|2189538-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34327|linux|2.6.22-1-k7|200809111311
2189654|2189654-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34327|linux|2.6.22-1-k7|200809111325
2190278|2190278-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|File-Find-Rule-Permissions|1.3|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809111426
2191574|2191574-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34327|linux|2.6.22-1-k7|200809111659
2192929|2192929-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Net-Random|2.0|i686-linux|5.8.8|linux|2.6.24.3|200809111332
2193628|2193628-b19f-3f77-b713-d32bba55d77f|pass|200809|andy@hexten.net|Data-Compare|1.21|i86pc-solaris-thread-multi|5.10.0|solaris|2.11|200809120152
2194022|2194022-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.10.0 patch 34325|freebsd|7.0-release|200809111951
2194874|2194874-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.8.8 patch 34327|freebsd|7.0-release|200809112155
2195859|2195859-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809120405
2195893|2195893-b19f-3f77-b713-d32bba55d77f|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Statistics-SerialCorrelation|1.1|i86pc-solaris-64int|5.10.0|solaris|2.11|200809120514
2196189|2196189-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|NestedMap|1.0|i686-linux|5.8.8|linux|2.6.24.3|200809120114
2196789|2196789-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200809120852
2197199|2197199-b19f-3f77-b713-d32bba55d77f|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Tie-Hash-Longest|1.1|i86pc-solaris-64int|5.10.0|solaris|2.11|200809121015
2197203|2197203-b19f-3f77-b713-d32bba55d77f|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Statistics-ChiSquare|0.5|i86pc-solaris-64int|5.10.0|solaris|2.11|200809121015
2197209|2197209-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Sort-MultipleFields|1.0|i686-linux|5.10.0|linux|2.6.24.3|200809120417
2197970|2197970-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|Sub-WrapPackages|1.2|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809121217
2200967|2200967-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|Palm-SMS|0.02|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809121821
2201560|2201560-b19f-3f77-b713-d32bba55d77f|pass|200809|JOST@cpan.org ("Josts Smokehouse")|NestedMap|1.0|i86pc-solaris-64int|5.10.0|solaris|2.11|200809122005
2202046|2202046-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34354|linux|2.6.22-1-k7|200809122121
2202341|2202341-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-STDOUT|1.03|i686-linux|5.8.8|linux|2.6.24.3|200809121558
2202370|2202370-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Games-Dice-Advanced|1.1|i686-linux|5.8.8|linux|2.6.24.3|200809121602
2203007|2203007-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34354|linux|2.6.22-1-k7|200809122334
2203270|2203270-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34354|linux|2.6.22-1-k7|200809122156
2203548|2203548-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|File-Overwrite|1.0|i686-linux|5.8.8|linux|2.6.24.3|200809121901
2204171|2204171-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Sub-WrapPackages|1.2|i686-linux|5.8.8|linux|2.6.24.3|200809130040
2206477|2206477-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24.5-smp|200809130456
2206563|2206563-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34354|linux|2.6.22-1-k7|200809130623
2206854|2206854-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|darwin-2level|5.8.8|darwin|8.10.1|200809130639
2207038|2207038-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34354|linux|2.6.22-1-k7|200809130738
2207162|2207162-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34354|linux|2.6.22-1-k7|200809130749
2207391|2207391-b19f-3f77-b713-d32bba55d77f|unknown|200809|JOST@cpan.org ("Josts Smokehouse")|Acme-Licence|1.0|i86pc-solaris-64int|5.10.0|solaris|2.11|200809130817
2209250|2209250-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809131320
2209305|2209305-b19f-3f77-b713-d32bba55d77f|pass|200809|bingos@cpan.org|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0 patch 34342|linux|2.6.24.5-smp|200809131218
2209944|2209944-b19f-3f77-b713-d32bba55d77f|pass|200809|bingos@cpan.org|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.8.8 patch 34347|linux|2.6.24.5-smp|200809131408
2210024|2210024-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.10.0|linux|2.6.24.3|200809130931
2210032|2210032-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Statistics-SerialCorrelation|1.1|i686-linux|5.8.8|linux|2.6.24.3|200809130932
2211035|2211035-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|Class-CanBeA|1.2|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809131806
2211723|2211723-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-CanBeA|1.2|i686-linux|5.10.0|linux|2.6.24.3|200809131419
2213262|2213262-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Sort-MultipleFields|1.0|i686-linux|5.8.8|linux|2.6.24.3|200809131828
2214291|2214291-b19f-3f77-b713-d32bba55d77f|unknown|200809|JOST@cpan.org ("Josts Smokehouse")|Acme-Pony|1.1.2|i86pc-solaris-64int|5.10.0|solaris|2.11|200809140253
2214362|2214362-b19f-3f77-b713-d32bba55d77f|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Tie-Hash-Rank|1.0.1|i86pc-solaris-64int|5.10.0|solaris|2.11|200809140301
2214565|2214565-b19f-3f77-b713-d32bba55d77f|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Tie-Scalar-Decay|1.1.1|i86pc-solaris-64int|5.10.0|solaris|2.11|200809140355
2216453|2216453-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 34342|linux|2.6.22-1-k7|200809140830
2216455|2216455-b19f-3f77-b713-d32bba55d77f|pass|200809|bingos@cpan.org|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.8.8 patch 34347|solaris|2.10|200809140729
2216472|2216472-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 34342|linux|2.6.22-1-k7|200809140834
2216530|2216530-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.8.8|linux|2.6.24.3|200809140241
2216571|2216571-b19f-3f77-b713-d32bba55d77f|pass|200809|bingos@cpan.org|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200809140745
2217634|2217634-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-CanBeA|1.2|i686-linux|5.8.8|linux|2.6.24.3|200809140716
2217915|2217915-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Scalar-Decay|1.1.1|i686-linux|5.10.0|linux|2.6.24.3|200809140815
2218095|2218095-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Statistics-ChiSquare|0.5|i686-linux|5.10.0|linux|2.6.24.3|200809140844
2218653|2218653-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|File-Find-Rule-Permissions|1.3|i686-linux|5.10.0|linux|2.6.24.3|200809141107
2220303|2220303-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|NestedMap|1.0|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809142146
2220467|2220467-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 34360|linux|2.6.22-1-k7|200809142250
2220769|2220769-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|Statistics-ChiSquare|0.5|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809142343
2221043|2221043-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34360|linux|2.6.22-1-k7|200809150019
2221441|2221441-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34360|linux|2.6.22-1-k7|200809150114
2221518|2221518-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34360|linux|2.6.22-1-k7|200809150145
2222888|2222888-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200809150543
2222968|2222968-b19f-3f77-b713-d32bba55d77f|pass|200809|bingos@cpan.org|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.10.0 patch 34342|freebsd|7.0-release|200809150451
2223014|2223014-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34360|linux|2.6.22-1-k7|200809150650
2223018|2223018-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.01|i686-linux|5.10.0|linux|2.6.24.3|200809150058
2223485|2223485-b19f-3f77-b713-d32bba55d77f|pass|200809|srezic@cpan.org|Tie-Hash-Longest|1.1|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809150710
2223644|2223644-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34360|linux|2.6.22-1-k7|200809150759
2226111|2226111-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Scalar-Decay|1.1.1|i686-linux|5.8.8|linux|2.6.24.3|200809151119
2226205|2226205-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|Statistics-ChiSquare|0.5|i686-linux|5.8.8|linux|2.6.24.3|200809151148
2226565|2226565-b19f-3f77-b713-d32bba55d77f|pass|200809|rick@jrmyers.net ("Rick Myers")|File-Find-Rule-Permissions|1.3|i686-linux|5.8.8|linux|2.6.24.3|200809151239
2228780|2228780-b19f-3f77-b713-d32bba55d77f|pass|200809|bingos@cpan.org|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.8.8 patch 34360|freebsd|7.0-release|200809160415
2229376|2229376-b19f-3f77-b713-d32bba55d77f|pass|200809|feldt@nhn.ou.edu|Data-Compare|1.21|sun4-solaris-thread-multi|5.8.8|solaris|2.8|200809151343
2232416|2232416-b19f-3f77-b713-d32bba55d77f|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0 patch 34372|linux|2.6.24-19-generic|200809170320
2233753|2233753-b19f-3f77-b713-d32bba55d77f|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34378|linux|2.6.22-1-k7|200809171121
2234064|2234064-b19f-3f77-b713-d32bba55d77f|pass|200809|dmdk1@comcast.net|Devel-CheckOS|1.43|i486-linux-gnu|5.8.8|linux|2.6.15.7|200809170640
2235521|2235521-b19f-3f77-b713-d32bba55d77f|pass|200809|stro@cpan.o