/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  ChallengeResponse,
  HttpStatusCode,
  MerchantAuthMethod,
} from "@gnu-taler/taler-util";
import {
  useChallengeHandler,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { Fragment, VNode, h } from "preact";
import { useState } from "preact/hooks";
import { AsyncButton } from "../../components/exception/AsyncButton.js";
import {
  FormErrors,
  FormProvider,
} from "../../components/form/FormProvider.js";
import { Input } from "../../components/form/Input.js";
import { NotificationCard } from "../../components/menu/index.js";
import { SolveMFAChallenges } from "../../components/SolveMFA.js";
import { useSessionContext } from "../../context/session.js";
import { Notification } from "../../utils/types.js";
import { FOREVER_REFRESHABLE_TOKEN } from "../login/index.js";

interface Form {
  password: string;
  repeat: string;
}
interface Props {
  onCancel: () => void;
  id: string;
  onReseted: () => void;
}

export function ResetAccount({
  onCancel,
  onReseted,
  id: instanceId,
}: Props): VNode {
  const { i18n } = useTranslationContext();
  const { state: session, lib, logIn } = useSessionContext();
  const [notif, setNotif] = useState<Notification | undefined>(undefined);
  const [value, setValue] = useState<Partial<Form>>({
    // password: "asd",
    // repeat: "asd",
  });

  const errors: FormErrors<Form> = {
    password: !value.password ? i18n.str`Required` : undefined,
    repeat: !value.repeat
      ? i18n.str`Required`
      : value.password !== value.repeat
        ? i18n.str`Doesn't match`
        : undefined,
  };

  function valueHandler(s: (d: Partial<Form>) => Partial<Form>): void {
    const next = s(value);
    const v: Form = {
      password: next.password ?? "",
      repeat: next.repeat ?? "",
    };
    setValue(v);
  }
  const mfa = useChallengeHandler();

  const [doReset, repeatReset] = mfa.withMfaHandler(
    ({ challengeIds, onChallengeRequired }) =>
      async function doResetImpl() {
        try {
          const resp = await lib
            .subInstanceApi(instanceId)
            .instance.forgotPasswordSelfProvision(
              {
                method: MerchantAuthMethod.TOKEN,
                password: value.password!,
              },
              {
                challengeIds,
              },
            );
          if (resp.type === "fail") {
            if (resp.case === HttpStatusCode.Accepted) {
              onChallengeRequired(resp.body);
            } else {
              setNotif({
                message: i18n.str`Failed to create account`,
                type: "ERROR",
                description: resp.detail?.hint,
              });
            }
            return;
          }
          //if auth has been updated, request a new access token
          const result = await lib.instance.createAccessToken(
            instanceId,
            value.password!,
            FOREVER_REFRESHABLE_TOKEN(i18n.str`Password reset`),
          );
          if (result.type === "ok") {
            const { access_token: token } = result.body;
            logIn(instanceId, token);
          }
          onReseted();
        } catch (error) {
          setNotif({
            message: i18n.str`Failed to create account`,
            type: "ERROR",
            description: error instanceof Error ? error.message : String(error),
          });
        }
      },
  );

  if (mfa.pendingChallenge) {
    return (
      <SolveMFAChallenges
        currentChallenge={mfa.pendingChallenge}
        onCompleted={repeatReset}
        onCancel={mfa.doCancelChallenge}
      />
    );
  }

  return (
    <Fragment>
      <NotificationCard notification={notif} />

      <div class="columns is-centered" style={{ margin: "auto" }}>
        <div class="column is-two-thirds ">
          <div class="modal-card" style={{ width: "100%", margin: 0 }}>
            <header
              class="modal-card-head"
              style={{ border: "1px solid", borderBottom: 0 }}
            >
              <p class="modal-card-title">
                <i18n.Translate>
                  Resetting access to the instance "{instanceId}"
                </i18n.Translate>
              </p>
            </header>
            <section
              class="modal-card-body"
              style={{ border: "1px solid", borderTop: 0, borderBottom: 0 }}
            >
              <FormProvider<Form>
                name="settings"
                errors={errors}
                object={value}
                valueHandler={valueHandler}
              >
                <Input<Form>
                  label={i18n.str`New password`}
                  inputType="password"
                  name="password"
                />
                <Input<Form>
                  label={i18n.str`Repeat password`}
                  inputType="password"
                  name="repeat"
                />
              </FormProvider>
            </section>
            <footer
              class="modal-card-foot "
              style={{
                justifyContent: "space-between",
                border: "1px solid",
                borderTop: 0,
              }}
            >
              <button class="button" onClick={onCancel}>
                <i18n.Translate>Cancel</i18n.Translate>
              </button>
              <AsyncButton disabled={!errors} onClick={doReset}>
                <i18n.Translate>Reset</i18n.Translate>
              </AsyncButton>
            </footer>
          </div>
        </div>
      </div>
    </Fragment>
  );
}
