/*
   This file is part of TALER
   Copyright (C) 2025 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_lookup_login_tokens.c
 * @brief Implementation of the lookup_login_tokens function for Postgres
 * @author Iván Ávalos
 */
#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "pg_lookup_login_tokens.h"
#include "pg_helper.h"

/**
 * Context used for TMH_PG_lookup_products().
 */
struct LookupLoginTokensContext
{
  /**
   * Function to call with the results.
   */
  TALER_MERCHANTDB_LoginTokensCallback cb;

  /**
   * Closure for @a cb.
   */
  void *cb_cls;

  /**
   * Did database result extraction fail?
   */
  bool extract_failed;
};


/**
 * Function to be called with the results of a SELECT statement
 * that has returned @a num_results results about products.
 *
 * @param[in,out] cls of type `struct LookupProductsContext *`
 * @param result the postgres result
 * @param num_results the number of results in @a result
 */
static void
lookup_login_tokens_cb (void *cls,
                        PGresult *result,
                        unsigned int num_results)
{
  struct LookupLoginTokensContext *plc = cls;

  for (unsigned int i = 0; i < num_results; i++)
  {
    uint32_t validity_scope;
    uint64_t serial;
    struct GNUNET_TIME_Timestamp expiration_time;
    struct GNUNET_TIME_Timestamp creation_time;
    char *description;
    struct TALER_MERCHANTDB_LoginTokenP token;
    struct GNUNET_PQ_ResultSpec rs[] = {
      GNUNET_PQ_result_spec_auto_from_type ("token",
                                            &token),
      GNUNET_PQ_result_spec_uint32 ("validity_scope",
                                    &validity_scope),
      GNUNET_PQ_result_spec_string ("description",
                                    &description),
      GNUNET_PQ_result_spec_timestamp ("creation_time",
                                       &creation_time),
      GNUNET_PQ_result_spec_timestamp ("expiration_time",
                                       &expiration_time),
      GNUNET_PQ_result_spec_uint64 ("serial",
                                    &serial),
      GNUNET_PQ_result_spec_end
    };

    if (GNUNET_OK !=
        GNUNET_PQ_extract_result (result,
                                  rs,
                                  i))
    {
      GNUNET_break (0);
      plc->extract_failed = true;
      return;
    }
    plc->cb (plc->cb_cls,
             creation_time,
             expiration_time,
             validity_scope,
             description,
             serial);
    GNUNET_PQ_cleanup_result (rs);
  }
}


enum GNUNET_DB_QueryStatus
TMH_PG_lookup_login_tokens (void *cls,
                            const char *instance_id,
                            uint64_t offset,
                            int64_t limit,
                            TALER_MERCHANTDB_LoginTokensCallback cb,
                            void *cb_cls)
{
  struct PostgresClosure *pg = cls;
  struct GNUNET_TIME_Timestamp now = GNUNET_TIME_timestamp_get ();
  uint64_t plimit = (uint64_t) ((limit < 0) ? -limit : limit);
  struct LookupLoginTokensContext plc = {
    .cb = cb,
    .cb_cls = cb_cls,
    /* Can be overwritten by the lookup_products_cb */
    .extract_failed = false,
  };
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_string (instance_id),
    GNUNET_PQ_query_param_timestamp (&now),
    GNUNET_PQ_query_param_uint64 (&offset),
    GNUNET_PQ_query_param_uint64 (&plimit),
    GNUNET_PQ_query_param_end
  };
  enum GNUNET_DB_QueryStatus qs;

  check_connection (pg);
  PREPARE (pg,
           "lookup_login_tokens_asc",
           "SELECT"
           " token"
           ",serial"
           ",creation_time"
           ",expiration_time"
           ",validity_scope"
           ",description"
           " FROM merchant_login_tokens"
           " JOIN merchant_instances"
           "   USING (merchant_serial)"
           " WHERE merchant_instances.merchant_id=$1"
           "   AND expiration_time > $2"
           "   AND serial > $3"
           " ORDER BY serial ASC"
           " LIMIT $4");
  PREPARE (pg,
           "lookup_login_tokens_desc",
           "SELECT"
           " token"
           ",serial"
           ",creation_time"
           ",expiration_time"
           ",validity_scope"
           ",description"
           " FROM merchant_login_tokens"
           " JOIN merchant_instances"
           "   USING (merchant_serial)"
           " WHERE merchant_instances.merchant_id=$1"
           "   AND expiration_time > $2"
           "   AND serial < $3"
           " ORDER BY serial DESC"
           " LIMIT $4");
  qs = GNUNET_PQ_eval_prepared_multi_select (
    pg->conn,
    (limit > 0)
    ? "lookup_login_tokens_asc"
    : "lookup_login_tokens_desc",
    params,
    &lookup_login_tokens_cb,
    &plc);
  /* If there was an error inside lookup_products_cb, return a hard error. */
  if (plc.extract_failed)
    return GNUNET_DB_STATUS_HARD_ERROR;
  return qs;
}
