/* Mapping tables for GBK handling.
   Copyright (C) 1999-2013 Free Software Foundation, Inc.
   This file is part of the GNU C Library.
   Contributed by Sean Chen <seanc@turbolinux.com.cn>, 1999.

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the GNU C Library; if not, see
   <http://www.gnu.org/licenses/>.  */

/*	This file is added for supporting gb18030-2000 code.
	by Wang ShouHua & Wu Jian
	Redflag Linux company
	www.redflag-linux.com
*/

/* The GB18030 standard was re-released on Nov. 2000 and many code values
 * were changed. The reimplementation of this module is based on the info
 * from Markus Scherer's site:
 * http://oss.software.ibm.com/cvs/icu/~checkout~/icu/source/tools/makeconv/gb18030/gb18030.html
 *
 *   Original - Yong Li <rigel863@yahoo.com>  Jan. 28, 2001
 */

/*   Dec 20 2001 Yu Shao <yshao@redhat.com>
 *   Revision, fixing bugs, adding missing entries.
 */

#include <dlfcn.h>
#include <gconv.h>
#include <stdint.h>
#include <wchar.h>

/* The tables in this file are generated from the charmap.
   In a first step, the charmap is converted to a simple table format:

      ./tst-table-charmap.sh GB18030 < ../localedata/charmaps/GB18030 \
      > GB18030.table
*/

/* Table for GB18030 -> UCS-4, containing the two-byte characters only.
   Generated through

   clisp tab.lisp < GB18030.table

   where tab.lisp is:
   ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
   (locally (declare (compile))
     (let ((count 0))
       (loop
	 (let ((line (read-line nil nil nil)))
	   (unless line (return))
	   (when (eql (position #\Tab line) 6)
	     (let ((gbk (parse-integer line :start 2 :end 6 :radix 16))
		   (ucs (parse-integer line :start 9 :radix 16)))
	       (when (<= ucs #xffff)
		 (when (= (mod count 4) 0)
		   (format t " "))
		 (format t " ~([0x~4,'0X] = 0x~4,'0X~),"
			   (let ((ch1 (ldb (byte 8 8) gbk))
				 (ch2 (ldb (byte 8 0) gbk)))
			     (+ (* (- ch1 #x81) 192) (- ch2 #x40)))
			   ucs)
		 (incf count)
		 (when (= (mod count 4) 0)
		   (format t "~%")))))))
       (format t "~%")))
   ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
*/

static const uint16_t __twobyte_to_ucs[] =
{
  [0x0000] = 0x4e02, [0x0001] = 0x4e04, [0x0002] = 0x4e05, [0x0003] = 0x4e06,
  [0x0004] = 0x4e0f, [0x0005] = 0x4e12, [0x0006] = 0x4e17, [0x0007] = 0x4e1f,
  [0x0008] = 0x4e20, [0x0009] = 0x4e21, [0x000a] = 0x4e23, [0x000b] = 0x4e26,
  [0x000c] = 0x4e29, [0x000d] = 0x4e2e, [0x000e] = 0x4e2f, [0x000f] = 0x4e31,
  [0x0010] = 0x4e33, [0x0011] = 0x4e35, [0x0012] = 0x4e37, [0x0013] = 0x4e3c,
  [0x0014] = 0x4e40, [0x0015] = 0x4e41, [0x0016] = 0x4e42, [0x0017] = 0x4e44,
  [0x0018] = 0x4e46, [0x0019] = 0x4e4a, [0x001a] = 0x4e51, [0x001b] = 0x4e55,
  [0x001c] = 0x4e57, [0x001d] = 0x4e5a, [0x001e] = 0x4e5b, [0x001f] = 0x4e62,
  [0x0020] = 0x4e63, [0x0021] = 0x4e64, [0x0022] = 0x4e65, [0x0023] = 0x4e67,
  [0x0024] = 0x4e68, [0x0025] = 0x4e6a, [0x0026] = 0x4e6b, [0x0027] = 0x4e6c,
  [0x0028] = 0x4e6d, [0x0029] = 0x4e6e, [0x002a] = 0x4e6f, [0x002b] = 0x4e72,
  [0x002c] = 0x4e74, [0x002d] = 0x4e75, [0x002e] = 0x4e76, [0x002f] = 0x4e77,
  [0x0030] = 0x4e78, [0x0031] = 0x4e79, [0x0032] = 0x4e7a, [0x0033] = 0x4e7b,
  [0x0034] = 0x4e7c, [0x0035] = 0x4e7d, [0x0036] = 0x4e7f, [0x0037] = 0x4e80,
  [0x0038] = 0x4e81, [0x0039] = 0x4e82, [0x003a] = 0x4e83, [0x003b] = 0x4e84,
  [0x003c] = 0x4e85, [0x003d] = 0x4e87, [0x003e] = 0x4e8a, [0x0040] = 0x4e90,
  [0x0041] = 0x4e96, [0x0042] = 0x4e97, [0x0043] = 0x4e99, [0x0044] = 0x4e9c,
  [0x0045] = 0x4e9d, [0x0046] = 0x4e9e, [0x0047] = 0x4ea3, [0x0048] = 0x4eaa,
  [0x0049] = 0x4eaf, [0x004a] = 0x4eb0, [0x004b] = 0x4eb1, [0x004c] = 0x4eb4,
  [0x004d] = 0x4eb6, [0x004e] = 0x4eb7, [0x004f] = 0x4eb8, [0x0050] = 0x4eb9,
  [0x0051] = 0x4ebc, [0x0052] = 0x4ebd, [0x0053] = 0x4ebe, [0x0054] = 0x4ec8,
  [0x0055] = 0x4ecc, [0x0056] = 0x4ecf, [0x0057] = 0x4ed0, [0x0058] = 0x4ed2,
  [0x0059] = 0x4eda, [0x005a] = 0x4edb, [0x005b] = 0x4edc, [0x005c] = 0x4ee0,
  [0x005d] = 0x4ee2, [0x005e] = 0x4ee6, [0x005f] = 0x4ee7, [0x0060] = 0x4ee9,
  [0x0061] = 0x4eed, [0x0062] = 0x4eee, [0x0063] = 0x4eef, [0x0064] = 0x4ef1,
  [0x0065] = 0x4ef4, [0x0066] = 0x4ef8, [0x0067] = 0x4ef9, [0x0068] = 0x4efa,
  [0x0069] = 0x4efc, [0x006a] = 0x4efe, [0x006b] = 0x4f00, [0x006c] = 0x4f02,
  [0x006d] = 0x4f03, [0x006e] = 0x4f04, [0x006f] = 0x4f05, [0x0070] = 0x4f06,
  [0x0071] = 0x4f07, [0x0072] = 0x4f08, [0x0073] = 0x4f0b, [0x0074] = 0x4f0c,
  [0x0075] = 0x4f12, [0x0076] = 0x4f13, [0x0077] = 0x4f14, [0x0078] = 0x4f15,
  [0x0079] = 0x4f16, [0x007a] = 0x4f1c, [0x007b] = 0x4f1d, [0x007c] = 0x4f21,
  [0x007d] = 0x4f23, [0x007e] = 0x4f28, [0x007f] = 0x4f29, [0x0080] = 0x4f2c,
  [0x0081] = 0x4f2d, [0x0082] = 0x4f2e, [0x0083] = 0x4f31, [0x0084] = 0x4f33,
  [0x0085] = 0x4f35, [0x0086] = 0x4f37, [0x0087] = 0x4f39, [0x0088] = 0x4f3b,
  [0x0089] = 0x4f3e, [0x008a] = 0x4f3f, [0x008b] = 0x4f40, [0x008c] = 0x4f41,
  [0x008d] = 0x4f42, [0x008e] = 0x4f44, [0x008f] = 0x4f45, [0x0090] = 0x4f47,
  [0x0091] = 0x4f48, [0x0092] = 0x4f49, [0x0093] = 0x4f4a, [0x0094] = 0x4f4b,
  [0x0095] = 0x4f4c, [0x0096] = 0x4f52, [0x0097] = 0x4f54, [0x0098] = 0x4f56,
  [0x0099] = 0x4f61, [0x009a] = 0x4f62, [0x009b] = 0x4f66, [0x009c] = 0x4f68,
  [0x009d] = 0x4f6a, [0x009e] = 0x4f6b, [0x009f] = 0x4f6d, [0x00a0] = 0x4f6e,
  [0x00a1] = 0x4f71, [0x00a2] = 0x4f72, [0x00a3] = 0x4f75, [0x00a4] = 0x4f77,
  [0x00a5] = 0x4f78, [0x00a6] = 0x4f79, [0x00a7] = 0x4f7a, [0x00a8] = 0x4f7d,
  [0x00a9] = 0x4f80, [0x00aa] = 0x4f81, [0x00ab] = 0x4f82, [0x00ac] = 0x4f85,
  [0x00ad] = 0x4f86, [0x00ae] = 0x4f87, [0x00af] = 0x4f8a, [0x00b0] = 0x4f8c,
  [0x00b1] = 0x4f8e, [0x00b2] = 0x4f90, [0x00b3] = 0x4f92, [0x00b4] = 0x4f93,
  [0x00b5] = 0x4f95, [0x00b6] = 0x4f96, [0x00b7] = 0x4f98, [0x00b8] = 0x4f99,
  [0x00b9] = 0x4f9a, [0x00ba] = 0x4f9c, [0x00bb] = 0x4f9e, [0x00bc] = 0x4f9f,
  [0x00bd] = 0x4fa1, [0x00be] = 0x4fa2, [0x00c0] = 0x4fa4, [0x00c1] = 0x4fab,
  [0x00c2] = 0x4fad, [0x00c3] = 0x4fb0, [0x00c4] = 0x4fb1, [0x00c5] = 0x4fb2,
  [0x00c6] = 0x4fb3, [0x00c7] = 0x4fb4, [0x00c8] = 0x4fb6, [0x00c9] = 0x4fb7,
  [0x00ca] = 0x4fb8, [0x00cb] = 0x4fb9, [0x00cc] = 0x4fba, [0x00cd] = 0x4fbb,
  [0x00ce] = 0x4fbc, [0x00cf] = 0x4fbd, [0x00d0] = 0x4fbe, [0x00d1] = 0x4fc0,
  [0x00d2] = 0x4fc1, [0x00d3] = 0x4fc2, [0x00d4] = 0x4fc6, [0x00d5] = 0x4fc7,
  [0x00d6] = 0x4fc8, [0x00d7] = 0x4fc9, [0x00d8] = 0x4fcb, [0x00d9] = 0x4fcc,
  [0x00da] = 0x4fcd, [0x00db] = 0x4fd2, [0x00dc] = 0x4fd3, [0x00dd] = 0x4fd4,
  [0x00de] = 0x4fd5, [0x00df] = 0x4fd6, [0x00e0] = 0x4fd9, [0x00e1] = 0x4fdb,
  [0x00e2] = 0x4fe0, [0x00e3] = 0x4fe2, [0x00e4] = 0x4fe4, [0x00e5] = 0x4fe5,
  [0x00e6] = 0x4fe7, [0x00e7] = 0x4feb, [0x00e8] = 0x4fec, [0x00e9] = 0x4ff0,
  [0x00ea] = 0x4ff2, [0x00eb] = 0x4ff4, [0x00ec] = 0x4ff5, [0x00ed] = 0x4ff6,
  [0x00ee] = 0x4ff7, [0x00ef] = 0x4ff9, [0x00f0] = 0x4ffb, [0x00f1] = 0x4ffc,
  [0x00f2] = 0x4ffd, [0x00f3] = 0x4fff, [0x00f4] = 0x5000, [0x00f5] = 0x5001,
  [0x00f6] = 0x5002, [0x00f7] = 0x5003, [0x00f8] = 0x5004, [0x00f9] = 0x5005,
  [0x00fa] = 0x5006, [0x00fb] = 0x5007, [0x00fc] = 0x5008, [0x00fd] = 0x5009,
  [0x00fe] = 0x500a, [0x0100] = 0x500b, [0x0101] = 0x500e, [0x0102] = 0x5010,
  [0x0103] = 0x5011, [0x0104] = 0x5013, [0x0105] = 0x5015, [0x0106] = 0x5016,
  [0x0107] = 0x5017, [0x0108] = 0x501b, [0x0109] = 0x501d, [0x010a] = 0x501e,
  [0x010b] = 0x5020, [0x010c] = 0x5022, [0x010d] = 0x5023, [0x010e] = 0x5024,
  [0x010f] = 0x5027, [0x0110] = 0x502b, [0x0111] = 0x502f, [0x0112] = 0x5030,
  [0x0113] = 0x5031, [0x0114] = 0x5032, [0x0115] = 0x5033, [0x0116] = 0x5034,
  [0x0117] = 0x5035, [0x0118] = 0x5036, [0x0119] = 0x5037, [0x011a] = 0x5038,
  [0x011b] = 0x5039, [0x011c] = 0x503b, [0x011d] = 0x503d, [0x011e] = 0x503f,
  [0x011f] = 0x5040, [0x0120] = 0x5041, [0x0121] = 0x5042, [0x0122] = 0x5044,
  [0x0123] = 0x5045, [0x0124] = 0x5046, [0x0125] = 0x5049, [0x0126] = 0x504a,
  [0x0127] = 0x504b, [0x0128] = 0x504d, [0x0129] = 0x5050, [0x012a] = 0x5051,
  [0x012b] = 0x5052, [0x012c] = 0x5053, [0x012d] = 0x5054, [0x012e] = 0x5056,
  [0x012f] = 0x5057, [0x0130] = 0x5058, [0x0131] = 0x5059, [0x0132] = 0x505b,
  [0x0133] = 0x505d, [0x0134] = 0x505e, [0x0135] = 0x505f, [0x0136] = 0x5060,
  [0x0137] = 0x5061, [0x0138] = 0x5062, [0x0139] = 0x5063, [0x013a] = 0x5064,
  [0x013b] = 0x5066, [0x013c] = 0x5067, [0x013d] = 0x5068, [0x013e] = 0x5069,
  [0x013f] = 0x506a, [0x0140] = 0x506b, [0x0141] = 0x506d, [0x0142] = 0x506e,
  [0x0143] = 0x506f, [0x0144] = 0x5070, [0x0145] = 0x5071, [0x0146] = 0x5072,
  [0x0147] = 0x5073, [0x0148] = 0x5074, [0x0149] = 0x5075, [0x014a] = 0x5078,
  [0x014b] = 0x5079, [0x014c] = 0x507a, [0x014d] = 0x507c, [0x014e] = 0x507d,
  [0x014f] = 0x5081, [0x0150] = 0x5082, [0x0151] = 0x5083, [0x0152] = 0x5084,
  [0x0153] = 0x5086, [0x0154] = 0x5087, [0x0155] = 0x5089, [0x0156] = 0x508a,
  [0x0157] = 0x508b, [0x0158] = 0x508c, [0x0159] = 0x508e, [0x015a] = 0x508f,
  [0x015b] = 0x5090, [0x015c] = 0x5091, [0x015d] = 0x5092, [0x015e] = 0x5093,
  [0x015f] = 0x5094, [0x0160] = 0x5095, [0x0161] = 0x5096, [0x0162] = 0x5097,
  [0x0163] = 0x5098, [0x0164] = 0x5099, [0x0165] = 0x509a, [0x0166] = 0x509b,
  [0x0167] = 0x509c, [0x0168] = 0x509d, [0x0169] = 0x509e, [0x016a] = 0x509f,
  [0x016b] = 0x50a0, [0x016c] = 0x50a1, [0x016d] = 0x50a2, [0x016e] = 0x50a4,
  [0x016f] = 0x50a6, [0x0170] = 0x50aa, [0x0171] = 0x50ab, [0x0172] = 0x50ad,
  [0x0173] = 0x50ae, [0x0174] = 0x50af, [0x0175] = 0x50b0, [0x0176] = 0x50b1,
  [0x0177] = 0x50b3, [0x0178] = 0x50b4, [0x0179] = 0x50b5, [0x017a] = 0x50b6,
  [0x017b] = 0x50b7, [0x017c] = 0x50b8, [0x017d] = 0x50b9, [0x017e] = 0x50bc,
  [0x0180] = 0x50bd, [0x0181] = 0x50be, [0x0182] = 0x50bf, [0x0183] = 0x50c0,
  [0x0184] = 0x50c1, [0x0185] = 0x50c2, [0x0186] = 0x50c3, [0x0187] = 0x50c4,
  [0x0188] = 0x50c5, [0x0189] = 0x50c6, [0x018a] = 0x50c7, [0x018b] = 0x50c8,
  [0x018c] = 0x50c9, [0x018d] = 0x50ca, [0x018e] = 0x50cb, [0x018f] = 0x50cc,
  [0x0190] = 0x50cd, [0x0191] = 0x50ce, [0x0192] = 0x50d0, [0x0193] = 0x50d1,
  [0x0194] = 0x50d2, [0x0195] = 0x50d3, [0x0196] = 0x50d4, [0x0197] = 0x50d5,
  [0x0198] = 0x50d7, [0x0199] = 0x50d8, [0x019a] = 0x50d9, [0x019b] = 0x50db,
  [0x019c] = 0x50dc, [0x019d] = 0x50dd, [0x019e] = 0x50de, [0x019f] = 0x50df,
  [0x01a0] = 0x50e0, [0x01a1] = 0x50e1, [0x01a2] = 0x50e2, [0x01a3] = 0x50e3,
  [0x01a4] = 0x50e4, [0x01a5] = 0x50e5, [0x01a6] = 0x50e8, [0x01a7] = 0x50e9,
  [0x01a8] = 0x50ea, [0x01a9] = 0x50eb, [0x01aa] = 0x50ef, [0x01ab] = 0x50f0,
  [0x01ac] = 0x50f1, [0x01ad] = 0x50f2, [0x01ae] = 0x50f4, [0x01af] = 0x50f6,
  [0x01b0] = 0x50f7, [0x01b1] = 0x50f8, [0x01b2] = 0x50f9, [0x01b3] = 0x50fa,
  [0x01b4] = 0x50fc, [0x01b5] = 0x50fd, [0x01b6] = 0x50fe, [0x01b7] = 0x50ff,
  [0x01b8] = 0x5100, [0x01b9] = 0x5101, [0x01ba] = 0x5102, [0x01bb] = 0x5103,
  [0x01bc] = 0x5104, [0x01bd] = 0x5105, [0x01be] = 0x5108, [0x01c0] = 0x5109,
  [0x01c1] = 0x510a, [0x01c2] = 0x510c, [0x01c3] = 0x510d, [0x01c4] = 0x510e,
  [0x01c5] = 0x510f, [0x01c6] = 0x5110, [0x01c7] = 0x5111, [0x01c8] = 0x5113,
  [0x01c9] = 0x5114, [0x01ca] = 0x5115, [0x01cb] = 0x5116, [0x01cc] = 0x5117,
  [0x01cd] = 0x5118, [0x01ce] = 0x5119, [0x01cf] = 0x511a, [0x01d0] = 0x511b,
  [0x01d1] = 0x511c, [0x01d2] = 0x511d, [0x01d3] = 0x511e, [0x01d4] = 0x511f,
  [0x01d5] = 0x5120, [0x01d6] = 0x5122, [0x01d7] = 0x5123, [0x01d8] = 0x5124,
  [0x01d9] = 0x5125, [0x01da] = 0x5126, [0x01db] = 0x5127, [0x01dc] = 0x5128,
  [0x01dd] = 0x5129, [0x01de] = 0x512a, [0x01df] = 0x512b, [0x01e0] = 0x512c,
  [0x01e1] = 0x512d, [0x01e2] = 0x512e, [0x01e3] = 0x512f, [0x01e4] = 0x5130,
  [0x01e5] = 0x5131, [0x01e6] = 0x5132, [0x01e7] = 0x5133, [0x01e8] = 0x5134,
  [0x01e9] = 0x5135, [0x01ea] = 0x5136, [0x01eb] = 0x5137, [0x01ec] = 0x5138,
  [0x01ed] = 0x5139, [0x01ee] = 0x513a, [0x01ef] = 0x513b, [0x01f0] = 0x513c,
  [0x01f1] = 0x513d, [0x01f2] = 0x513e, [0x01f3] = 0x5142, [0x01f4] = 0x5147,
  [0x01f5] = 0x514a, [0x01f6] = 0x514c, [0x01f7] = 0x514e, [0x01f8] = 0x514f,
  [0x01f9] = 0x5150, [0x01fa] = 0x5152, [0x01fb] = 0x5153, [0x01fc] = 0x5157,
  [0x01fd] = 0x5158, [0x01fe] = 0x5159, [0x01ff] = 0x515b, [0x0200] = 0x515d,
  [0x0201] = 0x515e, [0x0202] = 0x515f, [0x0203] = 0x5160, [0x0204] = 0x5161,
  [0x0205] = 0x5163, [0x0206] = 0x5164, [0x0207] = 0x5166, [0x0208] = 0x5167,
  [0x0209] = 0x5169, [0x020a] = 0x516a, [0x020b] = 0x516f, [0x020c] = 0x5172,
  [0x020d] = 0x517a, [0x020e] = 0x517e, [0x020f] = 0x517f, [0x0210] = 0x5183,
  [0x0211] = 0x5184, [0x0212] = 0x5186, [0x0213] = 0x5187, [0x0214] = 0x518a,
  [0x0215] = 0x518b, [0x0216] = 0x518e, [0x0217] = 0x518f, [0x0218] = 0x5190,
  [0x0219] = 0x5191, [0x021a] = 0x5193, [0x021b] = 0x5194, [0x021c] = 0x5198,
  [0x021d] = 0x519a, [0x021e] = 0x519d, [0x021f] = 0x519e, [0x0220] = 0x519f,
  [0x0221] = 0x51a1, [0x0222] = 0x51a3, [0x0223] = 0x51a6, [0x0224] = 0x51a7,
  [0x0225] = 0x51a8, [0x0226] = 0x51a9, [0x0227] = 0x51aa, [0x0228] = 0x51ad,
  [0x0229] = 0x51ae, [0x022a] = 0x51b4, [0x022b] = 0x51b8, [0x022c] = 0x51b9,
  [0x022d] = 0x51ba, [0x022e] = 0x51be, [0x022f] = 0x51bf, [0x0230] = 0x51c1,
  [0x0231] = 0x51c2, [0x0232] = 0x51c3, [0x0233] = 0x51c5, [0x0234] = 0x51c8,
  [0x0235] = 0x51ca, [0x0236] = 0x51cd, [0x0237] = 0x51ce, [0x0238] = 0x51d0,
  [0x0239] = 0x51d2, [0x023a] = 0x51d3, [0x023b] = 0x51d4, [0x023c] = 0x51d5,
  [0x023d] = 0x51d6, [0x023e] = 0x51d7, [0x0240] = 0x51d8, [0x0241] = 0x51d9,
  [0x0242] = 0x51da, [0x0243] = 0x51dc, [0x0244] = 0x51de, [0x0245] = 0x51df,
  [0x0246] = 0x51e2, [0x0247] = 0x51e3, [0x0248] = 0x51e5, [0x0249] = 0x51e6,
  [0x024a] = 0x51e7, [0x024b] = 0x51e8, [0x024c] = 0x51e9, [0x024d] = 0x51ea,
  [0x024e] = 0x51ec, [0x024f] = 0x51ee, [0x0250] = 0x51f1, [0x0251] = 0x51f2,
  [0x0252] = 0x51f4, [0x0253] = 0x51f7, [0x0254] = 0x51fe, [0x0255] = 0x5204,
  [0x0256] = 0x5205, [0x0257] = 0x5209, [0x0258] = 0x520b, [0x0259] = 0x520c,
  [0x025a] = 0x520f, [0x025b] = 0x5210, [0x025c] = 0x5213, [0x025d] = 0x5214,
  [0x025e] = 0x5215, [0x025f] = 0x521c, [0x0260] = 0x521e, [0x0261] = 0x521f,
  [0x0262] = 0x5221, [0x0263] = 0x5222, [0x0264] = 0x5223, [0x0265] = 0x5225,
  [0x0266] = 0x5226, [0x0267] = 0x5227, [0x0268] = 0x522a, [0x0269] = 0x522c,
  [0x026a] = 0x522f, [0x026b] = 0x5231, [0x026c] = 0x5232, [0x026d] = 0x5234,
  [0x026e] = 0x5235, [0x026f] = 0x523c, [0x0270] = 0x523e, [0x0271] = 0x5244,
  [0x0272] = 0x5245, [0x0273] = 0x5246, [0x0274] = 0x5247, [0x0275] = 0x5248,
  [0x0276] = 0x5249, [0x0277] = 0x524b, [0x0278] = 0x524e, [0x0279] = 0x524f,
  [0x027a] = 0x5252, [0x027b] = 0x5253, [0x027c] = 0x5255, [0x027d] = 0x5257,
  [0x027e] = 0x5258, [0x0280] = 0x5259, [0x0281] = 0x525a, [0x0282] = 0x525b,
  [0x0283] = 0x525d, [0x0284] = 0x525f, [0x0285] = 0x5260, [0x0286] = 0x5262,
  [0x0287] = 0x5263, [0x0288] = 0x5264, [0x0289] = 0x5266, [0x028a] = 0x5268,
  [0x028b] = 0x526b, [0x028c] = 0x526c, [0x028d] = 0x526d, [0x028e] = 0x526e,
  [0x028f] = 0x5270, [0x0290] = 0x5271, [0x0291] = 0x5273, [0x0292] = 0x5274,
  [0x0293] = 0x5275, [0x0294] = 0x5276, [0x0295] = 0x5277, [0x0296] = 0x5278,
  [0x0297] = 0x5279, [0x0298] = 0x527a, [0x0299] = 0x527b, [0x029a] = 0x527c,
  [0x029b] = 0x527e, [0x029c] = 0x5280, [0x029d] = 0x5283, [0x029e] = 0x5284,
  [0x029f] = 0x5285, [0x02a0] = 0x5286, [0x02a1] = 0x5287, [0x02a2] = 0x5289,
  [0x02a3] = 0x528a, [0x02a4] = 0x528b, [0x02a5] = 0x528c, [0x02a6] = 0x528d,
  [0x02a7] = 0x528e, [0x02a8] = 0x528f, [0x02a9] = 0x5291, [0x02aa] = 0x5292,
  [0x02ab] = 0x5294, [0x02ac] = 0x5295, [0x02ad] = 0x5296, [0x02ae] = 0x5297,
  [0x02af] = 0x5298, [0x02b0] = 0x5299, [0x02b1] = 0x529a, [0x02b2] = 0x529c,
  [0x02b3] = 0x52a4, [0x02b4] = 0x52a5, [0x02b5] = 0x52a6, [0x02b6] = 0x52a7,
  [0x02b7] = 0x52ae, [0x02b8] = 0x52af, [0x02b9] = 0x52b0, [0x02ba] = 0x52b4,
  [0x02bb] = 0x52b5, [0x02bc] = 0x52b6, [0x02bd] = 0x52b7, [0x02be] = 0x52b8,
  [0x02bf] = 0x52b9, [0x02c0] = 0x52ba, [0x02c1] = 0x52bb, [0x02c2] = 0x52bc,
  [0x02c3] = 0x52bd, [0x02c4] = 0x52c0, [0x02c5] = 0x52c1, [0x02c6] = 0x52c2,
  [0x02c7] = 0x52c4, [0x02c8] = 0x52c5, [0x02c9] = 0x52c6, [0x02ca] = 0x52c8,
  [0x02cb] = 0x52ca, [0x02cc] = 0x52cc, [0x02cd] = 0x52cd, [0x02ce] = 0x52ce,
  [0x02cf] = 0x52cf, [0x02d0] = 0x52d1, [0x02d1] = 0x52d3, [0x02d2] = 0x52d4,
  [0x02d3] = 0x52d5, [0x02d4] = 0x52d7, [0x02d5] = 0x52d9, [0x02d6] = 0x52da,
  [0x02d7] = 0x52db, [0x02d8] = 0x52dc, [0x02d9] = 0x52dd, [0x02da] = 0x52de,
  [0x02db] = 0x52e0, [0x02dc] = 0x52e1, [0x02dd] = 0x52e2, [0x02de] = 0x52e3,
  [0x02df] = 0x52e5, [0x02e0] = 0x52e6, [0x02e1] = 0x52e7, [0x02e2] = 0x52e8,
  [0x02e3] = 0x52e9, [0x02e4] = 0x52ea, [0x02e5] = 0x52eb, [0x02e6] = 0x52ec,
  [0x02e7] = 0x52ed, [0x02e8] = 0x52ee, [0x02e9] = 0x52ef, [0x02ea] = 0x52f1,
  [0x02eb] = 0x52f2, [0x02ec] = 0x52f3, [0x02ed] = 0x52f4, [0x02ee] = 0x52f5,
  [0x02ef] = 0x52f6, [0x02f0] = 0x52f7, [0x02f1] = 0x52f8, [0x02f2] = 0x52fb,
  [0x02f3] = 0x52fc, [0x02f4] = 0x52fd, [0x02f5] = 0x5301, [0x02f6] = 0x5302,
  [0x02f7] = 0x5303, [0x02f8] = 0x5304, [0x02f9] = 0x5307, [0x02fa] = 0x5309,
  [0x02fb] = 0x530a, [0x02fc] = 0x530b, [0x02fd] = 0x530c, [0x02fe] = 0x530e,
  [0x0300] = 0x5311, [0x0301] = 0x5312, [0x0302] = 0x5313, [0x0303] = 0x5314,
  [0x0304] = 0x5318, [0x0305] = 0x531b, [0x0306] = 0x531c, [0x0307] = 0x531e,
  [0x0308] = 0x531f, [0x0309] = 0x5322, [0x030a] = 0x5324, [0x030b] = 0x5325,
  [0x030c] = 0x5327, [0x030d] = 0x5328, [0x030e] = 0x5329, [0x030f] = 0x532b,
  [0x0310] = 0x532c, [0x0311] = 0x532d, [0x0312] = 0x532f, [0x0313] = 0x5330,
  [0x0314] = 0x5331, [0x0315] = 0x5332, [0x0316] = 0x5333, [0x0317] = 0x5334,
  [0x0318] = 0x5335, [0x0319] = 0x5336, [0x031a] = 0x5337, [0x031b] = 0x5338,
  [0x031c] = 0x533c, [0x031d] = 0x533d, [0x031e] = 0x5340, [0x031f] = 0x5342,
  [0x0320] = 0x5344, [0x0321] = 0x5346, [0x0322] = 0x534b, [0x0323] = 0x534c,
  [0x0324] = 0x534d, [0x0325] = 0x5350, [0x0326] = 0x5354, [0x0327] = 0x5358,
  [0x0328] = 0x5359, [0x0329] = 0x535b, [0x032a] = 0x535d, [0x032b] = 0x5365,
  [0x032c] = 0x5368, [0x032d] = 0x536a, [0x032e] = 0x536c, [0x032f] = 0x536d,
  [0x0330] = 0x5372, [0x0331] = 0x5376, [0x0332] = 0x5379, [0x0333] = 0x537b,
  [0x0334] = 0x537c, [0x0335] = 0x537d, [0x0336] = 0x537e, [0x0337] = 0x5380,
  [0x0338] = 0x5381, [0x0339] = 0x5383, [0x033a] = 0x5387, [0x033b] = 0x5388,
  [0x033c] = 0x538a, [0x033d] = 0x538e, [0x033e] = 0x538f, [0x0340] = 0x5390,
  [0x0341] = 0x5391, [0x0342] = 0x5392, [0x0343] = 0x5393, [0x0344] = 0x5394,
  [0x0345] = 0x5396, [0x0346] = 0x5397, [0x0347] = 0x5399, [0x0348] = 0x539b,
  [0x0349] = 0x539c, [0x034a] = 0x539e, [0x034b] = 0x53a0, [0x034c] = 0x53a1,
  [0x034d] = 0x53a4, [0x034e] = 0x53a7, [0x034f] = 0x53aa, [0x0350] = 0x53ab,
  [0x0351] = 0x53ac, [0x0352] = 0x53ad, [0x0353] = 0x53af, [0x0354] = 0x53b0,
  [0x0355] = 0x53b1, [0x0356] = 0x53b2, [0x0357] = 0x53b3, [0x0358] = 0x53b4,
  [0x0359] = 0x53b5, [0x035a] = 0x53b7, [0x035b] = 0x53b8, [0x035c] = 0x53b9,
  [0x035d] = 0x53ba, [0x035e] = 0x53bc, [0x035f] = 0x53bd, [0x0360] = 0x53be,
  [0x0361] = 0x53c0, [0x0362] = 0x53c3, [0x0363] = 0x53c4, [0x0364] = 0x53c5,
  [0x0365] = 0x53c6, [0x0366] = 0x53c7, [0x0367] = 0x53ce, [0x0368] = 0x53cf,
  [0x0369] = 0x53d0, [0x036a] = 0x53d2, [0x036b] = 0x53d3, [0x036c] = 0x53d5,
  [0x036d] = 0x53da, [0x036e] = 0x53dc, [0x036f] = 0x53dd, [0x0370] = 0x53de,
  [0x0371] = 0x53e1, [0x0372] = 0x53e2, [0x0373] = 0x53e7, [0x0374] = 0x53f4,
  [0x0375] = 0x53fa, [0x0376] = 0x53fe, [0x0377] = 0x53ff, [0x0378] = 0x5400,
  [0x0379] = 0x5402, [0x037a] = 0x5405, [0x037b] = 0x5407, [0x037c] = 0x540b,
  [0x037d] = 0x5414, [0x037e] = 0x5418, [0x037f] = 0x5419, [0x0380] = 0x541a,
  [0x0381] = 0x541c, [0x0382] = 0x5422, [0x0383] = 0x5424, [0x0384] = 0x5425,
  [0x0385] = 0x542a, [0x0386] = 0x5430, [0x0387] = 0x5433, [0x0388] = 0x5436,
  [0x0389] = 0x5437, [0x038a] = 0x543a, [0x038b] = 0x543d, [0x038c] = 0x543f,
  [0x038d] = 0x5441, [0x038e] = 0x5442, [0x038f] = 0x5444, [0x0390] = 0x5445,
  [0x0391] = 0x5447, [0x0392] = 0x5449, [0x0393] = 0x544c, [0x0394] = 0x544d,
  [0x0395] = 0x544e, [0x0396] = 0x544f, [0x0397] = 0x5451, [0x0398] = 0x545a,
  [0x0399] = 0x545d, [0x039a] = 0x545e, [0x039b] = 0x545f, [0x039c] = 0x5460,
  [0x039d] = 0x5461, [0x039e] = 0x5463, [0x039f] = 0x5465, [0x03a0] = 0x5467,
  [0x03a1] = 0x5469, [0x03a2] = 0x546a, [0x03a3] = 0x546b, [0x03a4] = 0x546c,
  [0x03a5] = 0x546d, [0x03a6] = 0x546e, [0x03a7] = 0x546f, [0x03a8] = 0x5470,
  [0x03a9] = 0x5474, [0x03aa] = 0x5479, [0x03ab] = 0x547a, [0x03ac] = 0x547e,
  [0x03ad] = 0x547f, [0x03ae] = 0x5481, [0x03af] = 0x5483, [0x03b0] = 0x5485,
  [0x03b1] = 0x5487, [0x03b2] = 0x5488, [0x03b3] = 0x5489, [0x03b4] = 0x548a,
  [0x03b5] = 0x548d, [0x03b6] = 0x5491, [0x03b7] = 0x5493, [0x03b8] = 0x5497,
  [0x03b9] = 0x5498, [0x03ba] = 0x549c, [0x03bb] = 0x549e, [0x03bc] = 0x549f,
  [0x03bd] = 0x54a0, [0x03be] = 0x54a1, [0x03c0] = 0x54a2, [0x03c1] = 0x54a5,
  [0x03c2] = 0x54ae, [0x03c3] = 0x54b0, [0x03c4] = 0x54b2, [0x03c5] = 0x54b5,
  [0x03c6] = 0x54b6, [0x03c7] = 0x54b7, [0x03c8] = 0x54b9, [0x03c9] = 0x54ba,
  [0x03ca] = 0x54bc, [0x03cb] = 0x54be, [0x03cc] = 0x54c3, [0x03cd] = 0x54c5,
  [0x03ce] = 0x54ca, [0x03cf] = 0x54cb, [0x03d0] = 0x54d6, [0x03d1] = 0x54d8,
  [0x03d2] = 0x54db, [0x03d3] = 0x54e0, [0x03d4] = 0x54e1, [0x03d5] = 0x54e2,
  [0x03d6] = 0x54e3, [0x03d7] = 0x54e4, [0x03d8] = 0x54eb, [0x03d9] = 0x54ec,
  [0x03da] = 0x54ef, [0x03db] = 0x54f0, [0x03dc] = 0x54f1, [0x03dd] = 0x54f4,
  [0x03de] = 0x54f5, [0x03df] = 0x54f6, [0x03e0] = 0x54f7, [0x03e1] = 0x54f8,
  [0x03e2] = 0x54f9, [0x03e3] = 0x54fb, [0x03e4] = 0x54fe, [0x03e5] = 0x5500,
  [0x03e6] = 0x5502, [0x03e7] = 0x5503, [0x03e8] = 0x5504, [0x03e9] = 0x5505,
  [0x03ea] = 0x5508, [0x03eb] = 0x550a, [0x03ec] = 0x550b, [0x03ed] = 0x550c,
  [0x03ee] = 0x550d, [0x03ef] = 0x550e, [0x03f0] = 0x5512, [0x03f1] = 0x5513,
  [0x03f2] = 0x5515, [0x03f3] = 0x5516, [0x03f4] = 0x5517, [0x03f5] = 0x5518,
  [0x03f6] = 0x5519, [0x03f7] = 0x551a, [0x03f8] = 0x551c, [0x03f9] = 0x551d,
  [0x03fa] = 0x551e, [0x03fb] = 0x551f, [0x03fc] = 0x5521, [0x03fd] = 0x5525,
  [0x03fe] = 0x5526, [0x0400] = 0x5528, [0x0401] = 0x5529, [0x0402] = 0x552b,
  [0x0403] = 0x552d, [0x0404] = 0x5532, [0x0405] = 0x5534, [0x0406] = 0x5535,
  [0x0407] = 0x5536, [0x0408] = 0x5538, [0x0409] = 0x5539, [0x040a] = 0x553a,
  [0x040b] = 0x553b, [0x040c] = 0x553d, [0x040d] = 0x5540, [0x040e] = 0x5542,
  [0x040f] = 0x5545, [0x0410] = 0x5547, [0x0411] = 0x5548, [0x0412] = 0x554b,
  [0x0413] = 0x554c, [0x0414] = 0x554d, [0x0415] = 0x554e, [0x0416] = 0x554f,
  [0x0417] = 0x5551, [0x0418] = 0x5552, [0x0419] = 0x5553, [0x041a] = 0x5554,
  [0x041b] = 0x5557, [0x041c] = 0x5558, [0x041d] = 0x5559, [0x041e] = 0x555a,
  [0x041f] = 0x555b, [0x0420] = 0x555d, [0x0421] = 0x555e, [0x0422] = 0x555f,
  [0x0423] = 0x5560, [0x0424] = 0x5562, [0x0425] = 0x5563, [0x0426] = 0x5568,
  [0x0427] = 0x5569, [0x0428] = 0x556b, [0x0429] = 0x556f, [0x042a] = 0x5570,
  [0x042b] = 0x5571, [0x042c] = 0x5572, [0x042d] = 0x5573, [0x042e] = 0x5574,
  [0x042f] = 0x5579, [0x0430] = 0x557a, [0x0431] = 0x557d, [0x0432] = 0x557f,
  [0x0433] = 0x5585, [0x0434] = 0x5586, [0x0435] = 0x558c, [0x0436] = 0x558d,
  [0x0437] = 0x558e, [0x0438] = 0x5590, [0x0439] = 0x5592, [0x043a] = 0x5593,
  [0x043b] = 0x5595, [0x043c] = 0x5596, [0x043d] = 0x5597, [0x043e] = 0x559a,
  [0x043f] = 0x559b, [0x0440] = 0x559e, [0x0441] = 0x55a0, [0x0442] = 0x55a1,
  [0x0443] = 0x55a2, [0x0444] = 0x55a3, [0x0445] = 0x55a4, [0x0446] = 0x55a5,
  [0x0447] = 0x55a6, [0x0448] = 0x55a8, [0x0449] = 0x55a9, [0x044a] = 0x55aa,
  [0x044b] = 0x55ab, [0x044c] = 0x55ac, [0x044d] = 0x55ad, [0x044e] = 0x55ae,
  [0x044f] = 0x55af, [0x0450] = 0x55b0, [0x0451] = 0x55b2, [0x0452] = 0x55b4,
  [0x0453] = 0x55b6, [0x0454] = 0x55b8, [0x0455] = 0x55ba, [0x0456] = 0x55bc,
  [0x0457] = 0x55bf, [0x0458] = 0x55c0, [0x0459] = 0x55c1, [0x045a] = 0x55c2,
  [0x045b] = 0x55c3, [0x045c] = 0x55c6, [0x045d] = 0x55c7, [0x045e] = 0x55c8,
  [0x045f] = 0x55ca, [0x0460] = 0x55cb, [0x0461] = 0x55ce, [0x0462] = 0x55cf,
  [0x0463] = 0x55d0, [0x0464] = 0x55d5, [0x0465] = 0x55d7, [0x0466] = 0x55d8,
  [0x0467] = 0x55d9, [0x0468] = 0x55da, [0x0469] = 0x55db, [0x046a] = 0x55de,
  [0x046b] = 0x55e0, [0x046c] = 0x55e2, [0x046d] = 0x55e7, [0x046e] = 0x55e9,
  [0x046f] = 0x55ed, [0x0470] = 0x55ee, [0x0471] = 0x55f0, [0x0472] = 0x55f1,
  [0x0473] = 0x55f4, [0x0474] = 0x55f6, [0x0475] = 0x55f8, [0x0476] = 0x55f9,
  [0x0477] = 0x55fa, [0x0478] = 0x55fb, [0x0479] = 0x55fc, [0x047a] = 0x55ff,
  [0x047b] = 0x5602, [0x047c] = 0x5603, [0x047d] = 0x5604, [0x047e] = 0x5605,
  [0x0480] = 0x5606, [0x0481] = 0x5607, [0x0482] = 0x560a, [0x0483] = 0x560b,
  [0x0484] = 0x560d, [0x0485] = 0x5610, [0x0486] = 0x5611, [0x0487] = 0x5612,
  [0x0488] = 0x5613, [0x0489] = 0x5614, [0x048a] = 0x5615, [0x048b] = 0x5616,
  [0x048c] = 0x5617, [0x048d] = 0x5619, [0x048e] = 0x561a, [0x048f] = 0x561c,
  [0x0490] = 0x561d, [0x0491] = 0x5620, [0x0492] = 0x5621, [0x0493] = 0x5622,
  [0x0494] = 0x5625, [0x0495] = 0x5626, [0x0496] = 0x5628, [0x0497] = 0x5629,
  [0x0498] = 0x562a, [0x0499] = 0x562b, [0x049a] = 0x562e, [0x049b] = 0x562f,
  [0x049c] = 0x5630, [0x049d] = 0x5633, [0x049e] = 0x5635, [0x049f] = 0x5637,
  [0x04a0] = 0x5638, [0x04a1] = 0x563a, [0x04a2] = 0x563c, [0x04a3] = 0x563d,
  [0x04a4] = 0x563e, [0x04a5] = 0x5640, [0x04a6] = 0x5641, [0x04a7] = 0x5642,
  [0x04a8] = 0x5643, [0x04a9] = 0x5644, [0x04aa] = 0x5645, [0x04ab] = 0x5646,
  [0x04ac] = 0x5647, [0x04ad] = 0x5648, [0x04ae] = 0x5649, [0x04af] = 0x564a,
  [0x04b0] = 0x564b, [0x04b1] = 0x564f, [0x04b2] = 0x5650, [0x04b3] = 0x5651,
  [0x04b4] = 0x5652, [0x04b5] = 0x5653, [0x04b6] = 0x5655, [0x04b7] = 0x5656,
  [0x04b8] = 0x565a, [0x04b9] = 0x565b, [0x04ba] = 0x565d, [0x04bb] = 0x565e,
  [0x04bc] = 0x565f, [0x04bd] = 0x5660, [0x04be] = 0x5661, [0x04c0] = 0x5663,
  [0x04c1] = 0x5665, [0x04c2] = 0x5666, [0x04c3] = 0x5667, [0x04c4] = 0x566d,
  [0x04c5] = 0x566e, [0x04c6] = 0x566f, [0x04c7] = 0x5670, [0x04c8] = 0x5672,
  [0x04c9] = 0x5673, [0x04ca] = 0x5674, [0x04cb] = 0x5675, [0x04cc] = 0x5677,
  [0x04cd] = 0x5678, [0x04ce] = 0x5679, [0x04cf] = 0x567a, [0x04d0] = 0x567d,
  [0x04d1] = 0x567e, [0x04d2] = 0x567f, [0x04d3] = 0x5680, [0x04d4] = 0x5681,
  [0x04d5] = 0x5682, [0x04d6] = 0x5683, [0x04d7] = 0x5684, [0x04d8] = 0x5687,
  [0x04d9] = 0x5688, [0x04da] = 0x5689, [0x04db] = 0x568a, [0x04dc] = 0x568b,
  [0x04dd] = 0x568c, [0x04de] = 0x568d, [0x04df] = 0x5690, [0x04e0] = 0x5691,
  [0x04e1] = 0x5692, [0x04e2] = 0x5694, [0x04e3] = 0x5695, [0x04e4] = 0x5696,
  [0x04e5] = 0x5697, [0x04e6] = 0x5698, [0x04e7] = 0x5699, [0x04e8] = 0x569a,
  [0x04e9] = 0x569b, [0x04ea] = 0x569c, [0x04eb] = 0x569d, [0x04ec] = 0x569e,
  [0x04ed] = 0x569f, [0x04ee] = 0x56a0, [0x04ef] = 0x56a1, [0x04f0] = 0x56a2,
  [0x04f1] = 0x56a4, [0x04f2] = 0x56a5, [0x04f3] = 0x56a6, [0x04f4] = 0x56a7,
  [0x04f5] = 0x56a8, [0x04f6] = 0x56a9, [0x04f7] = 0x56aa, [0x04f8] = 0x56ab,
  [0x04f9] = 0x56ac, [0x04fa] = 0x56ad, [0x04fb] = 0x56ae, [0x04fc] = 0x56b0,
  [0x04fd] = 0x56b1, [0x04fe] = 0x56b2, [0x04ff] = 0x56b3, [0x0500] = 0x56b4,
  [0x0501] = 0x56b5, [0x0502] = 0x56b6, [0x0503] = 0x56b8, [0x0504] = 0x56b9,
  [0x0505] = 0x56ba, [0x0506] = 0x56bb, [0x0507] = 0x56bd, [0x0508] = 0x56be,
  [0x0509] = 0x56bf, [0x050a] = 0x56c0, [0x050b] = 0x56c1, [0x050c] = 0x56c2,
  [0x050d] = 0x56c3, [0x050e] = 0x56c4, [0x050f] = 0x56c5, [0x0510] = 0x56c6,
  [0x0511] = 0x56c7, [0x0512] = 0x56c8, [0x0513] = 0x56c9, [0x0514] = 0x56cb,
  [0x0515] = 0x56cc, [0x0516] = 0x56cd, [0x0517] = 0x56ce, [0x0518] = 0x56cf,
  [0x0519] = 0x56d0, [0x051a] = 0x56d1, [0x051b] = 0x56d2, [0x051c] = 0x56d3,
  [0x051d] = 0x56d5, [0x051e] = 0x56d6, [0x051f] = 0x56d8, [0x0520] = 0x56d9,
  [0x0521] = 0x56dc, [0x0522] = 0x56e3, [0x0523] = 0x56e5, [0x0524] = 0x56e6,
  [0x0525] = 0x56e7, [0x0526] = 0x56e8, [0x0527] = 0x56e9, [0x0528] = 0x56ea,
  [0x0529] = 0x56ec, [0x052a] = 0x56ee, [0x052b] = 0x56ef, [0x052c] = 0x56f2,
  [0x052d] = 0x56f3, [0x052e] = 0x56f6, [0x052f] = 0x56f7, [0x0530] = 0x56f8,
  [0x0531] = 0x56fb, [0x0532] = 0x56fc, [0x0533] = 0x5700, [0x0534] = 0x5701,
  [0x0535] = 0x5702, [0x0536] = 0x5705, [0x0537] = 0x5707, [0x0538] = 0x570b,
  [0x0539] = 0x570c, [0x053a] = 0x570d, [0x053b] = 0x570e, [0x053c] = 0x570f,
  [0x053d] = 0x5710, [0x053e] = 0x5711, [0x0540] = 0x5712, [0x0541] = 0x5713,
  [0x0542] = 0x5714, [0x0543] = 0x5715, [0x0544] = 0x5716, [0x0545] = 0x5717,
  [0x0546] = 0x5718, [0x0547] = 0x5719, [0x0548] = 0x571a, [0x0549] = 0x571b,
  [0x054a] = 0x571d, [0x054b] = 0x571e, [0x054c] = 0x5720, [0x054d] = 0x5721,
  [0x054e] = 0x5722, [0x054f] = 0x5724, [0x0550] = 0x5725, [0x0551] = 0x5726,
  [0x0552] = 0x5727, [0x0553] = 0x572b, [0x0554] = 0x5731, [0x0555] = 0x5732,
  [0x0556] = 0x5734, [0x0557] = 0x5735, [0x0558] = 0x5736, [0x0559] = 0x5737,
  [0x055a] = 0x5738, [0x055b] = 0x573c, [0x055c] = 0x573d, [0x055d] = 0x573f,
  [0x055e] = 0x5741, [0x055f] = 0x5743, [0x0560] = 0x5744, [0x0561] = 0x5745,
  [0x0562] = 0x5746, [0x0563] = 0x5748, [0x0564] = 0x5749, [0x0565] = 0x574b,
  [0x0566] = 0x5752, [0x0567] = 0x5753, [0x0568] = 0x5754, [0x0569] = 0x5755,
  [0x056a] = 0x5756, [0x056b] = 0x5758, [0x056c] = 0x5759, [0x056d] = 0x5762,
  [0x056e] = 0x5763, [0x056f] = 0x5765, [0x0570] = 0x5767, [0x0571] = 0x576c,
  [0x0572] = 0x576e, [0x0573] = 0x5770, [0x0574] = 0x5771, [0x0575] = 0x5772,
  [0x0576] = 0x5774, [0x0577] = 0x5775, [0x0578] = 0x5778, [0x0579] = 0x5779,
  [0x057a] = 0x577a, [0x057b] = 0x577d, [0x057c] = 0x577e, [0x057d] = 0x577f,
  [0x057e] = 0x5780, [0x0580] = 0x5781, [0x0581] = 0x5787, [0x0582] = 0x5788,
  [0x0583] = 0x5789, [0x0584] = 0x578a, [0x0585] = 0x578d, [0x0586] = 0x578e,
  [0x0587] = 0x578f, [0x0588] = 0x5790, [0x0589] = 0x5791, [0x058a] = 0x5794,
  [0x058b] = 0x5795, [0x058c] = 0x5796, [0x058d] = 0x5797, [0x058e] = 0x5798,
  [0x058f] = 0x5799, [0x0590] = 0x579a, [0x0591] = 0x579c, [0x0592] = 0x579d,
  [0x0593] = 0x579e, [0x0594] = 0x579f, [0x0595] = 0x57a5, [0x0596] = 0x57a8,
  [0x0597] = 0x57aa, [0x0598] = 0x57ac, [0x0599] = 0x57af, [0x059a] = 0x57b0,
  [0x059b] = 0x57b1, [0x059c] = 0x57b3, [0x059d] = 0x57b5, [0x059e] = 0x57b6,
  [0x059f] = 0x57b7, [0x05a0] = 0x57b9, [0x05a1] = 0x57ba, [0x05a2] = 0x57bb,
  [0x05a3] = 0x57bc, [0x05a4] = 0x57bd, [0x05a5] = 0x57be, [0x05a6] = 0x57bf,
  [0x05a7] = 0x57c0, [0x05a8] = 0x57c1, [0x05a9] = 0x57c4, [0x05aa] = 0x57c5,
  [0x05ab] = 0x57c6, [0x05ac] = 0x57c7, [0x05ad] = 0x57c8, [0x05ae] = 0x57c9,
  [0x05af] = 0x57ca, [0x05b0] = 0x57cc, [0x05b1] = 0x57cd, [0x05b2] = 0x57d0,
  [0x05b3] = 0x57d1, [0x05b4] = 0x57d3, [0x05b5] = 0x57d6, [0x05b6] = 0x57d7,
  [0x05b7] = 0x57db, [0x05b8] = 0x57dc, [0x05b9] = 0x57de, [0x05ba] = 0x57e1,
  [0x05bb] = 0x57e2, [0x05bc] = 0x57e3, [0x05bd] = 0x57e5, [0x05be] = 0x57e6,
  [0x05bf] = 0x57e7, [0x05c0] = 0x57e8, [0x05c1] = 0x57e9, [0x05c2] = 0x57ea,
  [0x05c3] = 0x57eb, [0x05c4] = 0x57ec, [0x05c5] = 0x57ee, [0x05c6] = 0x57f0,
  [0x05c7] = 0x57f1, [0x05c8] = 0x57f2, [0x05c9] = 0x57f3, [0x05ca] = 0x57f5,
  [0x05cb] = 0x57f6, [0x05cc] = 0x57f7, [0x05cd] = 0x57fb, [0x05ce] = 0x57fc,
  [0x05cf] = 0x57fe, [0x05d0] = 0x57ff, [0x05d1] = 0x5801, [0x05d2] = 0x5803,
  [0x05d3] = 0x5804, [0x05d4] = 0x5805, [0x05d5] = 0x5808, [0x05d6] = 0x5809,
  [0x05d7] = 0x580a, [0x05d8] = 0x580c, [0x05d9] = 0x580e, [0x05da] = 0x580f,
  [0x05db] = 0x5810, [0x05dc] = 0x5812, [0x05dd] = 0x5813, [0x05de] = 0x5814,
  [0x05df] = 0x5816, [0x05e0] = 0x5817, [0x05e1] = 0x5818, [0x05e2] = 0x581a,
  [0x05e3] = 0x581b, [0x05e4] = 0x581c, [0x05e5] = 0x581d, [0x05e6] = 0x581f,
  [0x05e7] = 0x5822, [0x05e8] = 0x5823, [0x05e9] = 0x5825, [0x05ea] = 0x5826,
  [0x05eb] = 0x5827, [0x05ec] = 0x5828, [0x05ed] = 0x5829, [0x05ee] = 0x582b,
  [0x05ef] = 0x582c, [0x05f0] = 0x582d, [0x05f1] = 0x582e, [0x05f2] = 0x582f,
  [0x05f3] = 0x5831, [0x05f4] = 0x5832, [0x05f5] = 0x5833, [0x05f6] = 0x5834,
  [0x05f7] = 0x5836, [0x05f8] = 0x5837, [0x05f9] = 0x5838, [0x05fa] = 0x5839,
  [0x05fb] = 0x583a, [0x05fc] = 0x583b, [0x05fd] = 0x583c, [0x05fe] = 0x583d,
  [0x0600] = 0x583e, [0x0601] = 0x583f, [0x0602] = 0x5840, [0x0603] = 0x5841,
  [0x0604] = 0x5842, [0x0605] = 0x5843, [0x0606] = 0x5845, [0x0607] = 0x5846,
  [0x0608] = 0x5847, [0x0609] = 0x5848, [0x060a] = 0x5849, [0x060b] = 0x584a,
  [0x060c] = 0x584b, [0x060d] = 0x584e, [0x060e] = 0x584f, [0x060f] = 0x5850,
  [0x0610] = 0x5852, [0x0611] = 0x5853, [0x0612] = 0x5855, [0x0613] = 0x5856,
  [0x0614] = 0x5857, [0x0615] = 0x5859, [0x0616] = 0x585a, [0x0617] = 0x585b,
  [0x0618] = 0x585c, [0x0619] = 0x585d, [0x061a] = 0x585f, [0x061b] = 0x5860,
  [0x061c] = 0x5861, [0x061d] = 0x5862, [0x061e] = 0x5863, [0x061f] = 0x5864,
  [0x0620] = 0x5866, [0x0621] = 0x5867, [0x0622] = 0x5868, [0x0623] = 0x5869,
  [0x0624] = 0x586a, [0x0625] = 0x586d, [0x0626] = 0x586e, [0x0627] = 0x586f,
  [0x0628] = 0x5870, [0x0629] = 0x5871, [0x062a] = 0x5872, [0x062b] = 0x5873,
  [0x062c] = 0x5874, [0x062d] = 0x5875, [0x062e] = 0x5876, [0x062f] = 0x5877,
  [0x0630] = 0x5878, [0x0631] = 0x5879, [0x0632] = 0x587a, [0x0633] = 0x587b,
  [0x0634] = 0x587c, [0x0635] = 0x587d, [0x0636] = 0x587f, [0x0637] = 0x5882,
  [0x0638] = 0x5884, [0x0639] = 0x5886, [0x063a] = 0x5887, [0x063b] = 0x5888,
  [0x063c] = 0x588a, [0x063d] = 0x588b, [0x063e] = 0x588c, [0x0640] = 0x588d,
  [0x0641] = 0x588e, [0x0642] = 0x588f, [0x0643] = 0x5890, [0x0644] = 0x5891,
  [0x0645] = 0x5894, [0x0646] = 0x5895, [0x0647] = 0x5896, [0x0648] = 0x5897,
  [0x0649] = 0x5898, [0x064a] = 0x589b, [0x064b] = 0x589c, [0x064c] = 0x589d,
  [0x064d] = 0x58a0, [0x064e] = 0x58a1, [0x064f] = 0x58a2, [0x0650] = 0x58a3,
  [0x0651] = 0x58a4, [0x0652] = 0x58a5, [0x0653] = 0x58a6, [0x0654] = 0x58a7,
  [0x0655] = 0x58aa, [0x0656] = 0x58ab, [0x0657] = 0x58ac, [0x0658] = 0x58ad,
  [0x0659] = 0x58ae, [0x065a] = 0x58af, [0x065b] = 0x58b0, [0x065c] = 0x58b1,
  [0x065d] = 0x58b2, [0x065e] = 0x58b3, [0x065f] = 0x58b4, [0x0660] = 0x58b5,
  [0x0661] = 0x58b6, [0x0662] = 0x58b7, [0x0663] = 0x58b8, [0x0664] = 0x58b9,
  [0x0665] = 0x58ba, [0x0666] = 0x58bb, [0x0667] = 0x58bd, [0x0668] = 0x58be,
  [0x0669] = 0x58bf, [0x066a] = 0x58c0, [0x066b] = 0x58c2, [0x066c] = 0x58c3,
  [0x066d] = 0x58c4, [0x066e] = 0x58c6, [0x066f] = 0x58c7, [0x0670] = 0x58c8,
  [0x0671] = 0x58c9, [0x0672] = 0x58ca, [0x0673] = 0x58cb, [0x0674] = 0x58cc,
  [0x0675] = 0x58cd, [0x0676] = 0x58ce, [0x0677] = 0x58cf, [0x0678] = 0x58d0,
  [0x0679] = 0x58d2, [0x067a] = 0x58d3, [0x067b] = 0x58d4, [0x067c] = 0x58d6,
  [0x067d] = 0x58d7, [0x067e] = 0x58d8, [0x067f] = 0x58d9, [0x0680] = 0x58da,
  [0x0681] = 0x58db, [0x0682] = 0x58dc, [0x0683] = 0x58dd, [0x0684] = 0x58de,
  [0x0685] = 0x58df, [0x0686] = 0x58e0, [0x0687] = 0x58e1, [0x0688] = 0x58e2,
  [0x0689] = 0x58e3, [0x068a] = 0x58e5, [0x068b] = 0x58e6, [0x068c] = 0x58e7,
  [0x068d] = 0x58e8, [0x068e] = 0x58e9, [0x068f] = 0x58ea, [0x0690] = 0x58ed,
  [0x0691] = 0x58ef, [0x0692] = 0x58f1, [0x0693] = 0x58f2, [0x0694] = 0x58f4,
  [0x0695] = 0x58f5, [0x0696] = 0x58f7, [0x0697] = 0x58f8, [0x0698] = 0x58fa,
  [0x0699] = 0x58fb, [0x069a] = 0x58fc, [0x069b] = 0x58fd, [0x069c] = 0x58fe,
  [0x069d] = 0x58ff, [0x069e] = 0x5900, [0x069f] = 0x5901, [0x06a0] = 0x5903,
  [0x06a1] = 0x5905, [0x06a2] = 0x5906, [0x06a3] = 0x5908, [0x06a4] = 0x5909,
  [0x06a5] = 0x590a, [0x06a6] = 0x590b, [0x06a7] = 0x590c, [0x06a8] = 0x590e,
  [0x06a9] = 0x5910, [0x06aa] = 0x5911, [0x06ab] = 0x5912, [0x06ac] = 0x5913,
  [0x06ad] = 0x5917, [0x06ae] = 0x5918, [0x06af] = 0x591b, [0x06b0] = 0x591d,
  [0x06b1] = 0x591e, [0x06b2] = 0x5920, [0x06b3] = 0x5921, [0x06b4] = 0x5922,
  [0x06b5] = 0x5923, [0x06b6] = 0x5926, [0x06b7] = 0x5928, [0x06b8] = 0x592c,
  [0x06b9] = 0x5930, [0x06ba] = 0x5932, [0x06bb] = 0x5933, [0x06bc] = 0x5935,
  [0x06bd] = 0x5936, [0x06be] = 0x593b, [0x06c0] = 0x593d, [0x06c1] = 0x593e,
  [0x06c2] = 0x593f, [0x06c3] = 0x5940, [0x06c4] = 0x5943, [0x06c5] = 0x5945,
  [0x06c6] = 0x5946, [0x06c7] = 0x594a, [0x06c8] = 0x594c, [0x06c9] = 0x594d,
  [0x06ca] = 0x5950, [0x06cb] = 0x5952, [0x06cc] = 0x5953, [0x06cd] = 0x5959,
  [0x06ce] = 0x595b, [0x06cf] = 0x595c, [0x06d0] = 0x595d, [0x06d1] = 0x595e,
  [0x06d2] = 0x595f, [0x06d3] = 0x5961, [0x06d4] = 0x5963, [0x06d5] = 0x5964,
  [0x06d6] = 0x5966, [0x06d7] = 0x5967, [0x06d8] = 0x5968, [0x06d9] = 0x5969,
  [0x06da] = 0x596a, [0x06db] = 0x596b, [0x06dc] = 0x596c, [0x06dd] = 0x596d,
  [0x06de] = 0x596e, [0x06df] = 0x596f, [0x06e0] = 0x5970, [0x06e1] = 0x5971,
  [0x06e2] = 0x5972, [0x06e3] = 0x5975, [0x06e4] = 0x5977, [0x06e5] = 0x597a,
  [0x06e6] = 0x597b, [0x06e7] = 0x597c, [0x06e8] = 0x597e, [0x06e9] = 0x597f,
  [0x06ea] = 0x5980, [0x06eb] = 0x5985, [0x06ec] = 0x5989, [0x06ed] = 0x598b,
  [0x06ee] = 0x598c, [0x06ef] = 0x598e, [0x06f0] = 0x598f, [0x06f1] = 0x5990,
  [0x06f2] = 0x5991, [0x06f3] = 0x5994, [0x06f4] = 0x5995, [0x06f5] = 0x5998,
  [0x06f6] = 0x599a, [0x06f7] = 0x599b, [0x06f8] = 0x599c, [0x06f9] = 0x599d,
  [0x06fa] = 0x599f, [0x06fb] = 0x59a0, [0x06fc] = 0x59a1, [0x06fd] = 0x59a2,
  [0x06fe] = 0x59a6, [0x0700] = 0x59a7, [0x0701] = 0x59ac, [0x0702] = 0x59ad,
  [0x0703] = 0x59b0, [0x0704] = 0x59b1, [0x0705] = 0x59b3, [0x0706] = 0x59b4,
  [0x0707] = 0x59b5, [0x0708] = 0x59b6, [0x0709] = 0x59b7, [0x070a] = 0x59b8,
  [0x070b] = 0x59ba, [0x070c] = 0x59bc, [0x070d] = 0x59bd, [0x070e] = 0x59bf,
  [0x070f] = 0x59c0, [0x0710] = 0x59c1, [0x0711] = 0x59c2, [0x0712] = 0x59c3,
  [0x0713] = 0x59c4, [0x0714] = 0x59c5, [0x0715] = 0x59c7, [0x0716] = 0x59c8,
  [0x0717] = 0x59c9, [0x0718] = 0x59cc, [0x0719] = 0x59cd, [0x071a] = 0x59ce,
  [0x071b] = 0x59cf, [0x071c] = 0x59d5, [0x071d] = 0x59d6, [0x071e] = 0x59d9,
  [0x071f] = 0x59db, [0x0720] = 0x59de, [0x0721] = 0x59df, [0x0722] = 0x59e0,
  [0x0723] = 0x59e1, [0x0724] = 0x59e2, [0x0725] = 0x59e4, [0x0726] = 0x59e6,
  [0x0727] = 0x59e7, [0x0728] = 0x59e9, [0x0729] = 0x59ea, [0x072a] = 0x59eb,
  [0x072b] = 0x59ed, [0x072c] = 0x59ee, [0x072d] = 0x59ef, [0x072e] = 0x59f0,
  [0x072f] = 0x59f1, [0x0730] = 0x59f2, [0x0731] = 0x59f3, [0x0732] = 0x59f4,
  [0x0733] = 0x59f5, [0x0734] = 0x59f6, [0x0735] = 0x59f7, [0x0736] = 0x59f8,
  [0x0737] = 0x59fa, [0x0738] = 0x59fc, [0x0739] = 0x59fd, [0x073a] = 0x59fe,
  [0x073b] = 0x5a00, [0x073c] = 0x5a02, [0x073d] = 0x5a0a, [0x073e] = 0x5a0b,
  [0x073f] = 0x5a0d, [0x0740] = 0x5a0e, [0x0741] = 0x5a0f, [0x0742] = 0x5a10,
  [0x0743] = 0x5a12, [0x0744] = 0x5a14, [0x0745] = 0x5a15, [0x0746] = 0x5a16,
  [0x0747] = 0x5a17, [0x0748] = 0x5a19, [0x0749] = 0x5a1a, [0x074a] = 0x5a1b,
  [0x074b] = 0x5a1d, [0x074c] = 0x5a1e, [0x074d] = 0x5a21, [0x074e] = 0x5a22,
  [0x074f] = 0x5a24, [0x0750] = 0x5a26, [0x0751] = 0x5a27, [0x0752] = 0x5a28,
  [0x0753] = 0x5a2a, [0x0754] = 0x5a2b, [0x0755] = 0x5a2c, [0x0756] = 0x5a2d,
  [0x0757] = 0x5a2e, [0x0758] = 0x5a2f, [0x0759] = 0x5a30, [0x075a] = 0x5a33,
  [0x075b] = 0x5a35, [0x075c] = 0x5a37, [0x075d] = 0x5a38, [0x075e] = 0x5a39,
  [0x075f] = 0x5a3a, [0x0760] = 0x5a3b, [0x0761] = 0x5a3d, [0x0762] = 0x5a3e,
  [0x0763] = 0x5a3f, [0x0764] = 0x5a41, [0x0765] = 0x5a42, [0x0766] = 0x5a43,
  [0x0767] = 0x5a44, [0x0768] = 0x5a45, [0x0769] = 0x5a47, [0x076a] = 0x5a48,
  [0x076b] = 0x5a4b, [0x076c] = 0x5a4c, [0x076d] = 0x5a4d, [0x076e] = 0x5a4e,
  [0x076f] = 0x5a4f, [0x0770] = 0x5a50, [0x0771] = 0x5a51, [0x0772] = 0x5a52,
  [0x0773] = 0x5a53, [0x0774] = 0x5a54, [0x0775] = 0x5a56, [0x0776] = 0x5a57,
  [0x0777] = 0x5a58, [0x0778] = 0x5a59, [0x0779] = 0x5a5b, [0x077a] = 0x5a5c,
  [0x077b] = 0x5a5d, [0x077c] = 0x5a5e, [0x077d] = 0x5a5f, [0x077e] = 0x5a60,
  [0x0780] = 0x5a61, [0x0781] = 0x5a63, [0x0782] = 0x5a64, [0x0783] = 0x5a65,
  [0x0784] = 0x5a66, [0x0785] = 0x5a68, [0x0786] = 0x5a69, [0x0787] = 0x5a6b,
  [0x0788] = 0x5a6c, [0x0789] = 0x5a6d, [0x078a] = 0x5a6e, [0x078b] = 0x5a6f,
  [0x078c] = 0x5a70, [0x078d] = 0x5a71, [0x078e] = 0x5a72, [0x078f] = 0x5a73,
  [0x0790] = 0x5a78, [0x0791] = 0x5a79, [0x0792] = 0x5a7b, [0x0793] = 0x5a7c,
  [0x0794] = 0x5a7d, [0x0795] = 0x5a7e, [0x0796] = 0x5a80, [0x0797] = 0x5a81,
  [0x0798] = 0x5a82, [0x0799] = 0x5a83, [0x079a] = 0x5a84, [0x079b] = 0x5a85,
  [0x079c] = 0x5a86, [0x079d] = 0x5a87, [0x079e] = 0x5a88, [0x079f] = 0x5a89,
  [0x07a0] = 0x5a8a, [0x07a1] = 0x5a8b, [0x07a2] = 0x5a8c, [0x07a3] = 0x5a8d,
  [0x07a4] = 0x5a8e, [0x07a5] = 0x5a8f, [0x07a6] = 0x5a90, [0x07a7] = 0x5a91,
  [0x07a8] = 0x5a93, [0x07a9] = 0x5a94, [0x07aa] = 0x5a95, [0x07ab] = 0x5a96,
  [0x07ac] = 0x5a97, [0x07ad] = 0x5a98, [0x07ae] = 0x5a99, [0x07af] = 0x5a9c,
  [0x07b0] = 0x5a9d, [0x07b1] = 0x5a9e, [0x07b2] = 0x5a9f, [0x07b3] = 0x5aa0,
  [0x07b4] = 0x5aa1, [0x07b5] = 0x5aa2, [0x07b6] = 0x5aa3, [0x07b7] = 0x5aa4,
  [0x07b8] = 0x5aa5, [0x07b9] = 0x5aa6, [0x07ba] = 0x5aa7, [0x07bb] = 0x5aa8,
  [0x07bc] = 0x5aa9, [0x07bd] = 0x5aab, [0x07be] = 0x5aac, [0x07c0] = 0x5aad,
  [0x07c1] = 0x5aae, [0x07c2] = 0x5aaf, [0x07c3] = 0x5ab0, [0x07c4] = 0x5ab1,
  [0x07c5] = 0x5ab4, [0x07c6] = 0x5ab6, [0x07c7] = 0x5ab7, [0x07c8] = 0x5ab9,
  [0x07c9] = 0x5aba, [0x07ca] = 0x5abb, [0x07cb] = 0x5abc, [0x07cc] = 0x5abd,
  [0x07cd] = 0x5abf, [0x07ce] = 0x5ac0, [0x07cf] = 0x5ac3, [0x07d0] = 0x5ac4,
  [0x07d1] = 0x5ac5, [0x07d2] = 0x5ac6, [0x07d3] = 0x5ac7, [0x07d4] = 0x5ac8,
  [0x07d5] = 0x5aca, [0x07d6] = 0x5acb, [0x07d7] = 0x5acd, [0x07d8] = 0x5ace,
  [0x07d9] = 0x5acf, [0x07da] = 0x5ad0, [0x07db] = 0x5ad1, [0x07dc] = 0x5ad3,
  [0x07dd] = 0x5ad5, [0x07de] = 0x5ad7, [0x07df] = 0x5ad9, [0x07e0] = 0x5ada,
  [0x07e1] = 0x5adb, [0x07e2] = 0x5add, [0x07e3] = 0x5ade, [0x07e4] = 0x5adf,
  [0x07e5] = 0x5ae2, [0x07e6] = 0x5ae4, [0x07e7] = 0x5ae5, [0x07e8] = 0x5ae7,
  [0x07e9] = 0x5ae8, [0x07ea] = 0x5aea, [0x07eb] = 0x5aec, [0x07ec] = 0x5aed,
  [0x07ed] = 0x5aee, [0x07ee] = 0x5aef, [0x07ef] = 0x5af0, [0x07f0] = 0x5af2,
  [0x07f1] = 0x5af3, [0x07f2] = 0x5af4, [0x07f3] = 0x5af5, [0x07f4] = 0x5af6,
  [0x07f5] = 0x5af7, [0x07f6] = 0x5af8, [0x07f7] = 0x5af9, [0x07f8] = 0x5afa,
  [0x07f9] = 0x5afb, [0x07fa] = 0x5afc, [0x07fb] = 0x5afd, [0x07fc] = 0x5afe,
  [0x07fd] = 0x5aff, [0x07fe] = 0x5b00, [0x07ff] = 0x5b01, [0x0800] = 0x5b02,
  [0x0801] = 0x5b03, [0x0802] = 0x5b04, [0x0803] = 0x5b05, [0x0804] = 0x5b06,
  [0x0805] = 0x5b07, [0x0806] = 0x5b08, [0x0807] = 0x5b0a, [0x0808] = 0x5b0b,
  [0x0809] = 0x5b0c, [0x080a] = 0x5b0d, [0x080b] = 0x5b0e, [0x080c] = 0x5b0f,
  [0x080d] = 0x5b10, [0x080e] = 0x5b11, [0x080f] = 0x5b12, [0x0810] = 0x5b13,
  [0x0811] = 0x5b14, [0x0812] = 0x5b15, [0x0813] = 0x5b18, [0x0814] = 0x5b19,
  [0x0815] = 0x5b1a, [0x0816] = 0x5b1b, [0x0817] = 0x5b1c, [0x0818] = 0x5b1d,
  [0x0819] = 0x5b1e, [0x081a] = 0x5b1f, [0x081b] = 0x5b20, [0x081c] = 0x5b21,
  [0x081d] = 0x5b22, [0x081e] = 0x5b23, [0x081f] = 0x5b24, [0x0820] = 0x5b25,
  [0x0821] = 0x5b26, [0x0822] = 0x5b27, [0x0823] = 0x5b28, [0x0824] = 0x5b29,
  [0x0825] = 0x5b2a, [0x0826] = 0x5b2b, [0x0827] = 0x5b2c, [0x0828] = 0x5b2d,
  [0x0829] = 0x5b2e, [0x082a] = 0x5b2f, [0x082b] = 0x5b30, [0x082c] = 0x5b31,
  [0x082d] = 0x5b33, [0x082e] = 0x5b35, [0x082f] = 0x5b36, [0x0830] = 0x5b38,
  [0x0831] = 0x5b39, [0x0832] = 0x5b3a, [0x0833] = 0x5b3b, [0x0834] = 0x5b3c,
  [0x0835] = 0x5b3d, [0x0836] = 0x5b3e, [0x0837] = 0x5b3f, [0x0838] = 0x5b41,
  [0x0839] = 0x5b42, [0x083a] = 0x5b43, [0x083b] = 0x5b44, [0x083c] = 0x5b45,
  [0x083d] = 0x5b46, [0x083e] = 0x5b47, [0x0840] = 0x5b48, [0x0841] = 0x5b49,
  [0x0842] = 0x5b4a, [0x0843] = 0x5b4b, [0x0844] = 0x5b4c, [0x0845] = 0x5b4d,
  [0x0846] = 0x5b4e, [0x0847] = 0x5b4f, [0x0848] = 0x5b52, [0x0849] = 0x5b56,
  [0x084a] = 0x5b5e, [0x084b] = 0x5b60, [0x084c] = 0x5b61, [0x084d] = 0x5b67,
  [0x084e] = 0x5b68, [0x084f] = 0x5b6b, [0x0850] = 0x5b6d, [0x0851] = 0x5b6e,
  [0x0852] = 0x5b6f, [0x0853] = 0x5b72, [0x0854] = 0x5b74, [0x0855] = 0x5b76,
  [0x0856] = 0x5b77, [0x0857] = 0x5b78, [0x0858] = 0x5b79, [0x0859] = 0x5b7b,
  [0x085a] = 0x5b7c, [0x085b] = 0x5b7e, [0x085c] = 0x5b7f, [0x085d] = 0x5b82,
  [0x085e] = 0x5b86, [0x085f] = 0x5b8a, [0x0860] = 0x5b8d, [0x0861] = 0x5b8e,
  [0x0862] = 0x5b90, [0x0863] = 0x5b91, [0x0864] = 0x5b92, [0x0865] = 0x5b94,
  [0x0866] = 0x5b96, [0x0867] = 0x5b9f, [0x0868] = 0x5ba7, [0x0869] = 0x5ba8,
  [0x086a] = 0x5ba9, [0x086b] = 0x5bac, [0x086c] = 0x5bad, [0x086d] = 0x5bae,
  [0x086e] = 0x5baf, [0x086f] = 0x5bb1, [0x0870] = 0x5bb2, [0x0871] = 0x5bb7,
  [0x0872] = 0x5bba, [0x0873] = 0x5bbb, [0x0874] = 0x5bbc, [0x0875] = 0x5bc0,
  [0x0876] = 0x5bc1, [0x0877] = 0x5bc3, [0x0878] = 0x5bc8, [0x0879] = 0x5bc9,
  [0x087a] = 0x5bca, [0x087b] = 0x5bcb, [0x087c] = 0x5bcd, [0x087d] = 0x5bce,
  [0x087e] = 0x5bcf, [0x0880] = 0x5bd1, [0x0881] = 0x5bd4, [0x0882] = 0x5bd5,
  [0x0883] = 0x5bd6, [0x0884] = 0x5bd7, [0x0885] = 0x5bd8, [0x0886] = 0x5bd9,
  [0x0887] = 0x5bda, [0x0888] = 0x5bdb, [0x0889] = 0x5bdc, [0x088a] = 0x5be0,
  [0x088b] = 0x5be2, [0x088c] = 0x5be3, [0x088d] = 0x5be6, [0x088e] = 0x5be7,
  [0x088f] = 0x5be9, [0x0890] = 0x5bea, [0x0891] = 0x5beb, [0x0892] = 0x5bec,
  [0x0893] = 0x5bed, [0x0894] = 0x5bef, [0x0895] = 0x5bf1, [0x0896] = 0x5bf2,
  [0x0897] = 0x5bf3, [0x0898] = 0x5bf4, [0x0899] = 0x5bf5, [0x089a] = 0x5bf6,
  [0x089b] = 0x5bf7, [0x089c] = 0x5bfd, [0x089d] = 0x5bfe, [0x089e] = 0x5c00,
  [0x089f] = 0x5c02, [0x08a0] = 0x5c03, [0x08a1] = 0x5c05, [0x08a2] = 0x5c07,
  [0x08a3] = 0x5c08, [0x08a4] = 0x5c0b, [0x08a5] = 0x5c0c, [0x08a6] = 0x5c0d,
  [0x08a7] = 0x5c0e, [0x08a8] = 0x5c10, [0x08a9] = 0x5c12, [0x08aa] = 0x5c13,
  [0x08ab] = 0x5c17, [0x08ac] = 0x5c19, [0x08ad] = 0x5c1b, [0x08ae] = 0x5c1e,
  [0x08af] = 0x5c1f, [0x08b0] = 0x5c20, [0x08b1] = 0x5c21, [0x08b2] = 0x5c23,
  [0x08b3] = 0x5c26, [0x08b4] = 0x5c28, [0x08b5] = 0x5c29, [0x08b6] = 0x5c2a,
  [0x08b7] = 0x5c2b, [0x08b8] = 0x5c2d, [0x08b9] = 0x5c2e, [0x08ba] = 0x5c2f,
  [0x08bb] = 0x5c30, [0x08bc] = 0x5c32, [0x08bd] = 0x5c33, [0x08be] = 0x5c35,
  [0x08bf] = 0x5c36, [0x08c0] = 0x5c37, [0x08c1] = 0x5c43, [0x08c2] = 0x5c44,
  [0x08c3] = 0x5c46, [0x08c4] = 0x5c47, [0x08c5] = 0x5c4c, [0x08c6] = 0x5c4d,
  [0x08c7] = 0x5c52, [0x08c8] = 0x5c53, [0x08c9] = 0x5c54, [0x08ca] = 0x5c56,
  [0x08cb] = 0x5c57, [0x08cc] = 0x5c58, [0x08cd] = 0x5c5a, [0x08ce] = 0x5c5b,
  [0x08cf] = 0x5c5c, [0x08d0] = 0x5c5d, [0x08d1] = 0x5c5f, [0x08d2] = 0x5c62,
  [0x08d3] = 0x5c64, [0x08d4] = 0x5c67, [0x08d5] = 0x5c68, [0x08d6] = 0x5c69,
  [0x08d7] = 0x5c6a, [0x08d8] = 0x5c6b, [0x08d9] = 0x5c6c, [0x08da] = 0x5c6d,
  [0x08db] = 0x5c70, [0x08dc] = 0x5c72, [0x08dd] = 0x5c73, [0x08de] = 0x5c74,
  [0x08df] = 0x5c75, [0x08e0] = 0x5c76, [0x08e1] = 0x5c77, [0x08e2] = 0x5c78,
  [0x08e3] = 0x5c7b, [0x08e4] = 0x5c7c, [0x08e5] = 0x5c7d, [0x08e6] = 0x5c7e,
  [0x08e7] = 0x5c80, [0x08e8] = 0x5c83, [0x08e9] = 0x5c84, [0x08ea] = 0x5c85,
  [0x08eb] = 0x5c86, [0x08ec] = 0x5c87, [0x08ed] = 0x5c89, [0x08ee] = 0x5c8a,
  [0x08ef] = 0x5c8b, [0x08f0] = 0x5c8e, [0x08f1] = 0x5c8f, [0x08f2] = 0x5c92,
  [0x08f3] = 0x5c93, [0x08f4] = 0x5c95, [0x08f5] = 0x5c9d, [0x08f6] = 0x5c9e,
  [0x08f7] = 0x5c9f, [0x08f8] = 0x5ca0, [0x08f9] = 0x5ca1, [0x08fa] = 0x5ca4,
  [0x08fb] = 0x5ca5, [0x08fc] = 0x5ca6, [0x08fd] = 0x5ca7, [0x08fe] = 0x5ca8,
  [0x0900] = 0x5caa, [0x0901] = 0x5cae, [0x0902] = 0x5caf, [0x0903] = 0x5cb0,
  [0x0904] = 0x5cb2, [0x0905] = 0x5cb4, [0x0906] = 0x5cb6, [0x0907] = 0x5cb9,
  [0x0908] = 0x5cba, [0x0909] = 0x5cbb, [0x090a] = 0x5cbc, [0x090b] = 0x5cbe,
  [0x090c] = 0x5cc0, [0x090d] = 0x5cc2, [0x090e] = 0x5cc3, [0x090f] = 0x5cc5,
  [0x0910] = 0x5cc6, [0x0911] = 0x5cc7, [0x0912] = 0x5cc8, [0x0913] = 0x5cc9,
  [0x0914] = 0x5cca, [0x0915] = 0x5ccc, [0x0916] = 0x5ccd, [0x0917] = 0x5cce,
  [0x0918] = 0x5ccf, [0x0919] = 0x5cd0, [0x091a] = 0x5cd1, [0x091b] = 0x5cd3,
  [0x091c] = 0x5cd4, [0x091d] = 0x5cd5, [0x091e] = 0x5cd6, [0x091f] = 0x5cd7,
  [0x0920] = 0x5cd8, [0x0921] = 0x5cda, [0x0922] = 0x5cdb, [0x0923] = 0x5cdc,
  [0x0924] = 0x5cdd, [0x0925] = 0x5cde, [0x0926] = 0x5cdf, [0x0927] = 0x5ce0,
  [0x0928] = 0x5ce2, [0x0929] = 0x5ce3, [0x092a] = 0x5ce7, [0x092b] = 0x5ce9,
  [0x092c] = 0x5ceb, [0x092d] = 0x5cec, [0x092e] = 0x5cee, [0x092f] = 0x5cef,
  [0x0930] = 0x5cf1, [0x0931] = 0x5cf2, [0x0932] = 0x5cf3, [0x0933] = 0x5cf4,
  [0x0934] = 0x5cf5, [0x0935] = 0x5cf6, [0x0936] = 0x5cf7, [0x0937] = 0x5cf8,
  [0x0938] = 0x5cf9, [0x0939] = 0x5cfa, [0x093a] = 0x5cfc, [0x093b] = 0x5cfd,
  [0x093c] = 0x5cfe, [0x093d] = 0x5cff, [0x093e] = 0x5d00, [0x0940] = 0x5d01,
  [0x0941] = 0x5d04, [0x0942] = 0x5d05, [0x0943] = 0x5d08, [0x0944] = 0x5d09,
  [0x0945] = 0x5d0a, [0x0946] = 0x5d0b, [0x0947] = 0x5d0c, [0x0948] = 0x5d0d,
  [0x0949] = 0x5d0f, [0x094a] = 0x5d10, [0x094b] = 0x5d11, [0x094c] = 0x5d12,
  [0x094d] = 0x5d13, [0x094e] = 0x5d15, [0x094f] = 0x5d17, [0x0950] = 0x5d18,
  [0x0951] = 0x5d19, [0x0952] = 0x5d1a, [0x0953] = 0x5d1c, [0x0954] = 0x5d1d,
  [0x0955] = 0x5d1f, [0x0956] = 0x5d20, [0x0957] = 0x5d21, [0x0958] = 0x5d22,
  [0x0959] = 0x5d23, [0x095a] = 0x5d25, [0x095b] = 0x5d28, [0x095c] = 0x5d2a,
  [0x095d] = 0x5d2b, [0x095e] = 0x5d2c, [0x095f] = 0x5d2f, [0x0960] = 0x5d30,
  [0x0961] = 0x5d31, [0x0962] = 0x5d32, [0x0963] = 0x5d33, [0x0964] = 0x5d35,
  [0x0965] = 0x5d36, [0x0966] = 0x5d37, [0x0967] = 0x5d38, [0x0968] = 0x5d39,
  [0x0969] = 0x5d3a, [0x096a] = 0x5d3b, [0x096b] = 0x5d3c, [0x096c] = 0x5d3f,
  [0x096d] = 0x5d40, [0x096e] = 0x5d41, [0x096f] = 0x5d42, [0x0970] = 0x5d43,
  [0x0971] = 0x5d44, [0x0972] = 0x5d45, [0x0973] = 0x5d46, [0x0974] = 0x5d48,
  [0x0975] = 0x5d49, [0x0976] = 0x5d4d, [0x0977] = 0x5d4e, [0x0978] = 0x5d4f,
  [0x0979] = 0x5d50, [0x097a] = 0x5d51, [0x097b] = 0x5d52, [0x097c] = 0x5d53,
  [0x097d] = 0x5d54, [0x097e] = 0x5d55, [0x097f] = 0x5d56, [0x0980] = 0x5d57,
  [0x0981] = 0x5d59, [0x0982] = 0x5d5a, [0x0983] = 0x5d5c, [0x0984] = 0x5d5e,
  [0x0985] = 0x5d5f, [0x0986] = 0x5d60, [0x0987] = 0x5d61, [0x0988] = 0x5d62,
  [0x0989] = 0x5d63, [0x098a] = 0x5d64, [0x098b] = 0x5d65, [0x098c] = 0x5d66,
  [0x098d] = 0x5d67, [0x098e] = 0x5d68, [0x098f] = 0x5d6a, [0x0990] = 0x5d6d,
  [0x0991] = 0x5d6e, [0x0992] = 0x5d70, [0x0993] = 0x5d71, [0x0994] = 0x5d72,
  [0x0995] = 0x5d73, [0x0996] = 0x5d75, [0x0997] = 0x5d76, [0x0998] = 0x5d77,
  [0x0999] = 0x5d78, [0x099a] = 0x5d79, [0x099b] = 0x5d7a, [0x099c] = 0x5d7b,
  [0x099d] = 0x5d7c, [0x099e] = 0x5d7d, [0x099f] = 0x5d7e, [0x09a0] = 0x5d7f,
  [0x09a1] = 0x5d80, [0x09a2] = 0x5d81, [0x09a3] = 0x5d83, [0x09a4] = 0x5d84,
  [0x09a5] = 0x5d85, [0x09a6] = 0x5d86, [0x09a7] = 0x5d87, [0x09a8] = 0x5d88,
  [0x09a9] = 0x5d89, [0x09aa] = 0x5d8a, [0x09ab] = 0x5d8b, [0x09ac] = 0x5d8c,
  [0x09ad] = 0x5d8d, [0x09ae] = 0x5d8e, [0x09af] = 0x5d8f, [0x09b0] = 0x5d90,
  [0x09b1] = 0x5d91, [0x09b2] = 0x5d92, [0x09b3] = 0x5d93, [0x09b4] = 0x5d94,
  [0x09b5] = 0x5d95, [0x09b6] = 0x5d96, [0x09b7] = 0x5d97, [0x09b8] = 0x5d98,
  [0x09b9] = 0x5d9a, [0x09ba] = 0x5d9b, [0x09bb] = 0x5d9c, [0x09bc] = 0x5d9e,
  [0x09bd] = 0x5d9f, [0x09be] = 0x5da0, [0x09c0] = 0x5da1, [0x09c1] = 0x5da2,
  [0x09c2] = 0x5da3, [0x09c3] = 0x5da4, [0x09c4] = 0x5da5, [0x09c5] = 0x5da6,
  [0x09c6] = 0x5da7, [0x09c7] = 0x5da8, [0x09c8] = 0x5da9, [0x09c9] = 0x5daa,
  [0x09ca] = 0x5dab, [0x09cb] = 0x5dac, [0x09cc] = 0x5dad, [0x09cd] = 0x5dae,
  [0x09ce] = 0x5daf, [0x09cf] = 0x5db0, [0x09d0] = 0x5db1, [0x09d1] = 0x5db2,
  [0x09d2] = 0x5db3, [0x09d3] = 0x5db4, [0x09d4] = 0x5db5, [0x09d5] = 0x5db6,
  [0x09d6] = 0x5db8, [0x09d7] = 0x5db9, [0x09d8] = 0x5dba, [0x09d9] = 0x5dbb,
  [0x09da] = 0x5dbc, [0x09db] = 0x5dbd, [0x09dc] = 0x5dbe, [0x09dd] = 0x5dbf,
  [0x09de] = 0x5dc0, [0x09df] = 0x5dc1, [0x09e0] = 0x5dc2, [0x09e1] = 0x5dc3,
  [0x09e2] = 0x5dc4, [0x09e3] = 0x5dc6, [0x09e4] = 0x5dc7, [0x09e5] = 0x5dc8,
  [0x09e6] = 0x5dc9, [0x09e7] = 0x5dca, [0x09e8] = 0x5dcb, [0x09e9] = 0x5dcc,
  [0x09ea] = 0x5dce, [0x09eb] = 0x5dcf, [0x09ec] = 0x5dd0, [0x09ed] = 0x5dd1,
  [0x09ee] = 0x5dd2, [0x09ef] = 0x5dd3, [0x09f0] = 0x5dd4, [0x09f1] = 0x5dd5,
  [0x09f2] = 0x5dd6, [0x09f3] = 0x5dd7, [0x09f4] = 0x5dd8, [0x09f5] = 0x5dd9,
  [0x09f6] = 0x5dda, [0x09f7] = 0x5ddc, [0x09f8] = 0x5ddf, [0x09f9] = 0x5de0,
  [0x09fa] = 0x5de3, [0x09fb] = 0x5de4, [0x09fc] = 0x5dea, [0x09fd] = 0x5dec,
  [0x09fe] = 0x5ded, [0x0a00] = 0x5df0, [0x0a01] = 0x5df5, [0x0a02] = 0x5df6,
  [0x0a03] = 0x5df8, [0x0a04] = 0x5df9, [0x0a05] = 0x5dfa, [0x0a06] = 0x5dfb,
  [0x0a07] = 0x5dfc, [0x0a08] = 0x5dff, [0x0a09] = 0x5e00, [0x0a0a] = 0x5e04,
  [0x0a0b] = 0x5e07, [0x0a0c] = 0x5e09, [0x0a0d] = 0x5e0a, [0x0a0e] = 0x5e0b,
  [0x0a0f] = 0x5e0d, [0x0a10] = 0x5e0e, [0x0a11] = 0x5e12, [0x0a12] = 0x5e13,
  [0x0a13] = 0x5e17, [0x0a14] = 0x5e1e, [0x0a15] = 0x5e1f, [0x0a16] = 0x5e20,
  [0x0a17] = 0x5e21, [0x0a18] = 0x5e22, [0x0a19] = 0x5e23, [0x0a1a] = 0x5e24,
  [0x0a1b] = 0x5e25, [0x0a1c] = 0x5e28, [0x0a1d] = 0x5e29, [0x0a1e] = 0x5e2a,
  [0x0a1f] = 0x5e2b, [0x0a20] = 0x5e2c, [0x0a21] = 0x5e2f, [0x0a22] = 0x5e30,
  [0x0a23] = 0x5e32, [0x0a24] = 0x5e33, [0x0a25] = 0x5e34, [0x0a26] = 0x5e35,
  [0x0a27] = 0x5e36, [0x0a28] = 0x5e39, [0x0a29] = 0x5e3a, [0x0a2a] = 0x5e3e,
  [0x0a2b] = 0x5e3f, [0x0a2c] = 0x5e40, [0x0a2d] = 0x5e41, [0x0a2e] = 0x5e43,
  [0x0a2f] = 0x5e46, [0x0a30] = 0x5e47, [0x0a31] = 0x5e48, [0x0a32] = 0x5e49,
  [0x0a33] = 0x5e4a, [0x0a34] = 0x5e4b, [0x0a35] = 0x5e4d, [0x0a36] = 0x5e4e,
  [0x0a37] = 0x5e4f, [0x0a38] = 0x5e50, [0x0a39] = 0x5e51, [0x0a3a] = 0x5e52,
  [0x0a3b] = 0x5e53, [0x0a3c] = 0x5e56, [0x0a3d] = 0x5e57, [0x0a3e] = 0x5e58,
  [0x0a3f] = 0x5e59, [0x0a40] = 0x5e5a, [0x0a41] = 0x5e5c, [0x0a42] = 0x5e5d,
  [0x0a43] = 0x5e5f, [0x0a44] = 0x5e60, [0x0a45] = 0x5e63, [0x0a46] = 0x5e64,
  [0x0a47] = 0x5e65, [0x0a48] = 0x5e66, [0x0a49] = 0x5e67, [0x0a4a] = 0x5e68,
  [0x0a4b] = 0x5e69, [0x0a4c] = 0x5e6a, [0x0a4d] = 0x5e6b, [0x0a4e] = 0x5e6c,
  [0x0a4f] = 0x5e6d, [0x0a50] = 0x5e6e, [0x0a51] = 0x5e6f, [0x0a52] = 0x5e70,
  [0x0a53] = 0x5e71, [0x0a54] = 0x5e75, [0x0a55] = 0x5e77, [0x0a56] = 0x5e79,
  [0x0a57] = 0x5e7e, [0x0a58] = 0x5e81, [0x0a59] = 0x5e82, [0x0a5a] = 0x5e83,
  [0x0a5b] = 0x5e85, [0x0a5c] = 0x5e88, [0x0a5d] = 0x5e89, [0x0a5e] = 0x5e8c,
  [0x0a5f] = 0x5e8d, [0x0a60] = 0x5e8e, [0x0a61] = 0x5e92, [0x0a62] = 0x5e98,
  [0x0a63] = 0x5e9b, [0x0a64] = 0x5e9d, [0x0a65] = 0x5ea1, [0x0a66] = 0x5ea2,
  [0x0a67] = 0x5ea3, [0x0a68] = 0x5ea4, [0x0a69] = 0x5ea8, [0x0a6a] = 0x5ea9,
  [0x0a6b] = 0x5eaa, [0x0a6c] = 0x5eab, [0x0a6d] = 0x5eac, [0x0a6e] = 0x5eae,
  [0x0a6f] = 0x5eaf, [0x0a70] = 0x5eb0, [0x0a71] = 0x5eb1, [0x0a72] = 0x5eb2,
  [0x0a73] = 0x5eb4, [0x0a74] = 0x5eba, [0x0a75] = 0x5ebb, [0x0a76] = 0x5ebc,
  [0x0a77] = 0x5ebd, [0x0a78] = 0x5ebf, [0x0a79] = 0x5ec0, [0x0a7a] = 0x5ec1,
  [0x0a7b] = 0x5ec2, [0x0a7c] = 0x5ec3, [0x0a7d] = 0x5ec4, [0x0a7e] = 0x5ec5,
  [0x0a80] = 0x5ec6, [0x0a81] = 0x5ec7, [0x0a82] = 0x5ec8, [0x0a83] = 0x5ecb,
  [0x0a84] = 0x5ecc, [0x0a85] = 0x5ecd, [0x0a86] = 0x5ece, [0x0a87] = 0x5ecf,
  [0x0a88] = 0x5ed0, [0x0a89] = 0x5ed4, [0x0a8a] = 0x5ed5, [0x0a8b] = 0x5ed7,
  [0x0a8c] = 0x5ed8, [0x0a8d] = 0x5ed9, [0x0a8e] = 0x5eda, [0x0a8f] = 0x5edc,
  [0x0a90] = 0x5edd, [0x0a91] = 0x5ede, [0x0a92] = 0x5edf, [0x0a93] = 0x5ee0,
  [0x0a94] = 0x5ee1, [0x0a95] = 0x5ee2, [0x0a96] = 0x5ee3, [0x0a97] = 0x5ee4,
  [0x0a98] = 0x5ee5, [0x0a99] = 0x5ee6, [0x0a9a] = 0x5ee7, [0x0a9b] = 0x5ee9,
  [0x0a9c] = 0x5eeb, [0x0a9d] = 0x5eec, [0x0a9e] = 0x5eed, [0x0a9f] = 0x5eee,
  [0x0aa0] = 0x5eef, [0x0aa1] = 0x5ef0, [0x0aa2] = 0x5ef1, [0x0aa3] = 0x5ef2,
  [0x0aa4] = 0x5ef3, [0x0aa5] = 0x5ef5, [0x0aa6] = 0x5ef8, [0x0aa7] = 0x5ef9,
  [0x0aa8] = 0x5efb, [0x0aa9] = 0x5efc, [0x0aaa] = 0x5efd, [0x0aab] = 0x5f05,
  [0x0aac] = 0x5f06, [0x0aad] = 0x5f07, [0x0aae] = 0x5f09, [0x0aaf] = 0x5f0c,
  [0x0ab0] = 0x5f0d, [0x0ab1] = 0x5f0e, [0x0ab2] = 0x5f10, [0x0ab3] = 0x5f12,
  [0x0ab4] = 0x5f14, [0x0ab5] = 0x5f16, [0x0ab6] = 0x5f19, [0x0ab7] = 0x5f1a,
  [0x0ab8] = 0x5f1c, [0x0ab9] = 0x5f1d, [0x0aba] = 0x5f1e, [0x0abb] = 0x5f21,
  [0x0abc] = 0x5f22, [0x0abd] = 0x5f23, [0x0abe] = 0x5f24, [0x0ac0] = 0x5f28,
  [0x0ac1] = 0x5f2b, [0x0ac2] = 0x5f2c, [0x0ac3] = 0x5f2e, [0x0ac4] = 0x5f30,
  [0x0ac5] = 0x5f32, [0x0ac6] = 0x5f33, [0x0ac7] = 0x5f34, [0x0ac8] = 0x5f35,
  [0x0ac9] = 0x5f36, [0x0aca] = 0x5f37, [0x0acb] = 0x5f38, [0x0acc] = 0x5f3b,
  [0x0acd] = 0x5f3d, [0x0ace] = 0x5f3e, [0x0acf] = 0x5f3f, [0x0ad0] = 0x5f41,
  [0x0ad1] = 0x5f42, [0x0ad2] = 0x5f43, [0x0ad3] = 0x5f44, [0x0ad4] = 0x5f45,
  [0x0ad5] = 0x5f46, [0x0ad6] = 0x5f47, [0x0ad7] = 0x5f48, [0x0ad8] = 0x5f49,
  [0x0ad9] = 0x5f4a, [0x0ada] = 0x5f4b, [0x0adb] = 0x5f4c, [0x0adc] = 0x5f4d,
  [0x0add] = 0x5f4e, [0x0ade] = 0x5f4f, [0x0adf] = 0x5f51, [0x0ae0] = 0x5f54,
  [0x0ae1] = 0x5f59, [0x0ae2] = 0x5f5a, [0x0ae3] = 0x5f5b, [0x0ae4] = 0x5f5c,
  [0x0ae5] = 0x5f5e, [0x0ae6] = 0x5f5f, [0x0ae7] = 0x5f60, [0x0ae8] = 0x5f63,
  [0x0ae9] = 0x5f65, [0x0aea] = 0x5f67, [0x0aeb] = 0x5f68, [0x0aec] = 0x5f6b,
  [0x0aed] = 0x5f6e, [0x0aee] = 0x5f6f, [0x0aef] = 0x5f72, [0x0af0] = 0x5f74,
  [0x0af1] = 0x5f75, [0x0af2] = 0x5f76, [0x0af3] = 0x5f78, [0x0af4] = 0x5f7a,
  [0x0af5] = 0x5f7d, [0x0af6] = 0x5f7e, [0x0af7] = 0x5f7f, [0x0af8] = 0x5f83,
  [0x0af9] = 0x5f86, [0x0afa] = 0x5f8d, [0x0afb] = 0x5f8e, [0x0afc] = 0x5f8f,
  [0x0afd] = 0x5f91, [0x0afe] = 0x5f93, [0x0aff] = 0x5f94, [0x0b00] = 0x5f96,
  [0x0b01] = 0x5f9a, [0x0b02] = 0x5f9b, [0x0b03] = 0x5f9d, [0x0b04] = 0x5f9e,
  [0x0b05] = 0x5f9f, [0x0b06] = 0x5fa0, [0x0b07] = 0x5fa2, [0x0b08] = 0x5fa3,
  [0x0b09] = 0x5fa4, [0x0b0a] = 0x5fa5, [0x0b0b] = 0x5fa6, [0x0b0c] = 0x5fa7,
  [0x0b0d] = 0x5fa9, [0x0b0e] = 0x5fab, [0x0b0f] = 0x5fac, [0x0b10] = 0x5faf,
  [0x0b11] = 0x5fb0, [0x0b12] = 0x5fb1, [0x0b13] = 0x5fb2, [0x0b14] = 0x5fb3,
  [0x0b15] = 0x5fb4, [0x0b16] = 0x5fb6, [0x0b17] = 0x5fb8, [0x0b18] = 0x5fb9,
  [0x0b19] = 0x5fba, [0x0b1a] = 0x5fbb, [0x0b1b] = 0x5fbe, [0x0b1c] = 0x5fbf,
  [0x0b1d] = 0x5fc0, [0x0b1e] = 0x5fc1, [0x0b1f] = 0x5fc2, [0x0b20] = 0x5fc7,
  [0x0b21] = 0x5fc8, [0x0b22] = 0x5fca, [0x0b23] = 0x5fcb, [0x0b24] = 0x5fce,
  [0x0b25] = 0x5fd3, [0x0b26] = 0x5fd4, [0x0b27] = 0x5fd5, [0x0b28] = 0x5fda,
  [0x0b29] = 0x5fdb, [0x0b2a] = 0x5fdc, [0x0b2b] = 0x5fde, [0x0b2c] = 0x5fdf,
  [0x0b2d] = 0x5fe2, [0x0b2e] = 0x5fe3, [0x0b2f] = 0x5fe5, [0x0b30] = 0x5fe6,
  [0x0b31] = 0x5fe8, [0x0b32] = 0x5fe9, [0x0b33] = 0x5fec, [0x0b34] = 0x5fef,
  [0x0b35] = 0x5ff0, [0x0b36] = 0x5ff2, [0x0b37] = 0x5ff3, [0x0b38] = 0x5ff4,
  [0x0b39] = 0x5ff6, [0x0b3a] = 0x5ff7, [0x0b3b] = 0x5ff9, [0x0b3c] = 0x5ffa,
  [0x0b3d] = 0x5ffc, [0x0b3e] = 0x6007, [0x0b40] = 0x6008, [0x0b41] = 0x6009,
  [0x0b42] = 0x600b, [0x0b43] = 0x600c, [0x0b44] = 0x6010, [0x0b45] = 0x6011,
  [0x0b46] = 0x6013, [0x0b47] = 0x6017, [0x0b48] = 0x6018, [0x0b49] = 0x601a,
  [0x0b4a] = 0x601e, [0x0b4b] = 0x601f, [0x0b4c] = 0x6022, [0x0b4d] = 0x6023,
  [0x0b4e] = 0x6024, [0x0b4f] = 0x602c, [0x0b50] = 0x602d, [0x0b51] = 0x602e,
  [0x0b52] = 0x6030, [0x0b53] = 0x6031, [0x0b54] = 0x6032, [0x0b55] = 0x6033,
  [0x0b56] = 0x6034, [0x0b57] = 0x6036, [0x0b58] = 0x6037, [0x0b59] = 0x6038,
  [0x0b5a] = 0x6039, [0x0b5b] = 0x603a, [0x0b5c] = 0x603d, [0x0b5d] = 0x603e,
  [0x0b5e] = 0x6040, [0x0b5f] = 0x6044, [0x0b60] = 0x6045, [0x0b61] = 0x6046,
  [0x0b62] = 0x6047, [0x0b63] = 0x6048, [0x0b64] = 0x6049, [0x0b65] = 0x604a,
  [0x0b66] = 0x604c, [0x0b67] = 0x604e, [0x0b68] = 0x604f, [0x0b69] = 0x6051,
  [0x0b6a] = 0x6053, [0x0b6b] = 0x6054, [0x0b6c] = 0x6056, [0x0b6d] = 0x6057,
  [0x0b6e] = 0x6058, [0x0b6f] = 0x605b, [0x0b70] = 0x605c, [0x0b71] = 0x605e,
  [0x0b72] = 0x605f, [0x0b73] = 0x6060, [0x0b74] = 0x6061, [0x0b75] = 0x6065,
  [0x0b76] = 0x6066, [0x0b77] = 0x606e, [0x0b78] = 0x6071, [0x0b79] = 0x6072,
  [0x0b7a] = 0x6074, [0x0b7b] = 0x6075, [0x0b7c] = 0x6077, [0x0b7d] = 0x607e,
  [0x0b7e] = 0x6080, [0x0b80] = 0x6081, [0x0b81] = 0x6082, [0x0b82] = 0x6085,
  [0x0b83] = 0x6086, [0x0b84] = 0x6087, [0x0b85] = 0x6088, [0x0b86] = 0x608a,
  [0x0b87] = 0x608b, [0x0b88] = 0x608e, [0x0b89] = 0x608f, [0x0b8a] = 0x6090,
  [0x0b8b] = 0x6091, [0x0b8c] = 0x6093, [0x0b8d] = 0x6095, [0x0b8e] = 0x6097,
  [0x0b8f] = 0x6098, [0x0b90] = 0x6099, [0x0b91] = 0x609c, [0x0b92] = 0x609e,
  [0x0b93] = 0x60a1, [0x0b94] = 0x60a2, [0x0b95] = 0x60a4, [0x0b96] = 0x60a5,
  [0x0b97] = 0x60a7, [0x0b98] = 0x60a9, [0x0b99] = 0x60aa, [0x0b9a] = 0x60ae,
  [0x0b9b] = 0x60b0, [0x0b9c] = 0x60b3, [0x0b9d] = 0x60b5, [0x0b9e] = 0x60b6,
  [0x0b9f] = 0x60b7, [0x0ba0] = 0x60b9, [0x0ba1] = 0x60ba, [0x0ba2] = 0x60bd,
  [0x0ba3] = 0x60be, [0x0ba4] = 0x60bf, [0x0ba5] = 0x60c0, [0x0ba6] = 0x60c1,
  [0x0ba7] = 0x60c2, [0x0ba8] = 0x60c3, [0x0ba9] = 0x60c4, [0x0baa] = 0x60c7,
  [0x0bab] = 0x60c8, [0x0bac] = 0x60c9, [0x0bad] = 0x60cc, [0x0bae] = 0x60cd,
  [0x0baf] = 0x60ce, [0x0bb0] = 0x60cf, [0x0bb1] = 0x60d0, [0x0bb2] = 0x60d2,
  [0x0bb3] = 0x60d3, [0x0bb4] = 0x60d4, [0x0bb5] = 0x60d6, [0x0bb6] = 0x60d7,
  [0x0bb7] = 0x60d9, [0x0bb8] = 0x60db, [0x0bb9] = 0x60de, [0x0bba] = 0x60e1,
  [0x0bbb] = 0x60e2, [0x0bbc] = 0x60e3, [0x0bbd] = 0x60e4, [0x0bbe] = 0x60e5,
  [0x0bbf] = 0x60ea, [0x0bc0] = 0x60f1, [0x0bc1] = 0x60f2, [0x0bc2] = 0x60f5,
  [0x0bc3] = 0x60f7, [0x0bc4] = 0x60f8, [0x0bc5] = 0x60fb, [0x0bc6] = 0x60fc,
  [0x0bc7] = 0x60fd, [0x0bc8] = 0x60fe, [0x0bc9] = 0x60ff, [0x0bca] = 0x6102,
  [0x0bcb] = 0x6103, [0x0bcc] = 0x6104, [0x0bcd] = 0x6105, [0x0bce] = 0x6107,
  [0x0bcf] = 0x610a, [0x0bd0] = 0x610b, [0x0bd1] = 0x610c, [0x0bd2] = 0x6110,
  [0x0bd3] = 0x6111, [0x0bd4] = 0x6112, [0x0bd5] = 0x6113, [0x0bd6] = 0x6114,
  [0x0bd7] = 0x6116, [0x0bd8] = 0x6117, [0x0bd9] = 0x6118, [0x0bda] = 0x6119,
  [0x0bdb] = 0x611b, [0x0bdc] = 0x611c, [0x0bdd] = 0x611d, [0x0bde] = 0x611e,
  [0x0bdf] = 0x6121, [0x0be0] = 0x6122, [0x0be1] = 0x6125, [0x0be2] = 0x6128,
  [0x0be3] = 0x6129, [0x0be4] = 0x612a, [0x0be5] = 0x612c, [0x0be6] = 0x612d,
  [0x0be7] = 0x612e, [0x0be8] = 0x612f, [0x0be9] = 0x6130, [0x0bea] = 0x6131,
  [0x0beb] = 0x6132, [0x0bec] = 0x6133, [0x0bed] = 0x6134, [0x0bee] = 0x6135,
  [0x0bef] = 0x6136, [0x0bf0] = 0x6137, [0x0bf1] = 0x6138, [0x0bf2] = 0x6139,
  [0x0bf3] = 0x613a, [0x0bf4] = 0x613b, [0x0bf5] = 0x613c, [0x0bf6] = 0x613d,
  [0x0bf7] = 0x613e, [0x0bf8] = 0x6140, [0x0bf9] = 0x6141, [0x0bfa] = 0x6142,
  [0x0bfb] = 0x6143, [0x0bfc] = 0x6144, [0x0bfd] = 0x6145, [0x0bfe] = 0x6146,
  [0x0c00] = 0x6147, [0x0c01] = 0x6149, [0x0c02] = 0x614b, [0x0c03] = 0x614d,
  [0x0c04] = 0x614f, [0x0c05] = 0x6150, [0x0c06] = 0x6152, [0x0c07] = 0x6153,
  [0x0c08] = 0x6154, [0x0c09] = 0x6156, [0x0c0a] = 0x6157, [0x0c0b] = 0x6158,
  [0x0c0c] = 0x6159, [0x0c0d] = 0x615a, [0x0c0e] = 0x615b, [0x0c0f] = 0x615c,
  [0x0c10] = 0x615e, [0x0c11] = 0x615f, [0x0c12] = 0x6160, [0x0c13] = 0x6161,
  [0x0c14] = 0x6163, [0x0c15] = 0x6164, [0x0c16] = 0x6165, [0x0c17] = 0x6166,
  [0x0c18] = 0x6169, [0x0c19] = 0x616a, [0x0c1a] = 0x616b, [0x0c1b] = 0x616c,
  [0x0c1c] = 0x616d, [0x0c1d] = 0x616e, [0x0c1e] = 0x616f, [0x0c1f] = 0x6171,
  [0x0c20] = 0x6172, [0x0c21] = 0x6173, [0x0c22] = 0x6174, [0x0c23] = 0x6176,
  [0x0c24] = 0x6178, [0x0c25] = 0x6179, [0x0c26] = 0x617a, [0x0c27] = 0x617b,
  [0x0c28] = 0x617c, [0x0c29] = 0x617d, [0x0c2a] = 0x617e, [0x0c2b] = 0x617f,
  [0x0c2c] = 0x6180, [0x0c2d] = 0x6181, [0x0c2e] = 0x6182, [0x0c2f] = 0x6183,
  [0x0c30] = 0x6184, [0x0c31] = 0x6185, [0x0c32] = 0x6186, [0x0c33] = 0x6187,
  [0x0c34] = 0x6188, [0x0c35] = 0x6189, [0x0c36] = 0x618a, [0x0c37] = 0x618c,
  [0x0c38] = 0x618d, [0x0c39] = 0x618f, [0x0c3a] = 0x6190, [0x0c3b] = 0x6191,
  [0x0c3c] = 0x6192, [0x0c3d] = 0x6193, [0x0c3e] = 0x6195, [0x0c40] = 0x6196,
  [0x0c41] = 0x6197, [0x0c42] = 0x6198, [0x0c43] = 0x6199, [0x0c44] = 0x619a,
  [0x0c45] = 0x619b, [0x0c46] = 0x619c, [0x0c47] = 0x619e, [0x0c48] = 0x619f,
  [0x0c49] = 0x61a0, [0x0c4a] = 0x61a1, [0x0c4b] = 0x61a2, [0x0c4c] = 0x61a3,
  [0x0c4d] = 0x61a4, [0x0c4e] = 0x61a5, [0x0c4f] = 0x61a6, [0x0c50] = 0x61aa,
  [0x0c51] = 0x61ab, [0x0c52] = 0x61ad, [0x0c53] = 0x61ae, [0x0c54] = 0x61af,
  [0x0c55] = 0x61b0, [0x0c56] = 0x61b1, [0x0c57] = 0x61b2, [0x0c58] = 0x61b3,
  [0x0c59] = 0x61b4, [0x0c5a] = 0x61b5, [0x0c5b] = 0x61b6, [0x0c5c] = 0x61b8,
  [0x0c5d] = 0x61b9, [0x0c5e] = 0x61ba, [0x0c5f] = 0x61bb, [0x0c60] = 0x61bc,
  [0x0c61] = 0x61bd, [0x0c62] = 0x61bf, [0x0c63] = 0x61c0, [0x0c64] = 0x61c1,
  [0x0c65] = 0x61c3, [0x0c66] = 0x61c4, [0x0c67] = 0x61c5, [0x0c68] = 0x61c6,
  [0x0c69] = 0x61c7, [0x0c6a] = 0x61c9, [0x0c6b] = 0x61cc, [0x0c6c] = 0x61cd,
  [0x0c6d] = 0x61ce, [0x0c6e] = 0x61cf, [0x0c6f] = 0x61d0, [0x0c70] = 0x61d3,
  [0x0c71] = 0x61d5, [0x0c72] = 0x61d6, [0x0c73] = 0x61d7, [0x0c74] = 0x61d8,
  [0x0c75] = 0x61d9, [0x0c76] = 0x61da, [0x0c77] = 0x61db, [0x0c78] = 0x61dc,
  [0x0c79] = 0x61dd, [0x0c7a] = 0x61de, [0x0c7b] = 0x61df, [0x0c7c] = 0x61e0,
  [0x0c7d] = 0x61e1, [0x0c7e] = 0x61e2, [0x0c7f] = 0x61e3, [0x0c80] = 0x61e4,
  [0x0c81] = 0x61e5, [0x0c82] = 0x61e7, [0x0c83] = 0x61e8, [0x0c84] = 0x61e9,
  [0x0c85] = 0x61ea, [0x0c86] = 0x61eb, [0x0c87] = 0x61ec, [0x0c88] = 0x61ed,
  [0x0c89] = 0x61ee, [0x0c8a] = 0x61ef, [0x0c8b] = 0x61f0, [0x0c8c] = 0x61f1,
  [0x0c8d] = 0x61f2, [0x0c8e] = 0x61f3, [0x0c8f] = 0x61f4, [0x0c90] = 0x61f6,
  [0x0c91] = 0x61f7, [0x0c92] = 0x61f8, [0x0c93] = 0x61f9, [0x0c94] = 0x61fa,
  [0x0c95] = 0x61fb, [0x0c96] = 0x61fc, [0x0c97] = 0x61fd, [0x0c98] = 0x61fe,
  [0x0c99] = 0x6200, [0x0c9a] = 0x6201, [0x0c9b] = 0x6202, [0x0c9c] = 0x6203,
  [0x0c9d] = 0x6204, [0x0c9e] = 0x6205, [0x0c9f] = 0x6207, [0x0ca0] = 0x6209,
  [0x0ca1] = 0x6213, [0x0ca2] = 0x6214, [0x0ca3] = 0x6219, [0x0ca4] = 0x621c,
  [0x0ca5] = 0x621d, [0x0ca6] = 0x621e, [0x0ca7] = 0x6220, [0x0ca8] = 0x6223,
  [0x0ca9] = 0x6226, [0x0caa] = 0x6227, [0x0cab] = 0x6228, [0x0cac] = 0x6229,
  [0x0cad] = 0x622b, [0x0cae] = 0x622d, [0x0caf] = 0x622f, [0x0cb0] = 0x6230,
  [0x0cb1] = 0x6231, [0x0cb2] = 0x6232, [0x0cb3] = 0x6235, [0x0cb4] = 0x6236,
  [0x0cb5] = 0x6238, [0x0cb6] = 0x6239, [0x0cb7] = 0x623a, [0x0cb8] = 0x623b,
  [0x0cb9] = 0x623c, [0x0cba] = 0x6242, [0x0cbb] = 0x6244, [0x0cbc] = 0x6245,
  [0x0cbd] = 0x6246, [0x0cbe] = 0x624a, [0x0cc0] = 0x624f, [0x0cc1] = 0x6250,
  [0x0cc2] = 0x6255, [0x0cc3] = 0x6256, [0x0cc4] = 0x6257, [0x0cc5] = 0x6259,
  [0x0cc6] = 0x625a, [0x0cc7] = 0x625c, [0x0cc8] = 0x625d, [0x0cc9] = 0x625e,
  [0x0cca] = 0x625f, [0x0ccb] = 0x6260, [0x0ccc] = 0x6261, [0x0ccd] = 0x6262,
  [0x0cce] = 0x6264, [0x0ccf] = 0x6265, [0x0cd0] = 0x6268, [0x0cd1] = 0x6271,
  [0x0cd2] = 0x6272, [0x0cd3] = 0x6274, [0x0cd4] = 0x6275, [0x0cd5] = 0x6277,
  [0x0cd6] = 0x6278, [0x0cd7] = 0x627a, [0x0cd8] = 0x627b, [0x0cd9] = 0x627d,
  [0x0cda] = 0x6281, [0x0cdb] = 0x6282, [0x0cdc] = 0x6283, [0x0cdd] = 0x6285,
  [0x0cde] = 0x6286, [0x0cdf] = 0x6287, [0x0ce0] = 0x6288, [0x0ce1] = 0x628b,
  [0x0ce2] = 0x628c, [0x0ce3] = 0x628d, [0x0ce4] = 0x628e, [0x0ce5] = 0x628f,
  [0x0ce6] = 0x6290, [0x0ce7] = 0x6294, [0x0ce8] = 0x6299, [0x0ce9] = 0x629c,
  [0x0cea] = 0x629d, [0x0ceb] = 0x629e, [0x0cec] = 0x62a3, [0x0ced] = 0x62a6,
  [0x0cee] = 0x62a7, [0x0cef] = 0x62a9, [0x0cf0] = 0x62aa, [0x0cf1] = 0x62ad,
  [0x0cf2] = 0x62ae, [0x0cf3] = 0x62af, [0x0cf4] = 0x62b0, [0x0cf5] = 0x62b2,
  [0x0cf6] = 0x62b3, [0x0cf7] = 0x62b4, [0x0cf8] = 0x62b6, [0x0cf9] = 0x62b7,
  [0x0cfa] = 0x62b8, [0x0cfb] = 0x62ba, [0x0cfc] = 0x62be, [0x0cfd] = 0x62c0,
  [0x0cfe] = 0x62c1, [0x0d00] = 0x62c3, [0x0d01] = 0x62cb, [0x0d02] = 0x62cf,
  [0x0d03] = 0x62d1, [0x0d04] = 0x62d5, [0x0d05] = 0x62dd, [0x0d06] = 0x62de,
  [0x0d07] = 0x62e0, [0x0d08] = 0x62e1, [0x0d09] = 0x62e4, [0x0d0a] = 0x62ea,
  [0x0d0b] = 0x62eb, [0x0d0c] = 0x62f0, [0x0d0d] = 0x62f2, [0x0d0e] = 0x62f5,
  [0x0d0f] = 0x62f8, [0x0d10] = 0x62f9, [0x0d11] = 0x62fa, [0x0d12] = 0x62fb,
  [0x0d13] = 0x6300, [0x0d14] = 0x6303, [0x0d15] = 0x6304, [0x0d16] = 0x6305,
  [0x0d17] = 0x6306, [0x0d18] = 0x630a, [0x0d19] = 0x630b, [0x0d1a] = 0x630c,
  [0x0d1b] = 0x630d, [0x0d1c] = 0x630f, [0x0d1d] = 0x6310, [0x0d1e] = 0x6312,
  [0x0d1f] = 0x6313, [0x0d20] = 0x6314, [0x0d21] = 0x6315, [0x0d22] = 0x6317,
  [0x0d23] = 0x6318, [0x0d24] = 0x6319, [0x0d25] = 0x631c, [0x0d26] = 0x6326,
  [0x0d27] = 0x6327, [0x0d28] = 0x6329, [0x0d29] = 0x632c, [0x0d2a] = 0x632d,
  [0x0d2b] = 0x632e, [0x0d2c] = 0x6330, [0x0d2d] = 0x6331, [0x0d2e] = 0x6333,
  [0x0d2f] = 0x6334, [0x0d30] = 0x6335, [0x0d31] = 0x6336, [0x0d32] = 0x6337,
  [0x0d33] = 0x6338, [0x0d34] = 0x633b, [0x0d35] = 0x633c, [0x0d36] = 0x633e,
  [0x0d37] = 0x633f, [0x0d38] = 0x6340, [0x0d39] = 0x6341, [0x0d3a] = 0x6344,
  [0x0d3b] = 0x6347, [0x0d3c] = 0x6348, [0x0d3d] = 0x634a, [0x0d3e] = 0x6351,
  [0x0d3f] = 0x6352, [0x0d40] = 0x6353, [0x0d41] = 0x6354, [0x0d42] = 0x6356,
  [0x0d43] = 0x6357, [0x0d44] = 0x6358, [0x0d45] = 0x6359, [0x0d46] = 0x635a,
  [0x0d47] = 0x635b, [0x0d48] = 0x635c, [0x0d49] = 0x635d, [0x0d4a] = 0x6360,
  [0x0d4b] = 0x6364, [0x0d4c] = 0x6365, [0x0d4d] = 0x6366, [0x0d4e] = 0x6368,
  [0x0d4f] = 0x636a, [0x0d50] = 0x636b, [0x0d51] = 0x636c, [0x0d52] = 0x636f,
  [0x0d53] = 0x6370, [0x0d54] = 0x6372, [0x0d55] = 0x6373, [0x0d56] = 0x6374,
  [0x0d57] = 0x6375, [0x0d58] = 0x6378, [0x0d59] = 0x6379, [0x0d5a] = 0x637c,
  [0x0d5b] = 0x637d, [0x0d5c] = 0x637e, [0x0d5d] = 0x637f, [0x0d5e] = 0x6381,
  [0x0d5f] = 0x6383, [0x0d60] = 0x6384, [0x0d61] = 0x6385, [0x0d62] = 0x6386,
  [0x0d63] = 0x638b, [0x0d64] = 0x638d, [0x0d65] = 0x6391, [0x0d66] = 0x6393,
  [0x0d67] = 0x6394, [0x0d68] = 0x6395, [0x0d69] = 0x6397, [0x0d6a] = 0x6399,
  [0x0d6b] = 0x639a, [0x0d6c] = 0x639b, [0x0d6d] = 0x639c, [0x0d6e] = 0x639d,
  [0x0d6f] = 0x639e, [0x0d70] = 0x639f, [0x0d71] = 0x63a1, [0x0d72] = 0x63a4,
  [0x0d73] = 0x63a6, [0x0d74] = 0x63ab, [0x0d75] = 0x63af, [0x0d76] = 0x63b1,
  [0x0d77] = 0x63b2, [0x0d78] = 0x63b5, [0x0d79] = 0x63b6, [0x0d7a] = 0x63b9,
  [0x0d7b] = 0x63bb, [0x0d7c] = 0x63bd, [0x0d7d] = 0x63bf, [0x0d7e] = 0x63c0,
  [0x0d80] = 0x63c1, [0x0d81] = 0x63c2, [0x0d82] = 0x63c3, [0x0d83] = 0x63c5,
  [0x0d84] = 0x63c7, [0x0d85] = 0x63c8, [0x0d86] = 0x63ca, [0x0d87] = 0x63cb,
  [0x0d88] = 0x63cc, [0x0d89] = 0x63d1, [0x0d8a] = 0x63d3, [0x0d8b] = 0x63d4,
  [0x0d8c] = 0x63d5, [0x0d8d] = 0x63d7, [0x0d8e] = 0x63d8, [0x0d8f] = 0x63d9,
  [0x0d90] = 0x63da, [0x0d91] = 0x63db, [0x0d92] = 0x63dc, [0x0d93] = 0x63dd,
  [0x0d94] = 0x63df, [0x0d95] = 0x63e2, [0x0d96] = 0x63e4, [0x0d97] = 0x63e5,
  [0x0d98] = 0x63e6, [0x0d99] = 0x63e7, [0x0d9a] = 0x63e8, [0x0d9b] = 0x63eb,
  [0x0d9c] = 0x63ec, [0x0d9d] = 0x63ee, [0x0d9e] = 0x63ef, [0x0d9f] = 0x63f0,
  [0x0da0] = 0x63f1, [0x0da1] = 0x63f3, [0x0da2] = 0x63f5, [0x0da3] = 0x63f7,
  [0x0da4] = 0x63f9, [0x0da5] = 0x63fa, [0x0da6] = 0x63fb, [0x0da7] = 0x63fc,
  [0x0da8] = 0x63fe, [0x0da9] = 0x6403, [0x0daa] = 0x6404, [0x0dab] = 0x6406,
  [0x0dac] = 0x6407, [0x0dad] = 0x6408, [0x0dae] = 0x6409, [0x0daf] = 0x640a,
  [0x0db0] = 0x640d, [0x0db1] = 0x640e, [0x0db2] = 0x6411, [0x0db3] = 0x6412,
  [0x0db4] = 0x6415, [0x0db5] = 0x6416, [0x0db6] = 0x6417, [0x0db7] = 0x6418,
  [0x0db8] = 0x6419, [0x0db9] = 0x641a, [0x0dba] = 0x641d, [0x0dbb] = 0x641f,
  [0x0dbc] = 0x6422, [0x0dbd] = 0x6423, [0x0dbe] = 0x6424, [0x0dc0] = 0x6425,
  [0x0dc1] = 0x6427, [0x0dc2] = 0x6428, [0x0dc3] = 0x6429, [0x0dc4] = 0x642b,
  [0x0dc5] = 0x642e, [0x0dc6] = 0x642f, [0x0dc7] = 0x6430, [0x0dc8] = 0x6431,
  [0x0dc9] = 0x6432, [0x0dca] = 0x6433, [0x0dcb] = 0x6435, [0x0dcc] = 0x6436,
  [0x0dcd] = 0x6437, [0x0dce] = 0x6438, [0x0dcf] = 0x6439, [0x0dd0] = 0x643b,
  [0x0dd1] = 0x643c, [0x0dd2] = 0x643e, [0x0dd3] = 0x6440, [0x0dd4] = 0x6442,
  [0x0dd5] = 0x6443, [0x0dd6] = 0x6449, [0x0dd7] = 0x644b, [0x0dd8] = 0x644c,
  [0x0dd9] = 0x644d, [0x0dda] = 0x644e, [0x0ddb] = 0x644f, [0x0ddc] = 0x6450,
  [0x0ddd] = 0x6451, [0x0dde] = 0x6453, [0x0ddf] = 0x6455, [0x0de0] = 0x6456,
  [0x0de1] = 0x6457, [0x0de2] = 0x6459, [0x0de3] = 0x645a, [0x0de4] = 0x645b,
  [0x0de5] = 0x645c, [0x0de6] = 0x645d, [0x0de7] = 0x645f, [0x0de8] = 0x6460,
  [0x0de9] = 0x6461, [0x0dea] = 0x6462, [0x0deb] = 0x6463, [0x0dec] = 0x6464,
  [0x0ded] = 0x6465, [0x0dee] = 0x6466, [0x0def] = 0x6468, [0x0df0] = 0x646a,
  [0x0df1] = 0x646b, [0x0df2] = 0x646c, [0x0df3] = 0x646e, [0x0df4] = 0x646f,
  [0x0df5] = 0x6470, [0x0df6] = 0x6471, [0x0df7] = 0x6472, [0x0df8] = 0x6473,
  [0x0df9] = 0x6474, [0x0dfa] = 0x6475, [0x0dfb] = 0x6476, [0x0dfc] = 0x6477,
  [0x0dfd] = 0x647b, [0x0dfe] = 0x647c, [0x0dff] = 0x647d, [0x0e00] = 0x647e,
  [0x0e01] = 0x647f, [0x0e02] = 0x6480, [0x0e03] = 0x6481, [0x0e04] = 0x6483,
  [0x0e05] = 0x6486, [0x0e06] = 0x6488, [0x0e07] = 0x6489, [0x0e08] = 0x648a,
  [0x0e09] = 0x648b, [0x0e0a] = 0x648c, [0x0e0b] = 0x648d, [0x0e0c] = 0x648e,
  [0x0e0d] = 0x648f, [0x0e0e] = 0x6490, [0x0e0f] = 0x6493, [0x0e10] = 0x6494,
  [0x0e11] = 0x6497, [0x0e12] = 0x6498, [0x0e13] = 0x649a, [0x0e14] = 0x649b,
  [0x0e15] = 0x649c, [0x0e16] = 0x649d, [0x0e17] = 0x649f, [0x0e18] = 0x64a0,
  [0x0e19] = 0x64a1, [0x0e1a] = 0x64a2, [0x0e1b] = 0x64a3, [0x0e1c] = 0x64a5,
  [0x0e1d] = 0x64a6, [0x0e1e] = 0x64a7, [0x0e1f] = 0x64a8, [0x0e20] = 0x64aa,
  [0x0e21] = 0x64ab, [0x0e22] = 0x64af, [0x0e23] = 0x64b1, [0x0e24] = 0x64b2,
  [0x0e25] = 0x64b3, [0x0e26] = 0x64b4, [0x0e27] = 0x64b6, [0x0e28] = 0x64b9,
  [0x0e29] = 0x64bb, [0x0e2a] = 0x64bd, [0x0e2b] = 0x64be, [0x0e2c] = 0x64bf,
  [0x0e2d] = 0x64c1, [0x0e2e] = 0x64c3, [0x0e2f] = 0x64c4, [0x0e30] = 0x64c6,
  [0x0e31] = 0x64c7, [0x0e32] = 0x64c8, [0x0e33] = 0x64c9, [0x0e34] = 0x64ca,
  [0x0e35] = 0x64cb, [0x0e36] = 0x64cc, [0x0e37] = 0x64cf, [0x0e38] = 0x64d1,
  [0x0e39] = 0x64d3, [0x0e3a] = 0x64d4, [0x0e3b] = 0x64d5, [0x0e3c] = 0x64d6,
  [0x0e3d] = 0x64d9, [0x0e3e] = 0x64da, [0x0e40] = 0x64db, [0x0e41] = 0x64dc,
  [0x0e42] = 0x64dd, [0x0e43] = 0x64df, [0x0e44] = 0x64e0, [0x0e45] = 0x64e1,
  [0x0e46] = 0x64e3, [0x0e47] = 0x64e5, [0x0e48] = 0x64e7, [0x0e49] = 0x64e8,
  [0x0e4a] = 0x64e9, [0x0e4b] = 0x64ea, [0x0e4c] = 0x64eb, [0x0e4d] = 0x64ec,
  [0x0e4e] = 0x64ed, [0x0e4f] = 0x64ee, [0x0e50] = 0x64ef, [0x0e51] = 0x64f0,
  [0x0e52] = 0x64f1, [0x0e53] = 0x64f2, [0x0e54] = 0x64f3, [0x0e55] = 0x64f4,
  [0x0e56] = 0x64f5, [0x0e57] = 0x64f6, [0x0e58] = 0x64f7, [0x0e59] = 0x64f8,
  [0x0e5a] = 0x64f9, [0x0e5b] = 0x64fa, [0x0e5c] = 0x64fb, [0x0e5d] = 0x64fc,
  [0x0e5e] = 0x64fd, [0x0e5f] = 0x64fe, [0x0e60] = 0x64ff, [0x0e61] = 0x6501,
  [0x0e62] = 0x6502, [0x0e63] = 0x6503, [0x0e64] = 0x6504, [0x0e65] = 0x6505,
  [0x0e66] = 0x6506, [0x0e67] = 0x6507, [0x0e68] = 0x6508, [0x0e69] = 0x650a,
  [0x0e6a] = 0x650b, [0x0e6b] = 0x650c, [0x0e6c] = 0x650d, [0x0e6d] = 0x650e,
  [0x0e6e] = 0x650f, [0x0e6f] = 0x6510, [0x0e70] = 0x6511, [0x0e71] = 0x6513,
  [0x0e72] = 0x6514, [0x0e73] = 0x6515, [0x0e74] = 0x6516, [0x0e75] = 0x6517,
  [0x0e76] = 0x6519, [0x0e77] = 0x651a, [0x0e78] = 0x651b, [0x0e79] = 0x651c,
  [0x0e7a] = 0x651d, [0x0e7b] = 0x651e, [0x0e7c] = 0x651f, [0x0e7d] = 0x6520,
  [0x0e7e] = 0x6521, [0x0e80] = 0x6522, [0x0e81] = 0x6523, [0x0e82] = 0x6524,
  [0x0e83] = 0x6526, [0x0e84] = 0x6527, [0x0e85] = 0x6528, [0x0e86] = 0x6529,
  [0x0e87] = 0x652a, [0x0e88] = 0x652c, [0x0e89] = 0x652d, [0x0e8a] = 0x6530,
  [0x0e8b] = 0x6531, [0x0e8c] = 0x6532, [0x0e8d] = 0x6533, [0x0e8e] = 0x6537,
  [0x0e8f] = 0x653a, [0x0e90] = 0x653c, [0x0e91] = 0x653d, [0x0e92] = 0x6540,
  [0x0e93] = 0x6541, [0x0e94] = 0x6542, [0x0e95] = 0x6543, [0x0e96] = 0x6544,
  [0x0e97] = 0x6546, [0x0e98] = 0x6547, [0x0e99] = 0x654a, [0x0e9a] = 0x654b,
  [0x0e9b] = 0x654d, [0x0e9c] = 0x654e, [0x0e9d] = 0x6550, [0x0e9e] = 0x6552,
  [0x0e9f] = 0x6553, [0x0ea0] = 0x6554, [0x0ea1] = 0x6557, [0x0ea2] = 0x6558,
  [0x0ea3] = 0x655a, [0x0ea4] = 0x655c, [0x0ea5] = 0x655f, [0x0ea6] = 0x6560,
  [0x0ea7] = 0x6561, [0x0ea8] = 0x6564, [0x0ea9] = 0x6565, [0x0eaa] = 0x6567,
  [0x0eab] = 0x6568, [0x0eac] = 0x6569, [0x0ead] = 0x656a, [0x0eae] = 0x656d,
  [0x0eaf] = 0x656e, [0x0eb0] = 0x656f, [0x0eb1] = 0x6571, [0x0eb2] = 0x6573,
  [0x0eb3] = 0x6575, [0x0eb4] = 0x6576, [0x0eb5] = 0x6578, [0x0eb6] = 0x6579,
  [0x0eb7] = 0x657a, [0x0eb8] = 0x657b, [0x0eb9] = 0x657c, [0x0eba] = 0x657d,
  [0x0ebb] = 0x657e, [0x0ebc] = 0x657f, [0x0ebd] = 0x6580, [0x0ebe] = 0x6581,
  [0x0ebf] = 0x6582, [0x0ec0] = 0x6583, [0x0ec1] = 0x6584, [0x0ec2] = 0x6585,
  [0x0ec3] = 0x6586, [0x0ec4] = 0x6588, [0x0ec5] = 0x6589, [0x0ec6] = 0x658a,
  [0x0ec7] = 0x658d, [0x0ec8] = 0x658e, [0x0ec9] = 0x658f, [0x0eca] = 0x6592,
  [0x0ecb] = 0x6594, [0x0ecc] = 0x6595, [0x0ecd] = 0x6596, [0x0ece] = 0x6598,
  [0x0ecf] = 0x659a, [0x0ed0] = 0x659d, [0x0ed1] = 0x659e, [0x0ed2] = 0x65a0,
  [0x0ed3] = 0x65a2, [0x0ed4] = 0x65a3, [0x0ed5] = 0x65a6, [0x0ed6] = 0x65a8,
  [0x0ed7] = 0x65aa, [0x0ed8] = 0x65ac, [0x0ed9] = 0x65ae, [0x0eda] = 0x65b1,
  [0x0edb] = 0x65b2, [0x0edc] = 0x65b3, [0x0edd] = 0x65b4, [0x0ede] = 0x65b5,
  [0x0edf] = 0x65b6, [0x0ee0] = 0x65b7, [0x0ee1] = 0x65b8, [0x0ee2] = 0x65ba,
  [0x0ee3] = 0x65bb, [0x0ee4] = 0x65be, [0x0ee5] = 0x65bf, [0x0ee6] = 0x65c0,
  [0x0ee7] = 0x65c2, [0x0ee8] = 0x65c7, [0x0ee9] = 0x65c8, [0x0eea] = 0x65c9,
  [0x0eeb] = 0x65ca, [0x0eec] = 0x65cd, [0x0eed] = 0x65d0, [0x0eee] = 0x65d1,
  [0x0eef] = 0x65d3, [0x0ef0] = 0x65d4, [0x0ef1] = 0x65d5, [0x0ef2] = 0x65d8,
  [0x0ef3] = 0x65d9, [0x0ef4] = 0x65da, [0x0ef5] = 0x65db, [0x0ef6] = 0x65dc,
  [0x0ef7] = 0x65dd, [0x0ef8] = 0x65de, [0x0ef9] = 0x65df, [0x0efa] = 0x65e1,
  [0x0efb] = 0x65e3, [0x0efc] = 0x65e4, [0x0efd] = 0x65ea, [0x0efe] = 0x65eb,
  [0x0f00] = 0x65f2, [0x0f01] = 0x65f3, [0x0f02] = 0x65f4, [0x0f03] = 0x65f5,
  [0x0f04] = 0x65f8, [0x0f05] = 0x65f9, [0x0f06] = 0x65fb, [0x0f07] = 0x65fc,
  [0x0f08] = 0x65fd, [0x0f09] = 0x65fe, [0x0f0a] = 0x65ff, [0x0f0b] = 0x6601,
  [0x0f0c] = 0x6604, [0x0f0d] = 0x6605, [0x0f0e] = 0x6607, [0x0f0f] = 0x6608,
  [0x0f10] = 0x6609, [0x0f11] = 0x660b, [0x0f12] = 0x660d, [0x0f13] = 0x6610,
  [0x0f14] = 0x6611, [0x0f15] = 0x6612, [0x0f16] = 0x6616, [0x0f17] = 0x6617,
  [0x0f18] = 0x6618, [0x0f19] = 0x661a, [0x0f1a] = 0x661b, [0x0f1b] = 0x661c,
  [0x0f1c] = 0x661e, [0x0f1d] = 0x6621, [0x0f1e] = 0x6622, [0x0f1f] = 0x6623,
  [0x0f20] = 0x6624, [0x0f21] = 0x6626, [0x0f22] = 0x6629, [0x0f23] = 0x662a,
  [0x0f24] = 0x662b, [0x0f25] = 0x662c, [0x0f26] = 0x662e, [0x0f27] = 0x6630,
  [0x0f28] = 0x6632, [0x0f29] = 0x6633, [0x0f2a] = 0x6637, [0x0f2b] = 0x6638,
  [0x0f2c] = 0x6639, [0x0f2d] = 0x663a, [0x0f2e] = 0x663b, [0x0f2f] = 0x663d,
  [0x0f30] = 0x663f, [0x0f31] = 0x6640, [0x0f32] = 0x6642, [0x0f33] = 0x6644,
  [0x0f34] = 0x6645, [0x0f35] = 0x6646, [0x0f36] = 0x6647, [0x0f37] = 0x6648,
  [0x0f38] = 0x6649, [0x0f39] = 0x664a, [0x0f3a] = 0x664d, [0x0f3b] = 0x664e,
  [0x0f3c] = 0x6650, [0x0f3d] = 0x6651, [0x0f3e] = 0x6658, [0x0f40] = 0x6659,
  [0x0f41] = 0x665b, [0x0f42] = 0x665c, [0x0f43] = 0x665d, [0x0f44] = 0x665e,
  [0x0f45] = 0x6660, [0x0f46] = 0x6662, [0x0f47] = 0x6663, [0x0f48] = 0x6665,
  [0x0f49] = 0x6667, [0x0f4a] = 0x6669, [0x0f4b] = 0x666a, [0x0f4c] = 0x666b,
  [0x0f4d] = 0x666c, [0x0f4e] = 0x666d, [0x0f4f] = 0x6671, [0x0f50] = 0x6672,
  [0x0f51] = 0x6673, [0x0f52] = 0x6675, [0x0f53] = 0x6678, [0x0f54] = 0x6679,
  [0x0f55] = 0x667b, [0x0f56] = 0x667c, [0x0f57] = 0x667d, [0x0f58] = 0x667f,
  [0x0f59] = 0x6680, [0x0f5a] = 0x6681, [0x0f5b] = 0x6683, [0x0f5c] = 0x6685,
  [0x0f5d] = 0x6686, [0x0f5e] = 0x6688, [0x0f5f] = 0x6689, [0x0f60] = 0x668a,
  [0x0f61] = 0x668b, [0x0f62] = 0x668d, [0x0f63] = 0x668e, [0x0f64] = 0x668f,
  [0x0f65] = 0x6690, [0x0f66] = 0x6692, [0x0f67] = 0x6693, [0x0f68] = 0x6694,
  [0x0f69] = 0x6695, [0x0f6a] = 0x6698, [0x0f6b] = 0x6699, [0x0f6c] = 0x669a,
  [0x0f6d] = 0x669b, [0x0f6e] = 0x669c, [0x0f6f] = 0x669e, [0x0f70] = 0x669f,
  [0x0f71] = 0x66a0, [0x0f72] = 0x66a1, [0x0f73] = 0x66a2, [0x0f74] = 0x66a3,
  [0x0f75] = 0x66a4, [0x0f76] = 0x66a5, [0x0f77] = 0x66a6, [0x0f78] = 0x66a9,
  [0x0f79] = 0x66aa, [0x0f7a] = 0x66ab, [0x0f7b] = 0x66ac, [0x0f7c] = 0x66ad,
  [0x0f7d] = 0x66af, [0x0f7e] = 0x66b0, [0x0f7f] = 0x66b1, [0x0f80] = 0x66b2,
  [0x0f81] = 0x66b3, [0x0f82] = 0x66b5, [0x0f83] = 0x66b6, [0x0f84] = 0x66b7,
  [0x0f85] = 0x66b8, [0x0f86] = 0x66ba, [0x0f87] = 0x66bb, [0x0f88] = 0x66bc,
  [0x0f89] = 0x66bd, [0x0f8a] = 0x66bf, [0x0f8b] = 0x66c0, [0x0f8c] = 0x66c1,
  [0x0f8d] = 0x66c2, [0x0f8e] = 0x66c3, [0x0f8f] = 0x66c4, [0x0f90] = 0x66c5,
  [0x0f91] = 0x66c6, [0x0f92] = 0x66c7, [0x0f93] = 0x66c8, [0x0f94] = 0x66c9,
  [0x0f95] = 0x66ca, [0x0f96] = 0x66cb, [0x0f97] = 0x66cc, [0x0f98] = 0x66cd,
  [0x0f99] = 0x66ce, [0x0f9a] = 0x66cf, [0x0f9b] = 0x66d0, [0x0f9c] = 0x66d1,
  [0x0f9d] = 0x66d2, [0x0f9e] = 0x66d3, [0x0f9f] = 0x66d4, [0x0fa0] = 0x66d5,
  [0x0fa1] = 0x66d6, [0x0fa2] = 0x66d7, [0x0fa3] = 0x66d8, [0x0fa4] = 0x66da,
  [0x0fa5] = 0x66de, [0x0fa6] = 0x66df, [0x0fa7] = 0x66e0, [0x0fa8] = 0x66e1,
  [0x0fa9] = 0x66e2, [0x0faa] = 0x66e3, [0x0fab] = 0x66e4, [0x0fac] = 0x66e5,
  [0x0fad] = 0x66e7, [0x0fae] = 0x66e8, [0x0faf] = 0x66ea, [0x0fb0] = 0x66eb,
  [0x0fb1] = 0x66ec, [0x0fb2] = 0x66ed, [0x0fb3] = 0x66ee, [0x0fb4] = 0x66ef,
  [0x0fb5] = 0x66f1, [0x0fb6] = 0x66f5, [0x0fb7] = 0x66f6, [0x0fb8] = 0x66f8,
  [0x0fb9] = 0x66fa, [0x0fba] = 0x66fb, [0x0fbb] = 0x66fd, [0x0fbc] = 0x6701,
  [0x0fbd] = 0x6702, [0x0fbe] = 0x6703, [0x0fc0] = 0x6704, [0x0fc1] = 0x6705,
  [0x0fc2] = 0x6706, [0x0fc3] = 0x6707, [0x0fc4] = 0x670c, [0x0fc5] = 0x670e,
  [0x0fc6] = 0x670f, [0x0fc7] = 0x6711, [0x0fc8] = 0x6712, [0x0fc9] = 0x6713,
  [0x0fca] = 0x6716, [0x0fcb] = 0x6718, [0x0fcc] = 0x6719, [0x0fcd] = 0x671a,
  [0x0fce] = 0x671c, [0x0fcf] = 0x671e, [0x0fd0] = 0x6720, [0x0fd1] = 0x6721,
  [0x0fd2] = 0x6722, [0x0fd3] = 0x6723, [0x0fd4] = 0x6724, [0x0fd5] = 0x6725,
  [0x0fd6] = 0x6727, [0x0fd7] = 0x6729, [0x0fd8] = 0x672e, [0x0fd9] = 0x6730,
  [0x0fda] = 0x6732, [0x0fdb] = 0x6733, [0x0fdc] = 0x6736, [0x0fdd] = 0x6737,
  [0x0fde] = 0x6738, [0x0fdf] = 0x6739, [0x0fe0] = 0x673b, [0x0fe1] = 0x673c,
  [0x0fe2] = 0x673e, [0x0fe3] = 0x673f, [0x0fe4] = 0x6741, [0x0fe5] = 0x6744,
  [0x0fe6] = 0x6745, [0x0fe7] = 0x6747, [0x0fe8] = 0x674a, [0x0fe9] = 0x674b,
  [0x0fea] = 0x674d, [0x0feb] = 0x6752, [0x0fec] = 0x6754, [0x0fed] = 0x6755,
  [0x0fee] = 0x6757, [0x0fef] = 0x6758, [0x0ff0] = 0x6759, [0x0ff1] = 0x675a,
  [0x0ff2] = 0x675b, [0x0ff3] = 0x675d, [0x0ff4] = 0x6762, [0x0ff5] = 0x6763,
  [0x0ff6] = 0x6764, [0x0ff7] = 0x6766, [0x0ff8] = 0x6767, [0x0ff9] = 0x676b,
  [0x0ffa] = 0x676c, [0x0ffb] = 0x676e, [0x0ffc] = 0x6771, [0x0ffd] = 0x6774,
  [0x0ffe] = 0x6776, [0x1000] = 0x6778, [0x1001] = 0x6779, [0x1002] = 0x677a,
  [0x1003] = 0x677b, [0x1004] = 0x677d, [0x1005] = 0x6780, [0x1006] = 0x6782,
  [0x1007] = 0x6783, [0x1008] = 0x6785, [0x1009] = 0x6786, [0x100a] = 0x6788,
  [0x100b] = 0x678a, [0x100c] = 0x678c, [0x100d] = 0x678d, [0x100e] = 0x678e,
  [0x100f] = 0x678f, [0x1010] = 0x6791, [0x1011] = 0x6792, [0x1012] = 0x6793,
  [0x1013] = 0x6794, [0x1014] = 0x6796, [0x1015] = 0x6799, [0x1016] = 0x679b,
  [0x1017] = 0x679f, [0x1018] = 0x67a0, [0x1019] = 0x67a1, [0x101a] = 0x67a4,
  [0x101b] = 0x67a6, [0x101c] = 0x67a9, [0x101d] = 0x67ac, [0x101e] = 0x67ae,
  [0x101f] = 0x67b1, [0x1020] = 0x67b2, [0x1021] = 0x67b4, [0x1022] = 0x67b9,
  [0x1023] = 0x67ba, [0x1024] = 0x67bb, [0x1025] = 0x67bc, [0x1026] = 0x67bd,
  [0x1027] = 0x67be, [0x1028] = 0x67bf, [0x1029] = 0x67c0, [0x102a] = 0x67c2,
  [0x102b] = 0x67c5, [0x102c] = 0x67c6, [0x102d] = 0x67c7, [0x102e] = 0x67c8,
  [0x102f] = 0x67c9, [0x1030] = 0x67ca, [0x1031] = 0x67cb, [0x1032] = 0x67cc,
  [0x1033] = 0x67cd, [0x1034] = 0x67ce, [0x1035] = 0x67d5, [0x1036] = 0x67d6,
  [0x1037] = 0x67d7, [0x1038] = 0x67db, [0x1039] = 0x67df, [0x103a] = 0x67e1,
  [0x103b] = 0x67e3, [0x103c] = 0x67e4, [0x103d] = 0x67e6, [0x103e] = 0x67e7,
  [0x103f] = 0x67e8, [0x1040] = 0x67ea, [0x1041] = 0x67eb, [0x1042] = 0x67ed,
  [0x1043] = 0x67ee, [0x1044] = 0x67f2, [0x1045] = 0x67f5, [0x1046] = 0x67f6,
  [0x1047] = 0x67f7, [0x1048] = 0x67f8, [0x1049] = 0x67f9, [0x104a] = 0x67fa,
  [0x104b] = 0x67fb, [0x104c] = 0x67fc, [0x104d] = 0x67fe, [0x104e] = 0x6801,
  [0x104f] = 0x6802, [0x1050] = 0x6803, [0x1051] = 0x6804, [0x1052] = 0x6806,
  [0x1053] = 0x680d, [0x1054] = 0x6810, [0x1055] = 0x6812, [0x1056] = 0x6814,
  [0x1057] = 0x6815, [0x1058] = 0x6818, [0x1059] = 0x6819, [0x105a] = 0x681a,
  [0x105b] = 0x681b, [0x105c] = 0x681c, [0x105d] = 0x681e, [0x105e] = 0x681f,
  [0x105f] = 0x6820, [0x1060] = 0x6822, [0x1061] = 0x6823, [0x1062] = 0x6824,
  [0x1063] = 0x6825, [0x1064] = 0x6826, [0x1065] = 0x6827, [0x1066] = 0x6828,
  [0x1067] = 0x682b, [0x1068] = 0x682c, [0x1069] = 0x682d, [0x106a] = 0x682e,
  [0x106b] = 0x682f, [0x106c] = 0x6830, [0x106d] = 0x6831, [0x106e] = 0x6834,
  [0x106f] = 0x6835, [0x1070] = 0x6836, [0x1071] = 0x683a, [0x1072] = 0x683b,
  [0x1073] = 0x683f, [0x1074] = 0x6847, [0x1075] = 0x684b, [0x1076] = 0x684d,
  [0x1077] = 0x684f, [0x1078] = 0x6852, [0x1079] = 0x6856, [0x107a] = 0x6857,
  [0x107b] = 0x6858, [0x107c] = 0x6859, [0x107d] = 0x685a, [0x107e] = 0x685b,
  [0x1080] = 0x685c, [0x1081] = 0x685d, [0x1082] = 0x685e, [0x1083] = 0x685f,
  [0x1084] = 0x686a, [0x1085] = 0x686c, [0x1086] = 0x686d, [0x1087] = 0x686e,
  [0x1088] = 0x686f, [0x1089] = 0x6870, [0x108a] = 0x6871, [0x108b] = 0x6872,
  [0x108c] = 0x6873, [0x108d] = 0x6875, [0x108e] = 0x6878, [0x108f] = 0x6879,
  [0x1090] = 0x687a, [0x1091] = 0x687b, [0x1092] = 0x687c, [0x1093] = 0x687d,
  [0x1094] = 0x687e, [0x1095] = 0x687f, [0x1096] = 0x6880, [0x1097] = 0x6882,
  [0x1098] = 0x6884, [0x1099] = 0x6887, [0x109a] = 0x6888, [0x109b] = 0x6889,
  [0x109c] = 0x688a, [0x109d] = 0x688b, [0x109e] = 0x688c, [0x109f] = 0x688d,
  [0x10a0] = 0x688e, [0x10a1] = 0x6890, [0x10a2] = 0x6891, [0x10a3] = 0x6892,
  [0x10a4] = 0x6894, [0x10a5] = 0x6895, [0x10a6] = 0x6896, [0x10a7] = 0x6898,
  [0x10a8] = 0x6899, [0x10a9] = 0x689a, [0x10aa] = 0x689b, [0x10ab] = 0x689c,
  [0x10ac] = 0x689d, [0x10ad] = 0x689e, [0x10ae] = 0x689f, [0x10af] = 0x68a0,
  [0x10b0] = 0x68a1, [0x10b1] = 0x68a3, [0x10b2] = 0x68a4, [0x10b3] = 0x68a5,
  [0x10b4] = 0x68a9, [0x10b5] = 0x68aa, [0x10b6] = 0x68ab, [0x10b7] = 0x68ac,
  [0x10b8] = 0x68ae, [0x10b9] = 0x68b1, [0x10ba] = 0x68b2, [0x10bb] = 0x68b4,
  [0x10bc] = 0x68b6, [0x10bd] = 0x68b7, [0x10be] = 0x68b8, [0x10c0] = 0x68b9,
  [0x10c1] = 0x68ba, [0x10c2] = 0x68bb, [0x10c3] = 0x68bc, [0x10c4] = 0x68bd,
  [0x10c5] = 0x68be, [0x10c6] = 0x68bf, [0x10c7] = 0x68c1, [0x10c8] = 0x68c3,
  [0x10c9] = 0x68c4, [0x10ca] = 0x68c5, [0x10cb] = 0x68c6, [0x10cc] = 0x68c7,
  [0x10cd] = 0x68c8, [0x10ce] = 0x68ca, [0x10cf] = 0x68cc, [0x10d0] = 0x68ce,
  [0x10d1] = 0x68cf, [0x10d2] = 0x68d0, [0x10d3] = 0x68d1, [0x10d4] = 0x68d3,
  [0x10d5] = 0x68d4, [0x10d6] = 0x68d6, [0x10d7] = 0x68d7, [0x10d8] = 0x68d9,
  [0x10d9] = 0x68db, [0x10da] = 0x68dc, [0x10db] = 0x68dd, [0x10dc] = 0x68de,
  [0x10dd] = 0x68df, [0x10de] = 0x68e1, [0x10df] = 0x68e2, [0x10e0] = 0x68e4,
  [0x10e1] = 0x68e5, [0x10e2] = 0x68e6, [0x10e3] = 0x68e7, [0x10e4] = 0x68e8,
  [0x10e5] = 0x68e9, [0x10e6] = 0x68ea, [0x10e7] = 0x68eb, [0x10e8] = 0x68ec,
  [0x10e9] = 0x68ed, [0x10ea] = 0x68ef, [0x10eb] = 0x68f2, [0x10ec] = 0x68f3,
  [0x10ed] = 0x68f4, [0x10ee] = 0x68f6, [0x10ef] = 0x68f7, [0x10f0] = 0x68f8,
  [0x10f1] = 0x68fb, [0x10f2] = 0x68fd, [0x10f3] = 0x68fe, [0x10f4] = 0x68ff,
  [0x10f5] = 0x6900, [0x10f6] = 0x6902, [0x10f7] = 0x6903, [0x10f8] = 0x6904,
  [0x10f9] = 0x6906, [0x10fa] = 0x6907, [0x10fb] = 0x6908, [0x10fc] = 0x6909,
  [0x10fd] = 0x690a, [0x10fe] = 0x690c, [0x10ff] = 0x690f, [0x1100] = 0x6911,
  [0x1101] = 0x6913, [0x1102] = 0x6914, [0x1103] = 0x6915, [0x1104] = 0x6916,
  [0x1105] = 0x6917, [0x1106] = 0x6918, [0x1107] = 0x6919, [0x1108] = 0x691a,
  [0x1109] = 0x691b, [0x110a] = 0x691c, [0x110b] = 0x691d, [0x110c] = 0x691e,
  [0x110d] = 0x6921, [0x110e] = 0x6922, [0x110f] = 0x6923, [0x1110] = 0x6925,
  [0x1111] = 0x6926, [0x1112] = 0x6927, [0x1113] = 0x6928, [0x1114] = 0x6929,
  [0x1115] = 0x692a, [0x1116] = 0x692b, [0x1117] = 0x692c, [0x1118] = 0x692e,
  [0x1119] = 0x692f, [0x111a] = 0x6931, [0x111b] = 0x6932, [0x111c] = 0x6933,
  [0x111d] = 0x6935, [0x111e] = 0x6936, [0x111f] = 0x6937, [0x1120] = 0x6938,
  [0x1121] = 0x693a, [0x1122] = 0x693b, [0x1123] = 0x693c, [0x1124] = 0x693e,
  [0x1125] = 0x6940, [0x1126] = 0x6941, [0x1127] = 0x6943, [0x1128] = 0x6944,
  [0x1129] = 0x6945, [0x112a] = 0x6946, [0x112b] = 0x6947, [0x112c] = 0x6948,
  [0x112d] = 0x6949, [0x112e] = 0x694a, [0x112f] = 0x694b, [0x1130] = 0x694c,
  [0x1131] = 0x694d, [0x1132] = 0x694e, [0x1133] = 0x694f, [0x1134] = 0x6950,
  [0x1135] = 0x6951, [0x1136] = 0x6952, [0x1137] = 0x6953, [0x1138] = 0x6955,
  [0x1139] = 0x6956, [0x113a] = 0x6958, [0x113b] = 0x6959, [0x113c] = 0x695b,
  [0x113d] = 0x695c, [0x113e] = 0x695f, [0x1140] = 0x6961, [0x1141] = 0x6962,
  [0x1142] = 0x6964, [0x1143] = 0x6965, [0x1144] = 0x6967, [0x1145] = 0x6968,
  [0x1146] = 0x6969, [0x1147] = 0x696a, [0x1148] = 0x696c, [0x1149] = 0x696d,
  [0x114a] = 0x696f, [0x114b] = 0x6970, [0x114c] = 0x6972, [0x114d] = 0x6973,
  [0x114e] = 0x6974, [0x114f] = 0x6975, [0x1150] = 0x6976, [0x1151] = 0x697a,
  [0x1152] = 0x697b, [0x1153] = 0x697d, [0x1154] = 0x697e, [0x1155] = 0x697f,
  [0x1156] = 0x6981, [0x1157] = 0x6983, [0x1158] = 0x6985, [0x1159] = 0x698a,
  [0x115a] = 0x698b, [0x115b] = 0x698c, [0x115c] = 0x698e, [0x115d] = 0x698f,
  [0x115e] = 0x6990, [0x115f] = 0x6991, [0x1160] = 0x6992, [0x1161] = 0x6993,
  [0x1162] = 0x6996, [0x1163] = 0x6997, [0x1164] = 0x6999, [0x1165] = 0x699a,
  [0x1166] = 0x699d, [0x1167] = 0x699e, [0x1168] = 0x699f, [0x1169] = 0x69a0,
  [0x116a] = 0x69a1, [0x116b] = 0x69a2, [0x116c] = 0x69a3, [0x116d] = 0x69a4,
  [0x116e] = 0x69a5, [0x116f] = 0x69a6, [0x1170] = 0x69a9, [0x1171] = 0x69aa,
  [0x1172] = 0x69ac, [0x1173] = 0x69ae, [0x1174] = 0x69af, [0x1175] = 0x69b0,
  [0x1176] = 0x69b2, [0x1177] = 0x69b3, [0x1178] = 0x69b5, [0x1179] = 0x69b6,
  [0x117a] = 0x69b8, [0x117b] = 0x69b9, [0x117c] = 0x69ba, [0x117d] = 0x69bc,
  [0x117e] = 0x69bd, [0x1180] = 0x69be, [0x1181] = 0x69bf, [0x1182] = 0x69c0,
  [0x1183] = 0x69c2, [0x1184] = 0x69c3, [0x1185] = 0x69c4, [0x1186] = 0x69c5,
  [0x1187] = 0x69c6, [0x1188] = 0x69c7, [0x1189] = 0x69c8, [0x118a] = 0x69c9,
  [0x118b] = 0x69cb, [0x118c] = 0x69cd, [0x118d] = 0x69cf, [0x118e] = 0x69d1,
  [0x118f] = 0x69d2, [0x1190] = 0x69d3, [0x1191] = 0x69d5, [0x1192] = 0x69d6,
  [0x1193] = 0x69d7, [0x1194] = 0x69d8, [0x1195] = 0x69d9, [0x1196] = 0x69da,
  [0x1197] = 0x69dc, [0x1198] = 0x69dd, [0x1199] = 0x69de, [0x119a] = 0x69e1,
  [0x119b] = 0x69e2, [0x119c] = 0x69e3, [0x119d] = 0x69e4, [0x119e] = 0x69e5,
  [0x119f] = 0x69e6, [0x11a0] = 0x69e7, [0x11a1] = 0x69e8, [0x11a2] = 0x69e9,
  [0x11a3] = 0x69ea, [0x11a4] = 0x69eb, [0x11a5] = 0x69ec, [0x11a6] = 0x69ee,
  [0x11a7] = 0x69ef, [0x11a8] = 0x69f0, [0x11a9] = 0x69f1, [0x11aa] = 0x69f3,
  [0x11ab] = 0x69f4, [0x11ac] = 0x69f5, [0x11ad] = 0x69f6, [0x11ae] = 0x69f7,
  [0x11af] = 0x69f8, [0x11b0] = 0x69f9, [0x11b1] = 0x69fa, [0x11b2] = 0x69fb,
  [0x11b3] = 0x69fc, [0x11b4] = 0x69fe, [0x11b5] = 0x6a00, [0x11b6] = 0x6a01,
  [0x11b7] = 0x6a02, [0x11b8] = 0x6a03, [0x11b9] = 0x6a04, [0x11ba] = 0x6a05,
  [0x11bb] = 0x6a06, [0x11bc] = 0x6a07, [0x11bd] = 0x6a08, [0x11be] = 0x6a09,
  [0x11bf] = 0x6a0b, [0x11c0] = 0x6a0c, [0x11c1] = 0x6a0d, [0x11c2] = 0x6a0e,
  [0x11c3] = 0x6a0f, [0x11c4] = 0x6a10, [0x11c5] = 0x6a11, [0x11c6] = 0x6a12,
  [0x11c7] = 0x6a13, [0x11c8] = 0x6a14, [0x11c9] = 0x6a15, [0x11ca] = 0x6a16,
  [0x11cb] = 0x6a19, [0x11cc] = 0x6a1a, [0x11cd] = 0x6a1b, [0x11ce] = 0x6a1c,
  [0x11cf] = 0x6a1d, [0x11d0] = 0x6a1e, [0x11d1] = 0x6a20, [0x11d2] = 0x6a22,
  [0x11d3] = 0x6a23, [0x11d4] = 0x6a24, [0x11d5] = 0x6a25, [0x11d6] = 0x6a26,
  [0x11d7] = 0x6a27, [0x11d8] = 0x6a29, [0x11d9] = 0x6a2b, [0x11da] = 0x6a2c,
  [0x11db] = 0x6a2d, [0x11dc] = 0x6a2e, [0x11dd] = 0x6a30, [0x11de] = 0x6a32,
  [0x11df] = 0x6a33, [0x11e0] = 0x6a34, [0x11e1] = 0x6a36, [0x11e2] = 0x6a37,
  [0x11e3] = 0x6a38, [0x11e4] = 0x6a39, [0x11e5] = 0x6a3a, [0x11e6] = 0x6a3b,
  [0x11e7] = 0x6a3c, [0x11e8] = 0x6a3f, [0x11e9] = 0x6a40, [0x11ea] = 0x6a41,
  [0x11eb] = 0x6a42, [0x11ec] = 0x6a43, [0x11ed] = 0x6a45, [0x11ee] = 0x6a46,
  [0x11ef] = 0x6a48, [0x11f0] = 0x6a49, [0x11f1] = 0x6a4a, [0x11f2] = 0x6a4b,
  [0x11f3] = 0x6a4c, [0x11f4] = 0x6a4d, [0x11f5] = 0x6a4e, [0x11f6] = 0x6a4f,
  [0x11f7] = 0x6a51, [0x11f8] = 0x6a52, [0x11f9] = 0x6a53, [0x11fa] = 0x6a54,
  [0x11fb] = 0x6a55, [0x11fc] = 0x6a56, [0x11fd] = 0x6a57, [0x11fe] = 0x6a5a,
  [0x1200] = 0x6a5c, [0x1201] = 0x6a5d, [0x1202] = 0x6a5e, [0x1203] = 0x6a5f,
  [0x1204] = 0x6a60, [0x1205] = 0x6a62, [0x1206] = 0x6a63, [0x1207] = 0x6a64,
  [0x1208] = 0x6a66, [0x1209] = 0x6a67, [0x120a] = 0x6a68, [0x120b] = 0x6a69,
  [0x120c] = 0x6a6a, [0x120d] = 0x6a6b, [0x120e] = 0x6a6c, [0x120f] = 0x6a6d,
  [0x1210] = 0x6a6e, [0x1211] = 0x6a6f, [0x1212] = 0x6a70, [0x1213] = 0x6a72,
  [0x1214] = 0x6a73, [0x1215] = 0x6a74, [0x1216] = 0x6a75, [0x1217] = 0x6a76,
  [0x1218] = 0x6a77, [0x1219] = 0x6a78, [0x121a] = 0x6a7a, [0x121b] = 0x6a7b,
  [0x121c] = 0x6a7d, [0x121d] = 0x6a7e, [0x121e] = 0x6a7f, [0x121f] = 0x6a81,
  [0x1220] = 0x6a82, [0x1221] = 0x6a83, [0x1222] = 0x6a85, [0x1223] = 0x6a86,
  [0x1224] = 0x6a87, [0x1225] = 0x6a88, [0x1226] = 0x6a89, [0x1227] = 0x6a8a,
  [0x1228] = 0x6a8b, [0x1229] = 0x6a8c, [0x122a] = 0x6a8d, [0x122b] = 0x6a8f,
  [0x122c] = 0x6a92, [0x122d] = 0x6a93, [0x122e] = 0x6a94, [0x122f] = 0x6a95,
  [0x1230] = 0x6a96, [0x1231] = 0x6a98, [0x1232] = 0x6a99, [0x1233] = 0x6a9a,
  [0x1234] = 0x6a9b, [0x1235] = 0x6a9c, [0x1236] = 0x6a9d, [0x1237] = 0x6a9e,
  [0x1238] = 0x6a9f, [0x1239] = 0x6aa1, [0x123a] = 0x6aa2, [0x123b] = 0x6aa3,
  [0x123c] = 0x6aa4, [0x123d] = 0x6aa5, [0x123e] = 0x6aa6, [0x1240] = 0x6aa7,
  [0x1241] = 0x6aa8, [0x1242] = 0x6aaa, [0x1243] = 0x6aad, [0x1244] = 0x6aae,
  [0x1245] = 0x6aaf, [0x1246] = 0x6ab0, [0x1247] = 0x6ab1, [0x1248] = 0x6ab2,
  [0x1249] = 0x6ab3, [0x124a] = 0x6ab4, [0x124b] = 0x6ab5, [0x124c] = 0x6ab6,
  [0x124d] = 0x6ab7, [0x124e] = 0x6ab8, [0x124f] = 0x6ab9, [0x1250] = 0x6aba,
  [0x1251] = 0x6abb, [0x1252] = 0x6abc, [0x1253] = 0x6abd, [0x1254] = 0x6abe,
  [0x1255] = 0x6abf, [0x1256] = 0x6ac0, [0x1257] = 0x6ac1, [0x1258] = 0x6ac2,
  [0x1259] = 0x6ac3, [0x125a] = 0x6ac4, [0x125b] = 0x6ac5, [0x125c] = 0x6ac6,
  [0x125d] = 0x6ac7, [0x125e] = 0x6ac8, [0x125f] = 0x6ac9, [0x1260] = 0x6aca,
  [0x1261] = 0x6acb, [0x1262] = 0x6acc, [0x1263] = 0x6acd, [0x1264] = 0x6ace,
  [0x1265] = 0x6acf, [0x1266] = 0x6ad0, [0x1267] = 0x6ad1, [0x1268] = 0x6ad2,
  [0x1269] = 0x6ad3, [0x126a] = 0x6ad4, [0x126b] = 0x6ad5, [0x126c] = 0x6ad6,
  [0x126d] = 0x6ad7, [0x126e] = 0x6ad8, [0x126f] = 0x6ad9, [0x1270] = 0x6ada,
  [0x1271] = 0x6adb, [0x1272] = 0x6adc, [0x1273] = 0x6add, [0x1274] = 0x6ade,
  [0x1275] = 0x6adf, [0x1276] = 0x6ae0, [0x1277] = 0x6ae1, [0x1278] = 0x6ae2,
  [0x1279] = 0x6ae3, [0x127a] = 0x6ae4, [0x127b] = 0x6ae5, [0x127c] = 0x6ae6,
  [0x127d] = 0x6ae7, [0x127e] = 0x6ae8, [0x127f] = 0x6ae9, [0x1280] = 0x6aea,
  [0x1281] = 0x6aeb, [0x1282] = 0x6aec, [0x1283] = 0x6aed, [0x1284] = 0x6aee,
  [0x1285] = 0x6aef, [0x1286] = 0x6af0, [0x1287] = 0x6af1, [0x1288] = 0x6af2,
  [0x1289] = 0x6af3, [0x128a] = 0x6af4, [0x128b] = 0x6af5, [0x128c] = 0x6af6,
  [0x128d] = 0x6af7, [0x128e] = 0x6af8, [0x128f] = 0x6af9, [0x1290] = 0x6afa,
  [0x1291] = 0x6afb, [0x1292] = 0x6afc, [0x1293] = 0x6afd, [0x1294] = 0x6afe,
  [0x1295] = 0x6aff, [0x1296] = 0x6b00, [0x1297] = 0x6b01, [0x1298] = 0x6b02,
  [0x1299] = 0x6b03, [0x129a] = 0x6b04, [0x129b] = 0x6b05, [0x129c] = 0x6b06,
  [0x129d] = 0x6b07, [0x129e] = 0x6b08, [0x129f] = 0x6b09, [0x12a0] = 0x6b0a,
  [0x12a1] = 0x6b0b, [0x12a2] = 0x6b0c, [0x12a3] = 0x6b0d, [0x12a4] = 0x6b0e,
  [0x12a5] = 0x6b0f, [0x12a6] = 0x6b10, [0x12a7] = 0x6b11, [0x12a8] = 0x6b12,
  [0x12a9] = 0x6b13, [0x12aa] = 0x6b14, [0x12ab] = 0x6b15, [0x12ac] = 0x6b16,
  [0x12ad] = 0x6b17, [0x12ae] = 0x6b18, [0x12af] = 0x6b19, [0x12b0] = 0x6b1a,
  [0x12b1] = 0x6b1b, [0x12b2] = 0x6b1c, [0x12b3] = 0x6b1d, [0x12b4] = 0x6b1e,
  [0x12b5] = 0x6b1f, [0x12b6] = 0x6b25, [0x12b7] = 0x6b26, [0x12b8] = 0x6b28,
  [0x12b9] = 0x6b29, [0x12ba] = 0x6b2a, [0x12bb] = 0x6b2b, [0x12bc] = 0x6b2c,
  [0x12bd] = 0x6b2d, [0x12be] = 0x6b2e, [0x12c0] = 0x6b2f, [0x12c1] = 0x6b30,
  [0x12c2] = 0x6b31, [0x12c3] = 0x6b33, [0x12c4] = 0x6b34, [0x12c5] = 0x6b35,
  [0x12c6] = 0x6b36, [0x12c7] = 0x6b38, [0x12c8] = 0x6b3b, [0x12c9] = 0x6b3c,
  [0x12ca] = 0x6b3d, [0x12cb] = 0x6b3f, [0x12cc] = 0x6b40, [0x12cd] = 0x6b41,
  [0x12ce] = 0x6b42, [0x12cf] = 0x6b44, [0x12d0] = 0x6b45, [0x12d1] = 0x6b48,
  [0x12d2] = 0x6b4a, [0x12d3] = 0x6b4b, [0x12d4] = 0x6b4d, [0x12d5] = 0x6b4e,
  [0x12d6] = 0x6b4f, [0x12d7] = 0x6b50, [0x12d8] = 0x6b51, [0x12d9] = 0x6b52,
  [0x12da] = 0x6b53, [0x12db] = 0x6b54, [0x12dc] = 0x6b55, [0x12dd] = 0x6b56,
  [0x12de] = 0x6b57, [0x12df] = 0x6b58, [0x12e0] = 0x6b5a, [0x12e1] = 0x6b5b,
  [0x12e2] = 0x6b5c, [0x12e3] = 0x6b5d, [0x12e4] = 0x6b5e, [0x12e5] = 0x6b5f,
  [0x12e6] = 0x6b60, [0x12e7] = 0x6b61, [0x12e8] = 0x6b68, [0x12e9] = 0x6b69,
  [0x12ea] = 0x6b6b, [0x12eb] = 0x6b6c, [0x12ec] = 0x6b6d, [0x12ed] = 0x6b6e,
  [0x12ee] = 0x6b6f, [0x12ef] = 0x6b70, [0x12f0] = 0x6b71, [0x12f1] = 0x6b72,
  [0x12f2] = 0x6b73, [0x12f3] = 0x6b74, [0x12f4] = 0x6b75, [0x12f5] = 0x6b76,
  [0x12f6] = 0x6b77, [0x12f7] = 0x6b78, [0x12f8] = 0x6b7a, [0x12f9] = 0x6b7d,
  [0x12fa] = 0x6b7e, [0x12fb] = 0x6b7f, [0x12fc] = 0x6b80, [0x12fd] = 0x6b85,
  [0x12fe] = 0x6b88, [0x1300] = 0x6b8c, [0x1301] = 0x6b8e, [0x1302] = 0x6b8f,
  [0x1303] = 0x6b90, [0x1304] = 0x6b91, [0x1305] = 0x6b94, [0x1306] = 0x6b95,
  [0x1307] = 0x6b97, [0x1308] = 0x6b98, [0x1309] = 0x6b99, [0x130a] = 0x6b9c,
  [0x130b] = 0x6b9d, [0x130c] = 0x6b9e, [0x130d] = 0x6b9f, [0x130e] = 0x6ba0,
  [0x130f] = 0x6ba2, [0x1310] = 0x6ba3, [0x1311] = 0x6ba4, [0x1312] = 0x6ba5,
  [0x1313] = 0x6ba6, [0x1314] = 0x6ba7, [0x1315] = 0x6ba8, [0x1316] = 0x6ba9,
  [0x1317] = 0x6bab, [0x1318] = 0x6bac, [0x1319] = 0x6bad, [0x131a] = 0x6bae,
  [0x131b] = 0x6baf, [0x131c] = 0x6bb0, [0x131d] = 0x6bb1, [0x131e] = 0x6bb2,
  [0x131f] = 0x6bb6, [0x1320] = 0x6bb8, [0x1321] = 0x6bb9, [0x1322] = 0x6bba,
  [0x1323] = 0x6bbb, [0x1324] = 0x6bbc, [0x1325] = 0x6bbd, [0x1326] = 0x6bbe,
  [0x1327] = 0x6bc0, [0x1328] = 0x6bc3, [0x1329] = 0x6bc4, [0x132a] = 0x6bc6,
  [0x132b] = 0x6bc7, [0x132c] = 0x6bc8, [0x132d] = 0x6bc9, [0x132e] = 0x6bca,
  [0x132f] = 0x6bcc, [0x1330] = 0x6bce, [0x1331] = 0x6bd0, [0x1332] = 0x6bd1,
  [0x1333] = 0x6bd8, [0x1334] = 0x6bda, [0x1335] = 0x6bdc, [0x1336] = 0x6bdd,
  [0x1337] = 0x6bde, [0x1338] = 0x6bdf, [0x1339] = 0x6be0, [0x133a] = 0x6be2,
  [0x133b] = 0x6be3, [0x133c] = 0x6be4, [0x133d] = 0x6be5, [0x133e] = 0x6be6,
  [0x133f] = 0x6be7, [0x1340] = 0x6be8, [0x1341] = 0x6be9, [0x1342] = 0x6bec,
  [0x1343] = 0x6bed, [0x1344] = 0x6bee, [0x1345] = 0x6bf0, [0x1346] = 0x6bf1,
  [0x1347] = 0x6bf2, [0x1348] = 0x6bf4, [0x1349] = 0x6bf6, [0x134a] = 0x6bf7,
  [0x134b] = 0x6bf8, [0x134c] = 0x6bfa, [0x134d] = 0x6bfb, [0x134e] = 0x6bfc,
  [0x134f] = 0x6bfe, [0x1350] = 0x6bff, [0x1351] = 0x6c00, [0x1352] = 0x6c01,
  [0x1353] = 0x6c02, [0x1354] = 0x6c03, [0x1355] = 0x6c04, [0x1356] = 0x6c08,
  [0x1357] = 0x6c09, [0x1358] = 0x6c0a, [0x1359] = 0x6c0b, [0x135a] = 0x6c0c,
  [0x135b] = 0x6c0e, [0x135c] = 0x6c12, [0x135d] = 0x6c17, [0x135e] = 0x6c1c,
  [0x135f] = 0x6c1d, [0x1360] = 0x6c1e, [0x1361] = 0x6c20, [0x1362] = 0x6c23,
  [0x1363] = 0x6c25, [0x1364] = 0x6c2b, [0x1365] = 0x6c2c, [0x1366] = 0x6c2d,
  [0x1367] = 0x6c31, [0x1368] = 0x6c33, [0x1369] = 0x6c36, [0x136a] = 0x6c37,
  [0x136b] = 0x6c39, [0x136c] = 0x6c3a, [0x136d] = 0x6c3b, [0x136e] = 0x6c3c,
  [0x136f] = 0x6c3e, [0x1370] = 0x6c3f, [0x1371] = 0x6c43, [0x1372] = 0x6c44,
  [0x1373] = 0x6c45, [0x1374] = 0x6c48, [0x1375] = 0x6c4b, [0x1376] = 0x6c4c,
  [0x1377] = 0x6c4d, [0x1378] = 0x6c4e, [0x1379] = 0x6c4f, [0x137a] = 0x6c51,
  [0x137b] = 0x6c52, [0x137c] = 0x6c53, [0x137d] = 0x6c56, [0x137e] = 0x6c58,
  [0x1380] = 0x6c59, [0x1381] = 0x6c5a, [0x1382] = 0x6c62, [0x1383] = 0x6c63,
  [0x1384] = 0x6c65, [0x1385] = 0x6c66, [0x1386] = 0x6c67, [0x1387] = 0x6c6b,
  [0x1388] = 0x6c6c, [0x1389] = 0x6c6d, [0x138a] = 0x6c6e, [0x138b] = 0x6c6f,
  [0x138c] = 0x6c71, [0x138d] = 0x6c73, [0x138e] = 0x6c75, [0x138f] = 0x6c77,
  [0x1390] = 0x6c78, [0x1391] = 0x6c7a, [0x1392] = 0x6c7b, [0x1393] = 0x6c7c,
  [0x1394] = 0x6c7f, [0x1395] = 0x6c80, [0x1396] = 0x6c84, [0x1397] = 0x6c87,
  [0x1398] = 0x6c8a, [0x1399] = 0x6c8b, [0x139a] = 0x6c8d, [0x139b] = 0x6c8e,
  [0x139c] = 0x6c91, [0x139d] = 0x6c92, [0x139e] = 0x6c95, [0x139f] = 0x6c96,
  [0x13a0] = 0x6c97, [0x13a1] = 0x6c98, [0x13a2] = 0x6c9a, [0x13a3] = 0x6c9c,
  [0x13a4] = 0x6c9d, [0x13a5] = 0x6c9e, [0x13a6] = 0x6ca0, [0x13a7] = 0x6ca2,
  [0x13a8] = 0x6ca8, [0x13a9] = 0x6cac, [0x13aa] = 0x6caf, [0x13ab] = 0x6cb0,
  [0x13ac] = 0x6cb4, [0x13ad] = 0x6cb5, [0x13ae] = 0x6cb6, [0x13af] = 0x6cb7,
  [0x13b0] = 0x6cba, [0x13b1] = 0x6cc0, [0x13b2] = 0x6cc1, [0x13b3] = 0x6cc2,
  [0x13b4] = 0x6cc3, [0x13b5] = 0x6cc6, [0x13b6] = 0x6cc7, [0x13b7] = 0x6cc8,
  [0x13b8] = 0x6ccb, [0x13b9] = 0x6ccd, [0x13ba] = 0x6cce, [0x13bb] = 0x6ccf,
  [0x13bc] = 0x6cd1, [0x13bd] = 0x6cd2, [0x13be] = 0x6cd8, [0x13c0] = 0x6cd9,
  [0x13c1] = 0x6cda, [0x13c2] = 0x6cdc, [0x13c3] = 0x6cdd, [0x13c4] = 0x6cdf,
  [0x13c5] = 0x6ce4, [0x13c6] = 0x6ce6, [0x13c7] = 0x6ce7, [0x13c8] = 0x6ce9,
  [0x13c9] = 0x6cec, [0x13ca] = 0x6ced, [0x13cb] = 0x6cf2, [0x13cc] = 0x6cf4,
  [0x13cd] = 0x6cf9, [0x13ce] = 0x6cff, [0x13cf] = 0x6d00, [0x13d0] = 0x6d02,
  [0x13d1] = 0x6d03, [0x13d2] = 0x6d05, [0x13d3] = 0x6d06, [0x13d4] = 0x6d08,
  [0x13d5] = 0x6d09, [0x13d6] = 0x6d0a, [0x13d7] = 0x6d0d, [0x13d8] = 0x6d0f,
  [0x13d9] = 0x6d10, [0x13da] = 0x6d11, [0x13db] = 0x6d13, [0x13dc] = 0x6d14,
  [0x13dd] = 0x6d15, [0x13de] = 0x6d16, [0x13df] = 0x6d18, [0x13e0] = 0x6d1c,
  [0x13e1] = 0x6d1d, [0x13e2] = 0x6d1f, [0x13e3] = 0x6d20, [0x13e4] = 0x6d21,
  [0x13e5] = 0x6d22, [0x13e6] = 0x6d23, [0x13e7] = 0x6d24, [0x13e8] = 0x6d26,
  [0x13e9] = 0x6d28, [0x13ea] = 0x6d29, [0x13eb] = 0x6d2c, [0x13ec] = 0x6d2d,
  [0x13ed] = 0x6d2f, [0x13ee] = 0x6d30, [0x13ef] = 0x6d34, [0x13f0] = 0x6d36,
  [0x13f1] = 0x6d37, [0x13f2] = 0x6d38, [0x13f3] = 0x6d3a, [0x13f4] = 0x6d3f,
  [0x13f5] = 0x6d40, [0x13f6] = 0x6d42, [0x13f7] = 0x6d44, [0x13f8] = 0x6d49,
  [0x13f9] = 0x6d4c, [0x13fa] = 0x6d50, [0x13fb] = 0x6d55, [0x13fc] = 0x6d56,
  [0x13fd] = 0x6d57, [0x13fe] = 0x6d58, [0x13ff] = 0x6d5b, [0x1400] = 0x6d5d,
  [0x1401] = 0x6d5f, [0x1402] = 0x6d61, [0x1403] = 0x6d62, [0x1404] = 0x6d64,
  [0x1405] = 0x6d65, [0x1406] = 0x6d67, [0x1407] = 0x6d68, [0x1408] = 0x6d6b,
  [0x1409] = 0x6d6c, [0x140a] = 0x6d6d, [0x140b] = 0x6d70, [0x140c] = 0x6d71,
  [0x140d] = 0x6d72, [0x140e] = 0x6d73, [0x140f] = 0x6d75, [0x1410] = 0x6d76,
  [0x1411] = 0x6d79, [0x1412] = 0x6d7a, [0x1413] = 0x6d7b, [0x1414] = 0x6d7d,
  [0x1415] = 0x6d7e, [0x1416] = 0x6d7f, [0x1417] = 0x6d80, [0x1418] = 0x6d81,
  [0x1419] = 0x6d83, [0x141a] = 0x6d84, [0x141b] = 0x6d86, [0x141c] = 0x6d87,
  [0x141d] = 0x6d8a, [0x141e] = 0x6d8b, [0x141f] = 0x6d8d, [0x1420] = 0x6d8f,
  [0x1421] = 0x6d90, [0x1422] = 0x6d92, [0x1423] = 0x6d96, [0x1424] = 0x6d97,
  [0x1425] = 0x6d98, [0x1426] = 0x6d99, [0x1427] = 0x6d9a, [0x1428] = 0x6d9c,
  [0x1429] = 0x6da2, [0x142a] = 0x6da5, [0x142b] = 0x6dac, [0x142c] = 0x6dad,
  [0x142d] = 0x6db0, [0x142e] = 0x6db1, [0x142f] = 0x6db3, [0x1430] = 0x6db4,
  [0x1431] = 0x6db6, [0x1432] = 0x6db7, [0x1433] = 0x6db9, [0x1434] = 0x6dba,
  [0x1435] = 0x6dbb, [0x1436] = 0x6dbc, [0x1437] = 0x6dbd, [0x1438] = 0x6dbe,
  [0x1439] = 0x6dc1, [0x143a] = 0x6dc2, [0x143b] = 0x6dc3, [0x143c] = 0x6dc8,
  [0x143d] = 0x6dc9, [0x143e] = 0x6dca, [0x1440] = 0x6dcd, [0x1441] = 0x6dce,
  [0x1442] = 0x6dcf, [0x1443] = 0x6dd0, [0x1444] = 0x6dd2, [0x1445] = 0x6dd3,
  [0x1446] = 0x6dd4, [0x1447] = 0x6dd5, [0x1448] = 0x6dd7, [0x1449] = 0x6dda,
  [0x144a] = 0x6ddb, [0x144b] = 0x6ddc, [0x144c] = 0x6ddf, [0x144d] = 0x6de2,
  [0x144e] = 0x6de3, [0x144f] = 0x6de5, [0x1450] = 0x6de7, [0x1451] = 0x6de8,
  [0x1452] = 0x6de9, [0x1453] = 0x6dea, [0x1454] = 0x6ded, [0x1455] = 0x6def,
  [0x1456] = 0x6df0, [0x1457] = 0x6df2, [0x1458] = 0x6df4, [0x1459] = 0x6df5,
  [0x145a] = 0x6df6, [0x145b] = 0x6df8, [0x145c] = 0x6dfa, [0x145d] = 0x6dfd,
  [0x145e] = 0x6dfe, [0x145f] = 0x6dff, [0x1460] = 0x6e00, [0x1461] = 0x6e01,
  [0x1462] = 0x6e02, [0x1463] = 0x6e03, [0x1464] = 0x6e04, [0x1465] = 0x6e06,
  [0x1466] = 0x6e07, [0x1467] = 0x6e08, [0x1468] = 0x6e09, [0x1469] = 0x6e0b,
  [0x146a] = 0x6e0f, [0x146b] = 0x6e12, [0x146c] = 0x6e13, [0x146d] = 0x6e15,
  [0x146e] = 0x6e18, [0x146f] = 0x6e19, [0x1470] = 0x6e1b, [0x1471] = 0x6e1c,
  [0x1472] = 0x6e1e, [0x1473] = 0x6e1f, [0x1474] = 0x6e22, [0x1475] = 0x6e26,
  [0x1476] = 0x6e27, [0x1477] = 0x6e28, [0x1478] = 0x6e2a, [0x1479] = 0x6e2c,
  [0x147a] = 0x6e2e, [0x147b] = 0x6e30, [0x147c] = 0x6e31, [0x147d] = 0x6e33,
  [0x147e] = 0x6e35, [0x1480] = 0x6e36, [0x1481] = 0x6e37, [0x1482] = 0x6e39,
  [0x1483] = 0x6e3b, [0x1484] = 0x6e3c, [0x1485] = 0x6e3d, [0x1486] = 0x6e3e,
  [0x1487] = 0x6e3f, [0x1488] = 0x6e40, [0x1489] = 0x6e41, [0x148a] = 0x6e42,
  [0x148b] = 0x6e45, [0x148c] = 0x6e46, [0x148d] = 0x6e47, [0x148e] = 0x6e48,
  [0x148f] = 0x6e49, [0x1490] = 0x6e4a, [0x1491] = 0x6e4b, [0x1492] = 0x6e4c,
  [0x1493] = 0x6e4f, [0x1494] = 0x6e50, [0x1495] = 0x6e51, [0x1496] = 0x6e52,
  [0x1497] = 0x6e55, [0x1498] = 0x6e57, [0x1499] = 0x6e59, [0x149a] = 0x6e5a,
  [0x149b] = 0x6e5c, [0x149c] = 0x6e5d, [0x149d] = 0x6e5e, [0x149e] = 0x6e60,
  [0x149f] = 0x6e61, [0x14a0] = 0x6e62, [0x14a1] = 0x6e63, [0x14a2] = 0x6e64,
  [0x14a3] = 0x6e65, [0x14a4] = 0x6e66, [0x14a5] = 0x6e67, [0x14a6] = 0x6e68,
  [0x14a7] = 0x6e69, [0x14a8] = 0x6e6a, [0x14a9] = 0x6e6c, [0x14aa] = 0x6e6d,
  [0x14ab] = 0x6e6f, [0x14ac] = 0x6e70, [0x14ad] = 0x6e71, [0x14ae] = 0x6e72,
  [0x14af] = 0x6e73, [0x14b0] = 0x6e74, [0x14b1] = 0x6e75, [0x14b2] = 0x6e76,
  [0x14b3] = 0x6e77, [0x14b4] = 0x6e78, [0x14b5] = 0x6e79, [0x14b6] = 0x6e7a,
  [0x14b7] = 0x6e7b, [0x14b8] = 0x6e7c, [0x14b9] = 0x6e7d, [0x14ba] = 0x6e80,
  [0x14bb] = 0x6e81, [0x14bc] = 0x6e82, [0x14bd] = 0x6e84, [0x14be] = 0x6e87,
  [0x14bf] = 0x6e88, [0x14c0] = 0x6e8a, [0x14c1] = 0x6e8b, [0x14c2] = 0x6e8c,
  [0x14c3] = 0x6e8d, [0x14c4] = 0x6e8e, [0x14c5] = 0x6e91, [0x14c6] = 0x6e92,
  [0x14c7] = 0x6e93, [0x14c8] = 0x6e94, [0x14c9] = 0x6e95, [0x14ca] = 0x6e96,
  [0x14cb] = 0x6e97, [0x14cc] = 0x6e99, [0x14cd] = 0x6e9a, [0x14ce] = 0x6e9b,
  [0x14cf] = 0x6e9d, [0x14d0] = 0x6e9e, [0x14d1] = 0x6ea0, [0x14d2] = 0x6ea1,
  [0x14d3] = 0x6ea3, [0x14d4] = 0x6ea4, [0x14d5] = 0x6ea6, [0x14d6] = 0x6ea8,
  [0x14d7] = 0x6ea9, [0x14d8] = 0x6eab, [0x14d9] = 0x6eac, [0x14da] = 0x6ead,
  [0x14db] = 0x6eae, [0x14dc] = 0x6eb0, [0x14dd] = 0x6eb3, [0x14de] = 0x6eb5,
  [0x14df] = 0x6eb8, [0x14e0] = 0x6eb9, [0x14e1] = 0x6ebc, [0x14e2] = 0x6ebe,
  [0x14e3] = 0x6ebf, [0x14e4] = 0x6ec0, [0x14e5] = 0x6ec3, [0x14e6] = 0x6ec4,
  [0x14e7] = 0x6ec5, [0x14e8] = 0x6ec6, [0x14e9] = 0x6ec8, [0x14ea] = 0x6ec9,
  [0x14eb] = 0x6eca, [0x14ec] = 0x6ecc, [0x14ed] = 0x6ecd, [0x14ee] = 0x6ece,
  [0x14ef] = 0x6ed0, [0x14f0] = 0x6ed2, [0x14f1] = 0x6ed6, [0x14f2] = 0x6ed8,
  [0x14f3] = 0x6ed9, [0x14f4] = 0x6edb, [0x14f5] = 0x6edc, [0x14f6] = 0x6edd,
  [0x14f7] = 0x6ee3, [0x14f8] = 0x6ee7, [0x14f9] = 0x6eea, [0x14fa] = 0x6eeb,
  [0x14fb] = 0x6eec, [0x14fc] = 0x6eed, [0x14fd] = 0x6eee, [0x14fe] = 0x6eef,
  [0x1500] = 0x6ef0, [0x1501] = 0x6ef1, [0x1502] = 0x6ef2, [0x1503] = 0x6ef3,
  [0x1504] = 0x6ef5, [0x1505] = 0x6ef6, [0x1506] = 0x6ef7, [0x1507] = 0x6ef8,
  [0x1508] = 0x6efa, [0x1509] = 0x6efb, [0x150a] = 0x6efc, [0x150b] = 0x6efd,
  [0x150c] = 0x6efe, [0x150d] = 0x6eff, [0x150e] = 0x6f00, [0x150f] = 0x6f01,
  [0x1510] = 0x6f03, [0x1511] = 0x6f04, [0x1512] = 0x6f05, [0x1513] = 0x6f07,
  [0x1514] = 0x6f08, [0x1515] = 0x6f0a, [0x1516] = 0x6f0b, [0x1517] = 0x6f0c,
  [0x1518] = 0x6f0d, [0x1519] = 0x6f0e, [0x151a] = 0x6f10, [0x151b] = 0x6f11,
  [0x151c] = 0x6f12, [0x151d] = 0x6f16, [0x151e] = 0x6f17, [0x151f] = 0x6f18,
  [0x1520] = 0x6f19, [0x1521] = 0x6f1a, [0x1522] = 0x6f1b, [0x1523] = 0x6f1c,
  [0x1524] = 0x6f1d, [0x1525] = 0x6f1e, [0x1526] = 0x6f1f, [0x1527] = 0x6f21,
  [0x1528] = 0x6f22, [0x1529] = 0x6f23, [0x152a] = 0x6f25, [0x152b] = 0x6f26,
  [0x152c] = 0x6f27, [0x152d] = 0x6f28, [0x152e] = 0x6f2c, [0x152f] = 0x6f2e,
  [0x1530] = 0x6f30, [0x1531] = 0x6f32, [0x1532] = 0x6f34, [0x1533] = 0x6f35,
  [0x1534] = 0x6f37, [0x1535] = 0x6f38, [0x1536] = 0x6f39, [0x1537] = 0x6f3a,
  [0x1538] = 0x6f3b, [0x1539] = 0x6f3c, [0x153a] = 0x6f3d, [0x153b] = 0x6f3f,
  [0x153c] = 0x6f40, [0x153d] = 0x6f41, [0x153e] = 0x6f42, [0x1540] = 0x6f43,
  [0x1541] = 0x6f44, [0x1542] = 0x6f45, [0x1543] = 0x6f48, [0x1544] = 0x6f49,
  [0x1545] = 0x6f4a, [0x1546] = 0x6f4c, [0x1547] = 0x6f4e, [0x1548] = 0x6f4f,
  [0x1549] = 0x6f50, [0x154a] = 0x6f51, [0x154b] = 0x6f52, [0x154c] = 0x6f53,
  [0x154d] = 0x6f54, [0x154e] = 0x6f55, [0x154f] = 0x6f56, [0x1550] = 0x6f57,
  [0x1551] = 0x6f59, [0x1552] = 0x6f5a, [0x1553] = 0x6f5b, [0x1554] = 0x6f5d,
  [0x1555] = 0x6f5f, [0x1556] = 0x6f60, [0x1557] = 0x6f61, [0x1558] = 0x6f63,
  [0x1559] = 0x6f64, [0x155a] = 0x6f65, [0x155b] = 0x6f67, [0x155c] = 0x6f68,
  [0x155d] = 0x6f69, [0x155e] = 0x6f6a, [0x155f] = 0x6f6b, [0x1560] = 0x6f6c,
  [0x1561] = 0x6f6f, [0x1562] = 0x6f70, [0x1563] = 0x6f71, [0x1564] = 0x6f73,
  [0x1565] = 0x6f75, [0x1566] = 0x6f76, [0x1567] = 0x6f77, [0x1568] = 0x6f79,
  [0x1569] = 0x6f7b, [0x156a] = 0x6f7d, [0x156b] = 0x6f7e, [0x156c] = 0x6f7f,
  [0x156d] = 0x6f80, [0x156e] = 0x6f81, [0x156f] = 0x6f82, [0x1570] = 0x6f83,
  [0x1571] = 0x6f85, [0x1572] = 0x6f86, [0x1573] = 0x6f87, [0x1574] = 0x6f8a,
  [0x1575] = 0x6f8b, [0x1576] = 0x6f8f, [0x1577] = 0x6f90, [0x1578] = 0x6f91,
  [0x1579] = 0x6f92, [0x157a] = 0x6f93, [0x157b] = 0x6f94, [0x157c] = 0x6f95,
  [0x157d] = 0x6f96, [0x157e] = 0x6f97, [0x157f] = 0x6f98, [0x1580] = 0x6f99,
  [0x1581] = 0x6f9a, [0x1582] = 0x6f9b, [0x1583] = 0x6f9d, [0x1584] = 0x6f9e,
  [0x1585] = 0x6f9f, [0x1586] = 0x6fa0, [0x1587] = 0x6fa2, [0x1588] = 0x6fa3,
  [0x1589] = 0x6fa4, [0x158a] = 0x6fa5, [0x158b] = 0x6fa6, [0x158c] = 0x6fa8,
  [0x158d] = 0x6fa9, [0x158e] = 0x6faa, [0x158f] = 0x6fab, [0x1590] = 0x6fac,
  [0x1591] = 0x6fad, [0x1592] = 0x6fae, [0x1593] = 0x6faf, [0x1594] = 0x6fb0,
  [0x1595] = 0x6fb1, [0x1596] = 0x6fb2, [0x1597] = 0x6fb4, [0x1598] = 0x6fb5,
  [0x1599] = 0x6fb7, [0x159a] = 0x6fb8, [0x159b] = 0x6fba, [0x159c] = 0x6fbb,
  [0x159d] = 0x6fbc, [0x159e] = 0x6fbd, [0x159f] = 0x6fbe, [0x15a0] = 0x6fbf,
  [0x15a1] = 0x6fc1, [0x15a2] = 0x6fc3, [0x15a3] = 0x6fc4, [0x15a4] = 0x6fc5,
  [0x15a5] = 0x6fc6, [0x15a6] = 0x6fc7, [0x15a7] = 0x6fc8, [0x15a8] = 0x6fca,
  [0x15a9] = 0x6fcb, [0x15aa] = 0x6fcc, [0x15ab] = 0x6fcd, [0x15ac] = 0x6fce,
  [0x15ad] = 0x6fcf, [0x15ae] = 0x6fd0, [0x15af] = 0x6fd3, [0x15b0] = 0x6fd4,
  [0x15b1] = 0x6fd5, [0x15b2] = 0x6fd6, [0x15b3] = 0x6fd7, [0x15b4] = 0x6fd8,
  [0x15b5] = 0x6fd9, [0x15b6] = 0x6fda, [0x15b7] = 0x6fdb, [0x15b8] = 0x6fdc,
  [0x15b9] = 0x6fdd, [0x15ba] = 0x6fdf, [0x15bb] = 0x6fe2, [0x15bc] = 0x6fe3,
  [0x15bd] = 0x6fe4, [0x15be] = 0x6fe5, [0x15c0] = 0x6fe6, [0x15c1] = 0x6fe7,
  [0x15c2] = 0x6fe8, [0x15c3] = 0x6fe9, [0x15c4] = 0x6fea, [0x15c5] = 0x6feb,
  [0x15c6] = 0x6fec, [0x15c7] = 0x6fed, [0x15c8] = 0x6ff0, [0x15c9] = 0x6ff1,
  [0x15ca] = 0x6ff2, [0x15cb] = 0x6ff3, [0x15cc] = 0x6ff4, [0x15cd] = 0x6ff5,
  [0x15ce] = 0x6ff6, [0x15cf] = 0x6ff7, [0x15d0] = 0x6ff8, [0x15d1] = 0x6ff9,
  [0x15d2] = 0x6ffa, [0x15d3] = 0x6ffb, [0x15d4] = 0x6ffc, [0x15d5] = 0x6ffd,
  [0x15d6] = 0x6ffe, [0x15d7] = 0x6fff, [0x15d8] = 0x7000, [0x15d9] = 0x7001,
  [0x15da] = 0x7002, [0x15db] = 0x7003, [0x15dc] = 0x7004, [0x15dd] = 0x7005,
  [0x15de] = 0x7006, [0x15df] = 0x7007, [0x15e0] = 0x7008, [0x15e1] = 0x7009,
  [0x15e2] = 0x700a, [0x15e3] = 0x700b, [0x15e4] = 0x700c, [0x15e5] = 0x700d,
  [0x15e6] = 0x700e, [0x15e7] = 0x700f, [0x15e8] = 0x7010, [0x15e9] = 0x7012,
  [0x15ea] = 0x7013, [0x15eb] = 0x7014, [0x15ec] = 0x7015, [0x15ed] = 0x7016,
  [0x15ee] = 0x7017, [0x15ef] = 0x7018, [0x15f0] = 0x7019, [0x15f1] = 0x701c,
  [0x15f2] = 0x701d, [0x15f3] = 0x701e, [0x15f4] = 0x701f, [0x15f5] = 0x7020,
  [0x15f6] = 0x7021, [0x15f7] = 0x7022, [0x15f8] = 0x7024, [0x15f9] = 0x7025,
  [0x15fa] = 0x7026, [0x15fb] = 0x7027, [0x15fc] = 0x7028, [0x15fd] = 0x7029,
  [0x15fe] = 0x702a, [0x1600] = 0x702b, [0x1601] = 0x702c, [0x1602] = 0x702d,
  [0x1603] = 0x702e, [0x1604] = 0x702f, [0x1605] = 0x7030, [0x1606] = 0x7031,
  [0x1607] = 0x7032, [0x1608] = 0x7033, [0x1609] = 0x7034, [0x160a] = 0x7036,
  [0x160b] = 0x7037, [0x160c] = 0x7038, [0x160d] = 0x703a, [0x160e] = 0x703b,
  [0x160f] = 0x703c, [0x1610] = 0x703d, [0x1611] = 0x703e, [0x1612] = 0x703f,
  [0x1613] = 0x7040, [0x1614] = 0x7041, [0x1615] = 0x7042, [0x1616] = 0x7043,
  [0x1617] = 0x7044, [0x1618] = 0x7045, [0x1619] = 0x7046, [0x161a] = 0x7047,
  [0x161b] = 0x7048, [0x161c] = 0x7049, [0x161d] = 0x704a, [0x161e] = 0x704b,
  [0x161f] = 0x704d, [0x1620] = 0x704e, [0x1621] = 0x7050, [0x1622] = 0x7051,
  [0x1623] = 0x7052, [0x1624] = 0x7053, [0x1625] = 0x7054, [0x1626] = 0x7055,
  [0x1627] = 0x7056, [0x1628] = 0x7057, [0x1629] = 0x7058, [0x162a] = 0x7059,
  [0x162b] = 0x705a, [0x162c] = 0x705b, [0x162d] = 0x705c, [0x162e] = 0x705d,
  [0x162f] = 0x705f, [0x1630] = 0x7060, [0x1631] = 0x7061, [0x1632] = 0x7062,
  [0x1633] = 0x7063, [0x1634] = 0x7064, [0x1635] = 0x7065, [0x1636] = 0x7066,
  [0x1637] = 0x7067, [0x1638] = 0x7068, [0x1639] = 0x7069, [0x163a] = 0x706a,
  [0x163b] = 0x706e, [0x163c] = 0x7071, [0x163d] = 0x7072, [0x163e] = 0x7073,
  [0x163f] = 0x7074, [0x1640] = 0x7077, [0x1641] = 0x7079, [0x1642] = 0x707a,
  [0x1643] = 0x707b, [0x1644] = 0x707d, [0x1645] = 0x7081, [0x1646] = 0x7082,
  [0x1647] = 0x7083, [0x1648] = 0x7084, [0x1649] = 0x7086, [0x164a] = 0x7087,
  [0x164b] = 0x7088, [0x164c] = 0x708b, [0x164d] = 0x708c, [0x164e] = 0x708d,
  [0x164f] = 0x708f, [0x1650] = 0x7090, [0x1651] = 0x7091, [0x1652] = 0x7093,
  [0x1653] = 0x7097, [0x1654] = 0x7098, [0x1655] = 0x709a, [0x1656] = 0x709b,
  [0x1657] = 0x709e, [0x1658] = 0x709f, [0x1659] = 0x70a0, [0x165a] = 0x70a1,
  [0x165b] = 0x70a2, [0x165c] = 0x70a3, [0x165d] = 0x70a4, [0x165e] = 0x70a5,
  [0x165f] = 0x70a6, [0x1660] = 0x70a7, [0x1661] = 0x70a8, [0x1662] = 0x70a9,
  [0x1663] = 0x70aa, [0x1664] = 0x70b0, [0x1665] = 0x70b2, [0x1666] = 0x70b4,
  [0x1667] = 0x70b5, [0x1668] = 0x70b6, [0x1669] = 0x70ba, [0x166a] = 0x70be,
  [0x166b] = 0x70bf, [0x166c] = 0x70c4, [0x166d] = 0x70c5, [0x166e] = 0x70c6,
  [0x166f] = 0x70c7, [0x1670] = 0x70c9, [0x1671] = 0x70cb, [0x1672] = 0x70cc,
  [0x1673] = 0x70cd, [0x1674] = 0x70ce, [0x1675] = 0x70cf, [0x1676] = 0x70d0,
  [0x1677] = 0x70d1, [0x1678] = 0x70d2, [0x1679] = 0x70d3, [0x167a] = 0x70d4,
  [0x167b] = 0x70d5, [0x167c] = 0x70d6, [0x167d] = 0x70d7, [0x167e] = 0x70da,
  [0x1680] = 0x70dc, [0x1681] = 0x70dd, [0x1682] = 0x70de, [0x1683] = 0x70e0,
  [0x1684] = 0x70e1, [0x1685] = 0x70e2, [0x1686] = 0x70e3, [0x1687] = 0x70e5,
  [0x1688] = 0x70ea, [0x1689] = 0x70ee, [0x168a] = 0x70f0, [0x168b] = 0x70f1,
  [0x168c] = 0x70f2, [0x168d] = 0x70f3, [0x168e] = 0x70f4, [0x168f] = 0x70f5,
  [0x1690] = 0x70f6, [0x1691] = 0x70f8, [0x1692] = 0x70fa, [0x1693] = 0x70fb,
  [0x1694] = 0x70fc, [0x1695] = 0x70fe, [0x1696] = 0x70ff, [0x1697] = 0x7100,
  [0x1698] = 0x7101, [0x1699] = 0x7102, [0x169a] = 0x7103, [0x169b] = 0x7104,
  [0x169c] = 0x7105, [0x169d] = 0x7106, [0x169e] = 0x7107, [0x169f] = 0x7108,
  [0x16a0] = 0x710b, [0x16a1] = 0x710c, [0x16a2] = 0x710d, [0x16a3] = 0x710e,
  [0x16a4] = 0x710f, [0x16a5] = 0x7111, [0x16a6] = 0x7112, [0x16a7] = 0x7114,
  [0x16a8] = 0x7117, [0x16a9] = 0x711b, [0x16aa] = 0x711c, [0x16ab] = 0x711d,
  [0x16ac] = 0x711e, [0x16ad] = 0x711f, [0x16ae] = 0x7120, [0x16af] = 0x7121,
  [0x16b0] = 0x7122, [0x16b1] = 0x7123, [0x16b2] = 0x7124, [0x16b3] = 0x7125,
  [0x16b4] = 0x7127, [0x16b5] = 0x7128, [0x16b6] = 0x7129, [0x16b7] = 0x712a,
  [0x16b8] = 0x712b, [0x16b9] = 0x712c, [0x16ba] = 0x712d, [0x16bb] = 0x712e,
  [0x16bc] = 0x7132, [0x16bd] = 0x7133, [0x16be] = 0x7134, [0x16c0] = 0x7135,
  [0x16c1] = 0x7137, [0x16c2] = 0x7138, [0x16c3] = 0x7139, [0x16c4] = 0x713a,
  [0x16c5] = 0x713b, [0x16c6] = 0x713c, [0x16c7] = 0x713d, [0x16c8] = 0x713e,
  [0x16c9] = 0x713f, [0x16ca] = 0x7140, [0x16cb] = 0x7141, [0x16cc] = 0x7142,
  [0x16cd] = 0x7143, [0x16ce] = 0x7144, [0x16cf] = 0x7146, [0x16d0] = 0x7147,
  [0x16d1] = 0x7148, [0x16d2] = 0x7149, [0x16d3] = 0x714b, [0x16d4] = 0x714d,
  [0x16d5] = 0x714f, [0x16d6] = 0x7150, [0x16d7] = 0x7151, [0x16d8] = 0x7152,
  [0x16d9] = 0x7153, [0x16da] = 0x7154, [0x16db] = 0x7155, [0x16dc] = 0x7156,
  [0x16dd] = 0x7157, [0x16de] = 0x7158, [0x16df] = 0x7159, [0x16e0] = 0x715a,
  [0x16e1] = 0x715b, [0x16e2] = 0x715d, [0x16e3] = 0x715f, [0x16e4] = 0x7160,
  [0x16e5] = 0x7161, [0x16e6] = 0x7162, [0x16e7] = 0x7163, [0x16e8] = 0x7165,
  [0x16e9] = 0x7169, [0x16ea] = 0x716a, [0x16eb] = 0x716b, [0x16ec] = 0x716c,
  [0x16ed] = 0x716d, [0x16ee] = 0x716f, [0x16ef] = 0x7170, [0x16f0] = 0x7171,
  [0x16f1] = 0x7174, [0x16f2] = 0x7175, [0x16f3] = 0x7176, [0x16f4] = 0x7177,
  [0x16f5] = 0x7179, [0x16f6] = 0x717b, [0x16f7] = 0x717c, [0x16f8] = 0x717e,
  [0x16f9] = 0x717f, [0x16fa] = 0x7180, [0x16fb] = 0x7181, [0x16fc] = 0x7182,
  [0x16fd] = 0x7183, [0x16fe] = 0x7185, [0x16ff] = 0x7186, [0x1700] = 0x7187,
  [0x1701] = 0x7188, [0x1702] = 0x7189, [0x1703] = 0x718b, [0x1704] = 0x718c,
  [0x1705] = 0x718d, [0x1706] = 0x718e, [0x1707] = 0x7190, [0x1708] = 0x7191,
  [0x1709] = 0x7192, [0x170a] = 0x7193, [0x170b] = 0x7195, [0x170c] = 0x7196,
  [0x170d] = 0x7197, [0x170e] = 0x719a, [0x170f] = 0x719b, [0x1710] = 0x719c,
  [0x1711] = 0x719d, [0x1712] = 0x719e, [0x1713] = 0x71a1, [0x1714] = 0x71a2,
  [0x1715] = 0x71a3, [0x1716] = 0x71a4, [0x1717] = 0x71a5, [0x1718] = 0x71a6,
  [0x1719] = 0x71a7, [0x171a] = 0x71a9, [0x171b] = 0x71aa, [0x171c] = 0x71ab,
  [0x171d] = 0x71ad, [0x171e] = 0x71ae, [0x171f] = 0x71af, [0x1720] = 0x71b0,
  [0x1721] = 0x71b1, [0x1722] = 0x71b2, [0x1723] = 0x71b4, [0x1724] = 0x71b6,
  [0x1725] = 0x71b7, [0x1726] = 0x71b8, [0x1727] = 0x71ba, [0x1728] = 0x71bb,
  [0x1729] = 0x71bc, [0x172a] = 0x71bd, [0x172b] = 0x71be, [0x172c] = 0x71bf,
  [0x172d] = 0x71c0, [0x172e] = 0x71c1, [0x172f] = 0x71c2, [0x1730] = 0x71c4,
  [0x1731] = 0x71c5, [0x1732] = 0x71c6, [0x1733] = 0x71c7, [0x1734] = 0x71c8,
  [0x1735] = 0x71c9, [0x1736] = 0x71ca, [0x1737] = 0x71cb, [0x1738] = 0x71cc,
  [0x1739] = 0x71cd, [0x173a] = 0x71cf, [0x173b] = 0x71d0, [0x173c] = 0x71d1,
  [0x173d] = 0x71d2, [0x173e] = 0x71d3, [0x1740] = 0x71d6, [0x1741] = 0x71d7,
  [0x1742] = 0x71d8, [0x1743] = 0x71d9, [0x1744] = 0x71da, [0x1745] = 0x71db,
  [0x1746] = 0x71dc, [0x1747] = 0x71dd, [0x1748] = 0x71de, [0x1749] = 0x71df,
  [0x174a] = 0x71e1, [0x174b] = 0x71e2, [0x174c] = 0x71e3, [0x174d] = 0x71e4,
  [0x174e] = 0x71e6, [0x174f] = 0x71e8, [0x1750] = 0x71e9, [0x1751] = 0x71ea,
  [0x1752] = 0x71eb, [0x1753] = 0x71ec, [0x1754] = 0x71ed, [0x1755] = 0x71ef,
  [0x1756] = 0x71f0, [0x1757] = 0x71f1, [0x1758] = 0x71f2, [0x1759] = 0x71f3,
  [0x175a] = 0x71f4, [0x175b] = 0x71f5, [0x175c] = 0x71f6, [0x175d] = 0x71f7,
  [0x175e] = 0x71f8, [0x175f] = 0x71fa, [0x1760] = 0x71fb, [0x1761] = 0x71fc,
  [0x1762] = 0x71fd, [0x1763] = 0x71fe, [0x1764] = 0x71ff, [0x1765] = 0x7200,
  [0x1766] = 0x7201, [0x1767] = 0x7202, [0x1768] = 0x7203, [0x1769] = 0x7204,
  [0x176a] = 0x7205, [0x176b] = 0x7207, [0x176c] = 0x7208, [0x176d] = 0x7209,
  [0x176e] = 0x720a, [0x176f] = 0x720b, [0x1770] = 0x720c, [0x1771] = 0x720d,
  [0x1772] = 0x720e, [0x1773] = 0x720f, [0x1774] = 0x7210, [0x1775] = 0x7211,
  [0x1776] = 0x7212, [0x1777] = 0x7213, [0x1778] = 0x7214, [0x1779] = 0x7215,
  [0x177a] = 0x7216, [0x177b] = 0x7217, [0x177c] = 0x7218, [0x177d] = 0x7219,
  [0x177e] = 0x721a, [0x1780] = 0x721b, [0x1781] = 0x721c, [0x1782] = 0x721e,
  [0x1783] = 0x721f, [0x1784] = 0x7220, [0x1785] = 0x7221, [0x1786] = 0x7222,
  [0x1787] = 0x7223, [0x1788] = 0x7224, [0x1789] = 0x7225, [0x178a] = 0x7226,
  [0x178b] = 0x7227, [0x178c] = 0x7229, [0x178d] = 0x722b, [0x178e] = 0x722d,
  [0x178f] = 0x722e, [0x1790] = 0x722f, [0x1791] = 0x7232, [0x1792] = 0x7233,
  [0x1793] = 0x7234, [0x1794] = 0x723a, [0x1795] = 0x723c, [0x1796] = 0x723e,
  [0x1797] = 0x7240, [0x1798] = 0x7241, [0x1799] = 0x7242, [0x179a] = 0x7243,
  [0x179b] = 0x7244, [0x179c] = 0x7245, [0x179d] = 0x7246, [0x179e] = 0x7249,
  [0x179f] = 0x724a, [0x17a0] = 0x724b, [0x17a1] = 0x724e, [0x17a2] = 0x724f,
  [0x17a3] = 0x7250, [0x17a4] = 0x7251, [0x17a5] = 0x7253, [0x17a6] = 0x7254,
  [0x17a7] = 0x7255, [0x17a8] = 0x7257, [0x17a9] = 0x7258, [0x17aa] = 0x725a,
  [0x17ab] = 0x725c, [0x17ac] = 0x725e, [0x17ad] = 0x7260, [0x17ae] = 0x7263,
  [0x17af] = 0x7264, [0x17b0] = 0x7265, [0x17b1] = 0x7268, [0x17b2] = 0x726a,
  [0x17b3] = 0x726b, [0x17b4] = 0x726c, [0x17b5] = 0x726d, [0x17b6] = 0x7270,
  [0x17b7] = 0x7271, [0x17b8] = 0x7273, [0x17b9] = 0x7274, [0x17ba] = 0x7276,
  [0x17bb] = 0x7277, [0x17bc] = 0x7278, [0x17bd] = 0x727b, [0x17be] = 0x727c,
  [0x17bf] = 0x727d, [0x17c0] = 0x7282, [0x17c1] = 0x7283, [0x17c2] = 0x7285,
  [0x17c3] = 0x7286, [0x17c4] = 0x7287, [0x17c5] = 0x7288, [0x17c6] = 0x7289,
  [0x17c7] = 0x728c, [0x17c8] = 0x728e, [0x17c9] = 0x7290, [0x17ca] = 0x7291,
  [0x17cb] = 0x7293, [0x17cc] = 0x7294, [0x17cd] = 0x7295, [0x17ce] = 0x7296,
  [0x17cf] = 0x7297, [0x17d0] = 0x7298, [0x17d1] = 0x7299, [0x17d2] = 0x729a,
  [0x17d3] = 0x729b, [0x17d4] = 0x729c, [0x17d5] = 0x729d, [0x17d6] = 0x729e,
  [0x17d7] = 0x72a0, [0x17d8] = 0x72a1, [0x17d9] = 0x72a2, [0x17da] = 0x72a3,
  [0x17db] = 0x72a4, [0x17dc] = 0x72a5, [0x17dd] = 0x72a6, [0x17de] = 0x72a7,
  [0x17df] = 0x72a8, [0x17e0] = 0x72a9, [0x17e1] = 0x72aa, [0x17e2] = 0x72ab,
  [0x17e3] = 0x72ae, [0x17e4] = 0x72b1, [0x17e5] = 0x72b2, [0x17e6] = 0x72b3,
  [0x17e7] = 0x72b5, [0x17e8] = 0x72ba, [0x17e9] = 0x72bb, [0x17ea] = 0x72bc,
  [0x17eb] = 0x72bd, [0x17ec] = 0x72be, [0x17ed] = 0x72bf, [0x17ee] = 0x72c0,
  [0x17ef] = 0x72c5, [0x17f0] = 0x72c6, [0x17f1] = 0x72c7, [0x17f2] = 0x72c9,
  [0x17f3] = 0x72ca, [0x17f4] = 0x72cb, [0x17f5] = 0x72cc, [0x17f6] = 0x72cf,
  [0x17f7] = 0x72d1, [0x17f8] = 0x72d3, [0x17f9] = 0x72d4, [0x17fa] = 0x72d5,
  [0x17fb] = 0x72d6, [0x17fc] = 0x72d8, [0x17fd] = 0x72da, [0x17fe] = 0x72db,
  [0x1800] = 0xe4c6, [0x1801] = 0xe4c7, [0x1802] = 0xe4c8, [0x1803] = 0xe4c9,
  [0x1804] = 0xe4ca, [0x1805] = 0xe4cb, [0x1806] = 0xe4cc, [0x1807] = 0xe4cd,
  [0x1808] = 0xe4ce, [0x1809] = 0xe4cf, [0x180a] = 0xe4d0, [0x180b] = 0xe4d1,
  [0x180c] = 0xe4d2, [0x180d] = 0xe4d3, [0x180e] = 0xe4d4, [0x180f] = 0xe4d5,
  [0x1810] = 0xe4d6, [0x1811] = 0xe4d7, [0x1812] = 0xe4d8, [0x1813] = 0xe4d9,
  [0x1814] = 0xe4da, [0x1815] = 0xe4db, [0x1816] = 0xe4dc, [0x1817] = 0xe4dd,
  [0x1818] = 0xe4de, [0x1819] = 0xe4df, [0x181a] = 0xe4e0, [0x181b] = 0xe4e1,
  [0x181c] = 0xe4e2, [0x181d] = 0xe4e3, [0x181e] = 0xe4e4, [0x181f] = 0xe4e5,
  [0x1820] = 0xe4e6, [0x1821] = 0xe4e7, [0x1822] = 0xe4e8, [0x1823] = 0xe4e9,
  [0x1824] = 0xe4ea, [0x1825] = 0xe4eb, [0x1826] = 0xe4ec, [0x1827] = 0xe4ed,
  [0x1828] = 0xe4ee, [0x1829] = 0xe4ef, [0x182a] = 0xe4f0, [0x182b] = 0xe4f1,
  [0x182c] = 0xe4f2, [0x182d] = 0xe4f3, [0x182e] = 0xe4f4, [0x182f] = 0xe4f5,
  [0x1830] = 0xe4f6, [0x1831] = 0xe4f7, [0x1832] = 0xe4f8, [0x1833] = 0xe4f9,
  [0x1834] = 0xe4fa, [0x1835] = 0xe4fb, [0x1836] = 0xe4fc, [0x1837] = 0xe4fd,
  [0x1838] = 0xe4fe, [0x1839] = 0xe4ff, [0x183a] = 0xe500, [0x183b] = 0xe501,
  [0x183c] = 0xe502, [0x183d] = 0xe503, [0x183e] = 0xe504, [0x1840] = 0xe505,
  [0x1841] = 0xe506, [0x1842] = 0xe507, [0x1843] = 0xe508, [0x1844] = 0xe509,
  [0x1845] = 0xe50a, [0x1846] = 0xe50b, [0x1847] = 0xe50c, [0x1848] = 0xe50d,
  [0x1849] = 0xe50e, [0x184a] = 0xe50f, [0x184b] = 0xe510, [0x184c] = 0xe511,
  [0x184d] = 0xe512, [0x184e] = 0xe513, [0x184f] = 0xe514, [0x1850] = 0xe515,
  [0x1851] = 0xe516, [0x1852] = 0xe517, [0x1853] = 0xe518, [0x1854] = 0xe519,
  [0x1855] = 0xe51a, [0x1856] = 0xe51b, [0x1857] = 0xe51c, [0x1858] = 0xe51d,
  [0x1859] = 0xe51e, [0x185a] = 0xe51f, [0x185b] = 0xe520, [0x185c] = 0xe521,
  [0x185d] = 0xe522, [0x185e] = 0xe523, [0x185f] = 0xe524, [0x1860] = 0xe525,
  [0x1861] = 0x3000, [0x1862] = 0x3001, [0x1863] = 0x3002, [0x1864] = 0x00b7,
  [0x1865] = 0x02c9, [0x1866] = 0x02c7, [0x1867] = 0x00a8, [0x1868] = 0x3003,
  [0x1869] = 0x3005, [0x186a] = 0x2014, [0x186b] = 0xff5e, [0x186c] = 0x2016,
  [0x186d] = 0x2026, [0x186e] = 0x2018, [0x186f] = 0x2019, [0x1870] = 0x201c,
  [0x1871] = 0x201d, [0x1872] = 0x3014, [0x1873] = 0x3015, [0x1874] = 0x3008,
  [0x1875] = 0x3009, [0x1876] = 0x300a, [0x1877] = 0x300b, [0x1878] = 0x300c,
  [0x1879] = 0x300d, [0x187a] = 0x300e, [0x187b] = 0x300f, [0x187c] = 0x3016,
  [0x187d] = 0x3017, [0x187e] = 0x3010, [0x187f] = 0x3011, [0x1880] = 0x00b1,
  [0x1881] = 0x00d7, [0x1882] = 0x00f7, [0x1883] = 0x2236, [0x1884] = 0x2227,
  [0x1885] = 0x2228, [0x1886] = 0x2211, [0x1887] = 0x220f, [0x1888] = 0x222a,
  [0x1889] = 0x2229, [0x188a] = 0x2208, [0x188b] = 0x2237, [0x188c] = 0x221a,
  [0x188d] = 0x22a5, [0x188e] = 0x2225, [0x188f] = 0x2220, [0x1890] = 0x2312,
  [0x1891] = 0x2299, [0x1892] = 0x222b, [0x1893] = 0x222e, [0x1894] = 0x2261,
  [0x1895] = 0x224c, [0x1896] = 0x2248, [0x1897] = 0x223d, [0x1898] = 0x221d,
  [0x1899] = 0x2260, [0x189a] = 0x226e, [0x189b] = 0x226f, [0x189c] = 0x2264,
  [0x189d] = 0x2265, [0x189e] = 0x221e, [0x189f] = 0x2235, [0x18a0] = 0x2234,
  [0x18a1] = 0x2642, [0x18a2] = 0x2640, [0x18a3] = 0x00b0, [0x18a4] = 0x2032,
  [0x18a5] = 0x2033, [0x18a6] = 0x2103, [0x18a7] = 0xff04, [0x18a8] = 0x00a4,
  [0x18a9] = 0xffe0, [0x18aa] = 0xffe1, [0x18ab] = 0x2030, [0x18ac] = 0x00a7,
  [0x18ad] = 0x2116, [0x18ae] = 0x2606, [0x18af] = 0x2605, [0x18b0] = 0x25cb,
  [0x18b1] = 0x25cf, [0x18b2] = 0x25ce, [0x18b3] = 0x25c7, [0x18b4] = 0x25c6,
  [0x18b5] = 0x25a1, [0x18b6] = 0x25a0, [0x18b7] = 0x25b3, [0x18b8] = 0x25b2,
  [0x18b9] = 0x203b, [0x18ba] = 0x2192, [0x18bb] = 0x2190, [0x18bc] = 0x2191,
  [0x18bd] = 0x2193, [0x18be] = 0x3013, [0x18c0] = 0xe526, [0x18c1] = 0xe527,
  [0x18c2] = 0xe528, [0x18c3] = 0xe529, [0x18c4] = 0xe52a, [0x18c5] = 0xe52b,
  [0x18c6] = 0xe52c, [0x18c7] = 0xe52d, [0x18c8] = 0xe52e, [0x18c9] = 0xe52f,
  [0x18ca] = 0xe530, [0x18cb] = 0xe531, [0x18cc] = 0xe532, [0x18cd] = 0xe533,
  [0x18ce] = 0xe534, [0x18cf] = 0xe535, [0x18d0] = 0xe536, [0x18d1] = 0xe537,
  [0x18d2] = 0xe538, [0x18d3] = 0xe539, [0x18d4] = 0xe53a, [0x18d5] = 0xe53b,
  [0x18d6] = 0xe53c, [0x18d7] = 0xe53d, [0x18d8] = 0xe53e, [0x18d9] = 0xe53f,
  [0x18da] = 0xe540, [0x18db] = 0xe541, [0x18dc] = 0xe542, [0x18dd] = 0xe543,
  [0x18de] = 0xe544, [0x18df] = 0xe545, [0x18e0] = 0xe546, [0x18e1] = 0xe547,
  [0x18e2] = 0xe548, [0x18e3] = 0xe549, [0x18e4] = 0xe54a, [0x18e5] = 0xe54b,
  [0x18e6] = 0xe54c, [0x18e7] = 0xe54d, [0x18e8] = 0xe54e, [0x18e9] = 0xe54f,
  [0x18ea] = 0xe550, [0x18eb] = 0xe551, [0x18ec] = 0xe552, [0x18ed] = 0xe553,
  [0x18ee] = 0xe554, [0x18ef] = 0xe555, [0x18f0] = 0xe556, [0x18f1] = 0xe557,
  [0x18f2] = 0xe558, [0x18f3] = 0xe559, [0x18f4] = 0xe55a, [0x18f5] = 0xe55b,
  [0x18f6] = 0xe55c, [0x18f7] = 0xe55d, [0x18f8] = 0xe55e, [0x18f9] = 0xe55f,
  [0x18fa] = 0xe560, [0x18fb] = 0xe561, [0x18fc] = 0xe562, [0x18fd] = 0xe563,
  [0x18fe] = 0xe564, [0x1900] = 0xe565, [0x1901] = 0xe566, [0x1902] = 0xe567,
  [0x1903] = 0xe568, [0x1904] = 0xe569, [0x1905] = 0xe56a, [0x1906] = 0xe56b,
  [0x1907] = 0xe56c, [0x1908] = 0xe56d, [0x1909] = 0xe56e, [0x190a] = 0xe56f,
  [0x190b] = 0xe570, [0x190c] = 0xe571, [0x190d] = 0xe572, [0x190e] = 0xe573,
  [0x190f] = 0xe574, [0x1910] = 0xe575, [0x1911] = 0xe576, [0x1912] = 0xe577,
  [0x1913] = 0xe578, [0x1914] = 0xe579, [0x1915] = 0xe57a, [0x1916] = 0xe57b,
  [0x1917] = 0xe57c, [0x1918] = 0xe57d, [0x1919] = 0xe57e, [0x191a] = 0xe57f,
  [0x191b] = 0xe580, [0x191c] = 0xe581, [0x191d] = 0xe582, [0x191e] = 0xe583,
  [0x191f] = 0xe584, [0x1920] = 0xe585, [0x1921] = 0x2170, [0x1922] = 0x2171,
  [0x1923] = 0x2172, [0x1924] = 0x2173, [0x1925] = 0x2174, [0x1926] = 0x2175,
  [0x1927] = 0x2176, [0x1928] = 0x2177, [0x1929] = 0x2178, [0x192a] = 0x2179,
  [0x192b] = 0xe766, [0x192c] = 0xe767, [0x192d] = 0xe768, [0x192e] = 0xe769,
  [0x192f] = 0xe76a, [0x1930] = 0xe76b, [0x1931] = 0x2488, [0x1932] = 0x2489,
  [0x1933] = 0x248a, [0x1934] = 0x248b, [0x1935] = 0x248c, [0x1936] = 0x248d,
  [0x1937] = 0x248e, [0x1938] = 0x248f, [0x1939] = 0x2490, [0x193a] = 0x2491,
  [0x193b] = 0x2492, [0x193c] = 0x2493, [0x193d] = 0x2494, [0x193e] = 0x2495,
  [0x193f] = 0x2496, [0x1940] = 0x2497, [0x1941] = 0x2498, [0x1942] = 0x2499,
  [0x1943] = 0x249a, [0x1944] = 0x249b, [0x1945] = 0x2474, [0x1946] = 0x2475,
  [0x1947] = 0x2476, [0x1948] = 0x2477, [0x1949] = 0x2478, [0x194a] = 0x2479,
  [0x194b] = 0x247a, [0x194c] = 0x247b, [0x194d] = 0x247c, [0x194e] = 0x247d,
  [0x194f] = 0x247e, [0x1950] = 0x247f, [0x1951] = 0x2480, [0x1952] = 0x2481,
  [0x1953] = 0x2482, [0x1954] = 0x2483, [0x1955] = 0x2484, [0x1956] = 0x2485,
  [0x1957] = 0x2486, [0x1958] = 0x2487, [0x1959] = 0x2460, [0x195a] = 0x2461,
  [0x195b] = 0x2462, [0x195c] = 0x2463, [0x195d] = 0x2464, [0x195e] = 0x2465,
  [0x195f] = 0x2466, [0x1960] = 0x2467, [0x1961] = 0x2468, [0x1962] = 0x2469,
  [0x1963] = 0x20ac, [0x1964] = 0xe76d, [0x1965] = 0x3220, [0x1966] = 0x3221,
  [0x1967] = 0x3222, [0x1968] = 0x3223, [0x1969] = 0x3224, [0x196a] = 0x3225,
  [0x196b] = 0x3226, [0x196c] = 0x3227, [0x196d] = 0x3228, [0x196e] = 0x3229,
  [0x196f] = 0xe76e, [0x1970] = 0xe76f, [0x1971] = 0x2160, [0x1972] = 0x2161,
  [0x1973] = 0x2162, [0x1974] = 0x2163, [0x1975] = 0x2164, [0x1976] = 0x2165,
  [0x1977] = 0x2166, [0x1978] = 0x2167, [0x1979] = 0x2168, [0x197a] = 0x2169,
  [0x197b] = 0x216a, [0x197c] = 0x216b, [0x197d] = 0xe770, [0x197e] = 0xe771,
  [0x1980] = 0xe586, [0x1981] = 0xe587, [0x1982] = 0xe588, [0x1983] = 0xe589,
  [0x1984] = 0xe58a, [0x1985] = 0xe58b, [0x1986] = 0xe58c, [0x1987] = 0xe58d,
  [0x1988] = 0xe58e, [0x1989] = 0xe58f, [0x198a] = 0xe590, [0x198b] = 0xe591,
  [0x198c] = 0xe592, [0x198d] = 0xe593, [0x198e] = 0xe594, [0x198f] = 0xe595,
  [0x1990] = 0xe596, [0x1991] = 0xe597, [0x1992] = 0xe598, [0x1993] = 0xe599,
  [0x1994] = 0xe59a, [0x1995] = 0xe59b, [0x1996] = 0xe59c, [0x1997] = 0xe59d,
  [0x1998] = 0xe59e, [0x1999] = 0xe59f, [0x199a] = 0xe5a0, [0x199b] = 0xe5a1,
  [0x199c] = 0xe5a2, [0x199d] = 0xe5a3, [0x199e] = 0xe5a4, [0x199f] = 0xe5a5,
  [0x19a0] = 0xe5a6, [0x19a1] = 0xe5a7, [0x19a2] = 0xe5a8, [0x19a3] = 0xe5a9,
  [0x19a4] = 0xe5aa, [0x19a5] = 0xe5ab, [0x19a6] = 0xe5ac, [0x19a7] = 0xe5ad,
  [0x19a8] = 0xe5ae, [0x19a9] = 0xe5af, [0x19aa] = 0xe5b0, [0x19ab] = 0xe5b1,
  [0x19ac] = 0xe5b2, [0x19ad] = 0xe5b3, [0x19ae] = 0xe5b4, [0x19af] = 0xe5b5,
  [0x19b0] = 0xe5b6, [0x19b1] = 0xe5b7, [0x19b2] = 0xe5b8, [0x19b3] = 0xe5b9,
  [0x19b4] = 0xe5ba, [0x19b5] = 0xe5bb, [0x19b6] = 0xe5bc, [0x19b7] = 0xe5bd,
  [0x19b8] = 0xe5be, [0x19b9] = 0xe5bf, [0x19ba] = 0xe5c0, [0x19bb] = 0xe5c1,
  [0x19bc] = 0xe5c2, [0x19bd] = 0xe5c3, [0x19be] = 0xe5c4, [0x19c0] = 0xe5c5,
  [0x19c1] = 0xe5c6, [0x19c2] = 0xe5c7, [0x19c3] = 0xe5c8, [0x19c4] = 0xe5c9,
  [0x19c5] = 0xe5ca, [0x19c6] = 0xe5cb, [0x19c7] = 0xe5cc, [0x19c8] = 0xe5cd,
  [0x19c9] = 0xe5ce, [0x19ca] = 0xe5cf, [0x19cb] = 0xe5d0, [0x19cc] = 0xe5d1,
  [0x19cd] = 0xe5d2, [0x19ce] = 0xe5d3, [0x19cf] = 0xe5d4, [0x19d0] = 0xe5d5,
  [0x19d1] = 0xe5d6, [0x19d2] = 0xe5d7, [0x19d3] = 0xe5d8, [0x19d4] = 0xe5d9,
  [0x19d5] = 0xe5da, [0x19d6] = 0xe5db, [0x19d7] = 0xe5dc, [0x19d8] = 0xe5dd,
  [0x19d9] = 0xe5de, [0x19da] = 0xe5df, [0x19db] = 0xe5e0, [0x19dc] = 0xe5e1,
  [0x19dd] = 0xe5e2, [0x19de] = 0xe5e3, [0x19df] = 0xe5e4, [0x19e0] = 0xe5e5,
  [0x19e1] = 0xff01, [0x19e2] = 0xff02, [0x19e3] = 0xff03, [0x19e4] = 0xffe5,
  [0x19e5] = 0xff05, [0x19e6] = 0xff06, [0x19e7] = 0xff07, [0x19e8] = 0xff08,
  [0x19e9] = 0xff09, [0x19ea] = 0xff0a, [0x19eb] = 0xff0b, [0x19ec] = 0xff0c,
  [0x19ed] = 0xff0d, [0x19ee] = 0xff0e, [0x19ef] = 0xff0f, [0x19f0] = 0xff10,
  [0x19f1] = 0xff11, [0x19f2] = 0xff12, [0x19f3] = 0xff13, [0x19f4] = 0xff14,
  [0x19f5] = 0xff15, [0x19f6] = 0xff16, [0x19f7] = 0xff17, [0x19f8] = 0xff18,
  [0x19f9] = 0xff19, [0x19fa] = 0xff1a, [0x19fb] = 0xff1b, [0x19fc] = 0xff1c,
  [0x19fd] = 0xff1d, [0x19fe] = 0xff1e, [0x19ff] = 0xff1f, [0x1a00] = 0xff20,
  [0x1a01] = 0xff21, [0x1a02] = 0xff22, [0x1a03] = 0xff23, [0x1a04] = 0xff24,
  [0x1a05] = 0xff25, [0x1a06] = 0xff26, [0x1a07] = 0xff27, [0x1a08] = 0xff28,
  [0x1a09] = 0xff29, [0x1a0a] = 0xff2a, [0x1a0b] = 0xff2b, [0x1a0c] = 0xff2c,
  [0x1a0d] = 0xff2d, [0x1a0e] = 0xff2e, [0x1a0f] = 0xff2f, [0x1a10] = 0xff30,
  [0x1a11] = 0xff31, [0x1a12] = 0xff32, [0x1a13] = 0xff33, [0x1a14] = 0xff34,
  [0x1a15] = 0xff35, [0x1a16] = 0xff36, [0x1a17] = 0xff37, [0x1a18] = 0xff38,
  [0x1a19] = 0xff39, [0x1a1a] = 0xff3a, [0x1a1b] = 0xff3b, [0x1a1c] = 0xff3c,
  [0x1a1d] = 0xff3d, [0x1a1e] = 0xff3e, [0x1a1f] = 0xff3f, [0x1a20] = 0xff40,
  [0x1a21] = 0xff41, [0x1a22] = 0xff42, [0x1a23] = 0xff43, [0x1a24] = 0xff44,
  [0x1a25] = 0xff45, [0x1a26] = 0xff46, [0x1a27] = 0xff47, [0x1a28] = 0xff48,
  [0x1a29] = 0xff49, [0x1a2a] = 0xff4a, [0x1a2b] = 0xff4b, [0x1a2c] = 0xff4c,
  [0x1a2d] = 0xff4d, [0x1a2e] = 0xff4e, [0x1a2f] = 0xff4f, [0x1a30] = 0xff50,
  [0x1a31] = 0xff51, [0x1a32] = 0xff52, [0x1a33] = 0xff53, [0x1a34] = 0xff54,
  [0x1a35] = 0xff55, [0x1a36] = 0xff56, [0x1a37] = 0xff57, [0x1a38] = 0xff58,
  [0x1a39] = 0xff59, [0x1a3a] = 0xff5a, [0x1a3b] = 0xff5b, [0x1a3c] = 0xff5c,
  [0x1a3d] = 0xff5d, [0x1a3e] = 0xffe3, [0x1a40] = 0xe5e6, [0x1a41] = 0xe5e7,
  [0x1a42] = 0xe5e8, [0x1a43] = 0xe5e9, [0x1a44] = 0xe5ea, [0x1a45] = 0xe5eb,
  [0x1a46] = 0xe5ec, [0x1a47] = 0xe5ed, [0x1a48] = 0xe5ee, [0x1a49] = 0xe5ef,
  [0x1a4a] = 0xe5f0, [0x1a4b] = 0xe5f1, [0x1a4c] = 0xe5f2, [0x1a4d] = 0xe5f3,
  [0x1a4e] = 0xe5f4, [0x1a4f] = 0xe5f5, [0x1a50] = 0xe5f6, [0x1a51] = 0xe5f7,
  [0x1a52] = 0xe5f8, [0x1a53] = 0xe5f9, [0x1a54] = 0xe5fa, [0x1a55] = 0xe5fb,
  [0x1a56] = 0xe5fc, [0x1a57] = 0xe5fd, [0x1a58] = 0xe5fe, [0x1a59] = 0xe5ff,
  [0x1a5a] = 0xe600, [0x1a5b] = 0xe601, [0x1a5c] = 0xe602, [0x1a5d] = 0xe603,
  [0x1a5e] = 0xe604, [0x1a5f] = 0xe605, [0x1a60] = 0xe606, [0x1a61] = 0xe607,
  [0x1a62] = 0xe608, [0x1a63] = 0xe609, [0x1a64] = 0xe60a, [0x1a65] = 0xe60b,
  [0x1a66] = 0xe60c, [0x1a67] = 0xe60d, [0x1a68] = 0xe60e, [0x1a69] = 0xe60f,
  [0x1a6a] = 0xe610, [0x1a6b] = 0xe611, [0x1a6c] = 0xe612, [0x1a6d] = 0xe613,
  [0x1a6e] = 0xe614, [0x1a6f] = 0xe615, [0x1a70] = 0xe616, [0x1a71] = 0xe617,
  [0x1a72] = 0xe618, [0x1a73] = 0xe619, [0x1a74] = 0xe61a, [0x1a75] = 0xe61b,
  [0x1a76] = 0xe61c, [0x1a77] = 0xe61d, [0x1a78] = 0xe61e, [0x1a79] = 0xe61f,
  [0x1a7a] = 0xe620, [0x1a7b] = 0xe621, [0x1a7c] = 0xe622, [0x1a7d] = 0xe623,
  [0x1a7e] = 0xe624, [0x1a80] = 0xe625, [0x1a81] = 0xe626, [0x1a82] = 0xe627,
  [0x1a83] = 0xe628, [0x1a84] = 0xe629, [0x1a85] = 0xe62a, [0x1a86] = 0xe62b,
  [0x1a87] = 0xe62c, [0x1a88] = 0xe62d, [0x1a89] = 0xe62e, [0x1a8a] = 0xe62f,
  [0x1a8b] = 0xe630, [0x1a8c] = 0xe631, [0x1a8d] = 0xe632, [0x1a8e] = 0xe633,
  [0x1a8f] = 0xe634, [0x1a90] = 0xe635, [0x1a91] = 0xe636, [0x1a92] = 0xe637,
  [0x1a93] = 0xe638, [0x1a94] = 0xe639, [0x1a95] = 0xe63a, [0x1a96] = 0xe63b,
  [0x1a97] = 0xe63c, [0x1a98] = 0xe63d, [0x1a99] = 0xe63e, [0x1a9a] = 0xe63f,
  [0x1a9b] = 0xe640, [0x1a9c] = 0xe641, [0x1a9d] = 0xe642, [0x1a9e] = 0xe643,
  [0x1a9f] = 0xe644, [0x1aa0] = 0xe645, [0x1aa1] = 0x3041, [0x1aa2] = 0x3042,
  [0x1aa3] = 0x3043, [0x1aa4] = 0x3044, [0x1aa5] = 0x3045, [0x1aa6] = 0x3046,
  [0x1aa7] = 0x3047, [0x1aa8] = 0x3048, [0x1aa9] = 0x3049, [0x1aaa] = 0x304a,
  [0x1aab] = 0x304b, [0x1aac] = 0x304c, [0x1aad] = 0x304d, [0x1aae] = 0x304e,
  [0x1aaf] = 0x304f, [0x1ab0] = 0x3050, [0x1ab1] = 0x3051, [0x1ab2] = 0x3052,
  [0x1ab3] = 0x3053, [0x1ab4] = 0x3054, [0x1ab5] = 0x3055, [0x1ab6] = 0x3056,
  [0x1ab7] = 0x3057, [0x1ab8] = 0x3058, [0x1ab9] = 0x3059, [0x1aba] = 0x305a,
  [0x1abb] = 0x305b, [0x1abc] = 0x305c, [0x1abd] = 0x305d, [0x1abe] = 0x305e,
  [0x1abf] = 0x305f, [0x1ac0] = 0x3060, [0x1ac1] = 0x3061, [0x1ac2] = 0x3062,
  [0x1ac3] = 0x3063, [0x1ac4] = 0x3064, [0x1ac5] = 0x3065, [0x1ac6] = 0x3066,
  [0x1ac7] = 0x3067, [0x1ac8] = 0x3068, [0x1ac9] = 0x3069, [0x1aca] = 0x306a,
  [0x1acb] = 0x306b, [0x1acc] = 0x306c, [0x1acd] = 0x306d, [0x1ace] = 0x306e,
  [0x1acf] = 0x306f, [0x1ad0] = 0x3070, [0x1ad1] = 0x3071, [0x1ad2] = 0x3072,
  [0x1ad3] = 0x3073, [0x1ad4] = 0x3074, [0x1ad5] = 0x3075, [0x1ad6] = 0x3076,
  [0x1ad7] = 0x3077, [0x1ad8] = 0x3078, [0x1ad9] = 0x3079, [0x1ada] = 0x307a,
  [0x1adb] = 0x307b, [0x1adc] = 0x307c, [0x1add] = 0x307d, [0x1ade] = 0x307e,
  [0x1adf] = 0x307f, [0x1ae0] = 0x3080, [0x1ae1] = 0x3081, [0x1ae2] = 0x3082,
  [0x1ae3] = 0x3083, [0x1ae4] = 0x3084, [0x1ae5] = 0x3085, [0x1ae6] = 0x3086,
  [0x1ae7] = 0x3087, [0x1ae8] = 0x3088, [0x1ae9] = 0x3089, [0x1aea] = 0x308a,
  [0x1aeb] = 0x308b, [0x1aec] = 0x308c, [0x1aed] = 0x308d, [0x1aee] = 0x308e,
  [0x1aef] = 0x308f, [0x1af0] = 0x3090, [0x1af1] = 0x3091, [0x1af2] = 0x3092,
  [0x1af3] = 0x3093, [0x1af4] = 0xe772, [0x1af5] = 0xe773, [0x1af6] = 0xe774,
  [0x1af7] = 0xe775, [0x1af8] = 0xe776, [0x1af9] = 0xe777, [0x1afa] = 0xe778,
  [0x1afb] = 0xe779, [0x1afc] = 0xe77a, [0x1afd] = 0xe77b, [0x1afe] = 0xe77c,
  [0x1b00] = 0xe646, [0x1b01] = 0xe647, [0x1b02] = 0xe648, [0x1b03] = 0xe649,
  [0x1b04] = 0xe64a, [0x1b05] = 0xe64b, [0x1b06] = 0xe64c, [0x1b07] = 0xe64d,
  [0x1b08] = 0xe64e, [0x1b09] = 0xe64f, [0x1b0a] = 0xe650, [0x1b0b] = 0xe651,
  [0x1b0c] = 0xe652, [0x1b0d] = 0xe653, [0x1b0e] = 0xe654, [0x1b0f] = 0xe655,
  [0x1b10] = 0xe656, [0x1b11] = 0xe657, [0x1b12] = 0xe658, [0x1b13] = 0xe659,
  [0x1b14] = 0xe65a, [0x1b15] = 0xe65b, [0x1b16] = 0xe65c, [0x1b17] = 0xe65d,
  [0x1b18] = 0xe65e, [0x1b19] = 0xe65f, [0x1b1a] = 0xe660, [0x1b1b] = 0xe661,
  [0x1b1c] = 0xe662, [0x1b1d] = 0xe663, [0x1b1e] = 0xe664, [0x1b1f] = 0xe665,
  [0x1b20] = 0xe666, [0x1b21] = 0xe667, [0x1b22] = 0xe668, [0x1b23] = 0xe669,
  [0x1b24] = 0xe66a, [0x1b25] = 0xe66b, [0x1b26] = 0xe66c, [0x1b27] = 0xe66d,
  [0x1b28] = 0xe66e, [0x1b29] = 0xe66f, [0x1b2a] = 0xe670, [0x1b2b] = 0xe671,
  [0x1b2c] = 0xe672, [0x1b2d] = 0xe673, [0x1b2e] = 0xe674, [0x1b2f] = 0xe675,
  [0x1b30] = 0xe676, [0x1b31] = 0xe677, [0x1b32] = 0xe678, [0x1b33] = 0xe679,
  [0x1b34] = 0xe67a, [0x1b35] = 0xe67b, [0x1b36] = 0xe67c, [0x1b37] = 0xe67d,
  [0x1b38] = 0xe67e, [0x1b39] = 0xe67f, [0x1b3a] = 0xe680, [0x1b3b] = 0xe681,
  [0x1b3c] = 0xe682, [0x1b3d] = 0xe683, [0x1b3e] = 0xe684, [0x1b40] = 0xe685,
  [0x1b41] = 0xe686, [0x1b42] = 0xe687, [0x1b43] = 0xe688, [0x1b44] = 0xe689,
  [0x1b45] = 0xe68a, [0x1b46] = 0xe68b, [0x1b47] = 0xe68c, [0x1b48] = 0xe68d,
  [0x1b49] = 0xe68e, [0x1b4a] = 0xe68f, [0x1b4b] = 0xe690, [0x1b4c] = 0xe691,
  [0x1b4d] = 0xe692, [0x1b4e] = 0xe693, [0x1b4f] = 0xe694, [0x1b50] = 0xe695,
  [0x1b51] = 0xe696, [0x1b52] = 0xe697, [0x1b53] = 0xe698, [0x1b54] = 0xe699,
  [0x1b55] = 0xe69a, [0x1b56] = 0xe69b, [0x1b57] = 0xe69c, [0x1b58] = 0xe69d,
  [0x1b59] = 0xe69e, [0x1b5a] = 0xe69f, [0x1b5b] = 0xe6a0, [0x1b5c] = 0xe6a1,
  [0x1b5d] = 0xe6a2, [0x1b5e] = 0xe6a3, [0x1b5f] = 0xe6a4, [0x1b60] = 0xe6a5,
  [0x1b61] = 0x30a1, [0x1b62] = 0x30a2, [0x1b63] = 0x30a3, [0x1b64] = 0x30a4,
  [0x1b65] = 0x30a5, [0x1b66] = 0x30a6, [0x1b67] = 0x30a7, [0x1b68] = 0x30a8,
  [0x1b69] = 0x30a9, [0x1b6a] = 0x30aa, [0x1b6b] = 0x30ab, [0x1b6c] = 0x30ac,
  [0x1b6d] = 0x30ad, [0x1b6e] = 0x30ae, [0x1b6f] = 0x30af, [0x1b70] = 0x30b0,
  [0x1b71] = 0x30b1, [0x1b72] = 0x30b2, [0x1b73] = 0x30b3, [0x1b74] = 0x30b4,
  [0x1b75] = 0x30b5, [0x1b76] = 0x30b6, [0x1b77] = 0x30b7, [0x1b78] = 0x30b8,
  [0x1b79] = 0x30b9, [0x1b7a] = 0x30ba, [0x1b7b] = 0x30bb, [0x1b7c] = 0x30bc,
  [0x1b7d] = 0x30bd, [0x1b7e] = 0x30be, [0x1b7f] = 0x30bf, [0x1b80] = 0x30c0,
  [0x1b81] = 0x30c1, [0x1b82] = 0x30c2, [0x1b83] = 0x30c3, [0x1b84] = 0x30c4,
  [0x1b85] = 0x30c5, [0x1b86] = 0x30c6, [0x1b87] = 0x30c7, [0x1b88] = 0x30c8,
  [0x1b89] = 0x30c9, [0x1b8a] = 0x30ca, [0x1b8b] = 0x30cb, [0x1b8c] = 0x30cc,
  [0x1b8d] = 0x30cd, [0x1b8e] = 0x30ce, [0x1b8f] = 0x30cf, [0x1b90] = 0x30d0,
  [0x1b91] = 0x30d1, [0x1b92] = 0x30d2, [0x1b93] = 0x30d3, [0x1b94] = 0x30d4,
  [0x1b95] = 0x30d5, [0x1b96] = 0x30d6, [0x1b97] = 0x30d7, [0x1b98] = 0x30d8,
  [0x1b99] = 0x30d9, [0x1b9a] = 0x30da, [0x1b9b] = 0x30db, [0x1b9c] = 0x30dc,
  [0x1b9d] = 0x30dd, [0x1b9e] = 0x30de, [0x1b9f] = 0x30df, [0x1ba0] = 0x30e0,
  [0x1ba1] = 0x30e1, [0x1ba2] = 0x30e2, [0x1ba3] = 0x30e3, [0x1ba4] = 0x30e4,
  [0x1ba5] = 0x30e5, [0x1ba6] = 0x30e6, [0x1ba7] = 0x30e7, [0x1ba8] = 0x30e8,
  [0x1ba9] = 0x30e9, [0x1baa] = 0x30ea, [0x1bab] = 0x30eb, [0x1bac] = 0x30ec,
  [0x1bad] = 0x30ed, [0x1bae] = 0x30ee, [0x1baf] = 0x30ef, [0x1bb0] = 0x30f0,
  [0x1bb1] = 0x30f1, [0x1bb2] = 0x30f2, [0x1bb3] = 0x30f3, [0x1bb4] = 0x30f4,
  [0x1bb5] = 0x30f5, [0x1bb6] = 0x30f6, [0x1bb7] = 0xe77d, [0x1bb8] = 0xe77e,
  [0x1bb9] = 0xe77f, [0x1bba] = 0xe780, [0x1bbb] = 0xe781, [0x1bbc] = 0xe782,
  [0x1bbd] = 0xe783, [0x1bbe] = 0xe784, [0x1bc0] = 0xe6a6, [0x1bc1] = 0xe6a7,
  [0x1bc2] = 0xe6a8, [0x1bc3] = 0xe6a9, [0x1bc4] = 0xe6aa, [0x1bc5] = 0xe6ab,
  [0x1bc6] = 0xe6ac, [0x1bc7] = 0xe6ad, [0x1bc8] = 0xe6ae, [0x1bc9] = 0xe6af,
  [0x1bca] = 0xe6b0, [0x1bcb] = 0xe6b1, [0x1bcc] = 0xe6b2, [0x1bcd] = 0xe6b3,
  [0x1bce] = 0xe6b4, [0x1bcf] = 0xe6b5, [0x1bd0] = 0xe6b6, [0x1bd1] = 0xe6b7,
  [0x1bd2] = 0xe6b8, [0x1bd3] = 0xe6b9, [0x1bd4] = 0xe6ba, [0x1bd5] = 0xe6bb,
  [0x1bd6] = 0xe6bc, [0x1bd7] = 0xe6bd, [0x1bd8] = 0xe6be, [0x1bd9] = 0xe6bf,
  [0x1bda] = 0xe6c0, [0x1bdb] = 0xe6c1, [0x1bdc] = 0xe6c2, [0x1bdd] = 0xe6c3,
  [0x1bde] = 0xe6c4, [0x1bdf] = 0xe6c5, [0x1be0] = 0xe6c6, [0x1be1] = 0xe6c7,
  [0x1be2] = 0xe6c8, [0x1be3] = 0xe6c9, [0x1be4] = 0xe6ca, [0x1be5] = 0xe6cb,
  [0x1be6] = 0xe6cc, [0x1be7] = 0xe6cd, [0x1be8] = 0xe6ce, [0x1be9] = 0xe6cf,
  [0x1bea] = 0xe6d0, [0x1beb] = 0xe6d1, [0x1bec] = 0xe6d2, [0x1bed] = 0xe6d3,
  [0x1bee] = 0xe6d4, [0x1bef] = 0xe6d5, [0x1bf0] = 0xe6d6, [0x1bf1] = 0xe6d7,
  [0x1bf2] = 0xe6d8, [0x1bf3] = 0xe6d9, [0x1bf4] = 0xe6da, [0x1bf5] = 0xe6db,
  [0x1bf6] = 0xe6dc, [0x1bf7] = 0xe6dd, [0x1bf8] = 0xe6de, [0x1bf9] = 0xe6df,
  [0x1bfa] = 0xe6e0, [0x1bfb] = 0xe6e1, [0x1bfc] = 0xe6e2, [0x1bfd] = 0xe6e3,
  [0x1bfe] = 0xe6e4, [0x1c00] = 0xe6e5, [0x1c01] = 0xe6e6, [0x1c02] = 0xe6e7,
  [0x1c03] = 0xe6e8, [0x1c04] = 0xe6e9, [0x1c05] = 0xe6ea, [0x1c06] = 0xe6eb,
  [0x1c07] = 0xe6ec, [0x1c08] = 0xe6ed, [0x1c09] = 0xe6ee, [0x1c0a] = 0xe6ef,
  [0x1c0b] = 0xe6f0, [0x1c0c] = 0xe6f1, [0x1c0d] = 0xe6f2, [0x1c0e] = 0xe6f3,
  [0x1c0f] = 0xe6f4, [0x1c10] = 0xe6f5, [0x1c11] = 0xe6f6, [0x1c12] = 0xe6f7,
  [0x1c13] = 0xe6f8, [0x1c14] = 0xe6f9, [0x1c15] = 0xe6fa, [0x1c16] = 0xe6fb,
  [0x1c17] = 0xe6fc, [0x1c18] = 0xe6fd, [0x1c19] = 0xe6fe, [0x1c1a] = 0xe6ff,
  [0x1c1b] = 0xe700, [0x1c1c] = 0xe701, [0x1c1d] = 0xe702, [0x1c1e] = 0xe703,
  [0x1c1f] = 0xe704, [0x1c20] = 0xe705, [0x1c21] = 0x0391, [0x1c22] = 0x0392,
  [0x1c23] = 0x0393, [0x1c24] = 0x0394, [0x1c25] = 0x0395, [0x1c26] = 0x0396,
  [0x1c27] = 0x0397, [0x1c28] = 0x0398, [0x1c29] = 0x0399, [0x1c2a] = 0x039a,
  [0x1c2b] = 0x039b, [0x1c2c] = 0x039c, [0x1c2d] = 0x039d, [0x1c2e] = 0x039e,
  [0x1c2f] = 0x039f, [0x1c30] = 0x03a0, [0x1c31] = 0x03a1, [0x1c32] = 0x03a3,
  [0x1c33] = 0x03a4, [0x1c34] = 0x03a5, [0x1c35] = 0x03a6, [0x1c36] = 0x03a7,
  [0x1c37] = 0x03a8, [0x1c38] = 0x03a9, [0x1c39] = 0xe785, [0x1c3a] = 0xe786,
  [0x1c3b] = 0xe787, [0x1c3c] = 0xe788, [0x1c3d] = 0xe789, [0x1c3e] = 0xe78a,
  [0x1c3f] = 0xe78b, [0x1c40] = 0xe78c, [0x1c41] = 0x03b1, [0x1c42] = 0x03b2,
  [0x1c43] = 0x03b3, [0x1c44] = 0x03b4, [0x1c45] = 0x03b5, [0x1c46] = 0x03b6,
  [0x1c47] = 0x03b7, [0x1c48] = 0x03b8, [0x1c49] = 0x03b9, [0x1c4a] = 0x03ba,
  [0x1c4b] = 0x03bb, [0x1c4c] = 0x03bc, [0x1c4d] = 0x03bd, [0x1c4e] = 0x03be,
  [0x1c4f] = 0x03bf, [0x1c50] = 0x03c0, [0x1c51] = 0x03c1, [0x1c52] = 0x03c3,
  [0x1c53] = 0x03c4, [0x1c54] = 0x03c5, [0x1c55] = 0x03c6, [0x1c56] = 0x03c7,
  [0x1c57] = 0x03c8, [0x1c58] = 0x03c9, [0x1c59] = 0xfe10, [0x1c5a] = 0xfe12,
  [0x1c5b] = 0xfe11, [0x1c5c] = 0xfe13, [0x1c5d] = 0xfe14, [0x1c5e] = 0xfe15,
  [0x1c5f] = 0xfe16, [0x1c60] = 0xfe35, [0x1c61] = 0xfe36, [0x1c62] = 0xfe39,
  [0x1c63] = 0xfe3a, [0x1c64] = 0xfe3f, [0x1c65] = 0xfe40, [0x1c66] = 0xfe3d,
  [0x1c67] = 0xfe3e, [0x1c68] = 0xfe41, [0x1c69] = 0xfe42, [0x1c6a] = 0xfe43,
  [0x1c6b] = 0xfe44, [0x1c6c] = 0xfe17, [0x1c6d] = 0xfe18, [0x1c6e] = 0xfe3b,
  [0x1c6f] = 0xfe3c, [0x1c70] = 0xfe37, [0x1c71] = 0xfe38, [0x1c72] = 0xfe31,
  [0x1c73] = 0xfe19, [0x1c74] = 0xfe33, [0x1c75] = 0xfe34, [0x1c76] = 0xe797,
  [0x1c77] = 0xe798, [0x1c78] = 0xe799, [0x1c79] = 0xe79a, [0x1c7a] = 0xe79b,
  [0x1c7b] = 0xe79c, [0x1c7c] = 0xe79d, [0x1c7d] = 0xe79e, [0x1c7e] = 0xe79f,
  [0x1c80] = 0xe706, [0x1c81] = 0xe707, [0x1c82] = 0xe708, [0x1c83] = 0xe709,
  [0x1c84] = 0xe70a, [0x1c85] = 0xe70b, [0x1c86] = 0xe70c, [0x1c87] = 0xe70d,
  [0x1c88] = 0xe70e, [0x1c89] = 0xe70f, [0x1c8a] = 0xe710, [0x1c8b] = 0xe711,
  [0x1c8c] = 0xe712, [0x1c8d] = 0xe713, [0x1c8e] = 0xe714, [0x1c8f] = 0xe715,
  [0x1c90] = 0xe716, [0x1c91] = 0xe717, [0x1c92] = 0xe718, [0x1c93] = 0xe719,
  [0x1c94] = 0xe71a, [0x1c95] = 0xe71b, [0x1c96] = 0xe71c, [0x1c97] = 0xe71d,
  [0x1c98] = 0xe71e, [0x1c99] = 0xe71f, [0x1c9a] = 0xe720, [0x1c9b] = 0xe721,
  [0x1c9c] = 0xe722, [0x1c9d] = 0xe723, [0x1c9e] = 0xe724, [0x1c9f] = 0xe725,
  [0x1ca0] = 0xe726, [0x1ca1] = 0xe727, [0x1ca2] = 0xe728, [0x1ca3] = 0xe729,
  [0x1ca4] = 0xe72a, [0x1ca5] = 0xe72b, [0x1ca6] = 0xe72c, [0x1ca7] = 0xe72d,
  [0x1ca8] = 0xe72e, [0x1ca9] = 0xe72f, [0x1caa] = 0xe730, [0x1cab] = 0xe731,
  [0x1cac] = 0xe732, [0x1cad] = 0xe733, [0x1cae] = 0xe734, [0x1caf] = 0xe735,
  [0x1cb0] = 0xe736, [0x1cb1] = 0xe737, [0x1cb2] = 0xe738, [0x1cb3] = 0xe739,
  [0x1cb4] = 0xe73a, [0x1cb5] = 0xe73b, [0x1cb6] = 0xe73c, [0x1cb7] = 0xe73d,
  [0x1cb8] = 0xe73e, [0x1cb9] = 0xe73f, [0x1cba] = 0xe740, [0x1cbb] = 0xe741,
  [0x1cbc] = 0xe742, [0x1cbd] = 0xe743, [0x1cbe] = 0xe744, [0x1cc0] = 0xe745,
  [0x1cc1] = 0xe746, [0x1cc2] = 0xe747, [0x1cc3] = 0xe748, [0x1cc4] = 0xe749,
  [0x1cc5] = 0xe74a, [0x1cc6] = 0xe74b, [0x1cc7] = 0xe74c, [0x1cc8] = 0xe74d,
  [0x1cc9] = 0xe74e, [0x1cca] = 0xe74f, [0x1ccb] = 0xe750, [0x1ccc] = 0xe751,
  [0x1ccd] = 0xe752, [0x1cce] = 0xe753, [0x1ccf] = 0xe754, [0x1cd0] = 0xe755,
  [0x1cd1] = 0xe756, [0x1cd2] = 0xe757, [0x1cd3] = 0xe758, [0x1cd4] = 0xe759,
  [0x1cd5] = 0xe75a, [0x1cd6] = 0xe75b, [0x1cd7] = 0xe75c, [0x1cd8] = 0xe75d,
  [0x1cd9] = 0xe75e, [0x1cda] = 0xe75f, [0x1cdb] = 0xe760, [0x1cdc] = 0xe761,
  [0x1cdd] = 0xe762, [0x1cde] = 0xe763, [0x1cdf] = 0xe764, [0x1ce0] = 0xe765,
  [0x1ce1] = 0x0410, [0x1ce2] = 0x0411, [0x1ce3] = 0x0412, [0x1ce4] = 0x0413,
  [0x1ce5] = 0x0414, [0x1ce6] = 0x0415, [0x1ce7] = 0x0401, [0x1ce8] = 0x0416,
  [0x1ce9] = 0x0417, [0x1cea] = 0x0418, [0x1ceb] = 0x0419, [0x1cec] = 0x041a,
  [0x1ced] = 0x041b, [0x1cee] = 0x041c, [0x1cef] = 0x041d, [0x1cf0] = 0x041e,
  [0x1cf1] = 0x041f, [0x1cf2] = 0x0420, [0x1cf3] = 0x0421, [0x1cf4] = 0x0422,
  [0x1cf5] = 0x0423, [0x1cf6] = 0x0424, [0x1cf7] = 0x0425, [0x1cf8] = 0x0426,
  [0x1cf9] = 0x0427, [0x1cfa] = 0x0428, [0x1cfb] = 0x0429, [0x1cfc] = 0x042a,
  [0x1cfd] = 0x042b, [0x1cfe] = 0x042c, [0x1cff] = 0x042d, [0x1d00] = 0x042e,
  [0x1d01] = 0x042f, [0x1d02] = 0xe7a0, [0x1d03] = 0xe7a1, [0x1d04] = 0xe7a2,
  [0x1d05] = 0xe7a3, [0x1d06] = 0xe7a4, [0x1d07] = 0xe7a5, [0x1d08] = 0xe7a6,
  [0x1d09] = 0xe7a7, [0x1d0a] = 0xe7a8, [0x1d0b] = 0xe7a9, [0x1d0c] = 0xe7aa,
  [0x1d0d] = 0xe7ab, [0x1d0e] = 0xe7ac, [0x1d0f] = 0xe7ad, [0x1d10] = 0xe7ae,
  [0x1d11] = 0x0430, [0x1d12] = 0x0431, [0x1d13] = 0x0432, [0x1d14] = 0x0433,
  [0x1d15] = 0x0434, [0x1d16] = 0x0435, [0x1d17] = 0x0451, [0x1d18] = 0x0436,
  [0x1d19] = 0x0437, [0x1d1a] = 0x0438, [0x1d1b] = 0x0439, [0x1d1c] = 0x043a,
  [0x1d1d] = 0x043b, [0x1d1e] = 0x043c, [0x1d1f] = 0x043d, [0x1d20] = 0x043e,
  [0x1d21] = 0x043f, [0x1d22] = 0x0440, [0x1d23] = 0x0441, [0x1d24] = 0x0442,
  [0x1d25] = 0x0443, [0x1d26] = 0x0444, [0x1d27] = 0x0445, [0x1d28] = 0x0446,
  [0x1d29] = 0x0447, [0x1d2a] = 0x0448, [0x1d2b] = 0x0449, [0x1d2c] = 0x044a,
  [0x1d2d] = 0x044b, [0x1d2e] = 0x044c, [0x1d2f] = 0x044d, [0x1d30] = 0x044e,
  [0x1d31] = 0x044f, [0x1d32] = 0xe7af, [0x1d33] = 0xe7b0, [0x1d34] = 0xe7b1,
  [0x1d35] = 0xe7b2, [0x1d36] = 0xe7b3, [0x1d37] = 0xe7b4, [0x1d38] = 0xe7b5,
  [0x1d39] = 0xe7b6, [0x1d3a] = 0xe7b7, [0x1d3b] = 0xe7b8, [0x1d3c] = 0xe7b9,
  [0x1d3d] = 0xe7ba, [0x1d3e] = 0xe7bb, [0x1d40] = 0x02ca, [0x1d41] = 0x02cb,
  [0x1d42] = 0x02d9, [0x1d43] = 0x2013, [0x1d44] = 0x2015, [0x1d45] = 0x2025,
  [0x1d46] = 0x2035, [0x1d47] = 0x2105, [0x1d48] = 0x2109, [0x1d49] = 0x2196,
  [0x1d4a] = 0x2197, [0x1d4b] = 0x2198, [0x1d4c] = 0x2199, [0x1d4d] = 0x2215,
  [0x1d4e] = 0x221f, [0x1d4f] = 0x2223, [0x1d50] = 0x2252, [0x1d51] = 0x2266,
  [0x1d52] = 0x2267, [0x1d53] = 0x22bf, [0x1d54] = 0x2550, [0x1d55] = 0x2551,
  [0x1d56] = 0x2552, [0x1d57] = 0x2553, [0x1d58] = 0x2554, [0x1d59] = 0x2555,
  [0x1d5a] = 0x2556, [0x1d5b] = 0x2557, [0x1d5c] = 0x2558, [0x1d5d] = 0x2559,
  [0x1d5e] = 0x255a, [0x1d5f] = 0x255b, [0x1d60] = 0x255c, [0x1d61] = 0x255d,
  [0x1d62] = 0x255e, [0x1d63] = 0x255f, [0x1d64] = 0x2560, [0x1d65] = 0x2561,
  [0x1d66] = 0x2562, [0x1d67] = 0x2563, [0x1d68] = 0x2564, [0x1d69] = 0x2565,
  [0x1d6a] = 0x2566, [0x1d6b] = 0x2567, [0x1d6c] = 0x2568, [0x1d6d] = 0x2569,
  [0x1d6e] = 0x256a, [0x1d6f] = 0x256b, [0x1d70] = 0x256c, [0x1d71] = 0x256d,
  [0x1d72] = 0x256e, [0x1d73] = 0x256f, [0x1d74] = 0x2570, [0x1d75] = 0x2571,
  [0x1d76] = 0x2572, [0x1d77] = 0x2573, [0x1d78] = 0x2581, [0x1d79] = 0x2582,
  [0x1d7a] = 0x2583, [0x1d7b] = 0x2584, [0x1d7c] = 0x2585, [0x1d7d] = 0x2586,
  [0x1d7e] = 0x2587, [0x1d80] = 0x2588, [0x1d81] = 0x2589, [0x1d82] = 0x258a,
  [0x1d83] = 0x258b, [0x1d84] = 0x258c, [0x1d85] = 0x258d, [0x1d86] = 0x258e,
  [0x1d87] = 0x258f, [0x1d88] = 0x2593, [0x1d89] = 0x2594, [0x1d8a] = 0x2595,
  [0x1d8b] = 0x25bc, [0x1d8c] = 0x25bd, [0x1d8d] = 0x25e2, [0x1d8e] = 0x25e3,
  [0x1d8f] = 0x25e4, [0x1d90] = 0x25e5, [0x1d91] = 0x2609, [0x1d92] = 0x2295,
  [0x1d93] = 0x3012, [0x1d94] = 0x301d, [0x1d95] = 0x301e, [0x1d96] = 0xe7bc,
  [0x1d97] = 0xe7bd, [0x1d98] = 0xe7be, [0x1d99] = 0xe7bf, [0x1d9a] = 0xe7c0,
  [0x1d9b] = 0xe7c1, [0x1d9c] = 0xe7c2, [0x1d9d] = 0xe7c3, [0x1d9e] = 0xe7c4,
  [0x1d9f] = 0xe7c5, [0x1da0] = 0xe7c6, [0x1da1] = 0x0101, [0x1da2] = 0x00e1,
  [0x1da3] = 0x01ce, [0x1da4] = 0x00e0, [0x1da5] = 0x0113, [0x1da6] = 0x00e9,
  [0x1da7] = 0x011b, [0x1da8] = 0x00e8, [0x1da9] = 0x012b, [0x1daa] = 0x00ed,
  [0x1dab] = 0x01d0, [0x1dac] = 0x00ec, [0x1dad] = 0x014d, [0x1dae] = 0x00f3,
  [0x1daf] = 0x01d2, [0x1db0] = 0x00f2, [0x1db1] = 0x016b, [0x1db2] = 0x00fa,
  [0x1db3] = 0x01d4, [0x1db4] = 0x00f9, [0x1db5] = 0x01d6, [0x1db6] = 0x01d8,
  [0x1db7] = 0x01da, [0x1db8] = 0x01dc, [0x1db9] = 0x00fc, [0x1dba] = 0x00ea,
  [0x1dbb] = 0x0251, [0x1dbc] = 0x1e3f, [0x1dbd] = 0x0144, [0x1dbe] = 0x0148,
  [0x1dbf] = 0x01f9, [0x1dc0] = 0x0261, [0x1dc1] = 0xe7c9, [0x1dc2] = 0xe7ca,
  [0x1dc3] = 0xe7cb, [0x1dc4] = 0xe7cc, [0x1dc5] = 0x3105, [0x1dc6] = 0x3106,
  [0x1dc7] = 0x3107, [0x1dc8] = 0x3108, [0x1dc9] = 0x3109, [0x1dca] = 0x310a,
  [0x1dcb] = 0x310b, [0x1dcc] = 0x310c, [0x1dcd] = 0x310d, [0x1dce] = 0x310e,
  [0x1dcf] = 0x310f, [0x1dd0] = 0x3110, [0x1dd1] = 0x3111, [0x1dd2] = 0x3112,
  [0x1dd3] = 0x3113, [0x1dd4] = 0x3114, [0x1dd5] = 0x3115, [0x1dd6] = 0x3116,
  [0x1dd7] = 0x3117, [0x1dd8] = 0x3118, [0x1dd9] = 0x3119, [0x1dda] = 0x311a,
  [0x1ddb] = 0x311b, [0x1ddc] = 0x311c, [0x1ddd] = 0x311d, [0x1dde] = 0x311e,
  [0x1ddf] = 0x311f, [0x1de0] = 0x3120, [0x1de1] = 0x3121, [0x1de2] = 0x3122,
  [0x1de3] = 0x3123, [0x1de4] = 0x3124, [0x1de5] = 0x3125, [0x1de6] = 0x3126,
  [0x1de7] = 0x3127, [0x1de8] = 0x3128, [0x1de9] = 0x3129, [0x1dea] = 0xe7cd,
  [0x1deb] = 0xe7ce, [0x1dec] = 0xe7cf, [0x1ded] = 0xe7d0, [0x1dee] = 0xe7d1,
  [0x1def] = 0xe7d2, [0x1df0] = 0xe7d3, [0x1df1] = 0xe7d4, [0x1df2] = 0xe7d5,
  [0x1df3] = 0xe7d6, [0x1df4] = 0xe7d7, [0x1df5] = 0xe7d8, [0x1df6] = 0xe7d9,
  [0x1df7] = 0xe7da, [0x1df8] = 0xe7db, [0x1df9] = 0xe7dc, [0x1dfa] = 0xe7dd,
  [0x1dfb] = 0xe7de, [0x1dfc] = 0xe7df, [0x1dfd] = 0xe7e0, [0x1dfe] = 0xe7e1,
  [0x1e00] = 0x3021, [0x1e01] = 0x3022, [0x1e02] = 0x3023, [0x1e03] = 0x3024,
  [0x1e04] = 0x3025, [0x1e05] = 0x3026, [0x1e06] = 0x3027, [0x1e07] = 0x3028,
  [0x1e08] = 0x3029, [0x1e09] = 0x32a3, [0x1e0a] = 0x338e, [0x1e0b] = 0x338f,
  [0x1e0c] = 0x339c, [0x1e0d] = 0x339d, [0x1e0e] = 0x339e, [0x1e0f] = 0x33a1,
  [0x1e10] = 0x33c4, [0x1e11] = 0x33ce, [0x1e12] = 0x33d1, [0x1e13] = 0x33d2,
  [0x1e14] = 0x33d5, [0x1e15] = 0xfe30, [0x1e16] = 0xffe2, [0x1e17] = 0xffe4,
  [0x1e18] = 0xe7e2, [0x1e19] = 0x2121, [0x1e1a] = 0x3231, [0x1e1b] = 0xe7e3,
  [0x1e1c] = 0x2010, [0x1e1d] = 0xe7e4, [0x1e1e] = 0xe7e5, [0x1e1f] = 0xe7e6,
  [0x1e20] = 0x30fc, [0x1e21] = 0x309b, [0x1e22] = 0x309c, [0x1e23] = 0x30fd,
  [0x1e24] = 0x30fe, [0x1e25] = 0x3006, [0x1e26] = 0x309d, [0x1e27] = 0x309e,
  [0x1e28] = 0xfe49, [0x1e29] = 0xfe4a, [0x1e2a] = 0xfe4b, [0x1e2b] = 0xfe4c,
  [0x1e2c] = 0xfe4d, [0x1e2d] = 0xfe4e, [0x1e2e] = 0xfe4f, [0x1e2f] = 0xfe50,
  [0x1e30] = 0xfe51, [0x1e31] = 0xfe52, [0x1e32] = 0xfe54, [0x1e33] = 0xfe55,
  [0x1e34] = 0xfe56, [0x1e35] = 0xfe57, [0x1e36] = 0xfe59, [0x1e37] = 0xfe5a,
  [0x1e38] = 0xfe5b, [0x1e39] = 0xfe5c, [0x1e3a] = 0xfe5d, [0x1e3b] = 0xfe5e,
  [0x1e3c] = 0xfe5f, [0x1e3d] = 0xfe60, [0x1e3e] = 0xfe61, [0x1e40] = 0xfe62,
  [0x1e41] = 0xfe63, [0x1e42] = 0xfe64, [0x1e43] = 0xfe65, [0x1e44] = 0xfe66,
  [0x1e45] = 0xfe68, [0x1e46] = 0xfe69, [0x1e47] = 0xfe6a, [0x1e48] = 0xfe6b,
  [0x1e49] = 0x303e, [0x1e4a] = 0x2ff0, [0x1e4b] = 0x2ff1, [0x1e4c] = 0x2ff2,
  [0x1e4d] = 0x2ff3, [0x1e4e] = 0x2ff4, [0x1e4f] = 0x2ff5, [0x1e50] = 0x2ff6,
  [0x1e51] = 0x2ff7, [0x1e52] = 0x2ff8, [0x1e53] = 0x2ff9, [0x1e54] = 0x2ffa,
  [0x1e55] = 0x2ffb, [0x1e56] = 0x3007, [0x1e57] = 0xe7f4, [0x1e58] = 0xe7f5,
  [0x1e59] = 0xe7f6, [0x1e5a] = 0xe7f7, [0x1e5b] = 0xe7f8, [0x1e5c] = 0xe7f9,
  [0x1e5d] = 0xe7fa, [0x1e5e] = 0xe7fb, [0x1e5f] = 0xe7fc, [0x1e60] = 0xe7fd,
  [0x1e61] = 0xe7fe, [0x1e62] = 0xe7ff, [0x1e63] = 0xe800, [0x1e64] = 0x2500,
  [0x1e65] = 0x2501, [0x1e66] = 0x2502, [0x1e67] = 0x2503, [0x1e68] = 0x2504,
  [0x1e69] = 0x2505, [0x1e6a] = 0x2506, [0x1e6b] = 0x2507, [0x1e6c] = 0x2508,
  [0x1e6d] = 0x2509, [0x1e6e] = 0x250a, [0x1e6f] = 0x250b, [0x1e70] = 0x250c,
  [0x1e71] = 0x250d, [0x1e72] = 0x250e, [0x1e73] = 0x250f, [0x1e74] = 0x2510,
  [0x1e75] = 0x2511, [0x1e76] = 0x2512, [0x1e77] = 0x2513, [0x1e78] = 0x2514,
  [0x1e79] = 0x2515, [0x1e7a] = 0x2516, [0x1e7b] = 0x2517, [0x1e7c] = 0x2518,
  [0x1e7d] = 0x2519, [0x1e7e] = 0x251a, [0x1e7f] = 0x251b, [0x1e80] = 0x251c,
  [0x1e81] = 0x251d, [0x1e82] = 0x251e, [0x1e83] = 0x251f, [0x1e84] = 0x2520,
  [0x1e85] = 0x2521, [0x1e86] = 0x2522, [0x1e87] = 0x2523, [0x1e88] = 0x2524,
  [0x1e89] = 0x2525, [0x1e8a] = 0x2526, [0x1e8b] = 0x2527, [0x1e8c] = 0x2528,
  [0x1e8d] = 0x2529, [0x1e8e] = 0x252a, [0x1e8f] = 0x252b, [0x1e90] = 0x252c,
  [0x1e91] = 0x252d, [0x1e92] = 0x252e, [0x1e93] = 0x252f, [0x1e94] = 0x2530,
  [0x1e95] = 0x2531, [0x1e96] = 0x2532, [0x1e97] = 0x2533, [0x1e98] = 0x2534,
  [0x1e99] = 0x2535, [0x1e9a] = 0x2536, [0x1e9b] = 0x2537, [0x1e9c] = 0x2538,
  [0x1e9d] = 0x2539, [0x1e9e] = 0x253a, [0x1e9f] = 0x253b, [0x1ea0] = 0x253c,
  [0x1ea1] = 0x253d, [0x1ea2] = 0x253e, [0x1ea3] = 0x253f, [0x1ea4] = 0x2540,
  [0x1ea5] = 0x2541, [0x1ea6] = 0x2542, [0x1ea7] = 0x2543, [0x1ea8] = 0x2544,
  [0x1ea9] = 0x2545, [0x1eaa] = 0x2546, [0x1eab] = 0x2547, [0x1eac] = 0x2548,
  [0x1ead] = 0x2549, [0x1eae] = 0x254a, [0x1eaf] = 0x254b, [0x1eb0] = 0xe801,
  [0x1eb1] = 0xe802, [0x1eb2] = 0xe803, [0x1eb3] = 0xe804, [0x1eb4] = 0xe805,
  [0x1eb5] = 0xe806, [0x1eb6] = 0xe807, [0x1eb7] = 0xe808, [0x1eb8] = 0xe809,
  [0x1eb9] = 0xe80a, [0x1eba] = 0xe80b, [0x1ebb] = 0xe80c, [0x1ebc] = 0xe80d,
  [0x1ebd] = 0xe80e, [0x1ebe] = 0xe80f, [0x1ec0] = 0x72dc, [0x1ec1] = 0x72dd,
  [0x1ec2] = 0x72df, [0x1ec3] = 0x72e2, [0x1ec4] = 0x72e3, [0x1ec5] = 0x72e4,
  [0x1ec6] = 0x72e5, [0x1ec7] = 0x72e6, [0x1ec8] = 0x72e7, [0x1ec9] = 0x72ea,
  [0x1eca] = 0x72eb, [0x1ecb] = 0x72f5, [0x1ecc] = 0x72f6, [0x1ecd] = 0x72f9,
  [0x1ece] = 0x72fd, [0x1ecf] = 0x72fe, [0x1ed0] = 0x72ff, [0x1ed1] = 0x7300,
  [0x1ed2] = 0x7302, [0x1ed3] = 0x7304, [0x1ed4] = 0x7305, [0x1ed5] = 0x7306,
  [0x1ed6] = 0x7307, [0x1ed7] = 0x7308, [0x1ed8] = 0x7309, [0x1ed9] = 0x730b,
  [0x1eda] = 0x730c, [0x1edb] = 0x730d, [0x1edc] = 0x730f, [0x1edd] = 0x7310,
  [0x1ede] = 0x7311, [0x1edf] = 0x7312, [0x1ee0] = 0x7314, [0x1ee1] = 0x7318,
  [0x1ee2] = 0x7319, [0x1ee3] = 0x731a, [0x1ee4] = 0x731f, [0x1ee5] = 0x7320,
  [0x1ee6] = 0x7323, [0x1ee7] = 0x7324, [0x1ee8] = 0x7326, [0x1ee9] = 0x7327,
  [0x1eea] = 0x7328, [0x1eeb] = 0x732d, [0x1eec] = 0x732f, [0x1eed] = 0x7330,
  [0x1eee] = 0x7332, [0x1eef] = 0x7333, [0x1ef0] = 0x7335, [0x1ef1] = 0x7336,
  [0x1ef2] = 0x733a, [0x1ef3] = 0x733b, [0x1ef4] = 0x733c, [0x1ef5] = 0x733d,
  [0x1ef6] = 0x7340, [0x1ef7] = 0x7341, [0x1ef8] = 0x7342, [0x1ef9] = 0x7343,
  [0x1efa] = 0x7344, [0x1efb] = 0x7345, [0x1efc] = 0x7346, [0x1efd] = 0x7347,
  [0x1efe] = 0x7348, [0x1f00] = 0x7349, [0x1f01] = 0x734a, [0x1f02] = 0x734b,
  [0x1f03] = 0x734c, [0x1f04] = 0x734e, [0x1f05] = 0x734f, [0x1f06] = 0x7351,
  [0x1f07] = 0x7353, [0x1f08] = 0x7354, [0x1f09] = 0x7355, [0x1f0a] = 0x7356,
  [0x1f0b] = 0x7358, [0x1f0c] = 0x7359, [0x1f0d] = 0x735a, [0x1f0e] = 0x735b,
  [0x1f0f] = 0x735c, [0x1f10] = 0x735d, [0x1f11] = 0x735e, [0x1f12] = 0x735f,
  [0x1f13] = 0x7361, [0x1f14] = 0x7362, [0x1f15] = 0x7363, [0x1f16] = 0x7364,
  [0x1f17] = 0x7365, [0x1f18] = 0x7366, [0x1f19] = 0x7367, [0x1f1a] = 0x7368,
  [0x1f1b] = 0x7369, [0x1f1c] = 0x736a, [0x1f1d] = 0x736b, [0x1f1e] = 0x736e,
  [0x1f1f] = 0x7370, [0x1f20] = 0x7371, [0x1f21] = 0xe000, [0x1f22] = 0xe001,
  [0x1f23] = 0xe002, [0x1f24] = 0xe003, [0x1f25] = 0xe004, [0x1f26] = 0xe005,
  [0x1f27] = 0xe006, [0x1f28] = 0xe007, [0x1f29] = 0xe008, [0x1f2a] = 0xe009,
  [0x1f2b] = 0xe00a, [0x1f2c] = 0xe00b, [0x1f2d] = 0xe00c, [0x1f2e] = 0xe00d,
  [0x1f2f] = 0xe00e, [0x1f30] = 0xe00f, [0x1f31] = 0xe010, [0x1f32] = 0xe011,
  [0x1f33] = 0xe012, [0x1f34] = 0xe013, [0x1f35] = 0xe014, [0x1f36] = 0xe015,
  [0x1f37] = 0xe016, [0x1f38] = 0xe017, [0x1f39] = 0xe018, [0x1f3a] = 0xe019,
  [0x1f3b] = 0xe01a, [0x1f3c] = 0xe01b, [0x1f3d] = 0xe01c, [0x1f3e] = 0xe01d,
  [0x1f3f] = 0xe01e, [0x1f40] = 0xe01f, [0x1f41] = 0xe020, [0x1f42] = 0xe021,
  [0x1f43] = 0xe022, [0x1f44] = 0xe023, [0x1f45] = 0xe024, [0x1f46] = 0xe025,
  [0x1f47] = 0xe026, [0x1f48] = 0xe027, [0x1f49] = 0xe028, [0x1f4a] = 0xe029,
  [0x1f4b] = 0xe02a, [0x1f4c] = 0xe02b, [0x1f4d] = 0xe02c, [0x1f4e] = 0xe02d,
  [0x1f4f] = 0xe02e, [0x1f50] = 0xe02f, [0x1f51] = 0xe030, [0x1f52] = 0xe031,
  [0x1f53] = 0xe032, [0x1f54] = 0xe033, [0x1f55] = 0xe034, [0x1f56] = 0xe035,
  [0x1f57] = 0xe036, [0x1f58] = 0xe037, [0x1f59] = 0xe038, [0x1f5a] = 0xe039,
  [0x1f5b] = 0xe03a, [0x1f5c] = 0xe03b, [0x1f5d] = 0xe03c, [0x1f5e] = 0xe03d,
  [0x1f5f] = 0xe03e, [0x1f60] = 0xe03f, [0x1f61] = 0xe040, [0x1f62] = 0xe041,
  [0x1f63] = 0xe042, [0x1f64] = 0xe043, [0x1f65] = 0xe044, [0x1f66] = 0xe045,
  [0x1f67] = 0xe046, [0x1f68] = 0xe047, [0x1f69] = 0xe048, [0x1f6a] = 0xe049,
  [0x1f6b] = 0xe04a, [0x1f6c] = 0xe04b, [0x1f6d] = 0xe04c, [0x1f6e] = 0xe04d,
  [0x1f6f] = 0xe04e, [0x1f70] = 0xe04f, [0x1f71] = 0xe050, [0x1f72] = 0xe051,
  [0x1f73] = 0xe052, [0x1f74] = 0xe053, [0x1f75] = 0xe054, [0x1f76] = 0xe055,
  [0x1f77] = 0xe056, [0x1f78] = 0xe057, [0x1f79] = 0xe058, [0x1f7a] = 0xe059,
  [0x1f7b] = 0xe05a, [0x1f7c] = 0xe05b, [0x1f7d] = 0xe05c, [0x1f7e] = 0xe05d,
  [0x1f80] = 0x7372, [0x1f81] = 0x7373, [0x1f82] = 0x7374, [0x1f83] = 0x7375,
  [0x1f84] = 0x7376, [0x1f85] = 0x7377, [0x1f86] = 0x7378, [0x1f87] = 0x7379,
  [0x1f88] = 0x737a, [0x1f89] = 0x737b, [0x1f8a] = 0x737c, [0x1f8b] = 0x737d,
  [0x1f8c] = 0x737f, [0x1f8d] = 0x7380, [0x1f8e] = 0x7381, [0x1f8f] = 0x7382,
  [0x1f90] = 0x7383, [0x1f91] = 0x7385, [0x1f92] = 0x7386, [0x1f93] = 0x7388,
  [0x1f94] = 0x738a, [0x1f95] = 0x738c, [0x1f96] = 0x738d, [0x1f97] = 0x738f,
  [0x1f98] = 0x7390, [0x1f99] = 0x7392, [0x1f9a] = 0x7393, [0x1f9b] = 0x7394,
  [0x1f9c] = 0x7395, [0x1f9d] = 0x7397, [0x1f9e] = 0x7398, [0x1f9f] = 0x7399,
  [0x1fa0] = 0x739a, [0x1fa1] = 0x739c, [0x1fa2] = 0x739d, [0x1fa3] = 0x739e,
  [0x1fa4] = 0x73a0, [0x1fa5] = 0x73a1, [0x1fa6] = 0x73a3, [0x1fa7] = 0x73a4,
  [0x1fa8] = 0x73a5, [0x1fa9] = 0x73a6, [0x1faa] = 0x73a7, [0x1fab] = 0x73a8,
  [0x1fac] = 0x73aa, [0x1fad] = 0x73ac, [0x1fae] = 0x73ad, [0x1faf] = 0x73b1,
  [0x1fb0] = 0x73b4, [0x1fb1] = 0x73b5, [0x1fb2] = 0x73b6, [0x1fb3] = 0x73b8,
  [0x1fb4] = 0x73b9, [0x1fb5] = 0x73bc, [0x1fb6] = 0x73bd, [0x1fb7] = 0x73be,
  [0x1fb8] = 0x73bf, [0x1fb9] = 0x73c1, [0x1fba] = 0x73c3, [0x1fbb] = 0x73c4,
  [0x1fbc] = 0x73c5, [0x1fbd] = 0x73c6, [0x1fbe] = 0x73c7, [0x1fc0] = 0x73cb,
  [0x1fc1] = 0x73cc, [0x1fc2] = 0x73ce, [0x1fc3] = 0x73d2, [0x1fc4] = 0x73d3,
  [0x1fc5] = 0x73d4, [0x1fc6] = 0x73d5, [0x1fc7] = 0x73d6, [0x1fc8] = 0x73d7,
  [0x1fc9] = 0x73d8, [0x1fca] = 0x73da, [0x1fcb] = 0x73db, [0x1fcc] = 0x73dc,
  [0x1fcd] = 0x73dd, [0x1fce] = 0x73df, [0x1fcf] = 0x73e1, [0x1fd0] = 0x73e2,
  [0x1fd1] = 0x73e3, [0x1fd2] = 0x73e4, [0x1fd3] = 0x73e6, [0x1fd4] = 0x73e8,
  [0x1fd5] = 0x73ea, [0x1fd6] = 0x73eb, [0x1fd7] = 0x73ec, [0x1fd8] = 0x73ee,
  [0x1fd9] = 0x73ef, [0x1fda] = 0x73f0, [0x1fdb] = 0x73f1, [0x1fdc] = 0x73f3,
  [0x1fdd] = 0x73f4, [0x1fde] = 0x73f5, [0x1fdf] = 0x73f6, [0x1fe0] = 0x73f7,
  [0x1fe1] = 0xe05e, [0x1fe2] = 0xe05f, [0x1fe3] = 0xe060, [0x1fe4] = 0xe061,
  [0x1fe5] = 0xe062, [0x1fe6] = 0xe063, [0x1fe7] = 0xe064, [0x1fe8] = 0xe065,
  [0x1fe9] = 0xe066, [0x1fea] = 0xe067, [0x1feb] = 0xe068, [0x1fec] = 0xe069,
  [0x1fed] = 0xe06a, [0x1fee] = 0xe06b, [0x1fef] = 0xe06c, [0x1ff0] = 0xe06d,
  [0x1ff1] = 0xe06e, [0x1ff2] = 0xe06f, [0x1ff3] = 0xe070, [0x1ff4] = 0xe071,
  [0x1ff5] = 0xe072, [0x1ff6] = 0xe073, [0x1ff7] = 0xe074, [0x1ff8] = 0xe075,
  [0x1ff9] = 0xe076, [0x1ffa] = 0xe077, [0x1ffb] = 0xe078, [0x1ffc] = 0xe079,
  [0x1ffd] = 0xe07a, [0x1ffe] = 0xe07b, [0x1fff] = 0xe07c, [0x2000] = 0xe07d,
  [0x2001] = 0xe07e, [0x2002] = 0xe07f, [0x2003] = 0xe080, [0x2004] = 0xe081,
  [0x2005] = 0xe082, [0x2006] = 0xe083, [0x2007] = 0xe084, [0x2008] = 0xe085,
  [0x2009] = 0xe086, [0x200a] = 0xe087, [0x200b] = 0xe088, [0x200c] = 0xe089,
  [0x200d] = 0xe08a, [0x200e] = 0xe08b, [0x200f] = 0xe08c, [0x2010] = 0xe08d,
  [0x2011] = 0xe08e, [0x2012] = 0xe08f, [0x2013] = 0xe090, [0x2014] = 0xe091,
  [0x2015] = 0xe092, [0x2016] = 0xe093, [0x2017] = 0xe094, [0x2018] = 0xe095,
  [0x2019] = 0xe096, [0x201a] = 0xe097, [0x201b] = 0xe098, [0x201c] = 0xe099,
  [0x201d] = 0xe09a, [0x201e] = 0xe09b, [0x201f] = 0xe09c, [0x2020] = 0xe09d,
  [0x2021] = 0xe09e, [0x2022] = 0xe09f, [0x2023] = 0xe0a0, [0x2024] = 0xe0a1,
  [0x2025] = 0xe0a2, [0x2026] = 0xe0a3, [0x2027] = 0xe0a4, [0x2028] = 0xe0a5,
  [0x2029] = 0xe0a6, [0x202a] = 0xe0a7, [0x202b] = 0xe0a8, [0x202c] = 0xe0a9,
  [0x202d] = 0xe0aa, [0x202e] = 0xe0ab, [0x202f] = 0xe0ac, [0x2030] = 0xe0ad,
  [0x2031] = 0xe0ae, [0x2032] = 0xe0af, [0x2033] = 0xe0b0, [0x2034] = 0xe0b1,
  [0x2035] = 0xe0b2, [0x2036] = 0xe0b3, [0x2037] = 0xe0b4, [0x2038] = 0xe0b5,
  [0x2039] = 0xe0b6, [0x203a] = 0xe0b7, [0x203b] = 0xe0b8, [0x203c] = 0xe0b9,
  [0x203d] = 0xe0ba, [0x203e] = 0xe0bb, [0x2040] = 0x73f8, [0x2041] = 0x73f9,
  [0x2042] = 0x73fa, [0x2043] = 0x73fb, [0x2044] = 0x73fc, [0x2045] = 0x73fd,
  [0x2046] = 0x73fe, [0x2047] = 0x73ff, [0x2048] = 0x7400, [0x2049] = 0x7401,
  [0x204a] = 0x7402, [0x204b] = 0x7404, [0x204c] = 0x7407, [0x204d] = 0x7408,
  [0x204e] = 0x740b, [0x204f] = 0x740c, [0x2050] = 0x740d, [0x2051] = 0x740e,
  [0x2052] = 0x7411, [0x2053] = 0x7412, [0x2054] = 0x7413, [0x2055] = 0x7414,
  [0x2056] = 0x7415, [0x2057] = 0x7416, [0x2058] = 0x7417, [0x2059] = 0x7418,
  [0x205a] = 0x7419, [0x205b] = 0x741c, [0x205c] = 0x741d, [0x205d] = 0x741e,
  [0x205e] = 0x741f, [0x205f] = 0x7420, [0x2060] = 0x7421, [0x2061] = 0x7423,
  [0x2062] = 0x7424, [0x2063] = 0x7427, [0x2064] = 0x7429, [0x2065] = 0x742b,
  [0x2066] = 0x742d, [0x2067] = 0x742f, [0x2068] = 0x7431, [0x2069] = 0x7432,
  [0x206a] = 0x7437, [0x206b] = 0x7438, [0x206c] = 0x7439, [0x206d] = 0x743a,
  [0x206e] = 0x743b, [0x206f] = 0x743d, [0x2070] = 0x743e, [0x2071] = 0x743f,
  [0x2072] = 0x7440, [0x2073] = 0x7442, [0x2074] = 0x7443, [0x2075] = 0x7444,
  [0x2076] = 0x7445, [0x2077] = 0x7446, [0x2078] = 0x7447, [0x2079] = 0x7448,
  [0x207a] = 0x7449, [0x207b] = 0x744a, [0x207c] = 0x744b, [0x207d] = 0x744c,
  [0x207e] = 0x744d, [0x2080] = 0x744e, [0x2081] = 0x744f, [0x2082] = 0x7450,
  [0x2083] = 0x7451, [0x2084] = 0x7452, [0x2085] = 0x7453, [0x2086] = 0x7454,
  [0x2087] = 0x7456, [0x2088] = 0x7458, [0x2089] = 0x745d, [0x208a] = 0x7460,
  [0x208b] = 0x7461, [0x208c] = 0x7462, [0x208d] = 0x7463, [0x208e] = 0x7464,
  [0x208f] = 0x7465, [0x2090] = 0x7466, [0x2091] = 0x7467, [0x2092] = 0x7468,
  [0x2093] = 0x7469, [0x2094] = 0x746a, [0x2095] = 0x746b, [0x2096] = 0x746c,
  [0x2097] = 0x746e, [0x2098] = 0x746f, [0x2099] = 0x7471, [0x209a] = 0x7472,
  [0x209b] = 0x7473, [0x209c] = 0x7474, [0x209d] = 0x7475, [0x209e] = 0x7478,
  [0x209f] = 0x7479, [0x20a0] = 0x747a, [0x20a1] = 0xe0bc, [0x20a2] = 0xe0bd,
  [0x20a3] = 0xe0be, [0x20a4] = 0xe0bf, [0x20a5] = 0xe0c0, [0x20a6] = 0xe0c1,
  [0x20a7] = 0xe0c2, [0x20a8] = 0xe0c3, [0x20a9] = 0xe0c4, [0x20aa] = 0xe0c5,
  [0x20ab] = 0xe0c6, [0x20ac] = 0xe0c7, [0x20ad] = 0xe0c8, [0x20ae] = 0xe0c9,
  [0x20af] = 0xe0ca, [0x20b0] = 0xe0cb, [0x20b1] = 0xe0cc, [0x20b2] = 0xe0cd,
  [0x20b3] = 0xe0ce, [0x20b4] = 0xe0cf, [0x20b5] = 0xe0d0, [0x20b6] = 0xe0d1,
  [0x20b7] = 0xe0d2, [0x20b8] = 0xe0d3, [0x20b9] = 0xe0d4, [0x20ba] = 0xe0d5,
  [0x20bb] = 0xe0d6, [0x20bc] = 0xe0d7, [0x20bd] = 0xe0d8, [0x20be] = 0xe0d9,
  [0x20bf] = 0xe0da, [0x20c0] = 0xe0db, [0x20c1] = 0xe0dc, [0x20c2] = 0xe0dd,
  [0x20c3] = 0xe0de, [0x20c4] = 0xe0df, [0x20c5] = 0xe0e0, [0x20c6] = 0xe0e1,
  [0x20c7] = 0xe0e2, [0x20c8] = 0xe0e3, [0x20c9] = 0xe0e4, [0x20ca] = 0xe0e5,
  [0x20cb] = 0xe0e6, [0x20cc] = 0xe0e7, [0x20cd] = 0xe0e8, [0x20ce] = 0xe0e9,
  [0x20cf] = 0xe0ea, [0x20d0] = 0xe0eb, [0x20d1] = 0xe0ec, [0x20d2] = 0xe0ed,
  [0x20d3] = 0xe0ee, [0x20d4] = 0xe0ef, [0x20d5] = 0xe0f0, [0x20d6] = 0xe0f1,
  [0x20d7] = 0xe0f2, [0x20d8] = 0xe0f3, [0x20d9] = 0xe0f4, [0x20da] = 0xe0f5,
  [0x20db] = 0xe0f6, [0x20dc] = 0xe0f7, [0x20dd] = 0xe0f8, [0x20de] = 0xe0f9,
  [0x20df] = 0xe0fa, [0x20e0] = 0xe0fb, [0x20e1] = 0xe0fc, [0x20e2] = 0xe0fd,
  [0x20e3] = 0xe0fe, [0x20e4] = 0xe0ff, [0x20e5] = 0xe100, [0x20e6] = 0xe101,
  [0x20e7] = 0xe102, [0x20e8] = 0xe103, [0x20e9] = 0xe104, [0x20ea] = 0xe105,
  [0x20eb] = 0xe106, [0x20ec] = 0xe107, [0x20ed] = 0xe108, [0x20ee] = 0xe109,
  [0x20ef] = 0xe10a, [0x20f0] = 0xe10b, [0x20f1] = 0xe10c, [0x20f2] = 0xe10d,
  [0x20f3] = 0xe10e, [0x20f4] = 0xe10f, [0x20f5] = 0xe110, [0x20f6] = 0xe111,
  [0x20f7] = 0xe112, [0x20f8] = 0xe113, [0x20f9] = 0xe114, [0x20fa] = 0xe115,
  [0x20fb] = 0xe116, [0x20fc] = 0xe117, [0x20fd] = 0xe118, [0x20fe] = 0xe119,
  [0x2100] = 0x747b, [0x2101] = 0x747c, [0x2102] = 0x747d, [0x2103] = 0x747f,
  [0x2104] = 0x7482, [0x2105] = 0x7484, [0x2106] = 0x7485, [0x2107] = 0x7486,
  [0x2108] = 0x7488, [0x2109] = 0x7489, [0x210a] = 0x748a, [0x210b] = 0x748c,
  [0x210c] = 0x748d, [0x210d] = 0x748f, [0x210e] = 0x7491, [0x210f] = 0x7492,
  [0x2110] = 0x7493, [0x2111] = 0x7494, [0x2112] = 0x7495, [0x2113] = 0x7496,
  [0x2114] = 0x7497, [0x2115] = 0x7498, [0x2116] = 0x7499, [0x2117] = 0x749a,
  [0x2118] = 0x749b, [0x2119] = 0x749d, [0x211a] = 0x749f, [0x211b] = 0x74a0,
  [0x211c] = 0x74a1, [0x211d] = 0x74a2, [0x211e] = 0x74a3, [0x211f] = 0x74a4,
  [0x2120] = 0x74a5, [0x2121] = 0x74a6, [0x2122] = 0x74aa, [0x2123] = 0x74ab,
  [0x2124] = 0x74ac, [0x2125] = 0x74ad, [0x2126] = 0x74ae, [0x2127] = 0x74af,
  [0x2128] = 0x74b0, [0x2129] = 0x74b1, [0x212a] = 0x74b2, [0x212b] = 0x74b3,
  [0x212c] = 0x74b4, [0x212d] = 0x74b5, [0x212e] = 0x74b6, [0x212f] = 0x74b7,
  [0x2130] = 0x74b8, [0x2131] = 0x74b9, [0x2132] = 0x74bb, [0x2133] = 0x74bc,
  [0x2134] = 0x74bd, [0x2135] = 0x74be, [0x2136] = 0x74bf, [0x2137] = 0x74c0,
  [0x2138] = 0x74c1, [0x2139] = 0x74c2, [0x213a] = 0x74c3, [0x213b] = 0x74c4,
  [0x213c] = 0x74c5, [0x213d] = 0x74c6, [0x213e] = 0x74c7, [0x2140] = 0x74c8,
  [0x2141] = 0x74c9, [0x2142] = 0x74ca, [0x2143] = 0x74cb, [0x2144] = 0x74cc,
  [0x2145] = 0x74cd, [0x2146] = 0x74ce, [0x2147] = 0x74cf, [0x2148] = 0x74d0,
  [0x2149] = 0x74d1, [0x214a] = 0x74d3, [0x214b] = 0x74d4, [0x214c] = 0x74d5,
  [0x214d] = 0x74d6, [0x214e] = 0x74d7, [0x214f] = 0x74d8, [0x2150] = 0x74d9,
  [0x2151] = 0x74da, [0x2152] = 0x74db, [0x2153] = 0x74dd, [0x2154] = 0x74df,
  [0x2155] = 0x74e1, [0x2156] = 0x74e5, [0x2157] = 0x74e7, [0x2158] = 0x74e8,
  [0x2159] = 0x74e9, [0x215a] = 0x74ea, [0x215b] = 0x74eb, [0x215c] = 0x74ec,
  [0x215d] = 0x74ed, [0x215e] = 0x74f0, [0x215f] = 0x74f1, [0x2160] = 0x74f2,
  [0x2161] = 0xe11a, [0x2162] = 0xe11b, [0x2163] = 0xe11c, [0x2164] = 0xe11d,
  [0x2165] = 0xe11e, [0x2166] = 0xe11f, [0x2167] = 0xe120, [0x2168] = 0xe121,
  [0x2169] = 0xe122, [0x216a] = 0xe123, [0x216b] = 0xe124, [0x216c] = 0xe125,
  [0x216d] = 0xe126, [0x216e] = 0xe127, [0x216f] = 0xe128, [0x2170] = 0xe129,
  [0x2171] = 0xe12a, [0x2172] = 0xe12b, [0x2173] = 0xe12c, [0x2174] = 0xe12d,
  [0x2175] = 0xe12e, [0x2176] = 0xe12f, [0x2177] = 0xe130, [0x2178] = 0xe131,
  [0x2179] = 0xe132, [0x217a] = 0xe133, [0x217b] = 0xe134, [0x217c] = 0xe135,
  [0x217d] = 0xe136, [0x217e] = 0xe137, [0x217f] = 0xe138, [0x2180] = 0xe139,
  [0x2181] = 0xe13a, [0x2182] = 0xe13b, [0x2183] = 0xe13c, [0x2184] = 0xe13d,
  [0x2185] = 0xe13e, [0x2186] = 0xe13f, [0x2187] = 0xe140, [0x2188] = 0xe141,
  [0x2189] = 0xe142, [0x218a] = 0xe143, [0x218b] = 0xe144, [0x218c] = 0xe145,
  [0x218d] = 0xe146, [0x218e] = 0xe147, [0x218f] = 0xe148, [0x2190] = 0xe149,
  [0x2191] = 0xe14a, [0x2192] = 0xe14b, [0x2193] = 0xe14c, [0x2194] = 0xe14d,
  [0x2195] = 0xe14e, [0x2196] = 0xe14f, [0x2197] = 0xe150, [0x2198] = 0xe151,
  [0x2199] = 0xe152, [0x219a] = 0xe153, [0x219b] = 0xe154, [0x219c] = 0xe155,
  [0x219d] = 0xe156, [0x219e] = 0xe157, [0x219f] = 0xe158, [0x21a0] = 0xe159,
  [0x21a1] = 0xe15a, [0x21a2] = 0xe15b, [0x21a3] = 0xe15c, [0x21a4] = 0xe15d,
  [0x21a5] = 0xe15e, [0x21a6] = 0xe15f, [0x21a7] = 0xe160, [0x21a8] = 0xe161,
  [0x21a9] = 0xe162, [0x21aa] = 0xe163, [0x21ab] = 0xe164, [0x21ac] = 0xe165,
  [0x21ad] = 0xe166, [0x21ae] = 0xe167, [0x21af] = 0xe168, [0x21b0] = 0xe169,
  [0x21b1] = 0xe16a, [0x21b2] = 0xe16b, [0x21b3] = 0xe16c, [0x21b4] = 0xe16d,
  [0x21b5] = 0xe16e, [0x21b6] = 0xe16f, [0x21b7] = 0xe170, [0x21b8] = 0xe171,
  [0x21b9] = 0xe172, [0x21ba] = 0xe173, [0x21bb] = 0xe174, [0x21bc] = 0xe175,
  [0x21bd] = 0xe176, [0x21be] = 0xe177, [0x21c0] = 0x74f3, [0x21c1] = 0x74f5,
  [0x21c2] = 0x74f8, [0x21c3] = 0x74f9, [0x21c4] = 0x74fa, [0x21c5] = 0x74fb,
  [0x21c6] = 0x74fc, [0x21c7] = 0x74fd, [0x21c8] = 0x74fe, [0x21c9] = 0x7500,
  [0x21ca] = 0x7501, [0x21cb] = 0x7502, [0x21cc] = 0x7503, [0x21cd] = 0x7505,
  [0x21ce] = 0x7506, [0x21cf] = 0x7507, [0x21d0] = 0x7508, [0x21d1] = 0x7509,
  [0x21d2] = 0x750a, [0x21d3] = 0x750b, [0x21d4] = 0x750c, [0x21d5] = 0x750e,
  [0x21d6] = 0x7510, [0x21d7] = 0x7512, [0x21d8] = 0x7514, [0x21d9] = 0x7515,
  [0x21da] = 0x7516, [0x21db] = 0x7517, [0x21dc] = 0x751b, [0x21dd] = 0x751d,
  [0x21de] = 0x751e, [0x21df] = 0x7520, [0x21e0] = 0x7521, [0x21e1] = 0x7522,
  [0x21e2] = 0x7523, [0x21e3] = 0x7524, [0x21e4] = 0x7526, [0x21e5] = 0x7527,
  [0x21e6] = 0x752a, [0x21e7] = 0x752e, [0x21e8] = 0x7534, [0x21e9] = 0x7536,
  [0x21ea] = 0x7539, [0x21eb] = 0x753c, [0x21ec] = 0x753d, [0x21ed] = 0x753f,
  [0x21ee] = 0x7541, [0x21ef] = 0x7542, [0x21f0] = 0x7543, [0x21f1] = 0x7544,
  [0x21f2] = 0x7546, [0x21f3] = 0x7547, [0x21f4] = 0x7549, [0x21f5] = 0x754a,
  [0x21f6] = 0x754d, [0x21f7] = 0x7550, [0x21f8] = 0x7551, [0x21f9] = 0x7552,
  [0x21fa] = 0x7553, [0x21fb] = 0x7555, [0x21fc] = 0x7556, [0x21fd] = 0x7557,
  [0x21fe] = 0x7558, [0x2200] = 0x755d, [0x2201] = 0x755e, [0x2202] = 0x755f,
  [0x2203] = 0x7560, [0x2204] = 0x7561, [0x2205] = 0x7562, [0x2206] = 0x7563,
  [0x2207] = 0x7564, [0x2208] = 0x7567, [0x2209] = 0x7568, [0x220a] = 0x7569,
  [0x220b] = 0x756b, [0x220c] = 0x756c, [0x220d] = 0x756d, [0x220e] = 0x756e,
  [0x220f] = 0x756f, [0x2210] = 0x7570, [0x2211] = 0x7571, [0x2212] = 0x7573,
  [0x2213] = 0x7575, [0x2214] = 0x7576, [0x2215] = 0x7577, [0x2216] = 0x757a,
  [0x2217] = 0x757b, [0x2218] = 0x757c, [0x2219] = 0x757d, [0x221a] = 0x757e,
  [0x221b] = 0x7580, [0x221c] = 0x7581, [0x221d] = 0x7582, [0x221e] = 0x7584,
  [0x221f] = 0x7585, [0x2220] = 0x7587, [0x2221] = 0xe178, [0x2222] = 0xe179,
  [0x2223] = 0xe17a, [0x2224] = 0xe17b, [0x2225] = 0xe17c, [0x2226] = 0xe17d,
  [0x2227] = 0xe17e, [0x2228] = 0xe17f, [0x2229] = 0xe180, [0x222a] = 0xe181,
  [0x222b] = 0xe182, [0x222c] = 0xe183, [0x222d] = 0xe184, [0x222e] = 0xe185,
  [0x222f] = 0xe186, [0x2230] = 0xe187, [0x2231] = 0xe188, [0x2232] = 0xe189,
  [0x2233] = 0xe18a, [0x2234] = 0xe18b, [0x2235] = 0xe18c, [0x2236] = 0xe18d,
  [0x2237] = 0xe18e, [0x2238] = 0xe18f, [0x2239] = 0xe190, [0x223a] = 0xe191,
  [0x223b] = 0xe192, [0x223c] = 0xe193, [0x223d] = 0xe194, [0x223e] = 0xe195,
  [0x223f] = 0xe196, [0x2240] = 0xe197, [0x2241] = 0xe198, [0x2242] = 0xe199,
  [0x2243] = 0xe19a, [0x2244] = 0xe19b, [0x2245] = 0xe19c, [0x2246] = 0xe19d,
  [0x2247] = 0xe19e, [0x2248] = 0xe19f, [0x2249] = 0xe1a0, [0x224a] = 0xe1a1,
  [0x224b] = 0xe1a2, [0x224c] = 0xe1a3, [0x224d] = 0xe1a4, [0x224e] = 0xe1a5,
  [0x224f] = 0xe1a6, [0x2250] = 0xe1a7, [0x2251] = 0xe1a8, [0x2252] = 0xe1a9,
  [0x2253] = 0xe1aa, [0x2254] = 0xe1ab, [0x2255] = 0xe1ac, [0x2256] = 0xe1ad,
  [0x2257] = 0xe1ae, [0x2258] = 0xe1af, [0x2259] = 0xe1b0, [0x225a] = 0xe1b1,
  [0x225b] = 0xe1b2, [0x225c] = 0xe1b3, [0x225d] = 0xe1b4, [0x225e] = 0xe1b5,
  [0x225f] = 0xe1b6, [0x2260] = 0xe1b7, [0x2261] = 0xe1b8, [0x2262] = 0xe1b9,
  [0x2263] = 0xe1ba, [0x2264] = 0xe1bb, [0x2265] = 0xe1bc, [0x2266] = 0xe1bd,
  [0x2267] = 0xe1be, [0x2268] = 0xe1bf, [0x2269] = 0xe1c0, [0x226a] = 0xe1c1,
  [0x226b] = 0xe1c2, [0x226c] = 0xe1c3, [0x226d] = 0xe1c4, [0x226e] = 0xe1c5,
  [0x226f] = 0xe1c6, [0x2270] = 0xe1c7, [0x2271] = 0xe1c8, [0x2272] = 0xe1c9,
  [0x2273] = 0xe1ca, [0x2274] = 0xe1cb, [0x2275] = 0xe1cc, [0x2276] = 0xe1cd,
  [0x2277] = 0xe1ce, [0x2278] = 0xe1cf, [0x2279] = 0xe1d0, [0x227a] = 0xe1d1,
  [0x227b] = 0xe1d2, [0x227c] = 0xe1d3, [0x227d] = 0xe1d4, [0x227e] = 0xe1d5,
  [0x2280] = 0x7588, [0x2281] = 0x7589, [0x2282] = 0x758a, [0x2283] = 0x758c,
  [0x2284] = 0x758d, [0x2285] = 0x758e, [0x2286] = 0x7590, [0x2287] = 0x7593,
  [0x2288] = 0x7595, [0x2289] = 0x7598, [0x228a] = 0x759b, [0x228b] = 0x759c,
  [0x228c] = 0x759e, [0x228d] = 0x75a2, [0x228e] = 0x75a6, [0x228f] = 0x75a7,
  [0x2290] = 0x75a8, [0x2291] = 0x75a9, [0x2292] = 0x75aa, [0x2293] = 0x75ad,
  [0x2294] = 0x75b6, [0x2295] = 0x75b7, [0x2296] = 0x75ba, [0x2297] = 0x75bb,
  [0x2298] = 0x75bf, [0x2299] = 0x75c0, [0x229a] = 0x75c1, [0x229b] = 0x75c6,
  [0x229c] = 0x75cb, [0x229d] = 0x75cc, [0x229e] = 0x75ce, [0x229f] = 0x75cf,
  [0x22a0] = 0x75d0, [0x22a1] = 0x75d1, [0x22a2] = 0x75d3, [0x22a3] = 0x75d7,
  [0x22a4] = 0x75d9, [0x22a5] = 0x75da, [0x22a6] = 0x75dc, [0x22a7] = 0x75dd,
  [0x22a8] = 0x75df, [0x22a9] = 0x75e0, [0x22aa] = 0x75e1, [0x22ab] = 0x75e5,
  [0x22ac] = 0x75e9, [0x22ad] = 0x75ec, [0x22ae] = 0x75ed, [0x22af] = 0x75ee,
  [0x22b0] = 0x75ef, [0x22b1] = 0x75f2, [0x22b2] = 0x75f3, [0x22b3] = 0x75f5,
  [0x22b4] = 0x75f6, [0x22b5] = 0x75f7, [0x22b6] = 0x75f8, [0x22b7] = 0x75fa,
  [0x22b8] = 0x75fb, [0x22b9] = 0x75fd, [0x22ba] = 0x75fe, [0x22bb] = 0x7602,
  [0x22bc] = 0x7604, [0x22bd] = 0x7606, [0x22be] = 0x7607, [0x22c0] = 0x7608,
  [0x22c1] = 0x7609, [0x22c2] = 0x760b, [0x22c3] = 0x760d, [0x22c4] = 0x760e,
  [0x22c5] = 0x760f, [0x22c6] = 0x7611, [0x22c7] = 0x7612, [0x22c8] = 0x7613,
  [0x22c9] = 0x7614, [0x22ca] = 0x7616, [0x22cb] = 0x761a, [0x22cc] = 0x761c,
  [0x22cd] = 0x761d, [0x22ce] = 0x761e, [0x22cf] = 0x7621, [0x22d0] = 0x7623,
  [0x22d1] = 0x7627, [0x22d2] = 0x7628, [0x22d3] = 0x762c, [0x22d4] = 0x762e,
  [0x22d5] = 0x762f, [0x22d6] = 0x7631, [0x22d7] = 0x7632, [0x22d8] = 0x7636,
  [0x22d9] = 0x7637, [0x22da] = 0x7639, [0x22db] = 0x763a, [0x22dc] = 0x763b,
  [0x22dd] = 0x763d, [0x22de] = 0x7641, [0x22df] = 0x7642, [0x22e0] = 0x7644,
  [0x22e1] = 0xe1d6, [0x22e2] = 0xe1d7, [0x22e3] = 0xe1d8, [0x22e4] = 0xe1d9,
  [0x22e5] = 0xe1da, [0x22e6] = 0xe1db, [0x22e7] = 0xe1dc, [0x22e8] = 0xe1dd,
  [0x22e9] = 0xe1de, [0x22ea] = 0xe1df, [0x22eb] = 0xe1e0, [0x22ec] = 0xe1e1,
  [0x22ed] = 0xe1e2, [0x22ee] = 0xe1e3, [0x22ef] = 0xe1e4, [0x22f0] = 0xe1e5,
  [0x22f1] = 0xe1e6, [0x22f2] = 0xe1e7, [0x22f3] = 0xe1e8, [0x22f4] = 0xe1e9,
  [0x22f5] = 0xe1ea, [0x22f6] = 0xe1eb, [0x22f7] = 0xe1ec, [0x22f8] = 0xe1ed,
  [0x22f9] = 0xe1ee, [0x22fa] = 0xe1ef, [0x22fb] = 0xe1f0, [0x22fc] = 0xe1f1,
  [0x22fd] = 0xe1f2, [0x22fe] = 0xe1f3, [0x22ff] = 0xe1f4, [0x2300] = 0xe1f5,
  [0x2301] = 0xe1f6, [0x2302] = 0xe1f7, [0x2303] = 0xe1f8, [0x2304] = 0xe1f9,
  [0x2305] = 0xe1fa, [0x2306] = 0xe1fb, [0x2307] = 0xe1fc, [0x2308] = 0xe1fd,
  [0x2309] = 0xe1fe, [0x230a] = 0xe1ff, [0x230b] = 0xe200, [0x230c] = 0xe201,
  [0x230d] = 0xe202, [0x230e] = 0xe203, [0x230f] = 0xe204, [0x2310] = 0xe205,
  [0x2311] = 0xe206, [0x2312] = 0xe207, [0x2313] = 0xe208, [0x2314] = 0xe209,
  [0x2315] = 0xe20a, [0x2316] = 0xe20b, [0x2317] = 0xe20c, [0x2318] = 0xe20d,
  [0x2319] = 0xe20e, [0x231a] = 0xe20f, [0x231b] = 0xe210, [0x231c] = 0xe211,
  [0x231d] = 0xe212, [0x231e] = 0xe213, [0x231f] = 0xe214, [0x2320] = 0xe215,
  [0x2321] = 0xe216, [0x2322] = 0xe217, [0x2323] = 0xe218, [0x2324] = 0xe219,
  [0x2325] = 0xe21a, [0x2326] = 0xe21b, [0x2327] = 0xe21c, [0x2328] = 0xe21d,
  [0x2329] = 0xe21e, [0x232a] = 0xe21f, [0x232b] = 0xe220, [0x232c] = 0xe221,
  [0x232d] = 0xe222, [0x232e] = 0xe223, [0x232f] = 0xe224, [0x2330] = 0xe225,
  [0x2331] = 0xe226, [0x2332] = 0xe227, [0x2333] = 0xe228, [0x2334] = 0xe229,
  [0x2335] = 0xe22a, [0x2336] = 0xe22b, [0x2337] = 0xe22c, [0x2338] = 0xe22d,
  [0x2339] = 0xe22e, [0x233a] = 0xe22f, [0x233b] = 0xe230, [0x233c] = 0xe231,
  [0x233d] = 0xe232, [0x233e] = 0xe233, [0x2340] = 0x7645, [0x2341] = 0x7646,
  [0x2342] = 0x7647, [0x2343] = 0x7648, [0x2344] = 0x7649, [0x2345] = 0x764a,
  [0x2346] = 0x764b, [0x2347] = 0x764e, [0x2348] = 0x764f, [0x2349] = 0x7650,
  [0x234a] = 0x7651, [0x234b] = 0x7652, [0x234c] = 0x7653, [0x234d] = 0x7655,
  [0x234e] = 0x7657, [0x234f] = 0x7658, [0x2350] = 0x7659, [0x2351] = 0x765a,
  [0x2352] = 0x765b, [0x2353] = 0x765d, [0x2354] = 0x765f, [0x2355] = 0x7660,
  [0x2356] = 0x7661, [0x2357] = 0x7662, [0x2358] = 0x7664, [0x2359] = 0x7665,
  [0x235a] = 0x7666, [0x235b] = 0x7667, [0x235c] = 0x7668, [0x235d] = 0x7669,
  [0x235e] = 0x766a, [0x235f] = 0x766c, [0x2360] = 0x766d, [0x2361] = 0x766e,
  [0x2362] = 0x7670, [0x2363] = 0x7671, [0x2364] = 0x7672, [0x2365] = 0x7673,
  [0x2366] = 0x7674, [0x2367] = 0x7675, [0x2368] = 0x7676, [0x2369] = 0x7677,
  [0x236a] = 0x7679, [0x236b] = 0x767a, [0x236c] = 0x767c, [0x236d] = 0x767f,
  [0x236e] = 0x7680, [0x236f] = 0x7681, [0x2370] = 0x7683, [0x2371] = 0x7685,
  [0x2372] = 0x7689, [0x2373] = 0x768a, [0x2374] = 0x768c, [0x2375] = 0x768d,
  [0x2376] = 0x768f, [0x2377] = 0x7690, [0x2378] = 0x7692, [0x2379] = 0x7694,
  [0x237a] = 0x7695, [0x237b] = 0x7697, [0x237c] = 0x7698, [0x237d] = 0x769a,
  [0x237e] = 0x769b, [0x2380] = 0x769c, [0x2381] = 0x769d, [0x2382] = 0x769e,
  [0x2383] = 0x769f, [0x2384] = 0x76a0, [0x2385] = 0x76a1, [0x2386] = 0x76a2,
  [0x2387] = 0x76a3, [0x2388] = 0x76a5, [0x2389] = 0x76a6, [0x238a] = 0x76a7,
  [0x238b] = 0x76a8, [0x238c] = 0x76a9, [0x238d] = 0x76aa, [0x238e] = 0x76ab,
  [0x238f] = 0x76ac, [0x2390] = 0x76ad, [0x2391] = 0x76af, [0x2392] = 0x76b0,
  [0x2393] = 0x76b3, [0x2394] = 0x76b5, [0x2395] = 0x76b6, [0x2396] = 0x76b7,
  [0x2397] = 0x76b8, [0x2398] = 0x76b9, [0x2399] = 0x76ba, [0x239a] = 0x76bb,
  [0x239b] = 0x76bc, [0x239c] = 0x76bd, [0x239d] = 0x76be, [0x239e] = 0x76c0,
  [0x239f] = 0x76c1, [0x23a0] = 0x76c3, [0x23a1] = 0x554a, [0x23a2] = 0x963f,
  [0x23a3] = 0x57c3, [0x23a4] = 0x6328, [0x23a5] = 0x54ce, [0x23a6] = 0x5509,
  [0x23a7] = 0x54c0, [0x23a8] = 0x7691, [0x23a9] = 0x764c, [0x23aa] = 0x853c,
  [0x23ab] = 0x77ee, [0x23ac] = 0x827e, [0x23ad] = 0x788d, [0x23ae] = 0x7231,
  [0x23af] = 0x9698, [0x23b0] = 0x978d, [0x23b1] = 0x6c28, [0x23b2] = 0x5b89,
  [0x23b3] = 0x4ffa, [0x23b4] = 0x6309, [0x23b5] = 0x6697, [0x23b6] = 0x5cb8,
  [0x23b7] = 0x80fa, [0x23b8] = 0x6848, [0x23b9] = 0x80ae, [0x23ba] = 0x6602,
  [0x23bb] = 0x76ce, [0x23bc] = 0x51f9, [0x23bd] = 0x6556, [0x23be] = 0x71ac,
  [0x23bf] = 0x7ff1, [0x23c0] = 0x8884, [0x23c1] = 0x50b2, [0x23c2] = 0x5965,
  [0x23c3] = 0x61ca, [0x23c4] = 0x6fb3, [0x23c5] = 0x82ad, [0x23c6] = 0x634c,
  [0x23c7] = 0x6252, [0x23c8] = 0x53ed, [0x23c9] = 0x5427, [0x23ca] = 0x7b06,
  [0x23cb] = 0x516b, [0x23cc] = 0x75a4, [0x23cd] = 0x5df4, [0x23ce] = 0x62d4,
  [0x23cf] = 0x8dcb, [0x23d0] = 0x9776, [0x23d1] = 0x628a, [0x23d2] = 0x8019,
  [0x23d3] = 0x575d, [0x23d4] = 0x9738, [0x23d5] = 0x7f62, [0x23d6] = 0x7238,
  [0x23d7] = 0x767d, [0x23d8] = 0x67cf, [0x23d9] = 0x767e, [0x23da] = 0x6446,
  [0x23db] = 0x4f70, [0x23dc] = 0x8d25, [0x23dd] = 0x62dc, [0x23de] = 0x7a17,
  [0x23df] = 0x6591, [0x23e0] = 0x73ed, [0x23e1] = 0x642c, [0x23e2] = 0x6273,
  [0x23e3] = 0x822c, [0x23e4] = 0x9881, [0x23e5] = 0x677f, [0x23e6] = 0x7248,
  [0x23e7] = 0x626e, [0x23e8] = 0x62cc, [0x23e9] = 0x4f34, [0x23ea] = 0x74e3,
  [0x23eb] = 0x534a, [0x23ec] = 0x529e, [0x23ed] = 0x7eca, [0x23ee] = 0x90a6,
  [0x23ef] = 0x5e2e, [0x23f0] = 0x6886, [0x23f1] = 0x699c, [0x23f2] = 0x8180,
  [0x23f3] = 0x7ed1, [0x23f4] = 0x68d2, [0x23f5] = 0x78c5, [0x23f6] = 0x868c,
  [0x23f7] = 0x9551, [0x23f8] = 0x508d, [0x23f9] = 0x8c24, [0x23fa] = 0x82de,
  [0x23fb] = 0x80de, [0x23fc] = 0x5305, [0x23fd] = 0x8912, [0x23fe] = 0x5265,
  [0x2400] = 0x76c4, [0x2401] = 0x76c7, [0x2402] = 0x76c9, [0x2403] = 0x76cb,
  [0x2404] = 0x76cc, [0x2405] = 0x76d3, [0x2406] = 0x76d5, [0x2407] = 0x76d9,
  [0x2408] = 0x76da, [0x2409] = 0x76dc, [0x240a] = 0x76dd, [0x240b] = 0x76de,
  [0x240c] = 0x76e0, [0x240d] = 0x76e1, [0x240e] = 0x76e2, [0x240f] = 0x76e3,
  [0x2410] = 0x76e4, [0x2411] = 0x76e6, [0x2412] = 0x76e7, [0x2413] = 0x76e8,
  [0x2414] = 0x76e9, [0x2415] = 0x76ea, [0x2416] = 0x76eb, [0x2417] = 0x76ec,
  [0x2418] = 0x76ed, [0x2419] = 0x76f0, [0x241a] = 0x76f3, [0x241b] = 0x76f5,
  [0x241c] = 0x76f6, [0x241d] = 0x76f7, [0x241e] = 0x76fa, [0x241f] = 0x76fb,
  [0x2420] = 0x76fd, [0x2421] = 0x76ff, [0x2422] = 0x7700, [0x2423] = 0x7702,
  [0x2424] = 0x7703, [0x2425] = 0x7705, [0x2426] = 0x7706, [0x2427] = 0x770a,
  [0x2428] = 0x770c, [0x2429] = 0x770e, [0x242a] = 0x770f, [0x242b] = 0x7710,
  [0x242c] = 0x7711, [0x242d] = 0x7712, [0x242e] = 0x7713, [0x242f] = 0x7714,
  [0x2430] = 0x7715, [0x2431] = 0x7716, [0x2432] = 0x7717, [0x2433] = 0x7718,
  [0x2434] = 0x771b, [0x2435] = 0x771c, [0x2436] = 0x771d, [0x2437] = 0x771e,
  [0x2438] = 0x7721, [0x2439] = 0x7723, [0x243a] = 0x7724, [0x243b] = 0x7725,
  [0x243c] = 0x7727, [0x243d] = 0x772a, [0x243e] = 0x772b, [0x2440] = 0x772c,
  [0x2441] = 0x772e, [0x2442] = 0x7730, [0x2443] = 0x7731, [0x2444] = 0x7732,
  [0x2445] = 0x7733, [0x2446] = 0x7734, [0x2447] = 0x7739, [0x2448] = 0x773b,
  [0x2449] = 0x773d, [0x244a] = 0x773e, [0x244b] = 0x773f, [0x244c] = 0x7742,
  [0x244d] = 0x7744, [0x244e] = 0x7745, [0x244f] = 0x7746, [0x2450] = 0x7748,
  [0x2451] = 0x7749, [0x2452] = 0x774a, [0x2453] = 0x774b, [0x2454] = 0x774c,
  [0x2455] = 0x774d, [0x2456] = 0x774e, [0x2457] = 0x774f, [0x2458] = 0x7752,
  [0x2459] = 0x7753, [0x245a] = 0x7754, [0x245b] = 0x7755, [0x245c] = 0x7756,
  [0x245d] = 0x7757, [0x245e] = 0x7758, [0x245f] = 0x7759, [0x2460] = 0x775c,
  [0x2461] = 0x8584, [0x2462] = 0x96f9, [0x2463] = 0x4fdd, [0x2464] = 0x5821,
  [0x2465] = 0x9971, [0x2466] = 0x5b9d, [0x2467] = 0x62b1, [0x2468] = 0x62a5,
  [0x2469] = 0x66b4, [0x246a] = 0x8c79, [0x246b] = 0x9c8d, [0x246c] = 0x7206,
  [0x246d] = 0x676f, [0x246e] = 0x7891, [0x246f] = 0x60b2, [0x2470] = 0x5351,
  [0x2471] = 0x5317, [0x2472] = 0x8f88, [0x2473] = 0x80cc, [0x2474] = 0x8d1d,
  [0x2475] = 0x94a1, [0x2476] = 0x500d, [0x2477] = 0x72c8, [0x2478] = 0x5907,
  [0x2479] = 0x60eb, [0x247a] = 0x7119, [0x247b] = 0x88ab, [0x247c] = 0x5954,
  [0x247d] = 0x82ef, [0x247e] = 0x672c, [0x247f] = 0x7b28, [0x2480] = 0x5d29,
  [0x2481] = 0x7ef7, [0x2482] = 0x752d, [0x2483] = 0x6cf5, [0x2484] = 0x8e66,
  [0x2485] = 0x8ff8, [0x2486] = 0x903c, [0x2487] = 0x9f3b, [0x2488] = 0x6bd4,
  [0x2489] = 0x9119, [0x248a] = 0x7b14, [0x248b] = 0x5f7c, [0x248c] = 0x78a7,
  [0x248d] = 0x84d6, [0x248e] = 0x853d, [0x248f] = 0x6bd5, [0x2490] = 0x6bd9,
  [0x2491] = 0x6bd6, [0x2492] = 0x5e01, [0x2493] = 0x5e87, [0x2494] = 0x75f9,
  [0x2495] = 0x95ed, [0x2496] = 0x655d, [0x2497] = 0x5f0a, [0x2498] = 0x5fc5,
  [0x2499] = 0x8f9f, [0x249a] = 0x58c1, [0x249b] = 0x81c2, [0x249c] = 0x907f,
  [0x249d] = 0x965b, [0x249e] = 0x97ad, [0x249f] = 0x8fb9, [0x24a0] = 0x7f16,
  [0x24a1] = 0x8d2c, [0x24a2] = 0x6241, [0x24a3] = 0x4fbf, [0x24a4] = 0x53d8,
  [0x24a5] = 0x535e, [0x24a6] = 0x8fa8, [0x24a7] = 0x8fa9, [0x24a8] = 0x8fab,
  [0x24a9] = 0x904d, [0x24aa] = 0x6807, [0x24ab] = 0x5f6a, [0x24ac] = 0x8198,
  [0x24ad] = 0x8868, [0x24ae] = 0x9cd6, [0x24af] = 0x618b, [0x24b0] = 0x522b,
  [0x24b1] = 0x762a, [0x24b2] = 0x5f6c, [0x24b3] = 0x658c, [0x24b4] = 0x6fd2,
  [0x24b5] = 0x6ee8, [0x24b6] = 0x5bbe, [0x24b7] = 0x6448, [0x24b8] = 0x5175,
  [0x24b9] = 0x51b0, [0x24ba] = 0x67c4, [0x24bb] = 0x4e19, [0x24bc] = 0x79c9,
  [0x24bd] = 0x997c, [0x24be] = 0x70b3, [0x24c0] = 0x775d, [0x24c1] = 0x775e,
  [0x24c2] = 0x775f, [0x24c3] = 0x7760, [0x24c4] = 0x7764, [0x24c5] = 0x7767,
  [0x24c6] = 0x7769, [0x24c7] = 0x776a, [0x24c8] = 0x776d, [0x24c9] = 0x776e,
  [0x24ca] = 0x776f, [0x24cb] = 0x7770, [0x24cc] = 0x7771, [0x24cd] = 0x7772,
  [0x24ce] = 0x7773, [0x24cf] = 0x7774, [0x24d0] = 0x7775, [0x24d1] = 0x7776,
  [0x24d2] = 0x7777, [0x24d3] = 0x7778, [0x24d4] = 0x777a, [0x24d5] = 0x777b,
  [0x24d6] = 0x777c, [0x24d7] = 0x7781, [0x24d8] = 0x7782, [0x24d9] = 0x7783,
  [0x24da] = 0x7786, [0x24db] = 0x7787, [0x24dc] = 0x7788, [0x24dd] = 0x7789,
  [0x24de] = 0x778a, [0x24df] = 0x778b, [0x24e0] = 0x778f, [0x24e1] = 0x7790,
  [0x24e2] = 0x7793, [0x24e3] = 0x7794, [0x24e4] = 0x7795, [0x24e5] = 0x7796,
  [0x24e6] = 0x7797, [0x24e7] = 0x7798, [0x24e8] = 0x7799, [0x24e9] = 0x779a,
  [0x24ea] = 0x779b, [0x24eb] = 0x779c, [0x24ec] = 0x779d, [0x24ed] = 0x779e,
  [0x24ee] = 0x77a1, [0x24ef] = 0x77a3, [0x24f0] = 0x77a4, [0x24f1] = 0x77a6,
  [0x24f2] = 0x77a8, [0x24f3] = 0x77ab, [0x24f4] = 0x77ad, [0x24f5] = 0x77ae,
  [0x24f6] = 0x77af, [0x24f7] = 0x77b1, [0x24f8] = 0x77b2, [0x24f9] = 0x77b4,
  [0x24fa] = 0x77b6, [0x24fb] = 0x77b7, [0x24fc] = 0x77b8, [0x24fd] = 0x77b9,
  [0x24fe] = 0x77ba, [0x2500] = 0x77bc, [0x2501] = 0x77be, [0x2502] = 0x77c0,
  [0x2503] = 0x77c1, [0x2504] = 0x77c2, [0x2505] = 0x77c3, [0x2506] = 0x77c4,
  [0x2507] = 0x77c5, [0x2508] = 0x77c6, [0x2509] = 0x77c7, [0x250a] = 0x77c8,
  [0x250b] = 0x77c9, [0x250c] = 0x77ca, [0x250d] = 0x77cb, [0x250e] = 0x77cc,
  [0x250f] = 0x77ce, [0x2510] = 0x77cf, [0x2511] = 0x77d0, [0x2512] = 0x77d1,
  [0x2513] = 0x77d2, [0x2514] = 0x77d3, [0x2515] = 0x77d4, [0x2516] = 0x77d5,
  [0x2517] = 0x77d6, [0x2518] = 0x77d8, [0x2519] = 0x77d9, [0x251a] = 0x77da,
  [0x251b] = 0x77dd, [0x251c] = 0x77de, [0x251d] = 0x77df, [0x251e] = 0x77e0,
  [0x251f] = 0x77e1, [0x2520] = 0x77e4, [0x2521] = 0x75c5, [0x2522] = 0x5e76,
  [0x2523] = 0x73bb, [0x2524] = 0x83e0, [0x2525] = 0x64ad, [0x2526] = 0x62e8,
  [0x2527] = 0x94b5, [0x2528] = 0x6ce2, [0x2529] = 0x535a, [0x252a] = 0x52c3,
  [0x252b] = 0x640f, [0x252c] = 0x94c2, [0x252d] = 0x7b94, [0x252e] = 0x4f2f,
  [0x252f] = 0x5e1b, [0x2530] = 0x8236, [0x2531] = 0x8116, [0x2532] = 0x818a,
  [0x2533] = 0x6e24, [0x2534] = 0x6cca, [0x2535] = 0x9a73, [0x2536] = 0x6355,
  [0x2537] = 0x535c, [0x2538] = 0x54fa, [0x2539] = 0x8865, [0x253a] = 0x57e0,
  [0x253b] = 0x4e0d, [0x253c] = 0x5e03, [0x253d] = 0x6b65, [0x253e] = 0x7c3f,
  [0x253f] = 0x90e8, [0x2540] = 0x6016, [0x2541] = 0x64e6, [0x2542] = 0x731c,
  [0x2543] = 0x88c1, [0x2544] = 0x6750, [0x2545] = 0x624d, [0x2546] = 0x8d22,
  [0x2547] = 0x776c, [0x2548] = 0x8e29, [0x2549] = 0x91c7, [0x254a] = 0x5f69,
  [0x254b] = 0x83dc, [0x254c] = 0x8521, [0x254d] = 0x9910, [0x254e] = 0x53c2,
  [0x254f] = 0x8695, [0x2550] = 0x6b8b, [0x2551] = 0x60ed, [0x2552] = 0x60e8,
  [0x2553] = 0x707f, [0x2554] = 0x82cd, [0x2555] = 0x8231, [0x2556] = 0x4ed3,
  [0x2557] = 0x6ca7, [0x2558] = 0x85cf, [0x2559] = 0x64cd, [0x255a] = 0x7cd9,
  [0x255b] = 0x69fd, [0x255c] = 0x66f9, [0x255d] = 0x8349, [0x255e] = 0x5395,
  [0x255f] = 0x7b56, [0x2560] = 0x4fa7, [0x2561] = 0x518c, [0x2562] = 0x6d4b,
  [0x2563] = 0x5c42, [0x2564] = 0x8e6d, [0x2565] = 0x63d2, [0x2566] = 0x53c9,
  [0x2567] = 0x832c, [0x2568] = 0x8336, [0x2569] = 0x67e5, [0x256a] = 0x78b4,
  [0x256b] = 0x643d, [0x256c] = 0x5bdf, [0x256d] = 0x5c94, [0x256e] = 0x5dee,
  [0x256f] = 0x8be7, [0x2570] = 0x62c6, [0x2571] = 0x67f4, [0x2572] = 0x8c7a,
  [0x2573] = 0x6400, [0x2574] = 0x63ba, [0x2575] = 0x8749, [0x2576] = 0x998b,
  [0x2577] = 0x8c17, [0x2578] = 0x7f20, [0x2579] = 0x94f2, [0x257a] = 0x4ea7,
  [0x257b] = 0x9610, [0x257c] = 0x98a4, [0x257d] = 0x660c, [0x257e] = 0x7316,
  [0x2580] = 0x77e6, [0x2581] = 0x77e8, [0x2582] = 0x77ea, [0x2583] = 0x77ef,
  [0x2584] = 0x77f0, [0x2585] = 0x77f1, [0x2586] = 0x77f2, [0x2587] = 0x77f4,
  [0x2588] = 0x77f5, [0x2589] = 0x77f7, [0x258a] = 0x77f9, [0x258b] = 0x77fa,
  [0x258c] = 0x77fb, [0x258d] = 0x77fc, [0x258e] = 0x7803, [0x258f] = 0x7804,
  [0x2590] = 0x7805, [0x2591] = 0x7806, [0x2592] = 0x7807, [0x2593] = 0x7808,
  [0x2594] = 0x780a, [0x2595] = 0x780b, [0x2596] = 0x780e, [0x2597] = 0x780f,
  [0x2598] = 0x7810, [0x2599] = 0x7813, [0x259a] = 0x7815, [0x259b] = 0x7819,
  [0x259c] = 0x781b, [0x259d] = 0x781e, [0x259e] = 0x7820, [0x259f] = 0x7821,
  [0x25a0] = 0x7822, [0x25a1] = 0x7824, [0x25a2] = 0x7828, [0x25a3] = 0x782a,
  [0x25a4] = 0x782b, [0x25a5] = 0x782e, [0x25a6] = 0x782f, [0x25a7] = 0x7831,
  [0x25a8] = 0x7832, [0x25a9] = 0x7833, [0x25aa] = 0x7835, [0x25ab] = 0x7836,
  [0x25ac] = 0x783d, [0x25ad] = 0x783f, [0x25ae] = 0x7841, [0x25af] = 0x7842,
  [0x25b0] = 0x7843, [0x25b1] = 0x7844, [0x25b2] = 0x7846, [0x25b3] = 0x7848,
  [0x25b4] = 0x7849, [0x25b5] = 0x784a, [0x25b6] = 0x784b, [0x25b7] = 0x784d,
  [0x25b8] = 0x784f, [0x25b9] = 0x7851, [0x25ba] = 0x7853, [0x25bb] = 0x7854,
  [0x25bc] = 0x7858, [0x25bd] = 0x7859, [0x25be] = 0x785a, [0x25c0] = 0x785b,
  [0x25c1] = 0x785c, [0x25c2] = 0x785e, [0x25c3] = 0x785f, [0x25c4] = 0x7860,
  [0x25c5] = 0x7861, [0x25c6] = 0x7862, [0x25c7] = 0x7863, [0x25c8] = 0x7864,
  [0x25c9] = 0x7865, [0x25ca] = 0x7866, [0x25cb] = 0x7867, [0x25cc] = 0x7868,
  [0x25cd] = 0x7869, [0x25ce] = 0x786f, [0x25cf] = 0x7870, [0x25d0] = 0x7871,
  [0x25d1] = 0x7872, [0x25d2] = 0x7873, [0x25d3] = 0x7874, [0x25d4] = 0x7875,
  [0x25d5] = 0x7876, [0x25d6] = 0x7878, [0x25d7] = 0x7879, [0x25d8] = 0x787a,
  [0x25d9] = 0x787b, [0x25da] = 0x787d, [0x25db] = 0x787e, [0x25dc] = 0x787f,
  [0x25dd] = 0x7880, [0x25de] = 0x7881, [0x25df] = 0x7882, [0x25e0] = 0x7883,
  [0x25e1] = 0x573a, [0x25e2] = 0x5c1d, [0x25e3] = 0x5e38, [0x25e4] = 0x957f,
  [0x25e5] = 0x507f, [0x25e6] = 0x80a0, [0x25e7] = 0x5382, [0x25e8] = 0x655e,
  [0x25e9] = 0x7545, [0x25ea] = 0x5531, [0x25eb] = 0x5021, [0x25ec] = 0x8d85,
  [0x25ed] = 0x6284, [0x25ee] = 0x949e, [0x25ef] = 0x671d, [0x25f0] = 0x5632,
  [0x25f1] = 0x6f6e, [0x25f2] = 0x5de2, [0x25f3] = 0x5435, [0x25f4] = 0x7092,
  [0x25f5] = 0x8f66, [0x25f6] = 0x626f, [0x25f7] = 0x64a4, [0x25f8] = 0x63a3,
  [0x25f9] = 0x5f7b, [0x25fa] = 0x6f88, [0x25fb] = 0x90f4, [0x25fc] = 0x81e3,
  [0x25fd] = 0x8fb0, [0x25fe] = 0x5c18, [0x25ff] = 0x6668, [0x2600] = 0x5ff1,
  [0x2601] = 0x6c89, [0x2602] = 0x9648, [0x2603] = 0x8d81, [0x2604] = 0x886c,
  [0x2605] = 0x6491, [0x2606] = 0x79f0, [0x2607] = 0x57ce, [0x2608] = 0x6a59,
  [0x2609] = 0x6210, [0x260a] = 0x5448, [0x260b] = 0x4e58, [0x260c] = 0x7a0b,
  [0x260d] = 0x60e9, [0x260e] = 0x6f84, [0x260f] = 0x8bda, [0x2610] = 0x627f,
  [0x2611] = 0x901e, [0x2612] = 0x9a8b, [0x2613] = 0x79e4, [0x2614] = 0x5403,
  [0x2615] = 0x75f4, [0x2616] = 0x6301, [0x2617] = 0x5319, [0x2618] = 0x6c60,
  [0x2619] = 0x8fdf, [0x261a] = 0x5f1b, [0x261b] = 0x9a70, [0x261c] = 0x803b,
  [0x261d] = 0x9f7f, [0x261e] = 0x4f88, [0x261f] = 0x5c3a, [0x2620] = 0x8d64,
  [0x2621] = 0x7fc5, [0x2622] = 0x65a5, [0x2623] = 0x70bd, [0x2624] = 0x5145,
  [0x2625] = 0x51b2, [0x2626] = 0x866b, [0x2627] = 0x5d07, [0x2628] = 0x5ba0,
  [0x2629] = 0x62bd, [0x262a] = 0x916c, [0x262b] = 0x7574, [0x262c] = 0x8e0c,
  [0x262d] = 0x7a20, [0x262e] = 0x6101, [0x262f] = 0x7b79, [0x2630] = 0x4ec7,
  [0x2631] = 0x7ef8, [0x2632] = 0x7785, [0x2633] = 0x4e11, [0x2634] = 0x81ed,
  [0x2635] = 0x521d, [0x2636] = 0x51fa, [0x2637] = 0x6a71, [0x2638] = 0x53a8,
  [0x2639] = 0x8e87, [0x263a] = 0x9504, [0x263b] = 0x96cf, [0x263c] = 0x6ec1,
  [0x263d] = 0x9664, [0x263e] = 0x695a, [0x2640] = 0x7884, [0x2641] = 0x7885,
  [0x2642] = 0x7886, [0x2643] = 0x7888, [0x2644] = 0x788a, [0x2645] = 0x788b,
  [0x2646] = 0x788f, [0x2647] = 0x7890, [0x2648] = 0x7892, [0x2649] = 0x7894,
  [0x264a] = 0x7895, [0x264b] = 0x7896, [0x264c] = 0x7899, [0x264d] = 0x789d,
  [0x264e] = 0x789e, [0x264f] = 0x78a0, [0x2650] = 0x78a2, [0x2651] = 0x78a4,
  [0x2652] = 0x78a6, [0x2653] = 0x78a8, [0x2654] = 0x78a9, [0x2655] = 0x78aa,
  [0x2656] = 0x78ab, [0x2657] = 0x78ac, [0x2658] = 0x78ad, [0x2659] = 0x78ae,
  [0x265a] = 0x78af, [0x265b] = 0x78b5, [0x265c] = 0x78b6, [0x265d] = 0x78b7,
  [0x265e] = 0x78b8, [0x265f] = 0x78ba, [0x2660] = 0x78bb, [0x2661] = 0x78bc,
  [0x2662] = 0x78bd, [0x2663] = 0x78bf, [0x2664] = 0x78c0, [0x2665] = 0x78c2,
  [0x2666] = 0x78c3, [0x2667] = 0x78c4, [0x2668] = 0x78c6, [0x2669] = 0x78c7,
  [0x266a] = 0x78c8, [0x266b] = 0x78cc, [0x266c] = 0x78cd, [0x266d] = 0x78ce,
  [0x266e] = 0x78cf, [0x266f] = 0x78d1, [0x2670] = 0x78d2, [0x2671] = 0x78d3,
  [0x2672] = 0x78d6, [0x2673] = 0x78d7, [0x2674] = 0x78d8, [0x2675] = 0x78da,
  [0x2676] = 0x78db, [0x2677] = 0x78dc, [0x2678] = 0x78dd, [0x2679] = 0x78de,
  [0x267a] = 0x78df, [0x267b] = 0x78e0, [0x267c] = 0x78e1, [0x267d] = 0x78e2,
  [0x267e] = 0x78e3, [0x2680] = 0x78e4, [0x2681] = 0x78e5, [0x2682] = 0x78e6,
  [0x2683] = 0x78e7, [0x2684] = 0x78e9, [0x2685] = 0x78ea, [0x2686] = 0x78eb,
  [0x2687] = 0x78ed, [0x2688] = 0x78ee, [0x2689] = 0x78ef, [0x268a] = 0x78f0,
  [0x268b] = 0x78f1, [0x268c] = 0x78f3, [0x268d] = 0x78f5, [0x268e] = 0x78f6,
  [0x268f] = 0x78f8, [0x2690] = 0x78f9, [0x2691] = 0x78fb, [0x2692] = 0x78fc,
  [0x2693] = 0x78fd, [0x2694] = 0x78fe, [0x2695] = 0x78ff, [0x2696] = 0x7900,
  [0x2697] = 0x7902, [0x2698] = 0x7903, [0x2699] = 0x7904, [0x269a] = 0x7906,
  [0x269b] = 0x7907, [0x269c] = 0x7908, [0x269d] = 0x7909, [0x269e] = 0x790a,
  [0x269f] = 0x790b, [0x26a0] = 0x790c, [0x26a1] = 0x7840, [0x26a2] = 0x50a8,
  [0x26a3] = 0x77d7, [0x26a4] = 0x6410, [0x26a5] = 0x89e6, [0x26a6] = 0x5904,
  [0x26a7] = 0x63e3, [0x26a8] = 0x5ddd, [0x26a9] = 0x7a7f, [0x26aa] = 0x693d,
  [0x26ab] = 0x4f20, [0x26ac] = 0x8239, [0x26ad] = 0x5598, [0x26ae] = 0x4e32,
  [0x26af] = 0x75ae, [0x26b0] = 0x7a97, [0x26b1] = 0x5e62, [0x26b2] = 0x5e8a,
  [0x26b3] = 0x95ef, [0x26b4] = 0x521b, [0x26b5] = 0x5439, [0x26b6] = 0x708a,
  [0x26b7] = 0x6376, [0x26b8] = 0x9524, [0x26b9] = 0x5782, [0x26ba] = 0x6625,
  [0x26bb] = 0x693f, [0x26bc] = 0x9187, [0x26bd] = 0x5507, [0x26be] = 0x6df3,
  [0x26bf] = 0x7eaf, [0x26c0] = 0x8822, [0x26c1] = 0x6233, [0x26c2] = 0x7ef0,
  [0x26c3] = 0x75b5, [0x26c4] = 0x8328, [0x26c5] = 0x78c1, [0x26c6] = 0x96cc,
  [0x26c7] = 0x8f9e, [0x26c8] = 0x6148, [0x26c9] = 0x74f7, [0x26ca] = 0x8bcd,
  [0x26cb] = 0x6b64, [0x26cc] = 0x523a, [0x26cd] = 0x8d50, [0x26ce] = 0x6b21,
  [0x26cf] = 0x806a, [0x26d0] = 0x8471, [0x26d1] = 0x56f1, [0x26d2] = 0x5306,
  [0x26d3] = 0x4ece, [0x26d4] = 0x4e1b, [0x26d5] = 0x51d1, [0x26d6] = 0x7c97,
  [0x26d7] = 0x918b, [0x26d8] = 0x7c07, [0x26d9] = 0x4fc3, [0x26da] = 0x8e7f,
  [0x26db] = 0x7be1, [0x26dc] = 0x7a9c, [0x26dd] = 0x6467, [0x26de] = 0x5d14,
  [0x26df] = 0x50ac, [0x26e0] = 0x8106, [0x26e1] = 0x7601, [0x26e2] = 0x7cb9,
  [0x26e3] = 0x6dec, [0x26e4] = 0x7fe0, [0x26e5] = 0x6751, [0x26e6] = 0x5b58,
  [0x26e7] = 0x5bf8, [0x26e8] = 0x78cb, [0x26e9] = 0x64ae, [0x26ea] = 0x6413,
  [0x26eb] = 0x63aa, [0x26ec] = 0x632b, [0x26ed] = 0x9519, [0x26ee] = 0x642d,
  [0x26ef] = 0x8fbe, [0x26f0] = 0x7b54, [0x26f1] = 0x7629, [0x26f2] = 0x6253,
  [0x26f3] = 0x5927, [0x26f4] = 0x5446, [0x26f5] = 0x6b79, [0x26f6] = 0x50a3,
  [0x26f7] = 0x6234, [0x26f8] = 0x5e26, [0x26f9] = 0x6b86, [0x26fa] = 0x4ee3,
  [0x26fb] = 0x8d37, [0x26fc] = 0x888b, [0x26fd] = 0x5f85, [0x26fe] = 0x902e,
  [0x2700] = 0x790d, [0x2701] = 0x790e, [0x2702] = 0x790f, [0x2703] = 0x7910,
  [0x2704] = 0x7911, [0x2705] = 0x7912, [0x2706] = 0x7914, [0x2707] = 0x7915,
  [0x2708] = 0x7916, [0x2709] = 0x7917, [0x270a] = 0x7918, [0x270b] = 0x7919,
  [0x270c] = 0x791a, [0x270d] = 0x791b, [0x270e] = 0x791c, [0x270f] = 0x791d,
  [0x2710] = 0x791f, [0x2711] = 0x7920, [0x2712] = 0x7921, [0x2713] = 0x7922,
  [0x2714] = 0x7923, [0x2715] = 0x7925, [0x2716] = 0x7926, [0x2717] = 0x7927,
  [0x2718] = 0x7928, [0x2719] = 0x7929, [0x271a] = 0x792a, [0x271b] = 0x792b,
  [0x271c] = 0x792c, [0x271d] = 0x792d, [0x271e] = 0x792e, [0x271f] = 0x792f,
  [0x2720] = 0x7930, [0x2721] = 0x7931, [0x2722] = 0x7932, [0x2723] = 0x7933,
  [0x2724] = 0x7935, [0x2725] = 0x7936, [0x2726] = 0x7937, [0x2727] = 0x7938,
  [0x2728] = 0x7939, [0x2729] = 0x793d, [0x272a] = 0x793f, [0x272b] = 0x7942,
  [0x272c] = 0x7943, [0x272d] = 0x7944, [0x272e] = 0x7945, [0x272f] = 0x7947,
  [0x2730] = 0x794a, [0x2731] = 0x794b, [0x2732] = 0x794c, [0x2733] = 0x794d,
  [0x2734] = 0x794e, [0x2735] = 0x794f, [0x2736] = 0x7950, [0x2737] = 0x7951,
  [0x2738] = 0x7952, [0x2739] = 0x7954, [0x273a] = 0x7955, [0x273b] = 0x7958,
  [0x273c] = 0x7959, [0x273d] = 0x7961, [0x273e] = 0x7963, [0x2740] = 0x7964,
  [0x2741] = 0x7966, [0x2742] = 0x7969, [0x2743] = 0x796a, [0x2744] = 0x796b,
  [0x2745] = 0x796c, [0x2746] = 0x796e, [0x2747] = 0x7970, [0x2748] = 0x7971,
  [0x2749] = 0x7972, [0x274a] = 0x7973, [0x274b] = 0x7974, [0x274c] = 0x7975,
  [0x274d] = 0x7976, [0x274e] = 0x7979, [0x274f] = 0x797b, [0x2750] = 0x797c,
  [0x2751] = 0x797d, [0x2752] = 0x797e, [0x2753] = 0x797f, [0x2754] = 0x7982,
  [0x2755] = 0x7983, [0x2756] = 0x7986, [0x2757] = 0x7987, [0x2758] = 0x7988,
  [0x2759] = 0x7989, [0x275a] = 0x798b, [0x275b] = 0x798c, [0x275c] = 0x798d,
  [0x275d] = 0x798e, [0x275e] = 0x7990, [0x275f] = 0x7991, [0x2760] = 0x7992,
  [0x2761] = 0x6020, [0x2762] = 0x803d, [0x2763] = 0x62c5, [0x2764] = 0x4e39,
  [0x2765] = 0x5355, [0x2766] = 0x90f8, [0x2767] = 0x63b8, [0x2768] = 0x80c6,
  [0x2769] = 0x65e6, [0x276a] = 0x6c2e, [0x276b] = 0x4f46, [0x276c] = 0x60ee,
  [0x276d] = 0x6de1, [0x276e] = 0x8bde, [0x276f] = 0x5f39, [0x2770] = 0x86cb,
  [0x2771] = 0x5f53, [0x2772] = 0x6321, [0x2773] = 0x515a, [0x2774] = 0x8361,
  [0x2775] = 0x6863, [0x2776] = 0x5200, [0x2777] = 0x6363, [0x2778] = 0x8e48,
  [0x2779] = 0x5012, [0x277a] = 0x5c9b, [0x277b] = 0x7977, [0x277c] = 0x5bfc,
  [0x277d] = 0x5230, [0x277e] = 0x7a3b, [0x277f] = 0x60bc, [0x2780] = 0x9053,
  [0x2781] = 0x76d7, [0x2782] = 0x5fb7, [0x2783] = 0x5f97, [0x2784] = 0x7684,
  [0x2785] = 0x8e6c, [0x2786] = 0x706f, [0x2787] = 0x767b, [0x2788] = 0x7b49,
  [0x2789] = 0x77aa, [0x278a] = 0x51f3, [0x278b] = 0x9093, [0x278c] = 0x5824,
  [0x278d] = 0x4f4e, [0x278e] = 0x6ef4, [0x278f] = 0x8fea, [0x2790] = 0x654c,
  [0x2791] = 0x7b1b, [0x2792] = 0x72c4, [0x2793] = 0x6da4, [0x2794] = 0x7fdf,
  [0x2795] = 0x5ae1, [0x2796] = 0x62b5, [0x2797] = 0x5e95, [0x2798] = 0x5730,
  [0x2799] = 0x8482, [0x279a] = 0x7b2c, [0x279b] = 0x5e1d, [0x279c] = 0x5f1f,
  [0x279d] = 0x9012, [0x279e] = 0x7f14, [0x279f] = 0x98a0, [0x27a0] = 0x6382,
  [0x27a1] = 0x6ec7, [0x27a2] = 0x7898, [0x27a3] = 0x70b9, [0x27a4] = 0x5178,
  [0x27a5] = 0x975b, [0x27a6] = 0x57ab, [0x27a7] = 0x7535, [0x27a8] = 0x4f43,
  [0x27a9] = 0x7538, [0x27aa] = 0x5e97, [0x27ab] = 0x60e6, [0x27ac] = 0x5960,
  [0x27ad] = 0x6dc0, [0x27ae] = 0x6bbf, [0x27af] = 0x7889, [0x27b0] = 0x53fc,
  [0x27b1] = 0x96d5, [0x27b2] = 0x51cb, [0x27b3] = 0x5201, [0x27b4] = 0x6389,
  [0x27b5] = 0x540a, [0x27b6] = 0x9493, [0x27b7] = 0x8c03, [0x27b8] = 0x8dcc,
  [0x27b9] = 0x7239, [0x27ba] = 0x789f, [0x27bb] = 0x8776, [0x27bc] = 0x8fed,
  [0x27bd] = 0x8c0d, [0x27be] = 0x53e0, [0x27c0] = 0x7993, [0x27c1] = 0x7994,
  [0x27c2] = 0x7995, [0x27c3] = 0x7996, [0x27c4] = 0x7997, [0x27c5] = 0x7998,
  [0x27c6] = 0x7999, [0x27c7] = 0x799b, [0x27c8] = 0x799c, [0x27c9] = 0x799d,
  [0x27ca] = 0x799e, [0x27cb] = 0x799f, [0x27cc] = 0x79a0, [0x27cd] = 0x79a1,
  [0x27ce] = 0x79a2, [0x27cf] = 0x79a3, [0x27d0] = 0x79a4, [0x27d1] = 0x79a5,
  [0x27d2] = 0x79a6, [0x27d3] = 0x79a8, [0x27d4] = 0x79a9, [0x27d5] = 0x79aa,
  [0x27d6] = 0x79ab, [0x27d7] = 0x79ac, [0x27d8] = 0x79ad, [0x27d9] = 0x79ae,
  [0x27da] = 0x79af, [0x27db] = 0x79b0, [0x27dc] = 0x79b1, [0x27dd] = 0x79b2,
  [0x27de] = 0x79b4, [0x27df] = 0x79b5, [0x27e0] = 0x79b6, [0x27e1] = 0x79b7,
  [0x27e2] = 0x79b8, [0x27e3] = 0x79bc, [0x27e4] = 0x79bf, [0x27e5] = 0x79c2,
  [0x27e6] = 0x79c4, [0x27e7] = 0x79c5, [0x27e8] = 0x79c7, [0x27e9] = 0x79c8,
  [0x27ea] = 0x79ca, [0x27eb] = 0x79cc, [0x27ec] = 0x79ce, [0x27ed] = 0x79cf,
  [0x27ee] = 0x79d0, [0x27ef] = 0x79d3, [0x27f0] = 0x79d4, [0x27f1] = 0x79d6,
  [0x27f2] = 0x79d7, [0x27f3] = 0x79d9, [0x27f4] = 0x79da, [0x27f5] = 0x79db,
  [0x27f6] = 0x79dc, [0x27f7] = 0x79dd, [0x27f8] = 0x79de, [0x27f9] = 0x79e0,
  [0x27fa] = 0x79e1, [0x27fb] = 0x79e2, [0x27fc] = 0x79e5, [0x27fd] = 0x79e8,
  [0x27fe] = 0x79ea, [0x2800] = 0x79ec, [0x2801] = 0x79ee, [0x2802] = 0x79f1,
  [0x2803] = 0x79f2, [0x2804] = 0x79f3, [0x2805] = 0x79f4, [0x2806] = 0x79f5,
  [0x2807] = 0x79f6, [0x2808] = 0x79f7, [0x2809] = 0x79f9, [0x280a] = 0x79fa,
  [0x280b] = 0x79fc, [0x280c] = 0x79fe, [0x280d] = 0x79ff, [0x280e] = 0x7a01,
  [0x280f] = 0x7a04, [0x2810] = 0x7a05, [0x2811] = 0x7a07, [0x2812] = 0x7a08,
  [0x2813] = 0x7a09, [0x2814] = 0x7a0a, [0x2815] = 0x7a0c, [0x2816] = 0x7a0f,
  [0x2817] = 0x7a10, [0x2818] = 0x7a11, [0x2819] = 0x7a12, [0x281a] = 0x7a13,
  [0x281b] = 0x7a15, [0x281c] = 0x7a16, [0x281d] = 0x7a18, [0x281e] = 0x7a19,
  [0x281f] = 0x7a1b, [0x2820] = 0x7a1c, [0x2821] = 0x4e01, [0x2822] = 0x76ef,
  [0x2823] = 0x53ee, [0x2824] = 0x9489, [0x2825] = 0x9876, [0x2826] = 0x9f0e,
  [0x2827] = 0x952d, [0x2828] = 0x5b9a, [0x2829] = 0x8ba2, [0x282a] = 0x4e22,
  [0x282b] = 0x4e1c, [0x282c] = 0x51ac, [0x282d] = 0x8463, [0x282e] = 0x61c2,
  [0x282f] = 0x52a8, [0x2830] = 0x680b, [0x2831] = 0x4f97, [0x2832] = 0x606b,
  [0x2833] = 0x51bb, [0x2834] = 0x6d1e, [0x2835] = 0x515c, [0x2836] = 0x6296,
  [0x2837] = 0x6597, [0x2838] = 0x9661, [0x2839] = 0x8c46, [0x283a] = 0x9017,
  [0x283b] = 0x75d8, [0x283c] = 0x90fd, [0x283d] = 0x7763, [0x283e] = 0x6bd2,
  [0x283f] = 0x728a, [0x2840] = 0x72ec, [0x2841] = 0x8bfb, [0x2842] = 0x5835,
  [0x2843] = 0x7779, [0x2844] = 0x8d4c, [0x2845] = 0x675c, [0x2846] = 0x9540,
  [0x2847] = 0x809a, [0x2848] = 0x5ea6, [0x2849] = 0x6e21, [0x284a] = 0x5992,
  [0x284b] = 0x7aef, [0x284c] = 0x77ed, [0x284d] = 0x953b, [0x284e] = 0x6bb5,
  [0x284f] = 0x65ad, [0x2850] = 0x7f0e, [0x2851] = 0x5806, [0x2852] = 0x5151,
  [0x2853] = 0x961f, [0x2854] = 0x5bf9, [0x2855] = 0x58a9, [0x2856] = 0x5428,
  [0x2857] = 0x8e72, [0x2858] = 0x6566, [0x2859] = 0x987f, [0x285a] = 0x56e4,
  [0x285b] = 0x949d, [0x285c] = 0x76fe, [0x285d] = 0x9041, [0x285e] = 0x6387,
  [0x285f] = 0x54c6, [0x2860] = 0x591a, [0x2861] = 0x593a, [0x2862] = 0x579b,
  [0x2863] = 0x8eb2, [0x2864] = 0x6735, [0x2865] = 0x8dfa, [0x2866] = 0x8235,
  [0x2867] = 0x5241, [0x2868] = 0x60f0, [0x2869] = 0x5815, [0x286a] = 0x86fe,
  [0x286b] = 0x5ce8, [0x286c] = 0x9e45, [0x286d] = 0x4fc4, [0x286e] = 0x989d,
  [0x286f] = 0x8bb9, [0x2870] = 0x5a25, [0x2871] = 0x6076, [0x2872] = 0x5384,
  [0x2873] = 0x627c, [0x2874] = 0x904f, [0x2875] = 0x9102, [0x2876] = 0x997f,
  [0x2877] = 0x6069, [0x2878] = 0x800c, [0x2879] = 0x513f, [0x287a] = 0x8033,
  [0x287b] = 0x5c14, [0x287c] = 0x9975, [0x287d] = 0x6d31, [0x287e] = 0x4e8c,
  [0x2880] = 0x7a1d, [0x2881] = 0x7a1f, [0x2882] = 0x7a21, [0x2883] = 0x7a22,
  [0x2884] = 0x7a24, [0x2885] = 0x7a25, [0x2886] = 0x7a26, [0x2887] = 0x7a27,
  [0x2888] = 0x7a28, [0x2889] = 0x7a29, [0x288a] = 0x7a2a, [0x288b] = 0x7a2b,
  [0x288c] = 0x7a2c, [0x288d] = 0x7a2d, [0x288e] = 0x7a2e, [0x288f] = 0x7a2f,
  [0x2890] = 0x7a30, [0x2891] = 0x7a31, [0x2892] = 0x7a32, [0x2893] = 0x7a34,
  [0x2894] = 0x7a35, [0x2895] = 0x7a36, [0x2896] = 0x7a38, [0x2897] = 0x7a3a,
  [0x2898] = 0x7a3e, [0x2899] = 0x7a40, [0x289a] = 0x7a41, [0x289b] = 0x7a42,
  [0x289c] = 0x7a43, [0x289d] = 0x7a44, [0x289e] = 0x7a45, [0x289f] = 0x7a47,
  [0x28a0] = 0x7a48, [0x28a1] = 0x7a49, [0x28a2] = 0x7a4a, [0x28a3] = 0x7a4b,
  [0x28a4] = 0x7a4c, [0x28a5] = 0x7a4d, [0x28a6] = 0x7a4e, [0x28a7] = 0x7a4f,
  [0x28a8] = 0x7a50, [0x28a9] = 0x7a52, [0x28aa] = 0x7a53, [0x28ab] = 0x7a54,
  [0x28ac] = 0x7a55, [0x28ad] = 0x7a56, [0x28ae] = 0x7a58, [0x28af] = 0x7a59,
  [0x28b0] = 0x7a5a, [0x28b1] = 0x7a5b, [0x28b2] = 0x7a5c, [0x28b3] = 0x7a5d,
  [0x28b4] = 0x7a5e, [0x28b5] = 0x7a5f, [0x28b6] = 0x7a60, [0x28b7] = 0x7a61,
  [0x28b8] = 0x7a62, [0x28b9] = 0x7a63, [0x28ba] = 0x7a64, [0x28bb] = 0x7a65,
  [0x28bc] = 0x7a66, [0x28bd] = 0x7a67, [0x28be] = 0x7a68, [0x28c0] = 0x7a69,
  [0x28c1] = 0x7a6a, [0x28c2] = 0x7a6b, [0x28c3] = 0x7a6c, [0x28c4] = 0x7a6d,
  [0x28c5] = 0x7a6e, [0x28c6] = 0x7a6f, [0x28c7] = 0x7a71, [0x28c8] = 0x7a72,
  [0x28c9] = 0x7a73, [0x28ca] = 0x7a75, [0x28cb] = 0x7a7b, [0x28cc] = 0x7a7c,
  [0x28cd] = 0x7a7d, [0x28ce] = 0x7a7e, [0x28cf] = 0x7a82, [0x28d0] = 0x7a85,
  [0x28d1] = 0x7a87, [0x28d2] = 0x7a89, [0x28d3] = 0x7a8a, [0x28d4] = 0x7a8b,
  [0x28d5] = 0x7a8c, [0x28d6] = 0x7a8e, [0x28d7] = 0x7a8f, [0x28d8] = 0x7a90,
  [0x28d9] = 0x7a93, [0x28da] = 0x7a94, [0x28db] = 0x7a99, [0x28dc] = 0x7a9a,
  [0x28dd] = 0x7a9b, [0x28de] = 0x7a9e, [0x28df] = 0x7aa1, [0x28e0] = 0x7aa2,
  [0x28e1] = 0x8d30, [0x28e2] = 0x53d1, [0x28e3] = 0x7f5a, [0x28e4] = 0x7b4f,
  [0x28e5] = 0x4f10, [0x28e6] = 0x4e4f, [0x28e7] = 0x9600, [0x28e8] = 0x6cd5,
  [0x28e9] = 0x73d0, [0x28ea] = 0x85e9, [0x28eb] = 0x5e06, [0x28ec] = 0x756a,
  [0x28ed] = 0x7ffb, [0x28ee] = 0x6a0a, [0x28ef] = 0x77fe, [0x28f0] = 0x9492,
  [0x28f1] = 0x7e41, [0x28f2] = 0x51e1, [0x28f3] = 0x70e6, [0x28f4] = 0x53cd,
  [0x28f5] = 0x8fd4, [0x28f6] = 0x8303, [0x28f7] = 0x8d29, [0x28f8] = 0x72af,
  [0x28f9] = 0x996d, [0x28fa] = 0x6cdb, [0x28fb] = 0x574a, [0x28fc] = 0x82b3,
  [0x28fd] = 0x65b9, [0x28fe] = 0x80aa, [0x28ff] = 0x623f, [0x2900] = 0x9632,
  [0x2901] = 0x59a8, [0x2902] = 0x4eff, [0x2903] = 0x8bbf, [0x2904] = 0x7eba,
  [0x2905] = 0x653e, [0x2906] = 0x83f2, [0x2907] = 0x975e, [0x2908] = 0x5561,
  [0x2909] = 0x98de, [0x290a] = 0x80a5, [0x290b] = 0x532a, [0x290c] = 0x8bfd,
  [0x290d] = 0x5420, [0x290e] = 0x80ba, [0x290f] = 0x5e9f, [0x2910] = 0x6cb8,
  [0x2911] = 0x8d39, [0x2912] = 0x82ac, [0x2913] = 0x915a, [0x2914] = 0x5429,
  [0x2915] = 0x6c1b, [0x2916] = 0x5206, [0x2917] = 0x7eb7, [0x2918] = 0x575f,
  [0x2919] = 0x711a, [0x291a] = 0x6c7e, [0x291b] = 0x7c89, [0x291c] = 0x594b,
  [0x291d] = 0x4efd, [0x291e] = 0x5fff, [0x291f] = 0x6124, [0x2920] = 0x7caa,
  [0x2921] = 0x4e30, [0x2922] = 0x5c01, [0x2923] = 0x67ab, [0x2924] = 0x8702,
  [0x2925] = 0x5cf0, [0x2926] = 0x950b, [0x2927] = 0x98ce, [0x2928] = 0x75af,
  [0x2929] = 0x70fd, [0x292a] = 0x9022, [0x292b] = 0x51af, [0x292c] = 0x7f1d,
  [0x292d] = 0x8bbd, [0x292e] = 0x5949, [0x292f] = 0x51e4, [0x2930] = 0x4f5b,
  [0x2931] = 0x5426, [0x2932] = 0x592b, [0x2933] = 0x6577, [0x2934] = 0x80a4,
  [0x2935] = 0x5b75, [0x2936] = 0x6276, [0x2937] = 0x62c2, [0x2938] = 0x8f90,
  [0x2939] = 0x5e45, [0x293a] = 0x6c1f, [0x293b] = 0x7b26, [0x293c] = 0x4f0f,
  [0x293d] = 0x4fd8, [0x293e] = 0x670d, [0x2940] = 0x7aa3, [0x2941] = 0x7aa4,
  [0x2942] = 0x7aa7, [0x2943] = 0x7aa9, [0x2944] = 0x7aaa, [0x2945] = 0x7aab,
  [0x2946] = 0x7aae, [0x2947] = 0x7aaf, [0x2948] = 0x7ab0, [0x2949] = 0x7ab1,
  [0x294a] = 0x7ab2, [0x294b] = 0x7ab4, [0x294c] = 0x7ab5, [0x294d] = 0x7ab6,
  [0x294e] = 0x7ab7, [0x294f] = 0x7ab8, [0x2950] = 0x7ab9, [0x2951] = 0x7aba,
  [0x2952] = 0x7abb, [0x2953] = 0x7abc, [0x2954] = 0x7abd, [0x2955] = 0x7abe,
  [0x2956] = 0x7ac0, [0x2957] = 0x7ac1, [0x2958] = 0x7ac2, [0x2959] = 0x7ac3,
  [0x295a] = 0x7ac4, [0x295b] = 0x7ac5, [0x295c] = 0x7ac6, [0x295d] = 0x7ac7,
  [0x295e] = 0x7ac8, [0x295f] = 0x7ac9, [0x2960] = 0x7aca, [0x2961] = 0x7acc,
  [0x2962] = 0x7acd, [0x2963] = 0x7ace, [0x2964] = 0x7acf, [0x2965] = 0x7ad0,
  [0x2966] = 0x7ad1, [0x2967] = 0x7ad2, [0x2968] = 0x7ad3, [0x2969] = 0x7ad4,
  [0x296a] = 0x7ad5, [0x296b] = 0x7ad7, [0x296c] = 0x7ad8, [0x296d] = 0x7ada,
  [0x296e] = 0x7adb, [0x296f] = 0x7adc, [0x2970] = 0x7add, [0x2971] = 0x7ae1,
  [0x2972] = 0x7ae2, [0x2973] = 0x7ae4, [0x2974] = 0x7ae7, [0x2975] = 0x7ae8,
  [0x2976] = 0x7ae9, [0x2977] = 0x7aea, [0x2978] = 0x7aeb, [0x2979] = 0x7aec,
  [0x297a] = 0x7aee, [0x297b] = 0x7af0, [0x297c] = 0x7af1, [0x297d] = 0x7af2,
  [0x297e] = 0x7af3, [0x2980] = 0x7af4, [0x2981] = 0x7af5, [0x2982] = 0x7af6,
  [0x2983] = 0x7af7, [0x2984] = 0x7af8, [0x2985] = 0x7afb, [0x2986] = 0x7afc,
  [0x2987] = 0x7afe, [0x2988] = 0x7b00, [0x2989] = 0x7b01, [0x298a] = 0x7b02,
  [0x298b] = 0x7b05, [0x298c] = 0x7b07, [0x298d] = 0x7b09, [0x298e] = 0x7b0c,
  [0x298f] = 0x7b0d, [0x2990] = 0x7b0e, [0x2991] = 0x7b10, [0x2992] = 0x7b12,
  [0x2993] = 0x7b13, [0x2994] = 0x7b16, [0x2995] = 0x7b17, [0x2996] = 0x7b18,
  [0x2997] = 0x7b1a, [0x2998] = 0x7b1c, [0x2999] = 0x7b1d, [0x299a] = 0x7b1f,
  [0x299b] = 0x7b21, [0x299c] = 0x7b22, [0x299d] = 0x7b23, [0x299e] = 0x7b27,
  [0x299f] = 0x7b29, [0x29a0] = 0x7b2d, [0x29a1] = 0x6d6e, [0x29a2] = 0x6daa,
  [0x29a3] = 0x798f, [0x29a4] = 0x88b1, [0x29a5] = 0x5f17, [0x29a6] = 0x752b,
  [0x29a7] = 0x629a, [0x29a8] = 0x8f85, [0x29a9] = 0x4fef, [0x29aa] = 0x91dc,
  [0x29ab] = 0x65a7, [0x29ac] = 0x812f, [0x29ad] = 0x8151, [0x29ae] = 0x5e9c,
  [0x29af] = 0x8150, [0x29b0] = 0x8d74, [0x29b1] = 0x526f, [0x29b2] = 0x8986,
  [0x29b3] = 0x8d4b, [0x29b4] = 0x590d, [0x29b5] = 0x5085, [0x29b6] = 0x4ed8,
  [0x29b7] = 0x961c, [0x29b8] = 0x7236, [0x29b9] = 0x8179, [0x29ba] = 0x8d1f,
  [0x29bb] = 0x5bcc, [0x29bc] = 0x8ba3, [0x29bd] = 0x9644, [0x29be] = 0x5987,
  [0x29bf] = 0x7f1a, [0x29c0] = 0x5490, [0x29c1] = 0x5676, [0x29c2] = 0x560e,
  [0x29c3] = 0x8be5, [0x29c4] = 0x6539, [0x29c5] = 0x6982, [0x29c6] = 0x9499,
  [0x29c7] = 0x76d6, [0x29c8] = 0x6e89, [0x29c9] = 0x5e72, [0x29ca] = 0x7518,
  [0x29cb] = 0x6746, [0x29cc] = 0x67d1, [0x29cd] = 0x7aff, [0x29ce] = 0x809d,
  [0x29cf] = 0x8d76, [0x29d0] = 0x611f, [0x29d1] = 0x79c6, [0x29d2] = 0x6562,
  [0x29d3] = 0x8d63, [0x29d4] = 0x5188, [0x29d5] = 0x521a, [0x29d6] = 0x94a2,
  [0x29d7] = 0x7f38, [0x29d8] = 0x809b, [0x29d9] = 0x7eb2, [0x29da] = 0x5c97,
  [0x29db] = 0x6e2f, [0x29dc] = 0x6760, [0x29dd] = 0x7bd9, [0x29de] = 0x768b,
  [0x29df] = 0x9ad8, [0x29e0] = 0x818f, [0x29e1] = 0x7f94, [0x29e2] = 0x7cd5,
  [0x29e3] = 0x641e, [0x29e4] = 0x9550, [0x29e5] = 0x7a3f, [0x29e6] = 0x544a,
  [0x29e7] = 0x54e5, [0x29e8] = 0x6b4c, [0x29e9] = 0x6401, [0x29ea] = 0x6208,
  [0x29eb] = 0x9e3d, [0x29ec] = 0x80f3, [0x29ed] = 0x7599, [0x29ee] = 0x5272,
  [0x29ef] = 0x9769, [0x29f0] = 0x845b, [0x29f1] = 0x683c, [0x29f2] = 0x86e4,
  [0x29f3] = 0x9601, [0x29f4] = 0x9694, [0x29f5] = 0x94ec, [0x29f6] = 0x4e2a,
  [0x29f7] = 0x5404, [0x29f8] = 0x7ed9, [0x29f9] = 0x6839, [0x29fa] = 0x8ddf,
  [0x29fb] = 0x8015, [0x29fc] = 0x66f4, [0x29fd] = 0x5e9a, [0x29fe] = 0x7fb9,
  [0x2a00] = 0x7b2f, [0x2a01] = 0x7b30, [0x2a02] = 0x7b32, [0x2a03] = 0x7b34,
  [0x2a04] = 0x7b35, [0x2a05] = 0x7b36, [0x2a06] = 0x7b37, [0x2a07] = 0x7b39,
  [0x2a08] = 0x7b3b, [0x2a09] = 0x7b3d, [0x2a0a] = 0x7b3f, [0x2a0b] = 0x7b40,
  [0x2a0c] = 0x7b41, [0x2a0d] = 0x7b42, [0x2a0e] = 0x7b43, [0x2a0f] = 0x7b44,
  [0x2a10] = 0x7b46, [0x2a11] = 0x7b48, [0x2a12] = 0x7b4a, [0x2a13] = 0x7b4d,
  [0x2a14] = 0x7b4e, [0x2a15] = 0x7b53, [0x2a16] = 0x7b55, [0x2a17] = 0x7b57,
  [0x2a18] = 0x7b59, [0x2a19] = 0x7b5c, [0x2a1a] = 0x7b5e, [0x2a1b] = 0x7b5f,
  [0x2a1c] = 0x7b61, [0x2a1d] = 0x7b63, [0x2a1e] = 0x7b64, [0x2a1f] = 0x7b65,
  [0x2a20] = 0x7b66, [0x2a21] = 0x7b67, [0x2a22] = 0x7b68, [0x2a23] = 0x7b69,
  [0x2a24] = 0x7b6a, [0x2a25] = 0x7b6b, [0x2a26] = 0x7b6c, [0x2a27] = 0x7b6d,
  [0x2a28] = 0x7b6f, [0x2a29] = 0x7b70, [0x2a2a] = 0x7b73, [0x2a2b] = 0x7b74,
  [0x2a2c] = 0x7b76, [0x2a2d] = 0x7b78, [0x2a2e] = 0x7b7a, [0x2a2f] = 0x7b7c,
  [0x2a30] = 0x7b7d, [0x2a31] = 0x7b7f, [0x2a32] = 0x7b81, [0x2a33] = 0x7b82,
  [0x2a34] = 0x7b83, [0x2a35] = 0x7b84, [0x2a36] = 0x7b86, [0x2a37] = 0x7b87,
  [0x2a38] = 0x7b88, [0x2a39] = 0x7b89, [0x2a3a] = 0x7b8a, [0x2a3b] = 0x7b8b,
  [0x2a3c] = 0x7b8c, [0x2a3d] = 0x7b8e, [0x2a3e] = 0x7b8f, [0x2a40] = 0x7b91,
  [0x2a41] = 0x7b92, [0x2a42] = 0x7b93, [0x2a43] = 0x7b96, [0x2a44] = 0x7b98,
  [0x2a45] = 0x7b99, [0x2a46] = 0x7b9a, [0x2a47] = 0x7b9b, [0x2a48] = 0x7b9e,
  [0x2a49] = 0x7b9f, [0x2a4a] = 0x7ba0, [0x2a4b] = 0x7ba3, [0x2a4c] = 0x7ba4,
  [0x2a4d] = 0x7ba5, [0x2a4e] = 0x7bae, [0x2a4f] = 0x7baf, [0x2a50] = 0x7bb0,
  [0x2a51] = 0x7bb2, [0x2a52] = 0x7bb3, [0x2a53] = 0x7bb5, [0x2a54] = 0x7bb6,
  [0x2a55] = 0x7bb7, [0x2a56] = 0x7bb9, [0x2a57] = 0x7bba, [0x2a58] = 0x7bbb,
  [0x2a59] = 0x7bbc, [0x2a5a] = 0x7bbd, [0x2a5b] = 0x7bbe, [0x2a5c] = 0x7bbf,
  [0x2a5d] = 0x7bc0, [0x2a5e] = 0x7bc2, [0x2a5f] = 0x7bc3, [0x2a60] = 0x7bc4,
  [0x2a61] = 0x57c2, [0x2a62] = 0x803f, [0x2a63] = 0x6897, [0x2a64] = 0x5de5,
  [0x2a65] = 0x653b, [0x2a66] = 0x529f, [0x2a67] = 0x606d, [0x2a68] = 0x9f9a,
  [0x2a69] = 0x4f9b, [0x2a6a] = 0x8eac, [0x2a6b] = 0x516c, [0x2a6c] = 0x5bab,
  [0x2a6d] = 0x5f13, [0x2a6e] = 0x5de9, [0x2a6f] = 0x6c5e, [0x2a70] = 0x62f1,
  [0x2a71] = 0x8d21, [0x2a72] = 0x5171, [0x2a73] = 0x94a9, [0x2a74] = 0x52fe,
  [0x2a75] = 0x6c9f, [0x2a76] = 0x82df, [0x2a77] = 0x72d7, [0x2a78] = 0x57a2,
  [0x2a79] = 0x6784, [0x2a7a] = 0x8d2d, [0x2a7b] = 0x591f, [0x2a7c] = 0x8f9c,
  [0x2a7d] = 0x83c7, [0x2a7e] = 0x5495, [0x2a7f] = 0x7b8d, [0x2a80] = 0x4f30,
  [0x2a81] = 0x6cbd, [0x2a82] = 0x5b64, [0x2a83] = 0x59d1, [0x2a84] = 0x9f13,
  [0x2a85] = 0x53e4, [0x2a86] = 0x86ca, [0x2a87] = 0x9aa8, [0x2a88] = 0x8c37,
  [0x2a89] = 0x80a1, [0x2a8a] = 0x6545, [0x2a8b] = 0x987e, [0x2a8c] = 0x56fa,
  [0x2a8d] = 0x96c7, [0x2a8e] = 0x522e, [0x2a8f] = 0x74dc, [0x2a90] = 0x5250,
  [0x2a91] = 0x5be1, [0x2a92] = 0x6302, [0x2a93] = 0x8902, [0x2a94] = 0x4e56,
  [0x2a95] = 0x62d0, [0x2a96] = 0x602a, [0x2a97] = 0x68fa, [0x2a98] = 0x5173,
  [0x2a99] = 0x5b98, [0x2a9a] = 0x51a0, [0x2a9b] = 0x89c2, [0x2a9c] = 0x7ba1,
  [0x2a9d] = 0x9986, [0x2a9e] = 0x7f50, [0x2a9f] = 0x60ef, [0x2aa0] = 0x704c,
  [0x2aa1] = 0x8d2f, [0x2aa2] = 0x5149, [0x2aa3] = 0x5e7f, [0x2aa4] = 0x901b,
  [0x2aa5] = 0x7470, [0x2aa6] = 0x89c4, [0x2aa7] = 0x572d, [0x2aa8] = 0x7845,
  [0x2aa9] = 0x5f52, [0x2aaa] = 0x9f9f, [0x2aab] = 0x95fa, [0x2aac] = 0x8f68,
  [0x2aad] = 0x9b3c, [0x2aae] = 0x8be1, [0x2aaf] = 0x7678, [0x2ab0] = 0x6842,
  [0x2ab1] = 0x67dc, [0x2ab2] = 0x8dea, [0x2ab3] = 0x8d35, [0x2ab4] = 0x523d,
  [0x2ab5] = 0x8f8a, [0x2ab6] = 0x6eda, [0x2ab7] = 0x68cd, [0x2ab8] = 0x9505,
  [0x2ab9] = 0x90ed, [0x2aba] = 0x56fd, [0x2abb] = 0x679c, [0x2abc] = 0x88f9,
  [0x2abd] = 0x8fc7, [0x2abe] = 0x54c8, [0x2ac0] = 0x7bc5, [0x2ac1] = 0x7bc8,
  [0x2ac2] = 0x7bc9, [0x2ac3] = 0x7bca, [0x2ac4] = 0x7bcb, [0x2ac5] = 0x7bcd,
  [0x2ac6] = 0x7bce, [0x2ac7] = 0x7bcf, [0x2ac8] = 0x7bd0, [0x2ac9] = 0x7bd2,
  [0x2aca] = 0x7bd4, [0x2acb] = 0x7bd5, [0x2acc] = 0x7bd6, [0x2acd] = 0x7bd7,
  [0x2ace] = 0x7bd8, [0x2acf] = 0x7bdb, [0x2ad0] = 0x7bdc, [0x2ad1] = 0x7bde,
  [0x2ad2] = 0x7bdf, [0x2ad3] = 0x7be0, [0x2ad4] = 0x7be2, [0x2ad5] = 0x7be3,
  [0x2ad6] = 0x7be4, [0x2ad7] = 0x7be7, [0x2ad8] = 0x7be8, [0x2ad9] = 0x7be9,
  [0x2ada] = 0x7beb, [0x2adb] = 0x7bec, [0x2adc] = 0x7bed, [0x2add] = 0x7bef,
  [0x2ade] = 0x7bf0, [0x2adf] = 0x7bf2, [0x2ae0] = 0x7bf3, [0x2ae1] = 0x7bf4,
  [0x2ae2] = 0x7bf5, [0x2ae3] = 0x7bf6, [0x2ae4] = 0x7bf8, [0x2ae5] = 0x7bf9,
  [0x2ae6] = 0x7bfa, [0x2ae7] = 0x7bfb, [0x2ae8] = 0x7bfd, [0x2ae9] = 0x7bff,
  [0x2aea] = 0x7c00, [0x2aeb] = 0x7c01, [0x2aec] = 0x7c02, [0x2aed] = 0x7c03,
  [0x2aee] = 0x7c04, [0x2aef] = 0x7c05, [0x2af0] = 0x7c06, [0x2af1] = 0x7c08,
  [0x2af2] = 0x7c09, [0x2af3] = 0x7c0a, [0x2af4] = 0x7c0d, [0x2af5] = 0x7c0e,
  [0x2af6] = 0x7c10, [0x2af7] = 0x7c11, [0x2af8] = 0x7c12, [0x2af9] = 0x7c13,
  [0x2afa] = 0x7c14, [0x2afb] = 0x7c15, [0x2afc] = 0x7c17, [0x2afd] = 0x7c18,
  [0x2afe] = 0x7c19, [0x2b00] = 0x7c1a, [0x2b01] = 0x7c1b, [0x2b02] = 0x7c1c,
  [0x2b03] = 0x7c1d, [0x2b04] = 0x7c1e, [0x2b05] = 0x7c20, [0x2b06] = 0x7c21,
  [0x2b07] = 0x7c22, [0x2b08] = 0x7c23, [0x2b09] = 0x7c24, [0x2b0a] = 0x7c25,
  [0x2b0b] = 0x7c28, [0x2b0c] = 0x7c29, [0x2b0d] = 0x7c2b, [0x2b0e] = 0x7c2c,
  [0x2b0f] = 0x7c2d, [0x2b10] = 0x7c2e, [0x2b11] = 0x7c2f, [0x2b12] = 0x7c30,
  [0x2b13] = 0x7c31, [0x2b14] = 0x7c32, [0x2b15] = 0x7c33, [0x2b16] = 0x7c34,
  [0x2b17] = 0x7c35, [0x2b18] = 0x7c36, [0x2b19] = 0x7c37, [0x2b1a] = 0x7c39,
  [0x2b1b] = 0x7c3a, [0x2b1c] = 0x7c3b, [0x2b1d] = 0x7c3c, [0x2b1e] = 0x7c3d,
  [0x2b1f] = 0x7c3e, [0x2b20] = 0x7c42, [0x2b21] = 0x9ab8, [0x2b22] = 0x5b69,
  [0x2b23] = 0x6d77, [0x2b24] = 0x6c26, [0x2b25] = 0x4ea5, [0x2b26] = 0x5bb3,
  [0x2b27] = 0x9a87, [0x2b28] = 0x9163, [0x2b29] = 0x61a8, [0x2b2a] = 0x90af,
  [0x2b2b] = 0x97e9, [0x2b2c] = 0x542b, [0x2b2d] = 0x6db5, [0x2b2e] = 0x5bd2,
  [0x2b2f] = 0x51fd, [0x2b30] = 0x558a, [0x2b31] = 0x7f55, [0x2b32] = 0x7ff0,
  [0x2b33] = 0x64bc, [0x2b34] = 0x634d, [0x2b35] = 0x65f1, [0x2b36] = 0x61be,
  [0x2b37] = 0x608d, [0x2b38] = 0x710a, [0x2b39] = 0x6c57, [0x2b3a] = 0x6c49,
  [0x2b3b] = 0x592f, [0x2b3c] = 0x676d, [0x2b3d] = 0x822a, [0x2b3e] = 0x58d5,
  [0x2b3f] = 0x568e, [0x2b40] = 0x8c6a, [0x2b41] = 0x6beb, [0x2b42] = 0x90dd,
  [0x2b43] = 0x597d, [0x2b44] = 0x8017, [0x2b45] = 0x53f7, [0x2b46] = 0x6d69,
  [0x2b47] = 0x5475, [0x2b48] = 0x559d, [0x2b49] = 0x8377, [0x2b4a] = 0x83cf,
  [0x2b4b] = 0x6838, [0x2b4c] = 0x79be, [0x2b4d] = 0x548c, [0x2b4e] = 0x4f55,
  [0x2b4f] = 0x5408, [0x2b50] = 0x76d2, [0x2b51] = 0x8c89, [0x2b52] = 0x9602,
  [0x2b53] = 0x6cb3, [0x2b54] = 0x6db8, [0x2b55] = 0x8d6b, [0x2b56] = 0x8910,
  [0x2b57] = 0x9e64, [0x2b58] = 0x8d3a, [0x2b59] = 0x563f, [0x2b5a] = 0x9ed1,
  [0x2b5b] = 0x75d5, [0x2b5c] = 0x5f88, [0x2b5d] = 0x72e0, [0x2b5e] = 0x6068,
  [0x2b5f] = 0x54fc, [0x2b60] = 0x4ea8, [0x2b61] = 0x6a2a, [0x2b62] = 0x8861,
  [0x2b63] = 0x6052, [0x2b64] = 0x8f70, [0x2b65] = 0x54c4, [0x2b66] = 0x70d8,
  [0x2b67] = 0x8679, [0x2b68] = 0x9e3f, [0x2b69] = 0x6d2a, [0x2b6a] = 0x5b8f,
  [0x2b6b] = 0x5f18, [0x2b6c] = 0x7ea2, [0x2b6d] = 0x5589, [0x2b6e] = 0x4faf,
  [0x2b6f] = 0x7334, [0x2b70] = 0x543c, [0x2b71] = 0x539a, [0x2b72] = 0x5019,
  [0x2b73] = 0x540e, [0x2b74] = 0x547c, [0x2b75] = 0x4e4e, [0x2b76] = 0x5ffd,
  [0x2b77] = 0x745a, [0x2b78] = 0x58f6, [0x2b79] = 0x846b, [0x2b7a] = 0x80e1,
  [0x2b7b] = 0x8774, [0x2b7c] = 0x72d0, [0x2b7d] = 0x7cca, [0x2b7e] = 0x6e56,
  [0x2b80] = 0x7c43, [0x2b81] = 0x7c44, [0x2b82] = 0x7c45, [0x2b83] = 0x7c46,
  [0x2b84] = 0x7c47, [0x2b85] = 0x7c48, [0x2b86] = 0x7c49, [0x2b87] = 0x7c4a,
  [0x2b88] = 0x7c4b, [0x2b89] = 0x7c4c, [0x2b8a] = 0x7c4e, [0x2b8b] = 0x7c4f,
  [0x2b8c] = 0x7c50, [0x2b8d] = 0x7c51, [0x2b8e] = 0x7c52, [0x2b8f] = 0x7c53,
  [0x2b90] = 0x7c54, [0x2b91] = 0x7c55, [0x2b92] = 0x7c56, [0x2b93] = 0x7c57,
  [0x2b94] = 0x7c58, [0x2b95] = 0x7c59, [0x2b96] = 0x7c5a, [0x2b97] = 0x7c5b,
  [0x2b98] = 0x7c5c, [0x2b99] = 0x7c5d, [0x2b9a] = 0x7c5e, [0x2b9b] = 0x7c5f,
  [0x2b9c] = 0x7c60, [0x2b9d] = 0x7c61, [0x2b9e] = 0x7c62, [0x2b9f] = 0x7c63,
  [0x2ba0] = 0x7c64, [0x2ba1] = 0x7c65, [0x2ba2] = 0x7c66, [0x2ba3] = 0x7c67,
  [0x2ba4] = 0x7c68, [0x2ba5] = 0x7c69, [0x2ba6] = 0x7c6a, [0x2ba7] = 0x7c6b,
  [0x2ba8] = 0x7c6c, [0x2ba9] = 0x7c6d, [0x2baa] = 0x7c6e, [0x2bab] = 0x7c6f,
  [0x2bac] = 0x7c70, [0x2bad] = 0x7c71, [0x2bae] = 0x7c72, [0x2baf] = 0x7c75,
  [0x2bb0] = 0x7c76, [0x2bb1] = 0x7c77, [0x2bb2] = 0x7c78, [0x2bb3] = 0x7c79,
  [0x2bb4] = 0x7c7a, [0x2bb5] = 0x7c7e, [0x2bb6] = 0x7c7f, [0x2bb7] = 0x7c80,
  [0x2bb8] = 0x7c81, [0x2bb9] = 0x7c82, [0x2bba] = 0x7c83, [0x2bbb] = 0x7c84,
  [0x2bbc] = 0x7c85, [0x2bbd] = 0x7c86, [0x2bbe] = 0x7c87, [0x2bc0] = 0x7c88,
  [0x2bc1] = 0x7c8a, [0x2bc2] = 0x7c8b, [0x2bc3] = 0x7c8c, [0x2bc4] = 0x7c8d,
  [0x2bc5] = 0x7c8e, [0x2bc6] = 0x7c8f, [0x2bc7] = 0x7c90, [0x2bc8] = 0x7c93,
  [0x2bc9] = 0x7c94, [0x2bca] = 0x7c96, [0x2bcb] = 0x7c99, [0x2bcc] = 0x7c9a,
  [0x2bcd] = 0x7c9b, [0x2bce] = 0x7ca0, [0x2bcf] = 0x7ca1, [0x2bd0] = 0x7ca3,
  [0x2bd1] = 0x7ca6, [0x2bd2] = 0x7ca7, [0x2bd3] = 0x7ca8, [0x2bd4] = 0x7ca9,
  [0x2bd5] = 0x7cab, [0x2bd6] = 0x7cac, [0x2bd7] = 0x7cad, [0x2bd8] = 0x7caf,
  [0x2bd9] = 0x7cb0, [0x2bda] = 0x7cb4, [0x2bdb] = 0x7cb5, [0x2bdc] = 0x7cb6,
  [0x2bdd] = 0x7cb7, [0x2bde] = 0x7cb8, [0x2bdf] = 0x7cba, [0x2be0] = 0x7cbb,
  [0x2be1] = 0x5f27, [0x2be2] = 0x864e, [0x2be3] = 0x552c, [0x2be4] = 0x62a4,
  [0x2be5] = 0x4e92, [0x2be6] = 0x6caa, [0x2be7] = 0x6237, [0x2be8] = 0x82b1,
  [0x2be9] = 0x54d7, [0x2bea] = 0x534e, [0x2beb] = 0x733e, [0x2bec] = 0x6ed1,
  [0x2bed] = 0x753b, [0x2bee] = 0x5212, [0x2bef] = 0x5316, [0x2bf0] = 0x8bdd,
  [0x2bf1] = 0x69d0, [0x2bf2] = 0x5f8a, [0x2bf3] = 0x6000, [0x2bf4] = 0x6dee,
  [0x2bf5] = 0x574f, [0x2bf6] = 0x6b22, [0x2bf7] = 0x73af, [0x2bf8] = 0x6853,
  [0x2bf9] = 0x8fd8, [0x2bfa] = 0x7f13, [0x2bfb] = 0x6362, [0x2bfc] = 0x60a3,
  [0x2bfd] = 0x5524, [0x2bfe] = 0x75ea, [0x2bff] = 0x8c62, [0x2c00] = 0x7115,
  [0x2c01] = 0x6da3, [0x2c02] = 0x5ba6, [0x2c03] = 0x5e7b, [0x2c04] = 0x8352,
  [0x2c05] = 0x614c, [0x2c06] = 0x9ec4, [0x2c07] = 0x78fa, [0x2c08] = 0x8757,
  [0x2c09] = 0x7c27, [0x2c0a] = 0x7687, [0x2c0b] = 0x51f0, [0x2c0c] = 0x60f6,
  [0x2c0d] = 0x714c, [0x2c0e] = 0x6643, [0x2c0f] = 0x5e4c, [0x2c10] = 0x604d,
  [0x2c11] = 0x8c0e, [0x2c12] = 0x7070, [0x2c13] = 0x6325, [0x2c14] = 0x8f89,
  [0x2c15] = 0x5fbd, [0x2c16] = 0x6062, [0x2c17] = 0x86d4, [0x2c18] = 0x56de,
  [0x2c19] = 0x6bc1, [0x2c1a] = 0x6094, [0x2c1b] = 0x6167, [0x2c1c] = 0x5349,
  [0x2c1d] = 0x60e0, [0x2c1e] = 0x6666, [0x2c1f] = 0x8d3f, [0x2c20] = 0x79fd,
  [0x2c21] = 0x4f1a, [0x2c22] = 0x70e9, [0x2c23] = 0x6c47, [0x2c24] = 0x8bb3,
  [0x2c25] = 0x8bf2, [0x2c26] = 0x7ed8, [0x2c27] = 0x8364, [0x2c28] = 0x660f,
  [0x2c29] = 0x5a5a, [0x2c2a] = 0x9b42, [0x2c2b] = 0x6d51, [0x2c2c] = 0x6df7,
  [0x2c2d] = 0x8c41, [0x2c2e] = 0x6d3b, [0x2c2f] = 0x4f19, [0x2c30] = 0x706b,
  [0x2c31] = 0x83b7, [0x2c32] = 0x6216, [0x2c33] = 0x60d1, [0x2c34] = 0x970d,
  [0x2c35] = 0x8d27, [0x2c36] = 0x7978, [0x2c37] = 0x51fb, [0x2c38] = 0x573e,
  [0x2c39] = 0x57fa, [0x2c3a] = 0x673a, [0x2c3b] = 0x7578, [0x2c3c] = 0x7a3d,
  [0x2c3d] = 0x79ef, [0x2c3e] = 0x7b95, [0x2c40] = 0x7cbf, [0x2c41] = 0x7cc0,
  [0x2c42] = 0x7cc2, [0x2c43] = 0x7cc3, [0x2c44] = 0x7cc4, [0x2c45] = 0x7cc6,
  [0x2c46] = 0x7cc9, [0x2c47] = 0x7ccb, [0x2c48] = 0x7cce, [0x2c49] = 0x7ccf,
  [0x2c4a] = 0x7cd0, [0x2c4b] = 0x7cd1, [0x2c4c] = 0x7cd2, [0x2c4d] = 0x7cd3,
  [0x2c4e] = 0x7cd4, [0x2c4f] = 0x7cd8, [0x2c50] = 0x7cda, [0x2c51] = 0x7cdb,
  [0x2c52] = 0x7cdd, [0x2c53] = 0x7cde, [0x2c54] = 0x7ce1, [0x2c55] = 0x7ce2,
  [0x2c56] = 0x7ce3, [0x2c57] = 0x7ce4, [0x2c58] = 0x7ce5, [0x2c59] = 0x7ce6,
  [0x2c5a] = 0x7ce7, [0x2c5b] = 0x7ce9, [0x2c5c] = 0x7cea, [0x2c5d] = 0x7ceb,
  [0x2c5e] = 0x7cec, [0x2c5f] = 0x7ced, [0x2c60] = 0x7cee, [0x2c61] = 0x7cf0,
  [0x2c62] = 0x7cf1, [0x2c63] = 0x7cf2, [0x2c64] = 0x7cf3, [0x2c65] = 0x7cf4,
  [0x2c66] = 0x7cf5, [0x2c67] = 0x7cf6, [0x2c68] = 0x7cf7, [0x2c69] = 0x7cf9,
  [0x2c6a] = 0x7cfa, [0x2c6b] = 0x7cfc, [0x2c6c] = 0x7cfd, [0x2c6d] = 0x7cfe,
  [0x2c6e] = 0x7cff, [0x2c6f] = 0x7d00, [0x2c70] = 0x7d01, [0x2c71] = 0x7d02,
  [0x2c72] = 0x7d03, [0x2c73] = 0x7d04, [0x2c74] = 0x7d05, [0x2c75] = 0x7d06,
  [0x2c76] = 0x7d07, [0x2c77] = 0x7d08, [0x2c78] = 0x7d09, [0x2c79] = 0x7d0b,
  [0x2c7a] = 0x7d0c, [0x2c7b] = 0x7d0d, [0x2c7c] = 0x7d0e, [0x2c7d] = 0x7d0f,
  [0x2c7e] = 0x7d10, [0x2c80] = 0x7d11, [0x2c81] = 0x7d12, [0x2c82] = 0x7d13,
  [0x2c83] = 0x7d14, [0x2c84] = 0x7d15, [0x2c85] = 0x7d16, [0x2c86] = 0x7d17,
  [0x2c87] = 0x7d18, [0x2c88] = 0x7d19, [0x2c89] = 0x7d1a, [0x2c8a] = 0x7d1b,
  [0x2c8b] = 0x7d1c, [0x2c8c] = 0x7d1d, [0x2c8d] = 0x7d1e, [0x2c8e] = 0x7d1f,
  [0x2c8f] = 0x7d21, [0x2c90] = 0x7d23, [0x2c91] = 0x7d24, [0x2c92] = 0x7d25,
  [0x2c93] = 0x7d26, [0x2c94] = 0x7d28, [0x2c95] = 0x7d29, [0x2c96] = 0x7d2a,
  [0x2c97] = 0x7d2c, [0x2c98] = 0x7d2d, [0x2c99] = 0x7d2e, [0x2c9a] = 0x7d30,
  [0x2c9b] = 0x7d31, [0x2c9c] = 0x7d32, [0x2c9d] = 0x7d33, [0x2c9e] = 0x7d34,
  [0x2c9f] = 0x7d35, [0x2ca0] = 0x7d36, [0x2ca1] = 0x808c, [0x2ca2] = 0x9965,
  [0x2ca3] = 0x8ff9, [0x2ca4] = 0x6fc0, [0x2ca5] = 0x8ba5, [0x2ca6] = 0x9e21,
  [0x2ca7] = 0x59ec, [0x2ca8] = 0x7ee9, [0x2ca9] = 0x7f09, [0x2caa] = 0x5409,
  [0x2cab] = 0x6781, [0x2cac] = 0x68d8, [0x2cad] = 0x8f91, [0x2cae] = 0x7c4d,
  [0x2caf] = 0x96c6, [0x2cb0] = 0x53ca, [0x2cb1] = 0x6025, [0x2cb2] = 0x75be,
  [0x2cb3] = 0x6c72, [0x2cb4] = 0x5373, [0x2cb5] = 0x5ac9, [0x2cb6] = 0x7ea7,
  [0x2cb7] = 0x6324, [0x2cb8] = 0x51e0, [0x2cb9] = 0x810a, [0x2cba] = 0x5df1,
  [0x2cbb] = 0x84df, [0x2cbc] = 0x6280, [0x2cbd] = 0x5180, [0x2cbe] = 0x5b63,
  [0x2cbf] = 0x4f0e, [0x2cc0] = 0x796d, [0x2cc1] = 0x5242, [0x2cc2] = 0x60b8,
  [0x2cc3] = 0x6d4e, [0x2cc4] = 0x5bc4, [0x2cc5] = 0x5bc2, [0x2cc6] = 0x8ba1,
  [0x2cc7] = 0x8bb0, [0x2cc8] = 0x65e2, [0x2cc9] = 0x5fcc, [0x2cca] = 0x9645,
  [0x2ccb] = 0x5993, [0x2ccc] = 0x7ee7, [0x2ccd] = 0x7eaa, [0x2cce] = 0x5609,
  [0x2ccf] = 0x67b7, [0x2cd0] = 0x5939, [0x2cd1] = 0x4f73, [0x2cd2] = 0x5bb6,
  [0x2cd3] = 0x52a0, [0x2cd4] = 0x835a, [0x2cd5] = 0x988a, [0x2cd6] = 0x8d3e,
  [0x2cd7] = 0x7532, [0x2cd8] = 0x94be, [0x2cd9] = 0x5047, [0x2cda] = 0x7a3c,
  [0x2cdb] = 0x4ef7, [0x2cdc] = 0x67b6, [0x2cdd] = 0x9a7e, [0x2cde] = 0x5ac1,
  [0x2cdf] = 0x6b7c, [0x2ce0] = 0x76d1, [0x2ce1] = 0x575a, [0x2ce2] = 0x5c16,
  [0x2ce3] = 0x7b3a, [0x2ce4] = 0x95f4, [0x2ce5] = 0x714e, [0x2ce6] = 0x517c,
  [0x2ce7] = 0x80a9, [0x2ce8] = 0x8270, [0x2ce9] = 0x5978, [0x2cea] = 0x7f04,
  [0x2ceb] = 0x8327, [0x2cec] = 0x68c0, [0x2ced] = 0x67ec, [0x2cee] = 0x78b1,
  [0x2cef] = 0x7877, [0x2cf0] = 0x62e3, [0x2cf1] = 0x6361, [0x2cf2] = 0x7b80,
  [0x2cf3] = 0x4fed, [0x2cf4] = 0x526a, [0x2cf5] = 0x51cf, [0x2cf6] = 0x8350,
  [0x2cf7] = 0x69db, [0x2cf8] = 0x9274, [0x2cf9] = 0x8df5, [0x2cfa] = 0x8d31,
  [0x2cfb] = 0x89c1, [0x2cfc] = 0x952e, [0x2cfd] = 0x7bad, [0x2cfe] = 0x4ef6,
  [0x2d00] = 0x7d37, [0x2d01] = 0x7d38, [0x2d02] = 0x7d39, [0x2d03] = 0x7d3a,
  [0x2d04] = 0x7d3b, [0x2d05] = 0x7d3c, [0x2d06] = 0x7d3d, [0x2d07] = 0x7d3e,
  [0x2d08] = 0x7d3f, [0x2d09] = 0x7d40, [0x2d0a] = 0x7d41, [0x2d0b] = 0x7d42,
  [0x2d0c] = 0x7d43, [0x2d0d] = 0x7d44, [0x2d0e] = 0x7d45, [0x2d0f] = 0x7d46,
  [0x2d10] = 0x7d47, [0x2d11] = 0x7d48, [0x2d12] = 0x7d49, [0x2d13] = 0x7d4a,
  [0x2d14] = 0x7d4b, [0x2d15] = 0x7d4c, [0x2d16] = 0x7d4d, [0x2d17] = 0x7d4e,
  [0x2d18] = 0x7d4f, [0x2d19] = 0x7d50, [0x2d1a] = 0x7d51, [0x2d1b] = 0x7d52,
  [0x2d1c] = 0x7d53, [0x2d1d] = 0x7d54, [0x2d1e] = 0x7d55, [0x2d1f] = 0x7d56,
  [0x2d20] = 0x7d57, [0x2d21] = 0x7d58, [0x2d22] = 0x7d59, [0x2d23] = 0x7d5a,
  [0x2d24] = 0x7d5b, [0x2d25] = 0x7d5c, [0x2d26] = 0x7d5d, [0x2d27] = 0x7d5e,
  [0x2d28] = 0x7d5f, [0x2d29] = 0x7d60, [0x2d2a] = 0x7d61, [0x2d2b] = 0x7d62,
  [0x2d2c] = 0x7d63, [0x2d2d] = 0x7d64, [0x2d2e] = 0x7d65, [0x2d2f] = 0x7d66,
  [0x2d30] = 0x7d67, [0x2d31] = 0x7d68, [0x2d32] = 0x7d69, [0x2d33] = 0x7d6a,
  [0x2d34] = 0x7d6b, [0x2d35] = 0x7d6c, [0x2d36] = 0x7d6d, [0x2d37] = 0x7d6f,
  [0x2d38] = 0x7d70, [0x2d39] = 0x7d71, [0x2d3a] = 0x7d72, [0x2d3b] = 0x7d73,
  [0x2d3c] = 0x7d74, [0x2d3d] = 0x7d75, [0x2d3e] = 0x7d76, [0x2d40] = 0x7d78,
  [0x2d41] = 0x7d79, [0x2d42] = 0x7d7a, [0x2d43] = 0x7d7b, [0x2d44] = 0x7d7c,
  [0x2d45] = 0x7d7d, [0x2d46] = 0x7d7e, [0x2d47] = 0x7d7f, [0x2d48] = 0x7d80,
  [0x2d49] = 0x7d81, [0x2d4a] = 0x7d82, [0x2d4b] = 0x7d83, [0x2d4c] = 0x7d84,
  [0x2d4d] = 0x7d85, [0x2d4e] = 0x7d86, [0x2d4f] = 0x7d87, [0x2d50] = 0x7d88,
  [0x2d51] = 0x7d89, [0x2d52] = 0x7d8a, [0x2d53] = 0x7d8b, [0x2d54] = 0x7d8c,
  [0x2d55] = 0x7d8d, [0x2d56] = 0x7d8e, [0x2d57] = 0x7d8f, [0x2d58] = 0x7d90,
  [0x2d59] = 0x7d91, [0x2d5a] = 0x7d92, [0x2d5b] = 0x7d93, [0x2d5c] = 0x7d94,
  [0x2d5d] = 0x7d95, [0x2d5e] = 0x7d96, [0x2d5f] = 0x7d97, [0x2d60] = 0x7d98,
  [0x2d61] = 0x5065, [0x2d62] = 0x8230, [0x2d63] = 0x5251, [0x2d64] = 0x996f,
  [0x2d65] = 0x6e10, [0x2d66] = 0x6e85, [0x2d67] = 0x6da7, [0x2d68] = 0x5efa,
  [0x2d69] = 0x50f5, [0x2d6a] = 0x59dc, [0x2d6b] = 0x5c06, [0x2d6c] = 0x6d46,
  [0x2d6d] = 0x6c5f, [0x2d6e] = 0x7586, [0x2d6f] = 0x848b, [0x2d70] = 0x6868,
  [0x2d71] = 0x5956, [0x2d72] = 0x8bb2, [0x2d73] = 0x5320, [0x2d74] = 0x9171,
  [0x2d75] = 0x964d, [0x2d76] = 0x8549, [0x2d77] = 0x6912, [0x2d78] = 0x7901,
  [0x2d79] = 0x7126, [0x2d7a] = 0x80f6, [0x2d7b] = 0x4ea4, [0x2d7c] = 0x90ca,
  [0x2d7d] = 0x6d47, [0x2d7e] = 0x9a84, [0x2d7f] = 0x5a07, [0x2d80] = 0x56bc,
  [0x2d81] = 0x6405, [0x2d82] = 0x94f0, [0x2d83] = 0x77eb, [0x2d84] = 0x4fa5,
  [0x2d85] = 0x811a, [0x2d86] = 0x72e1, [0x2d87] = 0x89d2, [0x2d88] = 0x997a,
  [0x2d89] = 0x7f34, [0x2d8a] = 0x7ede, [0x2d8b] = 0x527f, [0x2d8c] = 0x6559,
  [0x2d8d] = 0x9175, [0x2d8e] = 0x8f7f, [0x2d8f] = 0x8f83, [0x2d90] = 0x53eb,
  [0x2d91] = 0x7a96, [0x2d92] = 0x63ed, [0x2d93] = 0x63a5, [0x2d94] = 0x7686,
  [0x2d95] = 0x79f8, [0x2d96] = 0x8857, [0x2d97] = 0x9636, [0x2d98] = 0x622a,
  [0x2d99] = 0x52ab, [0x2d9a] = 0x8282, [0x2d9b] = 0x6854, [0x2d9c] = 0x6770,
  [0x2d9d] = 0x6377, [0x2d9e] = 0x776b, [0x2d9f] = 0x7aed, [0x2da0] = 0x6d01,
  [0x2da1] = 0x7ed3, [0x2da2] = 0x89e3, [0x2da3] = 0x59d0, [0x2da4] = 0x6212,
  [0x2da5] = 0x85c9, [0x2da6] = 0x82a5, [0x2da7] = 0x754c, [0x2da8] = 0x501f,
  [0x2da9] = 0x4ecb, [0x2daa] = 0x75a5, [0x2dab] = 0x8beb, [0x2dac] = 0x5c4a,
  [0x2dad] = 0x5dfe, [0x2dae] = 0x7b4b, [0x2daf] = 0x65a4, [0x2db0] = 0x91d1,
  [0x2db1] = 0x4eca, [0x2db2] = 0x6d25, [0x2db3] = 0x895f, [0x2db4] = 0x7d27,
  [0x2db5] = 0x9526, [0x2db6] = 0x4ec5, [0x2db7] = 0x8c28, [0x2db8] = 0x8fdb,
  [0x2db9] = 0x9773, [0x2dba] = 0x664b, [0x2dbb] = 0x7981, [0x2dbc] = 0x8fd1,
  [0x2dbd] = 0x70ec, [0x2dbe] = 0x6d78, [0x2dc0] = 0x7d99, [0x2dc1] = 0x7d9a,
  [0x2dc2] = 0x7d9b, [0x2dc3] = 0x7d9c, [0x2dc4] = 0x7d9d, [0x2dc5] = 0x7d9e,
  [0x2dc6] = 0x7d9f, [0x2dc7] = 0x7da0, [0x2dc8] = 0x7da1, [0x2dc9] = 0x7da2,
  [0x2dca] = 0x7da3, [0x2dcb] = 0x7da4, [0x2dcc] = 0x7da5, [0x2dcd] = 0x7da7,
  [0x2dce] = 0x7da8, [0x2dcf] = 0x7da9, [0x2dd0] = 0x7daa, [0x2dd1] = 0x7dab,
  [0x2dd2] = 0x7dac, [0x2dd3] = 0x7dad, [0x2dd4] = 0x7daf, [0x2dd5] = 0x7db0,
  [0x2dd6] = 0x7db1, [0x2dd7] = 0x7db2, [0x2dd8] = 0x7db3, [0x2dd9] = 0x7db4,
  [0x2dda] = 0x7db5, [0x2ddb] = 0x7db6, [0x2ddc] = 0x7db7, [0x2ddd] = 0x7db8,
  [0x2dde] = 0x7db9, [0x2ddf] = 0x7dba, [0x2de0] = 0x7dbb, [0x2de1] = 0x7dbc,
  [0x2de2] = 0x7dbd, [0x2de3] = 0x7dbe, [0x2de4] = 0x7dbf, [0x2de5] = 0x7dc0,
  [0x2de6] = 0x7dc1, [0x2de7] = 0x7dc2, [0x2de8] = 0x7dc3, [0x2de9] = 0x7dc4,
  [0x2dea] = 0x7dc5, [0x2deb] = 0x7dc6, [0x2dec] = 0x7dc7, [0x2ded] = 0x7dc8,
  [0x2dee] = 0x7dc9, [0x2def] = 0x7dca, [0x2df0] = 0x7dcb, [0x2df1] = 0x7dcc,
  [0x2df2] = 0x7dcd, [0x2df3] = 0x7dce, [0x2df4] = 0x7dcf, [0x2df5] = 0x7dd0,
  [0x2df6] = 0x7dd1, [0x2df7] = 0x7dd2, [0x2df8] = 0x7dd3, [0x2df9] = 0x7dd4,
  [0x2dfa] = 0x7dd5, [0x2dfb] = 0x7dd6, [0x2dfc] = 0x7dd7, [0x2dfd] = 0x7dd8,
  [0x2dfe] = 0x7dd9, [0x2e00] = 0x7dda, [0x2e01] = 0x7ddb, [0x2e02] = 0x7ddc,
  [0x2e03] = 0x7ddd, [0x2e04] = 0x7dde, [0x2e05] = 0x7ddf, [0x2e06] = 0x7de0,
  [0x2e07] = 0x7de1, [0x2e08] = 0x7de2, [0x2e09] = 0x7de3, [0x2e0a] = 0x7de4,
  [0x2e0b] = 0x7de5, [0x2e0c] = 0x7de6, [0x2e0d] = 0x7de7, [0x2e0e] = 0x7de8,
  [0x2e0f] = 0x7de9, [0x2e10] = 0x7dea, [0x2e11] = 0x7deb, [0x2e12] = 0x7dec,
  [0x2e13] = 0x7ded, [0x2e14] = 0x7dee, [0x2e15] = 0x7def, [0x2e16] = 0x7df0,
  [0x2e17] = 0x7df1, [0x2e18] = 0x7df2, [0x2e19] = 0x7df3, [0x2e1a] = 0x7df4,
  [0x2e1b] = 0x7df5, [0x2e1c] = 0x7df6, [0x2e1d] = 0x7df7, [0x2e1e] = 0x7df8,
  [0x2e1f] = 0x7df9, [0x2e20] = 0x7dfa, [0x2e21] = 0x5c3d, [0x2e22] = 0x52b2,
  [0x2e23] = 0x8346, [0x2e24] = 0x5162, [0x2e25] = 0x830e, [0x2e26] = 0x775b,
  [0x2e27] = 0x6676, [0x2e28] = 0x9cb8, [0x2e29] = 0x4eac, [0x2e2a] = 0x60ca,
  [0x2e2b] = 0x7cbe, [0x2e2c] = 0x7cb3, [0x2e2d] = 0x7ecf, [0x2e2e] = 0x4e95,
  [0x2e2f] = 0x8b66, [0x2e30] = 0x666f, [0x2e31] = 0x9888, [0x2e32] = 0x9759,
  [0x2e33] = 0x5883, [0x2e34] = 0x656c, [0x2e35] = 0x955c, [0x2e36] = 0x5f84,
  [0x2e37] = 0x75c9, [0x2e38] = 0x9756, [0x2e39] = 0x7adf, [0x2e3a] = 0x7ade,
  [0x2e3b] = 0x51c0, [0x2e3c] = 0x70af, [0x2e3d] = 0x7a98, [0x2e3e] = 0x63ea,
  [0x2e3f] = 0x7a76, [0x2e40] = 0x7ea0, [0x2e41] = 0x7396, [0x2e42] = 0x97ed,
  [0x2e43] = 0x4e45, [0x2e44] = 0x7078, [0x2e45] = 0x4e5d, [0x2e46] = 0x9152,
  [0x2e47] = 0x53a9, [0x2e48] = 0x6551, [0x2e49] = 0x65e7, [0x2e4a] = 0x81fc,
  [0x2e4b] = 0x8205, [0x2e4c] = 0x548e, [0x2e4d] = 0x5c31, [0x2e4e] = 0x759a,
  [0x2e4f] = 0x97a0, [0x2e50] = 0x62d8, [0x2e51] = 0x72d9, [0x2e52] = 0x75bd,
  [0x2e53] = 0x5c45, [0x2e54] = 0x9a79, [0x2e55] = 0x83ca, [0x2e56] = 0x5c40,
  [0x2e57] = 0x5480, [0x2e58] = 0x77e9, [0x2e59] = 0x4e3e, [0x2e5a] = 0x6cae,
  [0x2e5b] = 0x805a, [0x2e5c] = 0x62d2, [0x2e5d] = 0x636e, [0x2e5e] = 0x5de8,
  [0x2e5f] = 0x5177, [0x2e60] = 0x8ddd, [0x2e61] = 0x8e1e, [0x2e62] = 0x952f,
  [0x2e63] = 0x4ff1, [0x2e64] = 0x53e5, [0x2e65] = 0x60e7, [0x2e66] = 0x70ac,
  [0x2e67] = 0x5267, [0x2e68] = 0x6350, [0x2e69] = 0x9e43, [0x2e6a] = 0x5a1f,
  [0x2e6b] = 0x5026, [0x2e6c] = 0x7737, [0x2e6d] = 0x5377, [0x2e6e] = 0x7ee2,
  [0x2e6f] = 0x6485, [0x2e70] = 0x652b, [0x2e71] = 0x6289, [0x2e72] = 0x6398,
  [0x2e73] = 0x5014, [0x2e74] = 0x7235, [0x2e75] = 0x89c9, [0x2e76] = 0x51b3,
  [0x2e77] = 0x8bc0, [0x2e78] = 0x7edd, [0x2e79] = 0x5747, [0x2e7a] = 0x83cc,
  [0x2e7b] = 0x94a7, [0x2e7c] = 0x519b, [0x2e7d] = 0x541b, [0x2e7e] = 0x5cfb,
  [0x2e80] = 0x7dfb, [0x2e81] = 0x7dfc, [0x2e82] = 0x7dfd, [0x2e83] = 0x7dfe,
  [0x2e84] = 0x7dff, [0x2e85] = 0x7e00, [0x2e86] = 0x7e01, [0x2e87] = 0x7e02,
  [0x2e88] = 0x7e03, [0x2e89] = 0x7e04, [0x2e8a] = 0x7e05, [0x2e8b] = 0x7e06,
  [0x2e8c] = 0x7e07, [0x2e8d] = 0x7e08, [0x2e8e] = 0x7e09, [0x2e8f] = 0x7e0a,
  [0x2e90] = 0x7e0b, [0x2e91] = 0x7e0c, [0x2e92] = 0x7e0d, [0x2e93] = 0x7e0e,
  [0x2e94] = 0x7e0f, [0x2e95] = 0x7e10, [0x2e96] = 0x7e11, [0x2e97] = 0x7e12,
  [0x2e98] = 0x7e13, [0x2e99] = 0x7e14, [0x2e9a] = 0x7e15, [0x2e9b] = 0x7e16,
  [0x2e9c] = 0x7e17, [0x2e9d] = 0x7e18, [0x2e9e] = 0x7e19, [0x2e9f] = 0x7e1a,
  [0x2ea0] = 0x7e1b, [0x2ea1] = 0x7e1c, [0x2ea2] = 0x7e1d, [0x2ea3] = 0x7e1e,
  [0x2ea4] = 0x7e1f, [0x2ea5] = 0x7e20, [0x2ea6] = 0x7e21, [0x2ea7] = 0x7e22,
  [0x2ea8] = 0x7e23, [0x2ea9] = 0x7e24, [0x2eaa] = 0x7e25, [0x2eab] = 0x7e26,
  [0x2eac] = 0x7e27, [0x2ead] = 0x7e28, [0x2eae] = 0x7e29, [0x2eaf] = 0x7e2a,
  [0x2eb0] = 0x7e2b, [0x2eb1] = 0x7e2c, [0x2eb2] = 0x7e2d, [0x2eb3] = 0x7e2e,
  [0x2eb4] = 0x7e2f, [0x2eb5] = 0x7e30, [0x2eb6] = 0x7e31, [0x2eb7] = 0x7e32,
  [0x2eb8] = 0x7e33, [0x2eb9] = 0x7e34, [0x2eba] = 0x7e35, [0x2ebb] = 0x7e36,
  [0x2ebc] = 0x7e37, [0x2ebd] = 0x7e38, [0x2ebe] = 0x7e39, [0x2ec0] = 0x7e3a,
  [0x2ec1] = 0x7e3c, [0x2ec2] = 0x7e3d, [0x2ec3] = 0x7e3e, [0x2ec4] = 0x7e3f,
  [0x2ec5] = 0x7e40, [0x2ec6] = 0x7e42, [0x2ec7] = 0x7e43, [0x2ec8] = 0x7e44,
  [0x2ec9] = 0x7e45, [0x2eca] = 0x7e46, [0x2ecb] = 0x7e48, [0x2ecc] = 0x7e49,
  [0x2ecd] = 0x7e4a, [0x2ece] = 0x7e4b, [0x2ecf] = 0x7e4c, [0x2ed0] = 0x7e4d,
  [0x2ed1] = 0x7e4e, [0x2ed2] = 0x7e4f, [0x2ed3] = 0x7e50, [0x2ed4] = 0x7e51,
  [0x2ed5] = 0x7e52, [0x2ed6] = 0x7e53, [0x2ed7] = 0x7e54, [0x2ed8] = 0x7e55,
  [0x2ed9] = 0x7e56, [0x2eda] = 0x7e57, [0x2edb] = 0x7e58, [0x2edc] = 0x7e59,
  [0x2edd] = 0x7e5a, [0x2ede] = 0x7e5b, [0x2edf] = 0x7e5c, [0x2ee0] = 0x7e5d,
  [0x2ee1] = 0x4fca, [0x2ee2] = 0x7ae3, [0x2ee3] = 0x6d5a, [0x2ee4] = 0x90e1,
  [0x2ee5] = 0x9a8f, [0x2ee6] = 0x5580, [0x2ee7] = 0x5496, [0x2ee8] = 0x5361,
  [0x2ee9] = 0x54af, [0x2eea] = 0x5f00, [0x2eeb] = 0x63e9, [0x2eec] = 0x6977,
  [0x2eed] = 0x51ef, [0x2eee] = 0x6168, [0x2eef] = 0x520a, [0x2ef0] = 0x582a,
  [0x2ef1] = 0x52d8, [0x2ef2] = 0x574e, [0x2ef3] = 0x780d, [0x2ef4] = 0x770b,
  [0x2ef5] = 0x5eb7, [0x2ef6] = 0x6177, [0x2ef7] = 0x7ce0, [0x2ef8] = 0x625b,
  [0x2ef9] = 0x6297, [0x2efa] = 0x4ea2, [0x2efb] = 0x7095, [0x2efc] = 0x8003,
  [0x2efd] = 0x62f7, [0x2efe] = 0x70e4, [0x2eff] = 0x9760, [0x2f00] = 0x5777,
  [0x2f01] = 0x82db, [0x2f02] = 0x67ef, [0x2f03] = 0x68f5, [0x2f04] = 0x78d5,
  [0x2f05] = 0x9897, [0x2f06] = 0x79d1, [0x2f07] = 0x58f3, [0x2f08] = 0x54b3,
  [0x2f09] = 0x53ef, [0x2f0a] = 0x6e34, [0x2f0b] = 0x514b, [0x2f0c] = 0x523b,
  [0x2f0d] = 0x5ba2, [0x2f0e] = 0x8bfe, [0x2f0f] = 0x80af, [0x2f10] = 0x5543,
  [0x2f11] = 0x57a6, [0x2f12] = 0x6073, [0x2f13] = 0x5751, [0x2f14] = 0x542d,
  [0x2f15] = 0x7a7a, [0x2f16] = 0x6050, [0x2f17] = 0x5b54, [0x2f18] = 0x63a7,
  [0x2f19] = 0x62a0, [0x2f1a] = 0x53e3, [0x2f1b] = 0x6263, [0x2f1c] = 0x5bc7,
  [0x2f1d] = 0x67af, [0x2f1e] = 0x54ed, [0x2f1f] = 0x7a9f, [0x2f20] = 0x82e6,
  [0x2f21] = 0x9177, [0x2f22] = 0x5e93, [0x2f23] = 0x88e4, [0x2f24] = 0x5938,
  [0x2f25] = 0x57ae, [0x2f26] = 0x630e, [0x2f27] = 0x8de8, [0x2f28] = 0x80ef,
  [0x2f29] = 0x5757, [0x2f2a] = 0x7b77, [0x2f2b] = 0x4fa9, [0x2f2c] = 0x5feb,
  [0x2f2d] = 0x5bbd, [0x2f2e] = 0x6b3e, [0x2f2f] = 0x5321, [0x2f30] = 0x7b50,
  [0x2f31] = 0x72c2, [0x2f32] = 0x6846, [0x2f33] = 0x77ff, [0x2f34] = 0x7736,
  [0x2f35] = 0x65f7, [0x2f36] = 0x51b5, [0x2f37] = 0x4e8f, [0x2f38] = 0x76d4,
  [0x2f39] = 0x5cbf, [0x2f3a] = 0x7aa5, [0x2f3b] = 0x8475, [0x2f3c] = 0x594e,
  [0x2f3d] = 0x9b41, [0x2f3e] = 0x5080, [0x2f40] = 0x7e5e, [0x2f41] = 0x7e5f,
  [0x2f42] = 0x7e60, [0x2f43] = 0x7e61, [0x2f44] = 0x7e62, [0x2f45] = 0x7e63,
  [0x2f46] = 0x7e64, [0x2f47] = 0x7e65, [0x2f48] = 0x7e66, [0x2f49] = 0x7e67,
  [0x2f4a] = 0x7e68, [0x2f4b] = 0x7e69, [0x2f4c] = 0x7e6a, [0x2f4d] = 0x7e6b,
  [0x2f4e] = 0x7e6c, [0x2f4f] = 0x7e6d, [0x2f50] = 0x7e6e, [0x2f51] = 0x7e6f,
  [0x2f52] = 0x7e70, [0x2f53] = 0x7e71, [0x2f54] = 0x7e72, [0x2f55] = 0x7e73,
  [0x2f56] = 0x7e74, [0x2f57] = 0x7e75, [0x2f58] = 0x7e76, [0x2f59] = 0x7e77,
  [0x2f5a] = 0x7e78, [0x2f5b] = 0x7e79, [0x2f5c] = 0x7e7a, [0x2f5d] = 0x7e7b,
  [0x2f5e] = 0x7e7c, [0x2f5f] = 0x7e7d, [0x2f60] = 0x7e7e, [0x2f61] = 0x7e7f,
  [0x2f62] = 0x7e80, [0x2f63] = 0x7e81, [0x2f64] = 0x7e83, [0x2f65] = 0x7e84,
  [0x2f66] = 0x7e85, [0x2f67] = 0x7e86, [0x2f68] = 0x7e87, [0x2f69] = 0x7e88,
  [0x2f6a] = 0x7e89, [0x2f6b] = 0x7e8a, [0x2f6c] = 0x7e8b, [0x2f6d] = 0x7e8c,
  [0x2f6e] = 0x7e8d, [0x2f6f] = 0x7e8e, [0x2f70] = 0x7e8f, [0x2f71] = 0x7e90,
  [0x2f72] = 0x7e91, [0x2f73] = 0x7e92, [0x2f74] = 0x7e93, [0x2f75] = 0x7e94,
  [0x2f76] = 0x7e95, [0x2f77] = 0x7e96, [0x2f78] = 0x7e97, [0x2f79] = 0x7e98,
  [0x2f7a] = 0x7e99, [0x2f7b] = 0x7e9a, [0x2f7c] = 0x7e9c, [0x2f7d] = 0x7e9d,
  [0x2f7e] = 0x7e9e, [0x2f80] = 0x7eae, [0x2f81] = 0x7eb4, [0x2f82] = 0x7ebb,
  [0x2f83] = 0x7ebc, [0x2f84] = 0x7ed6, [0x2f85] = 0x7ee4, [0x2f86] = 0x7eec,
  [0x2f87] = 0x7ef9, [0x2f88] = 0x7f0a, [0x2f89] = 0x7f10, [0x2f8a] = 0x7f1e,
  [0x2f8b] = 0x7f37, [0x2f8c] = 0x7f39, [0x2f8d] = 0x7f3b, [0x2f8e] = 0x7f3c,
  [0x2f8f] = 0x7f3d, [0x2f90] = 0x7f3e, [0x2f91] = 0x7f3f, [0x2f92] = 0x7f40,
  [0x2f93] = 0x7f41, [0x2f94] = 0x7f43, [0x2f95] = 0x7f46, [0x2f96] = 0x7f47,
  [0x2f97] = 0x7f48, [0x2f98] = 0x7f49, [0x2f99] = 0x7f4a, [0x2f9a] = 0x7f4b,
  [0x2f9b] = 0x7f4c, [0x2f9c] = 0x7f4d, [0x2f9d] = 0x7f4e, [0x2f9e] = 0x7f4f,
  [0x2f9f] = 0x7f52, [0x2fa0] = 0x7f53, [0x2fa1] = 0x9988, [0x2fa2] = 0x6127,
  [0x2fa3] = 0x6e83, [0x2fa4] = 0x5764, [0x2fa5] = 0x6606, [0x2fa6] = 0x6346,
  [0x2fa7] = 0x56f0, [0x2fa8] = 0x62ec, [0x2fa9] = 0x6269, [0x2faa] = 0x5ed3,
  [0x2fab] = 0x9614, [0x2fac] = 0x5783, [0x2fad] = 0x62c9, [0x2fae] = 0x5587,
  [0x2faf] = 0x8721, [0x2fb0] = 0x814a, [0x2fb1] = 0x8fa3, [0x2fb2] = 0x5566,
  [0x2fb3] = 0x83b1, [0x2fb4] = 0x6765, [0x2fb5] = 0x8d56, [0x2fb6] = 0x84dd,
  [0x2fb7] = 0x5a6a, [0x2fb8] = 0x680f, [0x2fb9] = 0x62e6, [0x2fba] = 0x7bee,
  [0x2fbb] = 0x9611, [0x2fbc] = 0x5170, [0x2fbd] = 0x6f9c, [0x2fbe] = 0x8c30,
  [0x2fbf] = 0x63fd, [0x2fc0] = 0x89c8, [0x2fc1] = 0x61d2, [0x2fc2] = 0x7f06,
  [0x2fc3] = 0x70c2, [0x2fc4] = 0x6ee5, [0x2fc5] = 0x7405, [0x2fc6] = 0x6994,
  [0x2fc7] = 0x72fc, [0x2fc8] = 0x5eca, [0x2fc9] = 0x90ce, [0x2fca] = 0x6717,
  [0x2fcb] = 0x6d6a, [0x2fcc] = 0x635e, [0x2fcd] = 0x52b3, [0x2fce] = 0x7262,
  [0x2fcf] = 0x8001, [0x2fd0] = 0x4f6c, [0x2fd1] = 0x59e5, [0x2fd2] = 0x916a,
  [0x2fd3] = 0x70d9, [0x2fd4] = 0x6d9d, [0x2fd5] = 0x52d2, [0x2fd6] = 0x4e50,
  [0x2fd7] = 0x96f7, [0x2fd8] = 0x956d, [0x2fd9] = 0x857e, [0x2fda] = 0x78ca,
  [0x2fdb] = 0x7d2f, [0x2fdc] = 0x5121, [0x2fdd] = 0x5792, [0x2fde] = 0x64c2,
  [0x2fdf] = 0x808b, [0x2fe0] = 0x7c7b, [0x2fe1] = 0x6cea, [0x2fe2] = 0x68f1,
  [0x2fe3] = 0x695e, [0x2fe4] = 0x51b7, [0x2fe5] = 0x5398, [0x2fe6] = 0x68a8,
  [0x2fe7] = 0x7281, [0x2fe8] = 0x9ece, [0x2fe9] = 0x7bf1, [0x2fea] = 0x72f8,
  [0x2feb] = 0x79bb, [0x2fec] = 0x6f13, [0x2fed] = 0x7406, [0x2fee] = 0x674e,
  [0x2fef] = 0x91cc, [0x2ff0] = 0x9ca4, [0x2ff1] = 0x793c, [0x2ff2] = 0x8389,
  [0x2ff3] = 0x8354, [0x2ff4] = 0x540f, [0x2ff5] = 0x6817, [0x2ff6] = 0x4e3d,
  [0x2ff7] = 0x5389, [0x2ff8] = 0x52b1, [0x2ff9] = 0x783e, [0x2ffa] = 0x5386,
  [0x2ffb] = 0x5229, [0x2ffc] = 0x5088, [0x2ffd] = 0x4f8b, [0x2ffe] = 0x4fd0,
  [0x3000] = 0x7f56, [0x3001] = 0x7f59, [0x3002] = 0x7f5b, [0x3003] = 0x7f5c,
  [0x3004] = 0x7f5d, [0x3005] = 0x7f5e, [0x3006] = 0x7f60, [0x3007] = 0x7f63,
  [0x3008] = 0x7f64, [0x3009] = 0x7f65, [0x300a] = 0x7f66, [0x300b] = 0x7f67,
  [0x300c] = 0x7f6b, [0x300d] = 0x7f6c, [0x300e] = 0x7f6d, [0x300f] = 0x7f6f,
  [0x3010] = 0x7f70, [0x3011] = 0x7f73, [0x3012] = 0x7f75, [0x3013] = 0x7f76,
  [0x3014] = 0x7f77, [0x3015] = 0x7f78, [0x3016] = 0x7f7a, [0x3017] = 0x7f7b,
  [0x3018] = 0x7f7c, [0x3019] = 0x7f7d, [0x301a] = 0x7f7f, [0x301b] = 0x7f80,
  [0x301c] = 0x7f82, [0x301d] = 0x7f83, [0x301e] = 0x7f84, [0x301f] = 0x7f85,
  [0x3020] = 0x7f86, [0x3021] = 0x7f87, [0x3022] = 0x7f88, [0x3023] = 0x7f89,
  [0x3024] = 0x7f8b, [0x3025] = 0x7f8d, [0x3026] = 0x7f8f, [0x3027] = 0x7f90,
  [0x3028] = 0x7f91, [0x3029] = 0x7f92, [0x302a] = 0x7f93, [0x302b] = 0x7f95,
  [0x302c] = 0x7f96, [0x302d] = 0x7f97, [0x302e] = 0x7f98, [0x302f] = 0x7f99,
  [0x3030] = 0x7f9b, [0x3031] = 0x7f9c, [0x3032] = 0x7fa0, [0x3033] = 0x7fa2,
  [0x3034] = 0x7fa3, [0x3035] = 0x7fa5, [0x3036] = 0x7fa6, [0x3037] = 0x7fa8,
  [0x3038] = 0x7fa9, [0x3039] = 0x7faa, [0x303a] = 0x7fab, [0x303b] = 0x7fac,
  [0x303c] = 0x7fad, [0x303d] = 0x7fae, [0x303e] = 0x7fb1, [0x3040] = 0x7fb3,
  [0x3041] = 0x7fb4, [0x3042] = 0x7fb5, [0x3043] = 0x7fb6, [0x3044] = 0x7fb7,
  [0x3045] = 0x7fba, [0x3046] = 0x7fbb, [0x3047] = 0x7fbe, [0x3048] = 0x7fc0,
  [0x3049] = 0x7fc2, [0x304a] = 0x7fc3, [0x304b] = 0x7fc4, [0x304c] = 0x7fc6,
  [0x304d] = 0x7fc7, [0x304e] = 0x7fc8, [0x304f] = 0x7fc9, [0x3050] = 0x7fcb,
  [0x3051] = 0x7fcd, [0x3052] = 0x7fcf, [0x3053] = 0x7fd0, [0x3054] = 0x7fd1,
  [0x3055] = 0x7fd2, [0x3056] = 0x7fd3, [0x3057] = 0x7fd6, [0x3058] = 0x7fd7,
  [0x3059] = 0x7fd9, [0x305a] = 0x7fda, [0x305b] = 0x7fdb, [0x305c] = 0x7fdc,
  [0x305d] = 0x7fdd, [0x305e] = 0x7fde, [0x305f] = 0x7fe2, [0x3060] = 0x7fe3,
  [0x3061] = 0x75e2, [0x3062] = 0x7acb, [0x3063] = 0x7c92, [0x3064] = 0x6ca5,
  [0x3065] = 0x96b6, [0x3066] = 0x529b, [0x3067] = 0x7483, [0x3068] = 0x54e9,
  [0x3069] = 0x4fe9, [0x306a] = 0x8054, [0x306b] = 0x83b2, [0x306c] = 0x8fde,
  [0x306d] = 0x9570, [0x306e] = 0x5ec9, [0x306f] = 0x601c, [0x3070] = 0x6d9f,
  [0x3071] = 0x5e18, [0x3072] = 0x655b, [0x3073] = 0x8138, [0x3074] = 0x94fe,
  [0x3075] = 0x604b, [0x3076] = 0x70bc, [0x3077] = 0x7ec3, [0x3078] = 0x7cae,
  [0x3079] = 0x51c9, [0x307a] = 0x6881, [0x307b] = 0x7cb1, [0x307c] = 0x826f,
  [0x307d] = 0x4e24, [0x307e] = 0x8f86, [0x307f] = 0x91cf, [0x3080] = 0x667e,
  [0x3081] = 0x4eae, [0x3082] = 0x8c05, [0x3083] = 0x64a9, [0x3084] = 0x804a,
  [0x3085] = 0x50da, [0x3086] = 0x7597, [0x3087] = 0x71ce, [0x3088] = 0x5be5,
  [0x3089] = 0x8fbd, [0x308a] = 0x6f66, [0x308b] = 0x4e86, [0x308c] = 0x6482,
  [0x308d] = 0x9563, [0x308e] = 0x5ed6, [0x308f] = 0x6599, [0x3090] = 0x5217,
  [0x3091] = 0x88c2, [0x3092] = 0x70c8, [0x3093] = 0x52a3, [0x3094] = 0x730e,
  [0x3095] = 0x7433, [0x3096] = 0x6797, [0x3097] = 0x78f7, [0x3098] = 0x9716,
  [0x3099] = 0x4e34, [0x309a] = 0x90bb, [0x309b] = 0x9cde, [0x309c] = 0x6dcb,
  [0x309d] = 0x51db, [0x309e] = 0x8d41, [0x309f] = 0x541d, [0x30a0] = 0x62ce,
  [0x30a1] = 0x73b2, [0x30a2] = 0x83f1, [0x30a3] = 0x96f6, [0x30a4] = 0x9f84,
  [0x30a5] = 0x94c3, [0x30a6] = 0x4f36, [0x30a7] = 0x7f9a, [0x30a8] = 0x51cc,
  [0x30a9] = 0x7075, [0x30aa] = 0x9675, [0x30ab] = 0x5cad, [0x30ac] = 0x9886,
  [0x30ad] = 0x53e6, [0x30ae] = 0x4ee4, [0x30af] = 0x6e9c, [0x30b0] = 0x7409,
  [0x30b1] = 0x69b4, [0x30b2] = 0x786b, [0x30b3] = 0x998f, [0x30b4] = 0x7559,
  [0x30b5] = 0x5218, [0x30b6] = 0x7624, [0x30b7] = 0x6d41, [0x30b8] = 0x67f3,
  [0x30b9] = 0x516d, [0x30ba] = 0x9f99, [0x30bb] = 0x804b, [0x30bc] = 0x5499,
  [0x30bd] = 0x7b3c, [0x30be] = 0x7abf, [0x30c0] = 0x7fe4, [0x30c1] = 0x7fe7,
  [0x30c2] = 0x7fe8, [0x30c3] = 0x7fea, [0x30c4] = 0x7feb, [0x30c5] = 0x7fec,
  [0x30c6] = 0x7fed, [0x30c7] = 0x7fef, [0x30c8] = 0x7ff2, [0x30c9] = 0x7ff4,
  [0x30ca] = 0x7ff5, [0x30cb] = 0x7ff6, [0x30cc] = 0x7ff7, [0x30cd] = 0x7ff8,
  [0x30ce] = 0x7ff9, [0x30cf] = 0x7ffa, [0x30d0] = 0x7ffd, [0x30d1] = 0x7ffe,
  [0x30d2] = 0x7fff, [0x30d3] = 0x8002, [0x30d4] = 0x8007, [0x30d5] = 0x8008,
  [0x30d6] = 0x8009, [0x30d7] = 0x800a, [0x30d8] = 0x800e, [0x30d9] = 0x800f,
  [0x30da] = 0x8011, [0x30db] = 0x8013, [0x30dc] = 0x801a, [0x30dd] = 0x801b,
  [0x30de] = 0x801d, [0x30df] = 0x801e, [0x30e0] = 0x801f, [0x30e1] = 0x8021,
  [0x30e2] = 0x8023, [0x30e3] = 0x8024, [0x30e4] = 0x802b, [0x30e5] = 0x802c,
  [0x30e6] = 0x802d, [0x30e7] = 0x802e, [0x30e8] = 0x802f, [0x30e9] = 0x8030,
  [0x30ea] = 0x8032, [0x30eb] = 0x8034, [0x30ec] = 0x8039, [0x30ed] = 0x803a,
  [0x30ee] = 0x803c, [0x30ef] = 0x803e, [0x30f0] = 0x8040, [0x30f1] = 0x8041,
  [0x30f2] = 0x8044, [0x30f3] = 0x8045, [0x30f4] = 0x8047, [0x30f5] = 0x8048,
  [0x30f6] = 0x8049, [0x30f7] = 0x804e, [0x30f8] = 0x804f, [0x30f9] = 0x8050,
  [0x30fa] = 0x8051, [0x30fb] = 0x8053, [0x30fc] = 0x8055, [0x30fd] = 0x8056,
  [0x30fe] = 0x8057, [0x3100] = 0x8059, [0x3101] = 0x805b, [0x3102] = 0x805c,
  [0x3103] = 0x805d, [0x3104] = 0x805e, [0x3105] = 0x805f, [0x3106] = 0x8060,
  [0x3107] = 0x8061, [0x3108] = 0x8062, [0x3109] = 0x8063, [0x310a] = 0x8064,
  [0x310b] = 0x8065, [0x310c] = 0x8066, [0x310d] = 0x8067, [0x310e] = 0x8068,
  [0x310f] = 0x806b, [0x3110] = 0x806c, [0x3111] = 0x806d, [0x3112] = 0x806e,
  [0x3113] = 0x806f, [0x3114] = 0x8070, [0x3115] = 0x8072, [0x3116] = 0x8073,
  [0x3117] = 0x8074, [0x3118] = 0x8075, [0x3119] = 0x8076, [0x311a] = 0x8077,
  [0x311b] = 0x8078, [0x311c] = 0x8079, [0x311d] = 0x807a, [0x311e] = 0x807b,
  [0x311f] = 0x807c, [0x3120] = 0x807d, [0x3121] = 0x9686, [0x3122] = 0x5784,
  [0x3123] = 0x62e2, [0x3124] = 0x9647, [0x3125] = 0x697c, [0x3126] = 0x5a04,
  [0x3127] = 0x6402, [0x3128] = 0x7bd3, [0x3129] = 0x6f0f, [0x312a] = 0x964b,
  [0x312b] = 0x82a6, [0x312c] = 0x5362, [0x312d] = 0x9885, [0x312e] = 0x5e90,
  [0x312f] = 0x7089, [0x3130] = 0x63b3, [0x3131] = 0x5364, [0x3132] = 0x864f,
  [0x3133] = 0x9c81, [0x3134] = 0x9e93, [0x3135] = 0x788c, [0x3136] = 0x9732,
  [0x3137] = 0x8def, [0x3138] = 0x8d42, [0x3139] = 0x9e7f, [0x313a] = 0x6f5e,
  [0x313b] = 0x7984, [0x313c] = 0x5f55, [0x313d] = 0x9646, [0x313e] = 0x622e,
  [0x313f] = 0x9a74, [0x3140] = 0x5415, [0x3141] = 0x94dd, [0x3142] = 0x4fa3,
  [0x3143] = 0x65c5, [0x3144] = 0x5c65, [0x3145] = 0x5c61, [0x3146] = 0x7f15,
  [0x3147] = 0x8651, [0x3148] = 0x6c2f, [0x3149] = 0x5f8b, [0x314a] = 0x7387,
  [0x314b] = 0x6ee4, [0x314c] = 0x7eff, [0x314d] = 0x5ce6, [0x314e] = 0x631b,
  [0x314f] = 0x5b6a, [0x3150] = 0x6ee6, [0x3151] = 0x5375, [0x3152] = 0x4e71,
  [0x3153] = 0x63a0, [0x3154] = 0x7565, [0x3155] = 0x62a1, [0x3156] = 0x8f6e,
  [0x3157] = 0x4f26, [0x3158] = 0x4ed1, [0x3159] = 0x6ca6, [0x315a] = 0x7eb6,
  [0x315b] = 0x8bba, [0x315c] = 0x841d, [0x315d] = 0x87ba, [0x315e] = 0x7f57,
  [0x315f] = 0x903b, [0x3160] = 0x9523, [0x3161] = 0x7ba9, [0x3162] = 0x9aa1,
  [0x3163] = 0x88f8, [0x3164] = 0x843d, [0x3165] = 0x6d1b, [0x3166] = 0x9a86,
  [0x3167] = 0x7edc, [0x3168] = 0x5988, [0x3169] = 0x9ebb, [0x316a] = 0x739b,
  [0x316b] = 0x7801, [0x316c] = 0x8682, [0x316d] = 0x9a6c, [0x316e] = 0x9a82,
  [0x316f] = 0x561b, [0x3170] = 0x5417, [0x3171] = 0x57cb, [0x3172] = 0x4e70,
  [0x3173] = 0x9ea6, [0x3174] = 0x5356, [0x3175] = 0x8fc8, [0x3176] = 0x8109,
  [0x3177] = 0x7792, [0x3178] = 0x9992, [0x3179] = 0x86ee, [0x317a] = 0x6ee1,
  [0x317b] = 0x8513, [0x317c] = 0x66fc, [0x317d] = 0x6162, [0x317e] = 0x6f2b,
  [0x3180] = 0x807e, [0x3181] = 0x8081, [0x3182] = 0x8082, [0x3183] = 0x8085,
  [0x3184] = 0x8088, [0x3185] = 0x808a, [0x3186] = 0x808d, [0x3187] = 0x808e,
  [0x3188] = 0x808f, [0x3189] = 0x8090, [0x318a] = 0x8091, [0x318b] = 0x8092,
  [0x318c] = 0x8094, [0x318d] = 0x8095, [0x318e] = 0x8097, [0x318f] = 0x8099,
  [0x3190] = 0x809e, [0x3191] = 0x80a3, [0x3192] = 0x80a6, [0x3193] = 0x80a7,
  [0x3194] = 0x80a8, [0x3195] = 0x80ac, [0x3196] = 0x80b0, [0x3197] = 0x80b3,
  [0x3198] = 0x80b5, [0x3199] = 0x80b6, [0x319a] = 0x80b8, [0x319b] = 0x80b9,
  [0x319c] = 0x80bb, [0x319d] = 0x80c5, [0x319e] = 0x80c7, [0x319f] = 0x80c8,
  [0x31a0] = 0x80c9, [0x31a1] = 0x80ca, [0x31a2] = 0x80cb, [0x31a3] = 0x80cf,
  [0x31a4] = 0x80d0, [0x31a5] = 0x80d1, [0x31a6] = 0x80d2, [0x31a7] = 0x80d3,
  [0x31a8] = 0x80d4, [0x31a9] = 0x80d5, [0x31aa] = 0x80d8, [0x31ab] = 0x80df,
  [0x31ac] = 0x80e0, [0x31ad] = 0x80e2, [0x31ae] = 0x80e3, [0x31af] = 0x80e6,
  [0x31b0] = 0x80ee, [0x31b1] = 0x80f5, [0x31b2] = 0x80f7, [0x31b3] = 0x80f9,
  [0x31b4] = 0x80fb, [0x31b5] = 0x80fe, [0x31b6] = 0x80ff, [0x31b7] = 0x8100,
  [0x31b8] = 0x8101, [0x31b9] = 0x8103, [0x31ba] = 0x8104, [0x31bb] = 0x8105,
  [0x31bc] = 0x8107, [0x31bd] = 0x8108, [0x31be] = 0x810b, [0x31c0] = 0x810c,
  [0x31c1] = 0x8115, [0x31c2] = 0x8117, [0x31c3] = 0x8119, [0x31c4] = 0x811b,
  [0x31c5] = 0x811c, [0x31c6] = 0x811d, [0x31c7] = 0x811f, [0x31c8] = 0x8120,
  [0x31c9] = 0x8121, [0x31ca] = 0x8122, [0x31cb] = 0x8123, [0x31cc] = 0x8124,
  [0x31cd] = 0x8125, [0x31ce] = 0x8126, [0x31cf] = 0x8127, [0x31d0] = 0x8128,
  [0x31d1] = 0x8129, [0x31d2] = 0x812a, [0x31d3] = 0x812b, [0x31d4] = 0x812d,
  [0x31d5] = 0x812e, [0x31d6] = 0x8130, [0x31d7] = 0x8133, [0x31d8] = 0x8134,
  [0x31d9] = 0x8135, [0x31da] = 0x8137, [0x31db] = 0x8139, [0x31dc] = 0x813a,
  [0x31dd] = 0x813b, [0x31de] = 0x813c, [0x31df] = 0x813d, [0x31e0] = 0x813f,
  [0x31e1] = 0x8c29, [0x31e2] = 0x8292, [0x31e3] = 0x832b, [0x31e4] = 0x76f2,
  [0x31e5] = 0x6c13, [0x31e6] = 0x5fd9, [0x31e7] = 0x83bd, [0x31e8] = 0x732b,
  [0x31e9] = 0x8305, [0x31ea] = 0x951a, [0x31eb] = 0x6bdb, [0x31ec] = 0x77db,
  [0x31ed] = 0x94c6, [0x31ee] = 0x536f, [0x31ef] = 0x8302, [0x31f0] = 0x5192,
  [0x31f1] = 0x5e3d, [0x31f2] = 0x8c8c, [0x31f3] = 0x8d38, [0x31f4] = 0x4e48,
  [0x31f5] = 0x73ab, [0x31f6] = 0x679a, [0x31f7] = 0x6885, [0x31f8] = 0x9176,
  [0x31f9] = 0x9709, [0x31fa] = 0x7164, [0x31fb] = 0x6ca1, [0x31fc] = 0x7709,
  [0x31fd] = 0x5a92, [0x31fe] = 0x9541, [0x31ff] = 0x6bcf, [0x3200] = 0x7f8e,
  [0x3201] = 0x6627, [0x3202] = 0x5bd0, [0x3203] = 0x59b9, [0x3204] = 0x5a9a,
  [0x3205] = 0x95e8, [0x3206] = 0x95f7, [0x3207] = 0x4eec, [0x3208] = 0x840c,
  [0x3209] = 0x8499, [0x320a] = 0x6aac, [0x320b] = 0x76df, [0x320c] = 0x9530,
  [0x320d] = 0x731b, [0x320e] = 0x68a6, [0x320f] = 0x5b5f, [0x3210] = 0x772f,
  [0x3211] = 0x919a, [0x3212] = 0x9761, [0x3213] = 0x7cdc, [0x3214] = 0x8ff7,
  [0x3215] = 0x8c1c, [0x3216] = 0x5f25, [0x3217] = 0x7c73, [0x3218] = 0x79d8,
  [0x3219] = 0x89c5, [0x321a] = 0x6ccc, [0x321b] = 0x871c, [0x321c] = 0x5bc6,
  [0x321d] = 0x5e42, [0x321e] = 0x68c9, [0x321f] = 0x7720, [0x3220] = 0x7ef5,
  [0x3221] = 0x5195, [0x3222] = 0x514d, [0x3223] = 0x52c9, [0x3224] = 0x5a29,
  [0x3225] = 0x7f05, [0x3226] = 0x9762, [0x3227] = 0x82d7, [0x3228] = 0x63cf,
  [0x3229] = 0x7784, [0x322a] = 0x85d0, [0x322b] = 0x79d2, [0x322c] = 0x6e3a,
  [0x322d] = 0x5e99, [0x322e] = 0x5999, [0x322f] = 0x8511, [0x3230] = 0x706d,
  [0x3231] = 0x6c11, [0x3232] = 0x62bf, [0x3233] = 0x76bf, [0x3234] = 0x654f,
  [0x3235] = 0x60af, [0x3236] = 0x95fd, [0x3237] = 0x660e, [0x3238] = 0x879f,
  [0x3239] = 0x9e23, [0x323a] = 0x94ed, [0x323b] = 0x540d, [0x323c] = 0x547d,
  [0x323d] = 0x8c2c, [0x323e] = 0x6478, [0x3240] = 0x8140, [0x3241] = 0x8141,
  [0x3242] = 0x8142, [0x3243] = 0x8143, [0x3244] = 0x8144, [0x3245] = 0x8145,
  [0x3246] = 0x8147, [0x3247] = 0x8149, [0x3248] = 0x814d, [0x3249] = 0x814e,
  [0x324a] = 0x814f, [0x324b] = 0x8152, [0x324c] = 0x8156, [0x324d] = 0x8157,
  [0x324e] = 0x8158, [0x324f] = 0x815b, [0x3250] = 0x815c, [0x3251] = 0x815d,
  [0x3252] = 0x815e, [0x3253] = 0x815f, [0x3254] = 0x8161, [0x3255] = 0x8162,
  [0x3256] = 0x8163, [0x3257] = 0x8164, [0x3258] = 0x8166, [0x3259] = 0x8168,
  [0x325a] = 0x816a, [0x325b] = 0x816b, [0x325c] = 0x816c, [0x325d] = 0x816f,
  [0x325e] = 0x8172, [0x325f] = 0x8173, [0x3260] = 0x8175, [0x3261] = 0x8176,
  [0x3262] = 0x8177, [0x3263] = 0x8178, [0x3264] = 0x8181, [0x3265] = 0x8183,
  [0x3266] = 0x8184, [0x3267] = 0x8185, [0x3268] = 0x8186, [0x3269] = 0x8187,
  [0x326a] = 0x8189, [0x326b] = 0x818b, [0x326c] = 0x818c, [0x326d] = 0x818d,
  [0x326e] = 0x818e, [0x326f] = 0x8190, [0x3270] = 0x8192, [0x3271] = 0x8193,
  [0x3272] = 0x8194, [0x3273] = 0x8195, [0x3274] = 0x8196, [0x3275] = 0x8197,
  [0x3276] = 0x8199, [0x3277] = 0x819a, [0x3278] = 0x819e, [0x3279] = 0x819f,
  [0x327a] = 0x81a0, [0x327b] = 0x81a1, [0x327c] = 0x81a2, [0x327d] = 0x81a4,
  [0x327e] = 0x81a5, [0x3280] = 0x81a7, [0x3281] = 0x81a9, [0x3282] = 0x81ab,
  [0x3283] = 0x81ac, [0x3284] = 0x81ad, [0x3285] = 0x81ae, [0x3286] = 0x81af,
  [0x3287] = 0x81b0, [0x3288] = 0x81b1, [0x3289] = 0x81b2, [0x328a] = 0x81b4,
  [0x328b] = 0x81b5, [0x328c] = 0x81b6, [0x328d] = 0x81b7, [0x328e] = 0x81b8,
  [0x328f] = 0x81b9, [0x3290] = 0x81bc, [0x3291] = 0x81bd, [0x3292] = 0x81be,
  [0x3293] = 0x81bf, [0x3294] = 0x81c4, [0x3295] = 0x81c5, [0x3296] = 0x81c7,
  [0x3297] = 0x81c8, [0x3298] = 0x81c9, [0x3299] = 0x81cb, [0x329a] = 0x81cd,
  [0x329b] = 0x81ce, [0x329c] = 0x81cf, [0x329d] = 0x81d0, [0x329e] = 0x81d1,
  [0x329f] = 0x81d2, [0x32a0] = 0x81d3, [0x32a1] = 0x6479, [0x32a2] = 0x8611,
  [0x32a3] = 0x6a21, [0x32a4] = 0x819c, [0x32a5] = 0x78e8, [0x32a6] = 0x6469,
  [0x32a7] = 0x9b54, [0x32a8] = 0x62b9, [0x32a9] = 0x672b, [0x32aa] = 0x83ab,
  [0x32ab] = 0x58a8, [0x32ac] = 0x9ed8, [0x32ad] = 0x6cab, [0x32ae] = 0x6f20,
  [0x32af] = 0x5bde, [0x32b0] = 0x964c, [0x32b1] = 0x8c0b, [0x32b2] = 0x725f,
  [0x32b3] = 0x67d0, [0x32b4] = 0x62c7, [0x32b5] = 0x7261, [0x32b6] = 0x4ea9,
  [0x32b7] = 0x59c6, [0x32b8] = 0x6bcd, [0x32b9] = 0x5893, [0x32ba] = 0x66ae,
  [0x32bb] = 0x5e55, [0x32bc] = 0x52df, [0x32bd] = 0x6155, [0x32be] = 0x6728,
  [0x32bf] = 0x76ee, [0x32c0] = 0x7766, [0x32c1] = 0x7267, [0x32c2] = 0x7a46,
  [0x32c3] = 0x62ff, [0x32c4] = 0x54ea, [0x32c5] = 0x5450, [0x32c6] = 0x94a0,
  [0x32c7] = 0x90a3, [0x32c8] = 0x5a1c, [0x32c9] = 0x7eb3, [0x32ca] = 0x6c16,
  [0x32cb] = 0x4e43, [0x32cc] = 0x5976, [0x32cd] = 0x8010, [0x32ce] = 0x5948,
  [0x32cf] = 0x5357, [0x32d0] = 0x7537, [0x32d1] = 0x96be, [0x32d2] = 0x56ca,
  [0x32d3] = 0x6320, [0x32d4] = 0x8111, [0x32d5] = 0x607c, [0x32d6] = 0x95f9,
  [0x32d7] = 0x6dd6, [0x32d8] = 0x5462, [0x32d9] = 0x9981, [0x32da] = 0x5185,
  [0x32db] = 0x5ae9, [0x32dc] = 0x80fd, [0x32dd] = 0x59ae, [0x32de] = 0x9713,
  [0x32df] = 0x502a, [0x32e0] = 0x6ce5, [0x32e1] = 0x5c3c, [0x32e2] = 0x62df,
  [0x32e3] = 0x4f60, [0x32e4] = 0x533f, [0x32e5] = 0x817b, [0x32e6] = 0x9006,
  [0x32e7] = 0x6eba, [0x32e8] = 0x852b, [0x32e9] = 0x62c8, [0x32ea] = 0x5e74,
  [0x32eb] = 0x78be, [0x32ec] = 0x64b5, [0x32ed] = 0x637b, [0x32ee] = 0x5ff5,
  [0x32ef] = 0x5a18, [0x32f0] = 0x917f, [0x32f1] = 0x9e1f, [0x32f2] = 0x5c3f,
  [0x32f3] = 0x634f, [0x32f4] = 0x8042, [0x32f5] = 0x5b7d, [0x32f6] = 0x556e,
  [0x32f7] = 0x954a, [0x32f8] = 0x954d, [0x32f9] = 0x6d85, [0x32fa] = 0x60a8,
  [0x32fb] = 0x67e0, [0x32fc] = 0x72de, [0x32fd] = 0x51dd, [0x32fe] = 0x5b81,
  [0x3300] = 0x81d4, [0x3301] = 0x81d5, [0x3302] = 0x81d6, [0x3303] = 0x81d7,
  [0x3304] = 0x81d8, [0x3305] = 0x81d9, [0x3306] = 0x81da, [0x3307] = 0x81db,
  [0x3308] = 0x81dc, [0x3309] = 0x81dd, [0x330a] = 0x81de, [0x330b] = 0x81df,
  [0x330c] = 0x81e0, [0x330d] = 0x81e1, [0x330e] = 0x81e2, [0x330f] = 0x81e4,
  [0x3310] = 0x81e5, [0x3311] = 0x81e6, [0x3312] = 0x81e8, [0x3313] = 0x81e9,
  [0x3314] = 0x81eb, [0x3315] = 0x81ee, [0x3316] = 0x81ef, [0x3317] = 0x81f0,
  [0x3318] = 0x81f1, [0x3319] = 0x81f2, [0x331a] = 0x81f5, [0x331b] = 0x81f6,
  [0x331c] = 0x81f7, [0x331d] = 0x81f8, [0x331e] = 0x81f9, [0x331f] = 0x81fa,
  [0x3320] = 0x81fd, [0x3321] = 0x81ff, [0x3322] = 0x8203, [0x3323] = 0x8207,
  [0x3324] = 0x8208, [0x3325] = 0x8209, [0x3326] = 0x820a, [0x3327] = 0x820b,
  [0x3328] = 0x820e, [0x3329] = 0x820f, [0x332a] = 0x8211, [0x332b] = 0x8213,
  [0x332c] = 0x8215, [0x332d] = 0x8216, [0x332e] = 0x8217, [0x332f] = 0x8218,
  [0x3330] = 0x8219, [0x3331] = 0x821a, [0x3332] = 0x821d, [0x3333] = 0x8220,
  [0x3334] = 0x8224, [0x3335] = 0x8225, [0x3336] = 0x8226, [0x3337] = 0x8227,
  [0x3338] = 0x8229, [0x3339] = 0x822e, [0x333a] = 0x8232, [0x333b] = 0x823a,
  [0x333c] = 0x823c, [0x333d] = 0x823d, [0x333e] = 0x823f, [0x3340] = 0x8240,
  [0x3341] = 0x8241, [0x3342] = 0x8242, [0x3343] = 0x8243, [0x3344] = 0x8245,
  [0x3345] = 0x8246, [0x3346] = 0x8248, [0x3347] = 0x824a, [0x3348] = 0x824c,
  [0x3349] = 0x824d, [0x334a] = 0x824e, [0x334b] = 0x8250, [0x334c] = 0x8251,
  [0x334d] = 0x8252, [0x334e] = 0x8253, [0x334f] = 0x8254, [0x3350] = 0x8255,
  [0x3351] = 0x8256, [0x3352] = 0x8257, [0x3353] = 0x8259, [0x3354] = 0x825b,
  [0x3355] = 0x825c, [0x3356] = 0x825d, [0x3357] = 0x825e, [0x3358] = 0x8260,
  [0x3359] = 0x8261, [0x335a] = 0x8262, [0x335b] = 0x8263, [0x335c] = 0x8264,
  [0x335d] = 0x8265, [0x335e] = 0x8266, [0x335f] = 0x8267, [0x3360] = 0x8269,
  [0x3361] = 0x62e7, [0x3362] = 0x6cde, [0x3363] = 0x725b, [0x3364] = 0x626d,
  [0x3365] = 0x94ae, [0x3366] = 0x7ebd, [0x3367] = 0x8113, [0x3368] = 0x6d53,
  [0x3369] = 0x519c, [0x336a] = 0x5f04, [0x336b] = 0x5974, [0x336c] = 0x52aa,
  [0x336d] = 0x6012, [0x336e] = 0x5973, [0x336f] = 0x6696, [0x3370] = 0x8650,
  [0x3371] = 0x759f, [0x3372] = 0x632a, [0x3373] = 0x61e6, [0x3374] = 0x7cef,
  [0x3375] = 0x8bfa, [0x3376] = 0x54e6, [0x3377] = 0x6b27, [0x3378] = 0x9e25,
  [0x3379] = 0x6bb4, [0x337a] = 0x85d5, [0x337b] = 0x5455, [0x337c] = 0x5076,
  [0x337d] = 0x6ca4, [0x337e] = 0x556a, [0x337f] = 0x8db4, [0x3380] = 0x722c,
  [0x3381] = 0x5e15, [0x3382] = 0x6015, [0x3383] = 0x7436, [0x3384] = 0x62cd,
  [0x3385] = 0x6392, [0x3386] = 0x724c, [0x3387] = 0x5f98, [0x3388] = 0x6e43,
  [0x3389] = 0x6d3e, [0x338a] = 0x6500, [0x338b] = 0x6f58, [0x338c] = 0x76d8,
  [0x338d] = 0x78d0, [0x338e] = 0x76fc, [0x338f] = 0x7554, [0x3390] = 0x5224,
  [0x3391] = 0x53db, [0x3392] = 0x4e53, [0x3393] = 0x5e9e, [0x3394] = 0x65c1,
  [0x3395] = 0x802a, [0x3396] = 0x80d6, [0x3397] = 0x629b, [0x3398] = 0x5486,
  [0x3399] = 0x5228, [0x339a] = 0x70ae, [0x339b] = 0x888d, [0x339c] = 0x8dd1,
  [0x339d] = 0x6ce1, [0x339e] = 0x5478, [0x339f] = 0x80da, [0x33a0] = 0x57f9,
  [0x33a1] = 0x88f4, [0x33a2] = 0x8d54, [0x33a3] = 0x966a, [0x33a4] = 0x914d,
  [0x33a5] = 0x4f69, [0x33a6] = 0x6c9b, [0x33a7] = 0x55b7, [0x33a8] = 0x76c6,
  [0x33a9] = 0x7830, [0x33aa] = 0x62a8, [0x33ab] = 0x70f9, [0x33ac] = 0x6f8e,
  [0x33ad] = 0x5f6d, [0x33ae] = 0x84ec, [0x33af] = 0x68da, [0x33b0] = 0x787c,
  [0x33b1] = 0x7bf7, [0x33b2] = 0x81a8, [0x33b3] = 0x670b, [0x33b4] = 0x9e4f,
  [0x33b5] = 0x6367, [0x33b6] = 0x78b0, [0x33b7] = 0x576f, [0x33b8] = 0x7812,
  [0x33b9] = 0x9739, [0x33ba] = 0x6279, [0x33bb] = 0x62ab, [0x33bc] = 0x5288,
  [0x33bd] = 0x7435, [0x33be] = 0x6bd7, [0x33c0] = 0x826a, [0x33c1] = 0x826b,
  [0x33c2] = 0x826c, [0x33c3] = 0x826d, [0x33c4] = 0x8271, [0x33c5] = 0x8275,
  [0x33c6] = 0x8276, [0x33c7] = 0x8277, [0x33c8] = 0x8278, [0x33c9] = 0x827b,
  [0x33ca] = 0x827c, [0x33cb] = 0x8280, [0x33cc] = 0x8281, [0x33cd] = 0x8283,
  [0x33ce] = 0x8285, [0x33cf] = 0x8286, [0x33d0] = 0x8287, [0x33d1] = 0x8289,
  [0x33d2] = 0x828c, [0x33d3] = 0x8290, [0x33d4] = 0x8293, [0x33d5] = 0x8294,
  [0x33d6] = 0x8295, [0x33d7] = 0x8296, [0x33d8] = 0x829a, [0x33d9] = 0x829b,
  [0x33da] = 0x829e, [0x33db] = 0x82a0, [0x33dc] = 0x82a2, [0x33dd] = 0x82a3,
  [0x33de] = 0x82a7, [0x33df] = 0x82b2, [0x33e0] = 0x82b5, [0x33e1] = 0x82b6,
  [0x33e2] = 0x82ba, [0x33e3] = 0x82bb, [0x33e4] = 0x82bc, [0x33e5] = 0x82bf,
  [0x33e6] = 0x82c0, [0x33e7] = 0x82c2, [0x33e8] = 0x82c3, [0x33e9] = 0x82c5,
  [0x33ea] = 0x82c6, [0x33eb] = 0x82c9, [0x33ec] = 0x82d0, [0x33ed] = 0x82d6,
  [0x33ee] = 0x82d9, [0x33ef] = 0x82da, [0x33f0] = 0x82dd, [0x33f1] = 0x82e2,
  [0x33f2] = 0x82e7, [0x33f3] = 0x82e8, [0x33f4] = 0x82e9, [0x33f5] = 0x82ea,
  [0x33f6] = 0x82ec, [0x33f7] = 0x82ed, [0x33f8] = 0x82ee, [0x33f9] = 0x82f0,
  [0x33fa] = 0x82f2, [0x33fb] = 0x82f3, [0x33fc] = 0x82f5, [0x33fd] = 0x82f6,
  [0x33fe] = 0x82f8, [0x3400] = 0x82fa, [0x3401] = 0x82fc, [0x3402] = 0x82fd,
  [0x3403] = 0x82fe, [0x3404] = 0x82ff, [0x3405] = 0x8300, [0x3406] = 0x830a,
  [0x3407] = 0x830b, [0x3408] = 0x830d, [0x3409] = 0x8310, [0x340a] = 0x8312,
  [0x340b] = 0x8313, [0x340c] = 0x8316, [0x340d] = 0x8318, [0x340e] = 0x8319,
  [0x340f] = 0x831d, [0x3410] = 0x831e, [0x3411] = 0x831f, [0x3412] = 0x8320,
  [0x3413] = 0x8321, [0x3414] = 0x8322, [0x3415] = 0x8323, [0x3416] = 0x8324,
  [0x3417] = 0x8325, [0x3418] = 0x8326, [0x3419] = 0x8329, [0x341a] = 0x832a,
  [0x341b] = 0x832e, [0x341c] = 0x8330, [0x341d] = 0x8332, [0x341e] = 0x8337,
  [0x341f] = 0x833b, [0x3420] = 0x833d, [0x3421] = 0x5564, [0x3422] = 0x813e,
  [0x3423] = 0x75b2, [0x3424] = 0x76ae, [0x3425] = 0x5339, [0x3426] = 0x75de,
  [0x3427] = 0x50fb, [0x3428] = 0x5c41, [0x3429] = 0x8b6c, [0x342a] = 0x7bc7,
  [0x342b] = 0x504f, [0x342c] = 0x7247, [0x342d] = 0x9a97, [0x342e] = 0x98d8,
  [0x342f] = 0x6f02, [0x3430] = 0x74e2, [0x3431] = 0x7968, [0x3432] = 0x6487,
  [0x3433] = 0x77a5, [0x3434] = 0x62fc, [0x3435] = 0x9891, [0x3436] = 0x8d2b,
  [0x3437] = 0x54c1, [0x3438] = 0x8058, [0x3439] = 0x4e52, [0x343a] = 0x576a,
  [0x343b] = 0x82f9, [0x343c] = 0x840d, [0x343d] = 0x5e73, [0x343e] = 0x51ed,
  [0x343f] = 0x74f6, [0x3440] = 0x8bc4, [0x3441] = 0x5c4f, [0x3442] = 0x5761,
  [0x3443] = 0x6cfc, [0x3444] = 0x9887, [0x3445] = 0x5a46, [0x3446] = 0x7834,
  [0x3447] = 0x9b44, [0x3448] = 0x8feb, [0x3449] = 0x7c95, [0x344a] = 0x5256,
  [0x344b] = 0x6251, [0x344c] = 0x94fa, [0x344d] = 0x4ec6, [0x344e] = 0x8386,
  [0x344f] = 0x8461, [0x3450] = 0x83e9, [0x3451] = 0x84b2, [0x3452] = 0x57d4,
  [0x3453] = 0x6734, [0x3454] = 0x5703, [0x3455] = 0x666e, [0x3456] = 0x6d66,
  [0x3457] = 0x8c31, [0x3458] = 0x66dd, [0x3459] = 0x7011, [0x345a] = 0x671f,
  [0x345b] = 0x6b3a, [0x345c] = 0x6816, [0x345d] = 0x621a, [0x345e] = 0x59bb,
  [0x345f] = 0x4e03, [0x3460] = 0x51c4, [0x3461] = 0x6f06, [0x3462] = 0x67d2,
  [0x3463] = 0x6c8f, [0x3464] = 0x5176, [0x3465] = 0x68cb, [0x3466] = 0x5947,
  [0x3467] = 0x6b67, [0x3468] = 0x7566, [0x3469] = 0x5d0e, [0x346a] = 0x8110,
  [0x346b] = 0x9f50, [0x346c] = 0x65d7, [0x346d] = 0x7948, [0x346e] = 0x7941,
  [0x346f] = 0x9a91, [0x3470] = 0x8d77, [0x3471] = 0x5c82, [0x3472] = 0x4e5e,
  [0x3473] = 0x4f01, [0x3474] = 0x542f, [0x3475] = 0x5951, [0x3476] = 0x780c,
  [0x3477] = 0x5668, [0x3478] = 0x6c14, [0x3479] = 0x8fc4, [0x347a] = 0x5f03,
  [0x347b] = 0x6c7d, [0x347c] = 0x6ce3, [0x347d] = 0x8bab, [0x347e] = 0x6390,
  [0x3480] = 0x833e, [0x3481] = 0x833f, [0x3482] = 0x8341, [0x3483] = 0x8342,
  [0x3484] = 0x8344, [0x3485] = 0x8345, [0x3486] = 0x8348, [0x3487] = 0x834a,
  [0x3488] = 0x834b, [0x3489] = 0x834c, [0x348a] = 0x834d, [0x348b] = 0x834e,
  [0x348c] = 0x8353, [0x348d] = 0x8355, [0x348e] = 0x8356, [0x348f] = 0x8357,
  [0x3490] = 0x8358, [0x3491] = 0x8359, [0x3492] = 0x835d, [0x3493] = 0x8362,
  [0x3494] = 0x8370, [0x3495] = 0x8371, [0x3496] = 0x8372, [0x3497] = 0x8373,
  [0x3498] = 0x8374, [0x3499] = 0x8375, [0x349a] = 0x8376, [0x349b] = 0x8379,
  [0x349c] = 0x837a, [0x349d] = 0x837e, [0x349e] = 0x837f, [0x349f] = 0x8380,
  [0x34a0] = 0x8381, [0x34a1] = 0x8382, [0x34a2] = 0x8383, [0x34a3] = 0x8384,
  [0x34a4] = 0x8387, [0x34a5] = 0x8388, [0x34a6] = 0x838a, [0x34a7] = 0x838b,
  [0x34a8] = 0x838c, [0x34a9] = 0x838d, [0x34aa] = 0x838f, [0x34ab] = 0x8390,
  [0x34ac] = 0x8391, [0x34ad] = 0x8394, [0x34ae] = 0x8395, [0x34af] = 0x8396,
  [0x34b0] = 0x8397, [0x34b1] = 0x8399, [0x34b2] = 0x839a, [0x34b3] = 0x839d,
  [0x34b4] = 0x839f, [0x34b5] = 0x83a1, [0x34b6] = 0x83a2, [0x34b7] = 0x83a3,
  [0x34b8] = 0x83a4, [0x34b9] = 0x83a5, [0x34ba] = 0x83a6, [0x34bb] = 0x83a7,
  [0x34bc] = 0x83ac, [0x34bd] = 0x83ad, [0x34be] = 0x83ae, [0x34c0] = 0x83af,
  [0x34c1] = 0x83b5, [0x34c2] = 0x83bb, [0x34c3] = 0x83be, [0x34c4] = 0x83bf,
  [0x34c5] = 0x83c2, [0x34c6] = 0x83c3, [0x34c7] = 0x83c4, [0x34c8] = 0x83c6,
  [0x34c9] = 0x83c8, [0x34ca] = 0x83c9, [0x34cb] = 0x83cb, [0x34cc] = 0x83cd,
  [0x34cd] = 0x83ce, [0x34ce] = 0x83d0, [0x34cf] = 0x83d1, [0x34d0] = 0x83d2,
  [0x34d1] = 0x83d3, [0x34d2] = 0x83d5, [0x34d3] = 0x83d7, [0x34d4] = 0x83d9,
  [0x34d5] = 0x83da, [0x34d6] = 0x83db, [0x34d7] = 0x83de, [0x34d8] = 0x83e2,
  [0x34d9] = 0x83e3, [0x34da] = 0x83e4, [0x34db] = 0x83e6, [0x34dc] = 0x83e7,
  [0x34dd] = 0x83e8, [0x34de] = 0x83eb, [0x34df] = 0x83ec, [0x34e0] = 0x83ed,
  [0x34e1] = 0x6070, [0x34e2] = 0x6d3d, [0x34e3] = 0x7275, [0x34e4] = 0x6266,
  [0x34e5] = 0x948e, [0x34e6] = 0x94c5, [0x34e7] = 0x5343, [0x34e8] = 0x8fc1,
  [0x34e9] = 0x7b7e, [0x34ea] = 0x4edf, [0x34eb] = 0x8c26, [0x34ec] = 0x4e7e,
  [0x34ed] = 0x9ed4, [0x34ee] = 0x94b1, [0x34ef] = 0x94b3, [0x34f0] = 0x524d,
  [0x34f1] = 0x6f5c, [0x34f2] = 0x9063, [0x34f3] = 0x6d45, [0x34f4] = 0x8c34,
  [0x34f5] = 0x5811, [0x34f6] = 0x5d4c, [0x34f7] = 0x6b20, [0x34f8] = 0x6b49,
  [0x34f9] = 0x67aa, [0x34fa] = 0x545b, [0x34fb] = 0x8154, [0x34fc] = 0x7f8c,
  [0x34fd] = 0x5899, [0x34fe] = 0x8537, [0x34ff] = 0x5f3a, [0x3500] = 0x62a2,
  [0x3501] = 0x6a47, [0x3502] = 0x9539, [0x3503] = 0x6572, [0x3504] = 0x6084,
  [0x3505] = 0x6865, [0x3506] = 0x77a7, [0x3507] = 0x4e54, [0x3508] = 0x4fa8,
  [0x3509] = 0x5de7, [0x350a] = 0x9798, [0x350b] = 0x64ac, [0x350c] = 0x7fd8,
  [0x350d] = 0x5ced, [0x350e] = 0x4fcf, [0x350f] = 0x7a8d, [0x3510] = 0x5207,
  [0x3511] = 0x8304, [0x3512] = 0x4e14, [0x3513] = 0x602f, [0x3514] = 0x7a83,
  [0x3515] = 0x94a6, [0x3516] = 0x4fb5, [0x3517] = 0x4eb2, [0x3518] = 0x79e6,
  [0x3519] = 0x7434, [0x351a] = 0x52e4, [0x351b] = 0x82b9, [0x351c] = 0x64d2,
  [0x351d] = 0x79bd, [0x351e] = 0x5bdd, [0x351f] = 0x6c81, [0x3520] = 0x9752,
  [0x3521] = 0x8f7b, [0x3522] = 0x6c22, [0x3523] = 0x503e, [0x3524] = 0x537f,
  [0x3525] = 0x6e05, [0x3526] = 0x64ce, [0x3527] = 0x6674, [0x3528] = 0x6c30,
  [0x3529] = 0x60c5, [0x352a] = 0x9877, [0x352b] = 0x8bf7, [0x352c] = 0x5e86,
  [0x352d] = 0x743c, [0x352e] = 0x7a77, [0x352f] = 0x79cb, [0x3530] = 0x4e18,
  [0x3531] = 0x90b1, [0x3532] = 0x7403, [0x3533] = 0x6c42, [0x3534] = 0x56da,
  [0x3535] = 0x914b, [0x3536] = 0x6cc5, [0x3537] = 0x8d8b, [0x3538] = 0x533a,
  [0x3539] = 0x86c6, [0x353a] = 0x66f2, [0x353b] = 0x8eaf, [0x353c] = 0x5c48,
  [0x353d] = 0x9a71, [0x353e] = 0x6e20, [0x3540] = 0x83ee, [0x3541] = 0x83ef,
  [0x3542] = 0x83f3, [0x3543] = 0x83f4, [0x3544] = 0x83f5, [0x3545] = 0x83f6,
  [0x3546] = 0x83f7, [0x3547] = 0x83fa, [0x3548] = 0x83fb, [0x3549] = 0x83fc,
  [0x354a] = 0x83fe, [0x354b] = 0x83ff, [0x354c] = 0x8400, [0x354d] = 0x8402,
  [0x354e] = 0x8405, [0x354f] = 0x8407, [0x3550] = 0x8408, [0x3551] = 0x8409,
  [0x3552] = 0x840a, [0x3553] = 0x8410, [0x3554] = 0x8412, [0x3555] = 0x8413,
  [0x3556] = 0x8414, [0x3557] = 0x8415, [0x3558] = 0x8416, [0x3559] = 0x8417,
  [0x355a] = 0x8419, [0x355b] = 0x841a, [0x355c] = 0x841b, [0x355d] = 0x841e,
  [0x355e] = 0x841f, [0x355f] = 0x8420, [0x3560] = 0x8421, [0x3561] = 0x8422,
  [0x3562] = 0x8423, [0x3563] = 0x8429, [0x3564] = 0x842a, [0x3565] = 0x842b,
  [0x3566] = 0x842c, [0x3567] = 0x842d, [0x3568] = 0x842e, [0x3569] = 0x842f,
  [0x356a] = 0x8430, [0x356b] = 0x8432, [0x356c] = 0x8433, [0x356d] = 0x8434,
  [0x356e] = 0x8435, [0x356f] = 0x8436, [0x3570] = 0x8437, [0x3571] = 0x8439,
  [0x3572] = 0x843a, [0x3573] = 0x843b, [0x3574] = 0x843e, [0x3575] = 0x843f,
  [0x3576] = 0x8440, [0x3577] = 0x8441, [0x3578] = 0x8442, [0x3579] = 0x8443,
  [0x357a] = 0x8444, [0x357b] = 0x8445, [0x357c] = 0x8447, [0x357d] = 0x8448,
  [0x357e] = 0x8449, [0x3580] = 0x844a, [0x3581] = 0x844b, [0x3582] = 0x844c,
  [0x3583] = 0x844d, [0x3584] = 0x844e, [0x3585] = 0x844f, [0x3586] = 0x8450,
  [0x3587] = 0x8452, [0x3588] = 0x8453, [0x3589] = 0x8454, [0x358a] = 0x8455,
  [0x358b] = 0x8456, [0x358c] = 0x8458, [0x358d] = 0x845d, [0x358e] = 0x845e,
  [0x358f] = 0x845f, [0x3590] = 0x8460, [0x3591] = 0x8462, [0x3592] = 0x8464,
  [0x3593] = 0x8465, [0x3594] = 0x8466, [0x3595] = 0x8467, [0x3596] = 0x8468,
  [0x3597] = 0x846a, [0x3598] = 0x846e, [0x3599] = 0x846f, [0x359a] = 0x8470,
  [0x359b] = 0x8472, [0x359c] = 0x8474, [0x359d] = 0x8477, [0x359e] = 0x8479,
  [0x359f] = 0x847b, [0x35a0] = 0x847c, [0x35a1] = 0x53d6, [0x35a2] = 0x5a36,
  [0x35a3] = 0x9f8b, [0x35a4] = 0x8da3, [0x35a5] = 0x53bb, [0x35a6] = 0x5708,
  [0x35a7] = 0x98a7, [0x35a8] = 0x6743, [0x35a9] = 0x919b, [0x35aa] = 0x6cc9,
  [0x35ab] = 0x5168, [0x35ac] = 0x75ca, [0x35ad] = 0x62f3, [0x35ae] = 0x72ac,
  [0x35af] = 0x5238, [0x35b0] = 0x529d, [0x35b1] = 0x7f3a, [0x35b2] = 0x7094,
  [0x35b3] = 0x7638, [0x35b4] = 0x5374, [0x35b5] = 0x9e4a, [0x35b6] = 0x69b7,
  [0x35b7] = 0x786e, [0x35b8] = 0x96c0, [0x35b9] = 0x88d9, [0x35ba] = 0x7fa4,
  [0x35bb] = 0x7136, [0x35bc] = 0x71c3, [0x35bd] = 0x5189, [0x35be] = 0x67d3,
  [0x35bf] = 0x74e4, [0x35c0] = 0x58e4, [0x35c1] = 0x6518, [0x35c2] = 0x56b7,
  [0x35c3] = 0x8ba9, [0x35c4] = 0x9976, [0x35c5] = 0x6270, [0x35c6] = 0x7ed5,
  [0x35c7] = 0x60f9, [0x35c8] = 0x70ed, [0x35c9] = 0x58ec, [0x35ca] = 0x4ec1,
  [0x35cb] = 0x4eba, [0x35cc] = 0x5fcd, [0x35cd] = 0x97e7, [0x35ce] = 0x4efb,
  [0x35cf] = 0x8ba4, [0x35d0] = 0x5203, [0x35d1] = 0x598a, [0x35d2] = 0x7eab,
  [0x35d3] = 0x6254, [0x35d4] = 0x4ecd, [0x35d5] = 0x65e5, [0x35d6] = 0x620e,
  [0x35d7] = 0x8338, [0x35d8] = 0x84c9, [0x35d9] = 0x8363, [0x35da] = 0x878d,
  [0x35db] = 0x7194, [0x35dc] = 0x6eb6, [0x35dd] = 0x5bb9, [0x35de] = 0x7ed2,
  [0x35df] = 0x5197, [0x35e0] = 0x63c9, [0x35e1] = 0x67d4, [0x35e2] = 0x8089,
  [0x35e3] = 0x8339, [0x35e4] = 0x8815, [0x35e5] = 0x5112, [0x35e6] = 0x5b7a,
  [0x35e7] = 0x5982, [0x35e8] = 0x8fb1, [0x35e9] = 0x4e73, [0x35ea] = 0x6c5d,
  [0x35eb] = 0x5165, [0x35ec] = 0x8925, [0x35ed] = 0x8f6f, [0x35ee] = 0x962e,
  [0x35ef] = 0x854a, [0x35f0] = 0x745e, [0x35f1] = 0x9510, [0x35f2] = 0x95f0,
  [0x35f3] = 0x6da6, [0x35f4] = 0x82e5, [0x35f5] = 0x5f31, [0x35f6] = 0x6492,
  [0x35f7] = 0x6d12, [0x35f8] = 0x8428, [0x35f9] = 0x816e, [0x35fa] = 0x9cc3,
  [0x35fb] = 0x585e, [0x35fc] = 0x8d5b, [0x35fd] = 0x4e09, [0x35fe] = 0x53c1,
  [0x3600] = 0x847d, [0x3601] = 0x847e, [0x3602] = 0x847f, [0x3603] = 0x8480,
  [0x3604] = 0x8481, [0x3605] = 0x8483, [0x3606] = 0x8484, [0x3607] = 0x8485,
  [0x3608] = 0x8486, [0x3609] = 0x848a, [0x360a] = 0x848d, [0x360b] = 0x848f,
  [0x360c] = 0x8490, [0x360d] = 0x8491, [0x360e] = 0x8492, [0x360f] = 0x8493,
  [0x3610] = 0x8494, [0x3611] = 0x8495, [0x3612] = 0x8496, [0x3613] = 0x8498,
  [0x3614] = 0x849a, [0x3615] = 0x849b, [0x3616] = 0x849d, [0x3617] = 0x849e,
  [0x3618] = 0x849f, [0x3619] = 0x84a0, [0x361a] = 0x84a2, [0x361b] = 0x84a3,
  [0x361c] = 0x84a4, [0x361d] = 0x84a5, [0x361e] = 0x84a6, [0x361f] = 0x84a7,
  [0x3620] = 0x84a8, [0x3621] = 0x84a9, [0x3622] = 0x84aa, [0x3623] = 0x84ab,
  [0x3624] = 0x84ac, [0x3625] = 0x84ad, [0x3626] = 0x84ae, [0x3627] = 0x84b0,
  [0x3628] = 0x84b1, [0x3629] = 0x84b3, [0x362a] = 0x84b5, [0x362b] = 0x84b6,
  [0x362c] = 0x84b7, [0x362d] = 0x84bb, [0x362e] = 0x84bc, [0x362f] = 0x84be,
  [0x3630] = 0x84c0, [0x3631] = 0x84c2, [0x3632] = 0x84c3, [0x3633] = 0x84c5,
  [0x3634] = 0x84c6, [0x3635] = 0x84c7, [0x3636] = 0x84c8, [0x3637] = 0x84cb,
  [0x3638] = 0x84cc, [0x3639] = 0x84ce, [0x363a] = 0x84cf, [0x363b] = 0x84d2,
  [0x363c] = 0x84d4, [0x363d] = 0x84d5, [0x363e] = 0x84d7, [0x3640] = 0x84d8,
  [0x3641] = 0x84d9, [0x3642] = 0x84da, [0x3643] = 0x84db, [0x3644] = 0x84dc,
  [0x3645] = 0x84de, [0x3646] = 0x84e1, [0x3647] = 0x84e2, [0x3648] = 0x84e4,
  [0x3649] = 0x84e7, [0x364a] = 0x84e8, [0x364b] = 0x84e9, [0x364c] = 0x84ea,
  [0x364d] = 0x84eb, [0x364e] = 0x84ed, [0x364f] = 0x84ee, [0x3650] = 0x84ef,
  [0x3651] = 0x84f1, [0x3652] = 0x84f2, [0x3653] = 0x84f3, [0x3654] = 0x84f4,
  [0x3655] = 0x84f5, [0x3656] = 0x84f6, [0x3657] = 0x84f7, [0x3658] = 0x84f8,
  [0x3659] = 0x84f9, [0x365a] = 0x84fa, [0x365b] = 0x84fb, [0x365c] = 0x84fd,
  [0x365d] = 0x84fe, [0x365e] = 0x8500, [0x365f] = 0x8501, [0x3660] = 0x8502,
  [0x3661] = 0x4f1e, [0x3662] = 0x6563, [0x3663] = 0x6851, [0x3664] = 0x55d3,
  [0x3665] = 0x4e27, [0x3666] = 0x6414, [0x3667] = 0x9a9a, [0x3668] = 0x626b,
  [0x3669] = 0x5ac2, [0x366a] = 0x745f, [0x366b] = 0x8272, [0x366c] = 0x6da9,
  [0x366d] = 0x68ee, [0x366e] = 0x50e7, [0x366f] = 0x838e, [0x3670] = 0x7802,
  [0x3671] = 0x6740, [0x3672] = 0x5239, [0x3673] = 0x6c99, [0x3674] = 0x7eb1,
  [0x3675] = 0x50bb, [0x3676] = 0x5565, [0x3677] = 0x715e, [0x3678] = 0x7b5b,
  [0x3679] = 0x6652, [0x367a] = 0x73ca, [0x367b] = 0x82eb, [0x367c] = 0x6749,
  [0x367d] = 0x5c71, [0x367e] = 0x5220, [0x367f] = 0x717d, [0x3680] = 0x886b,
  [0x3681] = 0x95ea, [0x3682] = 0x9655, [0x3683] = 0x64c5, [0x3684] = 0x8d61,
  [0x3685] = 0x81b3, [0x3686] = 0x5584, [0x3687] = 0x6c55, [0x3688] = 0x6247,
  [0x3689] = 0x7f2e, [0x368a] = 0x5892, [0x368b] = 0x4f24, [0x368c] = 0x5546,
  [0x368d] = 0x8d4f, [0x368e] = 0x664c, [0x368f] = 0x4e0a, [0x3690] = 0x5c1a,
  [0x3691] = 0x88f3, [0x3692] = 0x68a2, [0x3693] = 0x634e, [0x3694] = 0x7a0d,
  [0x3695] = 0x70e7, [0x3696] = 0x828d, [0x3697] = 0x52fa, [0x3698] = 0x97f6,
  [0x3699] = 0x5c11, [0x369a] = 0x54e8, [0x369b] = 0x90b5, [0x369c] = 0x7ecd,
  [0x369d] = 0x5962, [0x369e] = 0x8d4a, [0x369f] = 0x86c7, [0x36a0] = 0x820c,
  [0x36a1] = 0x820d, [0x36a2] = 0x8d66, [0x36a3] = 0x6444, [0x36a4] = 0x5c04,
  [0x36a5] = 0x6151, [0x36a6] = 0x6d89, [0x36a7] = 0x793e, [0x36a8] = 0x8bbe,
  [0x36a9] = 0x7837, [0x36aa] = 0x7533, [0x36ab] = 0x547b, [0x36ac] = 0x4f38,
  [0x36ad] = 0x8eab, [0x36ae] = 0x6df1, [0x36af] = 0x5a20, [0x36b0] = 0x7ec5,
  [0x36b1] = 0x795e, [0x36b2] = 0x6c88, [0x36b3] = 0x5ba1, [0x36b4] = 0x5a76,
  [0x36b5] = 0x751a, [0x36b6] = 0x80be, [0x36b7] = 0x614e, [0x36b8] = 0x6e17,
  [0x36b9] = 0x58f0, [0x36ba] = 0x751f, [0x36bb] = 0x7525, [0x36bc] = 0x7272,
  [0x36bd] = 0x5347, [0x36be] = 0x7ef3, [0x36c0] = 0x8503, [0x36c1] = 0x8504,
  [0x36c2] = 0x8505, [0x36c3] = 0x8506, [0x36c4] = 0x8507, [0x36c5] = 0x8508,
  [0x36c6] = 0x8509, [0x36c7] = 0x850a, [0x36c8] = 0x850b, [0x36c9] = 0x850d,
  [0x36ca] = 0x850e, [0x36cb] = 0x850f, [0x36cc] = 0x8510, [0x36cd] = 0x8512,
  [0x36ce] = 0x8514, [0x36cf] = 0x8515, [0x36d0] = 0x8516, [0x36d1] = 0x8518,
  [0x36d2] = 0x8519, [0x36d3] = 0x851b, [0x36d4] = 0x851c, [0x36d5] = 0x851d,
  [0x36d6] = 0x851e, [0x36d7] = 0x8520, [0x36d8] = 0x8522, [0x36d9] = 0x8523,
  [0x36da] = 0x8524, [0x36db] = 0x8525, [0x36dc] = 0x8526, [0x36dd] = 0x8527,
  [0x36de] = 0x8528, [0x36df] = 0x8529, [0x36e0] = 0x852a, [0x36e1] = 0x852d,
  [0x36e2] = 0x852e, [0x36e3] = 0x852f, [0x36e4] = 0x8530, [0x36e5] = 0x8531,
  [0x36e6] = 0x8532, [0x36e7] = 0x8533, [0x36e8] = 0x8534, [0x36e9] = 0x8535,
  [0x36ea] = 0x8536, [0x36eb] = 0x853e, [0x36ec] = 0x853f, [0x36ed] = 0x8540,
  [0x36ee] = 0x8541, [0x36ef] = 0x8542, [0x36f0] = 0x8544, [0x36f1] = 0x8545,
  [0x36f2] = 0x8546, [0x36f3] = 0x8547, [0x36f4] = 0x854b, [0x36f5] = 0x854c,
  [0x36f6] = 0x854d, [0x36f7] = 0x854e, [0x36f8] = 0x854f, [0x36f9] = 0x8550,
  [0x36fa] = 0x8551, [0x36fb] = 0x8552, [0x36fc] = 0x8553, [0x36fd] = 0x8554,
  [0x36fe] = 0x8555, [0x3700] = 0x8557, [0x3701] = 0x8558, [0x3702] = 0x855a,
  [0x3703] = 0x855b, [0x3704] = 0x855c, [0x3705] = 0x855d, [0x3706] = 0x855f,
  [0x3707] = 0x8560, [0x3708] = 0x8561, [0x3709] = 0x8562, [0x370a] = 0x8563,
  [0x370b] = 0x8565, [0x370c] = 0x8566, [0x370d] = 0x8567, [0x370e] = 0x8569,
  [0x370f] = 0x856a, [0x3710] = 0x856b, [0x3711] = 0x856c, [0x3712] = 0x856d,
  [0x3713] = 0x856e, [0x3714] = 0x856f, [0x3715] = 0x8570, [0x3716] = 0x8571,
  [0x3717] = 0x8573, [0x3718] = 0x8575, [0x3719] = 0x8576, [0x371a] = 0x8577,
  [0x371b] = 0x8578, [0x371c] = 0x857c, [0x371d] = 0x857d, [0x371e] = 0x857f,
  [0x371f] = 0x8580, [0x3720] = 0x8581, [0x3721] = 0x7701, [0x3722] = 0x76db,
  [0x3723] = 0x5269, [0x3724] = 0x80dc, [0x3725] = 0x5723, [0x3726] = 0x5e08,
  [0x3727] = 0x5931, [0x3728] = 0x72ee, [0x3729] = 0x65bd, [0x372a] = 0x6e7f,
  [0x372b] = 0x8bd7, [0x372c] = 0x5c38, [0x372d] = 0x8671, [0x372e] = 0x5341,
  [0x372f] = 0x77f3, [0x3730] = 0x62fe, [0x3731] = 0x65f6, [0x3732] = 0x4ec0,
  [0x3733] = 0x98df, [0x3734] = 0x8680, [0x3735] = 0x5b9e, [0x3736] = 0x8bc6,
  [0x3737] = 0x53f2, [0x3738] = 0x77e2, [0x3739] = 0x4f7f, [0x373a] = 0x5c4e,
  [0x373b] = 0x9a76, [0x373c] = 0x59cb, [0x373d] = 0x5f0f, [0x373e] = 0x793a,
  [0x373f] = 0x58eb, [0x3740] = 0x4e16, [0x3741] = 0x67ff, [0x3742] = 0x4e8b,
  [0x3743] = 0x62ed, [0x3744] = 0x8a93, [0x3745] = 0x901d, [0x3746] = 0x52bf,
  [0x3747] = 0x662f, [0x3748] = 0x55dc, [0x3749] = 0x566c, [0x374a] = 0x9002,
  [0x374b] = 0x4ed5, [0x374c] = 0x4f8d, [0x374d] = 0x91ca, [0x374e] = 0x9970,
  [0x374f] = 0x6c0f, [0x3750] = 0x5e02, [0x3751] = 0x6043, [0x3752] = 0x5ba4,
  [0x3753] = 0x89c6, [0x3754] = 0x8bd5, [0x3755] = 0x6536, [0x3756] = 0x624b,
  [0x3757] = 0x9996, [0x3758] = 0x5b88, [0x3759] = 0x5bff, [0x375a] = 0x6388,
  [0x375b] = 0x552e, [0x375c] = 0x53d7, [0x375d] = 0x7626, [0x375e] = 0x517d,
  [0x375f] = 0x852c, [0x3760] = 0x67a2, [0x3761] = 0x68b3, [0x3762] = 0x6b8a,
  [0x3763] = 0x6292, [0x3764] = 0x8f93, [0x3765] = 0x53d4, [0x3766] = 0x8212,
  [0x3767] = 0x6dd1, [0x3768] = 0x758f, [0x3769] = 0x4e66, [0x376a] = 0x8d4e,
  [0x376b] = 0x5b70, [0x376c] = 0x719f, [0x376d] = 0x85af, [0x376e] = 0x6691,
  [0x376f] = 0x66d9, [0x3770] = 0x7f72, [0x3771] = 0x8700, [0x3772] = 0x9ecd,
  [0x3773] = 0x9f20, [0x3774] = 0x5c5e, [0x3775] = 0x672f, [0x3776] = 0x8ff0,
  [0x3777] = 0x6811, [0x3778] = 0x675f, [0x3779] = 0x620d, [0x377a] = 0x7ad6,
  [0x377b] = 0x5885, [0x377c] = 0x5eb6, [0x377d] = 0x6570, [0x377e] = 0x6f31,
  [0x3780] = 0x8582, [0x3781] = 0x8583, [0x3782] = 0x8586, [0x3783] = 0x8588,
  [0x3784] = 0x8589, [0x3785] = 0x858a, [0x3786] = 0x858b, [0x3787] = 0x858c,
  [0x3788] = 0x858d, [0x3789] = 0x858e, [0x378a] = 0x8590, [0x378b] = 0x8591,
  [0x378c] = 0x8592, [0x378d] = 0x8593, [0x378e] = 0x8594, [0x378f] = 0x8595,
  [0x3790] = 0x8596, [0x3791] = 0x8597, [0x3792] = 0x8598, [0x3793] = 0x8599,
  [0x3794] = 0x859a, [0x3795] = 0x859d, [0x3796] = 0x859e, [0x3797] = 0x859f,
  [0x3798] = 0x85a0, [0x3799] = 0x85a1, [0x379a] = 0x85a2, [0x379b] = 0x85a3,
  [0x379c] = 0x85a5, [0x379d] = 0x85a6, [0x379e] = 0x85a7, [0x379f] = 0x85a9,
  [0x37a0] = 0x85ab, [0x37a1] = 0x85ac, [0x37a2] = 0x85ad, [0x37a3] = 0x85b1,
  [0x37a4] = 0x85b2, [0x37a5] = 0x85b3, [0x37a6] = 0x85b4, [0x37a7] = 0x85b5,
  [0x37a8] = 0x85b6, [0x37a9] = 0x85b8, [0x37aa] = 0x85ba, [0x37ab] = 0x85bb,
  [0x37ac] = 0x85bc, [0x37ad] = 0x85bd, [0x37ae] = 0x85be, [0x37af] = 0x85bf,
  [0x37b0] = 0x85c0, [0x37b1] = 0x85c2, [0x37b2] = 0x85c3, [0x37b3] = 0x85c4,
  [0x37b4] = 0x85c5, [0x37b5] = 0x85c6, [0x37b6] = 0x85c7, [0x37b7] = 0x85c8,
  [0x37b8] = 0x85ca, [0x37b9] = 0x85cb, [0x37ba] = 0x85cc, [0x37bb] = 0x85cd,
  [0x37bc] = 0x85ce, [0x37bd] = 0x85d1, [0x37be] = 0x85d2, [0x37c0] = 0x85d4,
  [0x37c1] = 0x85d6, [0x37c2] = 0x85d7, [0x37c3] = 0x85d8, [0x37c4] = 0x85d9,
  [0x37c5] = 0x85da, [0x37c6] = 0x85db, [0x37c7] = 0x85dd, [0x37c8] = 0x85de,
  [0x37c9] = 0x85df, [0x37ca] = 0x85e0, [0x37cb] = 0x85e1, [0x37cc] = 0x85e2,
  [0x37cd] = 0x85e3, [0x37ce] = 0x85e5, [0x37cf] = 0x85e6, [0x37d0] = 0x85e7,
  [0x37d1] = 0x85e8, [0x37d2] = 0x85ea, [0x37d3] = 0x85eb, [0x37d4] = 0x85ec,
  [0x37d5] = 0x85ed, [0x37d6] = 0x85ee, [0x37d7] = 0x85ef, [0x37d8] = 0x85f0,
  [0x37d9] = 0x85f1, [0x37da] = 0x85f2, [0x37db] = 0x85f3, [0x37dc] = 0x85f4,
  [0x37dd] = 0x85f5, [0x37de] = 0x85f6, [0x37df] = 0x85f7, [0x37e0] = 0x85f8,
  [0x37e1] = 0x6055, [0x37e2] = 0x5237, [0x37e3] = 0x800d, [0x37e4] = 0x6454,
  [0x37e5] = 0x8870, [0x37e6] = 0x7529, [0x37e7] = 0x5e05, [0x37e8] = 0x6813,
  [0x37e9] = 0x62f4, [0x37ea] = 0x971c, [0x37eb] = 0x53cc, [0x37ec] = 0x723d,
  [0x37ed] = 0x8c01, [0x37ee] = 0x6c34, [0x37ef] = 0x7761, [0x37f0] = 0x7a0e,
  [0x37f1] = 0x542e, [0x37f2] = 0x77ac, [0x37f3] = 0x987a, [0x37f4] = 0x821c,
  [0x37f5] = 0x8bf4, [0x37f6] = 0x7855, [0x37f7] = 0x6714, [0x37f8] = 0x70c1,
  [0x37f9] = 0x65af, [0x37fa] = 0x6495, [0x37fb] = 0x5636, [0x37fc] = 0x601d,
  [0x37fd] = 0x79c1, [0x37fe] = 0x53f8, [0x37ff] = 0x4e1d, [0x3800] = 0x6b7b,
  [0x3801] = 0x8086, [0x3802] = 0x5bfa, [0x3803] = 0x55e3, [0x3804] = 0x56db,
  [0x3805] = 0x4f3a, [0x3806] = 0x4f3c, [0x3807] = 0x9972, [0x3808] = 0x5df3,
  [0x3809] = 0x677e, [0x380a] = 0x8038, [0x380b] = 0x6002, [0x380c] = 0x9882,
  [0x380d] = 0x9001, [0x380e] = 0x5b8b, [0x380f] = 0x8bbc, [0x3810] = 0x8bf5,
  [0x3811] = 0x641c, [0x3812] = 0x8258, [0x3813] = 0x64de, [0x3814] = 0x55fd,
  [0x3815] = 0x82cf, [0x3816] = 0x9165, [0x3817] = 0x4fd7, [0x3818] = 0x7d20,
  [0x3819] = 0x901f, [0x381a] = 0x7c9f, [0x381b] = 0x50f3, [0x381c] = 0x5851,
  [0x381d] = 0x6eaf, [0x381e] = 0x5bbf, [0x381f] = 0x8bc9, [0x3820] = 0x8083,
  [0x3821] = 0x9178, [0x3822] = 0x849c, [0x3823] = 0x7b97, [0x3824] = 0x867d,
  [0x3825] = 0x968b, [0x3826] = 0x968f, [0x3827] = 0x7ee5, [0x3828] = 0x9ad3,
  [0x3829] = 0x788e, [0x382a] = 0x5c81, [0x382b] = 0x7a57, [0x382c] = 0x9042,
  [0x382d] = 0x96a7, [0x382e] = 0x795f, [0x382f] = 0x5b59, [0x3830] = 0x635f,
  [0x3831] = 0x7b0b, [0x3832] = 0x84d1, [0x3833] = 0x68ad, [0x3834] = 0x5506,
  [0x3835] = 0x7f29, [0x3836] = 0x7410, [0x3837] = 0x7d22, [0x3838] = 0x9501,
  [0x3839] = 0x6240, [0x383a] = 0x584c, [0x383b] = 0x4ed6, [0x383c] = 0x5b83,
  [0x383d] = 0x5979, [0x383e] = 0x5854, [0x3840] = 0x85f9, [0x3841] = 0x85fa,
  [0x3842] = 0x85fc, [0x3843] = 0x85fd, [0x3844] = 0x85fe, [0x3845] = 0x8600,
  [0x3846] = 0x8601, [0x3847] = 0x8602, [0x3848] = 0x8603, [0x3849] = 0x8604,
  [0x384a] = 0x8606, [0x384b] = 0x8607, [0x384c] = 0x8608, [0x384d] = 0x8609,
  [0x384e] = 0x860a, [0x384f] = 0x860b, [0x3850] = 0x860c, [0x3851] = 0x860d,
  [0x3852] = 0x860e, [0x3853] = 0x860f, [0x3854] = 0x8610, [0x3855] = 0x8612,
  [0x3856] = 0x8613, [0x3857] = 0x8614, [0x3858] = 0x8615, [0x3859] = 0x8617,
  [0x385a] = 0x8618, [0x385b] = 0x8619, [0x385c] = 0x861a, [0x385d] = 0x861b,
  [0x385e] = 0x861c, [0x385f] = 0x861d, [0x3860] = 0x861e, [0x3861] = 0x861f,
  [0x3862] = 0x8620, [0x3863] = 0x8621, [0x3864] = 0x8622, [0x3865] = 0x8623,
  [0x3866] = 0x8624, [0x3867] = 0x8625, [0x3868] = 0x8626, [0x3869] = 0x8628,
  [0x386a] = 0x862a, [0x386b] = 0x862b, [0x386c] = 0x862c, [0x386d] = 0x862d,
  [0x386e] = 0x862e, [0x386f] = 0x862f, [0x3870] = 0x8630, [0x3871] = 0x8631,
  [0x3872] = 0x8632, [0x3873] = 0x8633, [0x3874] = 0x8634, [0x3875] = 0x8635,
  [0x3876] = 0x8636, [0x3877] = 0x8637, [0x3878] = 0x8639, [0x3879] = 0x863a,
  [0x387a] = 0x863b, [0x387b] = 0x863d, [0x387c] = 0x863e, [0x387d] = 0x863f,
  [0x387e] = 0x8640, [0x3880] = 0x8641, [0x3881] = 0x8642, [0x3882] = 0x8643,
  [0x3883] = 0x8644, [0x3884] = 0x8645, [0x3885] = 0x8646, [0x3886] = 0x8647,
  [0x3887] = 0x8648, [0x3888] = 0x8649, [0x3889] = 0x864a, [0x388a] = 0x864b,
  [0x388b] = 0x864c, [0x388c] = 0x8652, [0x388d] = 0x8653, [0x388e] = 0x8655,
  [0x388f] = 0x8656, [0x3890] = 0x8657, [0x3891] = 0x8658, [0x3892] = 0x8659,
  [0x3893] = 0x865b, [0x3894] = 0x865c, [0x3895] = 0x865d, [0x3896] = 0x865f,
  [0x3897] = 0x8660, [0x3898] = 0x8661, [0x3899] = 0x8663, [0x389a] = 0x8664,
  [0x389b] = 0x8665, [0x389c] = 0x8666, [0x389d] = 0x8667, [0x389e] = 0x8668,
  [0x389f] = 0x8669, [0x38a0] = 0x866a, [0x38a1] = 0x736d, [0x38a2] = 0x631e,
  [0x38a3] = 0x8e4b, [0x38a4] = 0x8e0f, [0x38a5] = 0x80ce, [0x38a6] = 0x82d4,
  [0x38a7] = 0x62ac, [0x38a8] = 0x53f0, [0x38a9] = 0x6cf0, [0x38aa] = 0x915e,
  [0x38ab] = 0x592a, [0x38ac] = 0x6001, [0x38ad] = 0x6c70, [0x38ae] = 0x574d,
  [0x38af] = 0x644a, [0x38b0] = 0x8d2a, [0x38b1] = 0x762b, [0x38b2] = 0x6ee9,
  [0x38b3] = 0x575b, [0x38b4] = 0x6a80, [0x38b5] = 0x75f0, [0x38b6] = 0x6f6d,
  [0x38b7] = 0x8c2d, [0x38b8] = 0x8c08, [0x38b9] = 0x5766, [0x38ba] = 0x6bef,
  [0x38bb] = 0x8892, [0x38bc] = 0x78b3, [0x38bd] = 0x63a2, [0x38be] = 0x53f9,
  [0x38bf] = 0x70ad, [0x38c0] = 0x6c64, [0x38c1] = 0x5858, [0x38c2] = 0x642a,
  [0x38c3] = 0x5802, [0x38c4] = 0x68e0, [0x38c5] = 0x819b, [0x38c6] = 0x5510,
  [0x38c7] = 0x7cd6, [0x38c8] = 0x5018, [0x38c9] = 0x8eba, [0x38ca] = 0x6dcc,
  [0x38cb] = 0x8d9f, [0x38cc] = 0x70eb, [0x38cd] = 0x638f, [0x38ce] = 0x6d9b,
  [0x38cf] = 0x6ed4, [0x38d0] = 0x7ee6, [0x38d1] = 0x8404, [0x38d2] = 0x6843,
  [0x38d3] = 0x9003, [0x38d4] = 0x6dd8, [0x38d5] = 0x9676, [0x38d6] = 0x8ba8,
  [0x38d7] = 0x5957, [0x38d8] = 0x7279, [0x38d9] = 0x85e4, [0x38da] = 0x817e,
  [0x38db] = 0x75bc, [0x38dc] = 0x8a8a, [0x38dd] = 0x68af, [0x38de] = 0x5254,
  [0x38df] = 0x8e22, [0x38e0] = 0x9511, [0x38e1] = 0x63d0, [0x38e2] = 0x9898,
  [0x38e3] = 0x8e44, [0x38e4] = 0x557c, [0x38e5] = 0x4f53, [0x38e6] = 0x66ff,
  [0x38e7] = 0x568f, [0x38e8] = 0x60d5, [0x38e9] = 0x6d95, [0x38ea] = 0x5243,
  [0x38eb] = 0x5c49, [0x38ec] = 0x5929, [0x38ed] = 0x6dfb, [0x38ee] = 0x586b,
  [0x38ef] = 0x7530, [0x38f0] = 0x751c, [0x38f1] = 0x606c, [0x38f2] = 0x8214,
  [0x38f3] = 0x8146, [0x38f4] = 0x6311, [0x38f5] = 0x6761, [0x38f6] = 0x8fe2,
  [0x38f7] = 0x773a, [0x38f8] = 0x8df3, [0x38f9] = 0x8d34, [0x38fa] = 0x94c1,
  [0x38fb] = 0x5e16, [0x38fc] = 0x5385, [0x38fd] = 0x542c, [0x38fe] = 0x70c3,
  [0x3900] = 0x866d, [0x3901] = 0x866f, [0x3902] = 0x8670, [0x3903] = 0x8672,
  [0x3904] = 0x8673, [0x3905] = 0x8674, [0x3906] = 0x8675, [0x3907] = 0x8676,
  [0x3908] = 0x8677, [0x3909] = 0x8678, [0x390a] = 0x8683, [0x390b] = 0x8684,
  [0x390c] = 0x8685, [0x390d] = 0x8686, [0x390e] = 0x8687, [0x390f] = 0x8688,
  [0x3910] = 0x8689, [0x3911] = 0x868e, [0x3912] = 0x868f, [0x3913] = 0x8690,
  [0x3914] = 0x8691, [0x3915] = 0x8692, [0x3916] = 0x8694, [0x3917] = 0x8696,
  [0x3918] = 0x8697, [0x3919] = 0x8698, [0x391a] = 0x8699, [0x391b] = 0x869a,
  [0x391c] = 0x869b, [0x391d] = 0x869e, [0x391e] = 0x869f, [0x391f] = 0x86a0,
  [0x3920] = 0x86a1, [0x3921] = 0x86a2, [0x3922] = 0x86a5, [0x3923] = 0x86a6,
  [0x3924] = 0x86ab, [0x3925] = 0x86ad, [0x3926] = 0x86ae, [0x3927] = 0x86b2,
  [0x3928] = 0x86b3, [0x3929] = 0x86b7, [0x392a] = 0x86b8, [0x392b] = 0x86b9,
  [0x392c] = 0x86bb, [0x392d] = 0x86bc, [0x392e] = 0x86bd, [0x392f] = 0x86be,
  [0x3930] = 0x86bf, [0x3931] = 0x86c1, [0x3932] = 0x86c2, [0x3933] = 0x86c3,
  [0x3934] = 0x86c5, [0x3935] = 0x86c8, [0x3936] = 0x86cc, [0x3937] = 0x86cd,
  [0x3938] = 0x86d2, [0x3939] = 0x86d3, [0x393a] = 0x86d5, [0x393b] = 0x86d6,
  [0x393c] = 0x86d7, [0x393d] = 0x86da, [0x393e] = 0x86dc, [0x3940] = 0x86dd,
  [0x3941] = 0x86e0, [0x3942] = 0x86e1, [0x3943] = 0x86e2, [0x3944] = 0x86e3,
  [0x3945] = 0x86e5, [0x3946] = 0x86e6, [0x3947] = 0x86e7, [0x3948] = 0x86e8,
  [0x3949] = 0x86ea, [0x394a] = 0x86eb, [0x394b] = 0x86ec, [0x394c] = 0x86ef,
  [0x394d] = 0x86f5, [0x394e] = 0x86f6, [0x394f] = 0x86f7, [0x3950] = 0x86fa,
  [0x3951] = 0x86fb, [0x3952] = 0x86fc, [0x3953] = 0x86fd, [0x3954] = 0x86ff,
  [0x3955] = 0x8701, [0x3956] = 0x8704, [0x3957] = 0x8705, [0x3958] = 0x8706,
  [0x3959] = 0x870b, [0x395a] = 0x870c, [0x395b] = 0x870e, [0x395c] = 0x870f,
  [0x395d] = 0x8710, [0x395e] = 0x8711, [0x395f] = 0x8714, [0x3960] = 0x8716,
  [0x3961] = 0x6c40, [0x3962] = 0x5ef7, [0x3963] = 0x505c, [0x3964] = 0x4ead,
  [0x3965] = 0x5ead, [0x3966] = 0x633a, [0x3967] = 0x8247, [0x3968] = 0x901a,
  [0x3969] = 0x6850, [0x396a] = 0x916e, [0x396b] = 0x77b3, [0x396c] = 0x540c,
  [0x396d] = 0x94dc, [0x396e] = 0x5f64, [0x396f] = 0x7ae5, [0x3970] = 0x6876,
  [0x3971] = 0x6345, [0x3972] = 0x7b52, [0x3973] = 0x7edf, [0x3974] = 0x75db,
  [0x3975] = 0x5077, [0x3976] = 0x6295, [0x3977] = 0x5934, [0x3978] = 0x900f,
  [0x3979] = 0x51f8, [0x397a] = 0x79c3, [0x397b] = 0x7a81, [0x397c] = 0x56fe,
  [0x397d] = 0x5f92, [0x397e] = 0x9014, [0x397f] = 0x6d82, [0x3980] = 0x5c60,
  [0x3981] = 0x571f, [0x3982] = 0x5410, [0x3983] = 0x5154, [0x3984] = 0x6e4d,
  [0x3985] = 0x56e2, [0x3986] = 0x63a8, [0x3987] = 0x9893, [0x3988] = 0x817f,
  [0x3989] = 0x8715, [0x398a] = 0x892a, [0x398b] = 0x9000, [0x398c] = 0x541e,
  [0x398d] = 0x5c6f, [0x398e] = 0x81c0, [0x398f] = 0x62d6, [0x3990] = 0x6258,
  [0x3991] = 0x8131, [0x3992] = 0x9e35, [0x3993] = 0x9640, [0x3994] = 0x9a6e,
  [0x3995] = 0x9a7c, [0x3996] = 0x692d, [0x3997] = 0x59a5, [0x3998] = 0x62d3,
  [0x3999] = 0x553e, [0x399a] = 0x6316, [0x399b] = 0x54c7, [0x399c] = 0x86d9,
  [0x399d] = 0x6d3c, [0x399e] = 0x5a03, [0x399f] = 0x74e6, [0x39a0] = 0x889c,
  [0x39a1] = 0x6b6a, [0x39a2] = 0x5916, [0x39a3] = 0x8c4c, [0x39a4] = 0x5f2f,
  [0x39a5] = 0x6e7e, [0x39a6] = 0x73a9, [0x39a7] = 0x987d, [0x39a8] = 0x4e38,
  [0x39a9] = 0x70f7, [0x39aa] = 0x5b8c, [0x39ab] = 0x7897, [0x39ac] = 0x633d,
  [0x39ad] = 0x665a, [0x39ae] = 0x7696, [0x39af] = 0x60cb, [0x39b0] = 0x5b9b,
  [0x39b1] = 0x5a49, [0x39b2] = 0x4e07, [0x39b3] = 0x8155, [0x39b4] = 0x6c6a,
  [0x39b5] = 0x738b, [0x39b6] = 0x4ea1, [0x39b7] = 0x6789, [0x39b8] = 0x7f51,
  [0x39b9] = 0x5f80, [0x39ba] = 0x65fa, [0x39bb] = 0x671b, [0x39bc] = 0x5fd8,
  [0x39bd] = 0x5984, [0x39be] = 0x5a01, [0x39c0] = 0x8719, [0x39c1] = 0x871b,
  [0x39c2] = 0x871d, [0x39c3] = 0x871f, [0x39c4] = 0x8720, [0x39c5] = 0x8724,
  [0x39c6] = 0x8726, [0x39c7] = 0x8727, [0x39c8] = 0x8728, [0x39c9] = 0x872a,
  [0x39ca] = 0x872b, [0x39cb] = 0x872c, [0x39cc] = 0x872d, [0x39cd] = 0x872f,
  [0x39ce] = 0x8730, [0x39cf] = 0x8732, [0x39d0] = 0x8733, [0x39d1] = 0x8735,
  [0x39d2] = 0x8736, [0x39d3] = 0x8738, [0x39d4] = 0x8739, [0x39d5] = 0x873a,
  [0x39d6] = 0x873c, [0x39d7] = 0x873d, [0x39d8] = 0x8740, [0x39d9] = 0x8741,
  [0x39da] = 0x8742, [0x39db] = 0x8743, [0x39dc] = 0x8744, [0x39dd] = 0x8745,
  [0x39de] = 0x8746, [0x39df] = 0x874a, [0x39e0] = 0x874b, [0x39e1] = 0x874d,
  [0x39e2] = 0x874f, [0x39e3] = 0x8750, [0x39e4] = 0x8751, [0x39e5] = 0x8752,
  [0x39e6] = 0x8754, [0x39e7] = 0x8755, [0x39e8] = 0x8756, [0x39e9] = 0x8758,
  [0x39ea] = 0x875a, [0x39eb] = 0x875b, [0x39ec] = 0x875c, [0x39ed] = 0x875d,
  [0x39ee] = 0x875e, [0x39ef] = 0x875f, [0x39f0] = 0x8761, [0x39f1] = 0x8762,
  [0x39f2] = 0x8766, [0x39f3] = 0x8767, [0x39f4] = 0x8768, [0x39f5] = 0x8769,
  [0x39f6] = 0x876a, [0x39f7] = 0x876b, [0x39f8] = 0x876c, [0x39f9] = 0x876d,
  [0x39fa] = 0x876f, [0x39fb] = 0x8771, [0x39fc] = 0x8772, [0x39fd] = 0x8773,
  [0x39fe] = 0x8775, [0x3a00] = 0x8777, [0x3a01] = 0x8778, [0x3a02] = 0x8779,
  [0x3a03] = 0x877a, [0x3a04] = 0x877f, [0x3a05] = 0x8780, [0x3a06] = 0x8781,
  [0x3a07] = 0x8784, [0x3a08] = 0x8786, [0x3a09] = 0x8787, [0x3a0a] = 0x8789,
  [0x3a0b] = 0x878a, [0x3a0c] = 0x878c, [0x3a0d] = 0x878e, [0x3a0e] = 0x878f,
  [0x3a0f] = 0x8790, [0x3a10] = 0x8791, [0x3a11] = 0x8792, [0x3a12] = 0x8794,
  [0x3a13] = 0x8795, [0x3a14] = 0x8796, [0x3a15] = 0x8798, [0x3a16] = 0x8799,
  [0x3a17] = 0x879a, [0x3a18] = 0x879b, [0x3a19] = 0x879c, [0x3a1a] = 0x879d,
  [0x3a1b] = 0x879e, [0x3a1c] = 0x87a0, [0x3a1d] = 0x87a1, [0x3a1e] = 0x87a2,
  [0x3a1f] = 0x87a3, [0x3a20] = 0x87a4, [0x3a21] = 0x5dcd, [0x3a22] = 0x5fae,
  [0x3a23] = 0x5371, [0x3a24] = 0x97e6, [0x3a25] = 0x8fdd, [0x3a26] = 0x6845,
  [0x3a27] = 0x56f4, [0x3a28] = 0x552f, [0x3a29] = 0x60df, [0x3a2a] = 0x4e3a,
  [0x3a2b] = 0x6f4d, [0x3a2c] = 0x7ef4, [0x3a2d] = 0x82c7, [0x3a2e] = 0x840e,
  [0x3a2f] = 0x59d4, [0x3a30] = 0x4f1f, [0x3a31] = 0x4f2a, [0x3a32] = 0x5c3e,
  [0x3a33] = 0x7eac, [0x3a34] = 0x672a, [0x3a35] = 0x851a, [0x3a36] = 0x5473,
  [0x3a37] = 0x754f, [0x3a38] = 0x80c3, [0x3a39] = 0x5582, [0x3a3a] = 0x9b4f,
  [0x3a3b] = 0x4f4d, [0x3a3c] = 0x6e2d, [0x3a3d] = 0x8c13, [0x3a3e] = 0x5c09,
  [0x3a3f] = 0x6170, [0x3a40] = 0x536b, [0x3a41] = 0x761f, [0x3a42] = 0x6e29,
  [0x3a43] = 0x868a, [0x3a44] = 0x6587, [0x3a45] = 0x95fb, [0x3a46] = 0x7eb9,
  [0x3a47] = 0x543b, [0x3a48] = 0x7a33, [0x3a49] = 0x7d0a, [0x3a4a] = 0x95ee,
  [0x3a4b] = 0x55e1, [0x3a4c] = 0x7fc1, [0x3a4d] = 0x74ee, [0x3a4e] = 0x631d,
  [0x3a4f] = 0x8717, [0x3a50] = 0x6da1, [0x3a51] = 0x7a9d, [0x3a52] = 0x6211,
  [0x3a53] = 0x65a1, [0x3a54] = 0x5367, [0x3a55] = 0x63e1, [0x3a56] = 0x6c83,
  [0x3a57] = 0x5deb, [0x3a58] = 0x545c, [0x3a59] = 0x94a8, [0x3a5a] = 0x4e4c,
  [0x3a5b] = 0x6c61, [0x3a5c] = 0x8bec, [0x3a5d] = 0x5c4b, [0x3a5e] = 0x65e0,
  [0x3a5f] = 0x829c, [0x3a60] = 0x68a7, [0x3a61] = 0x543e, [0x3a62] = 0x5434,
  [0x3a63] = 0x6bcb, [0x3a64] = 0x6b66, [0x3a65] = 0x4e94, [0x3a66] = 0x6342,
  [0x3a67] = 0x5348, [0x3a68] = 0x821e, [0x3a69] = 0x4f0d, [0x3a6a] = 0x4fae,
  [0x3a6b] = 0x575e, [0x3a6c] = 0x620a, [0x3a6d] = 0x96fe, [0x3a6e] = 0x6664,
  [0x3a6f] = 0x7269, [0x3a70] = 0x52ff, [0x3a71] = 0x52a1, [0x3a72] = 0x609f,
  [0x3a73] = 0x8bef, [0x3a74] = 0x6614, [0x3a75] = 0x7199, [0x3a76] = 0x6790,
  [0x3a77] = 0x897f, [0x3a78] = 0x7852, [0x3a79] = 0x77fd, [0x3a7a] = 0x6670,
  [0x3a7b] = 0x563b, [0x3a7c] = 0x5438, [0x3a7d] = 0x9521, [0x3a7e] = 0x727a,
  [0x3a80] = 0x87a5, [0x3a81] = 0x87a6, [0x3a82] = 0x87a7, [0x3a83] = 0x87a9,
  [0x3a84] = 0x87aa, [0x3a85] = 0x87ae, [0x3a86] = 0x87b0, [0x3a87] = 0x87b1,
  [0x3a88] = 0x87b2, [0x3a89] = 0x87b4, [0x3a8a] = 0x87b6, [0x3a8b] = 0x87b7,
  [0x3a8c] = 0x87b8, [0x3a8d] = 0x87b9, [0x3a8e] = 0x87bb, [0x3a8f] = 0x87bc,
  [0x3a90] = 0x87be, [0x3a91] = 0x87bf, [0x3a92] = 0x87c1, [0x3a93] = 0x87c2,
  [0x3a94] = 0x87c3, [0x3a95] = 0x87c4, [0x3a96] = 0x87c5, [0x3a97] = 0x87c7,
  [0x3a98] = 0x87c8, [0x3a99] = 0x87c9, [0x3a9a] = 0x87cc, [0x3a9b] = 0x87cd,
  [0x3a9c] = 0x87ce, [0x3a9d] = 0x87cf, [0x3a9e] = 0x87d0, [0x3a9f] = 0x87d4,
  [0x3aa0] = 0x87d5, [0x3aa1] = 0x87d6, [0x3aa2] = 0x87d7, [0x3aa3] = 0x87d8,
  [0x3aa4] = 0x87d9, [0x3aa5] = 0x87da, [0x3aa6] = 0x87dc, [0x3aa7] = 0x87dd,
  [0x3aa8] = 0x87de, [0x3aa9] = 0x87df, [0x3aaa] = 0x87e1, [0x3aab] = 0x87e2,
  [0x3aac] = 0x87e3, [0x3aad] = 0x87e4, [0x3aae] = 0x87e6, [0x3aaf] = 0x87e7,
  [0x3ab0] = 0x87e8, [0x3ab1] = 0x87e9, [0x3ab2] = 0x87eb, [0x3ab3] = 0x87ec,
  [0x3ab4] = 0x87ed, [0x3ab5] = 0x87ef, [0x3ab6] = 0x87f0, [0x3ab7] = 0x87f1,
  [0x3ab8] = 0x87f2, [0x3ab9] = 0x87f3, [0x3aba] = 0x87f4, [0x3abb] = 0x87f5,
  [0x3abc] = 0x87f6, [0x3abd] = 0x87f7, [0x3abe] = 0x87f8, [0x3ac0] = 0x87fa,
  [0x3ac1] = 0x87fb, [0x3ac2] = 0x87fc, [0x3ac3] = 0x87fd, [0x3ac4] = 0x87ff,
  [0x3ac5] = 0x8800, [0x3ac6] = 0x8801, [0x3ac7] = 0x8802, [0x3ac8] = 0x8804,
  [0x3ac9] = 0x8805, [0x3aca] = 0x8806, [0x3acb] = 0x8807, [0x3acc] = 0x8808,
  [0x3acd] = 0x8809, [0x3ace] = 0x880b, [0x3acf] = 0x880c, [0x3ad0] = 0x880d,
  [0x3ad1] = 0x880e, [0x3ad2] = 0x880f, [0x3ad3] = 0x8810, [0x3ad4] = 0x8811,
  [0x3ad5] = 0x8812, [0x3ad6] = 0x8814, [0x3ad7] = 0x8817, [0x3ad8] = 0x8818,
  [0x3ad9] = 0x8819, [0x3ada] = 0x881a, [0x3adb] = 0x881c, [0x3adc] = 0x881d,
  [0x3add] = 0x881e, [0x3ade] = 0x881f, [0x3adf] = 0x8820, [0x3ae0] = 0x8823,
  [0x3ae1] = 0x7a00, [0x3ae2] = 0x606f, [0x3ae3] = 0x5e0c, [0x3ae4] = 0x6089,
  [0x3ae5] = 0x819d, [0x3ae6] = 0x5915, [0x3ae7] = 0x60dc, [0x3ae8] = 0x7184,
  [0x3ae9] = 0x70ef, [0x3aea] = 0x6eaa, [0x3aeb] = 0x6c50, [0x3aec] = 0x7280,
  [0x3aed] = 0x6a84, [0x3aee] = 0x88ad, [0x3aef] = 0x5e2d, [0x3af0] = 0x4e60,
  [0x3af1] = 0x5ab3, [0x3af2] = 0x559c, [0x3af3] = 0x94e3, [0x3af4] = 0x6d17,
  [0x3af5] = 0x7cfb, [0x3af6] = 0x9699, [0x3af7] = 0x620f, [0x3af8] = 0x7ec6,
  [0x3af9] = 0x778e, [0x3afa] = 0x867e, [0x3afb] = 0x5323, [0x3afc] = 0x971e,
  [0x3afd] = 0x8f96, [0x3afe] = 0x6687, [0x3aff] = 0x5ce1, [0x3b00] = 0x4fa0,
  [0x3b01] = 0x72ed, [0x3b02] = 0x4e0b, [0x3b03] = 0x53a6, [0x3b04] = 0x590f,
  [0x3b05] = 0x5413, [0x3b06] = 0x6380, [0x3b07] = 0x9528, [0x3b08] = 0x5148,
  [0x3b09] = 0x4ed9, [0x3b0a] = 0x9c9c, [0x3b0b] = 0x7ea4, [0x3b0c] = 0x54b8,
  [0x3b0d] = 0x8d24, [0x3b0e] = 0x8854, [0x3b0f] = 0x8237, [0x3b10] = 0x95f2,
  [0x3b11] = 0x6d8e, [0x3b12] = 0x5f26, [0x3b13] = 0x5acc, [0x3b14] = 0x663e,
  [0x3b15] = 0x9669, [0x3b16] = 0x73b0, [0x3b17] = 0x732e, [0x3b18] = 0x53bf,
  [0x3b19] = 0x817a, [0x3b1a] = 0x9985, [0x3b1b] = 0x7fa1, [0x3b1c] = 0x5baa,
  [0x3b1d] = 0x9677, [0x3b1e] = 0x9650, [0x3b1f] = 0x7ebf, [0x3b20] = 0x76f8,
  [0x3b21] = 0x53a2, [0x3b22] = 0x9576, [0x3b23] = 0x9999, [0x3b24] = 0x7bb1,
  [0x3b25] = 0x8944, [0x3b26] = 0x6e58, [0x3b27] = 0x4e61, [0x3b28] = 0x7fd4,
  [0x3b29] = 0x7965, [0x3b2a] = 0x8be6, [0x3b2b] = 0x60f3, [0x3b2c] = 0x54cd,
  [0x3b2d] = 0x4eab, [0x3b2e] = 0x9879, [0x3b2f] = 0x5df7, [0x3b30] = 0x6a61,
  [0x3b31] = 0x50cf, [0x3b32] = 0x5411, [0x3b33] = 0x8c61, [0x3b34] = 0x8427,
  [0x3b35] = 0x785d, [0x3b36] = 0x9704, [0x3b37] = 0x524a, [0x3b38] = 0x54ee,
  [0x3b39] = 0x56a3, [0x3b3a] = 0x9500, [0x3b3b] = 0x6d88, [0x3b3c] = 0x5bb5,
  [0x3b3d] = 0x6dc6, [0x3b3e] = 0x6653, [0x3b40] = 0x8824, [0x3b41] = 0x8825,
  [0x3b42] = 0x8826, [0x3b43] = 0x8827, [0x3b44] = 0x8828, [0x3b45] = 0x8829,
  [0x3b46] = 0x882a, [0x3b47] = 0x882b, [0x3b48] = 0x882c, [0x3b49] = 0x882d,
  [0x3b4a] = 0x882e, [0x3b4b] = 0x882f, [0x3b4c] = 0x8830, [0x3b4d] = 0x8831,
  [0x3b4e] = 0x8833, [0x3b4f] = 0x8834, [0x3b50] = 0x8835, [0x3b51] = 0x8836,
  [0x3b52] = 0x8837, [0x3b53] = 0x8838, [0x3b54] = 0x883a, [0x3b55] = 0x883b,
  [0x3b56] = 0x883d, [0x3b57] = 0x883e, [0x3b58] = 0x883f, [0x3b59] = 0x8841,
  [0x3b5a] = 0x8842, [0x3b5b] = 0x8843, [0x3b5c] = 0x8846, [0x3b5d] = 0x8847,
  [0x3b5e] = 0x8848, [0x3b5f] = 0x8849, [0x3b60] = 0x884a, [0x3b61] = 0x884b,
  [0x3b62] = 0x884e, [0x3b63] = 0x884f, [0x3b64] = 0x8850, [0x3b65] = 0x8851,
  [0x3b66] = 0x8852, [0x3b67] = 0x8853, [0x3b68] = 0x8855, [0x3b69] = 0x8856,
  [0x3b6a] = 0x8858, [0x3b6b] = 0x885a, [0x3b6c] = 0x885b, [0x3b6d] = 0x885c,
  [0x3b6e] = 0x885d, [0x3b6f] = 0x885e, [0x3b70] = 0x885f, [0x3b71] = 0x8860,
  [0x3b72] = 0x8866, [0x3b73] = 0x8867, [0x3b74] = 0x886a, [0x3b75] = 0x886d,
  [0x3b76] = 0x886f, [0x3b77] = 0x8871, [0x3b78] = 0x8873, [0x3b79] = 0x8874,
  [0x3b7a] = 0x8875, [0x3b7b] = 0x8876, [0x3b7c] = 0x8878, [0x3b7d] = 0x8879,
  [0x3b7e] = 0x887a, [0x3b80] = 0x887b, [0x3b81] = 0x887c, [0x3b82] = 0x8880,
  [0x3b83] = 0x8883, [0x3b84] = 0x8886, [0x3b85] = 0x8887, [0x3b86] = 0x8889,
  [0x3b87] = 0x888a, [0x3b88] = 0x888c, [0x3b89] = 0x888e, [0x3b8a] = 0x888f,
  [0x3b8b] = 0x8890, [0x3b8c] = 0x8891, [0x3b8d] = 0x8893, [0x3b8e] = 0x8894,
  [0x3b8f] = 0x8895, [0x3b90] = 0x8897, [0x3b91] = 0x8898, [0x3b92] = 0x8899,
  [0x3b93] = 0x889a, [0x3b94] = 0x889b, [0x3b95] = 0x889d, [0x3b96] = 0x889e,
  [0x3b97] = 0x889f, [0x3b98] = 0x88a0, [0x3b99] = 0x88a1, [0x3b9a] = 0x88a3,
  [0x3b9b] = 0x88a5, [0x3b9c] = 0x88a6, [0x3b9d] = 0x88a7, [0x3b9e] = 0x88a8,
  [0x3b9f] = 0x88a9, [0x3ba0] = 0x88aa, [0x3ba1] = 0x5c0f, [0x3ba2] = 0x5b5d,
  [0x3ba3] = 0x6821, [0x3ba4] = 0x8096, [0x3ba5] = 0x5578, [0x3ba6] = 0x7b11,
  [0x3ba7] = 0x6548, [0x3ba8] = 0x6954, [0x3ba9] = 0x4e9b, [0x3baa] = 0x6b47,
  [0x3bab] = 0x874e, [0x3bac] = 0x978b, [0x3bad] = 0x534f, [0x3bae] = 0x631f,
  [0x3baf] = 0x643a, [0x3bb0] = 0x90aa, [0x3bb1] = 0x659c, [0x3bb2] = 0x80c1,
  [0x3bb3] = 0x8c10, [0x3bb4] = 0x5199, [0x3bb5] = 0x68b0, [0x3bb6] = 0x5378,
  [0x3bb7] = 0x87f9, [0x3bb8] = 0x61c8, [0x3bb9] = 0x6cc4, [0x3bba] = 0x6cfb,
  [0x3bbb] = 0x8c22, [0x3bbc] = 0x5c51, [0x3bbd] = 0x85aa, [0x3bbe] = 0x82af,
  [0x3bbf] = 0x950c, [0x3bc0] = 0x6b23, [0x3bc1] = 0x8f9b, [0x3bc2] = 0x65b0,
  [0x3bc3] = 0x5ffb, [0x3bc4] = 0x5fc3, [0x3bc5] = 0x4fe1, [0x3bc6] = 0x8845,
  [0x3bc7] = 0x661f, [0x3bc8] = 0x8165, [0x3bc9] = 0x7329, [0x3bca] = 0x60fa,
  [0x3bcb] = 0x5174, [0x3bcc] = 0x5211, [0x3bcd] = 0x578b, [0x3bce] = 0x5f62,
  [0x3bcf] = 0x90a2, [0x3bd0] = 0x884c, [0x3bd1] = 0x9192, [0x3bd2] = 0x5e78,
  [0x3bd3] = 0x674f, [0x3bd4] = 0x6027, [0x3bd5] = 0x59d3, [0x3bd6] = 0x5144,
  [0x3bd7] = 0x51f6, [0x3bd8] = 0x80f8, [0x3bd9] = 0x5308, [0x3bda] = 0x6c79,
  [0x3bdb] = 0x96c4, [0x3bdc] = 0x718a, [0x3bdd] = 0x4f11, [0x3bde] = 0x4fee,
  [0x3bdf] = 0x7f9e, [0x3be0] = 0x673d, [0x3be1] = 0x55c5, [0x3be2] = 0x9508,
  [0x3be3] = 0x79c0, [0x3be4] = 0x8896, [0x3be5] = 0x7ee3, [0x3be6] = 0x589f,
  [0x3be7] = 0x620c, [0x3be8] = 0x9700, [0x3be9] = 0x865a, [0x3bea] = 0x5618,
  [0x3beb] = 0x987b, [0x3bec] = 0x5f90, [0x3bed] = 0x8bb8, [0x3bee] = 0x84c4,
  [0x3bef] = 0x9157, [0x3bf0] = 0x53d9, [0x3bf1] = 0x65ed, [0x3bf2] = 0x5e8f,
  [0x3bf3] = 0x755c, [0x3bf4] = 0x6064, [0x3bf5] = 0x7d6e, [0x3bf6] = 0x5a7f,
  [0x3bf7] = 0x7eea, [0x3bf8] = 0x7eed, [0x3bf9] = 0x8f69, [0x3bfa] = 0x55a7,
  [0x3bfb] = 0x5ba3, [0x3bfc] = 0x60ac, [0x3bfd] = 0x65cb, [0x3bfe] = 0x7384,
  [0x3c00] = 0x88ac, [0x3c01] = 0x88ae, [0x3c02] = 0x88af, [0x3c03] = 0x88b0,
  [0x3c04] = 0x88b2, [0x3c05] = 0x88b3, [0x3c06] = 0x88b4, [0x3c07] = 0x88b5,
  [0x3c08] = 0x88b6, [0x3c09] = 0x88b8, [0x3c0a] = 0x88b9, [0x3c0b] = 0x88ba,
  [0x3c0c] = 0x88bb, [0x3c0d] = 0x88bd, [0x3c0e] = 0x88be, [0x3c0f] = 0x88bf,
  [0x3c10] = 0x88c0, [0x3c11] = 0x88c3, [0x3c12] = 0x88c4, [0x3c13] = 0x88c7,
  [0x3c14] = 0x88c8, [0x3c15] = 0x88ca, [0x3c16] = 0x88cb, [0x3c17] = 0x88cc,
  [0x3c18] = 0x88cd, [0x3c19] = 0x88cf, [0x3c1a] = 0x88d0, [0x3c1b] = 0x88d1,
  [0x3c1c] = 0x88d3, [0x3c1d] = 0x88d6, [0x3c1e] = 0x88d7, [0x3c1f] = 0x88da,
  [0x3c20] = 0x88db, [0x3c21] = 0x88dc, [0x3c22] = 0x88dd, [0x3c23] = 0x88de,
  [0x3c24] = 0x88e0, [0x3c25] = 0x88e1, [0x3c26] = 0x88e6, [0x3c27] = 0x88e7,
  [0x3c28] = 0x88e9, [0x3c29] = 0x88ea, [0x3c2a] = 0x88eb, [0x3c2b] = 0x88ec,
  [0x3c2c] = 0x88ed, [0x3c2d] = 0x88ee, [0x3c2e] = 0x88ef, [0x3c2f] = 0x88f2,
  [0x3c30] = 0x88f5, [0x3c31] = 0x88f6, [0x3c32] = 0x88f7, [0x3c33] = 0x88fa,
  [0x3c34] = 0x88fb, [0x3c35] = 0x88fd, [0x3c36] = 0x88ff, [0x3c37] = 0x8900,
  [0x3c38] = 0x8901, [0x3c39] = 0x8903, [0x3c3a] = 0x8904, [0x3c3b] = 0x8905,
  [0x3c3c] = 0x8906, [0x3c3d] = 0x8907, [0x3c3e] = 0x8908, [0x3c40] = 0x8909,
  [0x3c41] = 0x890b, [0x3c42] = 0x890c, [0x3c43] = 0x890d, [0x3c44] = 0x890e,
  [0x3c45] = 0x890f, [0x3c46] = 0x8911, [0x3c47] = 0x8914, [0x3c48] = 0x8915,
  [0x3c49] = 0x8916, [0x3c4a] = 0x8917, [0x3c4b] = 0x8918, [0x3c4c] = 0x891c,
  [0x3c4d] = 0x891d, [0x3c4e] = 0x891e, [0x3c4f] = 0x891f, [0x3c50] = 0x8920,
  [0x3c51] = 0x8922, [0x3c52] = 0x8923, [0x3c53] = 0x8924, [0x3c54] = 0x8926,
  [0x3c55] = 0x8927, [0x3c56] = 0x8928, [0x3c57] = 0x8929, [0x3c58] = 0x892c,
  [0x3c59] = 0x892d, [0x3c5a] = 0x892e, [0x3c5b] = 0x892f, [0x3c5c] = 0x8931,
  [0x3c5d] = 0x8932, [0x3c5e] = 0x8933, [0x3c5f] = 0x8935, [0x3c60] = 0x8937,
  [0x3c61] = 0x9009, [0x3c62] = 0x7663, [0x3c63] = 0x7729, [0x3c64] = 0x7eda,
  [0x3c65] = 0x9774, [0x3c66] = 0x859b, [0x3c67] = 0x5b66, [0x3c68] = 0x7a74,
  [0x3c69] = 0x96ea, [0x3c6a] = 0x8840, [0x3c6b] = 0x52cb, [0x3c6c] = 0x718f,
  [0x3c6d] = 0x5faa, [0x3c6e] = 0x65ec, [0x3c6f] = 0x8be2, [0x3c70] = 0x5bfb,
  [0x3c71] = 0x9a6f, [0x3c72] = 0x5de1, [0x3c73] = 0x6b89, [0x3c74] = 0x6c5b,
  [0x3c75] = 0x8bad, [0x3c76] = 0x8baf, [0x3c77] = 0x900a, [0x3c78] = 0x8fc5,
  [0x3c79] = 0x538b, [0x3c7a] = 0x62bc, [0x3c7b] = 0x9e26, [0x3c7c] = 0x9e2d,
  [0x3c7d] = 0x5440, [0x3c7e] = 0x4e2b, [0x3c7f] = 0x82bd, [0x3c80] = 0x7259,
  [0x3c81] = 0x869c, [0x3c82] = 0x5d16, [0x3c83] = 0x8859, [0x3c84] = 0x6daf,
  [0x3c85] = 0x96c5, [0x3c86] = 0x54d1, [0x3c87] = 0x4e9a, [0x3c88] = 0x8bb6,
  [0x3c89] = 0x7109, [0x3c8a] = 0x54bd, [0x3c8b] = 0x9609, [0x3c8c] = 0x70df,
  [0x3c8d] = 0x6df9, [0x3c8e] = 0x76d0, [0x3c8f] = 0x4e25, [0x3c90] = 0x7814,
  [0x3c91] = 0x8712, [0x3c92] = 0x5ca9, [0x3c93] = 0x5ef6, [0x3c94] = 0x8a00,
  [0x3c95] = 0x989c, [0x3c96] = 0x960e, [0x3c97] = 0x708e, [0x3c98] = 0x6cbf,
  [0x3c99] = 0x5944, [0x3c9a] = 0x63a9, [0x3c9b] = 0x773c, [0x3c9c] = 0x884d,
  [0x3c9d] = 0x6f14, [0x3c9e] = 0x8273, [0x3c9f] = 0x5830, [0x3ca0] = 0x71d5,
  [0x3ca1] = 0x538c, [0x3ca2] = 0x781a, [0x3ca3] = 0x96c1, [0x3ca4] = 0x5501,
  [0x3ca5] = 0x5f66, [0x3ca6] = 0x7130, [0x3ca7] = 0x5bb4, [0x3ca8] = 0x8c1a,
  [0x3ca9] = 0x9a8c, [0x3caa] = 0x6b83, [0x3cab] = 0x592e, [0x3cac] = 0x9e2f,
  [0x3cad] = 0x79e7, [0x3cae] = 0x6768, [0x3caf] = 0x626c, [0x3cb0] = 0x4f6f,
  [0x3cb1] = 0x75a1, [0x3cb2] = 0x7f8a, [0x3cb3] = 0x6d0b, [0x3cb4] = 0x9633,
  [0x3cb5] = 0x6c27, [0x3cb6] = 0x4ef0, [0x3cb7] = 0x75d2, [0x3cb8] = 0x517b,
  [0x3cb9] = 0x6837, [0x3cba] = 0x6f3e, [0x3cbb] = 0x9080, [0x3cbc] = 0x8170,
  [0x3cbd] = 0x5996, [0x3cbe] = 0x7476, [0x3cc0] = 0x8938, [0x3cc1] = 0x8939,
  [0x3cc2] = 0x893a, [0x3cc3] = 0x893b, [0x3cc4] = 0x893c, [0x3cc5] = 0x893d,
  [0x3cc6] = 0x893e, [0x3cc7] = 0x893f, [0x3cc8] = 0x8940, [0x3cc9] = 0x8942,
  [0x3cca] = 0x8943, [0x3ccb] = 0x8945, [0x3ccc] = 0x8946, [0x3ccd] = 0x8947,
  [0x3cce] = 0x8948, [0x3ccf] = 0x8949, [0x3cd0] = 0x894a, [0x3cd1] = 0x894b,
  [0x3cd2] = 0x894c, [0x3cd3] = 0x894d, [0x3cd4] = 0x894e, [0x3cd5] = 0x894f,
  [0x3cd6] = 0x8950, [0x3cd7] = 0x8951, [0x3cd8] = 0x8952, [0x3cd9] = 0x8953,
  [0x3cda] = 0x8954, [0x3cdb] = 0x8955, [0x3cdc] = 0x8956, [0x3cdd] = 0x8957,
  [0x3cde] = 0x8958, [0x3cdf] = 0x8959, [0x3ce0] = 0x895a, [0x3ce1] = 0x895b,
  [0x3ce2] = 0x895c, [0x3ce3] = 0x895d, [0x3ce4] = 0x8960, [0x3ce5] = 0x8961,
  [0x3ce6] = 0x8962, [0x3ce7] = 0x8963, [0x3ce8] = 0x8964, [0x3ce9] = 0x8965,
  [0x3cea] = 0x8967, [0x3ceb] = 0x8968, [0x3cec] = 0x8969, [0x3ced] = 0x896a,
  [0x3cee] = 0x896b, [0x3cef] = 0x896c, [0x3cf0] = 0x896d, [0x3cf1] = 0x896e,
  [0x3cf2] = 0x896f, [0x3cf3] = 0x8970, [0x3cf4] = 0x8971, [0x3cf5] = 0x8972,
  [0x3cf6] = 0x8973, [0x3cf7] = 0x8974, [0x3cf8] = 0x8975, [0x3cf9] = 0x8976,
  [0x3cfa] = 0x8977, [0x3cfb] = 0x8978, [0x3cfc] = 0x8979, [0x3cfd] = 0x897a,
  [0x3cfe] = 0x897c, [0x3d00] = 0x897d, [0x3d01] = 0x897e, [0x3d02] = 0x8980,
  [0x3d03] = 0x8982, [0x3d04] = 0x8984, [0x3d05] = 0x8985, [0x3d06] = 0x8987,
  [0x3d07] = 0x8988, [0x3d08] = 0x8989, [0x3d09] = 0x898a, [0x3d0a] = 0x898b,
  [0x3d0b] = 0x898c, [0x3d0c] = 0x898d, [0x3d0d] = 0x898e, [0x3d0e] = 0x898f,
  [0x3d0f] = 0x8990, [0x3d10] = 0x8991, [0x3d11] = 0x8992, [0x3d12] = 0x8993,
  [0x3d13] = 0x8994, [0x3d14] = 0x8995, [0x3d15] = 0x8996, [0x3d16] = 0x8997,
  [0x3d17] = 0x8998, [0x3d18] = 0x8999, [0x3d19] = 0x899a, [0x3d1a] = 0x899b,
  [0x3d1b] = 0x899c, [0x3d1c] = 0x899d, [0x3d1d] = 0x899e, [0x3d1e] = 0x899f,
  [0x3d1f] = 0x89a0, [0x3d20] = 0x89a1, [0x3d21] = 0x6447, [0x3d22] = 0x5c27,
  [0x3d23] = 0x9065, [0x3d24] = 0x7a91, [0x3d25] = 0x8c23, [0x3d26] = 0x59da,
  [0x3d27] = 0x54ac, [0x3d28] = 0x8200, [0x3d29] = 0x836f, [0x3d2a] = 0x8981,
  [0x3d2b] = 0x8000, [0x3d2c] = 0x6930, [0x3d2d] = 0x564e, [0x3d2e] = 0x8036,
  [0x3d2f] = 0x7237, [0x3d30] = 0x91ce, [0x3d31] = 0x51b6, [0x3d32] = 0x4e5f,
  [0x3d33] = 0x9875, [0x3d34] = 0x6396, [0x3d35] = 0x4e1a, [0x3d36] = 0x53f6,
  [0x3d37] = 0x66f3, [0x3d38] = 0x814b, [0x3d39] = 0x591c, [0x3d3a] = 0x6db2,
  [0x3d3b] = 0x4e00, [0x3d3c] = 0x58f9, [0x3d3d] = 0x533b, [0x3d3e] = 0x63d6,
  [0x3d3f] = 0x94f1, [0x3d40] = 0x4f9d, [0x3d41] = 0x4f0a, [0x3d42] = 0x8863,
  [0x3d43] = 0x9890, [0x3d44] = 0x5937, [0x3d45] = 0x9057, [0x3d46] = 0x79fb,
  [0x3d47] = 0x4eea, [0x3d48] = 0x80f0, [0x3d49] = 0x7591, [0x3d4a] = 0x6c82,
  [0x3d4b] = 0x5b9c, [0x3d4c] = 0x59e8, [0x3d4d] = 0x5f5d, [0x3d4e] = 0x6905,
  [0x3d4f] = 0x8681, [0x3d50] = 0x501a, [0x3d51] = 0x5df2, [0x3d52] = 0x4e59,
  [0x3d53] = 0x77e3, [0x3d54] = 0x4ee5, [0x3d55] = 0x827a, [0x3d56] = 0x6291,
  [0x3d57] = 0x6613, [0x3d58] = 0x9091, [0x3d59] = 0x5c79, [0x3d5a] = 0x4ebf,
  [0x3d5b] = 0x5f79, [0x3d5c] = 0x81c6, [0x3d5d] = 0x9038, [0x3d5e] = 0x8084,
  [0x3d5f] = 0x75ab, [0x3d60] = 0x4ea6, [0x3d61] = 0x88d4, [0x3d62] = 0x610f,
  [0x3d63] = 0x6bc5, [0x3d64] = 0x5fc6, [0x3d65] = 0x4e49, [0x3d66] = 0x76ca,
  [0x3d67] = 0x6ea2, [0x3d68] = 0x8be3, [0x3d69] = 0x8bae, [0x3d6a] = 0x8c0a,
  [0x3d6b] = 0x8bd1, [0x3d6c] = 0x5f02, [0x3d6d] = 0x7ffc, [0x3d6e] = 0x7fcc,
  [0x3d6f] = 0x7ece, [0x3d70] = 0x8335, [0x3d71] = 0x836b, [0x3d72] = 0x56e0,
  [0x3d73] = 0x6bb7, [0x3d74] = 0x97f3, [0x3d75] = 0x9634, [0x3d76] = 0x59fb,
  [0x3d77] = 0x541f, [0x3d78] = 0x94f6, [0x3d79] = 0x6deb, [0x3d7a] = 0x5bc5,
  [0x3d7b] = 0x996e, [0x3d7c] = 0x5c39, [0x3d7d] = 0x5f15, [0x3d7e] = 0x9690,
  [0x3d80] = 0x89a2, [0x3d81] = 0x89a3, [0x3d82] = 0x89a4, [0x3d83] = 0x89a5,
  [0x3d84] = 0x89a6, [0x3d85] = 0x89a7, [0x3d86] = 0x89a8, [0x3d87] = 0x89a9,
  [0x3d88] = 0x89aa, [0x3d89] = 0x89ab, [0x3d8a] = 0x89ac, [0x3d8b] = 0x89ad,
  [0x3d8c] = 0x89ae, [0x3d8d] = 0x89af, [0x3d8e] = 0x89b0, [0x3d8f] = 0x89b1,
  [0x3d90] = 0x89b2, [0x3d91] = 0x89b3, [0x3d92] = 0x89b4, [0x3d93] = 0x89b5,
  [0x3d94] = 0x89b6, [0x3d95] = 0x89b7, [0x3d96] = 0x89b8, [0x3d97] = 0x89b9,
  [0x3d98] = 0x89ba, [0x3d99] = 0x89bb, [0x3d9a] = 0x89bc, [0x3d9b] = 0x89bd,
  [0x3d9c] = 0x89be, [0x3d9d] = 0x89bf, [0x3d9e] = 0x89c0, [0x3d9f] = 0x89c3,
  [0x3da0] = 0x89cd, [0x3da1] = 0x89d3, [0x3da2] = 0x89d4, [0x3da3] = 0x89d5,
  [0x3da4] = 0x89d7, [0x3da5] = 0x89d8, [0x3da6] = 0x89d9, [0x3da7] = 0x89db,
  [0x3da8] = 0x89dd, [0x3da9] = 0x89df, [0x3daa] = 0x89e0, [0x3dab] = 0x89e1,
  [0x3dac] = 0x89e2, [0x3dad] = 0x89e4, [0x3dae] = 0x89e7, [0x3daf] = 0x89e8,
  [0x3db0] = 0x89e9, [0x3db1] = 0x89ea, [0x3db2] = 0x89ec, [0x3db3] = 0x89ed,
  [0x3db4] = 0x89ee, [0x3db5] = 0x89f0, [0x3db6] = 0x89f1, [0x3db7] = 0x89f2,
  [0x3db8] = 0x89f4, [0x3db9] = 0x89f5, [0x3dba] = 0x89f6, [0x3dbb] = 0x89f7,
  [0x3dbc] = 0x89f8, [0x3dbd] = 0x89f9, [0x3dbe] = 0x89fa, [0x3dc0] = 0x89fb,
  [0x3dc1] = 0x89fc, [0x3dc2] = 0x89fd, [0x3dc3] = 0x89fe, [0x3dc4] = 0x89ff,
  [0x3dc5] = 0x8a01, [0x3dc6] = 0x8a02, [0x3dc7] = 0x8a03, [0x3dc8] = 0x8a04,
  [0x3dc9] = 0x8a05, [0x3dca] = 0x8a06, [0x3dcb] = 0x8a08, [0x3dcc] = 0x8a09,
  [0x3dcd] = 0x8a0a, [0x3dce] = 0x8a0b, [0x3dcf] = 0x8a0c, [0x3dd0] = 0x8a0d,
  [0x3dd1] = 0x8a0e, [0x3dd2] = 0x8a0f, [0x3dd3] = 0x8a10, [0x3dd4] = 0x8a11,
  [0x3dd5] = 0x8a12, [0x3dd6] = 0x8a13, [0x3dd7] = 0x8a14, [0x3dd8] = 0x8a15,
  [0x3dd9] = 0x8a16, [0x3dda] = 0x8a17, [0x3ddb] = 0x8a18, [0x3ddc] = 0x8a19,
  [0x3ddd] = 0x8a1a, [0x3dde] = 0x8a1b, [0x3ddf] = 0x8a1c, [0x3de0] = 0x8a1d,
  [0x3de1] = 0x5370, [0x3de2] = 0x82f1, [0x3de3] = 0x6a31, [0x3de4] = 0x5a74,
  [0x3de5] = 0x9e70, [0x3de6] = 0x5e94, [0x3de7] = 0x7f28, [0x3de8] = 0x83b9,
  [0x3de9] = 0x8424, [0x3dea] = 0x8425, [0x3deb] = 0x8367, [0x3dec] = 0x8747,
  [0x3ded] = 0x8fce, [0x3dee] = 0x8d62, [0x3def] = 0x76c8, [0x3df0] = 0x5f71,
  [0x3df1] = 0x9896, [0x3df2] = 0x786c, [0x3df3] = 0x6620, [0x3df4] = 0x54df,
  [0x3df5] = 0x62e5, [0x3df6] = 0x4f63, [0x3df7] = 0x81c3, [0x3df8] = 0x75c8,
  [0x3df9] = 0x5eb8, [0x3dfa] = 0x96cd, [0x3dfb] = 0x8e0a, [0x3dfc] = 0x86f9,
  [0x3dfd] = 0x548f, [0x3dfe] = 0x6cf3, [0x3dff] = 0x6d8c, [0x3e00] = 0x6c38,
  [0x3e01] = 0x607f, [0x3e02] = 0x52c7, [0x3e03] = 0x7528, [0x3e04] = 0x5e7d,
  [0x3e05] = 0x4f18, [0x3e06] = 0x60a0, [0x3e07] = 0x5fe7, [0x3e08] = 0x5c24,
  [0x3e09] = 0x7531, [0x3e0a] = 0x90ae, [0x3e0b] = 0x94c0, [0x3e0c] = 0x72b9,
  [0x3e0d] = 0x6cb9, [0x3e0e] = 0x6e38, [0x3e0f] = 0x9149, [0x3e10] = 0x6709,
  [0x3e11] = 0x53cb, [0x3e12] = 0x53f3, [0x3e13] = 0x4f51, [0x3e14] = 0x91c9,
  [0x3e15] = 0x8bf1, [0x3e16] = 0x53c8, [0x3e17] = 0x5e7c, [0x3e18] = 0x8fc2,
  [0x3e19] = 0x6de4, [0x3e1a] = 0x4e8e, [0x3e1b] = 0x76c2, [0x3e1c] = 0x6986,
  [0x3e1d] = 0x865e, [0x3e1e] = 0x611a, [0x3e1f] = 0x8206, [0x3e20] = 0x4f59,
  [0x3e21] = 0x4fde, [0x3e22] = 0x903e, [0x3e23] = 0x9c7c, [0x3e24] = 0x6109,
  [0x3e25] = 0x6e1d, [0x3e26] = 0x6e14, [0x3e27] = 0x9685, [0x3e28] = 0x4e88,
  [0x3e29] = 0x5a31, [0x3e2a] = 0x96e8, [0x3e2b] = 0x4e0e, [0x3e2c] = 0x5c7f,
  [0x3e2d] = 0x79b9, [0x3e2e] = 0x5b87, [0x3e2f] = 0x8bed, [0x3e30] = 0x7fbd,
  [0x3e31] = 0x7389, [0x3e32] = 0x57df, [0x3e33] = 0x828b, [0x3e34] = 0x90c1,
  [0x3e35] = 0x5401, [0x3e36] = 0x9047, [0x3e37] = 0x55bb, [0x3e38] = 0x5cea,
  [0x3e39] = 0x5fa1, [0x3e3a] = 0x6108, [0x3e3b] = 0x6b32, [0x3e3c] = 0x72f1,
  [0x3e3d] = 0x80b2, [0x3e3e] = 0x8a89, [0x3e40] = 0x8a1e, [0x3e41] = 0x8a1f,
  [0x3e42] = 0x8a20, [0x3e43] = 0x8a21, [0x3e44] = 0x8a22, [0x3e45] = 0x8a23,
  [0x3e46] = 0x8a24, [0x3e47] = 0x8a25, [0x3e48] = 0x8a26, [0x3e49] = 0x8a27,
  [0x3e4a] = 0x8a28, [0x3e4b] = 0x8a29, [0x3e4c] = 0x8a2a, [0x3e4d] = 0x8a2b,
  [0x3e4e] = 0x8a2c, [0x3e4f] = 0x8a2d, [0x3e50] = 0x8a2e, [0x3e51] = 0x8a2f,
  [0x3e52] = 0x8a30, [0x3e53] = 0x8a31, [0x3e54] = 0x8a32, [0x3e55] = 0x8a33,
  [0x3e56] = 0x8a34, [0x3e57] = 0x8a35, [0x3e58] = 0x8a36, [0x3e59] = 0x8a37,
  [0x3e5a] = 0x8a38, [0x3e5b] = 0x8a39, [0x3e5c] = 0x8a3a, [0x3e5d] = 0x8a3b,
  [0x3e5e] = 0x8a3c, [0x3e5f] = 0x8a3d, [0x3e60] = 0x8a3f, [0x3e61] = 0x8a40,
  [0x3e62] = 0x8a41, [0x3e63] = 0x8a42, [0x3e64] = 0x8a43, [0x3e65] = 0x8a44,
  [0x3e66] = 0x8a45, [0x3e67] = 0x8a46, [0x3e68] = 0x8a47, [0x3e69] = 0x8a49,
  [0x3e6a] = 0x8a4a, [0x3e6b] = 0x8a4b, [0x3e6c] = 0x8a4c, [0x3e6d] = 0x8a4d,
  [0x3e6e] = 0x8a4e, [0x3e6f] = 0x8a4f, [0x3e70] = 0x8a50, [0x3e71] = 0x8a51,
  [0x3e72] = 0x8a52, [0x3e73] = 0x8a53, [0x3e74] = 0x8a54, [0x3e75] = 0x8a55,
  [0x3e76] = 0x8a56, [0x3e77] = 0x8a57, [0x3e78] = 0x8a58, [0x3e79] = 0x8a59,
  [0x3e7a] = 0x8a5a, [0x3e7b] = 0x8a5b, [0x3e7c] = 0x8a5c, [0x3e7d] = 0x8a5d,
  [0x3e7e] = 0x8a5e, [0x3e80] = 0x8a5f, [0x3e81] = 0x8a60, [0x3e82] = 0x8a61,
  [0x3e83] = 0x8a62, [0x3e84] = 0x8a63, [0x3e85] = 0x8a64, [0x3e86] = 0x8a65,
  [0x3e87] = 0x8a66, [0x3e88] = 0x8a67, [0x3e89] = 0x8a68, [0x3e8a] = 0x8a69,
  [0x3e8b] = 0x8a6a, [0x3e8c] = 0x8a6b, [0x3e8d] = 0x8a6c, [0x3e8e] = 0x8a6d,
  [0x3e8f] = 0x8a6e, [0x3e90] = 0x8a6f, [0x3e91] = 0x8a70, [0x3e92] = 0x8a71,
  [0x3e93] = 0x8a72, [0x3e94] = 0x8a73, [0x3e95] = 0x8a74, [0x3e96] = 0x8a75,
  [0x3e97] = 0x8a76, [0x3e98] = 0x8a77, [0x3e99] = 0x8a78, [0x3e9a] = 0x8a7a,
  [0x3e9b] = 0x8a7b, [0x3e9c] = 0x8a7c, [0x3e9d] = 0x8a7d, [0x3e9e] = 0x8a7e,
  [0x3e9f] = 0x8a7f, [0x3ea0] = 0x8a80, [0x3ea1] = 0x6d74, [0x3ea2] = 0x5bd3,
  [0x3ea3] = 0x88d5, [0x3ea4] = 0x9884, [0x3ea5] = 0x8c6b, [0x3ea6] = 0x9a6d,
  [0x3ea7] = 0x9e33, [0x3ea8] = 0x6e0a, [0x3ea9] = 0x51a4, [0x3eaa] = 0x5143,
  [0x3eab] = 0x57a3, [0x3eac] = 0x8881, [0x3ead] = 0x539f, [0x3eae] = 0x63f4,
  [0x3eaf] = 0x8f95, [0x3eb0] = 0x56ed, [0x3eb1] = 0x5458, [0x3eb2] = 0x5706,
  [0x3eb3] = 0x733f, [0x3eb4] = 0x6e90, [0x3eb5] = 0x7f18, [0x3eb6] = 0x8fdc,
  [0x3eb7] = 0x82d1, [0x3eb8] = 0x613f, [0x3eb9] = 0x6028, [0x3eba] = 0x9662,
  [0x3ebb] = 0x66f0, [0x3ebc] = 0x7ea6, [0x3ebd] = 0x8d8a, [0x3ebe] = 0x8dc3,
  [0x3ebf] = 0x94a5, [0x3ec0] = 0x5cb3, [0x3ec1] = 0x7ca4, [0x3ec2] = 0x6708,
  [0x3ec3] = 0x60a6, [0x3ec4] = 0x9605, [0x3ec5] = 0x8018, [0x3ec6] = 0x4e91,
  [0x3ec7] = 0x90e7, [0x3ec8] = 0x5300, [0x3ec9] = 0x9668, [0x3eca] = 0x5141,
  [0x3ecb] = 0x8fd0, [0x3ecc] = 0x8574, [0x3ecd] = 0x915d, [0x3ece] = 0x6655,
  [0x3ecf] = 0x97f5, [0x3ed0] = 0x5b55, [0x3ed1] = 0x531d, [0x3ed2] = 0x7838,
  [0x3ed3] = 0x6742, [0x3ed4] = 0x683d, [0x3ed5] = 0x54c9, [0x3ed6] = 0x707e,
  [0x3ed7] = 0x5bb0, [0x3ed8] = 0x8f7d, [0x3ed9] = 0x518d, [0x3eda] = 0x5728,
  [0x3edb] = 0x54b1, [0x3edc] = 0x6512, [0x3edd] = 0x6682, [0x3ede] = 0x8d5e,
  [0x3edf] = 0x8d43, [0x3ee0] = 0x810f, [0x3ee1] = 0x846c, [0x3ee2] = 0x906d,
  [0x3ee3] = 0x7cdf, [0x3ee4] = 0x51ff, [0x3ee5] = 0x85fb, [0x3ee6] = 0x67a3,
  [0x3ee7] = 0x65e9, [0x3ee8] = 0x6fa1, [0x3ee9] = 0x86a4, [0x3eea] = 0x8e81,
  [0x3eeb] = 0x566a, [0x3eec] = 0x9020, [0x3eed] = 0x7682, [0x3eee] = 0x7076,
  [0x3eef] = 0x71e5, [0x3ef0] = 0x8d23, [0x3ef1] = 0x62e9, [0x3ef2] = 0x5219,
  [0x3ef3] = 0x6cfd, [0x3ef4] = 0x8d3c, [0x3ef5] = 0x600e, [0x3ef6] = 0x589e,
  [0x3ef7] = 0x618e, [0x3ef8] = 0x66fe, [0x3ef9] = 0x8d60, [0x3efa] = 0x624e,
  [0x3efb] = 0x55b3, [0x3efc] = 0x6e23, [0x3efd] = 0x672d, [0x3efe] = 0x8f67,
  [0x3f00] = 0x8a81, [0x3f01] = 0x8a82, [0x3f02] = 0x8a83, [0x3f03] = 0x8a84,
  [0x3f04] = 0x8a85, [0x3f05] = 0x8a86, [0x3f06] = 0x8a87, [0x3f07] = 0x8a88,
  [0x3f08] = 0x8a8b, [0x3f09] = 0x8a8c, [0x3f0a] = 0x8a8d, [0x3f0b] = 0x8a8e,
  [0x3f0c] = 0x8a8f, [0x3f0d] = 0x8a90, [0x3f0e] = 0x8a91, [0x3f0f] = 0x8a92,
  [0x3f10] = 0x8a94, [0x3f11] = 0x8a95, [0x3f12] = 0x8a96, [0x3f13] = 0x8a97,
  [0x3f14] = 0x8a98, [0x3f15] = 0x8a99, [0x3f16] = 0x8a9a, [0x3f17] = 0x8a9b,
  [0x3f18] = 0x8a9c, [0x3f19] = 0x8a9d, [0x3f1a] = 0x8a9e, [0x3f1b] = 0x8a9f,
  [0x3f1c] = 0x8aa0, [0x3f1d] = 0x8aa1, [0x3f1e] = 0x8aa2, [0x3f1f] = 0x8aa3,
  [0x3f20] = 0x8aa4, [0x3f21] = 0x8aa5, [0x3f22] = 0x8aa6, [0x3f23] = 0x8aa7,
  [0x3f24] = 0x8aa8, [0x3f25] = 0x8aa9, [0x3f26] = 0x8aaa, [0x3f27] = 0x8aab,
  [0x3f28] = 0x8aac, [0x3f29] = 0x8aad, [0x3f2a] = 0x8aae, [0x3f2b] = 0x8aaf,
  [0x3f2c] = 0x8ab0, [0x3f2d] = 0x8ab1, [0x3f2e] = 0x8ab2, [0x3f2f] = 0x8ab3,
  [0x3f30] = 0x8ab4, [0x3f31] = 0x8ab5, [0x3f32] = 0x8ab6, [0x3f33] = 0x8ab7,
  [0x3f34] = 0x8ab8, [0x3f35] = 0x8ab9, [0x3f36] = 0x8aba, [0x3f37] = 0x8abb,
  [0x3f38] = 0x8abc, [0x3f39] = 0x8abd, [0x3f3a] = 0x8abe, [0x3f3b] = 0x8abf,
  [0x3f3c] = 0x8ac0, [0x3f3d] = 0x8ac1, [0x3f3e] = 0x8ac2, [0x3f40] = 0x8ac3,
  [0x3f41] = 0x8ac4, [0x3f42] = 0x8ac5, [0x3f43] = 0x8ac6, [0x3f44] = 0x8ac7,
  [0x3f45] = 0x8ac8, [0x3f46] = 0x8ac9, [0x3f47] = 0x8aca, [0x3f48] = 0x8acb,
  [0x3f49] = 0x8acc, [0x3f4a] = 0x8acd, [0x3f4b] = 0x8ace, [0x3f4c] = 0x8acf,
  [0x3f4d] = 0x8ad0, [0x3f4e] = 0x8ad1, [0x3f4f] = 0x8ad2, [0x3f50] = 0x8ad3,
  [0x3f51] = 0x8ad4, [0x3f52] = 0x8ad5, [0x3f53] = 0x8ad6, [0x3f54] = 0x8ad7,
  [0x3f55] = 0x8ad8, [0x3f56] = 0x8ad9, [0x3f57] = 0x8ada, [0x3f58] = 0x8adb,
  [0x3f59] = 0x8adc, [0x3f5a] = 0x8add, [0x3f5b] = 0x8ade, [0x3f5c] = 0x8adf,
  [0x3f5d] = 0x8ae0, [0x3f5e] = 0x8ae1, [0x3f5f] = 0x8ae2, [0x3f60] = 0x8ae3,
  [0x3f61] = 0x94e1, [0x3f62] = 0x95f8, [0x3f63] = 0x7728, [0x3f64] = 0x6805,
  [0x3f65] = 0x69a8, [0x3f66] = 0x548b, [0x3f67] = 0x4e4d, [0x3f68] = 0x70b8,
  [0x3f69] = 0x8bc8, [0x3f6a] = 0x6458, [0x3f6b] = 0x658b, [0x3f6c] = 0x5b85,
  [0x3f6d] = 0x7a84, [0x3f6e] = 0x503a, [0x3f6f] = 0x5be8, [0x3f70] = 0x77bb,
  [0x3f71] = 0x6be1, [0x3f72] = 0x8a79, [0x3f73] = 0x7c98, [0x3f74] = 0x6cbe,
  [0x3f75] = 0x76cf, [0x3f76] = 0x65a9, [0x3f77] = 0x8f97, [0x3f78] = 0x5d2d,
  [0x3f79] = 0x5c55, [0x3f7a] = 0x8638, [0x3f7b] = 0x6808, [0x3f7c] = 0x5360,
  [0x3f7d] = 0x6218, [0x3f7e] = 0x7ad9, [0x3f7f] = 0x6e5b, [0x3f80] = 0x7efd,
  [0x3f81] = 0x6a1f, [0x3f82] = 0x7ae0, [0x3f83] = 0x5f70, [0x3f84] = 0x6f33,
  [0x3f85] = 0x5f20, [0x3f86] = 0x638c, [0x3f87] = 0x6da8, [0x3f88] = 0x6756,
  [0x3f89] = 0x4e08, [0x3f8a] = 0x5e10, [0x3f8b] = 0x8d26, [0x3f8c] = 0x4ed7,
  [0x3f8d] = 0x80c0, [0x3f8e] = 0x7634, [0x3f8f] = 0x969c, [0x3f90] = 0x62db,
  [0x3f91] = 0x662d, [0x3f92] = 0x627e, [0x3f93] = 0x6cbc, [0x3f94] = 0x8d75,
  [0x3f95] = 0x7167, [0x3f96] = 0x7f69, [0x3f97] = 0x5146, [0x3f98] = 0x8087,
  [0x3f99] = 0x53ec, [0x3f9a] = 0x906e, [0x3f9b] = 0x6298, [0x3f9c] = 0x54f2,
  [0x3f9d] = 0x86f0, [0x3f9e] = 0x8f99, [0x3f9f] = 0x8005, [0x3fa0] = 0x9517,
  [0x3fa1] = 0x8517, [0x3fa2] = 0x8fd9, [0x3fa3] = 0x6d59, [0x3fa4] = 0x73cd,
  [0x3fa5] = 0x659f, [0x3fa6] = 0x771f, [0x3fa7] = 0x7504, [0x3fa8] = 0x7827,
  [0x3fa9] = 0x81fb, [0x3faa] = 0x8d1e, [0x3fab] = 0x9488, [0x3fac] = 0x4fa6,
  [0x3fad] = 0x6795, [0x3fae] = 0x75b9, [0x3faf] = 0x8bca, [0x3fb0] = 0x9707,
  [0x3fb1] = 0x632f, [0x3fb2] = 0x9547, [0x3fb3] = 0x9635, [0x3fb4] = 0x84b8,
  [0x3fb5] = 0x6323, [0x3fb6] = 0x7741, [0x3fb7] = 0x5f81, [0x3fb8] = 0x72f0,
  [0x3fb9] = 0x4e89, [0x3fba] = 0x6014, [0x3fbb] = 0x6574, [0x3fbc] = 0x62ef,
  [0x3fbd] = 0x6b63, [0x3fbe] = 0x653f, [0x3fc0] = 0x8ae4, [0x3fc1] = 0x8ae5,
  [0x3fc2] = 0x8ae6, [0x3fc3] = 0x8ae7, [0x3fc4] = 0x8ae8, [0x3fc5] = 0x8ae9,
  [0x3fc6] = 0x8aea, [0x3fc7] = 0x8aeb, [0x3fc8] = 0x8aec, [0x3fc9] = 0x8aed,
  [0x3fca] = 0x8aee, [0x3fcb] = 0x8aef, [0x3fcc] = 0x8af0, [0x3fcd] = 0x8af1,
  [0x3fce] = 0x8af2, [0x3fcf] = 0x8af3, [0x3fd0] = 0x8af4, [0x3fd1] = 0x8af5,
  [0x3fd2] = 0x8af6, [0x3fd3] = 0x8af7, [0x3fd4] = 0x8af8, [0x3fd5] = 0x8af9,
  [0x3fd6] = 0x8afa, [0x3fd7] = 0x8afb, [0x3fd8] = 0x8afc, [0x3fd9] = 0x8afd,
  [0x3fda] = 0x8afe, [0x3fdb] = 0x8aff, [0x3fdc] = 0x8b00, [0x3fdd] = 0x8b01,
  [0x3fde] = 0x8b02, [0x3fdf] = 0x8b03, [0x3fe0] = 0x8b04, [0x3fe1] = 0x8b05,
  [0x3fe2] = 0x8b06, [0x3fe3] = 0x8b08, [0x3fe4] = 0x8b09, [0x3fe5] = 0x8b0a,
  [0x3fe6] = 0x8b0b, [0x3fe7] = 0x8b0c, [0x3fe8] = 0x8b0d, [0x3fe9] = 0x8b0e,
  [0x3fea] = 0x8b0f, [0x3feb] = 0x8b10, [0x3fec] = 0x8b11, [0x3fed] = 0x8b12,
  [0x3fee] = 0x8b13, [0x3fef] = 0x8b14, [0x3ff0] = 0x8b15, [0x3ff1] = 0x8b16,
  [0x3ff2] = 0x8b17, [0x3ff3] = 0x8b18, [0x3ff4] = 0x8b19, [0x3ff5] = 0x8b1a,
  [0x3ff6] = 0x8b1b, [0x3ff7] = 0x8b1c, [0x3ff8] = 0x8b1d, [0x3ff9] = 0x8b1e,
  [0x3ffa] = 0x8b1f, [0x3ffb] = 0x8b20, [0x3ffc] = 0x8b21, [0x3ffd] = 0x8b22,
  [0x3ffe] = 0x8b23, [0x4000] = 0x8b24, [0x4001] = 0x8b25, [0x4002] = 0x8b27,
  [0x4003] = 0x8b28, [0x4004] = 0x8b29, [0x4005] = 0x8b2a, [0x4006] = 0x8b2b,
  [0x4007] = 0x8b2c, [0x4008] = 0x8b2d, [0x4009] = 0x8b2e, [0x400a] = 0x8b2f,
  [0x400b] = 0x8b30, [0x400c] = 0x8b31, [0x400d] = 0x8b32, [0x400e] = 0x8b33,
  [0x400f] = 0x8b34, [0x4010] = 0x8b35, [0x4011] = 0x8b36, [0x4012] = 0x8b37,
  [0x4013] = 0x8b38, [0x4014] = 0x8b39, [0x4015] = 0x8b3a, [0x4016] = 0x8b3b,
  [0x4017] = 0x8b3c, [0x4018] = 0x8b3d, [0x4019] = 0x8b3e, [0x401a] = 0x8b3f,
  [0x401b] = 0x8b40, [0x401c] = 0x8b41, [0x401d] = 0x8b42, [0x401e] = 0x8b43,
  [0x401f] = 0x8b44, [0x4020] = 0x8b45, [0x4021] = 0x5e27, [0x4022] = 0x75c7,
  [0x4023] = 0x90d1, [0x4024] = 0x8bc1, [0x4025] = 0x829d, [0x4026] = 0x679d,
  [0x4027] = 0x652f, [0x4028] = 0x5431, [0x4029] = 0x8718, [0x402a] = 0x77e5,
  [0x402b] = 0x80a2, [0x402c] = 0x8102, [0x402d] = 0x6c41, [0x402e] = 0x4e4b,
  [0x402f] = 0x7ec7, [0x4030] = 0x804c, [0x4031] = 0x76f4, [0x4032] = 0x690d,
  [0x4033] = 0x6b96, [0x4034] = 0x6267, [0x4035] = 0x503c, [0x4036] = 0x4f84,
  [0x4037] = 0x5740, [0x4038] = 0x6307, [0x4039] = 0x6b62, [0x403a] = 0x8dbe,
  [0x403b] = 0x53ea, [0x403c] = 0x65e8, [0x403d] = 0x7eb8, [0x403e] = 0x5fd7,
  [0x403f] = 0x631a, [0x4040] = 0x63b7, [0x4041] = 0x81f3, [0x4042] = 0x81f4,
  [0x4043] = 0x7f6e, [0x4044] = 0x5e1c, [0x4045] = 0x5cd9, [0x4046] = 0x5236,
  [0x4047] = 0x667a, [0x4048] = 0x79e9, [0x4049] = 0x7a1a, [0x404a] = 0x8d28,
  [0x404b] = 0x7099, [0x404c] = 0x75d4, [0x404d] = 0x6ede, [0x404e] = 0x6cbb,
  [0x404f] = 0x7a92, [0x4050] = 0x4e2d, [0x4051] = 0x76c5, [0x4052] = 0x5fe0,
  [0x4053] = 0x949f, [0x4054] = 0x8877, [0x4055] = 0x7ec8, [0x4056] = 0x79cd,
  [0x4057] = 0x80bf, [0x4058] = 0x91cd, [0x4059] = 0x4ef2, [0x405a] = 0x4f17,
  [0x405b] = 0x821f, [0x405c] = 0x5468, [0x405d] = 0x5dde, [0x405e] = 0x6d32,
  [0x405f] = 0x8bcc, [0x4060] = 0x7ca5, [0x4061] = 0x8f74, [0x4062] = 0x8098,
  [0x4063] = 0x5e1a, [0x4064] = 0x5492, [0x4065] = 0x76b1, [0x4066] = 0x5b99,
  [0x4067] = 0x663c, [0x4068] = 0x9aa4, [0x4069] = 0x73e0, [0x406a] = 0x682a,
  [0x406b] = 0x86db, [0x406c] = 0x6731, [0x406d] = 0x732a, [0x406e] = 0x8bf8,
  [0x406f] = 0x8bdb, [0x4070] = 0x9010, [0x4071] = 0x7af9, [0x4072] = 0x70db,
  [0x4073] = 0x716e, [0x4074] = 0x62c4, [0x4075] = 0x77a9, [0x4076] = 0x5631,
  [0x4077] = 0x4e3b, [0x4078] = 0x8457, [0x4079] = 0x67f1, [0x407a] = 0x52a9,
  [0x407b] = 0x86c0, [0x407c] = 0x8d2e, [0x407d] = 0x94f8, [0x407e] = 0x7b51,
  [0x4080] = 0x8b46, [0x4081] = 0x8b47, [0x4082] = 0x8b48, [0x4083] = 0x8b49,
  [0x4084] = 0x8b4a, [0x4085] = 0x8b4b, [0x4086] = 0x8b4c, [0x4087] = 0x8b4d,
  [0x4088] = 0x8b4e, [0x4089] = 0x8b4f, [0x408a] = 0x8b50, [0x408b] = 0x8b51,
  [0x408c] = 0x8b52, [0x408d] = 0x8b53, [0x408e] = 0x8b54, [0x408f] = 0x8b55,
  [0x4090] = 0x8b56, [0x4091] = 0x8b57, [0x4092] = 0x8b58, [0x4093] = 0x8b59,
  [0x4094] = 0x8b5a, [0x4095] = 0x8b5b, [0x4096] = 0x8b5c, [0x4097] = 0x8b5d,
  [0x4098] = 0x8b5e, [0x4099] = 0x8b5f, [0x409a] = 0x8b60, [0x409b] = 0x8b61,
  [0x409c] = 0x8b62, [0x409d] = 0x8b63, [0x409e] = 0x8b64, [0x409f] = 0x8b65,
  [0x40a0] = 0x8b67, [0x40a1] = 0x8b68, [0x40a2] = 0x8b69, [0x40a3] = 0x8b6a,
  [0x40a4] = 0x8b6b, [0x40a5] = 0x8b6d, [0x40a6] = 0x8b6e, [0x40a7] = 0x8b6f,
  [0x40a8] = 0x8b70, [0x40a9] = 0x8b71, [0x40aa] = 0x8b72, [0x40ab] = 0x8b73,
  [0x40ac] = 0x8b74, [0x40ad] = 0x8b75, [0x40ae] = 0x8b76, [0x40af] = 0x8b77,
  [0x40b0] = 0x8b78, [0x40b1] = 0x8b79, [0x40b2] = 0x8b7a, [0x40b3] = 0x8b7b,
  [0x40b4] = 0x8b7c, [0x40b5] = 0x8b7d, [0x40b6] = 0x8b7e, [0x40b7] = 0x8b7f,
  [0x40b8] = 0x8b80, [0x40b9] = 0x8b81, [0x40ba] = 0x8b82, [0x40bb] = 0x8b83,
  [0x40bc] = 0x8b84, [0x40bd] = 0x8b85, [0x40be] = 0x8b86, [0x40c0] = 0x8b87,
  [0x40c1] = 0x8b88, [0x40c2] = 0x8b89, [0x40c3] = 0x8b8a, [0x40c4] = 0x8b8b,
  [0x40c5] = 0x8b8c, [0x40c6] = 0x8b8d, [0x40c7] = 0x8b8e, [0x40c8] = 0x8b8f,
  [0x40c9] = 0x8b90, [0x40ca] = 0x8b91, [0x40cb] = 0x8b92, [0x40cc] = 0x8b93,
  [0x40cd] = 0x8b94, [0x40ce] = 0x8b95, [0x40cf] = 0x8b96, [0x40d0] = 0x8b97,
  [0x40d1] = 0x8b98, [0x40d2] = 0x8b99, [0x40d3] = 0x8b9a, [0x40d4] = 0x8b9b,
  [0x40d5] = 0x8b9c, [0x40d6] = 0x8b9d, [0x40d7] = 0x8b9e, [0x40d8] = 0x8b9f,
  [0x40d9] = 0x8bac, [0x40da] = 0x8bb1, [0x40db] = 0x8bbb, [0x40dc] = 0x8bc7,
  [0x40dd] = 0x8bd0, [0x40de] = 0x8bea, [0x40df] = 0x8c09, [0x40e0] = 0x8c1e,
  [0x40e1] = 0x4f4f, [0x40e2] = 0x6ce8, [0x40e3] = 0x795d, [0x40e4] = 0x9a7b,
  [0x40e5] = 0x6293, [0x40e6] = 0x722a, [0x40e7] = 0x62fd, [0x40e8] = 0x4e13,
  [0x40e9] = 0x7816, [0x40ea] = 0x8f6c, [0x40eb] = 0x64b0, [0x40ec] = 0x8d5a,
  [0x40ed] = 0x7bc6, [0x40ee] = 0x6869, [0x40ef] = 0x5e84, [0x40f0] = 0x88c5,
  [0x40f1] = 0x5986, [0x40f2] = 0x649e, [0x40f3] = 0x58ee, [0x40f4] = 0x72b6,
  [0x40f5] = 0x690e, [0x40f6] = 0x9525, [0x40f7] = 0x8ffd, [0x40f8] = 0x8d58,
  [0x40f9] = 0x5760, [0x40fa] = 0x7f00, [0x40fb] = 0x8c06, [0x40fc] = 0x51c6,
  [0x40fd] = 0x6349, [0x40fe] = 0x62d9, [0x40ff] = 0x5353, [0x4100] = 0x684c,
  [0x4101] = 0x7422, [0x4102] = 0x8301, [0x4103] = 0x914c, [0x4104] = 0x5544,
  [0x4105] = 0x7740, [0x4106] = 0x707c, [0x4107] = 0x6d4a, [0x4108] = 0x5179,
  [0x4109] = 0x54a8, [0x410a] = 0x8d44, [0x410b] = 0x59ff, [0x410c] = 0x6ecb,
  [0x410d] = 0x6dc4, [0x410e] = 0x5b5c, [0x410f] = 0x7d2b, [0x4110] = 0x4ed4,
  [0x4111] = 0x7c7d, [0x4112] = 0x6ed3, [0x4113] = 0x5b50, [0x4114] = 0x81ea,
  [0x4115] = 0x6e0d, [0x4116] = 0x5b57, [0x4117] = 0x9b03, [0x4118] = 0x68d5,
  [0x4119] = 0x8e2a, [0x411a] = 0x5b97, [0x411b] = 0x7efc, [0x411c] = 0x603b,
  [0x411d] = 0x7eb5, [0x411e] = 0x90b9, [0x411f] = 0x8d70, [0x4120] = 0x594f,
  [0x4121] = 0x63cd, [0x4122] = 0x79df, [0x4123] = 0x8db3, [0x4124] = 0x5352,
  [0x4125] = 0x65cf, [0x4126] = 0x7956, [0x4127] = 0x8bc5, [0x4128] = 0x963b,
  [0x4129] = 0x7ec4, [0x412a] = 0x94bb, [0x412b] = 0x7e82, [0x412c] = 0x5634,
  [0x412d] = 0x9189, [0x412e] = 0x6700, [0x412f] = 0x7f6a, [0x4130] = 0x5c0a,
  [0x4131] = 0x9075, [0x4132] = 0x6628, [0x4133] = 0x5de6, [0x4134] = 0x4f50,
  [0x4135] = 0x67de, [0x4136] = 0x505a, [0x4137] = 0x4f5c, [0x4138] = 0x5750,
  [0x4139] = 0x5ea7, [0x413a] = 0xe810, [0x413b] = 0xe811, [0x413c] = 0xe812,
  [0x413d] = 0xe813, [0x413e] = 0xe814, [0x4140] = 0x8c38, [0x4141] = 0x8c39,
  [0x4142] = 0x8c3a, [0x4143] = 0x8c3b, [0x4144] = 0x8c3c, [0x4145] = 0x8c3d,
  [0x4146] = 0x8c3e, [0x4147] = 0x8c3f, [0x4148] = 0x8c40, [0x4149] = 0x8c42,
  [0x414a] = 0x8c43, [0x414b] = 0x8c44, [0x414c] = 0x8c45, [0x414d] = 0x8c48,
  [0x414e] = 0x8c4a, [0x414f] = 0x8c4b, [0x4150] = 0x8c4d, [0x4151] = 0x8c4e,
  [0x4152] = 0x8c4f, [0x4153] = 0x8c50, [0x4154] = 0x8c51, [0x4155] = 0x8c52,
  [0x4156] = 0x8c53, [0x4157] = 0x8c54, [0x4158] = 0x8c56, [0x4159] = 0x8c57,
  [0x415a] = 0x8c58, [0x415b] = 0x8c59, [0x415c] = 0x8c5b, [0x415d] = 0x8c5c,
  [0x415e] = 0x8c5d, [0x415f] = 0x8c5e, [0x4160] = 0x8c5f, [0x4161] = 0x8c60,
  [0x4162] = 0x8c63, [0x4163] = 0x8c64, [0x4164] = 0x8c65, [0x4165] = 0x8c66,
  [0x4166] = 0x8c67, [0x4167] = 0x8c68, [0x4168] = 0x8c69, [0x4169] = 0x8c6c,
  [0x416a] = 0x8c6d, [0x416b] = 0x8c6e, [0x416c] = 0x8c6f, [0x416d] = 0x8c70,
  [0x416e] = 0x8c71, [0x416f] = 0x8c72, [0x4170] = 0x8c74, [0x4171] = 0x8c75,
  [0x4172] = 0x8c76, [0x4173] = 0x8c77, [0x4174] = 0x8c7b, [0x4175] = 0x8c7c,
  [0x4176] = 0x8c7d, [0x4177] = 0x8c7e, [0x4178] = 0x8c7f, [0x4179] = 0x8c80,
  [0x417a] = 0x8c81, [0x417b] = 0x8c83, [0x417c] = 0x8c84, [0x417d] = 0x8c86,
  [0x417e] = 0x8c87, [0x4180] = 0x8c88, [0x4181] = 0x8c8b, [0x4182] = 0x8c8d,
  [0x4183] = 0x8c8e, [0x4184] = 0x8c8f, [0x4185] = 0x8c90, [0x4186] = 0x8c91,
  [0x4187] = 0x8c92, [0x4188] = 0x8c93, [0x4189] = 0x8c95, [0x418a] = 0x8c96,
  [0x418b] = 0x8c97, [0x418c] = 0x8c99, [0x418d] = 0x8c9a, [0x418e] = 0x8c9b,
  [0x418f] = 0x8c9c, [0x4190] = 0x8c9d, [0x4191] = 0x8c9e, [0x4192] = 0x8c9f,
  [0x4193] = 0x8ca0, [0x4194] = 0x8ca1, [0x4195] = 0x8ca2, [0x4196] = 0x8ca3,
  [0x4197] = 0x8ca4, [0x4198] = 0x8ca5, [0x4199] = 0x8ca6, [0x419a] = 0x8ca7,
  [0x419b] = 0x8ca8, [0x419c] = 0x8ca9, [0x419d] = 0x8caa, [0x419e] = 0x8cab,
  [0x419f] = 0x8cac, [0x41a0] = 0x8cad, [0x41a1] = 0x4e8d, [0x41a2] = 0x4e0c,
  [0x41a3] = 0x5140, [0x41a4] = 0x4e10, [0x41a5] = 0x5eff, [0x41a6] = 0x5345,
  [0x41a7] = 0x4e15, [0x41a8] = 0x4e98, [0x41a9] = 0x4e1e, [0x41aa] = 0x9b32,
  [0x41ab] = 0x5b6c, [0x41ac] = 0x5669, [0x41ad] = 0x4e28, [0x41ae] = 0x79ba,
  [0x41af] = 0x4e3f, [0x41b0] = 0x5315, [0x41b1] = 0x4e47, [0x41b2] = 0x592d,
  [0x41b3] = 0x723b, [0x41b4] = 0x536e, [0x41b5] = 0x6c10, [0x41b6] = 0x56df,
  [0x41b7] = 0x80e4, [0x41b8] = 0x9997, [0x41b9] = 0x6bd3, [0x41ba] = 0x777e,
  [0x41bb] = 0x9f17, [0x41bc] = 0x4e36, [0x41bd] = 0x4e9f, [0x41be] = 0x9f10,
  [0x41bf] = 0x4e5c, [0x41c0] = 0x4e69, [0x41c1] = 0x4e93, [0x41c2] = 0x8288,
  [0x41c3] = 0x5b5b, [0x41c4] = 0x556c, [0x41c5] = 0x560f, [0x41c6] = 0x4ec4,
  [0x41c7] = 0x538d, [0x41c8] = 0x539d, [0x41c9] = 0x53a3, [0x41ca] = 0x53a5,
  [0x41cb] = 0x53ae, [0x41cc] = 0x9765, [0x41cd] = 0x8d5d, [0x41ce] = 0x531a,
  [0x41cf] = 0x53f5, [0x41d0] = 0x5326, [0x41d1] = 0x532e, [0x41d2] = 0x533e,
  [0x41d3] = 0x8d5c, [0x41d4] = 0x5366, [0x41d5] = 0x5363, [0x41d6] = 0x5202,
  [0x41d7] = 0x5208, [0x41d8] = 0x520e, [0x41d9] = 0x522d, [0x41da] = 0x5233,
  [0x41db] = 0x523f, [0x41dc] = 0x5240, [0x41dd] = 0x524c, [0x41de] = 0x525e,
  [0x41df] = 0x5261, [0x41e0] = 0x525c, [0x41e1] = 0x84af, [0x41e2] = 0x527d,
  [0x41e3] = 0x5282, [0x41e4] = 0x5281, [0x41e5] = 0x5290, [0x41e6] = 0x5293,
  [0x41e7] = 0x5182, [0x41e8] = 0x7f54, [0x41e9] = 0x4ebb, [0x41ea] = 0x4ec3,
  [0x41eb] = 0x4ec9, [0x41ec] = 0x4ec2, [0x41ed] = 0x4ee8, [0x41ee] = 0x4ee1,
  [0x41ef] = 0x4eeb, [0x41f0] = 0x4ede, [0x41f1] = 0x4f1b, [0x41f2] = 0x4ef3,
  [0x41f3] = 0x4f22, [0x41f4] = 0x4f64, [0x41f5] = 0x4ef5, [0x41f6] = 0x4f25,
  [0x41f7] = 0x4f27, [0x41f8] = 0x4f09, [0x41f9] = 0x4f2b, [0x41fa] = 0x4f5e,
  [0x41fb] = 0x4f67, [0x41fc] = 0x6538, [0x41fd] = 0x4f5a, [0x41fe] = 0x4f5d,
  [0x4200] = 0x8cae, [0x4201] = 0x8caf, [0x4202] = 0x8cb0, [0x4203] = 0x8cb1,
  [0x4204] = 0x8cb2, [0x4205] = 0x8cb3, [0x4206] = 0x8cb4, [0x4207] = 0x8cb5,
  [0x4208] = 0x8cb6, [0x4209] = 0x8cb7, [0x420a] = 0x8cb8, [0x420b] = 0x8cb9,
  [0x420c] = 0x8cba, [0x420d] = 0x8cbb, [0x420e] = 0x8cbc, [0x420f] = 0x8cbd,
  [0x4210] = 0x8cbe, [0x4211] = 0x8cbf, [0x4212] = 0x8cc0, [0x4213] = 0x8cc1,
  [0x4214] = 0x8cc2, [0x4215] = 0x8cc3, [0x4216] = 0x8cc4, [0x4217] = 0x8cc5,
  [0x4218] = 0x8cc6, [0x4219] = 0x8cc7, [0x421a] = 0x8cc8, [0x421b] = 0x8cc9,
  [0x421c] = 0x8cca, [0x421d] = 0x8ccb, [0x421e] = 0x8ccc, [0x421f] = 0x8ccd,
  [0x4220] = 0x8cce, [0x4221] = 0x8ccf, [0x4222] = 0x8cd0, [0x4223] = 0x8cd1,
  [0x4224] = 0x8cd2, [0x4225] = 0x8cd3, [0x4226] = 0x8cd4, [0x4227] = 0x8cd5,
  [0x4228] = 0x8cd6, [0x4229] = 0x8cd7, [0x422a] = 0x8cd8, [0x422b] = 0x8cd9,
  [0x422c] = 0x8cda, [0x422d] = 0x8cdb, [0x422e] = 0x8cdc, [0x422f] = 0x8cdd,
  [0x4230] = 0x8cde, [0x4231] = 0x8cdf, [0x4232] = 0x8ce0, [0x4233] = 0x8ce1,
  [0x4234] = 0x8ce2, [0x4235] = 0x8ce3, [0x4236] = 0x8ce4, [0x4237] = 0x8ce5,
  [0x4238] = 0x8ce6, [0x4239] = 0x8ce7, [0x423a] = 0x8ce8, [0x423b] = 0x8ce9,
  [0x423c] = 0x8cea, [0x423d] = 0x8ceb, [0x423e] = 0x8cec, [0x4240] = 0x8ced,
  [0x4241] = 0x8cee, [0x4242] = 0x8cef, [0x4243] = 0x8cf0, [0x4244] = 0x8cf1,
  [0x4245] = 0x8cf2, [0x4246] = 0x8cf3, [0x4247] = 0x8cf4, [0x4248] = 0x8cf5,
  [0x4249] = 0x8cf6, [0x424a] = 0x8cf7, [0x424b] = 0x8cf8, [0x424c] = 0x8cf9,
  [0x424d] = 0x8cfa, [0x424e] = 0x8cfb, [0x424f] = 0x8cfc, [0x4250] = 0x8cfd,
  [0x4251] = 0x8cfe, [0x4252] = 0x8cff, [0x4253] = 0x8d00, [0x4254] = 0x8d01,
  [0x4255] = 0x8d02, [0x4256] = 0x8d03, [0x4257] = 0x8d04, [0x4258] = 0x8d05,
  [0x4259] = 0x8d06, [0x425a] = 0x8d07, [0x425b] = 0x8d08, [0x425c] = 0x8d09,
  [0x425d] = 0x8d0a, [0x425e] = 0x8d0b, [0x425f] = 0x8d0c, [0x4260] = 0x8d0d,
  [0x4261] = 0x4f5f, [0x4262] = 0x4f57, [0x4263] = 0x4f32, [0x4264] = 0x4f3d,
  [0x4265] = 0x4f76, [0x4266] = 0x4f74, [0x4267] = 0x4f91, [0x4268] = 0x4f89,
  [0x4269] = 0x4f83, [0x426a] = 0x4f8f, [0x426b] = 0x4f7e, [0x426c] = 0x4f7b,
  [0x426d] = 0x4faa, [0x426e] = 0x4f7c, [0x426f] = 0x4fac, [0x4270] = 0x4f94,
  [0x4271] = 0x4fe6, [0x4272] = 0x4fe8, [0x4273] = 0x4fea, [0x4274] = 0x4fc5,
  [0x4275] = 0x4fda, [0x4276] = 0x4fe3, [0x4277] = 0x4fdc, [0x4278] = 0x4fd1,
  [0x4279] = 0x4fdf, [0x427a] = 0x4ff8, [0x427b] = 0x5029, [0x427c] = 0x504c,
  [0x427d] = 0x4ff3, [0x427e] = 0x502c, [0x427f] = 0x500f, [0x4280] = 0x502e,
  [0x4281] = 0x502d, [0x4282] = 0x4ffe, [0x4283] = 0x501c, [0x4284] = 0x500c,
  [0x4285] = 0x5025, [0x4286] = 0x5028, [0x4287] = 0x507e, [0x4288] = 0x5043,
  [0x4289] = 0x5055, [0x428a] = 0x5048, [0x428b] = 0x504e, [0x428c] = 0x506c,
  [0x428d] = 0x507b, [0x428e] = 0x50a5, [0x428f] = 0x50a7, [0x4290] = 0x50a9,
  [0x4291] = 0x50ba, [0x4292] = 0x50d6, [0x4293] = 0x5106, [0x4294] = 0x50ed,
  [0x4295] = 0x50ec, [0x4296] = 0x50e6, [0x4297] = 0x50ee, [0x4298] = 0x5107,
  [0x4299] = 0x510b, [0x429a] = 0x4edd, [0x429b] = 0x6c3d, [0x429c] = 0x4f58,
  [0x429d] = 0x4f65, [0x429e] = 0x4fce, [0x429f] = 0x9fa0, [0x42a0] = 0x6c46,
  [0x42a1] = 0x7c74, [0x42a2] = 0x516e, [0x42a3] = 0x5dfd, [0x42a4] = 0x9ec9,
  [0x42a5] = 0x9998, [0x42a6] = 0x5181, [0x42a7] = 0x5914, [0x42a8] = 0x52f9,
  [0x42a9] = 0x530d, [0x42aa] = 0x8a07, [0x42ab] = 0x5310, [0x42ac] = 0x51eb,
  [0x42ad] = 0x5919, [0x42ae] = 0x5155, [0x42af] = 0x4ea0, [0x42b0] = 0x5156,
  [0x42b1] = 0x4eb3, [0x42b2] = 0x886e, [0x42b3] = 0x88a4, [0x42b4] = 0x4eb5,
  [0x42b5] = 0x8114, [0x42b6] = 0x88d2, [0x42b7] = 0x7980, [0x42b8] = 0x5b34,
  [0x42b9] = 0x8803, [0x42ba] = 0x7fb8, [0x42bb] = 0x51ab, [0x42bc] = 0x51b1,
  [0x42bd] = 0x51bd, [0x42be] = 0x51bc, [0x42c0] = 0x8d0e, [0x42c1] = 0x8d0f,
  [0x42c2] = 0x8d10, [0x42c3] = 0x8d11, [0x42c4] = 0x8d12, [0x42c5] = 0x8d13,
  [0x42c6] = 0x8d14, [0x42c7] = 0x8d15, [0x42c8] = 0x8d16, [0x42c9] = 0x8d17,
  [0x42ca] = 0x8d18, [0x42cb] = 0x8d19, [0x42cc] = 0x8d1a, [0x42cd] = 0x8d1b,
  [0x42ce] = 0x8d1c, [0x42cf] = 0x8d20, [0x42d0] = 0x8d51, [0x42d1] = 0x8d52,
  [0x42d2] = 0x8d57, [0x42d3] = 0x8d5f, [0x42d4] = 0x8d65, [0x42d5] = 0x8d68,
  [0x42d6] = 0x8d69, [0x42d7] = 0x8d6a, [0x42d8] = 0x8d6c, [0x42d9] = 0x8d6e,
  [0x42da] = 0x8d6f, [0x42db] = 0x8d71, [0x42dc] = 0x8d72, [0x42dd] = 0x8d78,
  [0x42de] = 0x8d79, [0x42df] = 0x8d7a, [0x42e0] = 0x8d7b, [0x42e1] = 0x8d7c,
  [0x42e2] = 0x8d7d, [0x42e3] = 0x8d7e, [0x42e4] = 0x8d7f, [0x42e5] = 0x8d80,
  [0x42e6] = 0x8d82, [0x42e7] = 0x8d83, [0x42e8] = 0x8d86, [0x42e9] = 0x8d87,
  [0x42ea] = 0x8d88, [0x42eb] = 0x8d89, [0x42ec] = 0x8d8c, [0x42ed] = 0x8d8d,
  [0x42ee] = 0x8d8e, [0x42ef] = 0x8d8f, [0x42f0] = 0x8d90, [0x42f1] = 0x8d92,
  [0x42f2] = 0x8d93, [0x42f3] = 0x8d95, [0x42f4] = 0x8d96, [0x42f5] = 0x8d97,
  [0x42f6] = 0x8d98, [0x42f7] = 0x8d99, [0x42f8] = 0x8d9a, [0x42f9] = 0x8d9b,
  [0x42fa] = 0x8d9c, [0x42fb] = 0x8d9d, [0x42fc] = 0x8d9e, [0x42fd] = 0x8da0,
  [0x42fe] = 0x8da1, [0x4300] = 0x8da2, [0x4301] = 0x8da4, [0x4302] = 0x8da5,
  [0x4303] = 0x8da6, [0x4304] = 0x8da7, [0x4305] = 0x8da8, [0x4306] = 0x8da9,
  [0x4307] = 0x8daa, [0x4308] = 0x8dab, [0x4309] = 0x8dac, [0x430a] = 0x8dad,
  [0x430b] = 0x8dae, [0x430c] = 0x8daf, [0x430d] = 0x8db0, [0x430e] = 0x8db2,
  [0x430f] = 0x8db6, [0x4310] = 0x8db7, [0x4311] = 0x8db9, [0x4312] = 0x8dbb,
  [0x4313] = 0x8dbd, [0x4314] = 0x8dc0, [0x4315] = 0x8dc1, [0x4316] = 0x8dc2,
  [0x4317] = 0x8dc5, [0x4318] = 0x8dc7, [0x4319] = 0x8dc8, [0x431a] = 0x8dc9,
  [0x431b] = 0x8dca, [0x431c] = 0x8dcd, [0x431d] = 0x8dd0, [0x431e] = 0x8dd2,
  [0x431f] = 0x8dd3, [0x4320] = 0x8dd4, [0x4321] = 0x51c7, [0x4322] = 0x5196,
  [0x4323] = 0x51a2, [0x4324] = 0x51a5, [0x4325] = 0x8ba0, [0x4326] = 0x8ba6,
  [0x4327] = 0x8ba7, [0x4328] = 0x8baa, [0x4329] = 0x8bb4, [0x432a] = 0x8bb5,
  [0x432b] = 0x8bb7, [0x432c] = 0x8bc2, [0x432d] = 0x8bc3, [0x432e] = 0x8bcb,
  [0x432f] = 0x8bcf, [0x4330] = 0x8bce, [0x4331] = 0x8bd2, [0x4332] = 0x8bd3,
  [0x4333] = 0x8bd4, [0x4334] = 0x8bd6, [0x4335] = 0x8bd8, [0x4336] = 0x8bd9,
  [0x4337] = 0x8bdc, [0x4338] = 0x8bdf, [0x4339] = 0x8be0, [0x433a] = 0x8be4,
  [0x433b] = 0x8be8, [0x433c] = 0x8be9, [0x433d] = 0x8bee, [0x433e] = 0x8bf0,
  [0x433f] = 0x8bf3, [0x4340] = 0x8bf6, [0x4341] = 0x8bf9, [0x4342] = 0x8bfc,
  [0x4343] = 0x8bff, [0x4344] = 0x8c00, [0x4345] = 0x8c02, [0x4346] = 0x8c04,
  [0x4347] = 0x8c07, [0x4348] = 0x8c0c, [0x4349] = 0x8c0f, [0x434a] = 0x8c11,
  [0x434b] = 0x8c12, [0x434c] = 0x8c14, [0x434d] = 0x8c15, [0x434e] = 0x8c16,
  [0x434f] = 0x8c19, [0x4350] = 0x8c1b, [0x4351] = 0x8c18, [0x4352] = 0x8c1d,
  [0x4353] = 0x8c1f, [0x4354] = 0x8c20, [0x4355] = 0x8c21, [0x4356] = 0x8c25,
  [0x4357] = 0x8c27, [0x4358] = 0x8c2a, [0x4359] = 0x8c2b, [0x435a] = 0x8c2e,
  [0x435b] = 0x8c2f, [0x435c] = 0x8c32, [0x435d] = 0x8c33, [0x435e] = 0x8c35,
  [0x435f] = 0x8c36, [0x4360] = 0x5369, [0x4361] = 0x537a, [0x4362] = 0x961d,
  [0x4363] = 0x9622, [0x4364] = 0x9621, [0x4365] = 0x9631, [0x4366] = 0x962a,
  [0x4367] = 0x963d, [0x4368] = 0x963c, [0x4369] = 0x9642, [0x436a] = 0x9649,
  [0x436b] = 0x9654, [0x436c] = 0x965f, [0x436d] = 0x9667, [0x436e] = 0x966c,
  [0x436f] = 0x9672, [0x4370] = 0x9674, [0x4371] = 0x9688, [0x4372] = 0x968d,
  [0x4373] = 0x9697, [0x4374] = 0x96b0, [0x4375] = 0x9097, [0x4376] = 0x909b,
  [0x4377] = 0x909d, [0x4378] = 0x9099, [0x4379] = 0x90ac, [0x437a] = 0x90a1,
  [0x437b] = 0x90b4, [0x437c] = 0x90b3, [0x437d] = 0x90b6, [0x437e] = 0x90ba,
  [0x4380] = 0x8dd5, [0x4381] = 0x8dd8, [0x4382] = 0x8dd9, [0x4383] = 0x8ddc,
  [0x4384] = 0x8de0, [0x4385] = 0x8de1, [0x4386] = 0x8de2, [0x4387] = 0x8de5,
  [0x4388] = 0x8de6, [0x4389] = 0x8de7, [0x438a] = 0x8de9, [0x438b] = 0x8ded,
  [0x438c] = 0x8dee, [0x438d] = 0x8df0, [0x438e] = 0x8df1, [0x438f] = 0x8df2,
  [0x4390] = 0x8df4, [0x4391] = 0x8df6, [0x4392] = 0x8dfc, [0x4393] = 0x8dfe,
  [0x4394] = 0x8dff, [0x4395] = 0x8e00, [0x4396] = 0x8e01, [0x4397] = 0x8e02,
  [0x4398] = 0x8e03, [0x4399] = 0x8e04, [0x439a] = 0x8e06, [0x439b] = 0x8e07,
  [0x439c] = 0x8e08, [0x439d] = 0x8e0b, [0x439e] = 0x8e0d, [0x439f] = 0x8e0e,
  [0x43a0] = 0x8e10, [0x43a1] = 0x8e11, [0x43a2] = 0x8e12, [0x43a3] = 0x8e13,
  [0x43a4] = 0x8e15, [0x43a5] = 0x8e16, [0x43a6] = 0x8e17, [0x43a7] = 0x8e18,
  [0x43a8] = 0x8e19, [0x43a9] = 0x8e1a, [0x43aa] = 0x8e1b, [0x43ab] = 0x8e1c,
  [0x43ac] = 0x8e20, [0x43ad] = 0x8e21, [0x43ae] = 0x8e24, [0x43af] = 0x8e25,
  [0x43b0] = 0x8e26, [0x43b1] = 0x8e27, [0x43b2] = 0x8e28, [0x43b3] = 0x8e2b,
  [0x43b4] = 0x8e2d, [0x43b5] = 0x8e30, [0x43b6] = 0x8e32, [0x43b7] = 0x8e33,
  [0x43b8] = 0x8e34, [0x43b9] = 0x8e36, [0x43ba] = 0x8e37, [0x43bb] = 0x8e38,
  [0x43bc] = 0x8e3b, [0x43bd] = 0x8e3c, [0x43be] = 0x8e3e, [0x43c0] = 0x8e3f,
  [0x43c1] = 0x8e43, [0x43c2] = 0x8e45, [0x43c3] = 0x8e46, [0x43c4] = 0x8e4c,
  [0x43c5] = 0x8e4d, [0x43c6] = 0x8e4e, [0x43c7] = 0x8e4f, [0x43c8] = 0x8e50,
  [0x43c9] = 0x8e53, [0x43ca] = 0x8e54, [0x43cb] = 0x8e55, [0x43cc] = 0x8e56,
  [0x43cd] = 0x8e57, [0x43ce] = 0x8e58, [0x43cf] = 0x8e5a, [0x43d0] = 0x8e5b,
  [0x43d1] = 0x8e5c, [0x43d2] = 0x8e5d, [0x43d3] = 0x8e5e, [0x43d4] = 0x8e5f,
  [0x43d5] = 0x8e60, [0x43d6] = 0x8e61, [0x43d7] = 0x8e62, [0x43d8] = 0x8e63,
  [0x43d9] = 0x8e64, [0x43da] = 0x8e65, [0x43db] = 0x8e67, [0x43dc] = 0x8e68,
  [0x43dd] = 0x8e6a, [0x43de] = 0x8e6b, [0x43df] = 0x8e6e, [0x43e0] = 0x8e71,
  [0x43e1] = 0x90b8, [0x43e2] = 0x90b0, [0x43e3] = 0x90cf, [0x43e4] = 0x90c5,
  [0x43e5] = 0x90be, [0x43e6] = 0x90d0, [0x43e7] = 0x90c4, [0x43e8] = 0x90c7,
  [0x43e9] = 0x90d3, [0x43ea] = 0x90e6, [0x43eb] = 0x90e2, [0x43ec] = 0x90dc,
  [0x43ed] = 0x90d7, [0x43ee] = 0x90db, [0x43ef] = 0x90eb, [0x43f0] = 0x90ef,
  [0x43f1] = 0x90fe, [0x43f2] = 0x9104, [0x43f3] = 0x9122, [0x43f4] = 0x911e,
  [0x43f5] = 0x9123, [0x43f6] = 0x9131, [0x43f7] = 0x912f, [0x43f8] = 0x9139,
  [0x43f9] = 0x9143, [0x43fa] = 0x9146, [0x43fb] = 0x520d, [0x43fc] = 0x5942,
  [0x43fd] = 0x52a2, [0x43fe] = 0x52ac, [0x43ff] = 0x52ad, [0x4400] = 0x52be,
  [0x4401] = 0x54ff, [0x4402] = 0x52d0, [0x4403] = 0x52d6, [0x4404] = 0x52f0,
  [0x4405] = 0x53df, [0x4406] = 0x71ee, [0x4407] = 0x77cd, [0x4408] = 0x5ef4,
  [0x4409] = 0x51f5, [0x440a] = 0x51fc, [0x440b] = 0x9b2f, [0x440c] = 0x53b6,
  [0x440d] = 0x5f01, [0x440e] = 0x755a, [0x440f] = 0x5def, [0x4410] = 0x574c,
  [0x4411] = 0x57a9, [0x4412] = 0x57a1, [0x4413] = 0x587e, [0x4414] = 0x58bc,
  [0x4415] = 0x58c5, [0x4416] = 0x58d1, [0x4417] = 0x5729, [0x4418] = 0x572c,
  [0x4419] = 0x572a, [0x441a] = 0x5733, [0x441b] = 0x5739, [0x441c] = 0x572e,
  [0x441d] = 0x572f, [0x441e] = 0x575c, [0x441f] = 0x573b, [0x4420] = 0x5742,
  [0x4421] = 0x5769, [0x4422] = 0x5785, [0x4423] = 0x576b, [0x4424] = 0x5786,
  [0x4425] = 0x577c, [0x4426] = 0x577b, [0x4427] = 0x5768, [0x4428] = 0x576d,
  [0x4429] = 0x5776, [0x442a] = 0x5773, [0x442b] = 0x57ad, [0x442c] = 0x57a4,
  [0x442d] = 0x578c, [0x442e] = 0x57b2, [0x442f] = 0x57cf, [0x4430] = 0x57a7,
  [0x4431] = 0x57b4, [0x4432] = 0x5793, [0x4433] = 0x57a0, [0x4434] = 0x57d5,
  [0x4435] = 0x57d8, [0x4436] = 0x57da, [0x4437] = 0x57d9, [0x4438] = 0x57d2,
  [0x4439] = 0x57b8, [0x443a] = 0x57f4, [0x443b] = 0x57ef, [0x443c] = 0x57f8,
  [0x443d] = 0x57e4, [0x443e] = 0x57dd, [0x4440] = 0x8e73, [0x4441] = 0x8e75,
  [0x4442] = 0x8e77, [0x4443] = 0x8e78, [0x4444] = 0x8e79, [0x4445] = 0x8e7a,
  [0x4446] = 0x8e7b, [0x4447] = 0x8e7d, [0x4448] = 0x8e7e, [0x4449] = 0x8e80,
  [0x444a] = 0x8e82, [0x444b] = 0x8e83, [0x444c] = 0x8e84, [0x444d] = 0x8e86,
  [0x444e] = 0x8e88, [0x444f] = 0x8e89, [0x4450] = 0x8e8a, [0x4451] = 0x8e8b,
  [0x4452] = 0x8e8c, [0x4453] = 0x8e8d, [0x4454] = 0x8e8e, [0x4455] = 0x8e91,
  [0x4456] = 0x8e92, [0x4457] = 0x8e93, [0x4458] = 0x8e95, [0x4459] = 0x8e96,
  [0x445a] = 0x8e97, [0x445b] = 0x8e98, [0x445c] = 0x8e99, [0x445d] = 0x8e9a,
  [0x445e] = 0x8e9b, [0x445f] = 0x8e9d, [0x4460] = 0x8e9f, [0x4461] = 0x8ea0,
  [0x4462] = 0x8ea1, [0x4463] = 0x8ea2, [0x4464] = 0x8ea3, [0x4465] = 0x8ea4,
  [0x4466] = 0x8ea5, [0x4467] = 0x8ea6, [0x4468] = 0x8ea7, [0x4469] = 0x8ea8,
  [0x446a] = 0x8ea9, [0x446b] = 0x8eaa, [0x446c] = 0x8ead, [0x446d] = 0x8eae,
  [0x446e] = 0x8eb0, [0x446f] = 0x8eb1, [0x4470] = 0x8eb3, [0x4471] = 0x8eb4,
  [0x4472] = 0x8eb5, [0x4473] = 0x8eb6, [0x4474] = 0x8eb7, [0x4475] = 0x8eb8,
  [0x4476] = 0x8eb9, [0x4477] = 0x8ebb, [0x4478] = 0x8ebc, [0x4479] = 0x8ebd,
  [0x447a] = 0x8ebe, [0x447b] = 0x8ebf, [0x447c] = 0x8ec0, [0x447d] = 0x8ec1,
  [0x447e] = 0x8ec2, [0x4480] = 0x8ec3, [0x4481] = 0x8ec4, [0x4482] = 0x8ec5,
  [0x4483] = 0x8ec6, [0x4484] = 0x8ec7, [0x4485] = 0x8ec8, [0x4486] = 0x8ec9,
  [0x4487] = 0x8eca, [0x4488] = 0x8ecb, [0x4489] = 0x8ecc, [0x448a] = 0x8ecd,
  [0x448b] = 0x8ecf, [0x448c] = 0x8ed0, [0x448d] = 0x8ed1, [0x448e] = 0x8ed2,
  [0x448f] = 0x8ed3, [0x4490] = 0x8ed4, [0x4491] = 0x8ed5, [0x4492] = 0x8ed6,
  [0x4493] = 0x8ed7, [0x4494] = 0x8ed8, [0x4495] = 0x8ed9, [0x4496] = 0x8eda,
  [0x4497] = 0x8edb, [0x4498] = 0x8edc, [0x4499] = 0x8edd, [0x449a] = 0x8ede,
  [0x449b] = 0x8edf, [0x449c] = 0x8ee0, [0x449d] = 0x8ee1, [0x449e] = 0x8ee2,
  [0x449f] = 0x8ee3, [0x44a0] = 0x8ee4, [0x44a1] = 0x580b, [0x44a2] = 0x580d,
  [0x44a3] = 0x57fd, [0x44a4] = 0x57ed, [0x44a5] = 0x5800, [0x44a6] = 0x581e,
  [0x44a7] = 0x5819, [0x44a8] = 0x5844, [0x44a9] = 0x5820, [0x44aa] = 0x5865,
  [0x44ab] = 0x586c, [0x44ac] = 0x5881, [0x44ad] = 0x5889, [0x44ae] = 0x589a,
  [0x44af] = 0x5880, [0x44b0] = 0x99a8, [0x44b1] = 0x9f19, [0x44b2] = 0x61ff,
  [0x44b3] = 0x8279, [0x44b4] = 0x827d, [0x44b5] = 0x827f, [0x44b6] = 0x828f,
  [0x44b7] = 0x828a, [0x44b8] = 0x82a8, [0x44b9] = 0x8284, [0x44ba] = 0x828e,
  [0x44bb] = 0x8291, [0x44bc] = 0x8297, [0x44bd] = 0x8299, [0x44be] = 0x82ab,
  [0x44bf] = 0x82b8, [0x44c0] = 0x82be, [0x44c1] = 0x82b0, [0x44c2] = 0x82c8,
  [0x44c3] = 0x82ca, [0x44c4] = 0x82e3, [0x44c5] = 0x8298, [0x44c6] = 0x82b7,
  [0x44c7] = 0x82ae, [0x44c8] = 0x82cb, [0x44c9] = 0x82cc, [0x44ca] = 0x82c1,
  [0x44cb] = 0x82a9, [0x44cc] = 0x82b4, [0x44cd] = 0x82a1, [0x44ce] = 0x82aa,
  [0x44cf] = 0x829f, [0x44d0] = 0x82c4, [0x44d1] = 0x82ce, [0x44d2] = 0x82a4,
  [0x44d3] = 0x82e1, [0x44d4] = 0x8309, [0x44d5] = 0x82f7, [0x44d6] = 0x82e4,
  [0x44d7] = 0x830f, [0x44d8] = 0x8307, [0x44d9] = 0x82dc, [0x44da] = 0x82f4,
  [0x44db] = 0x82d2, [0x44dc] = 0x82d8, [0x44dd] = 0x830c, [0x44de] = 0x82fb,
  [0x44df] = 0x82d3, [0x44e0] = 0x8311, [0x44e1] = 0x831a, [0x44e2] = 0x8306,
  [0x44e3] = 0x8314, [0x44e4] = 0x8315, [0x44e5] = 0x82e0, [0x44e6] = 0x82d5,
  [0x44e7] = 0x831c, [0x44e8] = 0x8351, [0x44e9] = 0x835b, [0x44ea] = 0x835c,
  [0x44eb] = 0x8308, [0x44ec] = 0x8392, [0x44ed] = 0x833c, [0x44ee] = 0x8334,
  [0x44ef] = 0x8331, [0x44f0] = 0x839b, [0x44f1] = 0x835e, [0x44f2] = 0x832f,
  [0x44f3] = 0x834f, [0x44f4] = 0x8347, [0x44f5] = 0x8343, [0x44f6] = 0x835f,
  [0x44f7] = 0x8340, [0x44f8] = 0x8317, [0x44f9] = 0x8360, [0x44fa] = 0x832d,
  [0x44fb] = 0x833a, [0x44fc] = 0x8333, [0x44fd] = 0x8366, [0x44fe] = 0x8365,
  [0x4500] = 0x8ee5, [0x4501] = 0x8ee6, [0x4502] = 0x8ee7, [0x4503] = 0x8ee8,
  [0x4504] = 0x8ee9, [0x4505] = 0x8eea, [0x4506] = 0x8eeb, [0x4507] = 0x8eec,
  [0x4508] = 0x8eed, [0x4509] = 0x8eee, [0x450a] = 0x8eef, [0x450b] = 0x8ef0,
  [0x450c] = 0x8ef1, [0x450d] = 0x8ef2, [0x450e] = 0x8ef3, [0x450f] = 0x8ef4,
  [0x4510] = 0x8ef5, [0x4511] = 0x8ef6, [0x4512] = 0x8ef7, [0x4513] = 0x8ef8,
  [0x4514] = 0x8ef9, [0x4515] = 0x8efa, [0x4516] = 0x8efb, [0x4517] = 0x8efc,
  [0x4518] = 0x8efd, [0x4519] = 0x8efe, [0x451a] = 0x8eff, [0x451b] = 0x8f00,
  [0x451c] = 0x8f01, [0x451d] = 0x8f02, [0x451e] = 0x8f03, [0x451f] = 0x8f04,
  [0x4520] = 0x8f05, [0x4521] = 0x8f06, [0x4522] = 0x8f07, [0x4523] = 0x8f08,
  [0x4524] = 0x8f09, [0x4525] = 0x8f0a, [0x4526] = 0x8f0b, [0x4527] = 0x8f0c,
  [0x4528] = 0x8f0d, [0x4529] = 0x8f0e, [0x452a] = 0x8f0f, [0x452b] = 0x8f10,
  [0x452c] = 0x8f11, [0x452d] = 0x8f12, [0x452e] = 0x8f13, [0x452f] = 0x8f14,
  [0x4530] = 0x8f15, [0x4531] = 0x8f16, [0x4532] = 0x8f17, [0x4533] = 0x8f18,
  [0x4534] = 0x8f19, [0x4535] = 0x8f1a, [0x4536] = 0x8f1b, [0x4537] = 0x8f1c,
  [0x4538] = 0x8f1d, [0x4539] = 0x8f1e, [0x453a] = 0x8f1f, [0x453b] = 0x8f20,
  [0x453c] = 0x8f21, [0x453d] = 0x8f22, [0x453e] = 0x8f23, [0x4540] = 0x8f24,
  [0x4541] = 0x8f25, [0x4542] = 0x8f26, [0x4543] = 0x8f27, [0x4544] = 0x8f28,
  [0x4545] = 0x8f29, [0x4546] = 0x8f2a, [0x4547] = 0x8f2b, [0x4548] = 0x8f2c,
  [0x4549] = 0x8f2d, [0x454a] = 0x8f2e, [0x454b] = 0x8f2f, [0x454c] = 0x8f30,
  [0x454d] = 0x8f31, [0x454e] = 0x8f32, [0x454f] = 0x8f33, [0x4550] = 0x8f34,
  [0x4551] = 0x8f35, [0x4552] = 0x8f36, [0x4553] = 0x8f37, [0x4554] = 0x8f38,
  [0x4555] = 0x8f39, [0x4556] = 0x8f3a, [0x4557] = 0x8f3b, [0x4558] = 0x8f3c,
  [0x4559] = 0x8f3d, [0x455a] = 0x8f3e, [0x455b] = 0x8f3f, [0x455c] = 0x8f40,
  [0x455d] = 0x8f41, [0x455e] = 0x8f42, [0x455f] = 0x8f43, [0x4560] = 0x8f44,
  [0x4561] = 0x8368, [0x4562] = 0x831b, [0x4563] = 0x8369, [0x4564] = 0x836c,
  [0x4565] = 0x836a, [0x4566] = 0x836d, [0x4567] = 0x836e, [0x4568] = 0x83b0,
  [0x4569] = 0x8378, [0x456a] = 0x83b3, [0x456b] = 0x83b4, [0x456c] = 0x83a0,
  [0x456d] = 0x83aa, [0x456e] = 0x8393, [0x456f] = 0x839c, [0x4570] = 0x8385,
  [0x4571] = 0x837c, [0x4572] = 0x83b6, [0x4573] = 0x83a9, [0x4574] = 0x837d,
  [0x4575] = 0x83b8, [0x4576] = 0x837b, [0x4577] = 0x8398, [0x4578] = 0x839e,
  [0x4579] = 0x83a8, [0x457a] = 0x83ba, [0x457b] = 0x83bc, [0x457c] = 0x83c1,
  [0x457d] = 0x8401, [0x457e] = 0x83e5, [0x457f] = 0x83d8, [0x4580] = 0x5807,
  [0x4581] = 0x8418, [0x4582] = 0x840b, [0x4583] = 0x83dd, [0x4584] = 0x83fd,
  [0x4585] = 0x83d6, [0x4586] = 0x841c, [0x4587] = 0x8438, [0x4588] = 0x8411,
  [0x4589] = 0x8406, [0x458a] = 0x83d4, [0x458b] = 0x83df, [0x458c] = 0x840f,
  [0x458d] = 0x8403, [0x458e] = 0x83f8, [0x458f] = 0x83f9, [0x4590] = 0x83ea,
  [0x4591] = 0x83c5, [0x4592] = 0x83c0, [0x4593] = 0x8426, [0x4594] = 0x83f0,
  [0x4595] = 0x83e1, [0x4596] = 0x845c, [0x4597] = 0x8451, [0x4598] = 0x845a,
  [0x4599] = 0x8459, [0x459a] = 0x8473, [0x459b] = 0x8487, [0x459c] = 0x8488,
  [0x459d] = 0x847a, [0x459e] = 0x8489, [0x459f] = 0x8478, [0x45a0] = 0x843c,
  [0x45a1] = 0x8446, [0x45a2] = 0x8469, [0x45a3] = 0x8476, [0x45a4] = 0x848c,
  [0x45a5] = 0x848e, [0x45a6] = 0x8431, [0x45a7] = 0x846d, [0x45a8] = 0x84c1,
  [0x45a9] = 0x84cd, [0x45aa] = 0x84d0, [0x45ab] = 0x84e6, [0x45ac] = 0x84bd,
  [0x45ad] = 0x84d3, [0x45ae] = 0x84ca, [0x45af] = 0x84bf, [0x45b0] = 0x84ba,
  [0x45b1] = 0x84e0, [0x45b2] = 0x84a1, [0x45b3] = 0x84b9, [0x45b4] = 0x84b4,
  [0x45b5] = 0x8497, [0x45b6] = 0x84e5, [0x45b7] = 0x84e3, [0x45b8] = 0x850c,
  [0x45b9] = 0x750d, [0x45ba] = 0x8538, [0x45bb] = 0x84f0, [0x45bc] = 0x8539,
  [0x45bd] = 0x851f, [0x45be] = 0x853a, [0x45c0] = 0x8f45, [0x45c1] = 0x8f46,
  [0x45c2] = 0x8f47, [0x45c3] = 0x8f48, [0x45c4] = 0x8f49, [0x45c5] = 0x8f4a,
  [0x45c6] = 0x8f4b, [0x45c7] = 0x8f4c, [0x45c8] = 0x8f4d, [0x45c9] = 0x8f4e,
  [0x45ca] = 0x8f4f, [0x45cb] = 0x8f50, [0x45cc] = 0x8f51, [0x45cd] = 0x8f52,
  [0x45ce] = 0x8f53, [0x45cf] = 0x8f54, [0x45d0] = 0x8f55, [0x45d1] = 0x8f56,
  [0x45d2] = 0x8f57, [0x45d3] = 0x8f58, [0x45d4] = 0x8f59, [0x45d5] = 0x8f5a,
  [0x45d6] = 0x8f5b, [0x45d7] = 0x8f5c, [0x45d8] = 0x8f5d, [0x45d9] = 0x8f5e,
  [0x45da] = 0x8f5f, [0x45db] = 0x8f60, [0x45dc] = 0x8f61, [0x45dd] = 0x8f62,
  [0x45de] = 0x8f63, [0x45df] = 0x8f64, [0x45e0] = 0x8f65, [0x45e1] = 0x8f6a,
  [0x45e2] = 0x8f80, [0x45e3] = 0x8f8c, [0x45e4] = 0x8f92, [0x45e5] = 0x8f9d,
  [0x45e6] = 0x8fa0, [0x45e7] = 0x8fa1, [0x45e8] = 0x8fa2, [0x45e9] = 0x8fa4,
  [0x45ea] = 0x8fa5, [0x45eb] = 0x8fa6, [0x45ec] = 0x8fa7, [0x45ed] = 0x8faa,
  [0x45ee] = 0x8fac, [0x45ef] = 0x8fad, [0x45f0] = 0x8fae, [0x45f1] = 0x8faf,
  [0x45f2] = 0x8fb2, [0x45f3] = 0x8fb3, [0x45f4] = 0x8fb4, [0x45f5] = 0x8fb5,
  [0x45f6] = 0x8fb7, [0x45f7] = 0x8fb8, [0x45f8] = 0x8fba, [0x45f9] = 0x8fbb,
  [0x45fa] = 0x8fbc, [0x45fb] = 0x8fbf, [0x45fc] = 0x8fc0, [0x45fd] = 0x8fc3,
  [0x45fe] = 0x8fc6, [0x4600] = 0x8fc9, [0x4601] = 0x8fca, [0x4602] = 0x8fcb,
  [0x4603] = 0x8fcc, [0x4604] = 0x8fcd, [0x4605] = 0x8fcf, [0x4606] = 0x8fd2,
  [0x4607] = 0x8fd6, [0x4608] = 0x8fd7, [0x4609] = 0x8fda, [0x460a] = 0x8fe0,
  [0x460b] = 0x8fe1, [0x460c] = 0x8fe3, [0x460d] = 0x8fe7, [0x460e] = 0x8fec,
  [0x460f] = 0x8fef, [0x4610] = 0x8ff1, [0x4611] = 0x8ff2, [0x4612] = 0x8ff4,
  [0x4613] = 0x8ff5, [0x4614] = 0x8ff6, [0x4615] = 0x8ffa, [0x4616] = 0x8ffb,
  [0x4617] = 0x8ffc, [0x4618] = 0x8ffe, [0x4619] = 0x8fff, [0x461a] = 0x9007,
  [0x461b] = 0x9008, [0x461c] = 0x900c, [0x461d] = 0x900e, [0x461e] = 0x9013,
  [0x461f] = 0x9015, [0x4620] = 0x9018, [0x4621] = 0x8556, [0x4622] = 0x853b,
  [0x4623] = 0x84ff, [0x4624] = 0x84fc, [0x4625] = 0x8559, [0x4626] = 0x8548,
  [0x4627] = 0x8568, [0x4628] = 0x8564, [0x4629] = 0x855e, [0x462a] = 0x857a,
  [0x462b] = 0x77a2, [0x462c] = 0x8543, [0x462d] = 0x8572, [0x462e] = 0x857b,
  [0x462f] = 0x85a4, [0x4630] = 0x85a8, [0x4631] = 0x8587, [0x4632] = 0x858f,
  [0x4633] = 0x8579, [0x4634] = 0x85ae, [0x4635] = 0x859c, [0x4636] = 0x8585,
  [0x4637] = 0x85b9, [0x4638] = 0x85b7, [0x4639] = 0x85b0, [0x463a] = 0x85d3,
  [0x463b] = 0x85c1, [0x463c] = 0x85dc, [0x463d] = 0x85ff, [0x463e] = 0x8627,
  [0x463f] = 0x8605, [0x4640] = 0x8629, [0x4641] = 0x8616, [0x4642] = 0x863c,
  [0x4643] = 0x5efe, [0x4644] = 0x5f08, [0x4645] = 0x593c, [0x4646] = 0x5941,
  [0x4647] = 0x8037, [0x4648] = 0x5955, [0x4649] = 0x595a, [0x464a] = 0x5958,
  [0x464b] = 0x530f, [0x464c] = 0x5c22, [0x464d] = 0x5c25, [0x464e] = 0x5c2c,
  [0x464f] = 0x5c34, [0x4650] = 0x624c, [0x4651] = 0x626a, [0x4652] = 0x629f,
  [0x4653] = 0x62bb, [0x4654] = 0x62ca, [0x4655] = 0x62da, [0x4656] = 0x62d7,
  [0x4657] = 0x62ee, [0x4658] = 0x6322, [0x4659] = 0x62f6, [0x465a] = 0x6339,
  [0x465b] = 0x634b, [0x465c] = 0x6343, [0x465d] = 0x63ad, [0x465e] = 0x63f6,
  [0x465f] = 0x6371, [0x4660] = 0x637a, [0x4661] = 0x638e, [0x4662] = 0x63b4,
  [0x4663] = 0x636d, [0x4664] = 0x63ac, [0x4665] = 0x638a, [0x4666] = 0x6369,
  [0x4667] = 0x63ae, [0x4668] = 0x63bc, [0x4669] = 0x63f2, [0x466a] = 0x63f8,
  [0x466b] = 0x63e0, [0x466c] = 0x63ff, [0x466d] = 0x63c4, [0x466e] = 0x63de,
  [0x466f] = 0x63ce, [0x4670] = 0x6452, [0x4671] = 0x63c6, [0x4672] = 0x63be,
  [0x4673] = 0x6445, [0x4674] = 0x6441, [0x4675] = 0x640b, [0x4676] = 0x641b,
  [0x4677] = 0x6420, [0x4678] = 0x640c, [0x4679] = 0x6426, [0x467a] = 0x6421,
  [0x467b] = 0x645e, [0x467c] = 0x6484, [0x467d] = 0x646d, [0x467e] = 0x6496,
  [0x4680] = 0x9019, [0x4681] = 0x901c, [0x4682] = 0x9023, [0x4683] = 0x9024,
  [0x4684] = 0x9025, [0x4685] = 0x9027, [0x4686] = 0x9028, [0x4687] = 0x9029,
  [0x4688] = 0x902a, [0x4689] = 0x902b, [0x468a] = 0x902c, [0x468b] = 0x9030,
  [0x468c] = 0x9031, [0x468d] = 0x9032, [0x468e] = 0x9033, [0x468f] = 0x9034,
  [0x4690] = 0x9037, [0x4691] = 0x9039, [0x4692] = 0x903a, [0x4693] = 0x903d,
  [0x4694] = 0x903f, [0x4695] = 0x9040, [0x4696] = 0x9043, [0x4697] = 0x9045,
  [0x4698] = 0x9046, [0x4699] = 0x9048, [0x469a] = 0x9049, [0x469b] = 0x904a,
  [0x469c] = 0x904b, [0x469d] = 0x904c, [0x469e] = 0x904e, [0x469f] = 0x9054,
  [0x46a0] = 0x9055, [0x46a1] = 0x9056, [0x46a2] = 0x9059, [0x46a3] = 0x905a,
  [0x46a4] = 0x905c, [0x46a5] = 0x905d, [0x46a6] = 0x905e, [0x46a7] = 0x905f,
  [0x46a8] = 0x9060, [0x46a9] = 0x9061, [0x46aa] = 0x9064, [0x46ab] = 0x9066,
  [0x46ac] = 0x9067, [0x46ad] = 0x9069, [0x46ae] = 0x906a, [0x46af] = 0x906b,
  [0x46b0] = 0x906c, [0x46b1] = 0x906f, [0x46b2] = 0x9070, [0x46b3] = 0x9071,
  [0x46b4] = 0x9072, [0x46b5] = 0x9073, [0x46b6] = 0x9076, [0x46b7] = 0x9077,
  [0x46b8] = 0x9078, [0x46b9] = 0x9079, [0x46ba] = 0x907a, [0x46bb] = 0x907b,
  [0x46bc] = 0x907c, [0x46bd] = 0x907e, [0x46be] = 0x9081, [0x46c0] = 0x9084,
  [0x46c1] = 0x9085, [0x46c2] = 0x9086, [0x46c3] = 0x9087, [0x46c4] = 0x9089,
  [0x46c5] = 0x908a, [0x46c6] = 0x908c, [0x46c7] = 0x908d, [0x46c8] = 0x908e,
  [0x46c9] = 0x908f, [0x46ca] = 0x9090, [0x46cb] = 0x9092, [0x46cc] = 0x9094,
  [0x46cd] = 0x9096, [0x46ce] = 0x9098, [0x46cf] = 0x909a, [0x46d0] = 0x909c,
  [0x46d1] = 0x909e, [0x46d2] = 0x909f, [0x46d3] = 0x90a0, [0x46d4] = 0x90a4,
  [0x46d5] = 0x90a5, [0x46d6] = 0x90a7, [0x46d7] = 0x90a8, [0x46d8] = 0x90a9,
  [0x46d9] = 0x90ab, [0x46da] = 0x90ad, [0x46db] = 0x90b2, [0x46dc] = 0x90b7,
  [0x46dd] = 0x90bc, [0x46de] = 0x90bd, [0x46df] = 0x90bf, [0x46e0] = 0x90c0,
  [0x46e1] = 0x647a, [0x46e2] = 0x64b7, [0x46e3] = 0x64b8, [0x46e4] = 0x6499,
  [0x46e5] = 0x64ba, [0x46e6] = 0x64c0, [0x46e7] = 0x64d0, [0x46e8] = 0x64d7,
  [0x46e9] = 0x64e4, [0x46ea] = 0x64e2, [0x46eb] = 0x6509, [0x46ec] = 0x6525,
  [0x46ed] = 0x652e, [0x46ee] = 0x5f0b, [0x46ef] = 0x5fd2, [0x46f0] = 0x7519,
  [0x46f1] = 0x5f11, [0x46f2] = 0x535f, [0x46f3] = 0x53f1, [0x46f4] = 0x53fd,
  [0x46f5] = 0x53e9, [0x46f6] = 0x53e8, [0x46f7] = 0x53fb, [0x46f8] = 0x5412,
  [0x46f9] = 0x5416, [0x46fa] = 0x5406, [0x46fb] = 0x544b, [0x46fc] = 0x5452,
  [0x46fd] = 0x5453, [0x46fe] = 0x5454, [0x46ff] = 0x5456, [0x4700] = 0x5443,
  [0x4701] = 0x5421, [0x4702] = 0x5457, [0x4703] = 0x5459, [0x4704] = 0x5423,
  [0x4705] = 0x5432, [0x4706] = 0x5482, [0x4707] = 0x5494, [0x4708] = 0x5477,
  [0x4709] = 0x5471, [0x470a] = 0x5464, [0x470b] = 0x549a, [0x470c] = 0x549b,
  [0x470d] = 0x5484, [0x470e] = 0x5476, [0x470f] = 0x5466, [0x4710] = 0x549d,
  [0x4711] = 0x54d0, [0x4712] = 0x54ad, [0x4713] = 0x54c2, [0x4714] = 0x54b4,
  [0x4715] = 0x54d2, [0x4716] = 0x54a7, [0x4717] = 0x54a6, [0x4718] = 0x54d3,
  [0x4719] = 0x54d4, [0x471a] = 0x5472, [0x471b] = 0x54a3, [0x471c] = 0x54d5,
  [0x471d] = 0x54bb, [0x471e] = 0x54bf, [0x471f] = 0x54cc, [0x4720] = 0x54d9,
  [0x4721] = 0x54da, [0x4722] = 0x54dc, [0x4723] = 0x54a9, [0x4724] = 0x54aa,
  [0x4725] = 0x54a4, [0x4726] = 0x54dd, [0x4727] = 0x54cf, [0x4728] = 0x54de,
  [0x4729] = 0x551b, [0x472a] = 0x54e7, [0x472b] = 0x5520, [0x472c] = 0x54fd,
  [0x472d] = 0x5514, [0x472e] = 0x54f3, [0x472f] = 0x5522, [0x4730] = 0x5523,
  [0x4731] = 0x550f, [0x4732] = 0x5511, [0x4733] = 0x5527, [0x4734] = 0x552a,
  [0x4735] = 0x5567, [0x4736] = 0x558f, [0x4737] = 0x55b5, [0x4738] = 0x5549,
  [0x4739] = 0x556d, [0x473a] = 0x5541, [0x473b] = 0x5555, [0x473c] = 0x553f,
  [0x473d] = 0x5550, [0x473e] = 0x553c, [0x4740] = 0x90c2, [0x4741] = 0x90c3,
  [0x4742] = 0x90c6, [0x4743] = 0x90c8, [0x4744] = 0x90c9, [0x4745] = 0x90cb,
  [0x4746] = 0x90cc, [0x4747] = 0x90cd, [0x4748] = 0x90d2, [0x4749] = 0x90d4,
  [0x474a] = 0x90d5, [0x474b] = 0x90d6, [0x474c] = 0x90d8, [0x474d] = 0x90d9,
  [0x474e] = 0x90da, [0x474f] = 0x90de, [0x4750] = 0x90df, [0x4751] = 0x90e0,
  [0x4752] = 0x90e3, [0x4753] = 0x90e4, [0x4754] = 0x90e5, [0x4755] = 0x90e9,
  [0x4756] = 0x90ea, [0x4757] = 0x90ec, [0x4758] = 0x90ee, [0x4759] = 0x90f0,
  [0x475a] = 0x90f1, [0x475b] = 0x90f2, [0x475c] = 0x90f3, [0x475d] = 0x90f5,
  [0x475e] = 0x90f6, [0x475f] = 0x90f7, [0x4760] = 0x90f9, [0x4761] = 0x90fa,
  [0x4762] = 0x90fb, [0x4763] = 0x90fc, [0x4764] = 0x90ff, [0x4765] = 0x9100,
  [0x4766] = 0x9101, [0x4767] = 0x9103, [0x4768] = 0x9105, [0x4769] = 0x9106,
  [0x476a] = 0x9107, [0x476b] = 0x9108, [0x476c] = 0x9109, [0x476d] = 0x910a,
  [0x476e] = 0x910b, [0x476f] = 0x910c, [0x4770] = 0x910d, [0x4771] = 0x910e,
  [0x4772] = 0x910f, [0x4773] = 0x9110, [0x4774] = 0x9111, [0x4775] = 0x9112,
  [0x4776] = 0x9113, [0x4777] = 0x9114, [0x4778] = 0x9115, [0x4779] = 0x9116,
  [0x477a] = 0x9117, [0x477b] = 0x9118, [0x477c] = 0x911a, [0x477d] = 0x911b,
  [0x477e] = 0x911c, [0x4780] = 0x911d, [0x4781] = 0x911f, [0x4782] = 0x9120,
  [0x4783] = 0x9121, [0x4784] = 0x9124, [0x4785] = 0x9125, [0x4786] = 0x9126,
  [0x4787] = 0x9127, [0x4788] = 0x9128, [0x4789] = 0x9129, [0x478a] = 0x912a,
  [0x478b] = 0x912b, [0x478c] = 0x912c, [0x478d] = 0x912d, [0x478e] = 0x912e,
  [0x478f] = 0x9130, [0x4790] = 0x9132, [0x4791] = 0x9133, [0x4792] = 0x9134,
  [0x4793] = 0x9135, [0x4794] = 0x9136, [0x4795] = 0x9137, [0x4796] = 0x9138,
  [0x4797] = 0x913a, [0x4798] = 0x913b, [0x4799] = 0x913c, [0x479a] = 0x913d,
  [0x479b] = 0x913e, [0x479c] = 0x913f, [0x479d] = 0x9140, [0x479e] = 0x9141,
  [0x479f] = 0x9142, [0x47a0] = 0x9144, [0x47a1] = 0x5537, [0x47a2] = 0x5556,
  [0x47a3] = 0x5575, [0x47a4] = 0x5576, [0x47a5] = 0x5577, [0x47a6] = 0x5533,
  [0x47a7] = 0x5530, [0x47a8] = 0x555c, [0x47a9] = 0x558b, [0x47aa] = 0x55d2,
  [0x47ab] = 0x5583, [0x47ac] = 0x55b1, [0x47ad] = 0x55b9, [0x47ae] = 0x5588,
  [0x47af] = 0x5581, [0x47b0] = 0x559f, [0x47b1] = 0x557e, [0x47b2] = 0x55d6,
  [0x47b3] = 0x5591, [0x47b4] = 0x557b, [0x47b5] = 0x55df, [0x47b6] = 0x55bd,
  [0x47b7] = 0x55be, [0x47b8] = 0x5594, [0x47b9] = 0x5599, [0x47ba] = 0x55ea,
  [0x47bb] = 0x55f7, [0x47bc] = 0x55c9, [0x47bd] = 0x561f, [0x47be] = 0x55d1,
  [0x47bf] = 0x55eb, [0x47c0] = 0x55ec, [0x47c1] = 0x55d4, [0x47c2] = 0x55e6,
  [0x47c3] = 0x55dd, [0x47c4] = 0x55c4, [0x47c5] = 0x55ef, [0x47c6] = 0x55e5,
  [0x47c7] = 0x55f2, [0x47c8] = 0x55f3, [0x47c9] = 0x55cc, [0x47ca] = 0x55cd,
  [0x47cb] = 0x55e8, [0x47cc] = 0x55f5, [0x47cd] = 0x55e4, [0x47ce] = 0x8f94,
  [0x47cf] = 0x561e, [0x47d0] = 0x5608, [0x47d1] = 0x560c, [0x47d2] = 0x5601,
  [0x47d3] = 0x5624, [0x47d4] = 0x5623, [0x47d5] = 0x55fe, [0x47d6] = 0x5600,
  [0x47d7] = 0x5627, [0x47d8] = 0x562d, [0x47d9] = 0x5658, [0x47da] = 0x5639,
  [0x47db] = 0x5657, [0x47dc] = 0x562c, [0x47dd] = 0x564d, [0x47de] = 0x5662,
  [0x47df] = 0x5659, [0x47e0] = 0x565c, [0x47e1] = 0x564c, [0x47e2] = 0x5654,
  [0x47e3] = 0x5686, [0x47e4] = 0x5664, [0x47e5] = 0x5671, [0x47e6] = 0x566b,
  [0x47e7] = 0x567b, [0x47e8] = 0x567c, [0x47e9] = 0x5685, [0x47ea] = 0x5693,
  [0x47eb] = 0x56af, [0x47ec] = 0x56d4, [0x47ed] = 0x56d7, [0x47ee] = 0x56dd,
  [0x47ef] = 0x56e1, [0x47f0] = 0x56f5, [0x47f1] = 0x56eb, [0x47f2] = 0x56f9,
  [0x47f3] = 0x56ff, [0x47f4] = 0x5704, [0x47f5] = 0x570a, [0x47f6] = 0x5709,
  [0x47f7] = 0x571c, [0x47f8] = 0x5e0f, [0x47f9] = 0x5e19, [0x47fa] = 0x5e14,
  [0x47fb] = 0x5e11, [0x47fc] = 0x5e31, [0x47fd] = 0x5e3b, [0x47fe] = 0x5e3c,
  [0x4800] = 0x9145, [0x4801] = 0x9147, [0x4802] = 0x9148, [0x4803] = 0x9151,
  [0x4804] = 0x9153, [0x4805] = 0x9154, [0x4806] = 0x9155, [0x4807] = 0x9156,
  [0x4808] = 0x9158, [0x4809] = 0x9159, [0x480a] = 0x915b, [0x480b] = 0x915c,
  [0x480c] = 0x915f, [0x480d] = 0x9160, [0x480e] = 0x9166, [0x480f] = 0x9167,
  [0x4810] = 0x9168, [0x4811] = 0x916b, [0x4812] = 0x916d, [0x4813] = 0x9173,
  [0x4814] = 0x917a, [0x4815] = 0x917b, [0x4816] = 0x917c, [0x4817] = 0x9180,
  [0x4818] = 0x9181, [0x4819] = 0x9182, [0x481a] = 0x9183, [0x481b] = 0x9184,
  [0x481c] = 0x9186, [0x481d] = 0x9188, [0x481e] = 0x918a, [0x481f] = 0x918e,
  [0x4820] = 0x918f, [0x4821] = 0x9193, [0x4822] = 0x9194, [0x4823] = 0x9195,
  [0x4824] = 0x9196, [0x4825] = 0x9197, [0x4826] = 0x9198, [0x4827] = 0x9199,
  [0x4828] = 0x919c, [0x4829] = 0x919d, [0x482a] = 0x919e, [0x482b] = 0x919f,
  [0x482c] = 0x91a0, [0x482d] = 0x91a1, [0x482e] = 0x91a4, [0x482f] = 0x91a5,
  [0x4830] = 0x91a6, [0x4831] = 0x91a7, [0x4832] = 0x91a8, [0x4833] = 0x91a9,
  [0x4834] = 0x91ab, [0x4835] = 0x91ac, [0x4836] = 0x91b0, [0x4837] = 0x91b1,
  [0x4838] = 0x91b2, [0x4839] = 0x91b3, [0x483a] = 0x91b6, [0x483b] = 0x91b7,
  [0x483c] = 0x91b8, [0x483d] = 0x91b9, [0x483e] = 0x91bb, [0x4840] = 0x91bc,
  [0x4841] = 0x91bd, [0x4842] = 0x91be, [0x4843] = 0x91bf, [0x4844] = 0x91c0,
  [0x4845] = 0x91c1, [0x4846] = 0x91c2, [0x4847] = 0x91c3, [0x4848] = 0x91c4,
  [0x4849] = 0x91c5, [0x484a] = 0x91c6, [0x484b] = 0x91c8, [0x484c] = 0x91cb,
  [0x484d] = 0x91d0, [0x484e] = 0x91d2, [0x484f] = 0x91d3, [0x4850] = 0x91d4,
  [0x4851] = 0x91d5, [0x4852] = 0x91d6, [0x4853] = 0x91d7, [0x4854] = 0x91d8,
  [0x4855] = 0x91d9, [0x4856] = 0x91da, [0x4857] = 0x91db, [0x4858] = 0x91dd,
  [0x4859] = 0x91de, [0x485a] = 0x91df, [0x485b] = 0x91e0, [0x485c] = 0x91e1,
  [0x485d] = 0x91e2, [0x485e] = 0x91e3, [0x485f] = 0x91e4, [0x4860] = 0x91e5,
  [0x4861] = 0x5e37, [0x4862] = 0x5e44, [0x4863] = 0x5e54, [0x4864] = 0x5e5b,
  [0x4865] = 0x5e5e, [0x4866] = 0x5e61, [0x4867] = 0x5c8c, [0x4868] = 0x5c7a,
  [0x4869] = 0x5c8d, [0x486a] = 0x5c90, [0x486b] = 0x5c96, [0x486c] = 0x5c88,
  [0x486d] = 0x5c98, [0x486e] = 0x5c99, [0x486f] = 0x5c91, [0x4870] = 0x5c9a,
  [0x4871] = 0x5c9c, [0x4872] = 0x5cb5, [0x4873] = 0x5ca2, [0x4874] = 0x5cbd,
  [0x4875] = 0x5cac, [0x4876] = 0x5cab, [0x4877] = 0x5cb1, [0x4878] = 0x5ca3,
  [0x4879] = 0x5cc1, [0x487a] = 0x5cb7, [0x487b] = 0x5cc4, [0x487c] = 0x5cd2,
  [0x487d] = 0x5ce4, [0x487e] = 0x5ccb, [0x487f] = 0x5ce5, [0x4880] = 0x5d02,
  [0x4881] = 0x5d03, [0x4882] = 0x5d27, [0x4883] = 0x5d26, [0x4884] = 0x5d2e,
  [0x4885] = 0x5d24, [0x4886] = 0x5d1e, [0x4887] = 0x5d06, [0x4888] = 0x5d1b,
  [0x4889] = 0x5d58, [0x488a] = 0x5d3e, [0x488b] = 0x5d34, [0x488c] = 0x5d3d,
  [0x488d] = 0x5d6c, [0x488e] = 0x5d5b, [0x488f] = 0x5d6f, [0x4890] = 0x5d5d,
  [0x4891] = 0x5d6b, [0x4892] = 0x5d4b, [0x4893] = 0x5d4a, [0x4894] = 0x5d69,
  [0x4895] = 0x5d74, [0x4896] = 0x5d82, [0x4897] = 0x5d99, [0x4898] = 0x5d9d,
  [0x4899] = 0x8c73, [0x489a] = 0x5db7, [0x489b] = 0x5dc5, [0x489c] = 0x5f73,
  [0x489d] = 0x5f77, [0x489e] = 0x5f82, [0x489f] = 0x5f87, [0x48a0] = 0x5f89,
  [0x48a1] = 0x5f8c, [0x48a2] = 0x5f95, [0x48a3] = 0x5f99, [0x48a4] = 0x5f9c,
  [0x48a5] = 0x5fa8, [0x48a6] = 0x5fad, [0x48a7] = 0x5fb5, [0x48a8] = 0x5fbc,
  [0x48a9] = 0x8862, [0x48aa] = 0x5f61, [0x48ab] = 0x72ad, [0x48ac] = 0x72b0,
  [0x48ad] = 0x72b4, [0x48ae] = 0x72b7, [0x48af] = 0x72b8, [0x48b0] = 0x72c3,
  [0x48b1] = 0x72c1, [0x48b2] = 0x72ce, [0x48b3] = 0x72cd, [0x48b4] = 0x72d2,
  [0x48b5] = 0x72e8, [0x48b6] = 0x72ef, [0x48b7] = 0x72e9, [0x48b8] = 0x72f2,
  [0x48b9] = 0x72f4, [0x48ba] = 0x72f7, [0x48bb] = 0x7301, [0x48bc] = 0x72f3,
  [0x48bd] = 0x7303, [0x48be] = 0x72fa, [0x48c0] = 0x91e6, [0x48c1] = 0x91e7,
  [0x48c2] = 0x91e8, [0x48c3] = 0x91e9, [0x48c4] = 0x91ea, [0x48c5] = 0x91eb,
  [0x48c6] = 0x91ec, [0x48c7] = 0x91ed, [0x48c8] = 0x91ee, [0x48c9] = 0x91ef,
  [0x48ca] = 0x91f0, [0x48cb] = 0x91f1, [0x48cc] = 0x91f2, [0x48cd] = 0x91f3,
  [0x48ce] = 0x91f4, [0x48cf] = 0x91f5, [0x48d0] = 0x91f6, [0x48d1] = 0x91f7,
  [0x48d2] = 0x91f8, [0x48d3] = 0x91f9, [0x48d4] = 0x91fa, [0x48d5] = 0x91fb,
  [0x48d6] = 0x91fc, [0x48d7] = 0x91fd, [0x48d8] = 0x91fe, [0x48d9] = 0x91ff,
  [0x48da] = 0x9200, [0x48db] = 0x9201, [0x48dc] = 0x9202, [0x48dd] = 0x9203,
  [0x48de] = 0x9204, [0x48df] = 0x9205, [0x48e0] = 0x9206, [0x48e1] = 0x9207,
  [0x48e2] = 0x9208, [0x48e3] = 0x9209, [0x48e4] = 0x920a, [0x48e5] = 0x920b,
  [0x48e6] = 0x920c, [0x48e7] = 0x920d, [0x48e8] = 0x920e, [0x48e9] = 0x920f,
  [0x48ea] = 0x9210, [0x48eb] = 0x9211, [0x48ec] = 0x9212, [0x48ed] = 0x9213,
  [0x48ee] = 0x9214, [0x48ef] = 0x9215, [0x48f0] = 0x9216, [0x48f1] = 0x9217,
  [0x48f2] = 0x9218, [0x48f3] = 0x9219, [0x48f4] = 0x921a, [0x48f5] = 0x921b,
  [0x48f6] = 0x921c, [0x48f7] = 0x921d, [0x48f8] = 0x921e, [0x48f9] = 0x921f,
  [0x48fa] = 0x9220, [0x48fb] = 0x9221, [0x48fc] = 0x9222, [0x48fd] = 0x9223,
  [0x48fe] = 0x9224, [0x4900] = 0x9225, [0x4901] = 0x9226, [0x4902] = 0x9227,
  [0x4903] = 0x9228, [0x4904] = 0x9229, [0x4905] = 0x922a, [0x4906] = 0x922b,
  [0x4907] = 0x922c, [0x4908] = 0x922d, [0x4909] = 0x922e, [0x490a] = 0x922f,
  [0x490b] = 0x9230, [0x490c] = 0x9231, [0x490d] = 0x9232, [0x490e] = 0x9233,
  [0x490f] = 0x9234, [0x4910] = 0x9235, [0x4911] = 0x9236, [0x4912] = 0x9237,
  [0x4913] = 0x9238, [0x4914] = 0x9239, [0x4915] = 0x923a, [0x4916] = 0x923b,
  [0x4917] = 0x923c, [0x4918] = 0x923d, [0x4919] = 0x923e, [0x491a] = 0x923f,
  [0x491b] = 0x9240, [0x491c] = 0x9241, [0x491d] = 0x9242, [0x491e] = 0x9243,
  [0x491f] = 0x9244, [0x4920] = 0x9245, [0x4921] = 0x72fb, [0x4922] = 0x7317,
  [0x4923] = 0x7313, [0x4924] = 0x7321, [0x4925] = 0x730a, [0x4926] = 0x731e,
  [0x4927] = 0x731d, [0x4928] = 0x7315, [0x4929] = 0x7322, [0x492a] = 0x7339,
  [0x492b] = 0x7325, [0x492c] = 0x732c, [0x492d] = 0x7338, [0x492e] = 0x7331,
  [0x492f] = 0x7350, [0x4930] = 0x734d, [0x4931] = 0x7357, [0x4932] = 0x7360,
  [0x4933] = 0x736c, [0x4934] = 0x736f, [0x4935] = 0x737e, [0x4936] = 0x821b,
  [0x4937] = 0x5925, [0x4938] = 0x98e7, [0x4939] = 0x5924, [0x493a] = 0x5902,
  [0x493b] = 0x9963, [0x493c] = 0x9967, [0x493d] = 0x9968, [0x493e] = 0x9969,
  [0x493f] = 0x996a, [0x4940] = 0x996b, [0x4941] = 0x996c, [0x4942] = 0x9974,
  [0x4943] = 0x9977, [0x4944] = 0x997d, [0x4945] = 0x9980, [0x4946] = 0x9984,
  [0x4947] = 0x9987, [0x4948] = 0x998a, [0x4949] = 0x998d, [0x494a] = 0x9990,
  [0x494b] = 0x9991, [0x494c] = 0x9993, [0x494d] = 0x9994, [0x494e] = 0x9995,
  [0x494f] = 0x5e80, [0x4950] = 0x5e91, [0x4951] = 0x5e8b, [0x4952] = 0x5e96,
  [0x4953] = 0x5ea5, [0x4954] = 0x5ea0, [0x4955] = 0x5eb9, [0x4956] = 0x5eb5,
  [0x4957] = 0x5ebe, [0x4958] = 0x5eb3, [0x4959] = 0x8d53, [0x495a] = 0x5ed2,
  [0x495b] = 0x5ed1, [0x495c] = 0x5edb, [0x495d] = 0x5ee8, [0x495e] = 0x5eea,
  [0x495f] = 0x81ba, [0x4960] = 0x5fc4, [0x4961] = 0x5fc9, [0x4962] = 0x5fd6,
  [0x4963] = 0x5fcf, [0x4964] = 0x6003, [0x4965] = 0x5fee, [0x4966] = 0x6004,
  [0x4967] = 0x5fe1, [0x4968] = 0x5fe4, [0x4969] = 0x5ffe, [0x496a] = 0x6005,
  [0x496b] = 0x6006, [0x496c] = 0x5fea, [0x496d] = 0x5fed, [0x496e] = 0x5ff8,
  [0x496f] = 0x6019, [0x4970] = 0x6035, [0x4971] = 0x6026, [0x4972] = 0x601b,
  [0x4973] = 0x600f, [0x4974] = 0x600d, [0x4975] = 0x6029, [0x4976] = 0x602b,
  [0x4977] = 0x600a, [0x4978] = 0x603f, [0x4979] = 0x6021, [0x497a] = 0x6078,
  [0x497b] = 0x6079, [0x497c] = 0x607b, [0x497d] = 0x607a, [0x497e] = 0x6042,
  [0x4980] = 0x9246, [0x4981] = 0x9247, [0x4982] = 0x9248, [0x4983] = 0x9249,
  [0x4984] = 0x924a, [0x4985] = 0x924b, [0x4986] = 0x924c, [0x4987] = 0x924d,
  [0x4988] = 0x924e, [0x4989] = 0x924f, [0x498a] = 0x9250, [0x498b] = 0x9251,
  [0x498c] = 0x9252, [0x498d] = 0x9253, [0x498e] = 0x9254, [0x498f] = 0x9255,
  [0x4990] = 0x9256, [0x4991] = 0x9257, [0x4992] = 0x9258, [0x4993] = 0x9259,
  [0x4994] = 0x925a, [0x4995] = 0x925b, [0x4996] = 0x925c, [0x4997] = 0x925d,
  [0x4998] = 0x925e, [0x4999] = 0x925f, [0x499a] = 0x9260, [0x499b] = 0x9261,
  [0x499c] = 0x9262, [0x499d] = 0x9263, [0x499e] = 0x9264, [0x499f] = 0x9265,
  [0x49a0] = 0x9266, [0x49a1] = 0x9267, [0x49a2] = 0x9268, [0x49a3] = 0x9269,
  [0x49a4] = 0x926a, [0x49a5] = 0x926b, [0x49a6] = 0x926c, [0x49a7] = 0x926d,
  [0x49a8] = 0x926e, [0x49a9] = 0x926f, [0x49aa] = 0x9270, [0x49ab] = 0x9271,
  [0x49ac] = 0x9272, [0x49ad] = 0x9273, [0x49ae] = 0x9275, [0x49af] = 0x9276,
  [0x49b0] = 0x9277, [0x49b1] = 0x9278, [0x49b2] = 0x9279, [0x49b3] = 0x927a,
  [0x49b4] = 0x927b, [0x49b5] = 0x927c, [0x49b6] = 0x927d, [0x49b7] = 0x927e,
  [0x49b8] = 0x927f, [0x49b9] = 0x9280, [0x49ba] = 0x9281, [0x49bb] = 0x9282,
  [0x49bc] = 0x9283, [0x49bd] = 0x9284, [0x49be] = 0x9285, [0x49c0] = 0x9286,
  [0x49c1] = 0x9287, [0x49c2] = 0x9288, [0x49c3] = 0x9289, [0x49c4] = 0x928a,
  [0x49c5] = 0x928b, [0x49c6] = 0x928c, [0x49c7] = 0x928d, [0x49c8] = 0x928f,
  [0x49c9] = 0x9290, [0x49ca] = 0x9291, [0x49cb] = 0x9292, [0x49cc] = 0x9293,
  [0x49cd] = 0x9294, [0x49ce] = 0x9295, [0x49cf] = 0x9296, [0x49d0] = 0x9297,
  [0x49d1] = 0x9298, [0x49d2] = 0x9299, [0x49d3] = 0x929a, [0x49d4] = 0x929b,
  [0x49d5] = 0x929c, [0x49d6] = 0x929d, [0x49d7] = 0x929e, [0x49d8] = 0x929f,
  [0x49d9] = 0x92a0, [0x49da] = 0x92a1, [0x49db] = 0x92a2, [0x49dc] = 0x92a3,
  [0x49dd] = 0x92a4, [0x49de] = 0x92a5, [0x49df] = 0x92a6, [0x49e0] = 0x92a7,
  [0x49e1] = 0x606a, [0x49e2] = 0x607d, [0x49e3] = 0x6096, [0x49e4] = 0x609a,
  [0x49e5] = 0x60ad, [0x49e6] = 0x609d, [0x49e7] = 0x6083, [0x49e8] = 0x6092,
  [0x49e9] = 0x608c, [0x49ea] = 0x609b, [0x49eb] = 0x60ec, [0x49ec] = 0x60bb,
  [0x49ed] = 0x60b1, [0x49ee] = 0x60dd, [0x49ef] = 0x60d8, [0x49f0] = 0x60c6,
  [0x49f1] = 0x60da, [0x49f2] = 0x60b4, [0x49f3] = 0x6120, [0x49f4] = 0x6126,
  [0x49f5] = 0x6115, [0x49f6] = 0x6123, [0x49f7] = 0x60f4, [0x49f8] = 0x6100,
  [0x49f9] = 0x610e, [0x49fa] = 0x612b, [0x49fb] = 0x614a, [0x49fc] = 0x6175,
  [0x49fd] = 0x61ac, [0x49fe] = 0x6194, [0x49ff] = 0x61a7, [0x4a00] = 0x61b7,
  [0x4a01] = 0x61d4, [0x4a02] = 0x61f5, [0x4a03] = 0x5fdd, [0x4a04] = 0x96b3,
  [0x4a05] = 0x95e9, [0x4a06] = 0x95eb, [0x4a07] = 0x95f1, [0x4a08] = 0x95f3,
  [0x4a09] = 0x95f5, [0x4a0a] = 0x95f6, [0x4a0b] = 0x95fc, [0x4a0c] = 0x95fe,
  [0x4a0d] = 0x9603, [0x4a0e] = 0x9604, [0x4a0f] = 0x9606, [0x4a10] = 0x9608,
  [0x4a11] = 0x960a, [0x4a12] = 0x960b, [0x4a13] = 0x960c, [0x4a14] = 0x960d,
  [0x4a15] = 0x960f, [0x4a16] = 0x9612, [0x4a17] = 0x9615, [0x4a18] = 0x9616,
  [0x4a19] = 0x9617, [0x4a1a] = 0x9619, [0x4a1b] = 0x961a, [0x4a1c] = 0x4e2c,
  [0x4a1d] = 0x723f, [0x4a1e] = 0x6215, [0x4a1f] = 0x6c35, [0x4a20] = 0x6c54,
  [0x4a21] = 0x6c5c, [0x4a22] = 0x6c4a, [0x4a23] = 0x6ca3, [0x4a24] = 0x6c85,
  [0x4a25] = 0x6c90, [0x4a26] = 0x6c94, [0x4a27] = 0x6c8c, [0x4a28] = 0x6c68,
  [0x4a29] = 0x6c69, [0x4a2a] = 0x6c74, [0x4a2b] = 0x6c76, [0x4a2c] = 0x6c86,
  [0x4a2d] = 0x6ca9, [0x4a2e] = 0x6cd0, [0x4a2f] = 0x6cd4, [0x4a30] = 0x6cad,
  [0x4a31] = 0x6cf7, [0x4a32] = 0x6cf8, [0x4a33] = 0x6cf1, [0x4a34] = 0x6cd7,
  [0x4a35] = 0x6cb2, [0x4a36] = 0x6ce0, [0x4a37] = 0x6cd6, [0x4a38] = 0x6cfa,
  [0x4a39] = 0x6ceb, [0x4a3a] = 0x6cee, [0x4a3b] = 0x6cb1, [0x4a3c] = 0x6cd3,
  [0x4a3d] = 0x6cef, [0x4a3e] = 0x6cfe, [0x4a40] = 0x92a8, [0x4a41] = 0x92a9,
  [0x4a42] = 0x92aa, [0x4a43] = 0x92ab, [0x4a44] = 0x92ac, [0x4a45] = 0x92ad,
  [0x4a46] = 0x92af, [0x4a47] = 0x92b0, [0x4a48] = 0x92b1, [0x4a49] = 0x92b2,
  [0x4a4a] = 0x92b3, [0x4a4b] = 0x92b4, [0x4a4c] = 0x92b5, [0x4a4d] = 0x92b6,
  [0x4a4e] = 0x92b7, [0x4a4f] = 0x92b8, [0x4a50] = 0x92b9, [0x4a51] = 0x92ba,
  [0x4a52] = 0x92bb, [0x4a53] = 0x92bc, [0x4a54] = 0x92bd, [0x4a55] = 0x92be,
  [0x4a56] = 0x92bf, [0x4a57] = 0x92c0, [0x4a58] = 0x92c1, [0x4a59] = 0x92c2,
  [0x4a5a] = 0x92c3, [0x4a5b] = 0x92c4, [0x4a5c] = 0x92c5, [0x4a5d] = 0x92c6,
  [0x4a5e] = 0x92c7, [0x4a5f] = 0x92c9, [0x4a60] = 0x92ca, [0x4a61] = 0x92cb,
  [0x4a62] = 0x92cc, [0x4a63] = 0x92cd, [0x4a64] = 0x92ce, [0x4a65] = 0x92cf,
  [0x4a66] = 0x92d0, [0x4a67] = 0x92d1, [0x4a68] = 0x92d2, [0x4a69] = 0x92d3,
  [0x4a6a] = 0x92d4, [0x4a6b] = 0x92d5, [0x4a6c] = 0x92d6, [0x4a6d] = 0x92d7,
  [0x4a6e] = 0x92d8, [0x4a6f] = 0x92d9, [0x4a70] = 0x92da, [0x4a71] = 0x92db,
  [0x4a72] = 0x92dc, [0x4a73] = 0x92dd, [0x4a74] = 0x92de, [0x4a75] = 0x92df,
  [0x4a76] = 0x92e0, [0x4a77] = 0x92e1, [0x4a78] = 0x92e2, [0x4a79] = 0x92e3,
  [0x4a7a] = 0x92e4, [0x4a7b] = 0x92e5, [0x4a7c] = 0x92e6, [0x4a7d] = 0x92e7,
  [0x4a7e] = 0x92e8, [0x4a80] = 0x92e9, [0x4a81] = 0x92ea, [0x4a82] = 0x92eb,
  [0x4a83] = 0x92ec, [0x4a84] = 0x92ed, [0x4a85] = 0x92ee, [0x4a86] = 0x92ef,
  [0x4a87] = 0x92f0, [0x4a88] = 0x92f1, [0x4a89] = 0x92f2, [0x4a8a] = 0x92f3,
  [0x4a8b] = 0x92f4, [0x4a8c] = 0x92f5, [0x4a8d] = 0x92f6, [0x4a8e] = 0x92f7,
  [0x4a8f] = 0x92f8, [0x4a90] = 0x92f9, [0x4a91] = 0x92fa, [0x4a92] = 0x92fb,
  [0x4a93] = 0x92fc, [0x4a94] = 0x92fd, [0x4a95] = 0x92fe, [0x4a96] = 0x92ff,
  [0x4a97] = 0x9300, [0x4a98] = 0x9301, [0x4a99] = 0x9302, [0x4a9a] = 0x9303,
  [0x4a9b] = 0x9304, [0x4a9c] = 0x9305, [0x4a9d] = 0x9306, [0x4a9e] = 0x9307,
  [0x4a9f] = 0x9308, [0x4aa0] = 0x9309, [0x4aa1] = 0x6d39, [0x4aa2] = 0x6d27,
  [0x4aa3] = 0x6d0c, [0x4aa4] = 0x6d43, [0x4aa5] = 0x6d48, [0x4aa6] = 0x6d07,
  [0x4aa7] = 0x6d04, [0x4aa8] = 0x6d19, [0x4aa9] = 0x6d0e, [0x4aaa] = 0x6d2b,
  [0x4aab] = 0x6d4d, [0x4aac] = 0x6d2e, [0x4aad] = 0x6d35, [0x4aae] = 0x6d1a,
  [0x4aaf] = 0x6d4f, [0x4ab0] = 0x6d52, [0x4ab1] = 0x6d54, [0x4ab2] = 0x6d33,
  [0x4ab3] = 0x6d91, [0x4ab4] = 0x6d6f, [0x4ab5] = 0x6d9e, [0x4ab6] = 0x6da0,
  [0x4ab7] = 0x6d5e, [0x4ab8] = 0x6d93, [0x4ab9] = 0x6d94, [0x4aba] = 0x6d5c,
  [0x4abb] = 0x6d60, [0x4abc] = 0x6d7c, [0x4abd] = 0x6d63, [0x4abe] = 0x6e1a,
  [0x4abf] = 0x6dc7, [0x4ac0] = 0x6dc5, [0x4ac1] = 0x6dde, [0x4ac2] = 0x6e0e,
  [0x4ac3] = 0x6dbf, [0x4ac4] = 0x6de0, [0x4ac5] = 0x6e11, [0x4ac6] = 0x6de6,
  [0x4ac7] = 0x6ddd, [0x4ac8] = 0x6dd9, [0x4ac9] = 0x6e16, [0x4aca] = 0x6dab,
  [0x4acb] = 0x6e0c, [0x4acc] = 0x6dae, [0x4acd] = 0x6e2b, [0x4ace] = 0x6e6e,
  [0x4acf] = 0x6e4e, [0x4ad0] = 0x6e6b, [0x4ad1] = 0x6eb2, [0x4ad2] = 0x6e5f,
  [0x4ad3] = 0x6e86, [0x4ad4] = 0x6e53, [0x4ad5] = 0x6e54, [0x4ad6] = 0x6e32,
  [0x4ad7] = 0x6e25, [0x4ad8] = 0x6e44, [0x4ad9] = 0x6edf, [0x4ada] = 0x6eb1,
  [0x4adb] = 0x6e98, [0x4adc] = 0x6ee0, [0x4add] = 0x6f2d, [0x4ade] = 0x6ee2,
  [0x4adf] = 0x6ea5, [0x4ae0] = 0x6ea7, [0x4ae1] = 0x6ebd, [0x4ae2] = 0x6ebb,
  [0x4ae3] = 0x6eb7, [0x4ae4] = 0x6ed7, [0x4ae5] = 0x6eb4, [0x4ae6] = 0x6ecf,
  [0x4ae7] = 0x6e8f, [0x4ae8] = 0x6ec2, [0x4ae9] = 0x6e9f, [0x4aea] = 0x6f62,
  [0x4aeb] = 0x6f46, [0x4aec] = 0x6f47, [0x4aed] = 0x6f24, [0x4aee] = 0x6f15,
  [0x4aef] = 0x6ef9, [0x4af0] = 0x6f2f, [0x4af1] = 0x6f36, [0x4af2] = 0x6f4b,
  [0x4af3] = 0x6f74, [0x4af4] = 0x6f2a, [0x4af5] = 0x6f09, [0x4af6] = 0x6f29,
  [0x4af7] = 0x6f89, [0x4af8] = 0x6f8d, [0x4af9] = 0x6f8c, [0x4afa] = 0x6f78,
  [0x4afb] = 0x6f72, [0x4afc] = 0x6f7c, [0x4afd] = 0x6f7a, [0x4afe] = 0x6fd1,
  [0x4b00] = 0x930a, [0x4b01] = 0x930b, [0x4b02] = 0x930c, [0x4b03] = 0x930d,
  [0x4b04] = 0x930e, [0x4b05] = 0x930f, [0x4b06] = 0x9310, [0x4b07] = 0x9311,
  [0x4b08] = 0x9312, [0x4b09] = 0x9313, [0x4b0a] = 0x9314, [0x4b0b] = 0x9315,
  [0x4b0c] = 0x9316, [0x4b0d] = 0x9317, [0x4b0e] = 0x9318, [0x4b0f] = 0x9319,
  [0x4b10] = 0x931a, [0x4b11] = 0x931b, [0x4b12] = 0x931c, [0x4b13] = 0x931d,
  [0x4b14] = 0x931e, [0x4b15] = 0x931f, [0x4b16] = 0x9320, [0x4b17] = 0x9321,
  [0x4b18] = 0x9322, [0x4b19] = 0x9323, [0x4b1a] = 0x9324, [0x4b1b] = 0x9325,
  [0x4b1c] = 0x9326, [0x4b1d] = 0x9327, [0x4b1e] = 0x9328, [0x4b1f] = 0x9329,
  [0x4b20] = 0x932a, [0x4b21] = 0x932b, [0x4b22] = 0x932c, [0x4b23] = 0x932d,
  [0x4b24] = 0x932e, [0x4b25] = 0x932f, [0x4b26] = 0x9330, [0x4b27] = 0x9331,
  [0x4b28] = 0x9332, [0x4b29] = 0x9333, [0x4b2a] = 0x9334, [0x4b2b] = 0x9335,
  [0x4b2c] = 0x9336, [0x4b2d] = 0x9337, [0x4b2e] = 0x9338, [0x4b2f] = 0x9339,
  [0x4b30] = 0x933a, [0x4b31] = 0x933b, [0x4b32] = 0x933c, [0x4b33] = 0x933d,
  [0x4b34] = 0x933f, [0x4b35] = 0x9340, [0x4b36] = 0x9341, [0x4b37] = 0x9342,
  [0x4b38] = 0x9343, [0x4b39] = 0x9344, [0x4b3a] = 0x9345, [0x4b3b] = 0x9346,
  [0x4b3c] = 0x9347, [0x4b3d] = 0x9348, [0x4b3e] = 0x9349, [0x4b40] = 0x934a,
  [0x4b41] = 0x934b, [0x4b42] = 0x934c, [0x4b43] = 0x934d, [0x4b44] = 0x934e,
  [0x4b45] = 0x934f, [0x4b46] = 0x9350, [0x4b47] = 0x9351, [0x4b48] = 0x9352,
  [0x4b49] = 0x9353, [0x4b4a] = 0x9354, [0x4b4b] = 0x9355, [0x4b4c] = 0x9356,
  [0x4b4d] = 0x9357, [0x4b4e] = 0x9358, [0x4b4f] = 0x9359, [0x4b50] = 0x935a,
  [0x4b51] = 0x935b, [0x4b52] = 0x935c, [0x4b53] = 0x935d, [0x4b54] = 0x935e,
  [0x4b55] = 0x935f, [0x4b56] = 0x9360, [0x4b57] = 0x9361, [0x4b58] = 0x9362,
  [0x4b59] = 0x9363, [0x4b5a] = 0x9364, [0x4b5b] = 0x9365, [0x4b5c] = 0x9366,
  [0x4b5d] = 0x9367, [0x4b5e] = 0x9368, [0x4b5f] = 0x9369, [0x4b60] = 0x936b,
  [0x4b61] = 0x6fc9, [0x4b62] = 0x6fa7, [0x4b63] = 0x6fb9, [0x4b64] = 0x6fb6,
  [0x4b65] = 0x6fc2, [0x4b66] = 0x6fe1, [0x4b67] = 0x6fee, [0x4b68] = 0x6fde,
  [0x4b69] = 0x6fe0, [0x4b6a] = 0x6fef, [0x4b6b] = 0x701a, [0x4b6c] = 0x7023,
  [0x4b6d] = 0x701b, [0x4b6e] = 0x7039, [0x4b6f] = 0x7035, [0x4b70] = 0x704f,
  [0x4b71] = 0x705e, [0x4b72] = 0x5b80, [0x4b73] = 0x5b84, [0x4b74] = 0x5b95,
  [0x4b75] = 0x5b93, [0x4b76] = 0x5ba5, [0x4b77] = 0x5bb8, [0x4b78] = 0x752f,
  [0x4b79] = 0x9a9e, [0x4b7a] = 0x6434, [0x4b7b] = 0x5be4, [0x4b7c] = 0x5bee,
  [0x4b7d] = 0x8930, [0x4b7e] = 0x5bf0, [0x4b7f] = 0x8e47, [0x4b80] = 0x8b07,
  [0x4b81] = 0x8fb6, [0x4b82] = 0x8fd3, [0x4b83] = 0x8fd5, [0x4b84] = 0x8fe5,
  [0x4b85] = 0x8fee, [0x4b86] = 0x8fe4, [0x4b87] = 0x8fe9, [0x4b88] = 0x8fe6,
  [0x4b89] = 0x8ff3, [0x4b8a] = 0x8fe8, [0x4b8b] = 0x9005, [0x4b8c] = 0x9004,
  [0x4b8d] = 0x900b, [0x4b8e] = 0x9026, [0x4b8f] = 0x9011, [0x4b90] = 0x900d,
  [0x4b91] = 0x9016, [0x4b92] = 0x9021, [0x4b93] = 0x9035, [0x4b94] = 0x9036,
  [0x4b95] = 0x902d, [0x4b96] = 0x902f, [0x4b97] = 0x9044, [0x4b98] = 0x9051,
  [0x4b99] = 0x9052, [0x4b9a] = 0x9050, [0x4b9b] = 0x9068, [0x4b9c] = 0x9058,
  [0x4b9d] = 0x9062, [0x4b9e] = 0x905b, [0x4b9f] = 0x66b9, [0x4ba0] = 0x9074,
  [0x4ba1] = 0x907d, [0x4ba2] = 0x9082, [0x4ba3] = 0x9088, [0x4ba4] = 0x9083,
  [0x4ba5] = 0x908b, [0x4ba6] = 0x5f50, [0x4ba7] = 0x5f57, [0x4ba8] = 0x5f56,
  [0x4ba9] = 0x5f58, [0x4baa] = 0x5c3b, [0x4bab] = 0x54ab, [0x4bac] = 0x5c50,
  [0x4bad] = 0x5c59, [0x4bae] = 0x5b71, [0x4baf] = 0x5c63, [0x4bb0] = 0x5c66,
  [0x4bb1] = 0x7fbc, [0x4bb2] = 0x5f2a, [0x4bb3] = 0x5f29, [0x4bb4] = 0x5f2d,
  [0x4bb5] = 0x8274, [0x4bb6] = 0x5f3c, [0x4bb7] = 0x9b3b, [0x4bb8] = 0x5c6e,
  [0x4bb9] = 0x5981, [0x4bba] = 0x5983, [0x4bbb] = 0x598d, [0x4bbc] = 0x59a9,
  [0x4bbd] = 0x59aa, [0x4bbe] = 0x59a3, [0x4bc0] = 0x936c, [0x4bc1] = 0x936d,
  [0x4bc2] = 0x936e, [0x4bc3] = 0x936f, [0x4bc4] = 0x9370, [0x4bc5] = 0x9371,
  [0x4bc6] = 0x9372, [0x4bc7] = 0x9373, [0x4bc8] = 0x9374, [0x4bc9] = 0x9375,
  [0x4bca] = 0x9376, [0x4bcb] = 0x9377, [0x4bcc] = 0x9378, [0x4bcd] = 0x9379,
  [0x4bce] = 0x937a, [0x4bcf] = 0x937b, [0x4bd0] = 0x937c, [0x4bd1] = 0x937d,
  [0x4bd2] = 0x937e, [0x4bd3] = 0x937f, [0x4bd4] = 0x9380, [0x4bd5] = 0x9381,
  [0x4bd6] = 0x9382, [0x4bd7] = 0x9383, [0x4bd8] = 0x9384, [0x4bd9] = 0x9385,
  [0x4bda] = 0x9386, [0x4bdb] = 0x9387, [0x4bdc] = 0x9388, [0x4bdd] = 0x9389,
  [0x4bde] = 0x938a, [0x4bdf] = 0x938b, [0x4be0] = 0x938c, [0x4be1] = 0x938d,
  [0x4be2] = 0x938e, [0x4be3] = 0x9390, [0x4be4] = 0x9391, [0x4be5] = 0x9392,
  [0x4be6] = 0x9393, [0x4be7] = 0x9394, [0x4be8] = 0x9395, [0x4be9] = 0x9396,
  [0x4bea] = 0x9397, [0x4beb] = 0x9398, [0x4bec] = 0x9399, [0x4bed] = 0x939a,
  [0x4bee] = 0x939b, [0x4bef] = 0x939c, [0x4bf0] = 0x939d, [0x4bf1] = 0x939e,
  [0x4bf2] = 0x939f, [0x4bf3] = 0x93a0, [0x4bf4] = 0x93a1, [0x4bf5] = 0x93a2,
  [0x4bf6] = 0x93a3, [0x4bf7] = 0x93a4, [0x4bf8] = 0x93a5, [0x4bf9] = 0x93a6,
  [0x4bfa] = 0x93a7, [0x4bfb] = 0x93a8, [0x4bfc] = 0x93a9, [0x4bfd] = 0x93aa,
  [0x4bfe] = 0x93ab, [0x4c00] = 0x93ac, [0x4c01] = 0x93ad, [0x4c02] = 0x93ae,
  [0x4c03] = 0x93af, [0x4c04] = 0x93b0, [0x4c05] = 0x93b1, [0x4c06] = 0x93b2,
  [0x4c07] = 0x93b3, [0x4c08] = 0x93b4, [0x4c09] = 0x93b5, [0x4c0a] = 0x93b6,
  [0x4c0b] = 0x93b7, [0x4c0c] = 0x93b8, [0x4c0d] = 0x93b9, [0x4c0e] = 0x93ba,
  [0x4c0f] = 0x93bb, [0x4c10] = 0x93bc, [0x4c11] = 0x93bd, [0x4c12] = 0x93be,
  [0x4c13] = 0x93bf, [0x4c14] = 0x93c0, [0x4c15] = 0x93c1, [0x4c16] = 0x93c2,
  [0x4c17] = 0x93c3, [0x4c18] = 0x93c4, [0x4c19] = 0x93c5, [0x4c1a] = 0x93c6,
  [0x4c1b] = 0x93c7, [0x4c1c] = 0x93c8, [0x4c1d] = 0x93c9, [0x4c1e] = 0x93cb,
  [0x4c1f] = 0x93cc, [0x4c20] = 0x93cd, [0x4c21] = 0x5997, [0x4c22] = 0x59ca,
  [0x4c23] = 0x59ab, [0x4c24] = 0x599e, [0x4c25] = 0x59a4, [0x4c26] = 0x59d2,
  [0x4c27] = 0x59b2, [0x4c28] = 0x59af, [0x4c29] = 0x59d7, [0x4c2a] = 0x59be,
  [0x4c2b] = 0x5a05, [0x4c2c] = 0x5a06, [0x4c2d] = 0x59dd, [0x4c2e] = 0x5a08,
  [0x4c2f] = 0x59e3, [0x4c30] = 0x59d8, [0x4c31] = 0x59f9, [0x4c32] = 0x5a0c,
  [0x4c33] = 0x5a09, [0x4c34] = 0x5a32, [0x4c35] = 0x5a34, [0x4c36] = 0x5a11,
  [0x4c37] = 0x5a23, [0x4c38] = 0x5a13, [0x4c39] = 0x5a40, [0x4c3a] = 0x5a67,
  [0x4c3b] = 0x5a4a, [0x4c3c] = 0x5a55, [0x4c3d] = 0x5a3c, [0x4c3e] = 0x5a62,
  [0x4c3f] = 0x5a75, [0x4c40] = 0x80ec, [0x4c41] = 0x5aaa, [0x4c42] = 0x5a9b,
  [0x4c43] = 0x5a77, [0x4c44] = 0x5a7a, [0x4c45] = 0x5abe, [0x4c46] = 0x5aeb,
  [0x4c47] = 0x5ab2, [0x4c48] = 0x5ad2, [0x4c49] = 0x5ad4, [0x4c4a] = 0x5ab8,
  [0x4c4b] = 0x5ae0, [0x4c4c] = 0x5ae3, [0x4c4d] = 0x5af1, [0x4c4e] = 0x5ad6,
  [0x4c4f] = 0x5ae6, [0x4c50] = 0x5ad8, [0x4c51] = 0x5adc, [0x4c52] = 0x5b09,
  [0x4c53] = 0x5b17, [0x4c54] = 0x5b16, [0x4c55] = 0x5b32, [0x4c56] = 0x5b37,
  [0x4c57] = 0x5b40, [0x4c58] = 0x5c15, [0x4c59] = 0x5c1c, [0x4c5a] = 0x5b5a,
  [0x4c5b] = 0x5b65, [0x4c5c] = 0x5b73, [0x4c5d] = 0x5b51, [0x4c5e] = 0x5b53,
  [0x4c5f] = 0x5b62, [0x4c60] = 0x9a75, [0x4c61] = 0x9a77, [0x4c62] = 0x9a78,
  [0x4c63] = 0x9a7a, [0x4c64] = 0x9a7f, [0x4c65] = 0x9a7d, [0x4c66] = 0x9a80,
  [0x4c67] = 0x9a81, [0x4c68] = 0x9a85, [0x4c69] = 0x9a88, [0x4c6a] = 0x9a8a,
  [0x4c6b] = 0x9a90, [0x4c6c] = 0x9a92, [0x4c6d] = 0x9a93, [0x4c6e] = 0x9a96,
  [0x4c6f] = 0x9a98, [0x4c70] = 0x9a9b, [0x4c71] = 0x9a9c, [0x4c72] = 0x9a9d,
  [0x4c73] = 0x9a9f, [0x4c74] = 0x9aa0, [0x4c75] = 0x9aa2, [0x4c76] = 0x9aa3,
  [0x4c77] = 0x9aa5, [0x4c78] = 0x9aa7, [0x4c79] = 0x7e9f, [0x4c7a] = 0x7ea1,
  [0x4c7b] = 0x7ea3, [0x4c7c] = 0x7ea5, [0x4c7d] = 0x7ea8, [0x4c7e] = 0x7ea9,
  [0x4c80] = 0x93ce, [0x4c81] = 0x93cf, [0x4c82] = 0x93d0, [0x4c83] = 0x93d1,
  [0x4c84] = 0x93d2, [0x4c85] = 0x93d3, [0x4c86] = 0x93d4, [0x4c87] = 0x93d5,
  [0x4c88] = 0x93d7, [0x4c89] = 0x93d8, [0x4c8a] = 0x93d9, [0x4c8b] = 0x93da,
  [0x4c8c] = 0x93db, [0x4c8d] = 0x93dc, [0x4c8e] = 0x93dd, [0x4c8f] = 0x93de,
  [0x4c90] = 0x93df, [0x4c91] = 0x93e0, [0x4c92] = 0x93e1, [0x4c93] = 0x93e2,
  [0x4c94] = 0x93e3, [0x4c95] = 0x93e4, [0x4c96] = 0x93e5, [0x4c97] = 0x93e6,
  [0x4c98] = 0x93e7, [0x4c99] = 0x93e8, [0x4c9a] = 0x93e9, [0x4c9b] = 0x93ea,
  [0x4c9c] = 0x93eb, [0x4c9d] = 0x93ec, [0x4c9e] = 0x93ed, [0x4c9f] = 0x93ee,
  [0x4ca0] = 0x93ef, [0x4ca1] = 0x93f0, [0x4ca2] = 0x93f1, [0x4ca3] = 0x93f2,
  [0x4ca4] = 0x93f3, [0x4ca5] = 0x93f4, [0x4ca6] = 0x93f5, [0x4ca7] = 0x93f6,
  [0x4ca8] = 0x93f7, [0x4ca9] = 0x93f8, [0x4caa] = 0x93f9, [0x4cab] = 0x93fa,
  [0x4cac] = 0x93fb, [0x4cad] = 0x93fc, [0x4cae] = 0x93fd, [0x4caf] = 0x93fe,
  [0x4cb0] = 0x93ff, [0x4cb1] = 0x9400, [0x4cb2] = 0x9401, [0x4cb3] = 0x9402,
  [0x4cb4] = 0x9403, [0x4cb5] = 0x9404, [0x4cb6] = 0x9405, [0x4cb7] = 0x9406,
  [0x4cb8] = 0x9407, [0x4cb9] = 0x9408, [0x4cba] = 0x9409, [0x4cbb] = 0x940a,
  [0x4cbc] = 0x940b, [0x4cbd] = 0x940c, [0x4cbe] = 0x940d, [0x4cc0] = 0x940e,
  [0x4cc1] = 0x940f, [0x4cc2] = 0x9410, [0x4cc3] = 0x9411, [0x4cc4] = 0x9412,
  [0x4cc5] = 0x9413, [0x4cc6] = 0x9414, [0x4cc7] = 0x9415, [0x4cc8] = 0x9416,
  [0x4cc9] = 0x9417, [0x4cca] = 0x9418, [0x4ccb] = 0x9419, [0x4ccc] = 0x941a,
  [0x4ccd] = 0x941b, [0x4cce] = 0x941c, [0x4ccf] = 0x941d, [0x4cd0] = 0x941e,
  [0x4cd1] = 0x941f, [0x4cd2] = 0x9420, [0x4cd3] = 0x9421, [0x4cd4] = 0x9422,
  [0x4cd5] = 0x9423, [0x4cd6] = 0x9424, [0x4cd7] = 0x9425, [0x4cd8] = 0x9426,
  [0x4cd9] = 0x9427, [0x4cda] = 0x9428, [0x4cdb] = 0x9429, [0x4cdc] = 0x942a,
  [0x4cdd] = 0x942b, [0x4cde] = 0x942c, [0x4cdf] = 0x942d, [0x4ce0] = 0x942e,
  [0x4ce1] = 0x7ead, [0x4ce2] = 0x7eb0, [0x4ce3] = 0x7ebe, [0x4ce4] = 0x7ec0,
  [0x4ce5] = 0x7ec1, [0x4ce6] = 0x7ec2, [0x4ce7] = 0x7ec9, [0x4ce8] = 0x7ecb,
  [0x4ce9] = 0x7ecc, [0x4cea] = 0x7ed0, [0x4ceb] = 0x7ed4, [0x4cec] = 0x7ed7,
  [0x4ced] = 0x7edb, [0x4cee] = 0x7ee0, [0x4cef] = 0x7ee1, [0x4cf0] = 0x7ee8,
  [0x4cf1] = 0x7eeb, [0x4cf2] = 0x7eee, [0x4cf3] = 0x7eef, [0x4cf4] = 0x7ef1,
  [0x4cf5] = 0x7ef2, [0x4cf6] = 0x7f0d, [0x4cf7] = 0x7ef6, [0x4cf8] = 0x7efa,
  [0x4cf9] = 0x7efb, [0x4cfa] = 0x7efe, [0x4cfb] = 0x7f01, [0x4cfc] = 0x7f02,
  [0x4cfd] = 0x7f03, [0x4cfe] = 0x7f07, [0x4cff] = 0x7f08, [0x4d00] = 0x7f0b,
  [0x4d01] = 0x7f0c, [0x4d02] = 0x7f0f, [0x4d03] = 0x7f11, [0x4d04] = 0x7f12,
  [0x4d05] = 0x7f17, [0x4d06] = 0x7f19, [0x4d07] = 0x7f1c, [0x4d08] = 0x7f1b,
  [0x4d09] = 0x7f1f, [0x4d0a] = 0x7f21, [0x4d0b] = 0x7f22, [0x4d0c] = 0x7f23,
  [0x4d0d] = 0x7f24, [0x4d0e] = 0x7f25, [0x4d0f] = 0x7f26, [0x4d10] = 0x7f27,
  [0x4d11] = 0x7f2a, [0x4d12] = 0x7f2b, [0x4d13] = 0x7f2c, [0x4d14] = 0x7f2d,
  [0x4d15] = 0x7f2f, [0x4d16] = 0x7f30, [0x4d17] = 0x7f31, [0x4d18] = 0x7f32,
  [0x4d19] = 0x7f33, [0x4d1a] = 0x7f35, [0x4d1b] = 0x5e7a, [0x4d1c] = 0x757f,
  [0x4d1d] = 0x5ddb, [0x4d1e] = 0x753e, [0x4d1f] = 0x9095, [0x4d20] = 0x738e,
  [0x4d21] = 0x7391, [0x4d22] = 0x73ae, [0x4d23] = 0x73a2, [0x4d24] = 0x739f,
  [0x4d25] = 0x73cf, [0x4d26] = 0x73c2, [0x4d27] = 0x73d1, [0x4d28] = 0x73b7,
  [0x4d29] = 0x73b3, [0x4d2a] = 0x73c0, [0x4d2b] = 0x73c9, [0x4d2c] = 0x73c8,
  [0x4d2d] = 0x73e5, [0x4d2e] = 0x73d9, [0x4d2f] = 0x987c, [0x4d30] = 0x740a,
  [0x4d31] = 0x73e9, [0x4d32] = 0x73e7, [0x4d33] = 0x73de, [0x4d34] = 0x73ba,
  [0x4d35] = 0x73f2, [0x4d36] = 0x740f, [0x4d37] = 0x742a, [0x4d38] = 0x745b,
  [0x4d39] = 0x7426, [0x4d3a] = 0x7425, [0x4d3b] = 0x7428, [0x4d3c] = 0x7430,
  [0x4d3d] = 0x742e, [0x4d3e] = 0x742c, [0x4d40] = 0x942f, [0x4d41] = 0x9430,
  [0x4d42] = 0x9431, [0x4d43] = 0x9432, [0x4d44] = 0x9433, [0x4d45] = 0x9434,
  [0x4d46] = 0x9435, [0x4d47] = 0x9436, [0x4d48] = 0x9437, [0x4d49] = 0x9438,
  [0x4d4a] = 0x9439, [0x4d4b] = 0x943a, [0x4d4c] = 0x943b, [0x4d4d] = 0x943c,
  [0x4d4e] = 0x943d, [0x4d4f] = 0x943f, [0x4d50] = 0x9440, [0x4d51] = 0x9441,
  [0x4d52] = 0x9442, [0x4d53] = 0x9443, [0x4d54] = 0x9444, [0x4d55] = 0x9445,
  [0x4d56] = 0x9446, [0x4d57] = 0x9447, [0x4d58] = 0x9448, [0x4d59] = 0x9449,
  [0x4d5a] = 0x944a, [0x4d5b] = 0x944b, [0x4d5c] = 0x944c, [0x4d5d] = 0x944d,
  [0x4d5e] = 0x944e, [0x4d5f] = 0x944f, [0x4d60] = 0x9450, [0x4d61] = 0x9451,
  [0x4d62] = 0x9452, [0x4d63] = 0x9453, [0x4d64] = 0x9454, [0x4d65] = 0x9455,
  [0x4d66] = 0x9456, [0x4d67] = 0x9457, [0x4d68] = 0x9458, [0x4d69] = 0x9459,
  [0x4d6a] = 0x945a, [0x4d6b] = 0x945b, [0x4d6c] = 0x945c, [0x4d6d] = 0x945d,
  [0x4d6e] = 0x945e, [0x4d6f] = 0x945f, [0x4d70] = 0x9460, [0x4d71] = 0x9461,
  [0x4d72] = 0x9462, [0x4d73] = 0x9463, [0x4d74] = 0x9464, [0x4d75] = 0x9465,
  [0x4d76] = 0x9466, [0x4d77] = 0x9467, [0x4d78] = 0x9468, [0x4d79] = 0x9469,
  [0x4d7a] = 0x946a, [0x4d7b] = 0x946c, [0x4d7c] = 0x946d, [0x4d7d] = 0x946e,
  [0x4d7e] = 0x946f, [0x4d80] = 0x9470, [0x4d81] = 0x9471, [0x4d82] = 0x9472,
  [0x4d83] = 0x9473, [0x4d84] = 0x9474, [0x4d85] = 0x9475, [0x4d86] = 0x9476,
  [0x4d87] = 0x9477, [0x4d88] = 0x9478, [0x4d89] = 0x9479, [0x4d8a] = 0x947a,
  [0x4d8b] = 0x947b, [0x4d8c] = 0x947c, [0x4d8d] = 0x947d, [0x4d8e] = 0x947e,
  [0x4d8f] = 0x947f, [0x4d90] = 0x9480, [0x4d91] = 0x9481, [0x4d92] = 0x9482,
  [0x4d93] = 0x9483, [0x4d94] = 0x9484, [0x4d95] = 0x9491, [0x4d96] = 0x9496,
  [0x4d97] = 0x9498, [0x4d98] = 0x94c7, [0x4d99] = 0x94cf, [0x4d9a] = 0x94d3,
  [0x4d9b] = 0x94d4, [0x4d9c] = 0x94da, [0x4d9d] = 0x94e6, [0x4d9e] = 0x94fb,
  [0x4d9f] = 0x951c, [0x4da0] = 0x9520, [0x4da1] = 0x741b, [0x4da2] = 0x741a,
  [0x4da3] = 0x7441, [0x4da4] = 0x745c, [0x4da5] = 0x7457, [0x4da6] = 0x7455,
  [0x4da7] = 0x7459, [0x4da8] = 0x7477, [0x4da9] = 0x746d, [0x4daa] = 0x747e,
  [0x4dab] = 0x749c, [0x4dac] = 0x748e, [0x4dad] = 0x7480, [0x4dae] = 0x7481,
  [0x4daf] = 0x7487, [0x4db0] = 0x748b, [0x4db1] = 0x749e, [0x4db2] = 0x74a8,
  [0x4db3] = 0x74a9, [0x4db4] = 0x7490, [0x4db5] = 0x74a7, [0x4db6] = 0x74d2,
  [0x4db7] = 0x74ba, [0x4db8] = 0x97ea, [0x4db9] = 0x97eb, [0x4dba] = 0x97ec,
  [0x4dbb] = 0x674c, [0x4dbc] = 0x6753, [0x4dbd] = 0x675e, [0x4dbe] = 0x6748,
  [0x4dbf] = 0x6769, [0x4dc0] = 0x67a5, [0x4dc1] = 0x6787, [0x4dc2] = 0x676a,
  [0x4dc3] = 0x6773, [0x4dc4] = 0x6798, [0x4dc5] = 0x67a7, [0x4dc6] = 0x6775,
  [0x4dc7] = 0x67a8, [0x4dc8] = 0x679e, [0x4dc9] = 0x67ad, [0x4dca] = 0x678b,
  [0x4dcb] = 0x6777, [0x4dcc] = 0x677c, [0x4dcd] = 0x67f0, [0x4dce] = 0x6809,
  [0x4dcf] = 0x67d8, [0x4dd0] = 0x680a, [0x4dd1] = 0x67e9, [0x4dd2] = 0x67b0,
  [0x4dd3] = 0x680c, [0x4dd4] = 0x67d9, [0x4dd5] = 0x67b5, [0x4dd6] = 0x67da,
  [0x4dd7] = 0x67b3, [0x4dd8] = 0x67dd, [0x4dd9] = 0x6800, [0x4dda] = 0x67c3,
  [0x4ddb] = 0x67b8, [0x4ddc] = 0x67e2, [0x4ddd] = 0x680e, [0x4dde] = 0x67c1,
  [0x4ddf] = 0x67fd, [0x4de0] = 0x6832, [0x4de1] = 0x6833, [0x4de2] = 0x6860,
  [0x4de3] = 0x6861, [0x4de4] = 0x684e, [0x4de5] = 0x6862, [0x4de6] = 0x6844,
  [0x4de7] = 0x6864, [0x4de8] = 0x6883, [0x4de9] = 0x681d, [0x4dea] = 0x6855,
  [0x4deb] = 0x6866, [0x4dec] = 0x6841, [0x4ded] = 0x6867, [0x4dee] = 0x6840,
  [0x4def] = 0x683e, [0x4df0] = 0x684a, [0x4df1] = 0x6849, [0x4df2] = 0x6829,
  [0x4df3] = 0x68b5, [0x4df4] = 0x688f, [0x4df5] = 0x6874, [0x4df6] = 0x6877,
  [0x4df7] = 0x6893, [0x4df8] = 0x686b, [0x4df9] = 0x68c2, [0x4dfa] = 0x696e,
  [0x4dfb] = 0x68fc, [0x4dfc] = 0x691f, [0x4dfd] = 0x6920, [0x4dfe] = 0x68f9,
  [0x4e00] = 0x9527, [0x4e01] = 0x9533, [0x4e02] = 0x953d, [0x4e03] = 0x9543,
  [0x4e04] = 0x9548, [0x4e05] = 0x954b, [0x4e06] = 0x9555, [0x4e07] = 0x955a,
  [0x4e08] = 0x9560, [0x4e09] = 0x956e, [0x4e0a] = 0x9574, [0x4e0b] = 0x9575,
  [0x4e0c] = 0x9577, [0x4e0d] = 0x9578, [0x4e0e] = 0x9579, [0x4e0f] = 0x957a,
  [0x4e10] = 0x957b, [0x4e11] = 0x957c, [0x4e12] = 0x957d, [0x4e13] = 0x957e,
  [0x4e14] = 0x9580, [0x4e15] = 0x9581, [0x4e16] = 0x9582, [0x4e17] = 0x9583,
  [0x4e18] = 0x9584, [0x4e19] = 0x9585, [0x4e1a] = 0x9586, [0x4e1b] = 0x9587,
  [0x4e1c] = 0x9588, [0x4e1d] = 0x9589, [0x4e1e] = 0x958a, [0x4e1f] = 0x958b,
  [0x4e20] = 0x958c, [0x4e21] = 0x958d, [0x4e22] = 0x958e, [0x4e23] = 0x958f,
  [0x4e24] = 0x9590, [0x4e25] = 0x9591, [0x4e26] = 0x9592, [0x4e27] = 0x9593,
  [0x4e28] = 0x9594, [0x4e29] = 0x9595, [0x4e2a] = 0x9596, [0x4e2b] = 0x9597,
  [0x4e2c] = 0x9598, [0x4e2d] = 0x9599, [0x4e2e] = 0x959a, [0x4e2f] = 0x959b,
  [0x4e30] = 0x959c, [0x4e31] = 0x959d, [0x4e32] = 0x959e, [0x4e33] = 0x959f,
  [0x4e34] = 0x95a0, [0x4e35] = 0x95a1, [0x4e36] = 0x95a2, [0x4e37] = 0x95a3,
  [0x4e38] = 0x95a4, [0x4e39] = 0x95a5, [0x4e3a] = 0x95a6, [0x4e3b] = 0x95a7,
  [0x4e3c] = 0x95a8, [0x4e3d] = 0x95a9, [0x4e3e] = 0x95aa, [0x4e40] = 0x95ab,
  [0x4e41] = 0x95ac, [0x4e42] = 0x95ad, [0x4e43] = 0x95ae, [0x4e44] = 0x95af,
  [0x4e45] = 0x95b0, [0x4e46] = 0x95b1, [0x4e47] = 0x95b2, [0x4e48] = 0x95b3,
  [0x4e49] = 0x95b4, [0x4e4a] = 0x95b5, [0x4e4b] = 0x95b6, [0x4e4c] = 0x95b7,
  [0x4e4d] = 0x95b8, [0x4e4e] = 0x95b9, [0x4e4f] = 0x95ba, [0x4e50] = 0x95bb,
  [0x4e51] = 0x95bc, [0x4e52] = 0x95bd, [0x4e53] = 0x95be, [0x4e54] = 0x95bf,
  [0x4e55] = 0x95c0, [0x4e56] = 0x95c1, [0x4e57] = 0x95c2, [0x4e58] = 0x95c3,
  [0x4e59] = 0x95c4, [0x4e5a] = 0x95c5, [0x4e5b] = 0x95c6, [0x4e5c] = 0x95c7,
  [0x4e5d] = 0x95c8, [0x4e5e] = 0x95c9, [0x4e5f] = 0x95ca, [0x4e60] = 0x95cb,
  [0x4e61] = 0x6924, [0x4e62] = 0x68f0, [0x4e63] = 0x690b, [0x4e64] = 0x6901,
  [0x4e65] = 0x6957, [0x4e66] = 0x68e3, [0x4e67] = 0x6910, [0x4e68] = 0x6971,
  [0x4e69] = 0x6939, [0x4e6a] = 0x6960, [0x4e6b] = 0x6942, [0x4e6c] = 0x695d,
  [0x4e6d] = 0x6984, [0x4e6e] = 0x696b, [0x4e6f] = 0x6980, [0x4e70] = 0x6998,
  [0x4e71] = 0x6978, [0x4e72] = 0x6934, [0x4e73] = 0x69cc, [0x4e74] = 0x6987,
  [0x4e75] = 0x6988, [0x4e76] = 0x69ce, [0x4e77] = 0x6989, [0x4e78] = 0x6966,
  [0x4e79] = 0x6963, [0x4e7a] = 0x6979, [0x4e7b] = 0x699b, [0x4e7c] = 0x69a7,
  [0x4e7d] = 0x69bb, [0x4e7e] = 0x69ab, [0x4e7f] = 0x69ad, [0x4e80] = 0x69d4,
  [0x4e81] = 0x69b1, [0x4e82] = 0x69c1, [0x4e83] = 0x69ca, [0x4e84] = 0x69df,
  [0x4e85] = 0x6995, [0x4e86] = 0x69e0, [0x4e87] = 0x698d, [0x4e88] = 0x69ff,
  [0x4e89] = 0x6a2f, [0x4e8a] = 0x69ed, [0x4e8b] = 0x6a17, [0x4e8c] = 0x6a18,
  [0x4e8d] = 0x6a65, [0x4e8e] = 0x69f2, [0x4e8f] = 0x6a44, [0x4e90] = 0x6a3e,
  [0x4e91] = 0x6aa0, [0x4e92] = 0x6a50, [0x4e93] = 0x6a5b, [0x4e94] = 0x6a35,
  [0x4e95] = 0x6a8e, [0x4e96] = 0x6a79, [0x4e97] = 0x6a3d, [0x4e98] = 0x6a28,
  [0x4e99] = 0x6a58, [0x4e9a] = 0x6a7c, [0x4e9b] = 0x6a91, [0x4e9c] = 0x6a90,
  [0x4e9d] = 0x6aa9, [0x4e9e] = 0x6a97, [0x4e9f] = 0x6aab, [0x4ea0] = 0x7337,
  [0x4ea1] = 0x7352, [0x4ea2] = 0x6b81, [0x4ea3] = 0x6b82, [0x4ea4] = 0x6b87,
  [0x4ea5] = 0x6b84, [0x4ea6] = 0x6b92, [0x4ea7] = 0x6b93, [0x4ea8] = 0x6b8d,
  [0x4ea9] = 0x6b9a, [0x4eaa] = 0x6b9b, [0x4eab] = 0x6ba1, [0x4eac] = 0x6baa,
  [0x4ead] = 0x8f6b, [0x4eae] = 0x8f6d, [0x4eaf] = 0x8f71, [0x4eb0] = 0x8f72,
  [0x4eb1] = 0x8f73, [0x4eb2] = 0x8f75, [0x4eb3] = 0x8f76, [0x4eb4] = 0x8f78,
  [0x4eb5] = 0x8f77, [0x4eb6] = 0x8f79, [0x4eb7] = 0x8f7a, [0x4eb8] = 0x8f7c,
  [0x4eb9] = 0x8f7e, [0x4eba] = 0x8f81, [0x4ebb] = 0x8f82, [0x4ebc] = 0x8f84,
  [0x4ebd] = 0x8f87, [0x4ebe] = 0x8f8b, [0x4ec0] = 0x95cc, [0x4ec1] = 0x95cd,
  [0x4ec2] = 0x95ce, [0x4ec3] = 0x95cf, [0x4ec4] = 0x95d0, [0x4ec5] = 0x95d1,
  [0x4ec6] = 0x95d2, [0x4ec7] = 0x95d3, [0x4ec8] = 0x95d4, [0x4ec9] = 0x95d5,
  [0x4eca] = 0x95d6, [0x4ecb] = 0x95d7, [0x4ecc] = 0x95d8, [0x4ecd] = 0x95d9,
  [0x4ece] = 0x95da, [0x4ecf] = 0x95db, [0x4ed0] = 0x95dc, [0x4ed1] = 0x95dd,
  [0x4ed2] = 0x95de, [0x4ed3] = 0x95df, [0x4ed4] = 0x95e0, [0x4ed5] = 0x95e1,
  [0x4ed6] = 0x95e2, [0x4ed7] = 0x95e3, [0x4ed8] = 0x95e4, [0x4ed9] = 0x95e5,
  [0x4eda] = 0x95e6, [0x4edb] = 0x95e7, [0x4edc] = 0x95ec, [0x4edd] = 0x95ff,
  [0x4ede] = 0x9607, [0x4edf] = 0x9613, [0x4ee0] = 0x9618, [0x4ee1] = 0x961b,
  [0x4ee2] = 0x961e, [0x4ee3] = 0x9620, [0x4ee4] = 0x9623, [0x4ee5] = 0x9624,
  [0x4ee6] = 0x9625, [0x4ee7] = 0x9626, [0x4ee8] = 0x9627, [0x4ee9] = 0x9628,
  [0x4eea] = 0x9629, [0x4eeb] = 0x962b, [0x4eec] = 0x962c, [0x4eed] = 0x962d,
  [0x4eee] = 0x962f, [0x4eef] = 0x9630, [0x4ef0] = 0x9637, [0x4ef1] = 0x9638,
  [0x4ef2] = 0x9639, [0x4ef3] = 0x963a, [0x4ef4] = 0x963e, [0x4ef5] = 0x9641,
  [0x4ef6] = 0x9643, [0x4ef7] = 0x964a, [0x4ef8] = 0x964e, [0x4ef9] = 0x964f,
  [0x4efa] = 0x9651, [0x4efb] = 0x9652, [0x4efc] = 0x9653, [0x4efd] = 0x9656,
  [0x4efe] = 0x9657, [0x4f00] = 0x9658, [0x4f01] = 0x9659, [0x4f02] = 0x965a,
  [0x4f03] = 0x965c, [0x4f04] = 0x965d, [0x4f05] = 0x965e, [0x4f06] = 0x9660,
  [0x4f07] = 0x9663, [0x4f08] = 0x9665, [0x4f09] = 0x9666, [0x4f0a] = 0x966b,
  [0x4f0b] = 0x966d, [0x4f0c] = 0x966e, [0x4f0d] = 0x966f, [0x4f0e] = 0x9670,
  [0x4f0f] = 0x9671, [0x4f10] = 0x9673, [0x4f11] = 0x9678, [0x4f12] = 0x9679,
  [0x4f13] = 0x967a, [0x4f14] = 0x967b, [0x4f15] = 0x967c, [0x4f16] = 0x967d,
  [0x4f17] = 0x967e, [0x4f18] = 0x967f, [0x4f19] = 0x9680, [0x4f1a] = 0x9681,
  [0x4f1b] = 0x9682, [0x4f1c] = 0x9683, [0x4f1d] = 0x9684, [0x4f1e] = 0x9687,
  [0x4f1f] = 0x9689, [0x4f20] = 0x968a, [0x4f21] = 0x8f8d, [0x4f22] = 0x8f8e,
  [0x4f23] = 0x8f8f, [0x4f24] = 0x8f98, [0x4f25] = 0x8f9a, [0x4f26] = 0x8ece,
  [0x4f27] = 0x620b, [0x4f28] = 0x6217, [0x4f29] = 0x621b, [0x4f2a] = 0x621f,
  [0x4f2b] = 0x6222, [0x4f2c] = 0x6221, [0x4f2d] = 0x6225, [0x4f2e] = 0x6224,
  [0x4f2f] = 0x622c, [0x4f30] = 0x81e7, [0x4f31] = 0x74ef, [0x4f32] = 0x74f4,
  [0x4f33] = 0x74ff, [0x4f34] = 0x750f, [0x4f35] = 0x7511, [0x4f36] = 0x7513,
  [0x4f37] = 0x6534, [0x4f38] = 0x65ee, [0x4f39] = 0x65ef, [0x4f3a] = 0x65f0,
  [0x4f3b] = 0x660a, [0x4f3c] = 0x6619, [0x4f3d] = 0x6772, [0x4f3e] = 0x6603,
  [0x4f3f] = 0x6615, [0x4f40] = 0x6600, [0x4f41] = 0x7085, [0x4f42] = 0x66f7,
  [0x4f43] = 0x661d, [0x4f44] = 0x6634, [0x4f45] = 0x6631, [0x4f46] = 0x6636,
  [0x4f47] = 0x6635, [0x4f48] = 0x8006, [0x4f49] = 0x665f, [0x4f4a] = 0x6654,
  [0x4f4b] = 0x6641, [0x4f4c] = 0x664f, [0x4f4d] = 0x6656, [0x4f4e] = 0x6661,
  [0x4f4f] = 0x6657, [0x4f50] = 0x6677, [0x4f51] = 0x6684, [0x4f52] = 0x668c,
  [0x4f53] = 0x66a7, [0x4f54] = 0x669d, [0x4f55] = 0x66be, [0x4f56] = 0x66db,
  [0x4f57] = 0x66dc, [0x4f58] = 0x66e6, [0x4f59] = 0x66e9, [0x4f5a] = 0x8d32,
  [0x4f5b] = 0x8d33, [0x4f5c] = 0x8d36, [0x4f5d] = 0x8d3b, [0x4f5e] = 0x8d3d,
  [0x4f5f] = 0x8d40, [0x4f60] = 0x8d45, [0x4f61] = 0x8d46, [0x4f62] = 0x8d48,
  [0x4f63] = 0x8d49, [0x4f64] = 0x8d47, [0x4f65] = 0x8d4d, [0x4f66] = 0x8d55,
  [0x4f67] = 0x8d59, [0x4f68] = 0x89c7, [0x4f69] = 0x89ca, [0x4f6a] = 0x89cb,
  [0x4f6b] = 0x89cc, [0x4f6c] = 0x89ce, [0x4f6d] = 0x89cf, [0x4f6e] = 0x89d0,
  [0x4f6f] = 0x89d1, [0x4f70] = 0x726e, [0x4f71] = 0x729f, [0x4f72] = 0x725d,
  [0x4f73] = 0x7266, [0x4f74] = 0x726f, [0x4f75] = 0x727e, [0x4f76] = 0x727f,
  [0x4f77] = 0x7284, [0x4f78] = 0x728b, [0x4f79] = 0x728d, [0x4f7a] = 0x728f,
  [0x4f7b] = 0x7292, [0x4f7c] = 0x6308, [0x4f7d] = 0x6332, [0x4f7e] = 0x63b0,
  [0x4f80] = 0x968c, [0x4f81] = 0x968e, [0x4f82] = 0x9691, [0x4f83] = 0x9692,
  [0x4f84] = 0x9693, [0x4f85] = 0x9695, [0x4f86] = 0x9696, [0x4f87] = 0x969a,
  [0x4f88] = 0x969b, [0x4f89] = 0x969d, [0x4f8a] = 0x969e, [0x4f8b] = 0x969f,
  [0x4f8c] = 0x96a0, [0x4f8d] = 0x96a1, [0x4f8e] = 0x96a2, [0x4f8f] = 0x96a3,
  [0x4f90] = 0x96a4, [0x4f91] = 0x96a5, [0x4f92] = 0x96a6, [0x4f93] = 0x96a8,
  [0x4f94] = 0x96a9, [0x4f95] = 0x96aa, [0x4f96] = 0x96ab, [0x4f97] = 0x96ac,
  [0x4f98] = 0x96ad, [0x4f99] = 0x96ae, [0x4f9a] = 0x96af, [0x4f9b] = 0x96b1,
  [0x4f9c] = 0x96b2, [0x4f9d] = 0x96b4, [0x4f9e] = 0x96b5, [0x4f9f] = 0x96b7,
  [0x4fa0] = 0x96b8, [0x4fa1] = 0x96ba, [0x4fa2] = 0x96bb, [0x4fa3] = 0x96bf,
  [0x4fa4] = 0x96c2, [0x4fa5] = 0x96c3, [0x4fa6] = 0x96c8, [0x4fa7] = 0x96ca,
  [0x4fa8] = 0x96cb, [0x4fa9] = 0x96d0, [0x4faa] = 0x96d1, [0x4fab] = 0x96d3,
  [0x4fac] = 0x96d4, [0x4fad] = 0x96d6, [0x4fae] = 0x96d7, [0x4faf] = 0x96d8,
  [0x4fb0] = 0x96d9, [0x4fb1] = 0x96da, [0x4fb2] = 0x96db, [0x4fb3] = 0x96dc,
  [0x4fb4] = 0x96dd, [0x4fb5] = 0x96de, [0x4fb6] = 0x96df, [0x4fb7] = 0x96e1,
  [0x4fb8] = 0x96e2, [0x4fb9] = 0x96e3, [0x4fba] = 0x96e4, [0x4fbb] = 0x96e5,
  [0x4fbc] = 0x96e6, [0x4fbd] = 0x96e7, [0x4fbe] = 0x96eb, [0x4fc0] = 0x96ec,
  [0x4fc1] = 0x96ed, [0x4fc2] = 0x96ee, [0x4fc3] = 0x96f0, [0x4fc4] = 0x96f1,
  [0x4fc5] = 0x96f2, [0x4fc6] = 0x96f4, [0x4fc7] = 0x96f5, [0x4fc8] = 0x96f8,
  [0x4fc9] = 0x96fa, [0x4fca] = 0x96fb, [0x4fcb] = 0x96fc, [0x4fcc] = 0x96fd,
  [0x4fcd] = 0x96ff, [0x4fce] = 0x9702, [0x4fcf] = 0x9703, [0x4fd0] = 0x9705,
  [0x4fd1] = 0x970a, [0x4fd2] = 0x970b, [0x4fd3] = 0x970c, [0x4fd4] = 0x9710,
  [0x4fd5] = 0x9711, [0x4fd6] = 0x9712, [0x4fd7] = 0x9714, [0x4fd8] = 0x9715,
  [0x4fd9] = 0x9717, [0x4fda] = 0x9718, [0x4fdb] = 0x9719, [0x4fdc] = 0x971a,
  [0x4fdd] = 0x971b, [0x4fde] = 0x971d, [0x4fdf] = 0x971f, [0x4fe0] = 0x9720,
  [0x4fe1] = 0x643f, [0x4fe2] = 0x64d8, [0x4fe3] = 0x8004, [0x4fe4] = 0x6bea,
  [0x4fe5] = 0x6bf3, [0x4fe6] = 0x6bfd, [0x4fe7] = 0x6bf5, [0x4fe8] = 0x6bf9,
  [0x4fe9] = 0x6c05, [0x4fea] = 0x6c07, [0x4feb] = 0x6c06, [0x4fec] = 0x6c0d,
  [0x4fed] = 0x6c15, [0x4fee] = 0x6c18, [0x4fef] = 0x6c19, [0x4ff0] = 0x6c1a,
  [0x4ff1] = 0x6c21, [0x4ff2] = 0x6c29, [0x4ff3] = 0x6c24, [0x4ff4] = 0x6c2a,
  [0x4ff5] = 0x6c32, [0x4ff6] = 0x6535, [0x4ff7] = 0x6555, [0x4ff8] = 0x656b,
  [0x4ff9] = 0x724d, [0x4ffa] = 0x7252, [0x4ffb] = 0x7256, [0x4ffc] = 0x7230,
  [0x4ffd] = 0x8662, [0x4ffe] = 0x5216, [0x4fff] = 0x809f, [0x5000] = 0x809c,
  [0x5001] = 0x8093, [0x5002] = 0x80bc, [0x5003] = 0x670a, [0x5004] = 0x80bd,
  [0x5005] = 0x80b1, [0x5006] = 0x80ab, [0x5007] = 0x80ad, [0x5008] = 0x80b4,
  [0x5009] = 0x80b7, [0x500a] = 0x80e7, [0x500b] = 0x80e8, [0x500c] = 0x80e9,
  [0x500d] = 0x80ea, [0x500e] = 0x80db, [0x500f] = 0x80c2, [0x5010] = 0x80c4,
  [0x5011] = 0x80d9, [0x5012] = 0x80cd, [0x5013] = 0x80d7, [0x5014] = 0x6710,
  [0x5015] = 0x80dd, [0x5016] = 0x80eb, [0x5017] = 0x80f1, [0x5018] = 0x80f4,
  [0x5019] = 0x80ed, [0x501a] = 0x810d, [0x501b] = 0x810e, [0x501c] = 0x80f2,
  [0x501d] = 0x80fc, [0x501e] = 0x6715, [0x501f] = 0x8112, [0x5020] = 0x8c5a,
  [0x5021] = 0x8136, [0x5022] = 0x811e, [0x5023] = 0x812c, [0x5024] = 0x8118,
  [0x5025] = 0x8132, [0x5026] = 0x8148, [0x5027] = 0x814c, [0x5028] = 0x8153,
  [0x5029] = 0x8174, [0x502a] = 0x8159, [0x502b] = 0x815a, [0x502c] = 0x8171,
  [0x502d] = 0x8160, [0x502e] = 0x8169, [0x502f] = 0x817c, [0x5030] = 0x817d,
  [0x5031] = 0x816d, [0x5032] = 0x8167, [0x5033] = 0x584d, [0x5034] = 0x5ab5,
  [0x5035] = 0x8188, [0x5036] = 0x8182, [0x5037] = 0x8191, [0x5038] = 0x6ed5,
  [0x5039] = 0x81a3, [0x503a] = 0x81aa, [0x503b] = 0x81cc, [0x503c] = 0x6726,
  [0x503d] = 0x81ca, [0x503e] = 0x81bb, [0x5040] = 0x9721, [0x5041] = 0x9722,
  [0x5042] = 0x9723, [0x5043] = 0x9724, [0x5044] = 0x9725, [0x5045] = 0x9726,
  [0x5046] = 0x9727, [0x5047] = 0x9728, [0x5048] = 0x9729, [0x5049] = 0x972b,
  [0x504a] = 0x972c, [0x504b] = 0x972e, [0x504c] = 0x972f, [0x504d] = 0x9731,
  [0x504e] = 0x9733, [0x504f] = 0x9734, [0x5050] = 0x9735, [0x5051] = 0x9736,
  [0x5052] = 0x9737, [0x5053] = 0x973a, [0x5054] = 0x973b, [0x5055] = 0x973c,
  [0x5056] = 0x973d, [0x5057] = 0x973f, [0x5058] = 0x9740, [0x5059] = 0x9741,
  [0x505a] = 0x9742, [0x505b] = 0x9743, [0x505c] = 0x9744, [0x505d] = 0x9745,
  [0x505e] = 0x9746, [0x505f] = 0x9747, [0x5060] = 0x9748, [0x5061] = 0x9749,
  [0x5062] = 0x974a, [0x5063] = 0x974b, [0x5064] = 0x974c, [0x5065] = 0x974d,
  [0x5066] = 0x974e, [0x5067] = 0x974f, [0x5068] = 0x9750, [0x5069] = 0x9751,
  [0x506a] = 0x9754, [0x506b] = 0x9755, [0x506c] = 0x9757, [0x506d] = 0x9758,
  [0x506e] = 0x975a, [0x506f] = 0x975c, [0x5070] = 0x975d, [0x5071] = 0x975f,
  [0x5072] = 0x9763, [0x5073] = 0x9764, [0x5074] = 0x9766, [0x5075] = 0x9767,
  [0x5076] = 0x9768, [0x5077] = 0x976a, [0x5078] = 0x976b, [0x5079] = 0x976c,
  [0x507a] = 0x976d, [0x507b] = 0x976e, [0x507c] = 0x976f, [0x507d] = 0x9770,
  [0x507e] = 0x9771, [0x5080] = 0x9772, [0x5081] = 0x9775, [0x5082] = 0x9777,
  [0x5083] = 0x9778, [0x5084] = 0x9779, [0x5085] = 0x977a, [0x5086] = 0x977b,
  [0x5087] = 0x977d, [0x5088] = 0x977e, [0x5089] = 0x977f, [0x508a] = 0x9780,
  [0x508b] = 0x9781, [0x508c] = 0x9782, [0x508d] = 0x9783, [0x508e] = 0x9784,
  [0x508f] = 0x9786, [0x5090] = 0x9787, [0x5091] = 0x9788, [0x5092] = 0x9789,
  [0x5093] = 0x978a, [0x5094] = 0x978c, [0x5095] = 0x978e, [0x5096] = 0x978f,
  [0x5097] = 0x9790, [0x5098] = 0x9793, [0x5099] = 0x9795, [0x509a] = 0x9796,
  [0x509b] = 0x9797, [0x509c] = 0x9799, [0x509d] = 0x979a, [0x509e] = 0x979b,
  [0x509f] = 0x979c, [0x50a0] = 0x979d, [0x50a1] = 0x81c1, [0x50a2] = 0x81a6,
  [0x50a3] = 0x6b24, [0x50a4] = 0x6b37, [0x50a5] = 0x6b39, [0x50a6] = 0x6b43,
  [0x50a7] = 0x6b46, [0x50a8] = 0x6b59, [0x50a9] = 0x98d1, [0x50aa] = 0x98d2,
  [0x50ab] = 0x98d3, [0x50ac] = 0x98d5, [0x50ad] = 0x98d9, [0x50ae] = 0x98da,
  [0x50af] = 0x6bb3, [0x50b0] = 0x5f40, [0x50b1] = 0x6bc2, [0x50b2] = 0x89f3,
  [0x50b3] = 0x6590, [0x50b4] = 0x9f51, [0x50b5] = 0x6593, [0x50b6] = 0x65bc,
  [0x50b7] = 0x65c6, [0x50b8] = 0x65c4, [0x50b9] = 0x65c3, [0x50ba] = 0x65cc,
  [0x50bb] = 0x65ce, [0x50bc] = 0x65d2, [0x50bd] = 0x65d6, [0x50be] = 0x7080,
  [0x50bf] = 0x709c, [0x50c0] = 0x7096, [0x50c1] = 0x709d, [0x50c2] = 0x70bb,
  [0x50c3] = 0x70c0, [0x50c4] = 0x70b7, [0x50c5] = 0x70ab, [0x50c6] = 0x70b1,
  [0x50c7] = 0x70e8, [0x50c8] = 0x70ca, [0x50c9] = 0x7110, [0x50ca] = 0x7113,
  [0x50cb] = 0x7116, [0x50cc] = 0x712f, [0x50cd] = 0x7131, [0x50ce] = 0x7173,
  [0x50cf] = 0x715c, [0x50d0] = 0x7168, [0x50d1] = 0x7145, [0x50d2] = 0x7172,
  [0x50d3] = 0x714a, [0x50d4] = 0x7178, [0x50d5] = 0x717a, [0x50d6] = 0x7198,
  [0x50d7] = 0x71b3, [0x50d8] = 0x71b5, [0x50d9] = 0x71a8, [0x50da] = 0x71a0,
  [0x50db] = 0x71e0, [0x50dc] = 0x71d4, [0x50dd] = 0x71e7, [0x50de] = 0x71f9,
  [0x50df] = 0x721d, [0x50e0] = 0x7228, [0x50e1] = 0x706c, [0x50e2] = 0x7118,
  [0x50e3] = 0x7166, [0x50e4] = 0x71b9, [0x50e5] = 0x623e, [0x50e6] = 0x623d,
  [0x50e7] = 0x6243, [0x50e8] = 0x6248, [0x50e9] = 0x6249, [0x50ea] = 0x793b,
  [0x50eb] = 0x7940, [0x50ec] = 0x7946, [0x50ed] = 0x7949, [0x50ee] = 0x795b,
  [0x50ef] = 0x795c, [0x50f0] = 0x7953, [0x50f1] = 0x795a, [0x50f2] = 0x7962,
  [0x50f3] = 0x7957, [0x50f4] = 0x7960, [0x50f5] = 0x796f, [0x50f6] = 0x7967,
  [0x50f7] = 0x797a, [0x50f8] = 0x7985, [0x50f9] = 0x798a, [0x50fa] = 0x799a,
  [0x50fb] = 0x79a7, [0x50fc] = 0x79b3, [0x50fd] = 0x5fd1, [0x50fe] = 0x5fd0,
  [0x5100] = 0x979e, [0x5101] = 0x979f, [0x5102] = 0x97a1, [0x5103] = 0x97a2,
  [0x5104] = 0x97a4, [0x5105] = 0x97a5, [0x5106] = 0x97a6, [0x5107] = 0x97a7,
  [0x5108] = 0x97a8, [0x5109] = 0x97a9, [0x510a] = 0x97aa, [0x510b] = 0x97ac,
  [0x510c] = 0x97ae, [0x510d] = 0x97b0, [0x510e] = 0x97b1, [0x510f] = 0x97b3,
  [0x5110] = 0x97b5, [0x5111] = 0x97b6, [0x5112] = 0x97b7, [0x5113] = 0x97b8,
  [0x5114] = 0x97b9, [0x5115] = 0x97ba, [0x5116] = 0x97bb, [0x5117] = 0x97bc,
  [0x5118] = 0x97bd, [0x5119] = 0x97be, [0x511a] = 0x97bf, [0x511b] = 0x97c0,
  [0x511c] = 0x97c1, [0x511d] = 0x97c2, [0x511e] = 0x97c3, [0x511f] = 0x97c4,
  [0x5120] = 0x97c5, [0x5121] = 0x97c6, [0x5122] = 0x97c7, [0x5123] = 0x97c8,
  [0x5124] = 0x97c9, [0x5125] = 0x97ca, [0x5126] = 0x97cb, [0x5127] = 0x97cc,
  [0x5128] = 0x97cd, [0x5129] = 0x97ce, [0x512a] = 0x97cf, [0x512b] = 0x97d0,
  [0x512c] = 0x97d1, [0x512d] = 0x97d2, [0x512e] = 0x97d3, [0x512f] = 0x97d4,
  [0x5130] = 0x97d5, [0x5131] = 0x97d6, [0x5132] = 0x97d7, [0x5133] = 0x97d8,
  [0x5134] = 0x97d9, [0x5135] = 0x97da, [0x5136] = 0x97db, [0x5137] = 0x97dc,
  [0x5138] = 0x97dd, [0x5139] = 0x97de, [0x513a] = 0x97df, [0x513b] = 0x97e0,
  [0x513c] = 0x97e1, [0x513d] = 0x97e2, [0x513e] = 0x97e3, [0x5140] = 0x97e4,
  [0x5141] = 0x97e5, [0x5142] = 0x97e8, [0x5143] = 0x97ee, [0x5144] = 0x97ef,
  [0x5145] = 0x97f0, [0x5146] = 0x97f1, [0x5147] = 0x97f2, [0x5148] = 0x97f4,
  [0x5149] = 0x97f7, [0x514a] = 0x97f8, [0x514b] = 0x97f9, [0x514c] = 0x97fa,
  [0x514d] = 0x97fb, [0x514e] = 0x97fc, [0x514f] = 0x97fd, [0x5150] = 0x97fe,
  [0x5151] = 0x97ff, [0x5152] = 0x9800, [0x5153] = 0x9801, [0x5154] = 0x9802,
  [0x5155] = 0x9803, [0x5156] = 0x9804, [0x5157] = 0x9805, [0x5158] = 0x9806,
  [0x5159] = 0x9807, [0x515a] = 0x9808, [0x515b] = 0x9809, [0x515c] = 0x980a,
  [0x515d] = 0x980b, [0x515e] = 0x980c, [0x515f] = 0x980d, [0x5160] = 0x980e,
  [0x5161] = 0x603c, [0x5162] = 0x605d, [0x5163] = 0x605a, [0x5164] = 0x6067,
  [0x5165] = 0x6041, [0x5166] = 0x6059, [0x5167] = 0x6063, [0x5168] = 0x60ab,
  [0x5169] = 0x6106, [0x516a] = 0x610d, [0x516b] = 0x615d, [0x516c] = 0x61a9,
  [0x516d] = 0x619d, [0x516e] = 0x61cb, [0x516f] = 0x61d1, [0x5170] = 0x6206,
  [0x5171] = 0x8080, [0x5172] = 0x807f, [0x5173] = 0x6c93, [0x5174] = 0x6cf6,
  [0x5175] = 0x6dfc, [0x5176] = 0x77f6, [0x5177] = 0x77f8, [0x5178] = 0x7800,
  [0x5179] = 0x7809, [0x517a] = 0x7817, [0x517b] = 0x7818, [0x517c] = 0x7811,
  [0x517d] = 0x65ab, [0x517e] = 0x782d, [0x517f] = 0x781c, [0x5180] = 0x781d,
  [0x5181] = 0x7839, [0x5182] = 0x783a, [0x5183] = 0x783b, [0x5184] = 0x781f,
  [0x5185] = 0x783c, [0x5186] = 0x7825, [0x5187] = 0x782c, [0x5188] = 0x7823,
  [0x5189] = 0x7829, [0x518a] = 0x784e, [0x518b] = 0x786d, [0x518c] = 0x7856,
  [0x518d] = 0x7857, [0x518e] = 0x7826, [0x518f] = 0x7850, [0x5190] = 0x7847,
  [0x5191] = 0x784c, [0x5192] = 0x786a, [0x5193] = 0x789b, [0x5194] = 0x7893,
  [0x5195] = 0x789a, [0x5196] = 0x7887, [0x5197] = 0x789c, [0x5198] = 0x78a1,
  [0x5199] = 0x78a3, [0x519a] = 0x78b2, [0x519b] = 0x78b9, [0x519c] = 0x78a5,
  [0x519d] = 0x78d4, [0x519e] = 0x78d9, [0x519f] = 0x78c9, [0x51a0] = 0x78ec,
  [0x51a1] = 0x78f2, [0x51a2] = 0x7905, [0x51a3] = 0x78f4, [0x51a4] = 0x7913,
  [0x51a5] = 0x7924, [0x51a6] = 0x791e, [0x51a7] = 0x7934, [0x51a8] = 0x9f9b,
  [0x51a9] = 0x9ef9, [0x51aa] = 0x9efb, [0x51ab] = 0x9efc, [0x51ac] = 0x76f1,
  [0x51ad] = 0x7704, [0x51ae] = 0x770d, [0x51af] = 0x76f9, [0x51b0] = 0x7707,
  [0x51b1] = 0x7708, [0x51b2] = 0x771a, [0x51b3] = 0x7722, [0x51b4] = 0x7719,
  [0x51b5] = 0x772d, [0x51b6] = 0x7726, [0x51b7] = 0x7735, [0x51b8] = 0x7738,
  [0x51b9] = 0x7750, [0x51ba] = 0x7751, [0x51bb] = 0x7747, [0x51bc] = 0x7743,
  [0x51bd] = 0x775a, [0x51be] = 0x7768, [0x51c0] = 0x980f, [0x51c1] = 0x9810,
  [0x51c2] = 0x9811, [0x51c3] = 0x9812, [0x51c4] = 0x9813, [0x51c5] = 0x9814,
  [0x51c6] = 0x9815, [0x51c7] = 0x9816, [0x51c8] = 0x9817, [0x51c9] = 0x9818,
  [0x51ca] = 0x9819, [0x51cb] = 0x981a, [0x51cc] = 0x981b, [0x51cd] = 0x981c,
  [0x51ce] = 0x981d, [0x51cf] = 0x981e, [0x51d0] = 0x981f, [0x51d1] = 0x9820,
  [0x51d2] = 0x9821, [0x51d3] = 0x9822, [0x51d4] = 0x9823, [0x51d5] = 0x9824,
  [0x51d6] = 0x9825, [0x51d7] = 0x9826, [0x51d8] = 0x9827, [0x51d9] = 0x9828,
  [0x51da] = 0x9829, [0x51db] = 0x982a, [0x51dc] = 0x982b, [0x51dd] = 0x982c,
  [0x51de] = 0x982d, [0x51df] = 0x982e, [0x51e0] = 0x982f, [0x51e1] = 0x9830,
  [0x51e2] = 0x9831, [0x51e3] = 0x9832, [0x51e4] = 0x9833, [0x51e5] = 0x9834,
  [0x51e6] = 0x9835, [0x51e7] = 0x9836, [0x51e8] = 0x9837, [0x51e9] = 0x9838,
  [0x51ea] = 0x9839, [0x51eb] = 0x983a, [0x51ec] = 0x983b, [0x51ed] = 0x983c,
  [0x51ee] = 0x983d, [0x51ef] = 0x983e, [0x51f0] = 0x983f, [0x51f1] = 0x9840,
  [0x51f2] = 0x9841, [0x51f3] = 0x9842, [0x51f4] = 0x9843, [0x51f5] = 0x9844,
  [0x51f6] = 0x9845, [0x51f7] = 0x9846, [0x51f8] = 0x9847, [0x51f9] = 0x9848,
  [0x51fa] = 0x9849, [0x51fb] = 0x984a, [0x51fc] = 0x984b, [0x51fd] = 0x984c,
  [0x51fe] = 0x984d, [0x5200] = 0x984e, [0x5201] = 0x984f, [0x5202] = 0x9850,
  [0x5203] = 0x9851, [0x5204] = 0x9852, [0x5205] = 0x9853, [0x5206] = 0x9854,
  [0x5207] = 0x9855, [0x5208] = 0x9856, [0x5209] = 0x9857, [0x520a] = 0x9858,
  [0x520b] = 0x9859, [0x520c] = 0x985a, [0x520d] = 0x985b, [0x520e] = 0x985c,
  [0x520f] = 0x985d, [0x5210] = 0x985e, [0x5211] = 0x985f, [0x5212] = 0x9860,
  [0x5213] = 0x9861, [0x5214] = 0x9862, [0x5215] = 0x9863, [0x5216] = 0x9864,
  [0x5217] = 0x9865, [0x5218] = 0x9866, [0x5219] = 0x9867, [0x521a] = 0x9868,
  [0x521b] = 0x9869, [0x521c] = 0x986a, [0x521d] = 0x986b, [0x521e] = 0x986c,
  [0x521f] = 0x986d, [0x5220] = 0x986e, [0x5221] = 0x7762, [0x5222] = 0x7765,
  [0x5223] = 0x777f, [0x5224] = 0x778d, [0x5225] = 0x777d, [0x5226] = 0x7780,
  [0x5227] = 0x778c, [0x5228] = 0x7791, [0x5229] = 0x779f, [0x522a] = 0x77a0,
  [0x522b] = 0x77b0, [0x522c] = 0x77b5, [0x522d] = 0x77bd, [0x522e] = 0x753a,
  [0x522f] = 0x7540, [0x5230] = 0x754e, [0x5231] = 0x754b, [0x5232] = 0x7548,
  [0x5233] = 0x755b, [0x5234] = 0x7572, [0x5235] = 0x7579, [0x5236] = 0x7583,
  [0x5237] = 0x7f58, [0x5238] = 0x7f61, [0x5239] = 0x7f5f, [0x523a] = 0x8a48,
  [0x523b] = 0x7f68, [0x523c] = 0x7f74, [0x523d] = 0x7f71, [0x523e] = 0x7f79,
  [0x523f] = 0x7f81, [0x5240] = 0x7f7e, [0x5241] = 0x76cd, [0x5242] = 0x76e5,
  [0x5243] = 0x8832, [0x5244] = 0x9485, [0x5245] = 0x9486, [0x5246] = 0x9487,
  [0x5247] = 0x948b, [0x5248] = 0x948a, [0x5249] = 0x948c, [0x524a] = 0x948d,
  [0x524b] = 0x948f, [0x524c] = 0x9490, [0x524d] = 0x9494, [0x524e] = 0x9497,
  [0x524f] = 0x9495, [0x5250] = 0x949a, [0x5251] = 0x949b, [0x5252] = 0x949c,
  [0x5253] = 0x94a3, [0x5254] = 0x94a4, [0x5255] = 0x94ab, [0x5256] = 0x94aa,
  [0x5257] = 0x94ad, [0x5258] = 0x94ac, [0x5259] = 0x94af, [0x525a] = 0x94b0,
  [0x525b] = 0x94b2, [0x525c] = 0x94b4, [0x525d] = 0x94b6, [0x525e] = 0x94b7,
  [0x525f] = 0x94b8, [0x5260] = 0x94b9, [0x5261] = 0x94ba, [0x5262] = 0x94bc,
  [0x5263] = 0x94bd, [0x5264] = 0x94bf, [0x5265] = 0x94c4, [0x5266] = 0x94c8,
  [0x5267] = 0x94c9, [0x5268] = 0x94ca, [0x5269] = 0x94cb, [0x526a] = 0x94cc,
  [0x526b] = 0x94cd, [0x526c] = 0x94ce, [0x526d] = 0x94d0, [0x526e] = 0x94d1,
  [0x526f] = 0x94d2, [0x5270] = 0x94d5, [0x5271] = 0x94d6, [0x5272] = 0x94d7,
  [0x5273] = 0x94d9, [0x5274] = 0x94d8, [0x5275] = 0x94db, [0x5276] = 0x94de,
  [0x5277] = 0x94df, [0x5278] = 0x94e0, [0x5279] = 0x94e2, [0x527a] = 0x94e4,
  [0x527b] = 0x94e5, [0x527c] = 0x94e7, [0x527d] = 0x94e8, [0x527e] = 0x94ea,
  [0x5280] = 0x986f, [0x5281] = 0x9870, [0x5282] = 0x9871, [0x5283] = 0x9872,
  [0x5284] = 0x9873, [0x5285] = 0x9874, [0x5286] = 0x988b, [0x5287] = 0x988e,
  [0x5288] = 0x9892, [0x5289] = 0x9895, [0x528a] = 0x9899, [0x528b] = 0x98a3,
  [0x528c] = 0x98a8, [0x528d] = 0x98a9, [0x528e] = 0x98aa, [0x528f] = 0x98ab,
  [0x5290] = 0x98ac, [0x5291] = 0x98ad, [0x5292] = 0x98ae, [0x5293] = 0x98af,
  [0x5294] = 0x98b0, [0x5295] = 0x98b1, [0x5296] = 0x98b2, [0x5297] = 0x98b3,
  [0x5298] = 0x98b4, [0x5299] = 0x98b5, [0x529a] = 0x98b6, [0x529b] = 0x98b7,
  [0x529c] = 0x98b8, [0x529d] = 0x98b9, [0x529e] = 0x98ba, [0x529f] = 0x98bb,
  [0x52a0] = 0x98bc, [0x52a1] = 0x98bd, [0x52a2] = 0x98be, [0x52a3] = 0x98bf,
  [0x52a4] = 0x98c0, [0x52a5] = 0x98c1, [0x52a6] = 0x98c2, [0x52a7] = 0x98c3,
  [0x52a8] = 0x98c4, [0x52a9] = 0x98c5, [0x52aa] = 0x98c6, [0x52ab] = 0x98c7,
  [0x52ac] = 0x98c8, [0x52ad] = 0x98c9, [0x52ae] = 0x98ca, [0x52af] = 0x98cb,
  [0x52b0] = 0x98cc, [0x52b1] = 0x98cd, [0x52b2] = 0x98cf, [0x52b3] = 0x98d0,
  [0x52b4] = 0x98d4, [0x52b5] = 0x98d6, [0x52b6] = 0x98d7, [0x52b7] = 0x98db,
  [0x52b8] = 0x98dc, [0x52b9] = 0x98dd, [0x52ba] = 0x98e0, [0x52bb] = 0x98e1,
  [0x52bc] = 0x98e2, [0x52bd] = 0x98e3, [0x52be] = 0x98e4, [0x52c0] = 0x98e5,
  [0x52c1] = 0x98e6, [0x52c2] = 0x98e9, [0x52c3] = 0x98ea, [0x52c4] = 0x98eb,
  [0x52c5] = 0x98ec, [0x52c6] = 0x98ed, [0x52c7] = 0x98ee, [0x52c8] = 0x98ef,
  [0x52c9] = 0x98f0, [0x52ca] = 0x98f1, [0x52cb] = 0x98f2, [0x52cc] = 0x98f3,
  [0x52cd] = 0x98f4, [0x52ce] = 0x98f5, [0x52cf] = 0x98f6, [0x52d0] = 0x98f7,
  [0x52d1] = 0x98f8, [0x52d2] = 0x98f9, [0x52d3] = 0x98fa, [0x52d4] = 0x98fb,
  [0x52d5] = 0x98fc, [0x52d6] = 0x98fd, [0x52d7] = 0x98fe, [0x52d8] = 0x98ff,
  [0x52d9] = 0x9900, [0x52da] = 0x9901, [0x52db] = 0x9902, [0x52dc] = 0x9903,
  [0x52dd] = 0x9904, [0x52de] = 0x9905, [0x52df] = 0x9906, [0x52e0] = 0x9907,
  [0x52e1] = 0x94e9, [0x52e2] = 0x94eb, [0x52e3] = 0x94ee, [0x52e4] = 0x94ef,
  [0x52e5] = 0x94f3, [0x52e6] = 0x94f4, [0x52e7] = 0x94f5, [0x52e8] = 0x94f7,
  [0x52e9] = 0x94f9, [0x52ea] = 0x94fc, [0x52eb] = 0x94fd, [0x52ec] = 0x94ff,
  [0x52ed] = 0x9503, [0x52ee] = 0x9502, [0x52ef] = 0x9506, [0x52f0] = 0x9507,
  [0x52f1] = 0x9509, [0x52f2] = 0x950a, [0x52f3] = 0x950d, [0x52f4] = 0x950e,
  [0x52f5] = 0x950f, [0x52f6] = 0x9512, [0x52f7] = 0x9513, [0x52f8] = 0x9514,
  [0x52f9] = 0x9515, [0x52fa] = 0x9516, [0x52fb] = 0x9518, [0x52fc] = 0x951b,
  [0x52fd] = 0x951d, [0x52fe] = 0x951e, [0x52ff] = 0x951f, [0x5300] = 0x9522,
  [0x5301] = 0x952a, [0x5302] = 0x952b, [0x5303] = 0x9529, [0x5304] = 0x952c,
  [0x5305] = 0x9531, [0x5306] = 0x9532, [0x5307] = 0x9534, [0x5308] = 0x9536,
  [0x5309] = 0x9537, [0x530a] = 0x9538, [0x530b] = 0x953c, [0x530c] = 0x953e,
  [0x530d] = 0x953f, [0x530e] = 0x9542, [0x530f] = 0x9535, [0x5310] = 0x9544,
  [0x5311] = 0x9545, [0x5312] = 0x9546, [0x5313] = 0x9549, [0x5314] = 0x954c,
  [0x5315] = 0x954e, [0x5316] = 0x954f, [0x5317] = 0x9552, [0x5318] = 0x9553,
  [0x5319] = 0x9554, [0x531a] = 0x9556, [0x531b] = 0x9557, [0x531c] = 0x9558,
  [0x531d] = 0x9559, [0x531e] = 0x955b, [0x531f] = 0x955e, [0x5320] = 0x955f,
  [0x5321] = 0x955d, [0x5322] = 0x9561, [0x5323] = 0x9562, [0x5324] = 0x9564,
  [0x5325] = 0x9565, [0x5326] = 0x9566, [0x5327] = 0x9567, [0x5328] = 0x9568,
  [0x5329] = 0x9569, [0x532a] = 0x956a, [0x532b] = 0x956b, [0x532c] = 0x956c,
  [0x532d] = 0x956f, [0x532e] = 0x9571, [0x532f] = 0x9572, [0x5330] = 0x9573,
  [0x5331] = 0x953a, [0x5332] = 0x77e7, [0x5333] = 0x77ec, [0x5334] = 0x96c9,
  [0x5335] = 0x79d5, [0x5336] = 0x79ed, [0x5337] = 0x79e3, [0x5338] = 0x79eb,
  [0x5339] = 0x7a06, [0x533a] = 0x5d47, [0x533b] = 0x7a03, [0x533c] = 0x7a02,
  [0x533d] = 0x7a1e, [0x533e] = 0x7a14, [0x5340] = 0x9908, [0x5341] = 0x9909,
  [0x5342] = 0x990a, [0x5343] = 0x990b, [0x5344] = 0x990c, [0x5345] = 0x990e,
  [0x5346] = 0x990f, [0x5347] = 0x9911, [0x5348] = 0x9912, [0x5349] = 0x9913,
  [0x534a] = 0x9914, [0x534b] = 0x9915, [0x534c] = 0x9916, [0x534d] = 0x9917,
  [0x534e] = 0x9918, [0x534f] = 0x9919, [0x5350] = 0x991a, [0x5351] = 0x991b,
  [0x5352] = 0x991c, [0x5353] = 0x991d, [0x5354] = 0x991e, [0x5355] = 0x991f,
  [0x5356] = 0x9920, [0x5357] = 0x9921, [0x5358] = 0x9922, [0x5359] = 0x9923,
  [0x535a] = 0x9924, [0x535b] = 0x9925, [0x535c] = 0x9926, [0x535d] = 0x9927,
  [0x535e] = 0x9928, [0x535f] = 0x9929, [0x5360] = 0x992a, [0x5361] = 0x992b,
  [0x5362] = 0x992c, [0x5363] = 0x992d, [0x5364] = 0x992f, [0x5365] = 0x9930,
  [0x5366] = 0x9931, [0x5367] = 0x9932, [0x5368] = 0x9933, [0x5369] = 0x9934,
  [0x536a] = 0x9935, [0x536b] = 0x9936, [0x536c] = 0x9937, [0x536d] = 0x9938,
  [0x536e] = 0x9939, [0x536f] = 0x993a, [0x5370] = 0x993b, [0x5371] = 0x993c,
  [0x5372] = 0x993d, [0x5373] = 0x993e, [0x5374] = 0x993f, [0x5375] = 0x9940,
  [0x5376] = 0x9941, [0x5377] = 0x9942, [0x5378] = 0x9943, [0x5379] = 0x9944,
  [0x537a] = 0x9945, [0x537b] = 0x9946, [0x537c] = 0x9947, [0x537d] = 0x9948,
  [0x537e] = 0x9949, [0x5380] = 0x994a, [0x5381] = 0x994b, [0x5382] = 0x994c,
  [0x5383] = 0x994d, [0x5384] = 0x994e, [0x5385] = 0x994f, [0x5386] = 0x9950,
  [0x5387] = 0x9951, [0x5388] = 0x9952, [0x5389] = 0x9953, [0x538a] = 0x9956,
  [0x538b] = 0x9957, [0x538c] = 0x9958, [0x538d] = 0x9959, [0x538e] = 0x995a,
  [0x538f] = 0x995b, [0x5390] = 0x995c, [0x5391] = 0x995d, [0x5392] = 0x995e,
  [0x5393] = 0x995f, [0x5394] = 0x9960, [0x5395] = 0x9961, [0x5396] = 0x9962,
  [0x5397] = 0x9964, [0x5398] = 0x9966, [0x5399] = 0x9973, [0x539a] = 0x9978,
  [0x539b] = 0x9979, [0x539c] = 0x997b, [0x539d] = 0x997e, [0x539e] = 0x9982,
  [0x539f] = 0x9983, [0x53a0] = 0x9989, [0x53a1] = 0x7a39, [0x53a2] = 0x7a37,
  [0x53a3] = 0x7a51, [0x53a4] = 0x9ecf, [0x53a5] = 0x99a5, [0x53a6] = 0x7a70,
  [0x53a7] = 0x7688, [0x53a8] = 0x768e, [0x53a9] = 0x7693, [0x53aa] = 0x7699,
  [0x53ab] = 0x76a4, [0x53ac] = 0x74de, [0x53ad] = 0x74e0, [0x53ae] = 0x752c,
  [0x53af] = 0x9e20, [0x53b0] = 0x9e22, [0x53b1] = 0x9e28, [0x53b2] = 0x9e29,
  [0x53b3] = 0x9e2a, [0x53b4] = 0x9e2b, [0x53b5] = 0x9e2c, [0x53b6] = 0x9e32,
  [0x53b7] = 0x9e31, [0x53b8] = 0x9e36, [0x53b9] = 0x9e38, [0x53ba] = 0x9e37,
  [0x53bb] = 0x9e39, [0x53bc] = 0x9e3a, [0x53bd] = 0x9e3e, [0x53be] = 0x9e41,
  [0x53bf] = 0x9e42, [0x53c0] = 0x9e44, [0x53c1] = 0x9e46, [0x53c2] = 0x9e47,
  [0x53c3] = 0x9e48, [0x53c4] = 0x9e49, [0x53c5] = 0x9e4b, [0x53c6] = 0x9e4c,
  [0x53c7] = 0x9e4e, [0x53c8] = 0x9e51, [0x53c9] = 0x9e55, [0x53ca] = 0x9e57,
  [0x53cb] = 0x9e5a, [0x53cc] = 0x9e5b, [0x53cd] = 0x9e5c, [0x53ce] = 0x9e5e,
  [0x53cf] = 0x9e63, [0x53d0] = 0x9e66, [0x53d1] = 0x9e67, [0x53d2] = 0x9e68,
  [0x53d3] = 0x9e69, [0x53d4] = 0x9e6a, [0x53d5] = 0x9e6b, [0x53d6] = 0x9e6c,
  [0x53d7] = 0x9e71, [0x53d8] = 0x9e6d, [0x53d9] = 0x9e73, [0x53da] = 0x7592,
  [0x53db] = 0x7594, [0x53dc] = 0x7596, [0x53dd] = 0x75a0, [0x53de] = 0x759d,
  [0x53df] = 0x75ac, [0x53e0] = 0x75a3, [0x53e1] = 0x75b3, [0x53e2] = 0x75b4,
  [0x53e3] = 0x75b8, [0x53e4] = 0x75c4, [0x53e5] = 0x75b1, [0x53e6] = 0x75b0,
  [0x53e7] = 0x75c3, [0x53e8] = 0x75c2, [0x53e9] = 0x75d6, [0x53ea] = 0x75cd,
  [0x53eb] = 0x75e3, [0x53ec] = 0x75e8, [0x53ed] = 0x75e6, [0x53ee] = 0x75e4,
  [0x53ef] = 0x75eb, [0x53f0] = 0x75e7, [0x53f1] = 0x7603, [0x53f2] = 0x75f1,
  [0x53f3] = 0x75fc, [0x53f4] = 0x75ff, [0x53f5] = 0x7610, [0x53f6] = 0x7600,
  [0x53f7] = 0x7605, [0x53f8] = 0x760c, [0x53f9] = 0x7617, [0x53fa] = 0x760a,
  [0x53fb] = 0x7625, [0x53fc] = 0x7618, [0x53fd] = 0x7615, [0x53fe] = 0x7619,
  [0x5400] = 0x998c, [0x5401] = 0x998e, [0x5402] = 0x999a, [0x5403] = 0x999b,
  [0x5404] = 0x999c, [0x5405] = 0x999d, [0x5406] = 0x999e, [0x5407] = 0x999f,
  [0x5408] = 0x99a0, [0x5409] = 0x99a1, [0x540a] = 0x99a2, [0x540b] = 0x99a3,
  [0x540c] = 0x99a4, [0x540d] = 0x99a6, [0x540e] = 0x99a7, [0x540f] = 0x99a9,
  [0x5410] = 0x99aa, [0x5411] = 0x99ab, [0x5412] = 0x99ac, [0x5413] = 0x99ad,
  [0x5414] = 0x99ae, [0x5415] = 0x99af, [0x5416] = 0x99b0, [0x5417] = 0x99b1,
  [0x5418] = 0x99b2, [0x5419] = 0x99b3, [0x541a] = 0x99b4, [0x541b] = 0x99b5,
  [0x541c] = 0x99b6, [0x541d] = 0x99b7, [0x541e] = 0x99b8, [0x541f] = 0x99b9,
  [0x5420] = 0x99ba, [0x5421] = 0x99bb, [0x5422] = 0x99bc, [0x5423] = 0x99bd,
  [0x5424] = 0x99be, [0x5425] = 0x99bf, [0x5426] = 0x99c0, [0x5427] = 0x99c1,
  [0x5428] = 0x99c2, [0x5429] = 0x99c3, [0x542a] = 0x99c4, [0x542b] = 0x99c5,
  [0x542c] = 0x99c6, [0x542d] = 0x99c7, [0x542e] = 0x99c8, [0x542f] = 0x99c9,
  [0x5430] = 0x99ca, [0x5431] = 0x99cb, [0x5432] = 0x99cc, [0x5433] = 0x99cd,
  [0x5434] = 0x99ce, [0x5435] = 0x99cf, [0x5436] = 0x99d0, [0x5437] = 0x99d1,
  [0x5438] = 0x99d2, [0x5439] = 0x99d3, [0x543a] = 0x99d4, [0x543b] = 0x99d5,
  [0x543c] = 0x99d6, [0x543d] = 0x99d7, [0x543e] = 0x99d8, [0x5440] = 0x99d9,
  [0x5441] = 0x99da, [0x5442] = 0x99db, [0x5443] = 0x99dc, [0x5444] = 0x99dd,
  [0x5445] = 0x99de, [0x5446] = 0x99df, [0x5447] = 0x99e0, [0x5448] = 0x99e1,
  [0x5449] = 0x99e2, [0x544a] = 0x99e3, [0x544b] = 0x99e4, [0x544c] = 0x99e5,
  [0x544d] = 0x99e6, [0x544e] = 0x99e7, [0x544f] = 0x99e8, [0x5450] = 0x99e9,
  [0x5451] = 0x99ea, [0x5452] = 0x99eb, [0x5453] = 0x99ec, [0x5454] = 0x99ed,
  [0x5455] = 0x99ee, [0x5456] = 0x99ef, [0x5457] = 0x99f0, [0x5458] = 0x99f1,
  [0x5459] = 0x99f2, [0x545a] = 0x99f3, [0x545b] = 0x99f4, [0x545c] = 0x99f5,
  [0x545d] = 0x99f6, [0x545e] = 0x99f7, [0x545f] = 0x99f8, [0x5460] = 0x99f9,
  [0x5461] = 0x761b, [0x5462] = 0x763c, [0x5463] = 0x7622, [0x5464] = 0x7620,
  [0x5465] = 0x7640, [0x5466] = 0x762d, [0x5467] = 0x7630, [0x5468] = 0x763f,
  [0x5469] = 0x7635, [0x546a] = 0x7643, [0x546b] = 0x763e, [0x546c] = 0x7633,
  [0x546d] = 0x764d, [0x546e] = 0x765e, [0x546f] = 0x7654, [0x5470] = 0x765c,
  [0x5471] = 0x7656, [0x5472] = 0x766b, [0x5473] = 0x766f, [0x5474] = 0x7fca,
  [0x5475] = 0x7ae6, [0x5476] = 0x7a78, [0x5477] = 0x7a79, [0x5478] = 0x7a80,
  [0x5479] = 0x7a86, [0x547a] = 0x7a88, [0x547b] = 0x7a95, [0x547c] = 0x7aa6,
  [0x547d] = 0x7aa0, [0x547e] = 0x7aac, [0x547f] = 0x7aa8, [0x5480] = 0x7aad,
  [0x5481] = 0x7ab3, [0x5482] = 0x8864, [0x5483] = 0x8869, [0x5484] = 0x8872,
  [0x5485] = 0x887d, [0x5486] = 0x887f, [0x5487] = 0x8882, [0x5488] = 0x88a2,
  [0x5489] = 0x88c6, [0x548a] = 0x88b7, [0x548b] = 0x88bc, [0x548c] = 0x88c9,
  [0x548d] = 0x88e2, [0x548e] = 0x88ce, [0x548f] = 0x88e3, [0x5490] = 0x88e5,
  [0x5491] = 0x88f1, [0x5492] = 0x891a, [0x5493] = 0x88fc, [0x5494] = 0x88e8,
  [0x5495] = 0x88fe, [0x5496] = 0x88f0, [0x5497] = 0x8921, [0x5498] = 0x8919,
  [0x5499] = 0x8913, [0x549a] = 0x891b, [0x549b] = 0x890a, [0x549c] = 0x8934,
  [0x549d] = 0x892b, [0x549e] = 0x8936, [0x549f] = 0x8941, [0x54a0] = 0x8966,
  [0x54a1] = 0x897b, [0x54a2] = 0x758b, [0x54a3] = 0x80e5, [0x54a4] = 0x76b2,
  [0x54a5] = 0x76b4, [0x54a6] = 0x77dc, [0x54a7] = 0x8012, [0x54a8] = 0x8014,
  [0x54a9] = 0x8016, [0x54aa] = 0x801c, [0x54ab] = 0x8020, [0x54ac] = 0x8022,
  [0x54ad] = 0x8025, [0x54ae] = 0x8026, [0x54af] = 0x8027, [0x54b0] = 0x8029,
  [0x54b1] = 0x8028, [0x54b2] = 0x8031, [0x54b3] = 0x800b, [0x54b4] = 0x8035,
  [0x54b5] = 0x8043, [0x54b6] = 0x8046, [0x54b7] = 0x804d, [0x54b8] = 0x8052,
  [0x54b9] = 0x8069, [0x54ba] = 0x8071, [0x54bb] = 0x8983, [0x54bc] = 0x9878,
  [0x54bd] = 0x9880, [0x54be] = 0x9883, [0x54c0] = 0x99fa, [0x54c1] = 0x99fb,
  [0x54c2] = 0x99fc, [0x54c3] = 0x99fd, [0x54c4] = 0x99fe, [0x54c5] = 0x99ff,
  [0x54c6] = 0x9a00, [0x54c7] = 0x9a01, [0x54c8] = 0x9a02, [0x54c9] = 0x9a03,
  [0x54ca] = 0x9a04, [0x54cb] = 0x9a05, [0x54cc] = 0x9a06, [0x54cd] = 0x9a07,
  [0x54ce] = 0x9a08, [0x54cf] = 0x9a09, [0x54d0] = 0x9a0a, [0x54d1] = 0x9a0b,
  [0x54d2] = 0x9a0c, [0x54d3] = 0x9a0d, [0x54d4] = 0x9a0e, [0x54d5] = 0x9a0f,
  [0x54d6] = 0x9a10, [0x54d7] = 0x9a11, [0x54d8] = 0x9a12, [0x54d9] = 0x9a13,
  [0x54da] = 0x9a14, [0x54db] = 0x9a15, [0x54dc] = 0x9a16, [0x54dd] = 0x9a17,
  [0x54de] = 0x9a18, [0x54df] = 0x9a19, [0x54e0] = 0x9a1a, [0x54e1] = 0x9a1b,
  [0x54e2] = 0x9a1c, [0x54e3] = 0x9a1d, [0x54e4] = 0x9a1e, [0x54e5] = 0x9a1f,
  [0x54e6] = 0x9a20, [0x54e7] = 0x9a21, [0x54e8] = 0x9a22, [0x54e9] = 0x9a23,
  [0x54ea] = 0x9a24, [0x54eb] = 0x9a25, [0x54ec] = 0x9a26, [0x54ed] = 0x9a27,
  [0x54ee] = 0x9a28, [0x54ef] = 0x9a29, [0x54f0] = 0x9a2a, [0x54f1] = 0x9a2b,
  [0x54f2] = 0x9a2c, [0x54f3] = 0x9a2d, [0x54f4] = 0x9a2e, [0x54f5] = 0x9a2f,
  [0x54f6] = 0x9a30, [0x54f7] = 0x9a31, [0x54f8] = 0x9a32, [0x54f9] = 0x9a33,
  [0x54fa] = 0x9a34, [0x54fb] = 0x9a35, [0x54fc] = 0x9a36, [0x54fd] = 0x9a37,
  [0x54fe] = 0x9a38, [0x5500] = 0x9a39, [0x5501] = 0x9a3a, [0x5502] = 0x9a3b,
  [0x5503] = 0x9a3c, [0x5504] = 0x9a3d, [0x5505] = 0x9a3e, [0x5506] = 0x9a3f,
  [0x5507] = 0x9a40, [0x5508] = 0x9a41, [0x5509] = 0x9a42, [0x550a] = 0x9a43,
  [0x550b] = 0x9a44, [0x550c] = 0x9a45, [0x550d] = 0x9a46, [0x550e] = 0x9a47,
  [0x550f] = 0x9a48, [0x5510] = 0x9a49, [0x5511] = 0x9a4a, [0x5512] = 0x9a4b,
  [0x5513] = 0x9a4c, [0x5514] = 0x9a4d, [0x5515] = 0x9a4e, [0x5516] = 0x9a4f,
  [0x5517] = 0x9a50, [0x5518] = 0x9a51, [0x5519] = 0x9a52, [0x551a] = 0x9a53,
  [0x551b] = 0x9a54, [0x551c] = 0x9a55, [0x551d] = 0x9a56, [0x551e] = 0x9a57,
  [0x551f] = 0x9a58, [0x5520] = 0x9a59, [0x5521] = 0x9889, [0x5522] = 0x988c,
  [0x5523] = 0x988d, [0x5524] = 0x988f, [0x5525] = 0x9894, [0x5526] = 0x989a,
  [0x5527] = 0x989b, [0x5528] = 0x989e, [0x5529] = 0x989f, [0x552a] = 0x98a1,
  [0x552b] = 0x98a2, [0x552c] = 0x98a5, [0x552d] = 0x98a6, [0x552e] = 0x864d,
  [0x552f] = 0x8654, [0x5530] = 0x866c, [0x5531] = 0x866e, [0x5532] = 0x867f,
  [0x5533] = 0x867a, [0x5534] = 0x867c, [0x5535] = 0x867b, [0x5536] = 0x86a8,
  [0x5537] = 0x868d, [0x5538] = 0x868b, [0x5539] = 0x86ac, [0x553a] = 0x869d,
  [0x553b] = 0x86a7, [0x553c] = 0x86a3, [0x553d] = 0x86aa, [0x553e] = 0x8693,
  [0x553f] = 0x86a9, [0x5540] = 0x86b6, [0x5541] = 0x86c4, [0x5542] = 0x86b5,
  [0x5543] = 0x86ce, [0x5544] = 0x86b0, [0x5545] = 0x86ba, [0x5546] = 0x86b1,
  [0x5547] = 0x86af, [0x5548] = 0x86c9, [0x5549] = 0x86cf, [0x554a] = 0x86b4,
  [0x554b] = 0x86e9, [0x554c] = 0x86f1, [0x554d] = 0x86f2, [0x554e] = 0x86ed,
  [0x554f] = 0x86f3, [0x5550] = 0x86d0, [0x5551] = 0x8713, [0x5552] = 0x86de,
  [0x5553] = 0x86f4, [0x5554] = 0x86df, [0x5555] = 0x86d8, [0x5556] = 0x86d1,
  [0x5557] = 0x8703, [0x5558] = 0x8707, [0x5559] = 0x86f8, [0x555a] = 0x8708,
  [0x555b] = 0x870a, [0x555c] = 0x870d, [0x555d] = 0x8709, [0x555e] = 0x8723,
  [0x555f] = 0x873b, [0x5560] = 0x871e, [0x5561] = 0x8725, [0x5562] = 0x872e,
  [0x5563] = 0x871a, [0x5564] = 0x873e, [0x5565] = 0x8748, [0x5566] = 0x8734,
  [0x5567] = 0x8731, [0x5568] = 0x8729, [0x5569] = 0x8737, [0x556a] = 0x873f,
  [0x556b] = 0x8782, [0x556c] = 0x8722, [0x556d] = 0x877d, [0x556e] = 0x877e,
  [0x556f] = 0x877b, [0x5570] = 0x8760, [0x5571] = 0x8770, [0x5572] = 0x874c,
  [0x5573] = 0x876e, [0x5574] = 0x878b, [0x5575] = 0x8753, [0x5576] = 0x8763,
  [0x5577] = 0x877c, [0x5578] = 0x8764, [0x5579] = 0x8759, [0x557a] = 0x8765,
  [0x557b] = 0x8793, [0x557c] = 0x87af, [0x557d] = 0x87a8, [0x557e] = 0x87d2,
  [0x5580] = 0x9a5a, [0x5581] = 0x9a5b, [0x5582] = 0x9a5c, [0x5583] = 0x9a5d,
  [0x5584] = 0x9a5e, [0x5585] = 0x9a5f, [0x5586] = 0x9a60, [0x5587] = 0x9a61,
  [0x5588] = 0x9a62, [0x5589] = 0x9a63, [0x558a] = 0x9a64, [0x558b] = 0x9a65,
  [0x558c] = 0x9a66, [0x558d] = 0x9a67, [0x558e] = 0x9a68, [0x558f] = 0x9a69,
  [0x5590] = 0x9a6a, [0x5591] = 0x9a6b, [0x5592] = 0x9a72, [0x5593] = 0x9a83,
  [0x5594] = 0x9a89, [0x5595] = 0x9a8d, [0x5596] = 0x9a8e, [0x5597] = 0x9a94,
  [0x5598] = 0x9a95, [0x5599] = 0x9a99, [0x559a] = 0x9aa6, [0x559b] = 0x9aa9,
  [0x559c] = 0x9aaa, [0x559d] = 0x9aab, [0x559e] = 0x9aac, [0x559f] = 0x9aad,
  [0x55a0] = 0x9aae, [0x55a1] = 0x9aaf, [0x55a2] = 0x9ab2, [0x55a3] = 0x9ab3,
  [0x55a4] = 0x9ab4, [0x55a5] = 0x9ab5, [0x55a6] = 0x9ab9, [0x55a7] = 0x9abb,
  [0x55a8] = 0x9abd, [0x55a9] = 0x9abe, [0x55aa] = 0x9abf, [0x55ab] = 0x9ac3,
  [0x55ac] = 0x9ac4, [0x55ad] = 0x9ac6, [0x55ae] = 0x9ac7, [0x55af] = 0x9ac8,
  [0x55b0] = 0x9ac9, [0x55b1] = 0x9aca, [0x55b2] = 0x9acd, [0x55b3] = 0x9ace,
  [0x55b4] = 0x9acf, [0x55b5] = 0x9ad0, [0x55b6] = 0x9ad2, [0x55b7] = 0x9ad4,
  [0x55b8] = 0x9ad5, [0x55b9] = 0x9ad6, [0x55ba] = 0x9ad7, [0x55bb] = 0x9ad9,
  [0x55bc] = 0x9ada, [0x55bd] = 0x9adb, [0x55be] = 0x9adc, [0x55c0] = 0x9add,
  [0x55c1] = 0x9ade, [0x55c2] = 0x9ae0, [0x55c3] = 0x9ae2, [0x55c4] = 0x9ae3,
  [0x55c5] = 0x9ae4, [0x55c6] = 0x9ae5, [0x55c7] = 0x9ae7, [0x55c8] = 0x9ae8,
  [0x55c9] = 0x9ae9, [0x55ca] = 0x9aea, [0x55cb] = 0x9aec, [0x55cc] = 0x9aee,
  [0x55cd] = 0x9af0, [0x55ce] = 0x9af1, [0x55cf] = 0x9af2, [0x55d0] = 0x9af3,
  [0x55d1] = 0x9af4, [0x55d2] = 0x9af5, [0x55d3] = 0x9af6, [0x55d4] = 0x9af7,
  [0x55d5] = 0x9af8, [0x55d6] = 0x9afa, [0x55d7] = 0x9afc, [0x55d8] = 0x9afd,
  [0x55d9] = 0x9afe, [0x55da] = 0x9aff, [0x55db] = 0x9b00, [0x55dc] = 0x9b01,
  [0x55dd] = 0x9b02, [0x55de] = 0x9b04, [0x55df] = 0x9b05, [0x55e0] = 0x9b06,
  [0x55e1] = 0x87c6, [0x55e2] = 0x8788, [0x55e3] = 0x8785, [0x55e4] = 0x87ad,
  [0x55e5] = 0x8797, [0x55e6] = 0x8783, [0x55e7] = 0x87ab, [0x55e8] = 0x87e5,
  [0x55e9] = 0x87ac, [0x55ea] = 0x87b5, [0x55eb] = 0x87b3, [0x55ec] = 0x87cb,
  [0x55ed] = 0x87d3, [0x55ee] = 0x87bd, [0x55ef] = 0x87d1, [0x55f0] = 0x87c0,
  [0x55f1] = 0x87ca, [0x55f2] = 0x87db, [0x55f3] = 0x87ea, [0x55f4] = 0x87e0,
  [0x55f5] = 0x87ee, [0x55f6] = 0x8816, [0x55f7] = 0x8813, [0x55f8] = 0x87fe,
  [0x55f9] = 0x880a, [0x55fa] = 0x881b, [0x55fb] = 0x8821, [0x55fc] = 0x8839,
  [0x55fd] = 0x883c, [0x55fe] = 0x7f36, [0x55ff] = 0x7f42, [0x5600] = 0x7f44,
  [0x5601] = 0x7f45, [0x5602] = 0x8210, [0x5603] = 0x7afa, [0x5604] = 0x7afd,
  [0x5605] = 0x7b08, [0x5606] = 0x7b03, [0x5607] = 0x7b04, [0x5608] = 0x7b15,
  [0x5609] = 0x7b0a, [0x560a] = 0x7b2b, [0x560b] = 0x7b0f, [0x560c] = 0x7b47,
  [0x560d] = 0x7b38, [0x560e] = 0x7b2a, [0x560f] = 0x7b19, [0x5610] = 0x7b2e,
  [0x5611] = 0x7b31, [0x5612] = 0x7b20, [0x5613] = 0x7b25, [0x5614] = 0x7b24,
  [0x5615] = 0x7b33, [0x5616] = 0x7b3e, [0x5617] = 0x7b1e, [0x5618] = 0x7b58,
  [0x5619] = 0x7b5a, [0x561a] = 0x7b45, [0x561b] = 0x7b75, [0x561c] = 0x7b4c,
  [0x561d] = 0x7b5d, [0x561e] = 0x7b60, [0x561f] = 0x7b6e, [0x5620] = 0x7b7b,
  [0x5621] = 0x7b62, [0x5622] = 0x7b72, [0x5623] = 0x7b71, [0x5624] = 0x7b90,
  [0x5625] = 0x7ba6, [0x5626] = 0x7ba7, [0x5627] = 0x7bb8, [0x5628] = 0x7bac,
  [0x5629] = 0x7b9d, [0x562a] = 0x7ba8, [0x562b] = 0x7b85, [0x562c] = 0x7baa,
  [0x562d] = 0x7b9c, [0x562e] = 0x7ba2, [0x562f] = 0x7bab, [0x5630] = 0x7bb4,
  [0x5631] = 0x7bd1, [0x5632] = 0x7bc1, [0x5633] = 0x7bcc, [0x5634] = 0x7bdd,
  [0x5635] = 0x7bda, [0x5636] = 0x7be5, [0x5637] = 0x7be6, [0x5638] = 0x7bea,
  [0x5639] = 0x7c0c, [0x563a] = 0x7bfe, [0x563b] = 0x7bfc, [0x563c] = 0x7c0f,
  [0x563d] = 0x7c16, [0x563e] = 0x7c0b, [0x5640] = 0x9b07, [0x5641] = 0x9b09,
  [0x5642] = 0x9b0a, [0x5643] = 0x9b0b, [0x5644] = 0x9b0c, [0x5645] = 0x9b0d,
  [0x5646] = 0x9b0e, [0x5647] = 0x9b10, [0x5648] = 0x9b11, [0x5649] = 0x9b12,
  [0x564a] = 0x9b14, [0x564b] = 0x9b15, [0x564c] = 0x9b16, [0x564d] = 0x9b17,
  [0x564e] = 0x9b18, [0x564f] = 0x9b19, [0x5650] = 0x9b1a, [0x5651] = 0x9b1b,
  [0x5652] = 0x9b1c, [0x5653] = 0x9b1d, [0x5654] = 0x9b1e, [0x5655] = 0x9b20,
  [0x5656] = 0x9b21, [0x5657] = 0x9b22, [0x5658] = 0x9b24, [0x5659] = 0x9b25,
  [0x565a] = 0x9b26, [0x565b] = 0x9b27, [0x565c] = 0x9b28, [0x565d] = 0x9b29,
  [0x565e] = 0x9b2a, [0x565f] = 0x9b2b, [0x5660] = 0x9b2c, [0x5661] = 0x9b2d,
  [0x5662] = 0x9b2e, [0x5663] = 0x9b30, [0x5664] = 0x9b31, [0x5665] = 0x9b33,
  [0x5666] = 0x9b34, [0x5667] = 0x9b35, [0x5668] = 0x9b36, [0x5669] = 0x9b37,
  [0x566a] = 0x9b38, [0x566b] = 0x9b39, [0x566c] = 0x9b3a, [0x566d] = 0x9b3d,
  [0x566e] = 0x9b3e, [0x566f] = 0x9b3f, [0x5670] = 0x9b40, [0x5671] = 0x9b46,
  [0x5672] = 0x9b4a, [0x5673] = 0x9b4b, [0x5674] = 0x9b4c, [0x5675] = 0x9b4e,
  [0x5676] = 0x9b50, [0x5677] = 0x9b52, [0x5678] = 0x9b53, [0x5679] = 0x9b55,
  [0x567a] = 0x9b56, [0x567b] = 0x9b57, [0x567c] = 0x9b58, [0x567d] = 0x9b59,
  [0x567e] = 0x9b5a, [0x5680] = 0x9b5b, [0x5681] = 0x9b5c, [0x5682] = 0x9b5d,
  [0x5683] = 0x9b5e, [0x5684] = 0x9b5f, [0x5685] = 0x9b60, [0x5686] = 0x9b61,
  [0x5687] = 0x9b62, [0x5688] = 0x9b63, [0x5689] = 0x9b64, [0x568a] = 0x9b65,
  [0x568b] = 0x9b66, [0x568c] = 0x9b67, [0x568d] = 0x9b68, [0x568e] = 0x9b69,
  [0x568f] = 0x9b6a, [0x5690] = 0x9b6b, [0x5691] = 0x9b6c, [0x5692] = 0x9b6d,
  [0x5693] = 0x9b6e, [0x5694] = 0x9b6f, [0x5695] = 0x9b70, [0x5696] = 0x9b71,
  [0x5697] = 0x9b72, [0x5698] = 0x9b73, [0x5699] = 0x9b74, [0x569a] = 0x9b75,
  [0x569b] = 0x9b76, [0x569c] = 0x9b77, [0x569d] = 0x9b78, [0x569e] = 0x9b79,
  [0x569f] = 0x9b7a, [0x56a0] = 0x9b7b, [0x56a1] = 0x7c1f, [0x56a2] = 0x7c2a,
  [0x56a3] = 0x7c26, [0x56a4] = 0x7c38, [0x56a5] = 0x7c41, [0x56a6] = 0x7c40,
  [0x56a7] = 0x81fe, [0x56a8] = 0x8201, [0x56a9] = 0x8202, [0x56aa] = 0x8204,
  [0x56ab] = 0x81ec, [0x56ac] = 0x8844, [0x56ad] = 0x8221, [0x56ae] = 0x8222,
  [0x56af] = 0x8223, [0x56b0] = 0x822d, [0x56b1] = 0x822f, [0x56b2] = 0x8228,
  [0x56b3] = 0x822b, [0x56b4] = 0x8238, [0x56b5] = 0x823b, [0x56b6] = 0x8233,
  [0x56b7] = 0x8234, [0x56b8] = 0x823e, [0x56b9] = 0x8244, [0x56ba] = 0x8249,
  [0x56bb] = 0x824b, [0x56bc] = 0x824f, [0x56bd] = 0x825a, [0x56be] = 0x825f,
  [0x56bf] = 0x8268, [0x56c0] = 0x887e, [0x56c1] = 0x8885, [0x56c2] = 0x8888,
  [0x56c3] = 0x88d8, [0x56c4] = 0x88df, [0x56c5] = 0x895e, [0x56c6] = 0x7f9d,
  [0x56c7] = 0x7f9f, [0x56c8] = 0x7fa7, [0x56c9] = 0x7faf, [0x56ca] = 0x7fb0,
  [0x56cb] = 0x7fb2, [0x56cc] = 0x7c7c, [0x56cd] = 0x6549, [0x56ce] = 0x7c91,
  [0x56cf] = 0x7c9d, [0x56d0] = 0x7c9c, [0x56d1] = 0x7c9e, [0x56d2] = 0x7ca2,
  [0x56d3] = 0x7cb2, [0x56d4] = 0x7cbc, [0x56d5] = 0x7cbd, [0x56d6] = 0x7cc1,
  [0x56d7] = 0x7cc7, [0x56d8] = 0x7ccc, [0x56d9] = 0x7ccd, [0x56da] = 0x7cc8,
  [0x56db] = 0x7cc5, [0x56dc] = 0x7cd7, [0x56dd] = 0x7ce8, [0x56de] = 0x826e,
  [0x56df] = 0x66a8, [0x56e0] = 0x7fbf, [0x56e1] = 0x7fce, [0x56e2] = 0x7fd5,
  [0x56e3] = 0x7fe5, [0x56e4] = 0x7fe1, [0x56e5] = 0x7fe6, [0x56e6] = 0x7fe9,
  [0x56e7] = 0x7fee, [0x56e8] = 0x7ff3, [0x56e9] = 0x7cf8, [0x56ea] = 0x7d77,
  [0x56eb] = 0x7da6, [0x56ec] = 0x7dae, [0x56ed] = 0x7e47, [0x56ee] = 0x7e9b,
  [0x56ef] = 0x9eb8, [0x56f0] = 0x9eb4, [0x56f1] = 0x8d73, [0x56f2] = 0x8d84,
  [0x56f3] = 0x8d94, [0x56f4] = 0x8d91, [0x56f5] = 0x8db1, [0x56f6] = 0x8d67,
  [0x56f7] = 0x8d6d, [0x56f8] = 0x8c47, [0x56f9] = 0x8c49, [0x56fa] = 0x914a,
  [0x56fb] = 0x9150, [0x56fc] = 0x914e, [0x56fd] = 0x914f, [0x56fe] = 0x9164,
  [0x5700] = 0x9b7c, [0x5701] = 0x9b7d, [0x5702] = 0x9b7e, [0x5703] = 0x9b7f,
  [0x5704] = 0x9b80, [0x5705] = 0x9b81, [0x5706] = 0x9b82, [0x5707] = 0x9b83,
  [0x5708] = 0x9b84, [0x5709] = 0x9b85, [0x570a] = 0x9b86, [0x570b] = 0x9b87,
  [0x570c] = 0x9b88, [0x570d] = 0x9b89, [0x570e] = 0x9b8a, [0x570f] = 0x9b8b,
  [0x5710] = 0x9b8c, [0x5711] = 0x9b8d, [0x5712] = 0x9b8e, [0x5713] = 0x9b8f,
  [0x5714] = 0x9b90, [0x5715] = 0x9b91, [0x5716] = 0x9b92, [0x5717] = 0x9b93,
  [0x5718] = 0x9b94, [0x5719] = 0x9b95, [0x571a] = 0x9b96, [0x571b] = 0x9b97,
  [0x571c] = 0x9b98, [0x571d] = 0x9b99, [0x571e] = 0x9b9a, [0x571f] = 0x9b9b,
  [0x5720] = 0x9b9c, [0x5721] = 0x9b9d, [0x5722] = 0x9b9e, [0x5723] = 0x9b9f,
  [0x5724] = 0x9ba0, [0x5725] = 0x9ba1, [0x5726] = 0x9ba2, [0x5727] = 0x9ba3,
  [0x5728] = 0x9ba4, [0x5729] = 0x9ba5, [0x572a] = 0x9ba6, [0x572b] = 0x9ba7,
  [0x572c] = 0x9ba8, [0x572d] = 0x9ba9, [0x572e] = 0x9baa, [0x572f] = 0x9bab,
  [0x5730] = 0x9bac, [0x5731] = 0x9bad, [0x5732] = 0x9bae, [0x5733] = 0x9baf,
  [0x5734] = 0x9bb0, [0x5735] = 0x9bb1, [0x5736] = 0x9bb2, [0x5737] = 0x9bb3,
  [0x5738] = 0x9bb4, [0x5739] = 0x9bb5, [0x573a] = 0x9bb6, [0x573b] = 0x9bb7,
  [0x573c] = 0x9bb8, [0x573d] = 0x9bb9, [0x573e] = 0x9bba, [0x5740] = 0x9bbb,
  [0x5741] = 0x9bbc, [0x5742] = 0x9bbd, [0x5743] = 0x9bbe, [0x5744] = 0x9bbf,
  [0x5745] = 0x9bc0, [0x5746] = 0x9bc1, [0x5747] = 0x9bc2, [0x5748] = 0x9bc3,
  [0x5749] = 0x9bc4, [0x574a] = 0x9bc5, [0x574b] = 0x9bc6, [0x574c] = 0x9bc7,
  [0x574d] = 0x9bc8, [0x574e] = 0x9bc9, [0x574f] = 0x9bca, [0x5750] = 0x9bcb,
  [0x5751] = 0x9bcc, [0x5752] = 0x9bcd, [0x5753] = 0x9bce, [0x5754] = 0x9bcf,
  [0x5755] = 0x9bd0, [0x5756] = 0x9bd1, [0x5757] = 0x9bd2, [0x5758] = 0x9bd3,
  [0x5759] = 0x9bd4, [0x575a] = 0x9bd5, [0x575b] = 0x9bd6, [0x575c] = 0x9bd7,
  [0x575d] = 0x9bd8, [0x575e] = 0x9bd9, [0x575f] = 0x9bda, [0x5760] = 0x9bdb,
  [0x5761] = 0x9162, [0x5762] = 0x9161, [0x5763] = 0x9170, [0x5764] = 0x9169,
  [0x5765] = 0x916f, [0x5766] = 0x917d, [0x5767] = 0x917e, [0x5768] = 0x9172,
  [0x5769] = 0x9174, [0x576a] = 0x9179, [0x576b] = 0x918c, [0x576c] = 0x9185,
  [0x576d] = 0x9190, [0x576e] = 0x918d, [0x576f] = 0x9191, [0x5770] = 0x91a2,
  [0x5771] = 0x91a3, [0x5772] = 0x91aa, [0x5773] = 0x91ad, [0x5774] = 0x91ae,
  [0x5775] = 0x91af, [0x5776] = 0x91b5, [0x5777] = 0x91b4, [0x5778] = 0x91ba,
  [0x5779] = 0x8c55, [0x577a] = 0x9e7e, [0x577b] = 0x8db8, [0x577c] = 0x8deb,
  [0x577d] = 0x8e05, [0x577e] = 0x8e59, [0x577f] = 0x8e69, [0x5780] = 0x8db5,
  [0x5781] = 0x8dbf, [0x5782] = 0x8dbc, [0x5783] = 0x8dba, [0x5784] = 0x8dc4,
  [0x5785] = 0x8dd6, [0x5786] = 0x8dd7, [0x5787] = 0x8dda, [0x5788] = 0x8dde,
  [0x5789] = 0x8dce, [0x578a] = 0x8dcf, [0x578b] = 0x8ddb, [0x578c] = 0x8dc6,
  [0x578d] = 0x8dec, [0x578e] = 0x8df7, [0x578f] = 0x8df8, [0x5790] = 0x8de3,
  [0x5791] = 0x8df9, [0x5792] = 0x8dfb, [0x5793] = 0x8de4, [0x5794] = 0x8e09,
  [0x5795] = 0x8dfd, [0x5796] = 0x8e14, [0x5797] = 0x8e1d, [0x5798] = 0x8e1f,
  [0x5799] = 0x8e2c, [0x579a] = 0x8e2e, [0x579b] = 0x8e23, [0x579c] = 0x8e2f,
  [0x579d] = 0x8e3a, [0x579e] = 0x8e40, [0x579f] = 0x8e39, [0x57a0] = 0x8e35,
  [0x57a1] = 0x8e3d, [0x57a2] = 0x8e31, [0x57a3] = 0x8e49, [0x57a4] = 0x8e41,
  [0x57a5] = 0x8e42, [0x57a6] = 0x8e51, [0x57a7] = 0x8e52, [0x57a8] = 0x8e4a,
  [0x57a9] = 0x8e70, [0x57aa] = 0x8e76, [0x57ab] = 0x8e7c, [0x57ac] = 0x8e6f,
  [0x57ad] = 0x8e74, [0x57ae] = 0x8e85, [0x57af] = 0x8e8f, [0x57b0] = 0x8e94,
  [0x57b1] = 0x8e90, [0x57b2] = 0x8e9c, [0x57b3] = 0x8e9e, [0x57b4] = 0x8c78,
  [0x57b5] = 0x8c82, [0x57b6] = 0x8c8a, [0x57b7] = 0x8c85, [0x57b8] = 0x8c98,
  [0x57b9] = 0x8c94, [0x57ba] = 0x659b, [0x57bb] = 0x89d6, [0x57bc] = 0x89de,
  [0x57bd] = 0x89da, [0x57be] = 0x89dc, [0x57c0] = 0x9bdc, [0x57c1] = 0x9bdd,
  [0x57c2] = 0x9bde, [0x57c3] = 0x9bdf, [0x57c4] = 0x9be0, [0x57c5] = 0x9be1,
  [0x57c6] = 0x9be2, [0x57c7] = 0x9be3, [0x57c8] = 0x9be4, [0x57c9] = 0x9be5,
  [0x57ca] = 0x9be6, [0x57cb] = 0x9be7, [0x57cc] = 0x9be8, [0x57cd] = 0x9be9,
  [0x57ce] = 0x9bea, [0x57cf] = 0x9beb, [0x57d0] = 0x9bec, [0x57d1] = 0x9bed,
  [0x57d2] = 0x9bee, [0x57d3] = 0x9bef, [0x57d4] = 0x9bf0, [0x57d5] = 0x9bf1,
  [0x57d6] = 0x9bf2, [0x57d7] = 0x9bf3, [0x57d8] = 0x9bf4, [0x57d9] = 0x9bf5,
  [0x57da] = 0x9bf6, [0x57db] = 0x9bf7, [0x57dc] = 0x9bf8, [0x57dd] = 0x9bf9,
  [0x57de] = 0x9bfa, [0x57df] = 0x9bfb, [0x57e0] = 0x9bfc, [0x57e1] = 0x9bfd,
  [0x57e2] = 0x9bfe, [0x57e3] = 0x9bff, [0x57e4] = 0x9c00, [0x57e5] = 0x9c01,
  [0x57e6] = 0x9c02, [0x57e7] = 0x9c03, [0x57e8] = 0x9c04, [0x57e9] = 0x9c05,
  [0x57ea] = 0x9c06, [0x57eb] = 0x9c07, [0x57ec] = 0x9c08, [0x57ed] = 0x9c09,
  [0x57ee] = 0x9c0a, [0x57ef] = 0x9c0b, [0x57f0] = 0x9c0c, [0x57f1] = 0x9c0d,
  [0x57f2] = 0x9c0e, [0x57f3] = 0x9c0f, [0x57f4] = 0x9c10, [0x57f5] = 0x9c11,
  [0x57f6] = 0x9c12, [0x57f7] = 0x9c13, [0x57f8] = 0x9c14, [0x57f9] = 0x9c15,
  [0x57fa] = 0x9c16, [0x57fb] = 0x9c17, [0x57fc] = 0x9c18, [0x57fd] = 0x9c19,
  [0x57fe] = 0x9c1a, [0x5800] = 0x9c1b, [0x5801] = 0x9c1c, [0x5802] = 0x9c1d,
  [0x5803] = 0x9c1e, [0x5804] = 0x9c1f, [0x5805] = 0x9c20, [0x5806] = 0x9c21,
  [0x5807] = 0x9c22, [0x5808] = 0x9c23, [0x5809] = 0x9c24, [0x580a] = 0x9c25,
  [0x580b] = 0x9c26, [0x580c] = 0x9c27, [0x580d] = 0x9c28, [0x580e] = 0x9c29,
  [0x580f] = 0x9c2a, [0x5810] = 0x9c2b, [0x5811] = 0x9c2c, [0x5812] = 0x9c2d,
  [0x5813] = 0x9c2e, [0x5814] = 0x9c2f, [0x5815] = 0x9c30, [0x5816] = 0x9c31,
  [0x5817] = 0x9c32, [0x5818] = 0x9c33, [0x5819] = 0x9c34, [0x581a] = 0x9c35,
  [0x581b] = 0x9c36, [0x581c] = 0x9c37, [0x581d] = 0x9c38, [0x581e] = 0x9c39,
  [0x581f] = 0x9c3a, [0x5820] = 0x9c3b, [0x5821] = 0x89e5, [0x5822] = 0x89eb,
  [0x5823] = 0x89ef, [0x5824] = 0x8a3e, [0x5825] = 0x8b26, [0x5826] = 0x9753,
  [0x5827] = 0x96e9, [0x5828] = 0x96f3, [0x5829] = 0x96ef, [0x582a] = 0x9706,
  [0x582b] = 0x9701, [0x582c] = 0x9708, [0x582d] = 0x970f, [0x582e] = 0x970e,
  [0x582f] = 0x972a, [0x5830] = 0x972d, [0x5831] = 0x9730, [0x5832] = 0x973e,
  [0x5833] = 0x9f80, [0x5834] = 0x9f83, [0x5835] = 0x9f85, [0x5836] = 0x9f86,
  [0x5837] = 0x9f87, [0x5838] = 0x9f88, [0x5839] = 0x9f89, [0x583a] = 0x9f8a,
  [0x583b] = 0x9f8c, [0x583c] = 0x9efe, [0x583d] = 0x9f0b, [0x583e] = 0x9f0d,
  [0x583f] = 0x96b9, [0x5840] = 0x96bc, [0x5841] = 0x96bd, [0x5842] = 0x96ce,
  [0x5843] = 0x96d2, [0x5844] = 0x77bf, [0x5845] = 0x96e0, [0x5846] = 0x928e,
  [0x5847] = 0x92ae, [0x5848] = 0x92c8, [0x5849] = 0x933e, [0x584a] = 0x936a,
  [0x584b] = 0x93ca, [0x584c] = 0x938f, [0x584d] = 0x943e, [0x584e] = 0x946b,
  [0x584f] = 0x9c7f, [0x5850] = 0x9c82, [0x5851] = 0x9c85, [0x5852] = 0x9c86,
  [0x5853] = 0x9c87, [0x5854] = 0x9c88, [0x5855] = 0x7a23, [0x5856] = 0x9c8b,
  [0x5857] = 0x9c8e, [0x5858] = 0x9c90, [0x5859] = 0x9c91, [0x585a] = 0x9c92,
  [0x585b] = 0x9c94, [0x585c] = 0x9c95, [0x585d] = 0x9c9a, [0x585e] = 0x9c9b,
  [0x585f] = 0x9c9e, [0x5860] = 0x9c9f, [0x5861] = 0x9ca0, [0x5862] = 0x9ca1,
  [0x5863] = 0x9ca2, [0x5864] = 0x9ca3, [0x5865] = 0x9ca5, [0x5866] = 0x9ca6,
  [0x5867] = 0x9ca7, [0x5868] = 0x9ca8, [0x5869] = 0x9ca9, [0x586a] = 0x9cab,
  [0x586b] = 0x9cad, [0x586c] = 0x9cae, [0x586d] = 0x9cb0, [0x586e] = 0x9cb1,
  [0x586f] = 0x9cb2, [0x5870] = 0x9cb3, [0x5871] = 0x9cb4, [0x5872] = 0x9cb5,
  [0x5873] = 0x9cb6, [0x5874] = 0x9cb7, [0x5875] = 0x9cba, [0x5876] = 0x9cbb,
  [0x5877] = 0x9cbc, [0x5878] = 0x9cbd, [0x5879] = 0x9cc4, [0x587a] = 0x9cc5,
  [0x587b] = 0x9cc6, [0x587c] = 0x9cc7, [0x587d] = 0x9cca, [0x587e] = 0x9ccb,
  [0x5880] = 0x9c3c, [0x5881] = 0x9c3d, [0x5882] = 0x9c3e, [0x5883] = 0x9c3f,
  [0x5884] = 0x9c40, [0x5885] = 0x9c41, [0x5886] = 0x9c42, [0x5887] = 0x9c43,
  [0x5888] = 0x9c44, [0x5889] = 0x9c45, [0x588a] = 0x9c46, [0x588b] = 0x9c47,
  [0x588c] = 0x9c48, [0x588d] = 0x9c49, [0x588e] = 0x9c4a, [0x588f] = 0x9c4b,
  [0x5890] = 0x9c4c, [0x5891] = 0x9c4d, [0x5892] = 0x9c4e, [0x5893] = 0x9c4f,
  [0x5894] = 0x9c50, [0x5895] = 0x9c51, [0x5896] = 0x9c52, [0x5897] = 0x9c53,
  [0x5898] = 0x9c54, [0x5899] = 0x9c55, [0x589a] = 0x9c56, [0x589b] = 0x9c57,
  [0x589c] = 0x9c58, [0x589d] = 0x9c59, [0x589e] = 0x9c5a, [0x589f] = 0x9c5b,
  [0x58a0] = 0x9c5c, [0x58a1] = 0x9c5d, [0x58a2] = 0x9c5e, [0x58a3] = 0x9c5f,
  [0x58a4] = 0x9c60, [0x58a5] = 0x9c61, [0x58a6] = 0x9c62, [0x58a7] = 0x9c63,
  [0x58a8] = 0x9c64, [0x58a9] = 0x9c65, [0x58aa] = 0x9c66, [0x58ab] = 0x9c67,
  [0x58ac] = 0x9c68, [0x58ad] = 0x9c69, [0x58ae] = 0x9c6a, [0x58af] = 0x9c6b,
  [0x58b0] = 0x9c6c, [0x58b1] = 0x9c6d, [0x58b2] = 0x9c6e, [0x58b3] = 0x9c6f,
  [0x58b4] = 0x9c70, [0x58b5] = 0x9c71, [0x58b6] = 0x9c72, [0x58b7] = 0x9c73,
  [0x58b8] = 0x9c74, [0x58b9] = 0x9c75, [0x58ba] = 0x9c76, [0x58bb] = 0x9c77,
  [0x58bc] = 0x9c78, [0x58bd] = 0x9c79, [0x58be] = 0x9c7a, [0x58c0] = 0x9c7b,
  [0x58c1] = 0x9c7d, [0x58c2] = 0x9c7e, [0x58c3] = 0x9c80, [0x58c4] = 0x9c83,
  [0x58c5] = 0x9c84, [0x58c6] = 0x9c89, [0x58c7] = 0x9c8a, [0x58c8] = 0x9c8c,
  [0x58c9] = 0x9c8f, [0x58ca] = 0x9c93, [0x58cb] = 0x9c96, [0x58cc] = 0x9c97,
  [0x58cd] = 0x9c98, [0x58ce] = 0x9c99, [0x58cf] = 0x9c9d, [0x58d0] = 0x9caa,
  [0x58d1] = 0x9cac, [0x58d2] = 0x9caf, [0x58d3] = 0x9cb9, [0x58d4] = 0x9cbe,
  [0x58d5] = 0x9cbf, [0x58d6] = 0x9cc0, [0x58d7] = 0x9cc1, [0x58d8] = 0x9cc2,
  [0x58d9] = 0x9cc8, [0x58da] = 0x9cc9, [0x58db] = 0x9cd1, [0x58dc] = 0x9cd2,
  [0x58dd] = 0x9cda, [0x58de] = 0x9cdb, [0x58df] = 0x9ce0, [0x58e0] = 0x9ce1,
  [0x58e1] = 0x9ccc, [0x58e2] = 0x9ccd, [0x58e3] = 0x9cce, [0x58e4] = 0x9ccf,
  [0x58e5] = 0x9cd0, [0x58e6] = 0x9cd3, [0x58e7] = 0x9cd4, [0x58e8] = 0x9cd5,
  [0x58e9] = 0x9cd7, [0x58ea] = 0x9cd8, [0x58eb] = 0x9cd9, [0x58ec] = 0x9cdc,
  [0x58ed] = 0x9cdd, [0x58ee] = 0x9cdf, [0x58ef] = 0x9ce2, [0x58f0] = 0x977c,
  [0x58f1] = 0x9785, [0x58f2] = 0x9791, [0x58f3] = 0x9792, [0x58f4] = 0x9794,
  [0x58f5] = 0x97af, [0x58f6] = 0x97ab, [0x58f7] = 0x97a3, [0x58f8] = 0x97b2,
  [0x58f9] = 0x97b4, [0x58fa] = 0x9ab1, [0x58fb] = 0x9ab0, [0x58fc] = 0x9ab7,
  [0x58fd] = 0x9e58, [0x58fe] = 0x9ab6, [0x58ff] = 0x9aba, [0x5900] = 0x9abc,
  [0x5901] = 0x9ac1, [0x5902] = 0x9ac0, [0x5903] = 0x9ac5, [0x5904] = 0x9ac2,
  [0x5905] = 0x9acb, [0x5906] = 0x9acc, [0x5907] = 0x9ad1, [0x5908] = 0x9b45,
  [0x5909] = 0x9b43, [0x590a] = 0x9b47, [0x590b] = 0x9b49, [0x590c] = 0x9b48,
  [0x590d] = 0x9b4d, [0x590e] = 0x9b51, [0x590f] = 0x98e8, [0x5910] = 0x990d,
  [0x5911] = 0x992e, [0x5912] = 0x9955, [0x5913] = 0x9954, [0x5914] = 0x9adf,
  [0x5915] = 0x9ae1, [0x5916] = 0x9ae6, [0x5917] = 0x9aef, [0x5918] = 0x9aeb,
  [0x5919] = 0x9afb, [0x591a] = 0x9aed, [0x591b] = 0x9af9, [0x591c] = 0x9b08,
  [0x591d] = 0x9b0f, [0x591e] = 0x9b13, [0x591f] = 0x9b1f, [0x5920] = 0x9b23,
  [0x5921] = 0x9ebd, [0x5922] = 0x9ebe, [0x5923] = 0x7e3b, [0x5924] = 0x9e82,
  [0x5925] = 0x9e87, [0x5926] = 0x9e88, [0x5927] = 0x9e8b, [0x5928] = 0x9e92,
  [0x5929] = 0x93d6, [0x592a] = 0x9e9d, [0x592b] = 0x9e9f, [0x592c] = 0x9edb,
  [0x592d] = 0x9edc, [0x592e] = 0x9edd, [0x592f] = 0x9ee0, [0x5930] = 0x9edf,
  [0x5931] = 0x9ee2, [0x5932] = 0x9ee9, [0x5933] = 0x9ee7, [0x5934] = 0x9ee5,
  [0x5935] = 0x9eea, [0x5936] = 0x9eef, [0x5937] = 0x9f22, [0x5938] = 0x9f2c,
  [0x5939] = 0x9f2f, [0x593a] = 0x9f39, [0x593b] = 0x9f37, [0x593c] = 0x9f3d,
  [0x593d] = 0x9f3e, [0x593e] = 0x9f44, [0x5940] = 0x9ce3, [0x5941] = 0x9ce4,
  [0x5942] = 0x9ce5, [0x5943] = 0x9ce6, [0x5944] = 0x9ce7, [0x5945] = 0x9ce8,
  [0x5946] = 0x9ce9, [0x5947] = 0x9cea, [0x5948] = 0x9ceb, [0x5949] = 0x9cec,
  [0x594a] = 0x9ced, [0x594b] = 0x9cee, [0x594c] = 0x9cef, [0x594d] = 0x9cf0,
  [0x594e] = 0x9cf1, [0x594f] = 0x9cf2, [0x5950] = 0x9cf3, [0x5951] = 0x9cf4,
  [0x5952] = 0x9cf5, [0x5953] = 0x9cf6, [0x5954] = 0x9cf7, [0x5955] = 0x9cf8,
  [0x5956] = 0x9cf9, [0x5957] = 0x9cfa, [0x5958] = 0x9cfb, [0x5959] = 0x9cfc,
  [0x595a] = 0x9cfd, [0x595b] = 0x9cfe, [0x595c] = 0x9cff, [0x595d] = 0x9d00,
  [0x595e] = 0x9d01, [0x595f] = 0x9d02, [0x5960] = 0x9d03, [0x5961] = 0x9d04,
  [0x5962] = 0x9d05, [0x5963] = 0x9d06, [0x5964] = 0x9d07, [0x5965] = 0x9d08,
  [0x5966] = 0x9d09, [0x5967] = 0x9d0a, [0x5968] = 0x9d0b, [0x5969] = 0x9d0c,
  [0x596a] = 0x9d0d, [0x596b] = 0x9d0e, [0x596c] = 0x9d0f, [0x596d] = 0x9d10,
  [0x596e] = 0x9d11, [0x596f] = 0x9d12, [0x5970] = 0x9d13, [0x5971] = 0x9d14,
  [0x5972] = 0x9d15, [0x5973] = 0x9d16, [0x5974] = 0x9d17, [0x5975] = 0x9d18,
  [0x5976] = 0x9d19, [0x5977] = 0x9d1a, [0x5978] = 0x9d1b, [0x5979] = 0x9d1c,
  [0x597a] = 0x9d1d, [0x597b] = 0x9d1e, [0x597c] = 0x9d1f, [0x597d] = 0x9d20,
  [0x597e] = 0x9d21, [0x5980] = 0x9d22, [0x5981] = 0x9d23, [0x5982] = 0x9d24,
  [0x5983] = 0x9d25, [0x5984] = 0x9d26, [0x5985] = 0x9d27, [0x5986] = 0x9d28,
  [0x5987] = 0x9d29, [0x5988] = 0x9d2a, [0x5989] = 0x9d2b, [0x598a] = 0x9d2c,
  [0x598b] = 0x9d2d, [0x598c] = 0x9d2e, [0x598d] = 0x9d2f, [0x598e] = 0x9d30,
  [0x598f] = 0x9d31, [0x5990] = 0x9d32, [0x5991] = 0x9d33, [0x5992] = 0x9d34,
  [0x5993] = 0x9d35, [0x5994] = 0x9d36, [0x5995] = 0x9d37, [0x5996] = 0x9d38,
  [0x5997] = 0x9d39, [0x5998] = 0x9d3a, [0x5999] = 0x9d3b, [0x599a] = 0x9d3c,
  [0x599b] = 0x9d3d, [0x599c] = 0x9d3e, [0x599d] = 0x9d3f, [0x599e] = 0x9d40,
  [0x599f] = 0x9d41, [0x59a0] = 0x9d42, [0x59a1] = 0xe234, [0x59a2] = 0xe235,
  [0x59a3] = 0xe236, [0x59a4] = 0xe237, [0x59a5] = 0xe238, [0x59a6] = 0xe239,
  [0x59a7] = 0xe23a, [0x59a8] = 0xe23b, [0x59a9] = 0xe23c, [0x59aa] = 0xe23d,
  [0x59ab] = 0xe23e, [0x59ac] = 0xe23f, [0x59ad] = 0xe240, [0x59ae] = 0xe241,
  [0x59af] = 0xe242, [0x59b0] = 0xe243, [0x59b1] = 0xe244, [0x59b2] = 0xe245,
  [0x59b3] = 0xe246, [0x59b4] = 0xe247, [0x59b5] = 0xe248, [0x59b6] = 0xe249,
  [0x59b7] = 0xe24a, [0x59b8] = 0xe24b, [0x59b9] = 0xe24c, [0x59ba] = 0xe24d,
  [0x59bb] = 0xe24e, [0x59bc] = 0xe24f, [0x59bd] = 0xe250, [0x59be] = 0xe251,
  [0x59bf] = 0xe252, [0x59c0] = 0xe253, [0x59c1] = 0xe254, [0x59c2] = 0xe255,
  [0x59c3] = 0xe256, [0x59c4] = 0xe257, [0x59c5] = 0xe258, [0x59c6] = 0xe259,
  [0x59c7] = 0xe25a, [0x59c8] = 0xe25b, [0x59c9] = 0xe25c, [0x59ca] = 0xe25d,
  [0x59cb] = 0xe25e, [0x59cc] = 0xe25f, [0x59cd] = 0xe260, [0x59ce] = 0xe261,
  [0x59cf] = 0xe262, [0x59d0] = 0xe263, [0x59d1] = 0xe264, [0x59d2] = 0xe265,
  [0x59d3] = 0xe266, [0x59d4] = 0xe267, [0x59d5] = 0xe268, [0x59d6] = 0xe269,
  [0x59d7] = 0xe26a, [0x59d8] = 0xe26b, [0x59d9] = 0xe26c, [0x59da] = 0xe26d,
  [0x59db] = 0xe26e, [0x59dc] = 0xe26f, [0x59dd] = 0xe270, [0x59de] = 0xe271,
  [0x59df] = 0xe272, [0x59e0] = 0xe273, [0x59e1] = 0xe274, [0x59e2] = 0xe275,
  [0x59e3] = 0xe276, [0x59e4] = 0xe277, [0x59e5] = 0xe278, [0x59e6] = 0xe279,
  [0x59e7] = 0xe27a, [0x59e8] = 0xe27b, [0x59e9] = 0xe27c, [0x59ea] = 0xe27d,
  [0x59eb] = 0xe27e, [0x59ec] = 0xe27f, [0x59ed] = 0xe280, [0x59ee] = 0xe281,
  [0x59ef] = 0xe282, [0x59f0] = 0xe283, [0x59f1] = 0xe284, [0x59f2] = 0xe285,
  [0x59f3] = 0xe286, [0x59f4] = 0xe287, [0x59f5] = 0xe288, [0x59f6] = 0xe289,
  [0x59f7] = 0xe28a, [0x59f8] = 0xe28b, [0x59f9] = 0xe28c, [0x59fa] = 0xe28d,
  [0x59fb] = 0xe28e, [0x59fc] = 0xe28f, [0x59fd] = 0xe290, [0x59fe] = 0xe291,
  [0x5a00] = 0x9d43, [0x5a01] = 0x9d44, [0x5a02] = 0x9d45, [0x5a03] = 0x9d46,
  [0x5a04] = 0x9d47, [0x5a05] = 0x9d48, [0x5a06] = 0x9d49, [0x5a07] = 0x9d4a,
  [0x5a08] = 0x9d4b, [0x5a09] = 0x9d4c, [0x5a0a] = 0x9d4d, [0x5a0b] = 0x9d4e,
  [0x5a0c] = 0x9d4f, [0x5a0d] = 0x9d50, [0x5a0e] = 0x9d51, [0x5a0f] = 0x9d52,
  [0x5a10] = 0x9d53, [0x5a11] = 0x9d54, [0x5a12] = 0x9d55, [0x5a13] = 0x9d56,
  [0x5a14] = 0x9d57, [0x5a15] = 0x9d58, [0x5a16] = 0x9d59, [0x5a17] = 0x9d5a,
  [0x5a18] = 0x9d5b, [0x5a19] = 0x9d5c, [0x5a1a] = 0x9d5d, [0x5a1b] = 0x9d5e,
  [0x5a1c] = 0x9d5f, [0x5a1d] = 0x9d60, [0x5a1e] = 0x9d61, [0x5a1f] = 0x9d62,
  [0x5a20] = 0x9d63, [0x5a21] = 0x9d64, [0x5a22] = 0x9d65, [0x5a23] = 0x9d66,
  [0x5a24] = 0x9d67, [0x5a25] = 0x9d68, [0x5a26] = 0x9d69, [0x5a27] = 0x9d6a,
  [0x5a28] = 0x9d6b, [0x5a29] = 0x9d6c, [0x5a2a] = 0x9d6d, [0x5a2b] = 0x9d6e,
  [0x5a2c] = 0x9d6f, [0x5a2d] = 0x9d70, [0x5a2e] = 0x9d71, [0x5a2f] = 0x9d72,
  [0x5a30] = 0x9d73, [0x5a31] = 0x9d74, [0x5a32] = 0x9d75, [0x5a33] = 0x9d76,
  [0x5a34] = 0x9d77, [0x5a35] = 0x9d78, [0x5a36] = 0x9d79, [0x5a37] = 0x9d7a,
  [0x5a38] = 0x9d7b, [0x5a39] = 0x9d7c, [0x5a3a] = 0x9d7d, [0x5a3b] = 0x9d7e,
  [0x5a3c] = 0x9d7f, [0x5a3d] = 0x9d80, [0x5a3e] = 0x9d81, [0x5a40] = 0x9d82,
  [0x5a41] = 0x9d83, [0x5a42] = 0x9d84, [0x5a43] = 0x9d85, [0x5a44] = 0x9d86,
  [0x5a45] = 0x9d87, [0x5a46] = 0x9d88, [0x5a47] = 0x9d89, [0x5a48] = 0x9d8a,
  [0x5a49] = 0x9d8b, [0x5a4a] = 0x9d8c, [0x5a4b] = 0x9d8d, [0x5a4c] = 0x9d8e,
  [0x5a4d] = 0x9d8f, [0x5a4e] = 0x9d90, [0x5a4f] = 0x9d91, [0x5a50] = 0x9d92,
  [0x5a51] = 0x9d93, [0x5a52] = 0x9d94, [0x5a53] = 0x9d95, [0x5a54] = 0x9d96,
  [0x5a55] = 0x9d97, [0x5a56] = 0x9d98, [0x5a57] = 0x9d99, [0x5a58] = 0x9d9a,
  [0x5a59] = 0x9d9b, [0x5a5a] = 0x9d9c, [0x5a5b] = 0x9d9d, [0x5a5c] = 0x9d9e,
  [0x5a5d] = 0x9d9f, [0x5a5e] = 0x9da0, [0x5a5f] = 0x9da1, [0x5a60] = 0x9da2,
  [0x5a61] = 0xe292, [0x5a62] = 0xe293, [0x5a63] = 0xe294, [0x5a64] = 0xe295,
  [0x5a65] = 0xe296, [0x5a66] = 0xe297, [0x5a67] = 0xe298, [0x5a68] = 0xe299,
  [0x5a69] = 0xe29a, [0x5a6a] = 0xe29b, [0x5a6b] = 0xe29c, [0x5a6c] = 0xe29d,
  [0x5a6d] = 0xe29e, [0x5a6e] = 0xe29f, [0x5a6f] = 0xe2a0, [0x5a70] = 0xe2a1,
  [0x5a71] = 0xe2a2, [0x5a72] = 0xe2a3, [0x5a73] = 0xe2a4, [0x5a74] = 0xe2a5,
  [0x5a75] = 0xe2a6, [0x5a76] = 0xe2a7, [0x5a77] = 0xe2a8, [0x5a78] = 0xe2a9,
  [0x5a79] = 0xe2aa, [0x5a7a] = 0xe2ab, [0x5a7b] = 0xe2ac, [0x5a7c] = 0xe2ad,
  [0x5a7d] = 0xe2ae, [0x5a7e] = 0xe2af, [0x5a7f] = 0xe2b0, [0x5a80] = 0xe2b1,
  [0x5a81] = 0xe2b2, [0x5a82] = 0xe2b3, [0x5a83] = 0xe2b4, [0x5a84] = 0xe2b5,
  [0x5a85] = 0xe2b6, [0x5a86] = 0xe2b7, [0x5a87] = 0xe2b8, [0x5a88] = 0xe2b9,
  [0x5a89] = 0xe2ba, [0x5a8a] = 0xe2bb, [0x5a8b] = 0xe2bc, [0x5a8c] = 0xe2bd,
  [0x5a8d] = 0xe2be, [0x5a8e] = 0xe2bf, [0x5a8f] = 0xe2c0, [0x5a90] = 0xe2c1,
  [0x5a91] = 0xe2c2, [0x5a92] = 0xe2c3, [0x5a93] = 0xe2c4, [0x5a94] = 0xe2c5,
  [0x5a95] = 0xe2c6, [0x5a96] = 0xe2c7, [0x5a97] = 0xe2c8, [0x5a98] = 0xe2c9,
  [0x5a99] = 0xe2ca, [0x5a9a] = 0xe2cb, [0x5a9b] = 0xe2cc, [0x5a9c] = 0xe2cd,
  [0x5a9d] = 0xe2ce, [0x5a9e] = 0xe2cf, [0x5a9f] = 0xe2d0, [0x5aa0] = 0xe2d1,
  [0x5aa1] = 0xe2d2, [0x5aa2] = 0xe2d3, [0x5aa3] = 0xe2d4, [0x5aa4] = 0xe2d5,
  [0x5aa5] = 0xe2d6, [0x5aa6] = 0xe2d7, [0x5aa7] = 0xe2d8, [0x5aa8] = 0xe2d9,
  [0x5aa9] = 0xe2da, [0x5aaa] = 0xe2db, [0x5aab] = 0xe2dc, [0x5aac] = 0xe2dd,
  [0x5aad] = 0xe2de, [0x5aae] = 0xe2df, [0x5aaf] = 0xe2e0, [0x5ab0] = 0xe2e1,
  [0x5ab1] = 0xe2e2, [0x5ab2] = 0xe2e3, [0x5ab3] = 0xe2e4, [0x5ab4] = 0xe2e5,
  [0x5ab5] = 0xe2e6, [0x5ab6] = 0xe2e7, [0x5ab7] = 0xe2e8, [0x5ab8] = 0xe2e9,
  [0x5ab9] = 0xe2ea, [0x5aba] = 0xe2eb, [0x5abb] = 0xe2ec, [0x5abc] = 0xe2ed,
  [0x5abd] = 0xe2ee, [0x5abe] = 0xe2ef, [0x5ac0] = 0x9da3, [0x5ac1] = 0x9da4,
  [0x5ac2] = 0x9da5, [0x5ac3] = 0x9da6, [0x5ac4] = 0x9da7, [0x5ac5] = 0x9da8,
  [0x5ac6] = 0x9da9, [0x5ac7] = 0x9daa, [0x5ac8] = 0x9dab, [0x5ac9] = 0x9dac,
  [0x5aca] = 0x9dad, [0x5acb] = 0x9dae, [0x5acc] = 0x9daf, [0x5acd] = 0x9db0,
  [0x5ace] = 0x9db1, [0x5acf] = 0x9db2, [0x5ad0] = 0x9db3, [0x5ad1] = 0x9db4,
  [0x5ad2] = 0x9db5, [0x5ad3] = 0x9db6, [0x5ad4] = 0x9db7, [0x5ad5] = 0x9db8,
  [0x5ad6] = 0x9db9, [0x5ad7] = 0x9dba, [0x5ad8] = 0x9dbb, [0x5ad9] = 0x9dbc,
  [0x5ada] = 0x9dbd, [0x5adb] = 0x9dbe, [0x5adc] = 0x9dbf, [0x5add] = 0x9dc0,
  [0x5ade] = 0x9dc1, [0x5adf] = 0x9dc2, [0x5ae0] = 0x9dc3, [0x5ae1] = 0x9dc4,
  [0x5ae2] = 0x9dc5, [0x5ae3] = 0x9dc6, [0x5ae4] = 0x9dc7, [0x5ae5] = 0x9dc8,
  [0x5ae6] = 0x9dc9, [0x5ae7] = 0x9dca, [0x5ae8] = 0x9dcb, [0x5ae9] = 0x9dcc,
  [0x5aea] = 0x9dcd, [0x5aeb] = 0x9dce, [0x5aec] = 0x9dcf, [0x5aed] = 0x9dd0,
  [0x5aee] = 0x9dd1, [0x5aef] = 0x9dd2, [0x5af0] = 0x9dd3, [0x5af1] = 0x9dd4,
  [0x5af2] = 0x9dd5, [0x5af3] = 0x9dd6, [0x5af4] = 0x9dd7, [0x5af5] = 0x9dd8,
  [0x5af6] = 0x9dd9, [0x5af7] = 0x9dda, [0x5af8] = 0x9ddb, [0x5af9] = 0x9ddc,
  [0x5afa] = 0x9ddd, [0x5afb] = 0x9dde, [0x5afc] = 0x9ddf, [0x5afd] = 0x9de0,
  [0x5afe] = 0x9de1, [0x5b00] = 0x9de2, [0x5b01] = 0x9de3, [0x5b02] = 0x9de4,
  [0x5b03] = 0x9de5, [0x5b04] = 0x9de6, [0x5b05] = 0x9de7, [0x5b06] = 0x9de8,
  [0x5b07] = 0x9de9, [0x5b08] = 0x9dea, [0x5b09] = 0x9deb, [0x5b0a] = 0x9dec,
  [0x5b0b] = 0x9ded, [0x5b0c] = 0x9dee, [0x5b0d] = 0x9def, [0x5b0e] = 0x9df0,
  [0x5b0f] = 0x9df1, [0x5b10] = 0x9df2, [0x5b11] = 0x9df3, [0x5b12] = 0x9df4,
  [0x5b13] = 0x9df5, [0x5b14] = 0x9df6, [0x5b15] = 0x9df7, [0x5b16] = 0x9df8,
  [0x5b17] = 0x9df9, [0x5b18] = 0x9dfa, [0x5b19] = 0x9dfb, [0x5b1a] = 0x9dfc,
  [0x5b1b] = 0x9dfd, [0x5b1c] = 0x9dfe, [0x5b1d] = 0x9dff, [0x5b1e] = 0x9e00,
  [0x5b1f] = 0x9e01, [0x5b20] = 0x9e02, [0x5b21] = 0xe2f0, [0x5b22] = 0xe2f1,
  [0x5b23] = 0xe2f2, [0x5b24] = 0xe2f3, [0x5b25] = 0xe2f4, [0x5b26] = 0xe2f5,
  [0x5b27] = 0xe2f6, [0x5b28] = 0xe2f7, [0x5b29] = 0xe2f8, [0x5b2a] = 0xe2f9,
  [0x5b2b] = 0xe2fa, [0x5b2c] = 0xe2fb, [0x5b2d] = 0xe2fc, [0x5b2e] = 0xe2fd,
  [0x5b2f] = 0xe2fe, [0x5b30] = 0xe2ff, [0x5b31] = 0xe300, [0x5b32] = 0xe301,
  [0x5b33] = 0xe302, [0x5b34] = 0xe303, [0x5b35] = 0xe304, [0x5b36] = 0xe305,
  [0x5b37] = 0xe306, [0x5b38] = 0xe307, [0x5b39] = 0xe308, [0x5b3a] = 0xe309,
  [0x5b3b] = 0xe30a, [0x5b3c] = 0xe30b, [0x5b3d] = 0xe30c, [0x5b3e] = 0xe30d,
  [0x5b3f] = 0xe30e, [0x5b40] = 0xe30f, [0x5b41] = 0xe310, [0x5b42] = 0xe311,
  [0x5b43] = 0xe312, [0x5b44] = 0xe313, [0x5b45] = 0xe314, [0x5b46] = 0xe315,
  [0x5b47] = 0xe316, [0x5b48] = 0xe317, [0x5b49] = 0xe318, [0x5b4a] = 0xe319,
  [0x5b4b] = 0xe31a, [0x5b4c] = 0xe31b, [0x5b4d] = 0xe31c, [0x5b4e] = 0xe31d,
  [0x5b4f] = 0xe31e, [0x5b50] = 0xe31f, [0x5b51] = 0xe320, [0x5b52] = 0xe321,
  [0x5b53] = 0xe322, [0x5b54] = 0xe323, [0x5b55] = 0xe324, [0x5b56] = 0xe325,
  [0x5b57] = 0xe326, [0x5b58] = 0xe327, [0x5b59] = 0xe328, [0x5b5a] = 0xe329,
  [0x5b5b] = 0xe32a, [0x5b5c] = 0xe32b, [0x5b5d] = 0xe32c, [0x5b5e] = 0xe32d,
  [0x5b5f] = 0xe32e, [0x5b60] = 0xe32f, [0x5b61] = 0xe330, [0x5b62] = 0xe331,
  [0x5b63] = 0xe332, [0x5b64] = 0xe333, [0x5b65] = 0xe334, [0x5b66] = 0xe335,
  [0x5b67] = 0xe336, [0x5b68] = 0xe337, [0x5b69] = 0xe338, [0x5b6a] = 0xe339,
  [0x5b6b] = 0xe33a, [0x5b6c] = 0xe33b, [0x5b6d] = 0xe33c, [0x5b6e] = 0xe33d,
  [0x5b6f] = 0xe33e, [0x5b70] = 0xe33f, [0x5b71] = 0xe340, [0x5b72] = 0xe341,
  [0x5b73] = 0xe342, [0x5b74] = 0xe343, [0x5b75] = 0xe344, [0x5b76] = 0xe345,
  [0x5b77] = 0xe346, [0x5b78] = 0xe347, [0x5b79] = 0xe348, [0x5b7a] = 0xe349,
  [0x5b7b] = 0xe34a, [0x5b7c] = 0xe34b, [0x5b7d] = 0xe34c, [0x5b7e] = 0xe34d,
  [0x5b80] = 0x9e03, [0x5b81] = 0x9e04, [0x5b82] = 0x9e05, [0x5b83] = 0x9e06,
  [0x5b84] = 0x9e07, [0x5b85] = 0x9e08, [0x5b86] = 0x9e09, [0x5b87] = 0x9e0a,
  [0x5b88] = 0x9e0b, [0x5b89] = 0x9e0c, [0x5b8a] = 0x9e0d, [0x5b8b] = 0x9e0e,
  [0x5b8c] = 0x9e0f, [0x5b8d] = 0x9e10, [0x5b8e] = 0x9e11, [0x5b8f] = 0x9e12,
  [0x5b90] = 0x9e13, [0x5b91] = 0x9e14, [0x5b92] = 0x9e15, [0x5b93] = 0x9e16,
  [0x5b94] = 0x9e17, [0x5b95] = 0x9e18, [0x5b96] = 0x9e19, [0x5b97] = 0x9e1a,
  [0x5b98] = 0x9e1b, [0x5b99] = 0x9e1c, [0x5b9a] = 0x9e1d, [0x5b9b] = 0x9e1e,
  [0x5b9c] = 0x9e24, [0x5b9d] = 0x9e27, [0x5b9e] = 0x9e2e, [0x5b9f] = 0x9e30,
  [0x5ba0] = 0x9e34, [0x5ba1] = 0x9e3b, [0x5ba2] = 0x9e3c, [0x5ba3] = 0x9e40,
  [0x5ba4] = 0x9e4d, [0x5ba5] = 0x9e50, [0x5ba6] = 0x9e52, [0x5ba7] = 0x9e53,
  [0x5ba8] = 0x9e54, [0x5ba9] = 0x9e56, [0x5baa] = 0x9e59, [0x5bab] = 0x9e5d,
  [0x5bac] = 0x9e5f, [0x5bad] = 0x9e60, [0x5bae] = 0x9e61, [0x5baf] = 0x9e62,
  [0x5bb0] = 0x9e65, [0x5bb1] = 0x9e6e, [0x5bb2] = 0x9e6f, [0x5bb3] = 0x9e72,
  [0x5bb4] = 0x9e74, [0x5bb5] = 0x9e75, [0x5bb6] = 0x9e76, [0x5bb7] = 0x9e77,
  [0x5bb8] = 0x9e78, [0x5bb9] = 0x9e79, [0x5bba] = 0x9e7a, [0x5bbb] = 0x9e7b,
  [0x5bbc] = 0x9e7c, [0x5bbd] = 0x9e7d, [0x5bbe] = 0x9e80, [0x5bc0] = 0x9e81,
  [0x5bc1] = 0x9e83, [0x5bc2] = 0x9e84, [0x5bc3] = 0x9e85, [0x5bc4] = 0x9e86,
  [0x5bc5] = 0x9e89, [0x5bc6] = 0x9e8a, [0x5bc7] = 0x9e8c, [0x5bc8] = 0x9e8d,
  [0x5bc9] = 0x9e8e, [0x5bca] = 0x9e8f, [0x5bcb] = 0x9e90, [0x5bcc] = 0x9e91,
  [0x5bcd] = 0x9e94, [0x5bce] = 0x9e95, [0x5bcf] = 0x9e96, [0x5bd0] = 0x9e97,
  [0x5bd1] = 0x9e98, [0x5bd2] = 0x9e99, [0x5bd3] = 0x9e9a, [0x5bd4] = 0x9e9b,
  [0x5bd5] = 0x9e9c, [0x5bd6] = 0x9e9e, [0x5bd7] = 0x9ea0, [0x5bd8] = 0x9ea1,
  [0x5bd9] = 0x9ea2, [0x5bda] = 0x9ea3, [0x5bdb] = 0x9ea4, [0x5bdc] = 0x9ea5,
  [0x5bdd] = 0x9ea7, [0x5bde] = 0x9ea8, [0x5bdf] = 0x9ea9, [0x5be0] = 0x9eaa,
  [0x5be1] = 0xe34e, [0x5be2] = 0xe34f, [0x5be3] = 0xe350, [0x5be4] = 0xe351,
  [0x5be5] = 0xe352, [0x5be6] = 0xe353, [0x5be7] = 0xe354, [0x5be8] = 0xe355,
  [0x5be9] = 0xe356, [0x5bea] = 0xe357, [0x5beb] = 0xe358, [0x5bec] = 0xe359,
  [0x5bed] = 0xe35a, [0x5bee] = 0xe35b, [0x5bef] = 0xe35c, [0x5bf0] = 0xe35d,
  [0x5bf1] = 0xe35e, [0x5bf2] = 0xe35f, [0x5bf3] = 0xe360, [0x5bf4] = 0xe361,
  [0x5bf5] = 0xe362, [0x5bf6] = 0xe363, [0x5bf7] = 0xe364, [0x5bf8] = 0xe365,
  [0x5bf9] = 0xe366, [0x5bfa] = 0xe367, [0x5bfb] = 0xe368, [0x5bfc] = 0xe369,
  [0x5bfd] = 0xe36a, [0x5bfe] = 0xe36b, [0x5bff] = 0xe36c, [0x5c00] = 0xe36d,
  [0x5c01] = 0xe36e, [0x5c02] = 0xe36f, [0x5c03] = 0xe370, [0x5c04] = 0xe371,
  [0x5c05] = 0xe372, [0x5c06] = 0xe373, [0x5c07] = 0xe374, [0x5c08] = 0xe375,
  [0x5c09] = 0xe376, [0x5c0a] = 0xe377, [0x5c0b] = 0xe378, [0x5c0c] = 0xe379,
  [0x5c0d] = 0xe37a, [0x5c0e] = 0xe37b, [0x5c0f] = 0xe37c, [0x5c10] = 0xe37d,
  [0x5c11] = 0xe37e, [0x5c12] = 0xe37f, [0x5c13] = 0xe380, [0x5c14] = 0xe381,
  [0x5c15] = 0xe382, [0x5c16] = 0xe383, [0x5c17] = 0xe384, [0x5c18] = 0xe385,
  [0x5c19] = 0xe386, [0x5c1a] = 0xe387, [0x5c1b] = 0xe388, [0x5c1c] = 0xe389,
  [0x5c1d] = 0xe38a, [0x5c1e] = 0xe38b, [0x5c1f] = 0xe38c, [0x5c20] = 0xe38d,
  [0x5c21] = 0xe38e, [0x5c22] = 0xe38f, [0x5c23] = 0xe390, [0x5c24] = 0xe391,
  [0x5c25] = 0xe392, [0x5c26] = 0xe393, [0x5c27] = 0xe394, [0x5c28] = 0xe395,
  [0x5c29] = 0xe396, [0x5c2a] = 0xe397, [0x5c2b] = 0xe398, [0x5c2c] = 0xe399,
  [0x5c2d] = 0xe39a, [0x5c2e] = 0xe39b, [0x5c2f] = 0xe39c, [0x5c30] = 0xe39d,
  [0x5c31] = 0xe39e, [0x5c32] = 0xe39f, [0x5c33] = 0xe3a0, [0x5c34] = 0xe3a1,
  [0x5c35] = 0xe3a2, [0x5c36] = 0xe3a3, [0x5c37] = 0xe3a4, [0x5c38] = 0xe3a5,
  [0x5c39] = 0xe3a6, [0x5c3a] = 0xe3a7, [0x5c3b] = 0xe3a8, [0x5c3c] = 0xe3a9,
  [0x5c3d] = 0xe3aa, [0x5c3e] = 0xe3ab, [0x5c40] = 0x9eab, [0x5c41] = 0x9eac,
  [0x5c42] = 0x9ead, [0x5c43] = 0x9eae, [0x5c44] = 0x9eaf, [0x5c45] = 0x9eb0,
  [0x5c46] = 0x9eb1, [0x5c47] = 0x9eb2, [0x5c48] = 0x9eb3, [0x5c49] = 0x9eb5,
  [0x5c4a] = 0x9eb6, [0x5c4b] = 0x9eb7, [0x5c4c] = 0x9eb9, [0x5c4d] = 0x9eba,
  [0x5c4e] = 0x9ebc, [0x5c4f] = 0x9ebf, [0x5c50] = 0x9ec0, [0x5c51] = 0x9ec1,
  [0x5c52] = 0x9ec2, [0x5c53] = 0x9ec3, [0x5c54] = 0x9ec5, [0x5c55] = 0x9ec6,
  [0x5c56] = 0x9ec7, [0x5c57] = 0x9ec8, [0x5c58] = 0x9eca, [0x5c59] = 0x9ecb,
  [0x5c5a] = 0x9ecc, [0x5c5b] = 0x9ed0, [0x5c5c] = 0x9ed2, [0x5c5d] = 0x9ed3,
  [0x5c5e] = 0x9ed5, [0x5c5f] = 0x9ed6, [0x5c60] = 0x9ed7, [0x5c61] = 0x9ed9,
  [0x5c62] = 0x9eda, [0x5c63] = 0x9ede, [0x5c64] = 0x9ee1, [0x5c65] = 0x9ee3,
  [0x5c66] = 0x9ee4, [0x5c67] = 0x9ee6, [0x5c68] = 0x9ee8, [0x5c69] = 0x9eeb,
  [0x5c6a] = 0x9eec, [0x5c6b] = 0x9eed, [0x5c6c] = 0x9eee, [0x5c6d] = 0x9ef0,
  [0x5c6e] = 0x9ef1, [0x5c6f] = 0x9ef2, [0x5c70] = 0x9ef3, [0x5c71] = 0x9ef4,
  [0x5c72] = 0x9ef5, [0x5c73] = 0x9ef6, [0x5c74] = 0x9ef7, [0x5c75] = 0x9ef8,
  [0x5c76] = 0x9efa, [0x5c77] = 0x9efd, [0x5c78] = 0x9eff, [0x5c79] = 0x9f00,
  [0x5c7a] = 0x9f01, [0x5c7b] = 0x9f02, [0x5c7c] = 0x9f03, [0x5c7d] = 0x9f04,
  [0x5c7e] = 0x9f05, [0x5c80] = 0x9f06, [0x5c81] = 0x9f07, [0x5c82] = 0x9f08,
  [0x5c83] = 0x9f09, [0x5c84] = 0x9f0a, [0x5c85] = 0x9f0c, [0x5c86] = 0x9f0f,
  [0x5c87] = 0x9f11, [0x5c88] = 0x9f12, [0x5c89] = 0x9f14, [0x5c8a] = 0x9f15,
  [0x5c8b] = 0x9f16, [0x5c8c] = 0x9f18, [0x5c8d] = 0x9f1a, [0x5c8e] = 0x9f1b,
  [0x5c8f] = 0x9f1c, [0x5c90] = 0x9f1d, [0x5c91] = 0x9f1e, [0x5c92] = 0x9f1f,
  [0x5c93] = 0x9f21, [0x5c94] = 0x9f23, [0x5c95] = 0x9f24, [0x5c96] = 0x9f25,
  [0x5c97] = 0x9f26, [0x5c98] = 0x9f27, [0x5c99] = 0x9f28, [0x5c9a] = 0x9f29,
  [0x5c9b] = 0x9f2a, [0x5c9c] = 0x9f2b, [0x5c9d] = 0x9f2d, [0x5c9e] = 0x9f2e,
  [0x5c9f] = 0x9f30, [0x5ca0] = 0x9f31, [0x5ca1] = 0xe3ac, [0x5ca2] = 0xe3ad,
  [0x5ca3] = 0xe3ae, [0x5ca4] = 0xe3af, [0x5ca5] = 0xe3b0, [0x5ca6] = 0xe3b1,
  [0x5ca7] = 0xe3b2, [0x5ca8] = 0xe3b3, [0x5ca9] = 0xe3b4, [0x5caa] = 0xe3b5,
  [0x5cab] = 0xe3b6, [0x5cac] = 0xe3b7, [0x5cad] = 0xe3b8, [0x5cae] = 0xe3b9,
  [0x5caf] = 0xe3ba, [0x5cb0] = 0xe3bb, [0x5cb1] = 0xe3bc, [0x5cb2] = 0xe3bd,
  [0x5cb3] = 0xe3be, [0x5cb4] = 0xe3bf, [0x5cb5] = 0xe3c0, [0x5cb6] = 0xe3c1,
  [0x5cb7] = 0xe3c2, [0x5cb8] = 0xe3c3, [0x5cb9] = 0xe3c4, [0x5cba] = 0xe3c5,
  [0x5cbb] = 0xe3c6, [0x5cbc] = 0xe3c7, [0x5cbd] = 0xe3c8, [0x5cbe] = 0xe3c9,
  [0x5cbf] = 0xe3ca, [0x5cc0] = 0xe3cb, [0x5cc1] = 0xe3cc, [0x5cc2] = 0xe3cd,
  [0x5cc3] = 0xe3ce, [0x5cc4] = 0xe3cf, [0x5cc5] = 0xe3d0, [0x5cc6] = 0xe3d1,
  [0x5cc7] = 0xe3d2, [0x5cc8] = 0xe3d3, [0x5cc9] = 0xe3d4, [0x5cca] = 0xe3d5,
  [0x5ccb] = 0xe3d6, [0x5ccc] = 0xe3d7, [0x5ccd] = 0xe3d8, [0x5cce] = 0xe3d9,
  [0x5ccf] = 0xe3da, [0x5cd0] = 0xe3db, [0x5cd1] = 0xe3dc, [0x5cd2] = 0xe3dd,
  [0x5cd3] = 0xe3de, [0x5cd4] = 0xe3df, [0x5cd5] = 0xe3e0, [0x5cd6] = 0xe3e1,
  [0x5cd7] = 0xe3e2, [0x5cd8] = 0xe3e3, [0x5cd9] = 0xe3e4, [0x5cda] = 0xe3e5,
  [0x5cdb] = 0xe3e6, [0x5cdc] = 0xe3e7, [0x5cdd] = 0xe3e8, [0x5cde] = 0xe3e9,
  [0x5cdf] = 0xe3ea, [0x5ce0] = 0xe3eb, [0x5ce1] = 0xe3ec, [0x5ce2] = 0xe3ed,
  [0x5ce3] = 0xe3ee, [0x5ce4] = 0xe3ef, [0x5ce5] = 0xe3f0, [0x5ce6] = 0xe3f1,
  [0x5ce7] = 0xe3f2, [0x5ce8] = 0xe3f3, [0x5ce9] = 0xe3f4, [0x5cea] = 0xe3f5,
  [0x5ceb] = 0xe3f6, [0x5cec] = 0xe3f7, [0x5ced] = 0xe3f8, [0x5cee] = 0xe3f9,
  [0x5cef] = 0xe3fa, [0x5cf0] = 0xe3fb, [0x5cf1] = 0xe3fc, [0x5cf2] = 0xe3fd,
  [0x5cf3] = 0xe3fe, [0x5cf4] = 0xe3ff, [0x5cf5] = 0xe400, [0x5cf6] = 0xe401,
  [0x5cf7] = 0xe402, [0x5cf8] = 0xe403, [0x5cf9] = 0xe404, [0x5cfa] = 0xe405,
  [0x5cfb] = 0xe406, [0x5cfc] = 0xe407, [0x5cfd] = 0xe408, [0x5cfe] = 0xe409,
  [0x5d00] = 0x9f32, [0x5d01] = 0x9f33, [0x5d02] = 0x9f34, [0x5d03] = 0x9f35,
  [0x5d04] = 0x9f36, [0x5d05] = 0x9f38, [0x5d06] = 0x9f3a, [0x5d07] = 0x9f3c,
  [0x5d08] = 0x9f3f, [0x5d09] = 0x9f40, [0x5d0a] = 0x9f41, [0x5d0b] = 0x9f42,
  [0x5d0c] = 0x9f43, [0x5d0d] = 0x9f45, [0x5d0e] = 0x9f46, [0x5d0f] = 0x9f47,
  [0x5d10] = 0x9f48, [0x5d11] = 0x9f49, [0x5d12] = 0x9f4a, [0x5d13] = 0x9f4b,
  [0x5d14] = 0x9f4c, [0x5d15] = 0x9f4d, [0x5d16] = 0x9f4e, [0x5d17] = 0x9f4f,
  [0x5d18] = 0x9f52, [0x5d19] = 0x9f53, [0x5d1a] = 0x9f54, [0x5d1b] = 0x9f55,
  [0x5d1c] = 0x9f56, [0x5d1d] = 0x9f57, [0x5d1e] = 0x9f58, [0x5d1f] = 0x9f59,
  [0x5d20] = 0x9f5a, [0x5d21] = 0x9f5b, [0x5d22] = 0x9f5c, [0x5d23] = 0x9f5d,
  [0x5d24] = 0x9f5e, [0x5d25] = 0x9f5f, [0x5d26] = 0x9f60, [0x5d27] = 0x9f61,
  [0x5d28] = 0x9f62, [0x5d29] = 0x9f63, [0x5d2a] = 0x9f64, [0x5d2b] = 0x9f65,
  [0x5d2c] = 0x9f66, [0x5d2d] = 0x9f67, [0x5d2e] = 0x9f68, [0x5d2f] = 0x9f69,
  [0x5d30] = 0x9f6a, [0x5d31] = 0x9f6b, [0x5d32] = 0x9f6c, [0x5d33] = 0x9f6d,
  [0x5d34] = 0x9f6e, [0x5d35] = 0x9f6f, [0x5d36] = 0x9f70, [0x5d37] = 0x9f71,
  [0x5d38] = 0x9f72, [0x5d39] = 0x9f73, [0x5d3a] = 0x9f74, [0x5d3b] = 0x9f75,
  [0x5d3c] = 0x9f76, [0x5d3d] = 0x9f77, [0x5d3e] = 0x9f78, [0x5d40] = 0x9f79,
  [0x5d41] = 0x9f7a, [0x5d42] = 0x9f7b, [0x5d43] = 0x9f7c, [0x5d44] = 0x9f7d,
  [0x5d45] = 0x9f7e, [0x5d46] = 0x9f81, [0x5d47] = 0x9f82, [0x5d48] = 0x9f8d,
  [0x5d49] = 0x9f8e, [0x5d4a] = 0x9f8f, [0x5d4b] = 0x9f90, [0x5d4c] = 0x9f91,
  [0x5d4d] = 0x9f92, [0x5d4e] = 0x9f93, [0x5d4f] = 0x9f94, [0x5d50] = 0x9f95,
  [0x5d51] = 0x9f96, [0x5d52] = 0x9f97, [0x5d53] = 0x9f98, [0x5d54] = 0x9f9c,
  [0x5d55] = 0x9f9d, [0x5d56] = 0x9f9e, [0x5d57] = 0x9fa1, [0x5d58] = 0x9fa2,
  [0x5d59] = 0x9fa3, [0x5d5a] = 0x9fa4, [0x5d5b] = 0x9fa5, [0x5d5c] = 0xf92c,
  [0x5d5d] = 0xf979, [0x5d5e] = 0xf995, [0x5d5f] = 0xf9e7, [0x5d60] = 0xf9f1,
  [0x5d61] = 0xe40a, [0x5d62] = 0xe40b, [0x5d63] = 0xe40c, [0x5d64] = 0xe40d,
  [0x5d65] = 0xe40e, [0x5d66] = 0xe40f, [0x5d67] = 0xe410, [0x5d68] = 0xe411,
  [0x5d69] = 0xe412, [0x5d6a] = 0xe413, [0x5d6b] = 0xe414, [0x5d6c] = 0xe415,
  [0x5d6d] = 0xe416, [0x5d6e] = 0xe417, [0x5d6f] = 0xe418, [0x5d70] = 0xe419,
  [0x5d71] = 0xe41a, [0x5d72] = 0xe41b, [0x5d73] = 0xe41c, [0x5d74] = 0xe41d,
  [0x5d75] = 0xe41e, [0x5d76] = 0xe41f, [0x5d77] = 0xe420, [0x5d78] = 0xe421,
  [0x5d79] = 0xe422, [0x5d7a] = 0xe423, [0x5d7b] = 0xe424, [0x5d7c] = 0xe425,
  [0x5d7d] = 0xe426, [0x5d7e] = 0xe427, [0x5d7f] = 0xe428, [0x5d80] = 0xe429,
  [0x5d81] = 0xe42a, [0x5d82] = 0xe42b, [0x5d83] = 0xe42c, [0x5d84] = 0xe42d,
  [0x5d85] = 0xe42e, [0x5d86] = 0xe42f, [0x5d87] = 0xe430, [0x5d88] = 0xe431,
  [0x5d89] = 0xe432, [0x5d8a] = 0xe433, [0x5d8b] = 0xe434, [0x5d8c] = 0xe435,
  [0x5d8d] = 0xe436, [0x5d8e] = 0xe437, [0x5d8f] = 0xe438, [0x5d90] = 0xe439,
  [0x5d91] = 0xe43a, [0x5d92] = 0xe43b, [0x5d93] = 0xe43c, [0x5d94] = 0xe43d,
  [0x5d95] = 0xe43e, [0x5d96] = 0xe43f, [0x5d97] = 0xe440, [0x5d98] = 0xe441,
  [0x5d99] = 0xe442, [0x5d9a] = 0xe443, [0x5d9b] = 0xe444, [0x5d9c] = 0xe445,
  [0x5d9d] = 0xe446, [0x5d9e] = 0xe447, [0x5d9f] = 0xe448, [0x5da0] = 0xe449,
  [0x5da1] = 0xe44a, [0x5da2] = 0xe44b, [0x5da3] = 0xe44c, [0x5da4] = 0xe44d,
  [0x5da5] = 0xe44e, [0x5da6] = 0xe44f, [0x5da7] = 0xe450, [0x5da8] = 0xe451,
  [0x5da9] = 0xe452, [0x5daa] = 0xe453, [0x5dab] = 0xe454, [0x5dac] = 0xe455,
  [0x5dad] = 0xe456, [0x5dae] = 0xe457, [0x5daf] = 0xe458, [0x5db0] = 0xe459,
  [0x5db1] = 0xe45a, [0x5db2] = 0xe45b, [0x5db3] = 0xe45c, [0x5db4] = 0xe45d,
  [0x5db5] = 0xe45e, [0x5db6] = 0xe45f, [0x5db7] = 0xe460, [0x5db8] = 0xe461,
  [0x5db9] = 0xe462, [0x5dba] = 0xe463, [0x5dbb] = 0xe464, [0x5dbc] = 0xe465,
  [0x5dbd] = 0xe466, [0x5dbe] = 0xe467, [0x5dc0] = 0xfa0c, [0x5dc1] = 0xfa0d,
  [0x5dc2] = 0xfa0e, [0x5dc3] = 0xfa0f, [0x5dc4] = 0xfa11, [0x5dc5] = 0xfa13,
  [0x5dc6] = 0xfa14, [0x5dc7] = 0xfa18, [0x5dc8] = 0xfa1f, [0x5dc9] = 0xfa20,
  [0x5dca] = 0xfa21, [0x5dcb] = 0xfa23, [0x5dcc] = 0xfa24, [0x5dcd] = 0xfa27,
  [0x5dce] = 0xfa28, [0x5dcf] = 0xfa29, [0x5dd0] = 0x2e81, [0x5dd4] = 0x2e84,
  [0x5dd5] = 0x3473, [0x5dd6] = 0x3447, [0x5dd7] = 0x2e88, [0x5dd8] = 0x2e8b,
  [0x5dd9] = 0x9fb4, [0x5dda] = 0x359e, [0x5ddb] = 0x361a, [0x5ddc] = 0x360e,
  [0x5ddd] = 0x2e8c, [0x5dde] = 0x2e97, [0x5ddf] = 0x396e, [0x5de0] = 0x3918,
  [0x5de1] = 0x9fb5, [0x5de2] = 0x39cf, [0x5de3] = 0x39df, [0x5de4] = 0x3a73,
  [0x5de5] = 0x39d0, [0x5de6] = 0x9fb6, [0x5de7] = 0x9fb7, [0x5de8] = 0x3b4e,
  [0x5de9] = 0x3c6e, [0x5dea] = 0x3ce0, [0x5deb] = 0x2ea7, [0x5ded] = 0x9fb8,
  [0x5dee] = 0x2eaa, [0x5def] = 0x4056, [0x5df0] = 0x415f, [0x5df1] = 0x2eae,
  [0x5df2] = 0x4337, [0x5df3] = 0x2eb3, [0x5df4] = 0x2eb6, [0x5df5] = 0x2eb7,
  [0x5df7] = 0x43b1, [0x5df8] = 0x43ac, [0x5df9] = 0x2ebb, [0x5dfa] = 0x43dd,
  [0x5dfb] = 0x44d6, [0x5dfc] = 0x4661, [0x5dfd] = 0x464c, [0x5dfe] = 0x9fb9,
  [0x5e00] = 0x4723, [0x5e01] = 0x4729, [0x5e02] = 0x477c, [0x5e03] = 0x478d,
  [0x5e04] = 0x2eca, [0x5e05] = 0x4947, [0x5e06] = 0x497a, [0x5e07] = 0x497d,
  [0x5e08] = 0x4982, [0x5e09] = 0x4983, [0x5e0a] = 0x4985, [0x5e0b] = 0x4986,
  [0x5e0c] = 0x499f, [0x5e0d] = 0x499b, [0x5e0e] = 0x49b7, [0x5e0f] = 0x49b6,
  [0x5e10] = 0x9fba, [0x5e12] = 0x4ca3, [0x5e13] = 0x4c9f, [0x5e14] = 0x4ca0,
  [0x5e15] = 0x4ca1, [0x5e16] = 0x4c77, [0x5e17] = 0x4ca2, [0x5e18] = 0x4d13,
  [0x5e19] = 0x4d14, [0x5e1a] = 0x4d15, [0x5e1b] = 0x4d16, [0x5e1c] = 0x4d17,
  [0x5e1d] = 0x4d18, [0x5e1e] = 0x4d19, [0x5e1f] = 0x4dae, [0x5e20] = 0x9fbb,
  [0x5e21] = 0xe468, [0x5e22] = 0xe469, [0x5e23] = 0xe46a, [0x5e24] = 0xe46b,
  [0x5e25] = 0xe46c, [0x5e26] = 0xe46d, [0x5e27] = 0xe46e, [0x5e28] = 0xe46f,
  [0x5e29] = 0xe470, [0x5e2a] = 0xe471, [0x5e2b] = 0xe472, [0x5e2c] = 0xe473,
  [0x5e2d] = 0xe474, [0x5e2e] = 0xe475, [0x5e2f] = 0xe476, [0x5e30] = 0xe477,
  [0x5e31] = 0xe478, [0x5e32] = 0xe479, [0x5e33] = 0xe47a, [0x5e34] = 0xe47b,
  [0x5e35] = 0xe47c, [0x5e36] = 0xe47d, [0x5e37] = 0xe47e, [0x5e38] = 0xe47f,
  [0x5e39] = 0xe480, [0x5e3a] = 0xe481, [0x5e3b] = 0xe482, [0x5e3c] = 0xe483,
  [0x5e3d] = 0xe484, [0x5e3e] = 0xe485, [0x5e3f] = 0xe486, [0x5e40] = 0xe487,
  [0x5e41] = 0xe488, [0x5e42] = 0xe489, [0x5e43] = 0xe48a, [0x5e44] = 0xe48b,
  [0x5e45] = 0xe48c, [0x5e46] = 0xe48d, [0x5e47] = 0xe48e, [0x5e48] = 0xe48f,
  [0x5e49] = 0xe490, [0x5e4a] = 0xe491, [0x5e4b] = 0xe492, [0x5e4c] = 0xe493,
  [0x5e4d] = 0xe494, [0x5e4e] = 0xe495, [0x5e4f] = 0xe496, [0x5e50] = 0xe497,
  [0x5e51] = 0xe498, [0x5e52] = 0xe499, [0x5e53] = 0xe49a, [0x5e54] = 0xe49b,
  [0x5e55] = 0xe49c, [0x5e56] = 0xe49d, [0x5e57] = 0xe49e, [0x5e58] = 0xe49f,
  [0x5e59] = 0xe4a0, [0x5e5a] = 0xe4a1, [0x5e5b] = 0xe4a2, [0x5e5c] = 0xe4a3,
  [0x5e5d] = 0xe4a4, [0x5e5e] = 0xe4a5, [0x5e5f] = 0xe4a6, [0x5e60] = 0xe4a7,
  [0x5e61] = 0xe4a8, [0x5e62] = 0xe4a9, [0x5e63] = 0xe4aa, [0x5e64] = 0xe4ab,
  [0x5e65] = 0xe4ac, [0x5e66] = 0xe4ad, [0x5e67] = 0xe4ae, [0x5e68] = 0xe4af,
  [0x5e69] = 0xe4b0, [0x5e6a] = 0xe4b1, [0x5e6b] = 0xe4b2, [0x5e6c] = 0xe4b3,
  [0x5e6d] = 0xe4b4, [0x5e6e] = 0xe4b5, [0x5e6f] = 0xe4b6, [0x5e70] = 0xe4b7,
  [0x5e71] = 0xe4b8, [0x5e72] = 0xe4b9, [0x5e73] = 0xe4ba, [0x5e74] = 0xe4bb,
  [0x5e75] = 0xe4bc, [0x5e76] = 0xe4bd, [0x5e77] = 0xe4be, [0x5e78] = 0xe4bf,
  [0x5e79] = 0xe4c0, [0x5e7a] = 0xe4c1, [0x5e7b] = 0xe4c2, [0x5e7c] = 0xe4c3,
  [0x5e7d] = 0xe4c4, [0x5e7e] = 0xe4c5,
};

/* Table for GB18030 -> UCS-4, containing the four-byte characters only,
   excluding four linear ranges.  Generated through

   clisp tab.lisp < GB18030.table

   where tab.lisp is:
   ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
   (locally (declare (compile))
     (let ((count 0))
       (loop
	 (let ((line (read-line nil nil nil)))
	   (unless line (return))
	   (when (eql (position #\Tab line) 10)
	     (let* ((gbk (parse-integer line :start 2 :end 10 :radix 16))
		    (ucs (parse-integer line :start 13 :radix 16))
		    (ch1 (ldb (byte 8 24) gbk))
		    (ch2 (ldb (byte 8 16) gbk))
		    (ch3 (ldb (byte 8 8) gbk))
		    (ch4 (ldb (byte 8 0) gbk))
		    (linear (+ (* (+ (* (+ (* (- ch1 #x81) 10)
					   (- ch2 #x30))
					126)
				     (- ch3 #x81))
				  10)
			       (- ch4 #x30)))
		    ; There are five ranges. The gap lengths are:
		    ;   (- #x1d21 #x334) = 6637,
		    ;   (- #x2c41 #x2403) = 2110,
		    ;   (- #x82bd #x4a79) = 14404,
		    ;   (- #x93d5 #x830e) = 4295.
		    (idx (cond ((<= 0 linear #x333) linear)
			       ((<= #x1d21 linear #x2402) (- linear 6637))
			       ((<= #x2c41 linear #x4a78) (- linear 6637 2110))
			       ((<= #x82bd linear #x830d) (- linear 6637 2110 14404))
			       ((<= #x93d5 linear #x99e1) (- linear 6637 2110 14404 4295))
			       (t nil))))
	       (when idx
		 (when (= (mod count 4) 0)
		   (format t " "))
		 (format t " ~([0x~4,'0X] = 0x~4,'0X~)," idx ucs)
		 (incf count)
		 (when (= (mod count 4) 0)
		   (format t "~%")))))))
       (format t "~%")))
   ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
*/

static const uint16_t __fourbyte_to_ucs[0x99e2 - 6637 - 2110 - 14404 - 4295] =
{
  [0x0000] = 0x0080, [0x0001] = 0x0081, [0x0002] = 0x0082, [0x0003] = 0x0083,
  [0x0004] = 0x0084, [0x0005] = 0x0085, [0x0006] = 0x0086, [0x0007] = 0x0087,
  [0x0008] = 0x0088, [0x0009] = 0x0089, [0x000a] = 0x008a, [0x000b] = 0x008b,
  [0x000c] = 0x008c, [0x000d] = 0x008d, [0x000e] = 0x008e, [0x000f] = 0x008f,
  [0x0010] = 0x0090, [0x0011] = 0x0091, [0x0012] = 0x0092, [0x0013] = 0x0093,
  [0x0014] = 0x0094, [0x0015] = 0x0095, [0x0016] = 0x0096, [0x0017] = 0x0097,
  [0x0018] = 0x0098, [0x0019] = 0x0099, [0x001a] = 0x009a, [0x001b] = 0x009b,
  [0x001c] = 0x009c, [0x001d] = 0x009d, [0x001e] = 0x009e, [0x001f] = 0x009f,
  [0x0020] = 0x00a0, [0x0021] = 0x00a1, [0x0022] = 0x00a2, [0x0023] = 0x00a3,
  [0x0024] = 0x00a5, [0x0025] = 0x00a6, [0x0026] = 0x00a9, [0x0027] = 0x00aa,
  [0x0028] = 0x00ab, [0x0029] = 0x00ac, [0x002a] = 0x00ad, [0x002b] = 0x00ae,
  [0x002c] = 0x00af, [0x002d] = 0x00b2, [0x002e] = 0x00b3, [0x002f] = 0x00b4,
  [0x0030] = 0x00b5, [0x0031] = 0x00b6, [0x0032] = 0x00b8, [0x0033] = 0x00b9,
  [0x0034] = 0x00ba, [0x0035] = 0x00bb, [0x0036] = 0x00bc, [0x0037] = 0x00bd,
  [0x0038] = 0x00be, [0x0039] = 0x00bf, [0x003a] = 0x00c0, [0x003b] = 0x00c1,
  [0x003c] = 0x00c2, [0x003d] = 0x00c3, [0x003e] = 0x00c4, [0x003f] = 0x00c5,
  [0x0040] = 0x00c6, [0x0041] = 0x00c7, [0x0042] = 0x00c8, [0x0043] = 0x00c9,
  [0x0044] = 0x00ca, [0x0045] = 0x00cb, [0x0046] = 0x00cc, [0x0047] = 0x00cd,
  [0x0048] = 0x00ce, [0x0049] = 0x00cf, [0x004a] = 0x00d0, [0x004b] = 0x00d1,
  [0x004c] = 0x00d2, [0x004d] = 0x00d3, [0x004e] = 0x00d4, [0x004f] = 0x00d5,
  [0x0050] = 0x00d6, [0x0051] = 0x00d8, [0x0052] = 0x00d9, [0x0053] = 0x00da,
  [0x0054] = 0x00db, [0x0055] = 0x00dc, [0x0056] = 0x00dd, [0x0057] = 0x00de,
  [0x0058] = 0x00df, [0x0059] = 0x00e2, [0x005a] = 0x00e3, [0x005b] = 0x00e4,
  [0x005c] = 0x00e5, [0x005d] = 0x00e6, [0x005e] = 0x00e7, [0x005f] = 0x00eb,
  [0x0060] = 0x00ee, [0x0061] = 0x00ef, [0x0062] = 0x00f0, [0x0063] = 0x00f1,
  [0x0064] = 0x00f4, [0x0065] = 0x00f5, [0x0066] = 0x00f6, [0x0067] = 0x00f8,
  [0x0068] = 0x00fb, [0x0069] = 0x00fd, [0x006a] = 0x00fe, [0x006b] = 0x00ff,
  [0x006c] = 0x0100, [0x006d] = 0x0102, [0x006e] = 0x0103, [0x006f] = 0x0104,
  [0x0070] = 0x0105, [0x0071] = 0x0106, [0x0072] = 0x0107, [0x0073] = 0x0108,
  [0x0074] = 0x0109, [0x0075] = 0x010a, [0x0076] = 0x010b, [0x0077] = 0x010c,
  [0x0078] = 0x010d, [0x0079] = 0x010e, [0x007a] = 0x010f, [0x007b] = 0x0110,
  [0x007c] = 0x0111, [0x007d] = 0x0112, [0x007e] = 0x0114, [0x007f] = 0x0115,
  [0x0080] = 0x0116, [0x0081] = 0x0117, [0x0082] = 0x0118, [0x0083] = 0x0119,
  [0x0084] = 0x011a, [0x0085] = 0x011c, [0x0086] = 0x011d, [0x0087] = 0x011e,
  [0x0088] = 0x011f, [0x0089] = 0x0120, [0x008a] = 0x0121, [0x008b] = 0x0122,
  [0x008c] = 0x0123, [0x008d] = 0x0124, [0x008e] = 0x0125, [0x008f] = 0x0126,
  [0x0090] = 0x0127, [0x0091] = 0x0128, [0x0092] = 0x0129, [0x0093] = 0x012a,
  [0x0094] = 0x012c, [0x0095] = 0x012d, [0x0096] = 0x012e, [0x0097] = 0x012f,
  [0x0098] = 0x0130, [0x0099] = 0x0131, [0x009a] = 0x0132, [0x009b] = 0x0133,
  [0x009c] = 0x0134, [0x009d] = 0x0135, [0x009e] = 0x0136, [0x009f] = 0x0137,
  [0x00a0] = 0x0138, [0x00a1] = 0x0139, [0x00a2] = 0x013a, [0x00a3] = 0x013b,
  [0x00a4] = 0x013c, [0x00a5] = 0x013d, [0x00a6] = 0x013e, [0x00a7] = 0x013f,
  [0x00a8] = 0x0140, [0x00a9] = 0x0141, [0x00aa] = 0x0142, [0x00ab] = 0x0143,
  [0x00ac] = 0x0145, [0x00ad] = 0x0146, [0x00ae] = 0x0147, [0x00af] = 0x0149,
  [0x00b0] = 0x014a, [0x00b1] = 0x014b, [0x00b2] = 0x014c, [0x00b3] = 0x014e,
  [0x00b4] = 0x014f, [0x00b5] = 0x0150, [0x00b6] = 0x0151, [0x00b7] = 0x0152,
  [0x00b8] = 0x0153, [0x00b9] = 0x0154, [0x00ba] = 0x0155, [0x00bb] = 0x0156,
  [0x00bc] = 0x0157, [0x00bd] = 0x0158, [0x00be] = 0x0159, [0x00bf] = 0x015a,
  [0x00c0] = 0x015b, [0x00c1] = 0x015c, [0x00c2] = 0x015d, [0x00c3] = 0x015e,
  [0x00c4] = 0x015f, [0x00c5] = 0x0160, [0x00c6] = 0x0161, [0x00c7] = 0x0162,
  [0x00c8] = 0x0163, [0x00c9] = 0x0164, [0x00ca] = 0x0165, [0x00cb] = 0x0166,
  [0x00cc] = 0x0167, [0x00cd] = 0x0168, [0x00ce] = 0x0169, [0x00cf] = 0x016a,
  [0x00d0] = 0x016c, [0x00d1] = 0x016d, [0x00d2] = 0x016e, [0x00d3] = 0x016f,
  [0x00d4] = 0x0170, [0x00d5] = 0x0171, [0x00d6] = 0x0172, [0x00d7] = 0x0173,
  [0x00d8] = 0x0174, [0x00d9] = 0x0175, [0x00da] = 0x0176, [0x00db] = 0x0177,
  [0x00dc] = 0x0178, [0x00dd] = 0x0179, [0x00de] = 0x017a, [0x00df] = 0x017b,
  [0x00e0] = 0x017c, [0x00e1] = 0x017d, [0x00e2] = 0x017e, [0x00e3] = 0x017f,
  [0x00e4] = 0x0180, [0x00e5] = 0x0181, [0x00e6] = 0x0182, [0x00e7] = 0x0183,
  [0x00e8] = 0x0184, [0x00e9] = 0x0185, [0x00ea] = 0x0186, [0x00eb] = 0x0187,
  [0x00ec] = 0x0188, [0x00ed] = 0x0189, [0x00ee] = 0x018a, [0x00ef] = 0x018b,
  [0x00f0] = 0x018c, [0x00f1] = 0x018d, [0x00f2] = 0x018e, [0x00f3] = 0x018f,
  [0x00f4] = 0x0190, [0x00f5] = 0x0191, [0x00f6] = 0x0192, [0x00f7] = 0x0193,
  [0x00f8] = 0x0194, [0x00f9] = 0x0195, [0x00fa] = 0x0196, [0x00fb] = 0x0197,
  [0x00fc] = 0x0198, [0x00fd] = 0x0199, [0x00fe] = 0x019a, [0x00ff] = 0x019b,
  [0x0100] = 0x019c, [0x0101] = 0x019d, [0x0102] = 0x019e, [0x0103] = 0x019f,
  [0x0104] = 0x01a0, [0x0105] = 0x01a1, [0x0106] = 0x01a2, [0x0107] = 0x01a3,
  [0x0108] = 0x01a4, [0x0109] = 0x01a5, [0x010a] = 0x01a6, [0x010b] = 0x01a7,
  [0x010c] = 0x01a8, [0x010d] = 0x01a9, [0x010e] = 0x01aa, [0x010f] = 0x01ab,
  [0x0110] = 0x01ac, [0x0111] = 0x01ad, [0x0112] = 0x01ae, [0x0113] = 0x01af,
  [0x0114] = 0x01b0, [0x0115] = 0x01b1, [0x0116] = 0x01b2, [0x0117] = 0x01b3,
  [0x0118] = 0x01b4, [0x0119] = 0x01b5, [0x011a] = 0x01b6, [0x011b] = 0x01b7,
  [0x011c] = 0x01b8, [0x011d] = 0x01b9, [0x011e] = 0x01ba, [0x011f] = 0x01bb,
  [0x0120] = 0x01bc, [0x0121] = 0x01bd, [0x0122] = 0x01be, [0x0123] = 0x01bf,
  [0x0124] = 0x01c0, [0x0125] = 0x01c1, [0x0126] = 0x01c2, [0x0127] = 0x01c3,
  [0x0128] = 0x01c4, [0x0129] = 0x01c5, [0x012a] = 0x01c6, [0x012b] = 0x01c7,
  [0x012c] = 0x01c8, [0x012d] = 0x01c9, [0x012e] = 0x01ca, [0x012f] = 0x01cb,
  [0x0130] = 0x01cc, [0x0131] = 0x01cd, [0x0132] = 0x01cf, [0x0133] = 0x01d1,
  [0x0134] = 0x01d3, [0x0135] = 0x01d5, [0x0136] = 0x01d7, [0x0137] = 0x01d9,
  [0x0138] = 0x01db, [0x0139] = 0x01dd, [0x013a] = 0x01de, [0x013b] = 0x01df,
  [0x013c] = 0x01e0, [0x013d] = 0x01e1, [0x013e] = 0x01e2, [0x013f] = 0x01e3,
  [0x0140] = 0x01e4, [0x0141] = 0x01e5, [0x0142] = 0x01e6, [0x0143] = 0x01e7,
  [0x0144] = 0x01e8, [0x0145] = 0x01e9, [0x0146] = 0x01ea, [0x0147] = 0x01eb,
  [0x0148] = 0x01ec, [0x0149] = 0x01ed, [0x014a] = 0x01ee, [0x014b] = 0x01ef,
  [0x014c] = 0x01f0, [0x014d] = 0x01f1, [0x014e] = 0x01f2, [0x014f] = 0x01f3,
  [0x0150] = 0x01f4, [0x0151] = 0x01f5, [0x0152] = 0x01f6, [0x0153] = 0x01f7,
  [0x0154] = 0x01f8, [0x0155] = 0x01fa, [0x0156] = 0x01fb, [0x0157] = 0x01fc,
  [0x0158] = 0x01fd, [0x0159] = 0x01fe, [0x015a] = 0x01ff, [0x015b] = 0x0200,
  [0x015c] = 0x0201, [0x015d] = 0x0202, [0x015e] = 0x0203, [0x015f] = 0x0204,
  [0x0160] = 0x0205, [0x0161] = 0x0206, [0x0162] = 0x0207, [0x0163] = 0x0208,
  [0x0164] = 0x0209, [0x0165] = 0x020a, [0x0166] = 0x020b, [0x0167] = 0x020c,
  [0x0168] = 0x020d, [0x0169] = 0x020e, [0x016a] = 0x020f, [0x016b] = 0x0210,
  [0x016c] = 0x0211, [0x016d] = 0x0212, [0x016e] = 0x0213, [0x016f] = 0x0214,
  [0x0170] = 0x0215, [0x0171] = 0x0216, [0x0172] = 0x0217, [0x0173] = 0x0218,
  [0x0174] = 0x0219, [0x0175] = 0x021a, [0x0176] = 0x021b, [0x0177] = 0x021c,
  [0x0178] = 0x021d, [0x0179] = 0x021e, [0x017a] = 0x021f, [0x017b] = 0x0220,
  [0x017c] = 0x0221, [0x017d] = 0x0222, [0x017e] = 0x0223, [0x017f] = 0x0224,
  [0x0180] = 0x0225, [0x0181] = 0x0226, [0x0182] = 0x0227, [0x0183] = 0x0228,
  [0x0184] = 0x0229, [0x0185] = 0x022a, [0x0186] = 0x022b, [0x0187] = 0x022c,
  [0x0188] = 0x022d, [0x0189] = 0x022e, [0x018a] = 0x022f, [0x018b] = 0x0230,
  [0x018c] = 0x0231, [0x018d] = 0x0232, [0x018e] = 0x0233, [0x018f] = 0x0234,
  [0x0190] = 0x0235, [0x0191] = 0x0236, [0x0192] = 0x0237, [0x0193] = 0x0238,
  [0x0194] = 0x0239, [0x0195] = 0x023a, [0x0196] = 0x023b, [0x0197] = 0x023c,
  [0x0198] = 0x023d, [0x0199] = 0x023e, [0x019a] = 0x023f, [0x019b] = 0x0240,
  [0x019c] = 0x0241, [0x019d] = 0x0242, [0x019e] = 0x0243, [0x019f] = 0x0244,
  [0x01a0] = 0x0245, [0x01a1] = 0x0246, [0x01a2] = 0x0247, [0x01a3] = 0x0248,
  [0x01a4] = 0x0249, [0x01a5] = 0x024a, [0x01a6] = 0x024b, [0x01a7] = 0x024c,
  [0x01a8] = 0x024d, [0x01a9] = 0x024e, [0x01aa] = 0x024f, [0x01ab] = 0x0250,
  [0x01ac] = 0x0252, [0x01ad] = 0x0253, [0x01ae] = 0x0254, [0x01af] = 0x0255,
  [0x01b0] = 0x0256, [0x01b1] = 0x0257, [0x01b2] = 0x0258, [0x01b3] = 0x0259,
  [0x01b4] = 0x025a, [0x01b5] = 0x025b, [0x01b6] = 0x025c, [0x01b7] = 0x025d,
  [0x01b8] = 0x025e, [0x01b9] = 0x025f, [0x01ba] = 0x0260, [0x01bb] = 0x0262,
  [0x01bc] = 0x0263, [0x01bd] = 0x0264, [0x01be] = 0x0265, [0x01bf] = 0x0266,
  [0x01c0] = 0x0267, [0x01c1] = 0x0268, [0x01c2] = 0x0269, [0x01c3] = 0x026a,
  [0x01c4] = 0x026b, [0x01c5] = 0x026c, [0x01c6] = 0x026d, [0x01c7] = 0x026e,
  [0x01c8] = 0x026f, [0x01c9] = 0x0270, [0x01ca] = 0x0271, [0x01cb] = 0x0272,
  [0x01cc] = 0x0273, [0x01cd] = 0x0274, [0x01ce] = 0x0275, [0x01cf] = 0x0276,
  [0x01d0] = 0x0277, [0x01d1] = 0x0278, [0x01d2] = 0x0279, [0x01d3] = 0x027a,
  [0x01d4] = 0x027b, [0x01d5] = 0x027c, [0x01d6] = 0x027d, [0x01d7] = 0x027e,
  [0x01d8] = 0x027f, [0x01d9] = 0x0280, [0x01da] = 0x0281, [0x01db] = 0x0282,
  [0x01dc] = 0x0283, [0x01dd] = 0x0284, [0x01de] = 0x0285, [0x01df] = 0x0286,
  [0x01e0] = 0x0287, [0x01e1] = 0x0288, [0x01e2] = 0x0289, [0x01e3] = 0x028a,
  [0x01e4] = 0x028b, [0x01e5] = 0x028c, [0x01e6] = 0x028d, [0x01e7] = 0x028e,
  [0x01e8] = 0x028f, [0x01e9] = 0x0290, [0x01ea] = 0x0291, [0x01eb] = 0x0292,
  [0x01ec] = 0x0293, [0x01ed] = 0x0294, [0x01ee] = 0x0295, [0x01ef] = 0x0296,
  [0x01f0] = 0x0297, [0x01f1] = 0x0298, [0x01f2] = 0x0299, [0x01f3] = 0x029a,
  [0x01f4] = 0x029b, [0x01f5] = 0x029c, [0x01f6] = 0x029d, [0x01f7] = 0x029e,
  [0x01f8] = 0x029f, [0x01f9] = 0x02a0, [0x01fa] = 0x02a1, [0x01fb] = 0x02a2,
  [0x01fc] = 0x02a3, [0x01fd] = 0x02a4, [0x01fe] = 0x02a5, [0x01ff] = 0x02a6,
  [0x0200] = 0x02a7, [0x0201] = 0x02a8, [0x0202] = 0x02a9, [0x0203] = 0x02aa,
  [0x0204] = 0x02ab, [0x0205] = 0x02ac, [0x0206] = 0x02ad, [0x0207] = 0x02ae,
  [0x0208] = 0x02af, [0x0209] = 0x02b0, [0x020a] = 0x02b1, [0x020b] = 0x02b2,
  [0x020c] = 0x02b3, [0x020d] = 0x02b4, [0x020e] = 0x02b5, [0x020f] = 0x02b6,
  [0x0210] = 0x02b7, [0x0211] = 0x02b8, [0x0212] = 0x02b9, [0x0213] = 0x02ba,
  [0x0214] = 0x02bb, [0x0215] = 0x02bc, [0x0216] = 0x02bd, [0x0217] = 0x02be,
  [0x0218] = 0x02bf, [0x0219] = 0x02c0, [0x021a] = 0x02c1, [0x021b] = 0x02c2,
  [0x021c] = 0x02c3, [0x021d] = 0x02c4, [0x021e] = 0x02c5, [0x021f] = 0x02c6,
  [0x0220] = 0x02c8, [0x0221] = 0x02cc, [0x0222] = 0x02cd, [0x0223] = 0x02ce,
  [0x0224] = 0x02cf, [0x0225] = 0x02d0, [0x0226] = 0x02d1, [0x0227] = 0x02d2,
  [0x0228] = 0x02d3, [0x0229] = 0x02d4, [0x022a] = 0x02d5, [0x022b] = 0x02d6,
  [0x022c] = 0x02d7, [0x022d] = 0x02d8, [0x022e] = 0x02da, [0x022f] = 0x02db,
  [0x0230] = 0x02dc, [0x0231] = 0x02dd, [0x0232] = 0x02de, [0x0233] = 0x02df,
  [0x0234] = 0x02e0, [0x0235] = 0x02e1, [0x0236] = 0x02e2, [0x0237] = 0x02e3,
  [0x0238] = 0x02e4, [0x0239] = 0x02e5, [0x023a] = 0x02e6, [0x023b] = 0x02e7,
  [0x023c] = 0x02e8, [0x023d] = 0x02e9, [0x023e] = 0x02ea, [0x023f] = 0x02eb,
  [0x0240] = 0x02ec, [0x0241] = 0x02ed, [0x0242] = 0x02ee, [0x0243] = 0x02ef,
  [0x0244] = 0x02f0, [0x0245] = 0x02f1, [0x0246] = 0x02f2, [0x0247] = 0x02f3,
  [0x0248] = 0x02f4, [0x0249] = 0x02f5, [0x024a] = 0x02f6, [0x024b] = 0x02f7,
  [0x024c] = 0x02f8, [0x024d] = 0x02f9, [0x024e] = 0x02fa, [0x024f] = 0x02fb,
  [0x0250] = 0x02fc, [0x0251] = 0x02fd, [0x0252] = 0x02fe, [0x0253] = 0x02ff,
  [0x0254] = 0x0300, [0x0255] = 0x0301, [0x0256] = 0x0302, [0x0257] = 0x0303,
  [0x0258] = 0x0304, [0x0259] = 0x0305, [0x025a] = 0x0306, [0x025b] = 0x0307,
  [0x025c] = 0x0308, [0x025d] = 0x0309, [0x025e] = 0x030a, [0x025f] = 0x030b,
  [0x0260] = 0x030c, [0x0261] = 0x030d, [0x0262] = 0x030e, [0x0263] = 0x030f,
  [0x0264] = 0x0310, [0x0265] = 0x0311, [0x0266] = 0x0312, [0x0267] = 0x0313,
  [0x0268] = 0x0314, [0x0269] = 0x0315, [0x026a] = 0x0316, [0x026b] = 0x0317,
  [0x026c] = 0x0318, [0x026d] = 0x0319, [0x026e] = 0x031a, [0x026f] = 0x031b,
  [0x0270] = 0x031c, [0x0271] = 0x031d, [0x0272] = 0x031e, [0x0273] = 0x031f,
  [0x0274] = 0x0320, [0x0275] = 0x0321, [0x0276] = 0x0322, [0x0277] = 0x0323,
  [0x0278] = 0x0324, [0x0279] = 0x0325, [0x027a] = 0x0326, [0x027b] = 0x0327,
  [0x027c] = 0x0328, [0x027d] = 0x0329, [0x027e] = 0x032a, [0x027f] = 0x032b,
  [0x0280] = 0x032c, [0x0281] = 0x032d, [0x0282] = 0x032e, [0x0283] = 0x032f,
  [0x0284] = 0x0330, [0x0285] = 0x0331, [0x0286] = 0x0332, [0x0287] = 0x0333,
  [0x0288] = 0x0334, [0x0289] = 0x0335, [0x028a] = 0x0336, [0x028b] = 0x0337,
  [0x028c] = 0x0338, [0x028d] = 0x0339, [0x028e] = 0x033a, [0x028f] = 0x033b,
  [0x0290] = 0x033c, [0x0291] = 0x033d, [0x0292] = 0x033e, [0x0293] = 0x033f,
  [0x0294] = 0x0340, [0x0295] = 0x0341, [0x0296] = 0x0342, [0x0297] = 0x0343,
  [0x0298] = 0x0344, [0x0299] = 0x0345, [0x029a] = 0x0346, [0x029b] = 0x0347,
  [0x029c] = 0x0348, [0x029d] = 0x0349, [0x029e] = 0x034a, [0x029f] = 0x034b,
  [0x02a0] = 0x034c, [0x02a1] = 0x034d, [0x02a2] = 0x034e, [0x02a3] = 0x034f,
  [0x02a4] = 0x0350, [0x02a5] = 0x0351, [0x02a6] = 0x0352, [0x02a7] = 0x0353,
  [0x02a8] = 0x0354, [0x02a9] = 0x0355, [0x02aa] = 0x0356, [0x02ab] = 0x0357,
  [0x02ac] = 0x0358, [0x02ad] = 0x0359, [0x02ae] = 0x035a, [0x02af] = 0x035b,
  [0x02b0] = 0x035c, [0x02b1] = 0x035d, [0x02b2] = 0x035e, [0x02b3] = 0x035f,
  [0x02b4] = 0x0360, [0x02b5] = 0x0361, [0x02b6] = 0x0362, [0x02b7] = 0x0363,
  [0x02b8] = 0x0364, [0x02b9] = 0x0365, [0x02ba] = 0x0366, [0x02bb] = 0x0367,
  [0x02bc] = 0x0368, [0x02bd] = 0x0369, [0x02be] = 0x036a, [0x02bf] = 0x036b,
  [0x02c0] = 0x036c, [0x02c1] = 0x036d, [0x02c2] = 0x036e, [0x02c3] = 0x036f,
  [0x02c4] = 0x0370, [0x02c5] = 0x0371, [0x02c6] = 0x0372, [0x02c7] = 0x0373,
  [0x02c8] = 0x0374, [0x02c9] = 0x0375, [0x02ca] = 0x0376, [0x02cb] = 0x0377,
  [0x02cc] = 0x0378, [0x02cd] = 0x0379, [0x02ce] = 0x037a, [0x02cf] = 0x037b,
  [0x02d0] = 0x037c, [0x02d1] = 0x037d, [0x02d2] = 0x037e, [0x02d3] = 0x037f,
  [0x02d4] = 0x0380, [0x02d5] = 0x0381, [0x02d6] = 0x0382, [0x02d7] = 0x0383,
  [0x02d8] = 0x0384, [0x02d9] = 0x0385, [0x02da] = 0x0386, [0x02db] = 0x0387,
  [0x02dc] = 0x0388, [0x02dd] = 0x0389, [0x02de] = 0x038a, [0x02df] = 0x038b,
  [0x02e0] = 0x038c, [0x02e1] = 0x038d, [0x02e2] = 0x038e, [0x02e3] = 0x038f,
  [0x02e4] = 0x0390, [0x02e5] = 0x03a2, [0x02e6] = 0x03aa, [0x02e7] = 0x03ab,
  [0x02e8] = 0x03ac, [0x02e9] = 0x03ad, [0x02ea] = 0x03ae, [0x02eb] = 0x03af,
  [0x02ec] = 0x03b0, [0x02ed] = 0x03c2, [0x02ee] = 0x03ca, [0x02ef] = 0x03cb,
  [0x02f0] = 0x03cc, [0x02f1] = 0x03cd, [0x02f2] = 0x03ce, [0x02f3] = 0x03cf,
  [0x02f4] = 0x03d0, [0x02f5] = 0x03d1, [0x02f6] = 0x03d2, [0x02f7] = 0x03d3,
  [0x02f8] = 0x03d4, [0x02f9] = 0x03d5, [0x02fa] = 0x03d6, [0x02fb] = 0x03d7,
  [0x02fc] = 0x03d8, [0x02fd] = 0x03d9, [0x02fe] = 0x03da, [0x02ff] = 0x03db,
  [0x0300] = 0x03dc, [0x0301] = 0x03dd, [0x0302] = 0x03de, [0x0303] = 0x03df,
  [0x0304] = 0x03e0, [0x0305] = 0x03e1, [0x0306] = 0x03e2, [0x0307] = 0x03e3,
  [0x0308] = 0x03e4, [0x0309] = 0x03e5, [0x030a] = 0x03e6, [0x030b] = 0x03e7,
  [0x030c] = 0x03e8, [0x030d] = 0x03e9, [0x030e] = 0x03ea, [0x030f] = 0x03eb,
  [0x0310] = 0x03ec, [0x0311] = 0x03ed, [0x0312] = 0x03ee, [0x0313] = 0x03ef,
  [0x0314] = 0x03f0, [0x0315] = 0x03f1, [0x0316] = 0x03f2, [0x0317] = 0x03f3,
  [0x0318] = 0x03f4, [0x0319] = 0x03f5, [0x031a] = 0x03f6, [0x031b] = 0x03f7,
  [0x031c] = 0x03f8, [0x031d] = 0x03f9, [0x031e] = 0x03fa, [0x031f] = 0x03fb,
  [0x0320] = 0x03fc, [0x0321] = 0x03fd, [0x0322] = 0x03fe, [0x0323] = 0x03ff,
  [0x0324] = 0x0400, [0x0325] = 0x0402, [0x0326] = 0x0403, [0x0327] = 0x0404,
  [0x0328] = 0x0405, [0x0329] = 0x0406, [0x032a] = 0x0407, [0x032b] = 0x0408,
  [0x032c] = 0x0409, [0x032d] = 0x040a, [0x032e] = 0x040b, [0x032f] = 0x040c,
  [0x0330] = 0x040d, [0x0331] = 0x040e, [0x0332] = 0x040f, [0x0333] = 0x0450,
  [0x0334] = 0xe7c7, [0x0335] = 0x1e40, [0x0336] = 0x1e41, [0x0337] = 0x1e42,
  [0x0338] = 0x1e43, [0x0339] = 0x1e44, [0x033a] = 0x1e45, [0x033b] = 0x1e46,
  [0x033c] = 0x1e47, [0x033d] = 0x1e48, [0x033e] = 0x1e49, [0x033f] = 0x1e4a,
  [0x0340] = 0x1e4b, [0x0341] = 0x1e4c, [0x0342] = 0x1e4d, [0x0343] = 0x1e4e,
  [0x0344] = 0x1e4f, [0x0345] = 0x1e50, [0x0346] = 0x1e51, [0x0347] = 0x1e52,
  [0x0348] = 0x1e53, [0x0349] = 0x1e54, [0x034a] = 0x1e55, [0x034b] = 0x1e56,
  [0x034c] = 0x1e57, [0x034d] = 0x1e58, [0x034e] = 0x1e59, [0x034f] = 0x1e5a,
  [0x0350] = 0x1e5b, [0x0351] = 0x1e5c, [0x0352] = 0x1e5d, [0x0353] = 0x1e5e,
  [0x0354] = 0x1e5f, [0x0355] = 0x1e60, [0x0356] = 0x1e61, [0x0357] = 0x1e62,
  [0x0358] = 0x1e63, [0x0359] = 0x1e64, [0x035a] = 0x1e65, [0x035b] = 0x1e66,
  [0x035c] = 0x1e67, [0x035d] = 0x1e68, [0x035e] = 0x1e69, [0x035f] = 0x1e6a,
  [0x0360] = 0x1e6b, [0x0361] = 0x1e6c, [0x0362] = 0x1e6d, [0x0363] = 0x1e6e,
  [0x0364] = 0x1e6f, [0x0365] = 0x1e70, [0x0366] = 0x1e71, [0x0367] = 0x1e72,
  [0x0368] = 0x1e73, [0x0369] = 0x1e74, [0x036a] = 0x1e75, [0x036b] = 0x1e76,
  [0x036c] = 0x1e77, [0x036d] = 0x1e78, [0x036e] = 0x1e79, [0x036f] = 0x1e7a,
  [0x0370] = 0x1e7b, [0x0371] = 0x1e7c, [0x0372] = 0x1e7d, [0x0373] = 0x1e7e,
  [0x0374] = 0x1e7f, [0x0375] = 0x1e80, [0x0376] = 0x1e81, [0x0377] = 0x1e82,
  [0x0378] = 0x1e83, [0x0379] = 0x1e84, [0x037a] = 0x1e85, [0x037b] = 0x1e86,
  [0x037c] = 0x1e87, [0x037d] = 0x1e88, [0x037e] = 0x1e89, [0x037f] = 0x1e8a,
  [0x0380] = 0x1e8b, [0x0381] = 0x1e8c, [0x0382] = 0x1e8d, [0x0383] = 0x1e8e,
  [0x0384] = 0x1e8f, [0x0385] = 0x1e90, [0x0386] = 0x1e91, [0x0387] = 0x1e92,
  [0x0388] = 0x1e93, [0x0389] = 0x1e94, [0x038a] = 0x1e95, [0x038b] = 0x1e96,
  [0x038c] = 0x1e97, [0x038d] = 0x1e98, [0x038e] = 0x1e99, [0x038f] = 0x1e9a,
  [0x0390] = 0x1e9b, [0x0391] = 0x1e9c, [0x0392] = 0x1e9d, [0x0393] = 0x1e9e,
  [0x0394] = 0x1e9f, [0x0395] = 0x1ea0, [0x0396] = 0x1ea1, [0x0397] = 0x1ea2,
  [0x0398] = 0x1ea3, [0x0399] = 0x1ea4, [0x039a] = 0x1ea5, [0x039b] = 0x1ea6,
  [0x039c] = 0x1ea7, [0x039d] = 0x1ea8, [0x039e] = 0x1ea9, [0x039f] = 0x1eaa,
  [0x03a0] = 0x1eab, [0x03a1] = 0x1eac, [0x03a2] = 0x1ead, [0x03a3] = 0x1eae,
  [0x03a4] = 0x1eaf, [0x03a5] = 0x1eb0, [0x03a6] = 0x1eb1, [0x03a7] = 0x1eb2,
  [0x03a8] = 0x1eb3, [0x03a9] = 0x1eb4, [0x03aa] = 0x1eb5, [0x03ab] = 0x1eb6,
  [0x03ac] = 0x1eb7, [0x03ad] = 0x1eb8, [0x03ae] = 0x1eb9, [0x03af] = 0x1eba,
  [0x03b0] = 0x1ebb, [0x03b1] = 0x1ebc, [0x03b2] = 0x1ebd, [0x03b3] = 0x1ebe,
  [0x03b4] = 0x1ebf, [0x03b5] = 0x1ec0, [0x03b6] = 0x1ec1, [0x03b7] = 0x1ec2,
  [0x03b8] = 0x1ec3, [0x03b9] = 0x1ec4, [0x03ba] = 0x1ec5, [0x03bb] = 0x1ec6,
  [0x03bc] = 0x1ec7, [0x03bd] = 0x1ec8, [0x03be] = 0x1ec9, [0x03bf] = 0x1eca,
  [0x03c0] = 0x1ecb, [0x03c1] = 0x1ecc, [0x03c2] = 0x1ecd, [0x03c3] = 0x1ece,
  [0x03c4] = 0x1ecf, [0x03c5] = 0x1ed0, [0x03c6] = 0x1ed1, [0x03c7] = 0x1ed2,
  [0x03c8] = 0x1ed3, [0x03c9] = 0x1ed4, [0x03ca] = 0x1ed5, [0x03cb] = 0x1ed6,
  [0x03cc] = 0x1ed7, [0x03cd] = 0x1ed8, [0x03ce] = 0x1ed9, [0x03cf] = 0x1eda,
  [0x03d0] = 0x1edb, [0x03d1] = 0x1edc, [0x03d2] = 0x1edd, [0x03d3] = 0x1ede,
  [0x03d4] = 0x1edf, [0x03d5] = 0x1ee0, [0x03d6] = 0x1ee1, [0x03d7] = 0x1ee2,
  [0x03d8] = 0x1ee3, [0x03d9] = 0x1ee4, [0x03da] = 0x1ee5, [0x03db] = 0x1ee6,
  [0x03dc] = 0x1ee7, [0x03dd] = 0x1ee8, [0x03de] = 0x1ee9, [0x03df] = 0x1eea,
  [0x03e0] = 0x1eeb, [0x03e1] = 0x1eec, [0x03e2] = 0x1eed, [0x03e3] = 0x1eee,
  [0x03e4] = 0x1eef, [0x03e5] = 0x1ef0, [0x03e6] = 0x1ef1, [0x03e7] = 0x1ef2,
  [0x03e8] = 0x1ef3, [0x03e9] = 0x1ef4, [0x03ea] = 0x1ef5, [0x03eb] = 0x1ef6,
  [0x03ec] = 0x1ef7, [0x03ed] = 0x1ef8, [0x03ee] = 0x1ef9, [0x03ef] = 0x1efa,
  [0x03f0] = 0x1efb, [0x03f1] = 0x1efc, [0x03f2] = 0x1efd, [0x03f3] = 0x1efe,
  [0x03f4] = 0x1eff, [0x03f5] = 0x1f00, [0x03f6] = 0x1f01, [0x03f7] = 0x1f02,
  [0x03f8] = 0x1f03, [0x03f9] = 0x1f04, [0x03fa] = 0x1f05, [0x03fb] = 0x1f06,
  [0x03fc] = 0x1f07, [0x03fd] = 0x1f08, [0x03fe] = 0x1f09, [0x03ff] = 0x1f0a,
  [0x0400] = 0x1f0b, [0x0401] = 0x1f0c, [0x0402] = 0x1f0d, [0x0403] = 0x1f0e,
  [0x0404] = 0x1f0f, [0x0405] = 0x1f10, [0x0406] = 0x1f11, [0x0407] = 0x1f12,
  [0x0408] = 0x1f13, [0x0409] = 0x1f14, [0x040a] = 0x1f15, [0x040b] = 0x1f16,
  [0x040c] = 0x1f17, [0x040d] = 0x1f18, [0x040e] = 0x1f19, [0x040f] = 0x1f1a,
  [0x0410] = 0x1f1b, [0x0411] = 0x1f1c, [0x0412] = 0x1f1d, [0x0413] = 0x1f1e,
  [0x0414] = 0x1f1f, [0x0415] = 0x1f20, [0x0416] = 0x1f21, [0x0417] = 0x1f22,
  [0x0418] = 0x1f23, [0x0419] = 0x1f24, [0x041a] = 0x1f25, [0x041b] = 0x1f26,
  [0x041c] = 0x1f27, [0x041d] = 0x1f28, [0x041e] = 0x1f29, [0x041f] = 0x1f2a,
  [0x0420] = 0x1f2b, [0x0421] = 0x1f2c, [0x0422] = 0x1f2d, [0x0423] = 0x1f2e,
  [0x0424] = 0x1f2f, [0x0425] = 0x1f30, [0x0426] = 0x1f31, [0x0427] = 0x1f32,
  [0x0428] = 0x1f33, [0x0429] = 0x1f34, [0x042a] = 0x1f35, [0x042b] = 0x1f36,
  [0x042c] = 0x1f37, [0x042d] = 0x1f38, [0x042e] = 0x1f39, [0x042f] = 0x1f3a,
  [0x0430] = 0x1f3b, [0x0431] = 0x1f3c, [0x0432] = 0x1f3d, [0x0433] = 0x1f3e,
  [0x0434] = 0x1f3f, [0x0435] = 0x1f40, [0x0436] = 0x1f41, [0x0437] = 0x1f42,
  [0x0438] = 0x1f43, [0x0439] = 0x1f44, [0x043a] = 0x1f45, [0x043b] = 0x1f46,
  [0x043c] = 0x1f47, [0x043d] = 0x1f48, [0x043e] = 0x1f49, [0x043f] = 0x1f4a,
  [0x0440] = 0x1f4b, [0x0441] = 0x1f4c, [0x0442] = 0x1f4d, [0x0443] = 0x1f4e,
  [0x0444] = 0x1f4f, [0x0445] = 0x1f50, [0x0446] = 0x1f51, [0x0447] = 0x1f52,
  [0x0448] = 0x1f53, [0x0449] = 0x1f54, [0x044a] = 0x1f55, [0x044b] = 0x1f56,
  [0x044c] = 0x1f57, [0x044d] = 0x1f58, [0x044e] = 0x1f59, [0x044f] = 0x1f5a,
  [0x0450] = 0x1f5b, [0x0451] = 0x1f5c, [0x0452] = 0x1f5d, [0x0453] = 0x1f5e,
  [0x0454] = 0x1f5f, [0x0455] = 0x1f60, [0x0456] = 0x1f61, [0x0457] = 0x1f62,
  [0x0458] = 0x1f63, [0x0459] = 0x1f64, [0x045a] = 0x1f65, [0x045b] = 0x1f66,
  [0x045c] = 0x1f67, [0x045d] = 0x1f68, [0x045e] = 0x1f69, [0x045f] = 0x1f6a,
  [0x0460] = 0x1f6b, [0x0461] = 0x1f6c, [0x0462] = 0x1f6d, [0x0463] = 0x1f6e,
  [0x0464] = 0x1f6f, [0x0465] = 0x1f70, [0x0466] = 0x1f71, [0x0467] = 0x1f72,
  [0x0468] = 0x1f73, [0x0469] = 0x1f74, [0x046a] = 0x1f75, [0x046b] = 0x1f76,
  [0x046c] = 0x1f77, [0x046d] = 0x1f78, [0x046e] = 0x1f79, [0x046f] = 0x1f7a,
  [0x0470] = 0x1f7b, [0x0471] = 0x1f7c, [0x0472] = 0x1f7d, [0x0473] = 0x1f7e,
  [0x0474] = 0x1f7f, [0x0475] = 0x1f80, [0x0476] = 0x1f81, [0x0477] = 0x1f82,
  [0x0478] = 0x1f83, [0x0479] = 0x1f84, [0x047a] = 0x1f85, [0x047b] = 0x1f86,
  [0x047c] = 0x1f87, [0x047d] = 0x1f88, [0x047e] = 0x1f89, [0x047f] = 0x1f8a,
  [0x0480] = 0x1f8b, [0x0481] = 0x1f8c, [0x0482] = 0x1f8d, [0x0483] = 0x1f8e,
  [0x0484] = 0x1f8f, [0x0485] = 0x1f90, [0x0486] = 0x1f91, [0x0487] = 0x1f92,
  [0x0488] = 0x1f93, [0x0489] = 0x1f94, [0x048a] = 0x1f95, [0x048b] = 0x1f96,
  [0x048c] = 0x1f97, [0x048d] = 0x1f98, [0x048e] = 0x1f99, [0x048f] = 0x1f9a,
  [0x0490] = 0x1f9b, [0x0491] = 0x1f9c, [0x0492] = 0x1f9d, [0x0493] = 0x1f9e,
  [0x0494] = 0x1f9f, [0x0495] = 0x1fa0, [0x0496] = 0x1fa1, [0x0497] = 0x1fa2,
  [0x0498] = 0x1fa3, [0x0499] = 0x1fa4, [0x049a] = 0x1fa5, [0x049b] = 0x1fa6,
  [0x049c] = 0x1fa7, [0x049d] = 0x1fa8, [0x049e] = 0x1fa9, [0x049f] = 0x1faa,
  [0x04a0] = 0x1fab, [0x04a1] = 0x1fac, [0x04a2] = 0x1fad, [0x04a3] = 0x1fae,
  [0x04a4] = 0x1faf, [0x04a5] = 0x1fb0, [0x04a6] = 0x1fb1, [0x04a7] = 0x1fb2,
  [0x04a8] = 0x1fb3, [0x04a9] = 0x1fb4, [0x04aa] = 0x1fb5, [0x04ab] = 0x1fb6,
  [0x04ac] = 0x1fb7, [0x04ad] = 0x1fb8, [0x04ae] = 0x1fb9, [0x04af] = 0x1fba,
  [0x04b0] = 0x1fbb, [0x04b1] = 0x1fbc, [0x04b2] = 0x1fbd, [0x04b3] = 0x1fbe,
  [0x04b4] = 0x1fbf, [0x04b5] = 0x1fc0, [0x04b6] = 0x1fc1, [0x04b7] = 0x1fc2,
  [0x04b8] = 0x1fc3, [0x04b9] = 0x1fc4, [0x04ba] = 0x1fc5, [0x04bb] = 0x1fc6,
  [0x04bc] = 0x1fc7, [0x04bd] = 0x1fc8, [0x04be] = 0x1fc9, [0x04bf] = 0x1fca,
  [0x04c0] = 0x1fcb, [0x04c1] = 0x1fcc, [0x04c2] = 0x1fcd, [0x04c3] = 0x1fce,
  [0x04c4] = 0x1fcf, [0x04c5] = 0x1fd0, [0x04c6] = 0x1fd1, [0x04c7] = 0x1fd2,
  [0x04c8] = 0x1fd3, [0x04c9] = 0x1fd4, [0x04ca] = 0x1fd5, [0x04cb] = 0x1fd6,
  [0x04cc] = 0x1fd7, [0x04cd] = 0x1fd8, [0x04ce] = 0x1fd9, [0x04cf] = 0x1fda,
  [0x04d0] = 0x1fdb, [0x04d1] = 0x1fdc, [0x04d2] = 0x1fdd, [0x04d3] = 0x1fde,
  [0x04d4] = 0x1fdf, [0x04d5] = 0x1fe0, [0x04d6] = 0x1fe1, [0x04d7] = 0x1fe2,
  [0x04d8] = 0x1fe3, [0x04d9] = 0x1fe4, [0x04da] = 0x1fe5, [0x04db] = 0x1fe6,
  [0x04dc] = 0x1fe7, [0x04dd] = 0x1fe8, [0x04de] = 0x1fe9, [0x04df] = 0x1fea,
  [0x04e0] = 0x1feb, [0x04e1] = 0x1fec, [0x04e2] = 0x1fed, [0x04e3] = 0x1fee,
  [0x04e4] = 0x1fef, [0x04e5] = 0x1ff0, [0x04e6] = 0x1ff1, [0x04e7] = 0x1ff2,
  [0x04e8] = 0x1ff3, [0x04e9] = 0x1ff4, [0x04ea] = 0x1ff5, [0x04eb] = 0x1ff6,
  [0x04ec] = 0x1ff7, [0x04ed] = 0x1ff8, [0x04ee] = 0x1ff9, [0x04ef] = 0x1ffa,
  [0x04f0] = 0x1ffb, [0x04f1] = 0x1ffc, [0x04f2] = 0x1ffd, [0x04f3] = 0x1ffe,
  [0x04f4] = 0x1fff, [0x04f5] = 0x2000, [0x04f6] = 0x2001, [0x04f7] = 0x2002,
  [0x04f8] = 0x2003, [0x04f9] = 0x2004, [0x04fa] = 0x2005, [0x04fb] = 0x2006,
  [0x04fc] = 0x2007, [0x04fd] = 0x2008, [0x04fe] = 0x2009, [0x04ff] = 0x200a,
  [0x0500] = 0x200b, [0x0501] = 0x200c, [0x0502] = 0x200d, [0x0503] = 0x200e,
  [0x0504] = 0x200f, [0x0505] = 0x2011, [0x0506] = 0x2012, [0x0507] = 0x2017,
  [0x0508] = 0x201a, [0x0509] = 0x201b, [0x050a] = 0x201e, [0x050b] = 0x201f,
  [0x050c] = 0x2020, [0x050d] = 0x2021, [0x050e] = 0x2022, [0x050f] = 0x2023,
  [0x0510] = 0x2024, [0x0511] = 0x2027, [0x0512] = 0x2028, [0x0513] = 0x2029,
  [0x0514] = 0x202a, [0x0515] = 0x202b, [0x0516] = 0x202c, [0x0517] = 0x202d,
  [0x0518] = 0x202e, [0x0519] = 0x202f, [0x051a] = 0x2031, [0x051b] = 0x2034,
  [0x051c] = 0x2036, [0x051d] = 0x2037, [0x051e] = 0x2038, [0x051f] = 0x2039,
  [0x0520] = 0x203a, [0x0521] = 0x203c, [0x0522] = 0x203d, [0x0523] = 0x203e,
  [0x0524] = 0x203f, [0x0525] = 0x2040, [0x0526] = 0x2041, [0x0527] = 0x2042,
  [0x0528] = 0x2043, [0x0529] = 0x2044, [0x052a] = 0x2045, [0x052b] = 0x2046,
  [0x052c] = 0x2047, [0x052d] = 0x2048, [0x052e] = 0x2049, [0x052f] = 0x204a,
  [0x0530] = 0x204b, [0x0531] = 0x204c, [0x0532] = 0x204d, [0x0533] = 0x204e,
  [0x0534] = 0x204f, [0x0535] = 0x2050, [0x0536] = 0x2051, [0x0537] = 0x2052,
  [0x0538] = 0x2053, [0x0539] = 0x2054, [0x053a] = 0x2055, [0x053b] = 0x2056,
  [0x053c] = 0x2057, [0x053d] = 0x2058, [0x053e] = 0x2059, [0x053f] = 0x205a,
  [0x0540] = 0x205b, [0x0541] = 0x205c, [0x0542] = 0x205d, [0x0543] = 0x205e,
  [0x0544] = 0x205f, [0x0545] = 0x2060, [0x0546] = 0x2061, [0x0547] = 0x2062,
  [0x0548] = 0x2063, [0x0549] = 0x2064, [0x054a] = 0x2065, [0x054b] = 0x2066,
  [0x054c] = 0x2067, [0x054d] = 0x2068, [0x054e] = 0x2069, [0x054f] = 0x206a,
  [0x0550] = 0x206b, [0x0551] = 0x206c, [0x0552] = 0x206d, [0x0553] = 0x206e,
  [0x0554] = 0x206f, [0x0555] = 0x2070, [0x0556] = 0x2071, [0x0557] = 0x2072,
  [0x0558] = 0x2073, [0x0559] = 0x2074, [0x055a] = 0x2075, [0x055b] = 0x2076,
  [0x055c] = 0x2077, [0x055d] = 0x2078, [0x055e] = 0x2079, [0x055f] = 0x207a,
  [0x0560] = 0x207b, [0x0561] = 0x207c, [0x0562] = 0x207d, [0x0563] = 0x207e,
  [0x0564] = 0x207f, [0x0565] = 0x2080, [0x0566] = 0x2081, [0x0567] = 0x2082,
  [0x0568] = 0x2083, [0x0569] = 0x2084, [0x056a] = 0x2085, [0x056b] = 0x2086,
  [0x056c] = 0x2087, [0x056d] = 0x2088, [0x056e] = 0x2089, [0x056f] = 0x208a,
  [0x0570] = 0x208b, [0x0571] = 0x208c, [0x0572] = 0x208d, [0x0573] = 0x208e,
  [0x0574] = 0x208f, [0x0575] = 0x2090, [0x0576] = 0x2091, [0x0577] = 0x2092,
  [0x0578] = 0x2093, [0x0579] = 0x2094, [0x057a] = 0x2095, [0x057b] = 0x2096,
  [0x057c] = 0x2097, [0x057d] = 0x2098, [0x057e] = 0x2099, [0x057f] = 0x209a,
  [0x0580] = 0x209b, [0x0581] = 0x209c, [0x0582] = 0x209d, [0x0583] = 0x209e,
  [0x0584] = 0x209f, [0x0585] = 0x20a0, [0x0586] = 0x20a1, [0x0587] = 0x20a2,
  [0x0588] = 0x20a3, [0x0589] = 0x20a4, [0x058a] = 0x20a5, [0x058b] = 0x20a6,
  [0x058c] = 0x20a7, [0x058d] = 0x20a8, [0x058e] = 0x20a9, [0x058f] = 0x20aa,
  [0x0590] = 0x20ab, [0x0591] = 0x20ad, [0x0592] = 0x20ae, [0x0593] = 0x20af,
  [0x0594] = 0x20b0, [0x0595] = 0x20b1, [0x0596] = 0x20b2, [0x0597] = 0x20b3,
  [0x0598] = 0x20b4, [0x0599] = 0x20b5, [0x059a] = 0x20b6, [0x059b] = 0x20b7,
  [0x059c] = 0x20b8, [0x059d] = 0x20b9, [0x059e] = 0x20ba, [0x059f] = 0x20bb,
  [0x05a0] = 0x20bc, [0x05a1] = 0x20bd, [0x05a2] = 0x20be, [0x05a3] = 0x20bf,
  [0x05a4] = 0x20c0, [0x05a5] = 0x20c1, [0x05a6] = 0x20c2, [0x05a7] = 0x20c3,
  [0x05a8] = 0x20c4, [0x05a9] = 0x20c5, [0x05aa] = 0x20c6, [0x05ab] = 0x20c7,
  [0x05ac] = 0x20c8, [0x05ad] = 0x20c9, [0x05ae] = 0x20ca, [0x05af] = 0x20cb,
  [0x05b0] = 0x20cc, [0x05b1] = 0x20cd, [0x05b2] = 0x20ce, [0x05b3] = 0x20cf,
  [0x05b4] = 0x20d0, [0x05b5] = 0x20d1, [0x05b6] = 0x20d2, [0x05b7] = 0x20d3,
  [0x05b8] = 0x20d4, [0x05b9] = 0x20d5, [0x05ba] = 0x20d6, [0x05bb] = 0x20d7,
  [0x05bc] = 0x20d8, [0x05bd] = 0x20d9, [0x05be] = 0x20da, [0x05bf] = 0x20db,
  [0x05c0] = 0x20dc, [0x05c1] = 0x20dd, [0x05c2] = 0x20de, [0x05c3] = 0x20df,
  [0x05c4] = 0x20e0, [0x05c5] = 0x20e1, [0x05c6] = 0x20e2, [0x05c7] = 0x20e3,
  [0x05c8] = 0x20e4, [0x05c9] = 0x20e5, [0x05ca] = 0x20e6, [0x05cb] = 0x20e7,
  [0x05cc] = 0x20e8, [0x05cd] = 0x20e9, [0x05ce] = 0x20ea, [0x05cf] = 0x20eb,
  [0x05d0] = 0x20ec, [0x05d1] = 0x20ed, [0x05d2] = 0x20ee, [0x05d3] = 0x20ef,
  [0x05d4] = 0x20f0, [0x05d5] = 0x20f1, [0x05d6] = 0x20f2, [0x05d7] = 0x20f3,
  [0x05d8] = 0x20f4, [0x05d9] = 0x20f5, [0x05da] = 0x20f6, [0x05db] = 0x20f7,
  [0x05dc] = 0x20f8, [0x05dd] = 0x20f9, [0x05de] = 0x20fa, [0x05df] = 0x20fb,
  [0x05e0] = 0x20fc, [0x05e1] = 0x20fd, [0x05e2] = 0x20fe, [0x05e3] = 0x20ff,
  [0x05e4] = 0x2100, [0x05e5] = 0x2101, [0x05e6] = 0x2102, [0x05e7] = 0x2104,
  [0x05e8] = 0x2106, [0x05e9] = 0x2107, [0x05ea] = 0x2108, [0x05eb] = 0x210a,
  [0x05ec] = 0x210b, [0x05ed] = 0x210c, [0x05ee] = 0x210d, [0x05ef] = 0x210e,
  [0x05f0] = 0x210f, [0x05f1] = 0x2110, [0x05f2] = 0x2111, [0x05f3] = 0x2112,
  [0x05f4] = 0x2113, [0x05f5] = 0x2114, [0x05f6] = 0x2115, [0x05f7] = 0x2117,
  [0x05f8] = 0x2118, [0x05f9] = 0x2119, [0x05fa] = 0x211a, [0x05fb] = 0x211b,
  [0x05fc] = 0x211c, [0x05fd] = 0x211d, [0x05fe] = 0x211e, [0x05ff] = 0x211f,
  [0x0600] = 0x2120, [0x0601] = 0x2122, [0x0602] = 0x2123, [0x0603] = 0x2124,
  [0x0604] = 0x2125, [0x0605] = 0x2126, [0x0606] = 0x2127, [0x0607] = 0x2128,
  [0x0608] = 0x2129, [0x0609] = 0x212a, [0x060a] = 0x212b, [0x060b] = 0x212c,
  [0x060c] = 0x212d, [0x060d] = 0x212e, [0x060e] = 0x212f, [0x060f] = 0x2130,
  [0x0610] = 0x2131, [0x0611] = 0x2132, [0x0612] = 0x2133, [0x0613] = 0x2134,
  [0x0614] = 0x2135, [0x0615] = 0x2136, [0x0616] = 0x2137, [0x0617] = 0x2138,
  [0x0618] = 0x2139, [0x0619] = 0x213a, [0x061a] = 0x213b, [0x061b] = 0x213c,
  [0x061c] = 0x213d, [0x061d] = 0x213e, [0x061e] = 0x213f, [0x061f] = 0x2140,
  [0x0620] = 0x2141, [0x0621] = 0x2142, [0x0622] = 0x2143, [0x0623] = 0x2144,
  [0x0624] = 0x2145, [0x0625] = 0x2146, [0x0626] = 0x2147, [0x0627] = 0x2148,
  [0x0628] = 0x2149, [0x0629] = 0x214a, [0x062a] = 0x214b, [0x062b] = 0x214c,
  [0x062c] = 0x214d, [0x062d] = 0x214e, [0x062e] = 0x214f, [0x062f] = 0x2150,
  [0x0630] = 0x2151, [0x0631] = 0x2152, [0x0632] = 0x2153, [0x0633] = 0x2154,
  [0x0634] = 0x2155, [0x0635] = 0x2156, [0x0636] = 0x2157, [0x0637] = 0x2158,
  [0x0638] = 0x2159, [0x0639] = 0x215a, [0x063a] = 0x215b, [0x063b] = 0x215c,
  [0x063c] = 0x215d, [0x063d] = 0x215e, [0x063e] = 0x215f, [0x063f] = 0x216c,
  [0x0640] = 0x216d, [0x0641] = 0x216e, [0x0642] = 0x216f, [0x0643] = 0x217a,
  [0x0644] = 0x217b, [0x0645] = 0x217c, [0x0646] = 0x217d, [0x0647] = 0x217e,
  [0x0648] = 0x217f, [0x0649] = 0x2180, [0x064a] = 0x2181, [0x064b] = 0x2182,
  [0x064c] = 0x2183, [0x064d] = 0x2184, [0x064e] = 0x2185, [0x064f] = 0x2186,
  [0x0650] = 0x2187, [0x0651] = 0x2188, [0x0652] = 0x2189, [0x0653] = 0x218a,
  [0x0654] = 0x218b, [0x0655] = 0x218c, [0x0656] = 0x218d, [0x0657] = 0x218e,
  [0x0658] = 0x218f, [0x0659] = 0x2194, [0x065a] = 0x2195, [0x065b] = 0x219a,
  [0x065c] = 0x219b, [0x065d] = 0x219c, [0x065e] = 0x219d, [0x065f] = 0x219e,
  [0x0660] = 0x219f, [0x0661] = 0x21a0, [0x0662] = 0x21a1, [0x0663] = 0x21a2,
  [0x0664] = 0x21a3, [0x0665] = 0x21a4, [0x0666] = 0x21a5, [0x0667] = 0x21a6,
  [0x0668] = 0x21a7, [0x0669] = 0x21a8, [0x066a] = 0x21a9, [0x066b] = 0x21aa,
  [0x066c] = 0x21ab, [0x066d] = 0x21ac, [0x066e] = 0x21ad, [0x066f] = 0x21ae,
  [0x0670] = 0x21af, [0x0671] = 0x21b0, [0x0672] = 0x21b1, [0x0673] = 0x21b2,
  [0x0674] = 0x21b3, [0x0675] = 0x21b4, [0x0676] = 0x21b5, [0x0677] = 0x21b6,
  [0x0678] = 0x21b7, [0x0679] = 0x21b8, [0x067a] = 0x21b9, [0x067b] = 0x21ba,
  [0x067c] = 0x21bb, [0x067d] = 0x21bc, [0x067e] = 0x21bd, [0x067f] = 0x21be,
  [0x0680] = 0x21bf, [0x0681] = 0x21c0, [0x0682] = 0x21c1, [0x0683] = 0x21c2,
  [0x0684] = 0x21c3, [0x0685] = 0x21c4, [0x0686] = 0x21c5, [0x0687] = 0x21c6,
  [0x0688] = 0x21c7, [0x0689] = 0x21c8, [0x068a] = 0x21c9, [0x068b] = 0x21ca,
  [0x068c] = 0x21cb, [0x068d] = 0x21cc, [0x068e] = 0x21cd, [0x068f] = 0x21ce,
  [0x0690] = 0x21cf, [0x0691] = 0x21d0, [0x0692] = 0x21d1, [0x0693] = 0x21d2,
  [0x0694] = 0x21d3, [0x0695] = 0x21d4, [0x0696] = 0x21d5, [0x0697] = 0x21d6,
  [0x0698] = 0x21d7, [0x0699] = 0x21d8, [0x069a] = 0x21d9, [0x069b] = 0x21da,
  [0x069c] = 0x21db, [0x069d] = 0x21dc, [0x069e] = 0x21dd, [0x069f] = 0x21de,
  [0x06a0] = 0x21df, [0x06a1] = 0x21e0, [0x06a2] = 0x21e1, [0x06a3] = 0x21e2,
  [0x06a4] = 0x21e3, [0x06a5] = 0x21e4, [0x06a6] = 0x21e5, [0x06a7] = 0x21e6,
  [0x06a8] = 0x21e7, [0x06a9] = 0x21e8, [0x06aa] = 0x21e9, [0x06ab] = 0x21ea,
  [0x06ac] = 0x21eb, [0x06ad] = 0x21ec, [0x06ae] = 0x21ed, [0x06af] = 0x21ee,
  [0x06b0] = 0x21ef, [0x06b1] = 0x21f0, [0x06b2] = 0x21f1, [0x06b3] = 0x21f2,
  [0x06b4] = 0x21f3, [0x06b5] = 0x21f4, [0x06b6] = 0x21f5, [0x06b7] = 0x21f6,
  [0x06b8] = 0x21f7, [0x06b9] = 0x21f8, [0x06ba] = 0x21f9, [0x06bb] = 0x21fa,
  [0x06bc] = 0x21fb, [0x06bd] = 0x21fc, [0x06be] = 0x21fd, [0x06bf] = 0x21fe,
  [0x06c0] = 0x21ff, [0x06c1] = 0x2200, [0x06c2] = 0x2201, [0x06c3] = 0x2202,
  [0x06c4] = 0x2203, [0x06c5] = 0x2204, [0x06c6] = 0x2205, [0x06c7] = 0x2206,
  [0x06c8] = 0x2207, [0x06c9] = 0x2209, [0x06ca] = 0x220a, [0x06cb] = 0x220b,
  [0x06cc] = 0x220c, [0x06cd] = 0x220d, [0x06ce] = 0x220e, [0x06cf] = 0x2210,
  [0x06d0] = 0x2212, [0x06d1] = 0x2213, [0x06d2] = 0x2214, [0x06d3] = 0x2216,
  [0x06d4] = 0x2217, [0x06d5] = 0x2218, [0x06d6] = 0x2219, [0x06d7] = 0x221b,
  [0x06d8] = 0x221c, [0x06d9] = 0x2221, [0x06da] = 0x2222, [0x06db] = 0x2224,
  [0x06dc] = 0x2226, [0x06dd] = 0x222c, [0x06de] = 0x222d, [0x06df] = 0x222f,
  [0x06e0] = 0x2230, [0x06e1] = 0x2231, [0x06e2] = 0x2232, [0x06e3] = 0x2233,
  [0x06e4] = 0x2238, [0x06e5] = 0x2239, [0x06e6] = 0x223a, [0x06e7] = 0x223b,
  [0x06e8] = 0x223c, [0x06e9] = 0x223e, [0x06ea] = 0x223f, [0x06eb] = 0x2240,
  [0x06ec] = 0x2241, [0x06ed] = 0x2242, [0x06ee] = 0x2243, [0x06ef] = 0x2244,
  [0x06f0] = 0x2245, [0x06f1] = 0x2246, [0x06f2] = 0x2247, [0x06f3] = 0x2249,
  [0x06f4] = 0x224a, [0x06f5] = 0x224b, [0x06f6] = 0x224d, [0x06f7] = 0x224e,
  [0x06f8] = 0x224f, [0x06f9] = 0x2250, [0x06fa] = 0x2251, [0x06fb] = 0x2253,
  [0x06fc] = 0x2254, [0x06fd] = 0x2255, [0x06fe] = 0x2256, [0x06ff] = 0x2257,
  [0x0700] = 0x2258, [0x0701] = 0x2259, [0x0702] = 0x225a, [0x0703] = 0x225b,
  [0x0704] = 0x225c, [0x0705] = 0x225d, [0x0706] = 0x225e, [0x0707] = 0x225f,
  [0x0708] = 0x2262, [0x0709] = 0x2263, [0x070a] = 0x2268, [0x070b] = 0x2269,
  [0x070c] = 0x226a, [0x070d] = 0x226b, [0x070e] = 0x226c, [0x070f] = 0x226d,
  [0x0710] = 0x2270, [0x0711] = 0x2271, [0x0712] = 0x2272, [0x0713] = 0x2273,
  [0x0714] = 0x2274, [0x0715] = 0x2275, [0x0716] = 0x2276, [0x0717] = 0x2277,
  [0x0718] = 0x2278, [0x0719] = 0x2279, [0x071a] = 0x227a, [0x071b] = 0x227b,
  [0x071c] = 0x227c, [0x071d] = 0x227d, [0x071e] = 0x227e, [0x071f] = 0x227f,
  [0x0720] = 0x2280, [0x0721] = 0x2281, [0x0722] = 0x2282, [0x0723] = 0x2283,
  [0x0724] = 0x2284, [0x0725] = 0x2285, [0x0726] = 0x2286, [0x0727] = 0x2287,
  [0x0728] = 0x2288, [0x0729] = 0x2289, [0x072a] = 0x228a, [0x072b] = 0x228b,
  [0x072c] = 0x228c, [0x072d] = 0x228d, [0x072e] = 0x228e, [0x072f] = 0x228f,
  [0x0730] = 0x2290, [0x0731] = 0x2291, [0x0732] = 0x2292, [0x0733] = 0x2293,
  [0x0734] = 0x2294, [0x0735] = 0x2296, [0x0736] = 0x2297, [0x0737] = 0x2298,
  [0x0738] = 0x229a, [0x0739] = 0x229b, [0x073a] = 0x229c, [0x073b] = 0x229d,
  [0x073c] = 0x229e, [0x073d] = 0x229f, [0x073e] = 0x22a0, [0x073f] = 0x22a1,
  [0x0740] = 0x22a2, [0x0741] = 0x22a3, [0x0742] = 0x22a4, [0x0743] = 0x22a6,
  [0x0744] = 0x22a7, [0x0745] = 0x22a8, [0x0746] = 0x22a9, [0x0747] = 0x22aa,
  [0x0748] = 0x22ab, [0x0749] = 0x22ac, [0x074a] = 0x22ad, [0x074b] = 0x22ae,
  [0x074c] = 0x22af, [0x074d] = 0x22b0, [0x074e] = 0x22b1, [0x074f] = 0x22b2,
  [0x0750] = 0x22b3, [0x0751] = 0x22b4, [0x0752] = 0x22b5, [0x0753] = 0x22b6,
  [0x0754] = 0x22b7, [0x0755] = 0x22b8, [0x0756] = 0x22b9, [0x0757] = 0x22ba,
  [0x0758] = 0x22bb, [0x0759] = 0x22bc, [0x075a] = 0x22bd, [0x075b] = 0x22be,
  [0x075c] = 0x22c0, [0x075d] = 0x22c1, [0x075e] = 0x22c2, [0x075f] = 0x22c3,
  [0x0760] = 0x22c4, [0x0761] = 0x22c5, [0x0762] = 0x22c6, [0x0763] = 0x22c7,
  [0x0764] = 0x22c8, [0x0765] = 0x22c9, [0x0766] = 0x22ca, [0x0767] = 0x22cb,
  [0x0768] = 0x22cc, [0x0769] = 0x22cd, [0x076a] = 0x22ce, [0x076b] = 0x22cf,
  [0x076c] = 0x22d0, [0x076d] = 0x22d1, [0x076e] = 0x22d2, [0x076f] = 0x22d3,
  [0x0770] = 0x22d4, [0x0771] = 0x22d5, [0x0772] = 0x22d6, [0x0773] = 0x22d7,
  [0x0774] = 0x22d8, [0x0775] = 0x22d9, [0x0776] = 0x22da, [0x0777] = 0x22db,
  [0x0778] = 0x22dc, [0x0779] = 0x22dd, [0x077a] = 0x22de, [0x077b] = 0x22df,
  [0x077c] = 0x22e0, [0x077d] = 0x22e1, [0x077e] = 0x22e2, [0x077f] = 0x22e3,
  [0x0780] = 0x22e4, [0x0781] = 0x22e5, [0x0782] = 0x22e6, [0x0783] = 0x22e7,
  [0x0784] = 0x22e8, [0x0785] = 0x22e9, [0x0786] = 0x22ea, [0x0787] = 0x22eb,
  [0x0788] = 0x22ec, [0x0789] = 0x22ed, [0x078a] = 0x22ee, [0x078b] = 0x22ef,
  [0x078c] = 0x22f0, [0x078d] = 0x22f1, [0x078e] = 0x22f2, [0x078f] = 0x22f3,
  [0x0790] = 0x22f4, [0x0791] = 0x22f5, [0x0792] = 0x22f6, [0x0793] = 0x22f7,
  [0x0794] = 0x22f8, [0x0795] = 0x22f9, [0x0796] = 0x22fa, [0x0797] = 0x22fb,
  [0x0798] = 0x22fc, [0x0799] = 0x22fd, [0x079a] = 0x22fe, [0x079b] = 0x22ff,
  [0x079c] = 0x2300, [0x079d] = 0x2301, [0x079e] = 0x2302, [0x079f] = 0x2303,
  [0x07a0] = 0x2304, [0x07a1] = 0x2305, [0x07a2] = 0x2306, [0x07a3] = 0x2307,
  [0x07a4] = 0x2308, [0x07a5] = 0x2309, [0x07a6] = 0x230a, [0x07a7] = 0x230b,
  [0x07a8] = 0x230c, [0x07a9] = 0x230d, [0x07aa] = 0x230e, [0x07ab] = 0x230f,
  [0x07ac] = 0x2310, [0x07ad] = 0x2311, [0x07ae] = 0x2313, [0x07af] = 0x2314,
  [0x07b0] = 0x2315, [0x07b1] = 0x2316, [0x07b2] = 0x2317, [0x07b3] = 0x2318,
  [0x07b4] = 0x2319, [0x07b5] = 0x231a, [0x07b6] = 0x231b, [0x07b7] = 0x231c,
  [0x07b8] = 0x231d, [0x07b9] = 0x231e, [0x07ba] = 0x231f, [0x07bb] = 0x2320,
  [0x07bc] = 0x2321, [0x07bd] = 0x2322, [0x07be] = 0x2323, [0x07bf] = 0x2324,
  [0x07c0] = 0x2325, [0x07c1] = 0x2326, [0x07c2] = 0x2327, [0x07c3] = 0x2328,
  [0x07c4] = 0x2329, [0x07c5] = 0x232a, [0x07c6] = 0x232b, [0x07c7] = 0x232c,
  [0x07c8] = 0x232d, [0x07c9] = 0x232e, [0x07ca] = 0x232f, [0x07cb] = 0x2330,
  [0x07cc] = 0x2331, [0x07cd] = 0x2332, [0x07ce] = 0x2333, [0x07cf] = 0x2334,
  [0x07d0] = 0x2335, [0x07d1] = 0x2336, [0x07d2] = 0x2337, [0x07d3] = 0x2338,
  [0x07d4] = 0x2339, [0x07d5] = 0x233a, [0x07d6] = 0x233b, [0x07d7] = 0x233c,
  [0x07d8] = 0x233d, [0x07d9] = 0x233e, [0x07da] = 0x233f, [0x07db] = 0x2340,
  [0x07dc] = 0x2341, [0x07dd] = 0x2342, [0x07de] = 0x2343, [0x07df] = 0x2344,
  [0x07e0] = 0x2345, [0x07e1] = 0x2346, [0x07e2] = 0x2347, [0x07e3] = 0x2348,
  [0x07e4] = 0x2349, [0x07e5] = 0x234a, [0x07e6] = 0x234b, [0x07e7] = 0x234c,
  [0x07e8] = 0x234d, [0x07e9] = 0x234e, [0x07ea] = 0x234f, [0x07eb] = 0x2350,
  [0x07ec] = 0x2351, [0x07ed] = 0x2352, [0x07ee] = 0x2353, [0x07ef] = 0x2354,
  [0x07f0] = 0x2355, [0x07f1] = 0x2356, [0x07f2] = 0x2357, [0x07f3] = 0x2358,
  [0x07f4] = 0x2359, [0x07f5] = 0x235a, [0x07f6] = 0x235b, [0x07f7] = 0x235c,
  [0x07f8] = 0x235d, [0x07f9] = 0x235e, [0x07fa] = 0x235f, [0x07fb] = 0x2360,
  [0x07fc] = 0x2361, [0x07fd] = 0x2362, [0x07fe] = 0x2363, [0x07ff] = 0x2364,
  [0x0800] = 0x2365, [0x0801] = 0x2366, [0x0802] = 0x2367, [0x0803] = 0x2368,
  [0x0804] = 0x2369, [0x0805] = 0x236a, [0x0806] = 0x236b, [0x0807] = 0x236c,
  [0x0808] = 0x236d, [0x0809] = 0x236e, [0x080a] = 0x236f, [0x080b] = 0x2370,
  [0x080c] = 0x2371, [0x080d] = 0x2372, [0x080e] = 0x2373, [0x080f] = 0x2374,
  [0x0810] = 0x2375, [0x0811] = 0x2376, [0x0812] = 0x2377, [0x0813] = 0x2378,
  [0x0814] = 0x2379, [0x0815] = 0x237a, [0x0816] = 0x237b, [0x0817] = 0x237c,
  [0x0818] = 0x237d, [0x0819] = 0x237e, [0x081a] = 0x237f, [0x081b] = 0x2380,
  [0x081c] = 0x2381, [0x081d] = 0x2382, [0x081e] = 0x2383, [0x081f] = 0x2384,
  [0x0820] = 0x2385, [0x0821] = 0x2386, [0x0822] = 0x2387, [0x0823] = 0x2388,
  [0x0824] = 0x2389, [0x0825] = 0x238a, [0x0826] = 0x238b, [0x0827] = 0x238c,
  [0x0828] = 0x238d, [0x0829] = 0x238e, [0x082a] = 0x238f, [0x082b] = 0x2390,
  [0x082c] = 0x2391, [0x082d] = 0x2392, [0x082e] = 0x2393, [0x082f] = 0x2394,
  [0x0830] = 0x2395, [0x0831] = 0x2396, [0x0832] = 0x2397, [0x0833] = 0x2398,
  [0x0834] = 0x2399, [0x0835] = 0x239a, [0x0836] = 0x239b, [0x0837] = 0x239c,
  [0x0838] = 0x239d, [0x0839] = 0x239e, [0x083a] = 0x239f, [0x083b] = 0x23a0,
  [0x083c] = 0x23a1, [0x083d] = 0x23a2, [0x083e] = 0x23a3, [0x083f] = 0x23a4,
  [0x0840] = 0x23a5, [0x0841] = 0x23a6, [0x0842] = 0x23a7, [0x0843] = 0x23a8,
  [0x0844] = 0x23a9, [0x0845] = 0x23aa, [0x0846] = 0x23ab, [0x0847] = 0x23ac,
  [0x0848] = 0x23ad, [0x0849] = 0x23ae, [0x084a] = 0x23af, [0x084b] = 0x23b0,
  [0x084c] = 0x23b1, [0x084d] = 0x23b2, [0x084e] = 0x23b3, [0x084f] = 0x23b4,
  [0x0850] = 0x23b5, [0x0851] = 0x23b6, [0x0852] = 0x23b7, [0x0853] = 0x23b8,
  [0x0854] = 0x23b9, [0x0855] = 0x23ba, [0x0856] = 0x23bb, [0x0857] = 0x23bc,
  [0x0858] = 0x23bd, [0x0859] = 0x23be, [0x085a] = 0x23bf, [0x085b] = 0x23c0,
  [0x085c] = 0x23c1, [0x085d] = 0x23c2, [0x085e] = 0x23c3, [0x085f] = 0x23c4,
  [0x0860] = 0x23c5, [0x0861] = 0x23c6, [0x0862] = 0x23c7, [0x0863] = 0x23c8,
  [0x0864] = 0x23c9, [0x0865] = 0x23ca, [0x0866] = 0x23cb, [0x0867] = 0x23cc,
  [0x0868] = 0x23cd, [0x0869] = 0x23ce, [0x086a] = 0x23cf, [0x086b] = 0x23d0,
  [0x086c] = 0x23d1, [0x086d] = 0x23d2, [0x086e] = 0x23d3, [0x086f] = 0x23d4,
  [0x0870] = 0x23d5, [0x0871] = 0x23d6, [0x0872] = 0x23d7, [0x0873] = 0x23d8,
  [0x0874] = 0x23d9, [0x0875] = 0x23da, [0x0876] = 0x23db, [0x0877] = 0x23dc,
  [0x0878] = 0x23dd, [0x0879] = 0x23de, [0x087a] = 0x23df, [0x087b] = 0x23e0,
  [0x087c] = 0x23e1, [0x087d] = 0x23e2, [0x087e] = 0x23e3, [0x087f] = 0x23e4,
  [0x0880] = 0x23e5, [0x0881] = 0x23e6, [0x0882] = 0x23e7, [0x0883] = 0x23e8,
  [0x0884] = 0x23e9, [0x0885] = 0x23ea, [0x0886] = 0x23eb, [0x0887] = 0x23ec,
  [0x0888] = 0x23ed, [0x0889] = 0x23ee, [0x088a] = 0x23ef, [0x088b] = 0x23f0,
  [0x088c] = 0x23f1, [0x088d] = 0x23f2, [0x088e] = 0x23f3, [0x088f] = 0x23f4,
  [0x0890] = 0x23f5, [0x0891] = 0x23f6, [0x0892] = 0x23f7, [0x0893] = 0x23f8,
  [0x0894] = 0x23f9, [0x0895] = 0x23fa, [0x0896] = 0x23fb, [0x0897] = 0x23fc,
  [0x0898] = 0x23fd, [0x0899] = 0x23fe, [0x089a] = 0x23ff, [0x089b] = 0x2400,
  [0x089c] = 0x2401, [0x089d] = 0x2402, [0x089e] = 0x2403, [0x089f] = 0x2404,
  [0x08a0] = 0x2405, [0x08a1] = 0x2406, [0x08a2] = 0x2407, [0x08a3] = 0x2408,
  [0x08a4] = 0x2409, [0x08a5] = 0x240a, [0x08a6] = 0x240b, [0x08a7] = 0x240c,
  [0x08a8] = 0x240d, [0x08a9] = 0x240e, [0x08aa] = 0x240f, [0x08ab] = 0x2410,
  [0x08ac] = 0x2411, [0x08ad] = 0x2412, [0x08ae] = 0x2413, [0x08af] = 0x2414,
  [0x08b0] = 0x2415, [0x08b1] = 0x2416, [0x08b2] = 0x2417, [0x08b3] = 0x2418,
  [0x08b4] = 0x2419, [0x08b5] = 0x241a, [0x08b6] = 0x241b, [0x08b7] = 0x241c,
  [0x08b8] = 0x241d, [0x08b9] = 0x241e, [0x08ba] = 0x241f, [0x08bb] = 0x2420,
  [0x08bc] = 0x2421, [0x08bd] = 0x2422, [0x08be] = 0x2423, [0x08bf] = 0x2424,
  [0x08c0] = 0x2425, [0x08c1] = 0x2426, [0x08c2] = 0x2427, [0x08c3] = 0x2428,
  [0x08c4] = 0x2429, [0x08c5] = 0x242a, [0x08c6] = 0x242b, [0x08c7] = 0x242c,
  [0x08c8] = 0x242d, [0x08c9] = 0x242e, [0x08ca] = 0x242f, [0x08cb] = 0x2430,
  [0x08cc] = 0x2431, [0x08cd] = 0x2432, [0x08ce] = 0x2433, [0x08cf] = 0x2434,
  [0x08d0] = 0x2435, [0x08d1] = 0x2436, [0x08d2] = 0x2437, [0x08d3] = 0x2438,
  [0x08d4] = 0x2439, [0x08d5] = 0x243a, [0x08d6] = 0x243b, [0x08d7] = 0x243c,
  [0x08d8] = 0x243d, [0x08d9] = 0x243e, [0x08da] = 0x243f, [0x08db] = 0x2440,
  [0x08dc] = 0x2441, [0x08dd] = 0x2442, [0x08de] = 0x2443, [0x08df] = 0x2444,
  [0x08e0] = 0x2445, [0x08e1] = 0x2446, [0x08e2] = 0x2447, [0x08e3] = 0x2448,
  [0x08e4] = 0x2449, [0x08e5] = 0x244a, [0x08e6] = 0x244b, [0x08e7] = 0x244c,
  [0x08e8] = 0x244d, [0x08e9] = 0x244e, [0x08ea] = 0x244f, [0x08eb] = 0x2450,
  [0x08ec] = 0x2451, [0x08ed] = 0x2452, [0x08ee] = 0x2453, [0x08ef] = 0x2454,
  [0x08f0] = 0x2455, [0x08f1] = 0x2456, [0x08f2] = 0x2457, [0x08f3] = 0x2458,
  [0x08f4] = 0x2459, [0x08f5] = 0x245a, [0x08f6] = 0x245b, [0x08f7] = 0x245c,
  [0x08f8] = 0x245d, [0x08f9] = 0x245e, [0x08fa] = 0x245f, [0x08fb] = 0x246a,
  [0x08fc] = 0x246b, [0x08fd] = 0x246c, [0x08fe] = 0x246d, [0x08ff] = 0x246e,
  [0x0900] = 0x246f, [0x0901] = 0x2470, [0x0902] = 0x2471, [0x0903] = 0x2472,
  [0x0904] = 0x2473, [0x0905] = 0x249c, [0x0906] = 0x249d, [0x0907] = 0x249e,
  [0x0908] = 0x249f, [0x0909] = 0x24a0, [0x090a] = 0x24a1, [0x090b] = 0x24a2,
  [0x090c] = 0x24a3, [0x090d] = 0x24a4, [0x090e] = 0x24a5, [0x090f] = 0x24a6,
  [0x0910] = 0x24a7, [0x0911] = 0x24a8, [0x0912] = 0x24a9, [0x0913] = 0x24aa,
  [0x0914] = 0x24ab, [0x0915] = 0x24ac, [0x0916] = 0x24ad, [0x0917] = 0x24ae,
  [0x0918] = 0x24af, [0x0919] = 0x24b0, [0x091a] = 0x24b1, [0x091b] = 0x24b2,
  [0x091c] = 0x24b3, [0x091d] = 0x24b4, [0x091e] = 0x24b5, [0x091f] = 0x24b6,
  [0x0920] = 0x24b7, [0x0921] = 0x24b8, [0x0922] = 0x24b9, [0x0923] = 0x24ba,
  [0x0924] = 0x24bb, [0x0925] = 0x24bc, [0x0926] = 0x24bd, [0x0927] = 0x24be,
  [0x0928] = 0x24bf, [0x0929] = 0x24c0, [0x092a] = 0x24c1, [0x092b] = 0x24c2,
  [0x092c] = 0x24c3, [0x092d] = 0x24c4, [0x092e] = 0x24c5, [0x092f] = 0x24c6,
  [0x0930] = 0x24c7, [0x0931] = 0x24c8, [0x0932] = 0x24c9, [0x0933] = 0x24ca,
  [0x0934] = 0x24cb, [0x0935] = 0x24cc, [0x0936] = 0x24cd, [0x0937] = 0x24ce,
  [0x0938] = 0x24cf, [0x0939] = 0x24d0, [0x093a] = 0x24d1, [0x093b] = 0x24d2,
  [0x093c] = 0x24d3, [0x093d] = 0x24d4, [0x093e] = 0x24d5, [0x093f] = 0x24d6,
  [0x0940] = 0x24d7, [0x0941] = 0x24d8, [0x0942] = 0x24d9, [0x0943] = 0x24da,
  [0x0944] = 0x24db, [0x0945] = 0x24dc, [0x0946] = 0x24dd, [0x0947] = 0x24de,
  [0x0948] = 0x24df, [0x0949] = 0x24e0, [0x094a] = 0x24e1, [0x094b] = 0x24e2,
  [0x094c] = 0x24e3, [0x094d] = 0x24e4, [0x094e] = 0x24e5, [0x094f] = 0x24e6,
  [0x0950] = 0x24e7, [0x0951] = 0x24e8, [0x0952] = 0x24e9, [0x0953] = 0x24ea,
  [0x0954] = 0x24eb, [0x0955] = 0x24ec, [0x0956] = 0x24ed, [0x0957] = 0x24ee,
  [0x0958] = 0x24ef, [0x0959] = 0x24f0, [0x095a] = 0x24f1, [0x095b] = 0x24f2,
  [0x095c] = 0x24f3, [0x095d] = 0x24f4, [0x095e] = 0x24f5, [0x095f] = 0x24f6,
  [0x0960] = 0x24f7, [0x0961] = 0x24f8, [0x0962] = 0x24f9, [0x0963] = 0x24fa,
  [0x0964] = 0x24fb, [0x0965] = 0x24fc, [0x0966] = 0x24fd, [0x0967] = 0x24fe,
  [0x0968] = 0x24ff, [0x0969] = 0x254c, [0x096a] = 0x254d, [0x096b] = 0x254e,
  [0x096c] = 0x254f, [0x096d] = 0x2574, [0x096e] = 0x2575, [0x096f] = 0x2576,
  [0x0970] = 0x2577, [0x0971] = 0x2578, [0x0972] = 0x2579, [0x0973] = 0x257a,
  [0x0974] = 0x257b, [0x0975] = 0x257c, [0x0976] = 0x257d, [0x0977] = 0x257e,
  [0x0978] = 0x257f, [0x0979] = 0x2580, [0x097a] = 0x2590, [0x097b] = 0x2591,
  [0x097c] = 0x2592, [0x097d] = 0x2596, [0x097e] = 0x2597, [0x097f] = 0x2598,
  [0x0980] = 0x2599, [0x0981] = 0x259a, [0x0982] = 0x259b, [0x0983] = 0x259c,
  [0x0984] = 0x259d, [0x0985] = 0x259e, [0x0986] = 0x259f, [0x0987] = 0x25a2,
  [0x0988] = 0x25a3, [0x0989] = 0x25a4, [0x098a] = 0x25a5, [0x098b] = 0x25a6,
  [0x098c] = 0x25a7, [0x098d] = 0x25a8, [0x098e] = 0x25a9, [0x098f] = 0x25aa,
  [0x0990] = 0x25ab, [0x0991] = 0x25ac, [0x0992] = 0x25ad, [0x0993] = 0x25ae,
  [0x0994] = 0x25af, [0x0995] = 0x25b0, [0x0996] = 0x25b1, [0x0997] = 0x25b4,
  [0x0998] = 0x25b5, [0x0999] = 0x25b6, [0x099a] = 0x25b7, [0x099b] = 0x25b8,
  [0x099c] = 0x25b9, [0x099d] = 0x25ba, [0x099e] = 0x25bb, [0x099f] = 0x25be,
  [0x09a0] = 0x25bf, [0x09a1] = 0x25c0, [0x09a2] = 0x25c1, [0x09a3] = 0x25c2,
  [0x09a4] = 0x25c3, [0x09a5] = 0x25c4, [0x09a6] = 0x25c5, [0x09a7] = 0x25c8,
  [0x09a8] = 0x25c9, [0x09a9] = 0x25ca, [0x09aa] = 0x25cc, [0x09ab] = 0x25cd,
  [0x09ac] = 0x25d0, [0x09ad] = 0x25d1, [0x09ae] = 0x25d2, [0x09af] = 0x25d3,
  [0x09b0] = 0x25d4, [0x09b1] = 0x25d5, [0x09b2] = 0x25d6, [0x09b3] = 0x25d7,
  [0x09b4] = 0x25d8, [0x09b5] = 0x25d9, [0x09b6] = 0x25da, [0x09b7] = 0x25db,
  [0x09b8] = 0x25dc, [0x09b9] = 0x25dd, [0x09ba] = 0x25de, [0x09bb] = 0x25df,
  [0x09bc] = 0x25e0, [0x09bd] = 0x25e1, [0x09be] = 0x25e6, [0x09bf] = 0x25e7,
  [0x09c0] = 0x25e8, [0x09c1] = 0x25e9, [0x09c2] = 0x25ea, [0x09c3] = 0x25eb,
  [0x09c4] = 0x25ec, [0x09c5] = 0x25ed, [0x09c6] = 0x25ee, [0x09c7] = 0x25ef,
  [0x09c8] = 0x25f0, [0x09c9] = 0x25f1, [0x09ca] = 0x25f2, [0x09cb] = 0x25f3,
  [0x09cc] = 0x25f4, [0x09cd] = 0x25f5, [0x09ce] = 0x25f6, [0x09cf] = 0x25f7,
  [0x09d0] = 0x25f8, [0x09d1] = 0x25f9, [0x09d2] = 0x25fa, [0x09d3] = 0x25fb,
  [0x09d4] = 0x25fc, [0x09d5] = 0x25fd, [0x09d6] = 0x25fe, [0x09d7] = 0x25ff,
  [0x09d8] = 0x2600, [0x09d9] = 0x2601, [0x09da] = 0x2602, [0x09db] = 0x2603,
  [0x09dc] = 0x2604, [0x09dd] = 0x2607, [0x09de] = 0x2608, [0x09df] = 0x260a,
  [0x09e0] = 0x260b, [0x09e1] = 0x260c, [0x09e2] = 0x260d, [0x09e3] = 0x260e,
  [0x09e4] = 0x260f, [0x09e5] = 0x2610, [0x09e6] = 0x2611, [0x09e7] = 0x2612,
  [0x09e8] = 0x2613, [0x09e9] = 0x2614, [0x09ea] = 0x2615, [0x09eb] = 0x2616,
  [0x09ec] = 0x2617, [0x09ed] = 0x2618, [0x09ee] = 0x2619, [0x09ef] = 0x261a,
  [0x09f0] = 0x261b, [0x09f1] = 0x261c, [0x09f2] = 0x261d, [0x09f3] = 0x261e,
  [0x09f4] = 0x261f, [0x09f5] = 0x2620, [0x09f6] = 0x2621, [0x09f7] = 0x2622,
  [0x09f8] = 0x2623, [0x09f9] = 0x2624, [0x09fa] = 0x2625, [0x09fb] = 0x2626,
  [0x09fc] = 0x2627, [0x09fd] = 0x2628, [0x09fe] = 0x2629, [0x09ff] = 0x262a,
  [0x0a00] = 0x262b, [0x0a01] = 0x262c, [0x0a02] = 0x262d, [0x0a03] = 0x262e,
  [0x0a04] = 0x262f, [0x0a05] = 0x2630, [0x0a06] = 0x2631, [0x0a07] = 0x2632,
  [0x0a08] = 0x2633, [0x0a09] = 0x2634, [0x0a0a] = 0x2635, [0x0a0b] = 0x2636,
  [0x0a0c] = 0x2637, [0x0a0d] = 0x2638, [0x0a0e] = 0x2639, [0x0a0f] = 0x263a,
  [0x0a10] = 0x263b, [0x0a11] = 0x263c, [0x0a12] = 0x263d, [0x0a13] = 0x263e,
  [0x0a14] = 0x263f, [0x0a15] = 0x2641, [0x0a16] = 0x2e82, [0x0a17] = 0x2e83,
  [0x0a18] = 0x2e85, [0x0a19] = 0x2e86, [0x0a1a] = 0x2e87, [0x0a1b] = 0x2e89,
  [0x0a1c] = 0x2e8a, [0x0a1d] = 0x2e8d, [0x0a1e] = 0x2e8e, [0x0a1f] = 0x2e8f,
  [0x0a20] = 0x2e90, [0x0a21] = 0x2e91, [0x0a22] = 0x2e92, [0x0a23] = 0x2e93,
  [0x0a24] = 0x2e94, [0x0a25] = 0x2e95, [0x0a26] = 0x2e96, [0x0a27] = 0x2e98,
  [0x0a28] = 0x2e99, [0x0a29] = 0x2e9a, [0x0a2a] = 0x2e9b, [0x0a2b] = 0x2e9c,
  [0x0a2c] = 0x2e9d, [0x0a2d] = 0x2e9e, [0x0a2e] = 0x2e9f, [0x0a2f] = 0x2ea0,
  [0x0a30] = 0x2ea1, [0x0a31] = 0x2ea2, [0x0a32] = 0x2ea3, [0x0a33] = 0x2ea4,
  [0x0a34] = 0x2ea5, [0x0a35] = 0x2ea6, [0x0a36] = 0x2ea8, [0x0a37] = 0x2ea9,
  [0x0a38] = 0x2eab, [0x0a39] = 0x2eac, [0x0a3a] = 0x2ead, [0x0a3b] = 0x2eaf,
  [0x0a3c] = 0x2eb0, [0x0a3d] = 0x2eb1, [0x0a3e] = 0x2eb2, [0x0a3f] = 0x2eb4,
  [0x0a40] = 0x2eb5, [0x0a41] = 0x2eb8, [0x0a42] = 0x2eb9, [0x0a43] = 0x2eba,
  [0x0a44] = 0x2ebc, [0x0a45] = 0x2ebd, [0x0a46] = 0x2ebe, [0x0a47] = 0x2ebf,
  [0x0a48] = 0x2ec0, [0x0a49] = 0x2ec1, [0x0a4a] = 0x2ec2, [0x0a4b] = 0x2ec3,
  [0x0a4c] = 0x2ec4, [0x0a4d] = 0x2ec5, [0x0a4e] = 0x2ec6, [0x0a4f] = 0x2ec7,
  [0x0a50] = 0x2ec8, [0x0a51] = 0x2ec9, [0x0a52] = 0x2ecb, [0x0a53] = 0x2ecc,
  [0x0a54] = 0x2ecd, [0x0a55] = 0x2ece, [0x0a56] = 0x2ecf, [0x0a57] = 0x2ed0,
  [0x0a58] = 0x2ed1, [0x0a59] = 0x2ed2, [0x0a5a] = 0x2ed3, [0x0a5b] = 0x2ed4,
  [0x0a5c] = 0x2ed5, [0x0a5d] = 0x2ed6, [0x0a5e] = 0x2ed7, [0x0a5f] = 0x2ed8,
  [0x0a60] = 0x2ed9, [0x0a61] = 0x2eda, [0x0a62] = 0x2edb, [0x0a63] = 0x2edc,
  [0x0a64] = 0x2edd, [0x0a65] = 0x2ede, [0x0a66] = 0x2edf, [0x0a67] = 0x2ee0,
  [0x0a68] = 0x2ee1, [0x0a69] = 0x2ee2, [0x0a6a] = 0x2ee3, [0x0a6b] = 0x2ee4,
  [0x0a6c] = 0x2ee5, [0x0a6d] = 0x2ee6, [0x0a6e] = 0x2ee7, [0x0a6f] = 0x2ee8,
  [0x0a70] = 0x2ee9, [0x0a71] = 0x2eea, [0x0a72] = 0x2eeb, [0x0a73] = 0x2eec,
  [0x0a74] = 0x2eed, [0x0a75] = 0x2eee, [0x0a76] = 0x2eef, [0x0a77] = 0x2ef0,
  [0x0a78] = 0x2ef1, [0x0a79] = 0x2ef2, [0x0a7a] = 0x2ef3, [0x0a7b] = 0x2ef4,
  [0x0a7c] = 0x2ef5, [0x0a7d] = 0x2ef6, [0x0a7e] = 0x2ef7, [0x0a7f] = 0x2ef8,
  [0x0a80] = 0x2ef9, [0x0a81] = 0x2efa, [0x0a82] = 0x2efb, [0x0a83] = 0x2efc,
  [0x0a84] = 0x2efd, [0x0a85] = 0x2efe, [0x0a86] = 0x2eff, [0x0a87] = 0x2f00,
  [0x0a88] = 0x2f01, [0x0a89] = 0x2f02, [0x0a8a] = 0x2f03, [0x0a8b] = 0x2f04,
  [0x0a8c] = 0x2f05, [0x0a8d] = 0x2f06, [0x0a8e] = 0x2f07, [0x0a8f] = 0x2f08,
  [0x0a90] = 0x2f09, [0x0a91] = 0x2f0a, [0x0a92] = 0x2f0b, [0x0a93] = 0x2f0c,
  [0x0a94] = 0x2f0d, [0x0a95] = 0x2f0e, [0x0a96] = 0x2f0f, [0x0a97] = 0x2f10,
  [0x0a98] = 0x2f11, [0x0a99] = 0x2f12, [0x0a9a] = 0x2f13, [0x0a9b] = 0x2f14,
  [0x0a9c] = 0x2f15, [0x0a9d] = 0x2f16, [0x0a9e] = 0x2f17, [0x0a9f] = 0x2f18,
  [0x0aa0] = 0x2f19, [0x0aa1] = 0x2f1a, [0x0aa2] = 0x2f1b, [0x0aa3] = 0x2f1c,
  [0x0aa4] = 0x2f1d, [0x0aa5] = 0x2f1e, [0x0aa6] = 0x2f1f, [0x0aa7] = 0x2f20,
  [0x0aa8] = 0x2f21, [0x0aa9] = 0x2f22, [0x0aaa] = 0x2f23, [0x0aab] = 0x2f24,
  [0x0aac] = 0x2f25, [0x0aad] = 0x2f26, [0x0aae] = 0x2f27, [0x0aaf] = 0x2f28,
  [0x0ab0] = 0x2f29, [0x0ab1] = 0x2f2a, [0x0ab2] = 0x2f2b, [0x0ab3] = 0x2f2c,
  [0x0ab4] = 0x2f2d, [0x0ab5] = 0x2f2e, [0x0ab6] = 0x2f2f, [0x0ab7] = 0x2f30,
  [0x0ab8] = 0x2f31, [0x0ab9] = 0x2f32, [0x0aba] = 0x2f33, [0x0abb] = 0x2f34,
  [0x0abc] = 0x2f35, [0x0abd] = 0x2f36, [0x0abe] = 0x2f37, [0x0abf] = 0x2f38,
  [0x0ac0] = 0x2f39, [0x0ac1] = 0x2f3a, [0x0ac2] = 0x2f3b, [0x0ac3] = 0x2f3c,
  [0x0ac4] = 0x2f3d, [0x0ac5] = 0x2f3e, [0x0ac6] = 0x2f3f, [0x0ac7] = 0x2f40,
  [0x0ac8] = 0x2f41, [0x0ac9] = 0x2f42, [0x0aca] = 0x2f43, [0x0acb] = 0x2f44,
  [0x0acc] = 0x2f45, [0x0acd] = 0x2f46, [0x0ace] = 0x2f47, [0x0acf] = 0x2f48,
  [0x0ad0] = 0x2f49, [0x0ad1] = 0x2f4a, [0x0ad2] = 0x2f4b, [0x0ad3] = 0x2f4c,
  [0x0ad4] = 0x2f4d, [0x0ad5] = 0x2f4e, [0x0ad6] = 0x2f4f, [0x0ad7] = 0x2f50,
  [0x0ad8] = 0x2f51, [0x0ad9] = 0x2f52, [0x0ada] = 0x2f53, [0x0adb] = 0x2f54,
  [0x0adc] = 0x2f55, [0x0add] = 0x2f56, [0x0ade] = 0x2f57, [0x0adf] = 0x2f58,
  [0x0ae0] = 0x2f59, [0x0ae1] = 0x2f5a, [0x0ae2] = 0x2f5b, [0x0ae3] = 0x2f5c,
  [0x0ae4] = 0x2f5d, [0x0ae5] = 0x2f5e, [0x0ae6] = 0x2f5f, [0x0ae7] = 0x2f60,
  [0x0ae8] = 0x2f61, [0x0ae9] = 0x2f62, [0x0aea] = 0x2f63, [0x0aeb] = 0x2f64,
  [0x0aec] = 0x2f65, [0x0aed] = 0x2f66, [0x0aee] = 0x2f67, [0x0aef] = 0x2f68,
  [0x0af0] = 0x2f69, [0x0af1] = 0x2f6a, [0x0af2] = 0x2f6b, [0x0af3] = 0x2f6c,
  [0x0af4] = 0x2f6d, [0x0af5] = 0x2f6e, [0x0af6] = 0x2f6f, [0x0af7] = 0x2f70,
  [0x0af8] = 0x2f71, [0x0af9] = 0x2f72, [0x0afa] = 0x2f73, [0x0afb] = 0x2f74,
  [0x0afc] = 0x2f75, [0x0afd] = 0x2f76, [0x0afe] = 0x2f77, [0x0aff] = 0x2f78,
  [0x0b00] = 0x2f79, [0x0b01] = 0x2f7a, [0x0b02] = 0x2f7b, [0x0b03] = 0x2f7c,
  [0x0b04] = 0x2f7d, [0x0b05] = 0x2f7e, [0x0b06] = 0x2f7f, [0x0b07] = 0x2f80,
  [0x0b08] = 0x2f81, [0x0b09] = 0x2f82, [0x0b0a] = 0x2f83, [0x0b0b] = 0x2f84,
  [0x0b0c] = 0x2f85, [0x0b0d] = 0x2f86, [0x0b0e] = 0x2f87, [0x0b0f] = 0x2f88,
  [0x0b10] = 0x2f89, [0x0b11] = 0x2f8a, [0x0b12] = 0x2f8b, [0x0b13] = 0x2f8c,
  [0x0b14] = 0x2f8d, [0x0b15] = 0x2f8e, [0x0b16] = 0x2f8f, [0x0b17] = 0x2f90,
  [0x0b18] = 0x2f91, [0x0b19] = 0x2f92, [0x0b1a] = 0x2f93, [0x0b1b] = 0x2f94,
  [0x0b1c] = 0x2f95, [0x0b1d] = 0x2f96, [0x0b1e] = 0x2f97, [0x0b1f] = 0x2f98,
  [0x0b20] = 0x2f99, [0x0b21] = 0x2f9a, [0x0b22] = 0x2f9b, [0x0b23] = 0x2f9c,
  [0x0b24] = 0x2f9d, [0x0b25] = 0x2f9e, [0x0b26] = 0x2f9f, [0x0b27] = 0x2fa0,
  [0x0b28] = 0x2fa1, [0x0b29] = 0x2fa2, [0x0b2a] = 0x2fa3, [0x0b2b] = 0x2fa4,
  [0x0b2c] = 0x2fa5, [0x0b2d] = 0x2fa6, [0x0b2e] = 0x2fa7, [0x0b2f] = 0x2fa8,
  [0x0b30] = 0x2fa9, [0x0b31] = 0x2faa, [0x0b32] = 0x2fab, [0x0b33] = 0x2fac,
  [0x0b34] = 0x2fad, [0x0b35] = 0x2fae, [0x0b36] = 0x2faf, [0x0b37] = 0x2fb0,
  [0x0b38] = 0x2fb1, [0x0b39] = 0x2fb2, [0x0b3a] = 0x2fb3, [0x0b3b] = 0x2fb4,
  [0x0b3c] = 0x2fb5, [0x0b3d] = 0x2fb6, [0x0b3e] = 0x2fb7, [0x0b3f] = 0x2fb8,
  [0x0b40] = 0x2fb9, [0x0b41] = 0x2fba, [0x0b42] = 0x2fbb, [0x0b43] = 0x2fbc,
  [0x0b44] = 0x2fbd, [0x0b45] = 0x2fbe, [0x0b46] = 0x2fbf, [0x0b47] = 0x2fc0,
  [0x0b48] = 0x2fc1, [0x0b49] = 0x2fc2, [0x0b4a] = 0x2fc3, [0x0b4b] = 0x2fc4,
  [0x0b4c] = 0x2fc5, [0x0b4d] = 0x2fc6, [0x0b4e] = 0x2fc7, [0x0b4f] = 0x2fc8,
  [0x0b50] = 0x2fc9, [0x0b51] = 0x2fca, [0x0b52] = 0x2fcb, [0x0b53] = 0x2fcc,
  [0x0b54] = 0x2fcd, [0x0b55] = 0x2fce, [0x0b56] = 0x2fcf, [0x0b57] = 0x2fd0,
  [0x0b58] = 0x2fd1, [0x0b59] = 0x2fd2, [0x0b5a] = 0x2fd3, [0x0b5b] = 0x2fd4,
  [0x0b5c] = 0x2fd5, [0x0b5d] = 0x2fd6, [0x0b5e] = 0x2fd7, [0x0b5f] = 0x2fd8,
  [0x0b60] = 0x2fd9, [0x0b61] = 0x2fda, [0x0b62] = 0x2fdb, [0x0b63] = 0x2fdc,
  [0x0b64] = 0x2fdd, [0x0b65] = 0x2fde, [0x0b66] = 0x2fdf, [0x0b67] = 0x2fe0,
  [0x0b68] = 0x2fe1, [0x0b69] = 0x2fe2, [0x0b6a] = 0x2fe3, [0x0b6b] = 0x2fe4,
  [0x0b6c] = 0x2fe5, [0x0b6d] = 0x2fe6, [0x0b6e] = 0x2fe7, [0x0b6f] = 0x2fe8,
  [0x0b70] = 0x2fe9, [0x0b71] = 0x2fea, [0x0b72] = 0x2feb, [0x0b73] = 0x2fec,
  [0x0b74] = 0x2fed, [0x0b75] = 0x2fee, [0x0b76] = 0x2fef, [0x0b77] = 0x2ffc,
  [0x0b78] = 0x2ffd, [0x0b79] = 0x2ffe, [0x0b7a] = 0x2fff, [0x0b7b] = 0x3004,
  [0x0b7c] = 0x3018, [0x0b7d] = 0x3019, [0x0b7e] = 0x301a, [0x0b7f] = 0x301b,
  [0x0b80] = 0x301c, [0x0b81] = 0x301f, [0x0b82] = 0x3020, [0x0b83] = 0x302a,
  [0x0b84] = 0x302b, [0x0b85] = 0x302c, [0x0b86] = 0x302d, [0x0b87] = 0x302e,
  [0x0b88] = 0x302f, [0x0b89] = 0x3030, [0x0b8a] = 0x3031, [0x0b8b] = 0x3032,
  [0x0b8c] = 0x3033, [0x0b8d] = 0x3034, [0x0b8e] = 0x3035, [0x0b8f] = 0x3036,
  [0x0b90] = 0x3037, [0x0b91] = 0x3038, [0x0b92] = 0x3039, [0x0b93] = 0x303a,
  [0x0b94] = 0x303b, [0x0b95] = 0x303c, [0x0b96] = 0x303d, [0x0b97] = 0x303f,
  [0x0b98] = 0x3040, [0x0b99] = 0x3094, [0x0b9a] = 0x3095, [0x0b9b] = 0x3096,
  [0x0b9c] = 0x3097, [0x0b9d] = 0x3098, [0x0b9e] = 0x3099, [0x0b9f] = 0x309a,
  [0x0ba0] = 0x309f, [0x0ba1] = 0x30a0, [0x0ba2] = 0x30f7, [0x0ba3] = 0x30f8,
  [0x0ba4] = 0x30f9, [0x0ba5] = 0x30fa, [0x0ba6] = 0x30fb, [0x0ba7] = 0x30ff,
  [0x0ba8] = 0x3100, [0x0ba9] = 0x3101, [0x0baa] = 0x3102, [0x0bab] = 0x3103,
  [0x0bac] = 0x3104, [0x0bad] = 0x312a, [0x0bae] = 0x312b, [0x0baf] = 0x312c,
  [0x0bb0] = 0x312d, [0x0bb1] = 0x312e, [0x0bb2] = 0x312f, [0x0bb3] = 0x3130,
  [0x0bb4] = 0x3131, [0x0bb5] = 0x3132, [0x0bb6] = 0x3133, [0x0bb7] = 0x3134,
  [0x0bb8] = 0x3135, [0x0bb9] = 0x3136, [0x0bba] = 0x3137, [0x0bbb] = 0x3138,
  [0x0bbc] = 0x3139, [0x0bbd] = 0x313a, [0x0bbe] = 0x313b, [0x0bbf] = 0x313c,
  [0x0bc0] = 0x313d, [0x0bc1] = 0x313e, [0x0bc2] = 0x313f, [0x0bc3] = 0x3140,
  [0x0bc4] = 0x3141, [0x0bc5] = 0x3142, [0x0bc6] = 0x3143, [0x0bc7] = 0x3144,
  [0x0bc8] = 0x3145, [0x0bc9] = 0x3146, [0x0bca] = 0x3147, [0x0bcb] = 0x3148,
  [0x0bcc] = 0x3149, [0x0bcd] = 0x314a, [0x0bce] = 0x314b, [0x0bcf] = 0x314c,
  [0x0bd0] = 0x314d, [0x0bd1] = 0x314e, [0x0bd2] = 0x314f, [0x0bd3] = 0x3150,
  [0x0bd4] = 0x3151, [0x0bd5] = 0x3152, [0x0bd6] = 0x3153, [0x0bd7] = 0x3154,
  [0x0bd8] = 0x3155, [0x0bd9] = 0x3156, [0x0bda] = 0x3157, [0x0bdb] = 0x3158,
  [0x0bdc] = 0x3159, [0x0bdd] = 0x315a, [0x0bde] = 0x315b, [0x0bdf] = 0x315c,
  [0x0be0] = 0x315d, [0x0be1] = 0x315e, [0x0be2] = 0x315f, [0x0be3] = 0x3160,
  [0x0be4] = 0x3161, [0x0be5] = 0x3162, [0x0be6] = 0x3163, [0x0be7] = 0x3164,
  [0x0be8] = 0x3165, [0x0be9] = 0x3166, [0x0bea] = 0x3167, [0x0beb] = 0x3168,
  [0x0bec] = 0x3169, [0x0bed] = 0x316a, [0x0bee] = 0x316b, [0x0bef] = 0x316c,
  [0x0bf0] = 0x316d, [0x0bf1] = 0x316e, [0x0bf2] = 0x316f, [0x0bf3] = 0x3170,
  [0x0bf4] = 0x3171, [0x0bf5] = 0x3172, [0x0bf6] = 0x3173, [0x0bf7] = 0x3174,
  [0x0bf8] = 0x3175, [0x0bf9] = 0x3176, [0x0bfa] = 0x3177, [0x0bfb] = 0x3178,
  [0x0bfc] = 0x3179, [0x0bfd] = 0x317a, [0x0bfe] = 0x317b, [0x0bff] = 0x317c,
  [0x0c00] = 0x317d, [0x0c01] = 0x317e, [0x0c02] = 0x317f, [0x0c03] = 0x3180,
  [0x0c04] = 0x3181, [0x0c05] = 0x3182, [0x0c06] = 0x3183, [0x0c07] = 0x3184,
  [0x0c08] = 0x3185, [0x0c09] = 0x3186, [0x0c0a] = 0x3187, [0x0c0b] = 0x3188,
  [0x0c0c] = 0x3189, [0x0c0d] = 0x318a, [0x0c0e] = 0x318b, [0x0c0f] = 0x318c,
  [0x0c10] = 0x318d, [0x0c11] = 0x318e, [0x0c12] = 0x318f, [0x0c13] = 0x3190,
  [0x0c14] = 0x3191, [0x0c15] = 0x3192, [0x0c16] = 0x3193, [0x0c17] = 0x3194,
  [0x0c18] = 0x3195, [0x0c19] = 0x3196, [0x0c1a] = 0x3197, [0x0c1b] = 0x3198,
  [0x0c1c] = 0x3199, [0x0c1d] = 0x319a, [0x0c1e] = 0x319b, [0x0c1f] = 0x319c,
  [0x0c20] = 0x319d, [0x0c21] = 0x319e, [0x0c22] = 0x319f, [0x0c23] = 0x31a0,
  [0x0c24] = 0x31a1, [0x0c25] = 0x31a2, [0x0c26] = 0x31a3, [0x0c27] = 0x31a4,
  [0x0c28] = 0x31a5, [0x0c29] = 0x31a6, [0x0c2a] = 0x31a7, [0x0c2b] = 0x31a8,
  [0x0c2c] = 0x31a9, [0x0c2d] = 0x31aa, [0x0c2e] = 0x31ab, [0x0c2f] = 0x31ac,
  [0x0c30] = 0x31ad, [0x0c31] = 0x31ae, [0x0c32] = 0x31af, [0x0c33] = 0x31b0,
  [0x0c34] = 0x31b1, [0x0c35] = 0x31b2, [0x0c36] = 0x31b3, [0x0c37] = 0x31b4,
  [0x0c38] = 0x31b5, [0x0c39] = 0x31b6, [0x0c3a] = 0x31b7, [0x0c3b] = 0x31b8,
  [0x0c3c] = 0x31b9, [0x0c3d] = 0x31ba, [0x0c3e] = 0x31bb, [0x0c3f] = 0x31bc,
  [0x0c40] = 0x31bd, [0x0c41] = 0x31be, [0x0c42] = 0x31bf, [0x0c43] = 0x31c0,
  [0x0c44] = 0x31c1, [0x0c45] = 0x31c2, [0x0c46] = 0x31c3, [0x0c47] = 0x31c4,
  [0x0c48] = 0x31c5, [0x0c49] = 0x31c6, [0x0c4a] = 0x31c7, [0x0c4b] = 0x31c8,
  [0x0c4c] = 0x31c9, [0x0c4d] = 0x31ca, [0x0c4e] = 0x31cb, [0x0c4f] = 0x31cc,
  [0x0c50] = 0x31cd, [0x0c51] = 0x31ce, [0x0c52] = 0x31cf, [0x0c53] = 0x31d0,
  [0x0c54] = 0x31d1, [0x0c55] = 0x31d2, [0x0c56] = 0x31d3, [0x0c57] = 0x31d4,
  [0x0c58] = 0x31d5, [0x0c59] = 0x31d6, [0x0c5a] = 0x31d7, [0x0c5b] = 0x31d8,
  [0x0c5c] = 0x31d9, [0x0c5d] = 0x31da, [0x0c5e] = 0x31db, [0x0c5f] = 0x31dc,
  [0x0c60] = 0x31dd, [0x0c61] = 0x31de, [0x0c62] = 0x31df, [0x0c63] = 0x31e0,
  [0x0c64] = 0x31e1, [0x0c65] = 0x31e2, [0x0c66] = 0x31e3, [0x0c67] = 0x31e4,
  [0x0c68] = 0x31e5, [0x0c69] = 0x31e6, [0x0c6a] = 0x31e7, [0x0c6b] = 0x31e8,
  [0x0c6c] = 0x31e9, [0x0c6d] = 0x31ea, [0x0c6e] = 0x31eb, [0x0c6f] = 0x31ec,
  [0x0c70] = 0x31ed, [0x0c71] = 0x31ee, [0x0c72] = 0x31ef, [0x0c73] = 0x31f0,
  [0x0c74] = 0x31f1, [0x0c75] = 0x31f2, [0x0c76] = 0x31f3, [0x0c77] = 0x31f4,
  [0x0c78] = 0x31f5, [0x0c79] = 0x31f6, [0x0c7a] = 0x31f7, [0x0c7b] = 0x31f8,
  [0x0c7c] = 0x31f9, [0x0c7d] = 0x31fa, [0x0c7e] = 0x31fb, [0x0c7f] = 0x31fc,
  [0x0c80] = 0x31fd, [0x0c81] = 0x31fe, [0x0c82] = 0x31ff, [0x0c83] = 0x3200,
  [0x0c84] = 0x3201, [0x0c85] = 0x3202, [0x0c86] = 0x3203, [0x0c87] = 0x3204,
  [0x0c88] = 0x3205, [0x0c89] = 0x3206, [0x0c8a] = 0x3207, [0x0c8b] = 0x3208,
  [0x0c8c] = 0x3209, [0x0c8d] = 0x320a, [0x0c8e] = 0x320b, [0x0c8f] = 0x320c,
  [0x0c90] = 0x320d, [0x0c91] = 0x320e, [0x0c92] = 0x320f, [0x0c93] = 0x3210,
  [0x0c94] = 0x3211, [0x0c95] = 0x3212, [0x0c96] = 0x3213, [0x0c97] = 0x3214,
  [0x0c98] = 0x3215, [0x0c99] = 0x3216, [0x0c9a] = 0x3217, [0x0c9b] = 0x3218,
  [0x0c9c] = 0x3219, [0x0c9d] = 0x321a, [0x0c9e] = 0x321b, [0x0c9f] = 0x321c,
  [0x0ca0] = 0x321d, [0x0ca1] = 0x321e, [0x0ca2] = 0x321f, [0x0ca3] = 0x322a,
  [0x0ca4] = 0x322b, [0x0ca5] = 0x322c, [0x0ca6] = 0x322d, [0x0ca7] = 0x322e,
  [0x0ca8] = 0x322f, [0x0ca9] = 0x3230, [0x0caa] = 0x3232, [0x0cab] = 0x3233,
  [0x0cac] = 0x3234, [0x0cad] = 0x3235, [0x0cae] = 0x3236, [0x0caf] = 0x3237,
  [0x0cb0] = 0x3238, [0x0cb1] = 0x3239, [0x0cb2] = 0x323a, [0x0cb3] = 0x323b,
  [0x0cb4] = 0x323c, [0x0cb5] = 0x323d, [0x0cb6] = 0x323e, [0x0cb7] = 0x323f,
  [0x0cb8] = 0x3240, [0x0cb9] = 0x3241, [0x0cba] = 0x3242, [0x0cbb] = 0x3243,
  [0x0cbc] = 0x3244, [0x0cbd] = 0x3245, [0x0cbe] = 0x3246, [0x0cbf] = 0x3247,
  [0x0cc0] = 0x3248, [0x0cc1] = 0x3249, [0x0cc2] = 0x324a, [0x0cc3] = 0x324b,
  [0x0cc4] = 0x324c, [0x0cc5] = 0x324d, [0x0cc6] = 0x324e, [0x0cc7] = 0x324f,
  [0x0cc8] = 0x3250, [0x0cc9] = 0x3251, [0x0cca] = 0x3252, [0x0ccb] = 0x3253,
  [0x0ccc] = 0x3254, [0x0ccd] = 0x3255, [0x0cce] = 0x3256, [0x0ccf] = 0x3257,
  [0x0cd0] = 0x3258, [0x0cd1] = 0x3259, [0x0cd2] = 0x325a, [0x0cd3] = 0x325b,
  [0x0cd4] = 0x325c, [0x0cd5] = 0x325d, [0x0cd6] = 0x325e, [0x0cd7] = 0x325f,
  [0x0cd8] = 0x3260, [0x0cd9] = 0x3261, [0x0cda] = 0x3262, [0x0cdb] = 0x3263,
  [0x0cdc] = 0x3264, [0x0cdd] = 0x3265, [0x0cde] = 0x3266, [0x0cdf] = 0x3267,
  [0x0ce0] = 0x3268, [0x0ce1] = 0x3269, [0x0ce2] = 0x326a, [0x0ce3] = 0x326b,
  [0x0ce4] = 0x326c, [0x0ce5] = 0x326d, [0x0ce6] = 0x326e, [0x0ce7] = 0x326f,
  [0x0ce8] = 0x3270, [0x0ce9] = 0x3271, [0x0cea] = 0x3272, [0x0ceb] = 0x3273,
  [0x0cec] = 0x3274, [0x0ced] = 0x3275, [0x0cee] = 0x3276, [0x0cef] = 0x3277,
  [0x0cf0] = 0x3278, [0x0cf1] = 0x3279, [0x0cf2] = 0x327a, [0x0cf3] = 0x327b,
  [0x0cf4] = 0x327c, [0x0cf5] = 0x327d, [0x0cf6] = 0x327e, [0x0cf7] = 0x327f,
  [0x0cf8] = 0x3280, [0x0cf9] = 0x3281, [0x0cfa] = 0x3282, [0x0cfb] = 0x3283,
  [0x0cfc] = 0x3284, [0x0cfd] = 0x3285, [0x0cfe] = 0x3286, [0x0cff] = 0x3287,
  [0x0d00] = 0x3288, [0x0d01] = 0x3289, [0x0d02] = 0x328a, [0x0d03] = 0x328b,
  [0x0d04] = 0x328c, [0x0d05] = 0x328d, [0x0d06] = 0x328e, [0x0d07] = 0x328f,
  [0x0d08] = 0x3290, [0x0d09] = 0x3291, [0x0d0a] = 0x3292, [0x0d0b] = 0x3293,
  [0x0d0c] = 0x3294, [0x0d0d] = 0x3295, [0x0d0e] = 0x3296, [0x0d0f] = 0x3297,
  [0x0d10] = 0x3298, [0x0d11] = 0x3299, [0x0d12] = 0x329a, [0x0d13] = 0x329b,
  [0x0d14] = 0x329c, [0x0d15] = 0x329d, [0x0d16] = 0x329e, [0x0d17] = 0x329f,
  [0x0d18] = 0x32a0, [0x0d19] = 0x32a1, [0x0d1a] = 0x32a2, [0x0d1b] = 0x32a4,
  [0x0d1c] = 0x32a5, [0x0d1d] = 0x32a6, [0x0d1e] = 0x32a7, [0x0d1f] = 0x32a8,
  [0x0d20] = 0x32a9, [0x0d21] = 0x32aa, [0x0d22] = 0x32ab, [0x0d23] = 0x32ac,
  [0x0d24] = 0x32ad, [0x0d25] = 0x32ae, [0x0d26] = 0x32af, [0x0d27] = 0x32b0,
  [0x0d28] = 0x32b1, [0x0d29] = 0x32b2, [0x0d2a] = 0x32b3, [0x0d2b] = 0x32b4,
  [0x0d2c] = 0x32b5, [0x0d2d] = 0x32b6, [0x0d2e] = 0x32b7, [0x0d2f] = 0x32b8,
  [0x0d30] = 0x32b9, [0x0d31] = 0x32ba, [0x0d32] = 0x32bb, [0x0d33] = 0x32bc,
  [0x0d34] = 0x32bd, [0x0d35] = 0x32be, [0x0d36] = 0x32bf, [0x0d37] = 0x32c0,
  [0x0d38] = 0x32c1, [0x0d39] = 0x32c2, [0x0d3a] = 0x32c3, [0x0d3b] = 0x32c4,
  [0x0d3c] = 0x32c5, [0x0d3d] = 0x32c6, [0x0d3e] = 0x32c7, [0x0d3f] = 0x32c8,
  [0x0d40] = 0x32c9, [0x0d41] = 0x32ca, [0x0d42] = 0x32cb, [0x0d43] = 0x32cc,
  [0x0d44] = 0x32cd, [0x0d45] = 0x32ce, [0x0d46] = 0x32cf, [0x0d47] = 0x32d0,
  [0x0d48] = 0x32d1, [0x0d49] = 0x32d2, [0x0d4a] = 0x32d3, [0x0d4b] = 0x32d4,
  [0x0d4c] = 0x32d5, [0x0d4d] = 0x32d6, [0x0d4e] = 0x32d7, [0x0d4f] = 0x32d8,
  [0x0d50] = 0x32d9, [0x0d51] = 0x32da, [0x0d52] = 0x32db, [0x0d53] = 0x32dc,
  [0x0d54] = 0x32dd, [0x0d55] = 0x32de, [0x0d56] = 0x32df, [0x0d57] = 0x32e0,
  [0x0d58] = 0x32e1, [0x0d59] = 0x32e2, [0x0d5a] = 0x32e3, [0x0d5b] = 0x32e4,
  [0x0d5c] = 0x32e5, [0x0d5d] = 0x32e6, [0x0d5e] = 0x32e7, [0x0d5f] = 0x32e8,
  [0x0d60] = 0x32e9, [0x0d61] = 0x32ea, [0x0d62] = 0x32eb, [0x0d63] = 0x32ec,
  [0x0d64] = 0x32ed, [0x0d65] = 0x32ee, [0x0d66] = 0x32ef, [0x0d67] = 0x32f0,
  [0x0d68] = 0x32f1, [0x0d69] = 0x32f2, [0x0d6a] = 0x32f3, [0x0d6b] = 0x32f4,
  [0x0d6c] = 0x32f5, [0x0d6d] = 0x32f6, [0x0d6e] = 0x32f7, [0x0d6f] = 0x32f8,
  [0x0d70] = 0x32f9, [0x0d71] = 0x32fa, [0x0d72] = 0x32fb, [0x0d73] = 0x32fc,
  [0x0d74] = 0x32fd, [0x0d75] = 0x32fe, [0x0d76] = 0x32ff, [0x0d77] = 0x3300,
  [0x0d78] = 0x3301, [0x0d79] = 0x3302, [0x0d7a] = 0x3303, [0x0d7b] = 0x3304,
  [0x0d7c] = 0x3305, [0x0d7d] = 0x3306, [0x0d7e] = 0x3307, [0x0d7f] = 0x3308,
  [0x0d80] = 0x3309, [0x0d81] = 0x330a, [0x0d82] = 0x330b, [0x0d83] = 0x330c,
  [0x0d84] = 0x330d, [0x0d85] = 0x330e, [0x0d86] = 0x330f, [0x0d87] = 0x3310,
  [0x0d88] = 0x3311, [0x0d89] = 0x3312, [0x0d8a] = 0x3313, [0x0d8b] = 0x3314,
  [0x0d8c] = 0x3315, [0x0d8d] = 0x3316, [0x0d8e] = 0x3317, [0x0d8f] = 0x3318,
  [0x0d90] = 0x3319, [0x0d91] = 0x331a, [0x0d92] = 0x331b, [0x0d93] = 0x331c,
  [0x0d94] = 0x331d, [0x0d95] = 0x331e, [0x0d96] = 0x331f, [0x0d97] = 0x3320,
  [0x0d98] = 0x3321, [0x0d99] = 0x3322, [0x0d9a] = 0x3323, [0x0d9b] = 0x3324,
  [0x0d9c] = 0x3325, [0x0d9d] = 0x3326, [0x0d9e] = 0x3327, [0x0d9f] = 0x3328,
  [0x0da0] = 0x3329, [0x0da1] = 0x332a, [0x0da2] = 0x332b, [0x0da3] = 0x332c,
  [0x0da4] = 0x332d, [0x0da5] = 0x332e, [0x0da6] = 0x332f, [0x0da7] = 0x3330,
  [0x0da8] = 0x3331, [0x0da9] = 0x3332, [0x0daa] = 0x3333, [0x0dab] = 0x3334,
  [0x0dac] = 0x3335, [0x0dad] = 0x3336, [0x0dae] = 0x3337, [0x0daf] = 0x3338,
  [0x0db0] = 0x3339, [0x0db1] = 0x333a, [0x0db2] = 0x333b, [0x0db3] = 0x333c,
  [0x0db4] = 0x333d, [0x0db5] = 0x333e, [0x0db6] = 0x333f, [0x0db7] = 0x3340,
  [0x0db8] = 0x3341, [0x0db9] = 0x3342, [0x0dba] = 0x3343, [0x0dbb] = 0x3344,
  [0x0dbc] = 0x3345, [0x0dbd] = 0x3346, [0x0dbe] = 0x3347, [0x0dbf] = 0x3348,
  [0x0dc0] = 0x3349, [0x0dc1] = 0x334a, [0x0dc2] = 0x334b, [0x0dc3] = 0x334c,
  [0x0dc4] = 0x334d, [0x0dc5] = 0x334e, [0x0dc6] = 0x334f, [0x0dc7] = 0x3350,
  [0x0dc8] = 0x3351, [0x0dc9] = 0x3352, [0x0dca] = 0x3353, [0x0dcb] = 0x3354,
  [0x0dcc] = 0x3355, [0x0dcd] = 0x3356, [0x0dce] = 0x3357, [0x0dcf] = 0x3358,
  [0x0dd0] = 0x3359, [0x0dd1] = 0x335a, [0x0dd2] = 0x335b, [0x0dd3] = 0x335c,
  [0x0dd4] = 0x335d, [0x0dd5] = 0x335e, [0x0dd6] = 0x335f, [0x0dd7] = 0x3360,
  [0x0dd8] = 0x3361, [0x0dd9] = 0x3362, [0x0dda] = 0x3363, [0x0ddb] = 0x3364,
  [0x0ddc] = 0x3365, [0x0ddd] = 0x3366, [0x0dde] = 0x3367, [0x0ddf] = 0x3368,
  [0x0de0] = 0x3369, [0x0de1] = 0x336a, [0x0de2] = 0x336b, [0x0de3] = 0x336c,
  [0x0de4] = 0x336d, [0x0de5] = 0x336e, [0x0de6] = 0x336f, [0x0de7] = 0x3370,
  [0x0de8] = 0x3371, [0x0de9] = 0x3372, [0x0dea] = 0x3373, [0x0deb] = 0x3374,
  [0x0dec] = 0x3375, [0x0ded] = 0x3376, [0x0dee] = 0x3377, [0x0def] = 0x3378,
  [0x0df0] = 0x3379, [0x0df1] = 0x337a, [0x0df2] = 0x337b, [0x0df3] = 0x337c,
  [0x0df4] = 0x337d, [0x0df5] = 0x337e, [0x0df6] = 0x337f, [0x0df7] = 0x3380,
  [0x0df8] = 0x3381, [0x0df9] = 0x3382, [0x0dfa] = 0x3383, [0x0dfb] = 0x3384,
  [0x0dfc] = 0x3385, [0x0dfd] = 0x3386, [0x0dfe] = 0x3387, [0x0dff] = 0x3388,
  [0x0e00] = 0x3389, [0x0e01] = 0x338a, [0x0e02] = 0x338b, [0x0e03] = 0x338c,
  [0x0e04] = 0x338d, [0x0e05] = 0x3390, [0x0e06] = 0x3391, [0x0e07] = 0x3392,
  [0x0e08] = 0x3393, [0x0e09] = 0x3394, [0x0e0a] = 0x3395, [0x0e0b] = 0x3396,
  [0x0e0c] = 0x3397, [0x0e0d] = 0x3398, [0x0e0e] = 0x3399, [0x0e0f] = 0x339a,
  [0x0e10] = 0x339b, [0x0e11] = 0x339f, [0x0e12] = 0x33a0, [0x0e13] = 0x33a2,
  [0x0e14] = 0x33a3, [0x0e15] = 0x33a4, [0x0e16] = 0x33a5, [0x0e17] = 0x33a6,
  [0x0e18] = 0x33a7, [0x0e19] = 0x33a8, [0x0e1a] = 0x33a9, [0x0e1b] = 0x33aa,
  [0x0e1c] = 0x33ab, [0x0e1d] = 0x33ac, [0x0e1e] = 0x33ad, [0x0e1f] = 0x33ae,
  [0x0e20] = 0x33af, [0x0e21] = 0x33b0, [0x0e22] = 0x33b1, [0x0e23] = 0x33b2,
  [0x0e24] = 0x33b3, [0x0e25] = 0x33b4, [0x0e26] = 0x33b5, [0x0e27] = 0x33b6,
  [0x0e28] = 0x33b7, [0x0e29] = 0x33b8, [0x0e2a] = 0x33b9, [0x0e2b] = 0x33ba,
  [0x0e2c] = 0x33bb, [0x0e2d] = 0x33bc, [0x0e2e] = 0x33bd, [0x0e2f] = 0x33be,
  [0x0e30] = 0x33bf, [0x0e31] = 0x33c0, [0x0e32] = 0x33c1, [0x0e33] = 0x33c2,
  [0x0e34] = 0x33c3, [0x0e35] = 0x33c5, [0x0e36] = 0x33c6, [0x0e37] = 0x33c7,
  [0x0e38] = 0x33c8, [0x0e39] = 0x33c9, [0x0e3a] = 0x33ca, [0x0e3b] = 0x33cb,
  [0x0e3c] = 0x33cc, [0x0e3d] = 0x33cd, [0x0e3e] = 0x33cf, [0x0e3f] = 0x33d0,
  [0x0e40] = 0x33d3, [0x0e41] = 0x33d4, [0x0e42] = 0x33d6, [0x0e43] = 0x33d7,
  [0x0e44] = 0x33d8, [0x0e45] = 0x33d9, [0x0e46] = 0x33da, [0x0e47] = 0x33db,
  [0x0e48] = 0x33dc, [0x0e49] = 0x33dd, [0x0e4a] = 0x33de, [0x0e4b] = 0x33df,
  [0x0e4c] = 0x33e0, [0x0e4d] = 0x33e1, [0x0e4e] = 0x33e2, [0x0e4f] = 0x33e3,
  [0x0e50] = 0x33e4, [0x0e51] = 0x33e5, [0x0e52] = 0x33e6, [0x0e53] = 0x33e7,
  [0x0e54] = 0x33e8, [0x0e55] = 0x33e9, [0x0e56] = 0x33ea, [0x0e57] = 0x33eb,
  [0x0e58] = 0x33ec, [0x0e59] = 0x33ed, [0x0e5a] = 0x33ee, [0x0e5b] = 0x33ef,
  [0x0e5c] = 0x33f0, [0x0e5d] = 0x33f1, [0x0e5e] = 0x33f2, [0x0e5f] = 0x33f3,
  [0x0e60] = 0x33f4, [0x0e61] = 0x33f5, [0x0e62] = 0x33f6, [0x0e63] = 0x33f7,
  [0x0e64] = 0x33f8, [0x0e65] = 0x33f9, [0x0e66] = 0x33fa, [0x0e67] = 0x33fb,
  [0x0e68] = 0x33fc, [0x0e69] = 0x33fd, [0x0e6a] = 0x33fe, [0x0e6b] = 0x33ff,
  [0x0e6c] = 0x3400, [0x0e6d] = 0x3401, [0x0e6e] = 0x3402, [0x0e6f] = 0x3403,
  [0x0e70] = 0x3404, [0x0e71] = 0x3405, [0x0e72] = 0x3406, [0x0e73] = 0x3407,
  [0x0e74] = 0x3408, [0x0e75] = 0x3409, [0x0e76] = 0x340a, [0x0e77] = 0x340b,
  [0x0e78] = 0x340c, [0x0e79] = 0x340d, [0x0e7a] = 0x340e, [0x0e7b] = 0x340f,
  [0x0e7c] = 0x3410, [0x0e7d] = 0x3411, [0x0e7e] = 0x3412, [0x0e7f] = 0x3413,
  [0x0e80] = 0x3414, [0x0e81] = 0x3415, [0x0e82] = 0x3416, [0x0e83] = 0x3417,
  [0x0e84] = 0x3418, [0x0e85] = 0x3419, [0x0e86] = 0x341a, [0x0e87] = 0x341b,
  [0x0e88] = 0x341c, [0x0e89] = 0x341d, [0x0e8a] = 0x341e, [0x0e8b] = 0x341f,
  [0x0e8c] = 0x3420, [0x0e8d] = 0x3421, [0x0e8e] = 0x3422, [0x0e8f] = 0x3423,
  [0x0e90] = 0x3424, [0x0e91] = 0x3425, [0x0e92] = 0x3426, [0x0e93] = 0x3427,
  [0x0e94] = 0x3428, [0x0e95] = 0x3429, [0x0e96] = 0x342a, [0x0e97] = 0x342b,
  [0x0e98] = 0x342c, [0x0e99] = 0x342d, [0x0e9a] = 0x342e, [0x0e9b] = 0x342f,
  [0x0e9c] = 0x3430, [0x0e9d] = 0x3431, [0x0e9e] = 0x3432, [0x0e9f] = 0x3433,
  [0x0ea0] = 0x3434, [0x0ea1] = 0x3435, [0x0ea2] = 0x3436, [0x0ea3] = 0x3437,
  [0x0ea4] = 0x3438, [0x0ea5] = 0x3439, [0x0ea6] = 0x343a, [0x0ea7] = 0x343b,
  [0x0ea8] = 0x343c, [0x0ea9] = 0x343d, [0x0eaa] = 0x343e, [0x0eab] = 0x343f,
  [0x0eac] = 0x3440, [0x0ead] = 0x3441, [0x0eae] = 0x3442, [0x0eaf] = 0x3443,
  [0x0eb0] = 0x3444, [0x0eb1] = 0x3445, [0x0eb2] = 0x3446, [0x0eb3] = 0x3448,
  [0x0eb4] = 0x3449, [0x0eb5] = 0x344a, [0x0eb6] = 0x344b, [0x0eb7] = 0x344c,
  [0x0eb8] = 0x344d, [0x0eb9] = 0x344e, [0x0eba] = 0x344f, [0x0ebb] = 0x3450,
  [0x0ebc] = 0x3451, [0x0ebd] = 0x3452, [0x0ebe] = 0x3453, [0x0ebf] = 0x3454,
  [0x0ec0] = 0x3455, [0x0ec1] = 0x3456, [0x0ec2] = 0x3457, [0x0ec3] = 0x3458,
  [0x0ec4] = 0x3459, [0x0ec5] = 0x345a, [0x0ec6] = 0x345b, [0x0ec7] = 0x345c,
  [0x0ec8] = 0x345d, [0x0ec9] = 0x345e, [0x0eca] = 0x345f, [0x0ecb] = 0x3460,
  [0x0ecc] = 0x3461, [0x0ecd] = 0x3462, [0x0ece] = 0x3463, [0x0ecf] = 0x3464,
  [0x0ed0] = 0x3465, [0x0ed1] = 0x3466, [0x0ed2] = 0x3467, [0x0ed3] = 0x3468,
  [0x0ed4] = 0x3469, [0x0ed5] = 0x346a, [0x0ed6] = 0x346b, [0x0ed7] = 0x346c,
  [0x0ed8] = 0x346d, [0x0ed9] = 0x346e, [0x0eda] = 0x346f, [0x0edb] = 0x3470,
  [0x0edc] = 0x3471, [0x0edd] = 0x3472, [0x0ede] = 0x3474, [0x0edf] = 0x3475,
  [0x0ee0] = 0x3476, [0x0ee1] = 0x3477, [0x0ee2] = 0x3478, [0x0ee3] = 0x3479,
  [0x0ee4] = 0x347a, [0x0ee5] = 0x347b, [0x0ee6] = 0x347c, [0x0ee7] = 0x347d,
  [0x0ee8] = 0x347e, [0x0ee9] = 0x347f, [0x0eea] = 0x3480, [0x0eeb] = 0x3481,
  [0x0eec] = 0x3482, [0x0eed] = 0x3483, [0x0eee] = 0x3484, [0x0eef] = 0x3485,
  [0x0ef0] = 0x3486, [0x0ef1] = 0x3487, [0x0ef2] = 0x3488, [0x0ef3] = 0x3489,
  [0x0ef4] = 0x348a, [0x0ef5] = 0x348b, [0x0ef6] = 0x348c, [0x0ef7] = 0x348d,
  [0x0ef8] = 0x348e, [0x0ef9] = 0x348f, [0x0efa] = 0x3490, [0x0efb] = 0x3491,
  [0x0efc] = 0x3492, [0x0efd] = 0x3493, [0x0efe] = 0x3494, [0x0eff] = 0x3495,
  [0x0f00] = 0x3496, [0x0f01] = 0x3497, [0x0f02] = 0x3498, [0x0f03] = 0x3499,
  [0x0f04] = 0x349a, [0x0f05] = 0x349b, [0x0f06] = 0x349c, [0x0f07] = 0x349d,
  [0x0f08] = 0x349e, [0x0f09] = 0x349f, [0x0f0a] = 0x34a0, [0x0f0b] = 0x34a1,
  [0x0f0c] = 0x34a2, [0x0f0d] = 0x34a3, [0x0f0e] = 0x34a4, [0x0f0f] = 0x34a5,
  [0x0f10] = 0x34a6, [0x0f11] = 0x34a7, [0x0f12] = 0x34a8, [0x0f13] = 0x34a9,
  [0x0f14] = 0x34aa, [0x0f15] = 0x34ab, [0x0f16] = 0x34ac, [0x0f17] = 0x34ad,
  [0x0f18] = 0x34ae, [0x0f19] = 0x34af, [0x0f1a] = 0x34b0, [0x0f1b] = 0x34b1,
  [0x0f1c] = 0x34b2, [0x0f1d] = 0x34b3, [0x0f1e] = 0x34b4, [0x0f1f] = 0x34b5,
  [0x0f20] = 0x34b6, [0x0f21] = 0x34b7, [0x0f22] = 0x34b8, [0x0f23] = 0x34b9,
  [0x0f24] = 0x34ba, [0x0f25] = 0x34bb, [0x0f26] = 0x34bc, [0x0f27] = 0x34bd,
  [0x0f28] = 0x34be, [0x0f29] = 0x34bf, [0x0f2a] = 0x34c0, [0x0f2b] = 0x34c1,
  [0x0f2c] = 0x34c2, [0x0f2d] = 0x34c3, [0x0f2e] = 0x34c4, [0x0f2f] = 0x34c5,
  [0x0f30] = 0x34c6, [0x0f31] = 0x34c7, [0x0f32] = 0x34c8, [0x0f33] = 0x34c9,
  [0x0f34] = 0x34ca, [0x0f35] = 0x34cb, [0x0f36] = 0x34cc, [0x0f37] = 0x34cd,
  [0x0f38] = 0x34ce, [0x0f39] = 0x34cf, [0x0f3a] = 0x34d0, [0x0f3b] = 0x34d1,
  [0x0f3c] = 0x34d2, [0x0f3d] = 0x34d3, [0x0f3e] = 0x34d4, [0x0f3f] = 0x34d5,
  [0x0f40] = 0x34d6, [0x0f41] = 0x34d7, [0x0f42] = 0x34d8, [0x0f43] = 0x34d9,
  [0x0f44] = 0x34da, [0x0f45] = 0x34db, [0x0f46] = 0x34dc, [0x0f47] = 0x34dd,
  [0x0f48] = 0x34de, [0x0f49] = 0x34df, [0x0f4a] = 0x34e0, [0x0f4b] = 0x34e1,
  [0x0f4c] = 0x34e2, [0x0f4d] = 0x34e3, [0x0f4e] = 0x34e4, [0x0f4f] = 0x34e5,
  [0x0f50] = 0x34e6, [0x0f51] = 0x34e7, [0x0f52] = 0x34e8, [0x0f53] = 0x34e9,
  [0x0f54] = 0x34ea, [0x0f55] = 0x34eb, [0x0f56] = 0x34ec, [0x0f57] = 0x34ed,
  [0x0f58] = 0x34ee, [0x0f59] = 0x34ef, [0x0f5a] = 0x34f0, [0x0f5b] = 0x34f1,
  [0x0f5c] = 0x34f2, [0x0f5d] = 0x34f3, [0x0f5e] = 0x34f4, [0x0f5f] = 0x34f5,
  [0x0f60] = 0x34f6, [0x0f61] = 0x34f7, [0x0f62] = 0x34f8, [0x0f63] = 0x34f9,
  [0x0f64] = 0x34fa, [0x0f65] = 0x34fb, [0x0f66] = 0x34fc, [0x0f67] = 0x34fd,
  [0x0f68] = 0x34fe, [0x0f69] = 0x34ff, [0x0f6a] = 0x3500, [0x0f6b] = 0x3501,
  [0x0f6c] = 0x3502, [0x0f6d] = 0x3503, [0x0f6e] = 0x3504, [0x0f6f] = 0x3505,
  [0x0f70] = 0x3506, [0x0f71] = 0x3507, [0x0f72] = 0x3508, [0x0f73] = 0x3509,
  [0x0f74] = 0x350a, [0x0f75] = 0x350b, [0x0f76] = 0x350c, [0x0f77] = 0x350d,
  [0x0f78] = 0x350e, [0x0f79] = 0x350f, [0x0f7a] = 0x3510, [0x0f7b] = 0x3511,
  [0x0f7c] = 0x3512, [0x0f7d] = 0x3513, [0x0f7e] = 0x3514, [0x0f7f] = 0x3515,
  [0x0f80] = 0x3516, [0x0f81] = 0x3517, [0x0f82] = 0x3518, [0x0f83] = 0x3519,
  [0x0f84] = 0x351a, [0x0f85] = 0x351b, [0x0f86] = 0x351c, [0x0f87] = 0x351d,
  [0x0f88] = 0x351e, [0x0f89] = 0x351f, [0x0f8a] = 0x3520, [0x0f8b] = 0x3521,
  [0x0f8c] = 0x3522, [0x0f8d] = 0x3523, [0x0f8e] = 0x3524, [0x0f8f] = 0x3525,
  [0x0f90] = 0x3526, [0x0f91] = 0x3527, [0x0f92] = 0x3528, [0x0f93] = 0x3529,
  [0x0f94] = 0x352a, [0x0f95] = 0x352b, [0x0f96] = 0x352c, [0x0f97] = 0x352d,
  [0x0f98] = 0x352e, [0x0f99] = 0x352f, [0x0f9a] = 0x3530, [0x0f9b] = 0x3531,
  [0x0f9c] = 0x3532, [0x0f9d] = 0x3533, [0x0f9e] = 0x3534, [0x0f9f] = 0x3535,
  [0x0fa0] = 0x3536, [0x0fa1] = 0x3537, [0x0fa2] = 0x3538, [0x0fa3] = 0x3539,
  [0x0fa4] = 0x353a, [0x0fa5] = 0x353b, [0x0fa6] = 0x353c, [0x0fa7] = 0x353d,
  [0x0fa8] = 0x353e, [0x0fa9] = 0x353f, [0x0faa] = 0x3540, [0x0fab] = 0x3541,
  [0x0fac] = 0x3542, [0x0fad] = 0x3543, [0x0fae] = 0x3544, [0x0faf] = 0x3545,
  [0x0fb0] = 0x3546, [0x0fb1] = 0x3547, [0x0fb2] = 0x3548, [0x0fb3] = 0x3549,
  [0x0fb4] = 0x354a, [0x0fb5] = 0x354b, [0x0fb6] = 0x354c, [0x0fb7] = 0x354d,
  [0x0fb8] = 0x354e, [0x0fb9] = 0x354f, [0x0fba] = 0x3550, [0x0fbb] = 0x3551,
  [0x0fbc] = 0x3552, [0x0fbd] = 0x3553, [0x0fbe] = 0x3554, [0x0fbf] = 0x3555,
  [0x0fc0] = 0x3556, [0x0fc1] = 0x3557, [0x0fc2] = 0x3558, [0x0fc3] = 0x3559,
  [0x0fc4] = 0x355a, [0x0fc5] = 0x355b, [0x0fc6] = 0x355c, [0x0fc7] = 0x355d,
  [0x0fc8] = 0x355e, [0x0fc9] = 0x355f, [0x0fca] = 0x3560, [0x0fcb] = 0x3561,
  [0x0fcc] = 0x3562, [0x0fcd] = 0x3563, [0x0fce] = 0x3564, [0x0fcf] = 0x3565,
  [0x0fd0] = 0x3566, [0x0fd1] = 0x3567, [0x0fd2] = 0x3568, [0x0fd3] = 0x3569,
  [0x0fd4] = 0x356a, [0x0fd5] = 0x356b, [0x0fd6] = 0x356c, [0x0fd7] = 0x356d,
  [0x0fd8] = 0x356e, [0x0fd9] = 0x356f, [0x0fda] = 0x3570, [0x0fdb] = 0x3571,
  [0x0fdc] = 0x3572, [0x0fdd] = 0x3573, [0x0fde] = 0x3574, [0x0fdf] = 0x3575,
  [0x0fe0] = 0x3576, [0x0fe1] = 0x3577, [0x0fe2] = 0x3578, [0x0fe3] = 0x3579,
  [0x0fe4] = 0x357a, [0x0fe5] = 0x357b, [0x0fe6] = 0x357c, [0x0fe7] = 0x357d,
  [0x0fe8] = 0x357e, [0x0fe9] = 0x357f, [0x0fea] = 0x3580, [0x0feb] = 0x3581,
  [0x0fec] = 0x3582, [0x0fed] = 0x3583, [0x0fee] = 0x3584, [0x0fef] = 0x3585,
  [0x0ff0] = 0x3586, [0x0ff1] = 0x3587, [0x0ff2] = 0x3588, [0x0ff3] = 0x3589,
  [0x0ff4] = 0x358a, [0x0ff5] = 0x358b, [0x0ff6] = 0x358c, [0x0ff7] = 0x358d,
  [0x0ff8] = 0x358e, [0x0ff9] = 0x358f, [0x0ffa] = 0x3590, [0x0ffb] = 0x3591,
  [0x0ffc] = 0x3592, [0x0ffd] = 0x3593, [0x0ffe] = 0x3594, [0x0fff] = 0x3595,
  [0x1000] = 0x3596, [0x1001] = 0x3597, [0x1002] = 0x3598, [0x1003] = 0x3599,
  [0x1004] = 0x359a, [0x1005] = 0x359b, [0x1006] = 0x359c, [0x1007] = 0x359d,
  [0x1008] = 0x359f, [0x1009] = 0x35a0, [0x100a] = 0x35a1, [0x100b] = 0x35a2,
  [0x100c] = 0x35a3, [0x100d] = 0x35a4, [0x100e] = 0x35a5, [0x100f] = 0x35a6,
  [0x1010] = 0x35a7, [0x1011] = 0x35a8, [0x1012] = 0x35a9, [0x1013] = 0x35aa,
  [0x1014] = 0x35ab, [0x1015] = 0x35ac, [0x1016] = 0x35ad, [0x1017] = 0x35ae,
  [0x1018] = 0x35af, [0x1019] = 0x35b0, [0x101a] = 0x35b1, [0x101b] = 0x35b2,
  [0x101c] = 0x35b3, [0x101d] = 0x35b4, [0x101e] = 0x35b5, [0x101f] = 0x35b6,
  [0x1020] = 0x35b7, [0x1021] = 0x35b8, [0x1022] = 0x35b9, [0x1023] = 0x35ba,
  [0x1024] = 0x35bb, [0x1025] = 0x35bc, [0x1026] = 0x35bd, [0x1027] = 0x35be,
  [0x1028] = 0x35bf, [0x1029] = 0x35c0, [0x102a] = 0x35c1, [0x102b] = 0x35c2,
  [0x102c] = 0x35c3, [0x102d] = 0x35c4, [0x102e] = 0x35c5, [0x102f] = 0x35c6,
  [0x1030] = 0x35c7, [0x1031] = 0x35c8, [0x1032] = 0x35c9, [0x1033] = 0x35ca,
  [0x1034] = 0x35cb, [0x1035] = 0x35cc, [0x1036] = 0x35cd, [0x1037] = 0x35ce,
  [0x1038] = 0x35cf, [0x1039] = 0x35d0, [0x103a] = 0x35d1, [0x103b] = 0x35d2,
  [0x103c] = 0x35d3, [0x103d] = 0x35d4, [0x103e] = 0x35d5, [0x103f] = 0x35d6,
  [0x1040] = 0x35d7, [0x1041] = 0x35d8, [0x1042] = 0x35d9, [0x1043] = 0x35da,
  [0x1044] = 0x35db, [0x1045] = 0x35dc, [0x1046] = 0x35dd, [0x1047] = 0x35de,
  [0x1048] = 0x35df, [0x1049] = 0x35e0, [0x104a] = 0x35e1, [0x104b] = 0x35e2,
  [0x104c] = 0x35e3, [0x104d] = 0x35e4, [0x104e] = 0x35e5, [0x104f] = 0x35e6,
  [0x1050] = 0x35e7, [0x1051] = 0x35e8, [0x1052] = 0x35e9, [0x1053] = 0x35ea,
  [0x1054] = 0x35eb, [0x1055] = 0x35ec, [0x1056] = 0x35ed, [0x1057] = 0x35ee,
  [0x1058] = 0x35ef, [0x1059] = 0x35f0, [0x105a] = 0x35f1, [0x105b] = 0x35f2,
  [0x105c] = 0x35f3, [0x105d] = 0x35f4, [0x105e] = 0x35f5, [0x105f] = 0x35f6,
  [0x1060] = 0x35f7, [0x1061] = 0x35f8, [0x1062] = 0x35f9, [0x1063] = 0x35fa,
  [0x1064] = 0x35fb, [0x1065] = 0x35fc, [0x1066] = 0x35fd, [0x1067] = 0x35fe,
  [0x1068] = 0x35ff, [0x1069] = 0x3600, [0x106a] = 0x3601, [0x106b] = 0x3602,
  [0x106c] = 0x3603, [0x106d] = 0x3604, [0x106e] = 0x3605, [0x106f] = 0x3606,
  [0x1070] = 0x3607, [0x1071] = 0x3608, [0x1072] = 0x3609, [0x1073] = 0x360a,
  [0x1074] = 0x360b, [0x1075] = 0x360c, [0x1076] = 0x360d, [0x1077] = 0x360f,
  [0x1078] = 0x3610, [0x1079] = 0x3611, [0x107a] = 0x3612, [0x107b] = 0x3613,
  [0x107c] = 0x3614, [0x107d] = 0x3615, [0x107e] = 0x3616, [0x107f] = 0x3617,
  [0x1080] = 0x3618, [0x1081] = 0x3619, [0x1082] = 0x361b, [0x1083] = 0x361c,
  [0x1084] = 0x361d, [0x1085] = 0x361e, [0x1086] = 0x361f, [0x1087] = 0x3620,
  [0x1088] = 0x3621, [0x1089] = 0x3622, [0x108a] = 0x3623, [0x108b] = 0x3624,
  [0x108c] = 0x3625, [0x108d] = 0x3626, [0x108e] = 0x3627, [0x108f] = 0x3628,
  [0x1090] = 0x3629, [0x1091] = 0x362a, [0x1092] = 0x362b, [0x1093] = 0x362c,
  [0x1094] = 0x362d, [0x1095] = 0x362e, [0x1096] = 0x362f, [0x1097] = 0x3630,
  [0x1098] = 0x3631, [0x1099] = 0x3632, [0x109a] = 0x3633, [0x109b] = 0x3634,
  [0x109c] = 0x3635, [0x109d] = 0x3636, [0x109e] = 0x3637, [0x109f] = 0x3638,
  [0x10a0] = 0x3639, [0x10a1] = 0x363a, [0x10a2] = 0x363b, [0x10a3] = 0x363c,
  [0x10a4] = 0x363d, [0x10a5] = 0x363e, [0x10a6] = 0x363f, [0x10a7] = 0x3640,
  [0x10a8] = 0x3641, [0x10a9] = 0x3642, [0x10aa] = 0x3643, [0x10ab] = 0x3644,
  [0x10ac] = 0x3645, [0x10ad] = 0x3646, [0x10ae] = 0x3647, [0x10af] = 0x3648,
  [0x10b0] = 0x3649, [0x10b1] = 0x364a, [0x10b2] = 0x364b, [0x10b3] = 0x364c,
  [0x10b4] = 0x364d, [0x10b5] = 0x364e, [0x10b6] = 0x364f, [0x10b7] = 0x3650,
  [0x10b8] = 0x3651, [0x10b9] = 0x3652, [0x10ba] = 0x3653, [0x10bb] = 0x3654,
  [0x10bc] = 0x3655, [0x10bd] = 0x3656, [0x10be] = 0x3657, [0x10bf] = 0x3658,
  [0x10c0] = 0x3659, [0x10c1] = 0x365a, [0x10c2] = 0x365b, [0x10c3] = 0x365c,
  [0x10c4] = 0x365d, [0x10c5] = 0x365e, [0x10c6] = 0x365f, [0x10c7] = 0x3660,
  [0x10c8] = 0x3661, [0x10c9] = 0x3662, [0x10ca] = 0x3663, [0x10cb] = 0x3664,
  [0x10cc] = 0x3665, [0x10cd] = 0x3666, [0x10ce] = 0x3667, [0x10cf] = 0x3668,
  [0x10d0] = 0x3669, [0x10d1] = 0x366a, [0x10d2] = 0x366b, [0x10d3] = 0x366c,
  [0x10d4] = 0x366d, [0x10d5] = 0x366e, [0x10d6] = 0x366f, [0x10d7] = 0x3670,
  [0x10d8] = 0x3671, [0x10d9] = 0x3672, [0x10da] = 0x3673, [0x10db] = 0x3674,
  [0x10dc] = 0x3675, [0x10dd] = 0x3676, [0x10de] = 0x3677, [0x10df] = 0x3678,
  [0x10e0] = 0x3679, [0x10e1] = 0x367a, [0x10e2] = 0x367b, [0x10e3] = 0x367c,
  [0x10e4] = 0x367d, [0x10e5] = 0x367e, [0x10e6] = 0x367f, [0x10e7] = 0x3680,
  [0x10e8] = 0x3681, [0x10e9] = 0x3682, [0x10ea] = 0x3683, [0x10eb] = 0x3684,
  [0x10ec] = 0x3685, [0x10ed] = 0x3686, [0x10ee] = 0x3687, [0x10ef] = 0x3688,
  [0x10f0] = 0x3689, [0x10f1] = 0x368a, [0x10f2] = 0x368b, [0x10f3] = 0x368c,
  [0x10f4] = 0x368d, [0x10f5] = 0x368e, [0x10f6] = 0x368f, [0x10f7] = 0x3690,
  [0x10f8] = 0x3691, [0x10f9] = 0x3692, [0x10fa] = 0x3693, [0x10fb] = 0x3694,
  [0x10fc] = 0x3695, [0x10fd] = 0x3696, [0x10fe] = 0x3697, [0x10ff] = 0x3698,
  [0x1100] = 0x3699, [0x1101] = 0x369a, [0x1102] = 0x369b, [0x1103] = 0x369c,
  [0x1104] = 0x369d, [0x1105] = 0x369e, [0x1106] = 0x369f, [0x1107] = 0x36a0,
  [0x1108] = 0x36a1, [0x1109] = 0x36a2, [0x110a] = 0x36a3, [0x110b] = 0x36a4,
  [0x110c] = 0x36a5, [0x110d] = 0x36a6, [0x110e] = 0x36a7, [0x110f] = 0x36a8,
  [0x1110] = 0x36a9, [0x1111] = 0x36aa, [0x1112] = 0x36ab, [0x1113] = 0x36ac,
  [0x1114] = 0x36ad, [0x1115] = 0x36ae, [0x1116] = 0x36af, [0x1117] = 0x36b0,
  [0x1118] = 0x36b1, [0x1119] = 0x36b2, [0x111a] = 0x36b3, [0x111b] = 0x36b4,
  [0x111c] = 0x36b5, [0x111d] = 0x36b6, [0x111e] = 0x36b7, [0x111f] = 0x36b8,
  [0x1120] = 0x36b9, [0x1121] = 0x36ba, [0x1122] = 0x36bb, [0x1123] = 0x36bc,
  [0x1124] = 0x36bd, [0x1125] = 0x36be, [0x1126] = 0x36bf, [0x1127] = 0x36c0,
  [0x1128] = 0x36c1, [0x1129] = 0x36c2, [0x112a] = 0x36c3, [0x112b] = 0x36c4,
  [0x112c] = 0x36c5, [0x112d] = 0x36c6, [0x112e] = 0x36c7, [0x112f] = 0x36c8,
  [0x1130] = 0x36c9, [0x1131] = 0x36ca, [0x1132] = 0x36cb, [0x1133] = 0x36cc,
  [0x1134] = 0x36cd, [0x1135] = 0x36ce, [0x1136] = 0x36cf, [0x1137] = 0x36d0,
  [0x1138] = 0x36d1, [0x1139] = 0x36d2, [0x113a] = 0x36d3, [0x113b] = 0x36d4,
  [0x113c] = 0x36d5, [0x113d] = 0x36d6, [0x113e] = 0x36d7, [0x113f] = 0x36d8,
  [0x1140] = 0x36d9, [0x1141] = 0x36da, [0x1142] = 0x36db, [0x1143] = 0x36dc,
  [0x1144] = 0x36dd, [0x1145] = 0x36de, [0x1146] = 0x36df, [0x1147] = 0x36e0,
  [0x1148] = 0x36e1, [0x1149] = 0x36e2, [0x114a] = 0x36e3, [0x114b] = 0x36e4,
  [0x114c] = 0x36e5, [0x114d] = 0x36e6, [0x114e] = 0x36e7, [0x114f] = 0x36e8,
  [0x1150] = 0x36e9, [0x1151] = 0x36ea, [0x1152] = 0x36eb, [0x1153] = 0x36ec,
  [0x1154] = 0x36ed, [0x1155] = 0x36ee, [0x1156] = 0x36ef, [0x1157] = 0x36f0,
  [0x1158] = 0x36f1, [0x1159] = 0x36f2, [0x115a] = 0x36f3, [0x115b] = 0x36f4,
  [0x115c] = 0x36f5, [0x115d] = 0x36f6, [0x115e] = 0x36f7, [0x115f] = 0x36f8,
  [0x1160] = 0x36f9, [0x1161] = 0x36fa, [0x1162] = 0x36fb, [0x1163] = 0x36fc,
  [0x1164] = 0x36fd, [0x1165] = 0x36fe, [0x1166] = 0x36ff, [0x1167] = 0x3700,
  [0x1168] = 0x3701, [0x1169] = 0x3702, [0x116a] = 0x3703, [0x116b] = 0x3704,
  [0x116c] = 0x3705, [0x116d] = 0x3706, [0x116e] = 0x3707, [0x116f] = 0x3708,
  [0x1170] = 0x3709, [0x1171] = 0x370a, [0x1172] = 0x370b, [0x1173] = 0x370c,
  [0x1174] = 0x370d, [0x1175] = 0x370e, [0x1176] = 0x370f, [0x1177] = 0x3710,
  [0x1178] = 0x3711, [0x1179] = 0x3712, [0x117a] = 0x3713, [0x117b] = 0x3714,
  [0x117c] = 0x3715, [0x117d] = 0x3716, [0x117e] = 0x3717, [0x117f] = 0x3718,
  [0x1180] = 0x3719, [0x1181] = 0x371a, [0x1182] = 0x371b, [0x1183] = 0x371c,
  [0x1184] = 0x371d, [0x1185] = 0x371e, [0x1186] = 0x371f, [0x1187] = 0x3720,
  [0x1188] = 0x3721, [0x1189] = 0x3722, [0x118a] = 0x3723, [0x118b] = 0x3724,
  [0x118c] = 0x3725, [0x118d] = 0x3726, [0x118e] = 0x3727, [0x118f] = 0x3728,
  [0x1190] = 0x3729, [0x1191] = 0x372a, [0x1192] = 0x372b, [0x1193] = 0x372c,
  [0x1194] = 0x372d, [0x1195] = 0x372e, [0x1196] = 0x372f, [0x1197] = 0x3730,
  [0x1198] = 0x3731, [0x1199] = 0x3732, [0x119a] = 0x3733, [0x119b] = 0x3734,
  [0x119c] = 0x3735, [0x119d] = 0x3736, [0x119e] = 0x3737, [0x119f] = 0x3738,
  [0x11a0] = 0x3739, [0x11a1] = 0x373a, [0x11a2] = 0x373b, [0x11a3] = 0x373c,
  [0x11a4] = 0x373d, [0x11a5] = 0x373e, [0x11a6] = 0x373f, [0x11a7] = 0x3740,
  [0x11a8] = 0x3741, [0x11a9] = 0x3742, [0x11aa] = 0x3743, [0x11ab] = 0x3744,
  [0x11ac] = 0x3745, [0x11ad] = 0x3746, [0x11ae] = 0x3747, [0x11af] = 0x3748,
  [0x11b0] = 0x3749, [0x11b1] = 0x374a, [0x11b2] = 0x374b, [0x11b3] = 0x374c,
  [0x11b4] = 0x374d, [0x11b5] = 0x374e, [0x11b6] = 0x374f, [0x11b7] = 0x3750,
  [0x11b8] = 0x3751, [0x11b9] = 0x3752, [0x11ba] = 0x3753, [0x11bb] = 0x3754,
  [0x11bc] = 0x3755, [0x11bd] = 0x3756, [0x11be] = 0x3757, [0x11bf] = 0x3758,
  [0x11c0] = 0x3759, [0x11c1] = 0x375a, [0x11c2] = 0x375b, [0x11c3] = 0x375c,
  [0x11c4] = 0x375d, [0x11c5] = 0x375e, [0x11c6] = 0x375f, [0x11c7] = 0x3760,
  [0x11c8] = 0x3761, [0x11c9] = 0x3762, [0x11ca] = 0x3763, [0x11cb] = 0x3764,
  [0x11cc] = 0x3765, [0x11cd] = 0x3766, [0x11ce] = 0x3767, [0x11cf] = 0x3768,
  [0x11d0] = 0x3769, [0x11d1] = 0x376a, [0x11d2] = 0x376b, [0x11d3] = 0x376c,
  [0x11d4] = 0x376d, [0x11d5] = 0x376e, [0x11d6] = 0x376f, [0x11d7] = 0x3770,
  [0x11d8] = 0x3771, [0x11d9] = 0x3772, [0x11da] = 0x3773, [0x11db] = 0x3774,
  [0x11dc] = 0x3775, [0x11dd] = 0x3776, [0x11de] = 0x3777, [0x11df] = 0x3778,
  [0x11e0] = 0x3779, [0x11e1] = 0x377a, [0x11e2] = 0x377b, [0x11e3] = 0x377c,
  [0x11e4] = 0x377d, [0x11e5] = 0x377e, [0x11e6] = 0x377f, [0x11e7] = 0x3780,
  [0x11e8] = 0x3781, [0x11e9] = 0x3782, [0x11ea] = 0x3783, [0x11eb] = 0x3784,
  [0x11ec] = 0x3785, [0x11ed] = 0x3786, [0x11ee] = 0x3787, [0x11ef] = 0x3788,
  [0x11f0] = 0x3789, [0x11f1] = 0x378a, [0x11f2] = 0x378b, [0x11f3] = 0x378c,
  [0x11f4] = 0x378d, [0x11f5] = 0x378e, [0x11f6] = 0x378f, [0x11f7] = 0x3790,
  [0x11f8] = 0x3791, [0x11f9] = 0x3792, [0x11fa] = 0x3793, [0x11fb] = 0x3794,
  [0x11fc] = 0x3795, [0x11fd] = 0x3796, [0x11fe] = 0x3797, [0x11ff] = 0x3798,
  [0x1200] = 0x3799, [0x1201] = 0x379a, [0x1202] = 0x379b, [0x1203] = 0x379c,
  [0x1204] = 0x379d, [0x1205] = 0x379e, [0x1206] = 0x379f, [0x1207] = 0x37a0,
  [0x1208] = 0x37a1, [0x1209] = 0x37a2, [0x120a] = 0x37a3, [0x120b] = 0x37a4,
  [0x120c] = 0x37a5, [0x120d] = 0x37a6, [0x120e] = 0x37a7, [0x120f] = 0x37a8,
  [0x1210] = 0x37a9, [0x1211] = 0x37aa, [0x1212] = 0x37ab, [0x1213] = 0x37ac,
  [0x1214] = 0x37ad, [0x1215] = 0x37ae, [0x1216] = 0x37af, [0x1217] = 0x37b0,
  [0x1218] = 0x37b1, [0x1219] = 0x37b2, [0x121a] = 0x37b3, [0x121b] = 0x37b4,
  [0x121c] = 0x37b5, [0x121d] = 0x37b6, [0x121e] = 0x37b7, [0x121f] = 0x37b8,
  [0x1220] = 0x37b9, [0x1221] = 0x37ba, [0x1222] = 0x37bb, [0x1223] = 0x37bc,
  [0x1224] = 0x37bd, [0x1225] = 0x37be, [0x1226] = 0x37bf, [0x1227] = 0x37c0,
  [0x1228] = 0x37c1, [0x1229] = 0x37c2, [0x122a] = 0x37c3, [0x122b] = 0x37c4,
  [0x122c] = 0x37c5, [0x122d] = 0x37c6, [0x122e] = 0x37c7, [0x122f] = 0x37c8,
  [0x1230] = 0x37c9, [0x1231] = 0x37ca, [0x1232] = 0x37cb, [0x1233] = 0x37cc,
  [0x1234] = 0x37cd, [0x1235] = 0x37ce, [0x1236] = 0x37cf, [0x1237] = 0x37d0,
  [0x1238] = 0x37d1, [0x1239] = 0x37d2, [0x123a] = 0x37d3, [0x123b] = 0x37d4,
  [0x123c] = 0x37d5, [0x123d] = 0x37d6, [0x123e] = 0x37d7, [0x123f] = 0x37d8,
  [0x1240] = 0x37d9, [0x1241] = 0x37da, [0x1242] = 0x37db, [0x1243] = 0x37dc,
  [0x1244] = 0x37dd, [0x1245] = 0x37de, [0x1246] = 0x37df, [0x1247] = 0x37e0,
  [0x1248] = 0x37e1, [0x1249] = 0x37e2, [0x124a] = 0x37e3, [0x124b] = 0x37e4,
  [0x124c] = 0x37e5, [0x124d] = 0x37e6, [0x124e] = 0x37e7, [0x124f] = 0x37e8,
  [0x1250] = 0x37e9, [0x1251] = 0x37ea, [0x1252] = 0x37eb, [0x1253] = 0x37ec,
  [0x1254] = 0x37ed, [0x1255] = 0x37ee, [0x1256] = 0x37ef, [0x1257] = 0x37f0,
  [0x1258] = 0x37f1, [0x1259] = 0x37f2, [0x125a] = 0x37f3, [0x125b] = 0x37f4,
  [0x125c] = 0x37f5, [0x125d] = 0x37f6, [0x125e] = 0x37f7, [0x125f] = 0x37f8,
  [0x1260] = 0x37f9, [0x1261] = 0x37fa, [0x1262] = 0x37fb, [0x1263] = 0x37fc,
  [0x1264] = 0x37fd, [0x1265] = 0x37fe, [0x1266] = 0x37ff, [0x1267] = 0x3800,
  [0x1268] = 0x3801, [0x1269] = 0x3802, [0x126a] = 0x3803, [0x126b] = 0x3804,
  [0x126c] = 0x3805, [0x126d] = 0x3806, [0x126e] = 0x3807, [0x126f] = 0x3808,
  [0x1270] = 0x3809, [0x1271] = 0x380a, [0x1272] = 0x380b, [0x1273] = 0x380c,
  [0x1274] = 0x380d, [0x1275] = 0x380e, [0x1276] = 0x380f, [0x1277] = 0x3810,
  [0x1278] = 0x3811, [0x1279] = 0x3812, [0x127a] = 0x3813, [0x127b] = 0x3814,
  [0x127c] = 0x3815, [0x127d] = 0x3816, [0x127e] = 0x3817, [0x127f] = 0x3818,
  [0x1280] = 0x3819, [0x1281] = 0x381a, [0x1282] = 0x381b, [0x1283] = 0x381c,
  [0x1284] = 0x381d, [0x1285] = 0x381e, [0x1286] = 0x381f, [0x1287] = 0x3820,
  [0x1288] = 0x3821, [0x1289] = 0x3822, [0x128a] = 0x3823, [0x128b] = 0x3824,
  [0x128c] = 0x3825, [0x128d] = 0x3826, [0x128e] = 0x3827, [0x128f] = 0x3828,
  [0x1290] = 0x3829, [0x1291] = 0x382a, [0x1292] = 0x382b, [0x1293] = 0x382c,
  [0x1294] = 0x382d, [0x1295] = 0x382e, [0x1296] = 0x382f, [0x1297] = 0x3830,
  [0x1298] = 0x3831, [0x1299] = 0x3832, [0x129a] = 0x3833, [0x129b] = 0x3834,
  [0x129c] = 0x3835, [0x129d] = 0x3836, [0x129e] = 0x3837, [0x129f] = 0x3838,
  [0x12a0] = 0x3839, [0x12a1] = 0x383a, [0x12a2] = 0x383b, [0x12a3] = 0x383c,
  [0x12a4] = 0x383d, [0x12a5] = 0x383e, [0x12a6] = 0x383f, [0x12a7] = 0x3840,
  [0x12a8] = 0x3841, [0x12a9] = 0x3842, [0x12aa] = 0x3843, [0x12ab] = 0x3844,
  [0x12ac] = 0x3845, [0x12ad] = 0x3846, [0x12ae] = 0x3847, [0x12af] = 0x3848,
  [0x12b0] = 0x3849, [0x12b1] = 0x384a, [0x12b2] = 0x384b, [0x12b3] = 0x384c,
  [0x12b4] = 0x384d, [0x12b5] = 0x384e, [0x12b6] = 0x384f, [0x12b7] = 0x3850,
  [0x12b8] = 0x3851, [0x12b9] = 0x3852, [0x12ba] = 0x3853, [0x12bb] = 0x3854,
  [0x12bc] = 0x3855, [0x12bd] = 0x3856, [0x12be] = 0x3857, [0x12bf] = 0x3858,
  [0x12c0] = 0x3859, [0x12c1] = 0x385a, [0x12c2] = 0x385b, [0x12c3] = 0x385c,
  [0x12c4] = 0x385d, [0x12c5] = 0x385e, [0x12c6] = 0x385f, [0x12c7] = 0x3860,
  [0x12c8] = 0x3861, [0x12c9] = 0x3862, [0x12ca] = 0x3863, [0x12cb] = 0x3864,
  [0x12cc] = 0x3865, [0x12cd] = 0x3866, [0x12ce] = 0x3867, [0x12cf] = 0x3868,
  [0x12d0] = 0x3869, [0x12d1] = 0x386a, [0x12d2] = 0x386b, [0x12d3] = 0x386c,
  [0x12d4] = 0x386d, [0x12d5] = 0x386e, [0x12d6] = 0x386f, [0x12d7] = 0x3870,
  [0x12d8] = 0x3871, [0x12d9] = 0x3872, [0x12da] = 0x3873, [0x12db] = 0x3874,
  [0x12dc] = 0x3875, [0x12dd] = 0x3876, [0x12de] = 0x3877, [0x12df] = 0x3878,
  [0x12e0] = 0x3879, [0x12e1] = 0x387a, [0x12e2] = 0x387b, [0x12e3] = 0x387c,
  [0x12e4] = 0x387d, [0x12e5] = 0x387e, [0x12e6] = 0x387f, [0x12e7] = 0x3880,
  [0x12e8] = 0x3881, [0x12e9] = 0x3882, [0x12ea] = 0x3883, [0x12eb] = 0x3884,
  [0x12ec] = 0x3885, [0x12ed] = 0x3886, [0x12ee] = 0x3887, [0x12ef] = 0x3888,
  [0x12f0] = 0x3889, [0x12f1] = 0x388a, [0x12f2] = 0x388b, [0x12f3] = 0x388c,
  [0x12f4] = 0x388d, [0x12f5] = 0x388e, [0x12f6] = 0x388f, [0x12f7] = 0x3890,
  [0x12f8] = 0x3891, [0x12f9] = 0x3892, [0x12fa] = 0x3893, [0x12fb] = 0x3894,
  [0x12fc] = 0x3895, [0x12fd] = 0x3896, [0x12fe] = 0x3897, [0x12ff] = 0x3898,
  [0x1300] = 0x3899, [0x1301] = 0x389a, [0x1302] = 0x389b, [0x1303] = 0x389c,
  [0x1304] = 0x389d, [0x1305] = 0x389e, [0x1306] = 0x389f, [0x1307] = 0x38a0,
  [0x1308] = 0x38a1, [0x1309] = 0x38a2, [0x130a] = 0x38a3, [0x130b] = 0x38a4,
  [0x130c] = 0x38a5, [0x130d] = 0x38a6, [0x130e] = 0x38a7, [0x130f] = 0x38a8,
  [0x1310] = 0x38a9, [0x1311] = 0x38aa, [0x1312] = 0x38ab, [0x1313] = 0x38ac,
  [0x1314] = 0x38ad, [0x1315] = 0x38ae, [0x1316] = 0x38af, [0x1317] = 0x38b0,
  [0x1318] = 0x38b1, [0x1319] = 0x38b2, [0x131a] = 0x38b3, [0x131b] = 0x38b4,
  [0x131c] = 0x38b5, [0x131d] = 0x38b6, [0x131e] = 0x38b7, [0x131f] = 0x38b8,
  [0x1320] = 0x38b9, [0x1321] = 0x38ba, [0x1322] = 0x38bb, [0x1323] = 0x38bc,
  [0x1324] = 0x38bd, [0x1325] = 0x38be, [0x1326] = 0x38bf, [0x1327] = 0x38c0,
  [0x1328] = 0x38c1, [0x1329] = 0x38c2, [0x132a] = 0x38c3, [0x132b] = 0x38c4,
  [0x132c] = 0x38c5, [0x132d] = 0x38c6, [0x132e] = 0x38c7, [0x132f] = 0x38c8,
  [0x1330] = 0x38c9, [0x1331] = 0x38ca, [0x1332] = 0x38cb, [0x1333] = 0x38cc,
  [0x1334] = 0x38cd, [0x1335] = 0x38ce, [0x1336] = 0x38cf, [0x1337] = 0x38d0,
  [0x1338] = 0x38d1, [0x1339] = 0x38d2, [0x133a] = 0x38d3, [0x133b] = 0x38d4,
  [0x133c] = 0x38d5, [0x133d] = 0x38d6, [0x133e] = 0x38d7, [0x133f] = 0x38d8,
  [0x1340] = 0x38d9, [0x1341] = 0x38da, [0x1342] = 0x38db, [0x1343] = 0x38dc,
  [0x1344] = 0x38dd, [0x1345] = 0x38de, [0x1346] = 0x38df, [0x1347] = 0x38e0,
  [0x1348] = 0x38e1, [0x1349] = 0x38e2, [0x134a] = 0x38e3, [0x134b] = 0x38e4,
  [0x134c] = 0x38e5, [0x134d] = 0x38e6, [0x134e] = 0x38e7, [0x134f] = 0x38e8,
  [0x1350] = 0x38e9, [0x1351] = 0x38ea, [0x1352] = 0x38eb, [0x1353] = 0x38ec,
  [0x1354] = 0x38ed, [0x1355] = 0x38ee, [0x1356] = 0x38ef, [0x1357] = 0x38f0,
  [0x1358] = 0x38f1, [0x1359] = 0x38f2, [0x135a] = 0x38f3, [0x135b] = 0x38f4,
  [0x135c] = 0x38f5, [0x135d] = 0x38f6, [0x135e] = 0x38f7, [0x135f] = 0x38f8,
  [0x1360] = 0x38f9, [0x1361] = 0x38fa, [0x1362] = 0x38fb, [0x1363] = 0x38fc,
  [0x1364] = 0x38fd, [0x1365] = 0x38fe, [0x1366] = 0x38ff, [0x1367] = 0x3900,
  [0x1368] = 0x3901, [0x1369] = 0x3902, [0x136a] = 0x3903, [0x136b] = 0x3904,
  [0x136c] = 0x3905, [0x136d] = 0x3906, [0x136e] = 0x3907, [0x136f] = 0x3908,
  [0x1370] = 0x3909, [0x1371] = 0x390a, [0x1372] = 0x390b, [0x1373] = 0x390c,
  [0x1374] = 0x390d, [0x1375] = 0x390e, [0x1376] = 0x390f, [0x1377] = 0x3910,
  [0x1378] = 0x3911, [0x1379] = 0x3912, [0x137a] = 0x3913, [0x137b] = 0x3914,
  [0x137c] = 0x3915, [0x137d] = 0x3916, [0x137e] = 0x3917, [0x137f] = 0x3919,
  [0x1380] = 0x391a, [0x1381] = 0x391b, [0x1382] = 0x391c, [0x1383] = 0x391d,
  [0x1384] = 0x391e, [0x1385] = 0x391f, [0x1386] = 0x3920, [0x1387] = 0x3921,
  [0x1388] = 0x3922, [0x1389] = 0x3923, [0x138a] = 0x3924, [0x138b] = 0x3925,
  [0x138c] = 0x3926, [0x138d] = 0x3927, [0x138e] = 0x3928, [0x138f] = 0x3929,
  [0x1390] = 0x392a, [0x1391] = 0x392b, [0x1392] = 0x392c, [0x1393] = 0x392d,
  [0x1394] = 0x392e, [0x1395] = 0x392f, [0x1396] = 0x3930, [0x1397] = 0x3931,
  [0x1398] = 0x3932, [0x1399] = 0x3933, [0x139a] = 0x3934, [0x139b] = 0x3935,
  [0x139c] = 0x3936, [0x139d] = 0x3937, [0x139e] = 0x3938, [0x139f] = 0x3939,
  [0x13a0] = 0x393a, [0x13a1] = 0x393b, [0x13a2] = 0x393c, [0x13a3] = 0x393d,
  [0x13a4] = 0x393e, [0x13a5] = 0x393f, [0x13a6] = 0x3940, [0x13a7] = 0x3941,
  [0x13a8] = 0x3942, [0x13a9] = 0x3943, [0x13aa] = 0x3944, [0x13ab] = 0x3945,
  [0x13ac] = 0x3946, [0x13ad] = 0x3947, [0x13ae] = 0x3948, [0x13af] = 0x3949,
  [0x13b0] = 0x394a, [0x13b1] = 0x394b, [0x13b2] = 0x394c, [0x13b3] = 0x394d,
  [0x13b4] = 0x394e, [0x13b5] = 0x394f, [0x13b6] = 0x3950, [0x13b7] = 0x3951,
  [0x13b8] = 0x3952, [0x13b9] = 0x3953, [0x13ba] = 0x3954, [0x13bb] = 0x3955,
  [0x13bc] = 0x3956, [0x13bd] = 0x3957, [0x13be] = 0x3958, [0x13bf] = 0x3959,
  [0x13c0] = 0x395a, [0x13c1] = 0x395b, [0x13c2] = 0x395c, [0x13c3] = 0x395d,
  [0x13c4] = 0x395e, [0x13c5] = 0x395f, [0x13c6] = 0x3960, [0x13c7] = 0x3961,
  [0x13c8] = 0x3962, [0x13c9] = 0x3963, [0x13ca] = 0x3964, [0x13cb] = 0x3965,
  [0x13cc] = 0x3966, [0x13cd] = 0x3967, [0x13ce] = 0x3968, [0x13cf] = 0x3969,
  [0x13d0] = 0x396a, [0x13d1] = 0x396b, [0x13d2] = 0x396c, [0x13d3] = 0x396d,
  [0x13d4] = 0x396f, [0x13d5] = 0x3970, [0x13d6] = 0x3971, [0x13d7] = 0x3972,
  [0x13d8] = 0x3973, [0x13d9] = 0x3974, [0x13da] = 0x3975, [0x13db] = 0x3976,
  [0x13dc] = 0x3977, [0x13dd] = 0x3978, [0x13de] = 0x3979, [0x13df] = 0x397a,
  [0x13e0] = 0x397b, [0x13e1] = 0x397c, [0x13e2] = 0x397d, [0x13e3] = 0x397e,
  [0x13e4] = 0x397f, [0x13e5] = 0x3980, [0x13e6] = 0x3981, [0x13e7] = 0x3982,
  [0x13e8] = 0x3983, [0x13e9] = 0x3984, [0x13ea] = 0x3985, [0x13eb] = 0x3986,
  [0x13ec] = 0x3987, [0x13ed] = 0x3988, [0x13ee] = 0x3989, [0x13ef] = 0x398a,
  [0x13f0] = 0x398b, [0x13f1] = 0x398c, [0x13f2] = 0x398d, [0x13f3] = 0x398e,
  [0x13f4] = 0x398f, [0x13f5] = 0x3990, [0x13f6] = 0x3991, [0x13f7] = 0x3992,
  [0x13f8] = 0x3993, [0x13f9] = 0x3994, [0x13fa] = 0x3995, [0x13fb] = 0x3996,
  [0x13fc] = 0x3997, [0x13fd] = 0x3998, [0x13fe] = 0x3999, [0x13ff] = 0x399a,
  [0x1400] = 0x399b, [0x1401] = 0x399c, [0x1402] = 0x399d, [0x1403] = 0x399e,
  [0x1404] = 0x399f, [0x1405] = 0x39a0, [0x1406] = 0x39a1, [0x1407] = 0x39a2,
  [0x1408] = 0x39a3, [0x1409] = 0x39a4, [0x140a] = 0x39a5, [0x140b] = 0x39a6,
  [0x140c] = 0x39a7, [0x140d] = 0x39a8, [0x140e] = 0x39a9, [0x140f] = 0x39aa,
  [0x1410] = 0x39ab, [0x1411] = 0x39ac, [0x1412] = 0x39ad, [0x1413] = 0x39ae,
  [0x1414] = 0x39af, [0x1415] = 0x39b0, [0x1416] = 0x39b1, [0x1417] = 0x39b2,
  [0x1418] = 0x39b3, [0x1419] = 0x39b4, [0x141a] = 0x39b5, [0x141b] = 0x39b6,
  [0x141c] = 0x39b7, [0x141d] = 0x39b8, [0x141e] = 0x39b9, [0x141f] = 0x39ba,
  [0x1420] = 0x39bb, [0x1421] = 0x39bc, [0x1422] = 0x39bd, [0x1423] = 0x39be,
  [0x1424] = 0x39bf, [0x1425] = 0x39c0, [0x1426] = 0x39c1, [0x1427] = 0x39c2,
  [0x1428] = 0x39c3, [0x1429] = 0x39c4, [0x142a] = 0x39c5, [0x142b] = 0x39c6,
  [0x142c] = 0x39c7, [0x142d] = 0x39c8, [0x142e] = 0x39c9, [0x142f] = 0x39ca,
  [0x1430] = 0x39cb, [0x1431] = 0x39cc, [0x1432] = 0x39cd, [0x1433] = 0x39ce,
  [0x1434] = 0x39d1, [0x1435] = 0x39d2, [0x1436] = 0x39d3, [0x1437] = 0x39d4,
  [0x1438] = 0x39d5, [0x1439] = 0x39d6, [0x143a] = 0x39d7, [0x143b] = 0x39d8,
  [0x143c] = 0x39d9, [0x143d] = 0x39da, [0x143e] = 0x39db, [0x143f] = 0x39dc,
  [0x1440] = 0x39dd, [0x1441] = 0x39de, [0x1442] = 0x39e0, [0x1443] = 0x39e1,
  [0x1444] = 0x39e2, [0x1445] = 0x39e3, [0x1446] = 0x39e4, [0x1447] = 0x39e5,
  [0x1448] = 0x39e6, [0x1449] = 0x39e7, [0x144a] = 0x39e8, [0x144b] = 0x39e9,
  [0x144c] = 0x39ea, [0x144d] = 0x39eb, [0x144e] = 0x39ec, [0x144f] = 0x39ed,
  [0x1450] = 0x39ee, [0x1451] = 0x39ef, [0x1452] = 0x39f0, [0x1453] = 0x39f1,
  [0x1454] = 0x39f2, [0x1455] = 0x39f3, [0x1456] = 0x39f4, [0x1457] = 0x39f5,
  [0x1458] = 0x39f6, [0x1459] = 0x39f7, [0x145a] = 0x39f8, [0x145b] = 0x39f9,
  [0x145c] = 0x39fa, [0x145d] = 0x39fb, [0x145e] = 0x39fc, [0x145f] = 0x39fd,
  [0x1460] = 0x39fe, [0x1461] = 0x39ff, [0x1462] = 0x3a00, [0x1463] = 0x3a01,
  [0x1464] = 0x3a02, [0x1465] = 0x3a03, [0x1466] = 0x3a04, [0x1467] = 0x3a05,
  [0x1468] = 0x3a06, [0x1469] = 0x3a07, [0x146a] = 0x3a08, [0x146b] = 0x3a09,
  [0x146c] = 0x3a0a, [0x146d] = 0x3a0b, [0x146e] = 0x3a0c, [0x146f] = 0x3a0d,
  [0x1470] = 0x3a0e, [0x1471] = 0x3a0f, [0x1472] = 0x3a10, [0x1473] = 0x3a11,
  [0x1474] = 0x3a12, [0x1475] = 0x3a13, [0x1476] = 0x3a14, [0x1477] = 0x3a15,
  [0x1478] = 0x3a16, [0x1479] = 0x3a17, [0x147a] = 0x3a18, [0x147b] = 0x3a19,
  [0x147c] = 0x3a1a, [0x147d] = 0x3a1b, [0x147e] = 0x3a1c, [0x147f] = 0x3a1d,
  [0x1480] = 0x3a1e, [0x1481] = 0x3a1f, [0x1482] = 0x3a20, [0x1483] = 0x3a21,
  [0x1484] = 0x3a22, [0x1485] = 0x3a23, [0x1486] = 0x3a24, [0x1487] = 0x3a25,
  [0x1488] = 0x3a26, [0x1489] = 0x3a27, [0x148a] = 0x3a28, [0x148b] = 0x3a29,
  [0x148c] = 0x3a2a, [0x148d] = 0x3a2b, [0x148e] = 0x3a2c, [0x148f] = 0x3a2d,
  [0x1490] = 0x3a2e, [0x1491] = 0x3a2f, [0x1492] = 0x3a30, [0x1493] = 0x3a31,
  [0x1494] = 0x3a32, [0x1495] = 0x3a33, [0x1496] = 0x3a34, [0x1497] = 0x3a35,
  [0x1498] = 0x3a36, [0x1499] = 0x3a37, [0x149a] = 0x3a38, [0x149b] = 0x3a39,
  [0x149c] = 0x3a3a, [0x149d] = 0x3a3b, [0x149e] = 0x3a3c, [0x149f] = 0x3a3d,
  [0x14a0] = 0x3a3e, [0x14a1] = 0x3a3f, [0x14a2] = 0x3a40, [0x14a3] = 0x3a41,
  [0x14a4] = 0x3a42, [0x14a5] = 0x3a43, [0x14a6] = 0x3a44, [0x14a7] = 0x3a45,
  [0x14a8] = 0x3a46, [0x14a9] = 0x3a47, [0x14aa] = 0x3a48, [0x14ab] = 0x3a49,
  [0x14ac] = 0x3a4a, [0x14ad] = 0x3a4b, [0x14ae] = 0x3a4c, [0x14af] = 0x3a4d,
  [0x14b0] = 0x3a4e, [0x14b1] = 0x3a4f, [0x14b2] = 0x3a50, [0x14b3] = 0x3a51,
  [0x14b4] = 0x3a52, [0x14b5] = 0x3a53, [0x14b6] = 0x3a54, [0x14b7] = 0x3a55,
  [0x14b8] = 0x3a56, [0x14b9] = 0x3a57, [0x14ba] = 0x3a58, [0x14bb] = 0x3a59,
  [0x14bc] = 0x3a5a, [0x14bd] = 0x3a5b, [0x14be] = 0x3a5c, [0x14bf] = 0x3a5d,
  [0x14c0] = 0x3a5e, [0x14c1] = 0x3a5f, [0x14c2] = 0x3a60, [0x14c3] = 0x3a61,
  [0x14c4] = 0x3a62, [0x14c5] = 0x3a63, [0x14c6] = 0x3a64, [0x14c7] = 0x3a65,
  [0x14c8] = 0x3a66, [0x14c9] = 0x3a67, [0x14ca] = 0x3a68, [0x14cb] = 0x3a69,
  [0x14cc] = 0x3a6a, [0x14cd] = 0x3a6b, [0x14ce] = 0x3a6c, [0x14cf] = 0x3a6d,
  [0x14d0] = 0x3a6e, [0x14d1] = 0x3a6f, [0x14d2] = 0x3a70, [0x14d3] = 0x3a71,
  [0x14d4] = 0x3a72, [0x14d5] = 0x3a74, [0x14d6] = 0x3a75, [0x14d7] = 0x3a76,
  [0x14d8] = 0x3a77, [0x14d9] = 0x3a78, [0x14da] = 0x3a79, [0x14db] = 0x3a7a,
  [0x14dc] = 0x3a7b, [0x14dd] = 0x3a7c, [0x14de] = 0x3a7d, [0x14df] = 0x3a7e,
  [0x14e0] = 0x3a7f, [0x14e1] = 0x3a80, [0x14e2] = 0x3a81, [0x14e3] = 0x3a82,
  [0x14e4] = 0x3a83, [0x14e5] = 0x3a84, [0x14e6] = 0x3a85, [0x14e7] = 0x3a86,
  [0x14e8] = 0x3a87, [0x14e9] = 0x3a88, [0x14ea] = 0x3a89, [0x14eb] = 0x3a8a,
  [0x14ec] = 0x3a8b, [0x14ed] = 0x3a8c, [0x14ee] = 0x3a8d, [0x14ef] = 0x3a8e,
  [0x14f0] = 0x3a8f, [0x14f1] = 0x3a90, [0x14f2] = 0x3a91, [0x14f3] = 0x3a92,
  [0x14f4] = 0x3a93, [0x14f5] = 0x3a94, [0x14f6] = 0x3a95, [0x14f7] = 0x3a96,
  [0x14f8] = 0x3a97, [0x14f9] = 0x3a98, [0x14fa] = 0x3a99, [0x14fb] = 0x3a9a,
  [0x14fc] = 0x3a9b, [0x14fd] = 0x3a9c, [0x14fe] = 0x3a9d, [0x14ff] = 0x3a9e,
  [0x1500] = 0x3a9f, [0x1501] = 0x3aa0, [0x1502] = 0x3aa1, [0x1503] = 0x3aa2,
  [0x1504] = 0x3aa3, [0x1505] = 0x3aa4, [0x1506] = 0x3aa5, [0x1507] = 0x3aa6,
  [0x1508] = 0x3aa7, [0x1509] = 0x3aa8, [0x150a] = 0x3aa9, [0x150b] = 0x3aaa,
  [0x150c] = 0x3aab, [0x150d] = 0x3aac, [0x150e] = 0x3aad, [0x150f] = 0x3aae,
  [0x1510] = 0x3aaf, [0x1511] = 0x3ab0, [0x1512] = 0x3ab1, [0x1513] = 0x3ab2,
  [0x1514] = 0x3ab3, [0x1515] = 0x3ab4, [0x1516] = 0x3ab5, [0x1517] = 0x3ab6,
  [0x1518] = 0x3ab7, [0x1519] = 0x3ab8, [0x151a] = 0x3ab9, [0x151b] = 0x3aba,
  [0x151c] = 0x3abb, [0x151d] = 0x3abc, [0x151e] = 0x3abd, [0x151f] = 0x3abe,
  [0x1520] = 0x3abf, [0x1521] = 0x3ac0, [0x1522] = 0x3ac1, [0x1523] = 0x3ac2,
  [0x1524] = 0x3ac3, [0x1525] = 0x3ac4, [0x1526] = 0x3ac5, [0x1527] = 0x3ac6,
  [0x1528] = 0x3ac7, [0x1529] = 0x3ac8, [0x152a] = 0x3ac9, [0x152b] = 0x3aca,
  [0x152c] = 0x3acb, [0x152d] = 0x3acc, [0x152e] = 0x3acd, [0x152f] = 0x3ace,
  [0x1530] = 0x3acf, [0x1531] = 0x3ad0, [0x1532] = 0x3ad1, [0x1533] = 0x3ad2,
  [0x1534] = 0x3ad3, [0x1535] = 0x3ad4, [0x1536] = 0x3ad5, [0x1537] = 0x3ad6,
  [0x1538] = 0x3ad7, [0x1539] = 0x3ad8, [0x153a] = 0x3ad9, [0x153b] = 0x3ada,
  [0x153c] = 0x3adb, [0x153d] = 0x3adc, [0x153e] = 0x3add, [0x153f] = 0x3ade,
  [0x1540] = 0x3adf, [0x1541] = 0x3ae0, [0x1542] = 0x3ae1, [0x1543] = 0x3ae2,
  [0x1544] = 0x3ae3, [0x1545] = 0x3ae4, [0x1546] = 0x3ae5, [0x1547] = 0x3ae6,
  [0x1548] = 0x3ae7, [0x1549] = 0x3ae8, [0x154a] = 0x3ae9, [0x154b] = 0x3aea,
  [0x154c] = 0x3aeb, [0x154d] = 0x3aec, [0x154e] = 0x3aed, [0x154f] = 0x3aee,
  [0x1550] = 0x3aef, [0x1551] = 0x3af0, [0x1552] = 0x3af1, [0x1553] = 0x3af2,
  [0x1554] = 0x3af3, [0x1555] = 0x3af4, [0x1556] = 0x3af5, [0x1557] = 0x3af6,
  [0x1558] = 0x3af7, [0x1559] = 0x3af8, [0x155a] = 0x3af9, [0x155b] = 0x3afa,
  [0x155c] = 0x3afb, [0x155d] = 0x3afc, [0x155e] = 0x3afd, [0x155f] = 0x3afe,
  [0x1560] = 0x3aff, [0x1561] = 0x3b00, [0x1562] = 0x3b01, [0x1563] = 0x3b02,
  [0x1564] = 0x3b03, [0x1565] = 0x3b04, [0x1566] = 0x3b05, [0x1567] = 0x3b06,
  [0x1568] = 0x3b07, [0x1569] = 0x3b08, [0x156a] = 0x3b09, [0x156b] = 0x3b0a,
  [0x156c] = 0x3b0b, [0x156d] = 0x3b0c, [0x156e] = 0x3b0d, [0x156f] = 0x3b0e,
  [0x1570] = 0x3b0f, [0x1571] = 0x3b10, [0x1572] = 0x3b11, [0x1573] = 0x3b12,
  [0x1574] = 0x3b13, [0x1575] = 0x3b14, [0x1576] = 0x3b15, [0x1577] = 0x3b16,
  [0x1578] = 0x3b17, [0x1579] = 0x3b18, [0x157a] = 0x3b19, [0x157b] = 0x3b1a,
  [0x157c] = 0x3b1b, [0x157d] = 0x3b1c, [0x157e] = 0x3b1d, [0x157f] = 0x3b1e,
  [0x1580] = 0x3b1f, [0x1581] = 0x3b20, [0x1582] = 0x3b21, [0x1583] = 0x3b22,
  [0x1584] = 0x3b23, [0x1585] = 0x3b24, [0x1586] = 0x3b25, [0x1587] = 0x3b26,
  [0x1588] = 0x3b27, [0x1589] = 0x3b28, [0x158a] = 0x3b29, [0x158b] = 0x3b2a,
  [0x158c] = 0x3b2b, [0x158d] = 0x3b2c, [0x158e] = 0x3b2d, [0x158f] = 0x3b2e,
  [0x1590] = 0x3b2f, [0x1591] = 0x3b30, [0x1592] = 0x3b31, [0x1593] = 0x3b32,
  [0x1594] = 0x3b33, [0x1595] = 0x3b34, [0x1596] = 0x3b35, [0x1597] = 0x3b36,
  [0x1598] = 0x3b37, [0x1599] = 0x3b38, [0x159a] = 0x3b39, [0x159b] = 0x3b3a,
  [0x159c] = 0x3b3b, [0x159d] = 0x3b3c, [0x159e] = 0x3b3d, [0x159f] = 0x3b3e,
  [0x15a0] = 0x3b3f, [0x15a1] = 0x3b40, [0x15a2] = 0x3b41, [0x15a3] = 0x3b42,
  [0x15a4] = 0x3b43, [0x15a5] = 0x3b44, [0x15a6] = 0x3b45, [0x15a7] = 0x3b46,
  [0x15a8] = 0x3b47, [0x15a9] = 0x3b48, [0x15aa] = 0x3b49, [0x15ab] = 0x3b4a,
  [0x15ac] = 0x3b4b, [0x15ad] = 0x3b4c, [0x15ae] = 0x3b4d, [0x15af] = 0x3b4f,
  [0x15b0] = 0x3b50, [0x15b1] = 0x3b51, [0x15b2] = 0x3b52, [0x15b3] = 0x3b53,
  [0x15b4] = 0x3b54, [0x15b5] = 0x3b55, [0x15b6] = 0x3b56, [0x15b7] = 0x3b57,
  [0x15b8] = 0x3b58, [0x15b9] = 0x3b59, [0x15ba] = 0x3b5a, [0x15bb] = 0x3b5b,
  [0x15bc] = 0x3b5c, [0x15bd] = 0x3b5d, [0x15be] = 0x3b5e, [0x15bf] = 0x3b5f,
  [0x15c0] = 0x3b60, [0x15c1] = 0x3b61, [0x15c2] = 0x3b62, [0x15c3] = 0x3b63,
  [0x15c4] = 0x3b64, [0x15c5] = 0x3b65, [0x15c6] = 0x3b66, [0x15c7] = 0x3b67,
  [0x15c8] = 0x3b68, [0x15c9] = 0x3b69, [0x15ca] = 0x3b6a, [0x15cb] = 0x3b6b,
  [0x15cc] = 0x3b6c, [0x15cd] = 0x3b6d, [0x15ce] = 0x3b6e, [0x15cf] = 0x3b6f,
  [0x15d0] = 0x3b70, [0x15d1] = 0x3b71, [0x15d2] = 0x3b72, [0x15d3] = 0x3b73,
  [0x15d4] = 0x3b74, [0x15d5] = 0x3b75, [0x15d6] = 0x3b76, [0x15d7] = 0x3b77,
  [0x15d8] = 0x3b78, [0x15d9] = 0x3b79, [0x15da] = 0x3b7a, [0x15db] = 0x3b7b,
  [0x15dc] = 0x3b7c, [0x15dd] = 0x3b7d, [0x15de] = 0x3b7e, [0x15df] = 0x3b7f,
  [0x15e0] = 0x3b80, [0x15e1] = 0x3b81, [0x15e2] = 0x3b82, [0x15e3] = 0x3b83,
  [0x15e4] = 0x3b84, [0x15e5] = 0x3b85, [0x15e6] = 0x3b86, [0x15e7] = 0x3b87,
  [0x15e8] = 0x3b88, [0x15e9] = 0x3b89, [0x15ea] = 0x3b8a, [0x15eb] = 0x3b8b,
  [0x15ec] = 0x3b8c, [0x15ed] = 0x3b8d, [0x15ee] = 0x3b8e, [0x15ef] = 0x3b8f,
  [0x15f0] = 0x3b90, [0x15f1] = 0x3b91, [0x15f2] = 0x3b92, [0x15f3] = 0x3b93,
  [0x15f4] = 0x3b94, [0x15f5] = 0x3b95, [0x15f6] = 0x3b96, [0x15f7] = 0x3b97,
  [0x15f8] = 0x3b98, [0x15f9] = 0x3b99, [0x15fa] = 0x3b9a, [0x15fb] = 0x3b9b,
  [0x15fc] = 0x3b9c, [0x15fd] = 0x3b9d, [0x15fe] = 0x3b9e, [0x15ff] = 0x3b9f,
  [0x1600] = 0x3ba0, [0x1601] = 0x3ba1, [0x1602] = 0x3ba2, [0x1603] = 0x3ba3,
  [0x1604] = 0x3ba4, [0x1605] = 0x3ba5, [0x1606] = 0x3ba6, [0x1607] = 0x3ba7,
  [0x1608] = 0x3ba8, [0x1609] = 0x3ba9, [0x160a] = 0x3baa, [0x160b] = 0x3bab,
  [0x160c] = 0x3bac, [0x160d] = 0x3bad, [0x160e] = 0x3bae, [0x160f] = 0x3baf,
  [0x1610] = 0x3bb0, [0x1611] = 0x3bb1, [0x1612] = 0x3bb2, [0x1613] = 0x3bb3,
  [0x1614] = 0x3bb4, [0x1615] = 0x3bb5, [0x1616] = 0x3bb6, [0x1617] = 0x3bb7,
  [0x1618] = 0x3bb8, [0x1619] = 0x3bb9, [0x161a] = 0x3bba, [0x161b] = 0x3bbb,
  [0x161c] = 0x3bbc, [0x161d] = 0x3bbd, [0x161e] = 0x3bbe, [0x161f] = 0x3bbf,
  [0x1620] = 0x3bc0, [0x1621] = 0x3bc1, [0x1622] = 0x3bc2, [0x1623] = 0x3bc3,
  [0x1624] = 0x3bc4, [0x1625] = 0x3bc5, [0x1626] = 0x3bc6, [0x1627] = 0x3bc7,
  [0x1628] = 0x3bc8, [0x1629] = 0x3bc9, [0x162a] = 0x3bca, [0x162b] = 0x3bcb,
  [0x162c] = 0x3bcc, [0x162d] = 0x3bcd, [0x162e] = 0x3bce, [0x162f] = 0x3bcf,
  [0x1630] = 0x3bd0, [0x1631] = 0x3bd1, [0x1632] = 0x3bd2, [0x1633] = 0x3bd3,
  [0x1634] = 0x3bd4, [0x1635] = 0x3bd5, [0x1636] = 0x3bd6, [0x1637] = 0x3bd7,
  [0x1638] = 0x3bd8, [0x1639] = 0x3bd9, [0x163a] = 0x3bda, [0x163b] = 0x3bdb,
  [0x163c] = 0x3bdc, [0x163d] = 0x3bdd, [0x163e] = 0x3bde, [0x163f] = 0x3bdf,
  [0x1640] = 0x3be0, [0x1641] = 0x3be1, [0x1642] = 0x3be2, [0x1643] = 0x3be3,
  [0x1644] = 0x3be4, [0x1645] = 0x3be5, [0x1646] = 0x3be6, [0x1647] = 0x3be7,
  [0x1648] = 0x3be8, [0x1649] = 0x3be9, [0x164a] = 0x3bea, [0x164b] = 0x3beb,
  [0x164c] = 0x3bec, [0x164d] = 0x3bed, [0x164e] = 0x3bee, [0x164f] = 0x3bef,
  [0x1650] = 0x3bf0, [0x1651] = 0x3bf1, [0x1652] = 0x3bf2, [0x1653] = 0x3bf3,
  [0x1654] = 0x3bf4, [0x1655] = 0x3bf5, [0x1656] = 0x3bf6, [0x1657] = 0x3bf7,
  [0x1658] = 0x3bf8, [0x1659] = 0x3bf9, [0x165a] = 0x3bfa, [0x165b] = 0x3bfb,
  [0x165c] = 0x3bfc, [0x165d] = 0x3bfd, [0x165e] = 0x3bfe, [0x165f] = 0x3bff,
  [0x1660] = 0x3c00, [0x1661] = 0x3c01, [0x1662] = 0x3c02, [0x1663] = 0x3c03,
  [0x1664] = 0x3c04, [0x1665] = 0x3c05, [0x1666] = 0x3c06, [0x1667] = 0x3c07,
  [0x1668] = 0x3c08, [0x1669] = 0x3c09, [0x166a] = 0x3c0a, [0x166b] = 0x3c0b,
  [0x166c] = 0x3c0c, [0x166d] = 0x3c0d, [0x166e] = 0x3c0e, [0x166f] = 0x3c0f,
  [0x1670] = 0x3c10, [0x1671] = 0x3c11, [0x1672] = 0x3c12, [0x1673] = 0x3c13,
  [0x1674] = 0x3c14, [0x1675] = 0x3c15, [0x1676] = 0x3c16, [0x1677] = 0x3c17,
  [0x1678] = 0x3c18, [0x1679] = 0x3c19, [0x167a] = 0x3c1a, [0x167b] = 0x3c1b,
  [0x167c] = 0x3c1c, [0x167d] = 0x3c1d, [0x167e] = 0x3c1e, [0x167f] = 0x3c1f,
  [0x1680] = 0x3c20, [0x1681] = 0x3c21, [0x1682] = 0x3c22, [0x1683] = 0x3c23,
  [0x1684] = 0x3c24, [0x1685] = 0x3c25, [0x1686] = 0x3c26, [0x1687] = 0x3c27,
  [0x1688] = 0x3c28, [0x1689] = 0x3c29, [0x168a] = 0x3c2a, [0x168b] = 0x3c2b,
  [0x168c] = 0x3c2c, [0x168d] = 0x3c2d, [0x168e] = 0x3c2e, [0x168f] = 0x3c2f,
  [0x1690] = 0x3c30, [0x1691] = 0x3c31, [0x1692] = 0x3c32, [0x1693] = 0x3c33,
  [0x1694] = 0x3c34, [0x1695] = 0x3c35, [0x1696] = 0x3c36, [0x1697] = 0x3c37,
  [0x1698] = 0x3c38, [0x1699] = 0x3c39, [0x169a] = 0x3c3a, [0x169b] = 0x3c3b,
  [0x169c] = 0x3c3c, [0x169d] = 0x3c3d, [0x169e] = 0x3c3e, [0x169f] = 0x3c3f,
  [0x16a0] = 0x3c40, [0x16a1] = 0x3c41, [0x16a2] = 0x3c42, [0x16a3] = 0x3c43,
  [0x16a4] = 0x3c44, [0x16a5] = 0x3c45, [0x16a6] = 0x3c46, [0x16a7] = 0x3c47,
  [0x16a8] = 0x3c48, [0x16a9] = 0x3c49, [0x16aa] = 0x3c4a, [0x16ab] = 0x3c4b,
  [0x16ac] = 0x3c4c, [0x16ad] = 0x3c4d, [0x16ae] = 0x3c4e, [0x16af] = 0x3c4f,
  [0x16b0] = 0x3c50, [0x16b1] = 0x3c51, [0x16b2] = 0x3c52, [0x16b3] = 0x3c53,
  [0x16b4] = 0x3c54, [0x16b5] = 0x3c55, [0x16b6] = 0x3c56, [0x16b7] = 0x3c57,
  [0x16b8] = 0x3c58, [0x16b9] = 0x3c59, [0x16ba] = 0x3c5a, [0x16bb] = 0x3c5b,
  [0x16bc] = 0x3c5c, [0x16bd] = 0x3c5d, [0x16be] = 0x3c5e, [0x16bf] = 0x3c5f,
  [0x16c0] = 0x3c60, [0x16c1] = 0x3c61, [0x16c2] = 0x3c62, [0x16c3] = 0x3c63,
  [0x16c4] = 0x3c64, [0x16c5] = 0x3c65, [0x16c6] = 0x3c66, [0x16c7] = 0x3c67,
  [0x16c8] = 0x3c68, [0x16c9] = 0x3c69, [0x16ca] = 0x3c6a, [0x16cb] = 0x3c6b,
  [0x16cc] = 0x3c6c, [0x16cd] = 0x3c6d, [0x16ce] = 0x3c6f, [0x16cf] = 0x3c70,
  [0x16d0] = 0x3c71, [0x16d1] = 0x3c72, [0x16d2] = 0x3c73, [0x16d3] = 0x3c74,
  [0x16d4] = 0x3c75, [0x16d5] = 0x3c76, [0x16d6] = 0x3c77, [0x16d7] = 0x3c78,
  [0x16d8] = 0x3c79, [0x16d9] = 0x3c7a, [0x16da] = 0x3c7b, [0x16db] = 0x3c7c,
  [0x16dc] = 0x3c7d, [0x16dd] = 0x3c7e, [0x16de] = 0x3c7f, [0x16df] = 0x3c80,
  [0x16e0] = 0x3c81, [0x16e1] = 0x3c82, [0x16e2] = 0x3c83, [0x16e3] = 0x3c84,
  [0x16e4] = 0x3c85, [0x16e5] = 0x3c86, [0x16e6] = 0x3c87, [0x16e7] = 0x3c88,
  [0x16e8] = 0x3c89, [0x16e9] = 0x3c8a, [0x16ea] = 0x3c8b, [0x16eb] = 0x3c8c,
  [0x16ec] = 0x3c8d, [0x16ed] = 0x3c8e, [0x16ee] = 0x3c8f, [0x16ef] = 0x3c90,
  [0x16f0] = 0x3c91, [0x16f1] = 0x3c92, [0x16f2] = 0x3c93, [0x16f3] = 0x3c94,
  [0x16f4] = 0x3c95, [0x16f5] = 0x3c96, [0x16f6] = 0x3c97, [0x16f7] = 0x3c98,
  [0x16f8] = 0x3c99, [0x16f9] = 0x3c9a, [0x16fa] = 0x3c9b, [0x16fb] = 0x3c9c,
  [0x16fc] = 0x3c9d, [0x16fd] = 0x3c9e, [0x16fe] = 0x3c9f, [0x16ff] = 0x3ca0,
  [0x1700] = 0x3ca1, [0x1701] = 0x3ca2, [0x1702] = 0x3ca3, [0x1703] = 0x3ca4,
  [0x1704] = 0x3ca5, [0x1705] = 0x3ca6, [0x1706] = 0x3ca7, [0x1707] = 0x3ca8,
  [0x1708] = 0x3ca9, [0x1709] = 0x3caa, [0x170a] = 0x3cab, [0x170b] = 0x3cac,
  [0x170c] = 0x3cad, [0x170d] = 0x3cae, [0x170e] = 0x3caf, [0x170f] = 0x3cb0,
  [0x1710] = 0x3cb1, [0x1711] = 0x3cb2, [0x1712] = 0x3cb3, [0x1713] = 0x3cb4,
  [0x1714] = 0x3cb5, [0x1715] = 0x3cb6, [0x1716] = 0x3cb7, [0x1717] = 0x3cb8,
  [0x1718] = 0x3cb9, [0x1719] = 0x3cba, [0x171a] = 0x3cbb, [0x171b] = 0x3cbc,
  [0x171c] = 0x3cbd, [0x171d] = 0x3cbe, [0x171e] = 0x3cbf, [0x171f] = 0x3cc0,
  [0x1720] = 0x3cc1, [0x1721] = 0x3cc2, [0x1722] = 0x3cc3, [0x1723] = 0x3cc4,
  [0x1724] = 0x3cc5, [0x1725] = 0x3cc6, [0x1726] = 0x3cc7, [0x1727] = 0x3cc8,
  [0x1728] = 0x3cc9, [0x1729] = 0x3cca, [0x172a] = 0x3ccb, [0x172b] = 0x3ccc,
  [0x172c] = 0x3ccd, [0x172d] = 0x3cce, [0x172e] = 0x3ccf, [0x172f] = 0x3cd0,
  [0x1730] = 0x3cd1, [0x1731] = 0x3cd2, [0x1732] = 0x3cd3, [0x1733] = 0x3cd4,
  [0x1734] = 0x3cd5, [0x1735] = 0x3cd6, [0x1736] = 0x3cd7, [0x1737] = 0x3cd8,
  [0x1738] = 0x3cd9, [0x1739] = 0x3cda, [0x173a] = 0x3cdb, [0x173b] = 0x3cdc,
  [0x173c] = 0x3cdd, [0x173d] = 0x3cde, [0x173e] = 0x3cdf, [0x173f] = 0x3ce1,
  [0x1740] = 0x3ce2, [0x1741] = 0x3ce3, [0x1742] = 0x3ce4, [0x1743] = 0x3ce5,
  [0x1744] = 0x3ce6, [0x1745] = 0x3ce7, [0x1746] = 0x3ce8, [0x1747] = 0x3ce9,
  [0x1748] = 0x3cea, [0x1749] = 0x3ceb, [0x174a] = 0x3cec, [0x174b] = 0x3ced,
  [0x174c] = 0x3cee, [0x174d] = 0x3cef, [0x174e] = 0x3cf0, [0x174f] = 0x3cf1,
  [0x1750] = 0x3cf2, [0x1751] = 0x3cf3, [0x1752] = 0x3cf4, [0x1753] = 0x3cf5,
  [0x1754] = 0x3cf6, [0x1755] = 0x3cf7, [0x1756] = 0x3cf8, [0x1757] = 0x3cf9,
  [0x1758] = 0x3cfa, [0x1759] = 0x3cfb, [0x175a] = 0x3cfc, [0x175b] = 0x3cfd,
  [0x175c] = 0x3cfe, [0x175d] = 0x3cff, [0x175e] = 0x3d00, [0x175f] = 0x3d01,
  [0x1760] = 0x3d02, [0x1761] = 0x3d03, [0x1762] = 0x3d04, [0x1763] = 0x3d05,
  [0x1764] = 0x3d06, [0x1765] = 0x3d07, [0x1766] = 0x3d08, [0x1767] = 0x3d09,
  [0x1768] = 0x3d0a, [0x1769] = 0x3d0b, [0x176a] = 0x3d0c, [0x176b] = 0x3d0d,
  [0x176c] = 0x3d0e, [0x176d] = 0x3d0f, [0x176e] = 0x3d10, [0x176f] = 0x3d11,
  [0x1770] = 0x3d12, [0x1771] = 0x3d13, [0x1772] = 0x3d14, [0x1773] = 0x3d15,
  [0x1774] = 0x3d16, [0x1775] = 0x3d17, [0x1776] = 0x3d18, [0x1777] = 0x3d19,
  [0x1778] = 0x3d1a, [0x1779] = 0x3d1b, [0x177a] = 0x3d1c, [0x177b] = 0x3d1d,
  [0x177c] = 0x3d1e, [0x177d] = 0x3d1f, [0x177e] = 0x3d20, [0x177f] = 0x3d21,
  [0x1780] = 0x3d22, [0x1781] = 0x3d23, [0x1782] = 0x3d24, [0x1783] = 0x3d25,
  [0x1784] = 0x3d26, [0x1785] = 0x3d27, [0x1786] = 0x3d28, [0x1787] = 0x3d29,
  [0x1788] = 0x3d2a, [0x1789] = 0x3d2b, [0x178a] = 0x3d2c, [0x178b] = 0x3d2d,
  [0x178c] = 0x3d2e, [0x178d] = 0x3d2f, [0x178e] = 0x3d30, [0x178f] = 0x3d31,
  [0x1790] = 0x3d32, [0x1791] = 0x3d33, [0x1792] = 0x3d34, [0x1793] = 0x3d35,
  [0x1794] = 0x3d36, [0x1795] = 0x3d37, [0x1796] = 0x3d38, [0x1797] = 0x3d39,
  [0x1798] = 0x3d3a, [0x1799] = 0x3d3b, [0x179a] = 0x3d3c, [0x179b] = 0x3d3d,
  [0x179c] = 0x3d3e, [0x179d] = 0x3d3f, [0x179e] = 0x3d40, [0x179f] = 0x3d41,
  [0x17a0] = 0x3d42, [0x17a1] = 0x3d43, [0x17a2] = 0x3d44, [0x17a3] = 0x3d45,
  [0x17a4] = 0x3d46, [0x17a5] = 0x3d47, [0x17a6] = 0x3d48, [0x17a7] = 0x3d49,
  [0x17a8] = 0x3d4a, [0x17a9] = 0x3d4b, [0x17aa] = 0x3d4c, [0x17ab] = 0x3d4d,
  [0x17ac] = 0x3d4e, [0x17ad] = 0x3d4f, [0x17ae] = 0x3d50, [0x17af] = 0x3d51,
  [0x17b0] = 0x3d52, [0x17b1] = 0x3d53, [0x17b2] = 0x3d54, [0x17b3] = 0x3d55,
  [0x17b4] = 0x3d56, [0x17b5] = 0x3d57, [0x17b6] = 0x3d58, [0x17b7] = 0x3d59,
  [0x17b8] = 0x3d5a, [0x17b9] = 0x3d5b, [0x17ba] = 0x3d5c, [0x17bb] = 0x3d5d,
  [0x17bc] = 0x3d5e, [0x17bd] = 0x3d5f, [0x17be] = 0x3d60, [0x17bf] = 0x3d61,
  [0x17c0] = 0x3d62, [0x17c1] = 0x3d63, [0x17c2] = 0x3d64, [0x17c3] = 0x3d65,
  [0x17c4] = 0x3d66, [0x17c5] = 0x3d67, [0x17c6] = 0x3d68, [0x17c7] = 0x3d69,
  [0x17c8] = 0x3d6a, [0x17c9] = 0x3d6b, [0x17ca] = 0x3d6c, [0x17cb] = 0x3d6d,
  [0x17cc] = 0x3d6e, [0x17cd] = 0x3d6f, [0x17ce] = 0x3d70, [0x17cf] = 0x3d71,
  [0x17d0] = 0x3d72, [0x17d1] = 0x3d73, [0x17d2] = 0x3d74, [0x17d3] = 0x3d75,
  [0x17d4] = 0x3d76, [0x17d5] = 0x3d77, [0x17d6] = 0x3d78, [0x17d7] = 0x3d79,
  [0x17d8] = 0x3d7a, [0x17d9] = 0x3d7b, [0x17da] = 0x3d7c, [0x17db] = 0x3d7d,
  [0x17dc] = 0x3d7e, [0x17dd] = 0x3d7f, [0x17de] = 0x3d80, [0x17df] = 0x3d81,
  [0x17e0] = 0x3d82, [0x17e1] = 0x3d83, [0x17e2] = 0x3d84, [0x17e3] = 0x3d85,
  [0x17e4] = 0x3d86, [0x17e5] = 0x3d87, [0x17e6] = 0x3d88, [0x17e7] = 0x3d89,
  [0x17e8] = 0x3d8a, [0x17e9] = 0x3d8b, [0x17ea] = 0x3d8c, [0x17eb] = 0x3d8d,
  [0x17ec] = 0x3d8e, [0x17ed] = 0x3d8f, [0x17ee] = 0x3d90, [0x17ef] = 0x3d91,
  [0x17f0] = 0x3d92, [0x17f1] = 0x3d93, [0x17f2] = 0x3d94, [0x17f3] = 0x3d95,
  [0x17f4] = 0x3d96, [0x17f5] = 0x3d97, [0x17f6] = 0x3d98, [0x17f7] = 0x3d99,
  [0x17f8] = 0x3d9a, [0x17f9] = 0x3d9b, [0x17fa] = 0x3d9c, [0x17fb] = 0x3d9d,
  [0x17fc] = 0x3d9e, [0x17fd] = 0x3d9f, [0x17fe] = 0x3da0, [0x17ff] = 0x3da1,
  [0x1800] = 0x3da2, [0x1801] = 0x3da3, [0x1802] = 0x3da4, [0x1803] = 0x3da5,
  [0x1804] = 0x3da6, [0x1805] = 0x3da7, [0x1806] = 0x3da8, [0x1807] = 0x3da9,
  [0x1808] = 0x3daa, [0x1809] = 0x3dab, [0x180a] = 0x3dac, [0x180b] = 0x3dad,
  [0x180c] = 0x3dae, [0x180d] = 0x3daf, [0x180e] = 0x3db0, [0x180f] = 0x3db1,
  [0x1810] = 0x3db2, [0x1811] = 0x3db3, [0x1812] = 0x3db4, [0x1813] = 0x3db5,
  [0x1814] = 0x3db6, [0x1815] = 0x3db7, [0x1816] = 0x3db8, [0x1817] = 0x3db9,
  [0x1818] = 0x3dba, [0x1819] = 0x3dbb, [0x181a] = 0x3dbc, [0x181b] = 0x3dbd,
  [0x181c] = 0x3dbe, [0x181d] = 0x3dbf, [0x181e] = 0x3dc0, [0x181f] = 0x3dc1,
  [0x1820] = 0x3dc2, [0x1821] = 0x3dc3, [0x1822] = 0x3dc4, [0x1823] = 0x3dc5,
  [0x1824] = 0x3dc6, [0x1825] = 0x3dc7, [0x1826] = 0x3dc8, [0x1827] = 0x3dc9,
  [0x1828] = 0x3dca, [0x1829] = 0x3dcb, [0x182a] = 0x3dcc, [0x182b] = 0x3dcd,
  [0x182c] = 0x3dce, [0x182d] = 0x3dcf, [0x182e] = 0x3dd0, [0x182f] = 0x3dd1,
  [0x1830] = 0x3dd2, [0x1831] = 0x3dd3, [0x1832] = 0x3dd4, [0x1833] = 0x3dd5,
  [0x1834] = 0x3dd6, [0x1835] = 0x3dd7, [0x1836] = 0x3dd8, [0x1837] = 0x3dd9,
  [0x1838] = 0x3dda, [0x1839] = 0x3ddb, [0x183a] = 0x3ddc, [0x183b] = 0x3ddd,
  [0x183c] = 0x3dde, [0x183d] = 0x3ddf, [0x183e] = 0x3de0, [0x183f] = 0x3de1,
  [0x1840] = 0x3de2, [0x1841] = 0x3de3, [0x1842] = 0x3de4, [0x1843] = 0x3de5,
  [0x1844] = 0x3de6, [0x1845] = 0x3de7, [0x1846] = 0x3de8, [0x1847] = 0x3de9,
  [0x1848] = 0x3dea, [0x1849] = 0x3deb, [0x184a] = 0x3dec, [0x184b] = 0x3ded,
  [0x184c] = 0x3dee, [0x184d] = 0x3def, [0x184e] = 0x3df0, [0x184f] = 0x3df1,
  [0x1850] = 0x3df2, [0x1851] = 0x3df3, [0x1852] = 0x3df4, [0x1853] = 0x3df5,
  [0x1854] = 0x3df6, [0x1855] = 0x3df7, [0x1856] = 0x3df8, [0x1857] = 0x3df9,
  [0x1858] = 0x3dfa, [0x1859] = 0x3dfb, [0x185a] = 0x3dfc, [0x185b] = 0x3dfd,
  [0x185c] = 0x3dfe, [0x185d] = 0x3dff, [0x185e] = 0x3e00, [0x185f] = 0x3e01,
  [0x1860] = 0x3e02, [0x1861] = 0x3e03, [0x1862] = 0x3e04, [0x1863] = 0x3e05,
  [0x1864] = 0x3e06, [0x1865] = 0x3e07, [0x1866] = 0x3e08, [0x1867] = 0x3e09,
  [0x1868] = 0x3e0a, [0x1869] = 0x3e0b, [0x186a] = 0x3e0c, [0x186b] = 0x3e0d,
  [0x186c] = 0x3e0e, [0x186d] = 0x3e0f, [0x186e] = 0x3e10, [0x186f] = 0x3e11,
  [0x1870] = 0x3e12, [0x1871] = 0x3e13, [0x1872] = 0x3e14, [0x1873] = 0x3e15,
  [0x1874] = 0x3e16, [0x1875] = 0x3e17, [0x1876] = 0x3e18, [0x1877] = 0x3e19,
  [0x1878] = 0x3e1a, [0x1879] = 0x3e1b, [0x187a] = 0x3e1c, [0x187b] = 0x3e1d,
  [0x187c] = 0x3e1e, [0x187d] = 0x3e1f, [0x187e] = 0x3e20, [0x187f] = 0x3e21,
  [0x1880] = 0x3e22, [0x1881] = 0x3e23, [0x1882] = 0x3e24, [0x1883] = 0x3e25,
  [0x1884] = 0x3e26, [0x1885] = 0x3e27, [0x1886] = 0x3e28, [0x1887] = 0x3e29,
  [0x1888] = 0x3e2a, [0x1889] = 0x3e2b, [0x188a] = 0x3e2c, [0x188b] = 0x3e2d,
  [0x188c] = 0x3e2e, [0x188d] = 0x3e2f, [0x188e] = 0x3e30, [0x188f] = 0x3e31,
  [0x1890] = 0x3e32, [0x1891] = 0x3e33, [0x1892] = 0x3e34, [0x1893] = 0x3e35,
  [0x1894] = 0x3e36, [0x1895] = 0x3e37, [0x1896] = 0x3e38, [0x1897] = 0x3e39,
  [0x1898] = 0x3e3a, [0x1899] = 0x3e3b, [0x189a] = 0x3e3c, [0x189b] = 0x3e3d,
  [0x189c] = 0x3e3e, [0x189d] = 0x3e3f, [0x189e] = 0x3e40, [0x189f] = 0x3e41,
  [0x18a0] = 0x3e42, [0x18a1] = 0x3e43, [0x18a2] = 0x3e44, [0x18a3] = 0x3e45,
  [0x18a4] = 0x3e46, [0x18a5] = 0x3e47, [0x18a6] = 0x3e48, [0x18a7] = 0x3e49,
  [0x18a8] = 0x3e4a, [0x18a9] = 0x3e4b, [0x18aa] = 0x3e4c, [0x18ab] = 0x3e4d,
  [0x18ac] = 0x3e4e, [0x18ad] = 0x3e4f, [0x18ae] = 0x3e50, [0x18af] = 0x3e51,
  [0x18b0] = 0x3e52, [0x18b1] = 0x3e53, [0x18b2] = 0x3e54, [0x18b3] = 0x3e55,
  [0x18b4] = 0x3e56, [0x18b5] = 0x3e57, [0x18b6] = 0x3e58, [0x18b7] = 0x3e59,
  [0x18b8] = 0x3e5a, [0x18b9] = 0x3e5b, [0x18ba] = 0x3e5c, [0x18bb] = 0x3e5d,
  [0x18bc] = 0x3e5e, [0x18bd] = 0x3e5f, [0x18be] = 0x3e60, [0x18bf] = 0x3e61,
  [0x18c0] = 0x3e62, [0x18c1] = 0x3e63, [0x18c2] = 0x3e64, [0x18c3] = 0x3e65,
  [0x18c4] = 0x3e66, [0x18c5] = 0x3e67, [0x18c6] = 0x3e68, [0x18c7] = 0x3e69,
  [0x18c8] = 0x3e6a, [0x18c9] = 0x3e6b, [0x18ca] = 0x3e6c, [0x18cb] = 0x3e6d,
  [0x18cc] = 0x3e6e, [0x18cd] = 0x3e6f, [0x18ce] = 0x3e70, [0x18cf] = 0x3e71,
  [0x18d0] = 0x3e72, [0x18d1] = 0x3e73, [0x18d2] = 0x3e74, [0x18d3] = 0x3e75,
  [0x18d4] = 0x3e76, [0x18d5] = 0x3e77, [0x18d6] = 0x3e78, [0x18d7] = 0x3e79,
  [0x18d8] = 0x3e7a, [0x18d9] = 0x3e7b, [0x18da] = 0x3e7c, [0x18db] = 0x3e7d,
  [0x18dc] = 0x3e7e, [0x18dd] = 0x3e7f, [0x18de] = 0x3e80, [0x18df] = 0x3e81,
  [0x18e0] = 0x3e82, [0x18e1] = 0x3e83, [0x18e2] = 0x3e84, [0x18e3] = 0x3e85,
  [0x18e4] = 0x3e86, [0x18e5] = 0x3e87, [0x18e6] = 0x3e88, [0x18e7] = 0x3e89,
  [0x18e8] = 0x3e8a, [0x18e9] = 0x3e8b, [0x18ea] = 0x3e8c, [0x18eb] = 0x3e8d,
  [0x18ec] = 0x3e8e, [0x18ed] = 0x3e8f, [0x18ee] = 0x3e90, [0x18ef] = 0x3e91,
  [0x18f0] = 0x3e92, [0x18f1] = 0x3e93, [0x18f2] = 0x3e94, [0x18f3] = 0x3e95,
  [0x18f4] = 0x3e96, [0x18f5] = 0x3e97, [0x18f6] = 0x3e98, [0x18f7] = 0x3e99,
  [0x18f8] = 0x3e9a, [0x18f9] = 0x3e9b, [0x18fa] = 0x3e9c, [0x18fb] = 0x3e9d,
  [0x18fc] = 0x3e9e, [0x18fd] = 0x3e9f, [0x18fe] = 0x3ea0, [0x18ff] = 0x3ea1,
  [0x1900] = 0x3ea2, [0x1901] = 0x3ea3, [0x1902] = 0x3ea4, [0x1903] = 0x3ea5,
  [0x1904] = 0x3ea6, [0x1905] = 0x3ea7, [0x1906] = 0x3ea8, [0x1907] = 0x3ea9,
  [0x1908] = 0x3eaa, [0x1909] = 0x3eab, [0x190a] = 0x3eac, [0x190b] = 0x3ead,
  [0x190c] = 0x3eae, [0x190d] = 0x3eaf, [0x190e] = 0x3eb0, [0x190f] = 0x3eb1,
  [0x1910] = 0x3eb2, [0x1911] = 0x3eb3, [0x1912] = 0x3eb4, [0x1913] = 0x3eb5,
  [0x1914] = 0x3eb6, [0x1915] = 0x3eb7, [0x1916] = 0x3eb8, [0x1917] = 0x3eb9,
  [0x1918] = 0x3eba, [0x1919] = 0x3ebb, [0x191a] = 0x3ebc, [0x191b] = 0x3ebd,
  [0x191c] = 0x3ebe, [0x191d] = 0x3ebf, [0x191e] = 0x3ec0, [0x191f] = 0x3ec1,
  [0x1920] = 0x3ec2, [0x1921] = 0x3ec3, [0x1922] = 0x3ec4, [0x1923] = 0x3ec5,
  [0x1924] = 0x3ec6, [0x1925] = 0x3ec7, [0x1926] = 0x3ec8, [0x1927] = 0x3ec9,
  [0x1928] = 0x3eca, [0x1929] = 0x3ecb, [0x192a] = 0x3ecc, [0x192b] = 0x3ecd,
  [0x192c] = 0x3ece, [0x192d] = 0x3ecf, [0x192e] = 0x3ed0, [0x192f] = 0x3ed1,
  [0x1930] = 0x3ed2, [0x1931] = 0x3ed3, [0x1932] = 0x3ed4, [0x1933] = 0x3ed5,
  [0x1934] = 0x3ed6, [0x1935] = 0x3ed7, [0x1936] = 0x3ed8, [0x1937] = 0x3ed9,
  [0x1938] = 0x3eda, [0x1939] = 0x3edb, [0x193a] = 0x3edc, [0x193b] = 0x3edd,
  [0x193c] = 0x3ede, [0x193d] = 0x3edf, [0x193e] = 0x3ee0, [0x193f] = 0x3ee1,
  [0x1940] = 0x3ee2, [0x1941] = 0x3ee3, [0x1942] = 0x3ee4, [0x1943] = 0x3ee5,
  [0x1944] = 0x3ee6, [0x1945] = 0x3ee7, [0x1946] = 0x3ee8, [0x1947] = 0x3ee9,
  [0x1948] = 0x3eea, [0x1949] = 0x3eeb, [0x194a] = 0x3eec, [0x194b] = 0x3eed,
  [0x194c] = 0x3eee, [0x194d] = 0x3eef, [0x194e] = 0x3ef0, [0x194f] = 0x3ef1,
  [0x1950] = 0x3ef2, [0x1951] = 0x3ef3, [0x1952] = 0x3ef4, [0x1953] = 0x3ef5,
  [0x1954] = 0x3ef6, [0x1955] = 0x3ef7, [0x1956] = 0x3ef8, [0x1957] = 0x3ef9,
  [0x1958] = 0x3efa, [0x1959] = 0x3efb, [0x195a] = 0x3efc, [0x195b] = 0x3efd,
  [0x195c] = 0x3efe, [0x195d] = 0x3eff, [0x195e] = 0x3f00, [0x195f] = 0x3f01,
  [0x1960] = 0x3f02, [0x1961] = 0x3f03, [0x1962] = 0x3f04, [0x1963] = 0x3f05,
  [0x1964] = 0x3f06, [0x1965] = 0x3f07, [0x1966] = 0x3f08, [0x1967] = 0x3f09,
  [0x1968] = 0x3f0a, [0x1969] = 0x3f0b, [0x196a] = 0x3f0c, [0x196b] = 0x3f0d,
  [0x196c] = 0x3f0e, [0x196d] = 0x3f0f, [0x196e] = 0x3f10, [0x196f] = 0x3f11,
  [0x1970] = 0x3f12, [0x1971] = 0x3f13, [0x1972] = 0x3f14, [0x1973] = 0x3f15,
  [0x1974] = 0x3f16, [0x1975] = 0x3f17, [0x1976] = 0x3f18, [0x1977] = 0x3f19,
  [0x1978] = 0x3f1a, [0x1979] = 0x3f1b, [0x197a] = 0x3f1c, [0x197b] = 0x3f1d,
  [0x197c] = 0x3f1e, [0x197d] = 0x3f1f, [0x197e] = 0x3f20, [0x197f] = 0x3f21,
  [0x1980] = 0x3f22, [0x1981] = 0x3f23, [0x1982] = 0x3f24, [0x1983] = 0x3f25,
  [0x1984] = 0x3f26, [0x1985] = 0x3f27, [0x1986] = 0x3f28, [0x1987] = 0x3f29,
  [0x1988] = 0x3f2a, [0x1989] = 0x3f2b, [0x198a] = 0x3f2c, [0x198b] = 0x3f2d,
  [0x198c] = 0x3f2e, [0x198d] = 0x3f2f, [0x198e] = 0x3f30, [0x198f] = 0x3f31,
  [0x1990] = 0x3f32, [0x1991] = 0x3f33, [0x1992] = 0x3f34, [0x1993] = 0x3f35,
  [0x1994] = 0x3f36, [0x1995] = 0x3f37, [0x1996] = 0x3f38, [0x1997] = 0x3f39,
  [0x1998] = 0x3f3a, [0x1999] = 0x3f3b, [0x199a] = 0x3f3c, [0x199b] = 0x3f3d,
  [0x199c] = 0x3f3e, [0x199d] = 0x3f3f, [0x199e] = 0x3f40, [0x199f] = 0x3f41,
  [0x19a0] = 0x3f42, [0x19a1] = 0x3f43, [0x19a2] = 0x3f44, [0x19a3] = 0x3f45,
  [0x19a4] = 0x3f46, [0x19a5] = 0x3f47, [0x19a6] = 0x3f48, [0x19a7] = 0x3f49,
  [0x19a8] = 0x3f4a, [0x19a9] = 0x3f4b, [0x19aa] = 0x3f4c, [0x19ab] = 0x3f4d,
  [0x19ac] = 0x3f4e, [0x19ad] = 0x3f4f, [0x19ae] = 0x3f50, [0x19af] = 0x3f51,
  [0x19b0] = 0x3f52, [0x19b1] = 0x3f53, [0x19b2] = 0x3f54, [0x19b3] = 0x3f55,
  [0x19b4] = 0x3f56, [0x19b5] = 0x3f57, [0x19b6] = 0x3f58, [0x19b7] = 0x3f59,
  [0x19b8] = 0x3f5a, [0x19b9] = 0x3f5b, [0x19ba] = 0x3f5c, [0x19bb] = 0x3f5d,
  [0x19bc] = 0x3f5e, [0x19bd] = 0x3f5f, [0x19be] = 0x3f60, [0x19bf] = 0x3f61,
  [0x19c0] = 0x3f62, [0x19c1] = 0x3f63, [0x19c2] = 0x3f64, [0x19c3] = 0x3f65,
  [0x19c4] = 0x3f66, [0x19c5] = 0x3f67, [0x19c6] = 0x3f68, [0x19c7] = 0x3f69,
  [0x19c8] = 0x3f6a, [0x19c9] = 0x3f6b, [0x19ca] = 0x3f6c, [0x19cb] = 0x3f6d,
  [0x19cc] = 0x3f6e, [0x19cd] = 0x3f6f, [0x19ce] = 0x3f70, [0x19cf] = 0x3f71,
  [0x19d0] = 0x3f72, [0x19d1] = 0x3f73, [0x19d2] = 0x3f74, [0x19d3] = 0x3f75,
  [0x19d4] = 0x3f76, [0x19d5] = 0x3f77, [0x19d6] = 0x3f78, [0x19d7] = 0x3f79,
  [0x19d8] = 0x3f7a, [0x19d9] = 0x3f7b, [0x19da] = 0x3f7c, [0x19db] = 0x3f7d,
  [0x19dc] = 0x3f7e, [0x19dd] = 0x3f7f, [0x19de] = 0x3f80, [0x19df] = 0x3f81,
  [0x19e0] = 0x3f82, [0x19e1] = 0x3f83, [0x19e2] = 0x3f84, [0x19e3] = 0x3f85,
  [0x19e4] = 0x3f86, [0x19e5] = 0x3f87, [0x19e6] = 0x3f88, [0x19e7] = 0x3f89,
  [0x19e8] = 0x3f8a, [0x19e9] = 0x3f8b, [0x19ea] = 0x3f8c, [0x19eb] = 0x3f8d,
  [0x19ec] = 0x3f8e, [0x19ed] = 0x3f8f, [0x19ee] = 0x3f90, [0x19ef] = 0x3f91,
  [0x19f0] = 0x3f92, [0x19f1] = 0x3f93, [0x19f2] = 0x3f94, [0x19f3] = 0x3f95,
  [0x19f4] = 0x3f96, [0x19f5] = 0x3f97, [0x19f6] = 0x3f98, [0x19f7] = 0x3f99,
  [0x19f8] = 0x3f9a, [0x19f9] = 0x3f9b, [0x19fa] = 0x3f9c, [0x19fb] = 0x3f9d,
  [0x19fc] = 0x3f9e, [0x19fd] = 0x3f9f, [0x19fe] = 0x3fa0, [0x19ff] = 0x3fa1,
  [0x1a00] = 0x3fa2, [0x1a01] = 0x3fa3, [0x1a02] = 0x3fa4, [0x1a03] = 0x3fa5,
  [0x1a04] = 0x3fa6, [0x1a05] = 0x3fa7, [0x1a06] = 0x3fa8, [0x1a07] = 0x3fa9,
  [0x1a08] = 0x3faa, [0x1a09] = 0x3fab, [0x1a0a] = 0x3fac, [0x1a0b] = 0x3fad,
  [0x1a0c] = 0x3fae, [0x1a0d] = 0x3faf, [0x1a0e] = 0x3fb0, [0x1a0f] = 0x3fb1,
  [0x1a10] = 0x3fb2, [0x1a11] = 0x3fb3, [0x1a12] = 0x3fb4, [0x1a13] = 0x3fb5,
  [0x1a14] = 0x3fb6, [0x1a15] = 0x3fb7, [0x1a16] = 0x3fb8, [0x1a17] = 0x3fb9,
  [0x1a18] = 0x3fba, [0x1a19] = 0x3fbb, [0x1a1a] = 0x3fbc, [0x1a1b] = 0x3fbd,
  [0x1a1c] = 0x3fbe, [0x1a1d] = 0x3fbf, [0x1a1e] = 0x3fc0, [0x1a1f] = 0x3fc1,
  [0x1a20] = 0x3fc2, [0x1a21] = 0x3fc3, [0x1a22] = 0x3fc4, [0x1a23] = 0x3fc5,
  [0x1a24] = 0x3fc6, [0x1a25] = 0x3fc7, [0x1a26] = 0x3fc8, [0x1a27] = 0x3fc9,
  [0x1a28] = 0x3fca, [0x1a29] = 0x3fcb, [0x1a2a] = 0x3fcc, [0x1a2b] = 0x3fcd,
  [0x1a2c] = 0x3fce, [0x1a2d] = 0x3fcf, [0x1a2e] = 0x3fd0, [0x1a2f] = 0x3fd1,
  [0x1a30] = 0x3fd2, [0x1a31] = 0x3fd3, [0x1a32] = 0x3fd4, [0x1a33] = 0x3fd5,
  [0x1a34] = 0x3fd6, [0x1a35] = 0x3fd7, [0x1a36] = 0x3fd8, [0x1a37] = 0x3fd9,
  [0x1a38] = 0x3fda, [0x1a39] = 0x3fdb, [0x1a3a] = 0x3fdc, [0x1a3b] = 0x3fdd,
  [0x1a3c] = 0x3fde, [0x1a3d] = 0x3fdf, [0x1a3e] = 0x3fe0, [0x1a3f] = 0x3fe1,
  [0x1a40] = 0x3fe2, [0x1a41] = 0x3fe3, [0x1a42] = 0x3fe4, [0x1a43] = 0x3fe5,
  [0x1a44] = 0x3fe6, [0x1a45] = 0x3fe7, [0x1a46] = 0x3fe8, [0x1a47] = 0x3fe9,
  [0x1a48] = 0x3fea, [0x1a49] = 0x3feb, [0x1a4a] = 0x3fec, [0x1a4b] = 0x3fed,
  [0x1a4c] = 0x3fee, [0x1a4d] = 0x3fef, [0x1a4e] = 0x3ff0, [0x1a4f] = 0x3ff1,
  [0x1a50] = 0x3ff2, [0x1a51] = 0x3ff3, [0x1a52] = 0x3ff4, [0x1a53] = 0x3ff5,
  [0x1a54] = 0x3ff6, [0x1a55] = 0x3ff7, [0x1a56] = 0x3ff8, [0x1a57] = 0x3ff9,
  [0x1a58] = 0x3ffa, [0x1a59] = 0x3ffb, [0x1a5a] = 0x3ffc, [0x1a5b] = 0x3ffd,
  [0x1a5c] = 0x3ffe, [0x1a5d] = 0x3fff, [0x1a5e] = 0x4000, [0x1a5f] = 0x4001,
  [0x1a60] = 0x4002, [0x1a61] = 0x4003, [0x1a62] = 0x4004, [0x1a63] = 0x4005,
  [0x1a64] = 0x4006, [0x1a65] = 0x4007, [0x1a66] = 0x4008, [0x1a67] = 0x4009,
  [0x1a68] = 0x400a, [0x1a69] = 0x400b, [0x1a6a] = 0x400c, [0x1a6b] = 0x400d,
  [0x1a6c] = 0x400e, [0x1a6d] = 0x400f, [0x1a6e] = 0x4010, [0x1a6f] = 0x4011,
  [0x1a70] = 0x4012, [0x1a71] = 0x4013, [0x1a72] = 0x4014, [0x1a73] = 0x4015,
  [0x1a74] = 0x4016, [0x1a75] = 0x4017, [0x1a76] = 0x4018, [0x1a77] = 0x4019,
  [0x1a78] = 0x401a, [0x1a79] = 0x401b, [0x1a7a] = 0x401c, [0x1a7b] = 0x401d,
  [0x1a7c] = 0x401e, [0x1a7d] = 0x401f, [0x1a7e] = 0x4020, [0x1a7f] = 0x4021,
  [0x1a80] = 0x4022, [0x1a81] = 0x4023, [0x1a82] = 0x4024, [0x1a83] = 0x4025,
  [0x1a84] = 0x4026, [0x1a85] = 0x4027, [0x1a86] = 0x4028, [0x1a87] = 0x4029,
  [0x1a88] = 0x402a, [0x1a89] = 0x402b, [0x1a8a] = 0x402c, [0x1a8b] = 0x402d,
  [0x1a8c] = 0x402e, [0x1a8d] = 0x402f, [0x1a8e] = 0x4030, [0x1a8f] = 0x4031,
  [0x1a90] = 0x4032, [0x1a91] = 0x4033, [0x1a92] = 0x4034, [0x1a93] = 0x4035,
  [0x1a94] = 0x4036, [0x1a95] = 0x4037, [0x1a96] = 0x4038, [0x1a97] = 0x4039,
  [0x1a98] = 0x403a, [0x1a99] = 0x403b, [0x1a9a] = 0x403c, [0x1a9b] = 0x403d,
  [0x1a9c] = 0x403e, [0x1a9d] = 0x403f, [0x1a9e] = 0x4040, [0x1a9f] = 0x4041,
  [0x1aa0] = 0x4042, [0x1aa1] = 0x4043, [0x1aa2] = 0x4044, [0x1aa3] = 0x4045,
  [0x1aa4] = 0x4046, [0x1aa5] = 0x4047, [0x1aa6] = 0x4048, [0x1aa7] = 0x4049,
  [0x1aa8] = 0x404a, [0x1aa9] = 0x404b, [0x1aaa] = 0x404c, [0x1aab] = 0x404d,
  [0x1aac] = 0x404e, [0x1aad] = 0x404f, [0x1aae] = 0x4050, [0x1aaf] = 0x4051,
  [0x1ab0] = 0x4052, [0x1ab1] = 0x4053, [0x1ab2] = 0x4054, [0x1ab3] = 0x4055,
  [0x1ab4] = 0x4057, [0x1ab5] = 0x4058, [0x1ab6] = 0x4059, [0x1ab7] = 0x405a,
  [0x1ab8] = 0x405b, [0x1ab9] = 0x405c, [0x1aba] = 0x405d, [0x1abb] = 0x405e,
  [0x1abc] = 0x405f, [0x1abd] = 0x4060, [0x1abe] = 0x4061, [0x1abf] = 0x4062,
  [0x1ac0] = 0x4063, [0x1ac1] = 0x4064, [0x1ac2] = 0x4065, [0x1ac3] = 0x4066,
  [0x1ac4] = 0x4067, [0x1ac5] = 0x4068, [0x1ac6] = 0x4069, [0x1ac7] = 0x406a,
  [0x1ac8] = 0x406b, [0x1ac9] = 0x406c, [0x1aca] = 0x406d, [0x1acb] = 0x406e,
  [0x1acc] = 0x406f, [0x1acd] = 0x4070, [0x1ace] = 0x4071, [0x1acf] = 0x4072,
  [0x1ad0] = 0x4073, [0x1ad1] = 0x4074, [0x1ad2] = 0x4075, [0x1ad3] = 0x4076,
  [0x1ad4] = 0x4077, [0x1ad5] = 0x4078, [0x1ad6] = 0x4079, [0x1ad7] = 0x407a,
  [0x1ad8] = 0x407b, [0x1ad9] = 0x407c, [0x1ada] = 0x407d, [0x1adb] = 0x407e,
  [0x1adc] = 0x407f, [0x1add] = 0x4080, [0x1ade] = 0x4081, [0x1adf] = 0x4082,
  [0x1ae0] = 0x4083, [0x1ae1] = 0x4084, [0x1ae2] = 0x4085, [0x1ae3] = 0x4086,
  [0x1ae4] = 0x4087, [0x1ae5] = 0x4088, [0x1ae6] = 0x4089, [0x1ae7] = 0x408a,
  [0x1ae8] = 0x408b, [0x1ae9] = 0x408c, [0x1aea] = 0x408d, [0x1aeb] = 0x408e,
  [0x1aec] = 0x408f, [0x1aed] = 0x4090, [0x1aee] = 0x4091, [0x1aef] = 0x4092,
  [0x1af0] = 0x4093, [0x1af1] = 0x4094, [0x1af2] = 0x4095, [0x1af3] = 0x4096,
  [0x1af4] = 0x4097, [0x1af5] = 0x4098, [0x1af6] = 0x4099, [0x1af7] = 0x409a,
  [0x1af8] = 0x409b, [0x1af9] = 0x409c, [0x1afa] = 0x409d, [0x1afb] = 0x409e,
  [0x1afc] = 0x409f, [0x1afd] = 0x40a0, [0x1afe] = 0x40a1, [0x1aff] = 0x40a2,
  [0x1b00] = 0x40a3, [0x1b01] = 0x40a4, [0x1b02] = 0x40a5, [0x1b03] = 0x40a6,
  [0x1b04] = 0x40a7, [0x1b05] = 0x40a8, [0x1b06] = 0x40a9, [0x1b07] = 0x40aa,
  [0x1b08] = 0x40ab, [0x1b09] = 0x40ac, [0x1b0a] = 0x40ad, [0x1b0b] = 0x40ae,
  [0x1b0c] = 0x40af, [0x1b0d] = 0x40b0, [0x1b0e] = 0x40b1, [0x1b0f] = 0x40b2,
  [0x1b10] = 0x40b3, [0x1b11] = 0x40b4, [0x1b12] = 0x40b5, [0x1b13] = 0x40b6,
  [0x1b14] = 0x40b7, [0x1b15] = 0x40b8, [0x1b16] = 0x40b9, [0x1b17] = 0x40ba,
  [0x1b18] = 0x40bb, [0x1b19] = 0x40bc, [0x1b1a] = 0x40bd, [0x1b1b] = 0x40be,
  [0x1b1c] = 0x40bf, [0x1b1d] = 0x40c0, [0x1b1e] = 0x40c1, [0x1b1f] = 0x40c2,
  [0x1b20] = 0x40c3, [0x1b21] = 0x40c4, [0x1b22] = 0x40c5, [0x1b23] = 0x40c6,
  [0x1b24] = 0x40c7, [0x1b25] = 0x40c8, [0x1b26] = 0x40c9, [0x1b27] = 0x40ca,
  [0x1b28] = 0x40cb, [0x1b29] = 0x40cc, [0x1b2a] = 0x40cd, [0x1b2b] = 0x40ce,
  [0x1b2c] = 0x40cf, [0x1b2d] = 0x40d0, [0x1b2e] = 0x40d1, [0x1b2f] = 0x40d2,
  [0x1b30] = 0x40d3, [0x1b31] = 0x40d4, [0x1b32] = 0x40d5, [0x1b33] = 0x40d6,
  [0x1b34] = 0x40d7, [0x1b35] = 0x40d8, [0x1b36] = 0x40d9, [0x1b37] = 0x40da,
  [0x1b38] = 0x40db, [0x1b39] = 0x40dc, [0x1b3a] = 0x40dd, [0x1b3b] = 0x40de,
  [0x1b3c] = 0x40df, [0x1b3d] = 0x40e0, [0x1b3e] = 0x40e1, [0x1b3f] = 0x40e2,
  [0x1b40] = 0x40e3, [0x1b41] = 0x40e4, [0x1b42] = 0x40e5, [0x1b43] = 0x40e6,
  [0x1b44] = 0x40e7, [0x1b45] = 0x40e8, [0x1b46] = 0x40e9, [0x1b47] = 0x40ea,
  [0x1b48] = 0x40eb, [0x1b49] = 0x40ec, [0x1b4a] = 0x40ed, [0x1b4b] = 0x40ee,
  [0x1b4c] = 0x40ef, [0x1b4d] = 0x40f0, [0x1b4e] = 0x40f1, [0x1b4f] = 0x40f2,
  [0x1b50] = 0x40f3, [0x1b51] = 0x40f4, [0x1b52] = 0x40f5, [0x1b53] = 0x40f6,
  [0x1b54] = 0x40f7, [0x1b55] = 0x40f8, [0x1b56] = 0x40f9, [0x1b57] = 0x40fa,
  [0x1b58] = 0x40fb, [0x1b59] = 0x40fc, [0x1b5a] = 0x40fd, [0x1b5b] = 0x40fe,
  [0x1b5c] = 0x40ff, [0x1b5d] = 0x4100, [0x1b5e] = 0x4101, [0x1b5f] = 0x4102,
  [0x1b60] = 0x4103, [0x1b61] = 0x4104, [0x1b62] = 0x4105, [0x1b63] = 0x4106,
  [0x1b64] = 0x4107, [0x1b65] = 0x4108, [0x1b66] = 0x4109, [0x1b67] = 0x410a,
  [0x1b68] = 0x410b, [0x1b69] = 0x410c, [0x1b6a] = 0x410d, [0x1b6b] = 0x410e,
  [0x1b6c] = 0x410f, [0x1b6d] = 0x4110, [0x1b6e] = 0x4111, [0x1b6f] = 0x4112,
  [0x1b70] = 0x4113, [0x1b71] = 0x4114, [0x1b72] = 0x4115, [0x1b73] = 0x4116,
  [0x1b74] = 0x4117, [0x1b75] = 0x4118, [0x1b76] = 0x4119, [0x1b77] = 0x411a,
  [0x1b78] = 0x411b, [0x1b79] = 0x411c, [0x1b7a] = 0x411d, [0x1b7b] = 0x411e,
  [0x1b7c] = 0x411f, [0x1b7d] = 0x4120, [0x1b7e] = 0x4121, [0x1b7f] = 0x4122,
  [0x1b80] = 0x4123, [0x1b81] = 0x4124, [0x1b82] = 0x4125, [0x1b83] = 0x4126,
  [0x1b84] = 0x4127, [0x1b85] = 0x4128, [0x1b86] = 0x4129, [0x1b87] = 0x412a,
  [0x1b88] = 0x412b, [0x1b89] = 0x412c, [0x1b8a] = 0x412d, [0x1b8b] = 0x412e,
  [0x1b8c] = 0x412f, [0x1b8d] = 0x4130, [0x1b8e] = 0x4131, [0x1b8f] = 0x4132,
  [0x1b90] = 0x4133, [0x1b91] = 0x4134, [0x1b92] = 0x4135, [0x1b93] = 0x4136,
  [0x1b94] = 0x4137, [0x1b95] = 0x4138, [0x1b96] = 0x4139, [0x1b97] = 0x413a,
  [0x1b98] = 0x413b, [0x1b99] = 0x413c, [0x1b9a] = 0x413d, [0x1b9b] = 0x413e,
  [0x1b9c] = 0x413f, [0x1b9d] = 0x4140, [0x1b9e] = 0x4141, [0x1b9f] = 0x4142,
  [0x1ba0] = 0x4143, [0x1ba1] = 0x4144, [0x1ba2] = 0x4145, [0x1ba3] = 0x4146,
  [0x1ba4] = 0x4147, [0x1ba5] = 0x4148, [0x1ba6] = 0x4149, [0x1ba7] = 0x414a,
  [0x1ba8] = 0x414b, [0x1ba9] = 0x414c, [0x1baa] = 0x414d, [0x1bab] = 0x414e,
  [0x1bac] = 0x414f, [0x1bad] = 0x4150, [0x1bae] = 0x4151, [0x1baf] = 0x4152,
  [0x1bb0] = 0x4153, [0x1bb1] = 0x4154, [0x1bb2] = 0x4155, [0x1bb3] = 0x4156,
  [0x1bb4] = 0x4157, [0x1bb5] = 0x4158, [0x1bb6] = 0x4159, [0x1bb7] = 0x415a,
  [0x1bb8] = 0x415b, [0x1bb9] = 0x415c, [0x1bba] = 0x415d, [0x1bbb] = 0x415e,
  [0x1bbc] = 0x4160, [0x1bbd] = 0x4161, [0x1bbe] = 0x4162, [0x1bbf] = 0x4163,
  [0x1bc0] = 0x4164, [0x1bc1] = 0x4165, [0x1bc2] = 0x4166, [0x1bc3] = 0x4167,
  [0x1bc4] = 0x4168, [0x1bc5] = 0x4169, [0x1bc6] = 0x416a, [0x1bc7] = 0x416b,
  [0x1bc8] = 0x416c, [0x1bc9] = 0x416d, [0x1bca] = 0x416e, [0x1bcb] = 0x416f,
  [0x1bcc] = 0x4170, [0x1bcd] = 0x4171, [0x1bce] = 0x4172, [0x1bcf] = 0x4173,
  [0x1bd0] = 0x4174, [0x1bd1] = 0x4175, [0x1bd2] = 0x4176, [0x1bd3] = 0x4177,
  [0x1bd4] = 0x4178, [0x1bd5] = 0x4179, [0x1bd6] = 0x417a, [0x1bd7] = 0x417b,
  [0x1bd8] = 0x417c, [0x1bd9] = 0x417d, [0x1bda] = 0x417e, [0x1bdb] = 0x417f,
  [0x1bdc] = 0x4180, [0x1bdd] = 0x4181, [0x1bde] = 0x4182, [0x1bdf] = 0x4183,
  [0x1be0] = 0x4184, [0x1be1] = 0x4185, [0x1be2] = 0x4186, [0x1be3] = 0x4187,
  [0x1be4] = 0x4188, [0x1be5] = 0x4189, [0x1be6] = 0x418a, [0x1be7] = 0x418b,
  [0x1be8] = 0x418c, [0x1be9] = 0x418d, [0x1bea] = 0x418e, [0x1beb] = 0x418f,
  [0x1bec] = 0x4190, [0x1bed] = 0x4191, [0x1bee] = 0x4192, [0x1bef] = 0x4193,
  [0x1bf0] = 0x4194, [0x1bf1] = 0x4195, [0x1bf2] = 0x4196, [0x1bf3] = 0x4197,
  [0x1bf4] = 0x4198, [0x1bf5] = 0x4199, [0x1bf6] = 0x419a, [0x1bf7] = 0x419b,
  [0x1bf8] = 0x419c, [0x1bf9] = 0x419d, [0x1bfa] = 0x419e, [0x1bfb] = 0x419f,
  [0x1bfc] = 0x41a0, [0x1bfd] = 0x41a1, [0x1bfe] = 0x41a2, [0x1bff] = 0x41a3,
  [0x1c00] = 0x41a4, [0x1c01] = 0x41a5, [0x1c02] = 0x41a6, [0x1c03] = 0x41a7,
  [0x1c04] = 0x41a8, [0x1c05] = 0x41a9, [0x1c06] = 0x41aa, [0x1c07] = 0x41ab,
  [0x1c08] = 0x41ac, [0x1c09] = 0x41ad, [0x1c0a] = 0x41ae, [0x1c0b] = 0x41af,
  [0x1c0c] = 0x41b0, [0x1c0d] = 0x41b1, [0x1c0e] = 0x41b2, [0x1c0f] = 0x41b3,
  [0x1c10] = 0x41b4, [0x1c11] = 0x41b5, [0x1c12] = 0x41b6, [0x1c13] = 0x41b7,
  [0x1c14] = 0x41b8, [0x1c15] = 0x41b9, [0x1c16] = 0x41ba, [0x1c17] = 0x41bb,
  [0x1c18] = 0x41bc, [0x1c19] = 0x41bd, [0x1c1a] = 0x41be, [0x1c1b] = 0x41bf,
  [0x1c1c] = 0x41c0, [0x1c1d] = 0x41c1, [0x1c1e] = 0x41c2, [0x1c1f] = 0x41c3,
  [0x1c20] = 0x41c4, [0x1c21] = 0x41c5, [0x1c22] = 0x41c6, [0x1c23] = 0x41c7,
  [0x1c24] = 0x41c8, [0x1c25] = 0x41c9, [0x1c26] = 0x41ca, [0x1c27] = 0x41cb,
  [0x1c28] = 0x41cc, [0x1c29] = 0x41cd, [0x1c2a] = 0x41ce, [0x1c2b] = 0x41cf,
  [0x1c2c] = 0x41d0, [0x1c2d] = 0x41d1, [0x1c2e] = 0x41d2, [0x1c2f] = 0x41d3,
  [0x1c30] = 0x41d4, [0x1c31] = 0x41d5, [0x1c32] = 0x41d6, [0x1c33] = 0x41d7,
  [0x1c34] = 0x41d8, [0x1c35] = 0x41d9, [0x1c36] = 0x41da, [0x1c37] = 0x41db,
  [0x1c38] = 0x41dc, [0x1c39] = 0x41dd, [0x1c3a] = 0x41de, [0x1c3b] = 0x41df,
  [0x1c3c] = 0x41e0, [0x1c3d] = 0x41e1, [0x1c3e] = 0x41e2, [0x1c3f] = 0x41e3,
  [0x1c40] = 0x41e4, [0x1c41] = 0x41e5, [0x1c42] = 0x41e6, [0x1c43] = 0x41e7,
  [0x1c44] = 0x41e8, [0x1c45] = 0x41e9, [0x1c46] = 0x41ea, [0x1c47] = 0x41eb,
  [0x1c48] = 0x41ec, [0x1c49] = 0x41ed, [0x1c4a] = 0x41ee, [0x1c4b] = 0x41ef,
  [0x1c4c] = 0x41f0, [0x1c4d] = 0x41f1, [0x1c4e] = 0x41f2, [0x1c4f] = 0x41f3,
  [0x1c50] = 0x41f4, [0x1c51] = 0x41f5, [0x1c52] = 0x41f6, [0x1c53] = 0x41f7,
  [0x1c54] = 0x41f8, [0x1c55] = 0x41f9, [0x1c56] = 0x41fa, [0x1c57] = 0x41fb,
  [0x1c58] = 0x41fc, [0x1c59] = 0x41fd, [0x1c5a] = 0x41fe, [0x1c5b] = 0x41ff,
  [0x1c5c] = 0x4200, [0x1c5d] = 0x4201, [0x1c5e] = 0x4202, [0x1c5f] = 0x4203,
  [0x1c60] = 0x4204, [0x1c61] = 0x4205, [0x1c62] = 0x4206, [0x1c63] = 0x4207,
  [0x1c64] = 0x4208, [0x1c65] = 0x4209, [0x1c66] = 0x420a, [0x1c67] = 0x420b,
  [0x1c68] = 0x420c, [0x1c69] = 0x420d, [0x1c6a] = 0x420e, [0x1c6b] = 0x420f,
  [0x1c6c] = 0x4210, [0x1c6d] = 0x4211, [0x1c6e] = 0x4212, [0x1c6f] = 0x4213,
  [0x1c70] = 0x4214, [0x1c71] = 0x4215, [0x1c72] = 0x4216, [0x1c73] = 0x4217,
  [0x1c74] = 0x4218, [0x1c75] = 0x4219, [0x1c76] = 0x421a, [0x1c77] = 0x421b,
  [0x1c78] = 0x421c, [0x1c79] = 0x421d, [0x1c7a] = 0x421e, [0x1c7b] = 0x421f,
  [0x1c7c] = 0x4220, [0x1c7d] = 0x4221, [0x1c7e] = 0x4222, [0x1c7f] = 0x4223,
  [0x1c80] = 0x4224, [0x1c81] = 0x4225, [0x1c82] = 0x4226, [0x1c83] = 0x4227,
  [0x1c84] = 0x4228, [0x1c85] = 0x4229, [0x1c86] = 0x422a, [0x1c87] = 0x422b,
  [0x1c88] = 0x422c, [0x1c89] = 0x422d, [0x1c8a] = 0x422e, [0x1c8b] = 0x422f,
  [0x1c8c] = 0x4230, [0x1c8d] = 0x4231, [0x1c8e] = 0x4232, [0x1c8f] = 0x4233,
  [0x1c90] = 0x4234, [0x1c91] = 0x4235, [0x1c92] = 0x4236, [0x1c93] = 0x4237,
  [0x1c94] = 0x4238, [0x1c95] = 0x4239, [0x1c96] = 0x423a, [0x1c97] = 0x423b,
  [0x1c98] = 0x423c, [0x1c99] = 0x423d, [0x1c9a] = 0x423e, [0x1c9b] = 0x423f,
  [0x1c9c] = 0x4240, [0x1c9d] = 0x4241, [0x1c9e] = 0x4242, [0x1c9f] = 0x4243,
  [0x1ca0] = 0x4244, [0x1ca1] = 0x4245, [0x1ca2] = 0x4246, [0x1ca3] = 0x4247,
  [0x1ca4] = 0x4248, [0x1ca5] = 0x4249, [0x1ca6] = 0x424a, [0x1ca7] = 0x424b,
  [0x1ca8] = 0x424c, [0x1ca9] = 0x424d, [0x1caa] = 0x424e, [0x1cab] = 0x424f,
  [0x1cac] = 0x4250, [0x1cad] = 0x4251, [0x1cae] = 0x4252, [0x1caf] = 0x4253,
  [0x1cb0] = 0x4254, [0x1cb1] = 0x4255, [0x1cb2] = 0x4256, [0x1cb3] = 0x4257,
  [0x1cb4] = 0x4258, [0x1cb5] = 0x4259, [0x1cb6] = 0x425a, [0x1cb7] = 0x425b,
  [0x1cb8] = 0x425c, [0x1cb9] = 0x425d, [0x1cba] = 0x425e, [0x1cbb] = 0x425f,
  [0x1cbc] = 0x4260, [0x1cbd] = 0x4261, [0x1cbe] = 0x4262, [0x1cbf] = 0x4263,
  [0x1cc0] = 0x4264, [0x1cc1] = 0x4265, [0x1cc2] = 0x4266, [0x1cc3] = 0x4267,
  [0x1cc4] = 0x4268, [0x1cc5] = 0x4269, [0x1cc6] = 0x426a, [0x1cc7] = 0x426b,
  [0x1cc8] = 0x426c, [0x1cc9] = 0x426d, [0x1cca] = 0x426e, [0x1ccb] = 0x426f,
  [0x1ccc] = 0x4270, [0x1ccd] = 0x4271, [0x1cce] = 0x4272, [0x1ccf] = 0x4273,
  [0x1cd0] = 0x4274, [0x1cd1] = 0x4275, [0x1cd2] = 0x4276, [0x1cd3] = 0x4277,
  [0x1cd4] = 0x4278, [0x1cd5] = 0x4279, [0x1cd6] = 0x427a, [0x1cd7] = 0x427b,
  [0x1cd8] = 0x427c, [0x1cd9] = 0x427d, [0x1cda] = 0x427e, [0x1cdb] = 0x427f,
  [0x1cdc] = 0x4280, [0x1cdd] = 0x4281, [0x1cde] = 0x4282, [0x1cdf] = 0x4283,
  [0x1ce0] = 0x4284, [0x1ce1] = 0x4285, [0x1ce2] = 0x4286, [0x1ce3] = 0x4287,
  [0x1ce4] = 0x4288, [0x1ce5] = 0x4289, [0x1ce6] = 0x428a, [0x1ce7] = 0x428b,
  [0x1ce8] = 0x428c, [0x1ce9] = 0x428d, [0x1cea] = 0x428e, [0x1ceb] = 0x428f,
  [0x1cec] = 0x4290, [0x1ced] = 0x4291, [0x1cee] = 0x4292, [0x1cef] = 0x4293,
  [0x1cf0] = 0x4294, [0x1cf1] = 0x4295, [0x1cf2] = 0x4296, [0x1cf3] = 0x4297,
  [0x1cf4] = 0x4298, [0x1cf5] = 0x4299, [0x1cf6] = 0x429a, [0x1cf7] = 0x429b,
  [0x1cf8] = 0x429c, [0x1cf9] = 0x429d, [0x1cfa] = 0x429e, [0x1cfb] = 0x429f,
  [0x1cfc] = 0x42a0, [0x1cfd] = 0x42a1, [0x1cfe] = 0x42a2, [0x1cff] = 0x42a3,
  [0x1d00] = 0x42a4, [0x1d01] = 0x42a5, [0x1d02] = 0x42a6, [0x1d03] = 0x42a7,
  [0x1d04] = 0x42a8, [0x1d05] = 0x42a9, [0x1d06] = 0x42aa, [0x1d07] = 0x42ab,
  [0x1d08] = 0x42ac, [0x1d09] = 0x42ad, [0x1d0a] = 0x42ae, [0x1d0b] = 0x42af,
  [0x1d0c] = 0x42b0, [0x1d0d] = 0x42b1, [0x1d0e] = 0x42b2, [0x1d0f] = 0x42b3,
  [0x1d10] = 0x42b4, [0x1d11] = 0x42b5, [0x1d12] = 0x42b6, [0x1d13] = 0x42b7,
  [0x1d14] = 0x42b8, [0x1d15] = 0x42b9, [0x1d16] = 0x42ba, [0x1d17] = 0x42bb,
  [0x1d18] = 0x42bc, [0x1d19] = 0x42bd, [0x1d1a] = 0x42be, [0x1d1b] = 0x42bf,
  [0x1d1c] = 0x42c0, [0x1d1d] = 0x42c1, [0x1d1e] = 0x42c2, [0x1d1f] = 0x42c3,
  [0x1d20] = 0x42c4, [0x1d21] = 0x42c5, [0x1d22] = 0x42c6, [0x1d23] = 0x42c7,
  [0x1d24] = 0x42c8, [0x1d25] = 0x42c9, [0x1d26] = 0x42ca, [0x1d27] = 0x42cb,
  [0x1d28] = 0x42cc, [0x1d29] = 0x42cd, [0x1d2a] = 0x42ce, [0x1d2b] = 0x42cf,
  [0x1d2c] = 0x42d0, [0x1d2d] = 0x42d1, [0x1d2e] = 0x42d2, [0x1d2f] = 0x42d3,
  [0x1d30] = 0x42d4, [0x1d31] = 0x42d5, [0x1d32] = 0x42d6, [0x1d33] = 0x42d7,
  [0x1d34] = 0x42d8, [0x1d35] = 0x42d9, [0x1d36] = 0x42da, [0x1d37] = 0x42db,
  [0x1d38] = 0x42dc, [0x1d39] = 0x42dd, [0x1d3a] = 0x42de, [0x1d3b] = 0x42df,
  [0x1d3c] = 0x42e0, [0x1d3d] = 0x42e1, [0x1d3e] = 0x42e2, [0x1d3f] = 0x42e3,
  [0x1d40] = 0x42e4, [0x1d41] = 0x42e5, [0x1d42] = 0x42e6, [0x1d43] = 0x42e7,
  [0x1d44] = 0x42e8, [0x1d45] = 0x42e9, [0x1d46] = 0x42ea, [0x1d47] = 0x42eb,
  [0x1d48] = 0x42ec, [0x1d49] = 0x42ed, [0x1d4a] = 0x42ee, [0x1d4b] = 0x42ef,
  [0x1d4c] = 0x42f0, [0x1d4d] = 0x42f1, [0x1d4e] = 0x42f2, [0x1d4f] = 0x42f3,
  [0x1d50] = 0x42f4, [0x1d51] = 0x42f5, [0x1d52] = 0x42f6, [0x1d53] = 0x42f7,
  [0x1d54] = 0x42f8, [0x1d55] = 0x42f9, [0x1d56] = 0x42fa, [0x1d57] = 0x42fb,
  [0x1d58] = 0x42fc, [0x1d59] = 0x42fd, [0x1d5a] = 0x42fe, [0x1d5b] = 0x42ff,
  [0x1d5c] = 0x4300, [0x1d5d] = 0x4301, [0x1d5e] = 0x4302, [0x1d5f] = 0x4303,
  [0x1d60] = 0x4304, [0x1d61] = 0x4305, [0x1d62] = 0x4306, [0x1d63] = 0x4307,
  [0x1d64] = 0x4308, [0x1d65] = 0x4309, [0x1d66] = 0x430a, [0x1d67] = 0x430b,
  [0x1d68] = 0x430c, [0x1d69] = 0x430d, [0x1d6a] = 0x430e, [0x1d6b] = 0x430f,
  [0x1d6c] = 0x4310, [0x1d6d] = 0x4311, [0x1d6e] = 0x4312, [0x1d6f] = 0x4313,
  [0x1d70] = 0x4314, [0x1d71] = 0x4315, [0x1d72] = 0x4316, [0x1d73] = 0x4317,
  [0x1d74] = 0x4318, [0x1d75] = 0x4319, [0x1d76] = 0x431a, [0x1d77] = 0x431b,
  [0x1d78] = 0x431c, [0x1d79] = 0x431d, [0x1d7a] = 0x431e, [0x1d7b] = 0x431f,
  [0x1d7c] = 0x4320, [0x1d7d] = 0x4321, [0x1d7e] = 0x4322, [0x1d7f] = 0x4323,
  [0x1d80] = 0x4324, [0x1d81] = 0x4325, [0x1d82] = 0x4326, [0x1d83] = 0x4327,
  [0x1d84] = 0x4328, [0x1d85] = 0x4329, [0x1d86] = 0x432a, [0x1d87] = 0x432b,
  [0x1d88] = 0x432c, [0x1d89] = 0x432d, [0x1d8a] = 0x432e, [0x1d8b] = 0x432f,
  [0x1d8c] = 0x4330, [0x1d8d] = 0x4331, [0x1d8e] = 0x4332, [0x1d8f] = 0x4333,
  [0x1d90] = 0x4334, [0x1d91] = 0x4335, [0x1d92] = 0x4336, [0x1d93] = 0x4338,
  [0x1d94] = 0x4339, [0x1d95] = 0x433a, [0x1d96] = 0x433b, [0x1d97] = 0x433c,
  [0x1d98] = 0x433d, [0x1d99] = 0x433e, [0x1d9a] = 0x433f, [0x1d9b] = 0x4340,
  [0x1d9c] = 0x4341, [0x1d9d] = 0x4342, [0x1d9e] = 0x4343, [0x1d9f] = 0x4344,
  [0x1da0] = 0x4345, [0x1da1] = 0x4346, [0x1da2] = 0x4347, [0x1da3] = 0x4348,
  [0x1da4] = 0x4349, [0x1da5] = 0x434a, [0x1da6] = 0x434b, [0x1da7] = 0x434c,
  [0x1da8] = 0x434d, [0x1da9] = 0x434e, [0x1daa] = 0x434f, [0x1dab] = 0x4350,
  [0x1dac] = 0x4351, [0x1dad] = 0x4352, [0x1dae] = 0x4353, [0x1daf] = 0x4354,
  [0x1db0] = 0x4355, [0x1db1] = 0x4356, [0x1db2] = 0x4357, [0x1db3] = 0x4358,
  [0x1db4] = 0x4359, [0x1db5] = 0x435a, [0x1db6] = 0x435b, [0x1db7] = 0x435c,
  [0x1db8] = 0x435d, [0x1db9] = 0x435e, [0x1dba] = 0x435f, [0x1dbb] = 0x4360,
  [0x1dbc] = 0x4361, [0x1dbd] = 0x4362, [0x1dbe] = 0x4363, [0x1dbf] = 0x4364,
  [0x1dc0] = 0x4365, [0x1dc1] = 0x4366, [0x1dc2] = 0x4367, [0x1dc3] = 0x4368,
  [0x1dc4] = 0x4369, [0x1dc5] = 0x436a, [0x1dc6] = 0x436b, [0x1dc7] = 0x436c,
  [0x1dc8] = 0x436d, [0x1dc9] = 0x436e, [0x1dca] = 0x436f, [0x1dcb] = 0x4370,
  [0x1dcc] = 0x4371, [0x1dcd] = 0x4372, [0x1dce] = 0x4373, [0x1dcf] = 0x4374,
  [0x1dd0] = 0x4375, [0x1dd1] = 0x4376, [0x1dd2] = 0x4377, [0x1dd3] = 0x4378,
  [0x1dd4] = 0x4379, [0x1dd5] = 0x437a, [0x1dd6] = 0x437b, [0x1dd7] = 0x437c,
  [0x1dd8] = 0x437d, [0x1dd9] = 0x437e, [0x1dda] = 0x437f, [0x1ddb] = 0x4380,
  [0x1ddc] = 0x4381, [0x1ddd] = 0x4382, [0x1dde] = 0x4383, [0x1ddf] = 0x4384,
  [0x1de0] = 0x4385, [0x1de1] = 0x4386, [0x1de2] = 0x4387, [0x1de3] = 0x4388,
  [0x1de4] = 0x4389, [0x1de5] = 0x438a, [0x1de6] = 0x438b, [0x1de7] = 0x438c,
  [0x1de8] = 0x438d, [0x1de9] = 0x438e, [0x1dea] = 0x438f, [0x1deb] = 0x4390,
  [0x1dec] = 0x4391, [0x1ded] = 0x4392, [0x1dee] = 0x4393, [0x1def] = 0x4394,
  [0x1df0] = 0x4395, [0x1df1] = 0x4396, [0x1df2] = 0x4397, [0x1df3] = 0x4398,
  [0x1df4] = 0x4399, [0x1df5] = 0x439a, [0x1df6] = 0x439b, [0x1df7] = 0x439c,
  [0x1df8] = 0x439d, [0x1df9] = 0x439e, [0x1dfa] = 0x439f, [0x1dfb] = 0x43a0,
  [0x1dfc] = 0x43a1, [0x1dfd] = 0x43a2, [0x1dfe] = 0x43a3, [0x1dff] = 0x43a4,
  [0x1e00] = 0x43a5, [0x1e01] = 0x43a6, [0x1e02] = 0x43a7, [0x1e03] = 0x43a8,
  [0x1e04] = 0x43a9, [0x1e05] = 0x43aa, [0x1e06] = 0x43ab, [0x1e07] = 0x43ad,
  [0x1e08] = 0x43ae, [0x1e09] = 0x43af, [0x1e0a] = 0x43b0, [0x1e0b] = 0x43b2,
  [0x1e0c] = 0x43b3, [0x1e0d] = 0x43b4, [0x1e0e] = 0x43b5, [0x1e0f] = 0x43b6,
  [0x1e10] = 0x43b7, [0x1e11] = 0x43b8, [0x1e12] = 0x43b9, [0x1e13] = 0x43ba,
  [0x1e14] = 0x43bb, [0x1e15] = 0x43bc, [0x1e16] = 0x43bd, [0x1e17] = 0x43be,
  [0x1e18] = 0x43bf, [0x1e19] = 0x43c0, [0x1e1a] = 0x43c1, [0x1e1b] = 0x43c2,
  [0x1e1c] = 0x43c3, [0x1e1d] = 0x43c4, [0x1e1e] = 0x43c5, [0x1e1f] = 0x43c6,
  [0x1e20] = 0x43c7, [0x1e21] = 0x43c8, [0x1e22] = 0x43c9, [0x1e23] = 0x43ca,
  [0x1e24] = 0x43cb, [0x1e25] = 0x43cc, [0x1e26] = 0x43cd, [0x1e27] = 0x43ce,
  [0x1e28] = 0x43cf, [0x1e29] = 0x43d0, [0x1e2a] = 0x43d1, [0x1e2b] = 0x43d2,
  [0x1e2c] = 0x43d3, [0x1e2d] = 0x43d4, [0x1e2e] = 0x43d5, [0x1e2f] = 0x43d6,
  [0x1e30] = 0x43d7, [0x1e31] = 0x43d8, [0x1e32] = 0x43d9, [0x1e33] = 0x43da,
  [0x1e34] = 0x43db, [0x1e35] = 0x43dc, [0x1e36] = 0x43de, [0x1e37] = 0x43df,
  [0x1e38] = 0x43e0, [0x1e39] = 0x43e1, [0x1e3a] = 0x43e2, [0x1e3b] = 0x43e3,
  [0x1e3c] = 0x43e4, [0x1e3d] = 0x43e5, [0x1e3e] = 0x43e6, [0x1e3f] = 0x43e7,
  [0x1e40] = 0x43e8, [0x1e41] = 0x43e9, [0x1e42] = 0x43ea, [0x1e43] = 0x43eb,
  [0x1e44] = 0x43ec, [0x1e45] = 0x43ed, [0x1e46] = 0x43ee, [0x1e47] = 0x43ef,
  [0x1e48] = 0x43f0, [0x1e49] = 0x43f1, [0x1e4a] = 0x43f2, [0x1e4b] = 0x43f3,
  [0x1e4c] = 0x43f4, [0x1e4d] = 0x43f5, [0x1e4e] = 0x43f6, [0x1e4f] = 0x43f7,
  [0x1e50] = 0x43f8, [0x1e51] = 0x43f9, [0x1e52] = 0x43fa, [0x1e53] = 0x43fb,
  [0x1e54] = 0x43fc, [0x1e55] = 0x43fd, [0x1e56] = 0x43fe, [0x1e57] = 0x43ff,
  [0x1e58] = 0x4400, [0x1e59] = 0x4401, [0x1e5a] = 0x4402, [0x1e5b] = 0x4403,
  [0x1e5c] = 0x4404, [0x1e5d] = 0x4405, [0x1e5e] = 0x4406, [0x1e5f] = 0x4407,
  [0x1e60] = 0x4408, [0x1e61] = 0x4409, [0x1e62] = 0x440a, [0x1e63] = 0x440b,
  [0x1e64] = 0x440c, [0x1e65] = 0x440d, [0x1e66] = 0x440e, [0x1e67] = 0x440f,
  [0x1e68] = 0x4410, [0x1e69] = 0x4411, [0x1e6a] = 0x4412, [0x1e6b] = 0x4413,
  [0x1e6c] = 0x4414, [0x1e6d] = 0x4415, [0x1e6e] = 0x4416, [0x1e6f] = 0x4417,
  [0x1e70] = 0x4418, [0x1e71] = 0x4419, [0x1e72] = 0x441a, [0x1e73] = 0x441b,
  [0x1e74] = 0x441c, [0x1e75] = 0x441d, [0x1e76] = 0x441e, [0x1e77] = 0x441f,
  [0x1e78] = 0x4420, [0x1e79] = 0x4421, [0x1e7a] = 0x4422, [0x1e7b] = 0x4423,
  [0x1e7c] = 0x4424, [0x1e7d] = 0x4425, [0x1e7e] = 0x4426, [0x1e7f] = 0x4427,
  [0x1e80] = 0x4428, [0x1e81] = 0x4429, [0x1e82] = 0x442a, [0x1e83] = 0x442b,
  [0x1e84] = 0x442c, [0x1e85] = 0x442d, [0x1e86] = 0x442e, [0x1e87] = 0x442f,
  [0x1e88] = 0x4430, [0x1e89] = 0x4431, [0x1e8a] = 0x4432, [0x1e8b] = 0x4433,
  [0x1e8c] = 0x4434, [0x1e8d] = 0x4435, [0x1e8e] = 0x4436, [0x1e8f] = 0x4437,
  [0x1e90] = 0x4438, [0x1e91] = 0x4439, [0x1e92] = 0x443a, [0x1e93] = 0x443b,
  [0x1e94] = 0x443c, [0x1e95] = 0x443d, [0x1e96] = 0x443e, [0x1e97] = 0x443f,
  [0x1e98] = 0x4440, [0x1e99] = 0x4441, [0x1e9a] = 0x4442, [0x1e9b] = 0x4443,
  [0x1e9c] = 0x4444, [0x1e9d] = 0x4445, [0x1e9e] = 0x4446, [0x1e9f] = 0x4447,
  [0x1ea0] = 0x4448, [0x1ea1] = 0x4449, [0x1ea2] = 0x444a, [0x1ea3] = 0x444b,
  [0x1ea4] = 0x444c, [0x1ea5] = 0x444d, [0x1ea6] = 0x444e, [0x1ea7] = 0x444f,
  [0x1ea8] = 0x4450, [0x1ea9] = 0x4451, [0x1eaa] = 0x4452, [0x1eab] = 0x4453,
  [0x1eac] = 0x4454, [0x1ead] = 0x4455, [0x1eae] = 0x4456, [0x1eaf] = 0x4457,
  [0x1eb0] = 0x4458, [0x1eb1] = 0x4459, [0x1eb2] = 0x445a, [0x1eb3] = 0x445b,
  [0x1eb4] = 0x445c, [0x1eb5] = 0x445d, [0x1eb6] = 0x445e, [0x1eb7] = 0x445f,
  [0x1eb8] = 0x4460, [0x1eb9] = 0x4461, [0x1eba] = 0x4462, [0x1ebb] = 0x4463,
  [0x1ebc] = 0x4464, [0x1ebd] = 0x4465, [0x1ebe] = 0x4466, [0x1ebf] = 0x4467,
  [0x1ec0] = 0x4468, [0x1ec1] = 0x4469, [0x1ec2] = 0x446a, [0x1ec3] = 0x446b,
  [0x1ec4] = 0x446c, [0x1ec5] = 0x446d, [0x1ec6] = 0x446e, [0x1ec7] = 0x446f,
  [0x1ec8] = 0x4470, [0x1ec9] = 0x4471, [0x1eca] = 0x4472, [0x1ecb] = 0x4473,
  [0x1ecc] = 0x4474, [0x1ecd] = 0x4475, [0x1ece] = 0x4476, [0x1ecf] = 0x4477,
  [0x1ed0] = 0x4478, [0x1ed1] = 0x4479, [0x1ed2] = 0x447a, [0x1ed3] = 0x447b,
  [0x1ed4] = 0x447c, [0x1ed5] = 0x447d, [0x1ed6] = 0x447e, [0x1ed7] = 0x447f,
  [0x1ed8] = 0x4480, [0x1ed9] = 0x4481, [0x1eda] = 0x4482, [0x1edb] = 0x4483,
  [0x1edc] = 0x4484, [0x1edd] = 0x4485, [0x1ede] = 0x4486, [0x1edf] = 0x4487,
  [0x1ee0] = 0x4488, [0x1ee1] = 0x4489, [0x1ee2] = 0x448a, [0x1ee3] = 0x448b,
  [0x1ee4] = 0x448c, [0x1ee5] = 0x448d, [0x1ee6] = 0x448e, [0x1ee7] = 0x448f,
  [0x1ee8] = 0x4490, [0x1ee9] = 0x4491, [0x1eea] = 0x4492, [0x1eeb] = 0x4493,
  [0x1eec] = 0x4494, [0x1eed] = 0x4495, [0x1eee] = 0x4496, [0x1eef] = 0x4497,
  [0x1ef0] = 0x4498, [0x1ef1] = 0x4499, [0x1ef2] = 0x449a, [0x1ef3] = 0x449b,
  [0x1ef4] = 0x449c, [0x1ef5] = 0x449d, [0x1ef6] = 0x449e, [0x1ef7] = 0x449f,
  [0x1ef8] = 0x44a0, [0x1ef9] = 0x44a1, [0x1efa] = 0x44a2, [0x1efb] = 0x44a3,
  [0x1efc] = 0x44a4, [0x1efd] = 0x44a5, [0x1efe] = 0x44a6, [0x1eff] = 0x44a7,
  [0x1f00] = 0x44a8, [0x1f01] = 0x44a9, [0x1f02] = 0x44aa, [0x1f03] = 0x44ab,
  [0x1f04] = 0x44ac, [0x1f05] = 0x44ad, [0x1f06] = 0x44ae, [0x1f07] = 0x44af,
  [0x1f08] = 0x44b0, [0x1f09] = 0x44b1, [0x1f0a] = 0x44b2, [0x1f0b] = 0x44b3,
  [0x1f0c] = 0x44b4, [0x1f0d] = 0x44b5, [0x1f0e] = 0x44b6, [0x1f0f] = 0x44b7,
  [0x1f10] = 0x44b8, [0x1f11] = 0x44b9, [0x1f12] = 0x44ba, [0x1f13] = 0x44bb,
  [0x1f14] = 0x44bc, [0x1f15] = 0x44bd, [0x1f16] = 0x44be, [0x1f17] = 0x44bf,
  [0x1f18] = 0x44c0, [0x1f19] = 0x44c1, [0x1f1a] = 0x44c2, [0x1f1b] = 0x44c3,
  [0x1f1c] = 0x44c4, [0x1f1d] = 0x44c5, [0x1f1e] = 0x44c6, [0x1f1f] = 0x44c7,
  [0x1f20] = 0x44c8, [0x1f21] = 0x44c9, [0x1f22] = 0x44ca, [0x1f23] = 0x44cb,
  [0x1f24] = 0x44cc, [0x1f25] = 0x44cd, [0x1f26] = 0x44ce, [0x1f27] = 0x44cf,
  [0x1f28] = 0x44d0, [0x1f29] = 0x44d1, [0x1f2a] = 0x44d2, [0x1f2b] = 0x44d3,
  [0x1f2c] = 0x44d4, [0x1f2d] = 0x44d5, [0x1f2e] = 0x44d7, [0x1f2f] = 0x44d8,
  [0x1f30] = 0x44d9, [0x1f31] = 0x44da, [0x1f32] = 0x44db, [0x1f33] = 0x44dc,
  [0x1f34] = 0x44dd, [0x1f35] = 0x44de, [0x1f36] = 0x44df, [0x1f37] = 0x44e0,
  [0x1f38] = 0x44e1, [0x1f39] = 0x44e2, [0x1f3a] = 0x44e3, [0x1f3b] = 0x44e4,
  [0x1f3c] = 0x44e5, [0x1f3d] = 0x44e6, [0x1f3e] = 0x44e7, [0x1f3f] = 0x44e8,
  [0x1f40] = 0x44e9, [0x1f41] = 0x44ea, [0x1f42] = 0x44eb, [0x1f43] = 0x44ec,
  [0x1f44] = 0x44ed, [0x1f45] = 0x44ee, [0x1f46] = 0x44ef, [0x1f47] = 0x44f0,
  [0x1f48] = 0x44f1, [0x1f49] = 0x44f2, [0x1f4a] = 0x44f3, [0x1f4b] = 0x44f4,
  [0x1f4c] = 0x44f5, [0x1f4d] = 0x44f6, [0x1f4e] = 0x44f7, [0x1f4f] = 0x44f8,
  [0x1f50] = 0x44f9, [0x1f51] = 0x44fa, [0x1f52] = 0x44fb, [0x1f53] = 0x44fc,
  [0x1f54] = 0x44fd, [0x1f55] = 0x44fe, [0x1f56] = 0x44ff, [0x1f57] = 0x4500,
  [0x1f58] = 0x4501, [0x1f59] = 0x4502, [0x1f5a] = 0x4503, [0x1f5b] = 0x4504,
  [0x1f5c] = 0x4505, [0x1f5d] = 0x4506, [0x1f5e] = 0x4507, [0x1f5f] = 0x4508,
  [0x1f60] = 0x4509, [0x1f61] = 0x450a, [0x1f62] = 0x450b, [0x1f63] = 0x450c,
  [0x1f64] = 0x450d, [0x1f65] = 0x450e, [0x1f66] = 0x450f, [0x1f67] = 0x4510,
  [0x1f68] = 0x4511, [0x1f69] = 0x4512, [0x1f6a] = 0x4513, [0x1f6b] = 0x4514,
  [0x1f6c] = 0x4515, [0x1f6d] = 0x4516, [0x1f6e] = 0x4517, [0x1f6f] = 0x4518,
  [0x1f70] = 0x4519, [0x1f71] = 0x451a, [0x1f72] = 0x451b, [0x1f73] = 0x451c,
  [0x1f74] = 0x451d, [0x1f75] = 0x451e, [0x1f76] = 0x451f, [0x1f77] = 0x4520,
  [0x1f78] = 0x4521, [0x1f79] = 0x4522, [0x1f7a] = 0x4523, [0x1f7b] = 0x4524,
  [0x1f7c] = 0x4525, [0x1f7d] = 0x4526, [0x1f7e] = 0x4527, [0x1f7f] = 0x4528,
  [0x1f80] = 0x4529, [0x1f81] = 0x452a, [0x1f82] = 0x452b, [0x1f83] = 0x452c,
  [0x1f84] = 0x452d, [0x1f85] = 0x452e, [0x1f86] = 0x452f, [0x1f87] = 0x4530,
  [0x1f88] = 0x4531, [0x1f89] = 0x4532, [0x1f8a] = 0x4533, [0x1f8b] = 0x4534,
  [0x1f8c] = 0x4535, [0x1f8d] = 0x4536, [0x1f8e] = 0x4537, [0x1f8f] = 0x4538,
  [0x1f90] = 0x4539, [0x1f91] = 0x453a, [0x1f92] = 0x453b, [0x1f93] = 0x453c,
  [0x1f94] = 0x453d, [0x1f95] = 0x453e, [0x1f96] = 0x453f, [0x1f97] = 0x4540,
  [0x1f98] = 0x4541, [0x1f99] = 0x4542, [0x1f9a] = 0x4543, [0x1f9b] = 0x4544,
  [0x1f9c] = 0x4545, [0x1f9d] = 0x4546, [0x1f9e] = 0x4547, [0x1f9f] = 0x4548,
  [0x1fa0] = 0x4549, [0x1fa1] = 0x454a, [0x1fa2] = 0x454b, [0x1fa3] = 0x454c,
  [0x1fa4] = 0x454d, [0x1fa5] = 0x454e, [0x1fa6] = 0x454f, [0x1fa7] = 0x4550,
  [0x1fa8] = 0x4551, [0x1fa9] = 0x4552, [0x1faa] = 0x4553, [0x1fab] = 0x4554,
  [0x1fac] = 0x4555, [0x1fad] = 0x4556, [0x1fae] = 0x4557, [0x1faf] = 0x4558,
  [0x1fb0] = 0x4559, [0x1fb1] = 0x455a, [0x1fb2] = 0x455b, [0x1fb3] = 0x455c,
  [0x1fb4] = 0x455d, [0x1fb5] = 0x455e, [0x1fb6] = 0x455f, [0x1fb7] = 0x4560,
  [0x1fb8] = 0x4561, [0x1fb9] = 0x4562, [0x1fba] = 0x4563, [0x1fbb] = 0x4564,
  [0x1fbc] = 0x4565, [0x1fbd] = 0x4566, [0x1fbe] = 0x4567, [0x1fbf] = 0x4568,
  [0x1fc0] = 0x4569, [0x1fc1] = 0x456a, [0x1fc2] = 0x456b, [0x1fc3] = 0x456c,
  [0x1fc4] = 0x456d, [0x1fc5] = 0x456e, [0x1fc6] = 0x456f, [0x1fc7] = 0x4570,
  [0x1fc8] = 0x4571, [0x1fc9] = 0x4572, [0x1fca] = 0x4573, [0x1fcb] = 0x4574,
  [0x1fcc] = 0x4575, [0x1fcd] = 0x4576, [0x1fce] = 0x4577, [0x1fcf] = 0x4578,
  [0x1fd0] = 0x4579, [0x1fd1] = 0x457a, [0x1fd2] = 0x457b, [0x1fd3] = 0x457c,
  [0x1fd4] = 0x457d, [0x1fd5] = 0x457e, [0x1fd6] = 0x457f, [0x1fd7] = 0x4580,
  [0x1fd8] = 0x4581, [0x1fd9] = 0x4582, [0x1fda] = 0x4583, [0x1fdb] = 0x4584,
  [0x1fdc] = 0x4585, [0x1fdd] = 0x4586, [0x1fde] = 0x4587, [0x1fdf] = 0x4588,
  [0x1fe0] = 0x4589, [0x1fe1] = 0x458a, [0x1fe2] = 0x458b, [0x1fe3] = 0x458c,
  [0x1fe4] = 0x458d, [0x1fe5] = 0x458e, [0x1fe6] = 0x458f, [0x1fe7] = 0x4590,
  [0x1fe8] = 0x4591, [0x1fe9] = 0x4592, [0x1fea] = 0x4593, [0x1feb] = 0x4594,
  [0x1fec] = 0x4595, [0x1fed] = 0x4596, [0x1fee] = 0x4597, [0x1fef] = 0x4598,
  [0x1ff0] = 0x4599, [0x1ff1] = 0x459a, [0x1ff2] = 0x459b, [0x1ff3] = 0x459c,
  [0x1ff4] = 0x459d, [0x1ff5] = 0x459e, [0x1ff6] = 0x459f, [0x1ff7] = 0x45a0,
  [0x1ff8] = 0x45a1, [0x1ff9] = 0x45a2, [0x1ffa] = 0x45a3, [0x1ffb] = 0x45a4,
  [0x1ffc] = 0x45a5, [0x1ffd] = 0x45a6, [0x1ffe] = 0x45a7, [0x1fff] = 0x45a8,
  [0x2000] = 0x45a9, [0x2001] = 0x45aa, [0x2002] = 0x45ab, [0x2003] = 0x45ac,
  [0x2004] = 0x45ad, [0x2005] = 0x45ae, [0x2006] = 0x45af, [0x2007] = 0x45b0,
  [0x2008] = 0x45b1, [0x2009] = 0x45b2, [0x200a] = 0x45b3, [0x200b] = 0x45b4,
  [0x200c] = 0x45b5, [0x200d] = 0x45b6, [0x200e] = 0x45b7, [0x200f] = 0x45b8,
  [0x2010] = 0x45b9, [0x2011] = 0x45ba, [0x2012] = 0x45bb, [0x2013] = 0x45bc,
  [0x2014] = 0x45bd, [0x2015] = 0x45be, [0x2016] = 0x45bf, [0x2017] = 0x45c0,
  [0x2018] = 0x45c1, [0x2019] = 0x45c2, [0x201a] = 0x45c3, [0x201b] = 0x45c4,
  [0x201c] = 0x45c5, [0x201d] = 0x45c6, [0x201e] = 0x45c7, [0x201f] = 0x45c8,
  [0x2020] = 0x45c9, [0x2021] = 0x45ca, [0x2022] = 0x45cb, [0x2023] = 0x45cc,
  [0x2024] = 0x45cd, [0x2025] = 0x45ce, [0x2026] = 0x45cf, [0x2027] = 0x45d0,
  [0x2028] = 0x45d1, [0x2029] = 0x45d2, [0x202a] = 0x45d3, [0x202b] = 0x45d4,
  [0x202c] = 0x45d5, [0x202d] = 0x45d6, [0x202e] = 0x45d7, [0x202f] = 0x45d8,
  [0x2030] = 0x45d9, [0x2031] = 0x45da, [0x2032] = 0x45db, [0x2033] = 0x45dc,
  [0x2034] = 0x45dd, [0x2035] = 0x45de, [0x2036] = 0x45df, [0x2037] = 0x45e0,
  [0x2038] = 0x45e1, [0x2039] = 0x45e2, [0x203a] = 0x45e3, [0x203b] = 0x45e4,
  [0x203c] = 0x45e5, [0x203d] = 0x45e6, [0x203e] = 0x45e7, [0x203f] = 0x45e8,
  [0x2040] = 0x45e9, [0x2041] = 0x45ea, [0x2042] = 0x45eb, [0x2043] = 0x45ec,
  [0x2044] = 0x45ed, [0x2045] = 0x45ee, [0x2046] = 0x45ef, [0x2047] = 0x45f0,
  [0x2048] = 0x45f1, [0x2049] = 0x45f2, [0x204a] = 0x45f3, [0x204b] = 0x45f4,
  [0x204c] = 0x45f5, [0x204d] = 0x45f6, [0x204e] = 0x45f7, [0x204f] = 0x45f8,
  [0x2050] = 0x45f9, [0x2051] = 0x45fa, [0x2052] = 0x45fb, [0x2053] = 0x45fc,
  [0x2054] = 0x45fd, [0x2055] = 0x45fe, [0x2056] = 0x45ff, [0x2057] = 0x4600,
  [0x2058] = 0x4601, [0x2059] = 0x4602, [0x205a] = 0x4603, [0x205b] = 0x4604,
  [0x205c] = 0x4605, [0x205d] = 0x4606, [0x205e] = 0x4607, [0x205f] = 0x4608,
  [0x2060] = 0x4609, [0x2061] = 0x460a, [0x2062] = 0x460b, [0x2063] = 0x460c,
  [0x2064] = 0x460d, [0x2065] = 0x460e, [0x2066] = 0x460f, [0x2067] = 0x4610,
  [0x2068] = 0x4611, [0x2069] = 0x4612, [0x206a] = 0x4613, [0x206b] = 0x4614,
  [0x206c] = 0x4615, [0x206d] = 0x4616, [0x206e] = 0x4617, [0x206f] = 0x4618,
  [0x2070] = 0x4619, [0x2071] = 0x461a, [0x2072] = 0x461b, [0x2073] = 0x461c,
  [0x2074] = 0x461d, [0x2075] = 0x461e, [0x2076] = 0x461f, [0x2077] = 0x4620,
  [0x2078] = 0x4621, [0x2079] = 0x4622, [0x207a] = 0x4623, [0x207b] = 0x4624,
  [0x207c] = 0x4625, [0x207d] = 0x4626, [0x207e] = 0x4627, [0x207f] = 0x4628,
  [0x2080] = 0x4629, [0x2081] = 0x462a, [0x2082] = 0x462b, [0x2083] = 0x462c,
  [0x2084] = 0x462d, [0x2085] = 0x462e, [0x2086] = 0x462f, [0x2087] = 0x4630,
  [0x2088] = 0x4631, [0x2089] = 0x4632, [0x208a] = 0x4633, [0x208b] = 0x4634,
  [0x208c] = 0x4635, [0x208d] = 0x4636, [0x208e] = 0x4637, [0x208f] = 0x4638,
  [0x2090] = 0x4639, [0x2091] = 0x463a, [0x2092] = 0x463b, [0x2093] = 0x463c,
  [0x2094] = 0x463d, [0x2095] = 0x463e, [0x2096] = 0x463f, [0x2097] = 0x4640,
  [0x2098] = 0x4641, [0x2099] = 0x4642, [0x209a] = 0x4643, [0x209b] = 0x4644,
  [0x209c] = 0x4645, [0x209d] = 0x4646, [0x209e] = 0x4647, [0x209f] = 0x4648,
  [0x20a0] = 0x4649, [0x20a1] = 0x464a, [0x20a2] = 0x464b, [0x20a3] = 0x464d,
  [0x20a4] = 0x464e, [0x20a5] = 0x464f, [0x20a6] = 0x4650, [0x20a7] = 0x4651,
  [0x20a8] = 0x4652, [0x20a9] = 0x4653, [0x20aa] = 0x4654, [0x20ab] = 0x4655,
  [0x20ac] = 0x4656, [0x20ad] = 0x4657, [0x20ae] = 0x4658, [0x20af] = 0x4659,
  [0x20b0] = 0x465a, [0x20b1] = 0x465b, [0x20b2] = 0x465c, [0x20b3] = 0x465d,
  [0x20b4] = 0x465e, [0x20b5] = 0x465f, [0x20b6] = 0x4660, [0x20b7] = 0x4662,
  [0x20b8] = 0x4663, [0x20b9] = 0x4664, [0x20ba] = 0x4665, [0x20bb] = 0x4666,
  [0x20bc] = 0x4667, [0x20bd] = 0x4668, [0x20be] = 0x4669, [0x20bf] = 0x466a,
  [0x20c0] = 0x466b, [0x20c1] = 0x466c, [0x20c2] = 0x466d, [0x20c3] = 0x466e,
  [0x20c4] = 0x466f, [0x20c5] = 0x4670, [0x20c6] = 0x4671, [0x20c7] = 0x4672,
  [0x20c8] = 0x4673, [0x20c9] = 0x4674, [0x20ca] = 0x4675, [0x20cb] = 0x4676,
  [0x20cc] = 0x4677, [0x20cd] = 0x4678, [0x20ce] = 0x4679, [0x20cf] = 0x467a,
  [0x20d0] = 0x467b, [0x20d1] = 0x467c, [0x20d2] = 0x467d, [0x20d3] = 0x467e,
  [0x20d4] = 0x467f, [0x20d5] = 0x4680, [0x20d6] = 0x4681, [0x20d7] = 0x4682,
  [0x20d8] = 0x4683, [0x20d9] = 0x4684, [0x20da] = 0x4685, [0x20db] = 0x4686,
  [0x20dc] = 0x4687, [0x20dd] = 0x4688, [0x20de] = 0x4689, [0x20df] = 0x468a,
  [0x20e0] = 0x468b, [0x20e1] = 0x468c, [0x20e2] = 0x468d, [0x20e3] = 0x468e,
  [0x20e4] = 0x468f, [0x20e5] = 0x4690, [0x20e6] = 0x4691, [0x20e7] = 0x4692,
  [0x20e8] = 0x4693, [0x20e9] = 0x4694, [0x20ea] = 0x4695, [0x20eb] = 0x4696,
  [0x20ec] = 0x4697, [0x20ed] = 0x4698, [0x20ee] = 0x4699, [0x20ef] = 0x469a,
  [0x20f0] = 0x469b, [0x20f1] = 0x469c, [0x20f2] = 0x469d, [0x20f3] = 0x469e,
  [0x20f4] = 0x469f, [0x20f5] = 0x46a0, [0x20f6] = 0x46a1, [0x20f7] = 0x46a2,
  [0x20f8] = 0x46a3, [0x20f9] = 0x46a4, [0x20fa] = 0x46a5, [0x20fb] = 0x46a6,
  [0x20fc] = 0x46a7, [0x20fd] = 0x46a8, [0x20fe] = 0x46a9, [0x20ff] = 0x46aa,
  [0x2100] = 0x46ab, [0x2101] = 0x46ac, [0x2102] = 0x46ad, [0x2103] = 0x46ae,
  [0x2104] = 0x46af, [0x2105] = 0x46b0, [0x2106] = 0x46b1, [0x2107] = 0x46b2,
  [0x2108] = 0x46b3, [0x2109] = 0x46b4, [0x210a] = 0x46b5, [0x210b] = 0x46b6,
  [0x210c] = 0x46b7, [0x210d] = 0x46b8, [0x210e] = 0x46b9, [0x210f] = 0x46ba,
  [0x2110] = 0x46bb, [0x2111] = 0x46bc, [0x2112] = 0x46bd, [0x2113] = 0x46be,
  [0x2114] = 0x46bf, [0x2115] = 0x46c0, [0x2116] = 0x46c1, [0x2117] = 0x46c2,
  [0x2118] = 0x46c3, [0x2119] = 0x46c4, [0x211a] = 0x46c5, [0x211b] = 0x46c6,
  [0x211c] = 0x46c7, [0x211d] = 0x46c8, [0x211e] = 0x46c9, [0x211f] = 0x46ca,
  [0x2120] = 0x46cb, [0x2121] = 0x46cc, [0x2122] = 0x46cd, [0x2123] = 0x46ce,
  [0x2124] = 0x46cf, [0x2125] = 0x46d0, [0x2126] = 0x46d1, [0x2127] = 0x46d2,
  [0x2128] = 0x46d3, [0x2129] = 0x46d4, [0x212a] = 0x46d5, [0x212b] = 0x46d6,
  [0x212c] = 0x46d7, [0x212d] = 0x46d8, [0x212e] = 0x46d9, [0x212f] = 0x46da,
  [0x2130] = 0x46db, [0x2131] = 0x46dc, [0x2132] = 0x46dd, [0x2133] = 0x46de,
  [0x2134] = 0x46df, [0x2135] = 0x46e0, [0x2136] = 0x46e1, [0x2137] = 0x46e2,
  [0x2138] = 0x46e3, [0x2139] = 0x46e4, [0x213a] = 0x46e5, [0x213b] = 0x46e6,
  [0x213c] = 0x46e7, [0x213d] = 0x46e8, [0x213e] = 0x46e9, [0x213f] = 0x46ea,
  [0x2140] = 0x46eb, [0x2141] = 0x46ec, [0x2142] = 0x46ed, [0x2143] = 0x46ee,
  [0x2144] = 0x46ef, [0x2145] = 0x46f0, [0x2146] = 0x46f1, [0x2147] = 0x46f2,
  [0x2148] = 0x46f3, [0x2149] = 0x46f4, [0x214a] = 0x46f5, [0x214b] = 0x46f6,
  [0x214c] = 0x46f7, [0x214d] = 0x46f8, [0x214e] = 0x46f9, [0x214f] = 0x46fa,
  [0x2150] = 0x46fb, [0x2151] = 0x46fc, [0x2152] = 0x46fd, [0x2153] = 0x46fe,
  [0x2154] = 0x46ff, [0x2155] = 0x4700, [0x2156] = 0x4701, [0x2157] = 0x4702,
  [0x2158] = 0x4703, [0x2159] = 0x4704, [0x215a] = 0x4705, [0x215b] = 0x4706,
  [0x215c] = 0x4707, [0x215d] = 0x4708, [0x215e] = 0x4709, [0x215f] = 0x470a,
  [0x2160] = 0x470b, [0x2161] = 0x470c, [0x2162] = 0x470d, [0x2163] = 0x470e,
  [0x2164] = 0x470f, [0x2165] = 0x4710, [0x2166] = 0x4711, [0x2167] = 0x4712,
  [0x2168] = 0x4713, [0x2169] = 0x4714, [0x216a] = 0x4715, [0x216b] = 0x4716,
  [0x216c] = 0x4717, [0x216d] = 0x4718, [0x216e] = 0x4719, [0x216f] = 0x471a,
  [0x2170] = 0x471b, [0x2171] = 0x471c, [0x2172] = 0x471d, [0x2173] = 0x471e,
  [0x2174] = 0x471f, [0x2175] = 0x4720, [0x2176] = 0x4721, [0x2177] = 0x4722,
  [0x2178] = 0x4724, [0x2179] = 0x4725, [0x217a] = 0x4726, [0x217b] = 0x4727,
  [0x217c] = 0x4728, [0x217d] = 0x472a, [0x217e] = 0x472b, [0x217f] = 0x472c,
  [0x2180] = 0x472d, [0x2181] = 0x472e, [0x2182] = 0x472f, [0x2183] = 0x4730,
  [0x2184] = 0x4731, [0x2185] = 0x4732, [0x2186] = 0x4733, [0x2187] = 0x4734,
  [0x2188] = 0x4735, [0x2189] = 0x4736, [0x218a] = 0x4737, [0x218b] = 0x4738,
  [0x218c] = 0x4739, [0x218d] = 0x473a, [0x218e] = 0x473b, [0x218f] = 0x473c,
  [0x2190] = 0x473d, [0x2191] = 0x473e, [0x2192] = 0x473f, [0x2193] = 0x4740,
  [0x2194] = 0x4741, [0x2195] = 0x4742, [0x2196] = 0x4743, [0x2197] = 0x4744,
  [0x2198] = 0x4745, [0x2199] = 0x4746, [0x219a] = 0x4747, [0x219b] = 0x4748,
  [0x219c] = 0x4749, [0x219d] = 0x474a, [0x219e] = 0x474b, [0x219f] = 0x474c,
  [0x21a0] = 0x474d, [0x21a1] = 0x474e, [0x21a2] = 0x474f, [0x21a3] = 0x4750,
  [0x21a4] = 0x4751, [0x21a5] = 0x4752, [0x21a6] = 0x4753, [0x21a7] = 0x4754,
  [0x21a8] = 0x4755, [0x21a9] = 0x4756, [0x21aa] = 0x4757, [0x21ab] = 0x4758,
  [0x21ac] = 0x4759, [0x21ad] = 0x475a, [0x21ae] = 0x475b, [0x21af] = 0x475c,
  [0x21b0] = 0x475d, [0x21b1] = 0x475e, [0x21b2] = 0x475f, [0x21b3] = 0x4760,
  [0x21b4] = 0x4761, [0x21b5] = 0x4762, [0x21b6] = 0x4763, [0x21b7] = 0x4764,
  [0x21b8] = 0x4765, [0x21b9] = 0x4766, [0x21ba] = 0x4767, [0x21bb] = 0x4768,
  [0x21bc] = 0x4769, [0x21bd] = 0x476a, [0x21be] = 0x476b, [0x21bf] = 0x476c,
  [0x21c0] = 0x476d, [0x21c1] = 0x476e, [0x21c2] = 0x476f, [0x21c3] = 0x4770,
  [0x21c4] = 0x4771, [0x21c5] = 0x4772, [0x21c6] = 0x4773, [0x21c7] = 0x4774,
  [0x21c8] = 0x4775, [0x21c9] = 0x4776, [0x21ca] = 0x4777, [0x21cb] = 0x4778,
  [0x21cc] = 0x4779, [0x21cd] = 0x477a, [0x21ce] = 0x477b, [0x21cf] = 0x477d,
  [0x21d0] = 0x477e, [0x21d1] = 0x477f, [0x21d2] = 0x4780, [0x21d3] = 0x4781,
  [0x21d4] = 0x4782, [0x21d5] = 0x4783, [0x21d6] = 0x4784, [0x21d7] = 0x4785,
  [0x21d8] = 0x4786, [0x21d9] = 0x4787, [0x21da] = 0x4788, [0x21db] = 0x4789,
  [0x21dc] = 0x478a, [0x21dd] = 0x478b, [0x21de] = 0x478c, [0x21df] = 0x478e,
  [0x21e0] = 0x478f, [0x21e1] = 0x4790, [0x21e2] = 0x4791, [0x21e3] = 0x4792,
  [0x21e4] = 0x4793, [0x21e5] = 0x4794, [0x21e6] = 0x4795, [0x21e7] = 0x4796,
  [0x21e8] = 0x4797, [0x21e9] = 0x4798, [0x21ea] = 0x4799, [0x21eb] = 0x479a,
  [0x21ec] = 0x479b, [0x21ed] = 0x479c, [0x21ee] = 0x479d, [0x21ef] = 0x479e,
  [0x21f0] = 0x479f, [0x21f1] = 0x47a0, [0x21f2] = 0x47a1, [0x21f3] = 0x47a2,
  [0x21f4] = 0x47a3, [0x21f5] = 0x47a4, [0x21f6] = 0x47a5, [0x21f7] = 0x47a6,
  [0x21f8] = 0x47a7, [0x21f9] = 0x47a8, [0x21fa] = 0x47a9, [0x21fb] = 0x47aa,
  [0x21fc] = 0x47ab, [0x21fd] = 0x47ac, [0x21fe] = 0x47ad, [0x21ff] = 0x47ae,
  [0x2200] = 0x47af, [0x2201] = 0x47b0, [0x2202] = 0x47b1, [0x2203] = 0x47b2,
  [0x2204] = 0x47b3, [0x2205] = 0x47b4, [0x2206] = 0x47b5, [0x2207] = 0x47b6,
  [0x2208] = 0x47b7, [0x2209] = 0x47b8, [0x220a] = 0x47b9, [0x220b] = 0x47ba,
  [0x220c] = 0x47bb, [0x220d] = 0x47bc, [0x220e] = 0x47bd, [0x220f] = 0x47be,
  [0x2210] = 0x47bf, [0x2211] = 0x47c0, [0x2212] = 0x47c1, [0x2213] = 0x47c2,
  [0x2214] = 0x47c3, [0x2215] = 0x47c4, [0x2216] = 0x47c5, [0x2217] = 0x47c6,
  [0x2218] = 0x47c7, [0x2219] = 0x47c8, [0x221a] = 0x47c9, [0x221b] = 0x47ca,
  [0x221c] = 0x47cb, [0x221d] = 0x47cc, [0x221e] = 0x47cd, [0x221f] = 0x47ce,
  [0x2220] = 0x47cf, [0x2221] = 0x47d0, [0x2222] = 0x47d1, [0x2223] = 0x47d2,
  [0x2224] = 0x47d3, [0x2225] = 0x47d4, [0x2226] = 0x47d5, [0x2227] = 0x47d6,
  [0x2228] = 0x47d7, [0x2229] = 0x47d8, [0x222a] = 0x47d9, [0x222b] = 0x47da,
  [0x222c] = 0x47db, [0x222d] = 0x47dc, [0x222e] = 0x47dd, [0x222f] = 0x47de,
  [0x2230] = 0x47df, [0x2231] = 0x47e0, [0x2232] = 0x47e1, [0x2233] = 0x47e2,
  [0x2234] = 0x47e3, [0x2235] = 0x47e4, [0x2236] = 0x47e5, [0x2237] = 0x47e6,
  [0x2238] = 0x47e7, [0x2239] = 0x47e8, [0x223a] = 0x47e9, [0x223b] = 0x47ea,
  [0x223c] = 0x47eb, [0x223d] = 0x47ec, [0x223e] = 0x47ed, [0x223f] = 0x47ee,
  [0x2240] = 0x47ef, [0x2241] = 0x47f0, [0x2242] = 0x47f1, [0x2243] = 0x47f2,
  [0x2244] = 0x47f3, [0x2245] = 0x47f4, [0x2246] = 0x47f5, [0x2247] = 0x47f6,
  [0x2248] = 0x47f7, [0x2249] = 0x47f8, [0x224a] = 0x47f9, [0x224b] = 0x47fa,
  [0x224c] = 0x47fb, [0x224d] = 0x47fc, [0x224e] = 0x47fd, [0x224f] = 0x47fe,
  [0x2250] = 0x47ff, [0x2251] = 0x4800, [0x2252] = 0x4801, [0x2253] = 0x4802,
  [0x2254] = 0x4803, [0x2255] = 0x4804, [0x2256] = 0x4805, [0x2257] = 0x4806,
  [0x2258] = 0x4807, [0x2259] = 0x4808, [0x225a] = 0x4809, [0x225b] = 0x480a,
  [0x225c] = 0x480b, [0x225d] = 0x480c, [0x225e] = 0x480d, [0x225f] = 0x480e,
  [0x2260] = 0x480f, [0x2261] = 0x4810, [0x2262] = 0x4811, [0x2263] = 0x4812,
  [0x2264] = 0x4813, [0x2265] = 0x4814, [0x2266] = 0x4815, [0x2267] = 0x4816,
  [0x2268] = 0x4817, [0x2269] = 0x4818, [0x226a] = 0x4819, [0x226b] = 0x481a,
  [0x226c] = 0x481b, [0x226d] = 0x481c, [0x226e] = 0x481d, [0x226f] = 0x481e,
  [0x2270] = 0x481f, [0x2271] = 0x4820, [0x2272] = 0x4821, [0x2273] = 0x4822,
  [0x2274] = 0x4823, [0x2275] = 0x4824, [0x2276] = 0x4825, [0x2277] = 0x4826,
  [0x2278] = 0x4827, [0x2279] = 0x4828, [0x227a] = 0x4829, [0x227b] = 0x482a,
  [0x227c] = 0x482b, [0x227d] = 0x482c, [0x227e] = 0x482d, [0x227f] = 0x482e,
  [0x2280] = 0x482f, [0x2281] = 0x4830, [0x2282] = 0x4831, [0x2283] = 0x4832,
  [0x2284] = 0x4833, [0x2285] = 0x4834, [0x2286] = 0x4835, [0x2287] = 0x4836,
  [0x2288] = 0x4837, [0x2289] = 0x4838, [0x228a] = 0x4839, [0x228b] = 0x483a,
  [0x228c] = 0x483b, [0x228d] = 0x483c, [0x228e] = 0x483d, [0x228f] = 0x483e,
  [0x2290] = 0x483f, [0x2291] = 0x4840, [0x2292] = 0x4841, [0x2293] = 0x4842,
  [0x2294] = 0x4843, [0x2295] = 0x4844, [0x2296] = 0x4845, [0x2297] = 0x4846,
  [0x2298] = 0x4847, [0x2299] = 0x4848, [0x229a] = 0x4849, [0x229b] = 0x484a,
  [0x229c] = 0x484b, [0x229d] = 0x484c, [0x229e] = 0x484d, [0x229f] = 0x484e,
  [0x22a0] = 0x484f, [0x22a1] = 0x4850, [0x22a2] = 0x4851, [0x22a3] = 0x4852,
  [0x22a4] = 0x4853, [0x22a5] = 0x4854, [0x22a6] = 0x4855, [0x22a7] = 0x4856,
  [0x22a8] = 0x4857, [0x22a9] = 0x4858, [0x22aa] = 0x4859, [0x22ab] = 0x485a,
  [0x22ac] = 0x485b, [0x22ad] = 0x485c, [0x22ae] = 0x485d, [0x22af] = 0x485e,
  [0x22b0] = 0x485f, [0x22b1] = 0x4860, [0x22b2] = 0x4861, [0x22b3] = 0x4862,
  [0x22b4] = 0x4863, [0x22b5] = 0x4864, [0x22b6] = 0x4865, [0x22b7] = 0x4866,
  [0x22b8] = 0x4867, [0x22b9] = 0x4868, [0x22ba] = 0x4869, [0x22bb] = 0x486a,
  [0x22bc] = 0x486b, [0x22bd] = 0x486c, [0x22be] = 0x486d, [0x22bf] = 0x486e,
  [0x22c0] = 0x486f, [0x22c1] = 0x4870, [0x22c2] = 0x4871, [0x22c3] = 0x4872,
  [0x22c4] = 0x4873, [0x22c5] = 0x4874, [0x22c6] = 0x4875, [0x22c7] = 0x4876,
  [0x22c8] = 0x4877, [0x22c9] = 0x4878, [0x22ca] = 0x4879, [0x22cb] = 0x487a,
  [0x22cc] = 0x487b, [0x22cd] = 0x487c, [0x22ce] = 0x487d, [0x22cf] = 0x487e,
  [0x22d0] = 0x487f, [0x22d1] = 0x4880, [0x22d2] = 0x4881, [0x22d3] = 0x4882,
  [0x22d4] = 0x4883, [0x22d5] = 0x4884, [0x22d6] = 0x4885, [0x22d7] = 0x4886,
  [0x22d8] = 0x4887, [0x22d9] = 0x4888, [0x22da] = 0x4889, [0x22db] = 0x488a,
  [0x22dc] = 0x488b, [0x22dd] = 0x488c, [0x22de] = 0x488d, [0x22df] = 0x488e,
  [0x22e0] = 0x488f, [0x22e1] = 0x4890, [0x22e2] = 0x4891, [0x22e3] = 0x4892,
  [0x22e4] = 0x4893, [0x22e5] = 0x4894, [0x22e6] = 0x4895, [0x22e7] = 0x4896,
  [0x22e8] = 0x4897, [0x22e9] = 0x4898, [0x22ea] = 0x4899, [0x22eb] = 0x489a,
  [0x22ec] = 0x489b, [0x22ed] = 0x489c, [0x22ee] = 0x489d, [0x22ef] = 0x489e,
  [0x22f0] = 0x489f, [0x22f1] = 0x48a0, [0x22f2] = 0x48a1, [0x22f3] = 0x48a2,
  [0x22f4] = 0x48a3, [0x22f5] = 0x48a4, [0x22f6] = 0x48a5, [0x22f7] = 0x48a6,
  [0x22f8] = 0x48a7, [0x22f9] = 0x48a8, [0x22fa] = 0x48a9, [0x22fb] = 0x48aa,
  [0x22fc] = 0x48ab, [0x22fd] = 0x48ac, [0x22fe] = 0x48ad, [0x22ff] = 0x48ae,
  [0x2300] = 0x48af, [0x2301] = 0x48b0, [0x2302] = 0x48b1, [0x2303] = 0x48b2,
  [0x2304] = 0x48b3, [0x2305] = 0x48b4, [0x2306] = 0x48b5, [0x2307] = 0x48b6,
  [0x2308] = 0x48b7, [0x2309] = 0x48b8, [0x230a] = 0x48b9, [0x230b] = 0x48ba,
  [0x230c] = 0x48bb, [0x230d] = 0x48bc, [0x230e] = 0x48bd, [0x230f] = 0x48be,
  [0x2310] = 0x48bf, [0x2311] = 0x48c0, [0x2312] = 0x48c1, [0x2313] = 0x48c2,
  [0x2314] = 0x48c3, [0x2315] = 0x48c4, [0x2316] = 0x48c5, [0x2317] = 0x48c6,
  [0x2318] = 0x48c7, [0x2319] = 0x48c8, [0x231a] = 0x48c9, [0x231b] = 0x48ca,
  [0x231c] = 0x48cb, [0x231d] = 0x48cc, [0x231e] = 0x48cd, [0x231f] = 0x48ce,
  [0x2320] = 0x48cf, [0x2321] = 0x48d0, [0x2322] = 0x48d1, [0x2323] = 0x48d2,
  [0x2324] = 0x48d3, [0x2325] = 0x48d4, [0x2326] = 0x48d5, [0x2327] = 0x48d6,
  [0x2328] = 0x48d7, [0x2329] = 0x48d8, [0x232a] = 0x48d9, [0x232b] = 0x48da,
  [0x232c] = 0x48db, [0x232d] = 0x48dc, [0x232e] = 0x48dd, [0x232f] = 0x48de,
  [0x2330] = 0x48df, [0x2331] = 0x48e0, [0x2332] = 0x48e1, [0x2333] = 0x48e2,
  [0x2334] = 0x48e3, [0x2335] = 0x48e4, [0x2336] = 0x48e5, [0x2337] = 0x48e6,
  [0x2338] = 0x48e7, [0x2339] = 0x48e8, [0x233a] = 0x48e9, [0x233b] = 0x48ea,
  [0x233c] = 0x48eb, [0x233d] = 0x48ec, [0x233e] = 0x48ed, [0x233f] = 0x48ee,
  [0x2340] = 0x48ef, [0x2341] = 0x48f0, [0x2342] = 0x48f1, [0x2343] = 0x48f2,
  [0x2344] = 0x48f3, [0x2345] = 0x48f4, [0x2346] = 0x48f5, [0x2347] = 0x48f6,
  [0x2348] = 0x48f7, [0x2349] = 0x48f8, [0x234a] = 0x48f9, [0x234b] = 0x48fa,
  [0x234c] = 0x48fb, [0x234d] = 0x48fc, [0x234e] = 0x48fd, [0x234f] = 0x48fe,
  [0x2350] = 0x48ff, [0x2351] = 0x4900, [0x2352] = 0x4901, [0x2353] = 0x4902,
  [0x2354] = 0x4903, [0x2355] = 0x4904, [0x2356] = 0x4905, [0x2357] = 0x4906,
  [0x2358] = 0x4907, [0x2359] = 0x4908, [0x235a] = 0x4909, [0x235b] = 0x490a,
  [0x235c] = 0x490b, [0x235d] = 0x490c, [0x235e] = 0x490d, [0x235f] = 0x490e,
  [0x2360] = 0x490f, [0x2361] = 0x4910, [0x2362] = 0x4911, [0x2363] = 0x4912,
  [0x2364] = 0x4913, [0x2365] = 0x4914, [0x2366] = 0x4915, [0x2367] = 0x4916,
  [0x2368] = 0x4917, [0x2369] = 0x4918, [0x236a] = 0x4919, [0x236b] = 0x491a,
  [0x236c] = 0x491b, [0x236d] = 0x491c, [0x236e] = 0x491d, [0x236f] = 0x491e,
  [0x2370] = 0x491f, [0x2371] = 0x4920, [0x2372] = 0x4921, [0x2373] = 0x4922,
  [0x2374] = 0x4923, [0x2375] = 0x4924, [0x2376] = 0x4925, [0x2377] = 0x4926,
  [0x2378] = 0x4927, [0x2379] = 0x4928, [0x237a] = 0x4929, [0x237b] = 0x492a,
  [0x237c] = 0x492b, [0x237d] = 0x492c, [0x237e] = 0x492d, [0x237f] = 0x492e,
  [0x2380] = 0x492f, [0x2381] = 0x4930, [0x2382] = 0x4931, [0x2383] = 0x4932,
  [0x2384] = 0x4933, [0x2385] = 0x4934, [0x2386] = 0x4935, [0x2387] = 0x4936,
  [0x2388] = 0x4937, [0x2389] = 0x4938, [0x238a] = 0x4939, [0x238b] = 0x493a,
  [0x238c] = 0x493b, [0x238d] = 0x493c, [0x238e] = 0x493d, [0x238f] = 0x493e,
  [0x2390] = 0x493f, [0x2391] = 0x4940, [0x2392] = 0x4941, [0x2393] = 0x4942,
  [0x2394] = 0x4943, [0x2395] = 0x4944, [0x2396] = 0x4945, [0x2397] = 0x4946,
  [0x2398] = 0x4948, [0x2399] = 0x4949, [0x239a] = 0x494a, [0x239b] = 0x494b,
  [0x239c] = 0x494c, [0x239d] = 0x494d, [0x239e] = 0x494e, [0x239f] = 0x494f,
  [0x23a0] = 0x4950, [0x23a1] = 0x4951, [0x23a2] = 0x4952, [0x23a3] = 0x4953,
  [0x23a4] = 0x4954, [0x23a5] = 0x4955, [0x23a6] = 0x4956, [0x23a7] = 0x4957,
  [0x23a8] = 0x4958, [0x23a9] = 0x4959, [0x23aa] = 0x495a, [0x23ab] = 0x495b,
  [0x23ac] = 0x495c, [0x23ad] = 0x495d, [0x23ae] = 0x495e, [0x23af] = 0x495f,
  [0x23b0] = 0x4960, [0x23b1] = 0x4961, [0x23b2] = 0x4962, [0x23b3] = 0x4963,
  [0x23b4] = 0x4964, [0x23b5] = 0x4965, [0x23b6] = 0x4966, [0x23b7] = 0x4967,
  [0x23b8] = 0x4968, [0x23b9] = 0x4969, [0x23ba] = 0x496a, [0x23bb] = 0x496b,
  [0x23bc] = 0x496c, [0x23bd] = 0x496d, [0x23be] = 0x496e, [0x23bf] = 0x496f,
  [0x23c0] = 0x4970, [0x23c1] = 0x4971, [0x23c2] = 0x4972, [0x23c3] = 0x4973,
  [0x23c4] = 0x4974, [0x23c5] = 0x4975, [0x23c6] = 0x4976, [0x23c7] = 0x4977,
  [0x23c8] = 0x4978, [0x23c9] = 0x4979, [0x23ca] = 0x497b, [0x23cb] = 0x497c,
  [0x23cc] = 0x497e, [0x23cd] = 0x497f, [0x23ce] = 0x4980, [0x23cf] = 0x4981,
  [0x23d0] = 0x4984, [0x23d1] = 0x4987, [0x23d2] = 0x4988, [0x23d3] = 0x4989,
  [0x23d4] = 0x498a, [0x23d5] = 0x498b, [0x23d6] = 0x498c, [0x23d7] = 0x498d,
  [0x23d8] = 0x498e, [0x23d9] = 0x498f, [0x23da] = 0x4990, [0x23db] = 0x4991,
  [0x23dc] = 0x4992, [0x23dd] = 0x4993, [0x23de] = 0x4994, [0x23df] = 0x4995,
  [0x23e0] = 0x4996, [0x23e1] = 0x4997, [0x23e2] = 0x4998, [0x23e3] = 0x4999,
  [0x23e4] = 0x499a, [0x23e5] = 0x499c, [0x23e6] = 0x499d, [0x23e7] = 0x499e,
  [0x23e8] = 0x49a0, [0x23e9] = 0x49a1, [0x23ea] = 0x49a2, [0x23eb] = 0x49a3,
  [0x23ec] = 0x49a4, [0x23ed] = 0x49a5, [0x23ee] = 0x49a6, [0x23ef] = 0x49a7,
  [0x23f0] = 0x49a8, [0x23f1] = 0x49a9, [0x23f2] = 0x49aa, [0x23f3] = 0x49ab,
  [0x23f4] = 0x49ac, [0x23f5] = 0x49ad, [0x23f6] = 0x49ae, [0x23f7] = 0x49af,
  [0x23f8] = 0x49b0, [0x23f9] = 0x49b1, [0x23fa] = 0x49b2, [0x23fb] = 0x49b3,
  [0x23fc] = 0x49b4, [0x23fd] = 0x49b5, [0x23fe] = 0x49b8, [0x23ff] = 0x49b9,
  [0x2400] = 0x49ba, [0x2401] = 0x49bb, [0x2402] = 0x49bc, [0x2403] = 0x49bd,
  [0x2404] = 0x49be, [0x2405] = 0x49bf, [0x2406] = 0x49c0, [0x2407] = 0x49c1,
  [0x2408] = 0x49c2, [0x2409] = 0x49c3, [0x240a] = 0x49c4, [0x240b] = 0x49c5,
  [0x240c] = 0x49c6, [0x240d] = 0x49c7, [0x240e] = 0x49c8, [0x240f] = 0x49c9,
  [0x2410] = 0x49ca, [0x2411] = 0x49cb, [0x2412] = 0x49cc, [0x2413] = 0x49cd,
  [0x2414] = 0x49ce, [0x2415] = 0x49cf, [0x2416] = 0x49d0, [0x2417] = 0x49d1,
  [0x2418] = 0x49d2, [0x2419] = 0x49d3, [0x241a] = 0x49d4, [0x241b] = 0x49d5,
  [0x241c] = 0x49d6, [0x241d] = 0x49d7, [0x241e] = 0x49d8, [0x241f] = 0x49d9,
  [0x2420] = 0x49da, [0x2421] = 0x49db, [0x2422] = 0x49dc, [0x2423] = 0x49dd,
  [0x2424] = 0x49de, [0x2425] = 0x49df, [0x2426] = 0x49e0, [0x2427] = 0x49e1,
  [0x2428] = 0x49e2, [0x2429] = 0x49e3, [0x242a] = 0x49e4, [0x242b] = 0x49e5,
  [0x242c] = 0x49e6, [0x242d] = 0x49e7, [0x242e] = 0x49e8, [0x242f] = 0x49e9,
  [0x2430] = 0x49ea, [0x2431] = 0x49eb, [0x2432] = 0x49ec, [0x2433] = 0x49ed,
  [0x2434] = 0x49ee, [0x2435] = 0x49ef, [0x2436] = 0x49f0, [0x2437] = 0x49f1,
  [0x2438] = 0x49f2, [0x2439] = 0x49f3, [0x243a] = 0x49f4, [0x243b] = 0x49f5,
  [0x243c] = 0x49f6, [0x243d] = 0x49f7, [0x243e] = 0x49f8, [0x243f] = 0x49f9,
  [0x2440] = 0x49fa, [0x2441] = 0x49fb, [0x2442] = 0x49fc, [0x2443] = 0x49fd,
  [0x2444] = 0x49fe, [0x2445] = 0x49ff, [0x2446] = 0x4a00, [0x2447] = 0x4a01,
  [0x2448] = 0x4a02, [0x2449] = 0x4a03, [0x244a] = 0x4a04, [0x244b] = 0x4a05,
  [0x244c] = 0x4a06, [0x244d] = 0x4a07, [0x244e] = 0x4a08, [0x244f] = 0x4a09,
  [0x2450] = 0x4a0a, [0x2451] = 0x4a0b, [0x2452] = 0x4a0c, [0x2453] = 0x4a0d,
  [0x2454] = 0x4a0e, [0x2455] = 0x4a0f, [0x2456] = 0x4a10, [0x2457] = 0x4a11,
  [0x2458] = 0x4a12, [0x2459] = 0x4a13, [0x245a] = 0x4a14, [0x245b] = 0x4a15,
  [0x245c] = 0x4a16, [0x245d] = 0x4a17, [0x245e] = 0x4a18, [0x245f] = 0x4a19,
  [0x2460] = 0x4a1a, [0x2461] = 0x4a1b, [0x2462] = 0x4a1c, [0x2463] = 0x4a1d,
  [0x2464] = 0x4a1e, [0x2465] = 0x4a1f, [0x2466] = 0x4a20, [0x2467] = 0x4a21,
  [0x2468] = 0x4a22, [0x2469] = 0x4a23, [0x246a] = 0x4a24, [0x246b] = 0x4a25,
  [0x246c] = 0x4a26, [0x246d] = 0x4a27, [0x246e] = 0x4a28, [0x246f] = 0x4a29,
  [0x2470] = 0x4a2a, [0x2471] = 0x4a2b, [0x2472] = 0x4a2c, [0x2473] = 0x4a2d,
  [0x2474] = 0x4a2e, [0x2475] = 0x4a2f, [0x2476] = 0x4a30, [0x2477] = 0x4a31,
  [0x2478] = 0x4a32, [0x2479] = 0x4a33, [0x247a] = 0x4a34, [0x247b] = 0x4a35,
  [0x247c] = 0x4a36, [0x247d] = 0x4a37, [0x247e] = 0x4a38, [0x247f] = 0x4a39,
  [0x2480] = 0x4a3a, [0x2481] = 0x4a3b, [0x2482] = 0x4a3c, [0x2483] = 0x4a3d,
  [0x2484] = 0x4a3e, [0x2485] = 0x4a3f, [0x2486] = 0x4a40, [0x2487] = 0x4a41,
  [0x2488] = 0x4a42, [0x2489] = 0x4a43, [0x248a] = 0x4a44, [0x248b] = 0x4a45,
  [0x248c] = 0x4a46, [0x248d] = 0x4a47, [0x248e] = 0x4a48, [0x248f] = 0x4a49,
  [0x2490] = 0x4a4a, [0x2491] = 0x4a4b, [0x2492] = 0x4a4c, [0x2493] = 0x4a4d,
  [0x2494] = 0x4a4e, [0x2495] = 0x4a4f, [0x2496] = 0x4a50, [0x2497] = 0x4a51,
  [0x2498] = 0x4a52, [0x2499] = 0x4a53, [0x249a] = 0x4a54, [0x249b] = 0x4a55,
  [0x249c] = 0x4a56, [0x249d] = 0x4a57, [0x249e] = 0x4a58, [0x249f] = 0x4a59,
  [0x24a0] = 0x4a5a, [0x24a1] = 0x4a5b, [0x24a2] = 0x4a5c, [0x24a3] = 0x4a5d,
  [0x24a4] = 0x4a5e, [0x24a5] = 0x4a5f, [0x24a6] = 0x4a60, [0x24a7] = 0x4a61,
  [0x24a8] = 0x4a62, [0x24a9] = 0x4a63, [0x24aa] = 0x4a64, [0x24ab] = 0x4a65,
  [0x24ac] = 0x4a66, [0x24ad] = 0x4a67, [0x24ae] = 0x4a68, [0x24af] = 0x4a69,
  [0x24b0] = 0x4a6a, [0x24b1] = 0x4a6b, [0x24b2] = 0x4a6c, [0x24b3] = 0x4a6d,
  [0x24b4] = 0x4a6e, [0x24b5] = 0x4a6f, [0x24b6] = 0x4a70, [0x24b7] = 0x4a71,
  [0x24b8] = 0x4a72, [0x24b9] = 0x4a73, [0x24ba] = 0x4a74, [0x24bb] = 0x4a75,
  [0x24bc] = 0x4a76, [0x24bd] = 0x4a77, [0x24be] = 0x4a78, [0x24bf] = 0x4a79,
  [0x24c0] = 0x4a7a, [0x24c1] = 0x4a7b, [0x24c2] = 0x4a7c, [0x24c3] = 0x4a7d,
  [0x24c4] = 0x4a7e, [0x24c5] = 0x4a7f, [0x24c6] = 0x4a80, [0x24c7] = 0x4a81,
  [0x24c8] = 0x4a82, [0x24c9] = 0x4a83, [0x24ca] = 0x4a84, [0x24cb] = 0x4a85,
  [0x24cc] = 0x4a86, [0x24cd] = 0x4a87, [0x24ce] = 0x4a88, [0x24cf] = 0x4a89,
  [0x24d0] = 0x4a8a, [0x24d1] = 0x4a8b, [0x24d2] = 0x4a8c, [0x24d3] = 0x4a8d,
  [0x24d4] = 0x4a8e, [0x24d5] = 0x4a8f, [0x24d6] = 0x4a90, [0x24d7] = 0x4a91,
  [0x24d8] = 0x4a92, [0x24d9] = 0x4a93, [0x24da] = 0x4a94, [0x24db] = 0x4a95,
  [0x24dc] = 0x4a96, [0x24dd] = 0x4a97, [0x24de] = 0x4a98, [0x24df] = 0x4a99,
  [0x24e0] = 0x4a9a, [0x24e1] = 0x4a9b, [0x24e2] = 0x4a9c, [0x24e3] = 0x4a9d,
  [0x24e4] = 0x4a9e, [0x24e5] = 0x4a9f, [0x24e6] = 0x4aa0, [0x24e7] = 0x4aa1,
  [0x24e8] = 0x4aa2, [0x24e9] = 0x4aa3, [0x24ea] = 0x4aa4, [0x24eb] = 0x4aa5,
  [0x24ec] = 0x4aa6, [0x24ed] = 0x4aa7, [0x24ee] = 0x4aa8, [0x24ef] = 0x4aa9,
  [0x24f0] = 0x4aaa, [0x24f1] = 0x4aab, [0x24f2] = 0x4aac, [0x24f3] = 0x4aad,
  [0x24f4] = 0x4aae, [0x24f5] = 0x4aaf, [0x24f6] = 0x4ab0, [0x24f7] = 0x4ab1,
  [0x24f8] = 0x4ab2, [0x24f9] = 0x4ab3, [0x24fa] = 0x4ab4, [0x24fb] = 0x4ab5,
  [0x24fc] = 0x4ab6, [0x24fd] = 0x4ab7, [0x24fe] = 0x4ab8, [0x24ff] = 0x4ab9,
  [0x2500] = 0x4aba, [0x2501] = 0x4abb, [0x2502] = 0x4abc, [0x2503] = 0x4abd,
  [0x2504] = 0x4abe, [0x2505] = 0x4abf, [0x2506] = 0x4ac0, [0x2507] = 0x4ac1,
  [0x2508] = 0x4ac2, [0x2509] = 0x4ac3, [0x250a] = 0x4ac4, [0x250b] = 0x4ac5,
  [0x250c] = 0x4ac6, [0x250d] = 0x4ac7, [0x250e] = 0x4ac8, [0x250f] = 0x4ac9,
  [0x2510] = 0x4aca, [0x2511] = 0x4acb, [0x2512] = 0x4acc, [0x2513] = 0x4acd,
  [0x2514] = 0x4ace, [0x2515] = 0x4acf, [0x2516] = 0x4ad0, [0x2517] = 0x4ad1,
  [0x2518] = 0x4ad2, [0x2519] = 0x4ad3, [0x251a] = 0x4ad4, [0x251b] = 0x4ad5,
  [0x251c] = 0x4ad6, [0x251d] = 0x4ad7, [0x251e] = 0x4ad8, [0x251f] = 0x4ad9,
  [0x2520] = 0x4ada, [0x2521] = 0x4adb, [0x2522] = 0x4adc, [0x2523] = 0x4add,
  [0x2524] = 0x4ade, [0x2525] = 0x4adf, [0x2526] = 0x4ae0, [0x2527] = 0x4ae1,
  [0x2528] = 0x4ae2, [0x2529] = 0x4ae3, [0x252a] = 0x4ae4, [0x252b] = 0x4ae5,
  [0x252c] = 0x4ae6, [0x252d] = 0x4ae7, [0x252e] = 0x4ae8, [0x252f] = 0x4ae9,
  [0x2530] = 0x4aea, [0x2531] = 0x4aeb, [0x2532] = 0x4aec, [0x2533] = 0x4aed,
  [0x2534] = 0x4aee, [0x2535] = 0x4aef, [0x2536] = 0x4af0, [0x2537] = 0x4af1,
  [0x2538] = 0x4af2, [0x2539] = 0x4af3, [0x253a] = 0x4af4, [0x253b] = 0x4af5,
  [0x253c] = 0x4af6, [0x253d] = 0x4af7, [0x253e] = 0x4af8, [0x253f] = 0x4af9,
  [0x2540] = 0x4afa, [0x2541] = 0x4afb, [0x2542] = 0x4afc, [0x2543] = 0x4afd,
  [0x2544] = 0x4afe, [0x2545] = 0x4aff, [0x2546] = 0x4b00, [0x2547] = 0x4b01,
  [0x2548] = 0x4b02, [0x2549] = 0x4b03, [0x254a] = 0x4b04, [0x254b] = 0x4b05,
  [0x254c] = 0x4b06, [0x254d] = 0x4b07, [0x254e] = 0x4b08, [0x254f] = 0x4b09,
  [0x2550] = 0x4b0a, [0x2551] = 0x4b0b, [0x2552] = 0x4b0c, [0x2553] = 0x4b0d,
  [0x2554] = 0x4b0e, [0x2555] = 0x4b0f, [0x2556] = 0x4b10, [0x2557] = 0x4b11,
  [0x2558] = 0x4b12, [0x2559] = 0x4b13, [0x255a] = 0x4b14, [0x255b] = 0x4b15,
  [0x255c] = 0x4b16, [0x255d] = 0x4b17, [0x255e] = 0x4b18, [0x255f] = 0x4b19,
  [0x2560] = 0x4b1a, [0x2561] = 0x4b1b, [0x2562] = 0x4b1c, [0x2563] = 0x4b1d,
  [0x2564] = 0x4b1e, [0x2565] = 0x4b1f, [0x2566] = 0x4b20, [0x2567] = 0x4b21,
  [0x2568] = 0x4b22, [0x2569] = 0x4b23, [0x256a] = 0x4b24, [0x256b] = 0x4b25,
  [0x256c] = 0x4b26, [0x256d] = 0x4b27, [0x256e] = 0x4b28, [0x256f] = 0x4b29,
  [0x2570] = 0x4b2a, [0x2571] = 0x4b2b, [0x2572] = 0x4b2c, [0x2573] = 0x4b2d,
  [0x2574] = 0x4b2e, [0x2575] = 0x4b2f, [0x2576] = 0x4b30, [0x2577] = 0x4b31,
  [0x2578] = 0x4b32, [0x2579] = 0x4b33, [0x257a] = 0x4b34, [0x257b] = 0x4b35,
  [0x257c] = 0x4b36, [0x257d] = 0x4b37, [0x257e] = 0x4b38, [0x257f] = 0x4b39,
  [0x2580] = 0x4b3a, [0x2581] = 0x4b3b, [0x2582] = 0x4b3c, [0x2583] = 0x4b3d,
  [0x2584] = 0x4b3e, [0x2585] = 0x4b3f, [0x2586] = 0x4b40, [0x2587] = 0x4b41,
  [0x2588] = 0x4b42, [0x2589] = 0x4b43, [0x258a] = 0x4b44, [0x258b] = 0x4b45,
  [0x258c] = 0x4b46, [0x258d] = 0x4b47, [0x258e] = 0x4b48, [0x258f] = 0x4b49,
  [0x2590] = 0x4b4a, [0x2591] = 0x4b4b, [0x2592] = 0x4b4c, [0x2593] = 0x4b4d,
  [0x2594] = 0x4b4e, [0x2595] = 0x4b4f, [0x2596] = 0x4b50, [0x2597] = 0x4b51,
  [0x2598] = 0x4b52, [0x2599] = 0x4b53, [0x259a] = 0x4b54, [0x259b] = 0x4b55,
  [0x259c] = 0x4b56, [0x259d] = 0x4b57, [0x259e] = 0x4b58, [0x259f] = 0x4b59,
  [0x25a0] = 0x4b5a, [0x25a1] = 0x4b5b, [0x25a2] = 0x4b5c, [0x25a3] = 0x4b5d,
  [0x25a4] = 0x4b5e, [0x25a5] = 0x4b5f, [0x25a6] = 0x4b60, [0x25a7] = 0x4b61,
  [0x25a8] = 0x4b62, [0x25a9] = 0x4b63, [0x25aa] = 0x4b64, [0x25ab] = 0x4b65,
  [0x25ac] = 0x4b66, [0x25ad] = 0x4b67, [0x25ae] = 0x4b68, [0x25af] = 0x4b69,
  [0x25b0] = 0x4b6a, [0x25b1] = 0x4b6b, [0x25b2] = 0x4b6c, [0x25b3] = 0x4b6d,
  [0x25b4] = 0x4b6e, [0x25b5] = 0x4b6f, [0x25b6] = 0x4b70, [0x25b7] = 0x4b71,
  [0x25b8] = 0x4b72, [0x25b9] = 0x4b73, [0x25ba] = 0x4b74, [0x25bb] = 0x4b75,
  [0x25bc] = 0x4b76, [0x25bd] = 0x4b77, [0x25be] = 0x4b78, [0x25bf] = 0x4b79,
  [0x25c0] = 0x4b7a, [0x25c1] = 0x4b7b, [0x25c2] = 0x4b7c, [0x25c3] = 0x4b7d,
  [0x25c4] = 0x4b7e, [0x25c5] = 0x4b7f, [0x25c6] = 0x4b80, [0x25c7] = 0x4b81,
  [0x25c8] = 0x4b82, [0x25c9] = 0x4b83, [0x25ca] = 0x4b84, [0x25cb] = 0x4b85,
  [0x25cc] = 0x4b86, [0x25cd] = 0x4b87, [0x25ce] = 0x4b88, [0x25cf] = 0x4b89,
  [0x25d0] = 0x4b8a, [0x25d1] = 0x4b8b, [0x25d2] = 0x4b8c, [0x25d3] = 0x4b8d,
  [0x25d4] = 0x4b8e, [0x25d5] = 0x4b8f, [0x25d6] = 0x4b90, [0x25d7] = 0x4b91,
  [0x25d8] = 0x4b92, [0x25d9] = 0x4b93, [0x25da] = 0x4b94, [0x25db] = 0x4b95,
  [0x25dc] = 0x4b96, [0x25dd] = 0x4b97, [0x25de] = 0x4b98, [0x25df] = 0x4b99,
  [0x25e0] = 0x4b9a, [0x25e1] = 0x4b9b, [0x25e2] = 0x4b9c, [0x25e3] = 0x4b9d,
  [0x25e4] = 0x4b9e, [0x25e5] = 0x4b9f, [0x25e6] = 0x4ba0, [0x25e7] = 0x4ba1,
  [0x25e8] = 0x4ba2, [0x25e9] = 0x4ba3, [0x25ea] = 0x4ba4, [0x25eb] = 0x4ba5,
  [0x25ec] = 0x4ba6, [0x25ed] = 0x4ba7, [0x25ee] = 0x4ba8, [0x25ef] = 0x4ba9,
  [0x25f0] = 0x4baa, [0x25f1] = 0x4bab, [0x25f2] = 0x4bac, [0x25f3] = 0x4bad,
  [0x25f4] = 0x4bae, [0x25f5] = 0x4baf, [0x25f6] = 0x4bb0, [0x25f7] = 0x4bb1,
  [0x25f8] = 0x4bb2, [0x25f9] = 0x4bb3, [0x25fa] = 0x4bb4, [0x25fb] = 0x4bb5,
  [0x25fc] = 0x4bb6, [0x25fd] = 0x4bb7, [0x25fe] = 0x4bb8, [0x25ff] = 0x4bb9,
  [0x2600] = 0x4bba, [0x2601] = 0x4bbb, [0x2602] = 0x4bbc, [0x2603] = 0x4bbd,
  [0x2604] = 0x4bbe, [0x2605] = 0x4bbf, [0x2606] = 0x4bc0, [0x2607] = 0x4bc1,
  [0x2608] = 0x4bc2, [0x2609] = 0x4bc3, [0x260a] = 0x4bc4, [0x260b] = 0x4bc5,
  [0x260c] = 0x4bc6, [0x260d] = 0x4bc7, [0x260e] = 0x4bc8, [0x260f] = 0x4bc9,
  [0x2610] = 0x4bca, [0x2611] = 0x4bcb, [0x2612] = 0x4bcc, [0x2613] = 0x4bcd,
  [0x2614] = 0x4bce, [0x2615] = 0x4bcf, [0x2616] = 0x4bd0, [0x2617] = 0x4bd1,
  [0x2618] = 0x4bd2, [0x2619] = 0x4bd3, [0x261a] = 0x4bd4, [0x261b] = 0x4bd5,
  [0x261c] = 0x4bd6, [0x261d] = 0x4bd7, [0x261e] = 0x4bd8, [0x261f] = 0x4bd9,
  [0x2620] = 0x4bda, [0x2621] = 0x4bdb, [0x2622] = 0x4bdc, [0x2623] = 0x4bdd,
  [0x2624] = 0x4bde, [0x2625] = 0x4bdf, [0x2626] = 0x4be0, [0x2627] = 0x4be1,
  [0x2628] = 0x4be2, [0x2629] = 0x4be3, [0x262a] = 0x4be4, [0x262b] = 0x4be5,
  [0x262c] = 0x4be6, [0x262d] = 0x4be7, [0x262e] = 0x4be8, [0x262f] = 0x4be9,
  [0x2630] = 0x4bea, [0x2631] = 0x4beb, [0x2632] = 0x4bec, [0x2633] = 0x4bed,
  [0x2634] = 0x4bee, [0x2635] = 0x4bef, [0x2636] = 0x4bf0, [0x2637] = 0x4bf1,
  [0x2638] = 0x4bf2, [0x2639] = 0x4bf3, [0x263a] = 0x4bf4, [0x263b] = 0x4bf5,
  [0x263c] = 0x4bf6, [0x263d] = 0x4bf7, [0x263e] = 0x4bf8, [0x263f] = 0x4bf9,
  [0x2640] = 0x4bfa, [0x2641] = 0x4bfb, [0x2642] = 0x4bfc, [0x2643] = 0x4bfd,
  [0x2644] = 0x4bfe, [0x2645] = 0x4bff, [0x2646] = 0x4c00, [0x2647] = 0x4c01,
  [0x2648] = 0x4c02, [0x2649] = 0x4c03, [0x264a] = 0x4c04, [0x264b] = 0x4c05,
  [0x264c] = 0x4c06, [0x264d] = 0x4c07, [0x264e] = 0x4c08, [0x264f] = 0x4c09,
  [0x2650] = 0x4c0a, [0x2651] = 0x4c0b, [0x2652] = 0x4c0c, [0x2653] = 0x4c0d,
  [0x2654] = 0x4c0e, [0x2655] = 0x4c0f, [0x2656] = 0x4c10, [0x2657] = 0x4c11,
  [0x2658] = 0x4c12, [0x2659] = 0x4c13, [0x265a] = 0x4c14, [0x265b] = 0x4c15,
  [0x265c] = 0x4c16, [0x265d] = 0x4c17, [0x265e] = 0x4c18, [0x265f] = 0x4c19,
  [0x2660] = 0x4c1a, [0x2661] = 0x4c1b, [0x2662] = 0x4c1c, [0x2663] = 0x4c1d,
  [0x2664] = 0x4c1e, [0x2665] = 0x4c1f, [0x2666] = 0x4c20, [0x2667] = 0x4c21,
  [0x2668] = 0x4c22, [0x2669] = 0x4c23, [0x266a] = 0x4c24, [0x266b] = 0x4c25,
  [0x266c] = 0x4c26, [0x266d] = 0x4c27, [0x266e] = 0x4c28, [0x266f] = 0x4c29,
  [0x2670] = 0x4c2a, [0x2671] = 0x4c2b, [0x2672] = 0x4c2c, [0x2673] = 0x4c2d,
  [0x2674] = 0x4c2e, [0x2675] = 0x4c2f, [0x2676] = 0x4c30, [0x2677] = 0x4c31,
  [0x2678] = 0x4c32, [0x2679] = 0x4c33, [0x267a] = 0x4c34, [0x267b] = 0x4c35,
  [0x267c] = 0x4c36, [0x267d] = 0x4c37, [0x267e] = 0x4c38, [0x267f] = 0x4c39,
  [0x2680] = 0x4c3a, [0x2681] = 0x4c3b, [0x2682] = 0x4c3c, [0x2683] = 0x4c3d,
  [0x2684] = 0x4c3e, [0x2685] = 0x4c3f, [0x2686] = 0x4c40, [0x2687] = 0x4c41,
  [0x2688] = 0x4c42, [0x2689] = 0x4c43, [0x268a] = 0x4c44, [0x268b] = 0x4c45,
  [0x268c] = 0x4c46, [0x268d] = 0x4c47, [0x268e] = 0x4c48, [0x268f] = 0x4c49,
  [0x2690] = 0x4c4a, [0x2691] = 0x4c4b, [0x2692] = 0x4c4c, [0x2693] = 0x4c4d,
  [0x2694] = 0x4c4e, [0x2695] = 0x4c4f, [0x2696] = 0x4c50, [0x2697] = 0x4c51,
  [0x2698] = 0x4c52, [0x2699] = 0x4c53, [0x269a] = 0x4c54, [0x269b] = 0x4c55,
  [0x269c] = 0x4c56, [0x269d] = 0x4c57, [0x269e] = 0x4c58, [0x269f] = 0x4c59,
  [0x26a0] = 0x4c5a, [0x26a1] = 0x4c5b, [0x26a2] = 0x4c5c, [0x26a3] = 0x4c5d,
  [0x26a4] = 0x4c5e, [0x26a5] = 0x4c5f, [0x26a6] = 0x4c60, [0x26a7] = 0x4c61,
  [0x26a8] = 0x4c62, [0x26a9] = 0x4c63, [0x26aa] = 0x4c64, [0x26ab] = 0x4c65,
  [0x26ac] = 0x4c66, [0x26ad] = 0x4c67, [0x26ae] = 0x4c68, [0x26af] = 0x4c69,
  [0x26b0] = 0x4c6a, [0x26b1] = 0x4c6b, [0x26b2] = 0x4c6c, [0x26b3] = 0x4c6d,
  [0x26b4] = 0x4c6e, [0x26b5] = 0x4c6f, [0x26b6] = 0x4c70, [0x26b7] = 0x4c71,
  [0x26b8] = 0x4c72, [0x26b9] = 0x4c73, [0x26ba] = 0x4c74, [0x26bb] = 0x4c75,
  [0x26bc] = 0x4c76, [0x26bd] = 0x4c78, [0x26be] = 0x4c79, [0x26bf] = 0x4c7a,
  [0x26c0] = 0x4c7b, [0x26c1] = 0x4c7c, [0x26c2] = 0x4c7d, [0x26c3] = 0x4c7e,
  [0x26c4] = 0x4c7f, [0x26c5] = 0x4c80, [0x26c6] = 0x4c81, [0x26c7] = 0x4c82,
  [0x26c8] = 0x4c83, [0x26c9] = 0x4c84, [0x26ca] = 0x4c85, [0x26cb] = 0x4c86,
  [0x26cc] = 0x4c87, [0x26cd] = 0x4c88, [0x26ce] = 0x4c89, [0x26cf] = 0x4c8a,
  [0x26d0] = 0x4c8b, [0x26d1] = 0x4c8c, [0x26d2] = 0x4c8d, [0x26d3] = 0x4c8e,
  [0x26d4] = 0x4c8f, [0x26d5] = 0x4c90, [0x26d6] = 0x4c91, [0x26d7] = 0x4c92,
  [0x26d8] = 0x4c93, [0x26d9] = 0x4c94, [0x26da] = 0x4c95, [0x26db] = 0x4c96,
  [0x26dc] = 0x4c97, [0x26dd] = 0x4c98, [0x26de] = 0x4c99, [0x26df] = 0x4c9a,
  [0x26e0] = 0x4c9b, [0x26e1] = 0x4c9c, [0x26e2] = 0x4c9d, [0x26e3] = 0x4c9e,
  [0x26e4] = 0x4ca4, [0x26e5] = 0x4ca5, [0x26e6] = 0x4ca6, [0x26e7] = 0x4ca7,
  [0x26e8] = 0x4ca8, [0x26e9] = 0x4ca9, [0x26ea] = 0x4caa, [0x26eb] = 0x4cab,
  [0x26ec] = 0x4cac, [0x26ed] = 0x4cad, [0x26ee] = 0x4cae, [0x26ef] = 0x4caf,
  [0x26f0] = 0x4cb0, [0x26f1] = 0x4cb1, [0x26f2] = 0x4cb2, [0x26f3] = 0x4cb3,
  [0x26f4] = 0x4cb4, [0x26f5] = 0x4cb5, [0x26f6] = 0x4cb6, [0x26f7] = 0x4cb7,
  [0x26f8] = 0x4cb8, [0x26f9] = 0x4cb9, [0x26fa] = 0x4cba, [0x26fb] = 0x4cbb,
  [0x26fc] = 0x4cbc, [0x26fd] = 0x4cbd, [0x26fe] = 0x4cbe, [0x26ff] = 0x4cbf,
  [0x2700] = 0x4cc0, [0x2701] = 0x4cc1, [0x2702] = 0x4cc2, [0x2703] = 0x4cc3,
  [0x2704] = 0x4cc4, [0x2705] = 0x4cc5, [0x2706] = 0x4cc6, [0x2707] = 0x4cc7,
  [0x2708] = 0x4cc8, [0x2709] = 0x4cc9, [0x270a] = 0x4cca, [0x270b] = 0x4ccb,
  [0x270c] = 0x4ccc, [0x270d] = 0x4ccd, [0x270e] = 0x4cce, [0x270f] = 0x4ccf,
  [0x2710] = 0x4cd0, [0x2711] = 0x4cd1, [0x2712] = 0x4cd2, [0x2713] = 0x4cd3,
  [0x2714] = 0x4cd4, [0x2715] = 0x4cd5, [0x2716] = 0x4cd6, [0x2717] = 0x4cd7,
  [0x2718] = 0x4cd8, [0x2719] = 0x4cd9, [0x271a] = 0x4cda, [0x271b] = 0x4cdb,
  [0x271c] = 0x4cdc, [0x271d] = 0x4cdd, [0x271e] = 0x4cde, [0x271f] = 0x4cdf,
  [0x2720] = 0x4ce0, [0x2721] = 0x4ce1, [0x2722] = 0x4ce2, [0x2723] = 0x4ce3,
  [0x2724] = 0x4ce4, [0x2725] = 0x4ce5, [0x2726] = 0x4ce6, [0x2727] = 0x4ce7,
  [0x2728] = 0x4ce8, [0x2729] = 0x4ce9, [0x272a] = 0x4cea, [0x272b] = 0x4ceb,
  [0x272c] = 0x4cec, [0x272d] = 0x4ced, [0x272e] = 0x4cee, [0x272f] = 0x4cef,
  [0x2730] = 0x4cf0, [0x2731] = 0x4cf1, [0x2732] = 0x4cf2, [0x2733] = 0x4cf3,
  [0x2734] = 0x4cf4, [0x2735] = 0x4cf5, [0x2736] = 0x4cf6, [0x2737] = 0x4cf7,
  [0x2738] = 0x4cf8, [0x2739] = 0x4cf9, [0x273a] = 0x4cfa, [0x273b] = 0x4cfb,
  [0x273c] = 0x4cfc, [0x273d] = 0x4cfd, [0x273e] = 0x4cfe, [0x273f] = 0x4cff,
  [0x2740] = 0x4d00, [0x2741] = 0x4d01, [0x2742] = 0x4d02, [0x2743] = 0x4d03,
  [0x2744] = 0x4d04, [0x2745] = 0x4d05, [0x2746] = 0x4d06, [0x2747] = 0x4d07,
  [0x2748] = 0x4d08, [0x2749] = 0x4d09, [0x274a] = 0x4d0a, [0x274b] = 0x4d0b,
  [0x274c] = 0x4d0c, [0x274d] = 0x4d0d, [0x274e] = 0x4d0e, [0x274f] = 0x4d0f,
  [0x2750] = 0x4d10, [0x2751] = 0x4d11, [0x2752] = 0x4d12, [0x2753] = 0x4d1a,
  [0x2754] = 0x4d1b, [0x2755] = 0x4d1c, [0x2756] = 0x4d1d, [0x2757] = 0x4d1e,
  [0x2758] = 0x4d1f, [0x2759] = 0x4d20, [0x275a] = 0x4d21, [0x275b] = 0x4d22,
  [0x275c] = 0x4d23, [0x275d] = 0x4d24, [0x275e] = 0x4d25, [0x275f] = 0x4d26,
  [0x2760] = 0x4d27, [0x2761] = 0x4d28, [0x2762] = 0x4d29, [0x2763] = 0x4d2a,
  [0x2764] = 0x4d2b, [0x2765] = 0x4d2c, [0x2766] = 0x4d2d, [0x2767] = 0x4d2e,
  [0x2768] = 0x4d2f, [0x2769] = 0x4d30, [0x276a] = 0x4d31, [0x276b] = 0x4d32,
  [0x276c] = 0x4d33, [0x276d] = 0x4d34, [0x276e] = 0x4d35, [0x276f] = 0x4d36,
  [0x2770] = 0x4d37, [0x2771] = 0x4d38, [0x2772] = 0x4d39, [0x2773] = 0x4d3a,
  [0x2774] = 0x4d3b, [0x2775] = 0x4d3c, [0x2776] = 0x4d3d, [0x2777] = 0x4d3e,
  [0x2778] = 0x4d3f, [0x2779] = 0x4d40, [0x277a] = 0x4d41, [0x277b] = 0x4d42,
  [0x277c] = 0x4d43, [0x277d] = 0x4d44, [0x277e] = 0x4d45, [0x277f] = 0x4d46,
  [0x2780] = 0x4d47, [0x2781] = 0x4d48, [0x2782] = 0x4d49, [0x2783] = 0x4d4a,
  [0x2784] = 0x4d4b, [0x2785] = 0x4d4c, [0x2786] = 0x4d4d, [0x2787] = 0x4d4e,
  [0x2788] = 0x4d4f, [0x2789] = 0x4d50, [0x278a] = 0x4d51, [0x278b] = 0x4d52,
  [0x278c] = 0x4d53, [0x278d] = 0x4d54, [0x278e] = 0x4d55, [0x278f] = 0x4d56,
  [0x2790] = 0x4d57, [0x2791] = 0x4d58, [0x2792] = 0x4d59, [0x2793] = 0x4d5a,
  [0x2794] = 0x4d5b, [0x2795] = 0x4d5c, [0x2796] = 0x4d5d, [0x2797] = 0x4d5e,
  [0x2798] = 0x4d5f, [0x2799] = 0x4d60, [0x279a] = 0x4d61, [0x279b] = 0x4d62,
  [0x279c] = 0x4d63, [0x279d] = 0x4d64, [0x279e] = 0x4d65, [0x279f] = 0x4d66,
  [0x27a0] = 0x4d67, [0x27a1] = 0x4d68, [0x27a2] = 0x4d69, [0x27a3] = 0x4d6a,
  [0x27a4] = 0x4d6b, [0x27a5] = 0x4d6c, [0x27a6] = 0x4d6d, [0x27a7] = 0x4d6e,
  [0x27a8] = 0x4d6f, [0x27a9] = 0x4d70, [0x27aa] = 0x4d71, [0x27ab] = 0x4d72,
  [0x27ac] = 0x4d73, [0x27ad] = 0x4d74, [0x27ae] = 0x4d75, [0x27af] = 0x4d76,
  [0x27b0] = 0x4d77, [0x27b1] = 0x4d78, [0x27b2] = 0x4d79, [0x27b3] = 0x4d7a,
  [0x27b4] = 0x4d7b, [0x27b5] = 0x4d7c, [0x27b6] = 0x4d7d, [0x27b7] = 0x4d7e,
  [0x27b8] = 0x4d7f, [0x27b9] = 0x4d80, [0x27ba] = 0x4d81, [0x27bb] = 0x4d82,
  [0x27bc] = 0x4d83, [0x27bd] = 0x4d84, [0x27be] = 0x4d85, [0x27bf] = 0x4d86,
  [0x27c0] = 0x4d87, [0x27c1] = 0x4d88, [0x27c2] = 0x4d89, [0x27c3] = 0x4d8a,
  [0x27c4] = 0x4d8b, [0x27c5] = 0x4d8c, [0x27c6] = 0x4d8d, [0x27c7] = 0x4d8e,
  [0x27c8] = 0x4d8f, [0x27c9] = 0x4d90, [0x27ca] = 0x4d91, [0x27cb] = 0x4d92,
  [0x27cc] = 0x4d93, [0x27cd] = 0x4d94, [0x27ce] = 0x4d95, [0x27cf] = 0x4d96,
  [0x27d0] = 0x4d97, [0x27d1] = 0x4d98, [0x27d2] = 0x4d99, [0x27d3] = 0x4d9a,
  [0x27d4] = 0x4d9b, [0x27d5] = 0x4d9c, [0x27d6] = 0x4d9d, [0x27d7] = 0x4d9e,
  [0x27d8] = 0x4d9f, [0x27d9] = 0x4da0, [0x27da] = 0x4da1, [0x27db] = 0x4da2,
  [0x27dc] = 0x4da3, [0x27dd] = 0x4da4, [0x27de] = 0x4da5, [0x27df] = 0x4da6,
  [0x27e0] = 0x4da7, [0x27e1] = 0x4da8, [0x27e2] = 0x4da9, [0x27e3] = 0x4daa,
  [0x27e4] = 0x4dab, [0x27e5] = 0x4dac, [0x27e6] = 0x4dad, [0x27e7] = 0x4daf,
  [0x27e8] = 0x4db0, [0x27e9] = 0x4db1, [0x27ea] = 0x4db2, [0x27eb] = 0x4db3,
  [0x27ec] = 0x4db4, [0x27ed] = 0x4db5, [0x27ee] = 0x4db6, [0x27ef] = 0x4db7,
  [0x27f0] = 0x4db8, [0x27f1] = 0x4db9, [0x27f2] = 0x4dba, [0x27f3] = 0x4dbb,
  [0x27f4] = 0x4dbc, [0x27f5] = 0x4dbd, [0x27f6] = 0x4dbe, [0x27f7] = 0x4dbf,
  [0x27f8] = 0x4dc0, [0x27f9] = 0x4dc1, [0x27fa] = 0x4dc2, [0x27fb] = 0x4dc3,
  [0x27fc] = 0x4dc4, [0x27fd] = 0x4dc5, [0x27fe] = 0x4dc6, [0x27ff] = 0x4dc7,
  [0x2800] = 0x4dc8, [0x2801] = 0x4dc9, [0x2802] = 0x4dca, [0x2803] = 0x4dcb,
  [0x2804] = 0x4dcc, [0x2805] = 0x4dcd, [0x2806] = 0x4dce, [0x2807] = 0x4dcf,
  [0x2808] = 0x4dd0, [0x2809] = 0x4dd1, [0x280a] = 0x4dd2, [0x280b] = 0x4dd3,
  [0x280c] = 0x4dd4, [0x280d] = 0x4dd5, [0x280e] = 0x4dd6, [0x280f] = 0x4dd7,
  [0x2810] = 0x4dd8, [0x2811] = 0x4dd9, [0x2812] = 0x4dda, [0x2813] = 0x4ddb,
  [0x2814] = 0x4ddc, [0x2815] = 0x4ddd, [0x2816] = 0x4dde, [0x2817] = 0x4ddf,
  [0x2818] = 0x4de0, [0x2819] = 0x4de1, [0x281a] = 0x4de2, [0x281b] = 0x4de3,
  [0x281c] = 0x4de4, [0x281d] = 0x4de5, [0x281e] = 0x4de6, [0x281f] = 0x4de7,
  [0x2820] = 0x4de8, [0x2821] = 0x4de9, [0x2822] = 0x4dea, [0x2823] = 0x4deb,
  [0x2824] = 0x4dec, [0x2825] = 0x4ded, [0x2826] = 0x4dee, [0x2827] = 0x4def,
  [0x2828] = 0x4df0, [0x2829] = 0x4df1, [0x282a] = 0x4df2, [0x282b] = 0x4df3,
  [0x282c] = 0x4df4, [0x282d] = 0x4df5, [0x282e] = 0x4df6, [0x282f] = 0x4df7,
  [0x2830] = 0x4df8, [0x2831] = 0x4df9, [0x2832] = 0x4dfa, [0x2833] = 0x4dfb,
  [0x2834] = 0x4dfc, [0x2835] = 0x4dfd, [0x2836] = 0x4dfe, [0x2837] = 0x4dff,
  [0x2838] = 0x9fa6, [0x2839] = 0x9fa7, [0x283a] = 0x9fa8, [0x283b] = 0x9fa9,
  [0x283c] = 0x9faa, [0x283d] = 0x9fab, [0x283e] = 0x9fac, [0x283f] = 0x9fad,
  [0x2840] = 0x9fae, [0x2841] = 0x9faf, [0x2842] = 0x9fb0, [0x2843] = 0x9fb1,
  [0x2844] = 0x9fb2, [0x2845] = 0x9fb3, [0x284e] = 0xe76c, [0x284f] = 0xe7c8,
  [0x2850] = 0xe7e7, [0x2851] = 0xe7e8, [0x2852] = 0xe7e9, [0x2853] = 0xe7ea,
  [0x2854] = 0xe7eb, [0x2855] = 0xe7ec, [0x2856] = 0xe7ed, [0x2857] = 0xe7ee,
  [0x2858] = 0xe7ef, [0x2859] = 0xe7f0, [0x285a] = 0xe7f1, [0x285b] = 0xe7f2,
  [0x285c] = 0xe7f3, [0x285d] = 0xe815, [0x285e] = 0xe819, [0x285f] = 0xe81a,
  [0x2860] = 0xe81b, [0x2861] = 0xe81c, [0x2862] = 0xe81d, [0x2863] = 0xe81f,
  [0x2864] = 0xe820, [0x2865] = 0xe821, [0x2866] = 0xe822, [0x2867] = 0xe823,
  [0x2868] = 0xe824, [0x2869] = 0xe825, [0x286a] = 0xe827, [0x286b] = 0xe828,
  [0x286c] = 0xe829, [0x286d] = 0xe82a, [0x286e] = 0xe82d, [0x286f] = 0xe82e,
  [0x2870] = 0xe82f, [0x2871] = 0xe830, [0x2872] = 0xe833, [0x2873] = 0xe834,
  [0x2874] = 0xe835, [0x2875] = 0xe836, [0x2876] = 0xe837, [0x2877] = 0xe838,
  [0x2878] = 0xe839, [0x2879] = 0xe83a, [0x287a] = 0xe83c, [0x287b] = 0xe83d,
  [0x287c] = 0xe83e, [0x287d] = 0xe83f, [0x287e] = 0xe840, [0x287f] = 0xe841,
  [0x2880] = 0xe842, [0x2881] = 0xe844, [0x2882] = 0xe845, [0x2883] = 0xe846,
  [0x2884] = 0xe847, [0x2885] = 0xe848, [0x2886] = 0xe849, [0x2887] = 0xe84a,
  [0x2888] = 0xe84b, [0x2889] = 0xe84c, [0x288a] = 0xe84d, [0x288b] = 0xe84e,
  [0x288c] = 0xe84f, [0x288d] = 0xe850, [0x288e] = 0xe851, [0x288f] = 0xe852,
  [0x2890] = 0xe853, [0x2891] = 0xe856, [0x2892] = 0xe857, [0x2893] = 0xe858,
  [0x2894] = 0xe859, [0x2895] = 0xe85a, [0x2896] = 0xe85b, [0x2897] = 0xe85c,
  [0x2898] = 0xe85d, [0x2899] = 0xe85e, [0x289a] = 0xe85f, [0x289b] = 0xe860,
  [0x289c] = 0xe861, [0x289d] = 0xe862, [0x289e] = 0xe863, [0x289f] = 0xf92d,
  [0x28a0] = 0xf92e, [0x28a1] = 0xf92f, [0x28a2] = 0xf930, [0x28a3] = 0xf931,
  [0x28a4] = 0xf932, [0x28a5] = 0xf933, [0x28a6] = 0xf934, [0x28a7] = 0xf935,
  [0x28a8] = 0xf936, [0x28a9] = 0xf937, [0x28aa] = 0xf938, [0x28ab] = 0xf939,
  [0x28ac] = 0xf93a, [0x28ad] = 0xf93b, [0x28ae] = 0xf93c, [0x28af] = 0xf93d,
  [0x28b0] = 0xf93e, [0x28b1] = 0xf93f, [0x28b2] = 0xf940, [0x28b3] = 0xf941,
  [0x28b4] = 0xf942, [0x28b5] = 0xf943, [0x28b6] = 0xf944, [0x28b7] = 0xf945,
  [0x28b8] = 0xf946, [0x28b9] = 0xf947, [0x28ba] = 0xf948, [0x28bb] = 0xf949,
  [0x28bc] = 0xf94a, [0x28bd] = 0xf94b, [0x28be] = 0xf94c, [0x28bf] = 0xf94d,
  [0x28c0] = 0xf94e, [0x28c1] = 0xf94f, [0x28c2] = 0xf950, [0x28c3] = 0xf951,
  [0x28c4] = 0xf952, [0x28c5] = 0xf953, [0x28c6] = 0xf954, [0x28c7] = 0xf955,
  [0x28c8] = 0xf956, [0x28c9] = 0xf957, [0x28ca] = 0xf958, [0x28cb] = 0xf959,
  [0x28cc] = 0xf95a, [0x28cd] = 0xf95b, [0x28ce] = 0xf95c, [0x28cf] = 0xf95d,
  [0x28d0] = 0xf95e, [0x28d1] = 0xf95f, [0x28d2] = 0xf960, [0x28d3] = 0xf961,
  [0x28d4] = 0xf962, [0x28d5] = 0xf963, [0x28d6] = 0xf964, [0x28d7] = 0xf965,
  [0x28d8] = 0xf966, [0x28d9] = 0xf967, [0x28da] = 0xf968, [0x28db] = 0xf969,
  [0x28dc] = 0xf96a, [0x28dd] = 0xf96b, [0x28de] = 0xf96c, [0x28df] = 0xf96d,
  [0x28e0] = 0xf96e, [0x28e1] = 0xf96f, [0x28e2] = 0xf970, [0x28e3] = 0xf971,
  [0x28e4] = 0xf972, [0x28e5] = 0xf973, [0x28e6] = 0xf974, [0x28e7] = 0xf975,
  [0x28e8] = 0xf976, [0x28e9] = 0xf977, [0x28ea] = 0xf978, [0x28eb] = 0xf97a,
  [0x28ec] = 0xf97b, [0x28ed] = 0xf97c, [0x28ee] = 0xf97d, [0x28ef] = 0xf97e,
  [0x28f0] = 0xf97f, [0x28f1] = 0xf980, [0x28f2] = 0xf981, [0x28f3] = 0xf982,
  [0x28f4] = 0xf983, [0x28f5] = 0xf984, [0x28f6] = 0xf985, [0x28f7] = 0xf986,
  [0x28f8] = 0xf987, [0x28f9] = 0xf988, [0x28fa] = 0xf989, [0x28fb] = 0xf98a,
  [0x28fc] = 0xf98b, [0x28fd] = 0xf98c, [0x28fe] = 0xf98d, [0x28ff] = 0xf98e,
  [0x2900] = 0xf98f, [0x2901] = 0xf990, [0x2902] = 0xf991, [0x2903] = 0xf992,
  [0x2904] = 0xf993, [0x2905] = 0xf994, [0x2906] = 0xf996, [0x2907] = 0xf997,
  [0x2908] = 0xf998, [0x2909] = 0xf999, [0x290a] = 0xf99a, [0x290b] = 0xf99b,
  [0x290c] = 0xf99c, [0x290d] = 0xf99d, [0x290e] = 0xf99e, [0x290f] = 0xf99f,
  [0x2910] = 0xf9a0, [0x2911] = 0xf9a1, [0x2912] = 0xf9a2, [0x2913] = 0xf9a3,
  [0x2914] = 0xf9a4, [0x2915] = 0xf9a5, [0x2916] = 0xf9a6, [0x2917] = 0xf9a7,
  [0x2918] = 0xf9a8, [0x2919] = 0xf9a9, [0x291a] = 0xf9aa, [0x291b] = 0xf9ab,
  [0x291c] = 0xf9ac, [0x291d] = 0xf9ad, [0x291e] = 0xf9ae, [0x291f] = 0xf9af,
  [0x2920] = 0xf9b0, [0x2921] = 0xf9b1, [0x2922] = 0xf9b2, [0x2923] = 0xf9b3,
  [0x2924] = 0xf9b4, [0x2925] = 0xf9b5, [0x2926] = 0xf9b6, [0x2927] = 0xf9b7,
  [0x2928] = 0xf9b8, [0x2929] = 0xf9b9, [0x292a] = 0xf9ba, [0x292b] = 0xf9bb,
  [0x292c] = 0xf9bc, [0x292d] = 0xf9bd, [0x292e] = 0xf9be, [0x292f] = 0xf9bf,
  [0x2930] = 0xf9c0, [0x2931] = 0xf9c1, [0x2932] = 0xf9c2, [0x2933] = 0xf9c3,
  [0x2934] = 0xf9c4, [0x2935] = 0xf9c5, [0x2936] = 0xf9c6, [0x2937] = 0xf9c7,
  [0x2938] = 0xf9c8, [0x2939] = 0xf9c9, [0x293a] = 0xf9ca, [0x293b] = 0xf9cb,
  [0x293c] = 0xf9cc, [0x293d] = 0xf9cd, [0x293e] = 0xf9ce, [0x293f] = 0xf9cf,
  [0x2940] = 0xf9d0, [0x2941] = 0xf9d1, [0x2942] = 0xf9d2, [0x2943] = 0xf9d3,
  [0x2944] = 0xf9d4, [0x2945] = 0xf9d5, [0x2946] = 0xf9d6, [0x2947] = 0xf9d7,
  [0x2948] = 0xf9d8, [0x2949] = 0xf9d9, [0x294a] = 0xf9da, [0x294b] = 0xf9db,
  [0x294c] = 0xf9dc, [0x294d] = 0xf9dd, [0x294e] = 0xf9de, [0x294f] = 0xf9df,
  [0x2950] = 0xf9e0, [0x2951] = 0xf9e1, [0x2952] = 0xf9e2, [0x2953] = 0xf9e3,
  [0x2954] = 0xf9e4, [0x2955] = 0xf9e5, [0x2956] = 0xf9e6, [0x2957] = 0xf9e8,
  [0x2958] = 0xf9e9, [0x2959] = 0xf9ea, [0x295a] = 0xf9eb, [0x295b] = 0xf9ec,
  [0x295c] = 0xf9ed, [0x295d] = 0xf9ee, [0x295e] = 0xf9ef, [0x295f] = 0xf9f0,
  [0x2960] = 0xf9f2, [0x2961] = 0xf9f3, [0x2962] = 0xf9f4, [0x2963] = 0xf9f5,
  [0x2964] = 0xf9f6, [0x2965] = 0xf9f7, [0x2966] = 0xf9f8, [0x2967] = 0xf9f9,
  [0x2968] = 0xf9fa, [0x2969] = 0xf9fb, [0x296a] = 0xf9fc, [0x296b] = 0xf9fd,
  [0x296c] = 0xf9fe, [0x296d] = 0xf9ff, [0x296e] = 0xfa00, [0x296f] = 0xfa01,
  [0x2970] = 0xfa02, [0x2971] = 0xfa03, [0x2972] = 0xfa04, [0x2973] = 0xfa05,
  [0x2974] = 0xfa06, [0x2975] = 0xfa07, [0x2976] = 0xfa08, [0x2977] = 0xfa09,
  [0x2978] = 0xfa0a, [0x2979] = 0xfa0b, [0x297a] = 0xfa10, [0x297b] = 0xfa12,
  [0x297c] = 0xfa15, [0x297d] = 0xfa16, [0x297e] = 0xfa17, [0x297f] = 0xfa19,
  [0x2980] = 0xfa1a, [0x2981] = 0xfa1b, [0x2982] = 0xfa1c, [0x2983] = 0xfa1d,
  [0x2984] = 0xfa1e, [0x2985] = 0xfa22, [0x2986] = 0xfa25, [0x2987] = 0xfa26,
  [0x2988] = 0xfa2a, [0x2989] = 0xfa2b, [0x298a] = 0xfa2c, [0x298b] = 0xfa2d,
  [0x298c] = 0xfa2e, [0x298d] = 0xfa2f, [0x298e] = 0xfa30, [0x298f] = 0xfa31,
  [0x2990] = 0xfa32, [0x2991] = 0xfa33, [0x2992] = 0xfa34, [0x2993] = 0xfa35,
  [0x2994] = 0xfa36, [0x2995] = 0xfa37, [0x2996] = 0xfa38, [0x2997] = 0xfa39,
  [0x2998] = 0xfa3a, [0x2999] = 0xfa3b, [0x299a] = 0xfa3c, [0x299b] = 0xfa3d,
  [0x299c] = 0xfa3e, [0x299d] = 0xfa3f, [0x299e] = 0xfa40, [0x299f] = 0xfa41,
  [0x29a0] = 0xfa42, [0x29a1] = 0xfa43, [0x29a2] = 0xfa44, [0x29a3] = 0xfa45,
  [0x29a4] = 0xfa46, [0x29a5] = 0xfa47, [0x29a6] = 0xfa48, [0x29a7] = 0xfa49,
  [0x29a8] = 0xfa4a, [0x29a9] = 0xfa4b, [0x29aa] = 0xfa4c, [0x29ab] = 0xfa4d,
  [0x29ac] = 0xfa4e, [0x29ad] = 0xfa4f, [0x29ae] = 0xfa50, [0x29af] = 0xfa51,
  [0x29b0] = 0xfa52, [0x29b1] = 0xfa53, [0x29b2] = 0xfa54, [0x29b3] = 0xfa55,
  [0x29b4] = 0xfa56, [0x29b5] = 0xfa57, [0x29b6] = 0xfa58, [0x29b7] = 0xfa59,
  [0x29b8] = 0xfa5a, [0x29b9] = 0xfa5b, [0x29ba] = 0xfa5c, [0x29bb] = 0xfa5d,
  [0x29bc] = 0xfa5e, [0x29bd] = 0xfa5f, [0x29be] = 0xfa60, [0x29bf] = 0xfa61,
  [0x29c0] = 0xfa62, [0x29c1] = 0xfa63, [0x29c2] = 0xfa64, [0x29c3] = 0xfa65,
  [0x29c4] = 0xfa66, [0x29c5] = 0xfa67, [0x29c6] = 0xfa68, [0x29c7] = 0xfa69,
  [0x29c8] = 0xfa6a, [0x29c9] = 0xfa6b, [0x29ca] = 0xfa6c, [0x29cb] = 0xfa6d,
  [0x29cc] = 0xfa6e, [0x29cd] = 0xfa6f, [0x29ce] = 0xfa70, [0x29cf] = 0xfa71,
  [0x29d0] = 0xfa72, [0x29d1] = 0xfa73, [0x29d2] = 0xfa74, [0x29d3] = 0xfa75,
  [0x29d4] = 0xfa76, [0x29d5] = 0xfa77, [0x29d6] = 0xfa78, [0x29d7] = 0xfa79,
  [0x29d8] = 0xfa7a, [0x29d9] = 0xfa7b, [0x29da] = 0xfa7c, [0x29db] = 0xfa7d,
  [0x29dc] = 0xfa7e, [0x29dd] = 0xfa7f, [0x29de] = 0xfa80, [0x29df] = 0xfa81,
  [0x29e0] = 0xfa82, [0x29e1] = 0xfa83, [0x29e2] = 0xfa84, [0x29e3] = 0xfa85,
  [0x29e4] = 0xfa86, [0x29e5] = 0xfa87, [0x29e6] = 0xfa88, [0x29e7] = 0xfa89,
  [0x29e8] = 0xfa8a, [0x29e9] = 0xfa8b, [0x29ea] = 0xfa8c, [0x29eb] = 0xfa8d,
  [0x29ec] = 0xfa8e, [0x29ed] = 0xfa8f, [0x29ee] = 0xfa90, [0x29ef] = 0xfa91,
  [0x29f0] = 0xfa92, [0x29f1] = 0xfa93, [0x29f2] = 0xfa94, [0x29f3] = 0xfa95,
  [0x29f4] = 0xfa96, [0x29f5] = 0xfa97, [0x29f6] = 0xfa98, [0x29f7] = 0xfa99,
  [0x29f8] = 0xfa9a, [0x29f9] = 0xfa9b, [0x29fa] = 0xfa9c, [0x29fb] = 0xfa9d,
  [0x29fc] = 0xfa9e, [0x29fd] = 0xfa9f, [0x29fe] = 0xfaa0, [0x29ff] = 0xfaa1,
  [0x2a00] = 0xfaa2, [0x2a01] = 0xfaa3, [0x2a02] = 0xfaa4, [0x2a03] = 0xfaa5,
  [0x2a04] = 0xfaa6, [0x2a05] = 0xfaa7, [0x2a06] = 0xfaa8, [0x2a07] = 0xfaa9,
  [0x2a08] = 0xfaaa, [0x2a09] = 0xfaab, [0x2a0a] = 0xfaac, [0x2a0b] = 0xfaad,
  [0x2a0c] = 0xfaae, [0x2a0d] = 0xfaaf, [0x2a0e] = 0xfab0, [0x2a0f] = 0xfab1,
  [0x2a10] = 0xfab2, [0x2a11] = 0xfab3, [0x2a12] = 0xfab4, [0x2a13] = 0xfab5,
  [0x2a14] = 0xfab6, [0x2a15] = 0xfab7, [0x2a16] = 0xfab8, [0x2a17] = 0xfab9,
  [0x2a18] = 0xfaba, [0x2a19] = 0xfabb, [0x2a1a] = 0xfabc, [0x2a1b] = 0xfabd,
  [0x2a1c] = 0xfabe, [0x2a1d] = 0xfabf, [0x2a1e] = 0xfac0, [0x2a1f] = 0xfac1,
  [0x2a20] = 0xfac2, [0x2a21] = 0xfac3, [0x2a22] = 0xfac4, [0x2a23] = 0xfac5,
  [0x2a24] = 0xfac6, [0x2a25] = 0xfac7, [0x2a26] = 0xfac8, [0x2a27] = 0xfac9,
  [0x2a28] = 0xfaca, [0x2a29] = 0xfacb, [0x2a2a] = 0xfacc, [0x2a2b] = 0xfacd,
  [0x2a2c] = 0xface, [0x2a2d] = 0xfacf, [0x2a2e] = 0xfad0, [0x2a2f] = 0xfad1,
  [0x2a30] = 0xfad2, [0x2a31] = 0xfad3, [0x2a32] = 0xfad4, [0x2a33] = 0xfad5,
  [0x2a34] = 0xfad6, [0x2a35] = 0xfad7, [0x2a36] = 0xfad8, [0x2a37] = 0xfad9,
  [0x2a38] = 0xfada, [0x2a39] = 0xfadb, [0x2a3a] = 0xfadc, [0x2a3b] = 0xfadd,
  [0x2a3c] = 0xfade, [0x2a3d] = 0xfadf, [0x2a3e] = 0xfae0, [0x2a3f] = 0xfae1,
  [0x2a40] = 0xfae2, [0x2a41] = 0xfae3, [0x2a42] = 0xfae4, [0x2a43] = 0xfae5,
  [0x2a44] = 0xfae6, [0x2a45] = 0xfae7, [0x2a46] = 0xfae8, [0x2a47] = 0xfae9,
  [0x2a48] = 0xfaea, [0x2a49] = 0xfaeb, [0x2a4a] = 0xfaec, [0x2a4b] = 0xfaed,
  [0x2a4c] = 0xfaee, [0x2a4d] = 0xfaef, [0x2a4e] = 0xfaf0, [0x2a4f] = 0xfaf1,
  [0x2a50] = 0xfaf2, [0x2a51] = 0xfaf3, [0x2a52] = 0xfaf4, [0x2a53] = 0xfaf5,
  [0x2a54] = 0xfaf6, [0x2a55] = 0xfaf7, [0x2a56] = 0xfaf8, [0x2a57] = 0xfaf9,
  [0x2a58] = 0xfafa, [0x2a59] = 0xfafb, [0x2a5a] = 0xfafc, [0x2a5b] = 0xfafd,
  [0x2a5c] = 0xfafe, [0x2a5d] = 0xfaff, [0x2a5e] = 0xfb00, [0x2a5f] = 0xfb01,
  [0x2a60] = 0xfb02, [0x2a61] = 0xfb03, [0x2a62] = 0xfb04, [0x2a63] = 0xfb05,
  [0x2a64] = 0xfb06, [0x2a65] = 0xfb07, [0x2a66] = 0xfb08, [0x2a67] = 0xfb09,
  [0x2a68] = 0xfb0a, [0x2a69] = 0xfb0b, [0x2a6a] = 0xfb0c, [0x2a6b] = 0xfb0d,
  [0x2a6c] = 0xfb0e, [0x2a6d] = 0xfb0f, [0x2a6e] = 0xfb10, [0x2a6f] = 0xfb11,
  [0x2a70] = 0xfb12, [0x2a71] = 0xfb13, [0x2a72] = 0xfb14, [0x2a73] = 0xfb15,
  [0x2a74] = 0xfb16, [0x2a75] = 0xfb17, [0x2a76] = 0xfb18, [0x2a77] = 0xfb19,
  [0x2a78] = 0xfb1a, [0x2a79] = 0xfb1b, [0x2a7a] = 0xfb1c, [0x2a7b] = 0xfb1d,
  [0x2a7c] = 0xfb1e, [0x2a7d] = 0xfb1f, [0x2a7e] = 0xfb20, [0x2a7f] = 0xfb21,
  [0x2a80] = 0xfb22, [0x2a81] = 0xfb23, [0x2a82] = 0xfb24, [0x2a83] = 0xfb25,
  [0x2a84] = 0xfb26, [0x2a85] = 0xfb27, [0x2a86] = 0xfb28, [0x2a87] = 0xfb29,
  [0x2a88] = 0xfb2a, [0x2a89] = 0xfb2b, [0x2a8a] = 0xfb2c, [0x2a8b] = 0xfb2d,
  [0x2a8c] = 0xfb2e, [0x2a8d] = 0xfb2f, [0x2a8e] = 0xfb30, [0x2a8f] = 0xfb31,
  [0x2a90] = 0xfb32, [0x2a91] = 0xfb33, [0x2a92] = 0xfb34, [0x2a93] = 0xfb35,
  [0x2a94] = 0xfb36, [0x2a95] = 0xfb37, [0x2a96] = 0xfb38, [0x2a97] = 0xfb39,
  [0x2a98] = 0xfb3a, [0x2a99] = 0xfb3b, [0x2a9a] = 0xfb3c, [0x2a9b] = 0xfb3d,
  [0x2a9c] = 0xfb3e, [0x2a9d] = 0xfb3f, [0x2a9e] = 0xfb40, [0x2a9f] = 0xfb41,
  [0x2aa0] = 0xfb42, [0x2aa1] = 0xfb43, [0x2aa2] = 0xfb44, [0x2aa3] = 0xfb45,
  [0x2aa4] = 0xfb46, [0x2aa5] = 0xfb47, [0x2aa6] = 0xfb48, [0x2aa7] = 0xfb49,
  [0x2aa8] = 0xfb4a, [0x2aa9] = 0xfb4b, [0x2aaa] = 0xfb4c, [0x2aab] = 0xfb4d,
  [0x2aac] = 0xfb4e, [0x2aad] = 0xfb4f, [0x2aae] = 0xfb50, [0x2aaf] = 0xfb51,
  [0x2ab0] = 0xfb52, [0x2ab1] = 0xfb53, [0x2ab2] = 0xfb54, [0x2ab3] = 0xfb55,
  [0x2ab4] = 0xfb56, [0x2ab5] = 0xfb57, [0x2ab6] = 0xfb58, [0x2ab7] = 0xfb59,
  [0x2ab8] = 0xfb5a, [0x2ab9] = 0xfb5b, [0x2aba] = 0xfb5c, [0x2abb] = 0xfb5d,
  [0x2abc] = 0xfb5e, [0x2abd] = 0xfb5f, [0x2abe] = 0xfb60, [0x2abf] = 0xfb61,
  [0x2ac0] = 0xfb62, [0x2ac1] = 0xfb63, [0x2ac2] = 0xfb64, [0x2ac3] = 0xfb65,
  [0x2ac4] = 0xfb66, [0x2ac5] = 0xfb67, [0x2ac6] = 0xfb68, [0x2ac7] = 0xfb69,
  [0x2ac8] = 0xfb6a, [0x2ac9] = 0xfb6b, [0x2aca] = 0xfb6c, [0x2acb] = 0xfb6d,
  [0x2acc] = 0xfb6e, [0x2acd] = 0xfb6f, [0x2ace] = 0xfb70, [0x2acf] = 0xfb71,
  [0x2ad0] = 0xfb72, [0x2ad1] = 0xfb73, [0x2ad2] = 0xfb74, [0x2ad3] = 0xfb75,
  [0x2ad4] = 0xfb76, [0x2ad5] = 0xfb77, [0x2ad6] = 0xfb78, [0x2ad7] = 0xfb79,
  [0x2ad8] = 0xfb7a, [0x2ad9] = 0xfb7b, [0x2ada] = 0xfb7c, [0x2adb] = 0xfb7d,
  [0x2adc] = 0xfb7e, [0x2add] = 0xfb7f, [0x2ade] = 0xfb80, [0x2adf] = 0xfb81,
  [0x2ae0] = 0xfb82, [0x2ae1] = 0xfb83, [0x2ae2] = 0xfb84, [0x2ae3] = 0xfb85,
  [0x2ae4] = 0xfb86, [0x2ae5] = 0xfb87, [0x2ae6] = 0xfb88, [0x2ae7] = 0xfb89,
  [0x2ae8] = 0xfb8a, [0x2ae9] = 0xfb8b, [0x2aea] = 0xfb8c, [0x2aeb] = 0xfb8d,
  [0x2aec] = 0xfb8e, [0x2aed] = 0xfb8f, [0x2aee] = 0xfb90, [0x2aef] = 0xfb91,
  [0x2af0] = 0xfb92, [0x2af1] = 0xfb93, [0x2af2] = 0xfb94, [0x2af3] = 0xfb95,
  [0x2af4] = 0xfb96, [0x2af5] = 0xfb97, [0x2af6] = 0xfb98, [0x2af7] = 0xfb99,
  [0x2af8] = 0xfb9a, [0x2af9] = 0xfb9b, [0x2afa] = 0xfb9c, [0x2afb] = 0xfb9d,
  [0x2afc] = 0xfb9e, [0x2afd] = 0xfb9f, [0x2afe] = 0xfba0, [0x2aff] = 0xfba1,
  [0x2b00] = 0xfba2, [0x2b01] = 0xfba3, [0x2b02] = 0xfba4, [0x2b03] = 0xfba5,
  [0x2b04] = 0xfba6, [0x2b05] = 0xfba7, [0x2b06] = 0xfba8, [0x2b07] = 0xfba9,
  [0x2b08] = 0xfbaa, [0x2b09] = 0xfbab, [0x2b0a] = 0xfbac, [0x2b0b] = 0xfbad,
  [0x2b0c] = 0xfbae, [0x2b0d] = 0xfbaf, [0x2b0e] = 0xfbb0, [0x2b0f] = 0xfbb1,
  [0x2b10] = 0xfbb2, [0x2b11] = 0xfbb3, [0x2b12] = 0xfbb4, [0x2b13] = 0xfbb5,
  [0x2b14] = 0xfbb6, [0x2b15] = 0xfbb7, [0x2b16] = 0xfbb8, [0x2b17] = 0xfbb9,
  [0x2b18] = 0xfbba, [0x2b19] = 0xfbbb, [0x2b1a] = 0xfbbc, [0x2b1b] = 0xfbbd,
  [0x2b1c] = 0xfbbe, [0x2b1d] = 0xfbbf, [0x2b1e] = 0xfbc0, [0x2b1f] = 0xfbc1,
  [0x2b20] = 0xfbc2, [0x2b21] = 0xfbc3, [0x2b22] = 0xfbc4, [0x2b23] = 0xfbc5,
  [0x2b24] = 0xfbc6, [0x2b25] = 0xfbc7, [0x2b26] = 0xfbc8, [0x2b27] = 0xfbc9,
  [0x2b28] = 0xfbca, [0x2b29] = 0xfbcb, [0x2b2a] = 0xfbcc, [0x2b2b] = 0xfbcd,
  [0x2b2c] = 0xfbce, [0x2b2d] = 0xfbcf, [0x2b2e] = 0xfbd0, [0x2b2f] = 0xfbd1,
  [0x2b30] = 0xfbd2, [0x2b31] = 0xfbd3, [0x2b32] = 0xfbd4, [0x2b33] = 0xfbd5,
  [0x2b34] = 0xfbd6, [0x2b35] = 0xfbd7, [0x2b36] = 0xfbd8, [0x2b37] = 0xfbd9,
  [0x2b38] = 0xfbda, [0x2b39] = 0xfbdb, [0x2b3a] = 0xfbdc, [0x2b3b] = 0xfbdd,
  [0x2b3c] = 0xfbde, [0x2b3d] = 0xfbdf, [0x2b3e] = 0xfbe0, [0x2b3f] = 0xfbe1,
  [0x2b40] = 0xfbe2, [0x2b41] = 0xfbe3, [0x2b42] = 0xfbe4, [0x2b43] = 0xfbe5,
  [0x2b44] = 0xfbe6, [0x2b45] = 0xfbe7, [0x2b46] = 0xfbe8, [0x2b47] = 0xfbe9,
  [0x2b48] = 0xfbea, [0x2b49] = 0xfbeb, [0x2b4a] = 0xfbec, [0x2b4b] = 0xfbed,
  [0x2b4c] = 0xfbee, [0x2b4d] = 0xfbef, [0x2b4e] = 0xfbf0, [0x2b4f] = 0xfbf1,
  [0x2b50] = 0xfbf2, [0x2b51] = 0xfbf3, [0x2b52] = 0xfbf4, [0x2b53] = 0xfbf5,
  [0x2b54] = 0xfbf6, [0x2b55] = 0xfbf7, [0x2b56] = 0xfbf8, [0x2b57] = 0xfbf9,
  [0x2b58] = 0xfbfa, [0x2b59] = 0xfbfb, [0x2b5a] = 0xfbfc, [0x2b5b] = 0xfbfd,
  [0x2b5c] = 0xfbfe, [0x2b5d] = 0xfbff, [0x2b5e] = 0xfc00, [0x2b5f] = 0xfc01,
  [0x2b60] = 0xfc02, [0x2b61] = 0xfc03, [0x2b62] = 0xfc04, [0x2b63] = 0xfc05,
  [0x2b64] = 0xfc06, [0x2b65] = 0xfc07, [0x2b66] = 0xfc08, [0x2b67] = 0xfc09,
  [0x2b68] = 0xfc0a, [0x2b69] = 0xfc0b, [0x2b6a] = 0xfc0c, [0x2b6b] = 0xfc0d,
  [0x2b6c] = 0xfc0e, [0x2b6d] = 0xfc0f, [0x2b6e] = 0xfc10, [0x2b6f] = 0xfc11,
  [0x2b70] = 0xfc12, [0x2b71] = 0xfc13, [0x2b72] = 0xfc14, [0x2b73] = 0xfc15,
  [0x2b74] = 0xfc16, [0x2b75] = 0xfc17, [0x2b76] = 0xfc18, [0x2b77] = 0xfc19,
  [0x2b78] = 0xfc1a, [0x2b79] = 0xfc1b, [0x2b7a] = 0xfc1c, [0x2b7b] = 0xfc1d,
  [0x2b7c] = 0xfc1e, [0x2b7d] = 0xfc1f, [0x2b7e] = 0xfc20, [0x2b7f] = 0xfc21,
  [0x2b80] = 0xfc22, [0x2b81] = 0xfc23, [0x2b82] = 0xfc24, [0x2b83] = 0xfc25,
  [0x2b84] = 0xfc26, [0x2b85] = 0xfc27, [0x2b86] = 0xfc28, [0x2b87] = 0xfc29,
  [0x2b88] = 0xfc2a, [0x2b89] = 0xfc2b, [0x2b8a] = 0xfc2c, [0x2b8b] = 0xfc2d,
  [0x2b8c] = 0xfc2e, [0x2b8d] = 0xfc2f, [0x2b8e] = 0xfc30, [0x2b8f] = 0xfc31,
  [0x2b90] = 0xfc32, [0x2b91] = 0xfc33, [0x2b92] = 0xfc34, [0x2b93] = 0xfc35,
  [0x2b94] = 0xfc36, [0x2b95] = 0xfc37, [0x2b96] = 0xfc38, [0x2b97] = 0xfc39,
  [0x2b98] = 0xfc3a, [0x2b99] = 0xfc3b, [0x2b9a] = 0xfc3c, [0x2b9b] = 0xfc3d,
  [0x2b9c] = 0xfc3e, [0x2b9d] = 0xfc3f, [0x2b9e] = 0xfc40, [0x2b9f] = 0xfc41,
  [0x2ba0] = 0xfc42, [0x2ba1] = 0xfc43, [0x2ba2] = 0xfc44, [0x2ba3] = 0xfc45,
  [0x2ba4] = 0xfc46, [0x2ba5] = 0xfc47, [0x2ba6] = 0xfc48, [0x2ba7] = 0xfc49,
  [0x2ba8] = 0xfc4a, [0x2ba9] = 0xfc4b, [0x2baa] = 0xfc4c, [0x2bab] = 0xfc4d,
  [0x2bac] = 0xfc4e, [0x2bad] = 0xfc4f, [0x2bae] = 0xfc50, [0x2baf] = 0xfc51,
  [0x2bb0] = 0xfc52, [0x2bb1] = 0xfc53, [0x2bb2] = 0xfc54, [0x2bb3] = 0xfc55,
  [0x2bb4] = 0xfc56, [0x2bb5] = 0xfc57, [0x2bb6] = 0xfc58, [0x2bb7] = 0xfc59,
  [0x2bb8] = 0xfc5a, [0x2bb9] = 0xfc5b, [0x2bba] = 0xfc5c, [0x2bbb] = 0xfc5d,
  [0x2bbc] = 0xfc5e, [0x2bbd] = 0xfc5f, [0x2bbe] = 0xfc60, [0x2bbf] = 0xfc61,
  [0x2bc0] = 0xfc62, [0x2bc1] = 0xfc63, [0x2bc2] = 0xfc64, [0x2bc3] = 0xfc65,
  [0x2bc4] = 0xfc66, [0x2bc5] = 0xfc67, [0x2bc6] = 0xfc68, [0x2bc7] = 0xfc69,
  [0x2bc8] = 0xfc6a, [0x2bc9] = 0xfc6b, [0x2bca] = 0xfc6c, [0x2bcb] = 0xfc6d,
  [0x2bcc] = 0xfc6e, [0x2bcd] = 0xfc6f, [0x2bce] = 0xfc70, [0x2bcf] = 0xfc71,
  [0x2bd0] = 0xfc72, [0x2bd1] = 0xfc73, [0x2bd2] = 0xfc74, [0x2bd3] = 0xfc75,
  [0x2bd4] = 0xfc76, [0x2bd5] = 0xfc77, [0x2bd6] = 0xfc78, [0x2bd7] = 0xfc79,
  [0x2bd8] = 0xfc7a, [0x2bd9] = 0xfc7b, [0x2bda] = 0xfc7c, [0x2bdb] = 0xfc7d,
  [0x2bdc] = 0xfc7e, [0x2bdd] = 0xfc7f, [0x2bde] = 0xfc80, [0x2bdf] = 0xfc81,
  [0x2be0] = 0xfc82, [0x2be1] = 0xfc83, [0x2be2] = 0xfc84, [0x2be3] = 0xfc85,
  [0x2be4] = 0xfc86, [0x2be5] = 0xfc87, [0x2be6] = 0xfc88, [0x2be7] = 0xfc89,
  [0x2be8] = 0xfc8a, [0x2be9] = 0xfc8b, [0x2bea] = 0xfc8c, [0x2beb] = 0xfc8d,
  [0x2bec] = 0xfc8e, [0x2bed] = 0xfc8f, [0x2bee] = 0xfc90, [0x2bef] = 0xfc91,
  [0x2bf0] = 0xfc92, [0x2bf1] = 0xfc93, [0x2bf2] = 0xfc94, [0x2bf3] = 0xfc95,
  [0x2bf4] = 0xfc96, [0x2bf5] = 0xfc97, [0x2bf6] = 0xfc98, [0x2bf7] = 0xfc99,
  [0x2bf8] = 0xfc9a, [0x2bf9] = 0xfc9b, [0x2bfa] = 0xfc9c, [0x2bfb] = 0xfc9d,
  [0x2bfc] = 0xfc9e, [0x2bfd] = 0xfc9f, [0x2bfe] = 0xfca0, [0x2bff] = 0xfca1,
  [0x2c00] = 0xfca2, [0x2c01] = 0xfca3, [0x2c02] = 0xfca4, [0x2c03] = 0xfca5,
  [0x2c04] = 0xfca6, [0x2c05] = 0xfca7, [0x2c06] = 0xfca8, [0x2c07] = 0xfca9,
  [0x2c08] = 0xfcaa, [0x2c09] = 0xfcab, [0x2c0a] = 0xfcac, [0x2c0b] = 0xfcad,
  [0x2c0c] = 0xfcae, [0x2c0d] = 0xfcaf, [0x2c0e] = 0xfcb0, [0x2c0f] = 0xfcb1,
  [0x2c10] = 0xfcb2, [0x2c11] = 0xfcb3, [0x2c12] = 0xfcb4, [0x2c13] = 0xfcb5,
  [0x2c14] = 0xfcb6, [0x2c15] = 0xfcb7, [0x2c16] = 0xfcb8, [0x2c17] = 0xfcb9,
  [0x2c18] = 0xfcba, [0x2c19] = 0xfcbb, [0x2c1a] = 0xfcbc, [0x2c1b] = 0xfcbd,
  [0x2c1c] = 0xfcbe, [0x2c1d] = 0xfcbf, [0x2c1e] = 0xfcc0, [0x2c1f] = 0xfcc1,
  [0x2c20] = 0xfcc2, [0x2c21] = 0xfcc3, [0x2c22] = 0xfcc4, [0x2c23] = 0xfcc5,
  [0x2c24] = 0xfcc6, [0x2c25] = 0xfcc7, [0x2c26] = 0xfcc8, [0x2c27] = 0xfcc9,
  [0x2c28] = 0xfcca, [0x2c29] = 0xfccb, [0x2c2a] = 0xfccc, [0x2c2b] = 0xfccd,
  [0x2c2c] = 0xfcce, [0x2c2d] = 0xfccf, [0x2c2e] = 0xfcd0, [0x2c2f] = 0xfcd1,
  [0x2c30] = 0xfcd2, [0x2c31] = 0xfcd3, [0x2c32] = 0xfcd4, [0x2c33] = 0xfcd5,
  [0x2c34] = 0xfcd6, [0x2c35] = 0xfcd7, [0x2c36] = 0xfcd8, [0x2c37] = 0xfcd9,
  [0x2c38] = 0xfcda, [0x2c39] = 0xfcdb, [0x2c3a] = 0xfcdc, [0x2c3b] = 0xfcdd,
  [0x2c3c] = 0xfcde, [0x2c3d] = 0xfcdf, [0x2c3e] = 0xfce0, [0x2c3f] = 0xfce1,
  [0x2c40] = 0xfce2, [0x2c41] = 0xfce3, [0x2c42] = 0xfce4, [0x2c43] = 0xfce5,
  [0x2c44] = 0xfce6, [0x2c45] = 0xfce7, [0x2c46] = 0xfce8, [0x2c47] = 0xfce9,
  [0x2c48] = 0xfcea, [0x2c49] = 0xfceb, [0x2c4a] = 0xfcec, [0x2c4b] = 0xfced,
  [0x2c4c] = 0xfcee, [0x2c4d] = 0xfcef, [0x2c4e] = 0xfcf0, [0x2c4f] = 0xfcf1,
  [0x2c50] = 0xfcf2, [0x2c51] = 0xfcf3, [0x2c52] = 0xfcf4, [0x2c53] = 0xfcf5,
  [0x2c54] = 0xfcf6, [0x2c55] = 0xfcf7, [0x2c56] = 0xfcf8, [0x2c57] = 0xfcf9,
  [0x2c58] = 0xfcfa, [0x2c59] = 0xfcfb, [0x2c5a] = 0xfcfc, [0x2c5b] = 0xfcfd,
  [0x2c5c] = 0xfcfe, [0x2c5d] = 0xfcff, [0x2c5e] = 0xfd00, [0x2c5f] = 0xfd01,
  [0x2c60] = 0xfd02, [0x2c61] = 0xfd03, [0x2c62] = 0xfd04, [0x2c63] = 0xfd05,
  [0x2c64] = 0xfd06, [0x2c65] = 0xfd07, [0x2c66] = 0xfd08, [0x2c67] = 0xfd09,
  [0x2c68] = 0xfd0a, [0x2c69] = 0xfd0b, [0x2c6a] = 0xfd0c, [0x2c6b] = 0xfd0d,
  [0x2c6c] = 0xfd0e, [0x2c6d] = 0xfd0f, [0x2c6e] = 0xfd10, [0x2c6f] = 0xfd11,
  [0x2c70] = 0xfd12, [0x2c71] = 0xfd13, [0x2c72] = 0xfd14, [0x2c73] = 0xfd15,
  [0x2c74] = 0xfd16, [0x2c75] = 0xfd17, [0x2c76] = 0xfd18, [0x2c77] = 0xfd19,
  [0x2c78] = 0xfd1a, [0x2c79] = 0xfd1b, [0x2c7a] = 0xfd1c, [0x2c7b] = 0xfd1d,
  [0x2c7c] = 0xfd1e, [0x2c7d] = 0xfd1f, [0x2c7e] = 0xfd20, [0x2c7f] = 0xfd21,
  [0x2c80] = 0xfd22, [0x2c81] = 0xfd23, [0x2c82] = 0xfd24, [0x2c83] = 0xfd25,
  [0x2c84] = 0xfd26, [0x2c85] = 0xfd27, [0x2c86] = 0xfd28, [0x2c87] = 0xfd29,
  [0x2c88] = 0xfd2a, [0x2c89] = 0xfd2b, [0x2c8a] = 0xfd2c, [0x2c8b] = 0xfd2d,
  [0x2c8c] = 0xfd2e, [0x2c8d] = 0xfd2f, [0x2c8e] = 0xfd30, [0x2c8f] = 0xfd31,
  [0x2c90] = 0xfd32, [0x2c91] = 0xfd33, [0x2c92] = 0xfd34, [0x2c93] = 0xfd35,
  [0x2c94] = 0xfd36, [0x2c95] = 0xfd37, [0x2c96] = 0xfd38, [0x2c97] = 0xfd39,
  [0x2c98] = 0xfd3a, [0x2c99] = 0xfd3b, [0x2c9a] = 0xfd3c, [0x2c9b] = 0xfd3d,
  [0x2c9c] = 0xfd3e, [0x2c9d] = 0xfd3f, [0x2c9e] = 0xfd40, [0x2c9f] = 0xfd41,
  [0x2ca0] = 0xfd42, [0x2ca1] = 0xfd43, [0x2ca2] = 0xfd44, [0x2ca3] = 0xfd45,
  [0x2ca4] = 0xfd46, [0x2ca5] = 0xfd47, [0x2ca6] = 0xfd48, [0x2ca7] = 0xfd49,
  [0x2ca8] = 0xfd4a, [0x2ca9] = 0xfd4b, [0x2caa] = 0xfd4c, [0x2cab] = 0xfd4d,
  [0x2cac] = 0xfd4e, [0x2cad] = 0xfd4f, [0x2cae] = 0xfd50, [0x2caf] = 0xfd51,
  [0x2cb0] = 0xfd52, [0x2cb1] = 0xfd53, [0x2cb2] = 0xfd54, [0x2cb3] = 0xfd55,
  [0x2cb4] = 0xfd56, [0x2cb5] = 0xfd57, [0x2cb6] = 0xfd58, [0x2cb7] = 0xfd59,
  [0x2cb8] = 0xfd5a, [0x2cb9] = 0xfd5b, [0x2cba] = 0xfd5c, [0x2cbb] = 0xfd5d,
  [0x2cbc] = 0xfd5e, [0x2cbd] = 0xfd5f, [0x2cbe] = 0xfd60, [0x2cbf] = 0xfd61,
  [0x2cc0] = 0xfd62, [0x2cc1] = 0xfd63, [0x2cc2] = 0xfd64, [0x2cc3] = 0xfd65,
  [0x2cc4] = 0xfd66, [0x2cc5] = 0xfd67, [0x2cc6] = 0xfd68, [0x2cc7] = 0xfd69,
  [0x2cc8] = 0xfd6a, [0x2cc9] = 0xfd6b, [0x2cca] = 0xfd6c, [0x2ccb] = 0xfd6d,
  [0x2ccc] = 0xfd6e, [0x2ccd] = 0xfd6f, [0x2cce] = 0xfd70, [0x2ccf] = 0xfd71,
  [0x2cd0] = 0xfd72, [0x2cd1] = 0xfd73, [0x2cd2] = 0xfd74, [0x2cd3] = 0xfd75,
  [0x2cd4] = 0xfd76, [0x2cd5] = 0xfd77, [0x2cd6] = 0xfd78, [0x2cd7] = 0xfd79,
  [0x2cd8] = 0xfd7a, [0x2cd9] = 0xfd7b, [0x2cda] = 0xfd7c, [0x2cdb] = 0xfd7d,
  [0x2cdc] = 0xfd7e, [0x2cdd] = 0xfd7f, [0x2cde] = 0xfd80, [0x2cdf] = 0xfd81,
  [0x2ce0] = 0xfd82, [0x2ce1] = 0xfd83, [0x2ce2] = 0xfd84, [0x2ce3] = 0xfd85,
  [0x2ce4] = 0xfd86, [0x2ce5] = 0xfd87, [0x2ce6] = 0xfd88, [0x2ce7] = 0xfd89,
  [0x2ce8] = 0xfd8a, [0x2ce9] = 0xfd8b, [0x2cea] = 0xfd8c, [0x2ceb] = 0xfd8d,
  [0x2cec] = 0xfd8e, [0x2ced] = 0xfd8f, [0x2cee] = 0xfd90, [0x2cef] = 0xfd91,
  [0x2cf0] = 0xfd92, [0x2cf1] = 0xfd93, [0x2cf2] = 0xfd94, [0x2cf3] = 0xfd95,
  [0x2cf4] = 0xfd96, [0x2cf5] = 0xfd97, [0x2cf6] = 0xfd98, [0x2cf7] = 0xfd99,
  [0x2cf8] = 0xfd9a, [0x2cf9] = 0xfd9b, [0x2cfa] = 0xfd9c, [0x2cfb] = 0xfd9d,
  [0x2cfc] = 0xfd9e, [0x2cfd] = 0xfd9f, [0x2cfe] = 0xfda0, [0x2cff] = 0xfda1,
  [0x2d00] = 0xfda2, [0x2d01] = 0xfda3, [0x2d02] = 0xfda4, [0x2d03] = 0xfda5,
  [0x2d04] = 0xfda6, [0x2d05] = 0xfda7, [0x2d06] = 0xfda8, [0x2d07] = 0xfda9,
  [0x2d08] = 0xfdaa, [0x2d09] = 0xfdab, [0x2d0a] = 0xfdac, [0x2d0b] = 0xfdad,
  [0x2d0c] = 0xfdae, [0x2d0d] = 0xfdaf, [0x2d0e] = 0xfdb0, [0x2d0f] = 0xfdb1,
  [0x2d10] = 0xfdb2, [0x2d11] = 0xfdb3, [0x2d12] = 0xfdb4, [0x2d13] = 0xfdb5,
  [0x2d14] = 0xfdb6, [0x2d15] = 0xfdb7, [0x2d16] = 0xfdb8, [0x2d17] = 0xfdb9,
  [0x2d18] = 0xfdba, [0x2d19] = 0xfdbb, [0x2d1a] = 0xfdbc, [0x2d1b] = 0xfdbd,
  [0x2d1c] = 0xfdbe, [0x2d1d] = 0xfdbf, [0x2d1e] = 0xfdc0, [0x2d1f] = 0xfdc1,
  [0x2d20] = 0xfdc2, [0x2d21] = 0xfdc3, [0x2d22] = 0xfdc4, [0x2d23] = 0xfdc5,
  [0x2d24] = 0xfdc6, [0x2d25] = 0xfdc7, [0x2d26] = 0xfdc8, [0x2d27] = 0xfdc9,
  [0x2d28] = 0xfdca, [0x2d29] = 0xfdcb, [0x2d2a] = 0xfdcc, [0x2d2b] = 0xfdcd,
  [0x2d2c] = 0xfdce, [0x2d2d] = 0xfdcf, [0x2d2e] = 0xfdd0, [0x2d2f] = 0xfdd1,
  [0x2d30] = 0xfdd2, [0x2d31] = 0xfdd3, [0x2d32] = 0xfdd4, [0x2d33] = 0xfdd5,
  [0x2d34] = 0xfdd6, [0x2d35] = 0xfdd7, [0x2d36] = 0xfdd8, [0x2d37] = 0xfdd9,
  [0x2d38] = 0xfdda, [0x2d39] = 0xfddb, [0x2d3a] = 0xfddc, [0x2d3b] = 0xfddd,
  [0x2d3c] = 0xfdde, [0x2d3d] = 0xfddf, [0x2d3e] = 0xfde0, [0x2d3f] = 0xfde1,
  [0x2d40] = 0xfde2, [0x2d41] = 0xfde3, [0x2d42] = 0xfde4, [0x2d43] = 0xfde5,
  [0x2d44] = 0xfde6, [0x2d45] = 0xfde7, [0x2d46] = 0xfde8, [0x2d47] = 0xfde9,
  [0x2d48] = 0xfdea, [0x2d49] = 0xfdeb, [0x2d4a] = 0xfdec, [0x2d4b] = 0xfded,
  [0x2d4c] = 0xfdee, [0x2d4d] = 0xfdef, [0x2d4e] = 0xfdf0, [0x2d4f] = 0xfdf1,
  [0x2d50] = 0xfdf2, [0x2d51] = 0xfdf3, [0x2d52] = 0xfdf4, [0x2d53] = 0xfdf5,
  [0x2d54] = 0xfdf6, [0x2d55] = 0xfdf7, [0x2d56] = 0xfdf8, [0x2d57] = 0xfdf9,
  [0x2d58] = 0xfdfa, [0x2d59] = 0xfdfb, [0x2d5a] = 0xfdfc, [0x2d5b] = 0xfdfd,
  [0x2d5c] = 0xfdfe, [0x2d5d] = 0xfdff, [0x2d5e] = 0xfe00, [0x2d5f] = 0xfe01,
  [0x2d60] = 0xfe02, [0x2d61] = 0xfe03, [0x2d62] = 0xfe04, [0x2d63] = 0xfe05,
  [0x2d64] = 0xfe06, [0x2d65] = 0xfe07, [0x2d66] = 0xfe08, [0x2d67] = 0xfe09,
  [0x2d68] = 0xfe0a, [0x2d69] = 0xfe0b, [0x2d6a] = 0xfe0c, [0x2d6b] = 0xfe0d,
  [0x2d6c] = 0xfe0e, [0x2d6d] = 0xfe0f, [0x2d78] = 0xfe1a, [0x2d79] = 0xfe1b,
  [0x2d7a] = 0xfe1c, [0x2d7b] = 0xfe1d, [0x2d7c] = 0xfe1e, [0x2d7d] = 0xfe1f,
  [0x2d7e] = 0xfe20, [0x2d7f] = 0xfe21, [0x2d80] = 0xfe22, [0x2d81] = 0xfe23,
  [0x2d82] = 0xfe24, [0x2d83] = 0xfe25, [0x2d84] = 0xfe26, [0x2d85] = 0xfe27,
  [0x2d86] = 0xfe28, [0x2d87] = 0xfe29, [0x2d88] = 0xfe2a, [0x2d89] = 0xfe2b,
  [0x2d8a] = 0xfe2c, [0x2d8b] = 0xfe2d, [0x2d8c] = 0xfe2e, [0x2d8d] = 0xfe2f,
  [0x2d8e] = 0xfe32, [0x2d8f] = 0xfe45, [0x2d90] = 0xfe46, [0x2d91] = 0xfe47,
  [0x2d92] = 0xfe48, [0x2d93] = 0xfe53, [0x2d94] = 0xfe58, [0x2d95] = 0xfe67,
  [0x2d96] = 0xfe6c, [0x2d97] = 0xfe6d, [0x2d98] = 0xfe6e, [0x2d99] = 0xfe6f,
  [0x2d9a] = 0xfe70, [0x2d9b] = 0xfe71, [0x2d9c] = 0xfe72, [0x2d9d] = 0xfe73,
  [0x2d9e] = 0xfe74, [0x2d9f] = 0xfe75, [0x2da0] = 0xfe76, [0x2da1] = 0xfe77,
  [0x2da2] = 0xfe78, [0x2da3] = 0xfe79, [0x2da4] = 0xfe7a, [0x2da5] = 0xfe7b,
  [0x2da6] = 0xfe7c, [0x2da7] = 0xfe7d, [0x2da8] = 0xfe7e, [0x2da9] = 0xfe7f,
  [0x2daa] = 0xfe80, [0x2dab] = 0xfe81, [0x2dac] = 0xfe82, [0x2dad] = 0xfe83,
  [0x2dae] = 0xfe84, [0x2daf] = 0xfe85, [0x2db0] = 0xfe86, [0x2db1] = 0xfe87,
  [0x2db2] = 0xfe88, [0x2db3] = 0xfe89, [0x2db4] = 0xfe8a, [0x2db5] = 0xfe8b,
  [0x2db6] = 0xfe8c, [0x2db7] = 0xfe8d, [0x2db8] = 0xfe8e, [0x2db9] = 0xfe8f,
  [0x2dba] = 0xfe90, [0x2dbb] = 0xfe91, [0x2dbc] = 0xfe92, [0x2dbd] = 0xfe93,
  [0x2dbe] = 0xfe94, [0x2dbf] = 0xfe95, [0x2dc0] = 0xfe96, [0x2dc1] = 0xfe97,
  [0x2dc2] = 0xfe98, [0x2dc3] = 0xfe99, [0x2dc4] = 0xfe9a, [0x2dc5] = 0xfe9b,
  [0x2dc6] = 0xfe9c, [0x2dc7] = 0xfe9d, [0x2dc8] = 0xfe9e, [0x2dc9] = 0xfe9f,
  [0x2dca] = 0xfea0, [0x2dcb] = 0xfea1, [0x2dcc] = 0xfea2, [0x2dcd] = 0xfea3,
  [0x2dce] = 0xfea4, [0x2dcf] = 0xfea5, [0x2dd0] = 0xfea6, [0x2dd1] = 0xfea7,
  [0x2dd2] = 0xfea8, [0x2dd3] = 0xfea9, [0x2dd4] = 0xfeaa, [0x2dd5] = 0xfeab,
  [0x2dd6] = 0xfeac, [0x2dd7] = 0xfead, [0x2dd8] = 0xfeae, [0x2dd9] = 0xfeaf,
  [0x2dda] = 0xfeb0, [0x2ddb] = 0xfeb1, [0x2ddc] = 0xfeb2, [0x2ddd] = 0xfeb3,
  [0x2dde] = 0xfeb4, [0x2ddf] = 0xfeb5, [0x2de0] = 0xfeb6, [0x2de1] = 0xfeb7,
  [0x2de2] = 0xfeb8, [0x2de3] = 0xfeb9, [0x2de4] = 0xfeba, [0x2de5] = 0xfebb,
  [0x2de6] = 0xfebc, [0x2de7] = 0xfebd, [0x2de8] = 0xfebe, [0x2de9] = 0xfebf,
  [0x2dea] = 0xfec0, [0x2deb] = 0xfec1, [0x2dec] = 0xfec2, [0x2ded] = 0xfec3,
  [0x2dee] = 0xfec4, [0x2def] = 0xfec5, [0x2df0] = 0xfec6, [0x2df1] = 0xfec7,
  [0x2df2] = 0xfec8, [0x2df3] = 0xfec9, [0x2df4] = 0xfeca, [0x2df5] = 0xfecb,
  [0x2df6] = 0xfecc, [0x2df7] = 0xfecd, [0x2df8] = 0xfece, [0x2df9] = 0xfecf,
  [0x2dfa] = 0xfed0, [0x2dfb] = 0xfed1, [0x2dfc] = 0xfed2, [0x2dfd] = 0xfed3,
  [0x2dfe] = 0xfed4, [0x2dff] = 0xfed5, [0x2e00] = 0xfed6, [0x2e01] = 0xfed7,
  [0x2e02] = 0xfed8, [0x2e03] = 0xfed9, [0x2e04] = 0xfeda, [0x2e05] = 0xfedb,
  [0x2e06] = 0xfedc, [0x2e07] = 0xfedd, [0x2e08] = 0xfede, [0x2e09] = 0xfedf,
  [0x2e0a] = 0xfee0, [0x2e0b] = 0xfee1, [0x2e0c] = 0xfee2, [0x2e0d] = 0xfee3,
  [0x2e0e] = 0xfee4, [0x2e0f] = 0xfee5, [0x2e10] = 0xfee6, [0x2e11] = 0xfee7,
  [0x2e12] = 0xfee8, [0x2e13] = 0xfee9, [0x2e14] = 0xfeea, [0x2e15] = 0xfeeb,
  [0x2e16] = 0xfeec, [0x2e17] = 0xfeed, [0x2e18] = 0xfeee, [0x2e19] = 0xfeef,
  [0x2e1a] = 0xfef0, [0x2e1b] = 0xfef1, [0x2e1c] = 0xfef2, [0x2e1d] = 0xfef3,
  [0x2e1e] = 0xfef4, [0x2e1f] = 0xfef5, [0x2e20] = 0xfef6, [0x2e21] = 0xfef7,
  [0x2e22] = 0xfef8, [0x2e23] = 0xfef9, [0x2e24] = 0xfefa, [0x2e25] = 0xfefb,
  [0x2e26] = 0xfefc, [0x2e27] = 0xfefd, [0x2e28] = 0xfefe, [0x2e29] = 0xfeff,
  [0x2e2a] = 0xff00, [0x2e2b] = 0xff5f, [0x2e2c] = 0xff60, [0x2e2d] = 0xff61,
  [0x2e2e] = 0xff62, [0x2e2f] = 0xff63, [0x2e30] = 0xff64, [0x2e31] = 0xff65,
  [0x2e32] = 0xff66, [0x2e33] = 0xff67, [0x2e34] = 0xff68, [0x2e35] = 0xff69,
  [0x2e36] = 0xff6a, [0x2e37] = 0xff6b, [0x2e38] = 0xff6c, [0x2e39] = 0xff6d,
  [0x2e3a] = 0xff6e, [0x2e3b] = 0xff6f, [0x2e3c] = 0xff70, [0x2e3d] = 0xff71,
  [0x2e3e] = 0xff72, [0x2e3f] = 0xff73, [0x2e40] = 0xff74, [0x2e41] = 0xff75,
  [0x2e42] = 0xff76, [0x2e43] = 0xff77, [0x2e44] = 0xff78, [0x2e45] = 0xff79,
  [0x2e46] = 0xff7a, [0x2e47] = 0xff7b, [0x2e48] = 0xff7c, [0x2e49] = 0xff7d,
  [0x2e4a] = 0xff7e, [0x2e4b] = 0xff7f, [0x2e4c] = 0xff80, [0x2e4d] = 0xff81,
  [0x2e4e] = 0xff82, [0x2e4f] = 0xff83, [0x2e50] = 0xff84, [0x2e51] = 0xff85,
  [0x2e52] = 0xff86, [0x2e53] = 0xff87, [0x2e54] = 0xff88, [0x2e55] = 0xff89,
  [0x2e56] = 0xff8a, [0x2e57] = 0xff8b, [0x2e58] = 0xff8c, [0x2e59] = 0xff8d,
  [0x2e5a] = 0xff8e, [0x2e5b] = 0xff8f, [0x2e5c] = 0xff90, [0x2e5d] = 0xff91,
  [0x2e5e] = 0xff92, [0x2e5f] = 0xff93, [0x2e60] = 0xff94, [0x2e61] = 0xff95,
  [0x2e62] = 0xff96, [0x2e63] = 0xff97, [0x2e64] = 0xff98, [0x2e65] = 0xff99,
  [0x2e66] = 0xff9a, [0x2e67] = 0xff9b, [0x2e68] = 0xff9c, [0x2e69] = 0xff9d,
  [0x2e6a] = 0xff9e, [0x2e6b] = 0xff9f, [0x2e6c] = 0xffa0, [0x2e6d] = 0xffa1,
  [0x2e6e] = 0xffa2, [0x2e6f] = 0xffa3, [0x2e70] = 0xffa4, [0x2e71] = 0xffa5,
  [0x2e72] = 0xffa6, [0x2e73] = 0xffa7, [0x2e74] = 0xffa8, [0x2e75] = 0xffa9,
  [0x2e76] = 0xffaa, [0x2e77] = 0xffab, [0x2e78] = 0xffac, [0x2e79] = 0xffad,
  [0x2e7a] = 0xffae, [0x2e7b] = 0xffaf, [0x2e7c] = 0xffb0, [0x2e7d] = 0xffb1,
  [0x2e7e] = 0xffb2, [0x2e7f] = 0xffb3, [0x2e80] = 0xffb4, [0x2e81] = 0xffb5,
  [0x2e82] = 0xffb6, [0x2e83] = 0xffb7, [0x2e84] = 0xffb8, [0x2e85] = 0xffb9,
  [0x2e86] = 0xffba, [0x2e87] = 0xffbb, [0x2e88] = 0xffbc, [0x2e89] = 0xffbd,
  [0x2e8a] = 0xffbe, [0x2e8b] = 0xffbf, [0x2e8c] = 0xffc0, [0x2e8d] = 0xffc1,
  [0x2e8e] = 0xffc2, [0x2e8f] = 0xffc3, [0x2e90] = 0xffc4, [0x2e91] = 0xffc5,
  [0x2e92] = 0xffc6, [0x2e93] = 0xffc7, [0x2e94] = 0xffc8, [0x2e95] = 0xffc9,
  [0x2e96] = 0xffca, [0x2e97] = 0xffcb, [0x2e98] = 0xffcc, [0x2e99] = 0xffcd,
  [0x2e9a] = 0xffce, [0x2e9b] = 0xffcf, [0x2e9c] = 0xffd0, [0x2e9d] = 0xffd1,
  [0x2e9e] = 0xffd2, [0x2e9f] = 0xffd3, [0x2ea0] = 0xffd4, [0x2ea1] = 0xffd5,
  [0x2ea2] = 0xffd6, [0x2ea3] = 0xffd7, [0x2ea4] = 0xffd8, [0x2ea5] = 0xffd9,
  [0x2ea6] = 0xffda, [0x2ea7] = 0xffdb, [0x2ea8] = 0xffdc, [0x2ea9] = 0xffdd,
  [0x2eaa] = 0xffde, [0x2eab] = 0xffdf,
};

/* Table for UCS-4 -> GB18030, for the range U+0080..U+9FBB.
   There are three kinds of two-byte entries (ch1, ch2):
     (ch1, ch2) = (0, 0) means an unmapped character,
     0x20 <= ch1 <= 0x80 means a four-byte character,
     ch1 >= 0x81 means a two-byte character.
   Generated through

   sort -k 2 < GB18030.table | clisp tab.lisp

   where tab.lisp is:
   ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
   (locally (declare (compile))
     (let ((count 0)
	   (range-start #x0080) (range-end #x9FBB))
       (flet ((output-pair (ucs entry)
		(when (= (mod count 3) 0)
		  (format t " "))
		(format t " ~([0x~4,'0X] = \"\\x~2,'0X\\x~2,'0X\"~),"
			; There are two ranges. The gap length is:
			;   (- #xd7a4 #xac00) = 11172.
			(cond ((<= range-start ucs range-end)
			       (- ucs range-start)))
			(ash entry -8) (ldb (byte 8 0) entry))
		(incf count)
		(when (= (mod count 3) 0)
		  (format t "~%"))))
	 (loop
	   (let ((line (read-line nil nil nil)))
	     (unless line (return))
	     (when (eql (position #\Tab line) 6)
	       (let* ((gbk (parse-integer line :start 2 :end 6 :radix 16))
		      (ucs (parse-integer line :start 9 :radix 16)))
		 (when (<= range-start ucs range-end)
		   (assert (>= gbk #x8100))
		   (output-pair ucs gbk))))
	     (when (eql (position #\Tab line) 10)
	       (let* ((gbk (parse-integer line :start 2 :end 10 :radix 16))
		      (ucs (parse-integer line :start 13 :radix 16))
		      (ch1 (ldb (byte 8 24) gbk))
		      (ch2 (ldb (byte 8 16) gbk))
		      (ch3 (ldb (byte 8 8) gbk))
		      (ch4 (ldb (byte 8 0) gbk))
		      (linear (+ (* (+ (* (+ (* (- ch1 #x81) 10)
					     (- ch2 #x30))
					  126)
				       (- ch3 #x81))
				    10)
				 (- ch4 #x30))))
		 (when (<= range-start ucs range-end)
		   (output-pair ucs (+ #x2000 linear))))))))
       (format t "~%")))
   ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
*/

static const unsigned char __ucs_to_gb18030_tab1[][2] =
{
  [0x0000] = "\x20\x00", [0x0001] = "\x20\x01", [0x0002] = "\x20\x02",
  [0x0003] = "\x20\x03", [0x0004] = "\x20\x04", [0x0005] = "\x20\x05",
  [0x0006] = "\x20\x06", [0x0007] = "\x20\x07", [0x0008] = "\x20\x08",
  [0x0009] = "\x20\x09", [0x000a] = "\x20\x0a", [0x000b] = "\x20\x0b",
  [0x000c] = "\x20\x0c", [0x000d] = "\x20\x0d", [0x000e] = "\x20\x0e",
  [0x000f] = "\x20\x0f", [0x0010] = "\x20\x10", [0x0011] = "\x20\x11",
  [0x0012] = "\x20\x12", [0x0013] = "\x20\x13", [0x0014] = "\x20\x14",
  [0x0015] = "\x20\x15", [0x0016] = "\x20\x16", [0x0017] = "\x20\x17",
  [0x0018] = "\x20\x18", [0x0019] = "\x20\x19", [0x001a] = "\x20\x1a",
  [0x001b] = "\x20\x1b", [0x001c] = "\x20\x1c", [0x001d] = "\x20\x1d",
  [0x001e] = "\x20\x1e", [0x001f] = "\x20\x1f", [0x0020] = "\x20\x20",
  [0x0021] = "\x20\x21", [0x0022] = "\x20\x22", [0x0023] = "\x20\x23",
  [0x0024] = "\xa1\xe8", [0x0025] = "\x20\x24", [0x0026] = "\x20\x25",
  [0x0027] = "\xa1\xec", [0x0028] = "\xa1\xa7", [0x0029] = "\x20\x26",
  [0x002a] = "\x20\x27", [0x002b] = "\x20\x28", [0x002c] = "\x20\x29",
  [0x002d] = "\x20\x2a", [0x002e] = "\x20\x2b", [0x002f] = "\x20\x2c",
  [0x0030] = "\xa1\xe3", [0x0031] = "\xa1\xc0", [0x0032] = "\x20\x2d",
  [0x0033] = "\x20\x2e", [0x0034] = "\x20\x2f", [0x0035] = "\x20\x30",
  [0x0036] = "\x20\x31", [0x0037] = "\xa1\xa4", [0x0038] = "\x20\x32",
  [0x0039] = "\x20\x33", [0x003a] = "\x20\x34", [0x003b] = "\x20\x35",
  [0x003c] = "\x20\x36", [0x003d] = "\x20\x37", [0x003e] = "\x20\x38",
  [0x003f] = "\x20\x39", [0x0040] = "\x20\x3a", [0x0041] = "\x20\x3b",
  [0x0042] = "\x20\x3c", [0x0043] = "\x20\x3d", [0x0044] = "\x20\x3e",
  [0x0045] = "\x20\x3f", [0x0046] = "\x20\x40", [0x0047] = "\x20\x41",
  [0x0048] = "\x20\x42", [0x0049] = "\x20\x43", [0x004a] = "\x20\x44",
  [0x004b] = "\x20\x45", [0x004c] = "\x20\x46", [0x004d] = "\x20\x47",
  [0x004e] = "\x20\x48", [0x004f] = "\x20\x49", [0x0050] = "\x20\x4a",
  [0x0051] = "\x20\x4b", [0x0052] = "\x20\x4c", [0x0053] = "\x20\x4d",
  [0x0054] = "\x20\x4e", [0x0055] = "\x20\x4f", [0x0056] = "\x20\x50",
  [0x0057] = "\xa1\xc1", [0x0058] = "\x20\x51", [0x0059] = "\x20\x52",
  [0x005a] = "\x20\x53", [0x005b] = "\x20\x54", [0x005c] = "\x20\x55",
  [0x005d] = "\x20\x56", [0x005e] = "\x20\x57", [0x005f] = "\x20\x58",
  [0x0060] = "\xa8\xa4", [0x0061] = "\xa8\xa2", [0x0062] = "\x20\x59",
  [0x0063] = "\x20\x5a", [0x0064] = "\x20\x5b", [0x0065] = "\x20\x5c",
  [0x0066] = "\x20\x5d", [0x0067] = "\x20\x5e", [0x0068] = "\xa8\xa8",
  [0x0069] = "\xa8\xa6", [0x006a] = "\xa8\xba", [0x006b] = "\x20\x5f",
  [0x006c] = "\xa8\xac", [0x006d] = "\xa8\xaa", [0x006e] = "\x20\x60",
  [0x006f] = "\x20\x61", [0x0070] = "\x20\x62", [0x0071] = "\x20\x63",
  [0x0072] = "\xa8\xb0", [0x0073] = "\xa8\xae", [0x0074] = "\x20\x64",
  [0x0075] = "\x20\x65", [0x0076] = "\x20\x66", [0x0077] = "\xa1\xc2",
  [0x0078] = "\x20\x67", [0x0079] = "\xa8\xb4", [0x007a] = "\xa8\xb2",
  [0x007b] = "\x20\x68", [0x007c] = "\xa8\xb9", [0x007d] = "\x20\x69",
  [0x007e] = "\x20\x6a", [0x007f] = "\x20\x6b", [0x0080] = "\x20\x6c",
  [0x0081] = "\xa8\xa1", [0x0082] = "\x20\x6d", [0x0083] = "\x20\x6e",
  [0x0084] = "\x20\x6f", [0x0085] = "\x20\x70", [0x0086] = "\x20\x71",
  [0x0087] = "\x20\x72", [0x0088] = "\x20\x73", [0x0089] = "\x20\x74",
  [0x008a] = "\x20\x75", [0x008b] = "\x20\x76", [0x008c] = "\x20\x77",
  [0x008d] = "\x20\x78", [0x008e] = "\x20\x79", [0x008f] = "\x20\x7a",
  [0x0090] = "\x20\x7b", [0x0091] = "\x20\x7c", [0x0092] = "\x20\x7d",
  [0x0093] = "\xa8\xa5", [0x0094] = "\x20\x7e", [0x0095] = "\x20\x7f",
  [0x0096] = "\x20\x80", [0x0097] = "\x20\x81", [0x0098] = "\x20\x82",
  [0x0099] = "\x20\x83", [0x009a] = "\x20\x84", [0x009b] = "\xa8\xa7",
  [0x009c] = "\x20\x85", [0x009d] = "\x20\x86", [0x009e] = "\x20\x87",
  [0x009f] = "\x20\x88", [0x00a0] = "\x20\x89", [0x00a1] = "\x20\x8a",
  [0x00a2] = "\x20\x8b", [0x00a3] = "\x20\x8c", [0x00a4] = "\x20\x8d",
  [0x00a5] = "\x20\x8e", [0x00a6] = "\x20\x8f", [0x00a7] = "\x20\x90",
  [0x00a8] = "\x20\x91", [0x00a9] = "\x20\x92", [0x00aa] = "\x20\x93",
  [0x00ab] = "\xa8\xa9", [0x00ac] = "\x20\x94", [0x00ad] = "\x20\x95",
  [0x00ae] = "\x20\x96", [0x00af] = "\x20\x97", [0x00b0] = "\x20\x98",
  [0x00b1] = "\x20\x99", [0x00b2] = "\x20\x9a", [0x00b3] = "\x20\x9b",
  [0x00b4] = "\x20\x9c", [0x00b5] = "\x20\x9d", [0x00b6] = "\x20\x9e",
  [0x00b7] = "\x20\x9f", [0x00b8] = "\x20\xa0", [0x00b9] = "\x20\xa1",
  [0x00ba] = "\x20\xa2", [0x00bb] = "\x20\xa3", [0x00bc] = "\x20\xa4",
  [0x00bd] = "\x20\xa5", [0x00be] = "\x20\xa6", [0x00bf] = "\x20\xa7",
  [0x00c0] = "\x20\xa8", [0x00c1] = "\x20\xa9", [0x00c2] = "\x20\xaa",
  [0x00c3] = "\x20\xab", [0x00c4] = "\xa8\xbd", [0x00c5] = "\x20\xac",
  [0x00c6] = "\x20\xad", [0x00c7] = "\x20\xae", [0x00c8] = "\xa8\xbe",
  [0x00c9] = "\x20\xaf", [0x00ca] = "\x20\xb0", [0x00cb] = "\x20\xb1",
  [0x00cc] = "\x20\xb2", [0x00cd] = "\xa8\xad", [0x00ce] = "\x20\xb3",
  [0x00cf] = "\x20\xb4", [0x00d0] = "\x20\xb5", [0x00d1] = "\x20\xb6",
  [0x00d2] = "\x20\xb7", [0x00d3] = "\x20\xb8", [0x00d4] = "\x20\xb9",
  [0x00d5] = "\x20\xba", [0x00d6] = "\x20\xbb", [0x00d7] = "\x20\xbc",
  [0x00d8] = "\x20\xbd", [0x00d9] = "\x20\xbe", [0x00da] = "\x20\xbf",
  [0x00db] = "\x20\xc0", [0x00dc] = "\x20\xc1", [0x00dd] = "\x20\xc2",
  [0x00de] = "\x20\xc3", [0x00df] = "\x20\xc4", [0x00e0] = "\x20\xc5",
  [0x00e1] = "\x20\xc6", [0x00e2] = "\x20\xc7", [0x00e3] = "\x20\xc8",
  [0x00e4] = "\x20\xc9", [0x00e5] = "\x20\xca", [0x00e6] = "\x20\xcb",
  [0x00e7] = "\x20\xcc", [0x00e8] = "\x20\xcd", [0x00e9] = "\x20\xce",
  [0x00ea] = "\x20\xcf", [0x00eb] = "\xa8\xb1", [0x00ec] = "\x20\xd0",
  [0x00ed] = "\x20\xd1", [0x00ee] = "\x20\xd2", [0x00ef] = "\x20\xd3",
  [0x00f0] = "\x20\xd4", [0x00f1] = "\x20\xd5", [0x00f2] = "\x20\xd6",
  [0x00f3] = "\x20\xd7", [0x00f4] = "\x20\xd8", [0x00f5] = "\x20\xd9",
  [0x00f6] = "\x20\xda", [0x00f7] = "\x20\xdb", [0x00f8] = "\x20\xdc",
  [0x00f9] = "\x20\xdd", [0x00fa] = "\x20\xde", [0x00fb] = "\x20\xdf",
  [0x00fc] = "\x20\xe0", [0x00fd] = "\x20\xe1", [0x00fe] = "\x20\xe2",
  [0x00ff] = "\x20\xe3", [0x0100] = "\x20\xe4", [0x0101] = "\x20\xe5",
  [0x0102] = "\x20\xe6", [0x0103] = "\x20\xe7", [0x0104] = "\x20\xe8",
  [0x0105] = "\x20\xe9", [0x0106] = "\x20\xea", [0x0107] = "\x20\xeb",
  [0x0108] = "\x20\xec", [0x0109] = "\x20\xed", [0x010a] = "\x20\xee",
  [0x010b] = "\x20\xef", [0x010c] = "\x20\xf0", [0x010d] = "\x20\xf1",
  [0x010e] = "\x20\xf2", [0x010f] = "\x20\xf3", [0x0110] = "\x20\xf4",
  [0x0111] = "\x20\xf5", [0x0112] = "\x20\xf6", [0x0113] = "\x20\xf7",
  [0x0114] = "\x20\xf8", [0x0115] = "\x20\xf9", [0x0116] = "\x20\xfa",
  [0x0117] = "\x20\xfb", [0x0118] = "\x20\xfc", [0x0119] = "\x20\xfd",
  [0x011a] = "\x20\xfe", [0x011b] = "\x20\xff", [0x011c] = "\x21\x00",
  [0x011d] = "\x21\x01", [0x011e] = "\x21\x02", [0x011f] = "\x21\x03",
  [0x0120] = "\x21\x04", [0x0121] = "\x21\x05", [0x0122] = "\x21\x06",
  [0x0123] = "\x21\x07", [0x0124] = "\x21\x08", [0x0125] = "\x21\x09",
  [0x0126] = "\x21\x0a", [0x0127] = "\x21\x0b", [0x0128] = "\x21\x0c",
  [0x0129] = "\x21\x0d", [0x012a] = "\x21\x0e", [0x012b] = "\x21\x0f",
  [0x012c] = "\x21\x10", [0x012d] = "\x21\x11", [0x012e] = "\x21\x12",
  [0x012f] = "\x21\x13", [0x0130] = "\x21\x14", [0x0131] = "\x21\x15",
  [0x0132] = "\x21\x16", [0x0133] = "\x21\x17", [0x0134] = "\x21\x18",
  [0x0135] = "\x21\x19", [0x0136] = "\x21\x1a", [0x0137] = "\x21\x1b",
  [0x0138] = "\x21\x1c", [0x0139] = "\x21\x1d", [0x013a] = "\x21\x1e",
  [0x013b] = "\x21\x1f", [0x013c] = "\x21\x20", [0x013d] = "\x21\x21",
  [0x013e] = "\x21\x22", [0x013f] = "\x21\x23", [0x0140] = "\x21\x24",
  [0x0141] = "\x21\x25", [0x0142] = "\x21\x26", [0x0143] = "\x21\x27",
  [0x0144] = "\x21\x28", [0x0145] = "\x21\x29", [0x0146] = "\x21\x2a",
  [0x0147] = "\x21\x2b", [0x0148] = "\x21\x2c", [0x0149] = "\x21\x2d",
  [0x014a] = "\x21\x2e", [0x014b] = "\x21\x2f", [0x014c] = "\x21\x30",
  [0x014d] = "\x21\x31", [0x014e] = "\xa8\xa3", [0x014f] = "\x21\x32",
  [0x0150] = "\xa8\xab", [0x0151] = "\x21\x33", [0x0152] = "\xa8\xaf",
  [0x0153] = "\x21\x34", [0x0154] = "\xa8\xb3", [0x0155] = "\x21\x35",
  [0x0156] = "\xa8\xb5", [0x0157] = "\x21\x36", [0x0158] = "\xa8\xb6",
  [0x0159] = "\x21\x37", [0x015a] = "\xa8\xb7", [0x015b] = "\x21\x38",
  [0x015c] = "\xa8\xb8", [0x015d] = "\x21\x39", [0x015e] = "\x21\x3a",
  [0x015f] = "\x21\x3b", [0x0160] = "\x21\x3c", [0x0161] = "\x21\x3d",
  [0x0162] = "\x21\x3e", [0x0163] = "\x21\x3f", [0x0164] = "\x21\x40",
  [0x0165] = "\x21\x41", [0x0166] = "\x21\x42", [0x0167] = "\x21\x43",
  [0x0168] = "\x21\x44", [0x0169] = "\x21\x45", [0x016a] = "\x21\x46",
  [0x016b] = "\x21\x47", [0x016c] = "\x21\x48", [0x016d] = "\x21\x49",
  [0x016e] = "\x21\x4a", [0x016f] = "\x21\x4b", [0x0170] = "\x21\x4c",
  [0x0171] = "\x21\x4d", [0x0172] = "\x21\x4e", [0x0173] = "\x21\x4f",
  [0x0174] = "\x21\x50", [0x0175] = "\x21\x51", [0x0176] = "\x21\x52",
  [0x0177] = "\x21\x53", [0x0178] = "\x21\x54", [0x0179] = "\xa8\xbf",
  [0x017a] = "\x21\x55", [0x017b] = "\x21\x56", [0x017c] = "\x21\x57",
  [0x017d] = "\x21\x58", [0x017e] = "\x21\x59", [0x017f] = "\x21\x5a",
  [0x0180] = "\x21\x5b", [0x0181] = "\x21\x5c", [0x0182] = "\x21\x5d",
  [0x0183] = "\x21\x5e", [0x0184] = "\x21\x5f", [0x0185] = "\x21\x60",
  [0x0186] = "\x21\x61", [0x0187] = "\x21\x62", [0x0188] = "\x21\x63",
  [0x0189] = "\x21\x64", [0x018a] = "\x21\x65", [0x018b] = "\x21\x66",
  [0x018c] = "\x21\x67", [0x018d] = "\x21\x68", [0x018e] = "\x21\x69",
  [0x018f] = "\x21\x6a", [0x0190] = "\x21\x6b", [0x0191] = "\x21\x6c",
  [0x0192] = "\x21\x6d", [0x0193] = "\x21\x6e", [0x0194] = "\x21\x6f",
  [0x0195] = "\x21\x70", [0x0196] = "\x21\x71", [0x0197] = "\x21\x72",
  [0x0198] = "\x21\x73", [0x0199] = "\x21\x74", [0x019a] = "\x21\x75",
  [0x019b] = "\x21\x76", [0x019c] = "\x21\x77", [0x019d] = "\x21\x78",
  [0x019e] = "\x21\x79", [0x019f] = "\x21\x7a", [0x01a0] = "\x21\x7b",
  [0x01a1] = "\x21\x7c", [0x01a2] = "\x21\x7d", [0x01a3] = "\x21\x7e",
  [0x01a4] = "\x21\x7f", [0x01a5] = "\x21\x80", [0x01a6] = "\x21\x81",
  [0x01a7] = "\x21\x82", [0x01a8] = "\x21\x83", [0x01a9] = "\x21\x84",
  [0x01aa] = "\x21\x85", [0x01ab] = "\x21\x86", [0x01ac] = "\x21\x87",
  [0x01ad] = "\x21\x88", [0x01ae] = "\x21\x89", [0x01af] = "\x21\x8a",
  [0x01b0] = "\x21\x8b", [0x01b1] = "\x21\x8c", [0x01b2] = "\x21\x8d",
  [0x01b3] = "\x21\x8e", [0x01b4] = "\x21\x8f", [0x01b5] = "\x21\x90",
  [0x01b6] = "\x21\x91", [0x01b7] = "\x21\x92", [0x01b8] = "\x21\x93",
  [0x01b9] = "\x21\x94", [0x01ba] = "\x21\x95", [0x01bb] = "\x21\x96",
  [0x01bc] = "\x21\x97", [0x01bd] = "\x21\x98", [0x01be] = "\x21\x99",
  [0x01bf] = "\x21\x9a", [0x01c0] = "\x21\x9b", [0x01c1] = "\x21\x9c",
  [0x01c2] = "\x21\x9d", [0x01c3] = "\x21\x9e", [0x01c4] = "\x21\x9f",
  [0x01c5] = "\x21\xa0", [0x01c6] = "\x21\xa1", [0x01c7] = "\x21\xa2",
  [0x01c8] = "\x21\xa3", [0x01c9] = "\x21\xa4", [0x01ca] = "\x21\xa5",
  [0x01cb] = "\x21\xa6", [0x01cc] = "\x21\xa7", [0x01cd] = "\x21\xa8",
  [0x01ce] = "\x21\xa9", [0x01cf] = "\x21\xaa", [0x01d0] = "\x21\xab",
  [0x01d1] = "\xa8\xbb", [0x01d2] = "\x21\xac", [0x01d3] = "\x21\xad",
  [0x01d4] = "\x21\xae", [0x01d5] = "\x21\xaf", [0x01d6] = "\x21\xb0",
  [0x01d7] = "\x21\xb1", [0x01d8] = "\x21\xb2", [0x01d9] = "\x21\xb3",
  [0x01da] = "\x21\xb4", [0x01db] = "\x21\xb5", [0x01dc] = "\x21\xb6",
  [0x01dd] = "\x21\xb7", [0x01de] = "\x21\xb8", [0x01df] = "\x21\xb9",
  [0x01e0] = "\x21\xba", [0x01e1] = "\xa8\xc0", [0x01e2] = "\x21\xbb",
  [0x01e3] = "\x21\xbc", [0x01e4] = "\x21\xbd", [0x01e5] = "\x21\xbe",
  [0x01e6] = "\x21\xbf", [0x01e7] = "\x21\xc0", [0x01e8] = "\x21\xc1",
  [0x01e9] = "\x21\xc2", [0x01ea] = "\x21\xc3", [0x01eb] = "\x21\xc4",
  [0x01ec] = "\x21\xc5", [0x01ed] = "\x21\xc6", [0x01ee] = "\x21\xc7",
  [0x01ef] = "\x21\xc8", [0x01f0] = "\x21\xc9", [0x01f1] = "\x21\xca",
  [0x01f2] = "\x21\xcb", [0x01f3] = "\x21\xcc", [0x01f4] = "\x21\xcd",
  [0x01f5] = "\x21\xce", [0x01f6] = "\x21\xcf", [0x01f7] = "\x21\xd0",
  [0x01f8] = "\x21\xd1", [0x01f9] = "\x21\xd2", [0x01fa] = "\x21\xd3",
  [0x01fb] = "\x21\xd4", [0x01fc] = "\x21\xd5", [0x01fd] = "\x21\xd6",
  [0x01fe] = "\x21\xd7", [0x01ff] = "\x21\xd8", [0x0200] = "\x21\xd9",
  [0x0201] = "\x21\xda", [0x0202] = "\x21\xdb", [0x0203] = "\x21\xdc",
  [0x0204] = "\x21\xdd", [0x0205] = "\x21\xde", [0x0206] = "\x21\xdf",
  [0x0207] = "\x21\xe0", [0x0208] = "\x21\xe1", [0x0209] = "\x21\xe2",
  [0x020a] = "\x21\xe3", [0x020b] = "\x21\xe4", [0x020c] = "\x21\xe5",
  [0x020d] = "\x21\xe6", [0x020e] = "\x21\xe7", [0x020f] = "\x21\xe8",
  [0x0210] = "\x21\xe9", [0x0211] = "\x21\xea", [0x0212] = "\x21\xeb",
  [0x0213] = "\x21\xec", [0x0214] = "\x21\xed", [0x0215] = "\x21\xee",
  [0x0216] = "\x21\xef", [0x0217] = "\x21\xf0", [0x0218] = "\x21\xf1",
  [0x0219] = "\x21\xf2", [0x021a] = "\x21\xf3", [0x021b] = "\x21\xf4",
  [0x021c] = "\x21\xf5", [0x021d] = "\x21\xf6", [0x021e] = "\x21\xf7",
  [0x021f] = "\x21\xf8", [0x0220] = "\x21\xf9", [0x0221] = "\x21\xfa",
  [0x0222] = "\x21\xfb", [0x0223] = "\x21\xfc", [0x0224] = "\x21\xfd",
  [0x0225] = "\x21\xfe", [0x0226] = "\x21\xff", [0x0227] = "\x22\x00",
  [0x0228] = "\x22\x01", [0x0229] = "\x22\x02", [0x022a] = "\x22\x03",
  [0x022b] = "\x22\x04", [0x022c] = "\x22\x05", [0x022d] = "\x22\x06",
  [0x022e] = "\x22\x07", [0x022f] = "\x22\x08", [0x0230] = "\x22\x09",
  [0x0231] = "\x22\x0a", [0x0232] = "\x22\x0b", [0x0233] = "\x22\x0c",
  [0x0234] = "\x22\x0d", [0x0235] = "\x22\x0e", [0x0236] = "\x22\x0f",
  [0x0237] = "\x22\x10", [0x0238] = "\x22\x11", [0x0239] = "\x22\x12",
  [0x023a] = "\x22\x13", [0x023b] = "\x22\x14", [0x023c] = "\x22\x15",
  [0x023d] = "\x22\x16", [0x023e] = "\x22\x17", [0x023f] = "\x22\x18",
  [0x0240] = "\x22\x19", [0x0241] = "\x22\x1a", [0x0242] = "\x22\x1b",
  [0x0243] = "\x22\x1c", [0x0244] = "\x22\x1d", [0x0245] = "\x22\x1e",
  [0x0246] = "\x22\x1f", [0x0247] = "\xa1\xa6", [0x0248] = "\x22\x20",
  [0x0249] = "\xa1\xa5", [0x024a] = "\xa8\x40", [0x024b] = "\xa8\x41",
  [0x024c] = "\x22\x21", [0x024d] = "\x22\x22", [0x024e] = "\x22\x23",
  [0x024f] = "\x22\x24", [0x0250] = "\x22\x25", [0x0251] = "\x22\x26",
  [0x0252] = "\x22\x27", [0x0253] = "\x22\x28", [0x0254] = "\x22\x29",
  [0x0255] = "\x22\x2a", [0x0256] = "\x22\x2b", [0x0257] = "\x22\x2c",
  [0x0258] = "\x22\x2d", [0x0259] = "\xa8\x42", [0x025a] = "\x22\x2e",
  [0x025b] = "\x22\x2f", [0x025c] = "\x22\x30", [0x025d] = "\x22\x31",
  [0x025e] = "\x22\x32", [0x025f] = "\x22\x33", [0x0260] = "\x22\x34",
  [0x0261] = "\x22\x35", [0x0262] = "\x22\x36", [0x0263] = "\x22\x37",
  [0x0264] = "\x22\x38", [0x0265] = "\x22\x39", [0x0266] = "\x22\x3a",
  [0x0267] = "\x22\x3b", [0x0268] = "\x22\x3c", [0x0269] = "\x22\x3d",
  [0x026a] = "\x22\x3e", [0x026b] = "\x22\x3f", [0x026c] = "\x22\x40",
  [0x026d] = "\x22\x41", [0x026e] = "\x22\x42", [0x026f] = "\x22\x43",
  [0x0270] = "\x22\x44", [0x0271] = "\x22\x45", [0x0272] = "\x22\x46",
  [0x0273] = "\x22\x47", [0x0274] = "\x22\x48", [0x0275] = "\x22\x49",
  [0x0276] = "\x22\x4a", [0x0277] = "\x22\x4b", [0x0278] = "\x22\x4c",
  [0x0279] = "\x22\x4d", [0x027a] = "\x22\x4e", [0x027b] = "\x22\x4f",
  [0x027c] = "\x22\x50", [0x027d] = "\x22\x51", [0x027e] = "\x22\x52",
  [0x027f] = "\x22\x53", [0x0280] = "\x22\x54", [0x0281] = "\x22\x55",
  [0x0282] = "\x22\x56", [0x0283] = "\x22\x57", [0x0284] = "\x22\x58",
  [0x0285] = "\x22\x59", [0x0286] = "\x22\x5a", [0x0287] = "\x22\x5b",
  [0x0288] = "\x22\x5c", [0x0289] = "\x22\x5d", [0x028a] = "\x22\x5e",
  [0x028b] = "\x22\x5f", [0x028c] = "\x22\x60", [0x028d] = "\x22\x61",
  [0x028e] = "\x22\x62", [0x028f] = "\x22\x63", [0x0290] = "\x22\x64",
  [0x0291] = "\x22\x65", [0x0292] = "\x22\x66", [0x0293] = "\x22\x67",
  [0x0294] = "\x22\x68", [0x0295] = "\x22\x69", [0x0296] = "\x22\x6a",
  [0x0297] = "\x22\x6b", [0x0298] = "\x22\x6c", [0x0299] = "\x22\x6d",
  [0x029a] = "\x22\x6e", [0x029b] = "\x22\x6f", [0x029c] = "\x22\x70",
  [0x029d] = "\x22\x71", [0x029e] = "\x22\x72", [0x029f] = "\x22\x73",
  [0x02a0] = "\x22\x74", [0x02a1] = "\x22\x75", [0x02a2] = "\x22\x76",
  [0x02a3] = "\x22\x77", [0x02a4] = "\x22\x78", [0x02a5] = "\x22\x79",
  [0x02a6] = "\x22\x7a", [0x02a7] = "\x22\x7b", [0x02a8] = "\x22\x7c",
  [0x02a9] = "\x22\x7d", [0x02aa] = "\x22\x7e", [0x02ab] = "\x22\x7f",
  [0x02ac] = "\x22\x80", [0x02ad] = "\x22\x81", [0x02ae] = "\x22\x82",
  [0x02af] = "\x22\x83", [0x02b0] = "\x22\x84", [0x02b1] = "\x22\x85",
  [0x02b2] = "\x22\x86", [0x02b3] = "\x22\x87", [0x02b4] = "\x22\x88",
  [0x02b5] = "\x22\x89", [0x02b6] = "\x22\x8a", [0x02b7] = "\x22\x8b",
  [0x02b8] = "\x22\x8c", [0x02b9] = "\x22\x8d", [0x02ba] = "\x22\x8e",
  [0x02bb] = "\x22\x8f", [0x02bc] = "\x22\x90", [0x02bd] = "\x22\x91",
  [0x02be] = "\x22\x92", [0x02bf] = "\x22\x93", [0x02c0] = "\x22\x94",
  [0x02c1] = "\x22\x95", [0x02c2] = "\x22\x96", [0x02c3] = "\x22\x97",
  [0x02c4] = "\x22\x98", [0x02c5] = "\x22\x99", [0x02c6] = "\x22\x9a",
  [0x02c7] = "\x22\x9b", [0x02c8] = "\x22\x9c", [0x02c9] = "\x22\x9d",
  [0x02ca] = "\x22\x9e", [0x02cb] = "\x22\x9f", [0x02cc] = "\x22\xa0",
  [0x02cd] = "\x22\xa1", [0x02ce] = "\x22\xa2", [0x02cf] = "\x22\xa3",
  [0x02d0] = "\x22\xa4", [0x02d1] = "\x22\xa5", [0x02d2] = "\x22\xa6",
  [0x02d3] = "\x22\xa7", [0x02d4] = "\x22\xa8", [0x02d5] = "\x22\xa9",
  [0x02d6] = "\x22\xaa", [0x02d7] = "\x22\xab", [0x02d8] = "\x22\xac",
  [0x02d9] = "\x22\xad", [0x02da] = "\x22\xae", [0x02db] = "\x22\xaf",
  [0x02dc] = "\x22\xb0", [0x02dd] = "\x22\xb1", [0x02de] = "\x22\xb2",
  [0x02df] = "\x22\xb3", [0x02e0] = "\x22\xb4", [0x02e1] = "\x22\xb5",
  [0x02e2] = "\x22\xb6", [0x02e3] = "\x22\xb7", [0x02e4] = "\x22\xb8",
  [0x02e5] = "\x22\xb9", [0x02e6] = "\x22\xba", [0x02e7] = "\x22\xbb",
  [0x02e8] = "\x22\xbc", [0x02e9] = "\x22\xbd", [0x02ea] = "\x22\xbe",
  [0x02eb] = "\x22\xbf", [0x02ec] = "\x22\xc0", [0x02ed] = "\x22\xc1",
  [0x02ee] = "\x22\xc2", [0x02ef] = "\x22\xc3", [0x02f0] = "\x22\xc4",
  [0x02f1] = "\x22\xc5", [0x02f2] = "\x22\xc6", [0x02f3] = "\x22\xc7",
  [0x02f4] = "\x22\xc8", [0x02f5] = "\x22\xc9", [0x02f6] = "\x22\xca",
  [0x02f7] = "\x22\xcb", [0x02f8] = "\x22\xcc", [0x02f9] = "\x22\xcd",
  [0x02fa] = "\x22\xce", [0x02fb] = "\x22\xcf", [0x02fc] = "\x22\xd0",
  [0x02fd] = "\x22\xd1", [0x02fe] = "\x22\xd2", [0x02ff] = "\x22\xd3",
  [0x0300] = "\x22\xd4", [0x0301] = "\x22\xd5", [0x0302] = "\x22\xd6",
  [0x0303] = "\x22\xd7", [0x0304] = "\x22\xd8", [0x0305] = "\x22\xd9",
  [0x0306] = "\x22\xda", [0x0307] = "\x22\xdb", [0x0308] = "\x22\xdc",
  [0x0309] = "\x22\xdd", [0x030a] = "\x22\xde", [0x030b] = "\x22\xdf",
  [0x030c] = "\x22\xe0", [0x030d] = "\x22\xe1", [0x030e] = "\x22\xe2",
  [0x030f] = "\x22\xe3", [0x0310] = "\x22\xe4", [0x0311] = "\xa6\xa1",
  [0x0312] = "\xa6\xa2", [0x0313] = "\xa6\xa3", [0x0314] = "\xa6\xa4",
  [0x0315] = "\xa6\xa5", [0x0316] = "\xa6\xa6", [0x0317] = "\xa6\xa7",
  [0x0318] = "\xa6\xa8", [0x0319] = "\xa6\xa9", [0x031a] = "\xa6\xaa",
  [0x031b] = "\xa6\xab", [0x031c] = "\xa6\xac", [0x031d] = "\xa6\xad",
  [0x031e] = "\xa6\xae", [0x031f] = "\xa6\xaf", [0x0320] = "\xa6\xb0",
  [0x0321] = "\xa6\xb1", [0x0322] = "\x22\xe5", [0x0323] = "\xa6\xb2",
  [0x0324] = "\xa6\xb3", [0x0325] = "\xa6\xb4", [0x0326] = "\xa6\xb5",
  [0x0327] = "\xa6\xb6", [0x0328] = "\xa6\xb7", [0x0329] = "\xa6\xb8",
  [0x032a] = "\x22\xe6", [0x032b] = "\x22\xe7", [0x032c] = "\x22\xe8",
  [0x032d] = "\x22\xe9", [0x032e] = "\x22\xea", [0x032f] = "\x22\xeb",
  [0x0330] = "\x22\xec", [0x0331] = "\xa6\xc1", [0x0332] = "\xa6\xc2",
  [0x0333] = "\xa6\xc3", [0x0334] = "\xa6\xc4", [0x0335] = "\xa6\xc5",
  [0x0336] = "\xa6\xc6", [0x0337] = "\xa6\xc7", [0x0338] = "\xa6\xc8",
  [0x0339] = "\xa6\xc9", [0x033a] = "\xa6\xca", [0x033b] = "\xa6\xcb",
  [0x033c] = "\xa6\xcc", [0x033d] = "\xa6\xcd", [0x033e] = "\xa6\xce",
  [0x033f] = "\xa6\xcf", [0x0340] = "\xa6\xd0", [0x0341] = "\xa6\xd1",
  [0x0342] = "\x22\xed", [0x0343] = "\xa6\xd2", [0x0344] = "\xa6\xd3",
  [0x0345] = "\xa6\xd4", [0x0346] = "\xa6\xd5", [0x0347] = "\xa6\xd6",
  [0x0348] = "\xa6\xd7", [0x0349] = "\xa6\xd8", [0x034a] = "\x22\xee",
  [0x034b] = "\x22\xef", [0x034c] = "\x22\xf0", [0x034d] = "\x22\xf1",
  [0x034e] = "\x22\xf2", [0x034f] = "\x22\xf3", [0x0350] = "\x22\xf4",
  [0x0351] = "\x22\xf5", [0x0352] = "\x22\xf6", [0x0353] = "\x22\xf7",
  [0x0354] = "\x22\xf8", [0x0355] = "\x22\xf9", [0x0356] = "\x22\xfa",
  [0x0357] = "\x22\xfb", [0x0358] = "\x22\xfc", [0x0359] = "\x22\xfd",
  [0x035a] = "\x22\xfe", [0x035b] = "\x22\xff", [0x035c] = "\x23\x00",
  [0x035d] = "\x23\x01", [0x035e] = "\x23\x02", [0x035f] = "\x23\x03",
  [0x0360] = "\x23\x04", [0x0361] = "\x23\x05", [0x0362] = "\x23\x06",
  [0x0363] = "\x23\x07", [0x0364] = "\x23\x08", [0x0365] = "\x23\x09",
  [0x0366] = "\x23\x0a", [0x0367] = "\x23\x0b", [0x0368] = "\x23\x0c",
  [0x0369] = "\x23\x0d", [0x036a] = "\x23\x0e", [0x036b] = "\x23\x0f",
  [0x036c] = "\x23\x10", [0x036d] = "\x23\x11", [0x036e] = "\x23\x12",
  [0x036f] = "\x23\x13", [0x0370] = "\x23\x14", [0x0371] = "\x23\x15",
  [0x0372] = "\x23\x16", [0x0373] = "\x23\x17", [0x0374] = "\x23\x18",
  [0x0375] = "\x23\x19", [0x0376] = "\x23\x1a", [0x0377] = "\x23\x1b",
  [0x0378] = "\x23\x1c", [0x0379] = "\x23\x1d", [0x037a] = "\x23\x1e",
  [0x037b] = "\x23\x1f", [0x037c] = "\x23\x20", [0x037d] = "\x23\x21",
  [0x037e] = "\x23\x22", [0x037f] = "\x23\x23", [0x0380] = "\x23\x24",
  [0x0381] = "\xa7\xa7", [0x0382] = "\x23\x25", [0x0383] = "\x23\x26",
  [0x0384] = "\x23\x27", [0x0385] = "\x23\x28", [0x0386] = "\x23\x29",
  [0x0387] = "\x23\x2a", [0x0388] = "\x23\x2b", [0x0389] = "\x23\x2c",
  [0x038a] = "\x23\x2d", [0x038b] = "\x23\x2e", [0x038c] = "\x23\x2f",
  [0x038d] = "\x23\x30", [0x038e] = "\x23\x31", [0x038f] = "\x23\x32",
  [0x0390] = "\xa7\xa1", [0x0391] = "\xa7\xa2", [0x0392] = "\xa7\xa3",
  [0x0393] = "\xa7\xa4", [0x0394] = "\xa7\xa5", [0x0395] = "\xa7\xa6",
  [0x0396] = "\xa7\xa8", [0x0397] = "\xa7\xa9", [0x0398] = "\xa7\xaa",
  [0x0399] = "\xa7\xab", [0x039a] = "\xa7\xac", [0x039b] = "\xa7\xad",
  [0x039c] = "\xa7\xae", [0x039d] = "\xa7\xaf", [0x039e] = "\xa7\xb0",
  [0x039f] = "\xa7\xb1", [0x03a0] = "\xa7\xb2", [0x03a1] = "\xa7\xb3",
  [0x03a2] = "\xa7\xb4", [0x03a3] = "\xa7\xb5", [0x03a4] = "\xa7\xb6",
  [0x03a5] = "\xa7\xb7", [0x03a6] = "\xa7\xb8", [0x03a7] = "\xa7\xb9",
  [0x03a8] = "\xa7\xba", [0x03a9] = "\xa7\xbb", [0x03aa] = "\xa7\xbc",
  [0x03ab] = "\xa7\xbd", [0x03ac] = "\xa7\xbe", [0x03ad] = "\xa7\xbf",
  [0x03ae] = "\xa7\xc0", [0x03af] = "\xa7\xc1", [0x03b0] = "\xa7\xd1",
  [0x03b1] = "\xa7\xd2", [0x03b2] = "\xa7\xd3", [0x03b3] = "\xa7\xd4",
  [0x03b4] = "\xa7\xd5", [0x03b5] = "\xa7\xd6", [0x03b6] = "\xa7\xd8",
  [0x03b7] = "\xa7\xd9", [0x03b8] = "\xa7\xda", [0x03b9] = "\xa7\xdb",
  [0x03ba] = "\xa7\xdc", [0x03bb] = "\xa7\xdd", [0x03bc] = "\xa7\xde",
  [0x03bd] = "\xa7\xdf", [0x03be] = "\xa7\xe0", [0x03bf] = "\xa7\xe1",
  [0x03c0] = "\xa7\xe2", [0x03c1] = "\xa7\xe3", [0x03c2] = "\xa7\xe4",
  [0x03c3] = "\xa7\xe5", [0x03c4] = "\xa7\xe6", [0x03c5] = "\xa7\xe7",
  [0x03c6] = "\xa7\xe8", [0x03c7] = "\xa7\xe9", [0x03c8] = "\xa7\xea",
  [0x03c9] = "\xa7\xeb", [0x03ca] = "\xa7\xec", [0x03cb] = "\xa7\xed",
  [0x03cc] = "\xa7\xee", [0x03cd] = "\xa7\xef", [0x03ce] = "\xa7\xf0",
  [0x03cf] = "\xa7\xf1", [0x03d0] = "\x23\x33", [0x03d1] = "\xa7\xd7",
  [0x03d2] = "\x23\x34", [0x03d3] = "\x23\x35", [0x03d4] = "\x23\x36",
  [0x03d5] = "\x23\x37", [0x03d6] = "\x23\x38", [0x03d7] = "\x23\x39",
  [0x03d8] = "\x23\x3a", [0x03d9] = "\x23\x3b", [0x03da] = "\x23\x3c",
  [0x03db] = "\x23\x3d", [0x03dc] = "\x23\x3e", [0x03dd] = "\x23\x3f",
  [0x03de] = "\x23\x40", [0x03df] = "\x23\x41", [0x03e0] = "\x23\x42",
  [0x03e1] = "\x23\x43", [0x03e2] = "\x23\x44", [0x03e3] = "\x23\x45",
  [0x03e4] = "\x23\x46", [0x03e5] = "\x23\x47", [0x03e6] = "\x23\x48",
  [0x03e7] = "\x23\x49", [0x03e8] = "\x23\x4a", [0x03e9] = "\x23\x4b",
  [0x03ea] = "\x23\x4c", [0x03eb] = "\x23\x4d", [0x03ec] = "\x23\x4e",
  [0x03ed] = "\x23\x4f", [0x03ee] = "\x23\x50", [0x03ef] = "\x23\x51",
  [0x03f0] = "\x23\x52", [0x03f1] = "\x23\x53", [0x03f2] = "\x23\x54",
  [0x03f3] = "\x23\x55", [0x03f4] = "\x23\x56", [0x03f5] = "\x23\x57",
  [0x03f6] = "\x23\x58", [0x03f7] = "\x23\x59", [0x03f8] = "\x23\x5a",
  [0x03f9] = "\x23\x5b", [0x03fa] = "\x23\x5c", [0x03fb] = "\x23\x5d",
  [0x03fc] = "\x23\x5e", [0x03fd] = "\x23\x5f", [0x03fe] = "\x23\x60",
  [0x03ff] = "\x23\x61", [0x0400] = "\x23\x62", [0x0401] = "\x23\x63",
  [0x0402] = "\x23\x64", [0x0403] = "\x23\x65", [0x0404] = "\x23\x66",
  [0x0405] = "\x23\x67", [0x0406] = "\x23\x68", [0x0407] = "\x23\x69",
  [0x0408] = "\x23\x6a", [0x0409] = "\x23\x6b", [0x040a] = "\x23\x6c",
  [0x040b] = "\x23\x6d", [0x040c] = "\x23\x6e", [0x040d] = "\x23\x6f",
  [0x040e] = "\x23\x70", [0x040f] = "\x23\x71", [0x0410] = "\x23\x72",
  [0x0411] = "\x23\x73", [0x0412] = "\x23\x74", [0x0413] = "\x23\x75",
  [0x0414] = "\x23\x76", [0x0415] = "\x23\x77", [0x0416] = "\x23\x78",
  [0x0417] = "\x23\x79", [0x0418] = "\x23\x7a", [0x0419] = "\x23\x7b",
  [0x041a] = "\x23\x7c", [0x041b] = "\x23\x7d", [0x041c] = "\x23\x7e",
  [0x041d] = "\x23\x7f", [0x041e] = "\x23\x80", [0x041f] = "\x23\x81",
  [0x0420] = "\x23\x82", [0x0421] = "\x23\x83", [0x0422] = "\x23\x84",
  [0x0423] = "\x23\x85", [0x0424] = "\x23\x86", [0x0425] = "\x23\x87",
  [0x0426] = "\x23\x88", [0x0427] = "\x23\x89", [0x0428] = "\x23\x8a",
  [0x0429] = "\x23\x8b", [0x042a] = "\x23\x8c", [0x042b] = "\x23\x8d",
  [0x042c] = "\x23\x8e", [0x042d] = "\x23\x8f", [0x042e] = "\x23\x90",
  [0x042f] = "\x23\x91", [0x0430] = "\x23\x92", [0x0431] = "\x23\x93",
  [0x0432] = "\x23\x94", [0x0433] = "\x23\x95", [0x0434] = "\x23\x96",
  [0x0435] = "\x23\x97", [0x0436] = "\x23\x98", [0x0437] = "\x23\x99",
  [0x0438] = "\x23\x9a", [0x0439] = "\x23\x9b", [0x043a] = "\x23\x9c",
  [0x043b] = "\x23\x9d", [0x043c] = "\x23\x9e", [0x043d] = "\x23\x9f",
  [0x043e] = "\x23\xa0", [0x043f] = "\x23\xa1", [0x0440] = "\x23\xa2",
  [0x0441] = "\x23\xa3", [0x0442] = "\x23\xa4", [0x0443] = "\x23\xa5",
  [0x0444] = "\x23\xa6", [0x0445] = "\x23\xa7", [0x0446] = "\x23\xa8",
  [0x0447] = "\x23\xa9", [0x0448] = "\x23\xaa", [0x0449] = "\x23\xab",
  [0x044a] = "\x23\xac", [0x044b] = "\x23\xad", [0x044c] = "\x23\xae",
  [0x044d] = "\x23\xaf", [0x044e] = "\x23\xb0", [0x044f] = "\x23\xb1",
  [0x0450] = "\x23\xb2", [0x0451] = "\x23\xb3", [0x0452] = "\x23\xb4",
  [0x0453] = "\x23\xb5", [0x0454] = "\x23\xb6", [0x0455] = "\x23\xb7",
  [0x0456] = "\x23\xb8", [0x0457] = "\x23\xb9", [0x0458] = "\x23\xba",
  [0x0459] = "\x23\xbb", [0x045a] = "\x23\xbc", [0x045b] = "\x23\xbd",
  [0x045c] = "\x23\xbe", [0x045d] = "\x23\xbf", [0x045e] = "\x23\xc0",
  [0x045f] = "\x23\xc1", [0x0460] = "\x23\xc2", [0x0461] = "\x23\xc3",
  [0x0462] = "\x23\xc4", [0x0463] = "\x23\xc5", [0x0464] = "\x23\xc6",
  [0x0465] = "\x23\xc7", [0x0466] = "\x23\xc8", [0x0467] = "\x23\xc9",
  [0x0468] = "\x23\xca", [0x0469] = "\x23\xcb", [0x046a] = "\x23\xcc",
  [0x046b] = "\x23\xcd", [0x046c] = "\x23\xce", [0x046d] = "\x23\xcf",
  [0x046e] = "\x23\xd0", [0x046f] = "\x23\xd1", [0x0470] = "\x23\xd2",
  [0x0471] = "\x23\xd3", [0x0472] = "\x23\xd4", [0x0473] = "\x23\xd5",
  [0x0474] = "\x23\xd6", [0x0475] = "\x23\xd7", [0x0476] = "\x23\xd8",
  [0x0477] = "\x23\xd9", [0x0478] = "\x23\xda", [0x0479] = "\x23\xdb",
  [0x047a] = "\x23\xdc", [0x047b] = "\x23\xdd", [0x047c] = "\x23\xde",
  [0x047d] = "\x23\xdf", [0x047e] = "\x23\xe0", [0x047f] = "\x23\xe1",
  [0x0480] = "\x23\xe2", [0x0481] = "\x23\xe3", [0x0482] = "\x23\xe4",
  [0x0483] = "\x23\xe5", [0x0484] = "\x23\xe6", [0x0485] = "\x23\xe7",
  [0x0486] = "\x23\xe8", [0x0487] = "\x23\xe9", [0x0488] = "\x23\xea",
  [0x0489] = "\x23\xeb", [0x048a] = "\x23\xec", [0x048b] = "\x23\xed",
  [0x048c] = "\x23\xee", [0x048d] = "\x23\xef", [0x048e] = "\x23\xf0",
  [0x048f] = "\x23\xf1", [0x0490] = "\x23\xf2", [0x0491] = "\x23\xf3",
  [0x0492] = "\x23\xf4", [0x0493] = "\x23\xf5", [0x0494] = "\x23\xf6",
  [0x0495] = "\x23\xf7", [0x0496] = "\x23\xf8", [0x0497] = "\x23\xf9",
  [0x0498] = "\x23\xfa", [0x0499] = "\x23\xfb", [0x049a] = "\x23\xfc",
  [0x049b] = "\x23\xfd", [0x049c] = "\x23\xfe", [0x049d] = "\x23\xff",
  [0x049e] = "\x24\x00", [0x049f] = "\x24\x01", [0x04a0] = "\x24\x02",
  [0x04a1] = "\x24\x03", [0x04a2] = "\x24\x04", [0x04a3] = "\x24\x05",
  [0x04a4] = "\x24\x06", [0x04a5] = "\x24\x07", [0x04a6] = "\x24\x08",
  [0x04a7] = "\x24\x09", [0x04a8] = "\x24\x0a", [0x04a9] = "\x24\x0b",
  [0x04aa] = "\x24\x0c", [0x04ab] = "\x24\x0d", [0x04ac] = "\x24\x0e",
  [0x04ad] = "\x24\x0f", [0x04ae] = "\x24\x10", [0x04af] = "\x24\x11",
  [0x04b0] = "\x24\x12", [0x04b1] = "\x24\x13", [0x04b2] = "\x24\x14",
  [0x04b3] = "\x24\x15", [0x04b4] = "\x24\x16", [0x04b5] = "\x24\x17",
  [0x04b6] = "\x24\x18", [0x04b7] = "\x24\x19", [0x04b8] = "\x24\x1a",
  [0x04b9] = "\x24\x1b", [0x04ba] = "\x24\x1c", [0x04bb] = "\x24\x1d",
  [0x04bc] = "\x24\x1e", [0x04bd] = "\x24\x1f", [0x04be] = "\x24\x20",
  [0x04bf] = "\x24\x21", [0x04c0] = "\x24\x22", [0x04c1] = "\x24\x23",
  [0x04c2] = "\x24\x24", [0x04c3] = "\x24\x25", [0x04c4] = "\x24\x26",
  [0x04c5] = "\x24\x27", [0x04c6] = "\x24\x28", [0x04c7] = "\x24\x29",
  [0x04c8] = "\x24\x2a", [0x04c9] = "\x24\x2b", [0x04ca] = "\x24\x2c",
  [0x04cb] = "\x24\x2d", [0x04cc] = "\x24\x2e", [0x04cd] = "\x24\x2f",
  [0x04ce] = "\x24\x30", [0x04cf] = "\x24\x31", [0x04d0] = "\x24\x32",
  [0x04d1] = "\x24\x33", [0x04d2] = "\x24\x34", [0x04d3] = "\x24\x35",
  [0x04d4] = "\x24\x36", [0x04d5] = "\x24\x37", [0x04d6] = "\x24\x38",
  [0x04d7] = "\x24\x39", [0x04d8] = "\x24\x3a", [0x04d9] = "\x24\x3b",
  [0x04da] = "\x24\x3c", [0x04db] = "\x24\x3d", [0x04dc] = "\x24\x3e",
  [0x04dd] = "\x24\x3f", [0x04de] = "\x24\x40", [0x04df] = "\x24\x41",
  [0x04e0] = "\x24\x42", [0x04e1] = "\x24\x43", [0x04e2] = "\x24\x44",
  [0x04e3] = "\x24\x45", [0x04e4] = "\x24\x46", [0x04e5] = "\x24\x47",
  [0x04e6] = "\x24\x48", [0x04e7] = "\x24\x49", [0x04e8] = "\x24\x4a",
  [0x04e9] = "\x24\x4b", [0x04ea] = "\x24\x4c", [0x04eb] = "\x24\x4d",
  [0x04ec] = "\x24\x4e", [0x04ed] = "\x24\x4f", [0x04ee] = "\x24\x50",
  [0x04ef] = "\x24\x51", [0x04f0] = "\x24\x52", [0x04f1] = "\x24\x53",
  [0x04f2] = "\x24\x54", [0x04f3] = "\x24\x55", [0x04f4] = "\x24\x56",
  [0x04f5] = "\x24\x57", [0x04f6] = "\x24\x58", [0x04f7] = "\x24\x59",
  [0x04f8] = "\x24\x5a", [0x04f9] = "\x24\x5b", [0x04fa] = "\x24\x5c",
  [0x04fb] = "\x24\x5d", [0x04fc] = "\x24\x5e", [0x04fd] = "\x24\x5f",
  [0x04fe] = "\x24\x60", [0x04ff] = "\x24\x61", [0x0500] = "\x24\x62",
  [0x0501] = "\x24\x63", [0x0502] = "\x24\x64", [0x0503] = "\x24\x65",
  [0x0504] = "\x24\x66", [0x0505] = "\x24\x67", [0x0506] = "\x24\x68",
  [0x0507] = "\x24\x69", [0x0508] = "\x24\x6a", [0x0509] = "\x24\x6b",
  [0x050a] = "\x24\x6c", [0x050b] = "\x24\x6d", [0x050c] = "\x24\x6e",
  [0x050d] = "\x24\x6f", [0x050e] = "\x24\x70", [0x050f] = "\x24\x71",
  [0x0510] = "\x24\x72", [0x0511] = "\x24\x73", [0x0512] = "\x24\x74",
  [0x0513] = "\x24\x75", [0x0514] = "\x24\x76", [0x0515] = "\x24\x77",
  [0x0516] = "\x24\x78", [0x0517] = "\x24\x79", [0x0518] = "\x24\x7a",
  [0x0519] = "\x24\x7b", [0x051a] = "\x24\x7c", [0x051b] = "\x24\x7d",
  [0x051c] = "\x24\x7e", [0x051d] = "\x24\x7f", [0x051e] = "\x24\x80",
  [0x051f] = "\x24\x81", [0x0520] = "\x24\x82", [0x0521] = "\x24\x83",
  [0x0522] = "\x24\x84", [0x0523] = "\x24\x85", [0x0524] = "\x24\x86",
  [0x0525] = "\x24\x87", [0x0526] = "\x24\x88", [0x0527] = "\x24\x89",
  [0x0528] = "\x24\x8a", [0x0529] = "\x24\x8b", [0x052a] = "\x24\x8c",
  [0x052b] = "\x24\x8d", [0x052c] = "\x24\x8e", [0x052d] = "\x24\x8f",
  [0x052e] = "\x24\x90", [0x052f] = "\x24\x91", [0x0530] = "\x24\x92",
  [0x0531] = "\x24\x93", [0x0532] = "\x24\x94", [0x0533] = "\x24\x95",
  [0x0534] = "\x24\x96", [0x0535] = "\x24\x97", [0x0536] = "\x24\x98",
  [0x0537] = "\x24\x99", [0x0538] = "\x24\x9a", [0x0539] = "\x24\x9b",
  [0x053a] = "\x24\x9c", [0x053b] = "\x24\x9d", [0x053c] = "\x24\x9e",
  [0x053d] = "\x24\x9f", [0x053e] = "\x24\xa0", [0x053f] = "\x24\xa1",
  [0x0540] = "\x24\xa2", [0x0541] = "\x24\xa3", [0x0542] = "\x24\xa4",
  [0x0543] = "\x24\xa5", [0x0544] = "\x24\xa6", [0x0545] = "\x24\xa7",
  [0x0546] = "\x24\xa8", [0x0547] = "\x24\xa9", [0x0548] = "\x24\xaa",
  [0x0549] = "\x24\xab", [0x054a] = "\x24\xac", [0x054b] = "\x24\xad",
  [0x054c] = "\x24\xae", [0x054d] = "\x24\xaf", [0x054e] = "\x24\xb0",
  [0x054f] = "\x24\xb1", [0x0550] = "\x24\xb2", [0x0551] = "\x24\xb3",
  [0x0552] = "\x24\xb4", [0x0553] = "\x24\xb5", [0x0554] = "\x24\xb6",
  [0x0555] = "\x24\xb7", [0x0556] = "\x24\xb8", [0x0557] = "\x24\xb9",
  [0x0558] = "\x24\xba", [0x0559] = "\x24\xbb", [0x055a] = "\x24\xbc",
  [0x055b] = "\x24\xbd", [0x055c] = "\x24\xbe", [0x055d] = "\x24\xbf",
  [0x055e] = "\x24\xc0", [0x055f] = "\x24\xc1", [0x0560] = "\x24\xc2",
  [0x0561] = "\x24\xc3", [0x0562] = "\x24\xc4", [0x0563] = "\x24\xc5",
  [0x0564] = "\x24\xc6", [0x0565] = "\x24\xc7", [0x0566] = "\x24\xc8",
  [0x0567] = "\x24\xc9", [0x0568] = "\x24\xca", [0x0569] = "\x24\xcb",
  [0x056a] = "\x24\xcc", [0x056b] = "\x24\xcd", [0x056c] = "\x24\xce",
  [0x056d] = "\x24\xcf", [0x056e] = "\x24\xd0", [0x056f] = "\x24\xd1",
  [0x0570] = "\x24\xd2", [0x0571] = "\x24\xd3", [0x0572] = "\x24\xd4",
  [0x0573] = "\x24\xd5", [0x0574] = "\x24\xd6", [0x0575] = "\x24\xd7",
  [0x0576] = "\x24\xd8", [0x0577] = "\x24\xd9", [0x0578] = "\x24\xda",
  [0x0579] = "\x24\xdb", [0x057a] = "\x24\xdc", [0x057b] = "\x24\xdd",
  [0x057c] = "\x24\xde", [0x057d] = "\x24\xdf", [0x057e] = "\x24\xe0",
  [0x057f] = "\x24\xe1", [0x0580] = "\x24\xe2", [0x0581] = "\x24\xe3",
  [0x0582] = "\x24\xe4", [0x0583] = "\x24\xe5", [0x0584] = "\x24\xe6",
  [0x0585] = "\x24\xe7", [0x0586] = "\x24\xe8", [0x0587] = "\x24\xe9",
  [0x0588] = "\x24\xea", [0x0589] = "\x24\xeb", [0x058a] = "\x24\xec",
  [0x058b] = "\x24\xed", [0x058c] = "\x24\xee", [0x058d] = "\x24\xef",
  [0x058e] = "\x24\xf0", [0x058f] = "\x24\xf1", [0x0590] = "\x24\xf2",
  [0x0591] = "\x24\xf3", [0x0592] = "\x24\xf4", [0x0593] = "\x24\xf5",
  [0x0594] = "\x24\xf6", [0x0595] = "\x24\xf7", [0x0596] = "\x24\xf8",
  [0x0597] = "\x24\xf9", [0x0598] = "\x24\xfa", [0x0599] = "\x24\xfb",
  [0x059a] = "\x24\xfc", [0x059b] = "\x24\xfd", [0x059c] = "\x24\xfe",
  [0x059d] = "\x24\xff", [0x059e] = "\x25\x00", [0x059f] = "\x25\x01",
  [0x05a0] = "\x25\x02", [0x05a1] = "\x25\x03", [0x05a2] = "\x25\x04",
  [0x05a3] = "\x25\x05", [0x05a4] = "\x25\x06", [0x05a5] = "\x25\x07",
  [0x05a6] = "\x25\x08", [0x05a7] = "\x25\x09", [0x05a8] = "\x25\x0a",
  [0x05a9] = "\x25\x0b", [0x05aa] = "\x25\x0c", [0x05ab] = "\x25\x0d",
  [0x05ac] = "\x25\x0e", [0x05ad] = "\x25\x0f", [0x05ae] = "\x25\x10",
  [0x05af] = "\x25\x11", [0x05b0] = "\x25\x12", [0x05b1] = "\x25\x13",
  [0x05b2] = "\x25\x14", [0x05b3] = "\x25\x15", [0x05b4] = "\x25\x16",
  [0x05b5] = "\x25\x17", [0x05b6] = "\x25\x18", [0x05b7] = "\x25\x19",
  [0x05b8] = "\x25\x1a", [0x05b9] = "\x25\x1b", [0x05ba] = "\x25\x1c",
  [0x05bb] = "\x25\x1d", [0x05bc] = "\x25\x1e", [0x05bd] = "\x25\x1f",
  [0x05be] = "\x25\x20", [0x05bf] = "\x25\x21", [0x05c0] = "\x25\x22",
  [0x05c1] = "\x25\x23", [0x05c2] = "\x25\x24", [0x05c3] = "\x25\x25",
  [0x05c4] = "\x25\x26", [0x05c5] = "\x25\x27", [0x05c6] = "\x25\x28",
  [0x05c7] = "\x25\x29", [0x05c8] = "\x25\x2a", [0x05c9] = "\x25\x2b",
  [0x05ca] = "\x25\x2c", [0x05cb] = "\x25\x2d", [0x05cc] = "\x25\x2e",
  [0x05cd] = "\x25\x2f", [0x05ce] = "\x25\x30", [0x05cf] = "\x25\x31",
  [0x05d0] = "\x25\x32", [0x05d1] = "\x25\x33", [0x05d2] = "\x25\x34",
  [0x05d3] = "\x25\x35", [0x05d4] = "\x25\x36", [0x05d5] = "\x25\x37",
  [0x05d6] = "\x25\x38", [0x05d7] = "\x25\x39", [0x05d8] = "\x25\x3a",
  [0x05d9] = "\x25\x3b", [0x05da] = "\x25\x3c", [0x05db] = "\x25\x3d",
  [0x05dc] = "\x25\x3e", [0x05dd] = "\x25\x3f", [0x05de] = "\x25\x40",
  [0x05df] = "\x25\x41", [0x05e0] = "\x25\x42", [0x05e1] = "\x25\x43",
  [0x05e2] = "\x25\x44", [0x05e3] = "\x25\x45", [0x05e4] = "\x25\x46",
  [0x05e5] = "\x25\x47", [0x05e6] = "\x25\x48", [0x05e7] = "\x25\x49",
  [0x05e8] = "\x25\x4a", [0x05e9] = "\x25\x4b", [0x05ea] = "\x25\x4c",
  [0x05eb] = "\x25\x4d", [0x05ec] = "\x25\x4e", [0x05ed] = "\x25\x4f",
  [0x05ee] = "\x25\x50", [0x05ef] = "\x25\x51", [0x05f0] = "\x25\x52",
  [0x05f1] = "\x25\x53", [0x05f2] = "\x25\x54", [0x05f3] = "\x25\x55",
  [0x05f4] = "\x25\x56", [0x05f5] = "\x25\x57", [0x05f6] = "\x25\x58",
  [0x05f7] = "\x25\x59", [0x05f8] = "\x25\x5a", [0x05f9] = "\x25\x5b",
  [0x05fa] = "\x25\x5c", [0x05fb] = "\x25\x5d", [0x05fc] = "\x25\x5e",
  [0x05fd] = "\x25\x5f", [0x05fe] = "\x25\x60", [0x05ff] = "\x25\x61",
  [0x0600] = "\x25\x62", [0x0601] = "\x25\x63", [0x0602] = "\x25\x64",
  [0x0603] = "\x25\x65", [0x0604] = "\x25\x66", [0x0605] = "\x25\x67",
  [0x0606] = "\x25\x68", [0x0607] = "\x25\x69", [0x0608] = "\x25\x6a",
  [0x0609] = "\x25\x6b", [0x060a] = "\x25\x6c", [0x060b] = "\x25\x6d",
  [0x060c] = "\x25\x6e", [0x060d] = "\x25\x6f", [0x060e] = "\x25\x70",
  [0x060f] = "\x25\x71", [0x0610] = "\x25\x72", [0x0611] = "\x25\x73",
  [0x0612] = "\x25\x74", [0x0613] = "\x25\x75", [0x0614] = "\x25\x76",
  [0x0615] = "\x25\x77", [0x0616] = "\x25\x78", [0x0617] = "\x25\x79",
  [0x0618] = "\x25\x7a", [0x0619] = "\x25\x7b", [0x061a] = "\x25\x7c",
  [0x061b] = "\x25\x7d", [0x061c] = "\x25\x7e", [0x061d] = "\x25\x7f",
  [0x061e] = "\x25\x80", [0x061f] = "\x25\x81", [0x0620] = "\x25\x82",
  [0x0621] = "\x25\x83", [0x0622] = "\x25\x84", [0x0623] = "\x25\x85",
  [0x0624] = "\x25\x86", [0x0625] = "\x25\x87", [0x0626] = "\x25\x88",
  [0x0627] = "\x25\x89", [0x0628] = "\x25\x8a", [0x0629] = "\x25\x8b",
  [0x062a] = "\x25\x8c", [0x062b] = "\x25\x8d", [0x062c] = "\x25\x8e",
  [0x062d] = "\x25\x8f", [0x062e] = "\x25\x90", [0x062f] = "\x25\x91",
  [0x0630] = "\x25\x92", [0x0631] = "\x25\x93", [0x0632] = "\x25\x94",
  [0x0633] = "\x25\x95", [0x0634] = "\x25\x96", [0x0635] = "\x25\x97",
  [0x0636] = "\x25\x98", [0x0637] = "\x25\x99", [0x0638] = "\x25\x9a",
  [0x0639] = "\x25\x9b", [0x063a] = "\x25\x9c", [0x063b] = "\x25\x9d",
  [0x063c] = "\x25\x9e", [0x063d] = "\x25\x9f", [0x063e] = "\x25\xa0",
  [0x063f] = "\x25\xa1", [0x0640] = "\x25\xa2", [0x0641] = "\x25\xa3",
  [0x0642] = "\x25\xa4", [0x0643] = "\x25\xa5", [0x0644] = "\x25\xa6",
  [0x0645] = "\x25\xa7", [0x0646] = "\x25\xa8", [0x0647] = "\x25\xa9",
  [0x0648] = "\x25\xaa", [0x0649] = "\x25\xab", [0x064a] = "\x25\xac",
  [0x064b] = "\x25\xad", [0x064c] = "\x25\xae", [0x064d] = "\x25\xaf",
  [0x064e] = "\x25\xb0", [0x064f] = "\x25\xb1", [0x0650] = "\x25\xb2",
  [0x0651] = "\x25\xb3", [0x0652] = "\x25\xb4", [0x0653] = "\x25\xb5",
  [0x0654] = "\x25\xb6", [0x0655] = "\x25\xb7", [0x0656] = "\x25\xb8",
  [0x0657] = "\x25\xb9", [0x0658] = "\x25\xba", [0x0659] = "\x25\xbb",
  [0x065a] = "\x25\xbc", [0x065b] = "\x25\xbd", [0x065c] = "\x25\xbe",
  [0x065d] = "\x25\xbf", [0x065e] = "\x25\xc0", [0x065f] = "\x25\xc1",
  [0x0660] = "\x25\xc2", [0x0661] = "\x25\xc3", [0x0662] = "\x25\xc4",
  [0x0663] = "\x25\xc5", [0x0664] = "\x25\xc6", [0x0665] = "\x25\xc7",
  [0x0666] = "\x25\xc8", [0x0667] = "\x25\xc9", [0x0668] = "\x25\xca",
  [0x0669] = "\x25\xcb", [0x066a] = "\x25\xcc", [0x066b] = "\x25\xcd",
  [0x066c] = "\x25\xce", [0x066d] = "\x25\xcf", [0x066e] = "\x25\xd0",
  [0x066f] = "\x25\xd1", [0x0670] = "\x25\xd2", [0x0671] = "\x25\xd3",
  [0x0672] = "\x25\xd4", [0x0673] = "\x25\xd5", [0x0674] = "\x25\xd6",
  [0x0675] = "\x25\xd7", [0x0676] = "\x25\xd8", [0x0677] = "\x25\xd9",
  [0x0678] = "\x25\xda", [0x0679] = "\x25\xdb", [0x067a] = "\x25\xdc",
  [0x067b] = "\x25\xdd", [0x067c] = "\x25\xde", [0x067d] = "\x25\xdf",
  [0x067e] = "\x25\xe0", [0x067f] = "\x25\xe1", [0x0680] = "\x25\xe2",
  [0x0681] = "\x25\xe3", [0x0682] = "\x25\xe4", [0x0683] = "\x25\xe5",
  [0x0684] = "\x25\xe6", [0x0685] = "\x25\xe7", [0x0686] = "\x25\xe8",
  [0x0687] = "\x25\xe9", [0x0688] = "\x25\xea", [0x0689] = "\x25\xeb",
  [0x068a] = "\x25\xec", [0x068b] = "\x25\xed", [0x068c] = "\x25\xee",
  [0x068d] = "\x25\xef", [0x068e] = "\x25\xf0", [0x068f] = "\x25\xf1",
  [0x0690] = "\x25\xf2", [0x0691] = "\x25\xf3", [0x0692] = "\x25\xf4",
  [0x0693] = "\x25\xf5", [0x0694] = "\x25\xf6", [0x0695] = "\x25\xf7",
  [0x0696] = "\x25\xf8", [0x0697] = "\x25\xf9", [0x0698] = "\x25\xfa",
  [0x0699] = "\x25\xfb", [0x069a] = "\x25\xfc", [0x069b] = "\x25\xfd",
  [0x069c] = "\x25\xfe", [0x069d] = "\x25\xff", [0x069e] = "\x26\x00",
  [0x069f] = "\x26\x01", [0x06a0] = "\x26\x02", [0x06a1] = "\x26\x03",
  [0x06a2] = "\x26\x04", [0x06a3] = "\x26\x05", [0x06a4] = "\x26\x06",
  [0x06a5] = "\x26\x07", [0x06a6] = "\x26\x08", [0x06a7] = "\x26\x09",
  [0x06a8] = "\x26\x0a", [0x06a9] = "\x26\x0b", [0x06aa] = "\x26\x0c",
  [0x06ab] = "\x26\x0d", [0x06ac] = "\x26\x0e", [0x06ad] = "\x26\x0f",
  [0x06ae] = "\x26\x10", [0x06af] = "\x26\x11", [0x06b0] = "\x26\x12",
  [0x06b1] = "\x26\x13", [0x06b2] = "\x26\x14", [0x06b3] = "\x26\x15",
  [0x06b4] = "\x26\x16", [0x06b5] = "\x26\x17", [0x06b6] = "\x26\x18",
  [0x06b7] = "\x26\x19", [0x06b8] = "\x26\x1a", [0x06b9] = "\x26\x1b",
  [0x06ba] = "\x26\x1c", [0x06bb] = "\x26\x1d", [0x06bc] = "\x26\x1e",
  [0x06bd] = "\x26\x1f", [0x06be] = "\x26\x20", [0x06bf] = "\x26\x21",
  [0x06c0] = "\x26\x22", [0x06c1] = "\x26\x23", [0x06c2] = "\x26\x24",
  [0x06c3] = "\x26\x25", [0x06c4] = "\x26\x26", [0x06c5] = "\x26\x27",
  [0x06c6] = "\x26\x28", [0x06c7] = "\x26\x29", [0x06c8] = "\x26\x2a",
  [0x06c9] = "\x26\x2b", [0x06ca] = "\x26\x2c", [0x06cb] = "\x26\x2d",
  [0x06cc] = "\x26\x2e", [0x06cd] = "\x26\x2f", [0x06ce] = "\x26\x30",
  [0x06cf] = "\x26\x31", [0x06d0] = "\x26\x32", [0x06d1] = "\x26\x33",
  [0x06d2] = "\x26\x34", [0x06d3] = "\x26\x35", [0x06d4] = "\x26\x36",
  [0x06d5] = "\x26\x37", [0x06d6] = "\x26\x38", [0x06d7] = "\x26\x39",
  [0x06d8] = "\x26\x3a", [0x06d9] = "\x26\x3b", [0x06da] = "\x26\x3c",
  [0x06db] = "\x26\x3d", [0x06dc] = "\x26\x3e", [0x06dd] = "\x26\x3f",
  [0x06de] = "\x26\x40", [0x06df] = "\x26\x41", [0x06e0] = "\x26\x42",
  [0x06e1] = "\x26\x43", [0x06e2] = "\x26\x44", [0x06e3] = "\x26\x45",
  [0x06e4] = "\x26\x46", [0x06e5] = "\x26\x47", [0x06e6] = "\x26\x48",
  [0x06e7] = "\x26\x49", [0x06e8] = "\x26\x4a", [0x06e9] = "\x26\x4b",
  [0x06ea] = "\x26\x4c", [0x06eb] = "\x26\x4d", [0x06ec] = "\x26\x4e",
  [0x06ed] = "\x26\x4f", [0x06ee] = "\x26\x50", [0x06ef] = "\x26\x51",
  [0x06f0] = "\x26\x52", [0x06f1] = "\x26\x53", [0x06f2] = "\x26\x54",
  [0x06f3] = "\x26\x55", [0x06f4] = "\x26\x56", [0x06f5] = "\x26\x57",
  [0x06f6] = "\x26\x58", [0x06f7] = "\x26\x59", [0x06f8] = "\x26\x5a",
  [0x06f9] = "\x26\x5b", [0x06fa] = "\x26\x5c", [0x06fb] = "\x26\x5d",
  [0x06fc] = "\x26\x5e", [0x06fd] = "\x26\x5f", [0x06fe] = "\x26\x60",
  [0x06ff] = "\x26\x61", [0x0700] = "\x26\x62", [0x0701] = "\x26\x63",
  [0x0702] = "\x26\x64", [0x0703] = "\x26\x65", [0x0704] = "\x26\x66",
  [0x0705] = "\x26\x67", [0x0706] = "\x26\x68", [0x0707] = "\x26\x69",
  [0x0708] = "\x26\x6a", [0x0709] = "\x26\x6b", [0x070a] = "\x26\x6c",
  [0x070b] = "\x26\x6d", [0x070c] = "\x26\x6e", [0x070d] = "\x26\x6f",
  [0x070e] = "\x26\x70", [0x070f] = "\x26\x71", [0x0710] = "\x26\x72",
  [0x0711] = "\x26\x73", [0x0712] = "\x26\x74", [0x0713] = "\x26\x75",
  [0x0714] = "\x26\x76", [0x0715] = "\x26\x77", [0x0716] = "\x26\x78",
  [0x0717] = "\x26\x79", [0x0718] = "\x26\x7a", [0x0719] = "\x26\x7b",
  [0x071a] = "\x26\x7c", [0x071b] = "\x26\x7d", [0x071c] = "\x26\x7e",
  [0x071d] = "\x26\x7f", [0x071e] = "\x26\x80", [0x071f] = "\x26\x81",
  [0x0720] = "\x26\x82", [0x0721] = "\x26\x83", [0x0722] = "\x26\x84",
  [0x0723] = "\x26\x85", [0x0724] = "\x26\x86", [0x0725] = "\x26\x87",
  [0x0726] = "\x26\x88", [0x0727] = "\x26\x89", [0x0728] = "\x26\x8a",
  [0x0729] = "\x26\x8b", [0x072a] = "\x26\x8c", [0x072b] = "\x26\x8d",
  [0x072c] = "\x26\x8e", [0x072d] = "\x26\x8f", [0x072e] = "\x26\x90",
  [0x072f] = "\x26\x91", [0x0730] = "\x26\x92", [0x0731] = "\x26\x93",
  [0x0732] = "\x26\x94", [0x0733] = "\x26\x95", [0x0734] = "\x26\x96",
  [0x0735] = "\x26\x97", [0x0736] = "\x26\x98", [0x0737] = "\x26\x99",
  [0x0738] = "\x26\x9a", [0x0739] = "\x26\x9b", [0x073a] = "\x26\x9c",
  [0x073b] = "\x26\x9d", [0x073c] = "\x26\x9e", [0x073d] = "\x26\x9f",
  [0x073e] = "\x26\xa0", [0x073f] = "\x26\xa1", [0x0740] = "\x26\xa2",
  [0x0741] = "\x26\xa3", [0x0742] = "\x26\xa4", [0x0743] = "\x26\xa5",
  [0x0744] = "\x26\xa6", [0x0745] = "\x26\xa7", [0x0746] = "\x26\xa8",
  [0x0747] = "\x26\xa9", [0x0748] = "\x26\xaa", [0x0749] = "\x26\xab",
  [0x074a] = "\x26\xac", [0x074b] = "\x26\xad", [0x074c] = "\x26\xae",
  [0x074d] = "\x26\xaf", [0x074e] = "\x26\xb0", [0x074f] = "\x26\xb1",
  [0x0750] = "\x26\xb2", [0x0751] = "\x26\xb3", [0x0752] = "\x26\xb4",
  [0x0753] = "\x26\xb5", [0x0754] = "\x26\xb6", [0x0755] = "\x26\xb7",
  [0x0756] = "\x26\xb8", [0x0757] = "\x26\xb9", [0x0758] = "\x26\xba",
  [0x0759] = "\x26\xbb", [0x075a] = "\x26\xbc", [0x075b] = "\x26\xbd",
  [0x075c] = "\x26\xbe", [0x075d] = "\x26\xbf", [0x075e] = "\x26\xc0",
  [0x075f] = "\x26\xc1", [0x0760] = "\x26\xc2", [0x0761] = "\x26\xc3",
  [0x0762] = "\x26\xc4", [0x0763] = "\x26\xc5", [0x0764] = "\x26\xc6",
  [0x0765] = "\x26\xc7", [0x0766] = "\x26\xc8", [0x0767] = "\x26\xc9",
  [0x0768] = "\x26\xca", [0x0769] = "\x26\xcb", [0x076a] = "\x26\xcc",
  [0x076b] = "\x26\xcd", [0x076c] = "\x26\xce", [0x076d] = "\x26\xcf",
  [0x076e] = "\x26\xd0", [0x076f] = "\x26\xd1", [0x0770] = "\x26\xd2",
  [0x0771] = "\x26\xd3", [0x0772] = "\x26\xd4", [0x0773] = "\x26\xd5",
  [0x0774] = "\x26\xd6", [0x0775] = "\x26\xd7", [0x0776] = "\x26\xd8",
  [0x0777] = "\x26\xd9", [0x0778] = "\x26\xda", [0x0779] = "\x26\xdb",
  [0x077a] = "\x26\xdc", [0x077b] = "\x26\xdd", [0x077c] = "\x26\xde",
  [0x077d] = "\x26\xdf", [0x077e] = "\x26\xe0", [0x077f] = "\x26\xe1",
  [0x0780] = "\x26\xe2", [0x0781] = "\x26\xe3", [0x0782] = "\x26\xe4",
  [0x0783] = "\x26\xe5", [0x0784] = "\x26\xe6", [0x0785] = "\x26\xe7",
  [0x0786] = "\x26\xe8", [0x0787] = "\x26\xe9", [0x0788] = "\x26\xea",
  [0x0789] = "\x26\xeb", [0x078a] = "\x26\xec", [0x078b] = "\x26\xed",
  [0x078c] = "\x26\xee", [0x078d] = "\x26\xef", [0x078e] = "\x26\xf0",
  [0x078f] = "\x26\xf1", [0x0790] = "\x26\xf2", [0x0791] = "\x26\xf3",
  [0x0792] = "\x26\xf4", [0x0793] = "\x26\xf5", [0x0794] = "\x26\xf6",
  [0x0795] = "\x26\xf7", [0x0796] = "\x26\xf8", [0x0797] = "\x26\xf9",
  [0x0798] = "\x26\xfa", [0x0799] = "\x26\xfb", [0x079a] = "\x26\xfc",
  [0x079b] = "\x26\xfd", [0x079c] = "\x26\xfe", [0x079d] = "\x26\xff",
  [0x079e] = "\x27\x00", [0x079f] = "\x27\x01", [0x07a0] = "\x27\x02",
  [0x07a1] = "\x27\x03", [0x07a2] = "\x27\x04", [0x07a3] = "\x27\x05",
  [0x07a4] = "\x27\x06", [0x07a5] = "\x27\x07", [0x07a6] = "\x27\x08",
  [0x07a7] = "\x27\x09", [0x07a8] = "\x27\x0a", [0x07a9] = "\x27\x0b",
  [0x07aa] = "\x27\x0c", [0x07ab] = "\x27\x0d", [0x07ac] = "\x27\x0e",
  [0x07ad] = "\x27\x0f", [0x07ae] = "\x27\x10", [0x07af] = "\x27\x11",
  [0x07b0] = "\x27\x12", [0x07b1] = "\x27\x13", [0x07b2] = "\x27\x14",
  [0x07b3] = "\x27\x15", [0x07b4] = "\x27\x16", [0x07b5] = "\x27\x17",
  [0x07b6] = "\x27\x18", [0x07b7] = "\x27\x19", [0x07b8] = "\x27\x1a",
  [0x07b9] = "\x27\x1b", [0x07ba] = "\x27\x1c", [0x07bb] = "\x27\x1d",
  [0x07bc] = "\x27\x1e", [0x07bd] = "\x27\x1f", [0x07be] = "\x27\x20",
  [0x07bf] = "\x27\x21", [0x07c0] = "\x27\x22", [0x07c1] = "\x27\x23",
  [0x07c2] = "\x27\x24", [0x07c3] = "\x27\x25", [0x07c4] = "\x27\x26",
  [0x07c5] = "\x27\x27", [0x07c6] = "\x27\x28", [0x07c7] = "\x27\x29",
  [0x07c8] = "\x27\x2a", [0x07c9] = "\x27\x2b", [0x07ca] = "\x27\x2c",
  [0x07cb] = "\x27\x2d", [0x07cc] = "\x27\x2e", [0x07cd] = "\x27\x2f",
  [0x07ce] = "\x27\x30", [0x07cf] = "\x27\x31", [0x07d0] = "\x27\x32",
  [0x07d1] = "\x27\x33", [0x07d2] = "\x27\x34", [0x07d3] = "\x27\x35",
  [0x07d4] = "\x27\x36", [0x07d5] = "\x27\x37", [0x07d6] = "\x27\x38",
  [0x07d7] = "\x27\x39", [0x07d8] = "\x27\x3a", [0x07d9] = "\x27\x3b",
  [0x07da] = "\x27\x3c", [0x07db] = "\x27\x3d", [0x07dc] = "\x27\x3e",
  [0x07dd] = "\x27\x3f", [0x07de] = "\x27\x40", [0x07df] = "\x27\x41",
  [0x07e0] = "\x27\x42", [0x07e1] = "\x27\x43", [0x07e2] = "\x27\x44",
  [0x07e3] = "\x27\x45", [0x07e4] = "\x27\x46", [0x07e5] = "\x27\x47",
  [0x07e6] = "\x27\x48", [0x07e7] = "\x27\x49", [0x07e8] = "\x27\x4a",
  [0x07e9] = "\x27\x4b", [0x07ea] = "\x27\x4c", [0x07eb] = "\x27\x4d",
  [0x07ec] = "\x27\x4e", [0x07ed] = "\x27\x4f", [0x07ee] = "\x27\x50",
  [0x07ef] = "\x27\x51", [0x07f0] = "\x27\x52", [0x07f1] = "\x27\x53",
  [0x07f2] = "\x27\x54", [0x07f3] = "\x27\x55", [0x07f4] = "\x27\x56",
  [0x07f5] = "\x27\x57", [0x07f6] = "\x27\x58", [0x07f7] = "\x27\x59",
  [0x07f8] = "\x27\x5a", [0x07f9] = "\x27\x5b", [0x07fa] = "\x27\x5c",
  [0x07fb] = "\x27\x5d", [0x07fc] = "\x27\x5e", [0x07fd] = "\x27\x5f",
  [0x07fe] = "\x27\x60", [0x07ff] = "\x27\x61", [0x0800] = "\x27\x62",
  [0x0801] = "\x27\x63", [0x0802] = "\x27\x64", [0x0803] = "\x27\x65",
  [0x0804] = "\x27\x66", [0x0805] = "\x27\x67", [0x0806] = "\x27\x68",
  [0x0807] = "\x27\x69", [0x0808] = "\x27\x6a", [0x0809] = "\x27\x6b",
  [0x080a] = "\x27\x6c", [0x080b] = "\x27\x6d", [0x080c] = "\x27\x6e",
  [0x080d] = "\x27\x6f", [0x080e] = "\x27\x70", [0x080f] = "\x27\x71",
  [0x0810] = "\x27\x72", [0x0811] = "\x27\x73", [0x0812] = "\x27\x74",
  [0x0813] = "\x27\x75", [0x0814] = "\x27\x76", [0x0815] = "\x27\x77",
  [0x0816] = "\x27\x78", [0x0817] = "\x27\x79", [0x0818] = "\x27\x7a",
  [0x0819] = "\x27\x7b", [0x081a] = "\x27\x7c", [0x081b] = "\x27\x7d",
  [0x081c] = "\x27\x7e", [0x081d] = "\x27\x7f", [0x081e] = "\x27\x80",
  [0x081f] = "\x27\x81", [0x0820] = "\x27\x82", [0x0821] = "\x27\x83",
  [0x0822] = "\x27\x84", [0x0823] = "\x27\x85", [0x0824] = "\x27\x86",
  [0x0825] = "\x27\x87", [0x0826] = "\x27\x88", [0x0827] = "\x27\x89",
  [0x0828] = "\x27\x8a", [0x0829] = "\x27\x8b", [0x082a] = "\x27\x8c",
  [0x082b] = "\x27\x8d", [0x082c] = "\x27\x8e", [0x082d] = "\x27\x8f",
  [0x082e] = "\x27\x90", [0x082f] = "\x27\x91", [0x0830] = "\x27\x92",
  [0x0831] = "\x27\x93", [0x0832] = "\x27\x94", [0x0833] = "\x27\x95",
  [0x0834] = "\x27\x96", [0x0835] = "\x27\x97", [0x0836] = "\x27\x98",
  [0x0837] = "\x27\x99", [0x0838] = "\x27\x9a", [0x0839] = "\x27\x9b",
  [0x083a] = "\x27\x9c", [0x083b] = "\x27\x9d", [0x083c] = "\x27\x9e",
  [0x083d] = "\x27\x9f", [0x083e] = "\x27\xa0", [0x083f] = "\x27\xa1",
  [0x0840] = "\x27\xa2", [0x0841] = "\x27\xa3", [0x0842] = "\x27\xa4",
  [0x0843] = "\x27\xa5", [0x0844] = "\x27\xa6", [0x0845] = "\x27\xa7",
  [0x0846] = "\x27\xa8", [0x0847] = "\x27\xa9", [0x0848] = "\x27\xaa",
  [0x0849] = "\x27\xab", [0x084a] = "\x27\xac", [0x084b] = "\x27\xad",
  [0x084c] = "\x27\xae", [0x084d] = "\x27\xaf", [0x084e] = "\x27\xb0",
  [0x084f] = "\x27\xb1", [0x0850] = "\x27\xb2", [0x0851] = "\x27\xb3",
  [0x0852] = "\x27\xb4", [0x0853] = "\x27\xb5", [0x0854] = "\x27\xb6",
  [0x0855] = "\x27\xb7", [0x0856] = "\x27\xb8", [0x0857] = "\x27\xb9",
  [0x0858] = "\x27\xba", [0x0859] = "\x27\xbb", [0x085a] = "\x27\xbc",
  [0x085b] = "\x27\xbd", [0x085c] = "\x27\xbe", [0x085d] = "\x27\xbf",
  [0x085e] = "\x27\xc0", [0x085f] = "\x27\xc1", [0x0860] = "\x27\xc2",
  [0x0861] = "\x27\xc3", [0x0862] = "\x27\xc4", [0x0863] = "\x27\xc5",
  [0x0864] = "\x27\xc6", [0x0865] = "\x27\xc7", [0x0866] = "\x27\xc8",
  [0x0867] = "\x27\xc9", [0x0868] = "\x27\xca", [0x0869] = "\x27\xcb",
  [0x086a] = "\x27\xcc", [0x086b] = "\x27\xcd", [0x086c] = "\x27\xce",
  [0x086d] = "\x27\xcf", [0x086e] = "\x27\xd0", [0x086f] = "\x27\xd1",
  [0x0870] = "\x27\xd2", [0x0871] = "\x27\xd3", [0x0872] = "\x27\xd4",
  [0x0873] = "\x27\xd5", [0x0874] = "\x27\xd6", [0x0875] = "\x27\xd7",
  [0x0876] = "\x27\xd8", [0x0877] = "\x27\xd9", [0x0878] = "\x27\xda",
  [0x0879] = "\x27\xdb", [0x087a] = "\x27\xdc", [0x087b] = "\x27\xdd",
  [0x087c] = "\x27\xde", [0x087d] = "\x27\xdf", [0x087e] = "\x27\xe0",
  [0x087f] = "\x27\xe1", [0x0880] = "\x27\xe2", [0x0881] = "\x27\xe3",
  [0x0882] = "\x27\xe4", [0x0883] = "\x27\xe5", [0x0884] = "\x27\xe6",
  [0x0885] = "\x27\xe7", [0x0886] = "\x27\xe8", [0x0887] = "\x27\xe9",
  [0x0888] = "\x27\xea", [0x0889] = "\x27\xeb", [0x088a] = "\x27\xec",
  [0x088b] = "\x27\xed", [0x088c] = "\x27\xee", [0x088d] = "\x27\xef",
  [0x088e] = "\x27\xf0", [0x088f] = "\x27\xf1", [0x0890] = "\x27\xf2",
  [0x0891] = "\x27\xf3", [0x0892] = "\x27\xf4", [0x0893] = "\x27\xf5",
  [0x0894] = "\x27\xf6", [0x0895] = "\x27\xf7", [0x0896] = "\x27\xf8",
  [0x0897] = "\x27\xf9", [0x0898] = "\x27\xfa", [0x0899] = "\x27\xfb",
  [0x089a] = "\x27\xfc", [0x089b] = "\x27\xfd", [0x089c] = "\x27\xfe",
  [0x089d] = "\x27\xff", [0x089e] = "\x28\x00", [0x089f] = "\x28\x01",
  [0x08a0] = "\x28\x02", [0x08a1] = "\x28\x03", [0x08a2] = "\x28\x04",
  [0x08a3] = "\x28\x05", [0x08a4] = "\x28\x06", [0x08a5] = "\x28\x07",
  [0x08a6] = "\x28\x08", [0x08a7] = "\x28\x09", [0x08a8] = "\x28\x0a",
  [0x08a9] = "\x28\x0b", [0x08aa] = "\x28\x0c", [0x08ab] = "\x28\x0d",
  [0x08ac] = "\x28\x0e", [0x08ad] = "\x28\x0f", [0x08ae] = "\x28\x10",
  [0x08af] = "\x28\x11", [0x08b0] = "\x28\x12", [0x08b1] = "\x28\x13",
  [0x08b2] = "\x28\x14", [0x08b3] = "\x28\x15", [0x08b4] = "\x28\x16",
  [0x08b5] = "\x28\x17", [0x08b6] = "\x28\x18", [0x08b7] = "\x28\x19",
  [0x08b8] = "\x28\x1a", [0x08b9] = "\x28\x1b", [0x08ba] = "\x28\x1c",
  [0x08bb] = "\x28\x1d", [0x08bc] = "\x28\x1e", [0x08bd] = "\x28\x1f",
  [0x08be] = "\x28\x20", [0x08bf] = "\x28\x21", [0x08c0] = "\x28\x22",
  [0x08c1] = "\x28\x23", [0x08c2] = "\x28\x24", [0x08c3] = "\x28\x25",
  [0x08c4] = "\x28\x26", [0x08c5] = "\x28\x27", [0x08c6] = "\x28\x28",
  [0x08c7] = "\x28\x29", [0x08c8] = "\x28\x2a", [0x08c9] = "\x28\x2b",
  [0x08ca] = "\x28\x2c", [0x08cb] = "\x28\x2d", [0x08cc] = "\x28\x2e",
  [0x08cd] = "\x28\x2f", [0x08ce] = "\x28\x30", [0x08cf] = "\x28\x31",
  [0x08d0] = "\x28\x32", [0x08d1] = "\x28\x33", [0x08d2] = "\x28\x34",
  [0x08d3] = "\x28\x35", [0x08d4] = "\x28\x36", [0x08d5] = "\x28\x37",
  [0x08d6] = "\x28\x38", [0x08d7] = "\x28\x39", [0x08d8] = "\x28\x3a",
  [0x08d9] = "\x28\x3b", [0x08da] = "\x28\x3c", [0x08db] = "\x28\x3d",
  [0x08dc] = "\x28\x3e", [0x08dd] = "\x28\x3f", [0x08de] = "\x28\x40",
  [0x08df] = "\x28\x41", [0x08e0] = "\x28\x42", [0x08e1] = "\x28\x43",
  [0x08e2] = "\x28\x44", [0x08e3] = "\x28\x45", [0x08e4] = "\x28\x46",
  [0x08e5] = "\x28\x47", [0x08e6] = "\x28\x48", [0x08e7] = "\x28\x49",
  [0x08e8] = "\x28\x4a", [0x08e9] = "\x28\x4b", [0x08ea] = "\x28\x4c",
  [0x08eb] = "\x28\x4d", [0x08ec] = "\x28\x4e", [0x08ed] = "\x28\x4f",
  [0x08ee] = "\x28\x50", [0x08ef] = "\x28\x51", [0x08f0] = "\x28\x52",
  [0x08f1] = "\x28\x53", [0x08f2] = "\x28\x54", [0x08f3] = "\x28\x55",
  [0x08f4] = "\x28\x56", [0x08f5] = "\x28\x57", [0x08f6] = "\x28\x58",
  [0x08f7] = "\x28\x59", [0x08f8] = "\x28\x5a", [0x08f9] = "\x28\x5b",
  [0x08fa] = "\x28\x5c", [0x08fb] = "\x28\x5d", [0x08fc] = "\x28\x5e",
  [0x08fd] = "\x28\x5f", [0x08fe] = "\x28\x60", [0x08ff] = "\x28\x61",
  [0x0900] = "\x28\x62", [0x0901] = "\x28\x63", [0x0902] = "\x28\x64",
  [0x0903] = "\x28\x65", [0x0904] = "\x28\x66", [0x0905] = "\x28\x67",
  [0x0906] = "\x28\x68", [0x0907] = "\x28\x69", [0x0908] = "\x28\x6a",
  [0x0909] = "\x28\x6b", [0x090a] = "\x28\x6c", [0x090b] = "\x28\x6d",
  [0x090c] = "\x28\x6e", [0x090d] = "\x28\x6f", [0x090e] = "\x28\x70",
  [0x090f] = "\x28\x71", [0x0910] = "\x28\x72", [0x0911] = "\x28\x73",
  [0x0912] = "\x28\x74", [0x0913] = "\x28\x75", [0x0914] = "\x28\x76",
  [0x0915] = "\x28\x77", [0x0916] = "\x28\x78", [0x0917] = "\x28\x79",
  [0x0918] = "\x28\x7a", [0x0919] = "\x28\x7b", [0x091a] = "\x28\x7c",
  [0x091b] = "\x28\x7d", [0x091c] = "\x28\x7e", [0x091d] = "\x28\x7f",
  [0x091e] = "\x28\x80", [0x091f] = "\x28\x81", [0x0920] = "\x28\x82",
  [0x0921] = "\x28\x83", [0x0922] = "\x28\x84", [0x0923] = "\x28\x85",
  [0x0924] = "\x28\x86", [0x0925] = "\x28\x87", [0x0926] = "\x28\x88",
  [0x0927] = "\x28\x89", [0x0928] = "\x28\x8a", [0x0929] = "\x28\x8b",
  [0x092a] = "\x28\x8c", [0x092b] = "\x28\x8d", [0x092c] = "\x28\x8e",
  [0x092d] = "\x28\x8f", [0x092e] = "\x28\x90", [0x092f] = "\x28\x91",
  [0x0930] = "\x28\x92", [0x0931] = "\x28\x93", [0x0932] = "\x28\x94",
  [0x0933] = "\x28\x95", [0x0934] = "\x28\x96", [0x0935] = "\x28\x97",
  [0x0936] = "\x28\x98", [0x0937] = "\x28\x99", [0x0938] = "\x28\x9a",
  [0x0939] = "\x28\x9b", [0x093a] = "\x28\x9c", [0x093b] = "\x28\x9d",
  [0x093c] = "\x28\x9e", [0x093d] = "\x28\x9f", [0x093e] = "\x28\xa0",
  [0x093f] = "\x28\xa1", [0x0940] = "\x28\xa2", [0x0941] = "\x28\xa3",
  [0x0942] = "\x28\xa4", [0x0943] = "\x28\xa5", [0x0944] = "\x28\xa6",
  [0x0945] = "\x28\xa7", [0x0946] = "\x28\xa8", [0x0947] = "\x28\xa9",
  [0x0948] = "\x28\xaa", [0x0949] = "\x28\xab", [0x094a] = "\x28\xac",
  [0x094b] = "\x28\xad", [0x094c] = "\x28\xae", [0x094d] = "\x28\xaf",
  [0x094e] = "\x28\xb0", [0x094f] = "\x28\xb1", [0x0950] = "\x28\xb2",
  [0x0951] = "\x28\xb3", [0x0952] = "\x28\xb4", [0x0953] = "\x28\xb5",
  [0x0954] = "\x28\xb6", [0x0955] = "\x28\xb7", [0x0956] = "\x28\xb8",
  [0x0957] = "\x28\xb9", [0x0958] = "\x28\xba", [0x0959] = "\x28\xbb",
  [0x095a] = "\x28\xbc", [0x095b] = "\x28\xbd", [0x095c] = "\x28\xbe",
  [0x095d] = "\x28\xbf", [0x095e] = "\x28\xc0", [0x095f] = "\x28\xc1",
  [0x0960] = "\x28\xc2", [0x0961] = "\x28\xc3", [0x0962] = "\x28\xc4",
  [0x0963] = "\x28\xc5", [0x0964] = "\x28\xc6", [0x0965] = "\x28\xc7",
  [0x0966] = "\x28\xc8", [0x0967] = "\x28\xc9", [0x0968] = "\x28\xca",
  [0x0969] = "\x28\xcb", [0x096a] = "\x28\xcc", [0x096b] = "\x28\xcd",
  [0x096c] = "\x28\xce", [0x096d] = "\x28\xcf", [0x096e] = "\x28\xd0",
  [0x096f] = "\x28\xd1", [0x0970] = "\x28\xd2", [0x0971] = "\x28\xd3",
  [0x0972] = "\x28\xd4", [0x0973] = "\x28\xd5", [0x0974] = "\x28\xd6",
  [0x0975] = "\x28\xd7", [0x0976] = "\x28\xd8", [0x0977] = "\x28\xd9",
  [0x0978] = "\x28\xda", [0x0979] = "\x28\xdb", [0x097a] = "\x28\xdc",
  [0x097b] = "\x28\xdd", [0x097c] = "\x28\xde", [0x097d] = "\x28\xdf",
  [0x097e] = "\x28\xe0", [0x097f] = "\x28\xe1", [0x0980] = "\x28\xe2",
  [0x0981] = "\x28\xe3", [0x0982] = "\x28\xe4", [0x0983] = "\x28\xe5",
  [0x0984] = "\x28\xe6", [0x0985] = "\x28\xe7", [0x0986] = "\x28\xe8",
  [0x0987] = "\x28\xe9", [0x0988] = "\x28\xea", [0x0989] = "\x28\xeb",
  [0x098a] = "\x28\xec", [0x098b] = "\x28\xed", [0x098c] = "\x28\xee",
  [0x098d] = "\x28\xef", [0x098e] = "\x28\xf0", [0x098f] = "\x28\xf1",
  [0x0990] = "\x28\xf2", [0x0991] = "\x28\xf3", [0x0992] = "\x28\xf4",
  [0x0993] = "\x28\xf5", [0x0994] = "\x28\xf6", [0x0995] = "\x28\xf7",
  [0x0996] = "\x28\xf8", [0x0997] = "\x28\xf9", [0x0998] = "\x28\xfa",
  [0x0999] = "\x28\xfb", [0x099a] = "\x28\xfc", [0x099b] = "\x28\xfd",
  [0x099c] = "\x28\xfe", [0x099d] = "\x28\xff", [0x099e] = "\x29\x00",
  [0x099f] = "\x29\x01", [0x09a0] = "\x29\x02", [0x09a1] = "\x29\x03",
  [0x09a2] = "\x29\x04", [0x09a3] = "\x29\x05", [0x09a4] = "\x29\x06",
  [0x09a5] = "\x29\x07", [0x09a6] = "\x29\x08", [0x09a7] = "\x29\x09",
  [0x09a8] = "\x29\x0a", [0x09a9] = "\x29\x0b", [0x09aa] = "\x29\x0c",
  [0x09ab] = "\x29\x0d", [0x09ac] = "\x29\x0e", [0x09ad] = "\x29\x0f",
  [0x09ae] = "\x29\x10", [0x09af] = "\x29\x11", [0x09b0] = "\x29\x12",
  [0x09b1] = "\x29\x13", [0x09b2] = "\x29\x14", [0x09b3] = "\x29\x15",
  [0x09b4] = "\x29\x16", [0x09b5] = "\x29\x17", [0x09b6] = "\x29\x18",
  [0x09b7] = "\x29\x19", [0x09b8] = "\x29\x1a", [0x09b9] = "\x29\x1b",
  [0x09ba] = "\x29\x1c", [0x09bb] = "\x29\x1d", [0x09bc] = "\x29\x1e",
  [0x09bd] = "\x29\x1f", [0x09be] = "\x29\x20", [0x09bf] = "\x29\x21",
  [0x09c0] = "\x29\x22", [0x09c1] = "\x29\x23", [0x09c2] = "\x29\x24",
  [0x09c3] = "\x29\x25", [0x09c4] = "\x29\x26", [0x09c5] = "\x29\x27",
  [0x09c6] = "\x29\x28", [0x09c7] = "\x29\x29", [0x09c8] = "\x29\x2a",
  [0x09c9] = "\x29\x2b", [0x09ca] = "\x29\x2c", [0x09cb] = "\x29\x2d",
  [0x09cc] = "\x29\x2e", [0x09cd] = "\x29\x2f", [0x09ce] = "\x29\x30",
  [0x09cf] = "\x29\x31", [0x09d0] = "\x29\x32", [0x09d1] = "\x29\x33",
  [0x09d2] = "\x29\x34", [0x09d3] = "\x29\x35", [0x09d4] = "\x29\x36",
  [0x09d5] = "\x29\x37", [0x09d6] = "\x29\x38", [0x09d7] = "\x29\x39",
  [0x09d8] = "\x29\x3a", [0x09d9] = "\x29\x3b", [0x09da] = "\x29\x3c",
  [0x09db] = "\x29\x3d", [0x09dc] = "\x29\x3e", [0x09dd] = "\x29\x3f",
  [0x09de] = "\x29\x40", [0x09df] = "\x29\x41", [0x09e0] = "\x29\x42",
  [0x09e1] = "\x29\x43", [0x09e2] = "\x29\x44", [0x09e3] = "\x29\x45",
  [0x09e4] = "\x29\x46", [0x09e5] = "\x29\x47", [0x09e6] = "\x29\x48",
  [0x09e7] = "\x29\x49", [0x09e8] = "\x29\x4a", [0x09e9] = "\x29\x4b",
  [0x09ea] = "\x29\x4c", [0x09eb] = "\x29\x4d", [0x09ec] = "\x29\x4e",
  [0x09ed] = "\x29\x4f", [0x09ee] = "\x29\x50", [0x09ef] = "\x29\x51",
  [0x09f0] = "\x29\x52", [0x09f1] = "\x29\x53", [0x09f2] = "\x29\x54",
  [0x09f3] = "\x29\x55", [0x09f4] = "\x29\x56", [0x09f5] = "\x29\x57",
  [0x09f6] = "\x29\x58", [0x09f7] = "\x29\x59", [0x09f8] = "\x29\x5a",
  [0x09f9] = "\x29\x5b", [0x09fa] = "\x29\x5c", [0x09fb] = "\x29\x5d",
  [0x09fc] = "\x29\x5e", [0x09fd] = "\x29\x5f", [0x09fe] = "\x29\x60",
  [0x09ff] = "\x29\x61", [0x0a00] = "\x29\x62", [0x0a01] = "\x29\x63",
  [0x0a02] = "\x29\x64", [0x0a03] = "\x29\x65", [0x0a04] = "\x29\x66",
  [0x0a05] = "\x29\x67", [0x0a06] = "\x29\x68", [0x0a07] = "\x29\x69",
  [0x0a08] = "\x29\x6a", [0x0a09] = "\x29\x6b", [0x0a0a] = "\x29\x6c",
  [0x0a0b] = "\x29\x6d", [0x0a0c] = "\x29\x6e", [0x0a0d] = "\x29\x6f",
  [0x0a0e] = "\x29\x70", [0x0a0f] = "\x29\x71", [0x0a10] = "\x29\x72",
  [0x0a11] = "\x29\x73", [0x0a12] = "\x29\x74", [0x0a13] = "\x29\x75",
  [0x0a14] = "\x29\x76", [0x0a15] = "\x29\x77", [0x0a16] = "\x29\x78",
  [0x0a17] = "\x29\x79", [0x0a18] = "\x29\x7a", [0x0a19] = "\x29\x7b",
  [0x0a1a] = "\x29\x7c", [0x0a1b] = "\x29\x7d", [0x0a1c] = "\x29\x7e",
  [0x0a1d] = "\x29\x7f", [0x0a1e] = "\x29\x80", [0x0a1f] = "\x29\x81",
  [0x0a20] = "\x29\x82", [0x0a21] = "\x29\x83", [0x0a22] = "\x29\x84",
  [0x0a23] = "\x29\x85", [0x0a24] = "\x29\x86", [0x0a25] = "\x29\x87",
  [0x0a26] = "\x29\x88", [0x0a27] = "\x29\x89", [0x0a28] = "\x29\x8a",
  [0x0a29] = "\x29\x8b", [0x0a2a] = "\x29\x8c", [0x0a2b] = "\x29\x8d",
  [0x0a2c] = "\x29\x8e", [0x0a2d] = "\x29\x8f", [0x0a2e] = "\x29\x90",
  [0x0a2f] = "\x29\x91", [0x0a30] = "\x29\x92", [0x0a31] = "\x29\x93",
  [0x0a32] = "\x29\x94", [0x0a33] = "\x29\x95", [0x0a34] = "\x29\x96",
  [0x0a35] = "\x29\x97", [0x0a36] = "\x29\x98", [0x0a37] = "\x29\x99",
  [0x0a38] = "\x29\x9a", [0x0a39] = "\x29\x9b", [0x0a3a] = "\x29\x9c",
  [0x0a3b] = "\x29\x9d", [0x0a3c] = "\x29\x9e", [0x0a3d] = "\x29\x9f",
  [0x0a3e] = "\x29\xa0", [0x0a3f] = "\x29\xa1", [0x0a40] = "\x29\xa2",
  [0x0a41] = "\x29\xa3", [0x0a42] = "\x29\xa4", [0x0a43] = "\x29\xa5",
  [0x0a44] = "\x29\xa6", [0x0a45] = "\x29\xa7", [0x0a46] = "\x29\xa8",
  [0x0a47] = "\x29\xa9", [0x0a48] = "\x29\xaa", [0x0a49] = "\x29\xab",
  [0x0a4a] = "\x29\xac", [0x0a4b] = "\x29\xad", [0x0a4c] = "\x29\xae",
  [0x0a4d] = "\x29\xaf", [0x0a4e] = "\x29\xb0", [0x0a4f] = "\x29\xb1",
  [0x0a50] = "\x29\xb2", [0x0a51] = "\x29\xb3", [0x0a52] = "\x29\xb4",
  [0x0a53] = "\x29\xb5", [0x0a54] = "\x29\xb6", [0x0a55] = "\x29\xb7",
  [0x0a56] = "\x29\xb8", [0x0a57] = "\x29\xb9", [0x0a58] = "\x29\xba",
  [0x0a59] = "\x29\xbb", [0x0a5a] = "\x29\xbc", [0x0a5b] = "\x29\xbd",
  [0x0a5c] = "\x29\xbe", [0x0a5d] = "\x29\xbf", [0x0a5e] = "\x29\xc0",
  [0x0a5f] = "\x29\xc1", [0x0a60] = "\x29\xc2", [0x0a61] = "\x29\xc3",
  [0x0a62] = "\x29\xc4", [0x0a63] = "\x29\xc5", [0x0a64] = "\x29\xc6",
  [0x0a65] = "\x29\xc7", [0x0a66] = "\x29\xc8", [0x0a67] = "\x29\xc9",
  [0x0a68] = "\x29\xca", [0x0a69] = "\x29\xcb", [0x0a6a] = "\x29\xcc",
  [0x0a6b] = "\x29\xcd", [0x0a6c] = "\x29\xce", [0x0a6d] = "\x29\xcf",
  [0x0a6e] = "\x29\xd0", [0x0a6f] = "\x29\xd1", [0x0a70] = "\x29\xd2",
  [0x0a71] = "\x29\xd3", [0x0a72] = "\x29\xd4", [0x0a73] = "\x29\xd5",
  [0x0a74] = "\x29\xd6", [0x0a75] = "\x29\xd7", [0x0a76] = "\x29\xd8",
  [0x0a77] = "\x29\xd9", [0x0a78] = "\x29\xda", [0x0a79] = "\x29\xdb",
  [0x0a7a] = "\x29\xdc", [0x0a7b] = "\x29\xdd", [0x0a7c] = "\x29\xde",
  [0x0a7d] = "\x29\xdf", [0x0a7e] = "\x29\xe0", [0x0a7f] = "\x29\xe1",
  [0x0a80] = "\x29\xe2", [0x0a81] = "\x29\xe3", [0x0a82] = "\x29\xe4",
  [0x0a83] = "\x29\xe5", [0x0a84] = "\x29\xe6", [0x0a85] = "\x29\xe7",
  [0x0a86] = "\x29\xe8", [0x0a87] = "\x29\xe9", [0x0a88] = "\x29\xea",
  [0x0a89] = "\x29\xeb", [0x0a8a] = "\x29\xec", [0x0a8b] = "\x29\xed",
  [0x0a8c] = "\x29\xee", [0x0a8d] = "\x29\xef", [0x0a8e] = "\x29\xf0",
  [0x0a8f] = "\x29\xf1", [0x0a90] = "\x29\xf2", [0x0a91] = "\x29\xf3",
  [0x0a92] = "\x29\xf4", [0x0a93] = "\x29\xf5", [0x0a94] = "\x29\xf6",
  [0x0a95] = "\x29\xf7", [0x0a96] = "\x29\xf8", [0x0a97] = "\x29\xf9",
  [0x0a98] = "\x29\xfa", [0x0a99] = "\x29\xfb", [0x0a9a] = "\x29\xfc",
  [0x0a9b] = "\x29\xfd", [0x0a9c] = "\x29\xfe", [0x0a9d] = "\x29\xff",
  [0x0a9e] = "\x2a\x00", [0x0a9f] = "\x2a\x01", [0x0aa0] = "\x2a\x02",
  [0x0aa1] = "\x2a\x03", [0x0aa2] = "\x2a\x04", [0x0aa3] = "\x2a\x05",
  [0x0aa4] = "\x2a\x06", [0x0aa5] = "\x2a\x07", [0x0aa6] = "\x2a\x08",
  [0x0aa7] = "\x2a\x09", [0x0aa8] = "\x2a\x0a", [0x0aa9] = "\x2a\x0b",
  [0x0aaa] = "\x2a\x0c", [0x0aab] = "\x2a\x0d", [0x0aac] = "\x2a\x0e",
  [0x0aad] = "\x2a\x0f", [0x0aae] = "\x2a\x10", [0x0aaf] = "\x2a\x11",
  [0x0ab0] = "\x2a\x12", [0x0ab1] = "\x2a\x13", [0x0ab2] = "\x2a\x14",
  [0x0ab3] = "\x2a\x15", [0x0ab4] = "\x2a\x16", [0x0ab5] = "\x2a\x17",
  [0x0ab6] = "\x2a\x18", [0x0ab7] = "\x2a\x19", [0x0ab8] = "\x2a\x1a",
  [0x0ab9] = "\x2a\x1b", [0x0aba] = "\x2a\x1c", [0x0abb] = "\x2a\x1d",
  [0x0abc] = "\x2a\x1e", [0x0abd] = "\x2a\x1f", [0x0abe] = "\x2a\x20",
  [0x0abf] = "\x2a\x21", [0x0ac0] = "\x2a\x22", [0x0ac1] = "\x2a\x23",
  [0x0ac2] = "\x2a\x24", [0x0ac3] = "\x2a\x25", [0x0ac4] = "\x2a\x26",
  [0x0ac5] = "\x2a\x27", [0x0ac6] = "\x2a\x28", [0x0ac7] = "\x2a\x29",
  [0x0ac8] = "\x2a\x2a", [0x0ac9] = "\x2a\x2b", [0x0aca] = "\x2a\x2c",
  [0x0acb] = "\x2a\x2d", [0x0acc] = "\x2a\x2e", [0x0acd] = "\x2a\x2f",
  [0x0ace] = "\x2a\x30", [0x0acf] = "\x2a\x31", [0x0ad0] = "\x2a\x32",
  [0x0ad1] = "\x2a\x33", [0x0ad2] = "\x2a\x34", [0x0ad3] = "\x2a\x35",
  [0x0ad4] = "\x2a\x36", [0x0ad5] = "\x2a\x37", [0x0ad6] = "\x2a\x38",
  [0x0ad7] = "\x2a\x39", [0x0ad8] = "\x2a\x3a", [0x0ad9] = "\x2a\x3b",
  [0x0ada] = "\x2a\x3c", [0x0adb] = "\x2a\x3d", [0x0adc] = "\x2a\x3e",
  [0x0add] = "\x2a\x3f", [0x0ade] = "\x2a\x40", [0x0adf] = "\x2a\x41",
  [0x0ae0] = "\x2a\x42", [0x0ae1] = "\x2a\x43", [0x0ae2] = "\x2a\x44",
  [0x0ae3] = "\x2a\x45", [0x0ae4] = "\x2a\x46", [0x0ae5] = "\x2a\x47",
  [0x0ae6] = "\x2a\x48", [0x0ae7] = "\x2a\x49", [0x0ae8] = "\x2a\x4a",
  [0x0ae9] = "\x2a\x4b", [0x0aea] = "\x2a\x4c", [0x0aeb] = "\x2a\x4d",
  [0x0aec] = "\x2a\x4e", [0x0aed] = "\x2a\x4f", [0x0aee] = "\x2a\x50",
  [0x0aef] = "\x2a\x51", [0x0af0] = "\x2a\x52", [0x0af1] = "\x2a\x53",
  [0x0af2] = "\x2a\x54", [0x0af3] = "\x2a\x55", [0x0af4] = "\x2a\x56",
  [0x0af5] = "\x2a\x57", [0x0af6] = "\x2a\x58", [0x0af7] = "\x2a\x59",
  [0x0af8] = "\x2a\x5a", [0x0af9] = "\x2a\x5b", [0x0afa] = "\x2a\x5c",
  [0x0afb] = "\x2a\x5d", [0x0afc] = "\x2a\x5e", [0x0afd] = "\x2a\x5f",
  [0x0afe] = "\x2a\x60", [0x0aff] = "\x2a\x61", [0x0b00] = "\x2a\x62",
  [0x0b01] = "\x2a\x63", [0x0b02] = "\x2a\x64", [0x0b03] = "\x2a\x65",
  [0x0b04] = "\x2a\x66", [0x0b05] = "\x2a\x67", [0x0b06] = "\x2a\x68",
  [0x0b07] = "\x2a\x69", [0x0b08] = "\x2a\x6a", [0x0b09] = "\x2a\x6b",
  [0x0b0a] = "\x2a\x6c", [0x0b0b] = "\x2a\x6d", [0x0b0c] = "\x2a\x6e",
  [0x0b0d] = "\x2a\x6f", [0x0b0e] = "\x2a\x70", [0x0b0f] = "\x2a\x71",
  [0x0b10] = "\x2a\x72", [0x0b11] = "\x2a\x73", [0x0b12] = "\x2a\x74",
  [0x0b13] = "\x2a\x75", [0x0b14] = "\x2a\x76", [0x0b15] = "\x2a\x77",
  [0x0b16] = "\x2a\x78", [0x0b17] = "\x2a\x79", [0x0b18] = "\x2a\x7a",
  [0x0b19] = "\x2a\x7b", [0x0b1a] = "\x2a\x7c", [0x0b1b] = "\x2a\x7d",
  [0x0b1c] = "\x2a\x7e", [0x0b1d] = "\x2a\x7f", [0x0b1e] = "\x2a\x80",
  [0x0b1f] = "\x2a\x81", [0x0b20] = "\x2a\x82", [0x0b21] = "\x2a\x83",
  [0x0b22] = "\x2a\x84", [0x0b23] = "\x2a\x85", [0x0b24] = "\x2a\x86",
  [0x0b25] = "\x2a\x87", [0x0b26] = "\x2a\x88", [0x0b27] = "\x2a\x89",
  [0x0b28] = "\x2a\x8a", [0x0b29] = "\x2a\x8b", [0x0b2a] = "\x2a\x8c",
  [0x0b2b] = "\x2a\x8d", [0x0b2c] = "\x2a\x8e", [0x0b2d] = "\x2a\x8f",
  [0x0b2e] = "\x2a\x90", [0x0b2f] = "\x2a\x91", [0x0b30] = "\x2a\x92",
  [0x0b31] = "\x2a\x93", [0x0b32] = "\x2a\x94", [0x0b33] = "\x2a\x95",
  [0x0b34] = "\x2a\x96", [0x0b35] = "\x2a\x97", [0x0b36] = "\x2a\x98",
  [0x0b37] = "\x2a\x99", [0x0b38] = "\x2a\x9a", [0x0b39] = "\x2a\x9b",
  [0x0b3a] = "\x2a\x9c", [0x0b3b] = "\x2a\x9d", [0x0b3c] = "\x2a\x9e",
  [0x0b3d] = "\x2a\x9f", [0x0b3e] = "\x2a\xa0", [0x0b3f] = "\x2a\xa1",
  [0x0b40] = "\x2a\xa2", [0x0b41] = "\x2a\xa3", [0x0b42] = "\x2a\xa4",
  [0x0b43] = "\x2a\xa5", [0x0b44] = "\x2a\xa6", [0x0b45] = "\x2a\xa7",
  [0x0b46] = "\x2a\xa8", [0x0b47] = "\x2a\xa9", [0x0b48] = "\x2a\xaa",
  [0x0b49] = "\x2a\xab", [0x0b4a] = "\x2a\xac", [0x0b4b] = "\x2a\xad",
  [0x0b4c] = "\x2a\xae", [0x0b4d] = "\x2a\xaf", [0x0b4e] = "\x2a\xb0",
  [0x0b4f] = "\x2a\xb1", [0x0b50] = "\x2a\xb2", [0x0b51] = "\x2a\xb3",
  [0x0b52] = "\x2a\xb4", [0x0b53] = "\x2a\xb5", [0x0b54] = "\x2a\xb6",
  [0x0b55] = "\x2a\xb7", [0x0b56] = "\x2a\xb8", [0x0b57] = "\x2a\xb9",
  [0x0b58] = "\x2a\xba", [0x0b59] = "\x2a\xbb", [0x0b5a] = "\x2a\xbc",
  [0x0b5b] = "\x2a\xbd", [0x0b5c] = "\x2a\xbe", [0x0b5d] = "\x2a\xbf",
  [0x0b5e] = "\x2a\xc0", [0x0b5f] = "\x2a\xc1", [0x0b60] = "\x2a\xc2",
  [0x0b61] = "\x2a\xc3", [0x0b62] = "\x2a\xc4", [0x0b63] = "\x2a\xc5",
  [0x0b64] = "\x2a\xc6", [0x0b65] = "\x2a\xc7", [0x0b66] = "\x2a\xc8",
  [0x0b67] = "\x2a\xc9", [0x0b68] = "\x2a\xca", [0x0b69] = "\x2a\xcb",
  [0x0b6a] = "\x2a\xcc", [0x0b6b] = "\x2a\xcd", [0x0b6c] = "\x2a\xce",
  [0x0b6d] = "\x2a\xcf", [0x0b6e] = "\x2a\xd0", [0x0b6f] = "\x2a\xd1",
  [0x0b70] = "\x2a\xd2", [0x0b71] = "\x2a\xd3", [0x0b72] = "\x2a\xd4",
  [0x0b73] = "\x2a\xd5", [0x0b74] = "\x2a\xd6", [0x0b75] = "\x2a\xd7",
  [0x0b76] = "\x2a\xd8", [0x0b77] = "\x2a\xd9", [0x0b78] = "\x2a\xda",
  [0x0b79] = "\x2a\xdb", [0x0b7a] = "\x2a\xdc", [0x0b7b] = "\x2a\xdd",
  [0x0b7c] = "\x2a\xde", [0x0b7d] = "\x2a\xdf", [0x0b7e] = "\x2a\xe0",
  [0x0b7f] = "\x2a\xe1", [0x0b80] = "\x2a\xe2", [0x0b81] = "\x2a\xe3",
  [0x0b82] = "\x2a\xe4", [0x0b83] = "\x2a\xe5", [0x0b84] = "\x2a\xe6",
  [0x0b85] = "\x2a\xe7", [0x0b86] = "\x2a\xe8", [0x0b87] = "\x2a\xe9",
  [0x0b88] = "\x2a\xea", [0x0b89] = "\x2a\xeb", [0x0b8a] = "\x2a\xec",
  [0x0b8b] = "\x2a\xed", [0x0b8c] = "\x2a\xee", [0x0b8d] = "\x2a\xef",
  [0x0b8e] = "\x2a\xf0", [0x0b8f] = "\x2a\xf1", [0x0b90] = "\x2a\xf2",
  [0x0b91] = "\x2a\xf3", [0x0b92] = "\x2a\xf4", [0x0b93] = "\x2a\xf5",
  [0x0b94] = "\x2a\xf6", [0x0b95] = "\x2a\xf7", [0x0b96] = "\x2a\xf8",
  [0x0b97] = "\x2a\xf9", [0x0b98] = "\x2a\xfa", [0x0b99] = "\x2a\xfb",
  [0x0b9a] = "\x2a\xfc", [0x0b9b] = "\x2a\xfd", [0x0b9c] = "\x2a\xfe",
  [0x0b9d] = "\x2a\xff", [0x0b9e] = "\x2b\x00", [0x0b9f] = "\x2b\x01",
  [0x0ba0] = "\x2b\x02", [0x0ba1] = "\x2b\x03", [0x0ba2] = "\x2b\x04",
  [0x0ba3] = "\x2b\x05", [0x0ba4] = "\x2b\x06", [0x0ba5] = "\x2b\x07",
  [0x0ba6] = "\x2b\x08", [0x0ba7] = "\x2b\x09", [0x0ba8] = "\x2b\x0a",
  [0x0ba9] = "\x2b\x0b", [0x0baa] = "\x2b\x0c", [0x0bab] = "\x2b\x0d",
  [0x0bac] = "\x2b\x0e", [0x0bad] = "\x2b\x0f", [0x0bae] = "\x2b\x10",
  [0x0baf] = "\x2b\x11", [0x0bb0] = "\x2b\x12", [0x0bb1] = "\x2b\x13",
  [0x0bb2] = "\x2b\x14", [0x0bb3] = "\x2b\x15", [0x0bb4] = "\x2b\x16",
  [0x0bb5] = "\x2b\x17", [0x0bb6] = "\x2b\x18", [0x0bb7] = "\x2b\x19",
  [0x0bb8] = "\x2b\x1a", [0x0bb9] = "\x2b\x1b", [0x0bba] = "\x2b\x1c",
  [0x0bbb] = "\x2b\x1d", [0x0bbc] = "\x2b\x1e", [0x0bbd] = "\x2b\x1f",
  [0x0bbe] = "\x2b\x20", [0x0bbf] = "\x2b\x21", [0x0bc0] = "\x2b\x22",
  [0x0bc1] = "\x2b\x23", [0x0bc2] = "\x2b\x24", [0x0bc3] = "\x2b\x25",
  [0x0bc4] = "\x2b\x26", [0x0bc5] = "\x2b\x27", [0x0bc6] = "\x2b\x28",
  [0x0bc7] = "\x2b\x29", [0x0bc8] = "\x2b\x2a", [0x0bc9] = "\x2b\x2b",
  [0x0bca] = "\x2b\x2c", [0x0bcb] = "\x2b\x2d", [0x0bcc] = "\x2b\x2e",
  [0x0bcd] = "\x2b\x2f", [0x0bce] = "\x2b\x30", [0x0bcf] = "\x2b\x31",
  [0x0bd0] = "\x2b\x32", [0x0bd1] = "\x2b\x33", [0x0bd2] = "\x2b\x34",
  [0x0bd3] = "\x2b\x35", [0x0bd4] = "\x2b\x36", [0x0bd5] = "\x2b\x37",
  [0x0bd6] = "\x2b\x38", [0x0bd7] = "\x2b\x39", [0x0bd8] = "\x2b\x3a",
  [0x0bd9] = "\x2b\x3b", [0x0bda] = "\x2b\x3c", [0x0bdb] = "\x2b\x3d",
  [0x0bdc] = "\x2b\x3e", [0x0bdd] = "\x2b\x3f", [0x0bde] = "\x2b\x40",
  [0x0bdf] = "\x2b\x41", [0x0be0] = "\x2b\x42", [0x0be1] = "\x2b\x43",
  [0x0be2] = "\x2b\x44", [0x0be3] = "\x2b\x45", [0x0be4] = "\x2b\x46",
  [0x0be5] = "\x2b\x47", [0x0be6] = "\x2b\x48", [0x0be7] = "\x2b\x49",
  [0x0be8] = "\x2b\x4a", [0x0be9] = "\x2b\x4b", [0x0bea] = "\x2b\x4c",
  [0x0beb] = "\x2b\x4d", [0x0bec] = "\x2b\x4e", [0x0bed] = "\x2b\x4f",
  [0x0bee] = "\x2b\x50", [0x0bef] = "\x2b\x51", [0x0bf0] = "\x2b\x52",
  [0x0bf1] = "\x2b\x53", [0x0bf2] = "\x2b\x54", [0x0bf3] = "\x2b\x55",
  [0x0bf4] = "\x2b\x56", [0x0bf5] = "\x2b\x57", [0x0bf6] = "\x2b\x58",
  [0x0bf7] = "\x2b\x59", [0x0bf8] = "\x2b\x5a", [0x0bf9] = "\x2b\x5b",
  [0x0bfa] = "\x2b\x5c", [0x0bfb] = "\x2b\x5d", [0x0bfc] = "\x2b\x5e",
  [0x0bfd] = "\x2b\x5f", [0x0bfe] = "\x2b\x60", [0x0bff] = "\x2b\x61",
  [0x0c00] = "\x2b\x62", [0x0c01] = "\x2b\x63", [0x0c02] = "\x2b\x64",
  [0x0c03] = "\x2b\x65", [0x0c04] = "\x2b\x66", [0x0c05] = "\x2b\x67",
  [0x0c06] = "\x2b\x68", [0x0c07] = "\x2b\x69", [0x0c08] = "\x2b\x6a",
  [0x0c09] = "\x2b\x6b", [0x0c0a] = "\x2b\x6c", [0x0c0b] = "\x2b\x6d",
  [0x0c0c] = "\x2b\x6e", [0x0c0d] = "\x2b\x6f", [0x0c0e] = "\x2b\x70",
  [0x0c0f] = "\x2b\x71", [0x0c10] = "\x2b\x72", [0x0c11] = "\x2b\x73",
  [0x0c12] = "\x2b\x74", [0x0c13] = "\x2b\x75", [0x0c14] = "\x2b\x76",
  [0x0c15] = "\x2b\x77", [0x0c16] = "\x2b\x78", [0x0c17] = "\x2b\x79",
  [0x0c18] = "\x2b\x7a", [0x0c19] = "\x2b\x7b", [0x0c1a] = "\x2b\x7c",
  [0x0c1b] = "\x2b\x7d", [0x0c1c] = "\x2b\x7e", [0x0c1d] = "\x2b\x7f",
  [0x0c1e] = "\x2b\x80", [0x0c1f] = "\x2b\x81", [0x0c20] = "\x2b\x82",
  [0x0c21] = "\x2b\x83", [0x0c22] = "\x2b\x84", [0x0c23] = "\x2b\x85",
  [0x0c24] = "\x2b\x86", [0x0c25] = "\x2b\x87", [0x0c26] = "\x2b\x88",
  [0x0c27] = "\x2b\x89", [0x0c28] = "\x2b\x8a", [0x0c29] = "\x2b\x8b",
  [0x0c2a] = "\x2b\x8c", [0x0c2b] = "\x2b\x8d", [0x0c2c] = "\x2b\x8e",
  [0x0c2d] = "\x2b\x8f", [0x0c2e] = "\x2b\x90", [0x0c2f] = "\x2b\x91",
  [0x0c30] = "\x2b\x92", [0x0c31] = "\x2b\x93", [0x0c32] = "\x2b\x94",
  [0x0c33] = "\x2b\x95", [0x0c34] = "\x2b\x96", [0x0c35] = "\x2b\x97",
  [0x0c36] = "\x2b\x98", [0x0c37] = "\x2b\x99", [0x0c38] = "\x2b\x9a",
  [0x0c39] = "\x2b\x9b", [0x0c3a] = "\x2b\x9c", [0x0c3b] = "\x2b\x9d",
  [0x0c3c] = "\x2b\x9e", [0x0c3d] = "\x2b\x9f", [0x0c3e] = "\x2b\xa0",
  [0x0c3f] = "\x2b\xa1", [0x0c40] = "\x2b\xa2", [0x0c41] = "\x2b\xa3",
  [0x0c42] = "\x2b\xa4", [0x0c43] = "\x2b\xa5", [0x0c44] = "\x2b\xa6",
  [0x0c45] = "\x2b\xa7", [0x0c46] = "\x2b\xa8", [0x0c47] = "\x2b\xa9",
  [0x0c48] = "\x2b\xaa", [0x0c49] = "\x2b\xab", [0x0c4a] = "\x2b\xac",
  [0x0c4b] = "\x2b\xad", [0x0c4c] = "\x2b\xae", [0x0c4d] = "\x2b\xaf",
  [0x0c4e] = "\x2b\xb0", [0x0c4f] = "\x2b\xb1", [0x0c50] = "\x2b\xb2",
  [0x0c51] = "\x2b\xb3", [0x0c52] = "\x2b\xb4", [0x0c53] = "\x2b\xb5",
  [0x0c54] = "\x2b\xb6", [0x0c55] = "\x2b\xb7", [0x0c56] = "\x2b\xb8",
  [0x0c57] = "\x2b\xb9", [0x0c58] = "\x2b\xba", [0x0c59] = "\x2b\xbb",
  [0x0c5a] = "\x2b\xbc", [0x0c5b] = "\x2b\xbd", [0x0c5c] = "\x2b\xbe",
  [0x0c5d] = "\x2b\xbf", [0x0c5e] = "\x2b\xc0", [0x0c5f] = "\x2b\xc1",
  [0x0c60] = "\x2b\xc2", [0x0c61] = "\x2b\xc3", [0x0c62] = "\x2b\xc4",
  [0x0c63] = "\x2b\xc5", [0x0c64] = "\x2b\xc6", [0x0c65] = "\x2b\xc7",
  [0x0c66] = "\x2b\xc8", [0x0c67] = "\x2b\xc9", [0x0c68] = "\x2b\xca",
  [0x0c69] = "\x2b\xcb", [0x0c6a] = "\x2b\xcc", [0x0c6b] = "\x2b\xcd",
  [0x0c6c] = "\x2b\xce", [0x0c6d] = "\x2b\xcf", [0x0c6e] = "\x2b\xd0",
  [0x0c6f] = "\x2b\xd1", [0x0c70] = "\x2b\xd2", [0x0c71] = "\x2b\xd3",
  [0x0c72] = "\x2b\xd4", [0x0c73] = "\x2b\xd5", [0x0c74] = "\x2b\xd6",
  [0x0c75] = "\x2b\xd7", [0x0c76] = "\x2b\xd8", [0x0c77] = "\x2b\xd9",
  [0x0c78] = "\x2b\xda", [0x0c79] = "\x2b\xdb", [0x0c7a] = "\x2b\xdc",
  [0x0c7b] = "\x2b\xdd", [0x0c7c] = "\x2b\xde", [0x0c7d] = "\x2b\xdf",
  [0x0c7e] = "\x2b\xe0", [0x0c7f] = "\x2b\xe1", [0x0c80] = "\x2b\xe2",
  [0x0c81] = "\x2b\xe3", [0x0c82] = "\x2b\xe4", [0x0c83] = "\x2b\xe5",
  [0x0c84] = "\x2b\xe6", [0x0c85] = "\x2b\xe7", [0x0c86] = "\x2b\xe8",
  [0x0c87] = "\x2b\xe9", [0x0c88] = "\x2b\xea", [0x0c89] = "\x2b\xeb",
  [0x0c8a] = "\x2b\xec", [0x0c8b] = "\x2b\xed", [0x0c8c] = "\x2b\xee",
  [0x0c8d] = "\x2b\xef", [0x0c8e] = "\x2b\xf0", [0x0c8f] = "\x2b\xf1",
  [0x0c90] = "\x2b\xf2", [0x0c91] = "\x2b\xf3", [0x0c92] = "\x2b\xf4",
  [0x0c93] = "\x2b\xf5", [0x0c94] = "\x2b\xf6", [0x0c95] = "\x2b\xf7",
  [0x0c96] = "\x2b\xf8", [0x0c97] = "\x2b\xf9", [0x0c98] = "\x2b\xfa",
  [0x0c99] = "\x2b\xfb", [0x0c9a] = "\x2b\xfc", [0x0c9b] = "\x2b\xfd",
  [0x0c9c] = "\x2b\xfe", [0x0c9d] = "\x2b\xff", [0x0c9e] = "\x2c\x00",
  [0x0c9f] = "\x2c\x01", [0x0ca0] = "\x2c\x02", [0x0ca1] = "\x2c\x03",
  [0x0ca2] = "\x2c\x04", [0x0ca3] = "\x2c\x05", [0x0ca4] = "\x2c\x06",
  [0x0ca5] = "\x2c\x07", [0x0ca6] = "\x2c\x08", [0x0ca7] = "\x2c\x09",
  [0x0ca8] = "\x2c\x0a", [0x0ca9] = "\x2c\x0b", [0x0caa] = "\x2c\x0c",
  [0x0cab] = "\x2c\x0d", [0x0cac] = "\x2c\x0e", [0x0cad] = "\x2c\x0f",
  [0x0cae] = "\x2c\x10", [0x0caf] = "\x2c\x11", [0x0cb0] = "\x2c\x12",
  [0x0cb1] = "\x2c\x13", [0x0cb2] = "\x2c\x14", [0x0cb3] = "\x2c\x15",
  [0x0cb4] = "\x2c\x16", [0x0cb5] = "\x2c\x17", [0x0cb6] = "\x2c\x18",
  [0x0cb7] = "\x2c\x19", [0x0cb8] = "\x2c\x1a", [0x0cb9] = "\x2c\x1b",
  [0x0cba] = "\x2c\x1c", [0x0cbb] = "\x2c\x1d", [0x0cbc] = "\x2c\x1e",
  [0x0cbd] = "\x2c\x1f", [0x0cbe] = "\x2c\x20", [0x0cbf] = "\x2c\x21",
  [0x0cc0] = "\x2c\x22", [0x0cc1] = "\x2c\x23", [0x0cc2] = "\x2c\x24",
  [0x0cc3] = "\x2c\x25", [0x0cc4] = "\x2c\x26", [0x0cc5] = "\x2c\x27",
  [0x0cc6] = "\x2c\x28", [0x0cc7] = "\x2c\x29", [0x0cc8] = "\x2c\x2a",
  [0x0cc9] = "\x2c\x2b", [0x0cca] = "\x2c\x2c", [0x0ccb] = "\x2c\x2d",
  [0x0ccc] = "\x2c\x2e", [0x0ccd] = "\x2c\x2f", [0x0cce] = "\x2c\x30",
  [0x0ccf] = "\x2c\x31", [0x0cd0] = "\x2c\x32", [0x0cd1] = "\x2c\x33",
  [0x0cd2] = "\x2c\x34", [0x0cd3] = "\x2c\x35", [0x0cd4] = "\x2c\x36",
  [0x0cd5] = "\x2c\x37", [0x0cd6] = "\x2c\x38", [0x0cd7] = "\x2c\x39",
  [0x0cd8] = "\x2c\x3a", [0x0cd9] = "\x2c\x3b", [0x0cda] = "\x2c\x3c",
  [0x0cdb] = "\x2c\x3d", [0x0cdc] = "\x2c\x3e", [0x0cdd] = "\x2c\x3f",
  [0x0cde] = "\x2c\x40", [0x0cdf] = "\x2c\x41", [0x0ce0] = "\x2c\x42",
  [0x0ce1] = "\x2c\x43", [0x0ce2] = "\x2c\x44", [0x0ce3] = "\x2c\x45",
  [0x0ce4] = "\x2c\x46", [0x0ce5] = "\x2c\x47", [0x0ce6] = "\x2c\x48",
  [0x0ce7] = "\x2c\x49", [0x0ce8] = "\x2c\x4a", [0x0ce9] = "\x2c\x4b",
  [0x0cea] = "\x2c\x4c", [0x0ceb] = "\x2c\x4d", [0x0cec] = "\x2c\x4e",
  [0x0ced] = "\x2c\x4f", [0x0cee] = "\x2c\x50", [0x0cef] = "\x2c\x51",
  [0x0cf0] = "\x2c\x52", [0x0cf1] = "\x2c\x53", [0x0cf2] = "\x2c\x54",
  [0x0cf3] = "\x2c\x55", [0x0cf4] = "\x2c\x56", [0x0cf5] = "\x2c\x57",
  [0x0cf6] = "\x2c\x58", [0x0cf7] = "\x2c\x59", [0x0cf8] = "\x2c\x5a",
  [0x0cf9] = "\x2c\x5b", [0x0cfa] = "\x2c\x5c", [0x0cfb] = "\x2c\x5d",
  [0x0cfc] = "\x2c\x5e", [0x0cfd] = "\x2c\x5f", [0x0cfe] = "\x2c\x60",
  [0x0cff] = "\x2c\x61", [0x0d00] = "\x2c\x62", [0x0d01] = "\x2c\x63",
  [0x0d02] = "\x2c\x64", [0x0d03] = "\x2c\x65", [0x0d04] = "\x2c\x66",
  [0x0d05] = "\x2c\x67", [0x0d06] = "\x2c\x68", [0x0d07] = "\x2c\x69",
  [0x0d08] = "\x2c\x6a", [0x0d09] = "\x2c\x6b", [0x0d0a] = "\x2c\x6c",
  [0x0d0b] = "\x2c\x6d", [0x0d0c] = "\x2c\x6e", [0x0d0d] = "\x2c\x6f",
  [0x0d0e] = "\x2c\x70", [0x0d0f] = "\x2c\x71", [0x0d10] = "\x2c\x72",
  [0x0d11] = "\x2c\x73", [0x0d12] = "\x2c\x74", [0x0d13] = "\x2c\x75",
  [0x0d14] = "\x2c\x76", [0x0d15] = "\x2c\x77", [0x0d16] = "\x2c\x78",
  [0x0d17] = "\x2c\x79", [0x0d18] = "\x2c\x7a", [0x0d19] = "\x2c\x7b",
  [0x0d1a] = "\x2c\x7c", [0x0d1b] = "\x2c\x7d", [0x0d1c] = "\x2c\x7e",
  [0x0d1d] = "\x2c\x7f", [0x0d1e] = "\x2c\x80", [0x0d1f] = "\x2c\x81",
  [0x0d20] = "\x2c\x82", [0x0d21] = "\x2c\x83", [0x0d22] = "\x2c\x84",
  [0x0d23] = "\x2c\x85", [0x0d24] = "\x2c\x86", [0x0d25] = "\x2c\x87",
  [0x0d26] = "\x2c\x88", [0x0d27] = "\x2c\x89", [0x0d28] = "\x2c\x8a",
  [0x0d29] = "\x2c\x8b", [0x0d2a] = "\x2c\x8c", [0x0d2b] = "\x2c\x8d",
  [0x0d2c] = "\x2c\x8e", [0x0d2d] = "\x2c\x8f", [0x0d2e] = "\x2c\x90",
  [0x0d2f] = "\x2c\x91", [0x0d30] = "\x2c\x92", [0x0d31] = "\x2c\x93",
  [0x0d32] = "\x2c\x94", [0x0d33] = "\x2c\x95", [0x0d34] = "\x2c\x96",
  [0x0d35] = "\x2c\x97", [0x0d36] = "\x2c\x98", [0x0d37] = "\x2c\x99",
  [0x0d38] = "\x2c\x9a", [0x0d39] = "\x2c\x9b", [0x0d3a] = "\x2c\x9c",
  [0x0d3b] = "\x2c\x9d", [0x0d3c] = "\x2c\x9e", [0x0d3d] = "\x2c\x9f",
  [0x0d3e] = "\x2c\xa0", [0x0d3f] = "\x2c\xa1", [0x0d40] = "\x2c\xa2",
  [0x0d41] = "\x2c\xa3", [0x0d42] = "\x2c\xa4", [0x0d43] = "\x2c\xa5",
  [0x0d44] = "\x2c\xa6", [0x0d45] = "\x2c\xa7", [0x0d46] = "\x2c\xa8",
  [0x0d47] = "\x2c\xa9", [0x0d48] = "\x2c\xaa", [0x0d49] = "\x2c\xab",
  [0x0d4a] = "\x2c\xac", [0x0d4b] = "\x2c\xad", [0x0d4c] = "\x2c\xae",
  [0x0d4d] = "\x2c\xaf", [0x0d4e] = "\x2c\xb0", [0x0d4f] = "\x2c\xb1",
  [0x0d50] = "\x2c\xb2", [0x0d51] = "\x2c\xb3", [0x0d52] = "\x2c\xb4",
  [0x0d53] = "\x2c\xb5", [0x0d54] = "\x2c\xb6", [0x0d55] = "\x2c\xb7",
  [0x0d56] = "\x2c\xb8", [0x0d57] = "\x2c\xb9", [0x0d58] = "\x2c\xba",
  [0x0d59] = "\x2c\xbb", [0x0d5a] = "\x2c\xbc", [0x0d5b] = "\x2c\xbd",
  [0x0d5c] = "\x2c\xbe", [0x0d5d] = "\x2c\xbf", [0x0d5e] = "\x2c\xc0",
  [0x0d5f] = "\x2c\xc1", [0x0d60] = "\x2c\xc2", [0x0d61] = "\x2c\xc3",
  [0x0d62] = "\x2c\xc4", [0x0d63] = "\x2c\xc5", [0x0d64] = "\x2c\xc6",
  [0x0d65] = "\x2c\xc7", [0x0d66] = "\x2c\xc8", [0x0d67] = "\x2c\xc9",
  [0x0d68] = "\x2c\xca", [0x0d69] = "\x2c\xcb", [0x0d6a] = "\x2c\xcc",
  [0x0d6b] = "\x2c\xcd", [0x0d6c] = "\x2c\xce", [0x0d6d] = "\x2c\xcf",
  [0x0d6e] = "\x2c\xd0", [0x0d6f] = "\x2c\xd1", [0x0d70] = "\x2c\xd2",
  [0x0d71] = "\x2c\xd3", [0x0d72] = "\x2c\xd4", [0x0d73] = "\x2c\xd5",
  [0x0d74] = "\x2c\xd6", [0x0d75] = "\x2c\xd7", [0x0d76] = "\x2c\xd8",
  [0x0d77] = "\x2c\xd9", [0x0d78] = "\x2c\xda", [0x0d79] = "\x2c\xdb",
  [0x0d7a] = "\x2c\xdc", [0x0d7b] = "\x2c\xdd", [0x0d7c] = "\x2c\xde",
  [0x0d7d] = "\x2c\xdf", [0x0d7e] = "\x2c\xe0", [0x0d7f] = "\x2c\xe1",
  [0x0d80] = "\x2c\xe2", [0x0d81] = "\x2c\xe3", [0x0d82] = "\x2c\xe4",
  [0x0d83] = "\x2c\xe5", [0x0d84] = "\x2c\xe6", [0x0d85] = "\x2c\xe7",
  [0x0d86] = "\x2c\xe8", [0x0d87] = "\x2c\xe9", [0x0d88] = "\x2c\xea",
  [0x0d89] = "\x2c\xeb", [0x0d8a] = "\x2c\xec", [0x0d8b] = "\x2c\xed",
  [0x0d8c] = "\x2c\xee", [0x0d8d] = "\x2c\xef", [0x0d8e] = "\x2c\xf0",
  [0x0d8f] = "\x2c\xf1", [0x0d90] = "\x2c\xf2", [0x0d91] = "\x2c\xf3",
  [0x0d92] = "\x2c\xf4", [0x0d93] = "\x2c\xf5", [0x0d94] = "\x2c\xf6",
  [0x0d95] = "\x2c\xf7", [0x0d96] = "\x2c\xf8", [0x0d97] = "\x2c\xf9",
  [0x0d98] = "\x2c\xfa", [0x0d99] = "\x2c\xfb", [0x0d9a] = "\x2c\xfc",
  [0x0d9b] = "\x2c\xfd", [0x0d9c] = "\x2c\xfe", [0x0d9d] = "\x2c\xff",
  [0x0d9e] = "\x2d\x00", [0x0d9f] = "\x2d\x01", [0x0da0] = "\x2d\x02",
  [0x0da1] = "\x2d\x03", [0x0da2] = "\x2d\x04", [0x0da3] = "\x2d\x05",
  [0x0da4] = "\x2d\x06", [0x0da5] = "\x2d\x07", [0x0da6] = "\x2d\x08",
  [0x0da7] = "\x2d\x09", [0x0da8] = "\x2d\x0a", [0x0da9] = "\x2d\x0b",
  [0x0daa] = "\x2d\x0c", [0x0dab] = "\x2d\x0d", [0x0dac] = "\x2d\x0e",
  [0x0dad] = "\x2d\x0f", [0x0dae] = "\x2d\x10", [0x0daf] = "\x2d\x11",
  [0x0db0] = "\x2d\x12", [0x0db1] = "\x2d\x13", [0x0db2] = "\x2d\x14",
  [0x0db3] = "\x2d\x15", [0x0db4] = "\x2d\x16", [0x0db5] = "\x2d\x17",
  [0x0db6] = "\x2d\x18", [0x0db7] = "\x2d\x19", [0x0db8] = "\x2d\x1a",
  [0x0db9] = "\x2d\x1b", [0x0dba] = "\x2d\x1c", [0x0dbb] = "\x2d\x1d",
  [0x0dbc] = "\x2d\x1e", [0x0dbd] = "\x2d\x1f", [0x0dbe] = "\x2d\x20",
  [0x0dbf] = "\x2d\x21", [0x0dc0] = "\x2d\x22", [0x0dc1] = "\x2d\x23",
  [0x0dc2] = "\x2d\x24", [0x0dc3] = "\x2d\x25", [0x0dc4] = "\x2d\x26",
  [0x0dc5] = "\x2d\x27", [0x0dc6] = "\x2d\x28", [0x0dc7] = "\x2d\x29",
  [0x0dc8] = "\x2d\x2a", [0x0dc9] = "\x2d\x2b", [0x0dca] = "\x2d\x2c",
  [0x0dcb] = "\x2d\x2d", [0x0dcc] = "\x2d\x2e", [0x0dcd] = "\x2d\x2f",
  [0x0dce] = "\x2d\x30", [0x0dcf] = "\x2d\x31", [0x0dd0] = "\x2d\x32",
  [0x0dd1] = "\x2d\x33", [0x0dd2] = "\x2d\x34", [0x0dd3] = "\x2d\x35",
  [0x0dd4] = "\x2d\x36", [0x0dd5] = "\x2d\x37", [0x0dd6] = "\x2d\x38",
  [0x0dd7] = "\x2d\x39", [0x0dd8] = "\x2d\x3a", [0x0dd9] = "\x2d\x3b",
  [0x0dda] = "\x2d\x3c", [0x0ddb] = "\x2d\x3d", [0x0ddc] = "\x2d\x3e",
  [0x0ddd] = "\x2d\x3f", [0x0dde] = "\x2d\x40", [0x0ddf] = "\x2d\x41",
  [0x0de0] = "\x2d\x42", [0x0de1] = "\x2d\x43", [0x0de2] = "\x2d\x44",
  [0x0de3] = "\x2d\x45", [0x0de4] = "\x2d\x46", [0x0de5] = "\x2d\x47",
  [0x0de6] = "\x2d\x48", [0x0de7] = "\x2d\x49", [0x0de8] = "\x2d\x4a",
  [0x0de9] = "\x2d\x4b", [0x0dea] = "\x2d\x4c", [0x0deb] = "\x2d\x4d",
  [0x0dec] = "\x2d\x4e", [0x0ded] = "\x2d\x4f", [0x0dee] = "\x2d\x50",
  [0x0def] = "\x2d\x51", [0x0df0] = "\x2d\x52", [0x0df1] = "\x2d\x53",
  [0x0df2] = "\x2d\x54", [0x0df3] = "\x2d\x55", [0x0df4] = "\x2d\x56",
  [0x0df5] = "\x2d\x57", [0x0df6] = "\x2d\x58", [0x0df7] = "\x2d\x59",
  [0x0df8] = "\x2d\x5a", [0x0df9] = "\x2d\x5b", [0x0dfa] = "\x2d\x5c",
  [0x0dfb] = "\x2d\x5d", [0x0dfc] = "\x2d\x5e", [0x0dfd] = "\x2d\x5f",
  [0x0dfe] = "\x2d\x60", [0x0dff] = "\x2d\x61", [0x0e00] = "\x2d\x62",
  [0x0e01] = "\x2d\x63", [0x0e02] = "\x2d\x64", [0x0e03] = "\x2d\x65",
  [0x0e04] = "\x2d\x66", [0x0e05] = "\x2d\x67", [0x0e06] = "\x2d\x68",
  [0x0e07] = "\x2d\x69", [0x0e08] = "\x2d\x6a", [0x0e09] = "\x2d\x6b",
  [0x0e0a] = "\x2d\x6c", [0x0e0b] = "\x2d\x6d", [0x0e0c] = "\x2d\x6e",
  [0x0e0d] = "\x2d\x6f", [0x0e0e] = "\x2d\x70", [0x0e0f] = "\x2d\x71",
  [0x0e10] = "\x2d\x72", [0x0e11] = "\x2d\x73", [0x0e12] = "\x2d\x74",
  [0x0e13] = "\x2d\x75", [0x0e14] = "\x2d\x76", [0x0e15] = "\x2d\x77",
  [0x0e16] = "\x2d\x78", [0x0e17] = "\x2d\x79", [0x0e18] = "\x2d\x7a",
  [0x0e19] = "\x2d\x7b", [0x0e1a] = "\x2d\x7c", [0x0e1b] = "\x2d\x7d",
  [0x0e1c] = "\x2d\x7e", [0x0e1d] = "\x2d\x7f", [0x0e1e] = "\x2d\x80",
  [0x0e1f] = "\x2d\x81", [0x0e20] = "\x2d\x82", [0x0e21] = "\x2d\x83",
  [0x0e22] = "\x2d\x84", [0x0e23] = "\x2d\x85", [0x0e24] = "\x2d\x86",
  [0x0e25] = "\x2d\x87", [0x0e26] = "\x2d\x88", [0x0e27] = "\x2d\x89",
  [0x0e28] = "\x2d\x8a", [0x0e29] = "\x2d\x8b", [0x0e2a] = "\x2d\x8c",
  [0x0e2b] = "\x2d\x8d", [0x0e2c] = "\x2d\x8e", [0x0e2d] = "\x2d\x8f",
  [0x0e2e] = "\x2d\x90", [0x0e2f] = "\x2d\x91", [0x0e30] = "\x2d\x92",
  [0x0e31] = "\x2d\x93", [0x0e32] = "\x2d\x94", [0x0e33] = "\x2d\x95",
  [0x0e34] = "\x2d\x96", [0x0e35] = "\x2d\x97", [0x0e36] = "\x2d\x98",
  [0x0e37] = "\x2d\x99", [0x0e38] = "\x2d\x9a", [0x0e39] = "\x2d\x9b",
  [0x0e3a] = "\x2d\x9c", [0x0e3b] = "\x2d\x9d", [0x0e3c] = "\x2d\x9e",
  [0x0e3d] = "\x2d\x9f", [0x0e3e] = "\x2d\xa0", [0x0e3f] = "\x2d\xa1",
  [0x0e40] = "\x2d\xa2", [0x0e41] = "\x2d\xa3", [0x0e42] = "\x2d\xa4",
  [0x0e43] = "\x2d\xa5", [0x0e44] = "\x2d\xa6", [0x0e45] = "\x2d\xa7",
  [0x0e46] = "\x2d\xa8", [0x0e47] = "\x2d\xa9", [0x0e48] = "\x2d\xaa",
  [0x0e49] = "\x2d\xab", [0x0e4a] = "\x2d\xac", [0x0e4b] = "\x2d\xad",
  [0x0e4c] = "\x2d\xae", [0x0e4d] = "\x2d\xaf", [0x0e4e] = "\x2d\xb0",
  [0x0e4f] = "\x2d\xb1", [0x0e50] = "\x2d\xb2", [0x0e51] = "\x2d\xb3",
  [0x0e52] = "\x2d\xb4", [0x0e53] = "\x2d\xb5", [0x0e54] = "\x2d\xb6",
  [0x0e55] = "\x2d\xb7", [0x0e56] = "\x2d\xb8", [0x0e57] = "\x2d\xb9",
  [0x0e58] = "\x2d\xba", [0x0e59] = "\x2d\xbb", [0x0e5a] = "\x2d\xbc",
  [0x0e5b] = "\x2d\xbd", [0x0e5c] = "\x2d\xbe", [0x0e5d] = "\x2d\xbf",
  [0x0e5e] = "\x2d\xc0", [0x0e5f] = "\x2d\xc1", [0x0e60] = "\x2d\xc2",
  [0x0e61] = "\x2d\xc3", [0x0e62] = "\x2d\xc4", [0x0e63] = "\x2d\xc5",
  [0x0e64] = "\x2d\xc6", [0x0e65] = "\x2d\xc7", [0x0e66] = "\x2d\xc8",
  [0x0e67] = "\x2d\xc9", [0x0e68] = "\x2d\xca", [0x0e69] = "\x2d\xcb",
  [0x0e6a] = "\x2d\xcc", [0x0e6b] = "\x2d\xcd", [0x0e6c] = "\x2d\xce",
  [0x0e6d] = "\x2d\xcf", [0x0e6e] = "\x2d\xd0", [0x0e6f] = "\x2d\xd1",
  [0x0e70] = "\x2d\xd2", [0x0e71] = "\x2d\xd3", [0x0e72] = "\x2d\xd4",
  [0x0e73] = "\x2d\xd5", [0x0e74] = "\x2d\xd6", [0x0e75] = "\x2d\xd7",
  [0x0e76] = "\x2d\xd8", [0x0e77] = "\x2d\xd9", [0x0e78] = "\x2d\xda",
  [0x0e79] = "\x2d\xdb", [0x0e7a] = "\x2d\xdc", [0x0e7b] = "\x2d\xdd",
  [0x0e7c] = "\x2d\xde", [0x0e7d] = "\x2d\xdf", [0x0e7e] = "\x2d\xe0",
  [0x0e7f] = "\x2d\xe1", [0x0e80] = "\x2d\xe2", [0x0e81] = "\x2d\xe3",
  [0x0e82] = "\x2d\xe4", [0x0e83] = "\x2d\xe5", [0x0e84] = "\x2d\xe6",
  [0x0e85] = "\x2d\xe7", [0x0e86] = "\x2d\xe8", [0x0e87] = "\x2d\xe9",
  [0x0e88] = "\x2d\xea", [0x0e89] = "\x2d\xeb", [0x0e8a] = "\x2d\xec",
  [0x0e8b] = "\x2d\xed", [0x0e8c] = "\x2d\xee", [0x0e8d] = "\x2d\xef",
  [0x0e8e] = "\x2d\xf0", [0x0e8f] = "\x2d\xf1", [0x0e90] = "\x2d\xf2",
  [0x0e91] = "\x2d\xf3", [0x0e92] = "\x2d\xf4", [0x0e93] = "\x2d\xf5",
  [0x0e94] = "\x2d\xf6", [0x0e95] = "\x2d\xf7", [0x0e96] = "\x2d\xf8",
  [0x0e97] = "\x2d\xf9", [0x0e98] = "\x2d\xfa", [0x0e99] = "\x2d\xfb",
  [0x0e9a] = "\x2d\xfc", [0x0e9b] = "\x2d\xfd", [0x0e9c] = "\x2d\xfe",
  [0x0e9d] = "\x2d\xff", [0x0e9e] = "\x2e\x00", [0x0e9f] = "\x2e\x01",
  [0x0ea0] = "\x2e\x02", [0x0ea1] = "\x2e\x03", [0x0ea2] = "\x2e\x04",
  [0x0ea3] = "\x2e\x05", [0x0ea4] = "\x2e\x06", [0x0ea5] = "\x2e\x07",
  [0x0ea6] = "\x2e\x08", [0x0ea7] = "\x2e\x09", [0x0ea8] = "\x2e\x0a",
  [0x0ea9] = "\x2e\x0b", [0x0eaa] = "\x2e\x0c", [0x0eab] = "\x2e\x0d",
  [0x0eac] = "\x2e\x0e", [0x0ead] = "\x2e\x0f", [0x0eae] = "\x2e\x10",
  [0x0eaf] = "\x2e\x11", [0x0eb0] = "\x2e\x12", [0x0eb1] = "\x2e\x13",
  [0x0eb2] = "\x2e\x14", [0x0eb3] = "\x2e\x15", [0x0eb4] = "\x2e\x16",
  [0x0eb5] = "\x2e\x17", [0x0eb6] = "\x2e\x18", [0x0eb7] = "\x2e\x19",
  [0x0eb8] = "\x2e\x1a", [0x0eb9] = "\x2e\x1b", [0x0eba] = "\x2e\x1c",
  [0x0ebb] = "\x2e\x1d", [0x0ebc] = "\x2e\x1e", [0x0ebd] = "\x2e\x1f",
  [0x0ebe] = "\x2e\x20", [0x0ebf] = "\x2e\x21", [0x0ec0] = "\x2e\x22",
  [0x0ec1] = "\x2e\x23", [0x0ec2] = "\x2e\x24", [0x0ec3] = "\x2e\x25",
  [0x0ec4] = "\x2e\x26", [0x0ec5] = "\x2e\x27", [0x0ec6] = "\x2e\x28",
  [0x0ec7] = "\x2e\x29", [0x0ec8] = "\x2e\x2a", [0x0ec9] = "\x2e\x2b",
  [0x0eca] = "\x2e\x2c", [0x0ecb] = "\x2e\x2d", [0x0ecc] = "\x2e\x2e",
  [0x0ecd] = "\x2e\x2f", [0x0ece] = "\x2e\x30", [0x0ecf] = "\x2e\x31",
  [0x0ed0] = "\x2e\x32", [0x0ed1] = "\x2e\x33", [0x0ed2] = "\x2e\x34",
  [0x0ed3] = "\x2e\x35", [0x0ed4] = "\x2e\x36", [0x0ed5] = "\x2e\x37",
  [0x0ed6] = "\x2e\x38", [0x0ed7] = "\x2e\x39", [0x0ed8] = "\x2e\x3a",
  [0x0ed9] = "\x2e\x3b", [0x0eda] = "\x2e\x3c", [0x0edb] = "\x2e\x3d",
  [0x0edc] = "\x2e\x3e", [0x0edd] = "\x2e\x3f", [0x0ede] = "\x2e\x40",
  [0x0edf] = "\x2e\x41", [0x0ee0] = "\x2e\x42", [0x0ee1] = "\x2e\x43",
  [0x0ee2] = "\x2e\x44", [0x0ee3] = "\x2e\x45", [0x0ee4] = "\x2e\x46",
  [0x0ee5] = "\x2e\x47", [0x0ee6] = "\x2e\x48", [0x0ee7] = "\x2e\x49",
  [0x0ee8] = "\x2e\x4a", [0x0ee9] = "\x2e\x4b", [0x0eea] = "\x2e\x4c",
  [0x0eeb] = "\x2e\x4d", [0x0eec] = "\x2e\x4e", [0x0eed] = "\x2e\x4f",
  [0x0eee] = "\x2e\x50", [0x0eef] = "\x2e\x51", [0x0ef0] = "\x2e\x52",
  [0x0ef1] = "\x2e\x53", [0x0ef2] = "\x2e\x54", [0x0ef3] = "\x2e\x55",
  [0x0ef4] = "\x2e\x56", [0x0ef5] = "\x2e\x57", [0x0ef6] = "\x2e\x58",
  [0x0ef7] = "\x2e\x59", [0x0ef8] = "\x2e\x5a", [0x0ef9] = "\x2e\x5b",
  [0x0efa] = "\x2e\x5c", [0x0efb] = "\x2e\x5d", [0x0efc] = "\x2e\x5e",
  [0x0efd] = "\x2e\x5f", [0x0efe] = "\x2e\x60", [0x0eff] = "\x2e\x61",
  [0x0f00] = "\x2e\x62", [0x0f01] = "\x2e\x63", [0x0f02] = "\x2e\x64",
  [0x0f03] = "\x2e\x65", [0x0f04] = "\x2e\x66", [0x0f05] = "\x2e\x67",
  [0x0f06] = "\x2e\x68", [0x0f07] = "\x2e\x69", [0x0f08] = "\x2e\x6a",
  [0x0f09] = "\x2e\x6b", [0x0f0a] = "\x2e\x6c", [0x0f0b] = "\x2e\x6d",
  [0x0f0c] = "\x2e\x6e", [0x0f0d] = "\x2e\x6f", [0x0f0e] = "\x2e\x70",
  [0x0f0f] = "\x2e\x71", [0x0f10] = "\x2e\x72", [0x0f11] = "\x2e\x73",
  [0x0f12] = "\x2e\x74", [0x0f13] = "\x2e\x75", [0x0f14] = "\x2e\x76",
  [0x0f15] = "\x2e\x77", [0x0f16] = "\x2e\x78", [0x0f17] = "\x2e\x79",
  [0x0f18] = "\x2e\x7a", [0x0f19] = "\x2e\x7b", [0x0f1a] = "\x2e\x7c",
  [0x0f1b] = "\x2e\x7d", [0x0f1c] = "\x2e\x7e", [0x0f1d] = "\x2e\x7f",
  [0x0f1e] = "\x2e\x80", [0x0f1f] = "\x2e\x81", [0x0f20] = "\x2e\x82",
  [0x0f21] = "\x2e\x83", [0x0f22] = "\x2e\x84", [0x0f23] = "\x2e\x85",
  [0x0f24] = "\x2e\x86", [0x0f25] = "\x2e\x87", [0x0f26] = "\x2e\x88",
  [0x0f27] = "\x2e\x89", [0x0f28] = "\x2e\x8a", [0x0f29] = "\x2e\x8b",
  [0x0f2a] = "\x2e\x8c", [0x0f2b] = "\x2e\x8d", [0x0f2c] = "\x2e\x8e",
  [0x0f2d] = "\x2e\x8f", [0x0f2e] = "\x2e\x90", [0x0f2f] = "\x2e\x91",
  [0x0f30] = "\x2e\x92", [0x0f31] = "\x2e\x93", [0x0f32] = "\x2e\x94",
  [0x0f33] = "\x2e\x95", [0x0f34] = "\x2e\x96", [0x0f35] = "\x2e\x97",
  [0x0f36] = "\x2e\x98", [0x0f37] = "\x2e\x99", [0x0f38] = "\x2e\x9a",
  [0x0f39] = "\x2e\x9b", [0x0f3a] = "\x2e\x9c", [0x0f3b] = "\x2e\x9d",
  [0x0f3c] = "\x2e\x9e", [0x0f3d] = "\x2e\x9f", [0x0f3e] = "\x2e\xa0",
  [0x0f3f] = "\x2e\xa1", [0x0f40] = "\x2e\xa2", [0x0f41] = "\x2e\xa3",
  [0x0f42] = "\x2e\xa4", [0x0f43] = "\x2e\xa5", [0x0f44] = "\x2e\xa6",
  [0x0f45] = "\x2e\xa7", [0x0f46] = "\x2e\xa8", [0x0f47] = "\x2e\xa9",
  [0x0f48] = "\x2e\xaa", [0x0f49] = "\x2e\xab", [0x0f4a] = "\x2e\xac",
  [0x0f4b] = "\x2e\xad", [0x0f4c] = "\x2e\xae", [0x0f4d] = "\x2e\xaf",
  [0x0f4e] = "\x2e\xb0", [0x0f4f] = "\x2e\xb1", [0x0f50] = "\x2e\xb2",
  [0x0f51] = "\x2e\xb3", [0x0f52] = "\x2e\xb4", [0x0f53] = "\x2e\xb5",
  [0x0f54] = "\x2e\xb6", [0x0f55] = "\x2e\xb7", [0x0f56] = "\x2e\xb8",
  [0x0f57] = "\x2e\xb9", [0x0f58] = "\x2e\xba", [0x0f59] = "\x2e\xbb",
  [0x0f5a] = "\x2e\xbc", [0x0f5b] = "\x2e\xbd", [0x0f5c] = "\x2e\xbe",
  [0x0f5d] = "\x2e\xbf", [0x0f5e] = "\x2e\xc0", [0x0f5f] = "\x2e\xc1",
  [0x0f60] = "\x2e\xc2", [0x0f61] = "\x2e\xc3", [0x0f62] = "\x2e\xc4",
  [0x0f63] = "\x2e\xc5", [0x0f64] = "\x2e\xc6", [0x0f65] = "\x2e\xc7",
  [0x0f66] = "\x2e\xc8", [0x0f67] = "\x2e\xc9", [0x0f68] = "\x2e\xca",
  [0x0f69] = "\x2e\xcb", [0x0f6a] = "\x2e\xcc", [0x0f6b] = "\x2e\xcd",
  [0x0f6c] = "\x2e\xce", [0x0f6d] = "\x2e\xcf", [0x0f6e] = "\x2e\xd0",
  [0x0f6f] = "\x2e\xd1", [0x0f70] = "\x2e\xd2", [0x0f71] = "\x2e\xd3",
  [0x0f72] = "\x2e\xd4", [0x0f73] = "\x2e\xd5", [0x0f74] = "\x2e\xd6",
  [0x0f75] = "\x2e\xd7", [0x0f76] = "\x2e\xd8", [0x0f77] = "\x2e\xd9",
  [0x0f78] = "\x2e\xda", [0x0f79] = "\x2e\xdb", [0x0f7a] = "\x2e\xdc",
  [0x0f7b] = "\x2e\xdd", [0x0f7c] = "\x2e\xde", [0x0f7d] = "\x2e\xdf",
  [0x0f7e] = "\x2e\xe0", [0x0f7f] = "\x2e\xe1", [0x0f80] = "\x2e\xe2",
  [0x0f81] = "\x2e\xe3", [0x0f82] = "\x2e\xe4", [0x0f83] = "\x2e\xe5",
  [0x0f84] = "\x2e\xe6", [0x0f85] = "\x2e\xe7", [0x0f86] = "\x2e\xe8",
  [0x0f87] = "\x2e\xe9", [0x0f88] = "\x2e\xea", [0x0f89] = "\x2e\xeb",
  [0x0f8a] = "\x2e\xec", [0x0f8b] = "\x2e\xed", [0x0f8c] = "\x2e\xee",
  [0x0f8d] = "\x2e\xef", [0x0f8e] = "\x2e\xf0", [0x0f8f] = "\x2e\xf1",
  [0x0f90] = "\x2e\xf2", [0x0f91] = "\x2e\xf3", [0x0f92] = "\x2e\xf4",
  [0x0f93] = "\x2e\xf5", [0x0f94] = "\x2e\xf6", [0x0f95] = "\x2e\xf7",
  [0x0f96] = "\x2e\xf8", [0x0f97] = "\x2e\xf9", [0x0f98] = "\x2e\xfa",
  [0x0f99] = "\x2e\xfb", [0x0f9a] = "\x2e\xfc", [0x0f9b] = "\x2e\xfd",
  [0x0f9c] = "\x2e\xfe", [0x0f9d] = "\x2e\xff", [0x0f9e] = "\x2f\x00",
  [0x0f9f] = "\x2f\x01", [0x0fa0] = "\x2f\x02", [0x0fa1] = "\x2f\x03",
  [0x0fa2] = "\x2f\x04", [0x0fa3] = "\x2f\x05", [0x0fa4] = "\x2f\x06",
  [0x0fa5] = "\x2f\x07", [0x0fa6] = "\x2f\x08", [0x0fa7] = "\x2f\x09",
  [0x0fa8] = "\x2f\x0a", [0x0fa9] = "\x2f\x0b", [0x0faa] = "\x2f\x0c",
  [0x0fab] = "\x2f\x0d", [0x0fac] = "\x2f\x0e", [0x0fad] = "\x2f\x0f",
  [0x0fae] = "\x2f\x10", [0x0faf] = "\x2f\x11", [0x0fb0] = "\x2f\x12",
  [0x0fb1] = "\x2f\x13", [0x0fb2] = "\x2f\x14", [0x0fb3] = "\x2f\x15",
  [0x0fb4] = "\x2f\x16", [0x0fb5] = "\x2f\x17", [0x0fb6] = "\x2f\x18",
  [0x0fb7] = "\x2f\x19", [0x0fb8] = "\x2f\x1a", [0x0fb9] = "\x2f\x1b",
  [0x0fba] = "\x2f\x1c", [0x0fbb] = "\x2f\x1d", [0x0fbc] = "\x2f\x1e",
  [0x0fbd] = "\x2f\x1f", [0x0fbe] = "\x2f\x20", [0x0fbf] = "\x2f\x21",
  [0x0fc0] = "\x2f\x22", [0x0fc1] = "\x2f\x23", [0x0fc2] = "\x2f\x24",
  [0x0fc3] = "\x2f\x25", [0x0fc4] = "\x2f\x26", [0x0fc5] = "\x2f\x27",
  [0x0fc6] = "\x2f\x28", [0x0fc7] = "\x2f\x29", [0x0fc8] = "\x2f\x2a",
  [0x0fc9] = "\x2f\x2b", [0x0fca] = "\x2f\x2c", [0x0fcb] = "\x2f\x2d",
  [0x0fcc] = "\x2f\x2e", [0x0fcd] = "\x2f\x2f", [0x0fce] = "\x2f\x30",
  [0x0fcf] = "\x2f\x31", [0x0fd0] = "\x2f\x32", [0x0fd1] = "\x2f\x33",
  [0x0fd2] = "\x2f\x34", [0x0fd3] = "\x2f\x35", [0x0fd4] = "\x2f\x36",
  [0x0fd5] = "\x2f\x37", [0x0fd6] = "\x2f\x38", [0x0fd7] = "\x2f\x39",
  [0x0fd8] = "\x2f\x3a", [0x0fd9] = "\x2f\x3b", [0x0fda] = "\x2f\x3c",
  [0x0fdb] = "\x2f\x3d", [0x0fdc] = "\x2f\x3e", [0x0fdd] = "\x2f\x3f",
  [0x0fde] = "\x2f\x40", [0x0fdf] = "\x2f\x41", [0x0fe0] = "\x2f\x42",
  [0x0fe1] = "\x2f\x43", [0x0fe2] = "\x2f\x44", [0x0fe3] = "\x2f\x45",
  [0x0fe4] = "\x2f\x46", [0x0fe5] = "\x2f\x47", [0x0fe6] = "\x2f\x48",
  [0x0fe7] = "\x2f\x49", [0x0fe8] = "\x2f\x4a", [0x0fe9] = "\x2f\x4b",
  [0x0fea] = "\x2f\x4c", [0x0feb] = "\x2f\x4d", [0x0fec] = "\x2f\x4e",
  [0x0fed] = "\x2f\x4f", [0x0fee] = "\x2f\x50", [0x0fef] = "\x2f\x51",
  [0x0ff0] = "\x2f\x52", [0x0ff1] = "\x2f\x53", [0x0ff2] = "\x2f\x54",
  [0x0ff3] = "\x2f\x55", [0x0ff4] = "\x2f\x56", [0x0ff5] = "\x2f\x57",
  [0x0ff6] = "\x2f\x58", [0x0ff7] = "\x2f\x59", [0x0ff8] = "\x2f\x5a",
  [0x0ff9] = "\x2f\x5b", [0x0ffa] = "\x2f\x5c", [0x0ffb] = "\x2f\x5d",
  [0x0ffc] = "\x2f\x5e", [0x0ffd] = "\x2f\x5f", [0x0ffe] = "\x2f\x60",
  [0x0fff] = "\x2f\x61", [0x1000] = "\x2f\x62", [0x1001] = "\x2f\x63",
  [0x1002] = "\x2f\x64", [0x1003] = "\x2f\x65", [0x1004] = "\x2f\x66",
  [0x1005] = "\x2f\x67", [0x1006] = "\x2f\x68", [0x1007] = "\x2f\x69",
  [0x1008] = "\x2f\x6a", [0x1009] = "\x2f\x6b", [0x100a] = "\x2f\x6c",
  [0x100b] = "\x2f\x6d", [0x100c] = "\x2f\x6e", [0x100d] = "\x2f\x6f",
  [0x100e] = "\x2f\x70", [0x100f] = "\x2f\x71", [0x1010] = "\x2f\x72",
  [0x1011] = "\x2f\x73", [0x1012] = "\x2f\x74", [0x1013] = "\x2f\x75",
  [0x1014] = "\x2f\x76", [0x1015] = "\x2f\x77", [0x1016] = "\x2f\x78",
  [0x1017] = "\x2f\x79", [0x1018] = "\x2f\x7a", [0x1019] = "\x2f\x7b",
  [0x101a] = "\x2f\x7c", [0x101b] = "\x2f\x7d", [0x101c] = "\x2f\x7e",
  [0x101d] = "\x2f\x7f", [0x101e] = "\x2f\x80", [0x101f] = "\x2f\x81",
  [0x1020] = "\x2f\x82", [0x1021] = "\x2f\x83", [0x1022] = "\x2f\x84",
  [0x1023] = "\x2f\x85", [0x1024] = "\x2f\x86", [0x1025] = "\x2f\x87",
  [0x1026] = "\x2f\x88", [0x1027] = "\x2f\x89", [0x1028] = "\x2f\x8a",
  [0x1029] = "\x2f\x8b", [0x102a] = "\x2f\x8c", [0x102b] = "\x2f\x8d",
  [0x102c] = "\x2f\x8e", [0x102d] = "\x2f\x8f", [0x102e] = "\x2f\x90",
  [0x102f] = "\x2f\x91", [0x1030] = "\x2f\x92", [0x1031] = "\x2f\x93",
  [0x1032] = "\x2f\x94", [0x1033] = "\x2f\x95", [0x1034] = "\x2f\x96",
  [0x1035] = "\x2f\x97", [0x1036] = "\x2f\x98", [0x1037] = "\x2f\x99",
  [0x1038] = "\x2f\x9a", [0x1039] = "\x2f\x9b", [0x103a] = "\x2f\x9c",
  [0x103b] = "\x2f\x9d", [0x103c] = "\x2f\x9e", [0x103d] = "\x2f\x9f",
  [0x103e] = "\x2f\xa0", [0x103f] = "\x2f\xa1", [0x1040] = "\x2f\xa2",
  [0x1041] = "\x2f\xa3", [0x1042] = "\x2f\xa4", [0x1043] = "\x2f\xa5",
  [0x1044] = "\x2f\xa6", [0x1045] = "\x2f\xa7", [0x1046] = "\x2f\xa8",
  [0x1047] = "\x2f\xa9", [0x1048] = "\x2f\xaa", [0x1049] = "\x2f\xab",
  [0x104a] = "\x2f\xac", [0x104b] = "\x2f\xad", [0x104c] = "\x2f\xae",
  [0x104d] = "\x2f\xaf", [0x104e] = "\x2f\xb0", [0x104f] = "\x2f\xb1",
  [0x1050] = "\x2f\xb2", [0x1051] = "\x2f\xb3", [0x1052] = "\x2f\xb4",
  [0x1053] = "\x2f\xb5", [0x1054] = "\x2f\xb6", [0x1055] = "\x2f\xb7",
  [0x1056] = "\x2f\xb8", [0x1057] = "\x2f\xb9", [0x1058] = "\x2f\xba",
  [0x1059] = "\x2f\xbb", [0x105a] = "\x2f\xbc", [0x105b] = "\x2f\xbd",
  [0x105c] = "\x2f\xbe", [0x105d] = "\x2f\xbf", [0x105e] = "\x2f\xc0",
  [0x105f] = "\x2f\xc1", [0x1060] = "\x2f\xc2", [0x1061] = "\x2f\xc3",
  [0x1062] = "\x2f\xc4", [0x1063] = "\x2f\xc5", [0x1064] = "\x2f\xc6",
  [0x1065] = "\x2f\xc7", [0x1066] = "\x2f\xc8", [0x1067] = "\x2f\xc9",
  [0x1068] = "\x2f\xca", [0x1069] = "\x2f\xcb", [0x106a] = "\x2f\xcc",
  [0x106b] = "\x2f\xcd", [0x106c] = "\x2f\xce", [0x106d] = "\x2f\xcf",
  [0x106e] = "\x2f\xd0", [0x106f] = "\x2f\xd1", [0x1070] = "\x2f\xd2",
  [0x1071] = "\x2f\xd3", [0x1072] = "\x2f\xd4", [0x1073] = "\x2f\xd5",
  [0x1074] = "\x2f\xd6", [0x1075] = "\x2f\xd7", [0x1076] = "\x2f\xd8",
  [0x1077] = "\x2f\xd9", [0x1078] = "\x2f\xda", [0x1079] = "\x2f\xdb",
  [0x107a] = "\x2f\xdc", [0x107b] = "\x2f\xdd", [0x107c] = "\x2f\xde",
  [0x107d] = "\x2f\xdf", [0x107e] = "\x2f\xe0", [0x107f] = "\x2f\xe1",
  [0x1080] = "\x2f\xe2", [0x1081] = "\x2f\xe3", [0x1082] = "\x2f\xe4",
  [0x1083] = "\x2f\xe5", [0x1084] = "\x2f\xe6", [0x1085] = "\x2f\xe7",
  [0x1086] = "\x2f\xe8", [0x1087] = "\x2f\xe9", [0x1088] = "\x2f\xea",
  [0x1089] = "\x2f\xeb", [0x108a] = "\x2f\xec", [0x108b] = "\x2f\xed",
  [0x108c] = "\x2f\xee", [0x108d] = "\x2f\xef", [0x108e] = "\x2f\xf0",
  [0x108f] = "\x2f\xf1", [0x1090] = "\x2f\xf2", [0x1091] = "\x2f\xf3",
  [0x1092] = "\x2f\xf4", [0x1093] = "\x2f\xf5", [0x1094] = "\x2f\xf6",
  [0x1095] = "\x2f\xf7", [0x1096] = "\x2f\xf8", [0x1097] = "\x2f\xf9",
  [0x1098] = "\x2f\xfa", [0x1099] = "\x2f\xfb", [0x109a] = "\x2f\xfc",
  [0x109b] = "\x2f\xfd", [0x109c] = "\x2f\xfe", [0x109d] = "\x2f\xff",
  [0x109e] = "\x30\x00", [0x109f] = "\x30\x01", [0x10a0] = "\x30\x02",
  [0x10a1] = "\x30\x03", [0x10a2] = "\x30\x04", [0x10a3] = "\x30\x05",
  [0x10a4] = "\x30\x06", [0x10a5] = "\x30\x07", [0x10a6] = "\x30\x08",
  [0x10a7] = "\x30\x09", [0x10a8] = "\x30\x0a", [0x10a9] = "\x30\x0b",
  [0x10aa] = "\x30\x0c", [0x10ab] = "\x30\x0d", [0x10ac] = "\x30\x0e",
  [0x10ad] = "\x30\x0f", [0x10ae] = "\x30\x10", [0x10af] = "\x30\x11",
  [0x10b0] = "\x30\x12", [0x10b1] = "\x30\x13", [0x10b2] = "\x30\x14",
  [0x10b3] = "\x30\x15", [0x10b4] = "\x30\x16", [0x10b5] = "\x30\x17",
  [0x10b6] = "\x30\x18", [0x10b7] = "\x30\x19", [0x10b8] = "\x30\x1a",
  [0x10b9] = "\x30\x1b", [0x10ba] = "\x30\x1c", [0x10bb] = "\x30\x1d",
  [0x10bc] = "\x30\x1e", [0x10bd] = "\x30\x1f", [0x10be] = "\x30\x20",
  [0x10bf] = "\x30\x21", [0x10c0] = "\x30\x22", [0x10c1] = "\x30\x23",
  [0x10c2] = "\x30\x24", [0x10c3] = "\x30\x25", [0x10c4] = "\x30\x26",
  [0x10c5] = "\x30\x27", [0x10c6] = "\x30\x28", [0x10c7] = "\x30\x29",
  [0x10c8] = "\x30\x2a", [0x10c9] = "\x30\x2b", [0x10ca] = "\x30\x2c",
  [0x10cb] = "\x30\x2d", [0x10cc] = "\x30\x2e", [0x10cd] = "\x30\x2f",
  [0x10ce] = "\x30\x30", [0x10cf] = "\x30\x31", [0x10d0] = "\x30\x32",
  [0x10d1] = "\x30\x33", [0x10d2] = "\x30\x34", [0x10d3] = "\x30\x35",
  [0x10d4] = "\x30\x36", [0x10d5] = "\x30\x37", [0x10d6] = "\x30\x38",
  [0x10d7] = "\x30\x39", [0x10d8] = "\x30\x3a", [0x10d9] = "\x30\x3b",
  [0x10da] = "\x30\x3c", [0x10db] = "\x30\x3d", [0x10dc] = "\x30\x3e",
  [0x10dd] = "\x30\x3f", [0x10de] = "\x30\x40", [0x10df] = "\x30\x41",
  [0x10e0] = "\x30\x42", [0x10e1] = "\x30\x43", [0x10e2] = "\x30\x44",
  [0x10e3] = "\x30\x45", [0x10e4] = "\x30\x46", [0x10e5] = "\x30\x47",
  [0x10e6] = "\x30\x48", [0x10e7] = "\x30\x49", [0x10e8] = "\x30\x4a",
  [0x10e9] = "\x30\x4b", [0x10ea] = "\x30\x4c", [0x10eb] = "\x30\x4d",
  [0x10ec] = "\x30\x4e", [0x10ed] = "\x30\x4f", [0x10ee] = "\x30\x50",
  [0x10ef] = "\x30\x51", [0x10f0] = "\x30\x52", [0x10f1] = "\x30\x53",
  [0x10f2] = "\x30\x54", [0x10f3] = "\x30\x55", [0x10f4] = "\x30\x56",
  [0x10f5] = "\x30\x57", [0x10f6] = "\x30\x58", [0x10f7] = "\x30\x59",
  [0x10f8] = "\x30\x5a", [0x10f9] = "\x30\x5b", [0x10fa] = "\x30\x5c",
  [0x10fb] = "\x30\x5d", [0x10fc] = "\x30\x5e", [0x10fd] = "\x30\x5f",
  [0x10fe] = "\x30\x60", [0x10ff] = "\x30\x61", [0x1100] = "\x30\x62",
  [0x1101] = "\x30\x63", [0x1102] = "\x30\x64", [0x1103] = "\x30\x65",
  [0x1104] = "\x30\x66", [0x1105] = "\x30\x67", [0x1106] = "\x30\x68",
  [0x1107] = "\x30\x69", [0x1108] = "\x30\x6a", [0x1109] = "\x30\x6b",
  [0x110a] = "\x30\x6c", [0x110b] = "\x30\x6d", [0x110c] = "\x30\x6e",
  [0x110d] = "\x30\x6f", [0x110e] = "\x30\x70", [0x110f] = "\x30\x71",
  [0x1110] = "\x30\x72", [0x1111] = "\x30\x73", [0x1112] = "\x30\x74",
  [0x1113] = "\x30\x75", [0x1114] = "\x30\x76", [0x1115] = "\x30\x77",
  [0x1116] = "\x30\x78", [0x1117] = "\x30\x79", [0x1118] = "\x30\x7a",
  [0x1119] = "\x30\x7b", [0x111a] = "\x30\x7c", [0x111b] = "\x30\x7d",
  [0x111c] = "\x30\x7e", [0x111d] = "\x30\x7f", [0x111e] = "\x30\x80",
  [0x111f] = "\x30\x81", [0x1120] = "\x30\x82", [0x1121] = "\x30\x83",
  [0x1122] = "\x30\x84", [0x1123] = "\x30\x85", [0x1124] = "\x30\x86",
  [0x1125] = "\x30\x87", [0x1126] = "\x30\x88", [0x1127] = "\x30\x89",
  [0x1128] = "\x30\x8a", [0x1129] = "\x30\x8b", [0x112a] = "\x30\x8c",
  [0x112b] = "\x30\x8d", [0x112c] = "\x30\x8e", [0x112d] = "\x30\x8f",
  [0x112e] = "\x30\x90", [0x112f] = "\x30\x91", [0x1130] = "\x30\x92",
  [0x1131] = "\x30\x93", [0x1132] = "\x30\x94", [0x1133] = "\x30\x95",
  [0x1134] = "\x30\x96", [0x1135] = "\x30\x97", [0x1136] = "\x30\x98",
  [0x1137] = "\x30\x99", [0x1138] = "\x30\x9a", [0x1139] = "\x30\x9b",
  [0x113a] = "\x30\x9c", [0x113b] = "\x30\x9d", [0x113c] = "\x30\x9e",
  [0x113d] = "\x30\x9f", [0x113e] = "\x30\xa0", [0x113f] = "\x30\xa1",
  [0x1140] = "\x30\xa2", [0x1141] = "\x30\xa3", [0x1142] = "\x30\xa4",
  [0x1143] = "\x30\xa5", [0x1144] = "\x30\xa6", [0x1145] = "\x30\xa7",
  [0x1146] = "\x30\xa8", [0x1147] = "\x30\xa9", [0x1148] = "\x30\xaa",
  [0x1149] = "\x30\xab", [0x114a] = "\x30\xac", [0x114b] = "\x30\xad",
  [0x114c] = "\x30\xae", [0x114d] = "\x30\xaf", [0x114e] = "\x30\xb0",
  [0x114f] = "\x30\xb1", [0x1150] = "\x30\xb2", [0x1151] = "\x30\xb3",
  [0x1152] = "\x30\xb4", [0x1153] = "\x30\xb5", [0x1154] = "\x30\xb6",
  [0x1155] = "\x30\xb7", [0x1156] = "\x30\xb8", [0x1157] = "\x30\xb9",
  [0x1158] = "\x30\xba", [0x1159] = "\x30\xbb", [0x115a] = "\x30\xbc",
  [0x115b] = "\x30\xbd", [0x115c] = "\x30\xbe", [0x115d] = "\x30\xbf",
  [0x115e] = "\x30\xc0", [0x115f] = "\x30\xc1", [0x1160] = "\x30\xc2",
  [0x1161] = "\x30\xc3", [0x1162] = "\x30\xc4", [0x1163] = "\x30\xc5",
  [0x1164] = "\x30\xc6", [0x1165] = "\x30\xc7", [0x1166] = "\x30\xc8",
  [0x1167] = "\x30\xc9", [0x1168] = "\x30\xca", [0x1169] = "\x30\xcb",
  [0x116a] = "\x30\xcc", [0x116b] = "\x30\xcd", [0x116c] = "\x30\xce",
  [0x116d] = "\x30\xcf", [0x116e] = "\x30\xd0", [0x116f] = "\x30\xd1",
  [0x1170] = "\x30\xd2", [0x1171] = "\x30\xd3", [0x1172] = "\x30\xd4",
  [0x1173] = "\x30\xd5", [0x1174] = "\x30\xd6", [0x1175] = "\x30\xd7",
  [0x1176] = "\x30\xd8", [0x1177] = "\x30\xd9", [0x1178] = "\x30\xda",
  [0x1179] = "\x30\xdb", [0x117a] = "\x30\xdc", [0x117b] = "\x30\xdd",
  [0x117c] = "\x30\xde", [0x117d] = "\x30\xdf", [0x117e] = "\x30\xe0",
  [0x117f] = "\x30\xe1", [0x1180] = "\x30\xe2", [0x1181] = "\x30\xe3",
  [0x1182] = "\x30\xe4", [0x1183] = "\x30\xe5", [0x1184] = "\x30\xe6",
  [0x1185] = "\x30\xe7", [0x1186] = "\x30\xe8", [0x1187] = "\x30\xe9",
  [0x1188] = "\x30\xea", [0x1189] = "\x30\xeb", [0x118a] = "\x30\xec",
  [0x118b] = "\x30\xed", [0x118c] = "\x30\xee", [0x118d] = "\x30\xef",
  [0x118e] = "\x30\xf0", [0x118f] = "\x30\xf1", [0x1190] = "\x30\xf2",
  [0x1191] = "\x30\xf3", [0x1192] = "\x30\xf4", [0x1193] = "\x30\xf5",
  [0x1194] = "\x30\xf6", [0x1195] = "\x30\xf7", [0x1196] = "\x30\xf8",
  [0x1197] = "\x30\xf9", [0x1198] = "\x30\xfa", [0x1199] = "\x30\xfb",
  [0x119a] = "\x30\xfc", [0x119b] = "\x30\xfd", [0x119c] = "\x30\xfe",
  [0x119d] = "\x30\xff", [0x119e] = "\x31\x00", [0x119f] = "\x31\x01",
  [0x11a0] = "\x31\x02", [0x11a1] = "\x31\x03", [0x11a2] = "\x31\x04",
  [0x11a3] = "\x31\x05", [0x11a4] = "\x31\x06", [0x11a5] = "\x31\x07",
  [0x11a6] = "\x31\x08", [0x11a7] = "\x31\x09", [0x11a8] = "\x31\x0a",
  [0x11a9] = "\x31\x0b", [0x11aa] = "\x31\x0c", [0x11ab] = "\x31\x0d",
  [0x11ac] = "\x31\x0e", [0x11ad] = "\x31\x0f", [0x11ae] = "\x31\x10",
  [0x11af] = "\x31\x11", [0x11b0] = "\x31\x12", [0x11b1] = "\x31\x13",
  [0x11b2] = "\x31\x14", [0x11b3] = "\x31\x15", [0x11b4] = "\x31\x16",
  [0x11b5] = "\x31\x17", [0x11b6] = "\x31\x18", [0x11b7] = "\x31\x19",
  [0x11b8] = "\x31\x1a", [0x11b9] = "\x31\x1b", [0x11ba] = "\x31\x1c",
  [0x11bb] = "\x31\x1d", [0x11bc] = "\x31\x1e", [0x11bd] = "\x31\x1f",
  [0x11be] = "\x31\x20", [0x11bf] = "\x31\x21", [0x11c0] = "\x31\x22",
  [0x11c1] = "\x31\x23", [0x11c2] = "\x31\x24", [0x11c3] = "\x31\x25",
  [0x11c4] = "\x31\x26", [0x11c5] = "\x31\x27", [0x11c6] = "\x31\x28",
  [0x11c7] = "\x31\x29", [0x11c8] = "\x31\x2a", [0x11c9] = "\x31\x2b",
  [0x11ca] = "\x31\x2c", [0x11cb] = "\x31\x2d", [0x11cc] = "\x31\x2e",
  [0x11cd] = "\x31\x2f", [0x11ce] = "\x31\x30", [0x11cf] = "\x31\x31",
  [0x11d0] = "\x31\x32", [0x11d1] = "\x31\x33", [0x11d2] = "\x31\x34",
  [0x11d3] = "\x31\x35", [0x11d4] = "\x31\x36", [0x11d5] = "\x31\x37",
  [0x11d6] = "\x31\x38", [0x11d7] = "\x31\x39", [0x11d8] = "\x31\x3a",
  [0x11d9] = "\x31\x3b", [0x11da] = "\x31\x3c", [0x11db] = "\x31\x3d",
  [0x11dc] = "\x31\x3e", [0x11dd] = "\x31\x3f", [0x11de] = "\x31\x40",
  [0x11df] = "\x31\x41", [0x11e0] = "\x31\x42", [0x11e1] = "\x31\x43",
  [0x11e2] = "\x31\x44", [0x11e3] = "\x31\x45", [0x11e4] = "\x31\x46",
  [0x11e5] = "\x31\x47", [0x11e6] = "\x31\x48", [0x11e7] = "\x31\x49",
  [0x11e8] = "\x31\x4a", [0x11e9] = "\x31\x4b", [0x11ea] = "\x31\x4c",
  [0x11eb] = "\x31\x4d", [0x11ec] = "\x31\x4e", [0x11ed] = "\x31\x4f",
  [0x11ee] = "\x31\x50", [0x11ef] = "\x31\x51", [0x11f0] = "\x31\x52",
  [0x11f1] = "\x31\x53", [0x11f2] = "\x31\x54", [0x11f3] = "\x31\x55",
  [0x11f4] = "\x31\x56", [0x11f5] = "\x31\x57", [0x11f6] = "\x31\x58",
  [0x11f7] = "\x31\x59", [0x11f8] = "\x31\x5a", [0x11f9] = "\x31\x5b",
  [0x11fa] = "\x31\x5c", [0x11fb] = "\x31\x5d", [0x11fc] = "\x31\x5e",
  [0x11fd] = "\x31\x5f", [0x11fe] = "\x31\x60", [0x11ff] = "\x31\x61",
  [0x1200] = "\x31\x62", [0x1201] = "\x31\x63", [0x1202] = "\x31\x64",
  [0x1203] = "\x31\x65", [0x1204] = "\x31\x66", [0x1205] = "\x31\x67",
  [0x1206] = "\x31\x68", [0x1207] = "\x31\x69", [0x1208] = "\x31\x6a",
  [0x1209] = "\x31\x6b", [0x120a] = "\x31\x6c", [0x120b] = "\x31\x6d",
  [0x120c] = "\x31\x6e", [0x120d] = "\x31\x6f", [0x120e] = "\x31\x70",
  [0x120f] = "\x31\x71", [0x1210] = "\x31\x72", [0x1211] = "\x31\x73",
  [0x1212] = "\x31\x74", [0x1213] = "\x31\x75", [0x1214] = "\x31\x76",
  [0x1215] = "\x31\x77", [0x1216] = "\x31\x78", [0x1217] = "\x31\x79",
  [0x1218] = "\x31\x7a", [0x1219] = "\x31\x7b", [0x121a] = "\x31\x7c",
  [0x121b] = "\x31\x7d", [0x121c] = "\x31\x7e", [0x121d] = "\x31\x7f",
  [0x121e] = "\x31\x80", [0x121f] = "\x31\x81", [0x1220] = "\x31\x82",
  [0x1221] = "\x31\x83", [0x1222] = "\x31\x84", [0x1223] = "\x31\x85",
  [0x1224] = "\x31\x86", [0x1225] = "\x31\x87", [0x1226] = "\x31\x88",
  [0x1227] = "\x31\x89", [0x1228] = "\x31\x8a", [0x1229] = "\x31\x8b",
  [0x122a] = "\x31\x8c", [0x122b] = "\x31\x8d", [0x122c] = "\x31\x8e",
  [0x122d] = "\x31\x8f", [0x122e] = "\x31\x90", [0x122f] = "\x31\x91",
  [0x1230] = "\x31\x92", [0x1231] = "\x31\x93", [0x1232] = "\x31\x94",
  [0x1233] = "\x31\x95", [0x1234] = "\x31\x96", [0x1235] = "\x31\x97",
  [0x1236] = "\x31\x98", [0x1237] = "\x31\x99", [0x1238] = "\x31\x9a",
  [0x1239] = "\x31\x9b", [0x123a] = "\x31\x9c", [0x123b] = "\x31\x9d",
  [0x123c] = "\x31\x9e", [0x123d] = "\x31\x9f", [0x123e] = "\x31\xa0",
  [0x123f] = "\x31\xa1", [0x1240] = "\x31\xa2", [0x1241] = "\x31\xa3",
  [0x1242] = "\x31\xa4", [0x1243] = "\x31\xa5", [0x1244] = "\x31\xa6",
  [0x1245] = "\x31\xa7", [0x1246] = "\x31\xa8", [0x1247] = "\x31\xa9",
  [0x1248] = "\x31\xaa", [0x1249] = "\x31\xab", [0x124a] = "\x31\xac",
  [0x124b] = "\x31\xad", [0x124c] = "\x31\xae", [0x124d] = "\x31\xaf",
  [0x124e] = "\x31\xb0", [0x124f] = "\x31\xb1", [0x1250] = "\x31\xb2",
  [0x1251] = "\x31\xb3", [0x1252] = "\x31\xb4", [0x1253] = "\x31\xb5",
  [0x1254] = "\x31\xb6", [0x1255] = "\x31\xb7", [0x1256] = "\x31\xb8",
  [0x1257] = "\x31\xb9", [0x1258] = "\x31\xba", [0x1259] = "\x31\xbb",
  [0x125a] = "\x31\xbc", [0x125b] = "\x31\xbd", [0x125c] = "\x31\xbe",
  [0x125d] = "\x31\xbf", [0x125e] = "\x31\xc0", [0x125f] = "\x31\xc1",
  [0x1260] = "\x31\xc2", [0x1261] = "\x31\xc3", [0x1262] = "\x31\xc4",
  [0x1263] = "\x31\xc5", [0x1264] = "\x31\xc6", [0x1265] = "\x31\xc7",
  [0x1266] = "\x31\xc8", [0x1267] = "\x31\xc9", [0x1268] = "\x31\xca",
  [0x1269] = "\x31\xcb", [0x126a] = "\x31\xcc", [0x126b] = "\x31\xcd",
  [0x126c] = "\x31\xce", [0x126d] = "\x31\xcf", [0x126e] = "\x31\xd0",
  [0x126f] = "\x31\xd1", [0x1270] = "\x31\xd2", [0x1271] = "\x31\xd3",
  [0x1272] = "\x31\xd4", [0x1273] = "\x31\xd5", [0x1274] = "\x31\xd6",
  [0x1275] = "\x31\xd7", [0x1276] = "\x31\xd8", [0x1277] = "\x31\xd9",
  [0x1278] = "\x31\xda", [0x1279] = "\x31\xdb", [0x127a] = "\x31\xdc",
  [0x127b] = "\x31\xdd", [0x127c] = "\x31\xde", [0x127d] = "\x31\xdf",
  [0x127e] = "\x31\xe0", [0x127f] = "\x31\xe1", [0x1280] = "\x31\xe2",
  [0x1281] = "\x31\xe3", [0x1282] = "\x31\xe4", [0x1283] = "\x31\xe5",
  [0x1284] = "\x31\xe6", [0x1285] = "\x31\xe7", [0x1286] = "\x31\xe8",
  [0x1287] = "\x31\xe9", [0x1288] = "\x31\xea", [0x1289] = "\x31\xeb",
  [0x128a] = "\x31\xec", [0x128b] = "\x31\xed", [0x128c] = "\x31\xee",
  [0x128d] = "\x31\xef", [0x128e] = "\x31\xf0", [0x128f] = "\x31\xf1",
  [0x1290] = "\x31\xf2", [0x1291] = "\x31\xf3", [0x1292] = "\x31\xf4",
  [0x1293] = "\x31\xf5", [0x1294] = "\x31\xf6", [0x1295] = "\x31\xf7",
  [0x1296] = "\x31\xf8", [0x1297] = "\x31\xf9", [0x1298] = "\x31\xfa",
  [0x1299] = "\x31\xfb", [0x129a] = "\x31\xfc", [0x129b] = "\x31\xfd",
  [0x129c] = "\x31\xfe", [0x129d] = "\x31\xff", [0x129e] = "\x32\x00",
  [0x129f] = "\x32\x01", [0x12a0] = "\x32\x02", [0x12a1] = "\x32\x03",
  [0x12a2] = "\x32\x04", [0x12a3] = "\x32\x05", [0x12a4] = "\x32\x06",
  [0x12a5] = "\x32\x07", [0x12a6] = "\x32\x08", [0x12a7] = "\x32\x09",
  [0x12a8] = "\x32\x0a", [0x12a9] = "\x32\x0b", [0x12aa] = "\x32\x0c",
  [0x12ab] = "\x32\x0d", [0x12ac] = "\x32\x0e", [0x12ad] = "\x32\x0f",
  [0x12ae] = "\x32\x10", [0x12af] = "\x32\x11", [0x12b0] = "\x32\x12",
  [0x12b1] = "\x32\x13", [0x12b2] = "\x32\x14", [0x12b3] = "\x32\x15",
  [0x12b4] = "\x32\x16", [0x12b5] = "\x32\x17", [0x12b6] = "\x32\x18",
  [0x12b7] = "\x32\x19", [0x12b8] = "\x32\x1a", [0x12b9] = "\x32\x1b",
  [0x12ba] = "\x32\x1c", [0x12bb] = "\x32\x1d", [0x12bc] = "\x32\x1e",
  [0x12bd] = "\x32\x1f", [0x12be] = "\x32\x20", [0x12bf] = "\x32\x21",
  [0x12c0] = "\x32\x22", [0x12c1] = "\x32\x23", [0x12c2] = "\x32\x24",
  [0x12c3] = "\x32\x25", [0x12c4] = "\x32\x26", [0x12c5] = "\x32\x27",
  [0x12c6] = "\x32\x28", [0x12c7] = "\x32\x29", [0x12c8] = "\x32\x2a",
  [0x12c9] = "\x32\x2b", [0x12ca] = "\x32\x2c", [0x12cb] = "\x32\x2d",
  [0x12cc] = "\x32\x2e", [0x12cd] = "\x32\x2f", [0x12ce] = "\x32\x30",
  [0x12cf] = "\x32\x31", [0x12d0] = "\x32\x32", [0x12d1] = "\x32\x33",
  [0x12d2] = "\x32\x34", [0x12d3] = "\x32\x35", [0x12d4] = "\x32\x36",
  [0x12d5] = "\x32\x37", [0x12d6] = "\x32\x38", [0x12d7] = "\x32\x39",
  [0x12d8] = "\x32\x3a", [0x12d9] = "\x32\x3b", [0x12da] = "\x32\x3c",
  [0x12db] = "\x32\x3d", [0x12dc] = "\x32\x3e", [0x12dd] = "\x32\x3f",
  [0x12de] = "\x32\x40", [0x12df] = "\x32\x41", [0x12e0] = "\x32\x42",
  [0x12e1] = "\x32\x43", [0x12e2] = "\x32\x44", [0x12e3] = "\x32\x45",
  [0x12e4] = "\x32\x46", [0x12e5] = "\x32\x47", [0x12e6] = "\x32\x48",
  [0x12e7] = "\x32\x49", [0x12e8] = "\x32\x4a", [0x12e9] = "\x32\x4b",
  [0x12ea] = "\x32\x4c", [0x12eb] = "\x32\x4d", [0x12ec] = "\x32\x4e",
  [0x12ed] = "\x32\x4f", [0x12ee] = "\x32\x50", [0x12ef] = "\x32\x51",
  [0x12f0] = "\x32\x52", [0x12f1] = "\x32\x53", [0x12f2] = "\x32\x54",
  [0x12f3] = "\x32\x55", [0x12f4] = "\x32\x56", [0x12f5] = "\x32\x57",
  [0x12f6] = "\x32\x58", [0x12f7] = "\x32\x59", [0x12f8] = "\x32\x5a",
  [0x12f9] = "\x32\x5b", [0x12fa] = "\x32\x5c", [0x12fb] = "\x32\x5d",
  [0x12fc] = "\x32\x5e", [0x12fd] = "\x32\x5f", [0x12fe] = "\x32\x60",
  [0x12ff] = "\x32\x61", [0x1300] = "\x32\x62", [0x1301] = "\x32\x63",
  [0x1302] = "\x32\x64", [0x1303] = "\x32\x65", [0x1304] = "\x32\x66",
  [0x1305] = "\x32\x67", [0x1306] = "\x32\x68", [0x1307] = "\x32\x69",
  [0x1308] = "\x32\x6a", [0x1309] = "\x32\x6b", [0x130a] = "\x32\x6c",
  [0x130b] = "\x32\x6d", [0x130c] = "\x32\x6e", [0x130d] = "\x32\x6f",
  [0x130e] = "\x32\x70", [0x130f] = "\x32\x71", [0x1310] = "\x32\x72",
  [0x1311] = "\x32\x73", [0x1312] = "\x32\x74", [0x1313] = "\x32\x75",
  [0x1314] = "\x32\x76", [0x1315] = "\x32\x77", [0x1316] = "\x32\x78",
  [0x1317] = "\x32\x79", [0x1318] = "\x32\x7a", [0x1319] = "\x32\x7b",
  [0x131a] = "\x32\x7c", [0x131b] = "\x32\x7d", [0x131c] = "\x32\x7e",
  [0x131d] = "\x32\x7f", [0x131e] = "\x32\x80", [0x131f] = "\x32\x81",
  [0x1320] = "\x32\x82", [0x1321] = "\x32\x83", [0x1322] = "\x32\x84",
  [0x1323] = "\x32\x85", [0x1324] = "\x32\x86", [0x1325] = "\x32\x87",
  [0x1326] = "\x32\x88", [0x1327] = "\x32\x89", [0x1328] = "\x32\x8a",
  [0x1329] = "\x32\x8b", [0x132a] = "\x32\x8c", [0x132b] = "\x32\x8d",
  [0x132c] = "\x32\x8e", [0x132d] = "\x32\x8f", [0x132e] = "\x32\x90",
  [0x132f] = "\x32\x91", [0x1330] = "\x32\x92", [0x1331] = "\x32\x93",
  [0x1332] = "\x32\x94", [0x1333] = "\x32\x95", [0x1334] = "\x32\x96",
  [0x1335] = "\x32\x97", [0x1336] = "\x32\x98", [0x1337] = "\x32\x99",
  [0x1338] = "\x32\x9a", [0x1339] = "\x32\x9b", [0x133a] = "\x32\x9c",
  [0x133b] = "\x32\x9d", [0x133c] = "\x32\x9e", [0x133d] = "\x32\x9f",
  [0x133e] = "\x32\xa0", [0x133f] = "\x32\xa1", [0x1340] = "\x32\xa2",
  [0x1341] = "\x32\xa3", [0x1342] = "\x32\xa4", [0x1343] = "\x32\xa5",
  [0x1344] = "\x32\xa6", [0x1345] = "\x32\xa7", [0x1346] = "\x32\xa8",
  [0x1347] = "\x32\xa9", [0x1348] = "\x32\xaa", [0x1349] = "\x32\xab",
  [0x134a] = "\x32\xac", [0x134b] = "\x32\xad", [0x134c] = "\x32\xae",
  [0x134d] = "\x32\xaf", [0x134e] = "\x32\xb0", [0x134f] = "\x32\xb1",
  [0x1350] = "\x32\xb2", [0x1351] = "\x32\xb3", [0x1352] = "\x32\xb4",
  [0x1353] = "\x32\xb5", [0x1354] = "\x32\xb6", [0x1355] = "\x32\xb7",
  [0x1356] = "\x32\xb8", [0x1357] = "\x32\xb9", [0x1358] = "\x32\xba",
  [0x1359] = "\x32\xbb", [0x135a] = "\x32\xbc", [0x135b] = "\x32\xbd",
  [0x135c] = "\x32\xbe", [0x135d] = "\x32\xbf", [0x135e] = "\x32\xc0",
  [0x135f] = "\x32\xc1", [0x1360] = "\x32\xc2", [0x1361] = "\x32\xc3",
  [0x1362] = "\x32\xc4", [0x1363] = "\x32\xc5", [0x1364] = "\x32\xc6",
  [0x1365] = "\x32\xc7", [0x1366] = "\x32\xc8", [0x1367] = "\x32\xc9",
  [0x1368] = "\x32\xca", [0x1369] = "\x32\xcb", [0x136a] = "\x32\xcc",
  [0x136b] = "\x32\xcd", [0x136c] = "\x32\xce", [0x136d] = "\x32\xcf",
  [0x136e] = "\x32\xd0", [0x136f] = "\x32\xd1", [0x1370] = "\x32\xd2",
  [0x1371] = "\x32\xd3", [0x1372] = "\x32\xd4", [0x1373] = "\x32\xd5",
  [0x1374] = "\x32\xd6", [0x1375] = "\x32\xd7", [0x1376] = "\x32\xd8",
  [0x1377] = "\x32\xd9", [0x1378] = "\x32\xda", [0x1379] = "\x32\xdb",
  [0x137a] = "\x32\xdc", [0x137b] = "\x32\xdd", [0x137c] = "\x32\xde",
  [0x137d] = "\x32\xdf", [0x137e] = "\x32\xe0", [0x137f] = "\x32\xe1",
  [0x1380] = "\x32\xe2", [0x1381] = "\x32\xe3", [0x1382] = "\x32\xe4",
  [0x1383] = "\x32\xe5", [0x1384] = "\x32\xe6", [0x1385] = "\x32\xe7",
  [0x1386] = "\x32\xe8", [0x1387] = "\x32\xe9", [0x1388] = "\x32\xea",
  [0x1389] = "\x32\xeb", [0x138a] = "\x32\xec", [0x138b] = "\x32\xed",
  [0x138c] = "\x32\xee", [0x138d] = "\x32\xef", [0x138e] = "\x32\xf0",
  [0x138f] = "\x32\xf1", [0x1390] = "\x32\xf2", [0x1391] = "\x32\xf3",
  [0x1392] = "\x32\xf4", [0x1393] = "\x32\xf5", [0x1394] = "\x32\xf6",
  [0x1395] = "\x32\xf7", [0x1396] = "\x32\xf8", [0x1397] = "\x32\xf9",
  [0x1398] = "\x32\xfa", [0x1399] = "\x32\xfb", [0x139a] = "\x32\xfc",
  [0x139b] = "\x32\xfd", [0x139c] = "\x32\xfe", [0x139d] = "\x32\xff",
  [0x139e] = "\x33\x00", [0x139f] = "\x33\x01", [0x13a0] = "\x33\x02",
  [0x13a1] = "\x33\x03", [0x13a2] = "\x33\x04", [0x13a3] = "\x33\x05",
  [0x13a4] = "\x33\x06", [0x13a5] = "\x33\x07", [0x13a6] = "\x33\x08",
  [0x13a7] = "\x33\x09", [0x13a8] = "\x33\x0a", [0x13a9] = "\x33\x0b",
  [0x13aa] = "\x33\x0c", [0x13ab] = "\x33\x0d", [0x13ac] = "\x33\x0e",
  [0x13ad] = "\x33\x0f", [0x13ae] = "\x33\x10", [0x13af] = "\x33\x11",
  [0x13b0] = "\x33\x12", [0x13b1] = "\x33\x13", [0x13b2] = "\x33\x14",
  [0x13b3] = "\x33\x15", [0x13b4] = "\x33\x16", [0x13b5] = "\x33\x17",
  [0x13b6] = "\x33\x18", [0x13b7] = "\x33\x19", [0x13b8] = "\x33\x1a",
  [0x13b9] = "\x33\x1b", [0x13ba] = "\x33\x1c", [0x13bb] = "\x33\x1d",
  [0x13bc] = "\x33\x1e", [0x13bd] = "\x33\x1f", [0x13be] = "\x33\x20",
  [0x13bf] = "\x33\x21", [0x13c0] = "\x33\x22", [0x13c1] = "\x33\x23",
  [0x13c2] = "\x33\x24", [0x13c3] = "\x33\x25", [0x13c4] = "\x33\x26",
  [0x13c5] = "\x33\x27", [0x13c6] = "\x33\x28", [0x13c7] = "\x33\x29",
  [0x13c8] = "\x33\x2a", [0x13c9] = "\x33\x2b", [0x13ca] = "\x33\x2c",
  [0x13cb] = "\x33\x2d", [0x13cc] = "\x33\x2e", [0x13cd] = "\x33\x2f",
  [0x13ce] = "\x33\x30", [0x13cf] = "\x33\x31", [0x13d0] = "\x33\x32",
  [0x13d1] = "\x33\x33", [0x13d2] = "\x33\x34", [0x13d3] = "\x33\x35",
  [0x13d4] = "\x33\x36", [0x13d5] = "\x33\x37", [0x13d6] = "\x33\x38",
  [0x13d7] = "\x33\x39", [0x13d8] = "\x33\x3a", [0x13d9] = "\x33\x3b",
  [0x13da] = "\x33\x3c", [0x13db] = "\x33\x3d", [0x13dc] = "\x33\x3e",
  [0x13dd] = "\x33\x3f", [0x13de] = "\x33\x40", [0x13df] = "\x33\x41",
  [0x13e0] = "\x33\x42", [0x13e1] = "\x33\x43", [0x13e2] = "\x33\x44",
  [0x13e3] = "\x33\x45", [0x13e4] = "\x33\x46", [0x13e5] = "\x33\x47",
  [0x13e6] = "\x33\x48", [0x13e7] = "\x33\x49", [0x13e8] = "\x33\x4a",
  [0x13e9] = "\x33\x4b", [0x13ea] = "\x33\x4c", [0x13eb] = "\x33\x4d",
  [0x13ec] = "\x33\x4e", [0x13ed] = "\x33\x4f", [0x13ee] = "\x33\x50",
  [0x13ef] = "\x33\x51", [0x13f0] = "\x33\x52", [0x13f1] = "\x33\x53",
  [0x13f2] = "\x33\x54", [0x13f3] = "\x33\x55", [0x13f4] = "\x33\x56",
  [0x13f5] = "\x33\x57", [0x13f6] = "\x33\x58", [0x13f7] = "\x33\x59",
  [0x13f8] = "\x33\x5a", [0x13f9] = "\x33\x5b", [0x13fa] = "\x33\x5c",
  [0x13fb] = "\x33\x5d", [0x13fc] = "\x33\x5e", [0x13fd] = "\x33\x5f",
  [0x13fe] = "\x33\x60", [0x13ff] = "\x33\x61", [0x1400] = "\x33\x62",
  [0x1401] = "\x33\x63", [0x1402] = "\x33\x64", [0x1403] = "\x33\x65",
  [0x1404] = "\x33\x66", [0x1405] = "\x33\x67", [0x1406] = "\x33\x68",
  [0x1407] = "\x33\x69", [0x1408] = "\x33\x6a", [0x1409] = "\x33\x6b",
  [0x140a] = "\x33\x6c", [0x140b] = "\x33\x6d", [0x140c] = "\x33\x6e",
  [0x140d] = "\x33\x6f", [0x140e] = "\x33\x70", [0x140f] = "\x33\x71",
  [0x1410] = "\x33\x72", [0x1411] = "\x33\x73", [0x1412] = "\x33\x74",
  [0x1413] = "\x33\x75", [0x1414] = "\x33\x76", [0x1415] = "\x33\x77",
  [0x1416] = "\x33\x78", [0x1417] = "\x33\x79", [0x1418] = "\x33\x7a",
  [0x1419] = "\x33\x7b", [0x141a] = "\x33\x7c", [0x141b] = "\x33\x7d",
  [0x141c] = "\x33\x7e", [0x141d] = "\x33\x7f", [0x141e] = "\x33\x80",
  [0x141f] = "\x33\x81", [0x1420] = "\x33\x82", [0x1421] = "\x33\x83",
  [0x1422] = "\x33\x84", [0x1423] = "\x33\x85", [0x1424] = "\x33\x86",
  [0x1425] = "\x33\x87", [0x1426] = "\x33\x88", [0x1427] = "\x33\x89",
  [0x1428] = "\x33\x8a", [0x1429] = "\x33\x8b", [0x142a] = "\x33\x8c",
  [0x142b] = "\x33\x8d", [0x142c] = "\x33\x8e", [0x142d] = "\x33\x8f",
  [0x142e] = "\x33\x90", [0x142f] = "\x33\x91", [0x1430] = "\x33\x92",
  [0x1431] = "\x33\x93", [0x1432] = "\x33\x94", [0x1433] = "\x33\x95",
  [0x1434] = "\x33\x96", [0x1435] = "\x33\x97", [0x1436] = "\x33\x98",
  [0x1437] = "\x33\x99", [0x1438] = "\x33\x9a", [0x1439] = "\x33\x9b",
  [0x143a] = "\x33\x9c", [0x143b] = "\x33\x9d", [0x143c] = "\x33\x9e",
  [0x143d] = "\x33\x9f", [0x143e] = "\x33\xa0", [0x143f] = "\x33\xa1",
  [0x1440] = "\x33\xa2", [0x1441] = "\x33\xa3", [0x1442] = "\x33\xa4",
  [0x1443] = "\x33\xa5", [0x1444] = "\x33\xa6", [0x1445] = "\x33\xa7",
  [0x1446] = "\x33\xa8", [0x1447] = "\x33\xa9", [0x1448] = "\x33\xaa",
  [0x1449] = "\x33\xab", [0x144a] = "\x33\xac", [0x144b] = "\x33\xad",
  [0x144c] = "\x33\xae", [0x144d] = "\x33\xaf", [0x144e] = "\x33\xb0",
  [0x144f] = "\x33\xb1", [0x1450] = "\x33\xb2", [0x1451] = "\x33\xb3",
  [0x1452] = "\x33\xb4", [0x1453] = "\x33\xb5", [0x1454] = "\x33\xb6",
  [0x1455] = "\x33\xb7", [0x1456] = "\x33\xb8", [0x1457] = "\x33\xb9",
  [0x1458] = "\x33\xba", [0x1459] = "\x33\xbb", [0x145a] = "\x33\xbc",
  [0x145b] = "\x33\xbd", [0x145c] = "\x33\xbe", [0x145d] = "\x33\xbf",
  [0x145e] = "\x33\xc0", [0x145f] = "\x33\xc1", [0x1460] = "\x33\xc2",
  [0x1461] = "\x33\xc3", [0x1462] = "\x33\xc4", [0x1463] = "\x33\xc5",
  [0x1464] = "\x33\xc6", [0x1465] = "\x33\xc7", [0x1466] = "\x33\xc8",
  [0x1467] = "\x33\xc9", [0x1468] = "\x33\xca", [0x1469] = "\x33\xcb",
  [0x146a] = "\x33\xcc", [0x146b] = "\x33\xcd", [0x146c] = "\x33\xce",
  [0x146d] = "\x33\xcf", [0x146e] = "\x33\xd0", [0x146f] = "\x33\xd1",
  [0x1470] = "\x33\xd2", [0x1471] = "\x33\xd3", [0x1472] = "\x33\xd4",
  [0x1473] = "\x33\xd5", [0x1474] = "\x33\xd6", [0x1475] = "\x33\xd7",
  [0x1476] = "\x33\xd8", [0x1477] = "\x33\xd9", [0x1478] = "\x33\xda",
  [0x1479] = "\x33\xdb", [0x147a] = "\x33\xdc", [0x147b] = "\x33\xdd",
  [0x147c] = "\x33\xde", [0x147d] = "\x33\xdf", [0x147e] = "\x33\xe0",
  [0x147f] = "\x33\xe1", [0x1480] = "\x33\xe2", [0x1481] = "\x33\xe3",
  [0x1482] = "\x33\xe4", [0x1483] = "\x33\xe5", [0x1484] = "\x33\xe6",
  [0x1485] = "\x33\xe7", [0x1486] = "\x33\xe8", [0x1487] = "\x33\xe9",
  [0x1488] = "\x33\xea", [0x1489] = "\x33\xeb", [0x148a] = "\x33\xec",
  [0x148b] = "\x33\xed", [0x148c] = "\x33\xee", [0x148d] = "\x33\xef",
  [0x148e] = "\x33\xf0", [0x148f] = "\x33\xf1", [0x1490] = "\x33\xf2",
  [0x1491] = "\x33\xf3", [0x1492] = "\x33\xf4", [0x1493] = "\x33\xf5",
  [0x1494] = "\x33\xf6", [0x1495] = "\x33\xf7", [0x1496] = "\x33\xf8",
  [0x1497] = "\x33\xf9", [0x1498] = "\x33\xfa", [0x1499] = "\x33\xfb",
  [0x149a] = "\x33\xfc", [0x149b] = "\x33\xfd", [0x149c] = "\x33\xfe",
  [0x149d] = "\x33\xff", [0x149e] = "\x34\x00", [0x149f] = "\x34\x01",
  [0x14a0] = "\x34\x02", [0x14a1] = "\x34\x03", [0x14a2] = "\x34\x04",
  [0x14a3] = "\x34\x05", [0x14a4] = "\x34\x06", [0x14a5] = "\x34\x07",
  [0x14a6] = "\x34\x08", [0x14a7] = "\x34\x09", [0x14a8] = "\x34\x0a",
  [0x14a9] = "\x34\x0b", [0x14aa] = "\x34\x0c", [0x14ab] = "\x34\x0d",
  [0x14ac] = "\x34\x0e", [0x14ad] = "\x34\x0f", [0x14ae] = "\x34\x10",
  [0x14af] = "\x34\x11", [0x14b0] = "\x34\x12", [0x14b1] = "\x34\x13",
  [0x14b2] = "\x34\x14", [0x14b3] = "\x34\x15", [0x14b4] = "\x34\x16",
  [0x14b5] = "\x34\x17", [0x14b6] = "\x34\x18", [0x14b7] = "\x34\x19",
  [0x14b8] = "\x34\x1a", [0x14b9] = "\x34\x1b", [0x14ba] = "\x34\x1c",
  [0x14bb] = "\x34\x1d", [0x14bc] = "\x34\x1e", [0x14bd] = "\x34\x1f",
  [0x14be] = "\x34\x20", [0x14bf] = "\x34\x21", [0x14c0] = "\x34\x22",
  [0x14c1] = "\x34\x23", [0x14c2] = "\x34\x24", [0x14c3] = "\x34\x25",
  [0x14c4] = "\x34\x26", [0x14c5] = "\x34\x27", [0x14c6] = "\x34\x28",
  [0x14c7] = "\x34\x29", [0x14c8] = "\x34\x2a", [0x14c9] = "\x34\x2b",
  [0x14ca] = "\x34\x2c", [0x14cb] = "\x34\x2d", [0x14cc] = "\x34\x2e",
  [0x14cd] = "\x34\x2f", [0x14ce] = "\x34\x30", [0x14cf] = "\x34\x31",
  [0x14d0] = "\x34\x32", [0x14d1] = "\x34\x33", [0x14d2] = "\x34\x34",
  [0x14d3] = "\x34\x35", [0x14d4] = "\x34\x36", [0x14d5] = "\x34\x37",
  [0x14d6] = "\x34\x38", [0x14d7] = "\x34\x39", [0x14d8] = "\x34\x3a",
  [0x14d9] = "\x34\x3b", [0x14da] = "\x34\x3c", [0x14db] = "\x34\x3d",
  [0x14dc] = "\x34\x3e", [0x14dd] = "\x34\x3f", [0x14de] = "\x34\x40",
  [0x14df] = "\x34\x41", [0x14e0] = "\x34\x42", [0x14e1] = "\x34\x43",
  [0x14e2] = "\x34\x44", [0x14e3] = "\x34\x45", [0x14e4] = "\x34\x46",
  [0x14e5] = "\x34\x47", [0x14e6] = "\x34\x48", [0x14e7] = "\x34\x49",
  [0x14e8] = "\x34\x4a", [0x14e9] = "\x34\x4b", [0x14ea] = "\x34\x4c",
  [0x14eb] = "\x34\x4d", [0x14ec] = "\x34\x4e", [0x14ed] = "\x34\x4f",
  [0x14ee] = "\x34\x50", [0x14ef] = "\x34\x51", [0x14f0] = "\x34\x52",
  [0x14f1] = "\x34\x53", [0x14f2] = "\x34\x54", [0x14f3] = "\x34\x55",
  [0x14f4] = "\x34\x56", [0x14f5] = "\x34\x57", [0x14f6] = "\x34\x58",
  [0x14f7] = "\x34\x59", [0x14f8] = "\x34\x5a", [0x14f9] = "\x34\x5b",
  [0x14fa] = "\x34\x5c", [0x14fb] = "\x34\x5d", [0x14fc] = "\x34\x5e",
  [0x14fd] = "\x34\x5f", [0x14fe] = "\x34\x60", [0x14ff] = "\x34\x61",
  [0x1500] = "\x34\x62", [0x1501] = "\x34\x63", [0x1502] = "\x34\x64",
  [0x1503] = "\x34\x65", [0x1504] = "\x34\x66", [0x1505] = "\x34\x67",
  [0x1506] = "\x34\x68", [0x1507] = "\x34\x69", [0x1508] = "\x34\x6a",
  [0x1509] = "\x34\x6b", [0x150a] = "\x34\x6c", [0x150b] = "\x34\x6d",
  [0x150c] = "\x34\x6e", [0x150d] = "\x34\x6f", [0x150e] = "\x34\x70",
  [0x150f] = "\x34\x71", [0x1510] = "\x34\x72", [0x1511] = "\x34\x73",
  [0x1512] = "\x34\x74", [0x1513] = "\x34\x75", [0x1514] = "\x34\x76",
  [0x1515] = "\x34\x77", [0x1516] = "\x34\x78", [0x1517] = "\x34\x79",
  [0x1518] = "\x34\x7a", [0x1519] = "\x34\x7b", [0x151a] = "\x34\x7c",
  [0x151b] = "\x34\x7d", [0x151c] = "\x34\x7e", [0x151d] = "\x34\x7f",
  [0x151e] = "\x34\x80", [0x151f] = "\x34\x81", [0x1520] = "\x34\x82",
  [0x1521] = "\x34\x83", [0x1522] = "\x34\x84", [0x1523] = "\x34\x85",
  [0x1524] = "\x34\x86", [0x1525] = "\x34\x87", [0x1526] = "\x34\x88",
  [0x1527] = "\x34\x89", [0x1528] = "\x34\x8a", [0x1529] = "\x34\x8b",
  [0x152a] = "\x34\x8c", [0x152b] = "\x34\x8d", [0x152c] = "\x34\x8e",
  [0x152d] = "\x34\x8f", [0x152e] = "\x34\x90", [0x152f] = "\x34\x91",
  [0x1530] = "\x34\x92", [0x1531] = "\x34\x93", [0x1532] = "\x34\x94",
  [0x1533] = "\x34\x95", [0x1534] = "\x34\x96", [0x1535] = "\x34\x97",
  [0x1536] = "\x34\x98", [0x1537] = "\x34\x99", [0x1538] = "\x34\x9a",
  [0x1539] = "\x34\x9b", [0x153a] = "\x34\x9c", [0x153b] = "\x34\x9d",
  [0x153c] = "\x34\x9e", [0x153d] = "\x34\x9f", [0x153e] = "\x34\xa0",
  [0x153f] = "\x34\xa1", [0x1540] = "\x34\xa2", [0x1541] = "\x34\xa3",
  [0x1542] = "\x34\xa4", [0x1543] = "\x34\xa5", [0x1544] = "\x34\xa6",
  [0x1545] = "\x34\xa7", [0x1546] = "\x34\xa8", [0x1547] = "\x34\xa9",
  [0x1548] = "\x34\xaa", [0x1549] = "\x34\xab", [0x154a] = "\x34\xac",
  [0x154b] = "\x34\xad", [0x154c] = "\x34\xae", [0x154d] = "\x34\xaf",
  [0x154e] = "\x34\xb0", [0x154f] = "\x34\xb1", [0x1550] = "\x34\xb2",
  [0x1551] = "\x34\xb3", [0x1552] = "\x34\xb4", [0x1553] = "\x34\xb5",
  [0x1554] = "\x34\xb6", [0x1555] = "\x34\xb7", [0x1556] = "\x34\xb8",
  [0x1557] = "\x34\xb9", [0x1558] = "\x34\xba", [0x1559] = "\x34\xbb",
  [0x155a] = "\x34\xbc", [0x155b] = "\x34\xbd", [0x155c] = "\x34\xbe",
  [0x155d] = "\x34\xbf", [0x155e] = "\x34\xc0", [0x155f] = "\x34\xc1",
  [0x1560] = "\x34\xc2", [0x1561] = "\x34\xc3", [0x1562] = "\x34\xc4",
  [0x1563] = "\x34\xc5", [0x1564] = "\x34\xc6", [0x1565] = "\x34\xc7",
  [0x1566] = "\x34\xc8", [0x1567] = "\x34\xc9", [0x1568] = "\x34\xca",
  [0x1569] = "\x34\xcb", [0x156a] = "\x34\xcc", [0x156b] = "\x34\xcd",
  [0x156c] = "\x34\xce", [0x156d] = "\x34\xcf", [0x156e] = "\x34\xd0",
  [0x156f] = "\x34\xd1", [0x1570] = "\x34\xd2", [0x1571] = "\x34\xd3",
  [0x1572] = "\x34\xd4", [0x1573] = "\x34\xd5", [0x1574] = "\x34\xd6",
  [0x1575] = "\x34\xd7", [0x1576] = "\x34\xd8", [0x1577] = "\x34\xd9",
  [0x1578] = "\x34\xda", [0x1579] = "\x34\xdb", [0x157a] = "\x34\xdc",
  [0x157b] = "\x34\xdd", [0x157c] = "\x34\xde", [0x157d] = "\x34\xdf",
  [0x157e] = "\x34\xe0", [0x157f] = "\x34\xe1", [0x1580] = "\x34\xe2",
  [0x1581] = "\x34\xe3", [0x1582] = "\x34\xe4", [0x1583] = "\x34\xe5",
  [0x1584] = "\x34\xe6", [0x1585] = "\x34\xe7", [0x1586] = "\x34\xe8",
  [0x1587] = "\x34\xe9", [0x1588] = "\x34\xea", [0x1589] = "\x34\xeb",
  [0x158a] = "\x34\xec", [0x158b] = "\x34\xed", [0x158c] = "\x34\xee",
  [0x158d] = "\x34\xef", [0x158e] = "\x34\xf0", [0x158f] = "\x34\xf1",
  [0x1590] = "\x34\xf2", [0x1591] = "\x34\xf3", [0x1592] = "\x34\xf4",
  [0x1593] = "\x34\xf5", [0x1594] = "\x34\xf6", [0x1595] = "\x34\xf7",
  [0x1596] = "\x34\xf8", [0x1597] = "\x34\xf9", [0x1598] = "\x34\xfa",
  [0x1599] = "\x34\xfb", [0x159a] = "\x34\xfc", [0x159b] = "\x34\xfd",
  [0x159c] = "\x34\xfe", [0x159d] = "\x34\xff", [0x159e] = "\x35\x00",
  [0x159f] = "\x35\x01", [0x15a0] = "\x35\x02", [0x15a1] = "\x35\x03",
  [0x15a2] = "\x35\x04", [0x15a3] = "\x35\x05", [0x15a4] = "\x35\x06",
  [0x15a5] = "\x35\x07", [0x15a6] = "\x35\x08", [0x15a7] = "\x35\x09",
  [0x15a8] = "\x35\x0a", [0x15a9] = "\x35\x0b", [0x15aa] = "\x35\x0c",
  [0x15ab] = "\x35\x0d", [0x15ac] = "\x35\x0e", [0x15ad] = "\x35\x0f",
  [0x15ae] = "\x35\x10", [0x15af] = "\x35\x11", [0x15b0] = "\x35\x12",
  [0x15b1] = "\x35\x13", [0x15b2] = "\x35\x14", [0x15b3] = "\x35\x15",
  [0x15b4] = "\x35\x16", [0x15b5] = "\x35\x17", [0x15b6] = "\x35\x18",
  [0x15b7] = "\x35\x19", [0x15b8] = "\x35\x1a", [0x15b9] = "\x35\x1b",
  [0x15ba] = "\x35\x1c", [0x15bb] = "\x35\x1d", [0x15bc] = "\x35\x1e",
  [0x15bd] = "\x35\x1f", [0x15be] = "\x35\x20", [0x15bf] = "\x35\x21",
  [0x15c0] = "\x35\x22", [0x15c1] = "\x35\x23", [0x15c2] = "\x35\x24",
  [0x15c3] = "\x35\x25", [0x15c4] = "\x35\x26", [0x15c5] = "\x35\x27",
  [0x15c6] = "\x35\x28", [0x15c7] = "\x35\x29", [0x15c8] = "\x35\x2a",
  [0x15c9] = "\x35\x2b", [0x15ca] = "\x35\x2c", [0x15cb] = "\x35\x2d",
  [0x15cc] = "\x35\x2e", [0x15cd] = "\x35\x2f", [0x15ce] = "\x35\x30",
  [0x15cf] = "\x35\x31", [0x15d0] = "\x35\x32", [0x15d1] = "\x35\x33",
  [0x15d2] = "\x35\x34", [0x15d3] = "\x35\x35", [0x15d4] = "\x35\x36",
  [0x15d5] = "\x35\x37", [0x15d6] = "\x35\x38", [0x15d7] = "\x35\x39",
  [0x15d8] = "\x35\x3a", [0x15d9] = "\x35\x3b", [0x15da] = "\x35\x3c",
  [0x15db] = "\x35\x3d", [0x15dc] = "\x35\x3e", [0x15dd] = "\x35\x3f",
  [0x15de] = "\x35\x40", [0x15df] = "\x35\x41", [0x15e0] = "\x35\x42",
  [0x15e1] = "\x35\x43", [0x15e2] = "\x35\x44", [0x15e3] = "\x35\x45",
  [0x15e4] = "\x35\x46", [0x15e5] = "\x35\x47", [0x15e6] = "\x35\x48",
  [0x15e7] = "\x35\x49", [0x15e8] = "\x35\x4a", [0x15e9] = "\x35\x4b",
  [0x15ea] = "\x35\x4c", [0x15eb] = "\x35\x4d", [0x15ec] = "\x35\x4e",
  [0x15ed] = "\x35\x4f", [0x15ee] = "\x35\x50", [0x15ef] = "\x35\x51",
  [0x15f0] = "\x35\x52", [0x15f1] = "\x35\x53", [0x15f2] = "\x35\x54",
  [0x15f3] = "\x35\x55", [0x15f4] = "\x35\x56", [0x15f5] = "\x35\x57",
  [0x15f6] = "\x35\x58", [0x15f7] = "\x35\x59", [0x15f8] = "\x35\x5a",
  [0x15f9] = "\x35\x5b", [0x15fa] = "\x35\x5c", [0x15fb] = "\x35\x5d",
  [0x15fc] = "\x35\x5e", [0x15fd] = "\x35\x5f", [0x15fe] = "\x35\x60",
  [0x15ff] = "\x35\x61", [0x1600] = "\x35\x62", [0x1601] = "\x35\x63",
  [0x1602] = "\x35\x64", [0x1603] = "\x35\x65", [0x1604] = "\x35\x66",
  [0x1605] = "\x35\x67", [0x1606] = "\x35\x68", [0x1607] = "\x35\x69",
  [0x1608] = "\x35\x6a", [0x1609] = "\x35\x6b", [0x160a] = "\x35\x6c",
  [0x160b] = "\x35\x6d", [0x160c] = "\x35\x6e", [0x160d] = "\x35\x6f",
  [0x160e] = "\x35\x70", [0x160f] = "\x35\x71", [0x1610] = "\x35\x72",
  [0x1611] = "\x35\x73", [0x1612] = "\x35\x74", [0x1613] = "\x35\x75",
  [0x1614] = "\x35\x76", [0x1615] = "\x35\x77", [0x1616] = "\x35\x78",
  [0x1617] = "\x35\x79", [0x1618] = "\x35\x7a", [0x1619] = "\x35\x7b",
  [0x161a] = "\x35\x7c", [0x161b] = "\x35\x7d", [0x161c] = "\x35\x7e",
  [0x161d] = "\x35\x7f", [0x161e] = "\x35\x80", [0x161f] = "\x35\x81",
  [0x1620] = "\x35\x82", [0x1621] = "\x35\x83", [0x1622] = "\x35\x84",
  [0x1623] = "\x35\x85", [0x1624] = "\x35\x86", [0x1625] = "\x35\x87",
  [0x1626] = "\x35\x88", [0x1627] = "\x35\x89", [0x1628] = "\x35\x8a",
  [0x1629] = "\x35\x8b", [0x162a] = "\x35\x8c", [0x162b] = "\x35\x8d",
  [0x162c] = "\x35\x8e", [0x162d] = "\x35\x8f", [0x162e] = "\x35\x90",
  [0x162f] = "\x35\x91", [0x1630] = "\x35\x92", [0x1631] = "\x35\x93",
  [0x1632] = "\x35\x94", [0x1633] = "\x35\x95", [0x1634] = "\x35\x96",
  [0x1635] = "\x35\x97", [0x1636] = "\x35\x98", [0x1637] = "\x35\x99",
  [0x1638] = "\x35\x9a", [0x1639] = "\x35\x9b", [0x163a] = "\x35\x9c",
  [0x163b] = "\x35\x9d", [0x163c] = "\x35\x9e", [0x163d] = "\x35\x9f",
  [0x163e] = "\x35\xa0", [0x163f] = "\x35\xa1", [0x1640] = "\x35\xa2",
  [0x1641] = "\x35\xa3", [0x1642] = "\x35\xa4", [0x1643] = "\x35\xa5",
  [0x1644] = "\x35\xa6", [0x1645] = "\x35\xa7", [0x1646] = "\x35\xa8",
  [0x1647] = "\x35\xa9", [0x1648] = "\x35\xaa", [0x1649] = "\x35\xab",
  [0x164a] = "\x35\xac", [0x164b] = "\x35\xad", [0x164c] = "\x35\xae",
  [0x164d] = "\x35\xaf", [0x164e] = "\x35\xb0", [0x164f] = "\x35\xb1",
  [0x1650] = "\x35\xb2", [0x1651] = "\x35\xb3", [0x1652] = "\x35\xb4",
  [0x1653] = "\x35\xb5", [0x1654] = "\x35\xb6", [0x1655] = "\x35\xb7",
  [0x1656] = "\x35\xb8", [0x1657] = "\x35\xb9", [0x1658] = "\x35\xba",
  [0x1659] = "\x35\xbb", [0x165a] = "\x35\xbc", [0x165b] = "\x35\xbd",
  [0x165c] = "\x35\xbe", [0x165d] = "\x35\xbf", [0x165e] = "\x35\xc0",
  [0x165f] = "\x35\xc1", [0x1660] = "\x35\xc2", [0x1661] = "\x35\xc3",
  [0x1662] = "\x35\xc4", [0x1663] = "\x35\xc5", [0x1664] = "\x35\xc6",
  [0x1665] = "\x35\xc7", [0x1666] = "\x35\xc8", [0x1667] = "\x35\xc9",
  [0x1668] = "\x35\xca", [0x1669] = "\x35\xcb", [0x166a] = "\x35\xcc",
  [0x166b] = "\x35\xcd", [0x166c] = "\x35\xce", [0x166d] = "\x35\xcf",
  [0x166e] = "\x35\xd0", [0x166f] = "\x35\xd1", [0x1670] = "\x35\xd2",
  [0x1671] = "\x35\xd3", [0x1672] = "\x35\xd4", [0x1673] = "\x35\xd5",
  [0x1674] = "\x35\xd6", [0x1675] = "\x35\xd7", [0x1676] = "\x35\xd8",
  [0x1677] = "\x35\xd9", [0x1678] = "\x35\xda", [0x1679] = "\x35\xdb",
  [0x167a] = "\x35\xdc", [0x167b] = "\x35\xdd", [0x167c] = "\x35\xde",
  [0x167d] = "\x35\xdf", [0x167e] = "\x35\xe0", [0x167f] = "\x35\xe1",
  [0x1680] = "\x35\xe2", [0x1681] = "\x35\xe3", [0x1682] = "\x35\xe4",
  [0x1683] = "\x35\xe5", [0x1684] = "\x35\xe6", [0x1685] = "\x35\xe7",
  [0x1686] = "\x35\xe8", [0x1687] = "\x35\xe9", [0x1688] = "\x35\xea",
  [0x1689] = "\x35\xeb", [0x168a] = "\x35\xec", [0x168b] = "\x35\xed",
  [0x168c] = "\x35\xee", [0x168d] = "\x35\xef", [0x168e] = "\x35\xf0",
  [0x168f] = "\x35\xf1", [0x1690] = "\x35\xf2", [0x1691] = "\x35\xf3",
  [0x1692] = "\x35\xf4", [0x1693] = "\x35\xf5", [0x1694] = "\x35\xf6",
  [0x1695] = "\x35\xf7", [0x1696] = "\x35\xf8", [0x1697] = "\x35\xf9",
  [0x1698] = "\x35\xfa", [0x1699] = "\x35\xfb", [0x169a] = "\x35\xfc",
  [0x169b] = "\x35\xfd", [0x169c] = "\x35\xfe", [0x169d] = "\x35\xff",
  [0x169e] = "\x36\x00", [0x169f] = "\x36\x01", [0x16a0] = "\x36\x02",
  [0x16a1] = "\x36\x03", [0x16a2] = "\x36\x04", [0x16a3] = "\x36\x05",
  [0x16a4] = "\x36\x06", [0x16a5] = "\x36\x07", [0x16a6] = "\x36\x08",
  [0x16a7] = "\x36\x09", [0x16a8] = "\x36\x0a", [0x16a9] = "\x36\x0b",
  [0x16aa] = "\x36\x0c", [0x16ab] = "\x36\x0d", [0x16ac] = "\x36\x0e",
  [0x16ad] = "\x36\x0f", [0x16ae] = "\x36\x10", [0x16af] = "\x36\x11",
  [0x16b0] = "\x36\x12", [0x16b1] = "\x36\x13", [0x16b2] = "\x36\x14",
  [0x16b3] = "\x36\x15", [0x16b4] = "\x36\x16", [0x16b5] = "\x36\x17",
  [0x16b6] = "\x36\x18", [0x16b7] = "\x36\x19", [0x16b8] = "\x36\x1a",
  [0x16b9] = "\x36\x1b", [0x16ba] = "\x36\x1c", [0x16bb] = "\x36\x1d",
  [0x16bc] = "\x36\x1e", [0x16bd] = "\x36\x1f", [0x16be] = "\x36\x20",
  [0x16bf] = "\x36\x21", [0x16c0] = "\x36\x22", [0x16c1] = "\x36\x23",
  [0x16c2] = "\x36\x24", [0x16c3] = "\x36\x25", [0x16c4] = "\x36\x26",
  [0x16c5] = "\x36\x27", [0x16c6] = "\x36\x28", [0x16c7] = "\x36\x29",
  [0x16c8] = "\x36\x2a", [0x16c9] = "\x36\x2b", [0x16ca] = "\x36\x2c",
  [0x16cb] = "\x36\x2d", [0x16cc] = "\x36\x2e", [0x16cd] = "\x36\x2f",
  [0x16ce] = "\x36\x30", [0x16cf] = "\x36\x31", [0x16d0] = "\x36\x32",
  [0x16d1] = "\x36\x33", [0x16d2] = "\x36\x34", [0x16d3] = "\x36\x35",
  [0x16d4] = "\x36\x36", [0x16d5] = "\x36\x37", [0x16d6] = "\x36\x38",
  [0x16d7] = "\x36\x39", [0x16d8] = "\x36\x3a", [0x16d9] = "\x36\x3b",
  [0x16da] = "\x36\x3c", [0x16db] = "\x36\x3d", [0x16dc] = "\x36\x3e",
  [0x16dd] = "\x36\x3f", [0x16de] = "\x36\x40", [0x16df] = "\x36\x41",
  [0x16e0] = "\x36\x42", [0x16e1] = "\x36\x43", [0x16e2] = "\x36\x44",
  [0x16e3] = "\x36\x45", [0x16e4] = "\x36\x46", [0x16e5] = "\x36\x47",
  [0x16e6] = "\x36\x48", [0x16e7] = "\x36\x49", [0x16e8] = "\x36\x4a",
  [0x16e9] = "\x36\x4b", [0x16ea] = "\x36\x4c", [0x16eb] = "\x36\x4d",
  [0x16ec] = "\x36\x4e", [0x16ed] = "\x36\x4f", [0x16ee] = "\x36\x50",
  [0x16ef] = "\x36\x51", [0x16f0] = "\x36\x52", [0x16f1] = "\x36\x53",
  [0x16f2] = "\x36\x54", [0x16f3] = "\x36\x55", [0x16f4] = "\x36\x56",
  [0x16f5] = "\x36\x57", [0x16f6] = "\x36\x58", [0x16f7] = "\x36\x59",
  [0x16f8] = "\x36\x5a", [0x16f9] = "\x36\x5b", [0x16fa] = "\x36\x5c",
  [0x16fb] = "\x36\x5d", [0x16fc] = "\x36\x5e", [0x16fd] = "\x36\x5f",
  [0x16fe] = "\x36\x60", [0x16ff] = "\x36\x61", [0x1700] = "\x36\x62",
  [0x1701] = "\x36\x63", [0x1702] = "\x36\x64", [0x1703] = "\x36\x65",
  [0x1704] = "\x36\x66", [0x1705] = "\x36\x67", [0x1706] = "\x36\x68",
  [0x1707] = "\x36\x69", [0x1708] = "\x36\x6a", [0x1709] = "\x36\x6b",
  [0x170a] = "\x36\x6c", [0x170b] = "\x36\x6d", [0x170c] = "\x36\x6e",
  [0x170d] = "\x36\x6f", [0x170e] = "\x36\x70", [0x170f] = "\x36\x71",
  [0x1710] = "\x36\x72", [0x1711] = "\x36\x73", [0x1712] = "\x36\x74",
  [0x1713] = "\x36\x75", [0x1714] = "\x36\x76", [0x1715] = "\x36\x77",
  [0x1716] = "\x36\x78", [0x1717] = "\x36\x79", [0x1718] = "\x36\x7a",
  [0x1719] = "\x36\x7b", [0x171a] = "\x36\x7c", [0x171b] = "\x36\x7d",
  [0x171c] = "\x36\x7e", [0x171d] = "\x36\x7f", [0x171e] = "\x36\x80",
  [0x171f] = "\x36\x81", [0x1720] = "\x36\x82", [0x1721] = "\x36\x83",
  [0x1722] = "\x36\x84", [0x1723] = "\x36\x85", [0x1724] = "\x36\x86",
  [0x1725] = "\x36\x87", [0x1726] = "\x36\x88", [0x1727] = "\x36\x89",
  [0x1728] = "\x36\x8a", [0x1729] = "\x36\x8b", [0x172a] = "\x36\x8c",
  [0x172b] = "\x36\x8d", [0x172c] = "\x36\x8e", [0x172d] = "\x36\x8f",
  [0x172e] = "\x36\x90", [0x172f] = "\x36\x91", [0x1730] = "\x36\x92",
  [0x1731] = "\x36\x93", [0x1732] = "\x36\x94", [0x1733] = "\x36\x95",
  [0x1734] = "\x36\x96", [0x1735] = "\x36\x97", [0x1736] = "\x36\x98",
  [0x1737] = "\x36\x99", [0x1738] = "\x36\x9a", [0x1739] = "\x36\x9b",
  [0x173a] = "\x36\x9c", [0x173b] = "\x36\x9d", [0x173c] = "\x36\x9e",
  [0x173d] = "\x36\x9f", [0x173e] = "\x36\xa0", [0x173f] = "\x36\xa1",
  [0x1740] = "\x36\xa2", [0x1741] = "\x36\xa3", [0x1742] = "\x36\xa4",
  [0x1743] = "\x36\xa5", [0x1744] = "\x36\xa6", [0x1745] = "\x36\xa7",
  [0x1746] = "\x36\xa8", [0x1747] = "\x36\xa9", [0x1748] = "\x36\xaa",
  [0x1749] = "\x36\xab", [0x174a] = "\x36\xac", [0x174b] = "\x36\xad",
  [0x174c] = "\x36\xae", [0x174d] = "\x36\xaf", [0x174e] = "\x36\xb0",
  [0x174f] = "\x36\xb1", [0x1750] = "\x36\xb2", [0x1751] = "\x36\xb3",
  [0x1752] = "\x36\xb4", [0x1753] = "\x36\xb5", [0x1754] = "\x36\xb6",
  [0x1755] = "\x36\xb7", [0x1756] = "\x36\xb8", [0x1757] = "\x36\xb9",
  [0x1758] = "\x36\xba", [0x1759] = "\x36\xbb", [0x175a] = "\x36\xbc",
  [0x175b] = "\x36\xbd", [0x175c] = "\x36\xbe", [0x175d] = "\x36\xbf",
  [0x175e] = "\x36\xc0", [0x175f] = "\x36\xc1", [0x1760] = "\x36\xc2",
  [0x1761] = "\x36\xc3", [0x1762] = "\x36\xc4", [0x1763] = "\x36\xc5",
  [0x1764] = "\x36\xc6", [0x1765] = "\x36\xc7", [0x1766] = "\x36\xc8",
  [0x1767] = "\x36\xc9", [0x1768] = "\x36\xca", [0x1769] = "\x36\xcb",
  [0x176a] = "\x36\xcc", [0x176b] = "\x36\xcd", [0x176c] = "\x36\xce",
  [0x176d] = "\x36\xcf", [0x176e] = "\x36\xd0", [0x176f] = "\x36\xd1",
  [0x1770] = "\x36\xd2", [0x1771] = "\x36\xd3", [0x1772] = "\x36\xd4",
  [0x1773] = "\x36\xd5", [0x1774] = "\x36\xd6", [0x1775] = "\x36\xd7",
  [0x1776] = "\x36\xd8", [0x1777] = "\x36\xd9", [0x1778] = "\x36\xda",
  [0x1779] = "\x36\xdb", [0x177a] = "\x36\xdc", [0x177b] = "\x36\xdd",
  [0x177c] = "\x36\xde", [0x177d] = "\x36\xdf", [0x177e] = "\x36\xe0",
  [0x177f] = "\x36\xe1", [0x1780] = "\x36\xe2", [0x1781] = "\x36\xe3",
  [0x1782] = "\x36\xe4", [0x1783] = "\x36\xe5", [0x1784] = "\x36\xe6",
  [0x1785] = "\x36\xe7", [0x1786] = "\x36\xe8", [0x1787] = "\x36\xe9",
  [0x1788] = "\x36\xea", [0x1789] = "\x36\xeb", [0x178a] = "\x36\xec",
  [0x178b] = "\x36\xed", [0x178c] = "\x36\xee", [0x178d] = "\x36\xef",
  [0x178e] = "\x36\xf0", [0x178f] = "\x36\xf1", [0x1790] = "\x36\xf2",
  [0x1791] = "\x36\xf3", [0x1792] = "\x36\xf4", [0x1793] = "\x36\xf5",
  [0x1794] = "\x36\xf6", [0x1795] = "\x36\xf7", [0x1796] = "\x36\xf8",
  [0x1797] = "\x36\xf9", [0x1798] = "\x36\xfa", [0x1799] = "\x36\xfb",
  [0x179a] = "\x36\xfc", [0x179b] = "\x36\xfd", [0x179c] = "\x36\xfe",
  [0x179d] = "\x36\xff", [0x179e] = "\x37\x00", [0x179f] = "\x37\x01",
  [0x17a0] = "\x37\x02", [0x17a1] = "\x37\x03", [0x17a2] = "\x37\x04",
  [0x17a3] = "\x37\x05", [0x17a4] = "\x37\x06", [0x17a5] = "\x37\x07",
  [0x17a6] = "\x37\x08", [0x17a7] = "\x37\x09", [0x17a8] = "\x37\x0a",
  [0x17a9] = "\x37\x0b", [0x17aa] = "\x37\x0c", [0x17ab] = "\x37\x0d",
  [0x17ac] = "\x37\x0e", [0x17ad] = "\x37\x0f", [0x17ae] = "\x37\x10",
  [0x17af] = "\x37\x11", [0x17b0] = "\x37\x12", [0x17b1] = "\x37\x13",
  [0x17b2] = "\x37\x14", [0x17b3] = "\x37\x15", [0x17b4] = "\x37\x16",
  [0x17b5] = "\x37\x17", [0x17b6] = "\x37\x18", [0x17b7] = "\x37\x19",
  [0x17b8] = "\x37\x1a", [0x17b9] = "\x37\x1b", [0x17ba] = "\x37\x1c",
  [0x17bb] = "\x37\x1d", [0x17bc] = "\x37\x1e", [0x17bd] = "\x37\x1f",
  [0x17be] = "\x37\x20", [0x17bf] = "\x37\x21", [0x17c0] = "\x37\x22",
  [0x17c1] = "\x37\x23", [0x17c2] = "\x37\x24", [0x17c3] = "\x37\x25",
  [0x17c4] = "\x37\x26", [0x17c5] = "\x37\x27", [0x17c6] = "\x37\x28",
  [0x17c7] = "\x37\x29", [0x17c8] = "\x37\x2a", [0x17c9] = "\x37\x2b",
  [0x17ca] = "\x37\x2c", [0x17cb] = "\x37\x2d", [0x17cc] = "\x37\x2e",
  [0x17cd] = "\x37\x2f", [0x17ce] = "\x37\x30", [0x17cf] = "\x37\x31",
  [0x17d0] = "\x37\x32", [0x17d1] = "\x37\x33", [0x17d2] = "\x37\x34",
  [0x17d3] = "\x37\x35", [0x17d4] = "\x37\x36", [0x17d5] = "\x37\x37",
  [0x17d6] = "\x37\x38", [0x17d7] = "\x37\x39", [0x17d8] = "\x37\x3a",
  [0x17d9] = "\x37\x3b", [0x17da] = "\x37\x3c", [0x17db] = "\x37\x3d",
  [0x17dc] = "\x37\x3e", [0x17dd] = "\x37\x3f", [0x17de] = "\x37\x40",
  [0x17df] = "\x37\x41", [0x17e0] = "\x37\x42", [0x17e1] = "\x37\x43",
  [0x17e2] = "\x37\x44", [0x17e3] = "\x37\x45", [0x17e4] = "\x37\x46",
  [0x17e5] = "\x37\x47", [0x17e6] = "\x37\x48", [0x17e7] = "\x37\x49",
  [0x17e8] = "\x37\x4a", [0x17e9] = "\x37\x4b", [0x17ea] = "\x37\x4c",
  [0x17eb] = "\x37\x4d", [0x17ec] = "\x37\x4e", [0x17ed] = "\x37\x4f",
  [0x17ee] = "\x37\x50", [0x17ef] = "\x37\x51", [0x17f0] = "\x37\x52",
  [0x17f1] = "\x37\x53", [0x17f2] = "\x37\x54", [0x17f3] = "\x37\x55",
  [0x17f4] = "\x37\x56", [0x17f5] = "\x37\x57", [0x17f6] = "\x37\x58",
  [0x17f7] = "\x37\x59", [0x17f8] = "\x37\x5a", [0x17f9] = "\x37\x5b",
  [0x17fa] = "\x37\x5c", [0x17fb] = "\x37\x5d", [0x17fc] = "\x37\x5e",
  [0x17fd] = "\x37\x5f", [0x17fe] = "\x37\x60", [0x17ff] = "\x37\x61",
  [0x1800] = "\x37\x62", [0x1801] = "\x37\x63", [0x1802] = "\x37\x64",
  [0x1803] = "\x37\x65", [0x1804] = "\x37\x66", [0x1805] = "\x37\x67",
  [0x1806] = "\x37\x68", [0x1807] = "\x37\x69", [0x1808] = "\x37\x6a",
  [0x1809] = "\x37\x6b", [0x180a] = "\x37\x6c", [0x180b] = "\x37\x6d",
  [0x180c] = "\x37\x6e", [0x180d] = "\x37\x6f", [0x180e] = "\x37\x70",
  [0x180f] = "\x37\x71", [0x1810] = "\x37\x72", [0x1811] = "\x37\x73",
  [0x1812] = "\x37\x74", [0x1813] = "\x37\x75", [0x1814] = "\x37\x76",
  [0x1815] = "\x37\x77", [0x1816] = "\x37\x78", [0x1817] = "\x37\x79",
  [0x1818] = "\x37\x7a", [0x1819] = "\x37\x7b", [0x181a] = "\x37\x7c",
  [0x181b] = "\x37\x7d", [0x181c] = "\x37\x7e", [0x181d] = "\x37\x7f",
  [0x181e] = "\x37\x80", [0x181f] = "\x37\x81", [0x1820] = "\x37\x82",
  [0x1821] = "\x37\x83", [0x1822] = "\x37\x84", [0x1823] = "\x37\x85",
  [0x1824] = "\x37\x86", [0x1825] = "\x37\x87", [0x1826] = "\x37\x88",
  [0x1827] = "\x37\x89", [0x1828] = "\x37\x8a", [0x1829] = "\x37\x8b",
  [0x182a] = "\x37\x8c", [0x182b] = "\x37\x8d", [0x182c] = "\x37\x8e",
  [0x182d] = "\x37\x8f", [0x182e] = "\x37\x90", [0x182f] = "\x37\x91",
  [0x1830] = "\x37\x92", [0x1831] = "\x37\x93", [0x1832] = "\x37\x94",
  [0x1833] = "\x37\x95", [0x1834] = "\x37\x96", [0x1835] = "\x37\x97",
  [0x1836] = "\x37\x98", [0x1837] = "\x37\x99", [0x1838] = "\x37\x9a",
  [0x1839] = "\x37\x9b", [0x183a] = "\x37\x9c", [0x183b] = "\x37\x9d",
  [0x183c] = "\x37\x9e", [0x183d] = "\x37\x9f", [0x183e] = "\x37\xa0",
  [0x183f] = "\x37\xa1", [0x1840] = "\x37\xa2", [0x1841] = "\x37\xa3",
  [0x1842] = "\x37\xa4", [0x1843] = "\x37\xa5", [0x1844] = "\x37\xa6",
  [0x1845] = "\x37\xa7", [0x1846] = "\x37\xa8", [0x1847] = "\x37\xa9",
  [0x1848] = "\x37\xaa", [0x1849] = "\x37\xab", [0x184a] = "\x37\xac",
  [0x184b] = "\x37\xad", [0x184c] = "\x37\xae", [0x184d] = "\x37\xaf",
  [0x184e] = "\x37\xb0", [0x184f] = "\x37\xb1", [0x1850] = "\x37\xb2",
  [0x1851] = "\x37\xb3", [0x1852] = "\x37\xb4", [0x1853] = "\x37\xb5",
  [0x1854] = "\x37\xb6", [0x1855] = "\x37\xb7", [0x1856] = "\x37\xb8",
  [0x1857] = "\x37\xb9", [0x1858] = "\x37\xba", [0x1859] = "\x37\xbb",
  [0x185a] = "\x37\xbc", [0x185b] = "\x37\xbd", [0x185c] = "\x37\xbe",
  [0x185d] = "\x37\xbf", [0x185e] = "\x37\xc0", [0x185f] = "\x37\xc1",
  [0x1860] = "\x37\xc2", [0x1861] = "\x37\xc3", [0x1862] = "\x37\xc4",
  [0x1863] = "\x37\xc5", [0x1864] = "\x37\xc6", [0x1865] = "\x37\xc7",
  [0x1866] = "\x37\xc8", [0x1867] = "\x37\xc9", [0x1868] = "\x37\xca",
  [0x1869] = "\x37\xcb", [0x186a] = "\x37\xcc", [0x186b] = "\x37\xcd",
  [0x186c] = "\x37\xce", [0x186d] = "\x37\xcf", [0x186e] = "\x37\xd0",
  [0x186f] = "\x37\xd1", [0x1870] = "\x37\xd2", [0x1871] = "\x37\xd3",
  [0x1872] = "\x37\xd4", [0x1873] = "\x37\xd5", [0x1874] = "\x37\xd6",
  [0x1875] = "\x37\xd7", [0x1876] = "\x37\xd8", [0x1877] = "\x37\xd9",
  [0x1878] = "\x37\xda", [0x1879] = "\x37\xdb", [0x187a] = "\x37\xdc",
  [0x187b] = "\x37\xdd", [0x187c] = "\x37\xde", [0x187d] = "\x37\xdf",
  [0x187e] = "\x37\xe0", [0x187f] = "\x37\xe1", [0x1880] = "\x37\xe2",
  [0x1881] = "\x37\xe3", [0x1882] = "\x37\xe4", [0x1883] = "\x37\xe5",
  [0x1884] = "\x37\xe6", [0x1885] = "\x37\xe7", [0x1886] = "\x37\xe8",
  [0x1887] = "\x37\xe9", [0x1888] = "\x37\xea", [0x1889] = "\x37\xeb",
  [0x188a] = "\x37\xec", [0x188b] = "\x37\xed", [0x188c] = "\x37\xee",
  [0x188d] = "\x37\xef", [0x188e] = "\x37\xf0", [0x188f] = "\x37\xf1",
  [0x1890] = "\x37\xf2", [0x1891] = "\x37\xf3", [0x1892] = "\x37\xf4",
  [0x1893] = "\x37\xf5", [0x1894] = "\x37\xf6", [0x1895] = "\x37\xf7",
  [0x1896] = "\x37\xf8", [0x1897] = "\x37\xf9", [0x1898] = "\x37\xfa",
  [0x1899] = "\x37\xfb", [0x189a] = "\x37\xfc", [0x189b] = "\x37\xfd",
  [0x189c] = "\x37\xfe", [0x189d] = "\x37\xff", [0x189e] = "\x38\x00",
  [0x189f] = "\x38\x01", [0x18a0] = "\x38\x02", [0x18a1] = "\x38\x03",
  [0x18a2] = "\x38\x04", [0x18a3] = "\x38\x05", [0x18a4] = "\x38\x06",
  [0x18a5] = "\x38\x07", [0x18a6] = "\x38\x08", [0x18a7] = "\x38\x09",
  [0x18a8] = "\x38\x0a", [0x18a9] = "\x38\x0b", [0x18aa] = "\x38\x0c",
  [0x18ab] = "\x38\x0d", [0x18ac] = "\x38\x0e", [0x18ad] = "\x38\x0f",
  [0x18ae] = "\x38\x10", [0x18af] = "\x38\x11", [0x18b0] = "\x38\x12",
  [0x18b1] = "\x38\x13", [0x18b2] = "\x38\x14", [0x18b3] = "\x38\x15",
  [0x18b4] = "\x38\x16", [0x18b5] = "\x38\x17", [0x18b6] = "\x38\x18",
  [0x18b7] = "\x38\x19", [0x18b8] = "\x38\x1a", [0x18b9] = "\x38\x1b",
  [0x18ba] = "\x38\x1c", [0x18bb] = "\x38\x1d", [0x18bc] = "\x38\x1e",
  [0x18bd] = "\x38\x1f", [0x18be] = "\x38\x20", [0x18bf] = "\x38\x21",
  [0x18c0] = "\x38\x22", [0x18c1] = "\x38\x23", [0x18c2] = "\x38\x24",
  [0x18c3] = "\x38\x25", [0x18c4] = "\x38\x26", [0x18c5] = "\x38\x27",
  [0x18c6] = "\x38\x28", [0x18c7] = "\x38\x29", [0x18c8] = "\x38\x2a",
  [0x18c9] = "\x38\x2b", [0x18ca] = "\x38\x2c", [0x18cb] = "\x38\x2d",
  [0x18cc] = "\x38\x2e", [0x18cd] = "\x38\x2f", [0x18ce] = "\x38\x30",
  [0x18cf] = "\x38\x31", [0x18d0] = "\x38\x32", [0x18d1] = "\x38\x33",
  [0x18d2] = "\x38\x34", [0x18d3] = "\x38\x35", [0x18d4] = "\x38\x36",
  [0x18d5] = "\x38\x37", [0x18d6] = "\x38\x38", [0x18d7] = "\x38\x39",
  [0x18d8] = "\x38\x3a", [0x18d9] = "\x38\x3b", [0x18da] = "\x38\x3c",
  [0x18db] = "\x38\x3d", [0x18dc] = "\x38\x3e", [0x18dd] = "\x38\x3f",
  [0x18de] = "\x38\x40", [0x18df] = "\x38\x41", [0x18e0] = "\x38\x42",
  [0x18e1] = "\x38\x43", [0x18e2] = "\x38\x44", [0x18e3] = "\x38\x45",
  [0x18e4] = "\x38\x46", [0x18e5] = "\x38\x47", [0x18e6] = "\x38\x48",
  [0x18e7] = "\x38\x49", [0x18e8] = "\x38\x4a", [0x18e9] = "\x38\x4b",
  [0x18ea] = "\x38\x4c", [0x18eb] = "\x38\x4d", [0x18ec] = "\x38\x4e",
  [0x18ed] = "\x38\x4f", [0x18ee] = "\x38\x50", [0x18ef] = "\x38\x51",
  [0x18f0] = "\x38\x52", [0x18f1] = "\x38\x53", [0x18f2] = "\x38\x54",
  [0x18f3] = "\x38\x55", [0x18f4] = "\x38\x56", [0x18f5] = "\x38\x57",
  [0x18f6] = "\x38\x58", [0x18f7] = "\x38\x59", [0x18f8] = "\x38\x5a",
  [0x18f9] = "\x38\x5b", [0x18fa] = "\x38\x5c", [0x18fb] = "\x38\x5d",
  [0x18fc] = "\x38\x5e", [0x18fd] = "\x38\x5f", [0x18fe] = "\x38\x60",
  [0x18ff] = "\x38\x61", [0x1900] = "\x38\x62", [0x1901] = "\x38\x63",
  [0x1902] = "\x38\x64", [0x1903] = "\x38\x65", [0x1904] = "\x38\x66",
  [0x1905] = "\x38\x67", [0x1906] = "\x38\x68", [0x1907] = "\x38\x69",
  [0x1908] = "\x38\x6a", [0x1909] = "\x38\x6b", [0x190a] = "\x38\x6c",
  [0x190b] = "\x38\x6d", [0x190c] = "\x38\x6e", [0x190d] = "\x38\x6f",
  [0x190e] = "\x38\x70", [0x190f] = "\x38\x71", [0x1910] = "\x38\x72",
  [0x1911] = "\x38\x73", [0x1912] = "\x38\x74", [0x1913] = "\x38\x75",
  [0x1914] = "\x38\x76", [0x1915] = "\x38\x77", [0x1916] = "\x38\x78",
  [0x1917] = "\x38\x79", [0x1918] = "\x38\x7a", [0x1919] = "\x38\x7b",
  [0x191a] = "\x38\x7c", [0x191b] = "\x38\x7d", [0x191c] = "\x38\x7e",
  [0x191d] = "\x38\x7f", [0x191e] = "\x38\x80", [0x191f] = "\x38\x81",
  [0x1920] = "\x38\x82", [0x1921] = "\x38\x83", [0x1922] = "\x38\x84",
  [0x1923] = "\x38\x85", [0x1924] = "\x38\x86", [0x1925] = "\x38\x87",
  [0x1926] = "\x38\x88", [0x1927] = "\x38\x89", [0x1928] = "\x38\x8a",
  [0x1929] = "\x38\x8b", [0x192a] = "\x38\x8c", [0x192b] = "\x38\x8d",
  [0x192c] = "\x38\x8e", [0x192d] = "\x38\x8f", [0x192e] = "\x38\x90",
  [0x192f] = "\x38\x91", [0x1930] = "\x38\x92", [0x1931] = "\x38\x93",
  [0x1932] = "\x38\x94", [0x1933] = "\x38\x95", [0x1934] = "\x38\x96",
  [0x1935] = "\x38\x97", [0x1936] = "\x38\x98", [0x1937] = "\x38\x99",
  [0x1938] = "\x38\x9a", [0x1939] = "\x38\x9b", [0x193a] = "\x38\x9c",
  [0x193b] = "\x38\x9d", [0x193c] = "\x38\x9e", [0x193d] = "\x38\x9f",
  [0x193e] = "\x38\xa0", [0x193f] = "\x38\xa1", [0x1940] = "\x38\xa2",
  [0x1941] = "\x38\xa3", [0x1942] = "\x38\xa4", [0x1943] = "\x38\xa5",
  [0x1944] = "\x38\xa6", [0x1945] = "\x38\xa7", [0x1946] = "\x38\xa8",
  [0x1947] = "\x38\xa9", [0x1948] = "\x38\xaa", [0x1949] = "\x38\xab",
  [0x194a] = "\x38\xac", [0x194b] = "\x38\xad", [0x194c] = "\x38\xae",
  [0x194d] = "\x38\xaf", [0x194e] = "\x38\xb0", [0x194f] = "\x38\xb1",
  [0x1950] = "\x38\xb2", [0x1951] = "\x38\xb3", [0x1952] = "\x38\xb4",
  [0x1953] = "\x38\xb5", [0x1954] = "\x38\xb6", [0x1955] = "\x38\xb7",
  [0x1956] = "\x38\xb8", [0x1957] = "\x38\xb9", [0x1958] = "\x38\xba",
  [0x1959] = "\x38\xbb", [0x195a] = "\x38\xbc", [0x195b] = "\x38\xbd",
  [0x195c] = "\x38\xbe", [0x195d] = "\x38\xbf", [0x195e] = "\x38\xc0",
  [0x195f] = "\x38\xc1", [0x1960] = "\x38\xc2", [0x1961] = "\x38\xc3",
  [0x1962] = "\x38\xc4", [0x1963] = "\x38\xc5", [0x1964] = "\x38\xc6",
  [0x1965] = "\x38\xc7", [0x1966] = "\x38\xc8", [0x1967] = "\x38\xc9",
  [0x1968] = "\x38\xca", [0x1969] = "\x38\xcb", [0x196a] = "\x38\xcc",
  [0x196b] = "\x38\xcd", [0x196c] = "\x38\xce", [0x196d] = "\x38\xcf",
  [0x196e] = "\x38\xd0", [0x196f] = "\x38\xd1", [0x1970] = "\x38\xd2",
  [0x1971] = "\x38\xd3", [0x1972] = "\x38\xd4", [0x1973] = "\x38\xd5",
  [0x1974] = "\x38\xd6", [0x1975] = "\x38\xd7", [0x1976] = "\x38\xd8",
  [0x1977] = "\x38\xd9", [0x1978] = "\x38\xda", [0x1979] = "\x38\xdb",
  [0x197a] = "\x38\xdc", [0x197b] = "\x38\xdd", [0x197c] = "\x38\xde",
  [0x197d] = "\x38\xdf", [0x197e] = "\x38\xe0", [0x197f] = "\x38\xe1",
  [0x1980] = "\x38\xe2", [0x1981] = "\x38\xe3", [0x1982] = "\x38\xe4",
  [0x1983] = "\x38\xe5", [0x1984] = "\x38\xe6", [0x1985] = "\x38\xe7",
  [0x1986] = "\x38\xe8", [0x1987] = "\x38\xe9", [0x1988] = "\x38\xea",
  [0x1989] = "\x38\xeb", [0x198a] = "\x38\xec", [0x198b] = "\x38\xed",
  [0x198c] = "\x38\xee", [0x198d] = "\x38\xef", [0x198e] = "\x38\xf0",
  [0x198f] = "\x38\xf1", [0x1990] = "\x38\xf2", [0x1991] = "\x38\xf3",
  [0x1992] = "\x38\xf4", [0x1993] = "\x38\xf5", [0x1994] = "\x38\xf6",
  [0x1995] = "\x38\xf7", [0x1996] = "\x38\xf8", [0x1997] = "\x38\xf9",
  [0x1998] = "\x38\xfa", [0x1999] = "\x38\xfb", [0x199a] = "\x38\xfc",
  [0x199b] = "\x38\xfd", [0x199c] = "\x38\xfe", [0x199d] = "\x38\xff",
  [0x199e] = "\x39\x00", [0x199f] = "\x39\x01", [0x19a0] = "\x39\x02",
  [0x19a1] = "\x39\x03", [0x19a2] = "\x39\x04", [0x19a3] = "\x39\x05",
  [0x19a4] = "\x39\x06", [0x19a5] = "\x39\x07", [0x19a6] = "\x39\x08",
  [0x19a7] = "\x39\x09", [0x19a8] = "\x39\x0a", [0x19a9] = "\x39\x0b",
  [0x19aa] = "\x39\x0c", [0x19ab] = "\x39\x0d", [0x19ac] = "\x39\x0e",
  [0x19ad] = "\x39\x0f", [0x19ae] = "\x39\x10", [0x19af] = "\x39\x11",
  [0x19b0] = "\x39\x12", [0x19b1] = "\x39\x13", [0x19b2] = "\x39\x14",
  [0x19b3] = "\x39\x15", [0x19b4] = "\x39\x16", [0x19b5] = "\x39\x17",
  [0x19b6] = "\x39\x18", [0x19b7] = "\x39\x19", [0x19b8] = "\x39\x1a",
  [0x19b9] = "\x39\x1b", [0x19ba] = "\x39\x1c", [0x19bb] = "\x39\x1d",
  [0x19bc] = "\x39\x1e", [0x19bd] = "\x39\x1f", [0x19be] = "\x39\x20",
  [0x19bf] = "\x39\x21", [0x19c0] = "\x39\x22", [0x19c1] = "\x39\x23",
  [0x19c2] = "\x39\x24", [0x19c3] = "\x39\x25", [0x19c4] = "\x39\x26",
  [0x19c5] = "\x39\x27", [0x19c6] = "\x39\x28", [0x19c7] = "\x39\x29",
  [0x19c8] = "\x39\x2a", [0x19c9] = "\x39\x2b", [0x19ca] = "\x39\x2c",
  [0x19cb] = "\x39\x2d", [0x19cc] = "\x39\x2e", [0x19cd] = "\x39\x2f",
  [0x19ce] = "\x39\x30", [0x19cf] = "\x39\x31", [0x19d0] = "\x39\x32",
  [0x19d1] = "\x39\x33", [0x19d2] = "\x39\x34", [0x19d3] = "\x39\x35",
  [0x19d4] = "\x39\x36", [0x19d5] = "\x39\x37", [0x19d6] = "\x39\x38",
  [0x19d7] = "\x39\x39", [0x19d8] = "\x39\x3a", [0x19d9] = "\x39\x3b",
  [0x19da] = "\x39\x3c", [0x19db] = "\x39\x3d", [0x19dc] = "\x39\x3e",
  [0x19dd] = "\x39\x3f", [0x19de] = "\x39\x40", [0x19df] = "\x39\x41",
  [0x19e0] = "\x39\x42", [0x19e1] = "\x39\x43", [0x19e2] = "\x39\x44",
  [0x19e3] = "\x39\x45", [0x19e4] = "\x39\x46", [0x19e5] = "\x39\x47",
  [0x19e6] = "\x39\x48", [0x19e7] = "\x39\x49", [0x19e8] = "\x39\x4a",
  [0x19e9] = "\x39\x4b", [0x19ea] = "\x39\x4c", [0x19eb] = "\x39\x4d",
  [0x19ec] = "\x39\x4e", [0x19ed] = "\x39\x4f", [0x19ee] = "\x39\x50",
  [0x19ef] = "\x39\x51", [0x19f0] = "\x39\x52", [0x19f1] = "\x39\x53",
  [0x19f2] = "\x39\x54", [0x19f3] = "\x39\x55", [0x19f4] = "\x39\x56",
  [0x19f5] = "\x39\x57", [0x19f6] = "\x39\x58", [0x19f7] = "\x39\x59",
  [0x19f8] = "\x39\x5a", [0x19f9] = "\x39\x5b", [0x19fa] = "\x39\x5c",
  [0x19fb] = "\x39\x5d", [0x19fc] = "\x39\x5e", [0x19fd] = "\x39\x5f",
  [0x19fe] = "\x39\x60", [0x19ff] = "\x39\x61", [0x1a00] = "\x39\x62",
  [0x1a01] = "\x39\x63", [0x1a02] = "\x39\x64", [0x1a03] = "\x39\x65",
  [0x1a04] = "\x39\x66", [0x1a05] = "\x39\x67", [0x1a06] = "\x39\x68",
  [0x1a07] = "\x39\x69", [0x1a08] = "\x39\x6a", [0x1a09] = "\x39\x6b",
  [0x1a0a] = "\x39\x6c", [0x1a0b] = "\x39\x6d", [0x1a0c] = "\x39\x6e",
  [0x1a0d] = "\x39\x6f", [0x1a0e] = "\x39\x70", [0x1a0f] = "\x39\x71",
  [0x1a10] = "\x39\x72", [0x1a11] = "\x39\x73", [0x1a12] = "\x39\x74",
  [0x1a13] = "\x39\x75", [0x1a14] = "\x39\x76", [0x1a15] = "\x39\x77",
  [0x1a16] = "\x39\x78", [0x1a17] = "\x39\x79", [0x1a18] = "\x39\x7a",
  [0x1a19] = "\x39\x7b", [0x1a1a] = "\x39\x7c", [0x1a1b] = "\x39\x7d",
  [0x1a1c] = "\x39\x7e", [0x1a1d] = "\x39\x7f", [0x1a1e] = "\x39\x80",
  [0x1a1f] = "\x39\x81", [0x1a20] = "\x39\x82", [0x1a21] = "\x39\x83",
  [0x1a22] = "\x39\x84", [0x1a23] = "\x39\x85", [0x1a24] = "\x39\x86",
  [0x1a25] = "\x39\x87", [0x1a26] = "\x39\x88", [0x1a27] = "\x39\x89",
  [0x1a28] = "\x39\x8a", [0x1a29] = "\x39\x8b", [0x1a2a] = "\x39\x8c",
  [0x1a2b] = "\x39\x8d", [0x1a2c] = "\x39\x8e", [0x1a2d] = "\x39\x8f",
  [0x1a2e] = "\x39\x90", [0x1a2f] = "\x39\x91", [0x1a30] = "\x39\x92",
  [0x1a31] = "\x39\x93", [0x1a32] = "\x39\x94", [0x1a33] = "\x39\x95",
  [0x1a34] = "\x39\x96", [0x1a35] = "\x39\x97", [0x1a36] = "\x39\x98",
  [0x1a37] = "\x39\x99", [0x1a38] = "\x39\x9a", [0x1a39] = "\x39\x9b",
  [0x1a3a] = "\x39\x9c", [0x1a3b] = "\x39\x9d", [0x1a3c] = "\x39\x9e",
  [0x1a3d] = "\x39\x9f", [0x1a3e] = "\x39\xa0", [0x1a3f] = "\x39\xa1",
  [0x1a40] = "\x39\xa2", [0x1a41] = "\x39\xa3", [0x1a42] = "\x39\xa4",
  [0x1a43] = "\x39\xa5", [0x1a44] = "\x39\xa6", [0x1a45] = "\x39\xa7",
  [0x1a46] = "\x39\xa8", [0x1a47] = "\x39\xa9", [0x1a48] = "\x39\xaa",
  [0x1a49] = "\x39\xab", [0x1a4a] = "\x39\xac", [0x1a4b] = "\x39\xad",
  [0x1a4c] = "\x39\xae", [0x1a4d] = "\x39\xaf", [0x1a4e] = "\x39\xb0",
  [0x1a4f] = "\x39\xb1", [0x1a50] = "\x39\xb2", [0x1a51] = "\x39\xb3",
  [0x1a52] = "\x39\xb4", [0x1a53] = "\x39\xb5", [0x1a54] = "\x39\xb6",
  [0x1a55] = "\x39\xb7", [0x1a56] = "\x39\xb8", [0x1a57] = "\x39\xb9",
  [0x1a58] = "\x39\xba", [0x1a59] = "\x39\xbb", [0x1a5a] = "\x39\xbc",
  [0x1a5b] = "\x39\xbd", [0x1a5c] = "\x39\xbe", [0x1a5d] = "\x39\xbf",
  [0x1a5e] = "\x39\xc0", [0x1a5f] = "\x39\xc1", [0x1a60] = "\x39\xc2",
  [0x1a61] = "\x39\xc3", [0x1a62] = "\x39\xc4", [0x1a63] = "\x39\xc5",
  [0x1a64] = "\x39\xc6", [0x1a65] = "\x39\xc7", [0x1a66] = "\x39\xc8",
  [0x1a67] = "\x39\xc9", [0x1a68] = "\x39\xca", [0x1a69] = "\x39\xcb",
  [0x1a6a] = "\x39\xcc", [0x1a6b] = "\x39\xcd", [0x1a6c] = "\x39\xce",
  [0x1a6d] = "\x39\xcf", [0x1a6e] = "\x39\xd0", [0x1a6f] = "\x39\xd1",
  [0x1a70] = "\x39\xd2", [0x1a71] = "\x39\xd3", [0x1a72] = "\x39\xd4",
  [0x1a73] = "\x39\xd5", [0x1a74] = "\x39\xd6", [0x1a75] = "\x39\xd7",
  [0x1a76] = "\x39\xd8", [0x1a77] = "\x39\xd9", [0x1a78] = "\x39\xda",
  [0x1a79] = "\x39\xdb", [0x1a7a] = "\x39\xdc", [0x1a7b] = "\x39\xdd",
  [0x1a7c] = "\x39\xde", [0x1a7d] = "\x39\xdf", [0x1a7e] = "\x39\xe0",
  [0x1a7f] = "\x39\xe1", [0x1a80] = "\x39\xe2", [0x1a81] = "\x39\xe3",
  [0x1a82] = "\x39\xe4", [0x1a83] = "\x39\xe5", [0x1a84] = "\x39\xe6",
  [0x1a85] = "\x39\xe7", [0x1a86] = "\x39\xe8", [0x1a87] = "\x39\xe9",
  [0x1a88] = "\x39\xea", [0x1a89] = "\x39\xeb", [0x1a8a] = "\x39\xec",
  [0x1a8b] = "\x39\xed", [0x1a8c] = "\x39\xee", [0x1a8d] = "\x39\xef",
  [0x1a8e] = "\x39\xf0", [0x1a8f] = "\x39\xf1", [0x1a90] = "\x39\xf2",
  [0x1a91] = "\x39\xf3", [0x1a92] = "\x39\xf4", [0x1a93] = "\x39\xf5",
  [0x1a94] = "\x39\xf6", [0x1a95] = "\x39\xf7", [0x1a96] = "\x39\xf8",
  [0x1a97] = "\x39\xf9", [0x1a98] = "\x39\xfa", [0x1a99] = "\x39\xfb",
  [0x1a9a] = "\x39\xfc", [0x1a9b] = "\x39\xfd", [0x1a9c] = "\x39\xfe",
  [0x1a9d] = "\x39\xff", [0x1a9e] = "\x3a\x00", [0x1a9f] = "\x3a\x01",
  [0x1aa0] = "\x3a\x02", [0x1aa1] = "\x3a\x03", [0x1aa2] = "\x3a\x04",
  [0x1aa3] = "\x3a\x05", [0x1aa4] = "\x3a\x06", [0x1aa5] = "\x3a\x07",
  [0x1aa6] = "\x3a\x08", [0x1aa7] = "\x3a\x09", [0x1aa8] = "\x3a\x0a",
  [0x1aa9] = "\x3a\x0b", [0x1aaa] = "\x3a\x0c", [0x1aab] = "\x3a\x0d",
  [0x1aac] = "\x3a\x0e", [0x1aad] = "\x3a\x0f", [0x1aae] = "\x3a\x10",
  [0x1aaf] = "\x3a\x11", [0x1ab0] = "\x3a\x12", [0x1ab1] = "\x3a\x13",
  [0x1ab2] = "\x3a\x14", [0x1ab3] = "\x3a\x15", [0x1ab4] = "\x3a\x16",
  [0x1ab5] = "\x3a\x17", [0x1ab6] = "\x3a\x18", [0x1ab7] = "\x3a\x19",
  [0x1ab8] = "\x3a\x1a", [0x1ab9] = "\x3a\x1b", [0x1aba] = "\x3a\x1c",
  [0x1abb] = "\x3a\x1d", [0x1abc] = "\x3a\x1e", [0x1abd] = "\x3a\x1f",
  [0x1abe] = "\x3a\x20", [0x1abf] = "\x3a\x21", [0x1ac0] = "\x3a\x22",
  [0x1ac1] = "\x3a\x23", [0x1ac2] = "\x3a\x24", [0x1ac3] = "\x3a\x25",
  [0x1ac4] = "\x3a\x26", [0x1ac5] = "\x3a\x27", [0x1ac6] = "\x3a\x28",
  [0x1ac7] = "\x3a\x29", [0x1ac8] = "\x3a\x2a", [0x1ac9] = "\x3a\x2b",
  [0x1aca] = "\x3a\x2c", [0x1acb] = "\x3a\x2d", [0x1acc] = "\x3a\x2e",
  [0x1acd] = "\x3a\x2f", [0x1ace] = "\x3a\x30", [0x1acf] = "\x3a\x31",
  [0x1ad0] = "\x3a\x32", [0x1ad1] = "\x3a\x33", [0x1ad2] = "\x3a\x34",
  [0x1ad3] = "\x3a\x35", [0x1ad4] = "\x3a\x36", [0x1ad5] = "\x3a\x37",
  [0x1ad6] = "\x3a\x38", [0x1ad7] = "\x3a\x39", [0x1ad8] = "\x3a\x3a",
  [0x1ad9] = "\x3a\x3b", [0x1ada] = "\x3a\x3c", [0x1adb] = "\x3a\x3d",
  [0x1adc] = "\x3a\x3e", [0x1add] = "\x3a\x3f", [0x1ade] = "\x3a\x40",
  [0x1adf] = "\x3a\x41", [0x1ae0] = "\x3a\x42", [0x1ae1] = "\x3a\x43",
  [0x1ae2] = "\x3a\x44", [0x1ae3] = "\x3a\x45", [0x1ae4] = "\x3a\x46",
  [0x1ae5] = "\x3a\x47", [0x1ae6] = "\x3a\x48", [0x1ae7] = "\x3a\x49",
  [0x1ae8] = "\x3a\x4a", [0x1ae9] = "\x3a\x4b", [0x1aea] = "\x3a\x4c",
  [0x1aeb] = "\x3a\x4d", [0x1aec] = "\x3a\x4e", [0x1aed] = "\x3a\x4f",
  [0x1aee] = "\x3a\x50", [0x1aef] = "\x3a\x51", [0x1af0] = "\x3a\x52",
  [0x1af1] = "\x3a\x53", [0x1af2] = "\x3a\x54", [0x1af3] = "\x3a\x55",
  [0x1af4] = "\x3a\x56", [0x1af5] = "\x3a\x57", [0x1af6] = "\x3a\x58",
  [0x1af7] = "\x3a\x59", [0x1af8] = "\x3a\x5a", [0x1af9] = "\x3a\x5b",
  [0x1afa] = "\x3a\x5c", [0x1afb] = "\x3a\x5d", [0x1afc] = "\x3a\x5e",
  [0x1afd] = "\x3a\x5f", [0x1afe] = "\x3a\x60", [0x1aff] = "\x3a\x61",
  [0x1b00] = "\x3a\x62", [0x1b01] = "\x3a\x63", [0x1b02] = "\x3a\x64",
  [0x1b03] = "\x3a\x65", [0x1b04] = "\x3a\x66", [0x1b05] = "\x3a\x67",
  [0x1b06] = "\x3a\x68", [0x1b07] = "\x3a\x69", [0x1b08] = "\x3a\x6a",
  [0x1b09] = "\x3a\x6b", [0x1b0a] = "\x3a\x6c", [0x1b0b] = "\x3a\x6d",
  [0x1b0c] = "\x3a\x6e", [0x1b0d] = "\x3a\x6f", [0x1b0e] = "\x3a\x70",
  [0x1b0f] = "\x3a\x71", [0x1b10] = "\x3a\x72", [0x1b11] = "\x3a\x73",
  [0x1b12] = "\x3a\x74", [0x1b13] = "\x3a\x75", [0x1b14] = "\x3a\x76",
  [0x1b15] = "\x3a\x77", [0x1b16] = "\x3a\x78", [0x1b17] = "\x3a\x79",
  [0x1b18] = "\x3a\x7a", [0x1b19] = "\x3a\x7b", [0x1b1a] = "\x3a\x7c",
  [0x1b1b] = "\x3a\x7d", [0x1b1c] = "\x3a\x7e", [0x1b1d] = "\x3a\x7f",
  [0x1b1e] = "\x3a\x80", [0x1b1f] = "\x3a\x81", [0x1b20] = "\x3a\x82",
  [0x1b21] = "\x3a\x83", [0x1b22] = "\x3a\x84", [0x1b23] = "\x3a\x85",
  [0x1b24] = "\x3a\x86", [0x1b25] = "\x3a\x87", [0x1b26] = "\x3a\x88",
  [0x1b27] = "\x3a\x89", [0x1b28] = "\x3a\x8a", [0x1b29] = "\x3a\x8b",
  [0x1b2a] = "\x3a\x8c", [0x1b2b] = "\x3a\x8d", [0x1b2c] = "\x3a\x8e",
  [0x1b2d] = "\x3a\x8f", [0x1b2e] = "\x3a\x90", [0x1b2f] = "\x3a\x91",
  [0x1b30] = "\x3a\x92", [0x1b31] = "\x3a\x93", [0x1b32] = "\x3a\x94",
  [0x1b33] = "\x3a\x95", [0x1b34] = "\x3a\x96", [0x1b35] = "\x3a\x97",
  [0x1b36] = "\x3a\x98", [0x1b37] = "\x3a\x99", [0x1b38] = "\x3a\x9a",
  [0x1b39] = "\x3a\x9b", [0x1b3a] = "\x3a\x9c", [0x1b3b] = "\x3a\x9d",
  [0x1b3c] = "\x3a\x9e", [0x1b3d] = "\x3a\x9f", [0x1b3e] = "\x3a\xa0",
  [0x1b3f] = "\x3a\xa1", [0x1b40] = "\x3a\xa2", [0x1b41] = "\x3a\xa3",
  [0x1b42] = "\x3a\xa4", [0x1b43] = "\x3a\xa5", [0x1b44] = "\x3a\xa6",
  [0x1b45] = "\x3a\xa7", [0x1b46] = "\x3a\xa8", [0x1b47] = "\x3a\xa9",
  [0x1b48] = "\x3a\xaa", [0x1b49] = "\x3a\xab", [0x1b4a] = "\x3a\xac",
  [0x1b4b] = "\x3a\xad", [0x1b4c] = "\x3a\xae", [0x1b4d] = "\x3a\xaf",
  [0x1b4e] = "\x3a\xb0", [0x1b4f] = "\x3a\xb1", [0x1b50] = "\x3a\xb2",
  [0x1b51] = "\x3a\xb3", [0x1b52] = "\x3a\xb4", [0x1b53] = "\x3a\xb5",
  [0x1b54] = "\x3a\xb6", [0x1b55] = "\x3a\xb7", [0x1b56] = "\x3a\xb8",
  [0x1b57] = "\x3a\xb9", [0x1b58] = "\x3a\xba", [0x1b59] = "\x3a\xbb",
  [0x1b5a] = "\x3a\xbc", [0x1b5b] = "\x3a\xbd", [0x1b5c] = "\x3a\xbe",
  [0x1b5d] = "\x3a\xbf", [0x1b5e] = "\x3a\xc0", [0x1b5f] = "\x3a\xc1",
  [0x1b60] = "\x3a\xc2", [0x1b61] = "\x3a\xc3", [0x1b62] = "\x3a\xc4",
  [0x1b63] = "\x3a\xc5", [0x1b64] = "\x3a\xc6", [0x1b65] = "\x3a\xc7",
  [0x1b66] = "\x3a\xc8", [0x1b67] = "\x3a\xc9", [0x1b68] = "\x3a\xca",
  [0x1b69] = "\x3a\xcb", [0x1b6a] = "\x3a\xcc", [0x1b6b] = "\x3a\xcd",
  [0x1b6c] = "\x3a\xce", [0x1b6d] = "\x3a\xcf", [0x1b6e] = "\x3a\xd0",
  [0x1b6f] = "\x3a\xd1", [0x1b70] = "\x3a\xd2", [0x1b71] = "\x3a\xd3",
  [0x1b72] = "\x3a\xd4", [0x1b73] = "\x3a\xd5", [0x1b74] = "\x3a\xd6",
  [0x1b75] = "\x3a\xd7", [0x1b76] = "\x3a\xd8", [0x1b77] = "\x3a\xd9",
  [0x1b78] = "\x3a\xda", [0x1b79] = "\x3a\xdb", [0x1b7a] = "\x3a\xdc",
  [0x1b7b] = "\x3a\xdd", [0x1b7c] = "\x3a\xde", [0x1b7d] = "\x3a\xdf",
  [0x1b7e] = "\x3a\xe0", [0x1b7f] = "\x3a\xe1", [0x1b80] = "\x3a\xe2",
  [0x1b81] = "\x3a\xe3", [0x1b82] = "\x3a\xe4", [0x1b83] = "\x3a\xe5",
  [0x1b84] = "\x3a\xe6", [0x1b85] = "\x3a\xe7", [0x1b86] = "\x3a\xe8",
  [0x1b87] = "\x3a\xe9", [0x1b88] = "\x3a\xea", [0x1b89] = "\x3a\xeb",
  [0x1b8a] = "\x3a\xec", [0x1b8b] = "\x3a\xed", [0x1b8c] = "\x3a\xee",
  [0x1b8d] = "\x3a\xef", [0x1b8e] = "\x3a\xf0", [0x1b8f] = "\x3a\xf1",
  [0x1b90] = "\x3a\xf2", [0x1b91] = "\x3a\xf3", [0x1b92] = "\x3a\xf4",
  [0x1b93] = "\x3a\xf5", [0x1b94] = "\x3a\xf6", [0x1b95] = "\x3a\xf7",
  [0x1b96] = "\x3a\xf8", [0x1b97] = "\x3a\xf9", [0x1b98] = "\x3a\xfa",
  [0x1b99] = "\x3a\xfb", [0x1b9a] = "\x3a\xfc", [0x1b9b] = "\x3a\xfd",
  [0x1b9c] = "\x3a\xfe", [0x1b9d] = "\x3a\xff", [0x1b9e] = "\x3b\x00",
  [0x1b9f] = "\x3b\x01", [0x1ba0] = "\x3b\x02", [0x1ba1] = "\x3b\x03",
  [0x1ba2] = "\x3b\x04", [0x1ba3] = "\x3b\x05", [0x1ba4] = "\x3b\x06",
  [0x1ba5] = "\x3b\x07", [0x1ba6] = "\x3b\x08", [0x1ba7] = "\x3b\x09",
  [0x1ba8] = "\x3b\x0a", [0x1ba9] = "\x3b\x0b", [0x1baa] = "\x3b\x0c",
  [0x1bab] = "\x3b\x0d", [0x1bac] = "\x3b\x0e", [0x1bad] = "\x3b\x0f",
  [0x1bae] = "\x3b\x10", [0x1baf] = "\x3b\x11", [0x1bb0] = "\x3b\x12",
  [0x1bb1] = "\x3b\x13", [0x1bb2] = "\x3b\x14", [0x1bb3] = "\x3b\x15",
  [0x1bb4] = "\x3b\x16", [0x1bb5] = "\x3b\x17", [0x1bb6] = "\x3b\x18",
  [0x1bb7] = "\x3b\x19", [0x1bb8] = "\x3b\x1a", [0x1bb9] = "\x3b\x1b",
  [0x1bba] = "\x3b\x1c", [0x1bbb] = "\x3b\x1d", [0x1bbc] = "\x3b\x1e",
  [0x1bbd] = "\x3b\x1f", [0x1bbe] = "\x3b\x20", [0x1bbf] = "\x3b\x21",
  [0x1bc0] = "\x3b\x22", [0x1bc1] = "\x3b\x23", [0x1bc2] = "\x3b\x24",
  [0x1bc3] = "\x3b\x25", [0x1bc4] = "\x3b\x26", [0x1bc5] = "\x3b\x27",
  [0x1bc6] = "\x3b\x28", [0x1bc7] = "\x3b\x29", [0x1bc8] = "\x3b\x2a",
  [0x1bc9] = "\x3b\x2b", [0x1bca] = "\x3b\x2c", [0x1bcb] = "\x3b\x2d",
  [0x1bcc] = "\x3b\x2e", [0x1bcd] = "\x3b\x2f", [0x1bce] = "\x3b\x30",
  [0x1bcf] = "\x3b\x31", [0x1bd0] = "\x3b\x32", [0x1bd1] = "\x3b\x33",
  [0x1bd2] = "\x3b\x34", [0x1bd3] = "\x3b\x35", [0x1bd4] = "\x3b\x36",
  [0x1bd5] = "\x3b\x37", [0x1bd6] = "\x3b\x38", [0x1bd7] = "\x3b\x39",
  [0x1bd8] = "\x3b\x3a", [0x1bd9] = "\x3b\x3b", [0x1bda] = "\x3b\x3c",
  [0x1bdb] = "\x3b\x3d", [0x1bdc] = "\x3b\x3e", [0x1bdd] = "\x3b\x3f",
  [0x1bde] = "\x3b\x40", [0x1bdf] = "\x3b\x41", [0x1be0] = "\x3b\x42",
  [0x1be1] = "\x3b\x43", [0x1be2] = "\x3b\x44", [0x1be3] = "\x3b\x45",
  [0x1be4] = "\x3b\x46", [0x1be5] = "\x3b\x47", [0x1be6] = "\x3b\x48",
  [0x1be7] = "\x3b\x49", [0x1be8] = "\x3b\x4a", [0x1be9] = "\x3b\x4b",
  [0x1bea] = "\x3b\x4c", [0x1beb] = "\x3b\x4d", [0x1bec] = "\x3b\x4e",
  [0x1bed] = "\x3b\x4f", [0x1bee] = "\x3b\x50", [0x1bef] = "\x3b\x51",
  [0x1bf0] = "\x3b\x52", [0x1bf1] = "\x3b\x53", [0x1bf2] = "\x3b\x54",
  [0x1bf3] = "\x3b\x55", [0x1bf4] = "\x3b\x56", [0x1bf5] = "\x3b\x57",
  [0x1bf6] = "\x3b\x58", [0x1bf7] = "\x3b\x59", [0x1bf8] = "\x3b\x5a",
  [0x1bf9] = "\x3b\x5b", [0x1bfa] = "\x3b\x5c", [0x1bfb] = "\x3b\x5d",
  [0x1bfc] = "\x3b\x5e", [0x1bfd] = "\x3b\x5f", [0x1bfe] = "\x3b\x60",
  [0x1bff] = "\x3b\x61", [0x1c00] = "\x3b\x62", [0x1c01] = "\x3b\x63",
  [0x1c02] = "\x3b\x64", [0x1c03] = "\x3b\x65", [0x1c04] = "\x3b\x66",
  [0x1c05] = "\x3b\x67", [0x1c06] = "\x3b\x68", [0x1c07] = "\x3b\x69",
  [0x1c08] = "\x3b\x6a", [0x1c09] = "\x3b\x6b", [0x1c0a] = "\x3b\x6c",
  [0x1c0b] = "\x3b\x6d", [0x1c0c] = "\x3b\x6e", [0x1c0d] = "\x3b\x6f",
  [0x1c0e] = "\x3b\x70", [0x1c0f] = "\x3b\x71", [0x1c10] = "\x3b\x72",
  [0x1c11] = "\x3b\x73", [0x1c12] = "\x3b\x74", [0x1c13] = "\x3b\x75",
  [0x1c14] = "\x3b\x76", [0x1c15] = "\x3b\x77", [0x1c16] = "\x3b\x78",
  [0x1c17] = "\x3b\x79", [0x1c18] = "\x3b\x7a", [0x1c19] = "\x3b\x7b",
  [0x1c1a] = "\x3b\x7c", [0x1c1b] = "\x3b\x7d", [0x1c1c] = "\x3b\x7e",
  [0x1c1d] = "\x3b\x7f", [0x1c1e] = "\x3b\x80", [0x1c1f] = "\x3b\x81",
  [0x1c20] = "\x3b\x82", [0x1c21] = "\x3b\x83", [0x1c22] = "\x3b\x84",
  [0x1c23] = "\x3b\x85", [0x1c24] = "\x3b\x86", [0x1c25] = "\x3b\x87",
  [0x1c26] = "\x3b\x88", [0x1c27] = "\x3b\x89", [0x1c28] = "\x3b\x8a",
  [0x1c29] = "\x3b\x8b", [0x1c2a] = "\x3b\x8c", [0x1c2b] = "\x3b\x8d",
  [0x1c2c] = "\x3b\x8e", [0x1c2d] = "\x3b\x8f", [0x1c2e] = "\x3b\x90",
  [0x1c2f] = "\x3b\x91", [0x1c30] = "\x3b\x92", [0x1c31] = "\x3b\x93",
  [0x1c32] = "\x3b\x94", [0x1c33] = "\x3b\x95", [0x1c34] = "\x3b\x96",
  [0x1c35] = "\x3b\x97", [0x1c36] = "\x3b\x98", [0x1c37] = "\x3b\x99",
  [0x1c38] = "\x3b\x9a", [0x1c39] = "\x3b\x9b", [0x1c3a] = "\x3b\x9c",
  [0x1c3b] = "\x3b\x9d", [0x1c3c] = "\x3b\x9e", [0x1c3d] = "\x3b\x9f",
  [0x1c3e] = "\x3b\xa0", [0x1c3f] = "\x3b\xa1", [0x1c40] = "\x3b\xa2",
  [0x1c41] = "\x3b\xa3", [0x1c42] = "\x3b\xa4", [0x1c43] = "\x3b\xa5",
  [0x1c44] = "\x3b\xa6", [0x1c45] = "\x3b\xa7", [0x1c46] = "\x3b\xa8",
  [0x1c47] = "\x3b\xa9", [0x1c48] = "\x3b\xaa", [0x1c49] = "\x3b\xab",
  [0x1c4a] = "\x3b\xac", [0x1c4b] = "\x3b\xad", [0x1c4c] = "\x3b\xae",
  [0x1c4d] = "\x3b\xaf", [0x1c4e] = "\x3b\xb0", [0x1c4f] = "\x3b\xb1",
  [0x1c50] = "\x3b\xb2", [0x1c51] = "\x3b\xb3", [0x1c52] = "\x3b\xb4",
  [0x1c53] = "\x3b\xb5", [0x1c54] = "\x3b\xb6", [0x1c55] = "\x3b\xb7",
  [0x1c56] = "\x3b\xb8", [0x1c57] = "\x3b\xb9", [0x1c58] = "\x3b\xba",
  [0x1c59] = "\x3b\xbb", [0x1c5a] = "\x3b\xbc", [0x1c5b] = "\x3b\xbd",
  [0x1c5c] = "\x3b\xbe", [0x1c5d] = "\x3b\xbf", [0x1c5e] = "\x3b\xc0",
  [0x1c5f] = "\x3b\xc1", [0x1c60] = "\x3b\xc2", [0x1c61] = "\x3b\xc3",
  [0x1c62] = "\x3b\xc4", [0x1c63] = "\x3b\xc5", [0x1c64] = "\x3b\xc6",
  [0x1c65] = "\x3b\xc7", [0x1c66] = "\x3b\xc8", [0x1c67] = "\x3b\xc9",
  [0x1c68] = "\x3b\xca", [0x1c69] = "\x3b\xcb", [0x1c6a] = "\x3b\xcc",
  [0x1c6b] = "\x3b\xcd", [0x1c6c] = "\x3b\xce", [0x1c6d] = "\x3b\xcf",
  [0x1c6e] = "\x3b\xd0", [0x1c6f] = "\x3b\xd1", [0x1c70] = "\x3b\xd2",
  [0x1c71] = "\x3b\xd3", [0x1c72] = "\x3b\xd4", [0x1c73] = "\x3b\xd5",
  [0x1c74] = "\x3b\xd6", [0x1c75] = "\x3b\xd7", [0x1c76] = "\x3b\xd8",
  [0x1c77] = "\x3b\xd9", [0x1c78] = "\x3b\xda", [0x1c79] = "\x3b\xdb",
  [0x1c7a] = "\x3b\xdc", [0x1c7b] = "\x3b\xdd", [0x1c7c] = "\x3b\xde",
  [0x1c7d] = "\x3b\xdf", [0x1c7e] = "\x3b\xe0", [0x1c7f] = "\x3b\xe1",
  [0x1c80] = "\x3b\xe2", [0x1c81] = "\x3b\xe3", [0x1c82] = "\x3b\xe4",
  [0x1c83] = "\x3b\xe5", [0x1c84] = "\x3b\xe6", [0x1c85] = "\x3b\xe7",
  [0x1c86] = "\x3b\xe8", [0x1c87] = "\x3b\xe9", [0x1c88] = "\x3b\xea",
  [0x1c89] = "\x3b\xeb", [0x1c8a] = "\x3b\xec", [0x1c8b] = "\x3b\xed",
  [0x1c8c] = "\x3b\xee", [0x1c8d] = "\x3b\xef", [0x1c8e] = "\x3b\xf0",
  [0x1c8f] = "\x3b\xf1", [0x1c90] = "\x3b\xf2", [0x1c91] = "\x3b\xf3",
  [0x1c92] = "\x3b\xf4", [0x1c93] = "\x3b\xf5", [0x1c94] = "\x3b\xf6",
  [0x1c95] = "\x3b\xf7", [0x1c96] = "\x3b\xf8", [0x1c97] = "\x3b\xf9",
  [0x1c98] = "\x3b\xfa", [0x1c99] = "\x3b\xfb", [0x1c9a] = "\x3b\xfc",
  [0x1c9b] = "\x3b\xfd", [0x1c9c] = "\x3b\xfe", [0x1c9d] = "\x3b\xff",
  [0x1c9e] = "\x3c\x00", [0x1c9f] = "\x3c\x01", [0x1ca0] = "\x3c\x02",
  [0x1ca1] = "\x3c\x03", [0x1ca2] = "\x3c\x04", [0x1ca3] = "\x3c\x05",
  [0x1ca4] = "\x3c\x06", [0x1ca5] = "\x3c\x07", [0x1ca6] = "\x3c\x08",
  [0x1ca7] = "\x3c\x09", [0x1ca8] = "\x3c\x0a", [0x1ca9] = "\x3c\x0b",
  [0x1caa] = "\x3c\x0c", [0x1cab] = "\x3c\x0d", [0x1cac] = "\x3c\x0e",
  [0x1cad] = "\x3c\x0f", [0x1cae] = "\x3c\x10", [0x1caf] = "\x3c\x11",
  [0x1cb0] = "\x3c\x12", [0x1cb1] = "\x3c\x13", [0x1cb2] = "\x3c\x14",
  [0x1cb3] = "\x3c\x15", [0x1cb4] = "\x3c\x16", [0x1cb5] = "\x3c\x17",
  [0x1cb6] = "\x3c\x18", [0x1cb7] = "\x3c\x19", [0x1cb8] = "\x3c\x1a",
  [0x1cb9] = "\x3c\x1b", [0x1cba] = "\x3c\x1c", [0x1cbb] = "\x3c\x1d",
  [0x1cbc] = "\x3c\x1e", [0x1cbd] = "\x3c\x1f", [0x1cbe] = "\x3c\x20",
  [0x1cbf] = "\x3c\x21", [0x1cc0] = "\x3c\x22", [0x1cc1] = "\x3c\x23",
  [0x1cc2] = "\x3c\x24", [0x1cc3] = "\x3c\x25", [0x1cc4] = "\x3c\x26",
  [0x1cc5] = "\x3c\x27", [0x1cc6] = "\x3c\x28", [0x1cc7] = "\x3c\x29",
  [0x1cc8] = "\x3c\x2a", [0x1cc9] = "\x3c\x2b", [0x1cca] = "\x3c\x2c",
  [0x1ccb] = "\x3c\x2d", [0x1ccc] = "\x3c\x2e", [0x1ccd] = "\x3c\x2f",
  [0x1cce] = "\x3c\x30", [0x1ccf] = "\x3c\x31", [0x1cd0] = "\x3c\x32",
  [0x1cd1] = "\x3c\x33", [0x1cd2] = "\x3c\x34", [0x1cd3] = "\x3c\x35",
  [0x1cd4] = "\x3c\x36", [0x1cd5] = "\x3c\x37", [0x1cd6] = "\x3c\x38",
  [0x1cd7] = "\x3c\x39", [0x1cd8] = "\x3c\x3a", [0x1cd9] = "\x3c\x3b",
  [0x1cda] = "\x3c\x3c", [0x1cdb] = "\x3c\x3d", [0x1cdc] = "\x3c\x3e",
  [0x1cdd] = "\x3c\x3f", [0x1cde] = "\x3c\x40", [0x1cdf] = "\x3c\x41",
  [0x1ce0] = "\x3c\x42", [0x1ce1] = "\x3c\x43", [0x1ce2] = "\x3c\x44",
  [0x1ce3] = "\x3c\x45", [0x1ce4] = "\x3c\x46", [0x1ce5] = "\x3c\x47",
  [0x1ce6] = "\x3c\x48", [0x1ce7] = "\x3c\x49", [0x1ce8] = "\x3c\x4a",
  [0x1ce9] = "\x3c\x4b", [0x1cea] = "\x3c\x4c", [0x1ceb] = "\x3c\x4d",
  [0x1cec] = "\x3c\x4e", [0x1ced] = "\x3c\x4f", [0x1cee] = "\x3c\x50",
  [0x1cef] = "\x3c\x51", [0x1cf0] = "\x3c\x52", [0x1cf1] = "\x3c\x53",
  [0x1cf2] = "\x3c\x54", [0x1cf3] = "\x3c\x55", [0x1cf4] = "\x3c\x56",
  [0x1cf5] = "\x3c\x57", [0x1cf6] = "\x3c\x58", [0x1cf7] = "\x3c\x59",
  [0x1cf8] = "\x3c\x5a", [0x1cf9] = "\x3c\x5b", [0x1cfa] = "\x3c\x5c",
  [0x1cfb] = "\x3c\x5d", [0x1cfc] = "\x3c\x5e", [0x1cfd] = "\x3c\x5f",
  [0x1cfe] = "\x3c\x60", [0x1cff] = "\x3c\x61", [0x1d00] = "\x3c\x62",
  [0x1d01] = "\x3c\x63", [0x1d02] = "\x3c\x64", [0x1d03] = "\x3c\x65",
  [0x1d04] = "\x3c\x66", [0x1d05] = "\x3c\x67", [0x1d06] = "\x3c\x68",
  [0x1d07] = "\x3c\x69", [0x1d08] = "\x3c\x6a", [0x1d09] = "\x3c\x6b",
  [0x1d0a] = "\x3c\x6c", [0x1d0b] = "\x3c\x6d", [0x1d0c] = "\x3c\x6e",
  [0x1d0d] = "\x3c\x6f", [0x1d0e] = "\x3c\x70", [0x1d0f] = "\x3c\x71",
  [0x1d10] = "\x3c\x72", [0x1d11] = "\x3c\x73", [0x1d12] = "\x3c\x74",
  [0x1d13] = "\x3c\x75", [0x1d14] = "\x3c\x76", [0x1d15] = "\x3c\x77",
  [0x1d16] = "\x3c\x78", [0x1d17] = "\x3c\x79", [0x1d18] = "\x3c\x7a",
  [0x1d19] = "\x3c\x7b", [0x1d1a] = "\x3c\x7c", [0x1d1b] = "\x3c\x7d",
  [0x1d1c] = "\x3c\x7e", [0x1d1d] = "\x3c\x7f", [0x1d1e] = "\x3c\x80",
  [0x1d1f] = "\x3c\x81", [0x1d20] = "\x3c\x82", [0x1d21] = "\x3c\x83",
  [0x1d22] = "\x3c\x84", [0x1d23] = "\x3c\x85", [0x1d24] = "\x3c\x86",
  [0x1d25] = "\x3c\x87", [0x1d26] = "\x3c\x88", [0x1d27] = "\x3c\x89",
  [0x1d28] = "\x3c\x8a", [0x1d29] = "\x3c\x8b", [0x1d2a] = "\x3c\x8c",
  [0x1d2b] = "\x3c\x8d", [0x1d2c] = "\x3c\x8e", [0x1d2d] = "\x3c\x8f",
  [0x1d2e] = "\x3c\x90", [0x1d2f] = "\x3c\x91", [0x1d30] = "\x3c\x92",
  [0x1d31] = "\x3c\x93", [0x1d32] = "\x3c\x94", [0x1d33] = "\x3c\x95",
  [0x1d34] = "\x3c\x96", [0x1d35] = "\x3c\x97", [0x1d36] = "\x3c\x98",
  [0x1d37] = "\x3c\x99", [0x1d38] = "\x3c\x9a", [0x1d39] = "\x3c\x9b",
  [0x1d3a] = "\x3c\x9c", [0x1d3b] = "\x3c\x9d", [0x1d3c] = "\x3c\x9e",
  [0x1d3d] = "\x3c\x9f", [0x1d3e] = "\x3c\xa0", [0x1d3f] = "\x3c\xa1",
  [0x1d40] = "\x3c\xa2", [0x1d41] = "\x3c\xa3", [0x1d42] = "\x3c\xa4",
  [0x1d43] = "\x3c\xa5", [0x1d44] = "\x3c\xa6", [0x1d45] = "\x3c\xa7",
  [0x1d46] = "\x3c\xa8", [0x1d47] = "\x3c\xa9", [0x1d48] = "\x3c\xaa",
  [0x1d49] = "\x3c\xab", [0x1d4a] = "\x3c\xac", [0x1d4b] = "\x3c\xad",
  [0x1d4c] = "\x3c\xae", [0x1d4d] = "\x3c\xaf", [0x1d4e] = "\x3c\xb0",
  [0x1d4f] = "\x3c\xb1", [0x1d50] = "\x3c\xb2", [0x1d51] = "\x3c\xb3",
  [0x1d52] = "\x3c\xb4", [0x1d53] = "\x3c\xb5", [0x1d54] = "\x3c\xb6",
  [0x1d55] = "\x3c\xb7", [0x1d56] = "\x3c\xb8", [0x1d57] = "\x3c\xb9",
  [0x1d58] = "\x3c\xba", [0x1d59] = "\x3c\xbb", [0x1d5a] = "\x3c\xbc",
  [0x1d5b] = "\x3c\xbd", [0x1d5c] = "\x3c\xbe", [0x1d5d] = "\x3c\xbf",
  [0x1d5e] = "\x3c\xc0", [0x1d5f] = "\x3c\xc1", [0x1d60] = "\x3c\xc2",
  [0x1d61] = "\x3c\xc3", [0x1d62] = "\x3c\xc4", [0x1d63] = "\x3c\xc5",
  [0x1d64] = "\x3c\xc6", [0x1d65] = "\x3c\xc7", [0x1d66] = "\x3c\xc8",
  [0x1d67] = "\x3c\xc9", [0x1d68] = "\x3c\xca", [0x1d69] = "\x3c\xcb",
  [0x1d6a] = "\x3c\xcc", [0x1d6b] = "\x3c\xcd", [0x1d6c] = "\x3c\xce",
  [0x1d6d] = "\x3c\xcf", [0x1d6e] = "\x3c\xd0", [0x1d6f] = "\x3c\xd1",
  [0x1d70] = "\x3c\xd2", [0x1d71] = "\x3c\xd3", [0x1d72] = "\x3c\xd4",
  [0x1d73] = "\x3c\xd5", [0x1d74] = "\x3c\xd6", [0x1d75] = "\x3c\xd7",
  [0x1d76] = "\x3c\xd8", [0x1d77] = "\x3c\xd9", [0x1d78] = "\x3c\xda",
  [0x1d79] = "\x3c\xdb", [0x1d7a] = "\x3c\xdc", [0x1d7b] = "\x3c\xdd",
  [0x1d7c] = "\x3c\xde", [0x1d7d] = "\x3c\xdf", [0x1d7e] = "\x3c\xe0",
  [0x1d7f] = "\x3c\xe1", [0x1d80] = "\x3c\xe2", [0x1d81] = "\x3c\xe3",
  [0x1d82] = "\x3c\xe4", [0x1d83] = "\x3c\xe5", [0x1d84] = "\x3c\xe6",
  [0x1d85] = "\x3c\xe7", [0x1d86] = "\x3c\xe8", [0x1d87] = "\x3c\xe9",
  [0x1d88] = "\x3c\xea", [0x1d89] = "\x3c\xeb", [0x1d8a] = "\x3c\xec",
  [0x1d8b] = "\x3c\xed", [0x1d8c] = "\x3c\xee", [0x1d8d] = "\x3c\xef",
  [0x1d8e] = "\x3c\xf0", [0x1d8f] = "\x3c\xf1", [0x1d90] = "\x3c\xf2",
  [0x1d91] = "\x3c\xf3", [0x1d92] = "\x3c\xf4", [0x1d93] = "\x3c\xf5",
  [0x1d94] = "\x3c\xf6", [0x1d95] = "\x3c\xf7", [0x1d96] = "\x3c\xf8",
  [0x1d97] = "\x3c\xf9", [0x1d98] = "\x3c\xfa", [0x1d99] = "\x3c\xfb",
  [0x1d9a] = "\x3c\xfc", [0x1d9b] = "\x3c\xfd", [0x1d9c] = "\x3c\xfe",
  [0x1d9d] = "\x3c\xff", [0x1d9e] = "\x3d\x00", [0x1d9f] = "\x3d\x01",
  [0x1da0] = "\x3d\x02", [0x1da1] = "\x3d\x03", [0x1da2] = "\x3d\x04",
  [0x1da3] = "\x3d\x05", [0x1da4] = "\x3d\x06", [0x1da5] = "\x3d\x07",
  [0x1da6] = "\x3d\x08", [0x1da7] = "\x3d\x09", [0x1da8] = "\x3d\x0a",
  [0x1da9] = "\x3d\x0b", [0x1daa] = "\x3d\x0c", [0x1dab] = "\x3d\x0d",
  [0x1dac] = "\x3d\x0e", [0x1dad] = "\x3d\x0f", [0x1dae] = "\x3d\x10",
  [0x1daf] = "\x3d\x11", [0x1db0] = "\x3d\x12", [0x1db1] = "\x3d\x13",
  [0x1db2] = "\x3d\x14", [0x1db3] = "\x3d\x15", [0x1db4] = "\x3d\x16",
  [0x1db5] = "\x3d\x17", [0x1db6] = "\x3d\x18", [0x1db7] = "\x3d\x19",
  [0x1db8] = "\x3d\x1a", [0x1db9] = "\x3d\x1b", [0x1dba] = "\x3d\x1c",
  [0x1dbb] = "\x3d\x1d", [0x1dbc] = "\x3d\x1e", [0x1dbd] = "\x3d\x1f",
  [0x1dbe] = "\x3d\x20", [0x1dbf] = "\xa8\xbc", [0x1dc0] = "\x3d\x22",
  [0x1dc1] = "\x3d\x23", [0x1dc2] = "\x3d\x24", [0x1dc3] = "\x3d\x25",
  [0x1dc4] = "\x3d\x26", [0x1dc5] = "\x3d\x27", [0x1dc6] = "\x3d\x28",
  [0x1dc7] = "\x3d\x29", [0x1dc8] = "\x3d\x2a", [0x1dc9] = "\x3d\x2b",
  [0x1dca] = "\x3d\x2c", [0x1dcb] = "\x3d\x2d", [0x1dcc] = "\x3d\x2e",
  [0x1dcd] = "\x3d\x2f", [0x1dce] = "\x3d\x30", [0x1dcf] = "\x3d\x31",
  [0x1dd0] = "\x3d\x32", [0x1dd1] = "\x3d\x33", [0x1dd2] = "\x3d\x34",
  [0x1dd3] = "\x3d\x35", [0x1dd4] = "\x3d\x36", [0x1dd5] = "\x3d\x37",
  [0x1dd6] = "\x3d\x38", [0x1dd7] = "\x3d\x39", [0x1dd8] = "\x3d\x3a",
  [0x1dd9] = "\x3d\x3b", [0x1dda] = "\x3d\x3c", [0x1ddb] = "\x3d\x3d",
  [0x1ddc] = "\x3d\x3e", [0x1ddd] = "\x3d\x3f", [0x1dde] = "\x3d\x40",
  [0x1ddf] = "\x3d\x41", [0x1de0] = "\x3d\x42", [0x1de1] = "\x3d\x43",
  [0x1de2] = "\x3d\x44", [0x1de3] = "\x3d\x45", [0x1de4] = "\x3d\x46",
  [0x1de5] = "\x3d\x47", [0x1de6] = "\x3d\x48", [0x1de7] = "\x3d\x49",
  [0x1de8] = "\x3d\x4a", [0x1de9] = "\x3d\x4b", [0x1dea] = "\x3d\x4c",
  [0x1deb] = "\x3d\x4d", [0x1dec] = "\x3d\x4e", [0x1ded] = "\x3d\x4f",
  [0x1dee] = "\x3d\x50", [0x1def] = "\x3d\x51", [0x1df0] = "\x3d\x52",
  [0x1df1] = "\x3d\x53", [0x1df2] = "\x3d\x54", [0x1df3] = "\x3d\x55",
  [0x1df4] = "\x3d\x56", [0x1df5] = "\x3d\x57", [0x1df6] = "\x3d\x58",
  [0x1df7] = "\x3d\x59", [0x1df8] = "\x3d\x5a", [0x1df9] = "\x3d\x5b",
  [0x1dfa] = "\x3d\x5c", [0x1dfb] = "\x3d\x5d", [0x1dfc] = "\x3d\x5e",
  [0x1dfd] = "\x3d\x5f", [0x1dfe] = "\x3d\x60", [0x1dff] = "\x3d\x61",
  [0x1e00] = "\x3d\x62", [0x1e01] = "\x3d\x63", [0x1e02] = "\x3d\x64",
  [0x1e03] = "\x3d\x65", [0x1e04] = "\x3d\x66", [0x1e05] = "\x3d\x67",
  [0x1e06] = "\x3d\x68", [0x1e07] = "\x3d\x69", [0x1e08] = "\x3d\x6a",
  [0x1e09] = "\x3d\x6b", [0x1e0a] = "\x3d\x6c", [0x1e0b] = "\x3d\x6d",
  [0x1e0c] = "\x3d\x6e", [0x1e0d] = "\x3d\x6f", [0x1e0e] = "\x3d\x70",
  [0x1e0f] = "\x3d\x71", [0x1e10] = "\x3d\x72", [0x1e11] = "\x3d\x73",
  [0x1e12] = "\x3d\x74", [0x1e13] = "\x3d\x75", [0x1e14] = "\x3d\x76",
  [0x1e15] = "\x3d\x77", [0x1e16] = "\x3d\x78", [0x1e17] = "\x3d\x79",
  [0x1e18] = "\x3d\x7a", [0x1e19] = "\x3d\x7b", [0x1e1a] = "\x3d\x7c",
  [0x1e1b] = "\x3d\x7d", [0x1e1c] = "\x3d\x7e", [0x1e1d] = "\x3d\x7f",
  [0x1e1e] = "\x3d\x80", [0x1e1f] = "\x3d\x81", [0x1e20] = "\x3d\x82",
  [0x1e21] = "\x3d\x83", [0x1e22] = "\x3d\x84", [0x1e23] = "\x3d\x85",
  [0x1e24] = "\x3d\x86", [0x1e25] = "\x3d\x87", [0x1e26] = "\x3d\x88",
  [0x1e27] = "\x3d\x89", [0x1e28] = "\x3d\x8a", [0x1e29] = "\x3d\x8b",
  [0x1e2a] = "\x3d\x8c", [0x1e2b] = "\x3d\x8d", [0x1e2c] = "\x3d\x8e",
  [0x1e2d] = "\x3d\x8f", [0x1e2e] = "\x3d\x90", [0x1e2f] = "\x3d\x91",
  [0x1e30] = "\x3d\x92", [0x1e31] = "\x3d\x93", [0x1e32] = "\x3d\x94",
  [0x1e33] = "\x3d\x95", [0x1e34] = "\x3d\x96", [0x1e35] = "\x3d\x97",
  [0x1e36] = "\x3d\x98", [0x1e37] = "\x3d\x99", [0x1e38] = "\x3d\x9a",
  [0x1e39] = "\x3d\x9b", [0x1e3a] = "\x3d\x9c", [0x1e3b] = "\x3d\x9d",
  [0x1e3c] = "\x3d\x9e", [0x1e3d] = "\x3d\x9f", [0x1e3e] = "\x3d\xa0",
  [0x1e3f] = "\x3d\xa1", [0x1e40] = "\x3d\xa2", [0x1e41] = "\x3d\xa3",
  [0x1e42] = "\x3d\xa4", [0x1e43] = "\x3d\xa5", [0x1e44] = "\x3d\xa6",
  [0x1e45] = "\x3d\xa7", [0x1e46] = "\x3d\xa8", [0x1e47] = "\x3d\xa9",
  [0x1e48] = "\x3d\xaa", [0x1e49] = "\x3d\xab", [0x1e4a] = "\x3d\xac",
  [0x1e4b] = "\x3d\xad", [0x1e4c] = "\x3d\xae", [0x1e4d] = "\x3d\xaf",
  [0x1e4e] = "\x3d\xb0", [0x1e4f] = "\x3d\xb1", [0x1e50] = "\x3d\xb2",
  [0x1e51] = "\x3d\xb3", [0x1e52] = "\x3d\xb4", [0x1e53] = "\x3d\xb5",
  [0x1e54] = "\x3d\xb6", [0x1e55] = "\x3d\xb7", [0x1e56] = "\x3d\xb8",
  [0x1e57] = "\x3d\xb9", [0x1e58] = "\x3d\xba", [0x1e59] = "\x3d\xbb",
  [0x1e5a] = "\x3d\xbc", [0x1e5b] = "\x3d\xbd", [0x1e5c] = "\x3d\xbe",
  [0x1e5d] = "\x3d\xbf", [0x1e5e] = "\x3d\xc0", [0x1e5f] = "\x3d\xc1",
  [0x1e60] = "\x3d\xc2", [0x1e61] = "\x3d\xc3", [0x1e62] = "\x3d\xc4",
  [0x1e63] = "\x3d\xc5", [0x1e64] = "\x3d\xc6", [0x1e65] = "\x3d\xc7",
  [0x1e66] = "\x3d\xc8", [0x1e67] = "\x3d\xc9", [0x1e68] = "\x3d\xca",
  [0x1e69] = "\x3d\xcb", [0x1e6a] = "\x3d\xcc", [0x1e6b] = "\x3d\xcd",
  [0x1e6c] = "\x3d\xce", [0x1e6d] = "\x3d\xcf", [0x1e6e] = "\x3d\xd0",
  [0x1e6f] = "\x3d\xd1", [0x1e70] = "\x3d\xd2", [0x1e71] = "\x3d\xd3",
  [0x1e72] = "\x3d\xd4", [0x1e73] = "\x3d\xd5", [0x1e74] = "\x3d\xd6",
  [0x1e75] = "\x3d\xd7", [0x1e76] = "\x3d\xd8", [0x1e77] = "\x3d\xd9",
  [0x1e78] = "\x3d\xda", [0x1e79] = "\x3d\xdb", [0x1e7a] = "\x3d\xdc",
  [0x1e7b] = "\x3d\xdd", [0x1e7c] = "\x3d\xde", [0x1e7d] = "\x3d\xdf",
  [0x1e7e] = "\x3d\xe0", [0x1e7f] = "\x3d\xe1", [0x1e80] = "\x3d\xe2",
  [0x1e81] = "\x3d\xe3", [0x1e82] = "\x3d\xe4", [0x1e83] = "\x3d\xe5",
  [0x1e84] = "\x3d\xe6", [0x1e85] = "\x3d\xe7", [0x1e86] = "\x3d\xe8",
  [0x1e87] = "\x3d\xe9", [0x1e88] = "\x3d\xea", [0x1e89] = "\x3d\xeb",
  [0x1e8a] = "\x3d\xec", [0x1e8b] = "\x3d\xed", [0x1e8c] = "\x3d\xee",
  [0x1e8d] = "\x3d\xef", [0x1e8e] = "\x3d\xf0", [0x1e8f] = "\x3d\xf1",
  [0x1e90] = "\x3d\xf2", [0x1e91] = "\x3d\xf3", [0x1e92] = "\x3d\xf4",
  [0x1e93] = "\x3d\xf5", [0x1e94] = "\x3d\xf6", [0x1e95] = "\x3d\xf7",
  [0x1e96] = "\x3d\xf8", [0x1e97] = "\x3d\xf9", [0x1e98] = "\x3d\xfa",
  [0x1e99] = "\x3d\xfb", [0x1e9a] = "\x3d\xfc", [0x1e9b] = "\x3d\xfd",
  [0x1e9c] = "\x3d\xfe", [0x1e9d] = "\x3d\xff", [0x1e9e] = "\x3e\x00",
  [0x1e9f] = "\x3e\x01", [0x1ea0] = "\x3e\x02", [0x1ea1] = "\x3e\x03",
  [0x1ea2] = "\x3e\x04", [0x1ea3] = "\x3e\x05", [0x1ea4] = "\x3e\x06",
  [0x1ea5] = "\x3e\x07", [0x1ea6] = "\x3e\x08", [0x1ea7] = "\x3e\x09",
  [0x1ea8] = "\x3e\x0a", [0x1ea9] = "\x3e\x0b", [0x1eaa] = "\x3e\x0c",
  [0x1eab] = "\x3e\x0d", [0x1eac] = "\x3e\x0e", [0x1ead] = "\x3e\x0f",
  [0x1eae] = "\x3e\x10", [0x1eaf] = "\x3e\x11", [0x1eb0] = "\x3e\x12",
  [0x1eb1] = "\x3e\x13", [0x1eb2] = "\x3e\x14", [0x1eb3] = "\x3e\x15",
  [0x1eb4] = "\x3e\x16", [0x1eb5] = "\x3e\x17", [0x1eb6] = "\x3e\x18",
  [0x1eb7] = "\x3e\x19", [0x1eb8] = "\x3e\x1a", [0x1eb9] = "\x3e\x1b",
  [0x1eba] = "\x3e\x1c", [0x1ebb] = "\x3e\x1d", [0x1ebc] = "\x3e\x1e",
  [0x1ebd] = "\x3e\x1f", [0x1ebe] = "\x3e\x20", [0x1ebf] = "\x3e\x21",
  [0x1ec0] = "\x3e\x22", [0x1ec1] = "\x3e\x23", [0x1ec2] = "\x3e\x24",
  [0x1ec3] = "\x3e\x25", [0x1ec4] = "\x3e\x26", [0x1ec5] = "\x3e\x27",
  [0x1ec6] = "\x3e\x28", [0x1ec7] = "\x3e\x29", [0x1ec8] = "\x3e\x2a",
  [0x1ec9] = "\x3e\x2b", [0x1eca] = "\x3e\x2c", [0x1ecb] = "\x3e\x2d",
  [0x1ecc] = "\x3e\x2e", [0x1ecd] = "\x3e\x2f", [0x1ece] = "\x3e\x30",
  [0x1ecf] = "\x3e\x31", [0x1ed0] = "\x3e\x32", [0x1ed1] = "\x3e\x33",
  [0x1ed2] = "\x3e\x34", [0x1ed3] = "\x3e\x35", [0x1ed4] = "\x3e\x36",
  [0x1ed5] = "\x3e\x37", [0x1ed6] = "\x3e\x38", [0x1ed7] = "\x3e\x39",
  [0x1ed8] = "\x3e\x3a", [0x1ed9] = "\x3e\x3b", [0x1eda] = "\x3e\x3c",
  [0x1edb] = "\x3e\x3d", [0x1edc] = "\x3e\x3e", [0x1edd] = "\x3e\x3f",
  [0x1ede] = "\x3e\x40", [0x1edf] = "\x3e\x41", [0x1ee0] = "\x3e\x42",
  [0x1ee1] = "\x3e\x43", [0x1ee2] = "\x3e\x44", [0x1ee3] = "\x3e\x45",
  [0x1ee4] = "\x3e\x46", [0x1ee5] = "\x3e\x47", [0x1ee6] = "\x3e\x48",
  [0x1ee7] = "\x3e\x49", [0x1ee8] = "\x3e\x4a", [0x1ee9] = "\x3e\x4b",
  [0x1eea] = "\x3e\x4c", [0x1eeb] = "\x3e\x4d", [0x1eec] = "\x3e\x4e",
  [0x1eed] = "\x3e\x4f", [0x1eee] = "\x3e\x50", [0x1eef] = "\x3e\x51",
  [0x1ef0] = "\x3e\x52", [0x1ef1] = "\x3e\x53", [0x1ef2] = "\x3e\x54",
  [0x1ef3] = "\x3e\x55", [0x1ef4] = "\x3e\x56", [0x1ef5] = "\x3e\x57",
  [0x1ef6] = "\x3e\x58", [0x1ef7] = "\x3e\x59", [0x1ef8] = "\x3e\x5a",
  [0x1ef9] = "\x3e\x5b", [0x1efa] = "\x3e\x5c", [0x1efb] = "\x3e\x5d",
  [0x1efc] = "\x3e\x5e", [0x1efd] = "\x3e\x5f", [0x1efe] = "\x3e\x60",
  [0x1eff] = "\x3e\x61", [0x1f00] = "\x3e\x62", [0x1f01] = "\x3e\x63",
  [0x1f02] = "\x3e\x64", [0x1f03] = "\x3e\x65", [0x1f04] = "\x3e\x66",
  [0x1f05] = "\x3e\x67", [0x1f06] = "\x3e\x68", [0x1f07] = "\x3e\x69",
  [0x1f08] = "\x3e\x6a", [0x1f09] = "\x3e\x6b", [0x1f0a] = "\x3e\x6c",
  [0x1f0b] = "\x3e\x6d", [0x1f0c] = "\x3e\x6e", [0x1f0d] = "\x3e\x6f",
  [0x1f0e] = "\x3e\x70", [0x1f0f] = "\x3e\x71", [0x1f10] = "\x3e\x72",
  [0x1f11] = "\x3e\x73", [0x1f12] = "\x3e\x74", [0x1f13] = "\x3e\x75",
  [0x1f14] = "\x3e\x76", [0x1f15] = "\x3e\x77", [0x1f16] = "\x3e\x78",
  [0x1f17] = "\x3e\x79", [0x1f18] = "\x3e\x7a", [0x1f19] = "\x3e\x7b",
  [0x1f1a] = "\x3e\x7c", [0x1f1b] = "\x3e\x7d", [0x1f1c] = "\x3e\x7e",
  [0x1f1d] = "\x3e\x7f", [0x1f1e] = "\x3e\x80", [0x1f1f] = "\x3e\x81",
  [0x1f20] = "\x3e\x82", [0x1f21] = "\x3e\x83", [0x1f22] = "\x3e\x84",
  [0x1f23] = "\x3e\x85", [0x1f24] = "\x3e\x86", [0x1f25] = "\x3e\x87",
  [0x1f26] = "\x3e\x88", [0x1f27] = "\x3e\x89", [0x1f28] = "\x3e\x8a",
  [0x1f29] = "\x3e\x8b", [0x1f2a] = "\x3e\x8c", [0x1f2b] = "\x3e\x8d",
  [0x1f2c] = "\x3e\x8e", [0x1f2d] = "\x3e\x8f", [0x1f2e] = "\x3e\x90",
  [0x1f2f] = "\x3e\x91", [0x1f30] = "\x3e\x92", [0x1f31] = "\x3e\x93",
  [0x1f32] = "\x3e\x94", [0x1f33] = "\x3e\x95", [0x1f34] = "\x3e\x96",
  [0x1f35] = "\x3e\x97", [0x1f36] = "\x3e\x98", [0x1f37] = "\x3e\x99",
  [0x1f38] = "\x3e\x9a", [0x1f39] = "\x3e\x9b", [0x1f3a] = "\x3e\x9c",
  [0x1f3b] = "\x3e\x9d", [0x1f3c] = "\x3e\x9e", [0x1f3d] = "\x3e\x9f",
  [0x1f3e] = "\x3e\xa0", [0x1f3f] = "\x3e\xa1", [0x1f40] = "\x3e\xa2",
  [0x1f41] = "\x3e\xa3", [0x1f42] = "\x3e\xa4", [0x1f43] = "\x3e\xa5",
  [0x1f44] = "\x3e\xa6", [0x1f45] = "\x3e\xa7", [0x1f46] = "\x3e\xa8",
  [0x1f47] = "\x3e\xa9", [0x1f48] = "\x3e\xaa", [0x1f49] = "\x3e\xab",
  [0x1f4a] = "\x3e\xac", [0x1f4b] = "\x3e\xad", [0x1f4c] = "\x3e\xae",
  [0x1f4d] = "\x3e\xaf", [0x1f4e] = "\x3e\xb0", [0x1f4f] = "\x3e\xb1",
  [0x1f50] = "\x3e\xb2", [0x1f51] = "\x3e\xb3", [0x1f52] = "\x3e\xb4",
  [0x1f53] = "\x3e\xb5", [0x1f54] = "\x3e\xb6", [0x1f55] = "\x3e\xb7",
  [0x1f56] = "\x3e\xb8", [0x1f57] = "\x3e\xb9", [0x1f58] = "\x3e\xba",
  [0x1f59] = "\x3e\xbb", [0x1f5a] = "\x3e\xbc", [0x1f5b] = "\x3e\xbd",
  [0x1f5c] = "\x3e\xbe", [0x1f5d] = "\x3e\xbf", [0x1f5e] = "\x3e\xc0",
  [0x1f5f] = "\x3e\xc1", [0x1f60] = "\x3e\xc2", [0x1f61] = "\x3e\xc3",
  [0x1f62] = "\x3e\xc4", [0x1f63] = "\x3e\xc5", [0x1f64] = "\x3e\xc6",
  [0x1f65] = "\x3e\xc7", [0x1f66] = "\x3e\xc8", [0x1f67] = "\x3e\xc9",
  [0x1f68] = "\x3e\xca", [0x1f69] = "\x3e\xcb", [0x1f6a] = "\x3e\xcc",
  [0x1f6b] = "\x3e\xcd", [0x1f6c] = "\x3e\xce", [0x1f6d] = "\x3e\xcf",
  [0x1f6e] = "\x3e\xd0", [0x1f6f] = "\x3e\xd1", [0x1f70] = "\x3e\xd2",
  [0x1f71] = "\x3e\xd3", [0x1f72] = "\x3e\xd4", [0x1f73] = "\x3e\xd5",
  [0x1f74] = "\x3e\xd6", [0x1f75] = "\x3e\xd7", [0x1f76] = "\x3e\xd8",
  [0x1f77] = "\x3e\xd9", [0x1f78] = "\x3e\xda", [0x1f79] = "\x3e\xdb",
  [0x1f7a] = "\x3e\xdc", [0x1f7b] = "\x3e\xdd", [0x1f7c] = "\x3e\xde",
  [0x1f7d] = "\x3e\xdf", [0x1f7e] = "\x3e\xe0", [0x1f7f] = "\x3e\xe1",
  [0x1f80] = "\x3e\xe2", [0x1f81] = "\x3e\xe3", [0x1f82] = "\x3e\xe4",
  [0x1f83] = "\x3e\xe5", [0x1f84] = "\x3e\xe6", [0x1f85] = "\x3e\xe7",
  [0x1f86] = "\x3e\xe8", [0x1f87] = "\x3e\xe9", [0x1f88] = "\x3e\xea",
  [0x1f89] = "\x3e\xeb", [0x1f8a] = "\x3e\xec", [0x1f8b] = "\x3e\xed",
  [0x1f8c] = "\x3e\xee", [0x1f8d] = "\x3e\xef", [0x1f8e] = "\x3e\xf0",
  [0x1f8f] = "\x3e\xf1", [0x1f90] = "\xa9\x5c", [0x1f91] = "\x3e\xf2",
  [0x1f92] = "\x3e\xf3", [0x1f93] = "\xa8\x43", [0x1f94] = "\xa1\xaa",
  [0x1f95] = "\xa8\x44", [0x1f96] = "\xa1\xac", [0x1f97] = "\x3e\xf4",
  [0x1f98] = "\xa1\xae", [0x1f99] = "\xa1\xaf", [0x1f9a] = "\x3e\xf5",
  [0x1f9b] = "\x3e\xf6", [0x1f9c] = "\xa1\xb0", [0x1f9d] = "\xa1\xb1",
  [0x1f9e] = "\x3e\xf7", [0x1f9f] = "\x3e\xf8", [0x1fa0] = "\x3e\xf9",
  [0x1fa1] = "\x3e\xfa", [0x1fa2] = "\x3e\xfb", [0x1fa3] = "\x3e\xfc",
  [0x1fa4] = "\x3e\xfd", [0x1fa5] = "\xa8\x45", [0x1fa6] = "\xa1\xad",
  [0x1fa7] = "\x3e\xfe", [0x1fa8] = "\x3e\xff", [0x1fa9] = "\x3f\x00",
  [0x1faa] = "\x3f\x01", [0x1fab] = "\x3f\x02", [0x1fac] = "\x3f\x03",
  [0x1fad] = "\x3f\x04", [0x1fae] = "\x3f\x05", [0x1faf] = "\x3f\x06",
  [0x1fb0] = "\xa1\xeb", [0x1fb1] = "\x3f\x07", [0x1fb2] = "\xa1\xe4",
  [0x1fb3] = "\xa1\xe5", [0x1fb4] = "\x3f\x08", [0x1fb5] = "\xa8\x46",
  [0x1fb6] = "\x3f\x09", [0x1fb7] = "\x3f\x0a", [0x1fb8] = "\x3f\x0b",
  [0x1fb9] = "\x3f\x0c", [0x1fba] = "\x3f\x0d", [0x1fbb] = "\xa1\xf9",
  [0x1fbc] = "\x3f\x0e", [0x1fbd] = "\x3f\x0f", [0x1fbe] = "\x3f\x10",
  [0x1fbf] = "\x3f\x11", [0x1fc0] = "\x3f\x12", [0x1fc1] = "\x3f\x13",
  [0x1fc2] = "\x3f\x14", [0x1fc3] = "\x3f\x15", [0x1fc4] = "\x3f\x16",
  [0x1fc5] = "\x3f\x17", [0x1fc6] = "\x3f\x18", [0x1fc7] = "\x3f\x19",
  [0x1fc8] = "\x3f\x1a", [0x1fc9] = "\x3f\x1b", [0x1fca] = "\x3f\x1c",
  [0x1fcb] = "\x3f\x1d", [0x1fcc] = "\x3f\x1e", [0x1fcd] = "\x3f\x1f",
  [0x1fce] = "\x3f\x20", [0x1fcf] = "\x3f\x21", [0x1fd0] = "\x3f\x22",
  [0x1fd1] = "\x3f\x23", [0x1fd2] = "\x3f\x24", [0x1fd3] = "\x3f\x25",
  [0x1fd4] = "\x3f\x26", [0x1fd5] = "\x3f\x27", [0x1fd6] = "\x3f\x28",
  [0x1fd7] = "\x3f\x29", [0x1fd8] = "\x3f\x2a", [0x1fd9] = "\x3f\x2b",
  [0x1fda] = "\x3f\x2c", [0x1fdb] = "\x3f\x2d", [0x1fdc] = "\x3f\x2e",
  [0x1fdd] = "\x3f\x2f", [0x1fde] = "\x3f\x30", [0x1fdf] = "\x3f\x31",
  [0x1fe0] = "\x3f\x32", [0x1fe1] = "\x3f\x33", [0x1fe2] = "\x3f\x34",
  [0x1fe3] = "\x3f\x35", [0x1fe4] = "\x3f\x36", [0x1fe5] = "\x3f\x37",
  [0x1fe6] = "\x3f\x38", [0x1fe7] = "\x3f\x39", [0x1fe8] = "\x3f\x3a",
  [0x1fe9] = "\x3f\x3b", [0x1fea] = "\x3f\x3c", [0x1feb] = "\x3f\x3d",
  [0x1fec] = "\x3f\x3e", [0x1fed] = "\x3f\x3f", [0x1fee] = "\x3f\x40",
  [0x1fef] = "\x3f\x41", [0x1ff0] = "\x3f\x42", [0x1ff1] = "\x3f\x43",
  [0x1ff2] = "\x3f\x44", [0x1ff3] = "\x3f\x45", [0x1ff4] = "\x3f\x46",
  [0x1ff5] = "\x3f\x47", [0x1ff6] = "\x3f\x48", [0x1ff7] = "\x3f\x49",
  [0x1ff8] = "\x3f\x4a", [0x1ff9] = "\x3f\x4b", [0x1ffa] = "\x3f\x4c",
  [0x1ffb] = "\x3f\x4d", [0x1ffc] = "\x3f\x4e", [0x1ffd] = "\x3f\x4f",
  [0x1ffe] = "\x3f\x50", [0x1fff] = "\x3f\x51", [0x2000] = "\x3f\x52",
  [0x2001] = "\x3f\x53", [0x2002] = "\x3f\x54", [0x2003] = "\x3f\x55",
  [0x2004] = "\x3f\x56", [0x2005] = "\x3f\x57", [0x2006] = "\x3f\x58",
  [0x2007] = "\x3f\x59", [0x2008] = "\x3f\x5a", [0x2009] = "\x3f\x5b",
  [0x200a] = "\x3f\x5c", [0x200b] = "\x3f\x5d", [0x200c] = "\x3f\x5e",
  [0x200d] = "\x3f\x5f", [0x200e] = "\x3f\x60", [0x200f] = "\x3f\x61",
  [0x2010] = "\x3f\x62", [0x2011] = "\x3f\x63", [0x2012] = "\x3f\x64",
  [0x2013] = "\x3f\x65", [0x2014] = "\x3f\x66", [0x2015] = "\x3f\x67",
  [0x2016] = "\x3f\x68", [0x2017] = "\x3f\x69", [0x2018] = "\x3f\x6a",
  [0x2019] = "\x3f\x6b", [0x201a] = "\x3f\x6c", [0x201b] = "\x3f\x6d",
  [0x201c] = "\x3f\x6e", [0x201d] = "\x3f\x6f", [0x201e] = "\x3f\x70",
  [0x201f] = "\x3f\x71", [0x2020] = "\x3f\x72", [0x2021] = "\x3f\x73",
  [0x2022] = "\x3f\x74", [0x2023] = "\x3f\x75", [0x2024] = "\x3f\x76",
  [0x2025] = "\x3f\x77", [0x2026] = "\x3f\x78", [0x2027] = "\x3f\x79",
  [0x2028] = "\x3f\x7a", [0x2029] = "\x3f\x7b", [0x202a] = "\x3f\x7c",
  [0x202b] = "\x3f\x7d", [0x202c] = "\xa2\xe3", [0x202d] = "\x3f\x7e",
  [0x202e] = "\x3f\x7f", [0x202f] = "\x3f\x80", [0x2030] = "\x3f\x81",
  [0x2031] = "\x3f\x82", [0x2032] = "\x3f\x83", [0x2033] = "\x3f\x84",
  [0x2034] = "\x3f\x85", [0x2035] = "\x3f\x86", [0x2036] = "\x3f\x87",
  [0x2037] = "\x3f\x88", [0x2038] = "\x3f\x89", [0x2039] = "\x3f\x8a",
  [0x203a] = "\x3f\x8b", [0x203b] = "\x3f\x8c", [0x203c] = "\x3f\x8d",
  [0x203d] = "\x3f\x8e", [0x203e] = "\x3f\x8f", [0x203f] = "\x3f\x90",
  [0x2040] = "\x3f\x91", [0x2041] = "\x3f\x92", [0x2042] = "\x3f\x93",
  [0x2043] = "\x3f\x94", [0x2044] = "\x3f\x95", [0x2045] = "\x3f\x96",
  [0x2046] = "\x3f\x97", [0x2047] = "\x3f\x98", [0x2048] = "\x3f\x99",
  [0x2049] = "\x3f\x9a", [0x204a] = "\x3f\x9b", [0x204b] = "\x3f\x9c",
  [0x204c] = "\x3f\x9d", [0x204d] = "\x3f\x9e", [0x204e] = "\x3f\x9f",
  [0x204f] = "\x3f\xa0", [0x2050] = "\x3f\xa1", [0x2051] = "\x3f\xa2",
  [0x2052] = "\x3f\xa3", [0x2053] = "\x3f\xa4", [0x2054] = "\x3f\xa5",
  [0x2055] = "\x3f\xa6", [0x2056] = "\x3f\xa7", [0x2057] = "\x3f\xa8",
  [0x2058] = "\x3f\xa9", [0x2059] = "\x3f\xaa", [0x205a] = "\x3f\xab",
  [0x205b] = "\x3f\xac", [0x205c] = "\x3f\xad", [0x205d] = "\x3f\xae",
  [0x205e] = "\x3f\xaf", [0x205f] = "\x3f\xb0", [0x2060] = "\x3f\xb1",
  [0x2061] = "\x3f\xb2", [0x2062] = "\x3f\xb3", [0x2063] = "\x3f\xb4",
  [0x2064] = "\x3f\xb5", [0x2065] = "\x3f\xb6", [0x2066] = "\x3f\xb7",
  [0x2067] = "\x3f\xb8", [0x2068] = "\x3f\xb9", [0x2069] = "\x3f\xba",
  [0x206a] = "\x3f\xbb", [0x206b] = "\x3f\xbc", [0x206c] = "\x3f\xbd",
  [0x206d] = "\x3f\xbe", [0x206e] = "\x3f\xbf", [0x206f] = "\x3f\xc0",
  [0x2070] = "\x3f\xc1", [0x2071] = "\x3f\xc2", [0x2072] = "\x3f\xc3",
  [0x2073] = "\x3f\xc4", [0x2074] = "\x3f\xc5", [0x2075] = "\x3f\xc6",
  [0x2076] = "\x3f\xc7", [0x2077] = "\x3f\xc8", [0x2078] = "\x3f\xc9",
  [0x2079] = "\x3f\xca", [0x207a] = "\x3f\xcb", [0x207b] = "\x3f\xcc",
  [0x207c] = "\x3f\xcd", [0x207d] = "\x3f\xce", [0x207e] = "\x3f\xcf",
  [0x207f] = "\x3f\xd0", [0x2080] = "\x3f\xd1", [0x2081] = "\x3f\xd2",
  [0x2082] = "\x3f\xd3", [0x2083] = "\xa1\xe6", [0x2084] = "\x3f\xd4",
  [0x2085] = "\xa8\x47", [0x2086] = "\x3f\xd5", [0x2087] = "\x3f\xd6",
  [0x2088] = "\x3f\xd7", [0x2089] = "\xa8\x48", [0x208a] = "\x3f\xd8",
  [0x208b] = "\x3f\xd9", [0x208c] = "\x3f\xda", [0x208d] = "\x3f\xdb",
  [0x208e] = "\x3f\xdc", [0x208f] = "\x3f\xdd", [0x2090] = "\x3f\xde",
  [0x2091] = "\x3f\xdf", [0x2092] = "\x3f\xe0", [0x2093] = "\x3f\xe1",
  [0x2094] = "\x3f\xe2", [0x2095] = "\x3f\xe3", [0x2096] = "\xa1\xed",
  [0x2097] = "\x3f\xe4", [0x2098] = "\x3f\xe5", [0x2099] = "\x3f\xe6",
  [0x209a] = "\x3f\xe7", [0x209b] = "\x3f\xe8", [0x209c] = "\x3f\xe9",
  [0x209d] = "\x3f\xea", [0x209e] = "\x3f\xeb", [0x209f] = "\x3f\xec",
  [0x20a0] = "\x3f\xed", [0x20a1] = "\xa9\x59", [0x20a2] = "\x3f\xee",
  [0x20a3] = "\x3f\xef", [0x20a4] = "\x3f\xf0", [0x20a5] = "\x3f\xf1",
  [0x20a6] = "\x3f\xf2", [0x20a7] = "\x3f\xf3", [0x20a8] = "\x3f\xf4",
  [0x20a9] = "\x3f\xf5", [0x20aa] = "\x3f\xf6", [0x20ab] = "\x3f\xf7",
  [0x20ac] = "\x3f\xf8", [0x20ad] = "\x3f\xf9", [0x20ae] = "\x3f\xfa",
  [0x20af] = "\x3f\xfb", [0x20b0] = "\x3f\xfc", [0x20b1] = "\x3f\xfd",
  [0x20b2] = "\x3f\xfe", [0x20b3] = "\x3f\xff", [0x20b4] = "\x40\x00",
  [0x20b5] = "\x40\x01", [0x20b6] = "\x40\x02", [0x20b7] = "\x40\x03",
  [0x20b8] = "\x40\x04", [0x20b9] = "\x40\x05", [0x20ba] = "\x40\x06",
  [0x20bb] = "\x40\x07", [0x20bc] = "\x40\x08", [0x20bd] = "\x40\x09",
  [0x20be] = "\x40\x0a", [0x20bf] = "\x40\x0b", [0x20c0] = "\x40\x0c",
  [0x20c1] = "\x40\x0d", [0x20c2] = "\x40\x0e", [0x20c3] = "\x40\x0f",
  [0x20c4] = "\x40\x10", [0x20c5] = "\x40\x11", [0x20c6] = "\x40\x12",
  [0x20c7] = "\x40\x13", [0x20c8] = "\x40\x14", [0x20c9] = "\x40\x15",
  [0x20ca] = "\x40\x16", [0x20cb] = "\x40\x17", [0x20cc] = "\x40\x18",
  [0x20cd] = "\x40\x19", [0x20ce] = "\x40\x1a", [0x20cf] = "\x40\x1b",
  [0x20d0] = "\x40\x1c", [0x20d1] = "\x40\x1d", [0x20d2] = "\x40\x1e",
  [0x20d3] = "\x40\x1f", [0x20d4] = "\x40\x20", [0x20d5] = "\x40\x21",
  [0x20d6] = "\x40\x22", [0x20d7] = "\x40\x23", [0x20d8] = "\x40\x24",
  [0x20d9] = "\x40\x25", [0x20da] = "\x40\x26", [0x20db] = "\x40\x27",
  [0x20dc] = "\x40\x28", [0x20dd] = "\x40\x29", [0x20de] = "\x40\x2a",
  [0x20df] = "\x40\x2b", [0x20e0] = "\xa2\xf1", [0x20e1] = "\xa2\xf2",
  [0x20e2] = "\xa2\xf3", [0x20e3] = "\xa2\xf4", [0x20e4] = "\xa2\xf5",
  [0x20e5] = "\xa2\xf6", [0x20e6] = "\xa2\xf7", [0x20e7] = "\xa2\xf8",
  [0x20e8] = "\xa2\xf9", [0x20e9] = "\xa2\xfa", [0x20ea] = "\xa2\xfb",
  [0x20eb] = "\xa2\xfc", [0x20ec] = "\x40\x2c", [0x20ed] = "\x40\x2d",
  [0x20ee] = "\x40\x2e", [0x20ef] = "\x40\x2f", [0x20f0] = "\xa2\xa1",
  [0x20f1] = "\xa2\xa2", [0x20f2] = "\xa2\xa3", [0x20f3] = "\xa2\xa4",
  [0x20f4] = "\xa2\xa5", [0x20f5] = "\xa2\xa6", [0x20f6] = "\xa2\xa7",
  [0x20f7] = "\xa2\xa8", [0x20f8] = "\xa2\xa9", [0x20f9] = "\xa2\xaa",
  [0x20fa] = "\x40\x30", [0x20fb] = "\x40\x31", [0x20fc] = "\x40\x32",
  [0x20fd] = "\x40\x33", [0x20fe] = "\x40\x34", [0x20ff] = "\x40\x35",
  [0x2100] = "\x40\x36", [0x2101] = "\x40\x37", [0x2102] = "\x40\x38",
  [0x2103] = "\x40\x39", [0x2104] = "\x40\x3a", [0x2105] = "\x40\x3b",
  [0x2106] = "\x40\x3c", [0x2107] = "\x40\x3d", [0x2108] = "\x40\x3e",
  [0x2109] = "\x40\x3f", [0x210a] = "\x40\x40", [0x210b] = "\x40\x41",
  [0x210c] = "\x40\x42", [0x210d] = "\x40\x43", [0x210e] = "\x40\x44",
  [0x210f] = "\x40\x45", [0x2110] = "\xa1\xfb", [0x2111] = "\xa1\xfc",
  [0x2112] = "\xa1\xfa", [0x2113] = "\xa1\xfd", [0x2114] = "\x40\x46",
  [0x2115] = "\x40\x47", [0x2116] = "\xa8\x49", [0x2117] = "\xa8\x4a",
  [0x2118] = "\xa8\x4b", [0x2119] = "\xa8\x4c", [0x211a] = "\x40\x48",
  [0x211b] = "\x40\x49", [0x211c] = "\x40\x4a", [0x211d] = "\x40\x4b",
  [0x211e] = "\x40\x4c", [0x211f] = "\x40\x4d", [0x2120] = "\x40\x4e",
  [0x2121] = "\x40\x4f", [0x2122] = "\x40\x50", [0x2123] = "\x40\x51",
  [0x2124] = "\x40\x52", [0x2125] = "\x40\x53", [0x2126] = "\x40\x54",
  [0x2127] = "\x40\x55", [0x2128] = "\x40\x56", [0x2129] = "\x40\x57",
  [0x212a] = "\x40\x58", [0x212b] = "\x40\x59", [0x212c] = "\x40\x5a",
  [0x212d] = "\x40\x5b", [0x212e] = "\x40\x5c", [0x212f] = "\x40\x5d",
  [0x2130] = "\x40\x5e", [0x2131] = "\x40\x5f", [0x2132] = "\x40\x60",
  [0x2133] = "\x40\x61", [0x2134] = "\x40\x62", [0x2135] = "\x40\x63",
  [0x2136] = "\x40\x64", [0x2137] = "\x40\x65", [0x2138] = "\x40\x66",
  [0x2139] = "\x40\x67", [0x213a] = "\x40\x68", [0x213b] = "\x40\x69",
  [0x213c] = "\x40\x6a", [0x213d] = "\x40\x6b", [0x213e] = "\x40\x6c",
  [0x213f] = "\x40\x6d", [0x2140] = "\x40\x6e", [0x2141] = "\x40\x6f",
  [0x2142] = "\x40\x70", [0x2143] = "\x40\x71", [0x2144] = "\x40\x72",
  [0x2145] = "\x40\x73", [0x2146] = "\x40\x74", [0x2147] = "\x40\x75",
  [0x2148] = "\x40\x76", [0x2149] = "\x40\x77", [0x214a] = "\x40\x78",
  [0x214b] = "\x40\x79", [0x214c] = "\x40\x7a", [0x214d] = "\x40\x7b",
  [0x214e] = "\x40\x7c", [0x214f] = "\x40\x7d", [0x2150] = "\x40\x7e",
  [0x2151] = "\x40\x7f", [0x2152] = "\x40\x80", [0x2153] = "\x40\x81",
  [0x2154] = "\x40\x82", [0x2155] = "\x40\x83", [0x2156] = "\x40\x84",
  [0x2157] = "\x40\x85", [0x2158] = "\x40\x86", [0x2159] = "\x40\x87",
  [0x215a] = "\x40\x88", [0x215b] = "\x40\x89", [0x215c] = "\x40\x8a",
  [0x215d] = "\x40\x8b", [0x215e] = "\x40\x8c", [0x215f] = "\x40\x8d",
  [0x2160] = "\x40\x8e", [0x2161] = "\x40\x8f", [0x2162] = "\x40\x90",
  [0x2163] = "\x40\x91", [0x2164] = "\x40\x92", [0x2165] = "\x40\x93",
  [0x2166] = "\x40\x94", [0x2167] = "\x40\x95", [0x2168] = "\x40\x96",
  [0x2169] = "\x40\x97", [0x216a] = "\x40\x98", [0x216b] = "\x40\x99",
  [0x216c] = "\x40\x9a", [0x216d] = "\x40\x9b", [0x216e] = "\x40\x9c",
  [0x216f] = "\x40\x9d", [0x2170] = "\x40\x9e", [0x2171] = "\x40\x9f",
  [0x2172] = "\x40\xa0", [0x2173] = "\x40\xa1", [0x2174] = "\x40\xa2",
  [0x2175] = "\x40\xa3", [0x2176] = "\x40\xa4", [0x2177] = "\x40\xa5",
  [0x2178] = "\x40\xa6", [0x2179] = "\x40\xa7", [0x217a] = "\x40\xa8",
  [0x217b] = "\x40\xa9", [0x217c] = "\x40\xaa", [0x217d] = "\x40\xab",
  [0x217e] = "\x40\xac", [0x217f] = "\x40\xad", [0x2180] = "\x40\xae",
  [0x2181] = "\x40\xaf", [0x2182] = "\x40\xb0", [0x2183] = "\x40\xb1",
  [0x2184] = "\x40\xb2", [0x2185] = "\x40\xb3", [0x2186] = "\x40\xb4",
  [0x2187] = "\x40\xb5", [0x2188] = "\xa1\xca", [0x2189] = "\x40\xb6",
  [0x218a] = "\x40\xb7", [0x218b] = "\x40\xb8", [0x218c] = "\x40\xb9",
  [0x218d] = "\x40\xba", [0x218e] = "\x40\xbb", [0x218f] = "\xa1\xc7",
  [0x2190] = "\x40\xbc", [0x2191] = "\xa1\xc6", [0x2192] = "\x40\xbd",
  [0x2193] = "\x40\xbe", [0x2194] = "\x40\xbf", [0x2195] = "\xa8\x4d",
  [0x2196] = "\x40\xc0", [0x2197] = "\x40\xc1", [0x2198] = "\x40\xc2",
  [0x2199] = "\x40\xc3", [0x219a] = "\xa1\xcc", [0x219b] = "\x40\xc4",
  [0x219c] = "\x40\xc5", [0x219d] = "\xa1\xd8", [0x219e] = "\xa1\xde",
  [0x219f] = "\xa8\x4e", [0x21a0] = "\xa1\xcf", [0x21a1] = "\x40\xc6",
  [0x21a2] = "\x40\xc7", [0x21a3] = "\xa8\x4f", [0x21a4] = "\x40\xc8",
  [0x21a5] = "\xa1\xce", [0x21a6] = "\x40\xc9", [0x21a7] = "\xa1\xc4",
  [0x21a8] = "\xa1\xc5", [0x21a9] = "\xa1\xc9", [0x21aa] = "\xa1\xc8",
  [0x21ab] = "\xa1\xd2", [0x21ac] = "\x40\xca", [0x21ad] = "\x40\xcb",
  [0x21ae] = "\xa1\xd3", [0x21af] = "\x40\xcc", [0x21b0] = "\x40\xcd",
  [0x21b1] = "\x40\xce", [0x21b2] = "\x40\xcf", [0x21b3] = "\x40\xd0",
  [0x21b4] = "\xa1\xe0", [0x21b5] = "\xa1\xdf", [0x21b6] = "\xa1\xc3",
  [0x21b7] = "\xa1\xcb", [0x21b8] = "\x40\xd1", [0x21b9] = "\x40\xd2",
  [0x21ba] = "\x40\xd3", [0x21bb] = "\x40\xd4", [0x21bc] = "\x40\xd5",
  [0x21bd] = "\xa1\xd7", [0x21be] = "\x40\xd6", [0x21bf] = "\x40\xd7",
  [0x21c0] = "\x40\xd8", [0x21c1] = "\x40\xd9", [0x21c2] = "\x40\xda",
  [0x21c3] = "\x40\xdb", [0x21c4] = "\x40\xdc", [0x21c5] = "\x40\xdd",
  [0x21c6] = "\x40\xde", [0x21c7] = "\x40\xdf", [0x21c8] = "\xa1\xd6",
  [0x21c9] = "\x40\xe0", [0x21ca] = "\x40\xe1", [0x21cb] = "\x40\xe2",
  [0x21cc] = "\xa1\xd5", [0x21cd] = "\x40\xe3", [0x21ce] = "\x40\xe4",
  [0x21cf] = "\x40\xe5", [0x21d0] = "\x40\xe6", [0x21d1] = "\x40\xe7",
  [0x21d2] = "\xa8\x50", [0x21d3] = "\x40\xe8", [0x21d4] = "\x40\xe9",
  [0x21d5] = "\x40\xea", [0x21d6] = "\x40\xeb", [0x21d7] = "\x40\xec",
  [0x21d8] = "\x40\xed", [0x21d9] = "\x40\xee", [0x21da] = "\x40\xef",
  [0x21db] = "\x40\xf0", [0x21dc] = "\x40\xf1", [0x21dd] = "\x40\xf2",
  [0x21de] = "\x40\xf3", [0x21df] = "\x40\xf4", [0x21e0] = "\xa1\xd9",
  [0x21e1] = "\xa1\xd4", [0x21e2] = "\x40\xf5", [0x21e3] = "\x40\xf6",
  [0x21e4] = "\xa1\xdc", [0x21e5] = "\xa1\xdd", [0x21e6] = "\xa8\x51",
  [0x21e7] = "\xa8\x52", [0x21e8] = "\x40\xf7", [0x21e9] = "\x40\xf8",
  [0x21ea] = "\x40\xf9", [0x21eb] = "\x40\xfa", [0x21ec] = "\x40\xfb",
  [0x21ed] = "\x40\xfc", [0x21ee] = "\xa1\xda", [0x21ef] = "\xa1\xdb",
  [0x21f0] = "\x40\xfd", [0x21f1] = "\x40\xfe", [0x21f2] = "\x40\xff",
  [0x21f3] = "\x41\x00", [0x21f4] = "\x41\x01", [0x21f5] = "\x41\x02",
  [0x21f6] = "\x41\x03", [0x21f7] = "\x41\x04", [0x21f8] = "\x41\x05",
  [0x21f9] = "\x41\x06", [0x21fa] = "\x41\x07", [0x21fb] = "\x41\x08",
  [0x21fc] = "\x41\x09", [0x21fd] = "\x41\x0a", [0x21fe] = "\x41\x0b",
  [0x21ff] = "\x41\x0c", [0x2200] = "\x41\x0d", [0x2201] = "\x41\x0e",
  [0x2202] = "\x41\x0f", [0x2203] = "\x41\x10", [0x2204] = "\x41\x11",
  [0x2205] = "\x41\x12", [0x2206] = "\x41\x13", [0x2207] = "\x41\x14",
  [0x2208] = "\x41\x15", [0x2209] = "\x41\x16", [0x220a] = "\x41\x17",
  [0x220b] = "\x41\x18", [0x220c] = "\x41\x19", [0x220d] = "\x41\x1a",
  [0x220e] = "\x41\x1b", [0x220f] = "\x41\x1c", [0x2210] = "\x41\x1d",
  [0x2211] = "\x41\x1e", [0x2212] = "\x41\x1f", [0x2213] = "\x41\x20",
  [0x2214] = "\x41\x21", [0x2215] = "\xa8\x92", [0x2216] = "\x41\x22",
  [0x2217] = "\x41\x23", [0x2218] = "\x41\x24", [0x2219] = "\xa1\xd1",
  [0x221a] = "\x41\x25", [0x221b] = "\x41\x26", [0x221c] = "\x41\x27",
  [0x221d] = "\x41\x28", [0x221e] = "\x41\x29", [0x221f] = "\x41\x2a",
  [0x2220] = "\x41\x2b", [0x2221] = "\x41\x2c", [0x2222] = "\x41\x2d",
  [0x2223] = "\x41\x2e", [0x2224] = "\x41\x2f", [0x2225] = "\xa1\xcd",
  [0x2226] = "\x41\x30", [0x2227] = "\x41\x31", [0x2228] = "\x41\x32",
  [0x2229] = "\x41\x33", [0x222a] = "\x41\x34", [0x222b] = "\x41\x35",
  [0x222c] = "\x41\x36", [0x222d] = "\x41\x37", [0x222e] = "\x41\x38",
  [0x222f] = "\x41\x39", [0x2230] = "\x41\x3a", [0x2231] = "\x41\x3b",
  [0x2232] = "\x41\x3c", [0x2233] = "\x41\x3d", [0x2234] = "\x41\x3e",
  [0x2235] = "\x41\x3f", [0x2236] = "\x41\x40", [0x2237] = "\x41\x41",
  [0x2238] = "\x41\x42", [0x2239] = "\x41\x43", [0x223a] = "\x41\x44",
  [0x223b] = "\x41\x45", [0x223c] = "\x41\x46", [0x223d] = "\x41\x47",
  [0x223e] = "\x41\x48", [0x223f] = "\xa8\x53", [0x2240] = "\x41\x49",
  [0x2241] = "\x41\x4a", [0x2242] = "\x41\x4b", [0x2243] = "\x41\x4c",
  [0x2244] = "\x41\x4d", [0x2245] = "\x41\x4e", [0x2246] = "\x41\x4f",
  [0x2247] = "\x41\x50", [0x2248] = "\x41\x51", [0x2249] = "\x41\x52",
  [0x224a] = "\x41\x53", [0x224b] = "\x41\x54", [0x224c] = "\x41\x55",
  [0x224d] = "\x41\x56", [0x224e] = "\x41\x57", [0x224f] = "\x41\x58",
  [0x2250] = "\x41\x59", [0x2251] = "\x41\x5a", [0x2252] = "\x41\x5b",
  [0x2253] = "\x41\x5c", [0x2254] = "\x41\x5d", [0x2255] = "\x41\x5e",
  [0x2256] = "\x41\x5f", [0x2257] = "\x41\x60", [0x2258] = "\x41\x61",
  [0x2259] = "\x41\x62", [0x225a] = "\x41\x63", [0x225b] = "\x41\x64",
  [0x225c] = "\x41\x65", [0x225d] = "\x41\x66", [0x225e] = "\x41\x67",
  [0x225f] = "\x41\x68", [0x2260] = "\x41\x69", [0x2261] = "\x41\x6a",
  [0x2262] = "\x41\x6b", [0x2263] = "\x41\x6c", [0x2264] = "\x41\x6d",
  [0x2265] = "\x41\x6e", [0x2266] = "\x41\x6f", [0x2267] = "\x41\x70",
  [0x2268] = "\x41\x71", [0x2269] = "\x41\x72", [0x226a] = "\x41\x73",
  [0x226b] = "\x41\x74", [0x226c] = "\x41\x75", [0x226d] = "\x41\x76",
  [0x226e] = "\x41\x77", [0x226f] = "\x41\x78", [0x2270] = "\x41\x79",
  [0x2271] = "\x41\x7a", [0x2272] = "\x41\x7b", [0x2273] = "\x41\x7c",
  [0x2274] = "\x41\x7d", [0x2275] = "\x41\x7e", [0x2276] = "\x41\x7f",
  [0x2277] = "\x41\x80", [0x2278] = "\x41\x81", [0x2279] = "\x41\x82",
  [0x227a] = "\x41\x83", [0x227b] = "\x41\x84", [0x227c] = "\x41\x85",
  [0x227d] = "\x41\x86", [0x227e] = "\x41\x87", [0x227f] = "\x41\x88",
  [0x2280] = "\x41\x89", [0x2281] = "\x41\x8a", [0x2282] = "\x41\x8b",
  [0x2283] = "\x41\x8c", [0x2284] = "\x41\x8d", [0x2285] = "\x41\x8e",
  [0x2286] = "\x41\x8f", [0x2287] = "\x41\x90", [0x2288] = "\x41\x91",
  [0x2289] = "\x41\x92", [0x228a] = "\x41\x93", [0x228b] = "\x41\x94",
  [0x228c] = "\x41\x95", [0x228d] = "\x41\x96", [0x228e] = "\x41\x97",
  [0x228f] = "\x41\x98", [0x2290] = "\x41\x99", [0x2291] = "\x41\x9a",
  [0x2292] = "\xa1\xd0", [0x2293] = "\x41\x9b", [0x2294] = "\x41\x9c",
  [0x2295] = "\x41\x9d", [0x2296] = "\x41\x9e", [0x2297] = "\x41\x9f",
  [0x2298] = "\x41\xa0", [0x2299] = "\x41\xa1", [0x229a] = "\x41\xa2",
  [0x229b] = "\x41\xa3", [0x229c] = "\x41\xa4", [0x229d] = "\x41\xa5",
  [0x229e] = "\x41\xa6", [0x229f] = "\x41\xa7", [0x22a0] = "\x41\xa8",
  [0x22a1] = "\x41\xa9", [0x22a2] = "\x41\xaa", [0x22a3] = "\x41\xab",
  [0x22a4] = "\x41\xac", [0x22a5] = "\x41\xad", [0x22a6] = "\x41\xae",
  [0x22a7] = "\x41\xaf", [0x22a8] = "\x41\xb0", [0x22a9] = "\x41\xb1",
  [0x22aa] = "\x41\xb2", [0x22ab] = "\x41\xb3", [0x22ac] = "\x41\xb4",
  [0x22ad] = "\x41\xb5", [0x22ae] = "\x41\xb6", [0x22af] = "\x41\xb7",
  [0x22b0] = "\x41\xb8", [0x22b1] = "\x41\xb9", [0x22b2] = "\x41\xba",
  [0x22b3] = "\x41\xbb", [0x22b4] = "\x41\xbc", [0x22b5] = "\x41\xbd",
  [0x22b6] = "\x41\xbe", [0x22b7] = "\x41\xbf", [0x22b8] = "\x41\xc0",
  [0x22b9] = "\x41\xc1", [0x22ba] = "\x41\xc2", [0x22bb] = "\x41\xc3",
  [0x22bc] = "\x41\xc4", [0x22bd] = "\x41\xc5", [0x22be] = "\x41\xc6",
  [0x22bf] = "\x41\xc7", [0x22c0] = "\x41\xc8", [0x22c1] = "\x41\xc9",
  [0x22c2] = "\x41\xca", [0x22c3] = "\x41\xcb", [0x22c4] = "\x41\xcc",
  [0x22c5] = "\x41\xcd", [0x22c6] = "\x41\xce", [0x22c7] = "\x41\xcf",
  [0x22c8] = "\x41\xd0", [0x22c9] = "\x41\xd1", [0x22ca] = "\x41\xd2",
  [0x22cb] = "\x41\xd3", [0x22cc] = "\x41\xd4", [0x22cd] = "\x41\xd5",
  [0x22ce] = "\x41\xd6", [0x22cf] = "\x41\xd7", [0x22d0] = "\x41\xd8",
  [0x22d1] = "\x41\xd9", [0x22d2] = "\x41\xda", [0x22d3] = "\x41\xdb",
  [0x22d4] = "\x41\xdc", [0x22d5] = "\x41\xdd", [0x22d6] = "\x41\xde",
  [0x22d7] = "\x41\xdf", [0x22d8] = "\x41\xe0", [0x22d9] = "\x41\xe1",
  [0x22da] = "\x41\xe2", [0x22db] = "\x41\xe3", [0x22dc] = "\x41\xe4",
  [0x22dd] = "\x41\xe5", [0x22de] = "\x41\xe6", [0x22df] = "\x41\xe7",
  [0x22e0] = "\x41\xe8", [0x22e1] = "\x41\xe9", [0x22e2] = "\x41\xea",
  [0x22e3] = "\x41\xeb", [0x22e4] = "\x41\xec", [0x22e5] = "\x41\xed",
  [0x22e6] = "\x41\xee", [0x22e7] = "\x41\xef", [0x22e8] = "\x41\xf0",
  [0x22e9] = "\x41\xf1", [0x22ea] = "\x41\xf2", [0x22eb] = "\x41\xf3",
  [0x22ec] = "\x41\xf4", [0x22ed] = "\x41\xf5", [0x22ee] = "\x41\xf6",
  [0x22ef] = "\x41\xf7", [0x22f0] = "\x41\xf8", [0x22f1] = "\x41\xf9",
  [0x22f2] = "\x41\xfa", [0x22f3] = "\x41\xfb", [0x22f4] = "\x41\xfc",
  [0x22f5] = "\x41\xfd", [0x22f6] = "\x41\xfe", [0x22f7] = "\x41\xff",
  [0x22f8] = "\x42\x00", [0x22f9] = "\x42\x01", [0x22fa] = "\x42\x02",
  [0x22fb] = "\x42\x03", [0x22fc] = "\x42\x04", [0x22fd] = "\x42\x05",
  [0x22fe] = "\x42\x06", [0x22ff] = "\x42\x07", [0x2300] = "\x42\x08",
  [0x2301] = "\x42\x09", [0x2302] = "\x42\x0a", [0x2303] = "\x42\x0b",
  [0x2304] = "\x42\x0c", [0x2305] = "\x42\x0d", [0x2306] = "\x42\x0e",
  [0x2307] = "\x42\x0f", [0x2308] = "\x42\x10", [0x2309] = "\x42\x11",
  [0x230a] = "\x42\x12", [0x230b] = "\x42\x13", [0x230c] = "\x42\x14",
  [0x230d] = "\x42\x15", [0x230e] = "\x42\x16", [0x230f] = "\x42\x17",
  [0x2310] = "\x42\x18", [0x2311] = "\x42\x19", [0x2312] = "\x42\x1a",
  [0x2313] = "\x42\x1b", [0x2314] = "\x42\x1c", [0x2315] = "\x42\x1d",
  [0x2316] = "\x42\x1e", [0x2317] = "\x42\x1f", [0x2318] = "\x42\x20",
  [0x2319] = "\x42\x21", [0x231a] = "\x42\x22", [0x231b] = "\x42\x23",
  [0x231c] = "\x42\x24", [0x231d] = "\x42\x25", [0x231e] = "\x42\x26",
  [0x231f] = "\x42\x27", [0x2320] = "\x42\x28", [0x2321] = "\x42\x29",
  [0x2322] = "\x42\x2a", [0x2323] = "\x42\x2b", [0x2324] = "\x42\x2c",
  [0x2325] = "\x42\x2d", [0x2326] = "\x42\x2e", [0x2327] = "\x42\x2f",
  [0x2328] = "\x42\x30", [0x2329] = "\x42\x31", [0x232a] = "\x42\x32",
  [0x232b] = "\x42\x33", [0x232c] = "\x42\x34", [0x232d] = "\x42\x35",
  [0x232e] = "\x42\x36", [0x232f] = "\x42\x37", [0x2330] = "\x42\x38",
  [0x2331] = "\x42\x39", [0x2332] = "\x42\x3a", [0x2333] = "\x42\x3b",
  [0x2334] = "\x42\x3c", [0x2335] = "\x42\x3d", [0x2336] = "\x42\x3e",
  [0x2337] = "\x42\x3f", [0x2338] = "\x42\x40", [0x2339] = "\x42\x41",
  [0x233a] = "\x42\x42", [0x233b] = "\x42\x43", [0x233c] = "\x42\x44",
  [0x233d] = "\x42\x45", [0x233e] = "\x42\x46", [0x233f] = "\x42\x47",
  [0x2340] = "\x42\x48", [0x2341] = "\x42\x49", [0x2342] = "\x42\x4a",
  [0x2343] = "\x42\x4b", [0x2344] = "\x42\x4c", [0x2345] = "\x42\x4d",
  [0x2346] = "\x42\x4e", [0x2347] = "\x42\x4f", [0x2348] = "\x42\x50",
  [0x2349] = "\x42\x51", [0x234a] = "\x42\x52", [0x234b] = "\x42\x53",
  [0x234c] = "\x42\x54", [0x234d] = "\x42\x55", [0x234e] = "\x42\x56",
  [0x234f] = "\x42\x57", [0x2350] = "\x42\x58", [0x2351] = "\x42\x59",
  [0x2352] = "\x42\x5a", [0x2353] = "\x42\x5b", [0x2354] = "\x42\x5c",
  [0x2355] = "\x42\x5d", [0x2356] = "\x42\x5e", [0x2357] = "\x42\x5f",
  [0x2358] = "\x42\x60", [0x2359] = "\x42\x61", [0x235a] = "\x42\x62",
  [0x235b] = "\x42\x63", [0x235c] = "\x42\x64", [0x235d] = "\x42\x65",
  [0x235e] = "\x42\x66", [0x235f] = "\x42\x67", [0x2360] = "\x42\x68",
  [0x2361] = "\x42\x69", [0x2362] = "\x42\x6a", [0x2363] = "\x42\x6b",
  [0x2364] = "\x42\x6c", [0x2365] = "\x42\x6d", [0x2366] = "\x42\x6e",
  [0x2367] = "\x42\x6f", [0x2368] = "\x42\x70", [0x2369] = "\x42\x71",
  [0x236a] = "\x42\x72", [0x236b] = "\x42\x73", [0x236c] = "\x42\x74",
  [0x236d] = "\x42\x75", [0x236e] = "\x42\x76", [0x236f] = "\x42\x77",
  [0x2370] = "\x42\x78", [0x2371] = "\x42\x79", [0x2372] = "\x42\x7a",
  [0x2373] = "\x42\x7b", [0x2374] = "\x42\x7c", [0x2375] = "\x42\x7d",
  [0x2376] = "\x42\x7e", [0x2377] = "\x42\x7f", [0x2378] = "\x42\x80",
  [0x2379] = "\x42\x81", [0x237a] = "\x42\x82", [0x237b] = "\x42\x83",
  [0x237c] = "\x42\x84", [0x237d] = "\x42\x85", [0x237e] = "\x42\x86",
  [0x237f] = "\x42\x87", [0x2380] = "\x42\x88", [0x2381] = "\x42\x89",
  [0x2382] = "\x42\x8a", [0x2383] = "\x42\x8b", [0x2384] = "\x42\x8c",
  [0x2385] = "\x42\x8d", [0x2386] = "\x42\x8e", [0x2387] = "\x42\x8f",
  [0x2388] = "\x42\x90", [0x2389] = "\x42\x91", [0x238a] = "\x42\x92",
  [0x238b] = "\x42\x93", [0x238c] = "\x42\x94", [0x238d] = "\x42\x95",
  [0x238e] = "\x42\x96", [0x238f] = "\x42\x97", [0x2390] = "\x42\x98",
  [0x2391] = "\x42\x99", [0x2392] = "\x42\x9a", [0x2393] = "\x42\x9b",
  [0x2394] = "\x42\x9c", [0x2395] = "\x42\x9d", [0x2396] = "\x42\x9e",
  [0x2397] = "\x42\x9f", [0x2398] = "\x42\xa0", [0x2399] = "\x42\xa1",
  [0x239a] = "\x42\xa2", [0x239b] = "\x42\xa3", [0x239c] = "\x42\xa4",
  [0x239d] = "\x42\xa5", [0x239e] = "\x42\xa6", [0x239f] = "\x42\xa7",
  [0x23a0] = "\x42\xa8", [0x23a1] = "\x42\xa9", [0x23a2] = "\x42\xaa",
  [0x23a3] = "\x42\xab", [0x23a4] = "\x42\xac", [0x23a5] = "\x42\xad",
  [0x23a6] = "\x42\xae", [0x23a7] = "\x42\xaf", [0x23a8] = "\x42\xb0",
  [0x23a9] = "\x42\xb1", [0x23aa] = "\x42\xb2", [0x23ab] = "\x42\xb3",
  [0x23ac] = "\x42\xb4", [0x23ad] = "\x42\xb5", [0x23ae] = "\x42\xb6",
  [0x23af] = "\x42\xb7", [0x23b0] = "\x42\xb8", [0x23b1] = "\x42\xb9",
  [0x23b2] = "\x42\xba", [0x23b3] = "\x42\xbb", [0x23b4] = "\x42\xbc",
  [0x23b5] = "\x42\xbd", [0x23b6] = "\x42\xbe", [0x23b7] = "\x42\xbf",
  [0x23b8] = "\x42\xc0", [0x23b9] = "\x42\xc1", [0x23ba] = "\x42\xc2",
  [0x23bb] = "\x42\xc3", [0x23bc] = "\x42\xc4", [0x23bd] = "\x42\xc5",
  [0x23be] = "\x42\xc6", [0x23bf] = "\x42\xc7", [0x23c0] = "\x42\xc8",
  [0x23c1] = "\x42\xc9", [0x23c2] = "\x42\xca", [0x23c3] = "\x42\xcb",
  [0x23c4] = "\x42\xcc", [0x23c5] = "\x42\xcd", [0x23c6] = "\x42\xce",
  [0x23c7] = "\x42\xcf", [0x23c8] = "\x42\xd0", [0x23c9] = "\x42\xd1",
  [0x23ca] = "\x42\xd2", [0x23cb] = "\x42\xd3", [0x23cc] = "\x42\xd4",
  [0x23cd] = "\x42\xd5", [0x23ce] = "\x42\xd6", [0x23cf] = "\x42\xd7",
  [0x23d0] = "\x42\xd8", [0x23d1] = "\x42\xd9", [0x23d2] = "\x42\xda",
  [0x23d3] = "\x42\xdb", [0x23d4] = "\x42\xdc", [0x23d5] = "\x42\xdd",
  [0x23d6] = "\x42\xde", [0x23d7] = "\x42\xdf", [0x23d8] = "\x42\xe0",
  [0x23d9] = "\x42\xe1", [0x23da] = "\x42\xe2", [0x23db] = "\x42\xe3",
  [0x23dc] = "\x42\xe4", [0x23dd] = "\x42\xe5", [0x23de] = "\x42\xe6",
  [0x23df] = "\x42\xe7", [0x23e0] = "\xa2\xd9", [0x23e1] = "\xa2\xda",
  [0x23e2] = "\xa2\xdb", [0x23e3] = "\xa2\xdc", [0x23e4] = "\xa2\xdd",
  [0x23e5] = "\xa2\xde", [0x23e6] = "\xa2\xdf", [0x23e7] = "\xa2\xe0",
  [0x23e8] = "\xa2\xe1", [0x23e9] = "\xa2\xe2", [0x23ea] = "\x42\xe8",
  [0x23eb] = "\x42\xe9", [0x23ec] = "\x42\xea", [0x23ed] = "\x42\xeb",
  [0x23ee] = "\x42\xec", [0x23ef] = "\x42\xed", [0x23f0] = "\x42\xee",
  [0x23f1] = "\x42\xef", [0x23f2] = "\x42\xf0", [0x23f3] = "\x42\xf1",
  [0x23f4] = "\xa2\xc5", [0x23f5] = "\xa2\xc6", [0x23f6] = "\xa2\xc7",
  [0x23f7] = "\xa2\xc8", [0x23f8] = "\xa2\xc9", [0x23f9] = "\xa2\xca",
  [0x23fa] = "\xa2\xcb", [0x23fb] = "\xa2\xcc", [0x23fc] = "\xa2\xcd",
  [0x23fd] = "\xa2\xce", [0x23fe] = "\xa2\xcf", [0x23ff] = "\xa2\xd0",
  [0x2400] = "\xa2\xd1", [0x2401] = "\xa2\xd2", [0x2402] = "\xa2\xd3",
  [0x2403] = "\xa2\xd4", [0x2404] = "\xa2\xd5", [0x2405] = "\xa2\xd6",
  [0x2406] = "\xa2\xd7", [0x2407] = "\xa2\xd8", [0x2408] = "\xa2\xb1",
  [0x2409] = "\xa2\xb2", [0x240a] = "\xa2\xb3", [0x240b] = "\xa2\xb4",
  [0x240c] = "\xa2\xb5", [0x240d] = "\xa2\xb6", [0x240e] = "\xa2\xb7",
  [0x240f] = "\xa2\xb8", [0x2410] = "\xa2\xb9", [0x2411] = "\xa2\xba",
  [0x2412] = "\xa2\xbb", [0x2413] = "\xa2\xbc", [0x2414] = "\xa2\xbd",
  [0x2415] = "\xa2\xbe", [0x2416] = "\xa2\xbf", [0x2417] = "\xa2\xc0",
  [0x2418] = "\xa2\xc1", [0x2419] = "\xa2\xc2", [0x241a] = "\xa2\xc3",
  [0x241b] = "\xa2\xc4", [0x241c] = "\x42\xf2", [0x241d] = "\x42\xf3",
  [0x241e] = "\x42\xf4", [0x241f] = "\x42\xf5", [0x2420] = "\x42\xf6",
  [0x2421] = "\x42\xf7", [0x2422] = "\x42\xf8", [0x2423] = "\x42\xf9",
  [0x2424] = "\x42\xfa", [0x2425] = "\x42\xfb", [0x2426] = "\x42\xfc",
  [0x2427] = "\x42\xfd", [0x2428] = "\x42\xfe", [0x2429] = "\x42\xff",
  [0x242a] = "\x43\x00", [0x242b] = "\x43\x01", [0x242c] = "\x43\x02",
  [0x242d] = "\x43\x03", [0x242e] = "\x43\x04", [0x242f] = "\x43\x05",
  [0x2430] = "\x43\x06", [0x2431] = "\x43\x07", [0x2432] = "\x43\x08",
  [0x2433] = "\x43\x09", [0x2434] = "\x43\x0a", [0x2435] = "\x43\x0b",
  [0x2436] = "\x43\x0c", [0x2437] = "\x43\x0d", [0x2438] = "\x43\x0e",
  [0x2439] = "\x43\x0f", [0x243a] = "\x43\x10", [0x243b] = "\x43\x11",
  [0x243c] = "\x43\x12", [0x243d] = "\x43\x13", [0x243e] = "\x43\x14",
  [0x243f] = "\x43\x15", [0x2440] = "\x43\x16", [0x2441] = "\x43\x17",
  [0x2442] = "\x43\x18", [0x2443] = "\x43\x19", [0x2444] = "\x43\x1a",
  [0x2445] = "\x43\x1b", [0x2446] = "\x43\x1c", [0x2447] = "\x43\x1d",
  [0x2448] = "\x43\x1e", [0x2449] = "\x43\x1f", [0x244a] = "\x43\x20",
  [0x244b] = "\x43\x21", [0x244c] = "\x43\x22", [0x244d] = "\x43\x23",
  [0x244e] = "\x43\x24", [0x244f] = "\x43\x25", [0x2450] = "\x43\x26",
  [0x2451] = "\x43\x27", [0x2452] = "\x43\x28", [0x2453] = "\x43\x29",
  [0x2454] = "\x43\x2a", [0x2455] = "\x43\x2b", [0x2456] = "\x43\x2c",
  [0x2457] = "\x43\x2d", [0x2458] = "\x43\x2e", [0x2459] = "\x43\x2f",
  [0x245a] = "\x43\x30", [0x245b] = "\x43\x31", [0x245c] = "\x43\x32",
  [0x245d] = "\x43\x33", [0x245e] = "\x43\x34", [0x245f] = "\x43\x35",
  [0x2460] = "\x43\x36", [0x2461] = "\x43\x37", [0x2462] = "\x43\x38",
  [0x2463] = "\x43\x39", [0x2464] = "\x43\x3a", [0x2465] = "\x43\x3b",
  [0x2466] = "\x43\x3c", [0x2467] = "\x43\x3d", [0x2468] = "\x43\x3e",
  [0x2469] = "\x43\x3f", [0x246a] = "\x43\x40", [0x246b] = "\x43\x41",
  [0x246c] = "\x43\x42", [0x246d] = "\x43\x43", [0x246e] = "\x43\x44",
  [0x246f] = "\x43\x45", [0x2470] = "\x43\x46", [0x2471] = "\x43\x47",
  [0x2472] = "\x43\x48", [0x2473] = "\x43\x49", [0x2474] = "\x43\x4a",
  [0x2475] = "\x43\x4b", [0x2476] = "\x43\x4c", [0x2477] = "\x43\x4d",
  [0x2478] = "\x43\x4e", [0x2479] = "\x43\x4f", [0x247a] = "\x43\x50",
  [0x247b] = "\x43\x51", [0x247c] = "\x43\x52", [0x247d] = "\x43\x53",
  [0x247e] = "\x43\x54", [0x247f] = "\x43\x55", [0x2480] = "\xa9\xa4",
  [0x2481] = "\xa9\xa5", [0x2482] = "\xa9\xa6", [0x2483] = "\xa9\xa7",
  [0x2484] = "\xa9\xa8", [0x2485] = "\xa9\xa9", [0x2486] = "\xa9\xaa",
  [0x2487] = "\xa9\xab", [0x2488] = "\xa9\xac", [0x2489] = "\xa9\xad",
  [0x248a] = "\xa9\xae", [0x248b] = "\xa9\xaf", [0x248c] = "\xa9\xb0",
  [0x248d] = "\xa9\xb1", [0x248e] = "\xa9\xb2", [0x248f] = "\xa9\xb3",
  [0x2490] = "\xa9\xb4", [0x2491] = "\xa9\xb5", [0x2492] = "\xa9\xb6",
  [0x2493] = "\xa9\xb7", [0x2494] = "\xa9\xb8", [0x2495] = "\xa9\xb9",
  [0x2496] = "\xa9\xba", [0x2497] = "\xa9\xbb", [0x2498] = "\xa9\xbc",
  [0x2499] = "\xa9\xbd", [0x249a] = "\xa9\xbe", [0x249b] = "\xa9\xbf",
  [0x249c] = "\xa9\xc0", [0x249d] = "\xa9\xc1", [0x249e] = "\xa9\xc2",
  [0x249f] = "\xa9\xc3", [0x24a0] = "\xa9\xc4", [0x24a1] = "\xa9\xc5",
  [0x24a2] = "\xa9\xc6", [0x24a3] = "\xa9\xc7", [0x24a4] = "\xa9\xc8",
  [0x24a5] = "\xa9\xc9", [0x24a6] = "\xa9\xca", [0x24a7] = "\xa9\xcb",
  [0x24a8] = "\xa9\xcc", [0x24a9] = "\xa9\xcd", [0x24aa] = "\xa9\xce",
  [0x24ab] = "\xa9\xcf", [0x24ac] = "\xa9\xd0", [0x24ad] = "\xa9\xd1",
  [0x24ae] = "\xa9\xd2", [0x24af] = "\xa9\xd3", [0x24b0] = "\xa9\xd4",
  [0x24b1] = "\xa9\xd5", [0x24b2] = "\xa9\xd6", [0x24b3] = "\xa9\xd7",
  [0x24b4] = "\xa9\xd8", [0x24b5] = "\xa9\xd9", [0x24b6] = "\xa9\xda",
  [0x24b7] = "\xa9\xdb", [0x24b8] = "\xa9\xdc", [0x24b9] = "\xa9\xdd",
  [0x24ba] = "\xa9\xde", [0x24bb] = "\xa9\xdf", [0x24bc] = "\xa9\xe0",
  [0x24bd] = "\xa9\xe1", [0x24be] = "\xa9\xe2", [0x24bf] = "\xa9\xe3",
  [0x24c0] = "\xa9\xe4", [0x24c1] = "\xa9\xe5", [0x24c2] = "\xa9\xe6",
  [0x24c3] = "\xa9\xe7", [0x24c4] = "\xa9\xe8", [0x24c5] = "\xa9\xe9",
  [0x24c6] = "\xa9\xea", [0x24c7] = "\xa9\xeb", [0x24c8] = "\xa9\xec",
  [0x24c9] = "\xa9\xed", [0x24ca] = "\xa9\xee", [0x24cb] = "\xa9\xef",
  [0x24cc] = "\x43\x56", [0x24cd] = "\x43\x57", [0x24ce] = "\x43\x58",
  [0x24cf] = "\x43\x59", [0x24d0] = "\xa8\x54", [0x24d1] = "\xa8\x55",
  [0x24d2] = "\xa8\x56", [0x24d3] = "\xa8\x57", [0x24d4] = "\xa8\x58",
  [0x24d5] = "\xa8\x59", [0x24d6] = "\xa8\x5a", [0x24d7] = "\xa8\x5b",
  [0x24d8] = "\xa8\x5c", [0x24d9] = "\xa8\x5d", [0x24da] = "\xa8\x5e",
  [0x24db] = "\xa8\x5f", [0x24dc] = "\xa8\x60", [0x24dd] = "\xa8\x61",
  [0x24de] = "\xa8\x62", [0x24df] = "\xa8\x63", [0x24e0] = "\xa8\x64",
  [0x24e1] = "\xa8\x65", [0x24e2] = "\xa8\x66", [0x24e3] = "\xa8\x67",
  [0x24e4] = "\xa8\x68", [0x24e5] = "\xa8\x69", [0x24e6] = "\xa8\x6a",
  [0x24e7] = "\xa8\x6b", [0x24e8] = "\xa8\x6c", [0x24e9] = "\xa8\x6d",
  [0x24ea] = "\xa8\x6e", [0x24eb] = "\xa8\x6f", [0x24ec] = "\xa8\x70",
  [0x24ed] = "\xa8\x71", [0x24ee] = "\xa8\x72", [0x24ef] = "\xa8\x73",
  [0x24f0] = "\xa8\x74", [0x24f1] = "\xa8\x75", [0x24f2] = "\xa8\x76",
  [0x24f3] = "\xa8\x77", [0x24f4] = "\x43\x5a", [0x24f5] = "\x43\x5b",
  [0x24f6] = "\x43\x5c", [0x24f7] = "\x43\x5d", [0x24f8] = "\x43\x5e",
  [0x24f9] = "\x43\x5f", [0x24fa] = "\x43\x60", [0x24fb] = "\x43\x61",
  [0x24fc] = "\x43\x62", [0x24fd] = "\x43\x63", [0x24fe] = "\x43\x64",
  [0x24ff] = "\x43\x65", [0x2500] = "\x43\x66", [0x2501] = "\xa8\x78",
  [0x2502] = "\xa8\x79", [0x2503] = "\xa8\x7a", [0x2504] = "\xa8\x7b",
  [0x2505] = "\xa8\x7c", [0x2506] = "\xa8\x7d", [0x2507] = "\xa8\x7e",
  [0x2508] = "\xa8\x80", [0x2509] = "\xa8\x81", [0x250a] = "\xa8\x82",
  [0x250b] = "\xa8\x83", [0x250c] = "\xa8\x84", [0x250d] = "\xa8\x85",
  [0x250e] = "\xa8\x86", [0x250f] = "\xa8\x87", [0x2510] = "\x43\x67",
  [0x2511] = "\x43\x68", [0x2512] = "\x43\x69", [0x2513] = "\xa8\x88",
  [0x2514] = "\xa8\x89", [0x2515] = "\xa8\x8a", [0x2516] = "\x43\x6a",
  [0x2517] = "\x43\x6b", [0x2518] = "\x43\x6c", [0x2519] = "\x43\x6d",
  [0x251a] = "\x43\x6e", [0x251b] = "\x43\x6f", [0x251c] = "\x43\x70",
  [0x251d] = "\x43\x71", [0x251e] = "\x43\x72", [0x251f] = "\x43\x73",
  [0x2520] = "\xa1\xf6", [0x2521] = "\xa1\xf5", [0x2522] = "\x43\x74",
  [0x2523] = "\x43\x75", [0x2524] = "\x43\x76", [0x2525] = "\x43\x77",
  [0x2526] = "\x43\x78", [0x2527] = "\x43\x79", [0x2528] = "\x43\x7a",
  [0x2529] = "\x43\x7b", [0x252a] = "\x43\x7c", [0x252b] = "\x43\x7d",
  [0x252c] = "\x43\x7e", [0x252d] = "\x43\x7f", [0x252e] = "\x43\x80",
  [0x252f] = "\x43\x81", [0x2530] = "\x43\x82", [0x2531] = "\x43\x83",
  [0x2532] = "\xa1\xf8", [0x2533] = "\xa1\xf7", [0x2534] = "\x43\x84",
  [0x2535] = "\x43\x85", [0x2536] = "\x43\x86", [0x2537] = "\x43\x87",
  [0x2538] = "\x43\x88", [0x2539] = "\x43\x89", [0x253a] = "\x43\x8a",
  [0x253b] = "\x43\x8b", [0x253c] = "\xa8\x8b", [0x253d] = "\xa8\x8c",
  [0x253e] = "\x43\x8c", [0x253f] = "\x43\x8d", [0x2540] = "\x43\x8e",
  [0x2541] = "\x43\x8f", [0x2542] = "\x43\x90", [0x2543] = "\x43\x91",
  [0x2544] = "\x43\x92", [0x2545] = "\x43\x93", [0x2546] = "\xa1\xf4",
  [0x2547] = "\xa1\xf3", [0x2548] = "\x43\x94", [0x2549] = "\x43\x95",
  [0x254a] = "\x43\x96", [0x254b] = "\xa1\xf0", [0x254c] = "\x43\x97",
  [0x254d] = "\x43\x98", [0x254e] = "\xa1\xf2", [0x254f] = "\xa1\xf1",
  [0x2550] = "\x43\x99", [0x2551] = "\x43\x9a", [0x2552] = "\x43\x9b",
  [0x2553] = "\x43\x9c", [0x2554] = "\x43\x9d", [0x2555] = "\x43\x9e",
  [0x2556] = "\x43\x9f", [0x2557] = "\x43\xa0", [0x2558] = "\x43\xa1",
  [0x2559] = "\x43\xa2", [0x255a] = "\x43\xa3", [0x255b] = "\x43\xa4",
  [0x255c] = "\x43\xa5", [0x255d] = "\x43\xa6", [0x255e] = "\x43\xa7",
  [0x255f] = "\x43\xa8", [0x2560] = "\x43\xa9", [0x2561] = "\x43\xaa",
  [0x2562] = "\xa8\x8d", [0x2563] = "\xa8\x8e", [0x2564] = "\xa8\x8f",
  [0x2565] = "\xa8\x90", [0x2566] = "\x43\xab", [0x2567] = "\x43\xac",
  [0x2568] = "\x43\xad", [0x2569] = "\x43\xae", [0x256a] = "\x43\xaf",
  [0x256b] = "\x43\xb0", [0x256c] = "\x43\xb1", [0x256d] = "\x43\xb2",
  [0x256e] = "\x43\xb3", [0x256f] = "\x43\xb4", [0x2570] = "\x43\xb5",
  [0x2571] = "\x43\xb6", [0x2572] = "\x43\xb7", [0x2573] = "\x43\xb8",
  [0x2574] = "\x43\xb9", [0x2575] = "\x43\xba", [0x2576] = "\x43\xbb",
  [0x2577] = "\x43\xbc", [0x2578] = "\x43\xbd", [0x2579] = "\x43\xbe",
  [0x257a] = "\x43\xbf", [0x257b] = "\x43\xc0", [0x257c] = "\x43\xc1",
  [0x257d] = "\x43\xc2", [0x257e] = "\x43\xc3", [0x257f] = "\x43\xc4",
  [0x2580] = "\x43\xc5", [0x2581] = "\x43\xc6", [0x2582] = "\x43\xc7",
  [0x2583] = "\x43\xc8", [0x2584] = "\x43\xc9", [0x2585] = "\xa1\xef",
  [0x2586] = "\xa1\xee", [0x2587] = "\x43\xca", [0x2588] = "\x43\xcb",
  [0x2589] = "\xa8\x91", [0x258a] = "\x43\xcc", [0x258b] = "\x43\xcd",
  [0x258c] = "\x43\xce", [0x258d] = "\x43\xcf", [0x258e] = "\x43\xd0",
  [0x258f] = "\x43\xd1", [0x2590] = "\x43\xd2", [0x2591] = "\x43\xd3",
  [0x2592] = "\x43\xd4", [0x2593] = "\x43\xd5", [0x2594] = "\x43\xd6",
  [0x2595] = "\x43\xd7", [0x2596] = "\x43\xd8", [0x2597] = "\x43\xd9",
  [0x2598] = "\x43\xda", [0x2599] = "\x43\xdb", [0x259a] = "\x43\xdc",
  [0x259b] = "\x43\xdd", [0x259c] = "\x43\xde", [0x259d] = "\x43\xdf",
  [0x259e] = "\x43\xe0", [0x259f] = "\x43\xe1", [0x25a0] = "\x43\xe2",
  [0x25a1] = "\x43\xe3", [0x25a2] = "\x43\xe4", [0x25a3] = "\x43\xe5",
  [0x25a4] = "\x43\xe6", [0x25a5] = "\x43\xe7", [0x25a6] = "\x43\xe8",
  [0x25a7] = "\x43\xe9", [0x25a8] = "\x43\xea", [0x25a9] = "\x43\xeb",
  [0x25aa] = "\x43\xec", [0x25ab] = "\x43\xed", [0x25ac] = "\x43\xee",
  [0x25ad] = "\x43\xef", [0x25ae] = "\x43\xf0", [0x25af] = "\x43\xf1",
  [0x25b0] = "\x43\xf2", [0x25b1] = "\x43\xf3", [0x25b2] = "\x43\xf4",
  [0x25b3] = "\x43\xf5", [0x25b4] = "\x43\xf6", [0x25b5] = "\x43\xf7",
  [0x25b6] = "\x43\xf8", [0x25b7] = "\x43\xf9", [0x25b8] = "\x43\xfa",
  [0x25b9] = "\x43\xfb", [0x25ba] = "\x43\xfc", [0x25bb] = "\x43\xfd",
  [0x25bc] = "\x43\xfe", [0x25bd] = "\x43\xff", [0x25be] = "\x44\x00",
  [0x25bf] = "\x44\x01", [0x25c0] = "\xa1\xe2", [0x25c1] = "\x44\x02",
  [0x25c2] = "\xa1\xe1", [0x25c3] = "\x44\x03", [0x25c4] = "\x44\x04",
  [0x25c5] = "\x44\x05", [0x25c6] = "\x44\x06", [0x25c7] = "\x44\x07",
  [0x25c8] = "\x44\x08", [0x25c9] = "\x44\x09", [0x25ca] = "\x44\x0a",
  [0x25cb] = "\x44\x0b", [0x25cc] = "\x44\x0c", [0x25cd] = "\x44\x0d",
  [0x25ce] = "\x44\x0e", [0x25cf] = "\x44\x0f", [0x25d0] = "\x44\x10",
  [0x25d1] = "\x44\x11", [0x25d2] = "\x44\x12", [0x25d3] = "\x44\x13",
  [0x25d4] = "\x44\x14", [0x25d5] = "\x44\x15", [0x25d6] = "\x44\x16",
  [0x25d7] = "\x44\x17", [0x25d8] = "\x44\x18", [0x25d9] = "\x44\x19",
  [0x25da] = "\x44\x1a", [0x25db] = "\x44\x1b", [0x25dc] = "\x44\x1c",
  [0x25dd] = "\x44\x1d", [0x25de] = "\x44\x1e", [0x25df] = "\x44\x1f",
  [0x25e0] = "\x44\x20", [0x25e1] = "\x44\x21", [0x25e2] = "\x44\x22",
  [0x25e3] = "\x44\x23", [0x25e4] = "\x44\x24", [0x25e5] = "\x44\x25",
  [0x25e6] = "\x44\x26", [0x25e7] = "\x44\x27", [0x25e8] = "\x44\x28",
  [0x25e9] = "\x44\x29", [0x25ea] = "\x44\x2a", [0x25eb] = "\x44\x2b",
  [0x25ec] = "\x44\x2c", [0x25ed] = "\x44\x2d", [0x25ee] = "\x44\x2e",
  [0x25ef] = "\x44\x2f", [0x25f0] = "\x44\x30", [0x25f1] = "\x44\x31",
  [0x25f2] = "\x44\x32", [0x25f3] = "\x44\x33", [0x25f4] = "\x44\x34",
  [0x25f5] = "\x44\x35", [0x25f6] = "\x44\x36", [0x25f7] = "\x44\x37",
  [0x25f8] = "\x44\x38", [0x25f9] = "\x44\x39", [0x25fa] = "\x44\x3a",
  [0x25fb] = "\x44\x3b", [0x25fc] = "\x44\x3c", [0x25fd] = "\x44\x3d",
  [0x25fe] = "\x44\x3e", [0x25ff] = "\x44\x3f", [0x2600] = "\x44\x40",
  [0x2601] = "\x44\x41", [0x2602] = "\x44\x42", [0x2603] = "\x44\x43",
  [0x2604] = "\x44\x44", [0x2605] = "\x44\x45", [0x2606] = "\x44\x46",
  [0x2607] = "\x44\x47", [0x2608] = "\x44\x48", [0x2609] = "\x44\x49",
  [0x260a] = "\x44\x4a", [0x260b] = "\x44\x4b", [0x260c] = "\x44\x4c",
  [0x260d] = "\x44\x4d", [0x260e] = "\x44\x4e", [0x260f] = "\x44\x4f",
  [0x2610] = "\x44\x50", [0x2611] = "\x44\x51", [0x2612] = "\x44\x52",
  [0x2613] = "\x44\x53", [0x2614] = "\x44\x54", [0x2615] = "\x44\x55",
  [0x2616] = "\x44\x56", [0x2617] = "\x44\x57", [0x2618] = "\x44\x58",
  [0x2619] = "\x44\x59", [0x261a] = "\x44\x5a", [0x261b] = "\x44\x5b",
  [0x261c] = "\x44\x5c", [0x261d] = "\x44\x5d", [0x261e] = "\x44\x5e",
  [0x261f] = "\x44\x5f", [0x2620] = "\x44\x60", [0x2621] = "\x44\x61",
  [0x2622] = "\x44\x62", [0x2623] = "\x44\x63", [0x2624] = "\x44\x64",
  [0x2625] = "\x44\x65", [0x2626] = "\x44\x66", [0x2627] = "\x44\x67",
  [0x2628] = "\x44\x68", [0x2629] = "\x44\x69", [0x262a] = "\x44\x6a",
  [0x262b] = "\x44\x6b", [0x262c] = "\x44\x6c", [0x262d] = "\x44\x6d",
  [0x262e] = "\x44\x6e", [0x262f] = "\x44\x6f", [0x2630] = "\x44\x70",
  [0x2631] = "\x44\x71", [0x2632] = "\x44\x72", [0x2633] = "\x44\x73",
  [0x2634] = "\x44\x74", [0x2635] = "\x44\x75", [0x2636] = "\x44\x76",
  [0x2637] = "\x44\x77", [0x2638] = "\x44\x78", [0x2639] = "\x44\x79",
  [0x263a] = "\x44\x7a", [0x263b] = "\x44\x7b", [0x263c] = "\x44\x7c",
  [0x263d] = "\x44\x7d", [0x263e] = "\x44\x7e", [0x263f] = "\x44\x7f",
  [0x2640] = "\x44\x80", [0x2641] = "\x44\x81", [0x2642] = "\x44\x82",
  [0x2643] = "\x44\x83", [0x2644] = "\x44\x84", [0x2645] = "\x44\x85",
  [0x2646] = "\x44\x86", [0x2647] = "\x44\x87", [0x2648] = "\x44\x88",
  [0x2649] = "\x44\x89", [0x264a] = "\x44\x8a", [0x264b] = "\x44\x8b",
  [0x264c] = "\x44\x8c", [0x264d] = "\x44\x8d", [0x264e] = "\x44\x8e",
  [0x264f] = "\x44\x8f", [0x2650] = "\x44\x90", [0x2651] = "\x44\x91",
  [0x2652] = "\x44\x92", [0x2653] = "\x44\x93", [0x2654] = "\x44\x94",
  [0x2655] = "\x44\x95", [0x2656] = "\x44\x96", [0x2657] = "\x44\x97",
  [0x2658] = "\x44\x98", [0x2659] = "\x44\x99", [0x265a] = "\x44\x9a",
  [0x265b] = "\x44\x9b", [0x265c] = "\x44\x9c", [0x265d] = "\x44\x9d",
  [0x265e] = "\x44\x9e", [0x265f] = "\x44\x9f", [0x2660] = "\x44\xa0",
  [0x2661] = "\x44\xa1", [0x2662] = "\x44\xa2", [0x2663] = "\x44\xa3",
  [0x2664] = "\x44\xa4", [0x2665] = "\x44\xa5", [0x2666] = "\x44\xa6",
  [0x2667] = "\x44\xa7", [0x2668] = "\x44\xa8", [0x2669] = "\x44\xa9",
  [0x266a] = "\x44\xaa", [0x266b] = "\x44\xab", [0x266c] = "\x44\xac",
  [0x266d] = "\x44\xad", [0x266e] = "\x44\xae", [0x266f] = "\x44\xaf",
  [0x2670] = "\x44\xb0", [0x2671] = "\x44\xb1", [0x2672] = "\x44\xb2",
  [0x2673] = "\x44\xb3", [0x2674] = "\x44\xb4", [0x2675] = "\x44\xb5",
  [0x2676] = "\x44\xb6", [0x2677] = "\x44\xb7", [0x2678] = "\x44\xb8",
  [0x2679] = "\x44\xb9", [0x267a] = "\x44\xba", [0x267b] = "\x44\xbb",
  [0x267c] = "\x44\xbc", [0x267d] = "\x44\xbd", [0x267e] = "\x44\xbe",
  [0x267f] = "\x44\xbf", [0x2680] = "\x44\xc0", [0x2681] = "\x44\xc1",
  [0x2682] = "\x44\xc2", [0x2683] = "\x44\xc3", [0x2684] = "\x44\xc4",
  [0x2685] = "\x44\xc5", [0x2686] = "\x44\xc6", [0x2687] = "\x44\xc7",
  [0x2688] = "\x44\xc8", [0x2689] = "\x44\xc9", [0x268a] = "\x44\xca",
  [0x268b] = "\x44\xcb", [0x268c] = "\x44\xcc", [0x268d] = "\x44\xcd",
  [0x268e] = "\x44\xce", [0x268f] = "\x44\xcf", [0x2690] = "\x44\xd0",
  [0x2691] = "\x44\xd1", [0x2692] = "\x44\xd2", [0x2693] = "\x44\xd3",
  [0x2694] = "\x44\xd4", [0x2695] = "\x44\xd5", [0x2696] = "\x44\xd6",
  [0x2697] = "\x44\xd7", [0x2698] = "\x44\xd8", [0x2699] = "\x44\xd9",
  [0x269a] = "\x44\xda", [0x269b] = "\x44\xdb", [0x269c] = "\x44\xdc",
  [0x269d] = "\x44\xdd", [0x269e] = "\x44\xde", [0x269f] = "\x44\xdf",
  [0x26a0] = "\x44\xe0", [0x26a1] = "\x44\xe1", [0x26a2] = "\x44\xe2",
  [0x26a3] = "\x44\xe3", [0x26a4] = "\x44\xe4", [0x26a5] = "\x44\xe5",
  [0x26a6] = "\x44\xe6", [0x26a7] = "\x44\xe7", [0x26a8] = "\x44\xe8",
  [0x26a9] = "\x44\xe9", [0x26aa] = "\x44\xea", [0x26ab] = "\x44\xeb",
  [0x26ac] = "\x44\xec", [0x26ad] = "\x44\xed", [0x26ae] = "\x44\xee",
  [0x26af] = "\x44\xef", [0x26b0] = "\x44\xf0", [0x26b1] = "\x44\xf1",
  [0x26b2] = "\x44\xf2", [0x26b3] = "\x44\xf3", [0x26b4] = "\x44\xf4",
  [0x26b5] = "\x44\xf5", [0x26b6] = "\x44\xf6", [0x26b7] = "\x44\xf7",
  [0x26b8] = "\x44\xf8", [0x26b9] = "\x44\xf9", [0x26ba] = "\x44\xfa",
  [0x26bb] = "\x44\xfb", [0x26bc] = "\x44\xfc", [0x26bd] = "\x44\xfd",
  [0x26be] = "\x44\xfe", [0x26bf] = "\x44\xff", [0x26c0] = "\x45\x00",
  [0x26c1] = "\x45\x01", [0x26c2] = "\x45\x02", [0x26c3] = "\x45\x03",
  [0x26c4] = "\x45\x04", [0x26c5] = "\x45\x05", [0x26c6] = "\x45\x06",
  [0x26c7] = "\x45\x07", [0x26c8] = "\x45\x08", [0x26c9] = "\x45\x09",
  [0x26ca] = "\x45\x0a", [0x26cb] = "\x45\x0b", [0x26cc] = "\x45\x0c",
  [0x26cd] = "\x45\x0d", [0x26ce] = "\x45\x0e", [0x26cf] = "\x45\x0f",
  [0x26d0] = "\x45\x10", [0x26d1] = "\x45\x11", [0x26d2] = "\x45\x12",
  [0x26d3] = "\x45\x13", [0x26d4] = "\x45\x14", [0x26d5] = "\x45\x15",
  [0x26d6] = "\x45\x16", [0x26d7] = "\x45\x17", [0x26d8] = "\x45\x18",
  [0x26d9] = "\x45\x19", [0x26da] = "\x45\x1a", [0x26db] = "\x45\x1b",
  [0x26dc] = "\x45\x1c", [0x26dd] = "\x45\x1d", [0x26de] = "\x45\x1e",
  [0x26df] = "\x45\x1f", [0x26e0] = "\x45\x20", [0x26e1] = "\x45\x21",
  [0x26e2] = "\x45\x22", [0x26e3] = "\x45\x23", [0x26e4] = "\x45\x24",
  [0x26e5] = "\x45\x25", [0x26e6] = "\x45\x26", [0x26e7] = "\x45\x27",
  [0x26e8] = "\x45\x28", [0x26e9] = "\x45\x29", [0x26ea] = "\x45\x2a",
  [0x26eb] = "\x45\x2b", [0x26ec] = "\x45\x2c", [0x26ed] = "\x45\x2d",
  [0x26ee] = "\x45\x2e", [0x26ef] = "\x45\x2f", [0x26f0] = "\x45\x30",
  [0x26f1] = "\x45\x31", [0x26f2] = "\x45\x32", [0x26f3] = "\x45\x33",
  [0x26f4] = "\x45\x34", [0x26f5] = "\x45\x35", [0x26f6] = "\x45\x36",
  [0x26f7] = "\x45\x37", [0x26f8] = "\x45\x38", [0x26f9] = "\x45\x39",
  [0x26fa] = "\x45\x3a", [0x26fb] = "\x45\x3b", [0x26fc] = "\x45\x3c",
  [0x26fd] = "\x45\x3d", [0x26fe] = "\x45\x3e", [0x26ff] = "\x45\x3f",
  [0x2700] = "\x45\x40", [0x2701] = "\x45\x41", [0x2702] = "\x45\x42",
  [0x2703] = "\x45\x43", [0x2704] = "\x45\x44", [0x2705] = "\x45\x45",
  [0x2706] = "\x45\x46", [0x2707] = "\x45\x47", [0x2708] = "\x45\x48",
  [0x2709] = "\x45\x49", [0x270a] = "\x45\x4a", [0x270b] = "\x45\x4b",
  [0x270c] = "\x45\x4c", [0x270d] = "\x45\x4d", [0x270e] = "\x45\x4e",
  [0x270f] = "\x45\x4f", [0x2710] = "\x45\x50", [0x2711] = "\x45\x51",
  [0x2712] = "\x45\x52", [0x2713] = "\x45\x53", [0x2714] = "\x45\x54",
  [0x2715] = "\x45\x55", [0x2716] = "\x45\x56", [0x2717] = "\x45\x57",
  [0x2718] = "\x45\x58", [0x2719] = "\x45\x59", [0x271a] = "\x45\x5a",
  [0x271b] = "\x45\x5b", [0x271c] = "\x45\x5c", [0x271d] = "\x45\x5d",
  [0x271e] = "\x45\x5e", [0x271f] = "\x45\x5f", [0x2720] = "\x45\x60",
  [0x2721] = "\x45\x61", [0x2722] = "\x45\x62", [0x2723] = "\x45\x63",
  [0x2724] = "\x45\x64", [0x2725] = "\x45\x65", [0x2726] = "\x45\x66",
  [0x2727] = "\x45\x67", [0x2728] = "\x45\x68", [0x2729] = "\x45\x69",
  [0x272a] = "\x45\x6a", [0x272b] = "\x45\x6b", [0x272c] = "\x45\x6c",
  [0x272d] = "\x45\x6d", [0x272e] = "\x45\x6e", [0x272f] = "\x45\x6f",
  [0x2730] = "\x45\x70", [0x2731] = "\x45\x71", [0x2732] = "\x45\x72",
  [0x2733] = "\x45\x73", [0x2734] = "\x45\x74", [0x2735] = "\x45\x75",
  [0x2736] = "\x45\x76", [0x2737] = "\x45\x77", [0x2738] = "\x45\x78",
  [0x2739] = "\x45\x79", [0x273a] = "\x45\x7a", [0x273b] = "\x45\x7b",
  [0x273c] = "\x45\x7c", [0x273d] = "\x45\x7d", [0x273e] = "\x45\x7e",
  [0x273f] = "\x45\x7f", [0x2740] = "\x45\x80", [0x2741] = "\x45\x81",
  [0x2742] = "\x45\x82", [0x2743] = "\x45\x83", [0x2744] = "\x45\x84",
  [0x2745] = "\x45\x85", [0x2746] = "\x45\x86", [0x2747] = "\x45\x87",
  [0x2748] = "\x45\x88", [0x2749] = "\x45\x89", [0x274a] = "\x45\x8a",
  [0x274b] = "\x45\x8b", [0x274c] = "\x45\x8c", [0x274d] = "\x45\x8d",
  [0x274e] = "\x45\x8e", [0x274f] = "\x45\x8f", [0x2750] = "\x45\x90",
  [0x2751] = "\x45\x91", [0x2752] = "\x45\x92", [0x2753] = "\x45\x93",
  [0x2754] = "\x45\x94", [0x2755] = "\x45\x95", [0x2756] = "\x45\x96",
  [0x2757] = "\x45\x97", [0x2758] = "\x45\x98", [0x2759] = "\x45\x99",
  [0x275a] = "\x45\x9a", [0x275b] = "\x45\x9b", [0x275c] = "\x45\x9c",
  [0x275d] = "\x45\x9d", [0x275e] = "\x45\x9e", [0x275f] = "\x45\x9f",
  [0x2760] = "\x45\xa0", [0x2761] = "\x45\xa1", [0x2762] = "\x45\xa2",
  [0x2763] = "\x45\xa3", [0x2764] = "\x45\xa4", [0x2765] = "\x45\xa5",
  [0x2766] = "\x45\xa6", [0x2767] = "\x45\xa7", [0x2768] = "\x45\xa8",
  [0x2769] = "\x45\xa9", [0x276a] = "\x45\xaa", [0x276b] = "\x45\xab",
  [0x276c] = "\x45\xac", [0x276d] = "\x45\xad", [0x276e] = "\x45\xae",
  [0x276f] = "\x45\xaf", [0x2770] = "\x45\xb0", [0x2771] = "\x45\xb1",
  [0x2772] = "\x45\xb2", [0x2773] = "\x45\xb3", [0x2774] = "\x45\xb4",
  [0x2775] = "\x45\xb5", [0x2776] = "\x45\xb6", [0x2777] = "\x45\xb7",
  [0x2778] = "\x45\xb8", [0x2779] = "\x45\xb9", [0x277a] = "\x45\xba",
  [0x277b] = "\x45\xbb", [0x277c] = "\x45\xbc", [0x277d] = "\x45\xbd",
  [0x277e] = "\x45\xbe", [0x277f] = "\x45\xbf", [0x2780] = "\x45\xc0",
  [0x2781] = "\x45\xc1", [0x2782] = "\x45\xc2", [0x2783] = "\x45\xc3",
  [0x2784] = "\x45\xc4", [0x2785] = "\x45\xc5", [0x2786] = "\x45\xc6",
  [0x2787] = "\x45\xc7", [0x2788] = "\x45\xc8", [0x2789] = "\x45\xc9",
  [0x278a] = "\x45\xca", [0x278b] = "\x45\xcb", [0x278c] = "\x45\xcc",
  [0x278d] = "\x45\xcd", [0x278e] = "\x45\xce", [0x278f] = "\x45\xcf",
  [0x2790] = "\x45\xd0", [0x2791] = "\x45\xd1", [0x2792] = "\x45\xd2",
  [0x2793] = "\x45\xd3", [0x2794] = "\x45\xd4", [0x2795] = "\x45\xd5",
  [0x2796] = "\x45\xd6", [0x2797] = "\x45\xd7", [0x2798] = "\x45\xd8",
  [0x2799] = "\x45\xd9", [0x279a] = "\x45\xda", [0x279b] = "\x45\xdb",
  [0x279c] = "\x45\xdc", [0x279d] = "\x45\xdd", [0x279e] = "\x45\xde",
  [0x279f] = "\x45\xdf", [0x27a0] = "\x45\xe0", [0x27a1] = "\x45\xe1",
  [0x27a2] = "\x45\xe2", [0x27a3] = "\x45\xe3", [0x27a4] = "\x45\xe4",
  [0x27a5] = "\x45\xe5", [0x27a6] = "\x45\xe6", [0x27a7] = "\x45\xe7",
  [0x27a8] = "\x45\xe8", [0x27a9] = "\x45\xe9", [0x27aa] = "\x45\xea",
  [0x27ab] = "\x45\xeb", [0x27ac] = "\x45\xec", [0x27ad] = "\x45\xed",
  [0x27ae] = "\x45\xee", [0x27af] = "\x45\xef", [0x27b0] = "\x45\xf0",
  [0x27b1] = "\x45\xf1", [0x27b2] = "\x45\xf2", [0x27b3] = "\x45\xf3",
  [0x27b4] = "\x45\xf4", [0x27b5] = "\x45\xf5", [0x27b6] = "\x45\xf6",
  [0x27b7] = "\x45\xf7", [0x27b8] = "\x45\xf8", [0x27b9] = "\x45\xf9",
  [0x27ba] = "\x45\xfa", [0x27bb] = "\x45\xfb", [0x27bc] = "\x45\xfc",
  [0x27bd] = "\x45\xfd", [0x27be] = "\x45\xfe", [0x27bf] = "\x45\xff",
  [0x27c0] = "\x46\x00", [0x27c1] = "\x46\x01", [0x27c2] = "\x46\x02",
  [0x27c3] = "\x46\x03", [0x27c4] = "\x46\x04", [0x27c5] = "\x46\x05",
  [0x27c6] = "\x46\x06", [0x27c7] = "\x46\x07", [0x27c8] = "\x46\x08",
  [0x27c9] = "\x46\x09", [0x27ca] = "\x46\x0a", [0x27cb] = "\x46\x0b",
  [0x27cc] = "\x46\x0c", [0x27cd] = "\x46\x0d", [0x27ce] = "\x46\x0e",
  [0x27cf] = "\x46\x0f", [0x27d0] = "\x46\x10", [0x27d1] = "\x46\x11",
  [0x27d2] = "\x46\x12", [0x27d3] = "\x46\x13", [0x27d4] = "\x46\x14",
  [0x27d5] = "\x46\x15", [0x27d6] = "\x46\x16", [0x27d7] = "\x46\x17",
  [0x27d8] = "\x46\x18", [0x27d9] = "\x46\x19", [0x27da] = "\x46\x1a",
  [0x27db] = "\x46\x1b", [0x27dc] = "\x46\x1c", [0x27dd] = "\x46\x1d",
  [0x27de] = "\x46\x1e", [0x27df] = "\x46\x1f", [0x27e0] = "\x46\x20",
  [0x27e1] = "\x46\x21", [0x27e2] = "\x46\x22", [0x27e3] = "\x46\x23",
  [0x27e4] = "\x46\x24", [0x27e5] = "\x46\x25", [0x27e6] = "\x46\x26",
  [0x27e7] = "\x46\x27", [0x27e8] = "\x46\x28", [0x27e9] = "\x46\x29",
  [0x27ea] = "\x46\x2a", [0x27eb] = "\x46\x2b", [0x27ec] = "\x46\x2c",
  [0x27ed] = "\x46\x2d", [0x27ee] = "\x46\x2e", [0x27ef] = "\x46\x2f",
  [0x27f0] = "\x46\x30", [0x27f1] = "\x46\x31", [0x27f2] = "\x46\x32",
  [0x27f3] = "\x46\x33", [0x27f4] = "\x46\x34", [0x27f5] = "\x46\x35",
  [0x27f6] = "\x46\x36", [0x27f7] = "\x46\x37", [0x27f8] = "\x46\x38",
  [0x27f9] = "\x46\x39", [0x27fa] = "\x46\x3a", [0x27fb] = "\x46\x3b",
  [0x27fc] = "\x46\x3c", [0x27fd] = "\x46\x3d", [0x27fe] = "\x46\x3e",
  [0x27ff] = "\x46\x3f", [0x2800] = "\x46\x40", [0x2801] = "\x46\x41",
  [0x2802] = "\x46\x42", [0x2803] = "\x46\x43", [0x2804] = "\x46\x44",
  [0x2805] = "\x46\x45", [0x2806] = "\x46\x46", [0x2807] = "\x46\x47",
  [0x2808] = "\x46\x48", [0x2809] = "\x46\x49", [0x280a] = "\x46\x4a",
  [0x280b] = "\x46\x4b", [0x280c] = "\x46\x4c", [0x280d] = "\x46\x4d",
  [0x280e] = "\x46\x4e", [0x280f] = "\x46\x4f", [0x2810] = "\x46\x50",
  [0x2811] = "\x46\x51", [0x2812] = "\x46\x52", [0x2813] = "\x46\x53",
  [0x2814] = "\x46\x54", [0x2815] = "\x46\x55", [0x2816] = "\x46\x56",
  [0x2817] = "\x46\x57", [0x2818] = "\x46\x58", [0x2819] = "\x46\x59",
  [0x281a] = "\x46\x5a", [0x281b] = "\x46\x5b", [0x281c] = "\x46\x5c",
  [0x281d] = "\x46\x5d", [0x281e] = "\x46\x5e", [0x281f] = "\x46\x5f",
  [0x2820] = "\x46\x60", [0x2821] = "\x46\x61", [0x2822] = "\x46\x62",
  [0x2823] = "\x46\x63", [0x2824] = "\x46\x64", [0x2825] = "\x46\x65",
  [0x2826] = "\x46\x66", [0x2827] = "\x46\x67", [0x2828] = "\x46\x68",
  [0x2829] = "\x46\x69", [0x282a] = "\x46\x6a", [0x282b] = "\x46\x6b",
  [0x282c] = "\x46\x6c", [0x282d] = "\x46\x6d", [0x282e] = "\x46\x6e",
  [0x282f] = "\x46\x6f", [0x2830] = "\x46\x70", [0x2831] = "\x46\x71",
  [0x2832] = "\x46\x72", [0x2833] = "\x46\x73", [0x2834] = "\x46\x74",
  [0x2835] = "\x46\x75", [0x2836] = "\x46\x76", [0x2837] = "\x46\x77",
  [0x2838] = "\x46\x78", [0x2839] = "\x46\x79", [0x283a] = "\x46\x7a",
  [0x283b] = "\x46\x7b", [0x283c] = "\x46\x7c", [0x283d] = "\x46\x7d",
  [0x283e] = "\x46\x7e", [0x283f] = "\x46\x7f", [0x2840] = "\x46\x80",
  [0x2841] = "\x46\x81", [0x2842] = "\x46\x82", [0x2843] = "\x46\x83",
  [0x2844] = "\x46\x84", [0x2845] = "\x46\x85", [0x2846] = "\x46\x86",
  [0x2847] = "\x46\x87", [0x2848] = "\x46\x88", [0x2849] = "\x46\x89",
  [0x284a] = "\x46\x8a", [0x284b] = "\x46\x8b", [0x284c] = "\x46\x8c",
  [0x284d] = "\x46\x8d", [0x284e] = "\x46\x8e", [0x284f] = "\x46\x8f",
  [0x2850] = "\x46\x90", [0x2851] = "\x46\x91", [0x2852] = "\x46\x92",
  [0x2853] = "\x46\x93", [0x2854] = "\x46\x94", [0x2855] = "\x46\x95",
  [0x2856] = "\x46\x96", [0x2857] = "\x46\x97", [0x2858] = "\x46\x98",
  [0x2859] = "\x46\x99", [0x285a] = "\x46\x9a", [0x285b] = "\x46\x9b",
  [0x285c] = "\x46\x9c", [0x285d] = "\x46\x9d", [0x285e] = "\x46\x9e",
  [0x285f] = "\x46\x9f", [0x2860] = "\x46\xa0", [0x2861] = "\x46\xa1",
  [0x2862] = "\x46\xa2", [0x2863] = "\x46\xa3", [0x2864] = "\x46\xa4",
  [0x2865] = "\x46\xa5", [0x2866] = "\x46\xa6", [0x2867] = "\x46\xa7",
  [0x2868] = "\x46\xa8", [0x2869] = "\x46\xa9", [0x286a] = "\x46\xaa",
  [0x286b] = "\x46\xab", [0x286c] = "\x46\xac", [0x286d] = "\x46\xad",
  [0x286e] = "\x46\xae", [0x286f] = "\x46\xaf", [0x2870] = "\x46\xb0",
  [0x2871] = "\x46\xb1", [0x2872] = "\x46\xb2", [0x2873] = "\x46\xb3",
  [0x2874] = "\x46\xb4", [0x2875] = "\x46\xb5", [0x2876] = "\x46\xb6",
  [0x2877] = "\x46\xb7", [0x2878] = "\x46\xb8", [0x2879] = "\x46\xb9",
  [0x287a] = "\x46\xba", [0x287b] = "\x46\xbb", [0x287c] = "\x46\xbc",
  [0x287d] = "\x46\xbd", [0x287e] = "\x46\xbe", [0x287f] = "\x46\xbf",
  [0x2880] = "\x46\xc0", [0x2881] = "\x46\xc1", [0x2882] = "\x46\xc2",
  [0x2883] = "\x46\xc3", [0x2884] = "\x46\xc4", [0x2885] = "\x46\xc5",
  [0x2886] = "\x46\xc6", [0x2887] = "\x46\xc7", [0x2888] = "\x46\xc8",
  [0x2889] = "\x46\xc9", [0x288a] = "\x46\xca", [0x288b] = "\x46\xcb",
  [0x288c] = "\x46\xcc", [0x288d] = "\x46\xcd", [0x288e] = "\x46\xce",
  [0x288f] = "\x46\xcf", [0x2890] = "\x46\xd0", [0x2891] = "\x46\xd1",
  [0x2892] = "\x46\xd2", [0x2893] = "\x46\xd3", [0x2894] = "\x46\xd4",
  [0x2895] = "\x46\xd5", [0x2896] = "\x46\xd6", [0x2897] = "\x46\xd7",
  [0x2898] = "\x46\xd8", [0x2899] = "\x46\xd9", [0x289a] = "\x46\xda",
  [0x289b] = "\x46\xdb", [0x289c] = "\x46\xdc", [0x289d] = "\x46\xdd",
  [0x289e] = "\x46\xde", [0x289f] = "\x46\xdf", [0x28a0] = "\x46\xe0",
  [0x28a1] = "\x46\xe1", [0x28a2] = "\x46\xe2", [0x28a3] = "\x46\xe3",
  [0x28a4] = "\x46\xe4", [0x28a5] = "\x46\xe5", [0x28a6] = "\x46\xe6",
  [0x28a7] = "\x46\xe7", [0x28a8] = "\x46\xe8", [0x28a9] = "\x46\xe9",
  [0x28aa] = "\x46\xea", [0x28ab] = "\x46\xeb", [0x28ac] = "\x46\xec",
  [0x28ad] = "\x46\xed", [0x28ae] = "\x46\xee", [0x28af] = "\x46\xef",
  [0x28b0] = "\x46\xf0", [0x28b1] = "\x46\xf1", [0x28b2] = "\x46\xf2",
  [0x28b3] = "\x46\xf3", [0x28b4] = "\x46\xf4", [0x28b5] = "\x46\xf5",
  [0x28b6] = "\x46\xf6", [0x28b7] = "\x46\xf7", [0x28b8] = "\x46\xf8",
  [0x28b9] = "\x46\xf9", [0x28ba] = "\x46\xfa", [0x28bb] = "\x46\xfb",
  [0x28bc] = "\x46\xfc", [0x28bd] = "\x46\xfd", [0x28be] = "\x46\xfe",
  [0x28bf] = "\x46\xff", [0x28c0] = "\x47\x00", [0x28c1] = "\x47\x01",
  [0x28c2] = "\x47\x02", [0x28c3] = "\x47\x03", [0x28c4] = "\x47\x04",
  [0x28c5] = "\x47\x05", [0x28c6] = "\x47\x06", [0x28c7] = "\x47\x07",
  [0x28c8] = "\x47\x08", [0x28c9] = "\x47\x09", [0x28ca] = "\x47\x0a",
  [0x28cb] = "\x47\x0b", [0x28cc] = "\x47\x0c", [0x28cd] = "\x47\x0d",
  [0x28ce] = "\x47\x0e", [0x28cf] = "\x47\x0f", [0x28d0] = "\x47\x10",
  [0x28d1] = "\x47\x11", [0x28d2] = "\x47\x12", [0x28d3] = "\x47\x13",
  [0x28d4] = "\x47\x14", [0x28d5] = "\x47\x15", [0x28d6] = "\x47\x16",
  [0x28d7] = "\x47\x17", [0x28d8] = "\x47\x18", [0x28d9] = "\x47\x19",
  [0x28da] = "\x47\x1a", [0x28db] = "\x47\x1b", [0x28dc] = "\x47\x1c",
  [0x28dd] = "\x47\x1d", [0x28de] = "\x47\x1e", [0x28df] = "\x47\x1f",
  [0x28e0] = "\x47\x20", [0x28e1] = "\x47\x21", [0x28e2] = "\x47\x22",
  [0x28e3] = "\x47\x23", [0x28e4] = "\x47\x24", [0x28e5] = "\x47\x25",
  [0x28e6] = "\x47\x26", [0x28e7] = "\x47\x27", [0x28e8] = "\x47\x28",
  [0x28e9] = "\x47\x29", [0x28ea] = "\x47\x2a", [0x28eb] = "\x47\x2b",
  [0x28ec] = "\x47\x2c", [0x28ed] = "\x47\x2d", [0x28ee] = "\x47\x2e",
  [0x28ef] = "\x47\x2f", [0x28f0] = "\x47\x30", [0x28f1] = "\x47\x31",
  [0x28f2] = "\x47\x32", [0x28f3] = "\x47\x33", [0x28f4] = "\x47\x34",
  [0x28f5] = "\x47\x35", [0x28f6] = "\x47\x36", [0x28f7] = "\x47\x37",
  [0x28f8] = "\x47\x38", [0x28f9] = "\x47\x39", [0x28fa] = "\x47\x3a",
  [0x28fb] = "\x47\x3b", [0x28fc] = "\x47\x3c", [0x28fd] = "\x47\x3d",
  [0x28fe] = "\x47\x3e", [0x28ff] = "\x47\x3f", [0x2900] = "\x47\x40",
  [0x2901] = "\x47\x41", [0x2902] = "\x47\x42", [0x2903] = "\x47\x43",
  [0x2904] = "\x47\x44", [0x2905] = "\x47\x45", [0x2906] = "\x47\x46",
  [0x2907] = "\x47\x47", [0x2908] = "\x47\x48", [0x2909] = "\x47\x49",
  [0x290a] = "\x47\x4a", [0x290b] = "\x47\x4b", [0x290c] = "\x47\x4c",
  [0x290d] = "\x47\x4d", [0x290e] = "\x47\x4e", [0x290f] = "\x47\x4f",
  [0x2910] = "\x47\x50", [0x2911] = "\x47\x51", [0x2912] = "\x47\x52",
  [0x2913] = "\x47\x53", [0x2914] = "\x47\x54", [0x2915] = "\x47\x55",
  [0x2916] = "\x47\x56", [0x2917] = "\x47\x57", [0x2918] = "\x47\x58",
  [0x2919] = "\x47\x59", [0x291a] = "\x47\x5a", [0x291b] = "\x47\x5b",
  [0x291c] = "\x47\x5c", [0x291d] = "\x47\x5d", [0x291e] = "\x47\x5e",
  [0x291f] = "\x47\x5f", [0x2920] = "\x47\x60", [0x2921] = "\x47\x61",
  [0x2922] = "\x47\x62", [0x2923] = "\x47\x63", [0x2924] = "\x47\x64",
  [0x2925] = "\x47\x65", [0x2926] = "\x47\x66", [0x2927] = "\x47\x67",
  [0x2928] = "\x47\x68", [0x2929] = "\x47\x69", [0x292a] = "\x47\x6a",
  [0x292b] = "\x47\x6b", [0x292c] = "\x47\x6c", [0x292d] = "\x47\x6d",
  [0x292e] = "\x47\x6e", [0x292f] = "\x47\x6f", [0x2930] = "\x47\x70",
  [0x2931] = "\x47\x71", [0x2932] = "\x47\x72", [0x2933] = "\x47\x73",
  [0x2934] = "\x47\x74", [0x2935] = "\x47\x75", [0x2936] = "\x47\x76",
  [0x2937] = "\x47\x77", [0x2938] = "\x47\x78", [0x2939] = "\x47\x79",
  [0x293a] = "\x47\x7a", [0x293b] = "\x47\x7b", [0x293c] = "\x47\x7c",
  [0x293d] = "\x47\x7d", [0x293e] = "\x47\x7e", [0x293f] = "\x47\x7f",
  [0x2940] = "\x47\x80", [0x2941] = "\x47\x81", [0x2942] = "\x47\x82",
  [0x2943] = "\x47\x83", [0x2944] = "\x47\x84", [0x2945] = "\x47\x85",
  [0x2946] = "\x47\x86", [0x2947] = "\x47\x87", [0x2948] = "\x47\x88",
  [0x2949] = "\x47\x89", [0x294a] = "\x47\x8a", [0x294b] = "\x47\x8b",
  [0x294c] = "\x47\x8c", [0x294d] = "\x47\x8d", [0x294e] = "\x47\x8e",
  [0x294f] = "\x47\x8f", [0x2950] = "\x47\x90", [0x2951] = "\x47\x91",
  [0x2952] = "\x47\x92", [0x2953] = "\x47\x93", [0x2954] = "\x47\x94",
  [0x2955] = "\x47\x95", [0x2956] = "\x47\x96", [0x2957] = "\x47\x97",
  [0x2958] = "\x47\x98", [0x2959] = "\x47\x99", [0x295a] = "\x47\x9a",
  [0x295b] = "\x47\x9b", [0x295c] = "\x47\x9c", [0x295d] = "\x47\x9d",
  [0x295e] = "\x47\x9e", [0x295f] = "\x47\x9f", [0x2960] = "\x47\xa0",
  [0x2961] = "\x47\xa1", [0x2962] = "\x47\xa2", [0x2963] = "\x47\xa3",
  [0x2964] = "\x47\xa4", [0x2965] = "\x47\xa5", [0x2966] = "\x47\xa6",
  [0x2967] = "\x47\xa7", [0x2968] = "\x47\xa8", [0x2969] = "\x47\xa9",
  [0x296a] = "\x47\xaa", [0x296b] = "\x47\xab", [0x296c] = "\x47\xac",
  [0x296d] = "\x47\xad", [0x296e] = "\x47\xae", [0x296f] = "\x47\xaf",
  [0x2970] = "\x47\xb0", [0x2971] = "\x47\xb1", [0x2972] = "\x47\xb2",
  [0x2973] = "\x47\xb3", [0x2974] = "\x47\xb4", [0x2975] = "\x47\xb5",
  [0x2976] = "\x47\xb6", [0x2977] = "\x47\xb7", [0x2978] = "\x47\xb8",
  [0x2979] = "\x47\xb9", [0x297a] = "\x47\xba", [0x297b] = "\x47\xbb",
  [0x297c] = "\x47\xbc", [0x297d] = "\x47\xbd", [0x297e] = "\x47\xbe",
  [0x297f] = "\x47\xbf", [0x2980] = "\x47\xc0", [0x2981] = "\x47\xc1",
  [0x2982] = "\x47\xc2", [0x2983] = "\x47\xc3", [0x2984] = "\x47\xc4",
  [0x2985] = "\x47\xc5", [0x2986] = "\x47\xc6", [0x2987] = "\x47\xc7",
  [0x2988] = "\x47\xc8", [0x2989] = "\x47\xc9", [0x298a] = "\x47\xca",
  [0x298b] = "\x47\xcb", [0x298c] = "\x47\xcc", [0x298d] = "\x47\xcd",
  [0x298e] = "\x47\xce", [0x298f] = "\x47\xcf", [0x2990] = "\x47\xd0",
  [0x2991] = "\x47\xd1", [0x2992] = "\x47\xd2", [0x2993] = "\x47\xd3",
  [0x2994] = "\x47\xd4", [0x2995] = "\x47\xd5", [0x2996] = "\x47\xd6",
  [0x2997] = "\x47\xd7", [0x2998] = "\x47\xd8", [0x2999] = "\x47\xd9",
  [0x299a] = "\x47\xda", [0x299b] = "\x47\xdb", [0x299c] = "\x47\xdc",
  [0x299d] = "\x47\xdd", [0x299e] = "\x47\xde", [0x299f] = "\x47\xdf",
  [0x29a0] = "\x47\xe0", [0x29a1] = "\x47\xe1", [0x29a2] = "\x47\xe2",
  [0x29a3] = "\x47\xe3", [0x29a4] = "\x47\xe4", [0x29a5] = "\x47\xe5",
  [0x29a6] = "\x47\xe6", [0x29a7] = "\x47\xe7", [0x29a8] = "\x47\xe8",
  [0x29a9] = "\x47\xe9", [0x29aa] = "\x47\xea", [0x29ab] = "\x47\xeb",
  [0x29ac] = "\x47\xec", [0x29ad] = "\x47\xed", [0x29ae] = "\x47\xee",
  [0x29af] = "\x47\xef", [0x29b0] = "\x47\xf0", [0x29b1] = "\x47\xf1",
  [0x29b2] = "\x47\xf2", [0x29b3] = "\x47\xf3", [0x29b4] = "\x47\xf4",
  [0x29b5] = "\x47\xf5", [0x29b6] = "\x47\xf6", [0x29b7] = "\x47\xf7",
  [0x29b8] = "\x47\xf8", [0x29b9] = "\x47\xf9", [0x29ba] = "\x47\xfa",
  [0x29bb] = "\x47\xfb", [0x29bc] = "\x47\xfc", [0x29bd] = "\x47\xfd",
  [0x29be] = "\x47\xfe", [0x29bf] = "\x47\xff", [0x29c0] = "\x48\x00",
  [0x29c1] = "\x48\x01", [0x29c2] = "\x48\x02", [0x29c3] = "\x48\x03",
  [0x29c4] = "\x48\x04", [0x29c5] = "\x48\x05", [0x29c6] = "\x48\x06",
  [0x29c7] = "\x48\x07", [0x29c8] = "\x48\x08", [0x29c9] = "\x48\x09",
  [0x29ca] = "\x48\x0a", [0x29cb] = "\x48\x0b", [0x29cc] = "\x48\x0c",
  [0x29cd] = "\x48\x0d", [0x29ce] = "\x48\x0e", [0x29cf] = "\x48\x0f",
  [0x29d0] = "\x48\x10", [0x29d1] = "\x48\x11", [0x29d2] = "\x48\x12",
  [0x29d3] = "\x48\x13", [0x29d4] = "\x48\x14", [0x29d5] = "\x48\x15",
  [0x29d6] = "\x48\x16", [0x29d7] = "\x48\x17", [0x29d8] = "\x48\x18",
  [0x29d9] = "\x48\x19", [0x29da] = "\x48\x1a", [0x29db] = "\x48\x1b",
  [0x29dc] = "\x48\x1c", [0x29dd] = "\x48\x1d", [0x29de] = "\x48\x1e",
  [0x29df] = "\x48\x1f", [0x29e0] = "\x48\x20", [0x29e1] = "\x48\x21",
  [0x29e2] = "\x48\x22", [0x29e3] = "\x48\x23", [0x29e4] = "\x48\x24",
  [0x29e5] = "\x48\x25", [0x29e6] = "\x48\x26", [0x29e7] = "\x48\x27",
  [0x29e8] = "\x48\x28", [0x29e9] = "\x48\x29", [0x29ea] = "\x48\x2a",
  [0x29eb] = "\x48\x2b", [0x29ec] = "\x48\x2c", [0x29ed] = "\x48\x2d",
  [0x29ee] = "\x48\x2e", [0x29ef] = "\x48\x2f", [0x29f0] = "\x48\x30",
  [0x29f1] = "\x48\x31", [0x29f2] = "\x48\x32", [0x29f3] = "\x48\x33",
  [0x29f4] = "\x48\x34", [0x29f5] = "\x48\x35", [0x29f6] = "\x48\x36",
  [0x29f7] = "\x48\x37", [0x29f8] = "\x48\x38", [0x29f9] = "\x48\x39",
  [0x29fa] = "\x48\x3a", [0x29fb] = "\x48\x3b", [0x29fc] = "\x48\x3c",
  [0x29fd] = "\x48\x3d", [0x29fe] = "\x48\x3e", [0x29ff] = "\x48\x3f",
  [0x2a00] = "\x48\x40", [0x2a01] = "\x48\x41", [0x2a02] = "\x48\x42",
  [0x2a03] = "\x48\x43", [0x2a04] = "\x48\x44", [0x2a05] = "\x48\x45",
  [0x2a06] = "\x48\x46", [0x2a07] = "\x48\x47", [0x2a08] = "\x48\x48",
  [0x2a09] = "\x48\x49", [0x2a0a] = "\x48\x4a", [0x2a0b] = "\x48\x4b",
  [0x2a0c] = "\x48\x4c", [0x2a0d] = "\x48\x4d", [0x2a0e] = "\x48\x4e",
  [0x2a0f] = "\x48\x4f", [0x2a10] = "\x48\x50", [0x2a11] = "\x48\x51",
  [0x2a12] = "\x48\x52", [0x2a13] = "\x48\x53", [0x2a14] = "\x48\x54",
  [0x2a15] = "\x48\x55", [0x2a16] = "\x48\x56", [0x2a17] = "\x48\x57",
  [0x2a18] = "\x48\x58", [0x2a19] = "\x48\x59", [0x2a1a] = "\x48\x5a",
  [0x2a1b] = "\x48\x5b", [0x2a1c] = "\x48\x5c", [0x2a1d] = "\x48\x5d",
  [0x2a1e] = "\x48\x5e", [0x2a1f] = "\x48\x5f", [0x2a20] = "\x48\x60",
  [0x2a21] = "\x48\x61", [0x2a22] = "\x48\x62", [0x2a23] = "\x48\x63",
  [0x2a24] = "\x48\x64", [0x2a25] = "\x48\x65", [0x2a26] = "\x48\x66",
  [0x2a27] = "\x48\x67", [0x2a28] = "\x48\x68", [0x2a29] = "\x48\x69",
  [0x2a2a] = "\x48\x6a", [0x2a2b] = "\x48\x6b", [0x2a2c] = "\x48\x6c",
  [0x2a2d] = "\x48\x6d", [0x2a2e] = "\x48\x6e", [0x2a2f] = "\x48\x6f",
  [0x2a30] = "\x48\x70", [0x2a31] = "\x48\x71", [0x2a32] = "\x48\x72",
  [0x2a33] = "\x48\x73", [0x2a34] = "\x48\x74", [0x2a35] = "\x48\x75",
  [0x2a36] = "\x48\x76", [0x2a37] = "\x48\x77", [0x2a38] = "\x48\x78",
  [0x2a39] = "\x48\x79", [0x2a3a] = "\x48\x7a", [0x2a3b] = "\x48\x7b",
  [0x2a3c] = "\x48\x7c", [0x2a3d] = "\x48\x7d", [0x2a3e] = "\x48\x7e",
  [0x2a3f] = "\x48\x7f", [0x2a40] = "\x48\x80", [0x2a41] = "\x48\x81",
  [0x2a42] = "\x48\x82", [0x2a43] = "\x48\x83", [0x2a44] = "\x48\x84",
  [0x2a45] = "\x48\x85", [0x2a46] = "\x48\x86", [0x2a47] = "\x48\x87",
  [0x2a48] = "\x48\x88", [0x2a49] = "\x48\x89", [0x2a4a] = "\x48\x8a",
  [0x2a4b] = "\x48\x8b", [0x2a4c] = "\x48\x8c", [0x2a4d] = "\x48\x8d",
  [0x2a4e] = "\x48\x8e", [0x2a4f] = "\x48\x8f", [0x2a50] = "\x48\x90",
  [0x2a51] = "\x48\x91", [0x2a52] = "\x48\x92", [0x2a53] = "\x48\x93",
  [0x2a54] = "\x48\x94", [0x2a55] = "\x48\x95", [0x2a56] = "\x48\x96",
  [0x2a57] = "\x48\x97", [0x2a58] = "\x48\x98", [0x2a59] = "\x48\x99",
  [0x2a5a] = "\x48\x9a", [0x2a5b] = "\x48\x9b", [0x2a5c] = "\x48\x9c",
  [0x2a5d] = "\x48\x9d", [0x2a5e] = "\x48\x9e", [0x2a5f] = "\x48\x9f",
  [0x2a60] = "\x48\xa0", [0x2a61] = "\x48\xa1", [0x2a62] = "\x48\xa2",
  [0x2a63] = "\x48\xa3", [0x2a64] = "\x48\xa4", [0x2a65] = "\x48\xa5",
  [0x2a66] = "\x48\xa6", [0x2a67] = "\x48\xa7", [0x2a68] = "\x48\xa8",
  [0x2a69] = "\x48\xa9", [0x2a6a] = "\x48\xaa", [0x2a6b] = "\x48\xab",
  [0x2a6c] = "\x48\xac", [0x2a6d] = "\x48\xad", [0x2a6e] = "\x48\xae",
  [0x2a6f] = "\x48\xaf", [0x2a70] = "\x48\xb0", [0x2a71] = "\x48\xb1",
  [0x2a72] = "\x48\xb2", [0x2a73] = "\x48\xb3", [0x2a74] = "\x48\xb4",
  [0x2a75] = "\x48\xb5", [0x2a76] = "\x48\xb6", [0x2a77] = "\x48\xb7",
  [0x2a78] = "\x48\xb8", [0x2a79] = "\x48\xb9", [0x2a7a] = "\x48\xba",
  [0x2a7b] = "\x48\xbb", [0x2a7c] = "\x48\xbc", [0x2a7d] = "\x48\xbd",
  [0x2a7e] = "\x48\xbe", [0x2a7f] = "\x48\xbf", [0x2a80] = "\x48\xc0",
  [0x2a81] = "\x48\xc1", [0x2a82] = "\x48\xc2", [0x2a83] = "\x48\xc3",
  [0x2a84] = "\x48\xc4", [0x2a85] = "\x48\xc5", [0x2a86] = "\x48\xc6",
  [0x2a87] = "\x48\xc7", [0x2a88] = "\x48\xc8", [0x2a89] = "\x48\xc9",
  [0x2a8a] = "\x48\xca", [0x2a8b] = "\x48\xcb", [0x2a8c] = "\x48\xcc",
  [0x2a8d] = "\x48\xcd", [0x2a8e] = "\x48\xce", [0x2a8f] = "\x48\xcf",
  [0x2a90] = "\x48\xd0", [0x2a91] = "\x48\xd1", [0x2a92] = "\x48\xd2",
  [0x2a93] = "\x48\xd3", [0x2a94] = "\x48\xd4", [0x2a95] = "\x48\xd5",
  [0x2a96] = "\x48\xd6", [0x2a97] = "\x48\xd7", [0x2a98] = "\x48\xd8",
  [0x2a99] = "\x48\xd9", [0x2a9a] = "\x48\xda", [0x2a9b] = "\x48\xdb",
  [0x2a9c] = "\x48\xdc", [0x2a9d] = "\x48\xdd", [0x2a9e] = "\x48\xde",
  [0x2a9f] = "\x48\xdf", [0x2aa0] = "\x48\xe0", [0x2aa1] = "\x48\xe1",
  [0x2aa2] = "\x48\xe2", [0x2aa3] = "\x48\xe3", [0x2aa4] = "\x48\xe4",
  [0x2aa5] = "\x48\xe5", [0x2aa6] = "\x48\xe6", [0x2aa7] = "\x48\xe7",
  [0x2aa8] = "\x48\xe8", [0x2aa9] = "\x48\xe9", [0x2aaa] = "\x48\xea",
  [0x2aab] = "\x48\xeb", [0x2aac] = "\x48\xec", [0x2aad] = "\x48\xed",
  [0x2aae] = "\x48\xee", [0x2aaf] = "\x48\xef", [0x2ab0] = "\x48\xf0",
  [0x2ab1] = "\x48\xf1", [0x2ab2] = "\x48\xf2", [0x2ab3] = "\x48\xf3",
  [0x2ab4] = "\x48\xf4", [0x2ab5] = "\x48\xf5", [0x2ab6] = "\x48\xf6",
  [0x2ab7] = "\x48\xf7", [0x2ab8] = "\x48\xf8", [0x2ab9] = "\x48\xf9",
  [0x2aba] = "\x48\xfa", [0x2abb] = "\x48\xfb", [0x2abc] = "\x48\xfc",
  [0x2abd] = "\x48\xfd", [0x2abe] = "\x48\xfe", [0x2abf] = "\x48\xff",
  [0x2ac0] = "\x49\x00", [0x2ac1] = "\x49\x01", [0x2ac2] = "\x49\x02",
  [0x2ac3] = "\x49\x03", [0x2ac4] = "\x49\x04", [0x2ac5] = "\x49\x05",
  [0x2ac6] = "\x49\x06", [0x2ac7] = "\x49\x07", [0x2ac8] = "\x49\x08",
  [0x2ac9] = "\x49\x09", [0x2aca] = "\x49\x0a", [0x2acb] = "\x49\x0b",
  [0x2acc] = "\x49\x0c", [0x2acd] = "\x49\x0d", [0x2ace] = "\x49\x0e",
  [0x2acf] = "\x49\x0f", [0x2ad0] = "\x49\x10", [0x2ad1] = "\x49\x11",
  [0x2ad2] = "\x49\x12", [0x2ad3] = "\x49\x13", [0x2ad4] = "\x49\x14",
  [0x2ad5] = "\x49\x15", [0x2ad6] = "\x49\x16", [0x2ad7] = "\x49\x17",
  [0x2ad8] = "\x49\x18", [0x2ad9] = "\x49\x19", [0x2ada] = "\x49\x1a",
  [0x2adb] = "\x49\x1b", [0x2adc] = "\x49\x1c", [0x2add] = "\x49\x1d",
  [0x2ade] = "\x49\x1e", [0x2adf] = "\x49\x1f", [0x2ae0] = "\x49\x20",
  [0x2ae1] = "\x49\x21", [0x2ae2] = "\x49\x22", [0x2ae3] = "\x49\x23",
  [0x2ae4] = "\x49\x24", [0x2ae5] = "\x49\x25", [0x2ae6] = "\x49\x26",
  [0x2ae7] = "\x49\x27", [0x2ae8] = "\x49\x28", [0x2ae9] = "\x49\x29",
  [0x2aea] = "\x49\x2a", [0x2aeb] = "\x49\x2b", [0x2aec] = "\x49\x2c",
  [0x2aed] = "\x49\x2d", [0x2aee] = "\x49\x2e", [0x2aef] = "\x49\x2f",
  [0x2af0] = "\x49\x30", [0x2af1] = "\x49\x31", [0x2af2] = "\x49\x32",
  [0x2af3] = "\x49\x33", [0x2af4] = "\x49\x34", [0x2af5] = "\x49\x35",
  [0x2af6] = "\x49\x36", [0x2af7] = "\x49\x37", [0x2af8] = "\x49\x38",
  [0x2af9] = "\x49\x39", [0x2afa] = "\x49\x3a", [0x2afb] = "\x49\x3b",
  [0x2afc] = "\x49\x3c", [0x2afd] = "\x49\x3d", [0x2afe] = "\x49\x3e",
  [0x2aff] = "\x49\x3f", [0x2b00] = "\x49\x40", [0x2b01] = "\x49\x41",
  [0x2b02] = "\x49\x42", [0x2b03] = "\x49\x43", [0x2b04] = "\x49\x44",
  [0x2b05] = "\x49\x45", [0x2b06] = "\x49\x46", [0x2b07] = "\x49\x47",
  [0x2b08] = "\x49\x48", [0x2b09] = "\x49\x49", [0x2b0a] = "\x49\x4a",
  [0x2b0b] = "\x49\x4b", [0x2b0c] = "\x49\x4c", [0x2b0d] = "\x49\x4d",
  [0x2b0e] = "\x49\x4e", [0x2b0f] = "\x49\x4f", [0x2b10] = "\x49\x50",
  [0x2b11] = "\x49\x51", [0x2b12] = "\x49\x52", [0x2b13] = "\x49\x53",
  [0x2b14] = "\x49\x54", [0x2b15] = "\x49\x55", [0x2b16] = "\x49\x56",
  [0x2b17] = "\x49\x57", [0x2b18] = "\x49\x58", [0x2b19] = "\x49\x59",
  [0x2b1a] = "\x49\x5a", [0x2b1b] = "\x49\x5b", [0x2b1c] = "\x49\x5c",
  [0x2b1d] = "\x49\x5d", [0x2b1e] = "\x49\x5e", [0x2b1f] = "\x49\x5f",
  [0x2b20] = "\x49\x60", [0x2b21] = "\x49\x61", [0x2b22] = "\x49\x62",
  [0x2b23] = "\x49\x63", [0x2b24] = "\x49\x64", [0x2b25] = "\x49\x65",
  [0x2b26] = "\x49\x66", [0x2b27] = "\x49\x67", [0x2b28] = "\x49\x68",
  [0x2b29] = "\x49\x69", [0x2b2a] = "\x49\x6a", [0x2b2b] = "\x49\x6b",
  [0x2b2c] = "\x49\x6c", [0x2b2d] = "\x49\x6d", [0x2b2e] = "\x49\x6e",
  [0x2b2f] = "\x49\x6f", [0x2b30] = "\x49\x70", [0x2b31] = "\x49\x71",
  [0x2b32] = "\x49\x72", [0x2b33] = "\x49\x73", [0x2b34] = "\x49\x74",
  [0x2b35] = "\x49\x75", [0x2b36] = "\x49\x76", [0x2b37] = "\x49\x77",
  [0x2b38] = "\x49\x78", [0x2b39] = "\x49\x79", [0x2b3a] = "\x49\x7a",
  [0x2b3b] = "\x49\x7b", [0x2b3c] = "\x49\x7c", [0x2b3d] = "\x49\x7d",
  [0x2b3e] = "\x49\x7e", [0x2b3f] = "\x49\x7f", [0x2b40] = "\x49\x80",
  [0x2b41] = "\x49\x81", [0x2b42] = "\x49\x82", [0x2b43] = "\x49\x83",
  [0x2b44] = "\x49\x84", [0x2b45] = "\x49\x85", [0x2b46] = "\x49\x86",
  [0x2b47] = "\x49\x87", [0x2b48] = "\x49\x88", [0x2b49] = "\x49\x89",
  [0x2b4a] = "\x49\x8a", [0x2b4b] = "\x49\x8b", [0x2b4c] = "\x49\x8c",
  [0x2b4d] = "\x49\x8d", [0x2b4e] = "\x49\x8e", [0x2b4f] = "\x49\x8f",
  [0x2b50] = "\x49\x90", [0x2b51] = "\x49\x91", [0x2b52] = "\x49\x92",
  [0x2b53] = "\x49\x93", [0x2b54] = "\x49\x94", [0x2b55] = "\x49\x95",
  [0x2b56] = "\x49\x96", [0x2b57] = "\x49\x97", [0x2b58] = "\x49\x98",
  [0x2b59] = "\x49\x99", [0x2b5a] = "\x49\x9a", [0x2b5b] = "\x49\x9b",
  [0x2b5c] = "\x49\x9c", [0x2b5d] = "\x49\x9d", [0x2b5e] = "\x49\x9e",
  [0x2b5f] = "\x49\x9f", [0x2b60] = "\x49\xa0", [0x2b61] = "\x49\xa1",
  [0x2b62] = "\x49\xa2", [0x2b63] = "\x49\xa3", [0x2b64] = "\x49\xa4",
  [0x2b65] = "\x49\xa5", [0x2b66] = "\x49\xa6", [0x2b67] = "\x49\xa7",
  [0x2b68] = "\x49\xa8", [0x2b69] = "\x49\xa9", [0x2b6a] = "\x49\xaa",
  [0x2b6b] = "\x49\xab", [0x2b6c] = "\x49\xac", [0x2b6d] = "\x49\xad",
  [0x2b6e] = "\x49\xae", [0x2b6f] = "\x49\xaf", [0x2b70] = "\x49\xb0",
  [0x2b71] = "\x49\xb1", [0x2b72] = "\x49\xb2", [0x2b73] = "\x49\xb3",
  [0x2b74] = "\x49\xb4", [0x2b75] = "\x49\xb5", [0x2b76] = "\x49\xb6",
  [0x2b77] = "\x49\xb7", [0x2b78] = "\x49\xb8", [0x2b79] = "\x49\xb9",
  [0x2b7a] = "\x49\xba", [0x2b7b] = "\x49\xbb", [0x2b7c] = "\x49\xbc",
  [0x2b7d] = "\x49\xbd", [0x2b7e] = "\x49\xbe", [0x2b7f] = "\x49\xbf",
  [0x2b80] = "\x49\xc0", [0x2b81] = "\x49\xc1", [0x2b82] = "\x49\xc2",
  [0x2b83] = "\x49\xc3", [0x2b84] = "\x49\xc4", [0x2b85] = "\x49\xc5",
  [0x2b86] = "\x49\xc6", [0x2b87] = "\x49\xc7", [0x2b88] = "\x49\xc8",
  [0x2b89] = "\x49\xc9", [0x2b8a] = "\x49\xca", [0x2b8b] = "\x49\xcb",
  [0x2b8c] = "\x49\xcc", [0x2b8d] = "\x49\xcd", [0x2b8e] = "\x49\xce",
  [0x2b8f] = "\x49\xcf", [0x2b90] = "\x49\xd0", [0x2b91] = "\x49\xd1",
  [0x2b92] = "\x49\xd2", [0x2b93] = "\x49\xd3", [0x2b94] = "\x49\xd4",
  [0x2b95] = "\x49\xd5", [0x2b96] = "\x49\xd6", [0x2b97] = "\x49\xd7",
  [0x2b98] = "\x49\xd8", [0x2b99] = "\x49\xd9", [0x2b9a] = "\x49\xda",
  [0x2b9b] = "\x49\xdb", [0x2b9c] = "\x49\xdc", [0x2b9d] = "\x49\xdd",
  [0x2b9e] = "\x49\xde", [0x2b9f] = "\x49\xdf", [0x2ba0] = "\x49\xe0",
  [0x2ba1] = "\x49\xe1", [0x2ba2] = "\x49\xe2", [0x2ba3] = "\x49\xe3",
  [0x2ba4] = "\x49\xe4", [0x2ba5] = "\x49\xe5", [0x2ba6] = "\x49\xe6",
  [0x2ba7] = "\x49\xe7", [0x2ba8] = "\x49\xe8", [0x2ba9] = "\x49\xe9",
  [0x2baa] = "\x49\xea", [0x2bab] = "\x49\xeb", [0x2bac] = "\x49\xec",
  [0x2bad] = "\x49\xed", [0x2bae] = "\x49\xee", [0x2baf] = "\x49\xef",
  [0x2bb0] = "\x49\xf0", [0x2bb1] = "\x49\xf1", [0x2bb2] = "\x49\xf2",
  [0x2bb3] = "\x49\xf3", [0x2bb4] = "\x49\xf4", [0x2bb5] = "\x49\xf5",
  [0x2bb6] = "\x49\xf6", [0x2bb7] = "\x49\xf7", [0x2bb8] = "\x49\xf8",
  [0x2bb9] = "\x49\xf9", [0x2bba] = "\x49\xfa", [0x2bbb] = "\x49\xfb",
  [0x2bbc] = "\x49\xfc", [0x2bbd] = "\x49\xfd", [0x2bbe] = "\x49\xfe",
  [0x2bbf] = "\x49\xff", [0x2bc0] = "\x4a\x00", [0x2bc1] = "\x4a\x01",
  [0x2bc2] = "\x4a\x02", [0x2bc3] = "\x4a\x03", [0x2bc4] = "\x4a\x04",
  [0x2bc5] = "\x4a\x05", [0x2bc6] = "\x4a\x06", [0x2bc7] = "\x4a\x07",
  [0x2bc8] = "\x4a\x08", [0x2bc9] = "\x4a\x09", [0x2bca] = "\x4a\x0a",
  [0x2bcb] = "\x4a\x0b", [0x2bcc] = "\x4a\x0c", [0x2bcd] = "\x4a\x0d",
  [0x2bce] = "\x4a\x0e", [0x2bcf] = "\x4a\x0f", [0x2bd0] = "\x4a\x10",
  [0x2bd1] = "\x4a\x11", [0x2bd2] = "\x4a\x12", [0x2bd3] = "\x4a\x13",
  [0x2bd4] = "\x4a\x14", [0x2bd5] = "\x4a\x15", [0x2bd6] = "\x4a\x16",
  [0x2bd7] = "\x4a\x17", [0x2bd8] = "\x4a\x18", [0x2bd9] = "\x4a\x19",
  [0x2bda] = "\x4a\x1a", [0x2bdb] = "\x4a\x1b", [0x2bdc] = "\x4a\x1c",
  [0x2bdd] = "\x4a\x1d", [0x2bde] = "\x4a\x1e", [0x2bdf] = "\x4a\x1f",
  [0x2be0] = "\x4a\x20", [0x2be1] = "\x4a\x21", [0x2be2] = "\x4a\x22",
  [0x2be3] = "\x4a\x23", [0x2be4] = "\x4a\x24", [0x2be5] = "\x4a\x25",
  [0x2be6] = "\x4a\x26", [0x2be7] = "\x4a\x27", [0x2be8] = "\x4a\x28",
  [0x2be9] = "\x4a\x29", [0x2bea] = "\x4a\x2a", [0x2beb] = "\x4a\x2b",
  [0x2bec] = "\x4a\x2c", [0x2bed] = "\x4a\x2d", [0x2bee] = "\x4a\x2e",
  [0x2bef] = "\x4a\x2f", [0x2bf0] = "\x4a\x30", [0x2bf1] = "\x4a\x31",
  [0x2bf2] = "\x4a\x32", [0x2bf3] = "\x4a\x33", [0x2bf4] = "\x4a\x34",
  [0x2bf5] = "\x4a\x35", [0x2bf6] = "\x4a\x36", [0x2bf7] = "\x4a\x37",
  [0x2bf8] = "\x4a\x38", [0x2bf9] = "\x4a\x39", [0x2bfa] = "\x4a\x3a",
  [0x2bfb] = "\x4a\x3b", [0x2bfc] = "\x4a\x3c", [0x2bfd] = "\x4a\x3d",
  [0x2bfe] = "\x4a\x3e", [0x2bff] = "\x4a\x3f", [0x2c00] = "\x4a\x40",
  [0x2c01] = "\x4a\x41", [0x2c02] = "\x4a\x42", [0x2c03] = "\x4a\x43",
  [0x2c04] = "\x4a\x44", [0x2c05] = "\x4a\x45", [0x2c06] = "\x4a\x46",
  [0x2c07] = "\x4a\x47", [0x2c08] = "\x4a\x48", [0x2c09] = "\x4a\x49",
  [0x2c0a] = "\x4a\x4a", [0x2c0b] = "\x4a\x4b", [0x2c0c] = "\x4a\x4c",
  [0x2c0d] = "\x4a\x4d", [0x2c0e] = "\x4a\x4e", [0x2c0f] = "\x4a\x4f",
  [0x2c10] = "\x4a\x50", [0x2c11] = "\x4a\x51", [0x2c12] = "\x4a\x52",
  [0x2c13] = "\x4a\x53", [0x2c14] = "\x4a\x54", [0x2c15] = "\x4a\x55",
  [0x2c16] = "\x4a\x56", [0x2c17] = "\x4a\x57", [0x2c18] = "\x4a\x58",
  [0x2c19] = "\x4a\x59", [0x2c1a] = "\x4a\x5a", [0x2c1b] = "\x4a\x5b",
  [0x2c1c] = "\x4a\x5c", [0x2c1d] = "\x4a\x5d", [0x2c1e] = "\x4a\x5e",
  [0x2c1f] = "\x4a\x5f", [0x2c20] = "\x4a\x60", [0x2c21] = "\x4a\x61",
  [0x2c22] = "\x4a\x62", [0x2c23] = "\x4a\x63", [0x2c24] = "\x4a\x64",
  [0x2c25] = "\x4a\x65", [0x2c26] = "\x4a\x66", [0x2c27] = "\x4a\x67",
  [0x2c28] = "\x4a\x68", [0x2c29] = "\x4a\x69", [0x2c2a] = "\x4a\x6a",
  [0x2c2b] = "\x4a\x6b", [0x2c2c] = "\x4a\x6c", [0x2c2d] = "\x4a\x6d",
  [0x2c2e] = "\x4a\x6e", [0x2c2f] = "\x4a\x6f", [0x2c30] = "\x4a\x70",
  [0x2c31] = "\x4a\x71", [0x2c32] = "\x4a\x72", [0x2c33] = "\x4a\x73",
  [0x2c34] = "\x4a\x74", [0x2c35] = "\x4a\x75", [0x2c36] = "\x4a\x76",
  [0x2c37] = "\x4a\x77", [0x2c38] = "\x4a\x78", [0x2c39] = "\x4a\x79",
  [0x2c3a] = "\x4a\x7a", [0x2c3b] = "\x4a\x7b", [0x2c3c] = "\x4a\x7c",
  [0x2c3d] = "\x4a\x7d", [0x2c3e] = "\x4a\x7e", [0x2c3f] = "\x4a\x7f",
  [0x2c40] = "\x4a\x80", [0x2c41] = "\x4a\x81", [0x2c42] = "\x4a\x82",
  [0x2c43] = "\x4a\x83", [0x2c44] = "\x4a\x84", [0x2c45] = "\x4a\x85",
  [0x2c46] = "\x4a\x86", [0x2c47] = "\x4a\x87", [0x2c48] = "\x4a\x88",
  [0x2c49] = "\x4a\x89", [0x2c4a] = "\x4a\x8a", [0x2c4b] = "\x4a\x8b",
  [0x2c4c] = "\x4a\x8c", [0x2c4d] = "\x4a\x8d", [0x2c4e] = "\x4a\x8e",
  [0x2c4f] = "\x4a\x8f", [0x2c50] = "\x4a\x90", [0x2c51] = "\x4a\x91",
  [0x2c52] = "\x4a\x92", [0x2c53] = "\x4a\x93", [0x2c54] = "\x4a\x94",
  [0x2c55] = "\x4a\x95", [0x2c56] = "\x4a\x96", [0x2c57] = "\x4a\x97",
  [0x2c58] = "\x4a\x98", [0x2c59] = "\x4a\x99", [0x2c5a] = "\x4a\x9a",
  [0x2c5b] = "\x4a\x9b", [0x2c5c] = "\x4a\x9c", [0x2c5d] = "\x4a\x9d",
  [0x2c5e] = "\x4a\x9e", [0x2c5f] = "\x4a\x9f", [0x2c60] = "\x4a\xa0",
  [0x2c61] = "\x4a\xa1", [0x2c62] = "\x4a\xa2", [0x2c63] = "\x4a\xa3",
  [0x2c64] = "\x4a\xa4", [0x2c65] = "\x4a\xa5", [0x2c66] = "\x4a\xa6",
  [0x2c67] = "\x4a\xa7", [0x2c68] = "\x4a\xa8", [0x2c69] = "\x4a\xa9",
  [0x2c6a] = "\x4a\xaa", [0x2c6b] = "\x4a\xab", [0x2c6c] = "\x4a\xac",
  [0x2c6d] = "\x4a\xad", [0x2c6e] = "\x4a\xae", [0x2c6f] = "\x4a\xaf",
  [0x2c70] = "\x4a\xb0", [0x2c71] = "\x4a\xb1", [0x2c72] = "\x4a\xb2",
  [0x2c73] = "\x4a\xb3", [0x2c74] = "\x4a\xb4", [0x2c75] = "\x4a\xb5",
  [0x2c76] = "\x4a\xb6", [0x2c77] = "\x4a\xb7", [0x2c78] = "\x4a\xb8",
  [0x2c79] = "\x4a\xb9", [0x2c7a] = "\x4a\xba", [0x2c7b] = "\x4a\xbb",
  [0x2c7c] = "\x4a\xbc", [0x2c7d] = "\x4a\xbd", [0x2c7e] = "\x4a\xbe",
  [0x2c7f] = "\x4a\xbf", [0x2c80] = "\x4a\xc0", [0x2c81] = "\x4a\xc1",
  [0x2c82] = "\x4a\xc2", [0x2c83] = "\x4a\xc3", [0x2c84] = "\x4a\xc4",
  [0x2c85] = "\x4a\xc5", [0x2c86] = "\x4a\xc6", [0x2c87] = "\x4a\xc7",
  [0x2c88] = "\x4a\xc8", [0x2c89] = "\x4a\xc9", [0x2c8a] = "\x4a\xca",
  [0x2c8b] = "\x4a\xcb", [0x2c8c] = "\x4a\xcc", [0x2c8d] = "\x4a\xcd",
  [0x2c8e] = "\x4a\xce", [0x2c8f] = "\x4a\xcf", [0x2c90] = "\x4a\xd0",
  [0x2c91] = "\x4a\xd1", [0x2c92] = "\x4a\xd2", [0x2c93] = "\x4a\xd3",
  [0x2c94] = "\x4a\xd4", [0x2c95] = "\x4a\xd5", [0x2c96] = "\x4a\xd6",
  [0x2c97] = "\x4a\xd7", [0x2c98] = "\x4a\xd8", [0x2c99] = "\x4a\xd9",
  [0x2c9a] = "\x4a\xda", [0x2c9b] = "\x4a\xdb", [0x2c9c] = "\x4a\xdc",
  [0x2c9d] = "\x4a\xdd", [0x2c9e] = "\x4a\xde", [0x2c9f] = "\x4a\xdf",
  [0x2ca0] = "\x4a\xe0", [0x2ca1] = "\x4a\xe1", [0x2ca2] = "\x4a\xe2",
  [0x2ca3] = "\x4a\xe3", [0x2ca4] = "\x4a\xe4", [0x2ca5] = "\x4a\xe5",
  [0x2ca6] = "\x4a\xe6", [0x2ca7] = "\x4a\xe7", [0x2ca8] = "\x4a\xe8",
  [0x2ca9] = "\x4a\xe9", [0x2caa] = "\x4a\xea", [0x2cab] = "\x4a\xeb",
  [0x2cac] = "\x4a\xec", [0x2cad] = "\x4a\xed", [0x2cae] = "\x4a\xee",
  [0x2caf] = "\x4a\xef", [0x2cb0] = "\x4a\xf0", [0x2cb1] = "\x4a\xf1",
  [0x2cb2] = "\x4a\xf2", [0x2cb3] = "\x4a\xf3", [0x2cb4] = "\x4a\xf4",
  [0x2cb5] = "\x4a\xf5", [0x2cb6] = "\x4a\xf6", [0x2cb7] = "\x4a\xf7",
  [0x2cb8] = "\x4a\xf8", [0x2cb9] = "\x4a\xf9", [0x2cba] = "\x4a\xfa",
  [0x2cbb] = "\x4a\xfb", [0x2cbc] = "\x4a\xfc", [0x2cbd] = "\x4a\xfd",
  [0x2cbe] = "\x4a\xfe", [0x2cbf] = "\x4a\xff", [0x2cc0] = "\x4b\x00",
  [0x2cc1] = "\x4b\x01", [0x2cc2] = "\x4b\x02", [0x2cc3] = "\x4b\x03",
  [0x2cc4] = "\x4b\x04", [0x2cc5] = "\x4b\x05", [0x2cc6] = "\x4b\x06",
  [0x2cc7] = "\x4b\x07", [0x2cc8] = "\x4b\x08", [0x2cc9] = "\x4b\x09",
  [0x2cca] = "\x4b\x0a", [0x2ccb] = "\x4b\x0b", [0x2ccc] = "\x4b\x0c",
  [0x2ccd] = "\x4b\x0d", [0x2cce] = "\x4b\x0e", [0x2ccf] = "\x4b\x0f",
  [0x2cd0] = "\x4b\x10", [0x2cd1] = "\x4b\x11", [0x2cd2] = "\x4b\x12",
  [0x2cd3] = "\x4b\x13", [0x2cd4] = "\x4b\x14", [0x2cd5] = "\x4b\x15",
  [0x2cd6] = "\x4b\x16", [0x2cd7] = "\x4b\x17", [0x2cd8] = "\x4b\x18",
  [0x2cd9] = "\x4b\x19", [0x2cda] = "\x4b\x1a", [0x2cdb] = "\x4b\x1b",
  [0x2cdc] = "\x4b\x1c", [0x2cdd] = "\x4b\x1d", [0x2cde] = "\x4b\x1e",
  [0x2cdf] = "\x4b\x1f", [0x2ce0] = "\x4b\x20", [0x2ce1] = "\x4b\x21",
  [0x2ce2] = "\x4b\x22", [0x2ce3] = "\x4b\x23", [0x2ce4] = "\x4b\x24",
  [0x2ce5] = "\x4b\x25", [0x2ce6] = "\x4b\x26", [0x2ce7] = "\x4b\x27",
  [0x2ce8] = "\x4b\x28", [0x2ce9] = "\x4b\x29", [0x2cea] = "\x4b\x2a",
  [0x2ceb] = "\x4b\x2b", [0x2cec] = "\x4b\x2c", [0x2ced] = "\x4b\x2d",
  [0x2cee] = "\x4b\x2e", [0x2cef] = "\x4b\x2f", [0x2cf0] = "\x4b\x30",
  [0x2cf1] = "\x4b\x31", [0x2cf2] = "\x4b\x32", [0x2cf3] = "\x4b\x33",
  [0x2cf4] = "\x4b\x34", [0x2cf5] = "\x4b\x35", [0x2cf6] = "\x4b\x36",
  [0x2cf7] = "\x4b\x37", [0x2cf8] = "\x4b\x38", [0x2cf9] = "\x4b\x39",
  [0x2cfa] = "\x4b\x3a", [0x2cfb] = "\x4b\x3b", [0x2cfc] = "\x4b\x3c",
  [0x2cfd] = "\x4b\x3d", [0x2cfe] = "\x4b\x3e", [0x2cff] = "\x4b\x3f",
  [0x2d00] = "\x4b\x40", [0x2d01] = "\x4b\x41", [0x2d02] = "\x4b\x42",
  [0x2d03] = "\x4b\x43", [0x2d04] = "\x4b\x44", [0x2d05] = "\x4b\x45",
  [0x2d06] = "\x4b\x46", [0x2d07] = "\x4b\x47", [0x2d08] = "\x4b\x48",
  [0x2d09] = "\x4b\x49", [0x2d0a] = "\x4b\x4a", [0x2d0b] = "\x4b\x4b",
  [0x2d0c] = "\x4b\x4c", [0x2d0d] = "\x4b\x4d", [0x2d0e] = "\x4b\x4e",
  [0x2d0f] = "\x4b\x4f", [0x2d10] = "\x4b\x50", [0x2d11] = "\x4b\x51",
  [0x2d12] = "\x4b\x52", [0x2d13] = "\x4b\x53", [0x2d14] = "\x4b\x54",
  [0x2d15] = "\x4b\x55", [0x2d16] = "\x4b\x56", [0x2d17] = "\x4b\x57",
  [0x2d18] = "\x4b\x58", [0x2d19] = "\x4b\x59", [0x2d1a] = "\x4b\x5a",
  [0x2d1b] = "\x4b\x5b", [0x2d1c] = "\x4b\x5c", [0x2d1d] = "\x4b\x5d",
  [0x2d1e] = "\x4b\x5e", [0x2d1f] = "\x4b\x5f", [0x2d20] = "\x4b\x60",
  [0x2d21] = "\x4b\x61", [0x2d22] = "\x4b\x62", [0x2d23] = "\x4b\x63",
  [0x2d24] = "\x4b\x64", [0x2d25] = "\x4b\x65", [0x2d26] = "\x4b\x66",
  [0x2d27] = "\x4b\x67", [0x2d28] = "\x4b\x68", [0x2d29] = "\x4b\x69",
  [0x2d2a] = "\x4b\x6a", [0x2d2b] = "\x4b\x6b", [0x2d2c] = "\x4b\x6c",
  [0x2d2d] = "\x4b\x6d", [0x2d2e] = "\x4b\x6e", [0x2d2f] = "\x4b\x6f",
  [0x2d30] = "\x4b\x70", [0x2d31] = "\x4b\x71", [0x2d32] = "\x4b\x72",
  [0x2d33] = "\x4b\x73", [0x2d34] = "\x4b\x74", [0x2d35] = "\x4b\x75",
  [0x2d36] = "\x4b\x76", [0x2d37] = "\x4b\x77", [0x2d38] = "\x4b\x78",
  [0x2d39] = "\x4b\x79", [0x2d3a] = "\x4b\x7a", [0x2d3b] = "\x4b\x7b",
  [0x2d3c] = "\x4b\x7c", [0x2d3d] = "\x4b\x7d", [0x2d3e] = "\x4b\x7e",
  [0x2d3f] = "\x4b\x7f", [0x2d40] = "\x4b\x80", [0x2d41] = "\x4b\x81",
  [0x2d42] = "\x4b\x82", [0x2d43] = "\x4b\x83", [0x2d44] = "\x4b\x84",
  [0x2d45] = "\x4b\x85", [0x2d46] = "\x4b\x86", [0x2d47] = "\x4b\x87",
  [0x2d48] = "\x4b\x88", [0x2d49] = "\x4b\x89", [0x2d4a] = "\x4b\x8a",
  [0x2d4b] = "\x4b\x8b", [0x2d4c] = "\x4b\x8c", [0x2d4d] = "\x4b\x8d",
  [0x2d4e] = "\x4b\x8e", [0x2d4f] = "\x4b\x8f", [0x2d50] = "\x4b\x90",
  [0x2d51] = "\x4b\x91", [0x2d52] = "\x4b\x92", [0x2d53] = "\x4b\x93",
  [0x2d54] = "\x4b\x94", [0x2d55] = "\x4b\x95", [0x2d56] = "\x4b\x96",
  [0x2d57] = "\x4b\x97", [0x2d58] = "\x4b\x98", [0x2d59] = "\x4b\x99",
  [0x2d5a] = "\x4b\x9a", [0x2d5b] = "\x4b\x9b", [0x2d5c] = "\x4b\x9c",
  [0x2d5d] = "\x4b\x9d", [0x2d5e] = "\x4b\x9e", [0x2d5f] = "\x4b\x9f",
  [0x2d60] = "\x4b\xa0", [0x2d61] = "\x4b\xa1", [0x2d62] = "\x4b\xa2",
  [0x2d63] = "\x4b\xa3", [0x2d64] = "\x4b\xa4", [0x2d65] = "\x4b\xa5",
  [0x2d66] = "\x4b\xa6", [0x2d67] = "\x4b\xa7", [0x2d68] = "\x4b\xa8",
  [0x2d69] = "\x4b\xa9", [0x2d6a] = "\x4b\xaa", [0x2d6b] = "\x4b\xab",
  [0x2d6c] = "\x4b\xac", [0x2d6d] = "\x4b\xad", [0x2d6e] = "\x4b\xae",
  [0x2d6f] = "\x4b\xaf", [0x2d70] = "\x4b\xb0", [0x2d71] = "\x4b\xb1",
  [0x2d72] = "\x4b\xb2", [0x2d73] = "\x4b\xb3", [0x2d74] = "\x4b\xb4",
  [0x2d75] = "\x4b\xb5", [0x2d76] = "\x4b\xb6", [0x2d77] = "\x4b\xb7",
  [0x2d78] = "\x4b\xb8", [0x2d79] = "\x4b\xb9", [0x2d7a] = "\x4b\xba",
  [0x2d7b] = "\x4b\xbb", [0x2d7c] = "\x4b\xbc", [0x2d7d] = "\x4b\xbd",
  [0x2d7e] = "\x4b\xbe", [0x2d7f] = "\x4b\xbf", [0x2d80] = "\x4b\xc0",
  [0x2d81] = "\x4b\xc1", [0x2d82] = "\x4b\xc2", [0x2d83] = "\x4b\xc3",
  [0x2d84] = "\x4b\xc4", [0x2d85] = "\x4b\xc5", [0x2d86] = "\x4b\xc6",
  [0x2d87] = "\x4b\xc7", [0x2d88] = "\x4b\xc8", [0x2d89] = "\x4b\xc9",
  [0x2d8a] = "\x4b\xca", [0x2d8b] = "\x4b\xcb", [0x2d8c] = "\x4b\xcc",
  [0x2d8d] = "\x4b\xcd", [0x2d8e] = "\x4b\xce", [0x2d8f] = "\x4b\xcf",
  [0x2d90] = "\x4b\xd0", [0x2d91] = "\x4b\xd1", [0x2d92] = "\x4b\xd2",
  [0x2d93] = "\x4b\xd3", [0x2d94] = "\x4b\xd4", [0x2d95] = "\x4b\xd5",
  [0x2d96] = "\x4b\xd6", [0x2d97] = "\x4b\xd7", [0x2d98] = "\x4b\xd8",
  [0x2d99] = "\x4b\xd9", [0x2d9a] = "\x4b\xda", [0x2d9b] = "\x4b\xdb",
  [0x2d9c] = "\x4b\xdc", [0x2d9d] = "\x4b\xdd", [0x2d9e] = "\x4b\xde",
  [0x2d9f] = "\x4b\xdf", [0x2da0] = "\x4b\xe0", [0x2da1] = "\x4b\xe1",
  [0x2da2] = "\x4b\xe2", [0x2da3] = "\x4b\xe3", [0x2da4] = "\x4b\xe4",
  [0x2da5] = "\x4b\xe5", [0x2da6] = "\x4b\xe6", [0x2da7] = "\x4b\xe7",
  [0x2da8] = "\x4b\xe8", [0x2da9] = "\x4b\xe9", [0x2daa] = "\x4b\xea",
  [0x2dab] = "\x4b\xeb", [0x2dac] = "\x4b\xec", [0x2dad] = "\x4b\xed",
  [0x2dae] = "\x4b\xee", [0x2daf] = "\x4b\xef", [0x2db0] = "\x4b\xf0",
  [0x2db1] = "\x4b\xf1", [0x2db2] = "\x4b\xf2", [0x2db3] = "\x4b\xf3",
  [0x2db4] = "\x4b\xf4", [0x2db5] = "\x4b\xf5", [0x2db6] = "\x4b\xf6",
  [0x2db7] = "\x4b\xf7", [0x2db8] = "\x4b\xf8", [0x2db9] = "\x4b\xf9",
  [0x2dba] = "\x4b\xfa", [0x2dbb] = "\x4b\xfb", [0x2dbc] = "\x4b\xfc",
  [0x2dbd] = "\x4b\xfd", [0x2dbe] = "\x4b\xfe", [0x2dbf] = "\x4b\xff",
  [0x2dc0] = "\x4c\x00", [0x2dc1] = "\x4c\x01", [0x2dc2] = "\x4c\x02",
  [0x2dc3] = "\x4c\x03", [0x2dc4] = "\x4c\x04", [0x2dc5] = "\x4c\x05",
  [0x2dc6] = "\x4c\x06", [0x2dc7] = "\x4c\x07", [0x2dc8] = "\x4c\x08",
  [0x2dc9] = "\x4c\x09", [0x2dca] = "\x4c\x0a", [0x2dcb] = "\x4c\x0b",
  [0x2dcc] = "\x4c\x0c", [0x2dcd] = "\x4c\x0d", [0x2dce] = "\x4c\x0e",
  [0x2dcf] = "\x4c\x0f", [0x2dd0] = "\x4c\x10", [0x2dd1] = "\x4c\x11",
  [0x2dd2] = "\x4c\x12", [0x2dd3] = "\x4c\x13", [0x2dd4] = "\x4c\x14",
  [0x2dd5] = "\x4c\x15", [0x2dd6] = "\x4c\x16", [0x2dd7] = "\x4c\x17",
  [0x2dd8] = "\x4c\x18", [0x2dd9] = "\x4c\x19", [0x2dda] = "\x4c\x1a",
  [0x2ddb] = "\x4c\x1b", [0x2ddc] = "\x4c\x1c", [0x2ddd] = "\x4c\x1d",
  [0x2dde] = "\x4c\x1e", [0x2ddf] = "\x4c\x1f", [0x2de0] = "\x4c\x20",
  [0x2de1] = "\x4c\x21", [0x2de2] = "\x4c\x22", [0x2de3] = "\x4c\x23",
  [0x2de4] = "\x4c\x24", [0x2de5] = "\x4c\x25", [0x2de6] = "\x4c\x26",
  [0x2de7] = "\x4c\x27", [0x2de8] = "\x4c\x28", [0x2de9] = "\x4c\x29",
  [0x2dea] = "\x4c\x2a", [0x2deb] = "\x4c\x2b", [0x2dec] = "\x4c\x2c",
  [0x2ded] = "\x4c\x2d", [0x2dee] = "\x4c\x2e", [0x2def] = "\x4c\x2f",
  [0x2df0] = "\x4c\x30", [0x2df1] = "\x4c\x31", [0x2df2] = "\x4c\x32",
  [0x2df3] = "\x4c\x33", [0x2df4] = "\x4c\x34", [0x2df5] = "\x4c\x35",
  [0x2df6] = "\x4c\x36", [0x2df7] = "\x4c\x37", [0x2df8] = "\x4c\x38",
  [0x2df9] = "\x4c\x39", [0x2dfa] = "\x4c\x3a", [0x2dfb] = "\x4c\x3b",
  [0x2dfc] = "\x4c\x3c", [0x2dfd] = "\x4c\x3d", [0x2dfe] = "\x4c\x3e",
  [0x2dff] = "\x4c\x3f", [0x2e00] = "\x4c\x40", [0x2e01] = "\xfe\x50",
  [0x2e02] = "\x4c\x41", [0x2e03] = "\x4c\x42", [0x2e04] = "\xfe\x54",
  [0x2e05] = "\x4c\x43", [0x2e06] = "\x4c\x44", [0x2e07] = "\x4c\x45",
  [0x2e08] = "\xfe\x57", [0x2e09] = "\x4c\x46", [0x2e0a] = "\x4c\x47",
  [0x2e0b] = "\xfe\x58", [0x2e0c] = "\xfe\x5d", [0x2e0d] = "\x4c\x48",
  [0x2e0e] = "\x4c\x49", [0x2e0f] = "\x4c\x4a", [0x2e10] = "\x4c\x4b",
  [0x2e11] = "\x4c\x4c", [0x2e12] = "\x4c\x4d", [0x2e13] = "\x4c\x4e",
  [0x2e14] = "\x4c\x4f", [0x2e15] = "\x4c\x50", [0x2e16] = "\x4c\x51",
  [0x2e17] = "\xfe\x5e", [0x2e18] = "\x4c\x52", [0x2e19] = "\x4c\x53",
  [0x2e1a] = "\x4c\x54", [0x2e1b] = "\x4c\x55", [0x2e1c] = "\x4c\x56",
  [0x2e1d] = "\x4c\x57", [0x2e1e] = "\x4c\x58", [0x2e1f] = "\x4c\x59",
  [0x2e20] = "\x4c\x5a", [0x2e21] = "\x4c\x5b", [0x2e22] = "\x4c\x5c",
  [0x2e23] = "\x4c\x5d", [0x2e24] = "\x4c\x5e", [0x2e25] = "\x4c\x5f",
  [0x2e26] = "\x4c\x60", [0x2e27] = "\xfe\x6b", [0x2e28] = "\x4c\x61",
  [0x2e29] = "\x4c\x62", [0x2e2a] = "\xfe\x6e", [0x2e2b] = "\x4c\x63",
  [0x2e2c] = "\x4c\x64", [0x2e2d] = "\x4c\x65", [0x2e2e] = "\xfe\x71",
  [0x2e2f] = "\x4c\x66", [0x2e30] = "\x4c\x67", [0x2e31] = "\x4c\x68",
  [0x2e32] = "\x4c\x69", [0x2e33] = "\xfe\x73", [0x2e34] = "\x4c\x6a",
  [0x2e35] = "\x4c\x6b", [0x2e36] = "\xfe\x74", [0x2e37] = "\xfe\x75",
  [0x2e38] = "\x4c\x6c", [0x2e39] = "\x4c\x6d", [0x2e3a] = "\x4c\x6e",
  [0x2e3b] = "\xfe\x79", [0x2e3c] = "\x4c\x6f", [0x2e3d] = "\x4c\x70",
  [0x2e3e] = "\x4c\x71", [0x2e3f] = "\x4c\x72", [0x2e40] = "\x4c\x73",
  [0x2e41] = "\x4c\x74", [0x2e42] = "\x4c\x75", [0x2e43] = "\x4c\x76",
  [0x2e44] = "\x4c\x77", [0x2e45] = "\x4c\x78", [0x2e46] = "\x4c\x79",
  [0x2e47] = "\x4c\x7a", [0x2e48] = "\x4c\x7b", [0x2e49] = "\x4c\x7c",
  [0x2e4a] = "\xfe\x84", [0x2e4b] = "\x4c\x7d", [0x2e4c] = "\x4c\x7e",
  [0x2e4d] = "\x4c\x7f", [0x2e4e] = "\x4c\x80", [0x2e4f] = "\x4c\x81",
  [0x2e50] = "\x4c\x82", [0x2e51] = "\x4c\x83", [0x2e52] = "\x4c\x84",
  [0x2e53] = "\x4c\x85", [0x2e54] = "\x4c\x86", [0x2e55] = "\x4c\x87",
  [0x2e56] = "\x4c\x88", [0x2e57] = "\x4c\x89", [0x2e58] = "\x4c\x8a",
  [0x2e59] = "\x4c\x8b", [0x2e5a] = "\x4c\x8c", [0x2e5b] = "\x4c\x8d",
  [0x2e5c] = "\x4c\x8e", [0x2e5d] = "\x4c\x8f", [0x2e5e] = "\x4c\x90",
  [0x2e5f] = "\x4c\x91", [0x2e60] = "\x4c\x92", [0x2e61] = "\x4c\x93",
  [0x2e62] = "\x4c\x94", [0x2e63] = "\x4c\x95", [0x2e64] = "\x4c\x96",
  [0x2e65] = "\x4c\x97", [0x2e66] = "\x4c\x98", [0x2e67] = "\x4c\x99",
  [0x2e68] = "\x4c\x9a", [0x2e69] = "\x4c\x9b", [0x2e6a] = "\x4c\x9c",
  [0x2e6b] = "\x4c\x9d", [0x2e6c] = "\x4c\x9e", [0x2e6d] = "\x4c\x9f",
  [0x2e6e] = "\x4c\xa0", [0x2e6f] = "\x4c\xa1", [0x2e70] = "\x4c\xa2",
  [0x2e71] = "\x4c\xa3", [0x2e72] = "\x4c\xa4", [0x2e73] = "\x4c\xa5",
  [0x2e74] = "\x4c\xa6", [0x2e75] = "\x4c\xa7", [0x2e76] = "\x4c\xa8",
  [0x2e77] = "\x4c\xa9", [0x2e78] = "\x4c\xaa", [0x2e79] = "\x4c\xab",
  [0x2e7a] = "\x4c\xac", [0x2e7b] = "\x4c\xad", [0x2e7c] = "\x4c\xae",
  [0x2e7d] = "\x4c\xaf", [0x2e7e] = "\x4c\xb0", [0x2e7f] = "\x4c\xb1",
  [0x2e80] = "\x4c\xb2", [0x2e81] = "\x4c\xb3", [0x2e82] = "\x4c\xb4",
  [0x2e83] = "\x4c\xb5", [0x2e84] = "\x4c\xb6", [0x2e85] = "\x4c\xb7",
  [0x2e86] = "\x4c\xb8", [0x2e87] = "\x4c\xb9", [0x2e88] = "\x4c\xba",
  [0x2e89] = "\x4c\xbb", [0x2e8a] = "\x4c\xbc", [0x2e8b] = "\x4c\xbd",
  [0x2e8c] = "\x4c\xbe", [0x2e8d] = "\x4c\xbf", [0x2e8e] = "\x4c\xc0",
  [0x2e8f] = "\x4c\xc1", [0x2e90] = "\x4c\xc2", [0x2e91] = "\x4c\xc3",
  [0x2e92] = "\x4c\xc4", [0x2e93] = "\x4c\xc5", [0x2e94] = "\x4c\xc6",
  [0x2e95] = "\x4c\xc7", [0x2e96] = "\x4c\xc8", [0x2e97] = "\x4c\xc9",
  [0x2e98] = "\x4c\xca", [0x2e99] = "\x4c\xcb", [0x2e9a] = "\x4c\xcc",
  [0x2e9b] = "\x4c\xcd", [0x2e9c] = "\x4c\xce", [0x2e9d] = "\x4c\xcf",
  [0x2e9e] = "\x4c\xd0", [0x2e9f] = "\x4c\xd1", [0x2ea0] = "\x4c\xd2",
  [0x2ea1] = "\x4c\xd3", [0x2ea2] = "\x4c\xd4", [0x2ea3] = "\x4c\xd5",
  [0x2ea4] = "\x4c\xd6", [0x2ea5] = "\x4c\xd7", [0x2ea6] = "\x4c\xd8",
  [0x2ea7] = "\x4c\xd9", [0x2ea8] = "\x4c\xda", [0x2ea9] = "\x4c\xdb",
  [0x2eaa] = "\x4c\xdc", [0x2eab] = "\x4c\xdd", [0x2eac] = "\x4c\xde",
  [0x2ead] = "\x4c\xdf", [0x2eae] = "\x4c\xe0", [0x2eaf] = "\x4c\xe1",
  [0x2eb0] = "\x4c\xe2", [0x2eb1] = "\x4c\xe3", [0x2eb2] = "\x4c\xe4",
  [0x2eb3] = "\x4c\xe5", [0x2eb4] = "\x4c\xe6", [0x2eb5] = "\x4c\xe7",
  [0x2eb6] = "\x4c\xe8", [0x2eb7] = "\x4c\xe9", [0x2eb8] = "\x4c\xea",
  [0x2eb9] = "\x4c\xeb", [0x2eba] = "\x4c\xec", [0x2ebb] = "\x4c\xed",
  [0x2ebc] = "\x4c\xee", [0x2ebd] = "\x4c\xef", [0x2ebe] = "\x4c\xf0",
  [0x2ebf] = "\x4c\xf1", [0x2ec0] = "\x4c\xf2", [0x2ec1] = "\x4c\xf3",
  [0x2ec2] = "\x4c\xf4", [0x2ec3] = "\x4c\xf5", [0x2ec4] = "\x4c\xf6",
  [0x2ec5] = "\x4c\xf7", [0x2ec6] = "\x4c\xf8", [0x2ec7] = "\x4c\xf9",
  [0x2ec8] = "\x4c\xfa", [0x2ec9] = "\x4c\xfb", [0x2eca] = "\x4c\xfc",
  [0x2ecb] = "\x4c\xfd", [0x2ecc] = "\x4c\xfe", [0x2ecd] = "\x4c\xff",
  [0x2ece] = "\x4d\x00", [0x2ecf] = "\x4d\x01", [0x2ed0] = "\x4d\x02",
  [0x2ed1] = "\x4d\x03", [0x2ed2] = "\x4d\x04", [0x2ed3] = "\x4d\x05",
  [0x2ed4] = "\x4d\x06", [0x2ed5] = "\x4d\x07", [0x2ed6] = "\x4d\x08",
  [0x2ed7] = "\x4d\x09", [0x2ed8] = "\x4d\x0a", [0x2ed9] = "\x4d\x0b",
  [0x2eda] = "\x4d\x0c", [0x2edb] = "\x4d\x0d", [0x2edc] = "\x4d\x0e",
  [0x2edd] = "\x4d\x0f", [0x2ede] = "\x4d\x10", [0x2edf] = "\x4d\x11",
  [0x2ee0] = "\x4d\x12", [0x2ee1] = "\x4d\x13", [0x2ee2] = "\x4d\x14",
  [0x2ee3] = "\x4d\x15", [0x2ee4] = "\x4d\x16", [0x2ee5] = "\x4d\x17",
  [0x2ee6] = "\x4d\x18", [0x2ee7] = "\x4d\x19", [0x2ee8] = "\x4d\x1a",
  [0x2ee9] = "\x4d\x1b", [0x2eea] = "\x4d\x1c", [0x2eeb] = "\x4d\x1d",
  [0x2eec] = "\x4d\x1e", [0x2eed] = "\x4d\x1f", [0x2eee] = "\x4d\x20",
  [0x2eef] = "\x4d\x21", [0x2ef0] = "\x4d\x22", [0x2ef1] = "\x4d\x23",
  [0x2ef2] = "\x4d\x24", [0x2ef3] = "\x4d\x25", [0x2ef4] = "\x4d\x26",
  [0x2ef5] = "\x4d\x27", [0x2ef6] = "\x4d\x28", [0x2ef7] = "\x4d\x29",
  [0x2ef8] = "\x4d\x2a", [0x2ef9] = "\x4d\x2b", [0x2efa] = "\x4d\x2c",
  [0x2efb] = "\x4d\x2d", [0x2efc] = "\x4d\x2e", [0x2efd] = "\x4d\x2f",
  [0x2efe] = "\x4d\x30", [0x2eff] = "\x4d\x31", [0x2f00] = "\x4d\x32",
  [0x2f01] = "\x4d\x33", [0x2f02] = "\x4d\x34", [0x2f03] = "\x4d\x35",
  [0x2f04] = "\x4d\x36", [0x2f05] = "\x4d\x37", [0x2f06] = "\x4d\x38",
  [0x2f07] = "\x4d\x39", [0x2f08] = "\x4d\x3a", [0x2f09] = "\x4d\x3b",
  [0x2f0a] = "\x4d\x3c", [0x2f0b] = "\x4d\x3d", [0x2f0c] = "\x4d\x3e",
  [0x2f0d] = "\x4d\x3f", [0x2f0e] = "\x4d\x40", [0x2f0f] = "\x4d\x41",
  [0x2f10] = "\x4d\x42", [0x2f11] = "\x4d\x43", [0x2f12] = "\x4d\x44",
  [0x2f13] = "\x4d\x45", [0x2f14] = "\x4d\x46", [0x2f15] = "\x4d\x47",
  [0x2f16] = "\x4d\x48", [0x2f17] = "\x4d\x49", [0x2f18] = "\x4d\x4a",
  [0x2f19] = "\x4d\x4b", [0x2f1a] = "\x4d\x4c", [0x2f1b] = "\x4d\x4d",
  [0x2f1c] = "\x4d\x4e", [0x2f1d] = "\x4d\x4f", [0x2f1e] = "\x4d\x50",
  [0x2f1f] = "\x4d\x51", [0x2f20] = "\x4d\x52", [0x2f21] = "\x4d\x53",
  [0x2f22] = "\x4d\x54", [0x2f23] = "\x4d\x55", [0x2f24] = "\x4d\x56",
  [0x2f25] = "\x4d\x57", [0x2f26] = "\x4d\x58", [0x2f27] = "\x4d\x59",
  [0x2f28] = "\x4d\x5a", [0x2f29] = "\x4d\x5b", [0x2f2a] = "\x4d\x5c",
  [0x2f2b] = "\x4d\x5d", [0x2f2c] = "\x4d\x5e", [0x2f2d] = "\x4d\x5f",
  [0x2f2e] = "\x4d\x60", [0x2f2f] = "\x4d\x61", [0x2f30] = "\x4d\x62",
  [0x2f31] = "\x4d\x63", [0x2f32] = "\x4d\x64", [0x2f33] = "\x4d\x65",
  [0x2f34] = "\x4d\x66", [0x2f35] = "\x4d\x67", [0x2f36] = "\x4d\x68",
  [0x2f37] = "\x4d\x69", [0x2f38] = "\x4d\x6a", [0x2f39] = "\x4d\x6b",
  [0x2f3a] = "\x4d\x6c", [0x2f3b] = "\x4d\x6d", [0x2f3c] = "\x4d\x6e",
  [0x2f3d] = "\x4d\x6f", [0x2f3e] = "\x4d\x70", [0x2f3f] = "\x4d\x71",
  [0x2f40] = "\x4d\x72", [0x2f41] = "\x4d\x73", [0x2f42] = "\x4d\x74",
  [0x2f43] = "\x4d\x75", [0x2f44] = "\x4d\x76", [0x2f45] = "\x4d\x77",
  [0x2f46] = "\x4d\x78", [0x2f47] = "\x4d\x79", [0x2f48] = "\x4d\x7a",
  [0x2f49] = "\x4d\x7b", [0x2f4a] = "\x4d\x7c", [0x2f4b] = "\x4d\x7d",
  [0x2f4c] = "\x4d\x7e", [0x2f4d] = "\x4d\x7f", [0x2f4e] = "\x4d\x80",
  [0x2f4f] = "\x4d\x81", [0x2f50] = "\x4d\x82", [0x2f51] = "\x4d\x83",
  [0x2f52] = "\x4d\x84", [0x2f53] = "\x4d\x85", [0x2f54] = "\x4d\x86",
  [0x2f55] = "\x4d\x87", [0x2f56] = "\x4d\x88", [0x2f57] = "\x4d\x89",
  [0x2f58] = "\x4d\x8a", [0x2f59] = "\x4d\x8b", [0x2f5a] = "\x4d\x8c",
  [0x2f5b] = "\x4d\x8d", [0x2f5c] = "\x4d\x8e", [0x2f5d] = "\x4d\x8f",
  [0x2f5e] = "\x4d\x90", [0x2f5f] = "\x4d\x91", [0x2f60] = "\x4d\x92",
  [0x2f61] = "\x4d\x93", [0x2f62] = "\x4d\x94", [0x2f63] = "\x4d\x95",
  [0x2f64] = "\x4d\x96", [0x2f65] = "\x4d\x97", [0x2f66] = "\x4d\x98",
  [0x2f67] = "\x4d\x99", [0x2f68] = "\x4d\x9a", [0x2f69] = "\x4d\x9b",
  [0x2f6a] = "\x4d\x9c", [0x2f6b] = "\x4d\x9d", [0x2f6c] = "\x4d\x9e",
  [0x2f6d] = "\x4d\x9f", [0x2f6e] = "\x4d\xa0", [0x2f6f] = "\x4d\xa1",
  [0x2f70] = "\xa9\x8a", [0x2f71] = "\xa9\x8b", [0x2f72] = "\xa9\x8c",
  [0x2f73] = "\xa9\x8d", [0x2f74] = "\xa9\x8e", [0x2f75] = "\xa9\x8f",
  [0x2f76] = "\xa9\x90", [0x2f77] = "\xa9\x91", [0x2f78] = "\xa9\x92",
  [0x2f79] = "\xa9\x93", [0x2f7a] = "\xa9\x94", [0x2f7b] = "\xa9\x95",
  [0x2f7c] = "\x4d\xa2", [0x2f7d] = "\x4d\xa3", [0x2f7e] = "\x4d\xa4",
  [0x2f7f] = "\x4d\xa5", [0x2f80] = "\xa1\xa1", [0x2f81] = "\xa1\xa2",
  [0x2f82] = "\xa1\xa3", [0x2f83] = "\xa1\xa8", [0x2f84] = "\x4d\xa6",
  [0x2f85] = "\xa1\xa9", [0x2f86] = "\xa9\x65", [0x2f87] = "\xa9\x96",
  [0x2f88] = "\xa1\xb4", [0x2f89] = "\xa1\xb5", [0x2f8a] = "\xa1\xb6",
  [0x2f8b] = "\xa1\xb7", [0x2f8c] = "\xa1\xb8", [0x2f8d] = "\xa1\xb9",
  [0x2f8e] = "\xa1\xba", [0x2f8f] = "\xa1\xbb", [0x2f90] = "\xa1\xbe",
  [0x2f91] = "\xa1\xbf", [0x2f92] = "\xa8\x93", [0x2f93] = "\xa1\xfe",
  [0x2f94] = "\xa1\xb2", [0x2f95] = "\xa1\xb3", [0x2f96] = "\xa1\xbc",
  [0x2f97] = "\xa1\xbd", [0x2f98] = "\x4d\xa7", [0x2f99] = "\x4d\xa8",
  [0x2f9a] = "\x4d\xa9", [0x2f9b] = "\x4d\xaa", [0x2f9c] = "\x4d\xab",
  [0x2f9d] = "\xa8\x94", [0x2f9e] = "\xa8\x95", [0x2f9f] = "\x4d\xac",
  [0x2fa0] = "\x4d\xad", [0x2fa1] = "\xa9\x40", [0x2fa2] = "\xa9\x41",
  [0x2fa3] = "\xa9\x42", [0x2fa4] = "\xa9\x43", [0x2fa5] = "\xa9\x44",
  [0x2fa6] = "\xa9\x45", [0x2fa7] = "\xa9\x46", [0x2fa8] = "\xa9\x47",
  [0x2fa9] = "\xa9\x48", [0x2faa] = "\x4d\xae", [0x2fab] = "\x4d\xaf",
  [0x2fac] = "\x4d\xb0", [0x2fad] = "\x4d\xb1", [0x2fae] = "\x4d\xb2",
  [0x2faf] = "\x4d\xb3", [0x2fb0] = "\x4d\xb4", [0x2fb1] = "\x4d\xb5",
  [0x2fb2] = "\x4d\xb6", [0x2fb3] = "\x4d\xb7", [0x2fb4] = "\x4d\xb8",
  [0x2fb5] = "\x4d\xb9", [0x2fb6] = "\x4d\xba", [0x2fb7] = "\x4d\xbb",
  [0x2fb8] = "\x4d\xbc", [0x2fb9] = "\x4d\xbd", [0x2fba] = "\x4d\xbe",
  [0x2fbb] = "\x4d\xbf", [0x2fbc] = "\x4d\xc0", [0x2fbd] = "\x4d\xc1",
  [0x2fbe] = "\xa9\x89", [0x2fbf] = "\x4d\xc2", [0x2fc0] = "\x4d\xc3",
  [0x2fc1] = "\xa4\xa1", [0x2fc2] = "\xa4\xa2", [0x2fc3] = "\xa4\xa3",
  [0x2fc4] = "\xa4\xa4", [0x2fc5] = "\xa4\xa5", [0x2fc6] = "\xa4\xa6",
  [0x2fc7] = "\xa4\xa7", [0x2fc8] = "\xa4\xa8", [0x2fc9] = "\xa4\xa9",
  [0x2fca] = "\xa4\xaa", [0x2fcb] = "\xa4\xab", [0x2fcc] = "\xa4\xac",
  [0x2fcd] = "\xa4\xad", [0x2fce] = "\xa4\xae", [0x2fcf] = "\xa4\xaf",
  [0x2fd0] = "\xa4\xb0", [0x2fd1] = "\xa4\xb1", [0x2fd2] = "\xa4\xb2",
  [0x2fd3] = "\xa4\xb3", [0x2fd4] = "\xa4\xb4", [0x2fd5] = "\xa4\xb5",
  [0x2fd6] = "\xa4\xb6", [0x2fd7] = "\xa4\xb7", [0x2fd8] = "\xa4\xb8",
  [0x2fd9] = "\xa4\xb9", [0x2fda] = "\xa4\xba", [0x2fdb] = "\xa4\xbb",
  [0x2fdc] = "\xa4\xbc", [0x2fdd] = "\xa4\xbd", [0x2fde] = "\xa4\xbe",
  [0x2fdf] = "\xa4\xbf", [0x2fe0] = "\xa4\xc0", [0x2fe1] = "\xa4\xc1",
  [0x2fe2] = "\xa4\xc2", [0x2fe3] = "\xa4\xc3", [0x2fe4] = "\xa4\xc4",
  [0x2fe5] = "\xa4\xc5", [0x2fe6] = "\xa4\xc6", [0x2fe7] = "\xa4\xc7",
  [0x2fe8] = "\xa4\xc8", [0x2fe9] = "\xa4\xc9", [0x2fea] = "\xa4\xca",
  [0x2feb] = "\xa4\xcb", [0x2fec] = "\xa4\xcc", [0x2fed] = "\xa4\xcd",
  [0x2fee] = "\xa4\xce", [0x2fef] = "\xa4\xcf", [0x2ff0] = "\xa4\xd0",
  [0x2ff1] = "\xa4\xd1", [0x2ff2] = "\xa4\xd2", [0x2ff3] = "\xa4\xd3",
  [0x2ff4] = "\xa4\xd4", [0x2ff5] = "\xa4\xd5", [0x2ff6] = "\xa4\xd6",
  [0x2ff7] = "\xa4\xd7", [0x2ff8] = "\xa4\xd8", [0x2ff9] = "\xa4\xd9",
  [0x2ffa] = "\xa4\xda", [0x2ffb] = "\xa4\xdb", [0x2ffc] = "\xa4\xdc",
  [0x2ffd] = "\xa4\xdd", [0x2ffe] = "\xa4\xde", [0x2fff] = "\xa4\xdf",
  [0x3000] = "\xa4\xe0", [0x3001] = "\xa4\xe1", [0x3002] = "\xa4\xe2",
  [0x3003] = "\xa4\xe3", [0x3004] = "\xa4\xe4", [0x3005] = "\xa4\xe5",
  [0x3006] = "\xa4\xe6", [0x3007] = "\xa4\xe7", [0x3008] = "\xa4\xe8",
  [0x3009] = "\xa4\xe9", [0x300a] = "\xa4\xea", [0x300b] = "\xa4\xeb",
  [0x300c] = "\xa4\xec", [0x300d] = "\xa4\xed", [0x300e] = "\xa4\xee",
  [0x300f] = "\xa4\xef", [0x3010] = "\xa4\xf0", [0x3011] = "\xa4\xf1",
  [0x3012] = "\xa4\xf2", [0x3013] = "\xa4\xf3", [0x3014] = "\x4d\xc4",
  [0x3015] = "\x4d\xc5", [0x3016] = "\x4d\xc6", [0x3017] = "\x4d\xc7",
  [0x3018] = "\x4d\xc8", [0x3019] = "\x4d\xc9", [0x301a] = "\x4d\xca",
  [0x301b] = "\xa9\x61", [0x301c] = "\xa9\x62", [0x301d] = "\xa9\x66",
  [0x301e] = "\xa9\x67", [0x301f] = "\x4d\xcb", [0x3020] = "\x4d\xcc",
  [0x3021] = "\xa5\xa1", [0x3022] = "\xa5\xa2", [0x3023] = "\xa5\xa3",
  [0x3024] = "\xa5\xa4", [0x3025] = "\xa5\xa5", [0x3026] = "\xa5\xa6",
  [0x3027] = "\xa5\xa7", [0x3028] = "\xa5\xa8", [0x3029] = "\xa5\xa9",
  [0x302a] = "\xa5\xaa", [0x302b] = "\xa5\xab", [0x302c] = "\xa5\xac",
  [0x302d] = "\xa5\xad", [0x302e] = "\xa5\xae", [0x302f] = "\xa5\xaf",
  [0x3030] = "\xa5\xb0", [0x3031] = "\xa5\xb1", [0x3032] = "\xa5\xb2",
  [0x3033] = "\xa5\xb3", [0x3034] = "\xa5\xb4", [0x3035] = "\xa5\xb5",
  [0x3036] = "\xa5\xb6", [0x3037] = "\xa5\xb7", [0x3038] = "\xa5\xb8",
  [0x3039] = "\xa5\xb9", [0x303a] = "\xa5\xba", [0x303b] = "\xa5\xbb",
  [0x303c] = "\xa5\xbc", [0x303d] = "\xa5\xbd", [0x303e] = "\xa5\xbe",
  [0x303f] = "\xa5\xbf", [0x3040] = "\xa5\xc0", [0x3041] = "\xa5\xc1",
  [0x3042] = "\xa5\xc2", [0x3043] = "\xa5\xc3", [0x3044] = "\xa5\xc4",
  [0x3045] = "\xa5\xc5", [0x3046] = "\xa5\xc6", [0x3047] = "\xa5\xc7",
  [0x3048] = "\xa5\xc8", [0x3049] = "\xa5\xc9", [0x304a] = "\xa5\xca",
  [0x304b] = "\xa5\xcb", [0x304c] = "\xa5\xcc", [0x304d] = "\xa5\xcd",
  [0x304e] = "\xa5\xce", [0x304f] = "\xa5\xcf", [0x3050] = "\xa5\xd0",
  [0x3051] = "\xa5\xd1", [0x3052] = "\xa5\xd2", [0x3053] = "\xa5\xd3",
  [0x3054] = "\xa5\xd4", [0x3055] = "\xa5\xd5", [0x3056] = "\xa5\xd6",
  [0x3057] = "\xa5\xd7", [0x3058] = "\xa5\xd8", [0x3059] = "\xa5\xd9",
  [0x305a] = "\xa5\xda", [0x305b] = "\xa5\xdb", [0x305c] = "\xa5\xdc",
  [0x305d] = "\xa5\xdd", [0x305e] = "\xa5\xde", [0x305f] = "\xa5\xdf",
  [0x3060] = "\xa5\xe0", [0x3061] = "\xa5\xe1", [0x3062] = "\xa5\xe2",
  [0x3063] = "\xa5\xe3", [0x3064] = "\xa5\xe4", [0x3065] = "\xa5\xe5",
  [0x3066] = "\xa5\xe6", [0x3067] = "\xa5\xe7", [0x3068] = "\xa5\xe8",
  [0x3069] = "\xa5\xe9", [0x306a] = "\xa5\xea", [0x306b] = "\xa5\xeb",
  [0x306c] = "\xa5\xec", [0x306d] = "\xa5\xed", [0x306e] = "\xa5\xee",
  [0x306f] = "\xa5\xef", [0x3070] = "\xa5\xf0", [0x3071] = "\xa5\xf1",
  [0x3072] = "\xa5\xf2", [0x3073] = "\xa5\xf3", [0x3074] = "\xa5\xf4",
  [0x3075] = "\xa5\xf5", [0x3076] = "\xa5\xf6", [0x3077] = "\x4d\xcd",
  [0x3078] = "\x4d\xce", [0x3079] = "\x4d\xcf", [0x307a] = "\x4d\xd0",
  [0x307b] = "\x4d\xd1", [0x307c] = "\xa9\x60", [0x307d] = "\xa9\x63",
  [0x307e] = "\xa9\x64", [0x307f] = "\x4d\xd2", [0x3080] = "\x4d\xd3",
  [0x3081] = "\x4d\xd4", [0x3082] = "\x4d\xd5", [0x3083] = "\x4d\xd6",
  [0x3084] = "\x4d\xd7", [0x3085] = "\xa8\xc5", [0x3086] = "\xa8\xc6",
  [0x3087] = "\xa8\xc7", [0x3088] = "\xa8\xc8", [0x3089] = "\xa8\xc9",
  [0x308a] = "\xa8\xca", [0x308b] = "\xa8\xcb", [0x308c] = "\xa8\xcc",
  [