/*
 * This file is part of LibEuFin.
 * Copyright (C) 2025 Taler Systems S.A.

 * LibEuFin is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation; either version 3, or
 * (at your option) any later version.

 * LibEuFin is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General
 * Public License for more details.

 * You should have received a copy of the GNU Affero General Public
 * License along with LibEuFin; see the file COPYING.  If not, see
 * <http://www.gnu.org/licenses/>
 */

import org.w3c.dom.Document
import com.github.ajalt.clikt.core.CliktCommand
import com.github.ajalt.clikt.testing.test
import kotlinx.coroutines.runBlocking
import io.ktor.http.*
import io.ktor.http.content.*
import org.junit.Test
import tech.libeufin.nexus.cli.LibeufinNexus
import tech.libeufin.nexus.*
import tech.libeufin.common.*
import tech.libeufin.common.test.*
import tech.libeufin.common.crypto.CryptoUtil
import tech.libeufin.ebics.test.*
import tech.libeufin.ebics.*
import kotlin.io.path.*
import kotlin.test.*
import java.time.LocalDate

@OptIn(kotlin.io.path.ExperimentalPathApi::class)
class EbicsTest {
    private val nexusCmd = LibeufinNexus()
    private val bank = EbicsState()
    private val args = "-L TRACE -c conf/fetch.conf"

    private fun ebicsSetup() {
        // Reset current keys
        val dir = Path("/tmp/ebics-test")
        dir.deleteRecursively()
        dir.createDirectories()

        // Set setup mock
        setMock(sequence {
            yield(bank::hev)
            yield(bank::ini)
            yield(bank::hia)
            yield(bank::hpb)
            yield(bank::hkd)
            yield(bank::receiptOk)
        })

        // Run setup
        nexusCmd.succeed("ebics-setup $args --auto-accept-keys")
    }

    @Test
    fun setup() {
        ebicsSetup()
    }

    @Test
    fun fetchPinnedDate() = setup { db, _ ->
        ebicsSetup()

        suspend fun resetCheckpoint() {
            db.serializable("DELETE FROM kv WHERE key=?") {
                bind(CHECKPOINT_KEY)
                executeUpdate()
            }
        }

        // Default transient
        setMock(sequence {
            yield(bank::haa)
            yield(bank::receiptOk)
            yield(bank::btdNoData)
        })
        nexusCmd.succeed("ebics-fetch $args --transient")

        // Pinned transient
        setMock(sequence {
            yield(bank::haa)
            yield(bank::receiptOk)
            yield(bank::btdNoDataPinned)
        })
        nexusCmd.succeed("ebics-fetch $args --transient --pinned-start 2024-06-05")

        // Init checkpoint
        setMock(sequence {
            yield(bank::hkd)
            yield(bank::receiptOk)
            yield(bank::btdNoData)
        })
        nexusCmd.succeed("ebics-fetch $args --transient --checkpoint")

        // Default checkpoint
        setMock(sequence {
            yield(bank::hkd)
            yield(bank::receiptOk)
            yield(bank::btdNoDataNow)
        })
        nexusCmd.succeed("ebics-fetch $args --transient --checkpoint")

        // Pinned checkpoint
        setMock(sequence {
            yield(bank::hkd)
            yield(bank::receiptOk)
            yield(bank::btdNoDataPinned)
        })
        nexusCmd.succeed("ebics-fetch $args --transient --checkpoint --pinned-start 2024-06-05")

        // Reset checkpoint
        resetCheckpoint()
        setMock(sequence {
            yield(bank::hkd)
            yield(bank::receiptOk)
            yield(bank::btdNoData)
        })
        nexusCmd.succeed("ebics-fetch $args --transient --checkpoint")

        // Reset checkpoint pinned
        resetCheckpoint()
        setMock(sequence {
            yield(bank::hkd)
            yield(bank::receiptOk)
            yield(bank::btdNoDataPinned)
        })
        nexusCmd.succeed("ebics-fetch $args --transient --checkpoint --pinned-start 2024-06-05")
    }

    @Test
    fun closePendingTransaction() = setup { db, _ ->
        ebicsSetup()

        // Failure before first segment
        setMock(sequence {
            // Failure to perform download
            yield(bank::failure)
            // Then continue
            yield(bank::haa)
            yield(bank::receiptOk)
            yield(bank::btdNoData)
        })
        nexusCmd.fail("ebics-fetch $args --transient")
        nexusCmd.succeed("ebics-fetch $args --transient")

        // Compliant server
        setMock(sequence {
            yield(bank::haa)
            yield(bank::receiptOk)
            // Failure to perform download
            yield(bank::initializeTx)
            yield(bank::failure)
            // Retry fail once
            yield(bank::failure)
            // Retry fail twice
            yield(bank::failure)
            // Retry succeed
            yield(bank::receiptErr)
            // Then continue
            yield(bank::haa)
            yield(bank::receiptOk)
            yield(bank::btdNoData)
        })
        nexusCmd.fail("ebics-fetch $args --transient")
        nexusCmd.fail("ebics-fetch $args --transient")
        nexusCmd.fail("ebics-fetch $args --transient")
        nexusCmd.succeed("ebics-fetch $args --transient")

        // Non compliant server
        setMock(sequence {
            yield(bank::haa)
            yield(bank::receiptOk)
            // Failure to perform download
            yield(bank::initializeTx)
            yield(bank::badRequest)
            // Retry fail
            yield(bank::failure)

            // Retry succeed
            yield(bank::badRequest)
            // Then continue
            yield(bank::haa)
            yield(bank::receiptOk)
            yield(bank::btdNoData)
        })
        nexusCmd.fail("ebics-fetch $args --transient")
        nexusCmd.fail("ebics-fetch $args --transient")
        nexusCmd.succeed("ebics-fetch $args --transient")
    }
}