/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)es_interactive.c	1.5 95/11/15 Sun Microsystems"

#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <arpa/inet.h>
#include <netinet/ip.h>
#include <netdb.h>
#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/xv_xrect.h>
#include <xview/notice.h>
#include "skiptool_ui.h"

#include <skip_proto.h>
#include <skip_types.h>
#include <skip_ioctl.h>
#include <skip_lib.h>

#include "es_glue.h"

#ifdef SYSV
#include <sys/systeminfo.h>
#endif

typedef struct listentry {
	struct listentry	*next;
	struct in_addr		addr;
} listentry_t;

#define	MAXMSGS			10
const int			inaddrsz = 4;
static int			skip_fd = -1;
static int			add_automatically = 0;
static listentry_t		*informed_hosts = NULL;

static char			*versions[]  = {
					"cleartext",
					"SKIP (SunScreen compatibility)",
					"SKIP V2"
				};
/*
 * cancel any further popup messages
 */
static void
cancel_interactive_mode()
{
	extern void		access_update();

	(void) xv_set(Skiptool_base_window->access_mode,
			PANEL_VALUE, NoAccess,
			NULL);
	access_update();
}

/*
 * popup a security warning message
 */
void
acl_warning(Panel_item owner, char *msgs[])
{
	Panel				panel;
	Xv_notice			notice; 
	int				notice_stat;

	if (owner == NULL) {
		panel = Skiptool_base_window->base_window;
	} else {
		panel = xv_get(owner, PANEL_PARENT_PANEL);
	}
	notice = xv_create(panel,
			NOTICE,
			XV_SHOW, TRUE,
			NOTICE_LOCK_SCREEN, FALSE,
			NOTICE_MESSAGE_STRINGS_ARRAY_PTR, msgs,
			NOTICE_BUTTON, "Confirm", NOTICE_NO,
			NOTICE_BUTTON, "Cancel further messages", NOTICE_YES,
			NOTICE_STATUS, &notice_stat,
			NULL);
	xv_destroy_safe(notice);
	if (notice_stat == NOTICE_YES) {
		cancel_interactive_mode();
	}
}

/*
 * prompt the user to accept a host or not
 */
static int
ask_user(Panel_item owner, char *msgs[])
{
	extern skiptool_base_window_objects	*Skiptool_base_window;
	const int			notice_cancel = 3;
	Panel				panel;
	Xv_notice			notice; 
	int				notice_stat;

	if (owner == NULL) {
		panel = Skiptool_base_window->base_window;
	} else {
		panel = xv_get(owner, PANEL_PARENT_PANEL);
	}
	(void) xv_set(Skiptool_base_window->base_window,
			FRAME_CLOSED, FALSE,
			NULL);
	notice = xv_create(Skiptool_base_window->base_window, 
			NOTICE,
			XV_SHOW, TRUE,
			NOTICE_LOCK_SCREEN, FALSE,
			NOTICE_MESSAGE_STRINGS_ARRAY_PTR, msgs,
			NOTICE_BUTTON, "Allow Access", NOTICE_YES,
			NOTICE_BUTTON, "No Access", NOTICE_NO,
			NOTICE_BUTTON, "Cancel further messages", notice_cancel,
			NOTICE_STATUS, &notice_stat,
			NULL);
	xv_destroy_safe(notice);
	if (notice_stat == notice_cancel) {
		cancel_interactive_mode();
		notice_stat = NOTICE_NO;
	}
	return (notice_stat);
}

/*
 * keep track of systems we have already informed the user about
 */
static boolean_t
already_notified(struct in_addr addr)
{
	listentry_t	*lp;

	for (lp = informed_hosts; lp; lp = lp->next) {
		if (memcmp((const char *)&addr, 
				(const char *)&lp->addr, inaddrsz) == 0) {
			return (B_TRUE);
		}
	}
	lp = (listentry_t *)calloc(1, sizeof(*lp));
	if (lp == NULL) {
		return (B_TRUE);
	}
	memcpy((char *)&lp->addr, (char *)&addr, inaddrsz);
	lp->next = informed_hosts;
	informed_hosts = lp;

	return (B_FALSE);
}

/*
 * delete entry from informed list
 */
void
inform_list_delete(struct in_addr addr)
{
	listentry_t	*lp, **prevlp;

	for (prevlp = &informed_hosts; (lp = *prevlp); prevlp = &lp->next) {
		if (memcmp((const char *)&addr,
				(const char *)&lp->addr, inaddrsz) == 0) {
			break;
		}
	}
	if (lp == NULL) {
		return;
	}
	*prevlp = lp->next;

	cfree(lp, 1, sizeof(*lp));
}

/*
 * free informed list
 */
static void
inform_list_free()
{
	listentry_t	*lp, *nlp;

	for (lp = informed_hosts; lp; lp = nlp) {
		nlp = lp->next;
		cfree(lp, 1, sizeof(*lp));
	}
	informed_hosts = NULL;
}

/*
 * Build a message array
 */
static void
add_msg(char *msgs[], char *msg)
{
	register int	i;

	for (i = 0; i < MAXMSGS; i++) {
		if (msgs[i] == NULL) {
			msgs[i] = strdup(msg);
			break;
		}
	}
}

/*
 * Destroy a message array
 */
static void
destroy_msgs(char *msgs[])
{
	register int	i;

	for (i = 0; i < MAXMSGS; i++) {
		if (msgs[i] != NULL) {
			free(msgs[i]);
			msgs[i] = NULL;
		}
	}
}

/*
 * handle incoming packet from an unknown destination
 */
static void
unknown_src(skip_es_notification_t	*skipmsgp)
{
	boolean_t		secure, is_bad = B_FALSE;
	skip_param_t		*params = &skipmsgp->params;
	struct ip		*ip;
	char			*kij_algname, *kp_algname, *versionstr;
	char			*r_mkeyid, *s_mkeyid;
	char			*msgs[MAXMSGS] = { NULL };
	char			msg[STRSZ], s[STRSZ];
	struct hostent		*hp;

	ip = (struct ip *) &skipmsgp->iphdr;

	switch (params->version) {
	case SKIP_NONE:
		versionstr = "in the clear";
		break;

	case SKIP_V1:
	case SKIP_V2:
		sprintf(s, "with %s using\n", versions[params->version]);
		versionstr = s;
		break;
	default:
		sprintf(s, "with bad version #%d\n", params->version);
		versionstr = s;
	}

	secure = (params->version != SKIP_NONE);

	hp = gethostbyaddr((char *) &ip->ip_src, inaddrsz, AF_INET);
	if (hp) {
		sprintf(msg, "\"%s\" (%s) is trying to connect %s",
			hp->h_name, inet_ntoa(ip->ip_src), versionstr);
	} else {
		sprintf(msg, "%s (unknown) is trying to connect %s",
			inet_ntoa(ip->ip_src), versionstr);
	}
	add_msg(msgs, msg);

	if (secure) {
		/*
		 * check we could support this host
		 */
		kij_algname = skip_kij_alg_to_name(params->kij_alg,
							params->version);
		kp_algname = skip_kp_alg_to_name(params->kp_alg,
							params->version);

		if (!skip_supported_kij_alg(params->kij_alg, params->version)) {
			sprintf(msg, "and unsupported key encryption "
				"algorithm #%d (%s)",
				params->kij_alg, kij_algname);

			add_msg(msgs, msg);
			is_bad = B_TRUE;
		}
		if (!skip_supported_kp_alg(params->kp_alg, params->version)) {
			sprintf(msg, "and unsupported traffic encryption "
				"algorithm #%d (%s)",
				params->kp_alg, kp_algname);
			add_msg(msgs, msg);
			is_bad = B_TRUE;
		}
		if ((int) params->r_nsid >= skip_max_nsids) {
			sprintf(msg, "and unsupported sender NSID #d",
				params->r_nsid);
			add_msg(msgs, msg);
			is_bad = B_TRUE;
		}

		if ((int) params->s_nsid >= skip_max_nsids) {
			sprintf(msg, "and unsupported receiver NSID #d ",
				params->s_nsid);
			add_msg(msgs, msg);
			is_bad = B_TRUE;
		}

		if (is_bad) {
			acl_warning(NULL, msgs);
			destroy_msgs(msgs);
			return;
		}

		/*
		 * build algorithm/key information messages
		 */
		sprintf(msg, "%s (key encryption) %s (traffic encryption)",
						kij_algname, kp_algname); 
		add_msg(msgs, msg);

		(void) skip_keyid_to_s(&params->r_mkeyid, params->r_nsid, s);
		sprintf(msg, "- its key id is %s %s",
				skip_nsids[params->r_nsid].name,
				s);
		add_msg(msgs, msg);

		sprintf(msg, "- local key id specified as %s",
				skip_nsids[params->s_nsid].name);
		add_msg(msgs, msg);
	}
	if (add_automatically || ask_user(NULL, msgs) == NOTICE_YES) {
		if (skip_allow_host(ifname, params)) {
			sprintf(msg, "Failed to add host %s (%s)",
				inet_ntoa(ip->ip_src), skip_errmsg);
			alert(NULL, msg, B_TRUE);
		} else {
			acl_build();
		}
	}
	destroy_msgs(msgs);
}

/*
 * handle incoming packet which doesn't match ACL
 */
static void
bad_params(skip_es_notification_t *skipmsgp)
{
	boolean_t		secure;
	skip_param_t		*in = &skipmsgp->params;
	skip_param_t		acl_params, *ref = &acl_params;
	struct ip		*ip;
	char			in_str[STRSZ], ref_str[STRSZ];
	char			*msgs[MAXMSGS] = { NULL };
	char			msg[STRSZ], s[STRSZ];
	struct hostent		*hp;

	ip = (struct ip *) &skipmsgp->iphdr;

	(void) skip_addr_to_host(&ip->ip_src, s);

	ref->ip_addr = ip->ip_src;

	if (skip_get_host(ifname, ref) != 0) {
		sprintf(msg, "Failed to get ACL entry for host %s (%s)",
			s, inet_ntoa(ip->ip_src), skip_errmsg);
		alert(NULL, msg, B_TRUE);
		return;
	}

	sprintf(msg, "\"%s\" (%s) is trying to connect using",
					s, inet_ntoa(ip->ip_src));

	add_msg(msgs, msg);

	/*
	 * check SKIP version
	 */
	if (in->version != ref->version) {
		if (((int) in->version >= SKIP_NONE) &&
			((int) in->version <= SKIP_V2)) {
			strcpy(in_str, versions[in->version]);
		} else {
			sprintf(s, "<bad SKIP version %d>", in->version);
		}

		if (((int) ref->version >= SKIP_NONE) &&
					((int) ref->version <= SKIP_V2)) {
			strcpy(ref_str, versions[ref->version]);
		} else {
			sprintf(ref_str, "<bad SKIP version %d>", in->version);
		}
		sprintf(msg, "%s instead of %s", in_str, ref_str);
		add_msg(msgs, msg);
		acl_warning(NULL, msgs);
		destroy_msgs(msgs);
		return;
	}

	/*
	 * check remote NSID
	 */
	if (in->r_nsid != ref->r_nsid) {
		if ((int) in->r_nsid >= skip_max_nsids) {
			sprintf(in_str, "remote NSID #%d "
				"<unknown>", in->r_nsid);
		} else {
			sprintf(in_str, "remote NSID #%d " "(%s)",
				in->r_nsid, skip_nsids[in->r_nsid].name);
		}
		if ((int) ref->r_nsid >= skip_max_nsids) {
			sprintf(ref_str, "#%d <unknown>",
				ref->r_nsid);
		} else {
			sprintf(ref_str, "#%d (%s)",
				ref->r_nsid, skip_nsids[ref->r_nsid].name);
		}
		sprintf(msg, "%s instead of %s", in_str, ref_str);
		add_msg(msgs, msg);
	} else {
		/*
		 * remote NSID is correct, check key value
		 */
		if (memcmp(in->r_mkeyid.buf, ref->r_mkeyid.buf,
							in->r_mkeyid.len)) {
			/*
			 * Key values do not match.  Check the key size is
			 * valid.
			 */
			if (skip_keyid_to_s(&in->r_mkeyid, in->r_nsid, in_str)
									< 0) {
				sprintf(in_str, "<bad key of length %d>",
						in->r_mkeyid.len);
			}
			if (skip_keyid_to_s(&ref->r_mkeyid, ref->r_nsid,
								ref_str) < 0) {
				sprintf(ref_str, "<bad key of length %d>",
						ref->r_mkeyid.len);
			}
			sprintf(msg, "with remote key %s", in_str);
			add_msg(msgs, msg);
			sprintf(msg, "instead of %s", ref_str);
			add_msg(msgs, msg);
		}
	}

	/*
	 * check local NSID
	 */
	if (in->s_nsid != ref->s_nsid) {
		if ((int) in->s_nsid >= skip_max_nsids) {
			sprintf(in_str, "local NSID #%d "
				"<unknown>", in->s_nsid);
		} else {
			sprintf(in_str, "local NSID #%d (%s)",
				in->s_nsid, skip_nsids[in->s_nsid].name);
		}
		if ((int) ref->s_nsid >= skip_max_nsids) {
			sprintf(ref_str, "#%d <unknown>",
				ref->s_nsid);
		} else {
			sprintf(ref_str, "#%d (%s)",
				ref->s_nsid, skip_nsids[ref->s_nsid].name);
		}
		sprintf(msg, "%s instead of %s", in_str, ref_str);
		add_msg(msgs, msg);
	}

	/*
	 * check Kij algs
	 */
	if (in->kij_alg != ref->kij_alg) {
		sprintf(msg, "key encryption algorithm #%d (%s) "
				"instead of #%d (%s)",
			in->kij_alg,
			skip_kij_alg_to_name(in->kij_alg, in->version),
			ref->kij_alg,
			skip_kij_alg_to_name(ref->kij_alg, ref->version));
		add_msg(msgs, msg);
	}
	/*
	 * check Kp algs
	 */
	if (in->kp_alg != ref->kp_alg) {
		sprintf(msg, "traffic encryption algorithm #%d (%s) "
				"instead of #%d (%s)",
			in->kp_alg,
			skip_kp_alg_to_name(in->kp_alg, in->version),
			ref->kp_alg,
			skip_kp_alg_to_name(ref->kp_alg, ref->version));
		add_msg(msgs, msg);
	}
	/*
	 * check MAC algs
	 */
	if (in->mac_alg != ref->mac_alg) {
		sprintf(msg, "authentication algorithm #%d "
				"instead of #%d",
			in->mac_alg,
			ref->mac_alg);
		add_msg(msgs, msg);
	}
	/*
	 * check compression algs
	 */
	if (in->comp_alg != ref->comp_alg) {
		sprintf(msg, "compression algorithm #%d "
				"instead of #%d",
			in->comp_alg,
			ref->comp_alg);
		add_msg(msgs, msg);
	}

	acl_warning(NULL, msgs);
	destroy_msgs(msgs);
}

/*
 * handle outgoing packet to an unknown destination
 */
static void
unknown_dst(skip_es_notification_t	*skipmsgp)
{
	struct ip		*ip;
	char			*msgs[MAXMSGS] = { NULL }, msg[STRSZ],*hostname;
	struct hostent		*hp;

	ip = (struct ip *) &skipmsgp->iphdr;

	hp = gethostbyaddr((char *) &ip->ip_dst, inaddrsz, AF_INET);
	if (hp) {
		sprintf(msg,
			"This system is trying to connect to \"%s\" "
			"(%s)", hp->h_name, inet_ntoa(ip->ip_dst));
		hostname = hp->h_name;
	} else {
		sprintf(msg,
			"This system is trying to connect to \"%s\" "
			"(unknown)", inet_ntoa(ip->ip_dst));
		hostname = inet_ntoa(ip->ip_dst);
	}
	add_msg(msgs, msg);

	if (add_automatically || ask_user(NULL, msgs) == NOTICE_YES) {
		if (skip_allow_host(ifname, &skipmsgp->params)) {
			sprintf(msg, "Failed to add host %s (%s)",
				hostname, skip_errmsg);
			alert(NULL, msg, B_TRUE);
		} else {
			acl_build();
		}
	}
	destroy_msgs(msgs);
}

static Notify_value
skip_msg(Notify_client client, int skip_fd)
{
	boolean_t		secure;
	int			rc;
	skip_es_notification_t	skipmsg;
	skip_param_t		*params = &skipmsg.params;
	struct ip		*ip;
	char			*h, msg[STRSZ], *msgs[MAXMSGS] = { NULL };
	struct hostent		*hp;

	rc = skip_get_message(skip_fd, &skipmsg);
	if (rc < 0) {
		perror("skip_get_message failed");
		return (NOTIFY_DONE);
	}

	ip = (struct ip *) &skipmsg.iphdr;

	switch(skipmsg.what) {

	case SkipUnknownSrc:
		if (already_notified(ip->ip_src)) {
			break;
		}
		unknown_src(&skipmsg);
		break;

	case SkipUnknownDst:
		if (already_notified(ip->ip_dst)) {
			break;
		}
		unknown_dst(&skipmsg);
		break;

	case SkipBadParams:
		if (already_notified(ip->ip_src)) {
			break;
		}
		bad_params(&skipmsg);
		break;

	case SkipAuthFailed:
		if (already_notified(ip->ip_src)) {
			break;
		}
		hp = gethostbyaddr((char *) &ip->ip_src, inaddrsz, AF_INET);
		if (hp) {
			sprintf(msg,
				"Warning: \"%s\" (%s) failed authentication",
				hp->h_name, inet_ntoa(ip->ip_src));
		} else {
			sprintf(msg,
				"Warning: %s (unknown) failed authentication",
				inet_ntoa(ip->ip_src));
		}
		add_msg(msgs, msg);

		sprintf(msg, "Its certificate/key id may be invalid");
		add_msg(msgs, msg);

		acl_warning(NULL, msgs);
		break;

	}
	destroy_msgs(msgs);
	return (NOTIFY_DONE);
}

static void
make_skip_interactive()
{
	extern skiptool_base_window_objects	*Skiptool_base_window;
	char					*msgs[MAXMSGS] = { NULL };

	if (skip_fd >= 0) {
		fprintf(stderr, "make_skip_interactive: internal error\n");
		return;
	}
	skip_fd = skip_set_mode(ifname, SkipInteractive);
	if (skip_fd < 0) {
		if (errno == EEXIST) {
			add_msg(msgs, "This mode is not available "
							"(already in use)");
		} else {
			add_msg(msgs, skip_errmsg);
		}
		acl_warning(NULL, msgs);
		cancel_interactive_mode();
		destroy_msgs(msgs);
		return;
	}
	notify_set_input_func(Skiptool_base_window->base_window,
							skip_msg, skip_fd);
}

static void
make_skip_noninteractive()
{
	extern skiptool_base_window_objects	*Skiptool_base_window;

	if (skip_fd < 0) {
		fprintf(stderr, "make_skip_noninteractive: internal error\n");
		return;
	}
	notify_set_input_func(Skiptool_base_window->base_window,
						NOTIFY_FUNC_NULL, skip_fd);
	close(skip_fd);
	skip_fd = -1;
	return;
}

/*
 * modify SKIP access control mode
 */
void
access_update()
{
	extern skiptool_base_window_objects	*Skiptool_base_window;
	int					enable, mode, current_mode;
	char					*msgs[MAXMSGS] = { NULL };

	enable = xv_get(Skiptool_base_window->access_ctrl, PANEL_VALUE);
	current_mode = skip_get_mode(ifname);
	mode = xv_get(Skiptool_base_window->access_mode, PANEL_VALUE);

	if (enable) {
		inform_list_free();
		if (current_mode == SkipAccessControlOff) {
			/*
			 * need to check for required servers
			 */
			if (required_systems_build() == EBUSY) {
				/*
				 * wait for user to answer
				 */
				return;
			}
		}
		acl_build();
		switch (mode) {
		case NoAccess:
			if (skip_fd >= 0) {
				make_skip_noninteractive();
			}
			if (skip_access_ctrl_on(ifname) < 0) {
				add_msg(msgs, skip_errmsg);
				acl_warning(Skiptool_base_window->access_ctrl,
					msgs);
			}
			break;

		case AskFirst:
			add_automatically = 0;
			if (skip_fd < 0) {
				make_skip_interactive();
			}
			break;

		case AddAutomatically:
			add_automatically = 1;
			if (skip_fd < 0) {
				make_skip_interactive();
			}
		}
	} else {
		if (skip_fd >= 0) {
			make_skip_noninteractive();
		}
		if (skip_access_ctrl_off(ifname) < 0) {
			add_msg(msgs, skip_errmsg);
			acl_warning(Skiptool_base_window->access_ctrl, msgs);
		}
	}
	destroy_msgs(msgs);
	show_mode();
}
