/******************************************************************************
**
**     File/Function name:      ssh_random.c
**
**				Copyright 1998 Tadayoshi Kohno
**				All rights reserved.  See the LICENSE file.
**
**     Purpose:                 seed the random number generator
**
**     Author/Date:             Tadayoshi Kohno, 14 March 1998
**
**     References:
**	Eric Young's documentation for SSLeay's random numbers
**
**     Notes:
**
**     Functions:
**	ssh_random_seed_simple	simple seeding of the random number generator
**
******************************************************************************/

#include <time.h>

#include "ssh_types.h"
#include "ssh_random.h"

#include "ssh_errno.h"

/******************************************************************************
**
**     File/Function name:      ssh_random_seed_simple
**
**     Purpose:                 a simple, small seeding of the random
**				number generator
**
**     Entry (pre) conditions:  none
**
**     Parameters:              none
**
**     Exit (post) conditions:  S_GOOD
**
**     Error conditions:        S_BAD		error getting time for seed
**
**     Side effects:            SSLeay's random number generator seeded
**				with a simple (small seed)
**
**     Author/Date:             Tadayoshi Kohno, 14 March 1998
**
**     References:
**
**     Notes:
**	Call this function to seed Eric Young's random number generator
**	Note that the random number state is static to his random C file.
**
**	This function uses time (gasp!) to seed the random number generator
**	and is used in the communication between hosts.  A better one
**	of these should be made for seeding the random numbers for creating
**	RSA keys (xxx).
**
**	Actually, a better one should be created in general (xxx).
**
******************************************************************************/

int ssh_random_seed_simple()
{
	time_t random_seed;		/* our random seed */

	if ((random_seed = time(NULL)) == -1)
	{
		ssh_errno_set(SSH_ERRNO_RANDOM_SEED);
		return(S_BAD);
	} 

	RAND_seed((unsigned char *) &random_seed, sizeof(random_seed));

	return(S_GOOD);
}



