package Beam::Minion::Command::worker;
our $VERSION = '0.001';
# ABSTRACT: Command to run a Beam::Minion worker

#pod =head1 SYNOPSIS
#pod
#pod     beam minion worker <container>
#pod
#pod =head1 DESCRIPTION
#pod
#pod This command takes a L<Beam::Wire> container (optionally searching
#pod C<BEAM_PATH> a la L<Beam::Runner>) and starts a L<Minion::Worker> worker that
#pod will run any service inside.
#pod
#pod Service jobs are added to the queue using the L<beam minion run
#pod command|Beam::Minion::Command::run>.
#pod
#pod =head1 SEE ALSO
#pod
#pod L<Beam::Minion>, L<Minion>
#pod
#pod =cut

use strict;
use warnings;
use Beam::Wire;
use Beam::Runner::Util qw( find_container_path );
use Beam::Minion::Util qw( minion_init_args );
use Mojolicious;
use Minion::Command::minion::worker;

sub run {
    my ( $class, $container ) = @_;
    my $app = Mojolicious->new;
    $app->plugin( Minion => { minion_init_args() } );
    my $minion = $app->minion;
    my $path = find_container_path( $container );
    my $wire = Beam::Wire->new( file => $path );
    my $config = $wire->config;
    for my $name ( keys %$config ) {
        next unless $wire->is_meta( $config->{ $name }, 1 );
        $minion->add_task( $name => sub {
            my ( $job, @args ) = @_;
            my $obj = $wire->get( $name );
            my $exit = $obj->run( @args );
            my $method = $exit ? 'fail' : 'finish';
            $job->$method( { exit => $exit } );
        } );
    }
    my $cmd = Minion::Command::minion::worker->new( app => $app );
    $cmd->run( '-q', $container );
}

1;

__END__

=pod

=head1 NAME

Beam::Minion::Command::worker - Command to run a Beam::Minion worker

=head1 VERSION

version 0.001

=head1 SYNOPSIS

    beam minion worker <container>

=head1 DESCRIPTION

This command takes a L<Beam::Wire> container (optionally searching
C<BEAM_PATH> a la L<Beam::Runner>) and starts a L<Minion::Worker> worker that
will run any service inside.

Service jobs are added to the queue using the L<beam minion run
command|Beam::Minion::Command::run>.

=head1 SEE ALSO

L<Beam::Minion>, L<Minion>

=head1 AUTHOR

Doug Bell <preaction@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by Doug Bell.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
