#!/usr/bin/perl

# Copyright 2007 Kevin Ryde

# This file is part of Devel::Mallinfo.

# Devel::Mallinfo is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 3, or (at your option) any later
# version.

# Devel::Mallinfo is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
# or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
# for more details.

# You should have received a copy of the GNU General Public License along
# with Devel::Mallinfo.  If not, see <http://www.gnu.org/licenses/>.



# Usage: config.pl [makeprog [exeext]]
#
# Generate a "config.h" file for use by Mallinfo.xs.  Normally this is run
# from Makefile.PL, with $(MAKE) and $(EXE_EXT) as arguments.  Those args
# are used to run "make" on feature test programs.  There's probably better
# ways to do C configury like this, but for one header, one struct, and the
# struct contents it's enough.
#
# Being separate from Makefile.PL lets this script use the $(CC) and rules
# in the Makefile.  But note only the settings actually in that file get
# used, so if you want to override MakeMaker's defaults you have to use say
#
#     perl Makefile.PL OPTIMIZE=-g   # right
#
# not a make-time style, as it doesn't get through this script
#
#     perl Makefile.PL
#     make OPTIMIZE=-g    # wrong
#

use strict;
use warnings;
use POSIX;

# defaults just for running "perl config.pl" to try it out
my $make   = $ARGV[0] || 'make';
my $exeext = $ARGV[1] || '';
print "config.pl using MAKE=$make, EXEEXT=$exeext\n";

sub write_file {
  my ($filename, $content) = @_;
  open (OUT, ">$filename") or die;
  print OUT $content;
  close OUT or die;
}

# run "make" on $target, return 1 for success, 0 for fail
sub make {
  my ($target) = @_;
  my $status = system ($make, $target);
  return (WIFEXITED($status) && WEXITSTATUS($status) == 0
          ? 1 : 0);
}


# malloc.h:
#   - freebsd has made malloc.h a deliberate #error, which is incredibly stupid
#   - darwin doesn't have a malloc.h at all
#
write_file ('confmallh.c',
'/* generated by config.pl */
#include <malloc.h>
int main (void) { return 0; }
');
my $have_malloc_h = make ("confmallh$exeext");


write_file ('confminfo.c',
'/* generated by config.pl */
#include <stdlib.h>
' .
($have_malloc_h ? "#include <malloc.h>\n" : "")
. 'int main (void) {
  struct mallinfo m;
  m = mallinfo ();
  return (m.arena != 0);
}
');
my $have_mallinfo = make ("confminfo$exeext");

if (! $have_mallinfo) {
    print <<EOF;
+--------------------------------------+
+ WARNING:                             |
+ WARNING: mallinfo() not available    |
+ WARNING: this module will do nothing |
+ WARNING:                             |
+--------------------------------------+
EOF
}


my @mallinfo_fields;
sub get_fields {
  if (! make ('confminfo.i')) {
    die "Oops, can't make confminfo.i";
  }
  open IN, '<confminfo.i' or die;
  my $preproc = do { local $/ = undef; <IN> }; # slurp
  close IN or die;
  
  if (! ($preproc =~ /struct\s+mallinfo\s*\{(.*?)}/s)) {
    print "\"struct mallinfo\" not found in confminfo.i preprocessed output\n";
    return;
  }

  my $body = $1;  # the insides of the struct
  while ($body =~ /([A-Za-z0-9_]+)\s*;/g) {  # field name before each ";"
    push @mallinfo_fields, $1;
  }
  if (@mallinfo_fields) {
    print "struct mallinfo fields found:\n  ",join(' ',@mallinfo_fields),"\n";
  } else {
    print "no fields matched in \"struct mallinfo\" body\n";
  }
}
if ($have_mallinfo) {
  get_fields();
  if (! @mallinfo_fields) {
    print <<EOF;
+----------------------------------------------+
+ WARNING:                                     |
+ WARNING: cannot grep struct mallinfo fields, |
+ WARNING: using default fallbacks             |
+ WARNING:                                     |
+----------------------------------------------+
EOF
    @mallinfo_fields = qw(arena ordblks smblks hblks hblkhd
                          usmblks fsmblks uordblks fordblks keepcost);
  }
}


open OUT, '>config.h' or die;
print OUT "/* generated by config.pl */

#define HAVE_MALLOC_H $have_malloc_h
#define HAVE_MALLINFO $have_mallinfo

#define STRUCT_MALLINFO_FIELDS \\
";
foreach my $field (@mallinfo_fields) {
  print OUT "  FIELD($field); \\\n";
}
print OUT "  /*end*/\n";
close OUT or die;

exit 0;
