package Pod::Weaver::Plugin::Ditaa;
$Pod::Weaver::Plugin::Ditaa::VERSION = '0.001005';
# ABSTRACT: include ditaa diagrams in your pod

use Moose;
with 'Pod::Weaver::Role::Dialect';

sub translate_dialect {
   Pod::Elemental::Transformer::Ditaa->new->transform_node($_[1])
}

package Pod::Elemental::Transformer::Ditaa {
$Pod::Elemental::Transformer::Ditaa::VERSION = '0.001005';
use Moose;
   with 'Pod::Elemental::Transformer';

   use Capture::Tiny 'capture';
   use autodie;
   use File::Temp;
   use IPC::System::Simple 'system';
   use IO::All;
   use MIME::Base64;
   use namespace::clean;

   sub transform_node {
      my ($self, $node) = @_;
      my $children = $node->children;

    my $x = 0;

    for (my $i = 0 ; $i < @$children; $i++) {
         my $para = $children->[$i];
         next
           unless $para->isa('Pod::Elemental::Element::Pod5::Region')
           and !$para->is_pod
           and $para->format_name eq 'ditaa';

         my $length = @{$para->children};
         confess 'ditaa transformer expects exec region to contain 1 Data para'
           unless $length == 1
           and $para->children->[0]->isa('Pod::Elemental::Element::Pod5::Data');

         $x++;
         my $text = $para->children->[0]->content;

         my %meta = ( label => "Figure $x" );;
         my ($meta, $rest) = split /\n\n/, $text, 2;

         if ($rest) {
            %meta = map { split qr/\s*:\s*/, $_, 2 } split "\n", $meta;
            $text = $rest;
         }

         my $new_doc = $self->_render_figure(
            %meta,
            text => $text,
            b64 => $self->_text_to_b64image(
               $text,
               split qr/\s+/, $para->content || '',
            ),
         );

         splice @$children, $i, 1, @{$new_doc->children};
      }

      return $node;
   }

   sub _text_to_b64image {
      my ($self, $text, @flags) = @_;

      my $tmp_text = tmpnam();
      my $tmp_img  = tmpnam() . '.png';
      open my $fh, '>', $tmp_text;
      print {$fh} $text;
      close $fh;

      my @cmd = ('ditaa', @flags, '-o', $tmp_text, $tmp_img);
      print STDERR join q( ), @cmd
         if $ENV{DITAA_TRACE};

      my $merged_out = capture { system @cmd };
      print STDERR $merged_out if $ENV{DITAA_TRACE};
      my $image = encode_base64(io->file($tmp_img)->binary->all, '');
      unlink $tmp_text unless $ENV{DITAA_TRACE} && $ENV{DITAA_TRACE} =~ m/keep/;
      unlink $tmp_img unless $ENV{DITAA_TRACE} && $ENV{DITAA_TRACE} =~ m/keep/;

      return $image
   }

   sub _render_figure {
      my ($self, %args) = @_;

      my $new_doc = Pod::Elemental->read_string(
         "\n\n=begin text\n\n$args{label}\n\n" .
         "$args{text}\n\n=end text\n\n" .
          qq(\n\n=begin html\n\n) .
             qq(<p><i>$args{label}</i>) .
             qq(<img src="data:image/png;base64,$args{b64}"></img></p>\n\n) .
          qq(=end html\n\n)
      );
      Pod::Elemental::Transformer::Pod5->transform_node($new_doc);
      shift @{$new_doc->children}
        while $new_doc->children->[0]
        ->isa('Pod::Elemental::Element::Pod5::Nonpod');

      return $new_doc
   }

}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Pod::Weaver::Plugin::Ditaa - include ditaa diagrams in your pod

=head1 VERSION

version 0.001005

=head1 SYNOPSIS

In your F<weaver.ini>:

 [@Default]
 [-Ditaa]

In the pod of one of your modules:

 =head1 HOW IT WORKS

 =begin ditaa

 label: How it works

    +--------+   +-------+    +-------+
    |        | --+ ditaa +--> |       |
    |  Text  |   +-------+    |diagram|
    |Document|   |!magic!|    |       |
    |     {d}|   |       |    |       |
    +---+----+   +-------+    +-------+
        :                         ^
        |       Lots of work      |
        +-------------------------+

 =end ditaa

=head1 DESCRIPTION

It has often been said that a picture is worth a thousand words.  I find that
sometimes a diagram truly can illuminate your design.  This L<Pod::Weaver>
plugin allows you to put L<ditaa|http://ditaa.sourceforge.net/> diagrams in your
pod and render the image for an html view.  In text mode it merely uses the text
diagram directly.

Note that you may put a C<label: Foo> at the top of your diagram, but if you
do not you will get a numbered label in the format C<Figure $i>.

=head1 IN ACTION

=begin text

Figure 1

label: How it works

   +--------+   +-------+    +-------+
   |        | --+ ditaa +--> |       |
   |  Text  |   +-------+    |diagram|
   |Document|   |!magic!|    |       |
   |     {d}|   |       |    |       |
   +---+----+   +-------+    +-------+
       :                         ^
       |       Lots of work      |
       +-------------------------+

=end text

=for html <p><i>Figure 1</i><img src="data:image/png;base64,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"></img></p>

=head1 SYNTAX

The ditaa syntax L<is documented here|http://ditaa.sourceforge.net/#usage>.

=head1 PASSING FLAGS TO DITAA

 =begin ditaa -r -S

 label: Passing Flags

    +--------+
    |        |
    |  Test  |
    |        |
    +---+----+

 =end ditaa

=begin text

Figure 2

label: Passing Flags

    +--------+
    |        |
    |  Test  |
    |        |
    +---+----+

=end text

=for html <p><i>Figure 2</i><img src="data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAPAAAACaCAIAAAAYWFL5AAAIDElEQVR42u3d30tTfRzAccu5SnEX2ViIXlhkEUFCF5G/w0FXgYQECmL6D3hb0FWXQZcFXWk/LGqQN2pRpLlZmQNhox88peScs80nx6YrN9N6PnToMFzPmU/Z4znb+33V2Zme5/s5r+fsnPHQk/ONKIPKYQQEaCLdg/5KZNgATYAmAjQRoIkATYAGNAGaCNBEgCYCNAEa0ARoIkATAZoI0ARoIkATAZoI0ESAJkATAZoI0ESAJgI0AfpXy/nexr5zY3/291enZrVaa2pqent7/4eD/um1qG3uhAG9CaCVPy8tLY2MjBw6dEheuXLlikFB623CgN400EovXryQV/bu3Wu4j2lA/8qYbt26pbyybdu28vLy8+fPx+Px5Hd2d3cfPHjQbDaLievXryf/KqfTabfbCwsLLRZLZWVlX1+fDkHHYjF5JS8vbz1L9vv9bW1tJSUlsl6bzdbc3Dw8PJx2V+qdgMPhOHbs2Pbt23fu3Hny5MnJyUn16Ddu3JBJyj+PTFVmqzGo/wRaY1HaB9VYlyFBnz17dmBgYHFxMRKJXLx4UXadO3cu+Z1yYt6/f+/z+aqqqmTz8ePHyt7BwUGTyVRbW/vu3bv5+fnW1lZFv95Aj46OrrlCayy5rq5ONu/duye3K4FA4ObNm3ILnnZXKujDhw+Pj49Ho9ELFy7Ipvysslemp45UOnr06EaB1liU9kE11mX4W47l5WXZtWfPnuR3ym2osil/kM3jx48rm0JZNj0ej7IZDAZlU64NurqHfvr0qXIPffny5fUsuaCgQDblErW6urrmnRq7UkE/f/5c2VxYWFCumsqmTE82XS6Xsim/LS3oX3goXLMo7YNqrMt4oEOh0JkzZ+TjRq616ry2bt2a/E45JcqmXG9ks6ioSNnMz89PnXVubq6uvuXYtWvXmm85tJdcXV2tvCKrq6io6OzsnJmZSbsr1Zn6cS9KkvfKHUjqSH//Cq29KO2DaqzLeKBPnDihfDbNzc0pl7TUc7NmEDKdZNAySp0/FK5Je8lyZ9Xe3l5aWqqykPOddpf2hfN/AK29KO2DaqzLeKDleS55qUNDQ6nn5t9uOerr62Xz7t27xgKtvWQ1uRNVHrPkEzntrvWDVj791RuS9dxy/P55XOdBNZZsGNANDQ3y50uXLsViMXl42rdvX+q5kWfBqakp9aHw0aNHyl65J5Pn4rKyMpmUfMLKe7q6uuQ9OgetvWTZK49H8rGTSCT6+/vldbvdnnbX+kErz2fyHCbz3MCHQu1FaR9UY13GAy1PtadPn5aPJHlqkfsnecJNPTfCVJ7Zd+zYIXbVLzGUxsbGGhsb5a5abt2Ki4tbWlqePHmic9DaSx4cHGxqarJarfLvqnwKd3R0fPjwIe2u9YNWv0GTecpUr169mvzg8cugtRelfVCNdRkDNOknr9crtvbv35/xBwV0xnbq1Cm32y3PbW/evFGeQ9Z87mXMQQGdFTkcjiNHjuTl5VksFrEl96+ZelBAU8YGaAI0EaCJAE0EaAI0oAnQRIAmAjQRoAnQgCZAEwGaCNBEgCZAEwGaCNBEgCbKDtAHDhzIoR/JNABtbNByFr+Reka//51dsVhsaWkpkUisrKwAGtDGBu33+0OhUDgcFtZiGtCANjboV69eTU5OBgIBMf3582dAA9rYoJ89e+bxeMR0MBhcXFwENKCNDfr+/fti+uXLl9PT05FIBNCANjboO3fuPHz40O12T0xMzM/PAxrQxgZ9+/btBw8ejI2NKf+/JUADGtCABjSgAQ1oQAMa0IAGNKABDWhAAxrQgAY0oAENaEADGtCABjSgAQ1oQAMa0IAGNKABDWhApzWh0Yb8ckADenOu0BvuD9CABjSgAf2H/blcLrvdXlhYaLFYKisr+/v71V0zMzNtbW0lJSVms9lmszU3Nzudzp/ezAAa0LoAPTQ0ZDKZamtrJyYmwuFwa2urvOfatWvK3rq6Otns7e2Nx+Ozs7M9PT01NTVcoQGtX9BCWV70er3KZigUks3y8nJls6CgQDblqpw8aEADWr+g8/PzU7/9yM3NVfZWV1crr8jbKioqOjs7A4EAoAGtd9Bzc3M//ZHp6en29vbS0lLVuhAHNKD1C7q+vl5edDgc2j8bjUbFlrxTbkLUF7ds2QJoQOsL9MjIiNlsLisrGx0dTSQSPp+vu7u7qqpK2dvQ0CBPhHL9Xl5eHhgYkB+32+3qzxYXF8srr1+/BjSgdfS1ndvtbmxsLCoqMplMYrSlpWV4eFj9DqSpqclqtQp6ufHo6OgIBoPqD3Z1ddlsNr62A/TmgOa/5QA0oAENaEADGtCABjSgAQ1oQAMa0IAGNKABDWhAAxrQgAY0oAENaEADGtCABjSgAU2ABjSgAQ1oQAMa0IAGNKABDWhAAxrQgAY0oAENaEADGtCABnQ2gd69e3cO/chisQDa2KClSCTi8/m8Xq/L5err6+vJ7mQCMgeZhsxEJgNo47WwsDA7O/v27dvx8XGn0zmQ3ckEZA4yDZmJTAbQxuvTp08fP370+/1yFj0ez2h2JxOQOcg0ZCYyGUAbr3g8LpciOX9yTZqamvoru5MJyBxkGjITmQygjdeXL1/kzMnVKBqNhsPhv7M7mYDMQaYhM5HJANqorf5oJbtT5/A1a8pM0JS1AZoATQRoIkATAZoADWgCNBGgiQBNBGgCNKAJ0ESAJgI0EaAJ0ESAJgI0EaCJAE2AJgI0EaCJAE20LtBEmfD3vTMCAjSRTvsHx0UMBMZpKEIAAAAASUVORK5CYII="></img></p>

To pass flags to C<ditaa> simply append the flags to the C<< =begin ditaa >>
directive.

=head1 DEBUGGING

Set the C<DITAA_TRACE> env var and you'll see all of the commands that this
plugin runs printed to C<STDERR>.  If you set the env var to C<keep> the
temporary files referenced in the command will not automatically be deleted, so
you can ensure that the text and image diagrams were created correctly.

=head1 PERL SUPPORT POLICY

Because this module is geared towards helping release code, as opposed to
helping run code, I only aim at supporting the last 3 releases of Perl.  So for
example, at the time of writing that would be 5.22, 5.20, and 5.18.  As an
author who is developing against Perl and using this to release modules, you can
use either L<perlbrew|http://perlbrew.pl/> or
L<plenv|https://github.com/tokuhirom/plenv> to get a more recent perl for
building releases.

Don't bother sending patches to support older versions; I could probably support
5.8 if I wanted, but this is more so that I can continue to use new perl
features.

=head1 AUTHOR

Arthur Axel "fREW" Schmidt <frioux+cpan@gmail.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by Arthur Axel "fREW" Schmidt.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
