(* Copyright (C) 1989-1992, Digital Equipment Corporation *)
(* All rights reserved. *)
(* See the file COPYRIGHT for a full description. *)
(* *)
(* Last modified on Tue Sep 22 09:27:50 PDT 1992 by mhb *)
(* modified on Sun Aug 16 17:45:05 PDT 1992 by meehan *)
(* modified on Fri Mar 27 02:14:10 1992 by steveg *)

(* {\em Resources\/} are arbitrary files that are associated with
   applications.  Resources can be bundled into an application
   using the "m3bundle" facility.  They may also be found in the
   file system.

   This interface supports retrieval of resources using a {\em
   search path}.  A search path is a list of elements, and each
   element is either a {\em path\/} or a {\em bundle}.  A path is
   a directory, implemented as a "Pathname.T".  It should already
   be fully expanded by having called "Pathname.Expand".  A
   bundle is a "Bundle.T" object, typically created by
   "m3bundle". *)

INTERFACE Rsrc;

IMPORT List, Rd, Thread;

TYPE 
  Path = List.T; (* of Pathname.T or Bundle.T *)

EXCEPTION NotFound;

PROCEDURE Open (name: TEXT; path: Path): Rd.T RAISES {NotFound};
(* Search each element of "path", from front to back, for the
   first occurrence of the resource called "name" and return a
   reader on the resource.  If the path element is a string "s",
   then a reader is returned if
| FileStream.OpenRead(s & "/" & name)
   is successful.  If the path element is a bundle "b", a reader
   is returned if
| TextRd.New(Bundle.Get(b, name))
   is successful.  The "NotFound" exception is raised if no
   element of "path" yields a successful reader on "name".  It is
   a checked runtime error if "path" contains an element that is
   neither a string nor a bundle. *)

PROCEDURE Get (name: TEXT; path: Path): TEXT
  RAISES {NotFound, Rd.Failure, Thread.Alerted};
(* A convenience procedure to retrieve the contents of the
   resource "name" as a "TEXT".  "Get" is logically equivalent to
|  VAR rd := Open(name);
|  BEGIN
|    TRY
|      RETURN Rd.GetText(rd, LAST(CARDINAL))
|    FINALLY
|      Rd.Close(rd)
|    END
|  END;
   The implementation is slightly more efficient, because it
   takes advantage of "Bundle.Get" procedure which returns the
   contents of the bundle element as a "TEXT".  The "Rd.Failure"
   exception is raised if "Rd.GetText" or "Rd.Close" report a
   problem.  The "Thread.Alerted" can be raised by the call to
   "Rd.GetText". *)

<* OBSOLETE *>
PROCEDURE BuildPath (a1, a2, a3, a4: REFANY := NIL): Path;
(* Build a "Path" from elements which are either a "Bundle.T" or
   a "TEXT".  If a "TEXT", the string is assumed to be the name
   of a directory, unless it starts with a dollar sign.  In the
   latter case, the "TEXT" is assumed to be environment variable.
   Environment variables are expanded, and undefined environment
   variables expand to the empty string.  The returned "Path"
   ignores all empty components.

   Note: This procedure is marked as obsolete because it will
   soon be superseded by the Interface Police's "Pathname"
   interface. *)

END Rsrc.




