/* -*- c -*-
 *
 * Author:      James Brister <brister@vix.com> -- berkeley-unix --
 * Start Date:  Thu Dec 28 13:34:47 1995
 * Project:     INN (innfeed)
 * File:        connection.c
 * RCSId:       $Id: connection.c,v 1.14 1996/04/13 02:51:33 brister Exp $
 * Copyright:   Copyright (c) 1996 by Internet Software Consortium
 *
 *              Permission to use, copy, modify, and distribute this
 *              software for any purpose with or without fee is hereby
 *              granted, provided that the above copyright notice and this
 *              permission notice appear in all copies.
 *
 *              THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE
 *              CONSORTIUM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *              SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 *              MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL INTERNET
 *              SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
 *              INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 *              WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 *              WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 *              TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE
 *              USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Description: The implementation of the Connection class.
 *
 *              The Connection object is what manages the NNTP
 *              protocol. If the remote doesn't do streaming, then the
 *              standard IHAVE lock-step protcol is performed. In the
 *              streaming situation we have two cases. One where we must
 *              send CHECK commands, and the other where we can directly
 *              send TAKETHIS commands without a prior CHECK.
 *
 *              The Connection object maintains four article queues. The first
 *              one is where new articles are put if they need to have an
 *              IHAVE or CHECK command sent for them. The second queue is
 *              where the articles move from the first after their IHAVE/CHECK
 *              command is sent, but the reply has not yet been seen. The
 *              third queue is where articles go after the IHAVE/CHECK reply
 *              has been seen (and the reply says to send the article). It is
 *              articles in the third queue that have the TAKETHIS command
 *              sent, or the body of an IHAVE.  The third queue is also where
 *              new articles go if the connection is running in TAKETHIS-only
 *              mode. The fourth queue is where the articles move to from the
 *              third queue after their IHAVE-body or TAKETHIS command has
 *              been sent. When the response to the IHAVE-body or TAKETHIS is
 *              received the articles are removed from the fourth queue and
 *              the Host object controlling this Connection is notified of
 *              the success or failure of the transfer.
 *
 *              The whole system is event-driven by the EndPoint class and the
 *              Host via calls to prepareRead() and prepareWrite() and
 *              prepareSleep().
 *           
 */

  /*

    We should probably store the results of gethostbyname in the connection so
    we can rotate through the address when one fails for connecting. Perhaps
    the gethostbyname should be done in the Host and the connection should just
    be given the address to use.

    Should we worry about articles being stuck on a queue for ever if the
    remote forgets to send a response to a CHECK?

    Perhaps instead of killing the connection on some of the more simple
    errors, we should perhaps try to flush the input and keep going.

    Worry about counter overflow.

    Worry about stats gathering when switch to non-check mode.

    */

#if ! defined (lint)
static const char *rcsid = "$Id: connection.c,v 1.14 1996/04/13 02:51:33 brister Exp $" ;
static void use_rcsid (const char *rid) {   /* Never called */
  use_rcsid (rcsid) ; use_rcsid (rid) ;
}
#endif

#include "config.h"

#include <stdlib.h>
#include <assert.h>
#include <string.h>

#if defined (DO_HAVE_UNISTD)
#include <unistd.h>
#endif

#include <stdio.h>
#include <errno.h>
#include <signal.h>
#include <sys/types.h>

#if defined (DO_NEED_TIME)
#include <time.h>
#endif
#include <sys/time.h>

#include <syslog.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <string.h>
#include <netdb.h>
#include <fcntl.h>

#if defined (DO_NEED_FILE)
#include <sys/file.h>
#endif

#include "buffer.h"
#include "connection.h"
#include "endpoint.h"
#include "host.h"
#include "article.h"
#include "msgs.h"




  /*
   * Private types.
   */

  /* We keep a linked list of articles the connection is trying to transmit */
typedef struct art_holder_s 
{
    Article article ;
    struct art_holder_s *next ;
} *ArtHolder ;


  /* The Connection class */
struct connection_s 
{
    Host myHost ;               /* the host who owns the connection */
    u_int ident ;               /* an identifier for syslogging. */
    EndPoint myEp ;             /* the endpoint the connection talks through */

    ArtHolder checkHead ;       /* head of article list to do CHECK/IHAVE */
    ArtHolder checkRespHead ;   /* head of list waiting on CHCEK/IHAVE
                                   response */
    ArtHolder takeHead ;        /* head of list of articles to send
                                   TAKETHIS/IHAVE-body */
    ArtHolder takeRespHead ;    /* list of articles waiting on
                                   TAKETHIS/IHAVE-body response */
    u_int articleQTotal ;       /* number of articles in all four queues */

    Buffer respBuffer ;         /* buffer all responses are read into */

    char *ipName ;              /* the ip name/address of the remote */
    u_int maxCheck ;            /* the max number of CHECKs to send */
    u_short port ;              /* the port number to use */

    u_int artTout ;             /* the timeout for getting articles from
                                   Host */
    TimeoutId artToutId ;       /* timeout id for article timeout */

    u_int respTout ;            /* the timeout for responses from remote */
    TimeoutId respToutId ;      /* timeout id for reponse timeout */

    u_int closePeriod ;         /* number of second max to keep connection
                                   open (due to nntpd history file
                                   behaviour) */
    TimeoutId closeId ;         /* timeout id for closing connection. */

    TimeoutId reopenId ;        /* timeout id for reopen attempts. */
    u_int reopenWait ;          /* number of seconds before trying to connect*/

    bool locked ;               /* true if we're processing a response */
    bool doesStreaming ;        /* true if remote will handle streaming */
    bool needsChecks ;          /* true if we issue CHECK commands in
                                   streaming mode (rather than just sending
                                   TAKETHIS commands) */

    bool closing ;              /* true if connection is closing down */
    bool flushing ;             /* true if after closed we re-connect */
    bool periodicClose ;        /* true if closing to due timeout */
    bool deleting ;             /* true if Connection will be free'd */
    time_t timeCon ;            /* the time the connect happened. */

      /* STATISTICS */
    u_int artsTaken ;           /* the number of articles INN gave this cxn */

    u_int checksIssued ;        /* the number of CHECKS/IHAVES we
                                   sent. Note that if we're running in
                                   TAKETHIS-only mode, then we add in the
                                   TAKETHIS commands too */
    u_int checksRefused ;       /* the number of response 435/438 */

    u_int takesRejected ;       /* the number of response 437/439 recevied */
    u_int takesOkayed ;         /* the number of response 235/239 received */

    double onThreshold ;        /* for no-CHECK mode */
    double offThreshold ;       /* for no-CHECK mode */
    double filterValue ;

    Connection next ;           /* for global list. */
};

static Connection gCxnList = NULL ;
static u_int gCxnCount = 0 ;

  /***************************************************
   *
   * Private function declarations.
   *
   ***************************************************/

  /* Misc functions */
static bool isLocked (Connection cxn) ;
static void unLockCxn (Connection cxn) ;
static void lockCxn (Connection cxn) ;
static void doSomeWrites (Connection cxn) ;
static void issueQUIT (Connection cxn) ;
static bool issueIHAVE (Connection cxn) ;
static bool issueStreamingCommands (Connection cxn) ;
static void abortConnection (Connection cxn) ;
static void setUpResponseTimeout (Connection cxn) ;
static Buffer buildCheckBuffer (Connection cxn) ;
static Buffer *buildTakethisBuffers (Connection cxn, Buffer checkBuffer) ;
static void resetConnection (Connection cxn) ;
static void deferAllArticles (Connection cxn) ;
static void incrFilter (Connection cxn) ;
static void decrFilter (Connection cxn) ;
static void delConnection (Connection cxn) ;
static bool writesNeeded (Connection cxn) ;
static void prepareReopenCbk (Connection cxn) ;
static void validateConnection (Connection cxn) ;


  /* I/O Callbacks */
static void connectionDone (EndPoint e, IoStatus i, Buffer *b, void *d) ;
static void getBanner (EndPoint e, IoStatus i, Buffer *b, void *d) ;
static void modeCmdIssued (EndPoint e, IoStatus i, Buffer *b, void *d) ;
static void getModeResponse (EndPoint e, IoStatus i, Buffer *b, void *d) ;
static void checkWriteDone (EndPoint e, IoStatus i, Buffer *b, void *d) ;
static void responseIsRead (EndPoint e, IoStatus i, Buffer *b, void *d) ;
static void ihaveBodyDone (EndPoint e, IoStatus i, Buffer *b, void *d) ;
static void quitWritten (EndPoint e, IoStatus i, Buffer *b, void *d) ;


  /* Timer callbacks */
static void articleTimeoutCbk (TimeoutId id, void *data) ;
static void responseTimeoutCbk (TimeoutId id, void *data) ;
static void periodicCxnCloseCbk (TimeoutId, void *data) ;
static void reopenTimeoutCbk (TimeoutId id, void *data) ;

  /* NNTP FSM routines. */
static bool processResponse205 (Connection cxn, char *response) ;
static bool processResponse235 (Connection cxn, char *response) ;
static bool processResponse238 (Connection cxn, char *response) ;
static bool processResponse239 (Connection cxn, char *response) ;
static bool processResponse335 (Connection cxn, char *response) ;
static bool processResponse400 (Connection cxn, char *response) ;
static bool processResponse431 (Connection cxn, char *response) ;
static bool processResponse435 (Connection cxn, char *response) ;
static bool processResponse436 (Connection cxn, char *response) ;
static bool processResponse437 (Connection cxn, char *response) ;
static bool processResponse438 (Connection cxn, char *response) ;
static bool processResponse439 (Connection cxn, char *response) ;


  /* Article queue management routines. */
static ArtHolder newArtHolder (Article art) ;
static void delArtHolder (ArtHolder artH) ;
static ArtHolder artHolderByMsgId (const char *msgid, ArtHolder head) ;
static void appendArtHolder (ArtHolder artH, ArtHolder *head, u_int *count) ;
static bool remArtHolder (ArtHolder art, ArtHolder *head, u_int *count) ;






  /***************************************************
   *
   * Public functions implementation.
   *
   ***************************************************/


  /* Create a new Connection object and return it. */
Connection newConnection (Host host,
                          u_int id,
                          const char *ipname,
                          u_int artTout,
                          u_int portNum,
                          u_int respTimeout,
                          u_int closePeriod,
                          double lowPassHigh,
                          double lowPassLow)
{
  Connection cxn ;
  bool croak = false ;
  const char *peerName = hostPeerName (host) ;

  if (ipname == NULL)
    {
      dprintf (1,"NULL ipname in newConnection\n") ;
      croak = true ;
    }

  if (ipname && strlen (ipname) == 0)
    {
      dprintf (1,"Empty ipname in newConnection\n") ;
      croak = true ;
    }

  if (artTout > MAX_MAXART_TOUT)
    {
      syslog (LOG_ERR, MAX_ARTTOUT_BIG, peerName, cxn->ident, artTout) ;
      croak = true ;
    }

  if (respTimeout > MAX_RESP_TOUT)
    {
      syslog (LOG_ERR, MAX_RESPTOUT_BIG, peerName, cxn->ident, respTimeout) ;
      croak = true ;
    }

  if (croak)
    return NULL ;

  cxn = CALLOC (struct connection_s, 1) ;
  ASSERT (cxn != NULL) ;

  cxn->myHost = host ;
  cxn->myEp = NULL ;
  cxn->ident = id ;

  cxn->checkHead = NULL ;
  cxn->checkRespHead = NULL ;
  cxn->takeHead = NULL ;
  cxn->takeRespHead = NULL ;
  cxn->articleQTotal = 0 ;

  cxn->respBuffer = newBuffer (BUFFER_SIZE) ;
  ASSERT (cxn->respBuffer != NULL) ;
  
  cxn->ipName = strdup (ipname) ;
  cxn->port = portNum ;
  cxn->maxCheck = 1 ;           /* may increase later */
  cxn->locked = false ;

  cxn->artTout = artTout ;
  cxn->artToutId = 0 ;
  
  cxn->respTout = respTimeout ;
  cxn->respToutId = 0 ;

  cxn->closePeriod = closePeriod ;
  cxn->closeId = 0 ;

  cxn->reopenWait = INITIAL_REESTABLISHMENT_PERIOD ;
  cxn->reopenId = 0 ;
  
  cxn->onThreshold = lowPassHigh ;
  cxn->offThreshold = lowPassLow ;

  resetConnection (cxn) ;

  cxn->next = gCxnList ;
  gCxnList = cxn ;
  gCxnCount++ ;
  
  return cxn ;
}

void gPrintCxnInfo (FILE *fp, u_int indentAmt)
{
  char indent [INDENT_BUFFER_SIZE] ;
  u_int i ;
  Connection cxn ;
  
  for (i = 0 ; i < MIN(INDENT_BUFFER_SIZE - 1,indentAmt) ; i++)
    indent [i] = ' ' ;
  indent [i] = '\0' ;

  fprintf (fp,"%sGlobal Connection list : (count %d) {\n",
           indent,gCxnCount) ;
  for (cxn = gCxnList ; cxn != NULL ; cxn = cxn->next)
    printCxnInfo (cxn,fp,indentAmt + INDENT_INCR) ;
  fprintf (fp,"%s}\n",indent) ;
}


void printCxnInfo (Connection cxn, FILE *fp, u_int indentAmt)
{
  char indent [INDENT_BUFFER_SIZE] ;
  u_int i ;
  ArtHolder artH ;

  for (i = 0 ; i < MIN(INDENT_BUFFER_SIZE - 1,indentAmt) ; i++)
    indent [i] = ' ' ;
  indent [i] = '\0' ;

  fprintf (fp,"%sConnection : %p {\n",indent,cxn) ;
  fprintf (fp,"%s    host : %p\n",indent, (void *) cxn->myHost) ;
  fprintf (fp,"%s    endpoint : %p\n",indent,cxn->myEp) ;
  fprintf (fp,"%s    ident : %d\n",indent,cxn->ident) ;
  fprintf (fp,"%s    ip-name : %s\n", indent, cxn->ipName) ;
  fprintf (fp,"%s    port-number : %d\n",indent,cxn->port) ;
  fprintf (fp,"%s    max-checks : %d\n",indent,cxn->maxCheck) ;
  fprintf (fp,"%s    locked : %s\n",indent,boolToString (cxn->locked)) ;
  fprintf (fp,"%s    does-streaming : %s\n",indent,
           boolToString (cxn->doesStreaming)) ;
  fprintf (fp,"%s    needs-checks : %s\n",indent,
           boolToString (cxn->needsChecks)) ;
  fprintf (fp,"%s    closing : %s\n",indent,boolToString (cxn->closing)) ;
  fprintf (fp,"%s    flushing : %s\n",indent,boolToString (cxn->flushing));
  fprintf (fp,"%s    periodic-close : %s\n",indent,
           boolToString (cxn->periodicClose)) ;
  fprintf (fp,"%s    deleting : %s\n",indent,
           boolToString (cxn->deleting)) ;
  fprintf (fp,"%s    time-connected : %ld\n",indent,(long) cxn->timeCon) ;
  fprintf (fp,"%s    articles from INN : %d\n",indent,cxn->artsTaken) ;
  fprintf (fp,"%s    articles offered : %d\n",indent,
           cxn->checksIssued) ;
  fprintf (fp,"%s    articles refused : %d\n",indent,
           cxn->checksRefused) ;
  fprintf (fp,"%s    articles rejected : %d\n",indent,
           cxn->takesRejected) ;
  fprintf (fp,"%s    articles accepted : %d\n",indent,
           cxn->takesOkayed) ;
  fprintf (fp,"%s    low-pass upper limit : %0.6f\n", indent,
           cxn->onThreshold) ;
  fprintf (fp,"%s    low-pass lower limit : %0.6f\n", indent,
           cxn->offThreshold) ;
  fprintf (fp,"%s    low-pass filter : %0.6f\n", indent,
           cxn->filterValue) ;
           
  fprintf (fp,"%s    article-timeout : %d\n",indent,cxn->artTout) ;
  fprintf (fp,"%s    article-callback : %d\n",indent,cxn->artToutId) ;

  fprintf (fp,"%s    response-timeout : %d\n",indent,cxn->respTout) ;
  fprintf (fp,"%s    response-callback : %d\n",indent,cxn->respToutId) ;

  fprintf (fp,"%s    closePeriod : %d\n",indent,cxn->closePeriod) ;
  fprintf (fp,"%s    closeId : %d\n",indent,cxn->closeId) ;
  
  fprintf (fp,"%s    reopen wait : %d\n",indent,cxn->reopenWait) ;
  fprintf (fp,"%s    reopen id : %d\n",indent,cxn->reopenId) ;

  fprintf (fp,"%s    CHECK queue {\n",indent) ;
  for (artH = cxn->checkHead ; artH != NULL ; artH = artH->next)
    printArticleInfo (artH->article,fp,indentAmt + INDENT_INCR) ;
  fprintf (fp,"%s    }\n",indent) ;
  
  fprintf (fp,"%s    CHECK Response queue {\n",indent) ;
  for (artH = cxn->checkRespHead ; artH != NULL ; artH = artH->next)
    printArticleInfo (artH->article,fp,indentAmt + INDENT_INCR) ;
  fprintf (fp,"%s    }\n",indent) ;
  
  fprintf (fp,"%s    TAKE queue {\n",indent) ;
  for (artH = cxn->takeHead ; artH != NULL ; artH = artH->next)
    printArticleInfo (artH->article,fp,indentAmt + INDENT_INCR) ;
  fprintf (fp,"%s    }\n",indent) ;
  
  fprintf (fp,"%s    TAKE response queue {\n",indent) ;
  for (artH = cxn->takeRespHead ; artH != NULL ; artH = artH->next)
    printArticleInfo (artH->article,fp,indentAmt + INDENT_INCR) ;
  fprintf (fp,"%s    }\n",indent) ;

  fprintf (fp,"%s    response buffer {\n",indent) ;
  printBufferInfo (cxn->respBuffer,fp,indentAmt + INDENT_INCR) ;
  fprintf (fp,"%s    }\n",indent) ;

  fprintf (fp,"%s}\n",indent) ;
}



  /* Create a new endpoint hooked to a non-blocking socket that is trying
     to conenct to the host info stored in the Connection. On fast machines
     connecting locally the connect() may have already succeeded when this
     returns, but typically the connect will still be running and when it
     completes the Connection will be notified via a write callback setup
     by prepareWrite below. If nothing goes wrong then this will return
     true (even if the connect() has not yet completed). If something fails
     (hostname lookup etc.) then it returns false. */
bool cxnConnect (Connection cxn)
{
  struct sockaddr_in cxnAddr ;
  int fd, rval ; 
  struct hostent *hostEnt ;
  struct in_addr ipAddr ;
  struct in_addr *addr ;
  const char *peerName = hostPeerName (cxn->myHost) ;

  ASSERT (!isLocked (cxn)) ;
          
    /* see if the ipName we're given is a dotted quad */
  if ( !inet_aton (cxn->ipName,&ipAddr) )
    {
      if ((hostEnt = gethostbyname (cxn->ipName)) == NULL)
        {
          syslog (LOG_ERR, HOST_RESOLV_ERROR, peerName, cxn->ident,
                  cxn->ipName, host_err_str ()) ;

          prepareReopenCbk (cxn) ;
          
          return false ;
        }
      addr = (struct in_addr *) hostEnt->h_addr_list [0] ;
    }
  else
    addr = &ipAddr ;            /* it was a dotted quad */

  memset (&cxnAddr, 0, sizeof (cxnAddr)) ;
  cxnAddr.sin_family = AF_INET ;
  cxnAddr.sin_port = htons(cxn->port) ;
  cxnAddr.sin_addr.s_addr = addr->s_addr ;

  if ((fd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
    {
      syslog (LOG_ERR, SOCKET_CREATE_ERROR, peerName, cxn->ident) ;
      dprintf (1,"Can't get a socket: %m\n") ;

      prepareReopenCbk (cxn) ;
      return false ;
    }

    /* set our file descriptor to non-blocking */
#if defined (NBIO_FCNTL)
  rval = fcntl (fd, F_SETFL, FNDELAY) ;
#else
  {
    int state = 1 ;
    rval = ioctl (fd, FIONBIO, (char *) &state) ;
  }
#endif
  
  if (rval < 0)
    {
      syslog (LOG_ERR, FCNTL_ERROR, peerName, cxn->ident) ;
      close (fd) ;

      prepareReopenCbk (cxn) ;
      return false ;
    }

  rval = connect (fd, (struct sockaddr *) &cxnAddr, sizeof (cxnAddr)) ;
  if (rval < 0 && errno != EINPROGRESS)
    {
      syslog (LOG_ERR, CONNECT_ERROR, peerName, cxn->ident) ;
      close (fd) ;

      prepareReopenCbk (cxn) ;
      return false ;
    }

  ASSERT (cxn->myEp == NULL) ;
  cxn->myEp = newEndPoint (fd) ;

  if (rval < 0)
      /* when the write callback gets done the connection went through */
    prepareWrite (cxn->myEp, NULL, connectionDone, cxn) ;
  else
    connectionDone (cxn->myEp, IoDone, NULL, cxn) ; 
  
  return true ;
}


  /* Tell the Connection to issue a QUIT and then delete its
     endpoint. Any articles it has it flushes out first. */
void cxnDisconnect (Connection cxn)
{
  ASSERT (cxn->closing == false) ;
  ASSERT (cxn->myEp != NULL) ;
  
  cxn->closing = true ;

    /* if we're in the middle of something do nothing yet. */
  if (cxn->checkHead != NULL || cxn->checkRespHead != NULL)
    return ;
  if (cxn->takeHead != NULL || cxn->takeRespHead != NULL)
    return ;

  if (cxn->timeCon == 0)
    return ;                    /* we're in the middle of connecting  */

    /* the connection is idle so send the QUIT down the pipe. We'll cleanup
       in the write-done callback. */
  issueQUIT (cxn) ;
}


  /* Tells the Connection to do a cxnDisconnect() and then cxnConnect() */
void cxnFlush (Connection cxn)
{
  if (cxn->closing)
    ASSERT (cxn->flushing == false) ;
  else
    {
      cxn->flushing = true ;        /* so we reconnect when QUIT is complete */
      
      cxnDisconnect (cxn) ;
    }
}



  /* Tells the Connection to do a disconnect and then when it is
     disconnected to delete itself. */
void cxnClose (Connection cxn)
{
    /* we may be in the middle of a periodic close. */
  if (cxn->flushing)
    {
      dprintf (1,"%s:%d Connection already being flushed\n",
               hostPeerName (cxn->myHost),cxn->ident) ;
      removeTimeout (cxn->closeId) ;
      cxn->closeId = 0 ;
      cxn->flushing = false ;
    }
  
  if (cxn->closing)
    {
      dprintf (1,"%s:%d Connection already closing\n",
               hostPeerName (cxn->myHost),cxn->ident) ;
      return ;
    }

  if (cxn->myEp != NULL)
    {
      dprintf (1,"%s:%d Issuing disconnect\n",hostPeerName (cxn->myHost),
               cxn->ident) ;
      cxn->deleting = true ;
      cxnDisconnect (cxn) ;
    }
  else
    {
      dprintf (1,"%s:%d Deleting connection\n",hostPeerName (cxn->myHost),
               cxn->ident) ;
      hostCxnGone (cxn->myHost,cxn) ;
      delConnection (cxn) ;
    }
}


  
  /* This is what the Host calls to get us to tranfer an article. If we're
     running the IHAVE sequence, then we can't take it if we've got an
     article already. If we're running the CHECK/TAKETHIS sequence, then
     we'll take as many up to our MAXCHECK limit.
     */

bool cxnTakeArticle (Connection cxn, Article art)
{
  validateConnection (cxn) ;
  
  if ( !cxnQueueArticle (cxn,art) )
    return false ;

  dprintf (5,"%s:%d connection took article.\n",
           hostPeerName (cxn->myHost), cxn->ident) ;

  if (cxn->myEp == NULL)        /* no network connection active so make one. */
    {
      ASSERT (!isLocked (cxn)) ;
      
      if ( !cxnConnect (cxn) )
        abortConnection (cxn) ;
    }
  else if (!isLocked (cxn))
    doSomeWrites (cxn) ;
  
  return true ;
}




  /* if there's room in the Connection then stick the article on the queue,
     otherwise return false.

     We set a timer to go off in the future. If the timer does go off then
     that means that no more articles have been received from INN since the
     timer was set and so we tear down the network connection to help reuse
     fds. The next time an article is received we remove the timeout set
     previously (if it hasn't already gone off). The network connection
     will be rebuilt as needed. */

bool cxnQueueArticle (Connection cxn, Article art)
{
  ArtHolder newArt ;

  if (cxn->timeCon != 0)
    ASSERT (cxn->doesStreaming || cxn->maxCheck == 1) ;

  if (cxn->closing == true)
    {
      dprintf (5,"%s:%d Refusing article due to closing == true\n",
               hostPeerName (cxn->myHost),cxn->ident) ;
      return false ;
    }
  
    /* check if we already have an article and we're doing IHAVE (in which
       case the Connections only take one article at a time. If the
       connection is in the middle of being set up (hasn't yet checked if
       the remote does streaming), then the Host will be lied to (slightly) */
  if (!cxn->doesStreaming && cxn->articleQTotal != 0)
    {
      dprintf (5,"%s:%d refusing article due to !doesStreaming && cxn->articleQTotal != 0)\n",
               hostPeerName (cxn->myHost),cxn->ident) ;
      return false ;
    }
  
  if (cxn->articleQTotal >= cxn->maxCheck)
    {
      dprintf (5,
               "%s:%d Refusing article due to articleQTotal >= maxCheck (%d > %d)\n",
               hostPeerName (cxn->myHost), cxn->ident,
               cxn->articleQTotal, cxn->maxCheck) ;
               
      return false ;  /* am full */
    }
  
  dprintf (5,"%s:%d accepting article %s\n",hostPeerName (cxn->myHost),
           cxn->ident,artMsgId (art)) ;
  
    /* now set the timeout for article reception, but only if connected */
  if (cxn->artTout > 0 && cxn->timeCon > 0)
    {
      removeTimeout (cxn->artToutId) ;
      cxn->artToutId = prepareSleep (articleTimeoutCbk, cxn->artTout, cxn) ;
    }

  cxn->artsTaken++ ;

  newArt = newArtHolder (art) ;

    /* Now add the article to the proper queue. */
  if ((cxn->doesStreaming && cxn->needsChecks) || !cxn->doesStreaming)
    appendArtHolder (newArt, &cxn->checkHead, &cxn->articleQTotal) ;
  else
    appendArtHolder (newArt, &cxn->takeHead, &cxn->articleQTotal) ;

  return true ;
}



  /* generate a log message for activity. Usually called by the timer
     callback */
void cxnLogStats (Connection cxn, bool final)
{
  const char *peerName ;
  time_t now = time (NULL) ;
  
  ASSERT (cxn != NULL) ;

  peerName = hostPeerName (cxn->myHost) ;

  syslog (LOG_NOTICE,STATS_MSG, peerName, cxn->ident,
          (final ? "final" : "checkpoint"), (long) (now - cxn->timeCon),
          cxn->checksIssued, cxn->takesOkayed, cxn->checksRefused,
          cxn->takesRejected) ;

  if (final)
    {
      cxn->artsTaken = 0 ;
      cxn->checksIssued = 0 ;
      cxn->checksRefused = 0 ;
      cxn->takesRejected = 0 ;
      cxn->takesOkayed = 0 ;

      if (cxn->timeCon > 0)
        cxn->timeCon = time (NULL) ;
    }
}



  /* return the number of articles the connection will accept. */
size_t cxnQueueSpace (Connection cxn)
{
  int rval = 0 ;
  
  ASSERT (cxn != NULL) ;

  if (!cxn->closing)
    rval = cxn->maxCheck - cxn->articleQTotal ;

  return rval ;
}







  /**********************************************************************/
  /**                       STATIC PRIVATE FUNCTIONS                   **/
  /**********************************************************************/




  /* this is the first stage of the NNTP FSM. This function is called when
     the network connection is setup and we should get ready to read the
     banner message.  */
static void connectionDone (EndPoint e, IoStatus i, Buffer *b, void *d)
{
  Buffer *readBuffers ;
  Connection cxn = (Connection) d ;
  const char *peerName ;
  int optval, rval, size ;
  
  ASSERT (b == NULL) ;
  ASSERT (cxn->timeCon == 0) ;
  ASSERT (cxn->artToutId == 0) ;
  ASSERT (cxn->respToutId == 0) ;

  size = sizeof (optval) ;
  peerName = hostPeerName (cxn->myHost) ;

  if (i != IoDone)
    {
      errno = endPointErrno (e) ;
      syslog (LOG_ERR,IO_FAILED,peerName,cxn->ident) ;

      abortConnection (cxn) ;
    }
    /* if the connect failed then the only way to know is by getting
       the SO_ERROR value out of the socket. */
  else if ((rval = getsockopt (endPointFd (e), SOL_SOCKET, SO_ERROR,
                               (GETSOCKOPT_ARG) &optval, &size)) != 0)
    {
      syslog (LOG_ERR,GETSOCKOPT_FAILED, peerName, cxn->ident) ;

      abortConnection (cxn) ;
    }
  else if (optval != 0)
    {
      errno = optval ;
      
      syslog (LOG_NOTICE,CONNECTION_FAILURE,peerName,cxn->ident) ;

      abortConnection (cxn) ;
    }
  else
    {
      cxn->timeCon = time (NULL) ;

      removeTimeout (cxn->reopenId) ;
      cxn->reopenId = 0 ;
      
      readBuffers = makeBufferArray (bufferTakeRef (cxn->respBuffer), NULL) ;
      
      if ( !prepareRead (e, readBuffers, getBanner, cxn, 1) )
        {
          syslog (LOG_ERR, PREPARE_READ_FAILED, peerName, cxn->ident) ;
          
          abortConnection (cxn) ;
        }
      else
        {
          setUpResponseTimeout (cxn) ;

            /* set up the callback for closing down the connection at regular
               intervals (due to problems with long running nntpd). */
          ASSERT (cxn->closeId == 0);
          if (cxn->closePeriod > 0)
            cxn->closeId = prepareSleep (periodicCxnCloseCbk,
                                         cxn->closePeriod,cxn) ;
          
            /* and for idle connections. */
          ASSERT (cxn->artToutId == 0);
          if (cxn->artTout > 0)
            cxn->artToutId = prepareSleep (articleTimeoutCbk,
                                           cxn->artTout, cxn) ;
        }
    }
}


  /* a processing error has occured (for example in parsing a
     response). We'll abort up the connection and notify the Host */
static void abortConnection (Connection cxn)
{
  ASSERT (cxn != NULL) ;
  ASSERT (cxn->reopenId == 0) ;
  
  if (cxn->myEp != NULL)
    {
      delEndPoint (cxn->myEp) ;
      cxn->myEp = NULL ;
    }

  cxn->closing = false ;
  cxn->flushing = false ;
  cxn->timeCon = 0 ;
  cxn->maxCheck = 1 ;

  removeTimeout (cxn->artToutId) ;
  cxn->artToutId = 0 ;

  removeTimeout (cxn->respToutId) ;
  cxn->respToutId = 0 ;

  removeTimeout (cxn->closeId) ;
  cxn->closeId = 0 ;

  deferAllArticles (cxn) ;      /* give any articles back to Host */

  bufferSetDataSize (cxn->respBuffer,0) ;
  
  if (isLocked (cxn))
    unLockCxn (cxn) ;

  hostCxnDead (cxn->myHost, cxn) ;

  if (!cxn->deleting && !cxn->flushing)
    prepareReopenCbk (cxn) ;
}


static void prepareReopenCbk (Connection cxn)
{
  ASSERT (cxn->reopenId == 0) ;

  dprintf (1,"%s:%d Setting up a reopen callback\n",
           hostPeerName (cxn->myHost), cxn->ident) ;
  
  cxn->reopenId = prepareSleep (reopenTimeoutCbk, cxn->reopenWait, cxn) ;

  cxn->reopenWait *= 2 ;
  if (cxn->reopenWait > MAX_REESTABLISHMENT_PERIOD)
    cxn->reopenWait = MAX_REESTABLISHMENT_PERIOD ;
}


void reopenTimeoutCbk (TimeoutId id, void *data)
{
  Connection cxn = (Connection) data ;

  ASSERT (id == cxn->reopenId) ;
  
  cxn->reopenId = 0 ;
  cxnConnect (cxn) ;
}




  /* reset all state variables to inital condition. */
static void resetConnection (Connection cxn)
{
  bufferSetDataSize (cxn->respBuffer,0) ;
  
  cxn->closing = false ;
  cxn->doesStreaming = false ;  /* who knows, next time around maybe... */
  cxn->needsChecks = true ;
  cxn->flushing = false ;
  cxn->periodicClose = false ;
  cxn->timeCon = 0 ;

  cxn->artsTaken = 0 ;
  cxn->checksIssued = 0 ;
  cxn->checksRefused = 0 ;
  cxn->takesRejected = 0 ;
  cxn->takesOkayed = 0 ;

  cxn->filterValue = 0.0 ;

  if (isLocked (cxn))
    unLockCxn (cxn) ;
}

  
  
  /* Give back any articles we have to the Host for later retrys. */
static void deferAllArticles (Connection cxn)
{
  ArtHolder p, q ;
  bool oldClosing = cxn->closing ;

    /* so that hostArticleDeferred doesn't queue any more up */
  cxn->closing = true ;
  
  for (q = NULL, p = cxn->checkHead ; p != NULL ; p = q)
    {
      q = p->next ;
      hostArticleDeferred (cxn->myHost, cxn, p->article) ;
      delArtHolder (p) ;
      cxn->articleQTotal-- ;
    }
  cxn->checkHead = NULL ;
  
  for (q = NULL, p = cxn->checkRespHead ; p != NULL ; p = q)
    {
      q = p->next ;
      hostArticleDeferred (cxn->myHost, cxn, p->article) ;
      delArtHolder (p) ;
      cxn->articleQTotal-- ;
    }
  cxn->checkRespHead = NULL ;
  
  for (q = NULL, p = cxn->takeHead ; p != NULL ; p = q)
    {
      q = p->next ;
      hostArticleDeferred (cxn->myHost, cxn, p->article) ;
      delArtHolder (p) ;
      cxn->articleQTotal-- ;
    }
  cxn->takeHead = NULL ;
  
  for (q = NULL, p = cxn->takeRespHead ; p != NULL ; p = q)
    {
      q = p->next ;
      hostArticleDeferred (cxn->myHost, cxn, p->article) ;
      delArtHolder (p) ;
      cxn->articleQTotal-- ;
    }
  cxn->takeRespHead = NULL ;

  ASSERT (cxn->articleQTotal == 0) ;
  
  cxn->closing = oldClosing ;
}




  /* timeout callback to close down long running connection. */
static void periodicCxnCloseCbk (TimeoutId id, void *data)
{
  Connection cxn = (Connection) data ;

  ASSERT (id == cxn->closeId) ;

  cxn->closeId = 0 ;
  cxn->periodicClose = true ;

  removeTimeout (cxn->artToutId) ;
  cxn->artToutId = 0 ;
 
  removeTimeout (cxn->respToutId) ;
  cxn->respToutId = 0 ;
  
  dprintf (1,"%s:%d Handling periodic connection close.\n",
           hostPeerName (cxn->myHost), cxn->ident) ;
  
  cxnFlush (cxn) ;
}


  /*
   * Timer callback for when the connection has not received an article from
   * INN. When that happens we tear down the network connection to help
   * recycle fds
   */
static void articleTimeoutCbk (TimeoutId id, void *data)
{
  Connection cxn = (Connection) data ;
  const char *peerName ;

  (void) id ;                   /* keep lint happy */

  ASSERT (cxn->artTout > 0) ;
  ASSERT (cxn->artToutId == id) ;

  cxn->artToutId = 0 ;
  
  peerName = hostPeerName (cxn->myHost) ;

  if (cxn->articleQTotal > 0)
    {
      syslog (LOG_WARNING, ARTICLE_TIMEOUT_W_Q_MSG, peerName, cxn->ident) ;
      cxn->artToutId = prepareSleep (articleTimeoutCbk, cxn->artTout, cxn) ;
    }
  else
    {
      syslog (LOG_WARNING, ARTICLE_TIMEOUT_MSG, peerName, cxn->ident) ;
      cxnDisconnect (cxn) ;
    }
}


  /* function to be called when the fd is not ready, but there is an
     article on tape or in the queue to be done. */
static void cxnWorkProc (EndPoint ep, void *data)
{
  Connection cxn = (Connection) data ;

  (void) ep ;                   /* keep lint happy */
  
  dprintf (2,"%s:%d calling work proc\n",
           hostPeerName (cxn->myHost),cxn->ident) ;
  
  if (writesNeeded (cxn))
    doSomeWrites (cxn) ;
  else
    {
      dprintf (2,"%s:%d no writes were needed....\n",
               hostPeerName (cxn->myHost), cxn->ident) ;
      addWorkCallback (cxn->myEp,NULL,NULL) ;
    }

  if (cxn->closing && !writesNeeded (cxn))
    issueQUIT (cxn) ;
}


  /* Called when there's probably an article to be pushed out to the
     remote. Even if the Connection has an article it's possible that
     nothing will be written if the article doesn't exist any more */
static void doSomeWrites (Connection cxn)
{
  bool doneSome = false ;
  
  ASSERT (!isLocked (cxn)) ;

    /* If there's a write pending we can't do anything now. */
  if ( writeIsPending (cxn->myEp) )
    return ;
  else if ( writesNeeded (cxn) )
    {
      if (cxn->doesStreaming)
        doneSome = issueStreamingCommands (cxn) ;
      else
        doneSome = issueIHAVE (cxn) ;

      if (!doneSome && (writesNeeded (cxn) || cxn->closing))
        {
            /* the article(s) on the queue were missing and no write was
               prepared. Set up the work proc so that next time through the
               select loop we can try again. */
          dprintf (2,"%s:%d Setting up work proc\n",
                   hostPeerName (cxn->myHost),cxn->ident) ;
          
          addWorkCallback (cxn->myEp,cxnWorkProc,cxn) ;
        }
    }
  else if (cxn->closing)
    {
      issueQUIT (cxn) ;
      doneSome = true ;
    }

  return  ;
}


  /* Queue up a buffer with the IHAVE command in it for the article at the
     head of the transmisson queue */
static bool issueIHAVE (Connection cxn)
{
  Buffer ihaveBuff, *writeArr ;
  ArtHolder artH ;
  Article article ;
  const char *msgid ;
  char *p ;
  u_int tmp ;
  size_t bufLen = 256 ;
  bool rval = false ;

  ASSERT (cxn->checkHead != NULL) ;
  ASSERT (cxn->takeHead == NULL) ;
  ASSERT (cxn->checkRespHead == NULL) ;
  ASSERT (cxn->takeRespHead == NULL) ;
  ASSERT (cxn->articleQTotal == 1) ;

  artH = cxn->checkHead ;
  article = cxn->checkHead->article ;
  msgid = artMsgId (artH->article) ;

  if (msgid == NULL)
    die ("msg id == NULL (%p)",artH->article) ;
  
  ASSERT (msgid != NULL) ;      /* I want the core dump right now. */

  if ( !artFileIsValid (artH->article) )
    {
      cxn->checkHead = NULL ;
      cxn->articleQTotal-- ;
      ASSERT (cxn->articleQTotal == 0) ;

      hostArticleIsMissing (cxn->myHost, cxn, artH->article) ; 

      delArtHolder (artH) ;
    }
  else
    {
      if ((tmp = (strlen (msgid) + 10)) > bufLen)
        bufLen = tmp ;

      ihaveBuff = newBuffer (bufLen) ;

      ASSERT (ihaveBuff != NULL) ;

      p = bufferBase (ihaveBuff) ;
      sprintf (p, "IHAVE %s\r\n", msgid) ;
      bufferSetDataSize (ihaveBuff, strlen (p)) ;

      dprintf (5,"%s:%d Command IHAVE %s\n",
               hostPeerName (cxn->myHost),cxn->ident,msgid) ;

      writeArr = makeBufferArray (ihaveBuff, NULL) ;
      prepareWrite (cxn->myEp, writeArr, checkWriteDone, cxn) ;

        /* now move the article to the second queue */
      cxn->checkRespHead = cxn->checkHead ;
      cxn->checkHead = NULL ;

      cxn->checksIssued++ ;
      hostArticleOffered (cxn->myHost, cxn) ;

      rval = true ;
    }

  return rval ;
}

  /*
   * Do a prepare write with the article body as the body portion of the
   * IHAVE command
   */
static void issueIHAVEBody (Connection cxn)
{
  Buffer *writeArray ;
  Article article ;

  ASSERT (cxn != NULL) ;
  ASSERT (cxn->takeHead != NULL) ;
  ASSERT (cxn->takeRespHead == NULL) ;
  ASSERT (cxn->checkHead == NULL) ;
  ASSERT (cxn->checkRespHead == NULL) ;
  ASSERT (cxn->takeHead->next == NULL) ;
  ASSERT (cxn->articleQTotal == 1) ;

  article = cxn->takeHead->article ;

  writeArray = artGetNntpBuffers (article) ;

  if ( !prepareWrite (cxn->myEp, writeArray, ihaveBodyDone, cxn) )
    die ("Preparewrite failed in issueIHAVEBody") ;

    /* now move the article to the last queue */
  cxn->takeRespHead = cxn->takeHead ;
  cxn->takeHead = NULL ;
  
  return ;
}



  /* Process the two command queues. Slaps all the CHECKs together and then
     does the TAKETHIS commands. */

static bool issueStreamingCommands (Connection cxn)
{
  Buffer checkBuffer = NULL ;   /* the buffer with the CHECK commands in it. */
  Buffer *writeArray = NULL ; 
  ArtHolder p ;
  bool rval = false ;

  ASSERT (cxn != NULL) ;
  ASSERT (cxn->myEp != NULL) ;

  checkBuffer = buildCheckBuffer (cxn) ; /* may be null if none to issue */

  writeArray = buildTakethisBuffers (cxn,checkBuffer) ; /* may be null */

    /* If not null, then writeArray will have checkBuffer (if it wasn't NULL)
       in the first spot and the takethis buffers after that. */
  if (writeArray)
    {
      prepareWrite (cxn->myEp, writeArray, checkWriteDone, cxn) ;
      rval = true ;

        /* Now shift the articles to their new queues. */
      for (p = cxn->checkRespHead ; p != NULL && p->next != NULL ; p = p->next)
          /* nada--finding end of queue*/ ;
      
      if (p == NULL)
        cxn->checkRespHead = cxn->checkHead ;
      else
        p->next = cxn->checkHead ;
      cxn->checkHead = NULL ;
      
      for (p = cxn->takeRespHead ; p != NULL && p->next != NULL ; p = p->next)
          /* nada--finding end of queue */ ;
      
      if (p == NULL)
        cxn->takeRespHead = cxn->takeHead ;
      else
        p->next = cxn->takeHead ;
      
      cxn->takeHead = NULL ;
    }

  return rval ;
}


  /* buil up the buffer of all the CHECK commands. */
static Buffer buildCheckBuffer (Connection cxn)
{
  ArtHolder p, q ;
  size_t lenBuff = 0 ;
  Buffer checkBuffer = NULL ;
  const char *peerName = hostPeerName (cxn->myHost) ;
  
  p = cxn->checkHead ;
  q = NULL ;
  while (p != NULL)
    {
      Article article = p->article ;
      const char *msgid ;
      
      msgid = artMsgId (article) ;

      lenBuff += (8 + strlen (msgid)) ; /* 8 == strlen("CHECK \r\n") */
      p = p->next ;
    }

  if (lenBuff > 0)
    lenBuff++ ;                 /* for the null byte */

    /* now build up the single buffer that contains all the CHECK commands */
  if (lenBuff > 0)
    {
      char *t ;
      size_t tlen = 0 ;

      checkBuffer = newBuffer (lenBuff) ;
      t = bufferBase (checkBuffer) ;

      p = cxn->checkHead ;
      while (p != NULL)
        {
          const char *msgid = artMsgId (p->article) ;

          q = p ;

          sprintf (t,"CHECK %s\r\n", msgid) ;
          dprintf (5,"%s:%d Command %s", peerName, cxn->ident, t) ;

          tlen += strlen (t) ;
          
          while ( *t ) t++ ;

          cxn->checksIssued++ ;
          hostArticleOffered (cxn->myHost,cxn) ;
            
          p = p->next ;
        }

      ASSERT (tlen + 1 == lenBuff) ;
      
      bufferSetDataSize (checkBuffer, tlen) ;
    }

  return checkBuffer ;
}



static Buffer *buildTakethisBuffers (Connection cxn, Buffer checkBuffer)
{
  size_t lenArray = 0 ;
  ArtHolder p, q ;
  Buffer *rval = NULL ;
  const char *peerName = hostPeerName (cxn->myHost) ;

  if (checkBuffer != NULL)
    lenArray++ ;
  
  if (cxn->takeHead != NULL)    /* some TAKETHIS commands to be done. */
    {
      Buffer takeBuffer ;
      u_int takeBuffLen  ;
      u_int writeIdx = 0 ;
      
        /* count up all the buffers we'll be writing. One extra each time for
           the TAKETHIS buffer*/

      for (p = cxn->takeHead ; p != NULL ; p = p->next)
        if (artContentsOk (p->article))
            lenArray += (1 + artNntpBufferCount (p->article)) ;

        /* now allocate the array for the buffers and put them all in it */
      rval = ALLOC (Buffer, lenArray + 1) ; /* 1 for the terminator */
      ASSERT (rval != NULL) ;
      
      if (checkBuffer != NULL)
        rval [writeIdx++] = checkBuffer ;

      q = NULL ;
      p = cxn->takeHead ;
      while (p != NULL)
        {
          char *t ;
          const char *msgid ;
          Article article ;
          Buffer *articleBuffers ;
          int i, nntpLen ;

          article = p->article ;
          nntpLen = artNntpBufferCount (article) ;
          msgid = artMsgId (article) ;

          if (nntpLen == 0)
            {                   /* file no longer valid so drop from queue */
              ArtHolder ta = p ;
              
              if (q == NULL)    /* it's the first in the queue */
                cxn->takeHead = p->next ;
              else
                q->next = p->next ;
              
              p = p->next ;
              ASSERT (cxn->articleQTotal > 0) ;
              cxn->articleQTotal-- ;
              
              hostArticleIsMissing (cxn->myHost, cxn, article) ;
              
              delArtHolder (ta) ;
            }
          else
            {
              articleBuffers = artGetNntpBuffers (article) ;
              
                /* set up the buffer with the TAKETHIS command in it.
                   12 == strlen ("TAKETHIS \n\r") */
              takeBuffLen = 12 + strlen (msgid) ;
              takeBuffer = newBuffer (takeBuffLen) ;
              t = bufferBase (takeBuffer) ;
              
              sprintf (t, "TAKETHIS %s\r\n", msgid) ;
              bufferSetDataSize (takeBuffer, strlen (t)) ;
              
              dprintf (5,"%s:%d Command %s", peerName, cxn->ident, t) ;
              
              rval [writeIdx++] = takeBuffer ;
              
                /* now add all the buffers that make up the body of the TAKETHIS
                   command  */
              for (i = 0 ; i < nntpLen ; i++)
                rval [writeIdx++] = bufferTakeRef (articleBuffers [i]) ;

#ifdef DEADBEEF
              deadBeef (articleBuffers, sizeof (Buffer *) * nntpLen) ;
#endif
              
              freeBufferArray (articleBuffers) ; 

              if ( !cxn->needsChecks )
                {
                    /* this isn't quite right. An article may be counted
                       twice if we switch to TAKETHIS-only mode after its
                       CHECK was issued, but before its TAKETHIS was done
                       just now. I'm not going to worry unless someone
                       complains. */
                  
                  cxn->checksIssued++ ;
                  hostArticleOffered (cxn->myHost,cxn) ;
                }

              p = p->next ;
            }
        }
      rval [writeIdx++] = NULL ;
    }
  else if (checkBuffer != NULL) /* no TAKETHIS to do, but some CHECKS */
    rval = makeBufferArray (checkBuffer, NULL) ;

  return rval ;
}


  /**********************************************************************

    ENDPOINT CALL BACK AREA.
    
    This section of code is where the callback functions that the EndPoint
    driver routines call into

    **********************************************************************/


  /* called when the write of the IHAVE-body data is finished */

static void ihaveBodyDone (EndPoint e, IoStatus i, Buffer *b, void *d)
{
  Connection cxn = (Connection) d ;

  ASSERT (e == cxn->myEp) ;

  if (i != IoDone)
    {
      errno = endPointErrno (e) ;
      syslog (LOG_ERR, IHAVE_WRITE_FAILED, hostPeerName (cxn->myHost),
              cxn->ident) ;

      abortConnection (cxn) ;
    }

  freeBufferArray (b) ;
    
  return ;
}

  /* called when a response has been read from the socket. This is where
     the bulk of the processing starts. */
static void responseIsRead (EndPoint e, IoStatus i, Buffer *b, void *d)
{
  Connection cxn = (Connection) d ;
  int code ;
  char *response ;
  char *endr ;
  char *bufBase ;
  u_int respSize ;
  char *rest = NULL ;
  Buffer buf ;
  const char *peerName ;
  bool cxnStillGood = false ;

  ASSERT (e == cxn->myEp) ;
  ASSERT (b != NULL) ;
  ASSERT (b [1] == NULL) ;
  ASSERT (b [0] == cxn->respBuffer) ;
  ASSERT (!isLocked (cxn)) ;
  
  bufferAddNullByte (b [0]) ;
  
  lockCxn (cxn) ;
  
  peerName = hostPeerName (cxn->myHost) ;
  
  if (i != IoDone)
    {                           /* uh oh. */
      errno = endPointErrno (e) ;
      syslog (LOG_ERR, RESPONSE_READ_FAILED, peerName, cxn->ident) ;

      freeBufferArray (b) ;

      abortConnection (cxn) ;
      
      return ;
    }

  buf = b [0] ;
  bufBase = bufferBase (buf) ;
      
    /* check that we have a full line response. If not expand the buffer and
       resubmit the read. */
  if (strchr (bufBase, '\n') == 0)
    {
      if (!expandBuffer (buf, BUFFER_EXPAND_AMOUNT))
        {
          syslog (LOG_ERR, CXN_BUFFER_EXPAND_ERROR, peerName, cxn->ident) ;
              
          freeBufferArray (b) ;
          
          abortConnection (cxn) ;
        }
      else if ( !prepareRead (cxn->myEp, b, responseIsRead, cxn, 1))
        {
          syslog (LOG_ERR, PREPARE_READ_FAILED, peerName, cxn->ident) ;

          freeBufferArray (b) ;
          
          abortConnection (cxn) ;
        }
      else
        {
          setUpResponseTimeout (cxn) ;
          unLockCxn (cxn) ;
        }
      
      return ;
    }
  
  freeBufferArray (b) ; /* connection still has reference to buffer */
      

    /* Now process all the full responses that we have. */
  response = bufBase ;
  respSize = bufferDataSize (cxn->respBuffer) ;
      
  while ((endr = strchr (response, '\n')) != NULL)
    {
      char *next = endr + 1 ;

      cxnStillGood = false ;
          
      if (*next == '\r')
        next++ ;
          
      endr-- ;
      if (*endr != '\r')
        endr++ ;

      if (next - endr != 2)     /* only a newline there. we'll live with it */
        syslog (LOG_ERR, NOCR_MSG, peerName, cxn->ident) ;

      *endr = '\0' ;

      if ( !getNntpResponse (response, &code, &rest) )
        {
          syslog (LOG_ERR, INVALID_RESP_FORMAT, peerName,
                  cxn->ident, response) ;

          abortConnection (cxn) ;

          return ;
        }

      dprintf (5,"%s:%d Response %d: %s\n", peerName, cxn->ident, code, response) ;

        /* now handle the response code. I'm not using symbolic names on
           purpose--the numbers are all you see in the RFC's. */
      switch (code)
        {
            /* The 205 may be in response to a periodic close, in which
               case the connection will have started a reestablishment of
               the network linke and cxnStillGood will be true. */
          case 205:             /* remote is closing down. */
            cxnStillGood = processResponse205 (cxn, response) ;
            break ;


            
              /* These three are from the CHECK command */
          case 238:             /* no such article found */
            incrFilter (cxn) ;
            cxnStillGood = processResponse238 (cxn, response) ;
            break ;

          case 431:             /* try again later (also for TAKETHIS) */
            decrFilter (cxn) ;
            cxnStillGood = processResponse431 (cxn, response) ;
            break ;

          case 438:             /* already have it */
            decrFilter (cxn) ;
            cxnStillGood = processResponse438 (cxn, response) ;
            break ;


            
              /* These are from the TAKETHIS command */
          case 239:             /* article transferred OK */
            incrFilter (cxn) ;
            cxnStillGood = processResponse239 (cxn, response) ;
            break ;

          case 439:             /* article rejected */
            decrFilter (cxn) ;
            cxnStillGood = processResponse439 (cxn, response) ;
            break ;

            
            
              /* These are from the IHAVE command */
          case 335:             /* send article */
            cxnStillGood = processResponse335 (cxn, response) ;
            break ;

          case 435:             /* article not wanted */
            cxnStillGood = processResponse435 (cxn, response) ;
            break ;

          case 436:             /* transfer failed try again later */
            cxnStillGood = processResponse436 (cxn, response) ;
            break ;

          case 437:             /* article rejected */
            cxnStillGood = processResponse437 (cxn, response) ;
            break ;

          case 400:             /* has stopped accepting articles */
            cxnStillGood = processResponse400 (cxn, response) ;
            break ;


            
              /* This is from the IHAVE-body */
          case 235:             /* article transfered OK (IHAVE-body*/
            cxnStillGood = processResponse235 (cxn, response) ;
            break ;


            
          default:
            syslog (LOG_ERR, UNKNOWN_RESPONSE, peerName, cxn->ident,
                    code, response) ;
            break ;
        }

      if ( !cxnStillGood )      /* we think we're stuffed */
        break ;
      
      validateConnection (cxn) ;

      response = next ;
    }

  dprintf (5,"%s:%d done with responses\n",hostPeerName (cxn->myHost),
           cxn->ident) ;
  
  if (cxnStillGood)
    {
      Buffer *bArr ;
      
        /* see if we need to drop in to or out of TAKETHIS-only mode */
      if (cxn->doesStreaming)
        {
          static changeNotified ;
          
          if ((cxn->filterValue > cxn->onThreshold) && cxn->needsChecks)
            {
              if (!changeNotified)
                {
                  syslog (LOG_NOTICE, STREAMING_MODE_SWITCH,
                          peerName, cxn->ident) ;
                  changeNotified = true ;
                }
              cxn->needsChecks = false ;
            }
          else if ((cxn->filterValue < cxn->offThreshold) &&
                   !cxn->needsChecks) 
            {
#if 0
              syslog (LOG_NOTICE, STREAMING_MODE_UNDO, peerName, cxn->ident) ;
#endif
              cxn->needsChecks = true ;
            }
        }
      

        /* We're done processing responses, so we're left with a partial
           response or an empty buffer */
      if (code != 205)
        {
          if (*response != '\0')
            {                       /* partial response */
              u_int leftAmt = respSize - (response - bufBase) ;

              dprintf (2,"%s:%d handling a partial response\n",
                       hostPeerName (cxn->myHost),cxn->ident) ;
              
                /* first we shift whats left in the buffer down to the
                   bottom */
              if (response != bufBase)
                {
                  memcpy (bufBase, response, leftAmt) ;
                  
                    /* the next read starts at the end of the data region in
                       the buffer. */
                  bufferSetDataSize (cxn->respBuffer, leftAmt) ;
                }
              else if (!expandBuffer (cxn->respBuffer, BUFFER_EXPAND_AMOUNT))
                {   /* expand 'cause buffer is full, but wasn't a complete
                       response. Unlikely to happen. */
                  syslog (LOG_ERR,CXN_BUFFER_EXPAND_ERROR,peerName,cxn->ident);
                  
                  abortConnection (cxn) ;
                  
                  return ;
                }
            }
          else
            bufferSetDataSize (cxn->respBuffer, 0) ;

          bArr = makeBufferArray (bufferTakeRef (cxn->respBuffer), NULL) ;
              
          if ( !prepareRead (e, bArr, responseIsRead, cxn, 1) )
            {
              syslog (LOG_ERR, PREPARE_READ_FAILED, peerName, cxn->ident) ;
              
              freeBufferArray (bArr) ;
              
              abortConnection (cxn) ;
              
              return ;
            }
          else
            {
              setUpResponseTimeout (cxn) ;

              unLockCxn (cxn) ;
              
              validateConnection (cxn) ;

              dprintf (5,"%s:%d about to do some writes\n",
                       hostPeerName (cxn->myHost),cxn->ident) ;
              
              doSomeWrites (cxn) ;
          
              validateConnection (cxn) ;
            }
        }
    }
  else if (cxn->deleting)
    {
      dprintf (1,"%s:%d Connection is being deleted\n",
               hostPeerName (cxn->myHost), cxn->ident) ;
      abortConnection (cxn) ;
      delConnection (cxn) ;
    }
  else
    {
      dprintf (1,"%s:%d Connection no longer any good.\n",
               hostPeerName (cxn->myHost), cxn->ident) ;
      abortConnection (cxn) ;   /* will set up reconnect callback */
    }
}



  /* Called when a command set (IHAVE, CHECK, TAKETHIS) has been written to
     the remote.  */
static void checkWriteDone (EndPoint e, IoStatus i, Buffer *b, void *d)
{
  Connection cxn = (Connection) d ;
  const char *peerName ;

  ASSERT (e == cxn->myEp) ;

  peerName = hostPeerName (cxn->myHost) ;
  
  freeBufferArray (b) ;
  
  if (i != IoDone)
    {
      errno = endPointErrno (e) ;
      syslog (LOG_ERR, COMMAND_WRITE_FAILED, peerName, cxn->ident) ;

      abortConnection (cxn) ;
    }
  else 
    {
        /* for an IHAVE we must wait for the response */
      if ( !cxn->doesStreaming )        
        return ;

        /* now we try to do more work if we can. */
      issueStreamingCommands (cxn) ;
    }
  
  return ;
}


  /* Called when the banner message has been read off the wire and is in the
     buffer(s). */
static void getBanner (EndPoint e, IoStatus i, Buffer *b, void *d)
{
  Buffer *modeCmdBuffers, *readBuffers ;
  Connection cxn = (Connection) d ;
  char *p = bufferBase (b[0]) ;
  int code ;
  bool isOk = false ;
  const char *peerName ;
  char *rest ;

  ASSERT (e == cxn->myEp) ;
  ASSERT (b[0] == cxn->respBuffer) ;
  ASSERT (b[1] == NULL) ;

  peerName = hostPeerName (cxn->myHost) ;

  bufferAddNullByte (b[0]) ;

  if (i == IoFailed)
    {
      errno = endPointErrno (cxn->myEp) ;
      syslog (LOG_ERR, BANNER_READ_FAILED, peerName, cxn->ident) ;

      abortConnection (cxn) ;
    }
  else if (strchr (p, '\n') == NULL)
    {                           /* partial read. expand buffer and retry */
      expandBuffer (b[0], BUFFER_EXPAND_AMOUNT) ;
      readBuffers = makeBufferArray (bufferTakeRef (b[0]), NULL) ;

      if ( !prepareRead (e, readBuffers, getBanner, cxn, 1) )
        {
          syslog (LOG_ERR, PREPARE_READ_FAILED, peerName, cxn->ident) ;
          
          abortConnection (cxn) ;
        }
      else
        setUpResponseTimeout (cxn) ;
    }
  else if ( !getNntpResponse (p, &code, &rest) )
    {
      trim_ws (p) ;
      
      syslog (LOG_ERR, RESPONSE_BAD_FORMAT, peerName, cxn->ident, p) ;

      abortConnection (cxn) ;
    }
  else
    {
      trim_ws (p) ;
      
      switch (code) 
        {
          case 200:             /* normal */
          case 201:             /* can transfer but not post -- old nntpd */
            isOk = true ;
            break ;

          case 400:
            abortConnection (cxn) ;
            hostCxnBlocked (cxn->myHost, cxn, rest) ;
            break ;

          default:
            syslog (LOG_NOTICE,UNKNOWN_BANNER, peerName, cxn->ident, code, p) ;
            dprintf (1,"%s:%d Unknown response code: %d: %s\n",
                     hostPeerName (cxn->myHost),cxn->ident, code, p) ;
            break ;
        }

      if ( isOk )
        {
          Buffer modeBuffer ;

#define  MODE_CMD "MODE STREAM\r\n"

          modeBuffer = newBuffer (strlen (MODE_CMD) + 1) ;
          p = bufferBase (modeBuffer) ;

            /* now issue the MODE STREAM command */
          dprintf (1,"%s:%d Issuing the streaming command: %s",
                   hostPeerName (cxn->myHost),cxn->ident,MODE_CMD) ;
          
          strcpy (p, MODE_CMD) ;

          bufferSetDataSize (modeBuffer, strlen (p)) ;

          modeCmdBuffers = makeBufferArray (modeBuffer, NULL) ;
          
          if ( !prepareWrite (e, modeCmdBuffers, modeCmdIssued, cxn) )
            {   
              syslog (LOG_ERR, PREPARE_WRITE_FAILED, peerName, cxn->ident) ;
              die ("Prepare write for mode stream failed") ;
            }
          
          bufferSetDataSize (cxn->respBuffer, 0) ;

          readBuffers = makeBufferArray (bufferTakeRef(cxn->respBuffer),NULL);

          if ( !prepareRead (e, readBuffers, getModeResponse, cxn, 1) )
            {
              syslog (LOG_ERR, PREPARE_READ_FAILED, peerName, cxn->ident) ;

              freeBufferArray (readBuffers) ;
              
              abortConnection (cxn) ;
            }
          else
            setUpResponseTimeout (cxn) ;
        }
    }

  freeBufferArray (b) ;
}



  /* Called when the MODE STREAM command has been written down the pipe. */
static void modeCmdIssued (EndPoint e, IoStatus i, Buffer *b, void *d)
{
  Connection cxn = (Connection) d ;

  ASSERT (e == cxn->myEp) ;

  if (i != IoDone)
    {
      dprintf (1,"%s:%d MODE STREAM command failed to write\n",
               hostPeerName (cxn->myHost), cxn->ident) ;

      syslog (LOG_ERR,MODE_STREAM_FAILED,hostPeerName (cxn->myHost),
              cxn->ident) ;
      
      abortConnection (cxn) ;
    }

  freeBufferArray (b) ;
}

  /* Process the remote's response to our MODE STREAM command */
static void getModeResponse (EndPoint e, IoStatus i, Buffer *b, void *d)
{
  Connection cxn = (Connection) d ;
  int code ;
  char *p = bufferBase (b[0]) ;
  Buffer *buffers ;
  const char *peerName ;

  ASSERT (e == cxn->myEp) ;
  ASSERT (b [0] == cxn->respBuffer) ;
  ASSERT (b [1] == NULL) ;      /* only ever one buffer on this read */
  ASSERT (!isLocked (cxn)) ;

  lockCxn (cxn) ;
  
  peerName = hostPeerName (cxn->myHost) ;

  bufferAddNullByte (b[0]) ;
  
  dprintf (1,"%s:%d Processing mode response: %s", /* no NL */
           hostPeerName (cxn->myHost), cxn->ident, p) ;
  
  if (i != IoDone)
    {
      errno = endPointErrno (e) ;
      syslog (LOG_ERR, RESPONSE_READ_FAILED, peerName, cxn->ident) ;

      abortConnection (cxn) ;
    }
  else if (strchr (p, '\n') == NULL) 
    {                           /* partial read */
      expandBuffer (b [0], BUFFER_EXPAND_AMOUNT) ;

      buffers = makeBufferArray (bufferTakeRef (b [0]), NULL) ;
      if ( !prepareRead (e, buffers, getModeResponse, cxn, 1) )
        {
          syslog (LOG_ERR, PREPARE_READ_FAILED, peerName, cxn->ident) ;

          freeBufferArray (buffers) ;
          
          abortConnection (cxn) ;
        }
      else
        {
          setUpResponseTimeout (cxn) ;
          unLockCxn (cxn) ;
        }
    }
  else if ( !getNntpResponse (p, &code, NULL) )
    {
      syslog (LOG_ERR, BAD_MODE_RESPONSE, peerName, cxn->ident, p) ;

      abortConnection (cxn) ;
    }
  else
    {
      syslog (LOG_NOTICE,CONNECTED,peerName, cxn->ident) ;
      
      switch (code)
        {
          case 203:                     /* will do streaming */
            hostRemoteStreams (cxn->myHost, cxn, true) ;
            cxn->doesStreaming = true ;
            cxn->maxCheck = hostMaxChecks (cxn->myHost) ;
            break ;

          default:                      /* won't do it */
            hostRemoteStreams (cxn->myHost, cxn, false) ;
            cxn->maxCheck = 1 ;
            break ;
        }

        /* one for the connection and one for the buffer array */
      ASSERT (bufferRefCount (cxn->respBuffer) == 2) ;

        /* there was only one line in there, right? */
      bufferSetDataSize (cxn->respBuffer, 0) ;
      buffers = makeBufferArray (bufferTakeRef (cxn->respBuffer), NULL) ;

      cxn->reopenWait = INITIAL_REESTABLISHMENT_PERIOD ;
      
        /* now we wait for articles from our Host, or we have some articles
           already. On infrequently used connections, the network link is
           torn down and rebuilt as needed. So we may be rebuilding the
           connection here in which case we have an article to send. */

      if ( !prepareRead (cxn->myEp, buffers, responseIsRead, cxn, 1) )
        {
          freeBufferArray (buffers) ;

          abortConnection (cxn) ;
        }
      else
        {
          setUpResponseTimeout (cxn) ;
          unLockCxn (cxn) ;

          /* may have been told to close while in the middle of connecting */
          if (cxn->closing)     
            issueQUIT (cxn) ;
          else if (writesNeeded (cxn) || hostGimmeArticle (cxn->myHost,cxn))
            doSomeWrites (cxn) ;
        }
    }

  freeBufferArray (b) ;
}



  /*
   * REPONSE CODE PROCESSING.
   */


  /* Handle the response 205 to our QUIT command, which means the remote is
     going away. If this returns true then the connection is still usable
     (or will become so). */
static bool processResponse205 (Connection cxn, char *response)
{
  bool rval = false ;

  (void) response ;             /* keep lint happy */

  if (cxn->periodicClose)
    {
      syslog (LOG_NOTICE, CXN_PERIODIC_CLOSE,
              hostPeerName (cxn->myHost), cxn->ident) ;

      delEndPoint (cxn->myEp) ;
      cxn->myEp = NULL ;

      removeTimeout (cxn->artToutId) ;
      cxn->artToutId = 0 ;
      
      removeTimeout (cxn->respToutId) ;
      cxn->respToutId = 0 ;
      
      removeTimeout (cxn->closeId) ;
      cxn->closeId = 0 ;

      if (hostLogConnectionStatsP ())
        cxnLogStats (cxn,true) ;
      
      resetConnection (cxn) ;
      hostCxnDead (cxn->myHost,cxn) ;
      
      rval = cxnConnect (cxn) ;
      if (!rval)
        syslog (LOG_NOTICE,"Periodic connect failed") ;
    }
  else if (cxn->closing)
    syslog (LOG_NOTICE, CXN_CLOSED, hostPeerName (cxn->myHost), cxn->ident) ;
  else
    syslog (LOG_ERR, CXN_DROPPING, hostPeerName (cxn->myHost), cxn->ident) ;

  return rval ;
}



  /* Handle a response code of 238 which is the "no such article" reply to
     the CHECK command (i.e. remote wants it). If this returns true then
     the connection is still usable. */
static bool processResponse238 (Connection cxn, char *response)
{
  const char *msgid ;
  ArtHolder artHolder ;
  const char *peerName = hostPeerName (cxn->myHost) ;
  bool rval = false ;

  ASSERT (cxn->doesStreaming) ;

  msgid = getMsgId (response) ;

  if (msgid == NULL || strlen (msgid) == 0)
    syslog (LOG_ERR, NOMSGID, peerName, cxn->ident, 238, response) ;
  else if ((artHolder = artHolderByMsgId (msgid, cxn->checkRespHead)) == NULL)
    syslog (LOG_ERR, INVALID_MSGID, peerName, cxn->ident, msgid, 238) ;
  else
    {
        /* now remove the article from the check queue and move it onto the
           transmit queue. Another function wil take care of transmitting */
      remArtHolder (artHolder, &cxn->checkRespHead, &cxn->articleQTotal) ;
      appendArtHolder (artHolder, &cxn->takeHead, &cxn->articleQTotal) ;

      rval = true ;
    }

  return rval ;
}


  /* process the response "try again later" to the CHECK command If this
     returns true then the connection is still usable.  */
static bool processResponse431 (Connection cxn, char *response)
{
  const char *msgid ;
  ArtHolder artHolder ;
  const char *peerName = hostPeerName (cxn->myHost) ;
  bool rval = false ;

  ASSERT (cxn->doesStreaming) ;

  msgid = getMsgId (response) ;

  if (msgid == NULL || strlen (msgid) == 0)
    syslog (LOG_ERR, NOMSGID, peerName, cxn->ident, 431, response) ;
  else if ((artHolder = artHolderByMsgId (msgid, cxn->checkRespHead)) == NULL)
    syslog (LOG_ERR, INVALID_MSGID, peerName, cxn->ident, msgid, 431) ;
  else
    {
        /* now remove the article from the queue and give it back to the
           Host to be retried later */
      remArtHolder (artHolder, &cxn->checkRespHead, &cxn->articleQTotal) ;

      hostArticleDeferred (cxn->myHost, cxn, artHolder->article) ;

      delArtHolder (artHolder) ;

      rval = true ;
    }

  return rval ;
}



  /* process the "already have it" response to the CHECK command.  If this
     returns true then the connection is still usable.  */
static bool processResponse438 (Connection cxn, char *response)
{
  const char *msgid ;
  ArtHolder artHolder ;
  const char *peerName = hostPeerName (cxn->myHost) ;
  bool rval = false ;

  ASSERT (cxn->doesStreaming) ;

  cxn->checksRefused++ ;
  msgid = getMsgId (response) ;

  if (msgid == NULL || strlen (msgid) == 0)
    syslog (LOG_ERR, NOMSGID, peerName, cxn->ident, 438, response) ;
  else if ((artHolder = artHolderByMsgId (msgid, cxn->checkRespHead)) == NULL)
    syslog (LOG_ERR, INVALID_MSGID, peerName, cxn->ident, msgid, 438) ;
  else
    {
      remArtHolder (artHolder, &cxn->checkRespHead, &cxn->articleQTotal) ;

      hostArticleNotWanted (cxn->myHost, cxn, artHolder->article);

      delArtHolder (artHolder) ;

      rval = true ;
    }

  return rval ;
}


  /* process the "article transferred ok" response to the TAKETHIS. If this
     returns true then the connection is still usable.  */
static bool processResponse239 (Connection cxn, char *response)
{
  const char *msgid ;
  ArtHolder artHolder ;
  const char *peerName = hostPeerName (cxn->myHost) ;
  bool rval = false ;

  ASSERT (cxn->doesStreaming) ;

  msgid = getMsgId (response) ;

  if (msgid == NULL || strlen (msgid) == 0)
    syslog (LOG_ERR, NOMSGID, peerName, cxn->ident, 239, response) ;
  else if ((artHolder = artHolderByMsgId (msgid, cxn->takeRespHead)) == NULL)
    syslog (LOG_ERR, INVALID_MSGID, peerName, cxn->ident, msgid, 239) ;
  else
    {
      remArtHolder (artHolder, &cxn->takeRespHead, &cxn->articleQTotal) ;

      cxn->takesOkayed++ ;
      hostArticleAccepted (cxn->myHost, cxn, artHolder->article) ;

      delArtHolder (artHolder) ;

      rval = true ;
    }

  return rval ;
}


  /* process a "article rejected do not try again" response to the
     TAKETHIS. This is presumably when we don't give a CHECK command first
     and the remote really had the article.

     If this returns true then the connection is still usable.  */
static bool processResponse439 (Connection cxn, char *response)
{
  const char *msgid ;
  ArtHolder artHolder ;
  const char *peerName = hostPeerName (cxn->myHost) ;
  bool rval = false ;

  ASSERT (cxn->doesStreaming) ;

  msgid = getMsgId (response) ;

  if (msgid == NULL || strlen (msgid) == 0)
    syslog (LOG_ERR, NOMSGID, peerName, cxn->ident, 439, response) ;
  else if ((artHolder = artHolderByMsgId (msgid, cxn->takeRespHead)) == NULL)
    syslog (LOG_ERR, INVALID_MSGID, peerName, cxn->ident, msgid, 439) ;
  else
    {
      remArtHolder (artHolder, &cxn->takeRespHead, &cxn->articleQTotal) ;

      cxn->takesRejected++ ;
      
      hostArticleRejected (cxn->myHost, cxn, artHolder->article) ;

      delArtHolder (artHolder) ;

      rval = true ;
    }

  return rval ;
}


  /* process the "article transferred ok" response to the IHAVE-body.  If
     this returns true then the connection is still usable.  */
static bool processResponse235 (Connection cxn, char *response) 
{
  ArtHolder artHolder ;

  (void) response ;             /* keep lint happy */
  
  ASSERT (cxn->takeRespHead != NULL) ;
  ASSERT (cxn->takeHead == NULL) ;
  ASSERT (cxn->checkHead == NULL) ;
  ASSERT (cxn->checkRespHead == NULL) ;
  ASSERT (cxn->articleQTotal == 1) ;

    /* now remove the article from the queue and tell the Host to forget
       about it. */
  artHolder = cxn->takeRespHead ;
  cxn->takeRespHead = NULL ;
  cxn->articleQTotal-- ;

  cxn->takesOkayed++ ;
  hostArticleAccepted (cxn->myHost, cxn, artHolder->article) ;

  delArtHolder (artHolder) ;

  return true ;
}



  /* process the "send article to be transfered" reponse to the IHAVE.  If
     this returns true then the connection is still usable. */
static bool processResponse335 (Connection cxn, char *response)
{
  ArtHolder artHolder ;

  (void) response ;             /* keep lint happy */

  ASSERT (cxn->checkRespHead != NULL) ;
  ASSERT (cxn->checkHead == NULL) ;
  ASSERT (cxn->takeHead == NULL) ;
  ASSERT (cxn->takeRespHead == NULL) ;
  ASSERT (cxn->checkRespHead->next == NULL) ;
  ASSERT (cxn->articleQTotal == 1) ;

  artHolder = cxn->checkHead ;

    /* now move the article into the third queue */
  cxn->takeHead = cxn->checkRespHead ;
  cxn->checkRespHead = NULL ;

  ASSERT (writeIsPending (cxn->myEp) == false) ;

  issueIHAVEBody (cxn) ;

  return true ;
}



  /* process the "not accepting articles" response. This could be to any of
     the IHAVE/CHECK/TAKETHIS command. */
static bool processResponse400 (Connection cxn, char *response)
{
    /* We may get a response 400 multiple times when in streaming mode. */
  if (!cxn->closing)
    {
      syslog (LOG_NOTICE,CXN_BLOCKED,hostPeerName(cxn->myHost),cxn->ident,
              response) ;
      
      deferAllArticles (cxn) ;
      
      cxnDisconnect (cxn) ;
    }
  
  return true ;
}



  /* process the "not wanted" reponse to the IHAVE. */
static bool processResponse435 (Connection cxn, char *response)
{
  ArtHolder artHolder ;

  (void) response ;             /* keep lint happy */

  ASSERT (cxn->checkRespHead != NULL) ;
  ASSERT (cxn->checkRespHead->next == NULL) ;
  ASSERT (cxn->checkHead == NULL) ;
  ASSERT (cxn->takeHead == NULL) ;
  ASSERT (cxn->takeRespHead == NULL) ;
  ASSERT (cxn->articleQTotal == 1) ;

  artHolder = cxn->checkRespHead ;
  cxn->checkRespHead = NULL ;
  cxn->articleQTotal-- ;

  cxn->checksRefused++ ;
  hostArticleNotWanted (cxn->myHost, cxn, artHolder->article) ;

  delArtHolder (artHolder) ;

#if 1
  dprintf (1,"%s:%d On exiting 435 article queue total is %d (%d %d %d %d)\n",
           hostPeerName (cxn->myHost), cxn->ident,
           cxn->articleQTotal,
           (int) (cxn->checkHead != NULL),
           (int) (cxn->checkRespHead != NULL),
           (int) (cxn->takeHead != NULL),
           (int) (cxn->takeRespHead != NULL));
#endif

  return true ;
}

  /* process the "transfer failed" response to the IHAVE-body. */
static bool processResponse436 (Connection cxn, char *response)
{
  ArtHolder artHolder ;
  
  (void) response ;             /* keep lint happy */

  ASSERT (cxn->takeRespHead != NULL) ;
  ASSERT (cxn->takeRespHead->next == NULL) ;
  ASSERT (cxn->checkHead == NULL) ;
  ASSERT (cxn->takeHead == NULL) ;
  ASSERT (cxn->checkRespHead == NULL) ;
  ASSERT (cxn->articleQTotal == 1) ;

    /* now remove the article from the queue and tell the Host it's gone */
  artHolder = cxn->takeRespHead ;
  cxn->takeRespHead = NULL ;

  cxn->articleQTotal-- ;
  hostArticleDeferred (cxn->myHost, cxn, artHolder->article) ;

  delArtHolder (artHolder) ;

  return true ;
}


  /* Process the "article rejected do not try again" response to the
     IHAVE-body. */
static bool processResponse437 (Connection cxn, char *response) 
{
  ArtHolder artHolder ;

  (void) response ;             /* keep lint happy */

  ASSERT (cxn->takeRespHead != NULL) ;
  ASSERT (cxn->takeRespHead->next == NULL) ;
  ASSERT (cxn->checkHead == NULL) ;
  ASSERT (cxn->takeHead == NULL) ;
  ASSERT (cxn->checkRespHead == NULL) ;
  ASSERT (cxn->articleQTotal == 1) ;

  artHolder = cxn->takeRespHead ;
  cxn->takeRespHead = NULL ;
  cxn->articleQTotal-- ;

  cxn->takesRejected++ ;
  hostArticleRejected (cxn->myHost, cxn, artHolder->article) ;

  delArtHolder (artHolder) ;

  return true ;
}



static ArtHolder newArtHolder (Article article)
{
  ArtHolder a = ALLOC (struct art_holder_s, 1) ;
  ASSERT (a != NULL) ;

  a->article = article ;
  a->next = NULL ;

  return a ;
}

  /* Deletes the article holder */
static void delArtHolder (ArtHolder artH)
{
  if (artH != NULL)
    {

#ifdef DEADBEEF
      deadBeef (artH, sizeof (struct art_holder_s)) ;
#endif
      
      FREE (artH) ;
    }
}


  /*
   * remove the article holder from the queue. Adjust the count and if nxtPtr
   * points at the element then adjust that too.
   */
static bool remArtHolder (ArtHolder artH, ArtHolder *head, u_int *count)
{
  ArtHolder h, i ;

  ASSERT (head != NULL) ;
  ASSERT (count != NULL) ;

  h = *head ;
  i = NULL ;
  while (h != NULL && h != artH)
    {
      i = h ;
      h = h->next ;
    }

  if (h == NULL)
    return false ;

  if (i == NULL)
    *head = (*head)->next ;
  else
    i->next = artH->next ;

  (*count)-- ;

  return true ;
}

  /* append the ArticleHolder to the queue */
static void appendArtHolder (ArtHolder artH, ArtHolder *head, u_int *count)
{
  ArtHolder p ;

  ASSERT (head != NULL) ;
  ASSERT (count != NULL) ;

  for (p = *head ; p != NULL && p->next != NULL ; p = p->next)
      /* nada */ ;

  if (p == NULL)
    *head = artH ;
  else
    p->next = artH ;

  artH->next = NULL ;
  (*count)++ ;
}

  /*
   * find the article holder on the queue by comparing the message-id.
   */
static ArtHolder artHolderByMsgId (const char *msgid, ArtHolder head)
{
  ASSERT (head != NULL) ;

  while (head != NULL)
    {
      if (strcmp (msgid, artMsgId (head->article)) == 0)
        return head ;

      head = head->next ;
    }

  return NULL ;
}

  /* for one reason or another we need to disconnect gracefully. We send a
     QUIT command. Note that this causes the EndPoint to get removed, but the
     Connection hangs around to accept articles from its Host. */
static void issueQUIT (Connection cxn)
{
  Buffer quitBuffer, *writeArray ;
  const char *peerName = hostPeerName (cxn->myHost) ;

  ASSERT (cxn->takeHead == NULL) ;
  ASSERT (cxn->checkHead == NULL) ;
  
  if (writeIsPending (cxn->myEp))
    syslog (LOG_ERR, QUIT_WHILE_WRITING, peerName, cxn->ident) ;
  else
    {
      quitBuffer = newBuffer (7) ;
      strcpy (bufferBase (quitBuffer), "QUIT\r\n") ;
      bufferSetDataSize (quitBuffer, 6) ;
      
      writeArray = makeBufferArray (quitBuffer, NULL) ;
      
      dprintf (1,"%s:%d Sending a quit command\n",
               hostPeerName (cxn->myHost),cxn->ident) ;
      
      prepareWrite (cxn->myEp, writeArray, quitWritten, cxn) ;
    }
}

  /* called when the write of the QUIT command has completed. It's up to this
     function to cleanup the EndPoint. */
static void quitWritten (EndPoint e, IoStatus i, Buffer *b, void *d)
{
  Connection cxn = (Connection) d ;
  const char *peerName ;

  peerName = hostPeerName (cxn->myHost) ;

  ASSERT (cxn->myEp == e) ;
  
  if (i != IoDone)
    {
      errno = endPointErrno (e) ;
      syslog (LOG_ERR, QUIT_WRITE_FAILED, peerName, cxn->ident) ;
    }

  freeBufferArray (b) ;
}


  /* This is called when the timeout for the reponse from the remote goes
     off. We tear down the connection and notify our host. */
static void responseTimeoutCbk (TimeoutId id, void *data)
{
  Connection cxn = (Connection) data ;
  const char *peerName ;

  (void) id ;                   /* keep lint happy */
  
  peerName = hostPeerName (cxn->myHost) ;

  delEndPoint (cxn->myEp) ;
  cxn->myEp = NULL ;

  syslog (LOG_WARNING, RESPONSE_TIMEOUT, peerName, cxn->ident) ;
  dprintf (1,"%s:%d shutting down non-repsonsive connection\n",
           hostPeerName (cxn->myHost), cxn->ident) ;

  abortConnection (cxn) ;
}

static void setUpResponseTimeout (Connection cxn)
{
  ASSERT (cxn != NULL) ;

    /* set up the response timer. */
  removeTimeout (cxn->respToutId) ;
  cxn->respToutId = 0 ;
  
  if (cxn->respTout > 0)
    cxn->respToutId = prepareSleep (responseTimeoutCbk, cxn->respTout, cxn) ;
}


  /* simulate (in a very loose sense) a mutex */
static void lockCxn (Connection cxn)
{
  ASSERT (cxn != NULL) ;
  
  cxn->locked = true ;
}


static bool isLocked (Connection cxn)
{
  return cxn->locked ;
}

static void unLockCxn (Connection cxn)
{
  ASSERT (cxn != NULL) ;
  ASSERT (cxn->locked == true) ;

  cxn->locked = false ;
}




  /* Will shutdown the network link rather
     abruptly. Called by the Host as needed. Doesn't actually delete the
     connection as that would typically cause a segfault. Just flags it as
     gone and then the code just before dropping back into the EndPoint
     will delete it. Note that if a callback comes through one endpoint, but
     another is actually deleted, then that other's memory won't be
     reclaimed. This needs fixing. */ 
static void delConnection (Connection cxn)
{
  if (cxn == NULL)
    return ;

  if (isLocked (cxn))
    {
      syslog (LOG_ERR,DELETE_LOCKED_CXN,
              hostPeerName (cxn->myHost),cxn->ident) ;
      return ;
    }

  delEndPoint (cxn->myEp) ;
  cxn->myEp = NULL ;

  ASSERT (cxn->checkHead == NULL) ;
  ASSERT (cxn->checkRespHead == NULL) ;
  ASSERT (cxn->takeHead == NULL) ;
  ASSERT (cxn->takeRespHead == NULL) ;

  delBuffer (cxn->respBuffer) ;

  cxn->ident = 0 ;
  cxn->timeCon = 0 ;
  

#ifdef DEADBEEF
  deadBeef (cxn->ipName,strlen (cxn->ipName) + 1) ;
#endif
  
  FREE (cxn->ipName) ;

  removeTimeout (cxn->artToutId) ;
  removeTimeout (cxn->respToutId) ;
  removeTimeout (cxn->closeId) ;

  hostCxnGone (cxn->myHost, cxn) ; /* tell the Host we're outta here. */

#ifdef DEADBEEF
  deadBeef (cxn, sizeof (struct connection_s)) ;
#endif

  FREE (cxn) ;
}


static void incrFilter (Connection cxn)
{
  cxn->filterValue *= 0.9 ;
  cxn->filterValue += 1.0 ;
}


static void decrFilter (Connection cxn)
{
  cxn->filterValue *= 0.9 ;
  cxn->filterValue -= 1.0 ;
}


  /* return true if we have articles we need to issue commands for. */
static bool writesNeeded (Connection cxn)
{
  return (cxn->checkHead != NULL || cxn->takeHead != NULL ? true : false) ;
}


  /* do some simple tests to make sure it's OK. */
static void validateConnection (Connection cxn)
{
  u_int i ;
  ArtHolder p ;

  i = 0 ;

    /* count up the articles the Connection has and make sure that matches. */
  for (p = cxn->takeHead ; p != NULL ; p = p->next)
    i++ ;

  for (p = cxn->takeRespHead ; p != NULL ; p = p->next)
    i++ ;

  for (p = cxn->checkHead ; p != NULL ; p = p->next)
    i++ ;

  for (p = cxn->checkRespHead ; p != NULL ; p = p->next)
    i++ ;

  ASSERT (i == cxn->articleQTotal) ;

  if (cxn->timeCon != 0)        /* connection wasn't reset */
    ASSERT (cxn->doesStreaming || cxn->maxCheck == 1) ;
}


#if 0
  /* flag the connection for closing. Causes a quit to be generated. When the
     quit completes the Connection object will be deleted. The Host will
     retry any articles still being processed by the Connection, but we still
     try to send thm anyway. */
void cxnClose (Connection cxn)
{
  ASSERT (cxn != NULL) ;
  ASSERT (cxn->closing == false) ;

  cxn->closing = true ;

  if (cxn->articleQTotal == 0)
    issueQUIT (cxn) ;
}
#endif






#if 0
static void pvtDelConnection (Connection cxn)
{
  ASSERT (cxn->isDeleted == true) ;

#ifdef DEADBEEF
  deadBeef (cxn, sizeof (struct connection_s)) ;
#endif
  
  FREE (cxn) ;
}



  /* Returns true if the Connection object has connected to the remote and
     has gone through any necessary initialization (like asking the remote if
     it does MODE STREAM) */
bool cxnIsConnected (Connection cxn)
{
  ASSERT (cxn->isDeleted == false) ;
  return (cxn->timeCon > 0) ;
}


size_t cxnArticleCount (Connection cxn)
{
  ASSERT (cxn != NULL) ;
  
  return cxn->articleQTotal ;
}









#endif

#if 0
static void statsTimeoutCbk (TimeoutId id, void *data) ;
#endif


#if 0
  /* this function is called periodically to have the connection register
     it's statistics. */
static void statsTimeoutCbk (TimeoutId id, void *data)
{
  Connection cxn = (Connection) data ;

  cxnLogStats (cxn) ;

  cxn->statsId = prepareSleep (statsTimeoutCbk, STATS_PERIOD, cxn) ;
}
#endif

