# pr_dowork.pl -- execute work loop
# SCCS Status     : @(#)@ pr_dowork.pl	3.24
# Author          : Johan Vromans
# Created On      : Thu Jun  4 22:14:50 1992
# Last Modified By: Johan Vromans
# Last Modified On: Sat Jan  2 15:11:05 1993
# Update Count    : 221
# Status          : OK

# Format of work queue entries.
# Fields are separated by \0 characters.
# Field 1 is the command, fields 2 .. are arguments.
# 
# Command  Arguments (* denotes optional arguments)
#  ---  -------------------------------------------
#   M   1: Recipient for feedback messages.
#   U   1: hostname!pathname for UUCP transfers.
#       2: Notify user on target host.
#   L   1: Limit for transfers, in Kbytes.
#   S   1: Filename to send.
#       2* List of comma-separated parts to (re)send.
#   P   1* Packing code for subsequent directories
#          Values: "zoo", "zip", "tar".
#          If omitted: no more packing.
#   E   1: Encoding to use:
#	   A: none
#          B: btoa
#          D: Dumas uuencode
#          X: xxencode
#          U: uuencode
#   D   1: directory
#   G   1: FTP subcommand
#	   O: open connection
#	      2: system name
#	   T: transfer type (not implemented)
#	      2: A: ascii, I: image
#	   G: get file
#	      2: file name
#	   C: close
#	   D: directory
#	      2: directory
#	   U: login info
#	      2: user name
#	      3: password
#   A   1: Archie request
#	   P: prog
#	      2: search arch (regexp)

# These are the standards commands.
# See 'userdefs.pl' how to add your own commands here.
$exe_tbl{'L'} = 'exe_setlimit';
$exe_tbl{'E'} = 'exe_setencoding';
$exe_tbl{'M'} = 'exe_setdestination';
$exe_tbl{'P'} = 'exe_setpacking';
$exe_tbl{'U'} = 'exe_setuucpdest';
$exe_tbl{'S'} = 'exe_send';
$exe_tbl{'D'} = 'exe_dir';
$exe_tbl{'G'} = 'exe_ftp';
$exe_tbl{'A'} = 'exe_archie';

sub work_loop {

    local ($entries);
    local (@work);
    local ($type);
    local (@queueq) = ();
    local ($proc);
    local ($result);

    # Local variables that retain their values between calls.
    local (*encoding) = *work_loop'encoding;	#';
    local (*packing)  = *work_loop'packing;	#';
    local (*limit)    = *work_loop'limit;	#';
    local (*uupath)   = *work_loop'uupath;	#';
    local (*uunote)   = *work_loop'uunote;	#';
    local (*ftphost)  = *work_loop'ftphost;	#';

    $encoding = $default_encoding unless defined $encoding;
    $limit = $limits[1] .'K' unless defined $limit;
    $ftpuser = "anonymous" unless defined $ftpuser;

    if ( $opt_debug || $opt_trace ) {
	print STDOUT ("=> Work queue:\n");
	local ($tally) = 0;
	foreach $i ( @workq ) {
	    $tally++;
	    printf STDOUT ("  %3d: %s\n", $tally, join(" ", &zu ($i)));
	}
	print STDOUT ("\n");
    }

    $entries = 0;

    # Process the work queue.
    # This will probably result in some files to be transferred. 

    foreach $work ( @workq ) {

	($type, @work) = &zu ($work);
	last unless defined $type;

	if ( defined ( $proc = $exe_tbl{$type} ) &&
	    ( $result = &$proc ) != 0 ) {
	    last if $result < 0;
	}
	else {
	    # Should not happen.
	    print STDOUT ("*** Mail Server internal error: ",
			  "Request type \"$type\" in work queue ***\n");
	}
    }
    @workq = ();

    # Close any pending FTP connection.
    if ( $ftphost && !$interactive ) {
	print STDOUT ("FTP Command execution:\n    CLOSE $ftphost\n");
	&ftp'close;		#';
	$ftphost = '';
	print STDOUT ("\n");
    }

    # Okay, let's see if we have something to transfer.

    if ( @queueq > 0 ) {
	print STDOUT ("Request results:\n");
	select (STDOUT);
	$~ = "request_header";
	write;
	$~ = "request_list";
	$: = " /.]";
	$= = 99999;
	local (%enc);

	foreach $entry ( @queueq ) {
	    local ($name, $size, $coding, $limit, $remarks) = &zu ($entry);
	    if ( $method eq 'U' ) {
		if ( $coding =~ /Z$/ ) {
		    $coding = 'Z';
		    $enc{'Z'} = 1;
		}
		else {
		    $coding = '';
		}
	    }
	    else {
		$enc{$coding} = 1 if $coding;
	    }
	    write;
	}

	print STDOUT ("\n") if %enc;
	foreach $enc ( sort keys (%enc) ) {
	    print STDOUT ("Encoding $enc means: ");
	    if ( $enc =~ /^[AP]/ ) {
		print STDOUT ("not encoded (plain file).\n");
		next;
	    }
	    print STDOUT ("data will be compressed before transfer.\n")
		if $enc eq 'Z';
	    print STDOUT ("compressed first, then ") if $enc =~ /.Z$/;
	    print STDOUT ("encoded with ") if $enc ne 'Z';
	    print STDOUT ("uuencode.\n") if $enc =~ /^U/;
	    print STDOUT ("btoa.\n") if $enc =~ /^B/;
	    print STDOUT ("xxencode.\n") if $enc =~ /^X/;
	    print STDOUT ("Dumas' uue.\n") if $enc =~ /^D/;
	}

	if ( $entries > 0 ) {
	    print STDOUT ("\nThe requests with status \"Queued\"",
			  " will be sent as soon as the load of\n",
			  "the server system permits, ",
			  "usually within 24 hours.\n");
	}
	else {
	    print STDOUT ("\nNo requests remain to be sent.\n")
		unless $interactive;
	}
    }
    else {
	print STDOUT ("\nNo requests remain to be sent.\n")
	    unless $interactive;
    }
}

################ Execute routines ################

sub exe_setlimit {
    $limit = $work[0] . 'K';
    1;
}

sub exe_setencoding {
    $encoding = $work[0];
    1;
}

sub exe_setdestination {
    $destination = $work[0];
    1;
}

sub exe_setpacking {
    $packing = $work[0];
    1;
}

sub exe_setuucpdest {
    ($uupath, $uunote) = @work;
    1;
}

sub exe_send {

    local (@found);		# return from search
    local ($name, $size, $date, $lib, $subdir); # elements of @found
    local ($request, $plist) = @work;
    local ($remarks) = "";
    local ($coding) = $encoding;
    local ($docompress) = 0;

    if ( $packing ) {
	@found = ();
	foreach $lib ( @libdirs ) {
	    print STDOUT ("Trying dir $lib/$request...\n")
		if $opt_debug;
	    push (@found, $lib)
		if -d "$lib/$request" && -r _;
	}
	if ( @found == 1 ) {
	    local ($lib) = $found[0];
	    print STDOUT ("Sizing dir $lib/$request... ")
		if $opt_debug;
	    $size = `$du -s $lib/$request` + 0;
	    print STDOUT ($size, " blocks.\n")
		if $opt_debug;
	    if ($size > $packing_limit) {
		push (@queueq, 
		      &zp ($request . "/ (" . $packing . ")",
			   "", "", "", "Request too big"));
	    }
	    else {

		# Put the request in the batch queue.
		if ( $opt_noqueue ) {
		    $remarks = "Tested OK";
		    $entries++;
		}
		elsif ( $method eq "M" ) {
		    $remarks =
			&enqueue ("MP", $recipient, $destination, '',
				  $request, "$lib/$request",
				  $coding, $limit, $packing,
				  $plist);
		}
		elsif ( $method eq "U" ) {
		    $remarks =
			&enqueue ("UP", $recipient, $uupath, $uunote, 
				  $request, "$lib/$request",
				  $coding, $limit, $packing,
				  $plist);
		}
		push (@queueq,
		      &zp ($request . "/ (" . $packing . ")",
			   int(($size+1) / 2) . "K",
			   $coding, $limit, $remarks));
	    }
	}
	elsif ( $icall_packing ) {
	    # Internal call.
	    return 0;
	}
	elsif ( @found == 0 ) {
	    push (@queueq, 
		  &zp ($request . "/ (" . $packing . ")",
		       "", "", "", "Not found"));
	}
	else {
	    # Ambiguous.
	    print STDOUT ("Directory \"$request\" is not unique in the archives.\n",
			  "This request has been skipped.\n\n");
	    push (@queueq, 
		  &zp ($request . "/ (" . $packing . ")",
		       "", "", "", "Ambiguous"));
	}
	return 1;
    }

    # Locate them.
    @found = &search ($request, 0);

    # If we are not successfull, try automatic packing.
    if ( @found != 1 && $auto_packing && $request =~ /\.(zoo|zip|tar|tar\.Z)$/ ) {
	local ($request) = $`;
	local ($packing) = $1;
	$packing = "tar" if $packing eq 'tar.Z';
	if ( ($packing eq "tar" && (-x $tar || -x $pdtar)) ||
	     ($packing eq "zip" && -x $zip) ||
	     ($packing eq "zoo" && -x $zoo) ) {

	    # Build @work, retain $work[1]
	    local (@work) = @work;
	    $work[0] = $request;

	    # Recursive call...
	    local ($icall_packing) = 1;
	    return 1 if &exe_send;
	}
    }

    # If we are not successfull, try $request w/o .Z extension.
    if ( @found != 1 && $auto_compress && $request =~ /\.Z$/ ) {
	local ($req) = substr ($request, 0, length ($request)-2);

	# Maybe 'foo.Z' may result in 'foo.shar.Z'...
	local ($extpat, @exts) if $auto_compress > 1;

	local (@cfound) = &search ($req, 0);
	if ( @cfound == 1 ) {
	    # We found a unique hit -- override earlier results.
	    @found = @cfound;
	    $docompress = 1;
	}
	else {
	    # Failed. Add to the list of possibilities.
	    push (@found, @cfound);
	}
    }


    if ( @found > 1 ) {
	print STDOUT ("Request \"$request\" is ambiguous:\n");
	&dolist ("Search", $request, *found);
	print STDOUT ("\n");
	push (@queueq, 
	      &zp ($request, "", "", "", "Ambiguous"));
	return 1;
    }

    ($name, $size, $date, $lib, $subdir) = &zu ($found[0]);

    # Make sure that we have one single file.
    if ( @found == 0 || ! -f $lib.$subdir.$name ) {
	push (@queueq,
	      &zp ($request, "", "", "", "Not found"));
	return 1;
    }

    if ( $docompress ) {
	$coding .= 'Z';
    }
    else {
	# Send some files in plain (ascii) format.
	$coding = "A" if ($name !~ /$extpat$/ || $+ eq ".shar")
	    && -T $lib.$subdir.$name ;
    }

    $size = int(($size+1023)/1024) . "K" unless $size =~ /K$/;

    # Put the request in the batch queue.
    if ( $opt_noqueue ) {
	$remarks = "Tested OK";
	$entries++;
    }
    elsif ( $method eq "M" ) {
	$remarks =
	    &enqueue ("M", $recipient, $destination, '',
		      $subdir.$name, $lib.$subdir.$name,
		      $coding, $limit, $plist);
    }
    elsif ( $method eq "U" ) {
	$remarks =
	    &enqueue ("U", $recipient, $uupath, $uunote, 
		      $subdir.$name, $lib.$subdir.$name,
		      $coding, $limit, $plist);
    }

    push (@queueq,
	  &zp ($subdir.$name, $size, $coding, $limit, $remarks));
    1;
}

sub exe_dir {

    local ($thefile) = &fttemp;
    local ($dir) = shift (@work);

    open (DIR, '>' . $thefile) && close (DIR);
    foreach $lib ( @libdirs ) {
	local ($here) = $lib . '/' . $dir;
	next unless -d $here;
	&do_system ("$dircmd '$here' >> $thefile 2>&1");
    }

    $size = -s $thefile;

    if ( $fb_limit == 0 || $size < ($fb_limit * 1024)) {
	print STDOUT ("Result from Dir $dir:\n");
	open (DIR, $thefile);
	while ( <DIR> ) {
	    next if /^total/i;
	    print STDOUT ("    $_");
	}
	close (DIR);
	unlink ($thefile);
	print STDOUT ("\n");
    }
    else {
	# Put the request in the batch queue.
	&ftqueue ("dir $dir", $thefile, 
		  'A', $limit, '', "dir $dir", 1);
    }
    1;
}

sub exe_ftp {

    require 'pr_ftp.pl';

    $type = shift (@work);
    if ( $type eq 'O' ) {
	return 1 if $ftphost eq $work[0];
	$ftppass = $recipient if $ftppass eq '';
	$ftppass = $uunote if $ftppass eq '?';
	&ftp_connect (shift (@work), $ftpuser, $ftppass);
	print STDOUT ("\n");
	return 1;
    }
    if ( $type eq 'C' ) {
	return 1 unless $ftphost;
	print STDOUT ("FTP Command execution:\n",
		      "    CLOSE $work[0]\n");
	&ftp'close;				#';
	$ftphost = '';
	print STDOUT ("\n");
	return 1;
    }
    if ( $type eq 'U' ) {
	($ftpuser, $ftppass) = @work;
	return 1;
    }
    if ( $ftphost eq '' ) {
	print STDOUT ('*** Mail Server internal error: ',
		      'No FTP host specified for ',
		      $type, " command\n");
	return 1;
    }

    if ( $type eq 'D' ) {
	local ($thefile) = &fttemp;
	local ($dir) = shift (@work);
	&ftp_dir ($dir, $thefile);
	local ($size) = -s $thefile;
	if ( $fb_limit == 0 || $size < ($fb_limit * 1024)) {
	    open (DIR, $thefile);
	    while ( <DIR> ) {
		next if /^total/i;
		print STDOUT ("    $_");
	    }
	    close (DIR);
	    unlink ($thefile);
	}
	else {
	    # Put the request in the batch queue.
	    &ftqueue ("dir $ftphost:$dir", $thefile, 'A',
		      $limit, '', "dir $ftphost:$dir", 1);
	}
	print STDOUT ("\n");
	return 1;
    }
    if ( $type eq 'G' ) {
	local ($thefile) = &ftp_get ($work[0]);
	&ftqueue ($ftphost . ':' .$work[0],
		  $thefile, $encoding, 
		  $limit, $work[1],
		  $ftphost . ':' .$work[0]);
	print STDOUT ("\n");
	return 1;
    }

    print STDOUT ('*** Mail Server internal error: FTP command ',
		  $type, " not yet implemented\n");
    1;
}


sub exe_archie {

    $type = shift (@work);
    if ( $type eq 'P' ) {
	local ($arg) = @work;
	&do_unix ("$archie -r '$arg'", "$archie -r '$arg'");
	return 1;
    }

    print STDOUT ('*** Mail Server internal error: Archie command ',
		  $type, " not yet implemented\n");
    1;
}

################ Support Routines ################

# Perform a command, and show the result.
sub do_unix {
    local ($cmd, $desc) = @_;
    local ($thefile) = &fttemp;

    &do_system ("$cmd > $thefile 2>&1");
    local ($size) = -s $thefile;
    if ( $fb_limit == 0 || $size < ($fb_limit * 1024)) {
	print STDOUT ("Local command execution:\n    ", $cmd, "\n");
	local (*F);
	open (F, $thefile);
	while ( <F> ) {
	    print STDOUT ("    ", $_);
	}
	close (F);
	unlink ($thefile);
    }
    else {
	# Put the request in the batch queue.
	$desc = "$cmd (cmd)" unless $desc;
	&ftqueue ($cmd, $thefile, $default_encoding,
		  $limit, '', $desc, 1);
    }
    print STDOUT ("\n");

}

sub do_system {
    local ($cmd) = @_;
    print STDOUT ("+ $cmd\n") if $opt_trace;
    system ($cmd);
}

# Enter a file to be sent into the queue, if it exists.
sub ftqueue {
    local ($get, $got, $coding, $limit, $plist, $desc, $try_compress) = (@_);
#   global ($remarks);

    local ($temp) = (index ($got, $tmpdir) == $[);
    local ($size) = -s $got;

    $desc = $got unless $desc;

    if ( $size <= 0 ) {
	push (@queueq, 
	      &zp ($desc, "", "", "", "Not found"));
	unlink $got if $temp;
	return;
    }

    if ( $try_compress ) {
	$coding = $default_encoding . 'Z';
    }

    # Put the request in the batch queue.
    if ( $opt_noqueue ) {
	$remarks = "Tested " . ($temp ? "Ok" : "OK");
	$entries++;
    }
    elsif ( $method eq "M" ) {
	$remarks =
	    &enqueue ($temp ? "m" : "M", 
		      $recipient, $destination, '', $get, $got,
		      $coding, $limit, $plist);
    }
    elsif ( $method eq "U" ) {
	$remarks =
	    &enqueue ($temp ? "u" : "U",
		      $recipient, $uupath, $uunote, $get, $got,
		      $coding, $limit, $plist);
    }
    $size = int (($size + 1023) / 1024) . 'K' unless $size =~ /K$/;
    push (@queueq, &zp ($desc, $size, $coding, $limit, $remarks));
}

sub enqueue {

    # Add a request to the queue.

    local (@work) = @_;

    if ( grep (/\t/, @work) ) {
	return "Refused";
    }

    if (open (BATCH, ">>$queue")) {
	if ( &locking (*BATCH, 1) == 1 ) {
	    seek (BATCH, 0, 2);
	    print BATCH (join ("\t", @work), "\n");
	    close (BATCH);
	    $entries++;
	    if ( defined $plist && $plist =~ /\S/ ) {
		local ($remarks) = "Queued (part";
		$remarks .= "s" if $plist =~ /,/;
		$remarks .= " ${plist} only)";
		return $remarks;
	    }
	    else {
		"Queued";
	    }

	}
	else {
	    "Queue error";
	}
    }
    else {
	"Cannot queue";
    }
}

format request_header =

  Request                                        Size  Enc  Limit  Status
  --------------------------------------------  -----  ---  -----  ------
.
format request_list =
  ^<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<@>>>>>>  @||  @>>>>  @<<<<<<<<<<<<<<<<<<<<<<<<<<<
$name, $size, $coding, $limit, $remarks
~~  ^<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
$name
.

1;
