;;;;
;;;; pexpr.scm 1.11
;;;;
;;;; psd -- a portable Scheme debugger, version 1.0
;;;; Copyright (C) 1992 Pertti Kellomaki, pk@cs.tut.fi

;;;; This program is free software; you can redistribute it and/or modify
;;;; it under the terms of the GNU General Public License as published by
;;;; the Free Software Foundation; either version 1, or (at your option)
;;;; any later version.

;;;; This program is distributed in the hope that it will be useful,
;;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
;;;; GNU General Public License for more details.

;;;; You should have received a copy of the GNU General Public License
;;;; along with this program; if not, write to the Free Software
;;;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;;;; See file COPYING in the psd distribution.

;;;; 
;;;; Written by Pertti Kellomaki, pk@cs.tut.fi
;;;;
;;;; This file contains the implementation of pexps, which are sexps
;;;; with position information.
;;;;

;;;
;;; Expressions. Each expression carries its starting and ending
;;; position and type tag.
;;; 

(define psd-make-expr
  (let ((list list))
    (lambda (type start end contents) (list type start end contents))))
(define psd-expr-type
  (let ((car car))
    (lambda (expr) (car expr))))
(define psd-expr-start
  (let ((cadr cadr))
    (lambda (expr) (cadr expr))))
(define psd-expr-end
  (let ((caddr caddr))
    (lambda (expr) (caddr expr))))
(define psd-expr-contents
  (let ((cadddr cadddr))
    (lambda (expr) (cadddr expr))))


;;;
;;; psd-cons, psd-car, psd-cdr, etc. work the same way as cons, car, cdr etc.,
;;; just with pexps instead of sexps.
;;;

(define psd-make-null (lambda (start end) (psd-make-expr 'null start end '())))

(define psd-null?
  (let ((eq? eq?))
    (lambda (x) (eq? 'null (psd-expr-type x)))))

(define psd-cons
  (let ((cons cons))
    (lambda (x y start end)
      (psd-make-expr 'pair start end (cons x y)))))

(define psd-pair?
  (let ((eq? eq?))
    (lambda (x) (eq? 'pair (psd-expr-type x)))))

(define psd-car
  (let ((car car) (error error))
    (lambda (x)
      (if (psd-pair? x)
	  (car (psd-expr-contents x))
	  (error "psd-car: argument not a psd-pair " x)))))

(define psd-cdr
  (let ((cdr cdr) (error error))
    (lambda (x)
      (if (psd-pair? x)
	  (cdr (psd-expr-contents x))
	  (error "psd-cdr: argument not a psd-pair " x)))))

(define psd-cadr (lambda (x) (psd-car (psd-cdr x))))
(define psd-caddr (lambda (x) (psd-cadr (psd-cdr x))))
(define psd-cddr (lambda (x) (psd-cdr (psd-cdr x))))
(define psd-cdddr (lambda (x) (psd-cdr (psd-cddr x))))
(define psd-caar (lambda (x) (psd-car (psd-car x))))
(define psd-cadar (lambda (x) (psd-car (psd-cdr (psd-car x)))))


;;;
;;; Atoms.
;;; 

(define psd-make-symbol
  (lambda (start end sym)
    (psd-make-expr 'symbol start end sym)))
(define psd-symbol?
  (let ((eq? eq?))
    (lambda (x) (eq? 'symbol (psd-expr-type x)))))
(define psd-make-number
  (lambda (start end num)
    (psd-make-expr 'number start end num)))
(define psd-number?
  (let ((eq? eq?))
    (lambda (x) (eq? 'number (psd-expr-type x)))))
(define psd-make-boolean
  (lambda (start end val)
    (psd-make-expr 'boolean start end val)))
(define psd-boolean?
  (let ((eq? eq?))
    (lambda (x) (eq? 'boolean (psd-expr-type x)))))
(define psd-make-string
  (lambda (start end val)
    (psd-make-expr 'string start end val)))
(define psd-string?
  (let ((eq? eq?))
    (lambda (x) (eq? 'string (psd-expr-type x)))))
(define psd-make-char
  (lambda (start end val)
    (psd-make-expr 'char start end val)))
(define psd-char?
  (let ((eq? eq?))
    (lambda (x) (eq? 'char (psd-expr-type x)))))
(define psd-make-vector
  (lambda (start end contents)
    (psd-make-expr 'vector start end contents)))
(define psd-vector?
  (let ((eq? eq?))
    (lambda (x) (eq? 'vector (psd-expr-type x)))))
(define psd-vector-contents psd-expr-contents)

;;;
;;; psd-map is a map that accepts a pexp and returns a sexp.
;;; 


(define psd-map
  (let ((not not) (cons cons))
    (lambda (proc pexp)
      (cond ((psd-null? pexp)
	     '())
	    ((not (psd-pair? pexp))
	     (proc pexp))
	    (else
	     (cons (proc (psd-car pexp))
		   (psd-map proc (psd-cdr pexp))))))))

;;;
;;; pexp->sexp converts a pexp into a sexp
;;; 

(define pexp->sexp
  (let ((cons cons) (eof-object? eof-object?))
    (lambda (pexp)
      (cond ((eof-object? pexp) pexp)
	    ((psd-null? pexp) '())
	    ((psd-pair? pexp)
	     (cons (pexp->sexp (psd-car pexp))
		   (pexp->sexp (psd-cdr pexp))))
	    ((psd-vector? pexp)
	     (apply vector (map pexp->sexp (psd-vector-contents pexp))))
	    (else
	     (psd-expr-contents pexp))))))
