;;;;;;;;;;;;;;;;;;;;;;;;;;; -*- Mode: Emacs-Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;
;; emp-sail.el -- Ship sail and support for Gnu Emacs Empire Tool (GEET)
;;
;; Copyright Robert Forsman, et.al. (GNU General Public License)
;;
;; Author          : Robert Forsman <thoth@manatee.cis.ufl.edu>
;; Created On      : Mon Jan  7 11:43:38 1991
;; Last Modified By: Lynn Slater x2048
;; Last Modified On: Thu Feb 14 19:16:49 1991
;; Update Count    : 30
;; Status          : GEET General Release 2d Patch 0
;; 
;; HISTORY
;; 23-Jan-1991		Lynn Slater x2048	
;;    Last Modified: Sat Jan 12 11:01:21 1991 #15 (Lynn Slater x2048)
;;    added Robert's changes, cleaned up some line splits (probably mail artifacts)
;;    made uses of sail-info-errorF consistently not have \n
;;      also added save-excursions
;;    added a map so Robert can add ship commands with little fuss
;;    updated table of contents
;; 22-Jan-1991		Robert Forsman (no clones please)
;;    added sail-fancy-nav.  It is not guaranteed to work yet.  Using
;;    it in an orders list is "counter-indicated" (whew, just using that
;;    word costs the Pentagon $2.2M).
;; 7-Jan-1991		Lynn Slater x2048	
;;    Last Modified: Mon Jan  7 11:46:51 1991 #1 (Lynn Slater x2048)
;;    made sail-trust-database into a user variable
;;
;; PURPOSE
;;    This code allows the user to group ships into fleets and
;;    order them to perform complicated commands that may stretch
;;    over many updates without having to remember what he(/she?)
;;    started doing 4-12 hours ago.
;;
;; TABLE OF CONTENTS
;;   sail-do-orders -- This function scans through the fleets and performs all orders.
;;   describe-sail-commands -- Offers function documentation on all registered sail commands
;; 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; The contents of this file ARE copyrighted but permission to use, modify,
;; and distribute this code is granted as described in the file
;; emp-install.el which should have been distributed with this file. These
;; terms constitute what the Free Software Foundation calls a COPYLEFT.
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;  This code is Alpha, not because I think it's broken, but because it's
;;; delicate.  Therefore, do not attempt to tread here unless you are a
;;; lisp hacker of *great power* or a man of great patience.  Do not use
;;; it unless you have read through this code.  You don't have to understand
;;; it, just read the function descriptions and comments.
;;; 
;;;   If you don't understand something and think you should, tell me
;;; (thoth@cis.ufl.edu) and I'll add documentation.  Do not hesitate to
;;; do this as you are doing us ALL a favor.
;;; 
;;; BUGS!: the load commands are not smart about ships max capacity
;;; and will wait forever for a ship to load up.
;;;   It has been brought to my attention that some sail commands will
;;; puke if one of the ships in the fleet has been sunk.  Please do
;;; not allow any of your ships to be sunk until I have fixed this.
;;; If you can not avoid this then tell me how it failed and I'll
;;; armor the code against future failures.
;;;
;;;   There is almost NO error checking yet and bogus commands will
;;; result in errors that are undebuggable by clueless users.
;;;
;;; 
;;; TODO: make the load commands smarter, get some error checking, add
;;; a nav command that can radar and lookout along the way and record
;;; the output.
;;; 

(provide 'emp-sail)
(require 'emp-ship)
(require 'emp-anal)
(require 'emp-help)
;;;

(defvar empire-sail-trust-database nil 
  "*Is your database always up-to-date before calling any sail code? ")

(defvar empire-show-sail-commands nil
  "*If t, sail commands are shown as they happen. This can help debug user
sail conditions.")

(defvar empire-fleets nil
  "The list of fleets maintained by Sail
An individual fleet is a list whose first element
is a list of ships in the fleet, the rest of the
list is orders for the fleet.

  A fleet's orders is a list.  Each element of the list is a form that
will be evaluated in a special way.  The sail-do-fleet-orders will
insert into the form the ship-list for the fleet so you only have to
provide the other parameters.  The function will return a *LIST* of
orders to insert in its place.  If you only wish to return a single
order, it must be a *LIST* of length one.  If you do not wish to
return any orders return the empty list () or nil.

Example orders
(sail-create-fleet '(32)
		   '( (sail-simple-nav \"jh\")
		      (sail-wait-for-commodity-min 'food 48 '(halt))
		      (sail-simple-nav \"gh\")
		      )
		   )
This sails the fleet out one sector and waits for all boats to
get at least 48 food and then sails the fleet back.

If you want this in an infinite loop and for a number of fishers try
sail-copy-orders.
(sail-create-fleet '(32 51)
		   '( (sail-copy-orders '( (sail-simple-nav \"jh\")
					   (sail-wait-for-commodity-min
					    'food 48 '(halt))
					   (sail-simple-nav \"gh\")
					   ))
		      )
		   )

")
(put 'empire-fleets 'empire-system t)



;;;
;;; fleet primitives
;;;
(defun sail-fleet-ships (fleet)
  "returns the list of ships in the FLEET"
  (car fleet)
  )

(defun sail-fleet-orders (fleet)
  "returns the orders list for FLEET"
  (cdr fleet)
  )

;;;
;;; verbosity routines
;;;
;; Yes, these functions DO change the current buffer.

(defun sail-info-clear (string)
  (switch-to-buffer (get-buffer-create "*Sail*"))
  (delete-region (point-min) (point-max))
  )

;; info messages are not printed unless the user has
;; empire-show-sail-commands set

(defmacro sail-info-messageF (format-string &rest data)
  (` (sail-info-message (format (, format-string) (,@ data)) ))
  )

(defun sail-info-message (string)
  (if empire-show-sail-commands
      (save-excursion
	(switch-to-buffer (get-buffer-create "*Sail*"))
	(goto-char (point-max))
	(insert string "\n")
	)
    )
  )

;; error messages are printed no matter what.

(defmacro sail-info-errorF (format-string &rest data)
  (` (sail-info-error (format (, format-string) (,@ data)) ))
  )

(defun sail-info-error (string)
  (save-excursion
    (switch-to-buffer (get-buffer-create "*Sail*"))
    (goto-char (point-max))
    ;; make the error show in batch mode via message
    (insert (message string) "\n"))
  )


;;;
;;; support routines
;;;

(defun ship-list-to-fleet-string (list)
  "turns a list of ship numbers into a string of slash-separated ship
 numbers suitable for all ship and plane commands"
  (mapconcat 'int-to-string list "/")
  )

(defun empire-ship-out-of-mobility (list)
  "returns t if any ships in LIST are out of mobility, nil otherwise"
  (let ( (ships-left list) )
    (while (and ships-left
		(< 0 (ship-recall (car ships-left) 'mob)))
      (setq ships-left (cdr ships-left))
      )
    (if ships-left t nil)
    ))

(defun empire-ships-same-sector (ship-list x y) ; presume not sunk
  (let ( (ships-left ship-list)
	 )
    (while (and ships-left
		(equal x (ship-recall (car ships-left) 'x))
		(equal y (ship-recall (car ships-left) 'y))
		)
      (setq ships-left (cdr ships-left))
      )
    (not ships-left)
    ))

(defun empire-ships-still-afloat (ship-list)
  (let ((ships))
    (while ship-list
      (if (ship-recall (car ship-list) 'x) ; verify not nil x,y
	  (setq ships (cons (car ship-list) ships))
	(sail-error-messageF "Warning: ship %s seems to be sunk\n" (car ship-list))
	)
      (setq ship-list (cdr ship-list))
      )
    (nreverse ships)))

(defun sail-fleet-ships-afloat (fleet)
  "returns the list of ships in the FLEET that are still floating"
  (empire-ships-still-afloat (car fleet))
  )


(defun empire-path-remaining (x1 y1 path x2 y2)
  "Returns the part of path NOT consumed while going from x1,y1 to
 x2,y2.  Be careful with self-intersecting paths since this routine
 assumes that you stopped the first time you crossed the destination"
  (let ( ( rval path )
	 ( sect (cons x1 y1) )
	 ( end-sect (cons x2 y2) )
	 )
    (while ( and (< 0 (length rval))
		 (not (equal sect end-sect)))
      (setq sect (sector-in-direction (car sect) (cdr sect)
				      (aref rval 0)))
      (setq rval (substring rval 1))
      )
    
    ;;return value
    ( cond
      ( (>= 0 (length rval))			"" )
      ( (string= "h" (substring rval 0 1))	(substring rval 1) )
      ( t					rval )
      )
    )
  )

;;;
;;; internal versions of complicated commands
;;;

(defun empire-simple-nav-ships (ship-list path)
  "This function navigates all the ships in SHIP-LIST along PATH
 and returns what portion of PATH was not navigated"
  (let* ( (fleet (ship-list-to-fleet-string ship-list))
	  (leader (car ship-list) )
	  (start-x (ship-recall leader 'x))
	  (start-y (ship-recall leader 'y))
	  old-point
	  end-x	end-y
	  )
    (if (empire-ship-out-of-mobility ship-list)
	path
      ( switch-to-empire-buffer-if-necessary
	(setq old-point (point))
	(send-empire-command-wait (format "navigate %s %s" fleet path))
	(goto-char old-point)
	(next-line 1) (beginning-of-line)
	(cond
	 ( (looking-at "^.* #\\([0-9]+\\) stopped at \\([-0-9]+\\),\\([-0-9]+\\)")
	   (setq end-x (read-buffer-regexp-match 'read-buffer-num 2)
		 end-y (read-buffer-regexp-match 'read-buffer-num 3))
	   (message "  Navigated ships %s to %s,%s" fleet end-x end-y)
	   (empire-path-remaining start-x start-y path end-x end-y)
	   )
	 ( t
	   (sail-info-errorF "  unhandled outcome error while navving ships %s" fleet)
	   "h" )
	 )
	)
      )
    ))

(defvar empire-navigation-regexp
  "^<\\([0-9]+\\.[0-9]+\\):\\([0-9]+\\.[0-9]+\\): *\\(-?[0-9]+\\),\\(-?[0-9]+\\)> *"
  "The regexp matching the prompt that appears when you are
navigating ships.  the following match-data positions are defined:
	1 - lead ship mobility
	2 - least ship mobility
	3 - current x
	4 - current y"
)

(defvar empire-navigation-radar-regexp
    "^Radar from (ship # or sector(s)) :"
    "The regexp matching the prompt that appears when you are
performing a radar from withing a navigation.  No match-data
positions are defined."
    )

(defvar empire-navigation-lookout-regexp
    "^ship(s)?"
    "The regexp matching the prompt that appears when you are
performing a lookout from withing a navigation.  No match-data
positions are defined."
    )

(defun define-handy-navigation-prompt-regexps ()
  (add-empire-prompt-regexp empire-navigation-regexp 'navigate-path-prompt)
  (add-empire-prompt-regexp empire-navigation-radar-regexp 'navigate-radar-prompt)
  (add-empire-prompt-regexp empire-navigation-lookout-regexp 'navigate-lookout-prompt)
  )

(defun sail-internal-lookout ( ship-list )
  "This internal function sends the command to radar from within a
nav command and parses the output.  It assumes you have defined the
lookout regexp ( navigate-lookout-prompt ). It makes many assumptions,
most of which I'm not aware of."
  (let ( start
	 end
	 )
    (insert "l")
    (empire-send-input)
    
    (if (not (eq (wait-for-empire-prompt)
		 'navigate-lookout-prompt)
	     )
	(throw 'navigation 'lookout-error)
      (insert (ship-list-to-fleet-string ship-list))
      (setq start (point))
      (empire-send-input)
      (if (not (eq (wait-for-empire-prompt)
		   'navigate-path-prompt))
	  (throw 'navigation 'lookout-didnot-finish) )
      (setq end (point))
      (narrow-to-region start end)
      (goto-char start)
      (empire-read-looks)
      ;;(debug)
      (widen)
      (goto-char end)
      )
    )
  )

(defun sail-internal-move-along ( path )
  (let ( navigation-result
	 start)
    (insert path)
    (setq start (point))
    (empire-send-input)
    (setq naviagation-result (wait-for-empire-prompt))
    ;;(debug)
    (goto-char start)
    (next-line 1) (beginning-of-line)
    (cond
     ( (looking-at "^.* #\\([0-9]+\\) stopped at \\([-0-9]+\\),\\([-0-9]+\\)")
       (throw 'navigation 'done) )
     ( (looking-at "^.* #\\([0-9]+\\) can't go to \\([-0-9]+\\),\\([-0-9]+\\)")
       (throw 'navigation 'blocked-path) )
     )
    (goto-char (point-max))
    ) )

(defun empire-fancy-nav-ships (ship-list path)
  "This function navigates all the ships in SHIP-LIST along PATH
 and returns what portion of PATH was not navigated.  This also
generates lookout commands at every point along the path.

  This does NOT work
  BUGS: sail-fancy-nav will not set adjacent sea sectors properly
during nav (I have this fixed).  sail-fancy-nav will not emit the
proper incomplete command when it can't finish a path.

  The second (still extant) makes it useless as a command in fleet
orders.
				<thoth@manatee.cis.ufl.edu>
"
  (let* ( (fleet (ship-list-to-fleet-string ship-list))
	  (leader (car ship-list) )
	  (start-x (ship-recall leader 'x))
	  (start-y (ship-recall leader 'y))
	  navigation-result	(path-left path)
	  command
	  (empire-symbol-prompt-alist empire-symbol-prompt-alist)
					; for navigation prompts
	  )
    (if (empire-ship-out-of-mobility ship-list)
	path

      (debug)
      (if (eq ?h (aref path-left (1- (length path-left))))
	  nil
	(setq path-left (concat path-left "h"))
	)
      (define-handy-navigation-prompt-regexps)
      ( switch-to-empire-buffer-if-necessary
	(move-to-empire-shell-prompt)
	(delete-region (point) (point-max));; insure that its clear
	(setq start (point))
	(setq command (format "navigate %s" fleet))
	(insert command)
	(empire-send-input)

	(setq problem
	      (catch 'navigation
		(if (not (eq (wait-for-empire-prompt)
			     'navigate-path-prompt))
		    (throw 'navigation 'failed)
		  )
		(while t ;;(not (eq ?h (aref path-left 0)))
		  (let ((now-x (string-to-int (substring empire-last-prompt
					  (match-beginning 3) (match-end 3))))
			(now-y (string-to-int (substring empire-last-prompt
					  (match-beginning 4) (match-end 4))))
			(des (position-of 'des))
			sect )
		    (setq sect (cons now-x now-y) )
		    ;; presume that we are a sea and all around us is a sea
		    (mapcar '(lambda (pt)
			       (if (not (recall (car pt) (cdr pt) des))
				   (record-des (car pt) (cdr pt) "." t t)))
			    (cons sect (sects-near sect)))
		    )
		  (sail-internal-lookout ship-list)
		  ;;(debug)
		  (sail-internal-move-along (substring path-left 0 1))
		  (setq path-left (substring path-left 1))
		  )
		;; NOTREACHED
		)
	      )
	
	( cond
	  ( (eq problem 'done) ;; no problem
	    (let ( ( end-x (read-buffer-regexp-match 'read-buffer-num 2) )
		   ( end-y (read-buffer-regexp-match 'read-buffer-num 3) )
		   )
	      (send-empire-command-and-parse-reply (format "ship %s" fleet)
						   'empire-read-ships)
	      (sail-info-messageF "  Navigated ships %s to %s,%s"
				  fleet end-x end-y)
	      ;;(empire-path-remaining start-x start-y path end-x end-y)
	      (debug)
	      (substring path-left 1)
	      )
	    )
	  ( (eq problem 'blocked-path)
	    (sail-info-errorF " can't move into sector %s,%s "
			      (read-buffer-regexp-match 'buffer-substring 2)
			      (read-buffer-regexp-match 'buffer-substring 3)
			      )
	    ;;(debug)
	    (goto-char (point-max))
	    (insert "h")
	    (empire-send-input)
	    (wait-for-empire-prompt)
	    "h" )
	  ( (eq problem 'lookout-error)
	    (sail-info-error "Sail didn't get ships prompt back after lookout request")
	    "h" )
	  ( (eq problem 'lookout-didnot-finish)
	    (sail-info-error "Sail didn't get path prompt back after lookout finished")
	    "h" )
	  ( t
	    (sail-info-errorF "unhandled error while navving %s" problem)
	    "h" )
	  )
	)
      )
    ))

;; this is total heiroglyphics.  If you can understand what
;; I'm doing, you're a better hacker than I am; please suggest
;; improvements.
(defun sail-load-ships-internal (comm ship-load-list command-name)
  "take a COMModity, a SHIP-LOAD-LIST (list of '(ship quant) pairs) and
 a COMMAND-NAME to use in the empire buffer and group the load commands
 together to reduce btu usage.  Not as smart as it could be, but I'm
 working on it.  I need to find out how to retrieve ship capabilities
 for TRUE smart loading."
  (let* ( harbor-level
	  x	y
	  curr-ships
	  curr-quant
	  curr-load-cdr
	  (ran-out nil)
	  )
    
    (while (and ship-load-list (not ran-out))
      (setq curr-load-cdr (cdar ship-load-list)
	    curr-ships (delq nil
			     (mapcar (` (lambda (load)
					  (if (equal (cdr load)
						     (quote (, curr-load-cdr)))
					      (car load)
					    nil) ))
				     ship-load-list))
	    curr-quant (car curr-load-cdr)
	    )
      (setq x (ship-recall (caar ship-load-list) 'x)
	    y (ship-recall (caar ship-load-list) 'y)
	    harbor-level (recall x y (dynamic-position-of comm))
	    )
      
      (if (or (string-equal command-name "unload")
	      (>= harbor-level (* curr-quant (length curr-ships))))
	  (setq ship-load-list
		(delq nil
		      (mapcar (` (lambda (load)
				   (if (equal (cdr load)
					      (quote (, curr-load-cdr ) ) )
				       nil
				     load) ))
			      ship-load-list)))
	(setq curr-quant (/ harbor-level (length curr-ships)))
	(setq ship-load-list
	      (mapcar (` (lambda (load)
			   (if (equal (cdr load)
				      (quote (, curr-load-cdr) ) )
			       (cons (car load)
				     (cons (, (- (car curr-load-cdr) curr-quant))
					   (cddr load)))
			     load) )
			 ) ship-load-list) )
	(setq ran-out t)
	)
      (if (< 0 curr-quant)
	  (progn
	    ( send-empire-command-wait
	      (format "\n%s %s %d %s\n" command-name
		      (ship-list-to-fleet-string curr-ships)
		      curr-quant comm))
	    ( send-empire-command-and-parse-reply
	      (format "dump %d,%d" x y))
	    ( send-empire-command-and-parse-reply
	      (format "ships %s" (ship-list-to-fleet-string curr-ships))
	      'empire-read-ships)
	    ( send-empire-command-and-parse-reply
	      (format "cargo %s" (ship-list-to-fleet-string curr-ships))
	      'empire-read-cargos)
	    )
	)
      )
    ship-load-list
    ))

(defun sail-calc-load-list (ship-list min-max comm quant-list)
  "take a SHIP-LIST, whether we want it to be a 'min or a 'max, a COMM,
 and a QUANT-LIST and calculate the exact deltas needed to bring each
 ship up to its exact quant."
  (let* ( ship-load-list
	  (ships-left ship-list)
	  (quants-left quant-list)
	  )
    (while ships-left
      (setq ship-load-list (cons (list (car ships-left)
				       (- (car quants-left)
					  (dynamic-ship-recall (car ships-left)
							       comm))
				       ) ship-load-list))
      (setq ships-left (cdr ships-left)
	    quants-left (cdr quants-left))
      )
    
    (setq ship-load-list (mapcar
			  ( cond
			    ( (eq min-max 'max)
			      '(lambda (n)
				 (list (car n) (- 0 (cadr n)) (cddr n) )) )
			    ( (eq min-max 'min) 'identity)
			    ( t (throw not-quote-min-or-max) )
			    ) ship-load-list) )
    (setq ship-load-list (delq nil (mapcar '(lambda (n)
					      (if (>= 0 (cadr n)) nil n))
					   ship-load-list)))
    ))

;;;
;;; exported routines
;;;

(defun sail-create-fleet (ship-list &optional orders)
  
  "This function creates a Sail fleet (not an empire fleet) from the
ships in SHIP-LIST.  All ships in a fleet must be in the same sector.
These ships are removed from any other fleets since ships can not
follow two different sets of orders at once.  If ORDERS are not
supplied, the fleet is created without them and you can use
sail-set-ship-orders on any ship in the fleet to give orders later.
sail-create-fleet returns the updated fleet list"
  
  (sail-defleet-ships ship-list)
  (setq empire-fleets (cons (cons ship-list orders) empire-fleets))
  )

(defun sail-defleet-ships (ship-list)
  "This function removes the ships in SHIP-LIST from any existing fleets
 and then deletes any orphan fleets"
  (do ( (fleets-left empire-fleets (cdr fleets-left))
	)
      ( (null fleets-left) nil )
    (do ( (ships-left ship-list (cdr ships-left))
	  )
	( (null ships-left) nil )
      (setcar (car fleets-left) (delq (car ships-left)
				      (sail-fleet-ships (car fleets-left)) ) )
      )
    )
  (sail-delete-orphan-fleets)
  )

(defun sail-delete-orphan-fleets ()
  "This function deletes any fleets that no longer have ships in them."
  (setq empire-fleets (delq nil (mapcar
		'(lambda (fleet)
		   (sail-set-fleet-ships fleet (delq nil (mapcar
			'(lambda (ship)
			   (if (eq empire-nation-number
				   (ship-recall ship 'owner) )
			       ship
			     nil))
			(sail-fleet-ships fleet))))
		   (if (null (sail-fleet-ships fleet))
		       nil
		     fleet
		     )
		   )
		empire-fleets)))
  )

(defun sail-fleet-of-ship (ship)
  "This returns the fleet that the ship belongs to"
  (let ( (fleets-left empire-fleets) )
    (while (and fleets-left
		(not (memq ship (sail-fleet-ships (car fleets-left)))))
      (setq fleets-left (cdr fleets-left))
      )
    ;; fleets-left can be nil if the ship isn't in any fleets
    (car-safe fleets-left)
    )
  )

(defun sail-set-fleet-ships (fleet ships)
  "sets the ships in FLEET to SHIPS"
  (setcar fleet ships)
  )

(defun sail-set-fleet-orders (fleet orders)
  "sets the orders of FLEET to ORDERS"
  (setcdr fleet orders)
  )

(defun sail-set-ship-orders (ship orders)
  "sets the orders of SHIP's fleet to ORDERS.  Notice it sets the orders
 for an entire fleet, not just a single ship."
  (sail-set-fleet-orders (sail-fleet-of-ship ship) orders)
  )

(defun sail-do-fleet-orders (fleet)
  "This actually performs the orders for a fleet.  It returns the new orders."
  (let* ( (orders (sail-fleet-orders fleet))
	  order
	  form
	  ( ships (sail-fleet-ships-afloat fleet))
	  )

    (sail-info-messageF "  Before ships %s orders\n\t%s" ships orders)
    (while (and orders
		(not (eq 'halt (setq order (car orders))))
		)
      (setq form (cons (car order)
		       (cons (list 'quote ships)
			     (cdr order))))

      (if empire-interaction-verbosity
	  (message " Ships %s order %s" ships order))
      (sail-info-messageF "    Doing command %s" form)
      (condition-case errorcode
	  (setq orders (append (eval form) (cdr orders)) )
	(error
	 (sail-info-errorF "  Ships %s sail order %s aborted with error %s"
			ships order errorcode)
	 (setq orders (cons 'halt orders))
	 ))
      (sail-info-messageF "    Did, new orders:\n\t%s" orders)
      )
    (sail-set-fleet-orders fleet (if (eq 'halt (car orders))
				     (cdr orders)
				   orders) )
    (sail-info-messageF "  After ships %s next time orders will be \n\t%s"
			ships (sail-fleet-orders fleet)))
  )


(defun sail-do-orders ()
  "This function scans through the fleets and performs all orders.
   See the variables empire-sail-trust-database and empire-fleets.

   Currently, there is no way to make fleets or orders except by
calling sail-create-fleet directly from lisp.  I can't think of any
other clever way to do it right now since giving orders is essentially
writing very simple lisp code"
  
  (interactive)

  (sail-delete-orphan-fleets)

  (if ( not empire-sail-trust-database )
      (progn
	(set-buffer (get-buffer-create empire-temp-buffer))
	(delete-region (point-min) (point-max))
	(execute-empire-command-with-buffer-output "ship *"
						   empire-temp-buffer)
	(sit-for 0)
	(execute-empire-command-with-buffer-output "cargo *"
						   empire-temp-buffer)
	(empire-read-ships t)
	(empire-read-cargos)

	))
  
  (let ( (fleets-left empire-fleets)
	 (empire-sail-trust-database t)	; we now trust the database (I hope)
	 )
    (while fleets-left
      (sail-do-fleet-orders (car fleets-left))
      (setq fleets-left (cdr fleets-left))
      )
    (message "")
    ))
(register-check-hook 'sail-do-orders "Ship Sailing" nil)



;;;
;;; fleet orders routines
;;;
(defun register-sail-command (symbol)
  "Records that the command is intended for use as orders for sail.
Later, editing modes may use this information.  Hopefully we can
insulate naive users from the tricky task of writing lisp.
"
  (put symbol 'empire-sail-command t)
  )

(defun empire-sail-commands ()
  "Returns list of all functions registered as sail orders."
  (let (vars)
    (mapatoms (function (lambda (sym)
			  (if (get sym 'empire-sail-command)
			      (setq vars (cons sym vars))))))
    (nreverse vars)
    ))

(defun describe-sail-commands ()
  "Offers function documentation on all registered sail commands
empire offers."
  (interactive)
  (with-output-to-temp-buffer "*Empire Sail Commands*"
    (mapcar '(lambda (function)
	       (prin1 function)
	       (princ ":\n ")
	       (if (documentation function)
		   (princ (documentation function))
		 (princ "not documented"))
	       (terpri)
	       (terpri)
	       )
	    (empire-sail-commands))))
(put 'describe-sail-commands 'empire t) ; make this command be described by
					; describe-empire-interative-functions


(defun sail-wait-for-condition-all (ship-list func &optional orders)
  "This function evaluates FUNC on every shipnumber in the fleet.
 If all evaluations return true then sail-wait-for-condition-all
 returns nil, allowing the fleet to continue on its journey.  If
 any one returns false, sail-wait-for-condition-all returns a list
 of all the optional ORDERS passed to it and a duplicate
 sail-wait-for-condition-all.
   Since sail-wait-for-condition-all could be waiting on a condition
 that is impossible in this turn it makes sure that there is a 'halt
 in the orders list when it clones.

 The functions are called with all ship data bound to local variables."
  
  (let ( (ships-left ship-list) )
    (while (and ships-left
		(empire-ship-pos-var-let-set-from-ship
		  (car ships-left)
		  (condition-case errorcode
                      (eval func)
		    (error
		     (sail-info-errorF "  Fleet %s sail command %s condition %s aborted with error %s"
				    (car ships-left)
				    'sail-wait-for-condition-all
				    func
				    errorcode)))
		  ))
      (setq ships-left (cdr ships-left))
      )
    (if ships-left
	(let ( (old-orders orders) )
	  (if (not (memq 'halt orders))
	      (setq orders (append orders '(halt))))
	  (append old-orders (list (` (sail-wait-for-condition-all
				       (quote (, func))
				       (quote (, orders)) ))))
	  )
      nil
      )
    ))
(register-sail-command 'sail-wait-for-condition-all)

(defun sail-copy-orders (ship-list orders)
  "This function returns a copy of its ORDERS and a duplicate
 sail-copy-orders.  This is useful for designing an infinite
 loop for shipping."
  (append orders (list (list 'sail-copy-orders (` (quote (, orders))))))
  )
(register-sail-command 'sail-copy-orders)

(defun sail-simple-nav (ship-list path)
  "This function navigates the fleet along a path until someone runs out
 of mobility.  If the path is finished, sail-simple-nav returns nil and
 the orders continue.  If there is still a ways to go, sail-simple-nav
 returns another sail-simple-nav command with the rest of the path in it."
  (let ( newpath
	 ( x (ship-recall (car ships) 'x) )
	 ( y (ship-recall (car ships) 'y) )
	 )
    (if (not (empire-ships-same-sector ship-list x y))
	(progn
	  (sail-set-fleet-orders fleet (cons 'halt orders))
	  (message (format "fleet %s is scattered or partially sunk" ship-list) )
	  )
      (setq newpath (empire-simple-nav-ships ship-list path
					     empire-sail-trust-database) )
      (if (< 0 (length newpath) )
	  (list 'halt (list 'sail-simple-nav newpath))
	'()
	)
      )
    ))
(register-sail-command 'sail-simple-nav)


(defun sail-fancy-nav (ship-list path curr-pos destination)
  "This function navigates the fleet along a path until someone runs out
 of mobility.  If the path is finished, sail-fancy-nav returns nil and
 the orders continue.  If there is still a ways to go, sail-fancy-nav
 returns another sail-fancy-nav command with the rest of the path in it."
  (let ( newpath
	 ( x (car curr-pos))
	 ( y (cdr curr-pos))
	 dest
	 )
    ( catch 'fit	; in case I have to throw a fit because of user error
      (if (not (empire-ships-same-sector ship-list x y))
	  (progn
	    (message (format "fleet %s is scattered or partially sunk
should be in sector %d,%d" ship-list x y) )
	    (throw 'fit (list 'halt (list 'sail-fancy-nav path curr-pos
					  destination)) )
	    ))
      (setq dest (destination-of path curr-pos))
      (if (not (equal destination dest))
	  (progn
	    (message (format "path %s from %d,%d does not lead to %d,%d"
			     path x y (car destination) (cdr destination)))
	    (throw 'fit (list 'halt (list 'sail-fancy-nav path curr-pos
					  destination)))
	    ))
      (setq newpath (empire-fancy-nav-ships ship-list path) )
      (if (< 0 (length newpath) )
	  (list 'halt (list 'sail-fancy-nav newpath curr-pos destination))
	'()
	))
    ))

(register-sail-command 'sail-fancy-nav)

(defun sail-wait-for-commodity-min (ship-list comm level &optional orders)
  "This function calls sail-wait-for-condition-all with a lambda form
 to test if the COMModity is >= to the LEVEL.
   If the ship has enough, then orders continue, otherwise the optional
 ORDERS are executed until the ship gets enough of the COMModity.
 Since sail-wait-for-commodity-min calls sail-wait-for-condition,
 copies of orders will contain sail-wait-for-condition with a lambda
 form instead of sail-wait-for-commodity-min."
  ( sail-wait-for-condition-all
    ship-list (list '<= level comm)
    orders
    )
  )
(register-sail-command 'sail-wait-for-commodity-min)

(defun sail-quota-commodity-min (ship-list comm quant-list)
  "This function emits commands necessary to bring each ship's COMModity
 stock *up* to the corresponding entry in QUANT-LIST.  If the user passes
 a number in place of QUANT-LIST it is duplicated APL style.

  Warning!  This function has not been tested under all the conditions
 forseen by the author and therefore may fail or misfunction under
 strenuous use.  Please break it and send back the pieces."
  (if (nlistp quant-list)
      (setq quant-list (make-list (length ship-list) quant-list))
    )
  (if (not(string-match (recall (ship-recall (car ship-list) 'x)
				(ship-recall (car ship-list) 'y)
				(position-of 'des))
			"h"))
      ;; bail if no harbor
      nil
    
    ;; else get on with the show
    (let* ( (ship-load-list (sail-calc-load-list ship-list 'min
						 comm quant-list))
	    (ships-left ship-list)
	    (quants-left quant-list)
	    )
      (setq ship-load-list (sail-load-ships-internal comm ship-load-list
						     "load"))
      (if (sail-calc-load-list ship-list 'min comm quant-list)
	  (list 'halt (list 'sail-quota-commodity-min (list 'quote
comm) (list 'quote quant-list)))
	nil)
      )
    )
  )
(register-sail-command 'sail-quota-commodity-min)

(defun sail-quota-commodity-max (ship-list comm quant-list)
  "This function emits commands necessary to bring each ship's COMModity
 stock *down* to the corresponding entry in QUANT-LIST.  If the user
 passes a number in place of QUANT-LIST it is duplicated APL style.

  Warning!  This function has not been tested under all the conditions
 forseen by the author and therefore may fail or misfunction under
 strenuous use.  Please break it and send back the pieces."
  (if (nlistp quant-list)
      (setq quant-list (make-list (length ship-list) quant-list))
    )
  (if (not(string-match (recall (ship-recall (car ship-list) 'x)
				(ship-recall (car ship-list) 'y)
				(position-of 'des))
			"h"))
      ;; bail if no harbor
      nil
    
    ;; else get on with the show
    (let* ( (ship-load-list (sail-calc-load-list ship-list 'max
						 comm quant-list))
	    (ships-left ship-list)
	    (quants-left quant-list)
	    )
      (setq ship-load-list (sail-load-ships-internal comm ship-load-list
						     "unload"))
      (if (sail-calc-load-list ship-list 'max comm quant-list)
	  (list 'halt (list 'sail-quota-commodity-max (list 'quote
comm) (list 'quote quant-list)))
	nil)
      )
    )
  )
(register-sail-command 'sail-quota-commodity-max)

(defun sail-load-commodity (ship-list comm quant-list)
  
  "This function emits commands necessary to increase each ship's
stock by the corresponding number in QUANT-LIST.  It does this by
checking current ship levels and mutating into a
sail-quota-commodity-min command.

  Alpha release: See the warnings on sail-quota-commodity-min."
  
  (if (nlistp quant-list)
      (setq quant-list (make-list (length ship-list) quant-list))
    )
  (let ( (ships-left ship-list)
	 (quants-left quant-list)
	 )
    (while ships-left
      (setcar quants-left (+ (car quants-left)
			     (dynamic-ship-recall (car ships-left) comm))
	      )
      (setq ships-left (cdr ships-left)
	    quants-left (cdr quants-left) )
      )
    (sail-quota-commodity-min ship-list comm quant-list)
    ))
(register-sail-command 'sail-load-commodity)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Map for user driving of ships
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-ship-commands-map nil
  "Keymap used from empire shell OR the map with ship manipulation commands.")

(let ((c-mp (make-sparse-keymap)))
  (define-key c-mp "?"	  'help-for-empire-ship-commands-map)
  (define-key c-mp "\C-h" 'help-for-empire-ship-commands-map)
  (define-key c-mp help-character 'help-for-empire-ship-commands-map)
  
  (define-key c-mp "f"		'empire-map-forget-ships)
  (define-key c-mp "d"		'describe-sail-commands)
  (define-key c-mp "o"		'sail-do-orders)
  (define-key c-mp "s"		'empire-read-ships) ; also on x map
  (define-key c-mp "c"		'empire-read-cargos); also on x map
						   
   ;; pings, lookouts, parse navs?
  (setq empire-ship-commands-map c-mp))

(make-help-screen help-for-empire-ship-commands-map
		  "d o s c f"
		  "You have discovered the empire tool ship commands
   From here, you can use the following options:

d	Describe all code that can be used as a sail order
o	Have the ships execute their Orders
s	extract ship data
c	extract cargo data
f	take all ship designations off the map. Will forget enemy ship positions
	  but will also forget obsolete data.

Use \\[help-for-empire-ship-commands-map] for help on redistribution.
Use \\[help-for-empire-extract-map] for help on data extraction.
Please use \\[describe-key] to find out more about any of the other keys."
		  empire-ship-commands-map)


;; Things to do

;; have a mode that will display and support editing of a ship/fleet's orders.
;; have more ship check hooks
;;   i.e. auto load of food, auto load of guns shells, mil

;; Separate fleet lists and orders into 2 data structures
;;   Would be nice to have alphanumeric ships lists initially extracted
;;   from empire ship commands but with additional multiple diget lists only
;;   (you want to do what?  I don't understand -thoth)
;;   known to the tool. The key of the list should key the orders.
;;   For example, I might have one set of orders for fleet N, another set
;;   for any destroyer (load if in port), one for carriers (load if in port),
;;   etc. Ship hooks would add ships to fleets based on their types.
;;
;;   Lynn wants ships to be able to follow two sets of orders at once. I
;;   think it's a good idea.  I'm working on it.
;;
;; need sail-add-list and sail-add-orders
;;      sail-add-ship?
;;
;; also need some sort of fleet/order display and some command to show all
;;   orders pertinent to a set of ships. Also need "canned" orders for
;;   beginners to use.
;; I am going to develop a list test it out.  The software has to work
;; first though :)
;;
