;;;;;;;;;;;;;;;;;;;;;;;;;;; -*- Mode: Emacs-Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;
;; emp-const.el -- Game specific constants for Gnu Emacs Empire Tool (GEET)
;; 
;; Copyright (c) 1990,1991 Lynn Randolph Slater, Jr
;; 
;; AFSID           : $__Header$
;; Author          : Lynn Slater x2048
;; Created On      : Fri Jan 25 11:24:38 1991
;; Last Modified By: Lynn Slater x2048
;; Last Modified On: Thu Feb 14 19:17:05 1991
;; Update Count    : 19
;; Status          : GEET General Release 2d Patch 0
;; 
;; HISTORY
;; 4-Feb-1991		Lynn Slater x2048	
;;    Last Modified: Mon Feb  4 11:38:28 1991 #10 (Lynn Slater x2048)
;;    added resource, bits, and mine info to empire-pos
;; PURPOSE
;; 	This file constains those values that may need changing from one
;; empire universe to another but not from one country to another. Edit
;; this to reflect local game changes.
;;
;; If you change this file, you MUST recompile GEET or some macros may have
;; used old data to encode locations and offsets into code you will be calling.
;; TABLE OF CONTENTS
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; The contents of this file ARE copyrighted but permission to use, modify,
;; and distribute this code is granted as described in the file
;; emp-install.el which should have been distributed with this file. These
;; terms constitute what the Free Software Foundation calls a COPYLEFT.
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(provide 'emp-const)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Sector data storage
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defconst empire-pos 			; Handled specially in save-empire
  (let ((index 2))
    (list	
     ;; The order of labels in the first part of this list
     ;; must match the data from the dump command or else
     ;; the read-dump will mess up.
     
     ;; If you change this, all code that uses position-of
     ;; (almost all code does) MUST be recompiled.
     
     ;; x, y, and des MUST have offsets of 0, 1, and 2
     (cons 'x		0)		; required offset
     (cons 'y		1)		; required offset
     (cons 'des		2)		; required offset
     
     (cons 'sdes	(setq index (1+ index)))
     (cons 'eff		(setq index (1+ index)))
     (cons 'mob		(setq index (1+ index)))
     (cons '*		(setq index (1+ index)))
     
     ;; Decision: I will chose to not store res levels
     (cons 'min		(setq index (1+ index)))
     (cons 'gold	(setq index (1+ index)))
     (cons 'fert	(setq index (1+ index)))
     (cons 'ocontent	(setq index (1+ index)))
     (cons 'uran	(setq index (1+ index)))
     
     (cons 'work	(setq index (1+ index)))
     (cons 'avail	(setq index (1+ index)))
     (cons 'terr	(setq index (1+ index)))
     
     (cons 'civ		(setq index (1+ index)))
     (cons 'mil		(setq index (1+ index)))
     (cons 'uw		(setq index (1+ index)))
     
     (cons 'food	(setq index (1+ index)))
     (cons 'shell	(setq index (1+ index)))
     (cons 'gun		(setq index (1+ index)))
     (cons 'pet		(setq index (1+ index)))
     (cons 'iron	(setq index (1+ index)))
     (cons 'dust	(setq index (1+ index)))
     (cons 'bar		(setq index (1+ index)))
     (cons 'oil		(setq index (1+ index)))
     (cons 'lcm		(setq index (1+ index)))
     (cons 'hcm		(setq index (1+ index)))
     (cons 'rad		(setq index (1+ index)))
     
     (cons 'u_del	nil)
     (cons 'f_del	nil)
     (cons 's_del	nil)
     (cons 'g_del	nil)
     (cons 'p_del	nil)
     (cons 'i_del	nil)
     (cons 'd_del	nil)
     (cons 'b_del	nil)
     (cons 'o_del	nil)
     (cons 'l_del	nil)
     (cons 'h_del	nil)
     (cons 'r_del	nil)
     (cons 'u_cut	nil)
     (cons 'f_cut	nil)
     (cons 's_cut	nil)
     (cons 'g_cut	nil)
     (cons 'p_cut	nil)
     (cons 'i_cut	nil)
     (cons 'd_cut	nil)
     (cons 'b_cut	nil)
     (cons 'o_cut	nil)
     (cons 'l_cut	nil)
     (cons 'h_cut	nil)
     (cons 'r_cut	nil)
     
     (cons 'dist_path	(setq index (1+ index)))
     (cons 'dist_x	(setq index (1+ index)))
     (cons 'dist_y	(setq index (1+ index)))
     
     (cons 'u_dist	nil)
     (cons 'f_dist	nil)
     (cons 's_dist	nil)
     (cons 'g_dist	nil)
     (cons 'p_dist	nil)
     (cons 'i_dist	nil)
     (cons 'd_dist	nil)
     (cons 'b_dist	nil)
     (cons 'o_dist	nil)
     (cons 'l_dist	nil)
     (cons 'h_dist	nil)
     (cons 'r_dist	nil)
     
     ;; that is the last of the required labels (in my
     ;; version of empire)
     
     (cons 'own		(setq index (1+ index)))
     
     ;; Production data
     (cons 'wkfc	(setq index (1+ index)))
     (cons 'will	(setq index (1+ index)))
     (cons 'make	(setq index (1+ index)))
     (cons 'use1	(setq index (1+ index)))
     (cons 'use2	(setq index (1+ index)))
     (cons 'use3	(setq index (1+ index)))
     (cons 'max1	(setq index (1+ index)))
     (cons 'max2	(setq index (1+ index)))
     (cons 'max3	(setq index (1+ index)))
     (cons 'max		(setq index (1+ index)))
     
     ;; Is there a mine field in these waters?
     (cons 'mine	(setq index (1+ index)))
     
     ;; Plane info
     (cons 'intercept	(setq index (1+ index)))
     
     ;; routing
     (cons 'dependents	(setq index (1+ index)))

     ;; survey
     (cons 'survey	(setq index (1+ index)))

     ;; bits
     (cons 'bits	(setq index (1+ index)))
     
     ;; growth
     ))
  "A alist of '(label . offset) that lets us know where to store each
thing. The labels must exactly match those of the dump command up to as
many labels as the dump command uses.
  The macro 'position-of accesses this list
  The function dynamic-position-of accesses this list

  A item of information that is not to be stored (because of disk space/memory
tradoffs) should have nil as its offset."
  )

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Planes
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
 (defvar empire-plane-flags
   '(VTOL tactical intercept light cargo bomber missile stealth satellite spy image)
   "Labels of the plane capability flags.")

;;; Different empire universes have different plane values. These values
;;; must be reflected in the empire-plane-cap constant compiled into this
;;; file. To make a new constant appropiate to your game, read the source
;;; file named below and uncomment this and evaluate it to create a new
;;; empire-plane-cap constant. after you run the function, cut the constant
;;; out of the temp buffer and place it in here.  Then you can kill the
;;; temporary buffer. 
; (defun create-empire-plane-cap ()
;   "This will create the empire-plane-cap constant from the plchrstr struct.
; This file is in the EMP/lib/global/plane.c file.
; 
; This code assumes that the fields in the structure are as follows:
; /*       name          lcm hcm $$$ tech acc load att/def ran mil gas flags */
; It will require modification if it changes.
; 
; It also uses the global 'empire-cap-pos', and requires that the variable names
; in this function match the key's in the global list.  It also requires the keys
; to be in numerical order.
; 
; This will only search while there are continguous plane definitions in the
; structure.  There is rudimentary 1st column comment and blank line skipping
; included.
; 
; To run this, place the cursor at the beginning of the first line of plane
; definitions (should be fighter 1).  Then type Esc-Esc-(create-empire-plane-cap)"
; 
;   (let ((output-buf (get-buffer-create "*empire-plane-cap*"))
; 	name lcm hcm cost tech acc load att def ran mil gas flags)
;     (save-excursion
;       (set-buffer output-buf)
;       (insert "(defconst empire-plane-cap
;   '(
;     "))
;     (while
; 	(looking-at "[ \t]*\"\\([---/ a-z1-9]*\\)\"[ \t]*,[ \t]*\\([0-9]+\\)[ \t]*,[ \t]*\\([0-9]+\\)[ \t]*,[ \t]*\\([0-9]+\\)[ \t]*,[ \t]*\\([0-9]+\\)[ \t]*,[ \t]*\\([0-9]+\\)[ \t]*,[ \t]*\\([0-9]+\\)[ \t]*,[ \t]*\\([0-9]+\\)[ \t]*\\(,\\)")
;       (setq name (buffer-substring (match-beginning 1) (match-end 1))
; 	    lcm (buffer-substring (match-beginning 2) (match-end 2))
; 	    hcm (buffer-substring (match-beginning 3) (match-end 3))
; 	    cost (buffer-substring (match-beginning 4) (match-end 4))
; 	    tech (buffer-substring (match-beginning 5) (match-end 5))
; 	    acc (buffer-substring (match-beginning 6) (match-end 6))
; 	    load (buffer-substring (match-beginning 7) (match-end 7))
; 	    att (buffer-substring (match-beginning 8) (match-end 8)))
;       (goto-char (match-end 9))
;       (if (looking-at "[ \t]*\\([0-9]+\\)[ \t]*,[ \t]*\\([0-9]+\\)[ \t]*,[ \t]*\\([0-9]+\\)[ \t]*,[ \t]*\\([0-9]+\\)[ \t]*,[ \t]*\\(P_[A-Z]|?\\)+[ \t]*,[ \t]*\n")
; 	  (setq def (buffer-substring (match-beginning 1) (match-end 1))
; 		ran (buffer-substring (match-beginning 2) (match-end 2))
; 		mil (buffer-substring (match-beginning 3) (match-end 3))
; 		gas (buffer-substring (match-beginning 4) (match-end 4))
; 		flags (buffer-substring (match-beginning 5) (match-end 5)))
; 	(unless (zerop (length name))
; 	  (message "Incomplete line parsed!")
; 	  (sit-for 3)))
; 
;       (forward-line 1)
;       (while (looking-at "[ \t]*/\\*\\|[ \t]*\n") (forward-line 1)) ;rudimentary comment skipping.
;       (when (not (zerop (length name)))
; 	(save-excursion
; 	  (set-buffer output-buf)
; 	  (insert "(\"")
; 	  ;; for some reason, names are 14 characters.
; 	  (let ((len (length name)))
; 	    (if (>= len 14)
; 		(insert (substring name 0 14))
; 	      (insert name)
; 	      (setq len (- 14 len))
; 	      (while (not (zerop len))
; 		(insert " ")
; 		(setq len (1- len)))))
; 	  (insert "\"")
; 	  ;; now insert the rest of the items in correct order.
; 	  (dolist (item (cdr empire-cap-pos))
; 	    (if (eq 'flags (car item))
; 		(insert " (" flags ")")
; 	      (insert " " (eval (car item)))))
; 	  (insert ")\n    "))))
;     ;; now insert the correct names for the flags.
;     (pop-to-buffer output-buf)
;     (set-buffer output-buf)
;     (insert ")
;   \"Defines what planes can do in empire\")\n")
;     (let ((start (point-min)))
;       (dolist (rep '(("P_V" . "VTOL") ("P_T" . "tactical") ("P_F" . "intercept")
; 		     ("P_L" . "light") ("P_C" . "cargo") ("P_B" . "bomber")
; 		     ("P_M" . "missile") ("P_X" . "stealth") ("P_O" . "satellite")
; 		     ("P_S" . "spy") ("P_I" . "image")))
; 	(goto-char start)
; 	(replace-string (car rep) (cdr rep)))
;       (goto-char start)
;       (replace-string "|" " ")))
;   (goto-char (point-min)))
; 

(defconst empire-plane-cap
  '(
    ("fighter 1     " (VTOL tactical intercept light) 1 1 30)
    ("lt bomber 1   " (VTOL tactical bomber light) 2 1 30)
    ("transport 1   " (cargo) 7 2 0)
    ("hvy bomber 1  " (bomber) 6 2 85)
    ("long bomber 1 " (bomber) 2 4 120)
    ("fighter 2     " (tactical intercept light) 1 1 25)
    ("lt bomber 2   " (tactical bomber light) 3 1 25)
    ("escort 2      " (intercept light) 0 2 0)
    ("chopper 1     " (cargo VTOL light) 5 1 0)
    ("transport 2   " (cargo) 9 3 0)
    ("long transport" (cargo) 3 5 0)
    ("hvy bomber 2  " (bomber) 10 3 75)
    ("long bomber 2 " (bomber) 4 5 75)
    ("jet fighter   " (tactical intercept light) 2 4 40)
    ("jet lt bomber " (tactical bomber light) 5 5 20)
    ("jet hvy bomber" (bomber) 16 8 65)
    ("jet transport " (cargo) 16 7 0)
    ("jet fighter 2 " (tactical intercept VTOL light) 2 5 30)
    ("jet f/b       " (bomber tactical) 8 6 40)
    ("spyplane 1    " (spy) 0 5 0)
    ("spyplane 2    " (spy image) 0 8 0)
    ("stealth bomber" (bomber stealth) 12 4 60)
    ("stealth fighte" (tactical intercept stealth) 3 3 30)
    ("chopper 2     " (tactical bomber VTOL light) 3 2 10)
    ("chopper 3     " (cargo VTOL light) 6 3 0)
    ("chopper 4     " (tactical bomber VTOL light) 5 3 10)
    ("ssm 1         " (missile light) 5 0 60)
    ("ssm 2         " (missile light) 7 0 50)
    ("irbm 1        " (missile) 5 7 60)
    ("irbm 2        " (missile) 7 8 50)
    ("icbm 1        " (missile) 7 9 70)
    ("icbm 2        " (missile) 10 10 50)
    ("icbm 3        " (missile) 15 11 20)
    ("icbm 4        " (missile) 30 12 20)
    ("slbm 1        " (missile light) 7 6 70)
    ("slbm 2        " (missile light) 11 7 70)
    ("sam 1         " (missile intercept) 4 6 30)
    ("sam 2         " (missile intercept light) 4 6 20)
    ("sam 3         " (missile intercept) 4 6 10)
    ("sam 4         " (missile intercept light) 4 6 10)
    ("landsat 1     " (satellite) 0 9 0)
    ("landsat 2     " (satellite image) 0 9 0)
    ("spysat 1      " (satellite spy) 0 9 0)
    ("spysat 2      " (satellite spy image) 0 9 0)
    ("anti-sat 1    " (satellite missile light) 6 4 70)
    ("anti-sat 2    " (satellite missile light) 8 5 60)
    ("anti-sat 3    " (satellite missile light) 8 6 50)
    ("anti-sat 4    " (satellite missile) 16 9 40)
    )
  "Defines what planes can do in empire")

(defvar empire-plane-pos		; Handled specially in save-empire
  (list
   (cons '\#	0)
   (cons 'type	1)
   (cons 'x	2)
   (cons 'y	3)
   (cons 'w	4)
   (cons 'eff	5)
   (cons 'mob	6)
   (cons 'att	7)
   (cons 'def	8)
   (cons 'tech	9)
   (cons 'ran	10)
   (cons 'hard	11)
   (cons 'ship	12)
   (cons 'nuke	13)
   )
  "Offsets of the plane data in the empire-planes list")

(defvar empire-cap-pos
  (let ((index 0))
    (list
      (cons	'name	0)				; name MUST be first.  It's the key.
       (cons	'flags	(setq index (1+ index)))	; this is a list
       (cons	'load	(setq index (1+ index)))
       (cons	'gas	(setq index (1+ index)))
       (cons	'acc	(setq index (1+ index)))
       ;; can't figure out a use for these right now...
       ;;(cons	'cost	(setq index (1+ index)))
       ;;(cons	'hcm	(setq index (1+ index)))
       ;;(cons	'lcm	(setq index (1+ index)))
       ;;(cons	'tech	(setq index (1+ index)))
       ;;(cons	'att	(setq index (1+ index)))
       ;;(cons	'def	(setq index (1+ index)))
       ;;(cons	'ran	(setq index (1+ index)))
       ;;(cons	'mil	(setq index (1+ index)))
       ))
   "Offsets of the plane capabilities in the empire-plane-cap constant.")

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Ships
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defconst empire-ship-pos
  ( list
    ;;	field name	index

    ;; attributes common to both reports
    ;;(cons 'shpnum	0) ; this is the key
    (cons 'type		1)
    (cons 'x		2)
    (cons 'y		3)
    ;; stuff from the ship report
    (cons 'flt		4)
    (cons 'eff		5)
    (cons 'civ		6)
    (cons 'mil		7)
    (cons 'uw		8)
    (cons 'food		9)
    (cons 'pln		10)
    (cons 'mob		11)
    (cons 'tech		12)
    ;; commodities from cargo
    (cons 'shell	13)
    (cons 'gun		14)
    (cons 'pet		15)
    (cons 'iron		16)
    (cons 'dust		17)
    (cons 'bar		18)
    (cons 'oil		19)
    (cons 'lcm		20)
    (cons 'hcm		21)
    (cons 'rad		22)
    (cons 'owner	23)
    ;; future orders information?
    ;; here
    )
  "Offsets of the ship data (extracted from the ship and cargo reports)
stored in empire-ships" 
  )

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Commodities
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defconst empire-basic-commodity-list
  '(
    ("bar")
    ("dust")
    ("food")
    ("gun")
    ("hcm")
    ("iron")
    ("lcm")
    ("oil")
    ("pet")
    ("rad")
    ("shell")
    ("uw")		;;; "uw" is really a commodity
    )
  "A list of `goods' commodities.  Used for completing-reads.
These names must match those named in `empire-pos'."
  )

(defconst empire-people-commodity-list
  '(
    ("civ")
    ("mil")
    )
  "A list of `people' commodities.  Used for completing-reads.
These names must match those named in `empire-pos'."
  )

(defconst empire-complete-commodity-list
  (append empire-people-commodity-list empire-basic-commodity-list)
  "A list of all things that can be moved, used to explore, etc."
  )

(defconst empire-survey-commodity-list
  (append
   '(
     ("eff")
     ("fert")
     ("gold")
     ("min")
     ("mob")
     ("ocontent")
     ("uran")
     ("work")
     )
   empire-complete-commodity-list)
  "A list of all things that can be surveyed."
  )


;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Types of things
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defconst empire-sector-types-regexp
  "^[-cudpimghw*aojkftrnl+)!#b%e=]"
  "A regular-expression matching the symbols that are sector types.")

(defconst empire-ship-types-regexp
  "^[TABHCODSUMGLYPFW]"
  "A regular-expression matching the symbols that are ship types. Ships
should be listed in the order of bigger to smaller,as seen by raidar. Thus,
an aircraft carrier is first and a patrol boat last.")

(defvar ukn nil "holds last unknown sector type")

(defun empire-code (type &optional pos)
  "Retuns empires' code for a given sector type"
  (cond
   ((string-equal type "mine") "m")
   ((string-equal type "oil") "o")
   ((string-equal type "agribusiness") "a")
   ((string-equal type "wilderness") "-")
   ((string-equal type "mountain") "^")
   ((string-equal type "fortress") "f")
   ((string-equal type "radar") ")")
   ((string-equal type "harbor") "h")
   ((string-equal type "gold") "g")
   ((string-equal type "shell") "i")
   ((string-equal type "enlistment") "e")
   ((string-equal type "technical") "t")
   ((string-equal type "library/school") "l")
   ((string-equal type "uranium") "u")
   ((string-equal type "highway") "+")
   ((string-equal type "refinery") "%")
   ((string-equal type "bridge") 
    (if (and pos (save-excursion (goto-char pos) (looking-at " span")))
	"=" "#"))
   ((string-equal type "bank") "b")
   ((string-equal type "airfield") "*")
   ((string-equal type "research") "r")
   ((string-equal type "light") "j")
   ((string-equal type "capital") "c")
   ((string-equal type "defense") "d")
   ((string-equal type "sea") ".")
   ((string-equal type "technical") "t")
   ((string-equal type "warehouse") "w")
   ((string-equal type "wasteland") "\\")
   ((string-equal type "nuclear")
    (if (and pos (save-excursion (goto-char pos) (looking-at " sub")))
	"N" "n"))
   ((string-equal type "park") "p")
   ((string-equal type "heavy")
    (if (and pos (save-excursion (goto-char pos) (looking-at " cruiser")))
	"H" "k"))
   ((string-equal type "tanker") "T")
   ((string-equal type "destroyer") "D")
   ((string-equal type "battleship") "B")
   ((string-equal type "aircraft") "A")
   ((string-equal type "cargo") "C")
   ((string-equal type "patrol") "P")
   ((string-equal type "ore") "O")
   ((string-equal type "slave") "S")
   ((string-equal type "submarine") "U")
   ((string-equal type "minesweep") "U")
   ((string-equal type "fishing") "F")
   ((string-equal type "yacht") "Y")

   ;; special-case "`" is shipping lane

   (t (setq ukn (cons type ukn))
      "&")
   ))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun initialize-sector-data ()
  "Initialize the sector data.  This has been made into a function for
development purposes; it is easier to force reinitialization of
variables this way."
  (progn
    ;;
    ;;	          name	produces	    move-cost	packaging-index
    ;;					  (0 = illegal)
    (define-sector "." sector-produces-nothing	0   empire-normal-packaging)
    (define-sector "?" sector-produces-nothing	0   empire-normal-packaging)
    (define-sector "^" sector-produces-nothing  127 empire-normal-packaging)
    (define-sector "s" sector-produces-nothing	0   empire-normal-packaging)
    (define-sector "\\" sector-produces-nothing	0   empire-normal-packaging)
    (define-sector "-" sector-produces-nothing	4   empire-normal-packaging)
    (define-sector "c" sector-produces-nothing	2   empire-normal-packaging)
    (define-sector "u" sector-produces-uranium	2   empire-normal-packaging)
    (define-sector "p" sector-produces-happiness 2   empire-normal-packaging)
    (define-sector "d" sector-produces-guns	2   empire-normal-packaging)
    (define-sector "i" sector-produces-shells	2   empire-normal-packaging)
    (define-sector "m" sector-produces-iron	2   empire-normal-packaging)
    (define-sector "g" sector-produces-dust	2   empire-normal-packaging)
    (define-sector "h" sector-produces-nothing	2   empire-normal-packaging)
    (define-sector "w" sector-produces-nothing	2   empire-warehouse-packaging)
    (define-sector "*" sector-produces-nothing	2   empire-normal-packaging)
    (define-sector "a" sector-produces-food	2   empire-normal-packaging)
    (define-sector "o" sector-produces-oil	2   empire-normal-packaging)
    (define-sector "j" sector-produces-lcm	2   empire-normal-packaging)
    (define-sector "k" sector-produces-hcm	2   empire-normal-packaging)
    (define-sector "f" sector-produces-nothing	2   empire-normal-packaging)
    (define-sector "t" sector-produces-tech	2   empire-normal-packaging)
    (define-sector "r" sector-produces-research	2   empire-normal-packaging)
    (define-sector "n" sector-produces-nothing	2   empire-normal-packaging)
    (define-sector "l" sector-produces-education 2   empire-normal-packaging)
    (define-sector "+" sector-produces-nothing	1   empire-normal-packaging)
    (define-sector ")" sector-produces-nothing	2   empire-normal-packaging)
    (define-sector "!" sector-produces-nothing	2   empire-normal-packaging)
    (define-sector "#" sector-produces-nothing	2   empire-normal-packaging)
    (define-sector "=" sector-produces-food	1   empire-normal-packaging)
    (define-sector "b" sector-produces-bar	2   empire-bank-packaging)
    (define-sector "%" sector-produces-petrol	2   empire-normal-packaging)
    (define-sector "e" sector-produces-nothing	2   empire-normal-packaging)
  )
)

(defun initialize-commodity-data ()
  "Initialize the commodity data.  This has been made into a function
for development purposes; it is easier to force reinitialization of
variables this way."
  (progn
    ;;					|-------- packaging -------|
    ;;					(0)	(1)	(2)	(3)
    ;;			name	lbs	reg	wareh.	urban	bank
    (define-commodity	'civ	1	[ 1	1	10	1 ])
    (define-commodity	'mil	1	[ 1	1	1	1 ])
    (define-commodity	'shell	1	[ 1	10	1	1 ])
    (define-commodity	'gun	10	[ 1	10	1	1 ])
    (define-commodity	'pet	1	[ 1	10	1	1 ])
    (define-commodity	'iron	1	[ 1	10	1	1 ])
    (define-commodity	'dust	5	[ 1	10	1	1 ])
    (define-commodity	'bar	50	[ 1	10	1	1 ])
    (define-commodity	'food	1	[ 1	10	1	1 ])
    (define-commodity	'oil	1	[ 1	10	1	1 ])
    (define-commodity	'lcm	1	[ 1	10	1	1 ])
    (define-commodity	'hcm	1	[ 1	10	1	1 ])
    (define-commodity	'uw	2	[ 1	10	1	1 ])
    (define-commodity	'rad	1	[ 1	10	1	1 ])
  )
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Most empire regular expressions for value returns are in emp-shell.el
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
